/*!\file:  DomainOutlineRead.cpp
 * \brief DomainOutlineRead.c: read the grid coordinates defined in a domain 
 * outline from Argus (.exp file). The first contour in the file is for 
 * the outside domain outline. The following contours represent holes in
 * the domain.
 */

#include "stdio.h"
#include "../Alloc/alloc.h"
#include "../../include/macros.h"
#include "../Exceptions/exceptions.h"

int DomainOutlineRead(int* pnprof,int** pprofngrids,double*** ppprofx,double*** ppprofy,char* domainname){

	
	/*Error management: */
	int noerr=1;
	int i,j,counter;

	/*I/O: */
	FILE* fid=NULL;
	char chardummy[256];
	int  idummy;
	double  ddummy;

	/*output: */
	int nprof; //number of profiles in the domainname file
	int* profngrids=NULL; //array holding the number of grids for the nprof profiles
	double** pprofx=NULL; //array of profiles x coordinates
	double** pprofy=NULL; //array of profiles y coordinates

	/*For each profile: */
	int n;
	double* x=NULL;
	double* y=NULL;

	/*open domain outline file for reading: */
	if ((fid=fopen(domainname,"r"))==NULL){
		ISSMERROR("%s%s","could not find domain file ",domainname); 
		noerr=0; goto cleanupandreturn;
	}

	/*Do a first pass through the domainname file, to figure out how many profiles 
	 *we need to read: */
	nprof=1;
	for(;;){
		fscanf(fid,"%s %s\n",chardummy,chardummy);
		fscanf(fid,"%s %s\n",chardummy,chardummy);
		fscanf(fid,"%s %s %s %s\n",chardummy,chardummy,chardummy,chardummy);
		fscanf(fid,"%u %s\n",&n,chardummy);
		fscanf(fid,"%s %s %s %s %s\n",chardummy,chardummy,chardummy,chardummy,chardummy);
		for (i=0;i<n;i++){
			fscanf(fid,"%lf %lf\n",&ddummy,&ddummy);
		}
		/*Ok, we have faked one profile reading, check whether we are at the end of the file, otherwise, keep fake reading next profile:*/
		if (feof(fid)){
			break;
		}
		nprof++;
	}
	#ifdef _ISSM_DEBUG_
		printf("Number of profiles in domain outline file: %i\n",nprof);
	#endif
	
	/*Allocate and initialize all the profiles: */
	profngrids=(int*)xmalloc(nprof*sizeof(int));
	pprofx=(double**)xmalloc(nprof*sizeof(double*));
	pprofy=(double**)xmalloc(nprof*sizeof(double*));
	for (i=0;i<nprof;i++){
		pprofx[i]=NULL;
		pprofy[i]=NULL;
	}

	/*Reaset file pointer to beginning of file: */
	fseek(fid,0,SEEK_SET);

	/*Start reading profiles: */
	for(counter=0;counter<nprof;counter++){

		/*Skip header: */
		fscanf(fid,"%s %s\n",chardummy,chardummy);
		fscanf(fid,"%s %s\n",chardummy,chardummy);
		fscanf(fid,"%s %s %s %s\n",chardummy,chardummy,chardummy,chardummy);
		
		/*Get number of profile grids: */
		fscanf(fid,"%u %s\n",&n,chardummy);
	
		/*Skip next line: */
		fscanf(fid,"%s %s %s %s %s\n",chardummy,chardummy,chardummy,chardummy,chardummy);

		/*Allocate grids: */
		x=(double*)xmalloc(n*sizeof(double));
		y=(double*)xmalloc(n*sizeof(double));
		

		/*Read grids: */
		for (i=0;i<n;i++){
			fscanf(fid,"%lf %lf\n",x+i,y+i);
		}

		/*Now check that we are dealing with open contours: */
		if((x[0]==x[n-1]) && (y[0]==y[n-1])){
			n=n-1;
		}

		/*Assign pointers: */
		profngrids[counter]=n;
		pprofx[counter]=x;
		pprofy[counter]=y;
	}

	/*close domain outline file: */
	fclose(fid);

	
	#ifdef _ISSM_DEBUG_
	for (i=0;i<nprof;i++){
		printf("Profile #%i\n",i);
		x=pprofx[i];
		y=pprofy[i];
		for (j=0;j<profngrids[i];j++){
			printf("   %lf %lf\n",x[j],y[j]);
		}
	}
	#endif


	cleanupandreturn: 
	/*Free ressources: */
	if(!noerr){
		xfree((void**)&x);
		xfree((void**)&y);
		n=-1;
	}
	
	/*Assign output pointers: */
	*pnprof=nprof;
	*pprofngrids=profngrids;
	*ppprofx=pprofx;
	*ppprofy=pprofy;
	return noerr;
}

