/* \file WriteParams.c:
 * \brief: interface for writing parameters dataset into a matlab structure
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif


#ifdef _SERIAL_
#include <mex.h>

#include "../objects/Param.h"
#include "./io.h"
#include "../shared/shared.h"
#include "../include/macros.h"

void WriteParams(DataHandle* pdataref,DataSet* parameters){

	/*output: */
	mxArray* dataref=NULL;
	mwSize nfields;
	const	char**	fnames=NULL;
	mwSize		onebyone[2] = {1,1};
	mwSize		ndim=2;
	mxArray*    pfield=NULL;
	mxArray*    pfield2=NULL;

	/*intermediary: */
	int         i,k;
	mxArray*    field=NULL;
	Param*      param=NULL;
	int         integer;
	double      ddouble;
	char*       string;
	double*     doublevec=NULL;
	double*     doublemat=NULL;
	Mat         mat=NULL;
	Vec         vec=NULL;
	char**      stringarray=NULL;
	double*     serial_vec=NULL;
	double*     serial_mat=NULL;
	mwSize      M,N;
	mwSize      dims[2]={0};

	/*Recover data from the parameters dataset: */
	nfields=(mwSize)parameters->Size()-1; //don't include Numpar

	fnames=(const char**)xmalloc(nfields*sizeof(char*));
	
	/*Build structure in matlab workspace with all the parameter fields: */
	for(i=0;i<nfields;i++){
		param=(Param*)parameters->GetObjectByOffset(i+1);
		fnames[i]=(const char*)xmalloc((strlen(param->GetParameterName())+1)*sizeof(char));
		strcpy((char*)fnames[i],param->GetParameterName());
	}
	/*Initialize structure: */
	dataref=mxCreateStructArray( ndim,onebyone,nfields,fnames);

	/*Fill each field: */
	for(i=0;i<nfields;i++){

		param=(Param*)parameters->GetObjectByOffset(i+1);
		
		switch(param->GetType()){
			case INTEGER:
				param->GetParameterValue(&integer);
				mxSetField( dataref, 0, param->GetParameterName(),mxCreateDoubleScalar((double)integer));
				break;

			case DOUBLE:
				param->GetParameterValue(&ddouble);
				mxSetField( dataref, 0, param->GetParameterName(),mxCreateDoubleScalar((double)ddouble));
				break;

			case STRING:
				param->GetParameterValue(&string);
				mxSetField( dataref, 0, param->GetParameterName(),mxCreateString(string));
				break;

			case STRINGARRAY:
				param->GetParameterValue(&stringarray);
				M=param->GetM();
				dims[0]=M;
				dims[1]=1;
				pfield=mxCreateCellArray(2,dims);
				for(k=0;k<M;k++){
					char* string=stringarray[k];
					mxSetCell(pfield,k,mxCreateString(string));
				}
				mxSetField( dataref, 0, param->GetParameterName(),pfield);
				break;

			case DOUBLEVEC:
				param->GetParameterValue(&doublevec);
				M=param->GetM();
				pfield=mxCreateDoubleMatrix(0,0,mxREAL);
				mxSetM(pfield,M);
				mxSetN(pfield,1);
				mxSetPr(pfield,doublevec);
				mxSetField( dataref, 0, param->GetParameterName(),pfield);
				break;

			case DOUBLEMAT:
				param->GetParameterValue(&doublemat);
				M=param->GetM();
				N=param->GetN();
				pfield=mxCreateDoubleMatrix(0,0,mxREAL);
				mxSetM(pfield,N);
				mxSetN(pfield,M);
				mxSetPr(pfield,doublemat);
				//transpose the matrix, written directly to matlab! from C to matlab.
				mexCallMATLAB(1,&pfield2, 1, &pfield, "'");
				mxSetField( dataref, 0, param->GetParameterName(),pfield2);
				break;
		
			case PETSCVEC:
				param->GetParameterValue(&vec);
				VecToMPISerial(&serial_vec,vec);
				VecFree(&vec);
				M=param->GetM();
				pfield=mxCreateDoubleMatrix(0,0,mxREAL);
				mxSetM(pfield,M);
				mxSetN(pfield,1);
				mxSetPr(pfield,serial_vec);
				mxSetField( dataref, 0, param->GetParameterName(),pfield);
				break;
		
			case PETSCMAT:
				param->GetParameterValue(&mat);
				MatToSerial(&serial_mat,mat);
				MatFree(&mat);
				M=param->GetM();
				N=param->GetN();
				pfield=mxCreateDoubleMatrix(0,0,mxREAL);
				mxSetM(pfield,M);
				mxSetN(pfield,N);
				mxSetPr(pfield,serial_mat);
				mxSetField( dataref, 0, param->GetParameterName(),pfield);
				break;
			default:
				ISSMERROR("%s%i","unknown parameter type: ",param->GetType());
				break;
		}
	}

	/*Assign output pointers:*/
	*pdataref=dataref;

}
#endif
