/*!\file Model.c
 * \brief: implementation of the Model object
 */

#include "./objects.h"
#include "../io/io.h"
#include "../shared/shared.h"
#include "../include/macros.h"
#include "../issm.h"

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "./Model.h"
#include "stdio.h"

/*Object constructors and destructor*/
/*FUNCTION Model::constructor {{{1*/
Model::Model(){

	femmodels=new DataSet(0);
	active=NULL;

}
/*}}}1*/
/*FUNCTION Model::destructor {{{1*/
Model::~Model(){

	delete femmodels;
	/*do not delete active, it just points to one of the femmodels!*/
}
/*}}}1*/

/*Object functions*/
/*FUNCTION Model::AddFormulation(ConstDataHandle MODEL, int analysis_type,int sub_analysis_type){{{1*/
void  Model::AddFormulation(ConstDataHandle MODEL, int analysis_type,int sub_analysis_type){

	/*FemModel: */
	FemModel*           femmodel=NULL;
	DataSet*            elements=NULL;
	DataSet*            nodes=NULL;
	DataSet*            vertices=NULL;
	DataSet*            constraints=NULL;
	DataSet*            loads=NULL;
	DataSet*            materials=NULL;
	DataSet*            parameters=NULL;
	DofVec*                 partition=NULL;
	DofVec*                 tpartition=NULL;
	DofVec*                 yg=NULL;
	Mat                 Rmg=NULL;
	Mat                 Gmn=NULL;
	NodeSets*           nodesets=NULL;
	Vec                 ys=NULL;
	Vec                 ys0=NULL;

	/*intermediary: */
	IoModel* iomodel=NULL;
	
	_printf_("   fill model with matlab workspace data\n");
	IoModelInit(&iomodel,MODEL); 

	_printf_("   specifying analysis\n");
	if (analysis_type!=0){
		iomodel->analysis_type=analysis_type;
	}
	if (sub_analysis_type!=0){
		iomodel->sub_analysis_type=sub_analysis_type;
	}

	_printf_("   create datasets:\n");
	CreateDataSets(&elements,&nodes,&vertices,&materials,&constraints,&loads,&parameters,iomodel,MODEL);

	_printf_("   create degrees of freedom: \n");
	Dofx( &partition,&tpartition,elements,nodes, vertices,parameters);
	
	_printf_("   create single point constraints: \n");
	SpcNodesx( &yg, nodes,constraints); 
	
	_printf_("   create rigid body constraints:\n");
	MpcNodesx( &Rmg, nodes,constraints); 
	
	_printf_("   create node sets:\n");
	BuildNodeSetsx(&nodesets, nodes);

	_printf_("   reducing single point constraints vector:\n");
	Reducevectorgtosx(&ys,&ys0, yg->vector,nodesets);
	
	_printf_("   normalizing rigid body constraints matrix:\n");
	NormalizeConstraintsx(&Gmn, Rmg,nodesets);

	_printf_("   configuring element and loads:\n");
	ConfigureObjectsx(elements, loads, nodes, vertices, materials,parameters);

	_printf_("   process parameters:\n");
	ProcessParamsx( parameters, partition->vector);

	_printf_("   free ressources:\n");
	DeleteIoModel(&iomodel);

	/*Use all the data created to create an femmodel: */
	femmodel=new FemModel(elements, nodes, vertices, constraints,loads,materials,parameters,
			              partition,tpartition,yg,Rmg,Gmn,nodesets,ys,ys0);

	/*Add to femmodels in model: */
	femmodels->AddObject((Object*)femmodel);

}
/*}}}1*/
/*FUNCTION Model::AddFormulation(ConstDataHandle MODEL, int analysis_type) {{{1*/
void  Model::AddFormulation(ConstDataHandle MODEL, int analysis_type){

	/*FemModel: */
	FemModel*           femmodel=NULL;
	DataSet*            elements=NULL;
	DataSet*            nodes=NULL;
	DataSet*            vertices=NULL;
	DataSet*            constraints=NULL;
	DataSet*            loads=NULL;
	DataSet*            materials=NULL;
	DataSet*            parameters=NULL;
	DofVec*                 partition=NULL;
	DofVec*                 tpartition=NULL;
	DofVec*                 yg=NULL;
	Mat                 Rmg=NULL;
	Mat                 Gmn=NULL;
	NodeSets*           nodesets=NULL;
	Vec                 ys=NULL;
	Vec                 ys0=NULL;

	/*intermediary: */
	IoModel* iomodel=NULL;
	int sub_analysis_type;
	
	_printf_("   fill model with matlab workspace data\n");
	IoModelInit(&iomodel,MODEL); 

	_printf_("   specifying analysis\n");
	if (analysis_type!=0){
		iomodel->analysis_type=analysis_type;
	}
	/*sub_analysis_type should default to none, as it was not specified: */
	sub_analysis_type=NoneAnalysisEnum;
	
	if (sub_analysis_type!=0){
		iomodel->sub_analysis_type=sub_analysis_type;
	}

	_printf_("   create datasets:\n");
	CreateDataSets(&elements,&nodes,&vertices,&materials,&constraints,&loads,&parameters,iomodel,MODEL);

	_printf_("   create degrees of freedom: \n");
	Dofx( &partition,&tpartition,elements,nodes, vertices,parameters);
	
	_printf_("   create single point constraints: \n");
	SpcNodesx( &yg, nodes,constraints); 
	
	_printf_("   create rigid body constraints:\n");
	MpcNodesx( &Rmg, nodes,constraints); 
	
	_printf_("   create node sets:\n");
	BuildNodeSetsx(&nodesets, nodes);

	_printf_("   reducing single point constraints vector:\n");
	Reducevectorgtosx(&ys,&ys0, yg->vector,nodesets);
	
	_printf_("   normalizing rigid body constraints matrix:\n");
	NormalizeConstraintsx(&Gmn, Rmg,nodesets);

	_printf_("   configuring element and loads:\n");
	ConfigureObjectsx(elements, loads, nodes, vertices, materials,parameters);

	_printf_("   process parameters:\n");
	ProcessParamsx( parameters, partition->vector);

	_printf_("   free ressources:\n");
	DeleteIoModel(&iomodel);

	/*Use all the data created to create an femmodel: */
	femmodel=new FemModel(elements,nodes,vertices, constraints,loads,materials,parameters,
			              partition,tpartition,yg,Rmg,Gmn,nodesets,ys,ys0);

	/*Add to femmodels in model: */
	femmodels->AddObject((Object*)femmodel);

}
/*}}}1*/
/*FUNCTION Model::DeepEcho {{{1*/

void Model::DeepEcho(void){

	femmodels->Echo();

}
/*}}}1*/
/*FUNCTION Model::Echo {{{1*/

void Model::Echo(void){

	printf("Models echo: \n");
	printf("   number of fem models: %i\n",femmodels->Size());

}
/*}}}1*/
/*FUNCTION Model::FindParam(int* pparameter,char* parametername {{{1*/

int   Model::FindParam(int* pparameter,char* parametername){

	FemModel* femmodel=NULL;

	/*no analysis_type or sub_analysis_type, find the parameter in the first dataset we find: */
	if (!femmodels->Size())ISSMERROR(" no fem models were found!");

	/*recover first femmodel: */
	femmodel=(FemModel*)femmodels->GetObjectByOffset(0);

	if(!femmodel)return 0;

	femmodel->FindParam(pparameter,parametername);

}
/*}}}1*/
/*FUNCTION Model::FindParam(double* pparameter,char* parametername ){{{1*/
int   Model::FindParam(double* pparameter,char* parametername){
	
	FemModel* femmodel=NULL;

	/*no analysis_type or sub_analysis_type, find the parameter in the first dataset we find: */
	if (!femmodels->Size())ISSMERROR(" no fem models were found!");

	/*recover first femmodel: */
	femmodel=(FemModel*)femmodels->GetObjectByOffset(0);
	
	if(!femmodel)return 0;
	
	femmodel->FindParam(pparameter,parametername);


}
/*}}}1*/
/*FUNCTION Model::FindParam(double** pparameter,int* pM, int *pN,char* parametername) {{{1*/
int   Model::FindParam(double** pparameter,int* pM, int *pN,char* parametername){
	
	FemModel* femmodel=NULL;

	/*no analysis_type or sub_analysis_type, find the parameter in the first dataset we find: */
	if (!femmodels->Size())ISSMERROR(" no fem models were found!");

	/*recover first femmodel: */
	femmodel=(FemModel*)femmodels->GetObjectByOffset(0);
	
	if(!femmodel)return 0;
	
	femmodel->FindParam(pparameter,pM, pN,parametername);


}
/*}}}1*/
/*FUNCTION Model::FindParam(char** pparameter,char* parametername) {{{1*/
int   Model::FindParam(char** pparameter,char* parametername){
	
	FemModel* femmodel=NULL;

	/*no analysis_type or sub_analysis_type, find the parameter in the first dataset we find: */
	if (!femmodels->Size())ISSMERROR(" no fem models were found!");

	/*recover first femmodel: */
	femmodel=(FemModel*)femmodels->GetObjectByOffset(0);
	
	if(!femmodel)return 0;
	
	femmodel->FindParam(pparameter,parametername);

}
/*}}}1*/
/*FUNCTION Model::FindParam(int* pparameter,char* parametername,int analysis_type,int sub_analysis_type) {{{1*/
int   Model::FindParam(int* pparameter,char* parametername,int analysis_type,int sub_analysis_type){
	
	FemModel* femmodel=NULL;

	/*find the correct formulation: */
	femmodel=this->GetFormulation(analysis_type,sub_analysis_type);
	
	if(!femmodel)return 0;

	/*extract our parameter from the found formulation: */
	femmodel->FindParam(pparameter,parametername);
}
/*}}}1*/
/*FUNCTION Model::FindParam(double* pparameter,char* parametername,int analysis_type,int sub_analysis_type) {{{1*/
int   Model::FindParam(double* pparameter,char* parametername,int analysis_type,int sub_analysis_type){
	
	FemModel* femmodel=NULL;

	/*find the correct formulation: */
	femmodel=this->GetFormulation(analysis_type,sub_analysis_type);
	
	if(!femmodel)return 0;

	/*extract our parameter from the found formulation: */
	femmodel->FindParam(pparameter,parametername);
}
/*}}}1*/
/*FUNCTION Model::FindParam(double** pparameter,int* pM, int* pN,char* parametername,int analysis_type,int sub_analysis_type) {{{1*/
int   Model::FindParam(double** pparameter,int* pM, int* pN,char* parametername,int analysis_type,int sub_analysis_type){

	FemModel* femmodel=NULL;

	/*find the correct formulation: */
	femmodel=this->GetFormulation(analysis_type,sub_analysis_type);
	
	if(!femmodel)return 0;

	/*extract our parameter from the found formulation: */
	femmodel->FindParam(pparameter,pM, pN,parametername);
}
/*}}}1*/
/*FUNCTION Model::FindParam(char** pparameter,char* parametername,int analysis_type,int sub_analysis_type) {{{1*/
int   Model::FindParam(char** pparameter,char* parametername,int analysis_type,int sub_analysis_type){

	FemModel* femmodel=NULL;
	
	/*find the correct formulation: */
	femmodel=this->GetFormulation(analysis_type,sub_analysis_type);
	
	if(!femmodel)return 0;

	/*extract our parameter from the found formulation: */
	femmodel->FindParam(pparameter,parametername);
}
/*}}}1*/
/*FUNCTION Model::FindParam(int* pparameter,char* parametername,int analysis_type) {{{1*/
int   Model::FindParam(int* pparameter,char* parametername,int analysis_type){
	
	FemModel* femmodel=NULL;

	/*find the correct formulation: */
	femmodel=this->GetFormulation(analysis_type);
	
	if(!femmodel)return 0;

	/*extract our parameter from the found formulation: */
	femmodel->FindParam(pparameter,parametername);
}
/*}}}1*/
/*FUNCTION Model::FindParam(double* pparameter,char* parametername,int analysis_type) {{{1*/
int   Model::FindParam(double* pparameter,char* parametername,int analysis_type){
	
	FemModel* femmodel=NULL;

	/*find the correct formulation: */
	femmodel=this->GetFormulation(analysis_type);
	
	if(!femmodel)return 0;

	/*extract our parameter from the found formulation: */
	femmodel->FindParam(pparameter,parametername);
}
/*}}}1*/
/*FUNCTION Model::FindParam(double** pparameter,int* pM, int* pN,char* parametername,int analysis_type) {{{1*/
int   Model::FindParam(double** pparameter,int* pM, int* pN,char* parametername,int analysis_type){

	FemModel* femmodel=NULL;

	/*find the correct formulation: */
	femmodel=this->GetFormulation(analysis_type);
	
	if(!femmodel)return 0;

	/*extract our parameter from the found formulation: */
	femmodel->FindParam(pparameter,pM,pN,parametername);
}
/*}}}1*/
/*FUNCTION Model::FindParam(char** pparameter,char* parametername,int analysis_type) {{{1*/
int   Model::FindParam(char** pparameter,char* parametername,int analysis_type){

	FemModel* femmodel=NULL;
	
	/*find the correct formulation: */
	femmodel=this->GetFormulation(analysis_type);
	
	if(!femmodel)return 0;

	/*extract our parameter from the found formulation: */
	femmodel->FindParam(pparameter,parametername);
}	
/*}}}1*/
/*FUNCTION Model::GetActiveFormulation {{{1*/
FemModel* Model::GetActiveFormulation(){return active;}
/*}}}1*/
/*FUNCTION Model::GetFormulation(int analysis_type,int sub_analysis_type) {{{1*/

FemModel* Model::GetFormulation(int analysis_type,int sub_analysis_type){
	
	int i;
	FemModel* femmodel=NULL;
	int femmodel_analysis_type;
	int femmodel_sub_analysis_type;
	int found=0;
	
	if (!femmodels->Size())ISSMERROR(" no fem models were found!");

	/*find femmodel with correct analysis_type and sub_analysis_type: */
	for(i=0;i<femmodels->Size();i++){

		femmodel=(FemModel*)femmodels->GetObjectByOffset(i);
		femmodel->FindParam(&femmodel_analysis_type,"analysis_type");
		femmodel->FindParam(&femmodel_sub_analysis_type,"sub_analysis_type");

		if((analysis_type==femmodel_analysis_type) && (sub_analysis_type==femmodel_sub_analysis_type)){
			found=1;
			break;
		}
	}

	if(!found)return NULL;
	else return femmodel;
}
/*}}}1*/
/*FUNCTION Model::GetFormulation(int analysis_type) {{{1*/

FemModel* Model::GetFormulation(int analysis_type){
	
	int i;
	FemModel* femmodel=NULL;
	int femmodel_analysis_type;
	int femmodel_sub_analysis_type;
	int found=0;
	
	if (!femmodels->Size())ISSMERROR(" no fem models were found!");

	/*find femmodel with correct analysis_type and sub_analysis_type: */
	for(i=0;i<femmodels->Size();i++){

		femmodel=(FemModel*)femmodels->GetObjectByOffset(i);
		femmodel->FindParam(&femmodel_analysis_type,"analysis_type");

		if((analysis_type==femmodel_analysis_type)){
			found=1;
			break;
		}
	}

	if(!found)return NULL;
	else return femmodel;
}
/*}}}1*/
/*FUNCTION Model::SetActiveFormulation {{{1*/
void* Model::SetActiveFormulation(FemModel* femmodel){
	active=femmodel;
}
/*}}}1*/
