/*!\file UpdateGeometryx
 * \brief: update geometry node by node
 */

#include "./UpdateGeometryx.h"

#include "../shared/shared.h"
#include "../include/macros.h"
#include "../toolkits/toolkits.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../DataSet/DataSet.h"

void UpdateGeometryx(Vec* poutthickness,Vec* poutbed,Vec* poutsurface, 
		DataSet* elements, DataSet* nodes, DataSet* vertices,DataSet* loads, DataSet* materials, DataSet* parameters,
		Vec vec_newthickness,Vec vec_bed,Vec vec_surface){

	int i;
	int dof;

	/*serialized input: */
	double* newthickness=NULL;
	double* bed=NULL;
	double* surface=NULL;

	/*objects: */
	Node* node=NULL;
	Object* object=NULL;
	Matpar* matpar=NULL;

	/*parameters: */
	double h,b,s;
	double rho_ice,rho_water;

	/*output: */
	Vec outthickness=NULL;
	Vec outbed=NULL;
	Vec outsurface=NULL;

	/*Duplicate inputs to outputs, do not copy values!: */
	VecDuplicate(vec_newthickness,&outthickness);
	VecDuplicate(vec_bed,&outbed);
	VecDuplicate(vec_surface,&outsurface);

	/*First, get elements and loads configured: */
	elements->Configure(elements,loads, nodes,vertices, materials,parameters);
	nodes->Configure(elements,loads, nodes,vertices, materials,parameters);
	materials->Configure(elements, loads, nodes,vertices, materials,parameters);
	loads->Configure(elements, loads, nodes,vertices, materials,parameters);
	parameters->Configure(elements,loads, nodes,vertices, materials,parameters);

	/*Serialize inputs :*/
	VecToMPISerial(&newthickness,vec_newthickness);
	VecToMPISerial(&bed,vec_bed);
	VecToMPISerial(&surface,vec_surface);

	/*First, find the matpar object in materials: */
	for(i=0;i<materials->Size();i++){
		Object* object=materials->GetObjectByOffset(i);
		if  (object->Enum()==MatparEnum){
			matpar=(Matpar*)object;
			break;
		}
	}

	/*recover material parameters: */
	rho_ice=matpar->GetRhoIce();
	rho_water=matpar->GetRhoWater();

	/*Go through nodes and for each node, update the thickness, bed and surface, using the 
	 * new thickness computed in the prognostic_core part of the transient solution sequence: */

	for(i=0;i<nodes->Size();i++){

		/*get i'th node: */
		node=(Node*)nodes->GetObjectByOffset(i);

		/*first, recover thickness, surface and bed for this node: */
		dof=node->GetDofList1();
		h=newthickness[dof];
		s=surface[dof];
		b=bed[dof];

		//First, check that thickness is >0
		if(h<0)h=10; 

		//For grids on ice sheet, the new bed remains the same, the new surface becomes bed+new_thickness. 
		if (node->IsOnSheet()){
			s=b+h;
		}

		//For grids on ice shelt, we have hydrostatic equilibrium (for now)
		if (node->IsOnShelf()){
			b=-rho_ice/rho_water*h;
			s=(1-rho_ice/rho_water)*h;
		}

		/*Ok, plug values of thickness, surafce and bed into our outputs: */
		VecSetValues(outthickness,1,&dof,&h,INSERT_VALUES);
		VecSetValues(outbed,1,&dof,&b,INSERT_VALUES);
		VecSetValues(outsurface,1,&dof,&s,INSERT_VALUES);
	}

	/*Assemble vectors: */
	VecAssemblyBegin(outthickness);
	VecAssemblyEnd(outthickness);

	VecAssemblyBegin(outbed);
	VecAssemblyEnd(outbed);

	VecAssemblyBegin(outsurface);
	VecAssemblyEnd(outsurface);

	/*Free ressources:*/
	xfree((void**)&newthickness);
	xfree((void**)&bed);
	xfree((void**)&surface);
	
	/*Assign output pointers: */
	*poutthickness=outthickness;
	*poutbed=outbed;
	*poutsurface=outsurface;
}
