/*!\file:  prognostic2.cpp
 * \brief: prognostic2 solution
 */ 

#include "../issm.h"
#include "./parallel.h"

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif


int main(int argc,char* *argv){
	
	/*I/O: */
	FILE* fid=NULL;
	char* inputfilename=NULL;
	char* outputfilename=NULL;
	char* lockname=NULL;
	int   numberofnodes;
	double waitonlock=0;

	Model* model=NULL;

	double* vx_g=NULL;
	double* vy_g=NULL;
	double* h_g_initial=NULL;
	double* m_g=NULL;
	double* a_g=NULL;
	double  dt;
	double  yts;
	int     qmu_analysis;

	/*Results: */
	DataSet* results=NULL;
	DataSet* processedresults=NULL;
	Result*  result=NULL;

	ParameterInputs* inputs=NULL;
	Param*   param=NULL;

	/*time*/
	double   start, finish;
	double   start_core, finish_core;
	double   start_init, finish_init;

	MODULEBOOT();

	#if !defined(_PARALLEL_) || (defined(_PARALLEL_) && !defined(_HAVE_PETSC_))
	ISSMERROR(" parallel executable was compiled without support of parallel libraries!");
	#endif

	/*Initialize Petsc and get start time*/
	PetscInitialize(&argc,&argv,(char *)0,"");  
	MPI_Barrier(MPI_COMM_WORLD); start=MPI_Wtime();

	/*Size and rank: */
	MPI_Comm_rank(MPI_COMM_WORLD,&my_rank);  
	MPI_Comm_size(MPI_COMM_WORLD,&num_procs); 

	_printf_("recover input file name and output file name:\n");
	inputfilename=argv[2];
	outputfilename=argv[3];
	lockname=argv[4];

	/*Initialize model structure: */
	MPI_Barrier(MPI_COMM_WORLD); start_init=MPI_Wtime();
	model=new Model();

	/*Open handle to data on disk: */
	fid=pfopen(inputfilename,"rb");

	_printf_("read and create finite element model:\n");
	model->AddFormulation(fid,Prognostic2AnalysisEnum);

	/*recover parameters: */
	model->FindParam(&waitonlock,"waitonlock");
	model->FindParam(&qmu_analysis,"qmu_analysis");

	_printf_("initialize inputs:\n");
	
	model->FindParam(&vx_g,NULL,NULL,"vx_g",Prognostic2AnalysisEnum);
	model->FindParam(&vy_g,NULL,NULL,"vy_g",Prognostic2AnalysisEnum);
	model->FindParam(&h_g_initial,NULL,NULL,"h_g",Prognostic2AnalysisEnum);
	model->FindParam(&m_g,NULL,NULL,"m_g",Prognostic2AnalysisEnum);
	model->FindParam(&a_g,NULL,NULL,"a_g",Prognostic2AnalysisEnum);
	model->FindParam(&dt,"dt");
	model->FindParam(&yts,"yts");
	model->FindParam(&numberofnodes,"numberofnodes");
	
	inputs=new ParameterInputs;
	inputs->Add("vx",vx_g,1,numberofnodes);
	inputs->Add("vy",vy_g,1,numberofnodes);
	inputs->Add("thickness",h_g_initial,1,numberofnodes);
	inputs->Add("melting",m_g,1,numberofnodes);
	inputs->Add("accumulation",a_g,1,numberofnodes);
	inputs->Add("dt",dt*yts);

	_printf_("initialize results:\n");
	results=new DataSet(ResultsEnum);
	MPI_Barrier(MPI_COMM_WORLD); finish_init=MPI_Wtime();

	/*are we running the solutoin sequence, or a qmu wrapper around it? : */
	if(!qmu_analysis){

		/*run prognostic2 analysis: */
		_printf_("call computational core:\n");
		MPI_Barrier(MPI_COMM_WORLD); start_core=MPI_Wtime( );
		prognostic2_core(results,model,inputs);
		MPI_Barrier(MPI_COMM_WORLD); finish_core=MPI_Wtime( );

	}
	else{

		/*run qmu analysis: */
		_printf_("calling qmu analysis on prognostic2 core:\n");
	
		#ifdef _HAVE_DAKOTA_ 
		MPI_Barrier(MPI_COMM_WORLD); start_core=MPI_Wtime( );
		Qmux(model,inputs,Prognostic2AnalysisEnum,NoneAnalysisEnum);
		MPI_Barrier(MPI_COMM_WORLD); finish_core=MPI_Wtime( );
	 	#else
		ISSMERROR(" Dakota not present, cannot do qmu!");
		#endif
	}

	/*Add analysis_type to results: */
	result=new Result(results->Size()+1,0,1,"analysis_type","prognostic2");
	results->AddObject(result);
	
	_printf_("process results:\n");
	ProcessResults(&processedresults,results,model,Prognostic2AnalysisEnum);
	
	_printf_("write results to disk:\n");
	OutputResults(processedresults,outputfilename);

	if (waitonlock>0){
		_printf_("write lock file:\n");
		WriteLockFile(lockname);
	}

	/*Free ressources:*/
	xfree((void**)&vx_g);
	xfree((void**)&vy_g);
	xfree((void**)&h_g_initial);
	xfree((void**)&m_g);
	xfree((void**)&a_g);
	delete processedresults;
	delete results;
	delete model;
	delete inputs;

	/*Get finish time and close*/
	MPI_Barrier(MPI_COMM_WORLD); finish = MPI_Wtime( );
	_printf_("\n   %-34s %f seconds  \n","Model initialization elapsed time:",finish_init-start_init);
	_printf_("   %-34s %f seconds  \n","Core solution elapsed time:",finish_core-start_core);
	_printf_("\n   %s %i hrs %i min %i sec\n\n","Total elapsed time:",int((finish-start)/3600),int(int(finish-start)%3600/60),int(finish-start)%60);
	_printf_("closing MPI and Petsc\n");
	PetscFinalize(); 

	/*end module: */
	MODULEEND();

	return 0; //unix success return;
}
