/*!\file Sing.c
 * \brief: implementation of the Sing object
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include "./Sing.h"
#include <string.h>
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../shared/shared.h"
#include "../DataSet/DataSet.h"
#include "../include/typedefs.h"
#include "../include/macros.h"

/*Object constructors and destructor*/
/*FUNCTION Sing::constructor {{{1*/
Sing::Sing(){
	return;
}
/*}}}*/
/*FUNCTION Sing constructor {{{1*/
Sing::Sing(int sing_id,int* sing_node_ids, int sing_matice_id, int sing_matpar_id, int sing_numpar_id, ElementProperties* singproperties): 
	hnodes(sing_node_ids,1),
	hmatice(&sing_matice_id,1),
	hmatpar(&sing_matpar_id,1),
	hnumpar(&sing_numpar_id,1),
	properties(singproperties)
{

	/*all the initialization has been done by the initializer, just fill in the id: */
	this->id=sing_id;

	return;
}
/*}}}*/
/*FUNCTION Sing other constructor {{{1*/
Sing::Sing(int sing_id,Hook* sing_hnodes, Hook* sing_hmatice, Hook* sing_hmatpar, Hook* sing_hnumpar, ElementProperties* sing_properties):
	hnodes(sing_hnodes),
	hmatice(sing_hmatice),
	hmatpar(sing_hmatpar),
	hnumpar(sing_hnumpar),
	properties(sing_properties)
{

	/*all the initialization has been done by the initializer, just fill in the id: */
	this->id=sing_id;

	return;
}
/*}}}*/
/*FUNCTION Sing iomodel constructor {{{1*/
Sing::Sing(int i, IoModel* iomodel){

	int sing_matice_id;
	int sing_matpar_id;
	int sing_numpar_id;
	
	int sing_g;
	double sing_h;
	double sing_k;


	/*id: */
	this->id=i+1;

	/*hooks: */
	sing_matice_id=i+1; //refers to the corresponding material property card
	sing_matpar_id=iomodel->numberofvertices+1;//refers to the corresponding matpar property card
	sing_numpar_id=1;
	sing_g=i+1;

	this->hnodes.Init(&sing_g,1);
	this->hmatice.Init(&sing_matice_id,1);
	this->hmatpar.Init(&sing_matpar_id,1);
	this->hnumpar.Init(&sing_numpar_id,1);

	/*properties: */
	sing_h=iomodel->thickness[i];
	sing_k=iomodel->drag[i];

	this->properties.Init(1,&sing_h, NULL, NULL, &sing_k, NULL, NULL, NULL, UNDEF, UNDEF, UNDEF, UNDEF, UNDEF,UNDEF, UNDEF,UNDEF,UNDEF);

}
/*}}}*/
/*FUNCTION Sing::destructor {{{1*/
Sing::~Sing(){
	return;
}
/*}}}*/

/*Object management*/
/*FUNCTION Sing::Configure {{{1*/
void  Sing::Configure(void* ploadsin, void* pnodesin,void* pmaterialsin,void* pparametersin){

	int i;
	
	DataSet* loadsin=NULL;
	DataSet* nodesin=NULL;
	DataSet* materialsin=NULL;
	DataSet* parametersin=NULL;

	/*Recover pointers :*/
	loadsin=(DataSet*)ploadsin;
	nodesin=(DataSet*)pnodesin;
	materialsin=(DataSet*)pmaterialsin;
	parametersin=(DataSet*)pparametersin;

	/*Take care of hooking up all objects for this element, ie links the objects in the hooks to their respective 
	 * datasets, using internal ids and offsets hidden in hooks: */
	hnodes.configure(nodesin);
	hmatice.configure(materialsin);
	hmatpar.configure(materialsin);
	hnumpar.configure(parametersin);

}
/*}}}*/
/*FUNCTION Sing::copy {{{1*/
Object* Sing::copy() {
	
	return new Sing(*this); 

}
/*}}}*/
/*FUNCTION Sing::DeepEcho {{{1*/
void Sing::DeepEcho(void){

	printf("Sing:\n");
	printf("   id: %i\n",id);
	hnodes.DeepEcho();
	hmatice.DeepEcho();
	hmatpar.DeepEcho();
	hnumpar.DeepEcho();
	properties.DeepEcho();

	return;
}
/*}}}*/
/*FUNCTION Sing::Demarshall {{{1*/
void  Sing::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   i;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/
	memcpy(&id,marshalled_dataset,sizeof(id));marshalled_dataset+=sizeof(id);

	/*demarshall hooks: */
	hnodes.Demarshall(&marshalled_dataset);
	hmatice.Demarshall(&marshalled_dataset);
	hmatpar.Demarshall(&marshalled_dataset);
	hnumpar.Demarshall(&marshalled_dataset);

	/*demarshall properties: */
	properties.Demarshall(&marshalled_dataset);

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Sing::Echo{{{1*/

void Sing::Echo(void){

	printf("Sing:\n");
	printf("   id: %i\n",id);
	hnodes.Echo();
	hmatice.Echo();
	hmatpar.Echo();
	hnumpar.Echo();
	properties.Echo();

	return;
}
/*}}}*/
/*FUNCTION Sing::Marshall {{{1*/
void  Sing::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_type=0;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of Sing: */
	enum_type=SingEnum();
	
	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	
	/*marshall Sing data: */
	memcpy(marshalled_dataset,&id,sizeof(id));marshalled_dataset+=sizeof(id);
	
	/*Marshall hooks: */
	hnodes.Marshall(&marshalled_dataset);
	hmatice.Marshall(&marshalled_dataset);
	hmatpar.Marshall(&marshalled_dataset);
	hnumpar.Marshall(&marshalled_dataset);

	/*Marshall properties: */
	properties.Marshall(&marshalled_dataset);

	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Sing::MashallSize {{{1*/
int   Sing::MarshallSize(){

	return sizeof(id)
		+hnodes.MarshallSize()
		+hmatice.MarshallSize()
		+hmatpar.MarshallSize()
		+hnumpar.MarshallSize()
		+properties.MarshallSize()
		+sizeof(int); //sizeof(int) for enum type
}
/*}}}*/
/*FUNCTION Sing::UpdateFromInputs {{{1*/
void  Sing::UpdateFromInputs(void* vinputs){

	int     dofs[1]={0};
	double  temperature;
	double  B;
	double  B_average;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;
	Numpar* numpar=NULL;

	ParameterInputs* inputs=NULL;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();
	numpar=(Numpar*)hnumpar.delivers();

	/*Update internal data if inputs holds new values: */
	//if (id==1) printf("WARNING if QMU: no hydrostatic equilibrium is applied here (conflict with prognostic, which does not have matpar)\n");
	//For now
	if(this->properties.h) inputs->Recover("thickness",&this->properties.h[0],1,dofs,1,(void**)nodes);
	//Later
	/*
		if(inputs->Recover("thickness",&new_h[0],1,dofs,1,(void**)nodes)){
	//density, needed later:
	double di=(this->matpar->GetRhoIce()/this->matpar->GetRhoWater());
	//Go through grids:
	for (i=0;i<1;i++){
	if(nodes[i]->IsOnShelf()){
	this->b[i]=this->b[i]-di*(new_h[i]-h[i]); //hydrostatic equilibrium;
	}
	this->s[i]=this->b[i]+new_h[i];
	this->h[i]=new_h[i];
	}
	}
	*/
	if (this->properties.k) inputs->Recover("drag",&this->properties.k[0],1,dofs,1,(void**)nodes);
	
	//Update material if necessary
	if(inputs->Recover("temperature_average",&temperature,1,dofs,1,(void**)nodes)){
		B_average=Paterson(temperature);
		matice->SetB(B_average);
	}
	else if(inputs->Recover("temperature",&temperature,1,dofs,1,(void**)nodes)){
		B=Paterson(temperature);
		matice->SetB(B);
	}
	
	if(inputs->Recover("B",&B,1,dofs,1,(void**)nodes)){
		matice->SetB(B);
	}

}
/*}}}*/
		
/*Object functions*/
/*FUNCTION Sing::ComputeBasalStress {{{1*/
void  Sing::ComputeBasalStress(Vec p_g,void* inputs,int analysis_type,int sub_analysis_type){

	ISSMERROR("Not implemented yet");

}
/*}}}*/
/*FUNCTION Sing::ComputePressure {{{1*/
void  Sing::ComputePressure(Vec p_g,void* inputs,int analysis_type,int sub_analysis_type){

	int i;
	const int numgrids=1;
	int doflist[numgrids];
	double pressure[numgrids];
	double rho_ice,g;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;
	Numpar* numpar=NULL;

	/*Get dof list on which we will plug the pressure values: */
	GetDofList1(&doflist[0]);

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();
	numpar=(Numpar*)hnumpar.delivers();

	/*pressure is lithostatic: */
	rho_ice=matpar->GetRhoIce();
	g=matpar->GetG();
	pressure[0]=rho_ice*g*this->properties.h[0];
	
	/*plug local pressure values into global pressure vector: */
	VecSetValues(p_g,numgrids,doflist,(const double*)pressure,INSERT_VALUES);

}
/*}}}*/
/*FUNCTION Sing::ComputeStrainRate {{{1*/
void  Sing::ComputeStrainRate(Vec p_g,void* inputs,int analysis_type,int sub_analysis_type){

	ISSMERROR("Not implemented yet");

}
/*}}}*/
/*FUNCTION Sing::CostFunction {{{1*/
double Sing::CostFunction(void*, int,int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::CreateKMatrix {{{1*/

void  Sing::CreateKMatrix(Mat Kgg,void* inputs,int analysis_type,int sub_analysis_type){

	/*Just branch to the correct element stiffness matrix generator, according to the type of analysis we are carrying out: */
	if ((analysis_type==DiagnosticAnalysisEnum()) && (sub_analysis_type==HutterAnalysisEnum())){

		CreateKMatrixDiagnosticHutter( Kgg,inputs,analysis_type,sub_analysis_type);

	}
	else{
		ISSMERROR(exprintf("%s%i%s%i%s\n","analysis: ",analysis_type," and sub_analysis_type: ",sub_analysis_type," not supported yet"));
	}

}
/*}}}*/
/*FUNCTION Sing::CreateKMatrixDiagnosticHutter {{{1*/

void  Sing::CreateKMatrixDiagnosticHutter(Mat Kgg,void* vinputs,int analysis_type,int sub_analysis_type){
	
	const int numgrids=1;
	const int NDOF2=2;
	const int numdofs=numgrids*NDOF2;
	double Ke_gg[numdofs][numdofs]={{1,0},{0,1}};
	int    doflist[numdofs];
	int    numberofdofspernode;

	ParameterInputs* inputs=NULL;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;
	
	GetDofList(&doflist[0],&numberofdofspernode);

	MatSetValues(Kgg,numdofs,doflist,numdofs,doflist,(const double*)Ke_gg,ADD_VALUES);

}
/*}}}*/
/*FUNCTION Sing::CreatePVector {{{1*/
void  Sing::CreatePVector(Vec pg,void* inputs,int analysis_type,int sub_analysis_type){
	
	/*Just branch to the correct load generator, according to the type of analysis we are carrying out: */
	if ((analysis_type==DiagnosticAnalysisEnum()) && (sub_analysis_type==HutterAnalysisEnum())){
	
			CreatePVectorDiagnosticHutter( pg,inputs,analysis_type,sub_analysis_type);

	}
	else{
		ISSMERROR(exprintf("%s%i%s"," analysis ",analysis_type," not supported yet"));
	}

}
/*}}}*/
/*FUNCTION Sing::CreatePVectorDiagnosticHutter {{{1*/

void Sing::CreatePVectorDiagnosticHutter( Vec pg, void* vinputs, int analysis_type,int sub_analysis_type){
	
	
	const int numgrids=1;
	const int NDOF2=2;
	const int numdofs=NDOF2*numgrids;
	int       doflist[numdofs];
	int       dofs[1]={0};
	int       found=0;
	double    slope[2];
	double    slope2;
	double    pe_g[numdofs]={0,0};
	double    ub,vb;
	double    constant_part;
	int       numberofdofspernode;
	double    rho_ice,gravity,n,B;
	double    thickness;

	/*dynamic objects pointed to by hooks: */
	Node**  node=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;
	Numpar* numpar=NULL;

	ParameterInputs* inputs=NULL;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;

	/*recover objects from hooks: */
	node=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();
	numpar=(Numpar*)hnumpar.delivers();

	found=inputs->Recover("surfaceslopex",&slope[0],1,dofs,numgrids,(void**)&node[0]);
	if(!found)ISSMERROR(" surfaceslopex missing from inputs!");
	found=inputs->Recover("surfaceslopey",&slope[1],1,dofs,numgrids,(void**)&node[0]);
	if(!found)ISSMERROR(" surfaceslopey missing from inputs!");

	GetDofList(&doflist[0],&numberofdofspernode);

	//compute slope2 
	slope2=pow(slope[0],2)+pow(slope[1],2);

	//compute ub
	rho_ice=matpar->GetRhoIce();
	gravity=matpar->GetG();
	n=matice->GetN();
	B=matice->GetB();
	thickness=this->properties.h[0];

	ub=-1.58*pow((double)10.0,(double)-10.0)*rho_ice*gravity*thickness*slope[0];
	vb=-1.58*pow((double)10.0,(double)-10.0)*rho_ice*gravity*thickness*slope[1];

	//compute constant_part
	constant_part=-2*pow(rho_ice*gravity,n)*pow(slope2,((n-1)/2));

	pe_g[0]=ub-2.0*pow(rho_ice*gravity,n)*pow(slope2,((n-1)/2.0))*pow(thickness,n)/(pow(B,n)*(n+1))*slope[0];
	pe_g[1]=vb-2.0*pow(rho_ice*gravity,n)*pow(slope2,((n-1)/2.0))*pow(thickness,n)/(pow(B,n)*(n+1))*slope[1];

	VecSetValues(pg,numdofs,doflist,(const double*)pe_g,ADD_VALUES);


}
/*}}}*/
/*FUNCTION Sing::Du {{{1*/
void  Sing::Du(_p_Vec*,void*,int,int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::Enum {{{1*/
int Sing::Enum(void){

	return SingEnum();

}
/*}}}*/
/*FUNCTION Sing::GetBedList {{{1*/
void  Sing::GetBedList(double*){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::GetDofList {{{1*/
void  Sing::GetDofList(int* doflist,int* pnumberofdofspernode){

	int i;
	int doflist_per_node[MAXDOFSPERNODE];
	int numberofdofspernode;
	
	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	
	nodes[0]->GetDofList(&doflist_per_node[0],&numberofdofspernode);
	for(i=0;i<numberofdofspernode;i++){
		doflist[i]=doflist_per_node[i];
	}

	/*Assign output pointers:*/
	*pnumberofdofspernode=numberofdofspernode;

}
/*}}}*/
/*FUNCTION Sing::GetDofList1 {{{1*/
void  Sing::GetDofList1(int* doflist){

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	
	int i;
	doflist[0]=nodes[0]->GetDofList1();

}
/*}}}*/
/*FUNCTION Sing::GetId {{{1*/
int    Sing::GetId(void){ return id; }
/*}}}*/
/*FUNCTION Sing::GetMatPar {{{1*/
void* Sing::GetMatPar(){

	/*dynamic objects pointed to by hooks: */
	Matpar* matpar=NULL;

	/*recover objects from hooks: */
	matpar=(Matpar*)hmatpar.delivers();

	return matpar;
}
/*}}}*/
/*FUNCTION Sing::GetName {{{1*/
char* Sing::GetName(void){
	return "sing";
}
/*}}}*/
/*FUNCTION Sing::GetNodes {{{1*/
void  Sing::GetNodes(void** vpnodes){
	
	Node** pnodes=NULL;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	
	/*recover nodes: */
	pnodes=(Node**)vpnodes;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	
	pnodes[0]=nodes[0];
}
/*}}}*/
/*FUNCTION Sing::GetOnBed {{{1*/
int   Sing::GetOnBed(){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::GetShelf {{{1*/
int   Sing::GetShelf(){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::GetThicknessList {{{1*/
void  Sing::GetThicknessList(double* thickness_list){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::Gradj {{{1*/
void  Sing::Gradj(_p_Vec*, void*, int, int ,char*){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::GradB {{{1*/
void  Sing::GradjB(_p_Vec*, void*, int,int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::GradjDrag {{{1*/
void  Sing::GradjDrag(_p_Vec*, void*, int,int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::MassFlux {{{1*/
double Sing::MassFlux( double* segment,double* ug){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::MaticeConfiguration {{{1*/
void  Sing::MaticeConfiguration(Matice* sing_matice,int sing_matice_offset){

	/*dynamic objects pointed to by hooks: */
	Matice* matice=NULL;

	/*recover objects from hooks: */
	matice=(Matice*)hmatice.delivers();
	
	ISSMERROR("not supported yet!");

	/*matice=sing_matice;
	matice_offset=sing_matice_offset;*/
}
/*}}}*/
/*FUNCTION Sing::MatparConfiguration {{{1*/
void  Sing::MatparConfiguration(Matpar* sing_matpar,int sing_matpar_offset){

	/*dynamic objects pointed to by hooks: */
	Matpar* matpar=NULL;

	/*recover objects from hooks: */
	matpar=(Matpar*)hmatpar.delivers();

	ISSMERROR("not supported yet!");

	/*matpar=sing_matpar;
	matpar_offset=sing_matpar_offset;*/

}
/*}}}*/
/*FUNCTION Sing::Misfit {{{1*/
double Sing::Misfit(void*, int,int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::MyRank {{{1*/
int    Sing::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}*/
/*FUNCTION Sing::SurfaceArea {{{1*/
double Sing::SurfaceArea(void*, int,int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::SetClone {{{1*/
void  Sing::SetClone(int* minranks){

	ISSMERROR("not implemented yet");
}
/*}}}1*/
