function md=transient3d(md);
%
%   Usage:
%      md=transient3d(md)
%
%   See also: TRANSIENT2D, TRANSIENT

%Build all models related to diagnostic
models.analysis_type=TransientAnalysisEnum(); %needed for processresults

displaystring(md.verbose,'%s',['reading diagnostic horiz model data']);
md.analysis_type=DiagnosticAnalysisEnum(); md.sub_analysis_type=HorizAnalysisEnum(); models.dh=CreateFemModel(md);

displaystring(md.verbose,'\n%s',['reading diagnostic vert model data']);
md.analysis_type=DiagnosticAnalysisEnum(); md.sub_analysis_type=VertAnalysisEnum(); models.dv=CreateFemModel(md);

displaystring(md.verbose,'\n%s',['reading diagnostic stokes model data']);
md.analysis_type=DiagnosticAnalysisEnum(); md.sub_analysis_type=StokesAnalysisEnum(); models.ds=CreateFemModel(md);

displaystring(md.verbose,'\n%s',['reading diagnostic hutter model data']);
md.analysis_type=DiagnosticAnalysisEnum(); md.sub_analysis_type=HutterAnalysisEnum(); models.dhu=CreateFemModel(md);

displaystring(md.verbose,'\n%s',['reading surface and bed slope computation model data']);
md.analysis_type=SlopecomputeAnalysisEnum(); md.sub_analysis_type=NoneAnalysisEnum(); models.sl=CreateFemModel(md);

displaystring(md.verbose,'%s',['reading prognostic model data']);
md.analysis_type=PrognosticAnalysisEnum(); models.p=CreateFemModel(md);

%Build all models related to thermal
displaystring(md.verbose,'%s',['reading thermal model data']);
md.analysis_type=ThermalAnalysisEnum(); models.t=CreateFemModel(md);

displaystring(md.verbose,'%s',['reading melting model data']);
md.analysis_type=MeltingAnalysisEnum(); models.m=CreateFemModel(md);


%initialize solution
results=struct('step',[],'time',[],'u_g',[],'p_g',[],'h_g',[],'s_g',[],'b_g',[]);
results.step=1;
results.time=0;
results.u_g=models.dh.parameters.u_g;
%results.u_g=models.dh.parameters.u_g(dofsetgen([1,2],3,length(models.dh.parameters.u_g)));
results.p_g=models.p.parameters.p_g;
results.h_g=models.p.parameters.h_g;
results.s_g=models.p.parameters.s_g;
results.b_g=models.p.parameters.b_g;
results.t_g=models.p.parameters.t_g;
results.m_g=models.p.parameters.m_g;
results.a_g=models.p.parameters.a_g;

%initialize inputs
displaystring(md.verbose,'\n%s',['setup inputs...']);
inputs=inputlist;
inputs=add(inputs,'velocity',results.u_g,'doublevec',3,models.p.parameters.numberofnodes);
inputs=add(inputs,'melting',results.m_g,'doublevec',1,models.p.parameters.numberofnodes);
inputs=add(inputs,'accumulation',results.a_g,'doublevec',1,models.p.parameters.numberofnodes);
inputs=add(inputs,'dt',models.p.parameters.dt*models.p.parameters.yts,'double');

% figure out number of dof: just for information purposes.
md.dof=modelsize(models);

%first time step is given by model. 
dt=models.p.parameters.dt;
finaltime=models.p.parameters.ndt;
time=dt;
yts=models.p.parameters.yts;
n=1; %counter

while  time<finaltime+dt, %make sure we run up to finaltime.

	displaystring(md.verbose,'\n%s%g%s%g%s%g\n','time [yr]: ',time,'    iteration number: ',n,'/',floor(finaltime/dt));

	results(n+1).step=n+1;
	results(n+1).time=time;

	%update inputs
	inputs=add(inputs,'thickness',results(n).h_g,'doublevec',1,models.p.parameters.numberofnodes);
	inputs=add(inputs,'surface',results(n).s_g,'doublevec',1,models.p.parameters.numberofnodes);
	inputs=add(inputs,'bed',results(n).b_g,'doublevec',1,models.p.parameters.numberofnodes);
	inputs=add(inputs,'velocity',results(n).u_g,'doublevec',3,models.p.parameters.numberofnodes);
	inputs=add(inputs,'pressure',results(n).p_g,'doublevec',1,models.p.parameters.numberofnodes);
	inputs=add(inputs,'temperature',results(n).t_g,'doublevec',1,models.t.parameters.numberofnodes);

	%Deal with temperature first 
	displaystring(md.verbose,'\n%s',['    computing temperatures...']);
	[results(n+1).t_g models.t.loads melting_offset]=thermal_core_nonlinear(models.t,inputs,ThermalAnalysisEnum(),TransientAnalysisEnum());
	inputs=add(inputs,'temperature',results(n+1).t_g,'doublevec',1,models.t.parameters.numberofnodes);
	
	displaystring(md.verbose,'\n%s',['    computing melting...']);
	inputs=add(inputs,'melting_offset',melting_offset,'double');
	results(n+1).m_g=diagnostic_core_linear(models.m,inputs,MeltingAnalysisEnum(),TransientAnalysisEnum());

	%Compute depth averaged temperature
	temperature_average=FieldDepthAverage(models.t.elements,models.t.nodes,models.t.vertices,models.t.loads,models.t.materials,models.t.parameters,results(n+1).t_g,'temperature');
	inputs=add(inputs,'temperature_average',temperature_average,'doublevec',1,models.t.parameters.numberofnodes);

	%Deal with velocities.
	rawresults=diagnostic_core(models,inputs);
	results(n+1).u_g=rawresults.u_g; results(n+1).p_g=rawresults.p_g;

	%compute new thickness
	displaystring(md.verbose,'\n%s',['    computing new thickness...']);
	inputs=add(inputs,'velocity',results(n+1).u_g,'doublevec',3,models.p.parameters.numberofnodes);
	rawresults=prognostic_core(models,inputs,PrognosticAnalysisEnum(),NoneAnalysisEnum());
	new_thickness=rawresults.h_g;

	%update surface and bed using the new thickness
	displaystring(md.verbose,'\n%s',['    updating geometry...']);
	[new_thickness,new_bed,new_surface]=UpdateGeometry(models.p.elements,models.p.nodes,models.p.vertices,models.p.loads,models.p.materials,models.p.parameters,new_thickness,results(n).b_g,results(n).s_g);

	%Record bed surface and thickness in the results
	results(n+1).h_g=new_thickness;
	results(n+1).b_g=new_bed;
	results(n+1).s_g=new_surface;

	%figure out if time stepping is good
	%displaystring(md.verbose,'   checking time stepping...'));
	%[back,dt,time]=TimeStepping(md,results,dt,time);
	%if back,
	%	continue;
	%end

	%update node positions
	displaystring(md.verbose,'\n%s',['    updating node positions...']);
	models.dh.nodes=UpdateVertexPositions(models.dh.vertices,new_bed,new_thickness);
	models.dv.nodes=UpdateVertexPositions(models.dv.vertices,new_bed,new_thickness);
	models.ds.nodes=UpdateVertexPositions(models.ds.vertices,new_bed,new_thickness);
	models.sl.nodes=UpdateVertexPositions(models.sl.vertices,new_bed,new_thickness);
	models.p.nodes=UpdateVertexPositions(models.p.vertices,new_bed,new_thickness);
	models.t.nodes=UpdateVertexPositions(models.t.vertices,new_bed,new_thickness);
	models.m.nodes=UpdateVertexPositions(models.m.vertices,new_bed,new_thickness);
	
	%update time and counter
	time=time+dt;
	n=n+1;
end

%process results
if ~isstruct(md.results), md.results=struct(); end
md.results.transient=processresults(models, results);
