/*!\file Matice.c
 * \brief: implementation of the Matice object
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include "./Matice.h"
#include <string.h>
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../shared/shared.h"
#include "../include/macros.h"
#include "../include/typedefs.h"
		
/*Object constructors and destructor*/
/*FUNCTION Matice::default constructor {{{1*/
Matice::Matice(){
	return;
}
/*}}}*/
/*FUNCTION Matice::constructor {{{1*/
Matice::Matice(int in_mid,double in_B,double in_n){
	this->Init(in_mid,in_B,in_n);
}
/*}}}*/
/*FUNCTION Matice::init {{{1*/
void Matice::Init(int in_mid,double in_B,double in_n){
	this->mid=in_mid;
	this->B=in_B;
	this->n=in_n;
}
/*}}}*/
/*FUNCTION Matice::constructor from iomodel {{{1*/
Matice::Matice(int i, IoModel* iomodel, int num_vertices){

	int j;
	
	/*needed for Init routine:*/
	int matice_mid;
	double matice_B;
	double matice_n;

	/*intermediary: */
	double B_avg;
	
	/*id: */
	matice_mid=i+1;  //same as element it is created for
 
	/*Compute B on the element if provided*/
	if (iomodel->B){
		B_avg=0;
		for(j=0;j<num_vertices;j++){
			B_avg+=*(iomodel->B+((int)*(iomodel->elements+num_vertices*i+j)-1));
		}
		B_avg=B_avg/num_vertices;
	}
	
	if (iomodel->B) matice_B=B_avg;
	else            matice_B=UNDEF;
	if (iomodel->n) matice_n=(double)*(iomodel->n+i);
	else            matice_n=UNDEF;

	this->Init(matice_mid,matice_B,matice_n);
}
/*}}}*/
/*FUNCTION Matice::destructor {{{1*/
Matice::~Matice(){
	return;
}
/*}}}*/

/*Object marshall*/
/*FUNCTION Matice::Marshall {{{1*/
void  Matice::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_type=0;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of Matice: */
	enum_type=MaticeEnum();
	
	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	
	/*marshall Matice data: */
	memcpy(marshalled_dataset,&mid,sizeof(mid));marshalled_dataset+=sizeof(mid);
	memcpy(marshalled_dataset,&B,sizeof(B));marshalled_dataset+=sizeof(B);
	memcpy(marshalled_dataset,&n,sizeof(n));marshalled_dataset+=sizeof(n);

	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Matice::MarshallSize{{{1*/
int   Matice::MarshallSize(){

	return sizeof(mid)+sizeof(B)+sizeof(n)+sizeof(int); //sizeof(int) for enum type
}
/*}}}*/
/*FUNCTION Matice::Demarshall {{{1*/
void  Matice::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/

	memcpy(&mid,marshalled_dataset,sizeof(mid));marshalled_dataset+=sizeof(mid);
	memcpy(&B,marshalled_dataset,sizeof(B));marshalled_dataset+=sizeof(B);
	memcpy(&n,marshalled_dataset,sizeof(n));marshalled_dataset+=sizeof(n);

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/

/*Object functions*/
/*FUNCTION Matice::copy {{{1*/
Object* Matice::copy() {
	return new Matice(*this); 
}
/*}}}*/
/*FUNCTION Matice::DeepEcho {{{1*/
void Matice::DeepEcho(void){

	printf("Matice:\n");
	printf("   mid: %i\n",mid);
	printf("   B: %g\n",B);
	printf("   n: %g\n",n);
	return;
}		
/*}}}*/
/*FUNCTION Matice::DistributeNumDofs {{{1*/
void  Matice::DistributeNumDofs(int* numdofspernode,int analysis_type){return;}
/*}}}*/
/*FUNCTION Matice::Echo {{{1*/
void Matice::Echo(void){

	printf("Matice:\n");
	printf("   mid: %i\n",mid);
	printf("   B: %g\n",B);
	printf("   n: %g\n",n);
	return;
}
/*}}}*/
/*FUNCTION Matice::Enum {{{1*/
int Matice::Enum(void){

	return MaticeEnum();

}
/*}}}*/
/*FUNCTION Matice::GetB {{{1*/
double Matice::GetB(){
	return B;
}
/*}}}*/
/*FUNCTION Matice::GetId {{{1*/
int    Matice::GetId(void){ return mid; }
/*}}}*/
/*FUNCTION Matice::GetN {{{1*/
double Matice::GetN(){
	return n;
}
/*}}}*/
/*FUNCTION Matice::GetName {{{1*/
char* Matice::GetName(void){
	return "matice";
}
/*}}}*/
/*FUNCTION Matice::GetViscosity2d {{{1*/
void  Matice::GetViscosity2d(double* pviscosity, double* epsilon){


	/*From a string tensor and a material object, return viscosity, using Glen's flow law.
	                                  2*B
	  viscosity= -------------------------------------------------------------------
	      2[ exx^2+eyy^2+exx*eyy+exy^2+exz^2+eyz^2 ]^[(n-1)/2n]
	 
	      where viscosity is the viscotiy, B the flow law parameter , (u,v) the velocity 
	      vector, and n the flow law exponent.
	 
	  If epsilon is NULL, it means this is the first time SystemMatrices is being run, and we 
	  return 10^14, initial viscosity.
*/



	/*output: */
	double viscosity;

	/*input strain rate: */
	double exx,eyy,exy;

	/*Intermediary value A and exponent e: */
	double A,e;

	if (n==1){
		/*Viscous behaviour! viscosity=B: */
		viscosity=B;
	}
	else{
		if((epsilon[0]==0) && (epsilon[1]==0) && (epsilon[2]==0)){
			viscosity=pow((double)10,(double)14);
		}
		else{
			/*Retrive strain rate components: */
			exx=*(epsilon+0);
			eyy=*(epsilon+1);
			exy=*(epsilon+2);

			/*Build viscosity: viscosity=2*B/(2*A^e) */
			A=pow(exx,2)+pow(eyy,2)+pow(exy,2)+exx*eyy;
			if(A==0){
				/*Maxiviscositym viscosity for 0 shear areas: */
				viscosity=4.5*pow((double)10,(double)17);
			}
			else{
				e=(n-1)/2/n;
				viscosity=2*B/(2*pow(A,e));
			}
		}
	}
	#ifdef _ISSM_DEBUG_
	printf("Viscosity %lf\n",viscosity);
	#endif

	/*Return: */
	*pviscosity=viscosity;
}
/*}}}*/
/*FUNCTION Matice::GetViscosity3d {{{1*/
void  Matice::GetViscosity3d(double* pviscosity3d, double* epsilon){

	/*Return viscosity accounting for steady state power law creep [Thomas and MacAyeal, 1982]: 
	 *
	 *                                 2*B
	 * viscosity3d= -------------------------------------------------------------------
	 *     2[ exx^2+eyy^2+exx*eyy+exy^2+exz^2+eyz^2 ]^[(n-1)/2n]
	 *
	 *     where mu is the viscotiy, B the flow law parameter , (u,v) the velocity 
	 *     vector, and n the flow law exponent.
	 *
	 * If epsilon is NULL, it means this is the first time Emg is being run, and we 
	 * return g, initial viscosity.
	 */
	
	/*output: */
	double viscosity3d;

	/*input strain rate: */
	double exx,eyy,exy,exz,eyz;

	/*Intermediary value A and exponent e: */
	double A,e;

	if (n==1){
		/*Viscous behaviour! viscosity3d=B: */
		viscosity3d=B;
	}
	else{
		if((epsilon[0]==0) && (epsilon[1]==0) && (epsilon[2]==0) && 
				(epsilon[3]==0) && (epsilon[4]==0)){
			viscosity3d=pow((double)10,(double)14);
		}
		else{

			/*Retrive strain rate components: */
			exx=*(epsilon+0);
			eyy=*(epsilon+1);
			exy=*(epsilon+2);
			exz=*(epsilon+3);
			eyz=*(epsilon+4);

			/*Build viscosity: viscosity3d=2*B/(2*A^e) */
			A=pow(exx,2)+pow(eyy,2)+pow(exy,2)+pow(exz,2)+pow(eyz,2)+exx*eyy;
			if(A==0){
				/*Maxiviscosity3dm viscosity for 0 shear areas: */
				viscosity3d=4.5*pow((double)10,(double)17);
			}
			else{
				e=(n-1)/2/n;
			
				viscosity3d=2*B/(2*pow(A,e));
			}
		}
	}
	#ifdef _ISSM_DEBUG_
	printf("Viscosity %lf\n",viscosity3d);
	#endif

	/*Assign output pointers:*/
	*pviscosity3d=viscosity3d;
}
/*}}}*/
/*FUNCTION Matice::GetViscosity3dStokes {{{1*/
void  Matice::GetViscosity3dStokes(double* pviscosity3d, double* epsilon){

	/*Return viscosity accounting for steady state power law creep [Thomas and MacAyeal, 1982]: 
	 *
	 *                                 2*B
	 * viscosity3d= -------------------------------------------------------------------
	 *     2[ exx^2+eyy^2+exx*eyy+exy^2+exz^2+eyz^2 ]^[(n-1)/2n]
	 *
	 *     where mu is the viscotiy, B the flow law parameter , (u,v) the velocity 
	 *     vector, and n the flow law exponent.
	 *
	 * If epsilon is NULL, it means this is the first time Emg is being run, and we 
	 * return g, initial viscosity.
	 */
	
	/*output: */
	double viscosity3d;

	/*input strain rate: */
	double exx,eyy,exy,exz,eyz,ezz;

	/*Intermediary value A and exponent e: */
	double A,e;
	double eps0;

	eps0=pow((double)10,(double)-27);
	
	if (n==1){
		/*Viscous behaviour! viscosity3d=B: */
		viscosity3d=B;
	}
	else{
		if((epsilon[0]==0) && (epsilon[1]==0) && (epsilon[2]==0) && 
				(epsilon[3]==0) && (epsilon[4]==0) && (epsilon[5]==0)){
			viscosity3d=pow((double)10,(double)14);
		}
		else{

			/*Retrive strain rate components: */
			exx=*(epsilon+0);
			eyy=*(epsilon+1);
			ezz=*(epsilon+2); //not used
			exy=*(epsilon+3);
			exz=*(epsilon+4);
			eyz=*(epsilon+5);

			/*Build viscosity: viscosity3d=2*B/(2*A^e) */
			A=pow(exx,2)+pow(eyy,2)+pow(exy,2)+pow(exz,2)+pow(eyz,2)+exx*eyy+pow(eps0,2);
			if(A==0){
				/*Maxiviscosity3dm viscosity for 0 shear areas: */
				viscosity3d=4.5*pow((double)10,(double)17);
			}
			else{
				e=(n-1)/2/n;
				viscosity3d=2*B/(2*pow(A,e));
			}
		}
	}

	#ifdef _ISSM_DEBUG_
	printf("Viscosity %lf\n",viscosity3d);
	#endif

	/*Assign output pointers:*/
	*pviscosity3d=viscosity3d;
}
/*}}}*/
/*FUNCTION Matice::GetViscosityComplement {{{1*/
void  Matice::GetViscosityComplement(double* pviscosity_complement, double* epsilon){

	/*Return viscosity accounting for steady state power law creep [Thomas and MacAyeal, 1982]: 
	 *
	 *                                  2* (1-n)/2n
	 * mu2= -------------------------------------------------------------------
	 *     2[ (du/dx)^2+(dv/dy)^2+1/4*(du/dy+dv/dx)^2+du/dx*dv/dy ]^[(3n-1)/2n]
	 *
	 *     where mu2 is the second viscosity, (u,v) the velocity 
	 *     vector, and n the flow law exponent.
	 *
	 * If epsilon is NULL, it means this is the first time Gradjb is being run, and we 
	 * return mu20, initial viscosity.
	 */
	
	/*output: */
	double viscosity_complement;

	/*input strain rate: */
	double exx,eyy,exy;

	/*Intermediary value A and exponent e: */
	double A,e;

	if(epsilon){
		exx=*(epsilon+0);
		eyy=*(epsilon+1);
		exy=*(epsilon+2);

		/*Build viscosity: mu2=B/(2*A^e) */
		A=pow(exx,2)+pow(eyy,2)+pow(exy,2)+exx*eyy;
		if(A==0){
			/*Maximum viscosity_complement for 0 shear areas: */
			viscosity_complement=4.5*pow((double)10,(double)17);
		}
		else{
			e=(n-1)/2/n;
		
			viscosity_complement=1/(2*pow(A,e));
		}
	}
	else{
		viscosity_complement=4.5*pow((double)10,(double)17);
	}
		
	#ifdef _ISSM_DEBUG_
	printf("viscosity_complement %lf\n",viscosity_complement);
	#endif

	/*Return: */
	*pviscosity_complement=viscosity_complement;
}
/*}}}*/
/*FUNCTION Matice::MyRank {{{1*/
int    Matice::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}*/
/*FUNCTION Matice::SetB {{{1*/
void  Matice::SetB(double B_param){
	B=B_param;
}
/*}}}*/
/*FUNCTION Matice::UpdateFromInputs {{{1*/
void  Matice::UpdateFromInputs(void* inputs){
	
	//ISSMERROR(" not supported yet!");

}
/*}}}*/
