/*!\file Node.c
 * \brief: implementation of the Node object
 */

/*Include files: {{{1*/
#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include "./Vertex.h"
#include "./Node.h"
#include "./Hook.h"
#include "./DofIndexing.h"
#include <string.h>
#include "../EnumDefinitions/EnumDefinitions.h"
#include "./ParameterInputs.h"
#include "../shared/shared.h"
#include "../include/typedefs.h"
#include "../include/macros.h"
/*}}}*/
/*Object constructors and destructors: {{{1*/
/*FUNCTION Node default constructor {{{2*/
Node::Node(){
	return;
}
/*}}}*/
/*FUNCTION Node constructor {{{2*/
Node::Node(int node_id,int node_vertex_id, int node_upper_node_id, int node_numdofs, NodeProperties* node_properties):
	indexing(node_numdofs),
	properties(node_properties),
    hvertex(&node_vertex_id,1),
    hupper_node(&node_upper_node_id,1){

	this->id=node_id;

	return;
}
/*}}}*/
/*FUNCTION Node other constructor {{{2*/
Node::Node(int node_id,DofIndexing* node_indexing, NodeProperties* node_properties, Hook* node_vertex, Hook* node_upper_node):
	    indexing(node_indexing),
		properties(node_properties),
		hvertex(node_vertex),
		hupper_node(node_upper_node) {

	    /*all the initialization has been done by the initializer, just fill in the id: */
	    this->id=node_id;

}
/*}}}*/
/*FUNCTION Node constructor from iomodel continuous Galerkin{{{2*/
Node::Node(int i, IoModel* iomodel){ //i is the node index

	int k;
	
	int numdofs;
	int vertex_id;
	int upper_node_id;

	/*id: */
	this->id=i+1; //matlab indexing

	/*indexing:*/
	DistributeNumDofs(&numdofs,iomodel->analysis_type,iomodel->sub_analysis_type); //number of dofs per node

	this->indexing.Init(numdofs);

	/*properties: */
	this->properties.Init(i,iomodel);

	/*hooks: */
	vertex_id=this->id; //node and vertex have the same id, as we are running galerkin continuous, with same number of nodes and vertices.

	if (strcmp(iomodel->meshtype,"3d")==0){
		if (isnan(iomodel->uppernodes[i])){
			upper_node_id=this->id; //nodes on surface do not have upper nodes, only themselves.
		}
		else{
			upper_node_id=(int)iomodel->uppernodes[i];
		}
	}
	else{
		/*If we are running 2d, upper_node does not mean much. Just point towards itself!:*/
		upper_node_id=this->id;
	}

	this->hvertex.Init(&vertex_id,1); //node id is the same as the vertex id, continuous galerkin!
	this->hupper_node.Init(&upper_node_id,1);

	/*set single point constraints: */

	/*Diagnostic Horiz*/
	if (iomodel->analysis_type==DiagnosticAnalysisEnum() && iomodel->sub_analysis_type==HorizAnalysisEnum()){
		if (strcmp(iomodel->meshtype,"3d")==0){
			/*We have a  3d mesh, we may have collapsed elements, hence dead grids. Freeze them out: */
			if (!iomodel->deadgrids) ISSMERROR("iomodel->deadgrids is NULL");
			if (iomodel->deadgrids[i]){
				for(k=1;k<=numdofs;k++){
					this->FreezeDof(k);
				}
			}
		}
		/*spc all nodes on hutter*/
		if (!iomodel->gridonhutter) ISSMERROR("iomodel->gridonhutter is NULL");
		if (iomodel->gridonhutter[i]){
			for(k=1;k<=numdofs;k++){
				this->FreezeDof(k);
			}
		}
	}

	/*Diagnostic Stokes*/
	if (iomodel->analysis_type==DiagnosticAnalysisEnum() && iomodel->sub_analysis_type==StokesAnalysisEnum()){
		/*On a 3d mesh, in stokes formualtions, only stokes grids are free, the others are frozen: */
		if (!iomodel->borderstokes) ISSMERROR("iomodel->borderstokes is NULL");
		if (iomodel->borderstokes[i]){
			//freeze everything except pressure
			this->FreezeDof(1);
			this->FreezeDof(2);
			this->FreezeDof(3);
		}
		else if (iomodel->gridonstokes[i]==0){
			for(k=1;k<=numdofs;k++){
				this->FreezeDof(k);
			}
		}
	}

	/*Diagnostic Hutter*/
	if (iomodel->analysis_type==DiagnosticAnalysisEnum() && iomodel->sub_analysis_type==HutterAnalysisEnum()){
		/*Spc all nodes that are not Hutter*/
		if (!iomodel->gridonhutter) ISSMERROR("iomodel->gridonhutter is NULL");
		if (!iomodel->gridonhutter[i]){
			for(k=1;k<=numdofs;k++){
				this->FreezeDof(k);
			}
		}
	}

	/*Prognostic/ Balancedvelocities/ Balancedthickness*/
	if (iomodel->analysis_type==PrognosticAnalysisEnum() || iomodel->analysis_type==BalancedvelocitiesAnalysisEnum() || iomodel->analysis_type==BalancedthicknessAnalysisEnum()){
		if (strcmp(iomodel->meshtype,"3d")==0){
			/*On a 3d mesh, we may have collapsed elements, hence dead grids. Freeze them out: */
			if (!iomodel->gridonbed) ISSMERROR("iomodel->gridonbed is NULL");
			if (!iomodel->gridonbed[i]){
				for(k=1;k<=numdofs;k++){
					this->FreezeDof(k);
				}
			}
		}
	}
}
/*}}}*/
/*FUNCTION Node constructor from iomodel discontinuous Galerkin{{{2*/
Node::Node(int i,int j,IoModel* iomodel){
	/* i -> index of the vertex in C indexing
	 * j -> index of the node in C indexing*/

	int numdofs;
	int vertex_id;
	int upper_node_id;

	/*id: */
	this->id=j+1; //matlab indexing

	/*indexing:*/
	DistributeNumDofs(&numdofs,iomodel->analysis_type,iomodel->sub_analysis_type); //number of dofs per node

	this->indexing.Init(numdofs);

	/*properties (come from vertex number i): */
	this->properties.Init(
				(int)iomodel->gridonbed[i],
				(int)iomodel->gridonsurface[i],
				(int)iomodel->gridoniceshelf[i],
				(int)iomodel->gridonicesheet[i]);

	/*hooks: */
	vertex_id=i+1; //matlab indexing

	if (strcmp(iomodel->meshtype,"3d")==0){
		if (isnan(iomodel->uppernodes[i])){
			upper_node_id=this->id; //nodes on surface do not have upper nodes, only themselves.
		}
		else{
			upper_node_id=(int)iomodel->uppernodes[i];
		}
	}
	else{
		/*If we are running 2d, upper_node does not mean much. Just point towards itself!:*/
		upper_node_id=this->id;
	}

	this->hvertex.Init(&vertex_id,1);
	this->hupper_node.Init(&upper_node_id,1);

}
/*}}}*/
/*FUNCTION Node destructor{{{2*/
Node::~Node(){
	return;
}
/*}}}*/
/*}}}*/
/*Object management: {{{1*/
/*FUNCTION Node Configure {{{2*/
void  Node::Configure(void* pnodes, void* pvertices){

	int i;

	DataSet* nodesin=NULL;
	DataSet* verticesin=NULL;

	/*Recover pointers :*/
	nodesin=(DataSet*)pnodes;
	verticesin=(DataSet*)pvertices;

	/*Take care of hooking up all objects for this element, ie links the objects in the hooks to their respective 
	 * datasets, using internal ids and ofthis->indexing.f_sets hidden in hooks: */
	hvertex.configure(verticesin);
	hupper_node.configure(nodesin);

}
/*}}}*/
/*FUNCTION Node copy {{{2*/
Object* Node::copy() {
		
	return new Node(this->id,&this->indexing, &this->properties, &this->hvertex,&this->hupper_node);

}

/*}}}*/
/*FUNCTION Node DeepEcho{{{2*/
void Node::DeepEcho(void){

	printf("Node:\n");
	printf("   id: %i\n",id);
	indexing.DeepEcho();
	properties.DeepEcho();
	hvertex.DeepEcho();
	hupper_node.DeepEcho();

}
/*}}}*/
/*FUNCTION Node Demarshall{{{2*/
void  Node::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/

	memcpy(&id,marshalled_dataset,sizeof(id));marshalled_dataset+=sizeof(id);
	
	/*demarshall objects: */
	indexing.Demarshall(&marshalled_dataset);
	properties.Demarshall(&marshalled_dataset);
	hvertex.Demarshall(&marshalled_dataset);
	hupper_node.Demarshall(&marshalled_dataset);
	
	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Node Echo{{{2*/
void Node::Echo(void){

	printf("Node:\n");
	printf("   id: %i\n",id);
	indexing.Echo();
	properties.Echo();
	hvertex.Echo();
	hupper_node.Echo();

}
/*}}}*/
/*FUNCTION Node Enum{{{2*/
int Node::Enum(void){

	return NodeEnum();

}
/*}}}*/
/*FUNCTION Node GetDof {{{2*/
int   Node::GetDof(int dofindex){

	return indexing.doflist[dofindex];

}
/*}}}*/
/*FUNCTION Node GetDofList1{{{2*/
int  Node::GetDofList1(void){

	Vertex* vertex=NULL;

	vertex=(Vertex*)this->hvertex.delivers();

	return vertex->dof;
}
/*}}}*/
/*FUNCTION Node GetDofList{{{2*/
void  Node::GetDofList(int* outdoflist,int* pnumberofdofspernode){

	int i;
	for(i=0;i<this->indexing.numberofdofs;i++){
		outdoflist[i]=indexing.doflist[i];
	}
	/*Assign output pointers:*/
	*pnumberofdofspernode=this->indexing.numberofdofs;
}
/*}}}*/
/*FUNCTION Node GetId{{{2*/
int    Node::GetId(void){ return id; }
/*}}}*/
/*FUNCTION Node GetName{{{2*/
char* Node::GetName(void){
	return "node";
}
/*}}}*/
/*FUNCTION Node GetVertexDof {{{2*/
int   Node::GetVertexDof(void){

	Vertex*  vertex=NULL;

	vertex=(Vertex*)hvertex.delivers();
	return vertex->dof;
}
/*}}}*/
/*FUNCTION Node Marshall{{{2*/
void  Node::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_type=0;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of Node: */
	enum_type=NodeEnum();
	
	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	
	/*marshall Node data: */
	memcpy(marshalled_dataset,&id,sizeof(id));marshalled_dataset+=sizeof(id);
	
	/*marshall objects: */
	indexing.Marshall(&marshalled_dataset);
	properties.Marshall(&marshalled_dataset);
	hvertex.Marshall(&marshalled_dataset);
	hupper_node.Marshall(&marshalled_dataset);

	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Node MarshallSize{{{2*/
int   Node::MarshallSize(){

	return sizeof(id)+
		+indexing.MarshallSize()+
		+properties.MarshallSize()+
		+hvertex.MarshallSize()+
		+hupper_node.MarshallSize()+
		sizeof(int); //sizeof(int) for enum type
}
/*}}}*/
/*FUNCTION Node SetVertexDof {{{2*/
void   Node::SetVertexDof(int in_dof){

	Vertex*  vertex=NULL;

	vertex=(Vertex*)hvertex.delivers();
	vertex->dof=in_dof;

}
/*}}}*/
/*}}}*/
/*Object numerics: {{{1*/
/*FUNCTION Node ApplyConstraints{{{2*/
void  Node::ApplyConstraint(Vec yg,int dof,double value){

	int index;

	/*First, dof should be added in the s set, describing which 
	 * dofs are constrained to a certain value (dirichlet boundary condition*/

	DofInSSet(dof-1);

	/*Second, we should add value into yg, at dof corresponding to doflist[dof], unless
	 *  we are a clone!*/

	if(!indexing.clone){

		index=indexing.doflist[dof-1]; //matlab indexing

		VecSetValues(yg,1,&index,&value,INSERT_VALUES);

	}

}
/*}}}*/
/*FUNCTION Node CreateVecSets {{{2*/
void  Node::CreateVecSets(Vec pv_g,Vec pv_m,Vec pv_n,Vec pv_f,Vec pv_s){

	double gvalue=1.0; //all nodes are in the g set;
	double value;

	int i;

	for(i=0;i<this->indexing.numberofdofs;i++){

		/*g set: */
		VecSetValues(pv_g,1,&indexing.doflist[i],&gvalue,INSERT_VALUES);
		
		/*m set: */
		value=(double)this->indexing.m_set[i];
		VecSetValues(pv_m,1,&indexing.doflist[i],&value,INSERT_VALUES);

		/*n set: */
		value=(double)this->indexing.n_set[i];
		VecSetValues(pv_n,1,&indexing.doflist[i],&value,INSERT_VALUES);

		/*f set: */
		value=(double)this->indexing.f_set[i];
		VecSetValues(pv_f,1,&indexing.doflist[i],&value,INSERT_VALUES);

		/*s set: */
		value=(double)this->indexing.s_set[i];
		VecSetValues(pv_s,1,&indexing.doflist[i],&value,INSERT_VALUES);

	}


}
/*}}}*/
/*FUNCTION Node DofInMSet{{{2*/
void  Node::DofInMSet(int dof){

	/*Put dof for this node into the m set (m set is for rigid body modes)*/

	this->indexing.m_set[dof]=1; //m and n are mutually exclusive (m for rigid body modes)
	this->indexing.n_set[dof]=0;
	this->indexing.f_set[dof]=0; //n splits into f (for which we solve) and s (single point constraints)
	this->indexing.s_set[dof]=0;
}
/*}}}*/
/*FUNCTION Node DofInSSet {{{2*/
void  Node::DofInSSet(int dof){

	/*Put dof for this node into the s set (ie, this dof will be constrained 
	 * to a fixed value during computations. */

	this->indexing.m_set[dof]=0; //m and n are mutually exclusive (m for rigid body modes)
	this->indexing.n_set[dof]=1;
	this->indexing.f_set[dof]=0; //n splits into f (for which we solve) and s (single point constraints)
	this->indexing.s_set[dof]=1;
}
/*}}}*/
/*FUNCTION Node DofIsInMSet{{{2*/
int  Node::DofIsInMSet(int dof){

	if (this->indexing.m_set[dof])return 1;
	else return 0;

}
/*}}}*/
/*FUNCTION Node FieldAverageOntoVertices{{{2*/
void  Node::FieldAverageOntoVertices(Vec field,double* field_serial,char* fieldname){

	ISSMERROR("NOT SUPPORTED YET"); // we need the vertex number !
}
/*}}}*/
/*FUNCTION Node FieldDepthAverageAtBase{{{2*/
void  Node::FieldDepthAverageAtBase(Vec field,double* field_serial,char* fieldname){

	/* node data: */
	int          vertexdof;
	int          dofx,dofy;
	int          isnodeonsurface;
	
	Node* node=NULL;
	Node* upper_node=NULL;
	double z1,z2,dz;
	double thickness;

	/*Are we on the base, not on the surface, and not on a clone node?:*/
	
	if(properties.onbed==1 & indexing.clone==0 &properties.onsurface==0){
			
		vertexdof=this->GetVertexDof();

		/*this node is on the bed. We are going to, follow the upper nodes until we reach the surface. At each upper node, 
		 * we'll grab the * field for this node, and add it to overall field: */

		if(strcmp(fieldname,"velocity")==0){

			/*field is a velocity, 2 dofs per node: */
			double velocity2[2];
			double velocity1[2];
			double velocity_average[2];
			double sum[2];

			sum[0]=0;
			sum[1]=0;
			thickness=0;

			/*get dofs for this base node velocity: we know there are two dofs in field_serial */
			dofx=2*vertexdof;
			dofy=2*vertexdof+1;

			node=this;
			for(;;){

				if (node->IsOnSurface())break;

				vertexdof=node->GetVertexDof();
				
				velocity1[0]=field_serial[2*vertexdof];
				velocity1[1]=field_serial[2*vertexdof+1];
				z1=node->GetZ();

				upper_node=node->GetUpperNode();
				vertexdof=upper_node->GetVertexDof();
			
				velocity2[0]=field_serial[2*vertexdof];
				velocity2[1]=field_serial[2*vertexdof+1];
				z2=upper_node->GetZ();

				dz=(z2-z1);
				thickness+=dz;
				velocity_average[0]=(velocity1[0]+velocity2[0])/2.0;
				velocity_average[1]=(velocity1[1]+velocity2[1])/2.0;

				sum[0]+=velocity_average[0]*dz;
				sum[1]+=velocity_average[1]*dz;

				/* get next node: */
				node=node->GetUpperNode();
			}

			sum[0]=sum[0]/thickness;
			sum[1]=sum[1]/thickness;

			/* Plfield velocity_average*deltaH/H into base of field: */
			VecSetValues(field,1,&dofx,&sum[0],INSERT_VALUES);
			VecSetValues(field,1,&dofy,&sum[1],INSERT_VALUES);
		}
		else{
			/*field is regular, 1 dof per node: */
			double field2;
			double field1;
			double field_average;
			double sum;

			sum=0;
			thickness=0;

			/*get dofs for this base node velocity: we know there are two dofs in field_serial */
			dofx=vertexdof;

			node=this;
			for(;;){

				if (node->IsOnSurface())break;

				vertexdof=node->GetVertexDof();
				
				field1=field_serial[vertexdof];
				z1=node->GetZ();

				upper_node=node->GetUpperNode();
				vertexdof=upper_node->GetVertexDof();
			
				field2=field_serial[vertexdof];
				z2=upper_node->GetZ();

				dz=(z2-z1);
				thickness+=dz;
				field_average=(field1+field2)/2.0;

				sum+=field_average*dz;

				/* get next node: */
				node=node->GetUpperNode();
			}

			sum=sum/thickness;

			/* Plug field_average*deltH/H into base of field: */
			VecSetValues(field,1,&dofx,&sum,INSERT_VALUES);
		}
	}
}
/*}}}*/
/*FUNCTION Node FieldExtrude {{{2*/
void  Node::FieldExtrude(Vec field,double* field_serial,char* field_name){
		
	/* node data: */
	int   numberofdofspernode;
	Node* node=NULL;
	int   i;

	/*Is this node on bed? :*/
	if (properties.onbed){

		if (strcmp(field_name,"velocity")==0){

			/* node data: */
			int          vertexdof;
			int          doflist[2];
			double       fieldnode[2];

			vertexdof=GetVertexDof();

			/*get dofs for this base node velocity: we know there are two dofs in field_serial */
			doflist[0]=2*vertexdof;
			doflist[1]=2*vertexdof+1;

			//initilaize node
			node=this;
		
			/*get field for this base node: */
			fieldnode[0]=field_serial[doflist[0]];
			fieldnode[1]=field_serial[doflist[1]];

			//go through all nodes which sit on top of this node, until we reach the surface, 
			//and plfield  field in field
			for(;;){

				vertexdof=node->GetVertexDof();
				doflist[0]=2*vertexdof;
				doflist[1]=2*vertexdof+1;
				VecSetValues(field,1,&doflist[0],&fieldnode[0],INSERT_VALUES);
				VecSetValues(field,1,&doflist[1],&fieldnode[1],INSERT_VALUES);

				if (node->IsOnSurface())break;
				/*get next node: */
				node=node->GetUpperNode();
			}
		} //if (strcmp(field_name,"velocity")==0)
		else if (strcmp(field_name,"gradj")==0){

			/* node data: */
			int          dof1;
			double       fieldel;

			//initilaize node and get dof1
			node=this;
			dof1=node->GetVertexDof();

			/*get field for this base node: */
			fieldel=field_serial[dof1];

			//go throfieldn all nodes which sit on top of this node, until we reach the surface, 
			//and plfield  field in field
			for(;;){

				dof1=node->GetVertexDof();
				VecSetValues(field,1,&dof1,&fieldel,INSERT_VALUES);

				if (node->IsOnSurface())break;
				/*get next node: */
				node=node->GetUpperNode();
			}
		}
		else if ( 
				(strcmp(field_name,"vx")==0) ||
				(strcmp(field_name,"vy")==0) ||
				(strcmp(field_name,"thickness")==0) ||
				(strcmp(field_name,"temperature")==0) ||
				(strcmp(field_name,"surface")==0)  ||
				(strcmp(field_name,"bed")==0)  || 
				(strcmp(field_name,"slopex")==0)  ||
				(strcmp(field_name,"slopey")==0) 
				){

			/* node data: */
			int          doflist;
			int          nodedofs;
			double       fieldnode;
				
			GetDofList(&doflist,&numberofdofspernode);

			//initilaize node
			node=this;

			/*get field for this bed node: */
			fieldnode=field_serial[doflist];

			//go through all nodes which sit on top of this node, until we reach the surface, 
			//and pltg  fieldnode in field:
			for(;;){

				node->GetDofList(&nodedofs,&numberofdofspernode);
				VecSetValues(field,1,&nodedofs,&fieldnode,INSERT_VALUES);

				if (node->IsOnSurface())break;
				/*get next node: */
				node=node->GetUpperNode();
			}
		}
		else ISSMERROR(exprintf("%s%s%s"," field ",field_name," not supported yet!"));

	} //if (extrude)
}
/*}}}*/
/*FUNCTION Node FreezeDof{{{2*/
void  Node::FreezeDof(int dof){
	
	DofInSSet(dof-1); //with 0 displacement for this dof.

}
/*}}}*/
/*FUNCTION Node GetNumberOfDofs{{{2*/
int   Node::GetNumberOfDofs(){
	
	return this->indexing.numberofdofs;

}
/*}}}*/
/*FUNCTION Node GetSigma {{{2*/
double Node::GetSigma(){
	Vertex* vertex=NULL;

	vertex=(Vertex*)hvertex.delivers();
	return vertex->sigma;
}
/*}}}*/
/*FUNCTION Node GetUpperNode {{{2*/
Node* Node::GetUpperNode(){
	Node* upper_node=NULL;
	upper_node=(Node*)hupper_node.delivers();
	return upper_node;
}
/*}}}*/
/*FUNCTION Node GetX {{{2*/
double Node::GetX(){
	Vertex* vertex=NULL;

	vertex=(Vertex*)hvertex.delivers();
	return vertex->x;
}
/*}}}*/
/*FUNCTION Node GetY {{{2*/
double Node::GetY(){
	Vertex* vertex=NULL;

	vertex=(Vertex*)hvertex.delivers();
	return vertex->y;
}
/*}}}*/
/*FUNCTION Node GetZ {{{2*/
double Node::GetZ(){
	Vertex* vertex=NULL;

	vertex=(Vertex*)hvertex.delivers();
	return vertex->z;
}
/*}}}*/
/*FUNCTION Node IsClone {{{2*/
int   Node::IsClone(){
	
	return indexing.clone;

}
/*}}}*/
/*FUNCTION Node IsOnBed {{{2*/
int   Node::IsOnBed(){
	return properties.onbed;
}
/*}}}*/
/*FUNCTION Node IsOnSheet {{{2*/
int   Node::IsOnSheet(){
	return properties.onsheet;
}		
/*}}}*/
/*FUNCTION Node IsOnShelf {{{2*/
int   Node::IsOnShelf(){
	return properties.onshelf;
}
/*}}}*/
/*FUNCTION Node IsOnSurface {{{2*/
int   Node::IsOnSurface(){
	return properties.onsurface;
}
/*}}}*/
/*FUNCTION Node MyRank{{{2*/
int    Node::MyRank(void){ 
	extern int my_rank;

	return my_rank; 
}
/*}}}*/
/*FUNCTION Node UpdateFromInputs {{{2*/
void  Node::UpdateFromInputs(void* vinputs){

	ISSMERROR("not used yet!");
	
}
/*}}}*/
/*}}}*/
/* DofObject routines: {{{1*
/*FUNCTION Node DistributeDofs{{{2*/
void  Node::DistributeDofs(int* pdofcount){

	int i;
	extern int my_rank;
	int dofcount;

	dofcount=*pdofcount;
	
	if(indexing.clone){
		/*This node is a clone! Don't distribute dofs, it will get them from another cpu!*/
		return;
	}

	/*This node should distribute dofs, go ahead: */
	for(i=0;i<this->indexing.numberofdofs;i++){
		indexing.doflist[i]=dofcount+i;
	}
	dofcount+=this->indexing.numberofdofs;

	/*Assign output pointers: */
	*pdofcount=dofcount;

}
/*}}}*/
/*FUNCTION Node OffsetDofs{{{2*/
void  Node::OffsetDofs(int dofcount){
	
	int i;
	extern int my_rank;
	
	if(indexing.clone){
		/*This node is a clone, don't offset the dofs!: */
		return;
	}

	/*This node should offset the dofs, go ahead: */
	for(i=0;i<this->indexing.numberofdofs;i++){
		indexing.doflist[i]+=dofcount;
	}
}
/*}}}*/
/*FUNCTION Node ShowTrueDofs{{{2*/
void  Node::ShowTrueDofs(int* truedofs){

	int j;
	extern int my_rank;
	
	/*Are we a clone? : */
	if(indexing.clone)return;

	/*Ok, we are not a clone, just plug our dofs into truedofs: */
	for(j=0;j<this->indexing.numberofdofs;j++){
		*(truedofs+this->indexing.numberofdofs*(id-1)+j)=indexing.doflist[j];
	}

}
/*}}}*/
/*FUNCTION Node UpdateCloneDofs{{{2*/
void  Node::UpdateCloneDofs(int* alltruedofs){

	int j;
	extern int my_rank;
	
	/*If we are not a clone, don't update, we already have dofs!: */
	if(indexing.clone==0)return;

	/*Ok, we are a clone node, but we did not create the dofs for this node. 
	 * Therefore, our doflist is garbage right now. Go pick it up in the alltruedofs: */
	for(j=0;j<this->indexing.numberofdofs;j++){
		indexing.doflist[j]=*(alltruedofs+this->indexing.numberofdofs*(id-1)+j);
	}
}
/*}}}*/
/*FUNCTION Node SetClone {{{2*/
void  Node::SetClone(int* minranks){

	extern int my_rank;

	if (minranks[id-1]==my_rank){
		indexing.clone=0;
	}
	else{
		/*!there is a cpu with lower rank that has the same node, 
		therefore, I am a clone*/
		indexing.clone=1; 	
	}

}
/*}}}*/
/*FUNCTION Node CreatePartition{{{2*/
void  Node::CreatePartition(Vec partition){ 

	int      idxm;
	double   value;

	idxm=(id-1);
	value=(double)this->indexing.doflist[0];
	ISSMASSERT(value>=0);

	VecSetValues(partition,1,&idxm,&value,INSERT_VALUES);

	return;
}
/*}}}*/
/*}}}*/
