/*!\file Vertex.c
 * \brief: implementation of the Vertex object
 */

/*Include files: {{{1*/
#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <string.h>
#include "./Vertex.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../include/macros.h"
#include "../shared/shared.h"
#include "../include/typedefs.h"
#include "../include/macros.h"
#include "../ModelProcessorx/IoModel.h"
/*}}}*/

/*Object constructors and destructor{{{1*/
/*FUNCTION Vertex default constructor {{{2*/
Vertex::Vertex(){
	return;
}
/*}}}*/
/*FUNCTION Vertex constructor {{{2*/
Vertex::Vertex(int tria_id, double tria_x, double tria_y, double tria_z, double tria_sigma){
	this->Init(tria_id, tria_x, tria_y, tria_z, tria_sigma);
}
/*}}}*/
/*FUNCTION Vertex init, used by constructor {{{2*/
void Vertex::Init(int tria_id, double tria_x, double tria_y, double tria_z, double tria_sigma){

	/*all the initialization has been done by the initializer, just fill in the id: */
	this->id=tria_id;
	this->x=tria_x;
	this->y=tria_y;
	this->z=tria_z;
	this->sigma=tria_sigma;
	this->dof=UNDEF;

	return;
}
/*}}}*/
/*FUNCTION Vertex constructor  from iomodel{{{2*/
Vertex::Vertex(int i, IoModel* iomodel){

	this->Init(i+1, iomodel->x[i],iomodel->y[i],iomodel->z[i],(iomodel->z[i]-iomodel->bed[i])/(iomodel->thickness[i]));

}
/*}}}*/
/*FUNCTION Vertex destructor {{{2*/
Vertex::~Vertex(){
	return;
}
/*}}}*/
/*}}}*/
/*Object management {{{1*/
/*FUNCTION copy {{{2*/
Object* Vertex::copy() {

	return new Vertex(*this); 

}
/*}}}*/
/*FUNCTION DeepEcho{{{2*/

void Vertex::DeepEcho(void){
	this->Echo();
}
/*}}}*/
/*FUNCTION Demarshall {{{2*/
void  Vertex::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   i;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/

	memcpy(&id,marshalled_dataset,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(&x,marshalled_dataset,sizeof(x));marshalled_dataset+=sizeof(x);
	memcpy(&y,marshalled_dataset,sizeof(y));marshalled_dataset+=sizeof(y);
	memcpy(&z,marshalled_dataset,sizeof(z));marshalled_dataset+=sizeof(z);
	memcpy(&sigma,marshalled_dataset,sizeof(sigma));marshalled_dataset+=sizeof(sigma);
	memcpy(&dof,marshalled_dataset,sizeof(dof));marshalled_dataset+=sizeof(dof);
	memcpy(&clone,marshalled_dataset,sizeof(clone));marshalled_dataset+=sizeof(clone);

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Echo{{{2*/

void Vertex::Echo(void){

	printf("Vertex:\n");
	printf("   id: %i\n",id);
	printf("   x: %g\n",x);
	printf("   y: %g\n",y);
	printf("   z: %g\n",z);
	printf("   sigma: %g\n",sigma);
	printf("   dof: %g\n",dof);
	printf("   clone: %g\n",clone);

	return;
}
/*}}}*/
/*FUNCTION Enum {{{2*/
int Vertex::Enum(void){

	return VertexEnum();

}
/*}}}*/
/*FUNCTION GetId{{{2*/
int    Vertex::GetId(void){ return id; }
/*}}}*/
/*FUNCTION GetName{{{2*/
char* Vertex::GetName(void){
	return "node";
}
/*}}}*/
/*FUNCTION Marshall {{{2*/
void  Vertex::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_type=0;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of Vertex: */
	enum_type=VertexEnum();

	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);

	/*marshall Vertex data: */
	memcpy(marshalled_dataset,&id,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(marshalled_dataset,&x,sizeof(x));marshalled_dataset+=sizeof(x);
	memcpy(marshalled_dataset,&y,sizeof(y));marshalled_dataset+=sizeof(y);
	memcpy(marshalled_dataset,&z,sizeof(z));marshalled_dataset+=sizeof(z);
	memcpy(marshalled_dataset,&sigma,sizeof(sigma));marshalled_dataset+=sizeof(sigma);
	memcpy(marshalled_dataset,&dof,sizeof(dof));marshalled_dataset+=sizeof(dof);
	memcpy(marshalled_dataset,&clone,sizeof(clone));marshalled_dataset+=sizeof(clone);

	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION MarshallSize {{{2*/
int   Vertex::MarshallSize(){
	
	return sizeof(id)+
		sizeof(x)+
		sizeof(y)+
		sizeof(z)+
		sizeof(sigma)+
		sizeof(dof)+
		sizeof(clone)+
		+sizeof(int); //sizeof(int) for enum type
}
/*}}}*/
/*FUNCTION MyRank {{{2*/
int    Vertex::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}*/
/*FUNCTION UpdateFromDakota {{{2*/
void  Vertex::UpdateFromDakota(void* vinputs){

	ISSMERROR("not supported yet!");
	
}
/*}}}*/
/*FUNCTION UpdateFromInputs {{{2*/
void  Vertex::UpdateFromInputs(void* vinputs){
	
	ParameterInputs* inputs=NULL;
	Vertex* vertex=NULL;
	int dof[1]={0};
	double coord[3];

	vertex=this;

	coord[0]=this->x; coord[1]=this->y; coord[2]=this->z;

	/*Recover parameter inputs: */
	inputs=(ParameterInputs*)vinputs;

	/*Update internal data if inputs holds new values: */
	inputs->Recover("x",&coord[0],1,dof,1,(void**)&vertex);
	inputs->Recover("y",&coord[1],1,dof,1,(void**)&vertex);
	inputs->Recover("z",&coord[2],1,dof,1,(void**)&vertex);
	
	ISSMERROR("not supported yet!");
	
}
/*}}}*/
/*FUNCTION UpdateVertexPosition {{{2*/
void  Vertex::UpdatePosition(double* thickness,double* bed){

	/*sigma remains constant. z=bed+sigma*thickness*/
	this->z=bed[dof]+sigma*thickness[this->dof];

}
/*}}}*/
/*}}}*/
/* DofObject routines: {{{1*
/*FUNCTION Vertex DistributeDofs{{{2*/
void  Vertex::DistributeDofs(int* pdofcount){

	int i;
	extern int my_rank;
	int dofcount;

	dofcount=*pdofcount;
	
	if(this->clone){
		/*This vertex is a clone! Don't distribute dofs, it will get them from another cpu!*/
		return;
	}

	/*This vertex should distribute his dof, go ahead: */
	this->dof=dofcount;
	dofcount++;

	/*Assign output pointers: */
	*pdofcount=dofcount;

}
/*}}}*/
/*FUNCTION Vertex OffsetDofs{{{2*/
void  Vertex::OffsetDofs(int dofcount){
	
	int i;
	extern int my_rank;
	
	if(this->clone){
		/*This vertex is a clone, don't offset the dofs!: */
		return;
	}

	/*This vertex should offset his dof, go ahead: */
	this->dof+=dofcount;
}
/*}}}*/
/*FUNCTION Vertex ShowTrueDofs{{{2*/
void  Vertex::ShowTrueDofs(int* truedofs){

	int j;
	extern int my_rank;
	
	/*Are we a clone? : */
	if(this->clone)return;

	/*Ok, we are not a clone, just plug our dof into truedofs: */
	truedofs[this->id-1]=this->dof;

}
/*}}}*/
/*FUNCTION Vertex UpdateCloneDofs{{{2*/
void  Vertex::UpdateCloneDofs(int* alltruedofs){

	int j;
	extern int my_rank;
	
	/*If we are not a clone, don't update, we already have dofs!: */
	if(this->clone==0)return;

	/*Ok, we are a clone node, but we did not create the dof for this vertex 
	 * Therefore, our dof is garbage right now. Go pick it up in the alltruedofs: */
	this->dof=alltruedofs[id-1];
}
/*}}}*/
/*FUNCTION Vertex SetClone {{{2*/
void  Vertex::SetClone(int* minranks){

	extern int my_rank;

	if (minranks[id-1]==my_rank){
		this->clone=0;
	}
	else{
		/*!there is a cpu with lower rank that has the same vertex, 
		therefore, I am a clone*/
		this->clone=1; 	
	}

}
/*}}}*/
/*}}}*/
