/*!\file Numericalflux.c
 * \brief: implementation of the Numericalflux object
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../shared/shared.h"
#include "../include/typedefs.h"
#include "../include/macros.h"
#include "./objects.h"

extern int my_rank;

/*Object constructors and destructor*/
/*FUNCTION Numericalflux::Numericalflux(){{{1*/
Numericalflux::Numericalflux(){
	return;
}
/*}}}*/
/*FUNCTION Numericalflux::Numericalflux(char numericalflux_type[NUMERICALFLUXSTRING],int numericalflux_fill...){{{1*/
Numericalflux::Numericalflux(int numericalflux_id,char numericalflux_type[NUMERICALFLUXSTRING], int numericalflux_node_ids[MAX_NUMERICALFLUX_NODES],int numericalflux_element_id){

	int i;
	
	strcpy(type,numericalflux_type);
	id=numericalflux_id;

	element_id=numericalflux_element_id;
	element_offset=UNDEF;
	element=NULL;

	for(i=0;i<MAX_NUMERICALFLUX_NODES;i++){
		node_ids[i]=numericalflux_node_ids[i];
		node_offsets[i]=UNDEF;
		nodes[i]=NULL;
	}

	return;
}
/*}}}*/
/*FUNCTION Numericalflux::~Numericalflux(){{{1*/
Numericalflux::~Numericalflux(){
	return;
}
/*}}}*/
/*FUNCTION Numericalflux::copy {{{1*/
Object* Numericalflux::copy() {
	return new Numericalflux(*this); 
}
/*}}}*/

/*Object marshall*/
/*FUNCTION Numericalflux::Demarshall {{{1*/
void  Numericalflux::Demarshall(char** pmarshalled_dataset){

	int i;
	char* marshalled_dataset=NULL;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/

	memcpy(&type,marshalled_dataset,sizeof(type));marshalled_dataset+=sizeof(type);
	memcpy(&id,marshalled_dataset,sizeof(id));marshalled_dataset+=sizeof(id);

	memcpy(&element_id,marshalled_dataset,sizeof(element_id));marshalled_dataset+=sizeof(element_id);
	memcpy(&element_offset,marshalled_dataset,sizeof(element_offset));marshalled_dataset+=sizeof(element_offset);
	element=NULL;
	
	memcpy(&node_ids,marshalled_dataset,sizeof(node_ids));marshalled_dataset+=sizeof(node_ids);
	memcpy(&node_offsets,marshalled_dataset,sizeof(node_offsets));marshalled_dataset+=sizeof(node_offsets);
	for(i=0;i<MAX_NUMERICALFLUX_NODES;i++)nodes[i]=NULL;

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Numericalflux::Marshall {{{1*/
void  Numericalflux::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_type=0;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of Numericalflux: */
	enum_type=NumericalfluxEnum();
	
	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	
	/*marshall Numericalflux data: */
	memcpy(marshalled_dataset,&type,sizeof(type));marshalled_dataset+=sizeof(type);
	memcpy(marshalled_dataset,&id,sizeof(id));marshalled_dataset+=sizeof(id);
	
	memcpy(marshalled_dataset,&element_id,sizeof(element_id));marshalled_dataset+=sizeof(element_id);
	memcpy(marshalled_dataset,&element_offset,sizeof(element_offset));marshalled_dataset+=sizeof(element_offset);

	memcpy(marshalled_dataset,&node_ids,sizeof(node_ids));marshalled_dataset+=sizeof(node_ids);
	memcpy(marshalled_dataset,&node_offsets,sizeof(node_offsets));marshalled_dataset+=sizeof(node_offsets);

	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Numericalflux::MarshallSize{{{1*/
int   Numericalflux::MarshallSize(){

	return sizeof(type)+
		sizeof(id)+
		sizeof(element_id)+
		sizeof(element_offset)+
		sizeof(node_ids)+
		sizeof(node_offsets)+
		sizeof(int); //sizeof(int) for enum type
}
/*}}}*/

/*Object functions*/
/*FUNCTION Numericalflux::Configure {{{1*/
void  Numericalflux::Configure(void* pelementsin,void* pnodesin,void* pmaterialsin){

	DataSet* elementsin=NULL;
	DataSet* nodesin=NULL;

	/*Recover pointers :*/
	elementsin=(DataSet*)pelementsin;
	nodesin=(DataSet*)pnodesin;
	
	/*Link this load with its nodes: */
	if (strcmp(type,"internal")==0){
		ResolvePointers((Object**)nodes,node_ids,node_offsets,4,nodesin);
	}
	else if (strcmp(type,"boundary")==0){
		ResolvePointers((Object**)nodes,node_ids,node_offsets,2,nodesin);
	}
	else ISSMERROR("type not supported yet");
	ResolvePointers((Object**)&element,&element_id,&element_offset,1,elementsin);

}
/*}}}*/
/*FUNCTION Numericalflux::CreateKMatrix {{{1*/
void  Numericalflux::CreateKMatrix(Mat Kgg,void* inputs,int analysis_type,int sub_analysis_type){

	/*No stiffness loads applied, do nothing: */
	return;

}
/*}}}*/
/*FUNCTION Numericalflux::CreatePVector {{{1*/
void  Numericalflux::CreatePVector(Vec pg, void* inputs, int analysis_type,int sub_analysis_type){

	/*No stiffness loads applied, do nothing: */
	return;
}
/*}}}*/
/*FUNCTION Numericalflux::DeepEcho {{{1*/
void Numericalflux::DeepEcho(void){

	int i;
	
	printf("Numericalflux:\n");
	printf("   type: %s\n",type);
	printf("   id: %i\n",id);
	
	printf("   element_id=%i\n",element_id);
	printf("   element_offset=%i\n",element_offset);
	if(element)element->Echo();
	if (strcmp(type,"internal")==0){
		printf("   node_ids=[%i,%i,%i,%i]\n",node_ids[0],node_ids[1],node_ids[2],node_ids[3]);
		printf("   node_offsets=[%i,%i,%i,%i]\n",node_offsets[0],node_offsets[1],node_offsets[2],node_offsets[3]);
		for(i=0;i<4;i++){
			if(nodes[i])nodes[i]->Echo();
		}
	}
	else{
		printf("   node_ids=[%i,%i,%i,%i]\n",node_ids[0],node_ids[1],node_ids[2],node_ids[3]);
		printf("   node_offsets=[%i,%i,%i,%i]\n",node_offsets[0],node_offsets[1],node_offsets[2],node_offsets[3]);
		for(i=0;i<2;i++){
			if(nodes[i])nodes[i]->Echo();
		}
	}
	return;
}		
/*}}}*/
/*FUNCTION Numericalflux::DistributeNumDofs {{{1*/
void  Numericalflux::DistributeNumDofs(int* numdofspernode,int analysis_type,int sub_analysis_type){
	return;
}
/*}}}*/
/*FUNCTION Numericalflux::Echo {{{1*/
void Numericalflux::Echo(void){

	printf("Numericalflux:\n");
	printf("   type: %s\n",type);
	printf("   id: %i\n",id);

	printf("   element_id=%i\n",element_id);
	printf("   element_offset=%i]\n",element_offset);

	if (strcmp(type,"internal")==0){
		printf("   node_ids=[%i,%i,%i,%i]\n",node_ids[0],node_ids[1],node_ids[2],node_ids[3]);
		printf("   node_offsets=[%i,%i,%i,%i]\n",node_offsets[0],node_offsets[1],node_offsets[2],node_offsets[3]);
	}
	else{
		printf("   node_ids=[%i,%i]\n",node_ids[0],node_ids[1]);
		printf("   node_offsets=[%i,%i]\n",node_offsets[0],node_offsets[1]);
	}

	return;
}
/*}}}*/
/*FUNCTION Numericalflux::Enum {{{1*/
int Numericalflux::Enum(void){

	return NumericalfluxEnum();

}
/*}}}*/
/*FUNCTION Numericalflux::GetB {{{1*/
void Numericalflux::GetB(double* B, double gauss_coord){

	const int numgrids=4;
	double l1l4[numgrids];

	/*Get nodal functions: */
	GetNodalFunctions(&l1l4[0],gauss_coord);

	/*Build B: */
	B[0] = +l1l4[0];
	B[1] = +l1l4[1];
	B[2] = -l1l4[0];
	B[3] = -l1l4[1];
}
/*}}}*/
/*FUNCTION Numericalflux::GetDofList{{{1*/

void  Numericalflux::GetDofList(int* doflist,int* pnumberofdofspernode){

	int i,j;
	int doflist_per_node[MAXDOFSPERNODE];
	int numberofdofspernode;
	
	if (strcmp(type,"internal")==0){
		for(i=0;i<4;i++){
			nodes[i]->GetDofList(&doflist_per_node[0],&numberofdofspernode);
			for(j=0;j<numberofdofspernode;j++){
				doflist[i*numberofdofspernode+j]=doflist_per_node[j];
			}
		}
	}
	else if (strcmp(type,"boundary")==0){
		for(i=0;i<2;i++){
			nodes[i]->GetDofList(&doflist_per_node[0],&numberofdofspernode);
			for(j=0;j<numberofdofspernode;j++){
				doflist[i*numberofdofspernode+j]=doflist_per_node[j];
			}
		}
	}
	else ISSMERROR(exprintf("type %s not supported yet",type));

	/*Assign output pointers:*/
	*pnumberofdofspernode=numberofdofspernode;
}
/*}}}*/
/*FUNCTION Numericalflux::GetId {{{1*/
int    Numericalflux::GetId(void){
	return id;
}
/*}}}*/
/*FUNCTION Numericalflux::GetJacobianDeterminant{{{1*/
void Numericalflux::GetJacobianDeterminant(double* pJdet,double xyz_list[4][3], double gauss_coord){

	double Jdet,length;

	length=sqrt(pow(xyz_list[1][0] - xyz_list[0][0],2.0) + pow(xyz_list[1][1] - xyz_list[0][1],2.0));
	Jdet=1.0/2.0*length;

	if(Jdet<0){
		ISSMERROR(" negative jacobian determinant!");
	}

	*pJdet=Jdet;
}
/*}}}*/
/*FUNCTION Numericalflux::GetL {{{1*/
void Numericalflux::GetL(double* L, double gauss_coord, int numdof){

	const int numgrids=4;
	double l1l4[numgrids];

	/*Check numdof*/
	ISSMASSERT(numdof==2 || numdof==4);

	/*Get nodal functions: */
	GetNodalFunctions(&l1l4[0],gauss_coord);

	/*Luild L: */
	L[0] = l1l4[0];
	L[1] = l1l4[1];
	if (numdof==4){
		L[2] = l1l4[2];
		L[3] = l1l4[3];
	}
}
/*}}}*/
/*FUNCTION Numericalflux::GetName {{{1*/
char* Numericalflux::GetName(void){
	return "numericalflux";
}
/*}}}*/
/*FUNCTION Numericalflux::GetNodalFunctions{{{1*/
void Numericalflux::GetNodalFunctions(double* l1l4, double gauss_coord){

	/*This routine returns the values of the nodal functions  at the gaussian point.*/

	l1l4[0]=-0.5*gauss_coord+0.5;
	l1l4[1]=+0.5*gauss_coord+0.5;
	l1l4[2]=-0.5*gauss_coord+0.5;
	l1l4[3]=+0.5*gauss_coord+0.5;

}
/*}}}*/
/*FUNCTION Numericalflux::GetNormal {{{1*/
void Numericalflux:: GetNormal(double* normal,double xyz_list[4][3]){

	/*Build unit outward pointing vector*/
	const int numgrids=4;
	double vector[2];
	double norm;

	vector[0]=xyz_list[1][0] - xyz_list[0][0];
	vector[1]=xyz_list[1][1] - xyz_list[0][1];

	norm=sqrt(pow(vector[0],2.0)+pow(vector[1],2.0));

	normal[0]= + vector[1]/norm;
	normal[1]= - vector[0]/norm;
}
/*}}}*/
/*FUNCTION Numericalflux::GetParameterValue {{{1*/
void Numericalflux::GetParameterValue(double* pp, double* plist, double gauss_coord){

	/*From node values of parameter p (plist[0],plist[1],plist[2]), return parameter value at gaussian 
	 * point specifie by gauss_l1l2l3: */

	/*nodal functions: */
	double l1l4[4];

	/*output: */
	double p;

	GetNodalFunctions(&l1l4[0],gauss_coord);

	p=l1l4[0]*plist[0]+l1l4[1]*plist[1];

	/*Assign output pointers:*/
	*pp=p;
}
/*}}}*/
/*FUNCTION Numericalflux::MyRank {{{1*/
int    Numericalflux::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}*/
/*FUNCTION Numericalflux::PenaltyCreateKMatrix {{{1*/
void  Numericalflux::PenaltyCreateKMatrix(Mat Kgg,void* inputs,double kmax,int analysis_type,int sub_analysis_type){

	if (strcmp(type,"internal")==0){

		PenaltyCreateKMatrixInternal(Kgg,inputs,kmax,analysis_type,sub_analysis_type);
	}
	else if (strcmp(type,"boundary")==0){

		PenaltyCreateKMatrixBoundary(Kgg,inputs,kmax,analysis_type,sub_analysis_type);
	}
	else ISSMERROR("type not supported yet");
}
/*}}}*/
/*FUNCTION Numericalflux::PenaltyCreateKMatrixInternal {{{1*/
void  Numericalflux::PenaltyCreateKMatrixInternal(Mat Kgg,void* vinputs,double kmax,int analysis_type,int sub_analysis_type){
	double   start, finish;
	start=MPI_Wtime();

	/* local declarations */
	int             i,j;

	/* node data: */
	const int numgrids=4;
	const int NDOF1=1;
	const int numdof=NDOF1*numgrids;
	double    xyz_list[numgrids][3];
	double    normal[2];
	int       doflist[numdof];
	int       numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* gauss_coords =NULL;
	double  gauss_coord;
	double* gauss_weights=NULL;
	double  gauss_weight;

	/* matrices: */
	double B[numgrids];
	double L[numgrids];
	double DL1,DL2;
	double Ke_gg1[numdof][numdof];
	double Ke_gg2[numdof][numdof];
	double Ke_gg[numdof][numdof]={0.0};
	double Jdet;

	/*input parameters for structural analysis (diagnostic): */
	double vx_list[numgrids]={0.0};
	double vy_list[numgrids]={0.0};
	double vx,vy;
	double UdotN;
	double dt;
	int    dofs[1]={0};
	int    found;

	ParameterInputs* inputs=NULL;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;

	/*recover extra inputs from users, at current convergence iteration: */
	found=inputs->Recover("vx_average",&vx_list[0],1,dofs,numgrids,(void**)nodes);
	if(!found)ISSMERROR(" could not find vx_average in inputs!");
	found=inputs->Recover("vy_average",&vy_list[0],1,dofs,numgrids,(void**)nodes);
	if(!found)ISSMERROR(" could not find vy_average in inputs!");
	found=inputs->Recover("dt",&dt);
	if(!found)ISSMERROR(" could not find dt in inputs!");

	/* Get node coordinates, dof list and normal vector: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);
	GetNormal(&normal[0],xyz_list);

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	num_gauss=2;
	GaussSegment(&gauss_coords, &gauss_weights,num_gauss);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){

		/*Pick up the gaussian point: */
		gauss_weight=gauss_weights[ig];
		gauss_coord =gauss_coords[ig];

		/* Get Jacobian determinant: */
		GetJacobianDeterminant(&Jdet,xyz_list,gauss_coord);

		//Get vx, vy and v.n
		GetParameterValue(&vx, &vx_list[0],gauss_coord);
		GetParameterValue(&vy, &vy_list[0],gauss_coord);
		UdotN=vx*normal[0]+vy*normal[1];

		/*Get L and B: */
		GetL(&L[0],gauss_coord,numdof);
		GetB(&B[0],gauss_coord);

		/*Compute DLs*/
		DL1=gauss_weight*Jdet*dt*UdotN/2;
		DL2=gauss_weight*Jdet*dt*fabs(UdotN)/2;

		/*  Do the triple products: */
		TripleMultiply(&B[0],1,numdof,1,
					&DL1,1,1,0,
					&L[0],1,numdof,0,
					&Ke_gg1[0][0],0);
		TripleMultiply(&B[0],1,numdof,1,
					&DL2,1,1,0,
					&B[0],1,numdof,0,
					&Ke_gg2[0][0],0);

		/* Add the Ke_gg_gaussian, and optionally Ke_gg_drag_gaussian onto Ke_gg: */
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg1[i][j];
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg2[i][j];

	} // for (ig=0; ig<num_gauss; ig++)

	/*Add Ke_gg to global matrix Kgg: */
	start=MPI_Wtime();
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);
	finish=MPI_Wtime();
	//printf("internal inserting K of edge %i matrix: %g\n",id,finish-start);

	xfree((void**)&gauss_coords);
	xfree((void**)&gauss_weights);
}
/*}}}*/
/*FUNCTION Numericalflux::PenaltyCreateKMatrixBoundary {{{1*/
void  Numericalflux::PenaltyCreateKMatrixBoundary(Mat Kgg,void* vinputs,double kmax,int analysis_type,int sub_analysis_type){
	double   start, finish;
	start=MPI_Wtime();

	/* local declarations */
	int             i,j;

	/* node data: */
	const int numgrids=2;
	const int NDOF1=1;
	const int numdof=NDOF1*numgrids;
	double    xyz_list[numgrids][3];
	double    normal[2];
	int       doflist[numdof];
	int       numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* gauss_coords =NULL;
	double  gauss_coord;
	double* gauss_weights=NULL;
	double  gauss_weight;

	/* matrices: */
	double L[numgrids];
	double DL;
	double Ke_gg[numdof][numdof]={0.0};
	double Ke_gg_gaussian[numdof][numdof];
	double Jdet;

	/*input parameters for structural analysis (diagnostic): */
	double vx_list[numgrids]={0.0};
	double vy_list[numgrids]={0.0};
	double vx,vy;
	double mean_vx;
	double mean_vy;
	double UdotN;
	double dt;
	int    dofs[1]={0};
	int    found;

	ParameterInputs* inputs=NULL;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;

	/*recover extra inputs from users, at current convergence iteration: */
	found=inputs->Recover("vx_average",&vx_list[0],1,dofs,numgrids,(void**)nodes);
	if(!found)ISSMERROR(" could not find vx_average in inputs!");
	found=inputs->Recover("vy_average",&vy_list[0],1,dofs,numgrids,(void**)nodes);
	if(!found)ISSMERROR(" could not find vy_average in inputs!");
	found=inputs->Recover("dt",&dt);
	if(!found)ISSMERROR(" could not find dt in inputs!");

	/* Get node coordinates, dof list and normal vector: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);
	GetNormal(&normal[0],xyz_list);

	/*Check wether it is an inflow or outflow BC*/
	mean_vx=(vx_list[0]+vx_list[1])/2.0;
	mean_vy=(vy_list[0]+vy_list[1])/2.0;
	UdotN=mean_vx*normal[0]+mean_vy*normal[1];
	if (UdotN<=0){
		/*(u,n)<0 -> inflow, PenaltyCreatePVector will take care of it*/
		return;
	}

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	num_gauss=2;
	GaussSegment(&gauss_coords, &gauss_weights,num_gauss);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){

		/*Pick up the gaussian point: */
		gauss_weight=gauss_weights[ig];
		gauss_coord =gauss_coords[ig];

		/* Get Jacobian determinant: */
		GetJacobianDeterminant(&Jdet,xyz_list,gauss_coord);

		//Get vx, vy and v.n
		GetParameterValue(&vx, &vx_list[0],gauss_coord);
		GetParameterValue(&vy, &vy_list[0],gauss_coord);
		UdotN=vx*normal[0]+vy*normal[1];

		/*Get L*/
		GetL(&L[0],gauss_coord,numdof);

		/*Compute DLs*/
		DL=gauss_weight*Jdet*dt*UdotN;

		/*Do triple product*/
		TripleMultiply(&L[0],1,numdof,1,
					&DL,1,1,0,
					&L[0],1,numdof,0,
					&Ke_gg_gaussian[0][0],0);

		/* Add the Ke_gg_gaussian, and optionally Ke_gg_drag_gaussian onto Ke_gg: */
		for( i=0; i<numdof; i++) for(j=0;j<numdof;j++) Ke_gg[i][j]+=Ke_gg_gaussian[i][j];

	} // for (ig=0; ig<num_gauss; ig++)

	/*Add Ke_gg to global matrix Kgg: */
	start=MPI_Wtime();
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);
	finish=MPI_Wtime();
	//printf("boundary inserting K of edge %i matrix: %g\n",id,finish-start);

	xfree((void**)&gauss_coords);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Numericalflux::PenaltyCreatePVector{{{1*/
void  Numericalflux::PenaltyCreatePVector(Vec pg,void* inputs,double kmax,int analysis_type,int sub_analysis_type){

	if (strcmp(type,"internal")==0){

		PenaltyCreatePVectorInternal(pg,inputs,kmax,analysis_type,sub_analysis_type);
	}
	else if (strcmp(type,"boundary")==0){

		PenaltyCreatePVectorBoundary(pg,inputs,kmax,analysis_type,sub_analysis_type);
	}
	else ISSMERROR("type not supported yet");
}
/*}}}*/
/*FUNCTION Numericalflux::PenaltyCreatePVectorInternal{{{1*/
void  Numericalflux::PenaltyCreatePVectorInternal(Vec pg,void* inputs,double kmax,int analysis_type,int sub_analysis_type){

	/*Nothing added to PVector*/
	return;

}
/*}}}*/
/*FUNCTION Numericalflux::PenaltyCreatePVectorBoundary{{{1*/
void  Numericalflux::PenaltyCreatePVectorBoundary(Vec pg,void* vinputs,double kmax,int analysis_type,int sub_analysis_type){

	/* local declarations */
	int             i,j;

	/* node data: */
	const int numgrids=2;
	const int NDOF1=1;
	const int numdof=NDOF1*numgrids;
	double    xyz_list[numgrids][3];
	double    normal[2];
	int       doflist[numdof];
	int       numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* gauss_coords =NULL;
	double  gauss_coord;
	double* gauss_weights=NULL;
	double  gauss_weight;

	/* matrices: */
	double L[numgrids];
	double DL;
	double pe_g[numdof]={0.0};
	double Jdet;

	/*input parameters for structural analysis (diagnostic): */
	double vx_list[numgrids]={0.0};
	double vy_list[numgrids]={0.0};
	double vx,vy;
	double mean_vx;
	double mean_vy;
	double UdotN;
	double dt;
	int    dofs[1]={0};
	int    found;

	ParameterInputs* inputs=NULL;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;

	/*recover extra inputs from users, at current convergence iteration: */
	found=inputs->Recover("vx_average",&vx_list[0],1,dofs,numgrids,(void**)nodes);
	if(!found)ISSMERROR(" could not find vx_average in inputs!");
	found=inputs->Recover("vy_average",&vy_list[0],1,dofs,numgrids,(void**)nodes);
	if(!found)ISSMERROR(" could not find vy_average in inputs!");
	found=inputs->Recover("dt",&dt);
	if(!found)ISSMERROR(" could not find dt in inputs!");

	/* Get node coordinates, dof list and normal vector: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);
	GetNormal(&normal[0],xyz_list);

	/*Check wether it is an inflow or outflow BC*/
	mean_vx=(vx_list[0]+vx_list[1])/2.0;
	mean_vy=(vy_list[0]+vy_list[1])/2.0;
	UdotN=mean_vx*normal[0]+mean_vy*normal[1];
	if (UdotN>0){
		/*(u,n)>0 -> outflow, PenaltyCreateKMatrix will take care of it*/
		return;
	}

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	num_gauss=2;
	GaussSegment(&gauss_coords, &gauss_weights,num_gauss);

	/* Start  looping on the number of gaussian points: */
	for (ig=0; ig<num_gauss; ig++){

		/*Pick up the gaussian point: */
		gauss_weight=gauss_weights[ig];
		gauss_coord =gauss_coords[ig];

		/* Get Jacobian determinant: */
		GetJacobianDeterminant(&Jdet,xyz_list,gauss_coord);

		//Get vx, vy and v.n
		GetParameterValue(&vx, &vx_list[0],gauss_coord);
		GetParameterValue(&vy, &vy_list[0],gauss_coord);
		UdotN=vx*normal[0]+vy*normal[1];

		/*Get L*/
		GetL(&L[0],gauss_coord,numdof);

		/*Compute DL*/
		DL= - gauss_weight*Jdet*dt*UdotN;

		/* Add value into pe_g: */
		for( i=0; i<numdof; i++) pe_g[i] += DL* L[i];

	} // for (ig=0; ig<num_gauss; ig++)

	/*Add pe_g to global matrix Kgg: */
	VecSetValues(pg,numdof,doflist,(const double*)pe_g,ADD_VALUES);

	xfree((void**)&gauss_coords);
	xfree((void**)&gauss_weights);

}
/*}}}*/
/*FUNCTION Numericalflux::UpdateFromInputs {{{1*/
void  Numericalflux::UpdateFromInputs(void* vinputs){

	/*Do nothing...*/
}
/*}}}*/
/*FUNCTION Numericalflux::SetClone {{{1*/
void  Numericalflux::SetClone(int* minranks){

	ISSMERROR("not implemented yet");
}
/*}}}1*/
