/*!\file: diagnostic_core.cpp
 * \brief: core of the diagnostic solution 
 */ 

#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../shared/shared.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "./parallel.h"
#include "../issm.h"

void diagnostic_core(DataSet* results,Model* model, ParameterInputs* inputs){

	extern int my_rank;

	/*fem models: */
	FemModel* fem_dh=NULL;
	FemModel* fem_dv=NULL;
	FemModel* fem_dhu=NULL;
	FemModel* fem_ds=NULL;
	FemModel* fem_sl=NULL;

	/*output: */
	Result* result=NULL;

	/*solutions: */
	Vec ug=NULL;
	Vec ug_horiz=NULL;
	Vec ug_vert=NULL;
	Vec ug_stokes=NULL;
	Vec pg=NULL;
	Vec slopex=NULL;
	Vec slopey=NULL;
	Vec riftproperties=NULL;
	double* u_g_initial=NULL;

	/*flags: */
	int verbose=0;
	int qmu_analysis=0;
	int dim=-1;
	int ishutter=0;
	int ismacayealpattyn=0;
	int isstokes=0;
	int numberofdofspernode_sl;
	int numberofdofspernode_dh;
	int numberofdofspernode_ds;
	int numberofnodes;
	int numrifts=0;

	double stokesreconditioning;

	/*dof recovery: */
	int dof01[2]={0,1};
	int dof2[1]={2};
	int dof012[3]={0,1,2};
	int dof3[1]={3};
	double* dofset=NULL;


	//first recover parameters common to all solutions
	model->FindParam(&verbose,"verbose");
	model->FindParam(&dim,"dim");
	model->FindParam(&ishutter,"ishutter");
	model->FindParam(&ismacayealpattyn,"ismacayealpattyn");
	model->FindParam(&numberofnodes,"numberofnodes");
	model->FindParam(&isstokes,"isstokes");
	model->FindParam(&stokesreconditioning,"stokesreconditioning");
	model->FindParam(&numrifts,"numrifts");
	model->FindParam(&qmu_analysis,"qmu_analysis");

	/*recover fem models: */
	fem_dh=model->GetFormulation(DiagnosticAnalysisEnum(),HorizAnalysisEnum());
	fem_dv=model->GetFormulation(DiagnosticAnalysisEnum(),VertAnalysisEnum());
	fem_ds=model->GetFormulation(DiagnosticAnalysisEnum(),StokesAnalysisEnum());
	fem_dhu=model->GetFormulation(DiagnosticAnalysisEnum(),HutterAnalysisEnum());
	fem_sl=model->GetFormulation(SlopecomputeAnalysisEnum());

	//specific parameters for specific models
	fem_dh->FindParam(&numberofdofspernode_dh,"numberofdofspernode");
	fem_sl->FindParam(&numberofdofspernode_sl,"numberofdofspernode");
	fem_ds->FindParam(&numberofdofspernode_ds,"numberofdofspernode");

	//for qmu analysis, be sure the velocity input we are starting from  is the one in the parameters: */
	if(qmu_analysis){
		model->FindParam(&u_g_initial,NULL,NULL,"u_g",DiagnosticAnalysisEnum(),HorizAnalysisEnum());
		inputs->Add("velocity",u_g_initial,3,numberofnodes);
	}

	if(ishutter){
			
		if(verbose)_printf_("%s\n","computing surface slope (x and y derivatives)...");
		diagnostic_core_linear(&slopex,fem_sl,inputs,SlopecomputeAnalysisEnum(),SurfaceXAnalysisEnum());
		diagnostic_core_linear(&slopey,fem_sl,inputs,SlopecomputeAnalysisEnum(),SurfaceYAnalysisEnum());

		if (dim==3){
		
			if(verbose)_printf_("%s\n","extruding slopes in 3d...");
			FieldExtrudex( slopex, fem_sl->elements,fem_sl->nodes,fem_sl->vertices,fem_sl->loads,fem_sl->materials,fem_sl->parameters,"slopex",0);
			FieldExtrudex( slopey, fem_sl->elements,fem_sl->nodes,fem_sl->vertices,fem_sl->loads,fem_sl->materials,fem_sl->parameters,"slopex",0);
		}

		if(verbose)_printf_("%s\n"," adding slopes in inputs...");
		inputs->Add("surfaceslopex",slopex,numberofdofspernode_sl,numberofnodes);
		inputs->Add("surfaceslopey",slopey,numberofdofspernode_sl,numberofnodes);
		VecFree(&slopex); VecFree(&slopey);

		if(verbose)_printf_("%s\n"," computing hutter velocities...");
		diagnostic_core_linear(&ug,fem_dhu,inputs,DiagnosticAnalysisEnum(),HutterAnalysisEnum());

		if(verbose)_printf_("%s\n"," computing pressure according to MacAyeal...");
		ComputePressurex( &pg,fem_dhu->elements,fem_dhu->nodes,fem_dhu->vertices,fem_dhu->loads,fem_dhu->materials, fem_dhu->parameters);

		if(verbose)_printf_("%s\n"," update boundary conditions for macyeal pattyn using hutter results...");
		if (ismacayealpattyn){
			VecFree(&fem_dh->yg->vector); VecFree(&fem_dh->ys);VecFree(&fem_dh->ys0);
			VecDuplicatePatch(&fem_dh->yg->vector,ug);
			Reducevectorgtosx(&fem_dh->ys,&fem_dh->ys0, fem_dh->yg->vector,fem_dh->nodesets);
		}

	}

	if (ismacayealpattyn){
		
		if(verbose)_printf_("%s\n"," computing horizontal velocities...");
		diagnostic_core_nonlinear(&ug,NULL,NULL,fem_dh->loads,fem_dh,inputs,DiagnosticAnalysisEnum(),HorizAnalysisEnum());

		if(dim==2){
			if(verbose)_printf_("%s\n"," computing pressure according to MacAyeal...");
			ComputePressurex( &pg,fem_dh->elements, fem_dh->nodes, fem_dh->vertices,fem_dh->loads,  fem_dh->materials, fem_dh->parameters);
		}

	}
	
	
	if (dim==3){

		if(verbose)_printf_("%s\n"," extruding horizontal velocities...");
		VecDuplicatePatch(&ug_horiz,ug); FieldExtrudex( ug_horiz,fem_dh->elements,fem_dh->nodes, fem_dh->vertices,fem_dh->loads,fem_dh->materials,fem_dh->parameters,"velocity",1);

		if(verbose)_printf_("%s\n"," computing vertical velocities...");
		inputs->Add("velocity",ug_horiz,numberofdofspernode_dh,numberofnodes);
		diagnostic_core_linear(&ug_vert,fem_dv,inputs,DiagnosticAnalysisEnum(),VertAnalysisEnum());

		if(verbose)_printf_("%s\n"," combining horizontal and vertical velocities...");
		VecFree(&ug); ug=NewVec(numberofnodes*3);

		xfree((void**)&dofset); dofset=dofsetgen(2,&dof01[0],3,numberofnodes*3); VecMerge(ug,ug_horiz,dofset,numberofnodes*2);
		xfree((void**)&dofset); dofset=dofsetgen(1,&dof2[0],3,numberofnodes*3); VecMerge(ug,ug_vert,dofset,numberofnodes*1);
		VecFree(&ug_horiz); VecFree(&ug_vert);

		if(verbose)_printf_("%s\n"," computing pressure according to Pattyn...");
		ComputePressurex( &pg,fem_dh->elements, fem_dh->nodes, fem_dh->vertices,fem_dh->loads,  fem_dh->materials,fem_dh->parameters);
		
		if (isstokes){

			//"recondition" pressure 
			VecScale(pg,1.0/stokesreconditioning);

			if(verbose)_printf_("%s\n","computing bed slope (x and y derivatives)...");
			diagnostic_core_linear(&slopex,fem_sl,inputs,SlopecomputeAnalysisEnum(),BedXAnalysisEnum());
			diagnostic_core_linear(&slopey,fem_sl,inputs,SlopecomputeAnalysisEnum(),BedYAnalysisEnum());
			FieldExtrudex( slopex, fem_sl->elements,fem_sl->nodes,fem_sl->vertices,fem_sl->loads,fem_sl->materials,fem_sl->parameters,"slopex",0);
			FieldExtrudex( slopey, fem_sl->elements,fem_sl->nodes,fem_sl->vertices,fem_sl->loads,fem_sl->materials,fem_sl->parameters,"slopey",0);

			inputs->Add("bedslopex",slopex,numberofdofspernode_sl,numberofnodes);
			inputs->Add("bedslopey",slopey,numberofdofspernode_sl,numberofnodes);
			VecFree(&slopex); VecFree(&slopey);
			
			//recombine ug and pg: 
			ug_stokes=NewVec(fem_ds->nodesets->GetGSize());
			xfree((void**)&dofset);dofset=dofsetgen(3,dof012,4,numberofnodes*4); VecMerge(ug_stokes,ug,dofset,numberofnodes*3);
			xfree((void**)&dofset); dofset=dofsetgen(1,dof3,4,numberofnodes*4); VecMerge(ug_stokes,pg,dofset,numberofnodes);

			inputs->Add("velocity",ug_stokes,numberofdofspernode_ds,numberofnodes);
			VecFree(&ug_stokes);

			if(verbose)_printf_("%s\n"," update boundary conditions for stokes using velocities previously computed...");
			xfree((void**)&dofset);dofset=dofsetgen(3,dof012,4,numberofnodes*4); VecMerge(fem_ds->yg->vector,ug,dofset,3*numberofnodes);
			VecFree(&fem_ds->ys); VecFree(&fem_ds->ys0);
			Reducevectorgtosx(&fem_ds->ys,&fem_ds->ys0, fem_ds->yg->vector,fem_ds->nodesets);

			if(verbose)_printf_("%s\n"," computing stokes velocities and pressure ...");
			VecFree(&ug);
			diagnostic_core_nonlinear(&ug,NULL,NULL,NULL,fem_ds,inputs,DiagnosticAnalysisEnum(),StokesAnalysisEnum());
		
			//decondition" pressure
			VecFree(&pg);	
			xfree((void**)&dofset);dofset=dofsetgen(1,dof3,4,numberofnodes*4); VecPartition(&pg, ug, dofset, numberofnodes*1);
			VecScale(pg,stokesreconditioning);
		}
	}
	
	/*Plug results into output dataset: */
	result=new Result(results->Size()+1,0,1,"u_g",ug);
	results->AddObject(result);
	result=new Result(results->Size()+1,0,1,"p_g",pg);
	results->AddObject(result);

	/*output if we have rifts: */
	if(numrifts){
		OutputRiftsx( &riftproperties,fem_dh->loads,numrifts);
		result=new Result(results->Size()+1,0,1,"riftproperties",riftproperties);
		results->AddObject(result);
	}

	/*Free ressources: */
	VecFree(&ug);
	VecFree(&pg);
	xfree((void**)&dofset);
	xfree((void**)&u_g_initial);

}
