/*!\file ElementProperties.c
 * \brief: implementation of the ElementProperties object
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include "./ElementProperties.h"
#include <string.h>
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../shared/shared.h"
#include "../DataSet/DataSet.h"
#include "../include/typedefs.h"
#include "../include/macros.h"

/*Object constructors and destructor*/
/*FUNCTION ElementProperties default constructor {{{1*/
ElementProperties::ElementProperties(){
	
	int i;

	/*initialize to 0 every value: */
	this->numnodes=0;
	this->h=NULL;
	this->s=NULL;
	this->b=NULL;
	this->k=NULL;
	this->melting=NULL;
	this->accumulation=NULL;
	this->geothermalflux=NULL;
	
	this->friction_type=0;
	this->p=0;
	this->q=0;
	this->shelf=0;
	this->onbed=0;
	this->onwater=0;
	this->onsurface=0;
	this->collapse=0;
	this->thermal_steadystate=0;
	return;
}
/*}}}*/
/*FUNCTION ElementProperties constructor {{{1*/

ElementProperties::ElementProperties(int elementproperties_numnodes, double* elementproperties_h,double* elementproperties_s,double* elementproperties_b,
		double* elementproperties_k,double* elementproperties_melting,double* elementproperties_accumulation,
		double* elementproperties_geothermalflux, int elementproperties_friction_type,double elementproperties_p,
		double elementproperties_q, int elementproperties_shelf, int elementproperties_onbed, bool elementproperties_onwater, 
		int elementproperties_onsurface, int elementproperties_collapse, int elementproperties_thermal_steadystate){

	this->Init(elementproperties_numnodes, elementproperties_h,elementproperties_s,elementproperties_b,
		elementproperties_k,elementproperties_melting,elementproperties_accumulation,
		elementproperties_geothermalflux, elementproperties_friction_type,elementproperties_p,
		elementproperties_q, elementproperties_shelf, elementproperties_onbed, elementproperties_onwater,
		elementproperties_onsurface, elementproperties_collapse, elementproperties_thermal_steadystate);
}
/*}}}*/
/*FUNCTION ElementProperties Initialize propreties, used by constructor{{{1*/
void ElementProperties::Init(int elementproperties_numnodes, double* elementproperties_h,double* elementproperties_s,double* elementproperties_b,
		double* elementproperties_k,double* elementproperties_melting,double* elementproperties_accumulation,
		double* elementproperties_geothermalflux, int elementproperties_friction_type,double elementproperties_p,
		double elementproperties_q, int elementproperties_shelf, int elementproperties_onbed, bool elementproperties_onwater, 
		int elementproperties_onsurface, int elementproperties_collapse, int elementproperties_thermal_steadystate){

	int i;

	this->numnodes=elementproperties_numnodes;

	if(elementproperties_h)this->h=(double*)xmalloc(this->numnodes*sizeof(double));
	else this->h=NULL;
	if(elementproperties_s)this->s=(double*)xmalloc(this->numnodes*sizeof(double));
	else this->s=NULL;
	if(elementproperties_b)this->b=(double*)xmalloc(this->numnodes*sizeof(double));
	else this->b=NULL;
	if(elementproperties_k)this->k=(double*)xmalloc(this->numnodes*sizeof(double));
	else this->k=NULL;
	if(elementproperties_melting)this->melting=(double*)xmalloc(this->numnodes*sizeof(double));
	else this->melting=NULL;
	if(elementproperties_accumulation)this->accumulation=(double*)xmalloc(this->numnodes*sizeof(double));
	else this->accumulation=NULL;
	if(elementproperties_geothermalflux)this->geothermalflux=(double*)xmalloc(this->numnodes*sizeof(double));
	else this->geothermalflux=NULL;

	for(i=0;i<this->numnodes;i++){
		if(elementproperties_h)this->h[i]=elementproperties_h[i];
		if(elementproperties_s)this->s[i]=elementproperties_s[i];
		if(elementproperties_b)this->b[i]=elementproperties_b[i];
		if(elementproperties_k)this->k[i]=elementproperties_k[i];
		if(elementproperties_melting)this->melting[i]=elementproperties_melting[i];
		if(elementproperties_accumulation)this->accumulation[i]=elementproperties_accumulation[i];
		if(elementproperties_geothermalflux)this->geothermalflux[i]=elementproperties_geothermalflux[i];
	}
	
	this->friction_type=elementproperties_friction_type;
	this->p=elementproperties_p;
	this->q=elementproperties_q;
	this->shelf=elementproperties_shelf;
	this->onbed=elementproperties_onbed;
	this->onwater=elementproperties_onwater;
	this->onsurface=elementproperties_onsurface;
	this->collapse=elementproperties_collapse;
	this->thermal_steadystate=elementproperties_thermal_steadystate;

	return;
}
/*}}}*/
/*FUNCTION ElementProperties destructor {{{1*/
ElementProperties::~ElementProperties(){ //destructor

	if(this->h)xfree((void**)&this->h);
	if(this->s)xfree((void**)&this->s);
	if(this->b)xfree((void**)&this->b);
	if(this->k)xfree((void**)&this->k);
	if(this->melting)xfree((void**)&this->melting);
	if(this->accumulation)xfree((void**)&this->accumulation);
	if(this->geothermalflux)xfree((void**)&this->geothermalflux);

	return;
}
/*}}}*/
/*FUNCTION ElementProperties Marshall{{{1*/
void  ElementProperties::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_type=0;
	int   nill=0;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of ElementProperties: */
	enum_type=ElementPropertiesEnum();
	
	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	
	memcpy(marshalled_dataset,&this->numnodes,sizeof(this->numnodes));marshalled_dataset+=sizeof(this->numnodes);

	if(this->h){
		memcpy(marshalled_dataset,&this->numnodes,sizeof(this->numnodes));marshalled_dataset+=sizeof(this->numnodes); 
		memcpy(marshalled_dataset,this->h,this->numnodes*sizeof(double));marshalled_dataset+=this->numnodes*sizeof(double);
	}
	else{ memcpy(marshalled_dataset,&nill,sizeof(nill));marshalled_dataset+=sizeof(nill); }

	if(this->s){
		memcpy(marshalled_dataset,&this->numnodes,sizeof(this->numnodes));marshalled_dataset+=sizeof(this->numnodes); 
		memcpy(marshalled_dataset,this->s,this->numnodes*sizeof(double));marshalled_dataset+=this->numnodes*sizeof(double);
	}
	else{ memcpy(marshalled_dataset,&nill,sizeof(nill));marshalled_dataset+=sizeof(nill); }

	if(this->b){
		memcpy(marshalled_dataset,&this->numnodes,sizeof(this->numnodes));marshalled_dataset+=sizeof(this->numnodes); 
		memcpy(marshalled_dataset,this->b,this->numnodes*sizeof(double));marshalled_dataset+=this->numnodes*sizeof(double);
	}
	else{ memcpy(marshalled_dataset,&nill,sizeof(nill));marshalled_dataset+=sizeof(nill); }

	if(this->k){
		memcpy(marshalled_dataset,&this->numnodes,sizeof(this->numnodes));marshalled_dataset+=sizeof(this->numnodes); 
		memcpy(marshalled_dataset,this->k,this->numnodes*sizeof(double));marshalled_dataset+=this->numnodes*sizeof(double);
	}
	else{ memcpy(marshalled_dataset,&nill,sizeof(nill));marshalled_dataset+=sizeof(nill); }

	if(this->melting){
		memcpy(marshalled_dataset,&this->numnodes,sizeof(this->numnodes));marshalled_dataset+=sizeof(this->numnodes); 
		memcpy(marshalled_dataset,this->melting,this->numnodes*sizeof(double));marshalled_dataset+=this->numnodes*sizeof(double);
	}
	else{ memcpy(marshalled_dataset,&nill,sizeof(nill));marshalled_dataset+=sizeof(nill); }

	if(this->accumulation){
		memcpy(marshalled_dataset,&this->numnodes,sizeof(this->numnodes));marshalled_dataset+=sizeof(this->numnodes); 
		memcpy(marshalled_dataset,this->accumulation,this->numnodes*sizeof(double));marshalled_dataset+=this->numnodes*sizeof(double);
	}
	else{ memcpy(marshalled_dataset,&nill,sizeof(nill));marshalled_dataset+=sizeof(nill); }


	if(this->geothermalflux){
		memcpy(marshalled_dataset,&this->numnodes,sizeof(this->numnodes));marshalled_dataset+=sizeof(this->numnodes); 
		memcpy(marshalled_dataset,this->geothermalflux,this->numnodes*sizeof(double));marshalled_dataset+=this->numnodes*sizeof(double);
	}
	else{ memcpy(marshalled_dataset,&nill,sizeof(nill));marshalled_dataset+=sizeof(nill); }

	memcpy(marshalled_dataset,&friction_type,sizeof(friction_type));marshalled_dataset+=sizeof(friction_type);
	memcpy(marshalled_dataset,&p,sizeof(p));marshalled_dataset+=sizeof(p);
	memcpy(marshalled_dataset,&q,sizeof(q));marshalled_dataset+=sizeof(q);
	memcpy(marshalled_dataset,&shelf,sizeof(shelf));marshalled_dataset+=sizeof(shelf);
	memcpy(marshalled_dataset,&onbed,sizeof(onbed));marshalled_dataset+=sizeof(onbed);
	memcpy(marshalled_dataset,&onwater,sizeof(onwater));marshalled_dataset+=sizeof(onwater);
	memcpy(marshalled_dataset,&onsurface,sizeof(onsurface));marshalled_dataset+=sizeof(onsurface);
	memcpy(marshalled_dataset,&collapse,sizeof(collapse));marshalled_dataset+=sizeof(collapse);
	memcpy(marshalled_dataset,&thermal_steadystate,sizeof(thermal_steadystate));marshalled_dataset+=sizeof(thermal_steadystate);

	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION ElementProperties MarshallSize{{{1*/
int ElementProperties::MarshallSize(){

	int size;

	size=sizeof(numnodes);

	if(this->h)size+=sizeof(int)+numnodes*sizeof(double); else size+=sizeof(int);
	if(this->s)size+=sizeof(int)+numnodes*sizeof(double); else size+=sizeof(int);
	if(this->b)size+=sizeof(int)+numnodes*sizeof(double); else size+=sizeof(int);
	if(this->k)size+=sizeof(int)+numnodes*sizeof(double); else size+=sizeof(int);
	if(this->melting)size+=sizeof(int)+numnodes*sizeof(double); else size+=sizeof(int);
	if(this->accumulation)size+=sizeof(int)+numnodes*sizeof(double); else size+=sizeof(int);
	if(this->geothermalflux)size+=sizeof(int)+numnodes*sizeof(double); else size+=sizeof(int);

	size+= sizeof(friction_type)+
		sizeof(p)+
		sizeof(q)+
		sizeof(shelf)+
		sizeof(onbed)+
		sizeof(onwater)+
		sizeof(onsurface)+
		sizeof(collapse)+
		sizeof(thermal_steadystate)+
		sizeof(int); //sizeof(int) for enum type

	return size;

}
/*}}}*/
/*FUNCTION ElementProperties Demarshall{{{1*/
void  ElementProperties::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   nonnill=0;
	int   enum_type;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/* Recover enum_type as ElementProperties is not directly called by dataset */
	memcpy(&enum_type,marshalled_dataset,sizeof(int)); marshalled_dataset+=sizeof(int);

	memcpy(&numnodes,marshalled_dataset,sizeof(numnodes));marshalled_dataset+=sizeof(numnodes);
	
	memcpy(&nonnill,marshalled_dataset,sizeof(nonnill));marshalled_dataset+=sizeof(nonnill);
	if(nonnill){
		this->h=(double*)xmalloc(this->numnodes*sizeof(double));
		memcpy(this->h,marshalled_dataset,this->numnodes*sizeof(double));marshalled_dataset+=this->numnodes*sizeof(double);
	}

	memcpy(&nonnill,marshalled_dataset,sizeof(nonnill));marshalled_dataset+=sizeof(nonnill);
	if(nonnill){
		this->s=(double*)xmalloc(this->numnodes*sizeof(double));
		memcpy(this->s,marshalled_dataset,this->numnodes*sizeof(double));marshalled_dataset+=this->numnodes*sizeof(double);
	}
	
	memcpy(&nonnill,marshalled_dataset,sizeof(nonnill));marshalled_dataset+=sizeof(nonnill);
	if(nonnill){
		this->b=(double*)xmalloc(this->numnodes*sizeof(double));
		memcpy(this->b,marshalled_dataset,this->numnodes*sizeof(double));marshalled_dataset+=this->numnodes*sizeof(double);
	}	

	memcpy(&nonnill,marshalled_dataset,sizeof(nonnill));marshalled_dataset+=sizeof(nonnill);
	if(nonnill){
		this->k=(double*)xmalloc(this->numnodes*sizeof(double));
		memcpy(this->k,marshalled_dataset,this->numnodes*sizeof(double));marshalled_dataset+=this->numnodes*sizeof(double);
	}
	
	memcpy(&nonnill,marshalled_dataset,sizeof(nonnill));marshalled_dataset+=sizeof(nonnill);
	if(nonnill){
		this->melting=(double*)xmalloc(this->numnodes*sizeof(double));
		memcpy(this->melting,marshalled_dataset,this->numnodes*sizeof(double));marshalled_dataset+=this->numnodes*sizeof(double);
	}
	
	memcpy(&nonnill,marshalled_dataset,sizeof(nonnill));marshalled_dataset+=sizeof(nonnill);
	if(nonnill){
		this->accumulation=(double*)xmalloc(this->numnodes*sizeof(double));
		memcpy(this->accumulation,marshalled_dataset,this->numnodes*sizeof(double));marshalled_dataset+=this->numnodes*sizeof(double);
	}
	
	memcpy(&nonnill,marshalled_dataset,sizeof(nonnill));marshalled_dataset+=sizeof(nonnill);
	if(nonnill){
		this->geothermalflux=(double*)xmalloc(this->numnodes*sizeof(double));
		memcpy(this->geothermalflux,marshalled_dataset,this->numnodes*sizeof(double));marshalled_dataset+=this->numnodes*sizeof(double);
	}
	
	
	memcpy(&friction_type,marshalled_dataset,sizeof(friction_type));marshalled_dataset+=sizeof(friction_type);
	memcpy(&p,marshalled_dataset,sizeof(p));marshalled_dataset+=sizeof(p);
	memcpy(&q,marshalled_dataset,sizeof(q));marshalled_dataset+=sizeof(q);
	memcpy(&shelf,marshalled_dataset,sizeof(shelf));marshalled_dataset+=sizeof(shelf);
	memcpy(&onbed,marshalled_dataset,sizeof(onbed));marshalled_dataset+=sizeof(onbed);
	memcpy(&onwater,marshalled_dataset,sizeof(onwater));marshalled_dataset+=sizeof(onwater);
	memcpy(&onsurface,marshalled_dataset,sizeof(onsurface));marshalled_dataset+=sizeof(onsurface);
	memcpy(&collapse,marshalled_dataset,sizeof(collapse));marshalled_dataset+=sizeof(collapse);
	memcpy(&thermal_steadystate,marshalled_dataset,sizeof(thermal_steadystate));marshalled_dataset+=sizeof(thermal_steadystate);

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION ElementProperties Echo{{{1*/
void  ElementProperties::Echo(void){

	int i;
	
	if (h)              {printf("   h=[\n"); for(i=0;i<numnodes;i++)printf("%g ",this->h[i]);printf("]\n");}
	if (s)              {printf("   s=[\n"); for(i=0;i<numnodes;i++)printf("%g ",this->s[i]);printf("]\n");}
	if (b)              {printf("   b=[\n"); for(i=0;i<numnodes;i++)printf("%g ",this->b[i]);printf("]\n");}
	if (k)              {printf("   k=[\n"); for(i=0;i<numnodes;i++)printf("%g ",this->k[i]);printf("]\n");}
	if (melting)        {printf("   melting=[\n"); for(i=0;i<numnodes;i++)printf("%g ",this->melting[i]);printf("]\n");}
	if (accumulation)   {printf("   accumulation=[\n"); for(i=0;i<numnodes;i++)printf("%g ",this->accumulation[i]);printf("]\n");}
	if (geothermalflux) {printf("   geothermalflux=[\n"); for(i=0;i<numnodes;i++)printf("%g ",this->geothermalflux[i]);printf("]\n");}
	printf("   friction_type: %i\n",friction_type);
	printf("   p: %g\n",p);
	printf("   q: %g\n",q);
	printf("   shelf: %i\n",shelf);
	printf("   onbed: %i\n",onbed);
	printf("   onwater: %i\n",onwater);
	printf("   onsurface: %i\n",onsurface);
	printf("   collapse: %i\n",collapse);
	printf("   thermal_steadystate: %i\n",thermal_steadystate);
}
/*}}}*/
/*FUNCTION ElementProperties DeepEcho{{{1*/
void  ElementProperties::DeepEcho(void){
	this->Echo();
}
/*}}}*/
/*FUNCTION ElementProperties Spawn{{{1*/
ElementProperties* ElementProperties::Spawn(int* indices, int numindices){

	int i;
	
	int     sub_numnodes=0;
	double* sub_h=NULL;
	double* sub_s=NULL;
	double* sub_b=NULL;
	double* sub_k=NULL;
	double* sub_melting=NULL;
	double* sub_accumulation=NULL;
	double* sub_geothermalflux=NULL;

	int     sub_friction_type;
	double  sub_p;
	double  sub_q;
	int     sub_shelf;
	int     sub_onbed;
	bool    sub_onwater;
	int     sub_onsurface;
	int     sub_collapse;
	int     sub_thermal_steadystate;

	ElementProperties* sub_properties = NULL;

	sub_numnodes=numindices;

	/*go pickup the correct objects, for the correct ids: */
	if(this->h)sub_h=(double*)xmalloc(numindices*sizeof(double));
	if(this->s)sub_s=(double*)xmalloc(numindices*sizeof(double));
	if(this->b)sub_b=(double*)xmalloc(numindices*sizeof(double));
	if(this->k)sub_k=(double*)xmalloc(numindices*sizeof(double));
	if(this->melting)sub_melting=(double*)xmalloc(numindices*sizeof(double));
	if(this->accumulation)sub_accumulation=(double*)xmalloc(numindices*sizeof(double));
	if(this->geothermalflux)sub_geothermalflux=(double*)xmalloc(numindices*sizeof(double));

	for(i=0;i<numindices;i++){
		if(this->h)sub_h[i]=this->h[indices[i]];
		if(this->s)sub_s[i]=this->s[indices[i]];
		if(this->b)sub_b[i]=this->b[indices[i]];
		if(this->k)sub_k[i]=this->k[indices[i]];
		if(this->melting)sub_melting[i]=this->melting[indices[i]];
		if(this->accumulation)sub_accumulation[i]=this->accumulation[indices[i]];
		if(this->geothermalflux)sub_geothermalflux[i]=this->geothermalflux[indices[i]];
	}

	sub_friction_type=this->friction_type;
	sub_p=this->p;
	sub_q=this->q;
	sub_shelf=this->shelf;
	sub_onbed=this->onbed;
	sub_onwater=this->onwater;
	sub_onsurface=this->onsurface;
	sub_collapse=this->collapse;
	sub_thermal_steadystate=this->thermal_steadystate;

	sub_properties=new ElementProperties(numindices,sub_h, sub_s, sub_b, sub_k, sub_melting, sub_accumulation, sub_geothermalflux, sub_friction_type, sub_p, sub_q, sub_shelf, sub_onbed, sub_onwater, sub_onsurface, sub_collapse, sub_thermal_steadystate);


	xfree((void**)&sub_h);
	xfree((void**)&sub_s);
	xfree((void**)&sub_b);
	xfree((void**)&sub_k);
	xfree((void**)&sub_melting);
	xfree((void**)&sub_accumulation);
	xfree((void**)&sub_geothermalflux);

	return sub_properties;

}
/*}}}*/
