/*!\file Hook.cpp
 * \brief: implementation of the Hook object: see Hook.h for more explanations.
 */


#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "./Object.h"
#include "../DataSet/DataSet.h"
#include "./Hook.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "./ParameterInputs.h"
#include "../shared/shared.h"
#include "../include/typedefs.h"
#include "../include/macros.h"


/*Constructor/Destructors*/
/*FUNCTION Hook::Hook(){{{1*/
Hook::Hook(){
	this->num=0;
	this->objects=NULL;
	this->ids=NULL;
	this->offsets=NULL;
	return;
}
/*}}}*/
/*FUNCTION Hook::Hook(int* ids, int num){{{1*/
Hook::Hook(int* in_ids, int in_num){
	this->Init(in_ids,in_num);
}
/*}}}*/
/*FUNCTION Hook::Init(int* ids, int num){{{1*/
Hook::Init(int* in_ids, int in_num){

	int i;
	this->num=in_num;
	
	/*Allocate: */
	this->objects=(Object**)xmalloc(this->num*sizeof(Object*));
	this->ids=(int*)xmalloc(this->num*sizeof(int));
	this->offsets=(int*)xmalloc(this->num*sizeof(int));

	/*Copy ids: */
	for (i=0;i<this->num;i++){
		this->ids[i]=in_ids[i];
		this->objects[i]=NULL;
		this->offsets[i]=0;
	}
}
/*}}}*/
/*FUNCTION Hook::Hook(Object** hook_objects, int* hook_ids, int* hook_offsets,int hook_num){{{1*/
Hook::Hook(Object** hook_objects, int* hook_ids, int* hook_offsets,int hook_num){

	/*just plug in: */
	this->num=hook_num;
	this->objects=hook_objects;
	this->offsets=hook_offsets;
	this->ids=hook_ids;

}
/*}}}*/
/*FUNCTION Hook::Hook(Hook* input) {{{1*/
Hook::Hook(Hook* input){

	int i;
	Object** input_objects=NULL;
	int*     input_ids=NULL;
	int*     input_offsets=NULL;

	/*get internals of input: */
	this->num=input->GetNum();
	input_objects=input->GetObjects();
	input_ids=input->GetIds();
	input_offsets=input->GetOffsets();

	if(this->num){
		this->objects=(Object**)xmalloc(this->num*sizeof(Object*));
		this->ids=(int*)xmalloc(this->num*sizeof(int));
		this->offsets=(int*)xmalloc(this->num*sizeof(int));
	}
	

	for(i=0;i<this->num;i++){
		this->objects[i]=input_objects[i];
		this->offsets[i]=input_offsets[i];
		this->ids[i]=input_ids[i];
	}
}
/*}}}*/
/*FUNCTION Hook::~Hook(){{{1*/
Hook::~Hook(){
	/*deallocate: */
	xfree((void**)&this->objects);
	xfree((void**)&this->ids);
	xfree((void**)&this->offsets);
	return;
}
/*}}}*/

/*Object marshalling*/
/*FUNCTION Hook::Marshall{{{1*/
void Hook::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_type=0;
	int   i;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of Hook: */
	enum_type=HookEnum();
	
	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	
	/*marshall Hook data: */
	memcpy(marshalled_dataset,&num,sizeof(num));marshalled_dataset+=sizeof(num);
	for(i=0;i<num;i++){
		memcpy(marshalled_dataset,&this->ids[i],sizeof(int));marshalled_dataset+=sizeof(int);
		memcpy(marshalled_dataset,&this->offsets[i],sizeof(int));marshalled_dataset+=sizeof(int);
	}

	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Hook::MarshallSize{{{1*/
int Hook::MarshallSize(){

	return 
		sizeof(num)+
		num*sizeof(int)+
		num*sizeof(int)+
		sizeof(int); //sizeof(int) for enum type
}
/*}}}*/
/*FUNCTION Hook::Demarshall{{{1*/
void Hook::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   i;
	int   enum_type;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of object since Hook is not directly called by DataSet: */
	memcpy(&enum_type,marshalled_dataset,sizeof(int)); marshalled_dataset+=sizeof(int);

	memcpy(&num,marshalled_dataset,sizeof(num));marshalled_dataset+=sizeof(num);
	
	/*allocate: */
	this->ids=(int*)xmalloc(num*sizeof(int));
	this->offsets=(int*)xmalloc(num*sizeof(int));
	
	/*demarshall allocated ids and offsets: */
	for (i=0;i<num;i++){
		memcpy(&this->ids[i],marshalled_dataset,sizeof(int));marshalled_dataset+=sizeof(int);
		memcpy(&this->offsets[i],marshalled_dataset,sizeof(int));marshalled_dataset+=sizeof(int);
	}

	/*nullify object pointers */
	this->objects=(Object**)xmalloc(num*sizeof(Object*));
	for (i=0;i<num;i++){
		this->objects[i]=NULL;
	}

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/

/*Object processing: */
/*FUNCTION Hook::configure{{{1*/
void Hook::configure(DataSet* dataset){

	/*intermediary: */
	Object* object=NULL;
	int i;

	for(i=0;i<this->num;i++){

		/*is this object id -1? If so, drop this search, it was not requested: */
		if (this->ids[i]==-1) continue;

		/*Check whether existing this->objects are correct: */
		if(this->objects[i]){
			if(this->objects[i]->GetId()==this->ids[i]) continue; //this node is good.
			else this->objects[i]=NULL; //this node was incorrect, reset it.
		}

		/*May be the object this->offsets into this->objects are valid?: */
		if(this->offsets[i]!=UNDEF){
			/* Look at the this->offsets[i]'th node in the nodes dataset. If it has the correct id, 
			 * we are good: */
			object=(Object*)dataset->GetObjectByOffset(this->offsets[i]);
			if (object->GetId()==this->ids[i]){
				this->objects[i]=object;
				continue;
			}
			else this->offsets[i]=UNDEF; //object offset was wrong, reset it.
		}

		/*Now, for this->objects that did not get resolved, and for which we have no offset, chase them in the dataset, by id: */
		if(this->objects[i]==NULL){
			this->objects[i]=(Object*)dataset->GetObjectById(this->offsets+i,this->ids[i]); //remember the offset for later on.
			/*check the id is correct!: */
			if (this->objects[i]->GetId()!=this->ids[i]) ISSMERROR(exprintf("%s%i%s%i%s"," wrong id: ",this->objects[i]->GetId()," vs ",this->ids[i],"  in resolved pointer!"));
		}
	}
}
/*}}}*/
/*FUNCTION Hook::Echo{{{1*/
void Hook::Echo(void){

	int i;
	if (num){
		printf("Hook: \n");
		printf("  num=%i\n",this->num);
		printf("  ids:\n    ");
		for (i=0;i<this->num;i++){
			printf("%i ",this->ids[i]);
		}
		printf("\n   offsets:\n   ");
		for (i=0;i<this->num;i++){
			printf("%i ",this->offsets[i]);
		}
	}
	else{
		printf("Hook: num=0 \n");
	}
}
/*}}}*/
/*FUNCTION Hook::DeepEcho{{{1*/
void Hook::DeepEcho(void){

	int i;
	if (num){
		printf("Hook: \n");
		printf("  num=%i\n",this->num);
		printf("  ids:\n    ");
		for (i=0;i<this->num;i++){
			printf("%i ",this->ids[i]);
		}
		printf("\n   offsets:\n   ");
		for (i=0;i<this->num;i++){
			printf("%i ",this->offsets[i]);
		}
		printf("\n   objects:\n   ");
		for (i=0;i<this->num;i++){
			printf("      object %i\n",i);
			objects[i]->Echo();
		}
	}
	else{
		printf("Hook: num=0 \n");
	}
}
/*}}}*/

/*Object functions*/
/*FUNCTION Hook::delivers{{{1*/
Object* Hook::delivers(void){
	
	/*first, check that we only have one T object in our object list: */
	if (this->num!=1) ISSMERROR(exprintf("%s%i%s\n"," trying to delivery a single hook object when hook holds ",this->num," objects"));

	/*check NULL: */
	if (this->objects==NULL) ISSMERROR("hook is not pointing to any object, objects pointer is NULL");

	return *objects;
}

/*}}}*/
/*FUNCTION Hook::deliverp{{{1*/
Object** Hook::deliverp(void){
	return objects;
}
/*}}}*/
/*FUNCTION Hook::GetIds{{{1*/
int* Hook::GetIds(void){
	return this->ids;
}
/*}}}*/
/*FUNCTION Hook::GetNum{{{1*/
int Hook::GetNum(void){
	return this->num;
}
/*}}}*/
/*FUNCTION Hook::GetObjects{{{1*/
Object** Hook::GetObjects(void){
	return this->objects;
}
/*}}}*/
/*FUNCTION Hook::GetOffsets{{{1*/
int* Hook::GetOffsets(void){
	return this->offsets;
}
/*}}}*/
/*FUNCTION Hook::Spawn{{{1*/
Hook* Hook::Spawn(int* indices, int numindices){

	int i;
	Object** subobjects=NULL;
	int       subnum;
	int*      suboffsets=NULL;
	int*      subids=NULL;

	/*go pickup the correct objects, ids and offsets :*/
	subnum=numindices;
	if(subnum<1) ISSMERROR("Trying to spawn an empty ElementProperties!");

	subobjects=(Object**)xmalloc(subnum*sizeof(Object*));
	subids=(int*)xmalloc(subnum*sizeof(int));
	suboffsets=(int*)xmalloc(subnum*sizeof(int));

	for(i=0;i<subnum;i++){
		subobjects[i]=this->objects[indices[i]];
		subids[i]=this->ids[indices[i]];
		suboffsets[i]=this->offsets[indices[i]];
	}

	return new Hook(subobjects,subids,suboffsets,subnum);
}
/*}}}*/
