/*!\file Beam.c
 * \brief: implementation of the Beam object
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include "./Beam.h"
#include <string.h>
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../shared/shared.h"
#include "../DataSet/DataSet.h"
#include "../include/typedefs.h"
#include "../include/macros.h"

/*Object constructors and destructor*/
/*FUNCTION Beam constructor {{{1*/
Beam::Beam(){
	return;
}
/*}}}*/
/*FUNCTION Beam constructor {{{1*/
Beam::Beam(int beam_id,int* beam_node_ids, int beam_matice_id, int beam_matpar_id, int beam_numpar_id, ElementProperties* beamproperties): 
	hnodes(beam_node_ids,2),
	hmatice(&beam_matice_id,1),
	hmatpar(&beam_matpar_id,1),
	hnumpar(&beam_numpar_id,1),
	properties(beamproperties)
{

	/*all the initialization has been done by the initializer, just fill in the id: */
	this->id=beam_id;

	return;
}
/*}}}*/
/*FUNCTION Beam other constructor {{{1*/
Beam::Beam(int beam_id,Hook* beam_hnodes, Hook* beam_hmatice, Hook* beam_hmatpar, Hook* beam_hnumpar, ElementProperties* beam_properties):
	hnodes(beam_hnodes),
	hmatice(beam_hmatice),
	hmatpar(beam_hmatpar),
	hnumpar(beam_hnumpar),
	properties(beam_properties)
{

	/*all the initialization has been done by the initializer, just fill in the id: */
	this->id=beam_id;

	return;
}
/*}}}*/
/*FUNCTION Beam destructor {{{1*/
Beam::~Beam(){
	return;
}
/*}}}*/

/*Object management*/
/*FUNCTION Beam Configure{{{1*/
void  Beam::Configure(void* ploadsin,void* pnodesin,void* pmaterialsin,void* pparametersin){

	int i;
	
	DataSet* loadsin=NULL;
	DataSet* nodesin=NULL;
	DataSet* materialsin=NULL;
	DataSet* parametersin=NULL;

	/*Recover pointers :*/
	loadsin=(DataSet*)ploadsin;
	nodesin=(DataSet*)pnodesin;
	materialsin=(DataSet*)pmaterialsin;
	parametersin=(DataSet*)pparametersin;

	/*Take care of hooking up all objects for this element, ie links the objects in the hooks to their respective 
	 * datasets, using internal ids and offsets hidden in hooks: */
	hnodes.configure(nodesin);
	hmatice.configure(materialsin);
	hmatpar.configure(materialsin);
	hnumpar.configure(parametersin);

}
/*}}}*/
/*FUNCTION Beam copy{{{1*/
Object* Beam::copy() {
	
	return new Beam(this->id,&this->hnodes,&this->hmatice,&this->hmatpar,&this->hnumpar,&this->properties);

}
/*}}}*/
/*FUNCTION Beam DeepEcho{{{1*/
void Beam::DeepEcho(void){

	printf("Beam:\n");
	printf("   id: %i\n",id);
	hnodes.DeepEcho();
	hmatice.DeepEcho();
	hmatpar.DeepEcho();
	hnumpar.DeepEcho();
	properties.DeepEcho();

	return;
}
/*}}}*/
/*FUNCTION Beam Demarshall{{{1*/
void  Beam::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   i;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/
	memcpy(&id,marshalled_dataset,sizeof(id));marshalled_dataset+=sizeof(id);

	/*demarshall hooks: */
	hnodes.Demarshall(&marshalled_dataset);
	hmatice.Demarshall(&marshalled_dataset);
	hmatpar.Demarshall(&marshalled_dataset);
	hnumpar.Demarshall(&marshalled_dataset);

	/*demarshall properties: */
	properties.Demarshall(&marshalled_dataset);

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Beam Echo {{{1*/
void Beam::Echo(void){

	printf("Beam:\n");
	printf("   id: %i\n",id);
	hnodes.Echo();
	hmatice.Echo();
	hmatpar.Echo();
	hnumpar.Echo();
	properties.Echo();

	return;
}
/*}}}*/
/*FUNCTION Beam Marshall{{{1*/
void  Beam::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_type=0;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of Beam: */
	enum_type=BeamEnum();
	
	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	
	/*marshall Beam data: */
	memcpy(marshalled_dataset,&id,sizeof(id));marshalled_dataset+=sizeof(id);

	/*Marshall hooks: */
	hnodes.Marshall(&marshalled_dataset);
	hmatice.Marshall(&marshalled_dataset);
	hmatpar.Marshall(&marshalled_dataset);
	hnumpar.Marshall(&marshalled_dataset);

	/*Marshall properties: */
	properties.Marshall(&marshalled_dataset);

	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Beam MarshallSize{{{1*/
int   Beam::MarshallSize(){
	
	return sizeof(id)
		+hnodes.MarshallSize()
		+hmatice.MarshallSize()
		+hmatpar.MarshallSize()
		+hnumpar.MarshallSize()
		+properties.MarshallSize()
		+sizeof(int); //sizeof(int) for enum type
}
/*}}}*/
/*FUNCTION Beam UpdateFromInputs{{{1*/
void  Beam::UpdateFromInputs(void* vinputs){

	int     dofs[1]={0};
	double  temperature_list[2];
	double  temperature_average;
	double  B_list[2];
	double  B_average;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;
	Numpar* numpar=NULL;

	ParameterInputs* inputs=NULL;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();
	numpar=(Numpar*)hnumpar.delivers();

	/*Update internal data if inputs holds new values: */
	//if (id==1) printf("WARNING if QMU: no hydrostatic equilibrium is applied here (conflict with prognostic, which does not have matpar)\n");
	//For now
	if(this->properties.h) inputs->Recover("thickness",&this->properties.h[0],1,dofs,2,(void**)nodes);
	//Later
	/*
		if(inputs->Recover("thickness",&new_h[0],1,dofs,2,(void**)nodes)){
	//density, needed later:
	double di=(this->matpar->GetRhoIce()/this->matpar->GetRhoWater());
	//Go through grids:
	for (i=0;i<2;i++){
	if(nodes[i]->IsOnShelf()){
	this->b[i]=this->b[i]-di*(new_h[i]-h[i]); //hydrostatic equilibrium;
	}
	this->s[i]=this->b[i]+new_h[i];
	this->h[i]=new_h[i];
	}
	}
	*/
	if (this->properties.s)              inputs->Recover("surface",&this->properties.s[0],1,dofs,2,(void**)nodes);
	if (this->properties.b)              inputs->Recover("bed",&this->properties.b[0],1,dofs,2,(void**)nodes);
	if (this->properties.k)              inputs->Recover("drag",&this->properties.k[0],1,dofs,2,(void**)nodes);
	if (this->properties.melting)        inputs->Recover("melting",&this->properties.melting[0],1,dofs,2,(void**)nodes);
	if (this->properties.accumulation)   inputs->Recover("accumulation",&this->properties.accumulation[0],1,dofs,2,(void**)nodes);
	if (this->properties.geothermalflux) inputs->Recover("geothermalflux",&this->properties.geothermalflux[0],1,dofs,2,(void**)nodes);
	
	//Update material if necessary
	if(inputs->Recover("temperature_average",&temperature_list[0],1,dofs,2,(void**)nodes)){
		temperature_average=(temperature_list[0]+temperature_list[1])/2.0;
		B_average=Paterson(temperature_average);
		matice->SetB(B_average);
	}
	
	if(inputs->Recover("B",&B_list[0],1,dofs,2,(void**)nodes)){
		B_average=(B_list[0]+B_list[1])/2.0;
		matice->SetB(B_average);
	}

}
/*}}}*/

/*Object functions*/
/*FUNCTION Beam ComputePressure{{{1*/
void  Beam::ComputePressure(Vec p_g){

	int i;
	const int numgrids=2;
	int doflist[numgrids];
	double pressure[numgrids];
	double rho_ice,g;
	double xyz_list[numgrids][3];

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;
	Numpar* numpar=NULL;

	
	/*Get dof list on which we will plug the pressure values: */
	GetDofList1(&doflist[0]);

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();
	numpar=(Numpar*)hnumpar.delivers();

	/*Get node data: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
        
	/*Get dof list on which we will plug the pressure values: */
	GetDofList1(&doflist[0]);

	/*pressure is lithostatic: */
	rho_ice=matpar->GetRhoIce();
	g=matpar->GetG();
	for(i=0;i<numgrids;i++){
		pressure[i]=rho_ice*g*(this->properties.s[i]-xyz_list[i][2]);
	}
	
	/*plug local pressure values into global pressure vector: */
	VecSetValues(p_g,numgrids,doflist,(const double*)pressure,INSERT_VALUES);

}
/*}}}*/
/*FUNCTION Beam CostFunction{{{1*/
double Beam::CostFunction(void*,int,int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam CreateKMatrix{{{1*/

void  Beam::CreateKMatrix(Mat Kgg,void* inputs,int analysis_type,int sub_analysis_type){

	/*Just branch to the correct element stiffness matrix generator, according to the type of analysis we are carrying out: */
	if (analysis_type==DiagnosticAnalysisEnum()) {
	
		if (sub_analysis_type==HutterAnalysisEnum()) {

			CreateKMatrixDiagnosticHutter( Kgg,inputs,analysis_type,sub_analysis_type);
		}
		else 
			ISSMERROR(exprintf("%s%i%s\n","sub_analysis_type: ",sub_analysis_type," not supported yet"));
	}
	else{
		ISSMERROR(exprintf("%s%i%s\n","analysis: ",analysis_type," not supported yet"));
	}

}
/*}}}*/
/*FUNCTION Beam CreateKMatrixDiagnosticHutter{{{1*/

void  Beam::CreateKMatrixDiagnosticHutter(Mat Kgg,void* vinputs,int analysis_type,int sub_analysis_type){
	
	
	const int numgrids=2;
	const int NDOF2=2;
	const int numdofs=NDOF2*numgrids;
	int       doflist[numdofs];
	double    Ke_gg[numdofs][numdofs]={{0,0,0,0},{0,0,0,0},{0,0,0,0},{0,0,0,0}};
	int       numberofdofspernode;
	
	GetDofList(&doflist[0],&numberofdofspernode);

	if (this->properties.onbed){
		Ke_gg[0][0]=1;
		Ke_gg[1][1]=1;
		Ke_gg[2][0]=-1;
		Ke_gg[2][2]=1;
		Ke_gg[3][1]=-1;
		Ke_gg[3][3]=1;
	}
	else{
		Ke_gg[2][0]=-1;
		Ke_gg[2][2]=1;
		Ke_gg[3][1]=-1;
		Ke_gg[3][3]=1;
	}

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdofs,doflist,numdofs,doflist,(const double*)Ke_gg,ADD_VALUES);

}
/*}}}*/
/*FUNCTION Beam CreatePVector{{{1*/
void  Beam::CreatePVector(Vec pg,void* inputs,int analysis_type,int sub_analysis_type){
	
	/*Just branch to the correct load generator, according to the type of analysis we are carrying out: */
	if (analysis_type==DiagnosticAnalysisEnum()) {
		if (sub_analysis_type==HutterAnalysisEnum()) {
			CreatePVectorDiagnosticHutter( pg,inputs,analysis_type,sub_analysis_type);
		}
		else
			ISSMERROR(exprintf("%s%i%s"," analysis ",analysis_type," not supported yet"));
	}
	else{
		ISSMERROR(exprintf("%s%i%s"," analysis ",analysis_type," not supported yet"));
	}

}
/*}}}*/
/*FUNCTION Beam CreatePVectorDiagnosticHutter{{{1*/

void Beam::CreatePVectorDiagnosticHutter( Vec pg, void* vinputs, int analysis_type,int sub_analysis_type){

	int i,j,k;
	
	const int numgrids=2;
	const int NDOF2=2;
	const int numdofs=NDOF2*numgrids;
	int       doflist[numdofs];
	double    pe_g[4]={0,0,0,0};
	double    pe_g_gaussian[4]={0,0,0,0};
	int       found=0;
	int       dofs[1]={0};
	double    xyz_list[numgrids][3];
	double    z_list[numgrids];
	double    constant_part;
	int       numberofdofspernode;

	/*gaussian points: */
	int      num_gauss;
	double*  segment_gauss_coord=NULL;
	double   gauss_coord;
	double*  gauss_weights=NULL;
	double   gauss_weight;
	int      ig;
	double   l1l2[2];
	double   slope[2];
	double   slope2;

	double   z_g;
	double   rho_ice,gravity,n,B;
	double   Jdet;
	double   ub,vb;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;
	Numpar* numpar=NULL;

	ParameterInputs* inputs=NULL;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();
	numpar=(Numpar*)hnumpar.delivers();

	/*recover doflist: */
	GetDofList(&doflist[0],&numberofdofspernode);

	/*recover parameters: */
	rho_ice=matpar->GetRhoIce();
	gravity=matpar->GetG();
	n=matice->GetN();
	B=matice->GetB();

	//recover extra inputs
	found=inputs->Recover("surfaceslopex",&slope[0],1,dofs,1,(void**)&nodes[0]); //only recover from fist node, second node will have the same slope
	if(!found)ISSMERROR(" surfaceslopex missing from inputs!");
	
	found=inputs->Recover("surfaceslopey",&slope[1],1,dofs,1,(void**)&nodes[0]);//only recover from fist node, second node will have the same slope
	if(!found)ISSMERROR(" surfaceslopey missing from inputs!");

	//Get all element grid data:
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	for(i=0;i<numgrids;i++)z_list[i]=xyz_list[i][2];
	
	//compute slope2 slopex and slopy
	slope2=pow(slope[0],2)+pow(slope[1],2);

	//%compute constant_part
	constant_part=-2*pow(rho_ice*gravity,n)*pow(slope2,((n-1)/2));

	//Get gaussian points and weights. order 2 since we have a product of 2 nodal functions
	num_gauss=3;
	GaussSegment(&segment_gauss_coord, &gauss_weights, num_gauss);

	//Start  looping on the number of gaussian points:
	for(ig=0;ig<num_gauss;ig++){

		//Pick up the gaussian point and its weight:
		gauss_weight=gauss_weights[ig];
		gauss_coord=segment_gauss_coord[ig];

		//compute constant_part
		GetParameterValue(&z_g, &z_list[0],gauss_coord);

		//Get Jacobian determinant:
		GetJacobianDeterminant(&Jdet, &z_list[0],gauss_coord);
		
		for(j=0;j<NDOF2;j++){
			pe_g_gaussian[NDOF2+j]=constant_part*pow((this->properties.s[0]-z_g)/B,n)*slope[j]*Jdet*gauss_weight;
		}
		
		//add pe_gaussian vector to pe:
		for(j=0;j<numdofs;j++){
			pe_g[j]+=pe_g_gaussian[j];
		}
	} //for(ig=0;ig<num_gauss;ig++)

	//Deal with lower surface
	if (this->properties.onbed){

		//compute ub
		constant_part=-1.58*pow((double)10.0,-(double)10.0)*rho_ice*gravity*this->properties.h[0];
		ub=constant_part*slope[0];
		vb=constant_part*slope[1];

		//Add to pe: 
		pe_g[0]+=ub;
		pe_g[1]+=vb;
	}

	/*Add pe_g to global vector pg: */
	VecSetValues(pg,numdofs,doflist,(const double*)pe_g,ADD_VALUES);

	cleanup_and_return: 
	xfree((void**)&segment_gauss_coord);
	xfree((void**)&gauss_weights);
}
/*}}}*/
/*FUNCTION Beam Du{{{1*/
void  Beam::Du(_p_Vec*,void*,int,int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam Enum{{{1*/
int Beam::Enum(void){

	return BeamEnum();

}
/*}}}*/
/*FUNCTION Beam GetBedList{{{1*/
void  Beam::GetBedList(double*){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam GetDofList{{{1*/
void  Beam::GetDofList(int* doflist,int* pnumberofdofspernode){

	int i,j;
	int doflist_per_node[MAXDOFSPERNODE];
	int numberofdofspernode;
	
	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	
	for(i=0;i<2;i++){
		nodes[i]->GetDofList(&doflist_per_node[0],&numberofdofspernode);
		for(j=0;j<numberofdofspernode;j++){
			doflist[i*numberofdofspernode+j]=doflist_per_node[j];
		}
	}

	/*Assign output pointers:*/
	*pnumberofdofspernode=numberofdofspernode;

}
/*}}}*/
/*FUNCTION Beam GetDofList1{{{1*/
void  Beam::GetDofList1(int* doflist){

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	
	int i;
	for(i=0;i<2;i++){
		doflist[i]=nodes[i]->GetDofList1();
	}

}
/*}}}*/
/*FUNCTION Beam GetId{{{1*/
int    Beam::GetId(void){ return id; }
/*}}}*/
/*FUNCTION Beam GetJacobianDeterminant{{{1*/
void Beam::GetJacobianDeterminant(double* pJdet,double* z_list, double gauss_coord){


	double Jdet;

	Jdet=1.0/2.0*(z_list[1]-z_list[0]);

	if(Jdet<0){
		ISSMERROR(" negative jacobian determinant!");
	}
	
	*pJdet=Jdet;
}
/*}}}*/
/*FUNCTION Beam GetMatPar{{{1*/
void* Beam::GetMatPar(){

	/*dynamic objects pointed to by hooks: */
	Matpar* matpar=NULL;

	/*recover objects from hooks: */
	matpar=(Matpar*)hmatpar.delivers();

	return matpar;
}
/*}}}*/
/*FUNCTION Beam GetName{{{1*/
char* Beam::GetName(void){
	return "beam";
}
/*}}}*/
/*FUNCTION Beam GetNodalFunctions{{{1*/
void Beam::GetNodalFunctions(double* l1l2, double gauss_coord){
	
	/*This routine returns the values of the nodal functions  at the gaussian point.*/

	/*First nodal function: */
	l1l2[0]=.5*gauss_coord+.5;

	/*Second nodal function: */
	l1l2[1]=-.5*gauss_coord+.5;
}
/*}}}*/
/*FUNCTION Beam GetNodes{{{1*/
void  Beam::GetNodes(void** vpnodes){
	int i;
	Node** pnodes=NULL;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	
	/*recover nodes: */
	pnodes=(Node**)vpnodes;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	
	for(i=0;i<3;i++){
		pnodes[i]=nodes[i];
	}
}
/*}}}*/
/*FUNCTION Beam GetOnBed{{{1*/
int   Beam::GetOnBed(){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam GetParameterValue{{{1*/
void Beam::GetParameterValue(double* pvalue, double* value_list,double gauss_coord){

	double l1l2[2];
	
	GetNodalFunctions(&l1l2[0],gauss_coord);

	*pvalue=l1l2[0]*value_list[0]+l1l2[1]*value_list[1];
}
/*}}}*/
/*FUNCTION Beam GetShelf{{{1*/
int   Beam::GetShelf(){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam GetThicknessList{{{1*/
void  Beam::GetThicknessList(double* thickness_list){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam Gradj{{{1*/
void  Beam::Gradj(_p_Vec*, void*, int, int,char*){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam GradjB{{{1*/
void  Beam::GradjB(_p_Vec*, void*, int, int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam GradjDrag{{{1*/
void  Beam::GradjDrag(_p_Vec*, void*, int,int ){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam MassFlux{{{1*/
double Beam::MassFlux( double* segment,double* ug){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam MaticeConfiguration{{{1*/
void  Beam::MaticeConfiguration(Matice* beam_matice,int beam_matice_offset){

	/*dynamic objects pointed to by hooks: */
	Matice* matice=NULL;

	/*recover objects from hooks: */
	matice=(Matice*)hmatice.delivers();
	
	ISSMERROR("not supported yet!");

	/*matice=beam_matice;
	matice_offset=beam_matice_offset;*/
}
/*}}}*/
/*FUNCTION Beam MatparConfiguration{{{1*/
void  Beam::MatparConfiguration(Matpar* beam_matpar,int beam_matpar_offset){

	/*dynamic objects pointed to by hooks: */
	Matpar* matpar=NULL;

	/*recover objects from hooks: */
	matpar=(Matpar*)hmatpar.delivers();

	ISSMERROR("not supported yet!");

	/*matpar=beam_matpar;
	matpar_offset=beam_matpar_offset;*/

}
/*}}}*/
/*FUNCTION Beam Misfit{{{1*/
double Beam::Misfit(void*,int,int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Beam MyRank{{{1*/
int    Beam::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}*/
/*FUNCTION Beam NodeConfiguration{{{1*/
void  Beam::NodeConfiguration(int* beam_node_ids,Node* beam_nodes[2],int* beam_node_offsets){

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();

	ISSMERROR("not supported yet!");

	/*int i;
	for(i=0;i<2;i++){
		node_ids[i]=beam_node_ids[i];
		nodes[i]=beam_nodes[i];
		node_offsets[i]=beam_node_offsets[i];
	}*/

}
/*}}}*/
/*FUNCTION Beam SurfaceArea{{{1*/
double Beam::SurfaceArea(void*,int,int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
