/*!\file ParameterInputs.c
 * \brief: implementation of the ParameterInputs object
 */


#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../shared/shared.h"
#include "../include/typedefs.h"
#include "../include/macros.h"
#include "./objects.h"

/*Object constructors and destructor*/
/*FUNCTION ParameterInputs::constructor {{{1*/
ParameterInputs::ParameterInputs(){

	dataset=new DataSet();

}
/*}}}*/
/*FUNCTION ParameterInputs::destructor {{{1*/
ParameterInputs::~ParameterInputs(){

	delete dataset;

}
/*}}}*/
/*FUNCTION ParameterInputs::purge {{{1*/
void ParameterInputs::purge(char* name){

	int i;
	Input* input=NULL;

	/*Go through dataset, and figure out if an Input 
	 * already has the name "name". If so, delete it : */
	
	for(i=0;i<dataset->Size();i++){
		input=(Input*)dataset->GetObjectByOffset(i);

		if (input->IsSameName(name)){
			/*delete object: */
			dataset->DeleteObject(input);
		}
	}
}
/*}}}*/

/*Object functions*/
/*FUNCTION ParameterInputs::Add(char* name,char* string) {{{1*/
void ParameterInputs::Add(char* name,char* string){

	Input* input=NULL;

	/*First, purge object with same name: */
	this->purge(name);

	/*We are sure an input of the same name does not exist. Create new 
	 * input: */
	input=new Input(name,string);

	/*Add input to dataset: */
	dataset->AddObject(input);

}
/*}}}*/
/*FUNCTION ParameterInputs::Add(char* name,int integer) {{{1*/
void ParameterInputs::Add(char* name,int integer){
	
	Input* input=NULL;

	/*First, purge object with same name: */
	this->purge(name);

	/*We are sure an input of the same name does not exist. Create new 
	 * input: */
	input=new Input(name,integer);

	/*Add input to dataset: */
	dataset->AddObject(input);
}
/*}}}*/
/*FUNCTION ParameterInputs::Add(char* name,double scalar) {{{1*/
void ParameterInputs::Add(char* name,double scalar){

	Input* input=NULL;

	/*First, purge object with same name: */
	this->purge(name);

	/*We are sure an input of the same name does not exist. Create new 
	 * input: */
	input=new Input(name,scalar);

	/*Add input to dataset: */
	dataset->AddObject(input);

}
/*}}}*/
/*FUNCTION ParameterInputs::Add(char* name,double* vector,int ndof,int numberofnodes) {{{1*/
void ParameterInputs::Add(char* name,double* vector,int ndof,int numberofnodes){
	
	Input* input=NULL;

	/*First, purge object with same name: */
	this->purge(name);

	/*We are sure an input of the same name does not exist. Create new 
	 * input: */
	input=new Input(name,vector,ndof,numberofnodes);
	
	/*Add input to dataset: */
	dataset->AddObject(input);

}
/*}}}*/
/*FUNCTION ParameterInputs::Add(char* name,Vec petscvector,int ndof, int numberofnodes) {{{1*/
void ParameterInputs::Add(char* name,Vec petscvector,int ndof, int numberofnodes){
	
	Input* input=NULL;

	/*First, purge object with same name: */
	this->purge(name);

	/*We are sure an input of the same name does not exist. Create new 
	 * input: */
	input=new Input(name,petscvector,ndof,numberofnodes);

	/*Add input to dataset: */
	dataset->AddObject(input);

}
/*}}}*/
/*FUNCTION ParameterInputs::DeepEcho {{{1*/
void ParameterInputs::DeepEcho(){

	printf("ParameterInputs echo: \n");
	dataset->DeepEcho();

}
/*}}}*/
/*FUNCTION ParameterInputs::Echo {{{1*/
void ParameterInputs::Echo(){

	printf("ParameterInputs echo: \n");
	dataset->Echo();

}
/*}}}*/
/*FUNCTION ParameterInputs::Get {{{1*/
Vec ParameterInputs::Get(char* name,int* dofs, int numdofs){
	
	int i;
	Input* input=NULL;
	int found=0;
	
	/*output: */
	Vec ug=NULL;

	/*Go through dataset, and figure out if an Input 
	 * has the name "name": */
	for(i=0;i<dataset->Size();i++){
		input=(Input*)dataset->GetObjectByOffset(i);

		if (input->IsSameName(name)){
			found=1;
			break;
		}
	}

	if(found==0)return NULL;
	
	/*call submethod: */
	ug=input->Get(dofs,numdofs);

	return ug;
}
/*}}}*/
/*FUNCTION ParameterInputs::IsPresent(char* name,char* string) {{{1*/
int ParameterInputs::IsPresent(char* name){

	/*Intermediary*/
	int i;
	Input* input=NULL;

	/*Go through dataset, and figure out if the input is present*/
	for(i=0;i<dataset->Size();i++){
		input=(Input*)dataset->GetObjectByOffset(i);
		if (input->IsPresent(name)) return 1;
	}

	/*Input not found... return 0*/
	return 0;
}
/*}}}*/
/*FUNCTION ParameterInputs::Recover(char* name, char** pstring) {{{1*/
int ParameterInputs::Recover(char* name, char** pstring){

	int found=0;
	int i;
	Input* input=NULL;
	char* string=NULL;

	/*Go through dataset, and figure out if an Input 
	 * has the name "name": */
	for(i=0;i<dataset->Size();i++){
		input=(Input*)dataset->GetObjectByOffset(i);

		if (input->IsSameName(name)){

			/*Get string out of this input: */
			input->Recover(&string);
			found=1;break;
		}
	}
	/*Assign output pointers:*/
	*pstring=string;
	return found;

}
/*}}}*/
/*FUNCTION ParameterInputs::Recover(char* name, int* pinteger) {{{1*/
int ParameterInputs::Recover(char* name, int* pinteger){
	
	int found=0;
	int i;
	Input* input=NULL;
	int integer;

	/*Go through dataset, and figure out if an Input 
	 * has the name "name": */
	for(i=0;i<dataset->Size();i++){
		input=(Input*)dataset->GetObjectByOffset(i);

		if (input->IsSameName(name)){

			/*Get string out of this input: */
			input->Recover(&integer);
			found=1; break;

		}
	}
	/*Assign output pointers:*/
	*pinteger=integer;

	return found;
}
/*}}}*/
/*FUNCTION ParameterInputs::Recover(char* name, double* pscalar) {{{1*/
int ParameterInputs::Recover(char* name, double* pscalar){

	int found=0;
	int i;
	Input* input=NULL;
	double scalar;

	/*Go through dataset, and figure out if an Input 
	 * has the name "name": */
	for(i=0;i<dataset->Size();i++){
		input=(Input*)dataset->GetObjectByOffset(i);

		if (input->IsSameName(name)){

			/*Get string out of this input: */
			input->Recover(&scalar);
			found=1; break;

		}
	}
	/*Assign output pointers:*/
	*pscalar=scalar;

	return found;

}
/*}}}*/
/*FUNCTION ParameterInputs::Recover(char* name,double* values, int ndof, int* dofs,int numnodes,void** pnodes) {{{1*/
int ParameterInputs::Recover(char* name,double* values, int ndof, int* dofs,int numnodes,void** pnodes){

	int i;
	Node** nodes=NULL;
	Input* input=NULL;
	int found=0;

	/*First recover nodes pointer :*/
	nodes=(Node**)pnodes;

	/*Go through dataset, and figure out if an Input 
	 * has the name "name": */
	for(i=0;i<dataset->Size();i++){
		input=(Input*)dataset->GetObjectByOffset(i);

		if (input->IsSameName(name)){
			found=1;
			break;
		}
	}

	if(found==0)return 0;
	
	/*call submethod: */
	input->Recover(values,ndof,dofs,numnodes,(void**)nodes);
	return 1;
}
/*}}}*/
/*FUNCTION ParameterInputs::UpdateFromDakota {{{1*/
void ParameterInputs::UpdateFromDakota(double* variables,char** variables_descriptors,int numvariables,DataSet* parameters,double* qmu_part,int qmu_npart){

	/*Go through all dakota descriptors, ex: "rho_ice","thermal_conductivity","thickness1","thickness2", etc ..., and 
	 * for each descriptor, take the variable value and plug it into the inputs: */

	int     i,j,k;
	int     found;

	double* distributed_values=NULL;
	char*   root=NULL; //root name of distributed variable, ex: thickness, drag, etc ...
	double* parameter=NULL;
	int     numberofnodes;

	char*   descriptor=NULL;

	/*retrive some necessary parameters first: */
	found=parameters->FindParam(&numberofnodes,"numberofnodes");
	if(!found)ISSMERROR("coud not find numberofnodes in parameters dataset!");

	for(i=0;i<numvariables;i++){
	
		descriptor=variables_descriptors[i];

		/*From descriptor, figure out if the variable is distributed (distributed implies there is a numeric value at the 
		 * end of the descriptor, for ex: thickness1, thickness10, etc .... If it is distributed, the next qmu_npart (number 
		 * of partitions in the distributed variable) variable are the values for each partition of the distributed variable: */
		if (!isdistributed(&root,descriptor)){
			
			/*Ok, variable is not distributed, just add it to inputs: */
			this->Add(descriptor,variables[i]);
		}
		else{
			
			/*Ok, variable is distributed. Root name of variable is also known. Now, allocate distributed_values and fill the 
			 * distributed_values with the next qmu_npart variables: */

			distributed_values=(double*)xmalloc(qmu_npart*sizeof(double));
			for(j=0;j<qmu_npart;j++){
				distributed_values[j]=variables[i+j];
			}

			/*Now, pick up the parameter corresponding to root: */
			found=parameters->FindParam(&parameter,NULL,NULL,root);
			if(!found)ISSMERROR(exprintf("%s%s%s"," could not find parameter ",root," for Dakota input update"));
			
			/*We've got the parameter, we need to update it using qmu_part (a partitioning vector), and the distributed_values: */
			for(k=0;k<numberofnodes;k++){
				parameter[k]=parameter[k]*distributed_values[(int)qmu_part[k]];
			}

			#ifdef _ISSM_DEBUG_
				PetscSynchronizedPrintf(MPI_COMM_WORLD,"Parameter vetor:");
				PetscSynchronizedFlush(MPI_COMM_WORLD);
				for(k=0;k<numberofnodes;k++){
					PetscSynchronizedPrintf(MPI_COMM_WORLD," node %i value %g\n",k+1,parameter[k]);
					PetscSynchronizedFlush(MPI_COMM_WORLD);
				}
			#endif
			  

			/*Add parameter to inputs: */
			this->Add(root,parameter,1,numberofnodes);

			/*increment i to skip the distributed values just collected: */
			i+=qmu_npart-1; //careful, the for loop will add 1.

			/*Free allocations: */
			xfree((void**)&parameter);
			xfree((void**)&distributed_values);
			xfree((void**)&root);

		}
	}

	ISSMERROR("create UpdateFromDakota routine, to call UpdateFromDakota in the elements, so it does not conflict
			with UpdateFromInputs! in the elements");


	/*Free ressources:*/
	xfree((void**)&distributed_values);
	xfree((void**)&root);
	xfree((void**)&parameter);

}
/*}}}*/
