/*!\file Numericalflux.c
 * \brief: implementation of the Numericalflux object
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../shared/shared.h"
#include "../include/typedefs.h"
#include "../include/macros.h"
#include "./objects.h"

/*Object constructors and destructor*/
/*FUNCTION Numericalflux::Numericalflux(){{{1*/
Numericalflux::Numericalflux(){
	return;
}
/*}}}*/
/*FUNCTION Numericalflux::Numericalflux(char numericalflux_type[NUMERICALFLUXSTRING],int numericalflux_fill...){{{1*/
Numericalflux::Numericalflux(int numericalflux_id,char numericalflux_type[NUMERICALFLUXSTRING], int numericalflux_node_ids[MAX_NUMERICALFLUX_NODES],int numericalflux_element_ids[MAX_NUMERICALFLUX_ELEMS]){

	int i;
	
	strcpy(type,numericalflux_type);
	id=numericalflux_id;
	
	for(i=0;i<MAX_NUMERICALFLUX_ELEMS;i++){
		element_ids[i]=numericalflux_element_ids[i];
		element_offsets[i]=UNDEF;
		elements[i]=NULL;
	}

	for(i=0;i<MAX_NUMERICALFLUX_NODES;i++){
		node_ids[i]=numericalflux_node_ids[i];
		node_offsets[i]=UNDEF;
		nodes[i]=NULL;
	}

	return;
}
/*}}}*/
/*FUNCTION Numericalflux::~Numericalflux(){{{1*/
Numericalflux::~Numericalflux(){
	return;
}
/*}}}*/
/*FUNCTION Numericalflux::copy {{{1*/
Object* Numericalflux::copy() {
	return new Numericalflux(*this); 
}
/*}}}*/

/*Object marshall*/
/*FUNCTION Numericalflux::Demarshall {{{1*/
void  Numericalflux::Demarshall(char** pmarshalled_dataset){

	int i;
	char* marshalled_dataset=NULL;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/

	memcpy(&type,marshalled_dataset,sizeof(type));marshalled_dataset+=sizeof(type);
	memcpy(&id,marshalled_dataset,sizeof(id));marshalled_dataset+=sizeof(id);

	memcpy(&element_ids,marshalled_dataset,sizeof(element_ids));marshalled_dataset+=sizeof(element_ids);
	memcpy(&element_offsets,marshalled_dataset,sizeof(element_offsets));marshalled_dataset+=sizeof(element_offsets);
	for(i=0;i<MAX_NUMERICALFLUX_ELEMS;i++)elements[i]=NULL;
	
	memcpy(&node_ids,marshalled_dataset,sizeof(node_ids));marshalled_dataset+=sizeof(node_ids);
	memcpy(&node_offsets,marshalled_dataset,sizeof(node_offsets));marshalled_dataset+=sizeof(node_offsets);
	for(i=0;i<MAX_NUMERICALFLUX_NODES;i++)nodes[i]=NULL;

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Numericalflux::Marshall {{{1*/
void  Numericalflux::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_type=0;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of Numericalflux: */
	enum_type=NumericalfluxEnum();
	
	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	
	/*marshall Numericalflux data: */
	memcpy(marshalled_dataset,&type,sizeof(type));marshalled_dataset+=sizeof(type);
	memcpy(marshalled_dataset,&id,sizeof(id));marshalled_dataset+=sizeof(id);
	
	memcpy(marshalled_dataset,&element_ids,sizeof(element_ids));marshalled_dataset+=sizeof(element_ids);
	memcpy(marshalled_dataset,&element_offsets,sizeof(element_offsets));marshalled_dataset+=sizeof(element_offsets);

	memcpy(marshalled_dataset,&node_ids,sizeof(node_ids));marshalled_dataset+=sizeof(node_ids);
	memcpy(marshalled_dataset,&node_offsets,sizeof(node_offsets));marshalled_dataset+=sizeof(node_offsets);

	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Numericalflux::MarshallSize{{{1*/
int   Numericalflux::MarshallSize(){

	return sizeof(type)+
		sizeof(id)+
		sizeof(element_ids)+
		sizeof(element_offsets)+
		sizeof(node_ids)+
		sizeof(node_offsets)+
		sizeof(int); //sizeof(int) for enum type
}
/*}}}*/

/*Object functions*/
/*FUNCTION Numericalflux::Configure {{{1*/
void  Numericalflux::Configure(void* pelementsin,void* pnodesin,void* pmaterialsin){

	DataSet* elementsin=NULL;
	DataSet* nodesin=NULL;

	/*Recover pointers :*/
	elementsin=(DataSet*)pelementsin;
	nodesin=(DataSet*)pnodesin;
	
	/*Link this load with its nodes: */
	if (strcmp(type,"internal")==0){
		ResolvePointers((Object**)nodes,node_ids,node_offsets,4,nodesin);
		ResolvePointers((Object**)elements,element_ids,element_offsets,2,elementsin);
	}
	else if (strcmp(type,"boundary")==0){
		ResolvePointers((Object**)nodes,node_ids,node_offsets,2,nodesin);
		ResolvePointers((Object**)elements,element_ids,element_offsets,1,elementsin);
	}
	else ISSMERROR("type not supported yet");

}
/*}}}*/
/*FUNCTION Numericalflux::CreateKMatrix {{{1*/
void  Numericalflux::CreateKMatrix(Mat Kgg,void* inputs,int analysis_type,int sub_analysis_type){

	/*No stiffness loads applied, do nothing: */
	return;

}
/*}}}*/
/*FUNCTION Numericalflux::CreatePVector {{{1*/
void  Numericalflux::CreatePVector(Vec pg, void* inputs, int analysis_type,int sub_analysis_type){

	/*No stiffness loads applied, do nothing: */
	return;
}
/*}}}*/
/*FUNCTION Numericalflux::DeepEcho {{{1*/
void Numericalflux::DeepEcho(void){

	int i;
	
	printf("Numericalflux:\n");
	printf("   type: %s\n",type);
	printf("   id: %i\n",id);
	
	if (strcmp(type,"internal")==0){
		printf("   element_ids=[%i,%i]\n",element_ids[0],element_ids[1]);
		printf("   element_offsets=[%i,%i]\n",element_offsets[0],element_offsets[1]);
		for(i=0;i<2;i++){
			if(elements[i])elements[i]->Echo();
		}
		printf("   node_ids=[%i,%i,%i,%i]\n",node_ids[0],node_ids[1],node_ids[2],node_ids[3]);
		printf("   node_offsets=[%i,%i,%i,%i]\n",node_offsets[0],node_offsets[1],node_offsets[2],node_offsets[3]);
		for(i=0;i<4;i++){
			if(nodes[i])nodes[i]->Echo();
		}
	}
	else{
		printf("   element_ids=[%i,%i]\n",element_ids[0],element_ids[1]);
		printf("   element_offsets=[%i,%i]\n",element_offsets[0],element_offsets[1]);
		for(i=0;i<1;i++){
			if(elements[i])elements[i]->Echo();
		}
		printf("   node_ids=[%i,%i,%i,%i]\n",node_ids[0],node_ids[1],node_ids[2],node_ids[3]);
		printf("   node_offsets=[%i,%i,%i,%i]\n",node_offsets[0],node_offsets[1],node_offsets[2],node_offsets[3]);
		for(i=0;i<2;i++){
			if(nodes[i])nodes[i]->Echo();
		}
	}
	return;
}		
/*}}}*/
/*FUNCTION Numericalflux::DistributeNumDofs {{{1*/
void  Numericalflux::DistributeNumDofs(int* numdofspernode,int analysis_type,int sub_analysis_type){
	return;
}
/*}}}*/
/*FUNCTION Numericalflux::Echo {{{1*/
void Numericalflux::Echo(void){

	printf("Numericalflux:\n");
	printf("   type: %s\n",type);
	printf("   id: %i\n",id);

	if (strcmp(type,"internal")==0){
		printf("   element_ids=[%i,%i]\n",element_ids[0],element_ids[1]);
		printf("   element_offsets=[%i,%i]\n",element_offsets[0],element_offsets[1]);
		printf("   node_ids=[%i,%i,%i,%i]\n",node_ids[0],node_ids[1],node_ids[2],node_ids[3]);
		printf("   node_offsets=[%i,%i,%i,%i]\n",node_offsets[0],node_offsets[1],node_offsets[2],node_offsets[3]);
	}
	else{
		printf("   element_ids=%i\n",element_ids[0]);
		printf("   element_offsets=%i\n",element_offsets[0]);
		printf("   node_ids=[%i,%i]\n",node_ids[0],node_ids[1]);
		printf("   node_offsets=[%i,%i]\n",node_offsets[0],node_offsets[1]);
	}

	return;
}
/*}}}*/
/*FUNCTION Numericalflux::Enum {{{1*/
int Numericalflux::Enum(void){

	return NumericalfluxEnum();

}
/*}}}*/
/*FUNCTION Numericalflux::GetDofList{{{1*/

void  Numericalflux::GetDofList(int* doflist,int* pnumberofdofspernode){

	int i,j;
	int doflist_per_node[MAXDOFSPERNODE];
	int numberofdofspernode;
	
	if (strcmp(type,"internal")==0){
		for(i=0;i<4;i++){
			nodes[i]->GetDofList(&doflist_per_node[0],&numberofdofspernode);
			for(j=0;j<numberofdofspernode;j++){
				doflist[i*numberofdofspernode+j]=doflist_per_node[j];
			}
		}
	}
	else if (strcmp(type,"boundary")){
		for(i=0;i<2;i++){
			nodes[i]->GetDofList(&doflist_per_node[0],&numberofdofspernode);
			for(j=0;j<numberofdofspernode;j++){
				doflist[i*numberofdofspernode+j]=doflist_per_node[j];
			}
		}
	}
	else ISSMERROR("type not supported yet");

	/*Assign output pointers:*/
	*pnumberofdofspernode=numberofdofspernode;
}
/*}}}*/
/*FUNCTION Numericalflux::GetId {{{1*/
int    Numericalflux::GetId(void){
	return id;
}
/*}}}*/
/*FUNCTION Numericalflux::GetJacobianDeterminant{{{1*/
void Numericalflux::GetJacobianDeterminant(double* pJdet,double xyz_list[4][3], double gauss_coord){

	double Jdet,length;

	length=sqrt(pow(xyz_list[1][0] - xyz_list[0][0],2.0) + pow(xyz_list[1][1] - xyz_list[0][1],2.0));
	Jdet=1.0/2.0*length;

	if(Jdet<0){
		ISSMERROR(" negative jacobian determinant!");
	}

	*pJdet=Jdet;
}
/*}}}*/
/*FUNCTION Numericalflux::GetName {{{1*/
char* Numericalflux::GetName(void){
	return "numericalflux";
}
/*}}}*/
/*FUNCTION Numericalflux::GetNodalFunctions{{{1*/
void Numericalflux::GetNodalFunctions(double* l1l4, double gauss_coord){

	/*This routine returns the values of the nodal functions  at the gaussian point.*/

	l1l4[0]=-0.5*gauss_coord+0.5;
	l1l4[1]=+0.5*gauss_coord+0.5;
	l1l4[0]=-0.5*gauss_coord+0.5;
	l1l4[1]=+0.5*gauss_coord+0.5;

}
/*}}}*/
/*FUNCTION Numericalflux::MyRank {{{1*/
int    Numericalflux::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}*/
/*FUNCTION Numericalflux::PenaltyCreateKMatrix {{{1*/
void  Numericalflux::PenaltyCreateKMatrix(Mat Kgg,void* inputs,double kmax,int analysis_type,int sub_analysis_type){

	if (strcmp(type,"internal")==0){

		PenaltyCreateKMatrixInternal(Kgg,inputs,kmax,analysis_type,sub_analysis_type);
	}
	else if (strcmp(type,"boundary")==0){

		PenaltyCreateKMatrixBoundary(Kgg,inputs,kmax,analysis_type,sub_analysis_type);
	}
	else ISSMERROR("type not supported yet");
}
/*}}}*/
/*FUNCTION Numericalflux::PenaltyCreateKMatrixInternal {{{1*/
void  Numericalflux::PenaltyCreateKMatrixInternal(Mat Kgg,void* inputs,double kmax,int analysis_type,int sub_analysis_type){

	ISSMERROR("not supported yet");

}
/*}}}*/
/*FUNCTION Numericalflux::PenaltyCreateKMatrixBoundary {{{1*/
void  Numericalflux::PenaltyCreateKMatrixBoundary(Mat Kgg,void* inputs,double kmax,int analysis_type,int sub_analysis_type){

	ISSMERROR("type not supported yet");

}
/*}}}*/
/*FUNCTION Numericalflux::PenaltyCreatePVector{{{1*/
void  Numericalflux::PenaltyCreatePVector(Vec pg,void* inputs,double kmax,int analysis_type,int sub_analysis_type){

	if (strcmp(type,"internal")==0){

		PenaltyCreatePVectorInternal(pg,inputs,kmax,analysis_type,sub_analysis_type);
	}
	else if (strcmp(type,"boundary")==0){

		PenaltyCreatePVectorBoundary(pg,inputs,kmax,analysis_type,sub_analysis_type);
	}
	else ISSMERROR("type not supported yet");
}
/*}}}*/
/*FUNCTION Numericalflux::PenaltyCreatePVectorInternal{{{1*/
void  Numericalflux::PenaltyCreatePVectorInternal(Vec pg,void* inputs,double kmax,int analysis_type,int sub_analysis_type){

	/*Nothing added to PVector*/
	return;

}
/*}}}*/
/*FUNCTION Numericalflux::PenaltyCreatePVectorBoundary{{{1*/
void  Numericalflux::PenaltyCreatePVectorBoundary(Vec pg,void* inputs,double kmax,int analysis_type,int sub_analysis_type){

	ISSMERROR("type not supported yet");

}
/*}}}*/
/*FUNCTION Numericalflux::UpdateFromInputs {{{1*/
void  Numericalflux::UpdateFromInputs(void* vinputs){

	/*Do nothing...*/
}
/*}}}*/
