/*
 * \file DataSet.c
 * \brief: implementation of the DataSet class
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <vector>
#include <functional>
#include <algorithm>
#include <iostream>

#include "./DataSet.h"
#include "../shared/shared.h"
#include "../include/macros.h"
#include "../EnumDefinitions/EnumDefinitions.h"

using namespace std;

/*Constructors/Destructors*/
/*FUNCTION DataSet::DataSet(){{{1*/
DataSet::DataSet(){
	
	sorted=0;
	sorted_ids=NULL;
	id_offsets=NULL;

}
/*}}}*/
/*FUNCTION DataSet::DataSet(int dataset_enum){{{1*/
DataSet::DataSet(int dataset_enum){
	enum_type=dataset_enum;
	
	sorted=0;
	sorted_ids=NULL;
	id_offsets=NULL;

}
/*}}}*/
/*FUNCTION DataSet::Copy{{{1*/
DataSet*   DataSet::Copy(void){


	DataSet* copy=NULL;
	vector<Object*>::iterator object;
	Object* object_copy=NULL;

	copy=new DataSet(enum_type);

	copy->sorted=sorted;
	copy->presorted=presorted;
	if(sorted_ids){
		copy->sorted_ids=(int*)xmalloc(objects.size()*sizeof(int));
		memcpy(copy->sorted_ids,sorted_ids,objects.size()*sizeof(int));
	}
	if(id_offsets){
		copy->id_offsets=(int*)xmalloc(objects.size()*sizeof(int));
		memcpy(copy->id_offsets,id_offsets,objects.size()*sizeof(int));
	}

	/*Now we need to deep copy the objects: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Call echo on object: */
		object_copy = (*object)->copy();
		copy->AddObject(object_copy);
	}
	return copy;
}
/*}}}*/
/*FUNCTION DataSet::~DataSet{{{1*/
DataSet::~DataSet(){
	clear();
	xfree((void**)&sorted_ids);
	xfree((void**)&id_offsets);
}
/*}}}*/

/*I/O*/
/*FUNCTION DataSet::Marshall{{{1*/
char* DataSet::Marshall(){

	vector<Object*>::iterator object;
	int                       object_size;
	int                       marshalled_dataset_size=0;
	char*                     marshalled_dataset=NULL;
	char*                     old_marshalled_dataset=NULL;

	/*First get size of marshalled dataset: */
	object_size=(int)objects.size();

	marshalled_dataset_size=MarshallSize();
	
	/*Allocate marshalled dataset: */
	marshalled_dataset=(char*)xmalloc(marshalled_dataset_size*sizeof(char)); 

	/*Keep track of old_marshalled_dataset: */
	old_marshalled_dataset=marshalled_dataset;

	/*Store internals of dataset first: */
	memcpy(marshalled_dataset,&object_size,sizeof(int)); marshalled_dataset+=sizeof(int);
	memcpy(marshalled_dataset,&sorted,sizeof(int)); marshalled_dataset+=sizeof(int);
	if(sorted){
		if(object_size)memcpy(marshalled_dataset,sorted_ids,object_size*sizeof(int)); marshalled_dataset+=object_size*sizeof(int);
		if(object_size)memcpy(marshalled_dataset,id_offsets,object_size*sizeof(int)); marshalled_dataset+=object_size*sizeof(int);
	}

	for ( object=objects.begin() ; object < objects.end(); object++ ){
		(*object)->Marshall(&marshalled_dataset);
	}

	/* Ok, marshalled_dataset now points to the end of the original marshalled_dataset pointer 
	 * before  we started the loop on objects. Get object to point right again: */
	marshalled_dataset-=marshalled_dataset_size;

	/*We should be back to old_marshalled_dataset: check and abort if that's not the case, 
	 * because this is a nasty error: */
	if (marshalled_dataset!=old_marshalled_dataset){
		ISSMERROR("final marshalled dataset is different from initial one!"); 
		abort();
	}

	/*Return: */
	return marshalled_dataset;
}
/*}}}*/
/*FUNCTION DataSet::MarshallSize{{{1*/
int DataSet::MarshallSize(){

	vector<Object*>::iterator object;
	int                      marshalled_dataset_size=0;


	for ( object=objects.begin() ; object < objects.end(); object++ ){
		marshalled_dataset_size+= (*object)->MarshallSize();
	}

	marshalled_dataset_size+=sizeof(int); //objects size
	marshalled_dataset_size+=sizeof(int); //sorted size
	if(sorted){
		marshalled_dataset_size+=(int)objects.size()*sizeof(int); //sorted ids
		marshalled_dataset_size+=(int)objects.size()*sizeof(int); //id offsets
	}

	return marshalled_dataset_size;
}
/*}}}*/
/*FUNCTION DataSetDemarshall{{{1*/
DataSet* DataSetDemarshall(char* marshalled_dataset){

	int i;

	DataSet* dataset=NULL;
	int      numobjects=0;
	int      enum_type;
	Object*  object=NULL;
	int      sorted;
	int*     sorted_ids=NULL;
	int*     id_offsets=NULL;

	/*initialize dataset: */
	dataset=new DataSet();

	/*Get internals first: */
	memcpy(&numobjects,marshalled_dataset,sizeof(int)); marshalled_dataset+=sizeof(int);
	memcpy(&sorted,marshalled_dataset,sizeof(int)); marshalled_dataset+=sizeof(int);
	if(sorted){
		if(numobjects){
			sorted_ids=(int*)xmalloc(numobjects*sizeof(int));
			id_offsets=(int*)xmalloc(numobjects*sizeof(int));
			memcpy(sorted_ids,marshalled_dataset,numobjects*sizeof(int)); marshalled_dataset+=numobjects*sizeof(int);
			memcpy(id_offsets,marshalled_dataset,numobjects*sizeof(int)); marshalled_dataset+=numobjects*sizeof(int);
		}
		dataset->SetSorting(sorted_ids,id_offsets);
	}

#ifdef _ISSM_DEBUG_
	_printf_("Number of objects in dataset being demarshalled: %i\n",numobjects);
#endif

	for(i=0;i<numobjects;i++){

		/*get enum type of object: */
		memcpy(&enum_type,marshalled_dataset,sizeof(int)); marshalled_dataset+=sizeof(int);

		if(enum_type==NodeEnum()){
			Node* node=NULL;
			node=new Node();
			node->Demarshall(&marshalled_dataset);
			dataset->AddObject(node);
		}
		else if(enum_type==ParamEnum()){
			Param* param=NULL;
			param=new Param();
			param->Demarshall(&marshalled_dataset);
			dataset->AddObject(param);
		}
		else if(enum_type==NumparEnum()){
			Numpar* numpar=NULL;
			numpar=new Numpar();
			numpar->Demarshall(&marshalled_dataset);
			dataset->AddObject(numpar);
		}
		else if(enum_type==TriaEnum()){
			Tria* tria=NULL;
			tria=new Tria();
			tria->Demarshall(&marshalled_dataset);
			dataset->AddObject(tria);
		}
		else if(enum_type==SingEnum()){
			Sing* sing=NULL;
			sing=new Sing();
			sing->Demarshall(&marshalled_dataset);
			dataset->AddObject(sing);
		}
		else if(enum_type==BeamEnum()){
			Beam* beam=NULL;
			beam=new Beam();
			beam->Demarshall(&marshalled_dataset);
			dataset->AddObject(beam);
		}
		else if(enum_type==PentaEnum()){
			Penta* penta=NULL;
			penta=new Penta();
			penta->Demarshall(&marshalled_dataset);
			dataset->AddObject(penta);
		}
		else if(enum_type==MaticeEnum()){
			Matice* matice=NULL;
			matice=new Matice();
			matice->Demarshall(&marshalled_dataset);
			dataset->AddObject(matice);
		}
		else if(enum_type==MatparEnum()){
			Matpar* matpar=NULL;
			matpar=new Matpar();
			matpar->Demarshall(&marshalled_dataset);
			dataset->AddObject(matpar);
		}
		else if(enum_type==SpcEnum()){
			Spc* spc=NULL;
			spc=new Spc();
			spc->Demarshall(&marshalled_dataset);
			dataset->AddObject(spc);
		}
		else if(enum_type==PengridEnum()){
			Pengrid* pengrid=NULL;
			pengrid=new Pengrid();
			pengrid->Demarshall(&marshalled_dataset);
			dataset->AddObject(pengrid);
		}
		else if(enum_type==PenpairEnum()){
			Penpair* penpair=NULL;
			penpair=new Penpair();
			penpair->Demarshall(&marshalled_dataset);
			dataset->AddObject(penpair);
		}
		else if(enum_type==IcefrontEnum()){
			Icefront* icefront=NULL;
			icefront=new Icefront();
			icefront->Demarshall(&marshalled_dataset);
			dataset->AddObject(icefront);
		}
		else if(enum_type==RgbEnum()){
			Rgb* rgb=NULL;
			rgb=new Rgb();
			rgb->Demarshall(&marshalled_dataset);
			dataset->AddObject(rgb);
		}
		else if(enum_type==RiftfrontEnum()){
			Riftfront* riftfront=NULL;
			riftfront=new Riftfront();
			riftfront->Demarshall(&marshalled_dataset);
			dataset->AddObject(riftfront);
		}
		else{
			ISSMERROR(exprintf("%s%i"," could not recognize enum type: ",enum_type));
		}

	}
	return dataset;

}
/*}}}*/

/*Specific methods*/
/*FUNCTION DataSet::AddObject{{{1*/
int  DataSet::AddObject(Object* object){

	objects.push_back(object);

	return 1;
}
/*}}}*/
/*FUNCTION DataSet::clear{{{1*/
void  DataSet::clear(){

	vector<Object*>::iterator object;

	for ( object=objects.begin() ; object < objects.end(); object++ ){
		delete (*object);
	}
	objects.clear();
}
/*}}}*/
/*FUNCTION DataSet::DeleteObject{{{1*/
int  DataSet::DeleteObject(Object* object){

	vector<Object*>::iterator iterator;

	if(object){
		iterator = find(objects.begin(), objects.end(),object);
		delete *iterator;
		objects.erase(iterator);
	}

}
/*}}}*/
/*FUNCTION DataSet::DeepEcho{{{1*/
void DataSet::DeepEcho(){


	vector<Object*>::iterator object;

	if(this==NULL)ISSMERROR(" trying to echo a NULL dataset");

	_printf_("DataSet echo: %i objects\n",objects.size());

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Call deep echo on object: */
		(*object)->DeepEcho();

	}
	return;
}
/*}}}*/
/*FUNCTION DataSet::Echo{{{1*/
void DataSet::Echo(){


	vector<Object*>::iterator object;

	if(this==NULL)ISSMERROR(" trying to echo a NULL dataset");

	_printf_("DataSet echo: %i objects\n",objects.size());

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Call echo on object: */
		(*object)->Echo();

	}
	return;
}
/*}}}*/
/*FUNCTION DataSet::FindParam(double* pscalar, char* name){{{1*/
int   DataSet::FindParam(double* pscalar, char* name){
	
	/*Go through a dataset, and find a Param* object 
	 *which parameter name is "name" : */
	
	vector<Object*>::iterator object;
	Param* param=NULL;

	int found=0;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Find param type objects: */
		if((*object)->Enum()==ParamEnum()){

			/*Ok, this object is a parameter, recover it and ask which name it has: */
			param=(Param*)(*object);

			if (strcmp(param->GetParameterName(),name)==0){
				/*Ok, this is the one! Recover the value of this parameter: */
				param->GetParameterValue(pscalar);
				found=1;
				break;
			}
		}
	}
	return found;
}
/*}}}*/
/*FUNCTION DataSet::FindParam(int* pinteger,char* name){{{1*/
int   DataSet::FindParam(int* pinteger,char* name){
	
	
	/*Go through a dataset, and find a Param* object 
	 *which parameter name is "name" : */
	
	vector<Object*>::iterator object;
	Param* param=NULL;

	int found=0;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Find param type objects: */
		if((*object)->Enum()==ParamEnum()){

			/*Ok, this object is a parameter, recover it and ask which name it has: */
			param=(Param*)(*object);

			if (strcmp(param->GetParameterName(),name)==0){
				/*Ok, this is the one! Recover the value of this parameter: */
				param->GetParameterValue(pinteger);
				found=1;
				break;
			}
		}
	}
	return found;
}
/*}}}*/
/*FUNCTION DataSet::FindParam(char** pstring,char* name){{{1*/
int   DataSet::FindParam(char** pstring,char* name){
	
	/*Go through a dataset, and find a Param* object 
	 *which parameter name is "name" : */
	
	vector<Object*>::iterator object;
	Param* param=NULL;

	int found=0;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Find param type objects: */
		if((*object)->Enum()==ParamEnum()){

			/*Ok, this object is a parameter, recover it and ask which name it has: */
			param=(Param*)(*object);

			if (strcmp(param->GetParameterName(),name)==0){
				/*Ok, this is the one! Recover the value of this parameter: */
				param->GetParameterValue(pstring);
				found=1;
				break;
			}
		}
	}
	return found;

}
/*}}}*/
/*FUNCTION DataSet::FindParam(char*** pstringarray,int* pM,char* name){{{1*/
int   DataSet::FindParam(char*** pstringarray,int* pM,char* name){
	
	/*Go through a dataset, and find a Param* object 
	 *which parameter name is "name" : */
	
	vector<Object*>::iterator object;
	Param* param=NULL;

	int found=0;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Find param type objects: */
		if((*object)->Enum()==ParamEnum()){

			/*Ok, this object is a parameter, recover it and ask which name it has: */
			param=(Param*)(*object);

			if (strcmp(param->GetParameterName(),name)==0){
				/*Ok, this is the one! Recover the value of this parameter: */
				param->GetParameterValue(pstringarray);
				if(pM)*pM=param->GetM();
				found=1;
				break;
			}
		}
	}
	return found;

}
/*}}}*/
/*FUNCTION DataSet::FindParam(double** pdoublearray,int* pM, int* pN,char* name){{{1*/
int   DataSet::FindParam(double** pdoublearray,int* pM, int* pN,char* name){
	
	/*Go through a dataset, and find a Param* object 
	 *which parameter name is "name" : */
	
	vector<Object*>::iterator object;
	Param* param=NULL;

	int found=0;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Find param type objects: */
		if((*object)->Enum()==ParamEnum()){

			/*Ok, this object is a parameter, recover it and ask which name it has: */
			param=(Param*)(*object);

			if (strcmp(param->GetParameterName(),name)==0){
				/*Ok, this is the one! Recover the value of this parameter: */
				param->GetParameterValue(pdoublearray);
				if(pM)param->GetM();
				if(pN)param->GetN();
				found=1;
				break;
			}
		}
	}
	return found;

}
/*}}}*/
/*FUNCTION DataSet::FindParam(Vec* pvec,char* name){{{1*/
int   DataSet::FindParam(Vec* pvec,char* name){
	
	/*Go through a dataset, and find a Param* object 
	 *which parameter name is "name" : */
	
	vector<Object*>::iterator object;
	Param* param=NULL;

	int found=0;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Find param type objects: */
		if((*object)->Enum()==ParamEnum()){

			/*Ok, this object is a parameter, recover it and ask which name it has: */
			param=(Param*)(*object);

			if (strcmp(param->GetParameterName(),name)==0){
				/*Ok, this is the one! Recover the value of this parameter: */
				param->GetParameterValue(pvec);
				found=1;
				break;
			}
		}
	}
	return found;

}
/*}}}*/
/*FUNCTION DataSet::FindParamMat* pmat,char* name){{{1*/
int   DataSet::FindParam(Mat* pmat,char* name){
	
	/*Go through a dataset, and find a Param* object 
	 *which parameter name is "name" : */
	
	vector<Object*>::iterator object;
	Param* param=NULL;

	int found=0;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Find param type objects: */
		if((*object)->Enum()==ParamEnum()){

			/*Ok, this object is a parameter, recover it and ask which name it has: */
			param=(Param*)(*object);

			if (strcmp(param->GetParameterName(),name)==0){
				/*Ok, this is the one! Recover the value of this parameter: */
				param->GetParameterValue(pmat);
				found=1;
				break;
			}
		}
	}
	return found;

}
/*}}}*/
/*FUNCTION DataSet::FindParamObject{{{1*/
Object*   DataSet::FindParamObject(char* name){

	/*Go through a dataset, and find a Param* object 
	 *which parameter name is "name" : */

	vector<Object*>::iterator object;
	Param* param=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Find param type objects: */
		if((*object)->Enum()==ParamEnum()){

			/*Ok, this object is a parameter, recover it and ask which name it has: */
			param=(Param*)(*object);

			if (strcmp(param->GetParameterName(),name)==0){
				/*Ok, this is the one! Return the object: */
				return (*object);
			}
		}
	}
	return NULL;
}
/*}}}*/
/*FUNCTION DataSet::FindResult(Vec* presult,char* name){{{1*/
int   DataSet::FindResult(Vec* presult,char* name){

	/*Go through a dataset, and find a Result* object 
	 *whith result name is "name" : */
	
	vector<Object*>::iterator object;
	Result* result=NULL;

	int found=0;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Find param type objects: */
		if((*object)->Enum()==ResultEnum()){

			/*Ok, this object is a result,recover it and ask which name it has: */
			result=(Result*)(*object);

			if (strcmp(result->GetFieldName(),name)==0){
				/*Ok, this is the one! Recover the value of this result: */
				result->GetField(presult);
				found=1;
				break;
			}
		}
	}
	return found;
}
/*}}}*/
/*FUNCTION DataSet::FindResult(void* pvalue, char* name){{{1*/
int   DataSet::FindResult(void* pvalue, char* name){

	/*Go through a dataset, and find a Result* object 
	 *which field name is "name" : */
	
	vector<Object*>::iterator object;
	Result* result=NULL;

	int found=0;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Find param type objects: */
		if((*object)->Enum()==ResultEnum()){

			/*Ok, this object is a result, recover it and ask which name it has: */
			result=(Result*)(*object);

			if (strcmp(result->GetFieldName(),name)==0){
				/*Ok, this is the one! Recover the value of this result: */
				double** field=NULL;
				field=(double**)pvalue; //not very safe, but hey!
				result->GetField(field);
				found=1;
				break;
			}
		}
	}
	return found;
}
/*}}}*/
/*FUNCTION DataSet::GetEnum(){{{1*/
int  DataSet::GetEnum(){
	return enum_type;
}
/*}}}*/
/*FUNCTION DataSet::GetEnum(int offset){{{1*/
int   DataSet::GetEnum(int offset){

	return objects[offset]->Enum();

}
/*}}}*/
/*FUNCTION DataSet::GetObjectByOffset{{{1*/
Object* DataSet::GetObjectByOffset(int offset){

	return objects[offset];

}
/*}}}*/
/*FUNCTION DataSet::GetObjectById{{{1*/
Object* DataSet::GetObjectById(int* poffset,int eid){

	int id_offset;
	int offset;
	int i;

	if(!sorted)ISSMERROR(" trying to binary search on a non-sorted dataset!");

	/*Carry out a binary search on the sorted_ids: */
	if(!binary_search(&id_offset,eid, sorted_ids,objects.size())){
		ISSMERROR(exprintf("%s%i"," could not find object with id ",eid));
	}

	/*Convert  the id offset into sorted offset: */
	offset=id_offsets[id_offset];

	/*Assign output pointers if requested:*/
	if (poffset)*poffset=offset;

	/*Return object at offset position in objects :*/
	return objects[offset];
}
/*}}}*/
/*FUNCTION DataSet::NodeRank{{{1*/
void   DataSet::NodeRank(int* ranks){

	/*Go through a dataset, and find a Node* object. For this object, 
	 * ask it to report its cpu rank: */

	int node_rank;
	int id;
	vector<Object*>::iterator object;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Call echo on object: */
		if((*object)->Enum()==NodeEnum()){

			/*Ok, this object is a node, ask which rank it has: */
			node_rank=(*object)->MyRank();

			/*Which id does it have: */
			id=(*object)->GetId();

			/*Plug rank into ranks, according to id: */
			ranks[id-1]=node_rank; //Matlab ids start at 1.
		}
	}
	return;
}
/*}}}*/
/*FUNCTION DataSet::Presort{{{1*/
void DataSet::Presort(){

	/*vector of objects is already sorted, just allocate the sorted ids and their
	 * offsets:*/
	int i;

	if(objects.size()){
		sorted_ids=(int*)xmalloc(objects.size()*sizeof(int));
		id_offsets=(int*)xmalloc(objects.size()*sizeof(int));
		for(i=0;i<objects.size();i++){
			id_offsets[i]=i;
			sorted_ids[i]=objects[i]->GetId();
		}
	}

	/*set sorted flag: */
	sorted=1;
}
/*}}}*/
/*FUNCTION DataSet::SetSorting{{{1*/
void DataSet::SetSorting(int* in_sorted_ids,int* in_id_offsets){

	sorted=1;
	sorted_ids=in_sorted_ids;
	id_offsets=in_id_offsets;
}
/*}}}*/
/*FUNCTION DataSet::Size{{{1*/
int  DataSet::Size(void){

	return objects.size();
}
/*}}}*/
/*FUNCTION DataSet::Sort{{{1*/
void DataSet::Sort(){

	/*Only sort if we are not already sorted: */
	if(!sorted){
		ISSMERROR(" not implemented yet!");
	}
}
/*}}}*/

/*Objects methods*/
/*FUNCTION DataSet::ComputePressure{{{1*/
void DataSet::ComputePressure(Vec p_g){

	vector<Object*>::iterator object;
	Element* element=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if(EnumIsElement((*object)->Enum())){

			element=(Element*)(*object);
			element->ComputePressure(p_g);
		}
	}

}
/*}}}*/
/*FUNCTION DataSet::Configure{{{1*/
void DataSet::Configure(DataSet* elements,DataSet* loads, DataSet* nodes, DataSet* materials,DataSet* parameters){

	vector<Object*>::iterator object;
	Element* element=NULL;
	Load* load=NULL;
	Node* node=NULL;
	Numpar* numpar=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if(EnumIsElement((*object)->Enum())){

			element=(Element*)(*object);
			element->Configure(loads,nodes,materials,parameters);
		}
		if(EnumIsLoad((*object)->Enum())){

			load=(Load*)(*object);

			load->Configure(elements,nodes,materials);
		}

		if((*object)->Enum()==NodeEnum()){
			node=(Node*)(*object);
			node->Configure(nodes);
		}

		if((*object)->Enum()==NumparEnum()){
			numpar=(Numpar*)(*object);
			numpar->Configure(parameters);
		}

	}

}
/*}}}*/
/*FUNCTION DataSet::CostFunction{{{1*/
void  DataSet::CostFunction(double* pJ,void* inputs,int analysis_type,int sub_analysis_type){

	double J=0;;

	vector<Object*>::iterator object;
	Element* element=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if(EnumIsElement((*object)->Enum())){

			element=(Element*)(*object);
			J+=element->CostFunction(inputs,analysis_type,sub_analysis_type);

		}
	}

	/*Assign output pointers:*/
	*pJ=J;

}
/*}}}*/
/*FUNCTION DataSet::CreateKMatrix{{{1*/
void  DataSet::CreateKMatrix(Mat Kgg,void* inputs,int analysis_type,int sub_analysis_type){

	vector<Object*>::iterator object;
	Element* element=NULL;
	Load* load=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){
		
		if(EnumIsElement((*object)->Enum())){

			element=(Element*)(*object);
			element->CreateKMatrix(Kgg,inputs,analysis_type,sub_analysis_type);
		}
		if(EnumIsLoad((*object)->Enum())){

			load=(Load*)(*object);
			load->CreateKMatrix(Kgg,inputs,analysis_type,sub_analysis_type);
		}
	}

}
/*}}}*/
/*FUNCTION DataSet::CreatePartitioningVector{{{1*/
void  DataSet::CreatePartitioningVector(Vec* ppartition,int numberofnodes,int numdofspernode){

	/*output: */
	Vec partition=NULL;
	vector<Object*>::iterator object;
	Node* node=NULL;

	/*Create partition vector: */
	partition=NewVec(numberofnodes);

	/*Go through all nodes, and ask each node to plug its 1D doflist in 
	 * partition. The location where each node plugs its doflist into 
	 * partition is determined by its (id-1)*3 (ie, serial * organisation of the dofs).
	 */

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Check this is a node: */
		if((*object)->Enum()==NodeEnum()){

			node=(Node*)(*object);

			/*Ok, this object is a node, ask it to plug values into partition: */
			node->CreatePartition(partition);

		}
	}

	/*Assemble the petsc vector: */
	VecAssemblyBegin(partition);
	VecAssemblyEnd(partition);

	/*Assign output pointers: */
	*ppartition=partition;

	return;
}
/*}}}*/
/*FUNCTION DataSet::CreatePVector{{{1*/
void  DataSet::CreatePVector(Vec pg,void* inputs,int analysis_type,int sub_analysis_type){

	vector<Object*>::iterator object;
	Element* element=NULL;
	Load* load=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if(EnumIsElement((*object)->Enum())){

			element=(Element*)(*object);
			element->CreatePVector(pg,inputs,analysis_type,sub_analysis_type);
		}
		if(EnumIsLoad((*object)->Enum())){

			load=(Load*)(*object);
			load->CreatePVector(pg,inputs,analysis_type,sub_analysis_type);
		}		
	}

}
/*}}}*/
/*FUNCTION DataSet::DistributeDofs{{{1*/
void  DataSet::DistributeDofs(int numberofnodes,int numdofspernode){

	int  dofcount=0;
	int  dofcount1=0;
	int* alldofcount=NULL;
	int* alldofcount1=NULL;
	int* borderdofs=NULL;
	int* borderdofs1=NULL;
	int* allborderdofs=NULL;
	int* allborderdofs1=NULL;
	int  i;
	vector<Object*>::iterator object;
	Node* node=NULL;

	extern int num_procs;
	extern int my_rank;

	dofcount=0;
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Check this is a node: */
		if((*object)->Enum()==NodeEnum()){

			node=(Node*)(*object);

			/*Ok, this object is a node, ask it to distribute dofs, and update the dofcount: */
			node->DistributeDofs(&dofcount,&dofcount1);

		}
	}

	/*Ok, now every node has distributed dofs, but locally, and with a dof count starting from 
	 *0. This means the dofs between all the cpus are not synchronized! We need to synchronize all 
	 *dof on all cpus, and use those to update the dofs of every node: */

	alldofcount=(int*)xmalloc(num_procs*sizeof(int));
	MPI_Gather(&dofcount,1,MPI_INT,alldofcount,1,MPI_INT,0,MPI_COMM_WORLD);
	MPI_Bcast(alldofcount,num_procs,MPI_INT,0,MPI_COMM_WORLD);

	alldofcount1=(int*)xmalloc(num_procs*sizeof(int));
	MPI_Gather(&dofcount1,1,MPI_INT,alldofcount1,1,MPI_INT,0,MPI_COMM_WORLD);
	MPI_Bcast(alldofcount1,num_procs,MPI_INT,0,MPI_COMM_WORLD);

	/*Ok, now every cpu should start its own dof count at the end of the dofcount 
	 * from cpu-1. : */
	dofcount=0;
	dofcount1=0;
	if(my_rank==0){
		dofcount=0;
		dofcount1=0;
	}
	else{
		for(i=0;i<my_rank;i++){
			dofcount+=alldofcount[i];
			dofcount1+=alldofcount1[i];
		}
	}


	/*Ok, now every cpu knows where his dofs should start. Update the dof count: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Check this is a node: */
		if((*object)->Enum()==NodeEnum()){

			node=(Node*)(*object);

			/*Ok, this object is a node, ask it to update his dofs: */
			node->UpdateDofs(dofcount,dofcount1);

		}
	}

	/*Finally, remember that cpus may have skipped some nodes, when these nodes were 
	 * clones: */
	borderdofs=(int*)xcalloc(numberofnodes*numdofspernode,sizeof(int));
	allborderdofs=(int*)xcalloc(numberofnodes*numdofspernode,sizeof(int));
	borderdofs1=(int*)xcalloc(numberofnodes*3,sizeof(int));
	allborderdofs1=(int*)xcalloc(numberofnodes*3,sizeof(int));

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Check this is a node: */
		if((*object)->Enum()==NodeEnum()){

			node=(Node*)(*object);

			/*Ok, let this object show its border dofs, if is is a border dof: */
			node->ShowBorderDofs(borderdofs,borderdofs1);

		}
	}
	MPI_Allreduce ( (void*)borderdofs,(void*)allborderdofs,numberofnodes*numdofspernode,MPI_INT,MPI_MAX,MPI_COMM_WORLD);
	MPI_Allreduce ( (void*)borderdofs1,(void*)allborderdofs1,numberofnodes*3,MPI_INT,MPI_MAX,MPI_COMM_WORLD);

	/*Ok, now every cpu knows everyone else's border node dofs, update the border dofs accordingly: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Check this is a node: */
		if((*object)->Enum()==NodeEnum()){

			node=(Node*)(*object);

			/*Ok, this object is a node, ask it to update his dofs: */
			node->UpdateBorderDofs(allborderdofs,allborderdofs1);

		}
	}


	/* Free ressources: */
	xfree((void**)&alldofcount);
	xfree((void**)&borderdofs);
	xfree((void**)&allborderdofs);
	xfree((void**)&alldofcount1);
	xfree((void**)&borderdofs1);
	xfree((void**)&allborderdofs1);

	return;

}
/*}}}*/
/*FUNCTION DataSet::Du{{{1*/
void  DataSet::Du(Vec du_g,void* inputs,int analysis_type,int sub_analysis_type){


	vector<Object*>::iterator object;
	Element* element=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if(EnumIsElement((*object)->Enum())){

			element=(Element*)(*object);
			element->Du(du_g,inputs,analysis_type,sub_analysis_type);
		}
	}


}
/*}}}*/
/*FUNCTION DataSet::FieldDepthAverageAtBase{{{1*/
void  DataSet::FieldDepthAverageAtBase(Vec field,double* field_serial,char* fieldname){

	vector<Object*>::iterator object;
	Node* node=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if((*object)->Enum()==NodeEnum()){

			node=(Node*)(*object);
			node->FieldDepthAverageAtBase(field,field_serial,fieldname);
		}
	}

}
/*}}}*/
/*FUNCTION DataSet::FieldExtrude{{{1*/
void  DataSet::FieldExtrude(Vec field,double* field_serial,char* field_name, int collapse){

	vector<Object*>::iterator object;
	Penta* penta=NULL;
	Node*  node=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if((*object)->Enum()==PentaEnum()){

			penta=(Penta*)(*object);
			penta->FieldExtrude(field,field_serial,field_name,collapse);

		}
		if((*object)->Enum()==NodeEnum()){

			node=(Node*)(*object);
			node->FieldExtrude(field,field_serial,field_name);

		}
	}

}
/*}}}*/
/*FUNCTION DataSet::FlagClones{{{1*/
void  DataSet::FlagClones(int numberofnodes){

	int i;
	extern int num_procs;

	int* ranks=NULL;
	int* minranks=NULL;

	vector<Object*>::iterator object;
	Node* node=NULL;

	/*Allocate ranks: */
	ranks=(int*)xmalloc(numberofnodes*sizeof(int));
	minranks=(int*)xmalloc(numberofnodes*sizeof(int));

	for(i=0;i<numberofnodes;i++)ranks[i]=num_procs; //no cpu can have rank num_procs. This is the maximum limit.

	/*Now go through all our nodes and ask them to report to who they belong (which rank): */
	NodeRank(ranks);

#ifdef _ISSM_DEBUG_
	for(i=0;i<numberofnodes;i++){
		_printf_("%i\n",ranks[i]);
	}
#endif

	/*We need to take the minimum rank for each node, and every cpu needs to get that result. That way, 
	 * when we start building the dof list for all nodes, a cpu can check whether its node already has been 
	 * dealt with by another cpu. We take the minimum because we are going to manage dof assignment in increasing 
	 * order of cpu rank. This is also why we initialized this array to num_procs.*/
	MPI_Allreduce ( (void*)ranks,(void*)minranks,numberofnodes,MPI_INT,MPI_MIN,MPI_COMM_WORLD);

#ifdef _ISSM_DEBUG_
	for(i=0;i<numberofnodes;i++){
		_printf_("%i\n",minranks[i]);
	}
#endif

	/*Now go through all nodes, and use minranks to flag which nodes are cloned: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Check this is a node: */
		if((*object)->Enum()==NodeEnum()){

			node=(Node*)(*object);

			/*Ok, this object is a node, ask it to plug values into partition: */
			node->SetClone(minranks);

		}
	}

	/*Free ressources: */
	xfree((void**)&ranks); 
	xfree((void**)&minranks);

}
/*}}}*/
/*FUNCTION DataSet::FlagNodeSets{{{1*/
void DataSet::FlagNodeSets(Vec pv_g, Vec pv_m, Vec pv_n, Vec pv_f, Vec pv_s){

	vector<Object*>::iterator object;
	Node* node=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Check this is a single point constraint (spc): */
		if((*object)->Enum()==NodeEnum()){

			node=(Node*)(*object);

			/*Plug set values intp our 4 set vectors: */
			node->CreateVecSets(pv_g,pv_m,pv_n,pv_f,pv_s);

		}
	}

	/*Assemble: */
	VecAssemblyBegin(pv_g);
	VecAssemblyEnd(pv_g);

	VecAssemblyBegin(pv_m);
	VecAssemblyEnd(pv_m);

	VecAssemblyBegin(pv_n);
	VecAssemblyEnd(pv_n);

	VecAssemblyBegin(pv_f);
	VecAssemblyEnd(pv_f);

	VecAssemblyBegin(pv_s);
	VecAssemblyEnd(pv_s);

}
/*}}}*/
/*FUNCTION DataSet::Gradj{{{1*/
void  DataSet::Gradj(Vec grad_g,void* inputs,int analysis_type,int sub_analysis_type,char* control_type){


	vector<Object*>::iterator object;
	Element* element=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if(EnumIsElement((*object)->Enum())){

			element=(Element*)(*object);
			element->Gradj(grad_g,inputs,analysis_type,sub_analysis_type,control_type);
		}
	}


}		
/*}}}*/
/*FUNCTION DataSet::MeltingIsPresent{{{1*/
int   DataSet::MeltingIsPresent(){

	int found=0;
	int mpi_found=0;

	vector<Object*>::iterator object;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if((*object)->Enum()==PengridEnum()){
			found=1;
			break;
		}
	}	
	
	#ifdef _PARALLEL_
	MPI_Reduce (&found,&mpi_found,1,MPI_INT,MPI_SUM,0,MPI_COMM_WORLD );
	MPI_Bcast(&mpi_found,1,MPI_INT,0,MPI_COMM_WORLD);                
	found=mpi_found;
	#endif

	return found;
}
/*}}}*/
/*FUNCTION DataSet::MeltingConstraints{{{1*/
void  DataSet::MeltingConstraints(int* pconverged, int* pnum_unstable_constraints,void* inputs,int analysis_type,int sub_analysis_type){

	/* generic object pointer: */
	vector<Object*>::iterator object;
	Pengrid* pengrid=NULL;

	int unstable=0;
	int num_unstable_constraints=0;
	int converged=0;
	int sum_num_unstable_constraints=0;

	num_unstable_constraints=0;	

	/*Enforce constraints: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){
		
		if((*object)->Enum()==PengridEnum()){

			pengrid=(Pengrid*)(*object);

			pengrid->PenaltyConstrain(&unstable,inputs,analysis_type,sub_analysis_type);

			num_unstable_constraints+=unstable;
		}
	}

	#ifdef _PARALLEL_
	MPI_Reduce (&num_unstable_constraints,&sum_num_unstable_constraints,1,MPI_INT,MPI_SUM,0,MPI_COMM_WORLD );
	MPI_Bcast(&sum_num_unstable_constraints,1,MPI_INT,0,MPI_COMM_WORLD);                
	num_unstable_constraints=sum_num_unstable_constraints;
	#endif

	/*Have we converged? : */
	if (num_unstable_constraints==0) converged=1;
	else converged=0;

	/*Assign output pointers: */
	*pconverged=converged;
	*pnum_unstable_constraints=num_unstable_constraints;
}
/*}}}*/
/*FUNCTION DataSet::Misfit{{{1*/
void  DataSet::Misfit(double* pJ,void* inputs,int analysis_type,int sub_analysis_type){

	double J=0;;

	vector<Object*>::iterator object;
	Element* element=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if(EnumIsElement((*object)->Enum())){

			element=(Element*)(*object);
			J+=element->Misfit(inputs,analysis_type,sub_analysis_type);

		}
	}

	/*Assign output pointers:*/
	*pJ=J;

}
/*}}}*/
/*FUNCTION DataSet::NumberOfDofs{{{1*/
int   DataSet::NumberOfDofs(){


	vector<Object*>::iterator object;
	Node* node=NULL;
	int   numdofs=0;
	int   allnumdofs;

	/*Now go through all nodes, and get how many dofs they own, unless they are clone nodes: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Check this is a node: */
		if((*object)->Enum()==NodeEnum()){

			node=(Node*)(*object);

			/*Ok, this object is a node, ask it to plug values into partition: */
			if (!node->IsClone()){

				numdofs+=node->GetNumberOfDofs();

			}
		}
	}

	/*Gather from all cpus: */
	MPI_Allreduce ( (void*)&numdofs,(void*)&allnumdofs,1,MPI_INT,MPI_SUM,MPI_COMM_WORLD);

	return allnumdofs;
}
/*}}}*/
/*FUNCTION DataSet::NumberOfRgbs{{{1*/
int   DataSet::NumberOfRgbs(){

	vector<Object*>::iterator object;
	Rgb* rgb=NULL;
	int  count=0;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Check this is a single point constraint (spc): */
		if((*object)->Enum()==RgbEnum()){ //we assume uniqueness of all Rgbs, no error checking here.

			count++;

		}
	}

	return count;
}
/*}}}*/
/*FUNCTION DataSet::OutputRifts{{{1*/
void  DataSet::OutputRifts(Vec riftproperties){

	vector<Object*>::iterator object;
	Riftfront* riftfront=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if((*object)->Enum()==RiftfrontEnum()){

			riftfront=(Riftfront*)(*object);
			riftfront->OutputProperties(riftproperties);
		}
	}



}
/*}}}*/
/*FUNCTION DataSet::PenaltyCreateKMatrix{{{1*/
void  DataSet::PenaltyCreateKMatrix(Mat Kgg,void* inputs,double kmax,int analysis_type,int sub_analysis_type){

	vector<Object*>::iterator object;
	Load* load=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){
		
		if(EnumIsLoad((*object)->Enum())){

			load=(Load*)(*object);
			load->PenaltyCreateKMatrix(Kgg,inputs,kmax,analysis_type,sub_analysis_type);
		}
	}

}
/*}}}*/
/*FUNCTION DataSet::PenaltyCreatePVector{{{1*/
void  DataSet::PenaltyCreatePVector(Vec pg,void* inputs,double kmax,int analysis_type,int sub_analysis_type){

	vector<Object*>::iterator object;
	Load* load=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if(EnumIsLoad((*object)->Enum())){

			load=(Load*)(*object);
			load->PenaltyCreatePVector(pg,inputs,kmax,analysis_type,sub_analysis_type);
		}		
	}

}
/*}}}*/
/*FUNCTION DataSet::RiftIsPresent{{{1*/
int   DataSet::RiftIsPresent(){

	int i;

	vector<Object*>::iterator object;
	Penpair* penpair=NULL;
	int found=0;
	int mpi_found=0;

	/*go though loads, and figure out if one of the loads is a PenPair with numdof=2: */
	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if((*object)->Enum()==PenpairEnum()){

			penpair=(Penpair*)(*object);
		}
	}

#ifdef _PARALLEL_
	MPI_Reduce (&found,&mpi_found,1,MPI_INT,MPI_SUM,0,MPI_COMM_WORLD );
	MPI_Bcast(&mpi_found,1,MPI_INT,0,MPI_COMM_WORLD);                
	found=mpi_found;
#endif

	return found;
}
/*}}}*/
/*FUNCTION DataSet::SetupMpcs{{{1*/
void DataSet::SetupMpcs(Mat Rmg,DataSet* nodes){

	vector<Object*>::iterator object;
	Rgb* rgb=NULL;
	int  nodeid1;
	int  nodeid2;
	int  dof;

	int  dof1;
	int  dof2;


	Node* node1=NULL;
	Node* node2=NULL;

	int count=-1;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Check this is a single point constraint (spc): */
		if((*object)->Enum()==RgbEnum()){ //we assume uniqueness of all Rgbs, no error checking here.

			/*we found an rgb, increment counter, so that row index for Rmg is up to date: */
			count++;
			rgb=(Rgb*)(*object);


			nodeid1=rgb->GetNodeId1();
			nodeid2=rgb->GetNodeId2();
			dof=rgb->GetDof();

			/*For this rgb, find the nodes that go with it: */
			node1=(Node*)nodes->GetObjectById(NULL,nodeid1);
			node2=(Node*)nodes->GetObjectById(NULL,nodeid2);

			if ((node1 && !node2) || (!node1 && node2)){
				/*we are missing one node, not good!*/
				ISSMERROR(exprintf("%s%p%s%p"," in Rgb, missing one node. node1: ",node1," node2: ",node2));
			}

			if(!node1 && !node2){
				/*That's ok, this Rgb can't find those nodes, so leave them alone. They are probably not on this 
				 * cpu!*/
			}
			else{
				/*Ok, this cpu owns both nodes. Put dof for node1 into m set, unless it is already there, 
				 * in which case node2 gets into the m set: */
				if(node1->DofIsInMSet(dof-1)){
					node2->DofInMSet(dof-1);
				}
				else{
					node1->DofInMSet(dof-1);
				}

				/*Plug values into Rmg. We essentially want dofs from node1 and node2 to be the 
				 *same: */
				dof1=node1->GetDof(dof-1); //matlab indexing
				dof2=node2->GetDof(dof-1); //matlab indexing

				MatSetValue(Rmg,count,dof1,1.0,INSERT_VALUES);
				MatSetValue(Rmg,count,dof2,-1.0,INSERT_VALUES);

			}
		}
	}
}
/*}}}*/
/*FUNCTION DataSet::SetupSpcs{{{1*/
void   DataSet::SetupSpcs(DataSet* nodes,Vec yg){

	vector<Object*>::iterator object;
	Spc* spc=NULL;
	Node* node=NULL;

	int nodeid;
	int dof;
	double value;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		/*Check this is a single point constraint (spc): */
		if((*object)->Enum()==SpcEnum()){

			spc=(Spc*)(*object);

			/*Ok, this object is a constraint. Get the nodeid from the node it applies to: */
			nodeid=spc->GetNodeId();
			dof=spc->GetDof();
			value=spc->GetValue();

			/*Now, chase through nodes and find the corect node: */
			node=(Node*)nodes->GetObjectById(NULL,nodeid);

			/*Apply constraint: */
			if(node){ //in case the spc is dealing with a node on another cpu
				node->ApplyConstraint(yg,dof,value);
			}

		}
	}

	/*Assemble yg: */
	VecAssemblyBegin(yg);
	VecAssemblyEnd(yg);
}
/*}}}*/
/*FUNCTION DataSet::SurfaceArea{{{1*/
void  DataSet::SurfaceArea(double* pS,void* inputs,int analysis_type,int sub_analysis_type){

	double S=0;;

	vector<Object*>::iterator object;
	Element* element=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if(EnumIsElement((*object)->Enum())){

			element=(Element*)(*object);
			S+=element->SurfaceArea(inputs,analysis_type,sub_analysis_type);

		}
	}

	/*Assign output pointers:*/
	*pS=S;

}
/*}}}*/
/*FUNCTION DataSet::UpdateFromInputs{{{1*/
void  DataSet::UpdateFromInputs(void* inputs){

	vector<Object*>::iterator object;
	Element* element=NULL;
	Load* load=NULL;
	Material* material=NULL;
	Node* node=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if(EnumIsElement((*object)->Enum())){

			element=(Element*)(*object);
			element->UpdateFromInputs(inputs);
		}
		if(EnumIsLoad((*object)->Enum())){

			load=(Load*)(*object);
			load->UpdateFromInputs(inputs);
		}
		if(EnumIsMaterial((*object)->Enum())){

			material=(Material*)(*object);
			material->UpdateFromInputs(inputs);
		}
		if((*object)->Enum()==NodeEnum()){
			node=(Node*)(*object);
			node->UpdateFromInputs(inputs);
		}		
	}

}
/*}}}*/
/*FUNCTION DataSet::UpdateNodePositions{{{1*/
void  DataSet::UpdateNodePositions(double* thickness,double* bed){

	vector<Object*>::iterator object;
	Node* node=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		if((*object)->Enum()==NodeEnum()){

			node=(Node*)(*object);
			node->UpdateNodePosition(thickness,bed);
		}
	}
}
/*}}}*/
