function [elements,grids,loads,constraints,materials,part,tpart]=ModelProcessorDiagnosticVert(md);
%MODELPROCESSORDIAGNOSTICVERT - process model for a vertical diagnostic solution
%
%   This routine uses all the informations in the model md and put them
%   into different structures (grids, elements, loads, constrained,materials)
%   that will be used to create the stiffness matrix and load vector.
%   After this routine, the model md should not be called until the end of the
%   solution sequence.
%
%   Usage:
%      [elements,grids,loads,constraints,materials,part,tpart]=ModelProcessorDiagnosticVert(md);

global cluster

if cluster,
	%We are running in parallel, we need to partition the elements 
	element_partitioning=MeshPartition(md,numlabs);
else
	%We are running in serial, all elements belong to the same partition.
	element_partitioning=ones(md.numberofelements,1);
	labindex=1; %older versions of matlab do not include the parallel toolbox labindex variable.
end

%Allocate grids and elements
elements=struct('element',cell(md.numberofelements,1));
materials=struct('material',cell(md.numberofelements+1,1));
mygrids=zeros(md.numberofgrids,1); %this array determines grid partitioning.

%3d elements
pos=find(element_partitioning==labindex);
[elements(pos).element]=deal(pentaelem);

elements(pos)=SetStructureField(elements(pos),'element','type','pentaelem');
elements(pos)=SetStructureField(elements(pos),'element','id',pos);
elements(pos)=SetStructureField(elements(pos),'element','g',md.elements(pos,1:6));
elements(pos)=SetStructureField(elements(pos),'element','h',md.thickness(md.elements(pos,1:6)));
elements(pos)=SetStructureField(elements(pos),'element','s',md.surface(md.elements(pos,1:6)));
elements(pos)=SetStructureField(elements(pos),'element','b',md.bed(md.elements(pos,1:6)));
elements(pos)=SetStructureField(elements(pos),'element','friction_type',md.drag_type);
elements(pos)=SetStructureField(elements(pos),'element','viscosity_overshoot',md.viscosity_overshoot);
elements(pos)=SetStructureField(elements(pos),'element','k',md.drag(md.elements(pos,1:6)));
elements(pos)=SetStructureField(elements(pos),'element','p',md.p(pos));
elements(pos)=SetStructureField(elements(pos),'element','q',md.q(pos));
elements(pos)=SetStructureField(elements(pos),'element','shelf',md.elementoniceshelf(pos));
elements(pos)=SetStructureField(elements(pos),'element','onbed',md.elementonbed(pos));
elements(pos)=SetStructureField(elements(pos),'element','onsurface',md.elementonsurface(pos));
elements(pos)=SetStructureField(elements(pos),'element','meanvel',md.meanvel);
elements(pos)=SetStructureField(elements(pos),'element','epsvel',md.epsvel);
elements(pos)=SetStructureField(elements(pos),'element','collapse',0);
elements(pos)=SetStructureField(elements(pos),'element','matid',pos);
elements(pos)=SetStructureField(elements(pos),'element','collapse',zeros(length(pos),1)); %no collapsed formulation
elements(pos)=SetStructureField(elements(pos),'element','melting',full(md.melting(md.elements(pos,1:6))/md.yts));
elements(pos)=SetStructureField(elements(pos),'element','accumulation',md.accumulation(md.elements(pos,1:6))/md.yts);

[materials(pos).material]=deal(matice);

materials(pos)=SetStructureField(materials(pos),'material','id',pos);
materials(pos)=SetStructureField(materials(pos),'material','B',md.B(md.elements(pos,1:6))*[1;1;1;1;1;1]/6);
materials(pos)=SetStructureField(materials(pos),'material','n',md.n(pos));

%Add physical constants in materials
[materials(end).constants]=deal(matpar);
materials(end)=SetStructureField(materials(end),'constants','g',md.g);
materials(end)=SetStructureField(materials(end),'constants','rho_ice',md.rho_ice);
materials(end)=SetStructureField(materials(end),'constants','rho_water',md.rho_water);
materials(end)=SetStructureField(materials(end),'constants','thermalconductivity',md.thermalconductivity);
materials(end)=SetStructureField(materials(end),'constants','heatcapacity',md.heatcapacity);
materials(end)=SetStructureField(materials(end),'constants','latentheat',md.latentheat);
materials(end)=SetStructureField(materials(end),'constants','beta',md.beta);
materials(end)=SetStructureField(materials(end),'constants','meltingpoint',md.meltingpoint);
materials(end)=SetStructureField(materials(end),'constants','mixed_layer_capacity',md.mixed_layer_capacity);
materials(end)=SetStructureField(materials(end),'constants','thermal_exchange_velocity',md.thermal_exchange_velocity);

if cluster, 
	%Figure out which grids from the partitioning belong to different element partitions. We'll 
	%call them 'border' grids.
	bordergrids=double(gplus(mygrids)>1);
else
	bordergrids=zeros(md.numberofgrids,1); %no partitioning serially.
end

%Get the grids set up:
grids=struct('grid',cell(md.numberofgrids,1));

pos=[1:md.numberofgrids]';
[grids(pos).grid]=deal(node);
grids(pos)=SetStructureField(grids(pos),'grid','id',pos);
grids(pos)=SetStructureField(grids(pos),'grid','x',md.x(pos));
grids(pos)=SetStructureField(grids(pos),'grid','y',md.y(pos));
grids(pos)=SetStructureField(grids(pos),'grid','z',md.z(pos));
grids(pos)=SetStructureField(grids(pos),'grid','onbed',md.gridonbed(pos));
grids(pos)=SetStructureField(grids(pos),'grid','border',bordergrids(pos));

%spc degrees of freedom:
%	 for each grid, 6 degrees of freedom are allowed. These dofs are numbered from 1 to 6. The first 3
%    deal with the (x,y,z) velocities, or deformations. The last 3 deal with the (x,y,z) rotations.
%    If a certain degree of freedom i (1<=i<=6) is constrained to the value 0, the number i should be added to the
%    gridset field of a grid.
%    The gridset field holds all the numbers corresponding to the dofs that have been constrained to 0 value. Because
%    we do not know firshand how many dofs have been constrained for a certain grid, we need a flexible way
%    of keeping track of these constraints. Hence gridset is a string array, of no given size, with no given
%    numerical order.
%    Ex: if a grid has 0 values for the x and z deformations, and 0 values for the y rotation, we could add any of the
%    following strings to the gridset: '135', '153', '315', etc ...
grids(pos)=SetStructureField(grids(pos),'grid','gridset','12456');

%Boundary conditions:

%penalties: not applied here
loads=struct('load',cell([0,1]));

%No single constraints
constraints=struct('constraint',cell(0,0));

%Last thing, return a partitioning vector, and its transpose.
[part,tpart]=PartitioningVector(md,grids);

end %end function
