/* \file WriteData.c:
 * \brief: general interface for writing data
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "../include/macros.h"
#include "./shared/shared.h"
#include "./io.h"

#undef __FUNCT__ 
#define __FUNCT__  "WriteData"

#ifdef _SERIAL_

#include <mex.h>
#include <assert.h>

/*Several prototypes for WriteData, according to type: */

/*DataSet: */
void WriteData(mxArray** pdataref,DataSet* dataset){

	mxArray* dataref=NULL;
	char* marshalled_dataset=NULL;
	int   marshalled_dataset_size;

	/*Write a dataset: */
	if(dataset){
			/* marshall the dataset: */
			marshalled_dataset=dataset->Marshall();
			marshalled_dataset_size=dataset->MarshallSize();
			
			dataref = mxCreateDoubleMatrix(0,0,mxREAL);
			mxSetM(dataref,(mwSize)(marshalled_dataset_size/sizeof(double)));
			mxSetN(dataref,(mwSize)1);
			mxSetPr(dataref,(double*)marshalled_dataset);	
	}
	else{
		/* return empty matrix: */
		dataref=mxCreateDoubleMatrix(0,0,mxREAL);
	}
	*pdataref=dataref;
	
}

void WriteData(mxArray** pdataref,Mat matrix){
		
	mxArray* dataref=NULL;
	
	if(matrix){
		
		/*call toolkit routine: */
		PetscMatrixToMatlabMatrix(&dataref,matrix);
	}
	else{
		dataref = mxCreateDoubleMatrix(0,0,mxREAL);
	}

	*pdataref=dataref;
}

void WriteData(mxArray** pdataref,double* matrix, int M,int N){
	
	mxArray* dataref=NULL;
	mxArray* tdataref=NULL;
		
	if(matrix){
		
		/*data is a double* pointer. Copy into a matrix: */
		tdataref = mxCreateDoubleMatrix(0,0,mxREAL);
		mxSetM(tdataref,(mwSize)N);
		mxSetN(tdataref,(mwSize)M);
		mxSetPr(tdataref,(double*)matrix);

		//transpose
		mexCallMATLAB(1,&dataref,1,&tdataref, "'");
	}
	else{
		dataref = mxCreateDoubleMatrix(0,0,mxREAL);
	}
	*pdataref=dataref;
}


void WriteData(mxArray** pdataref,Vec vector){
	
	mxArray* dataref=NULL;
	
	if(vector){
		
		/*call toolkit routine: */
		PetscVectorToMatlabVector(&dataref,vector);
	}
	else{
		dataref = mxCreateDoubleMatrix(0,0,mxREAL);
	}
	*pdataref=dataref;

}


void WriteData(mxArray** pdataref,DofVec* dofvec){
	
	mxArray* mxvector=NULL;
	mxArray* structdataref=NULL;
	mxArray* dataref=NULL;

	int nfields=4;
	const	char*	fnames[nfields];
	mwSize		onebyone[2] = {1,1};
	mwSize		ndim=2;

	fnames[0] = "name";
	fnames[1] = "numdofs";
	fnames[2] = "numentries";
	fnames[3] = "vector";
	
	if(dofvec){
		
		/*call toolkit routine: */
		if(dofvec->vector)PetscVectorToMatlabVector(&mxvector,dofvec->vector);
		else mxvector=mxCreateDoubleMatrix(0,0,mxREAL);

		/*use the mxvector to create a dofvec object: */
		structdataref=mxCreateStructArray( ndim,onebyone,nfields,fnames);
	
		mxSetField( structdataref, 0, "name",mxCreateString(dofvec->name));
		mxSetField( structdataref, 0, "numdofs",mxCreateDoubleScalar(dofvec->numdofs));
		mxSetField( structdataref, 0, "numentries",mxCreateDoubleScalar(dofvec->numentries));
		mxSetField( structdataref, 0, "vector",mxvector);


		/*transform structure into dofvec class: */
		mexCallMATLAB( 1, &dataref, 1, &structdataref, "dofvec");
	}
	else{
		dataref = mxCreateDoubleMatrix(0,0,mxREAL);
	}
	*pdataref=dataref;
}

void WriteData(mxArray** pdataref,double* vector, int M){
	
	mxArray* dataref=NULL;

	if(vector){

		/*data is a double* pointer. Copy into a vector: */
		dataref = mxCreateDoubleMatrix(0,0,mxREAL);
		mxSetM(dataref,(mwSize)M);
		mxSetN(dataref,(mwSize)1);
		mxSetPr(dataref,vector);
	}
	else{
		dataref = mxCreateDoubleMatrix(0,0,mxREAL);
	}

	*pdataref=dataref;
}

void WriteData(mxArray** pdataref,double scalar){

	*pdataref=mxCreateDoubleScalar(scalar);
}

void WriteData(mxArray** pdataref,int integer){

		*pdataref=mxCreateDoubleScalar((double)integer);

}

void WriteData(mxArray** pdataref,char* string){

		*pdataref=mxCreateString(string);
}

void WriteData(mxArray** pbamgmesh_mat,BamgMesh* bm){

	/*Output*/
	mxArray*    bamgmesh_mat=NULL;

	/*Intermediary*/
	int         i,j;
	mxArray*    pfield=NULL;
	mxArray*    pfield2=NULL;
	int         numfields=15;
	const char* fnames[numfields];
	int         fsize[numfields][2];
	double**    fpointer[numfields];
	mwSize      ndim=2;
	mwSize      dimensions[2]={1,1};

	/*Build fnames and fsize names and sizes of each field*/
	i=-1;
	fnames[++i] = "Triangles";                fsize[i][0]=bm->TrianglesSize[0];                 fsize[i][1]=bm->TrianglesSize[1];                fpointer[i]=&bm->Triangles;
	fnames[++i] = "Vertices";                 fsize[i][0]=bm->VerticesSize[0];                  fsize[i][1]=bm->VerticesSize[1];                 fpointer[i]=&bm->Vertices;
	fnames[++i] = "Edges";                    fsize[i][0]=bm->EdgesSize[0];                     fsize[i][1]=bm->EdgesSize[1];                    fpointer[i]=&bm->Edges;
	fnames[++i] = "Segments";                 fsize[i][0]=bm->SegmentsSize[0];                  fsize[i][1]=bm->SegmentsSize[1];                 fpointer[i]=&bm->Segments;
	fnames[++i] = "ElementEdges";             fsize[i][0]=bm->ElementEdgesSize[0];              fsize[i][1]=bm->ElementEdgesSize[1];             fpointer[i]=&bm->ElementEdges;
	fnames[++i] = "CrackedEdges";             fsize[i][0]=bm->CrackedEdgesSize[0];              fsize[i][1]=bm->CrackedEdgesSize[1];             fpointer[i]=&bm->CrackedEdges;
	fnames[++i] = "Quadrilaterals";           fsize[i][0]=bm->QuadrilateralsSize[0];            fsize[i][1]=bm->QuadrilateralsSize[1];           fpointer[i]=&bm->Quadrilaterals;
	fnames[++i] = "VerticesOnGeometricVertex";fsize[i][0]=bm->VerticesOnGeometricVertexSize[0]; fsize[i][1]=bm->VerticesOnGeometricVertexSize[1];fpointer[i]=&bm->VerticesOnGeometricVertex;
	fnames[++i] = "VerticesOnGeometricEdge";  fsize[i][0]=bm->VerticesOnGeometricEdgeSize[0];   fsize[i][1]=bm->VerticesOnGeometricEdgeSize[1];  fpointer[i]=&bm->VerticesOnGeometricEdge;
	fnames[++i] = "EdgesOnGeometricEdge";     fsize[i][0]=bm->EdgesOnGeometricEdgeSize[0];      fsize[i][1]=bm->EdgesOnGeometricEdgeSize[1];     fpointer[i]=&bm->EdgesOnGeometricEdge;
	fnames[++i] = "SubDomains";               fsize[i][0]=bm->SubDomainsSize[0];                fsize[i][1]=bm->SubDomainsSize[1];               fpointer[i]=&bm->SubDomains;
	fnames[++i] = "SubDomainsFromGeom";       fsize[i][0]=bm->SubDomainsFromGeomSize[0];        fsize[i][1]=bm->SubDomainsFromGeomSize[1];       fpointer[i]=&bm->SubDomainsFromGeom;
	fnames[++i] = "ElementConnectivity";      fsize[i][0]=bm->ElementConnectivitySize[0];       fsize[i][1]=bm->ElementConnectivitySize[1];      fpointer[i]=&bm->ElementConnectivity;
	fnames[++i] = "NodalConnectivity";        fsize[i][0]=bm->NodalConnectivitySize[0];         fsize[i][1]=bm->NodalConnectivitySize[1];        fpointer[i]=&bm->NodalConnectivity;
	fnames[++i] = "NodalElementConnectivity"; fsize[i][0]=bm->NodalElementConnectivitySize[0];  fsize[i][1]=bm->NodalElementConnectivitySize[1]; fpointer[i]=&bm->NodalElementConnectivity;
	assert(i==numfields-1);

	/*Initialize Matlab structure*/
	bamgmesh_mat=mxCreateStructArray(ndim,dimensions,numfields,fnames);

	/*Add every field to structure*/
	for(i=0;i<numfields;i++){
		pfield=mxCreateDoubleMatrix(0,0,mxREAL);
		mxSetM(pfield,fsize[i][1]);
		mxSetN(pfield,fsize[i][0]);
		mxSetPr(pfield,*(fpointer[i]));
		mexCallMATLAB(1,&pfield2,1,&pfield,"transpose");//transpose
		mxSetField(bamgmesh_mat,0,fnames[i],pfield2);
	}

	/*Assign output pointer*/
	*pbamgmesh_mat=bamgmesh_mat;
}

void WriteData(mxArray** pbamggeom_mat, BamgGeom* bg){

	/*Output*/
	mxArray*    bamggeom_mat=NULL;

	/*Intermediary*/
	int         i,j;
	mxArray*    pfield=NULL;
	mxArray*    pfield2=NULL;
	int         numfields=7;
	const char* fnames[numfields];
	int         fsize[numfields][2];
	double**    fpointer[numfields];
	mwSize      ndim=2;
	mwSize      dimensions[2]={1,1};

	/*Build fnames and fsize names and sizes of each field*/
	i=-1;
	fnames[++i] = "Vertices";        fsize[i][0]=bg->VerticesSize[0];        fsize[i][1]=bg->VerticesSize[1];        fpointer[i]=&bg->Vertices;
	fnames[++i] = "Edges";           fsize[i][0]=bg->EdgesSize[0];           fsize[i][1]=bg->EdgesSize[1];           fpointer[i]=&bg->Edges;
	fnames[++i] = "TangentAtEdges";  fsize[i][0]=bg->TangentAtEdgesSize[0];  fsize[i][1]=bg->TangentAtEdgesSize[1];  fpointer[i]=&bg->TangentAtEdges;
	fnames[++i] = "RequiredVertices";fsize[i][0]=bg->RequiredVerticesSize[0];fsize[i][1]=bg->RequiredVerticesSize[1];fpointer[i]=&bg->RequiredVertices;
	fnames[++i] = "RequiredEdges";   fsize[i][0]=bg->RequiredEdgesSize[0];   fsize[i][1]=bg->RequiredEdgesSize[1];   fpointer[i]=&bg->RequiredEdges;
	fnames[++i] = "CrackedEdges";    fsize[i][0]=bg->CrackedEdgesSize[0];    fsize[i][1]=bg->CrackedEdgesSize[1];    fpointer[i]=&bg->CrackedEdges;
	fnames[++i] = "SubDomains";      fsize[i][0]=bg->SubDomainsSize[0];      fsize[i][1]=bg->SubDomainsSize[1];      fpointer[i]=&bg->SubDomains;
	assert(i==numfields-1);

	/*Initialize Matlab structure*/
	bamggeom_mat=mxCreateStructArray(ndim,dimensions,numfields,fnames);

	/*Add every field tu structure*/
	for(i=0;i<numfields;i++){
		pfield=mxCreateDoubleMatrix(0,0,mxREAL);
		mxSetM(pfield,fsize[i][1]);
		mxSetN(pfield,fsize[i][0]);
		mxSetPr(pfield,*(fpointer[i]));
		mexCallMATLAB(1,&pfield2,1,&pfield,"transpose");//transpose
		mxSetField(bamggeom_mat,0,fnames[i],pfield2);
	}

	/*Assign output pointer*/
	*pbamggeom_mat=bamggeom_mat;
}

#else
void WriteData(int* pdummy,void* data,char* data_type){
	
	/*In parallel mode, WriteData is not used, instead we access the data directly through pointers in the solution sequences. */
}
#endif
