function md=bamg(md,varargin)
%BAMG - mesh generation
%
%   Available options (for more details see ISSM website http://issm.jpl.nasa.gov/):
%
%   - domain: followed by an ARGUS file that prescribes the domain outline (rifts and holes)
%   - hmin  : minimum edge length (default is 10^-100)
%   - hmax  : maximum esge length (default is 10^100)
%   - hVertices   : imposed edge length for each vertex (geometry or mesh)
%   - hminVertices: minimum edge length for each vertex (mesh)
%   - hmaxVertices: maximum edge length for each vertex (mesh)
%
%   - anisomax    : maximum ration between the smallest and largest edges (default is 10^30)
%   - coef        : coefficient applied to the metric (2-> twice as many elements, default is 1)
%   - cutoff      : scalar used to compute the metric when metric type 2 or 3 are applied
%   - err         : error used to generate the metric from a field
%   - errg        : geometrical error (default is 0.1)
%   - field       : field of the model that will be used to compute the metric
%                   to apply several fields, use one column per field
%   - gradation   : maximum ration between two adjacent edges
%   - Hessiantype : 0 -> use double P2 projection (default)
%                   1 -> use Green formula
%   - KeepVertices: try to keep initial vertices when adaptation is done on an existing mesh (default 1)
%   - MaximalAngleOfCorner: maximal angle of corners in degree (default is 10)
%   - maxnbv      : maximum number of vertices used to allocate memory (default is 10^6)
%   - maxsubdiv   : maximum subdivision of exisiting elements (default is 10)
%   - metric      : matrix (numberofgrids x 3) used as a metric
%   - Metrictype  : 1 -> absolute error          c/(err coeff^2) * Abs(H)        (default)
%                   2 -> relative error          c/(err coeff^2) * Abs(H)/max(s,cutoff*max(s))
%                   3 -> rescaled absolute error c/(err coeff^2) * Abs(H)/(smax-smin)
%   - nbjacoby    : correction used by Hessiantype=1 (default is 1)
%   - NbSmooth    : number of metric smoothing procedure (default is 3)
%   - omega       : relaxation parameter of the smoothing procedure (default is 1.8)
%   - power       : power applied to the metric (default is 1)
%   - splitcorners : split triangles whuch have 3 vertices on the outline (default is 1)
%   - geometricalmetric : Take the geometry into account to generate the metric (default is 0)
%   - verbose     : level of verbosity (default is 1)
%
%   Examples:
%      md=bamg(md,'domain','DomainOutline.exp','hmax',3000);
%      md=bamg(md,'field',[md.vel_obs md.thickness],'hmax',20000,'hmin',1000);
%      md=bamg(md,'metric',A,'hmin',1000,'hmax',20000,'gradation',3,'anisomax',1);

%process options
options=pairoptions(varargin{:});
options=deleteduplicates(options,1);

%initialize the structures required as input of Bamg
bamg_options=struct();
bamg_geometry=bamggeom;
bamg_mesh=bamgmesh;

% Bamg Geometry parameters {{{1
if exist(options,'domain'),

	%Check that file exists
	domainfile=getfieldvalueerr(options,'domain');
	if ~exist(domainfile,'file')
		error(['bamg error message: file ' domainfile ' not found ']);
	end

	%Build geometry 
	domain=expread(domainfile);
	count=0;
	for i=1:length(domain),

		%Check that the domain is closed
		if (domain(i).x(1)~=domain(i).x(end) | domain(i).y(1)~=domain(i).y(end)),
			error('bamg error message: all contours provided in ''domain'' should be closed');
		end
		%Checks that all holes are INSIDE the principla domain outline
		if i>1,
			flags=ContourToNodes(domain(i).x,domain(i).y,domain(1),0);
			if any(~flags),
				error('bamg error message: All holes should be stricly inside the principal domain');
			end
		end

		%Add all points to bamg_geometry
		nods=domain(i).nods-1; %the domain are closed 1=end;
		bamg_geometry.Vertices=[bamg_geometry.Vertices; [domain(i).x(1:nods) domain(i).y(1:nods) ones(nods,1)]];
		bamg_geometry.Edges=[bamg_geometry.Edges; [transpose(count+1:count+nods) transpose([count+2:count+nods count+1])  1*ones(nods,1)]];

		%update counter
		count=count+nods;
	end

	%take care of rifts
	if exist(options,'rifts'),

		%Check that file exists
		riftfile=getfieldvalueerr(options,'rifts');
		if ~exist(riftfile,'file')
			error(['bamg error message: file ' riftfile ' not found ']);
		end
		rift=expread(riftfile);

		for i=1:length(rift),

			%detect wether all points of the rift are inside the domain
			flags=ContourToNodes(rift(i).x,rift(i).y,domain(1),0);
			if ~flags,
				error('one Rift has all his points outside of the domain outline'),

			elseif any(~flags),
				%We LOTS of work to do
				disp('Rift tip outside of or on the domain has been detected and is being processed...');

				%check that only one point is outsie (for now)
				if sum(~flags)~=1,
					error('bamg error message: only one point outside of the domain is supported yet');
				end

				%Move tip outside to the first position
				if flags(1)==0,
					%OK, first point is outside (do nothing),
				elseif (flags(end)==0),
					rift(i).x=flipud(rift(i).x);
					rift(i).y=flipud(rift(i).y);
				else
					error('bamg error message: only a rift tip can be outside of the domain');
				end

				%Get cordinate of intersection point
				x1=rift(i).x(1); y1=rift(i).y(1);
				x2=rift(i).x(2); y2=rift(i).y(2);
				for j=1:length(domain(1).x)-1;
					if SegIntersect([x1 y1; x2 y2],[domain(1).x(j) domain(1).y(j); domain(1).x(j+1) domain(1).y(j+1)]),

						%Get position of the two nodes of the edge in domain
						i1=j;
						i2=mod(j+1,domain(1).nods);

						%rift is crossing edge [i1 i2] of the domain
						%Get coordinate of intersection point (http://mathworld.wolfram.com/Line-LineIntersection.html)
						x3=domain(1).x(i1); y3=domain(1).y(i1);
						x4=domain(1).x(i2); y4=domain(1).y(i2);
						x=det([det([x1 y1; x2 y2])  x1-x2;det([x3 y3; x4 y4])  x3-x4])/det([x1-x2 y1-y2;x3-x4 y3-y4]);
						y=det([det([x1 y1; x2 y2])  y1-y2;det([x3 y3; x4 y4])  y3-y4])/det([x1-x2 y1-y2;x3-x4 y3-y4]);

						%Add intersection point to Vertices
						bamg_geometry.Vertices=[bamg_geometry.Vertices; x y 1];
						count=count+1;

						%Decompose the crossing edge in 2 subedges
						pos=find(bamg_geometry.Edges(:,1)==i1 & bamg_geometry.Edges(:,2)==i2);
						if isempty(pos) error('bamg error message: a problem occured...'); end
						bamg_geometry.Edges=[bamg_geometry.Edges(1:pos-1,:);...
							bamg_geometry.Edges(pos,1) count                           bamg_geometry.Edges(pos,3);...
							count                      bamg_geometry.Edges(pos,2)   bamg_geometry.Edges(pos,3);...
							bamg_geometry.Edges(pos+1:end,:)];

						%OK, no we can add our own rift
						nods=rift(i).nods-1;
						bamg_geometry.Vertices=[bamg_geometry.Vertices; [rift(i).x(2:end) rift(i).y(2:end) ones(nods,1)]];
						bamg_geometry.Edges=[bamg_geometry.Edges;...
							count  count+1  2 ;...
							[transpose(count+1:count+nods-1) transpose([count+2:count+nods])  2*ones(nods-1,1)]];
						count=count+nods;

						break;
					end
				end
			else
				nods=rift(i).nods-1;
				bamg_geometry.Vertices=[bamg_geometry.Vertices; [rift(i).x(:) rift(i).y(:) ones(nods+1,1)]];
				bamg_geometry.Edges=[bamg_geometry.Edges; [transpose(count+1:count+nods) transpose([count+2:count+nods+1])  2*ones(nods,1)]];
				count=count+nods+1;
			end
		end
	end

	if exist(options,'hVertices'),
		bamg_geometry.hVertices=getfieldvalueerr(options,'hVertices');
		if length(bamg_geometry.hVertices)~=nods,
			error(['hVertices option should be of size [' num2str(nods) ',1]']);
		end
	end

elseif isstruct(md.bamg),

	bamg_geometry=bamggeom(md.bamg.geometry); %TEST

else

	%do nothing...

end

%update other fields of bamg_geometry
bamg_geometry.NumVertices=size(bamg_geometry.Vertices,1);
bamg_geometry.NumEdges=size(bamg_geometry.Edges,1);
bamg_geometry.NumSubDomains=size(bamg_geometry.SubDomains,1);
%}}}

% Bamg Mesh parameters {{{1
if (~exist(options,'domain') & md.numberofgrids~=0 & strcmpi(md.type,'2d')),

	if isstruct(md.bamg),
		bamg_mesh=bamgmesh(md.bamg.mesh);
	else
		bamg_mesh.NumVertices=md.numberofgrids;
		bamg_mesh.Vertices=[md.x md.y ones(md.numberofgrids,1)];
		bamg_mesh.NumTriangles=md.numberofelements;
		bamg_mesh.Triangles=[md.elements ones(md.numberofelements,1)];
	end

	if exist(options,'hVertices'),
		bamg_mesh.hVertices=getfieldvalueerr(options,'hVertices');
		if length(bamg_mesh.hVertices)~=md.numberofgrids,
			error(['hVertices option should be of size [' num2str(md.numberofgrids) ',1]']);
		end
	end

	if isstruct(md.rifts)
		error('bamg error message: rifts not supported yet. Do meshprocessrift AFTER bamg');
	end
end
%}}}

% Bamg Options {{{1
bamg_options.anisomax=getfieldvalue(options,'anisomax',10^30);
bamg_options.coef=getfieldvalue(options,'coef',1);
bamg_options.cutoff=getfieldvalue(options,'cutoff',10^-5);
bamg_options.err=getfieldvalue(options,'err',0.01);
bamg_options.errg=getfieldvalue(options,'errg',0.1);
bamg_options.field=getfieldvalue(options,'field',[]);
bamg_options.gradation=getfieldvalue(options,'gradation',1.5);
bamg_options.Hessiantype=getfieldvalue(options,'Hessiantype',0);
bamg_options.hmin=getfieldvalue(options,'hmin',10^-100);
bamg_options.hmax=getfieldvalue(options,'hmax',10^100);
bamg_options.hminVertices=getfieldvalue(options,'hminVertices',[]);
bamg_options.hmaxVertices=getfieldvalue(options,'hmaxVertices',[]);
bamg_options.KeepVertices=getfieldvalue(options,'KeepVertices',1);
bamg_options.MaximalAngleOfCorner=getfieldvalue(options,'MaximalAngleOfCorner',10);
bamg_options.maxnbv=getfieldvalue(options,'maxnbv',10^6);
bamg_options.maxsubdiv=getfieldvalue(options,'maxsubdiv',10);
bamg_options.metric=getfieldvalue(options,'metric',[]);
bamg_options.Metrictype=getfieldvalue(options,'Metrictype',0);
bamg_options.nbjacobi=getfieldvalue(options,'nbjacobi',1);
bamg_options.NbSmooth=getfieldvalue(options,'NbSmooth',3);
bamg_options.omega=getfieldvalue(options,'omega',1.8);
bamg_options.power=getfieldvalue(options,'power',1);
bamg_options.splitcorners=getfieldvalue(options,'splitcorners',1);
bamg_options.geometricalmetric=getfieldvalue(options,'geometricalmetric',0);
bamg_options.verbose=getfieldvalue(options,'verbose',1);
%}}}

%call Bamg
[bamgmesh_out bamggeom_out]=Bamg(bamg_mesh,bamg_geometry,bamg_options);

% plug results onto model
md.bamg=struct();
md.bamg.mesh=bamgmesh(bamgmesh_out);
md.bamg.geometry=bamggeom(bamggeom_out);
md.x=bamgmesh_out.Vertices(:,1);
md.y=bamgmesh_out.Vertices(:,2);
md.elements=bamgmesh_out.Triangles(:,1:3);
md.segments=bamgmesh_out.Segments;
md.segmentmarkers=bamgmesh_out.SegmentsMarkers;

%Fill in rest of fields:
md.type='2d';
md.numberofelements=length(md.elements);
md.numberofgrids=length(md.x);
md.z=zeros(md.numberofgrids,1);
md.gridonbed=ones(md.numberofgrids,1);
md.gridonwater=zeros(md.numberofgrids,1);
md.gridonsurface=ones(md.numberofgrids,1);
md.elementonbed=ones(md.numberofelements,1);
md.elementonsurface=ones(md.numberofelements,1);
md.gridonboundary=zeros(md.numberofgrids,1); md.gridonboundary(md.segments(:,1:2))=1;
md.counter=1;
