/*!\file Bamgx
 * \brief: use Bamg capabilities.
 */

#include "./Bamgx.h"

#undef __FUNCT__ 
#define __FUNCT__ "Bamgx"

#include "../shared/shared.h"
#include "../include/macros.h"
#include "../toolkits/toolkits.h"

/*Bamg: */
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "BamgObjects.h"
#include "objects/QuadTree.h"

using namespace bamg;
using namespace std;

int Bamgx(BamgMesh* bamgmesh_out,BamgGeom* bamggeom_out,BamgMesh* bamgmesh_in,BamgGeom* bamggeom_in,BamgOpts* bamgopts){

	/*Bamg options*/
	double anisomax;
	int    maxnbv;
	double hmin,hmax;
	double coef,maxsubdiv;
	int    verbosity;
	int    NbSmooth;
	double omega;

	/*intermediary*/
	int noerr=1;
	int i,j,num;
	int allquad=0;
	double costheta=2;
	double hminaniso=1e-100; 

	Triangles* Thr=NULL;
	Triangles* Thb=NULL;

	/*Bamg options*/
	anisomax=bamgopts->anisomax;
	NbSmooth=bamgopts->NbSmooth;
	omega=bamgopts->omega;
	coef=bamgopts->coef;
	maxnbv=bamgopts->maxnbv;
	maxsubdiv=bamgopts->maxsubdiv;
	hmin=bamgopts->hmin;
	hmax=bamgopts->hmax;
	verbosity=bamgopts->verbose;

	// some verification
	if ( maxsubdiv > 10 || maxsubdiv <= 1.0){
		throw ErrorException(__FUNCT__,exprintf("maxsubdiv (%g) should be in ]1,10]",maxsubdiv));
	}
	// no metric -> no smoothing 
	if (bamgopts->metric==NULL){
		NbSmooth=0; 
	}

	/*If no mesh in input, generate one*/

	if(bamgmesh_in->NumTriangles==0){
		/*Mesh generation {{{1*/

		if (verbosity>0) printf("Construction of a mesh from a given geometry\n");
		if (verbosity>1) printf("   Processing geometry...\n");
		Geometry Gh(bamggeom_in,bamgopts);
		if (verbosity>10){
			Gh.Echo();
		}

		//get hmin and hmax from geometry to generate the metric
		hmin = Max(hmin,Gh.MinimalHmin());
		hmax = Min(hmax,Gh.MaximalHmax());

		//build metric if not given in input
		if (verbosity>1) printf("   Generating Metric...\n");
		for(i=0;i<Gh.nbv;i++){
			Metric M=Gh[i];
			MatVVP2x2 Vp(M/coef);
			Vp.Maxh(hmax);
			Vp.Minh(hmin);
			Gh.vertices[i].m = Vp;
		}

		//generate mesh
		if (verbosity>1) printf("   Generating Mesh...\n");
		Triangles Th(maxnbv,Gh,bamgopts);
		if(bamgopts->splitcorners) Th.SplitInternalEdgeWithBorderVertices();
		Th.ReNumberingTheTriangleBySubDomain();

		//Build output
		if (verbosity>1) printf("   Write Mesh...\n");
		Th.WriteMesh(bamgmesh_out,bamgopts);
		if (verbosity>1) printf("   Write Geometry...\n");
		Gh.WriteGeometry(bamggeom_out,bamgopts);

		//clean up
	//	delete &Th;
	//	delete &Gh;
		/*}}}*/
	}

	else{
		/*Anisotropic mesh adaptation {{{1*/

		// read background mesh 
		if (verbosity>0) printf("Anisotropic mesh adaptation\n");
		if (verbosity>1) printf("   Processing initial mesh and geometry...\n");
		Triangles BTh(bamgmesh_in,bamgopts); 
		hmin = Max(hmin,BTh.MinimalHmin());
		hmax = Min(hmax,BTh.MaximalHmax());

		//Make Quadtree from background mesh
		BTh.MakeQuadTree();

		//build metric if not given in input
		if (bamgopts->metric){
			if (verbosity>1) printf("   Processing Metric...\n");
			BTh.ReadMetric(bamgopts,hmin,hmax,coef);
		}
		else { 

			// init with hmax 
			Metric Mhmax(hmax);
			for (int iv=0;iv<BTh.nbv;iv++) BTh[iv].m = Mhmax;
			// change using hVertices if required
			if (bamgmesh_in->hVertices){
				for (i=0;i<BTh.nbv;i++){
					if (!isnan(bamgmesh_in->hVertices[i])){
						BTh[i].m=Metric((float)bamgmesh_in->hVertices[i]);
					}
				}
			}

			//use present fields to generate metric if present
			if (bamgopts->field){
				if (verbosity>1) printf("   Generating Metric from solution field...\n");
				BTh.AddMetric(bamgopts);
			}
		}

		BTh.AddGeometryMetric(bamgopts);
		if (verbosity>1) printf("   Smoothing Metric...");
		if(bamgopts->gradation) BTh.SmoothMetric(bamgopts->gradation);
		if (verbosity>1) printf(" done\n");
		BTh.MaxSubDivision(maxsubdiv);
		BTh.BoundAnisotropy(anisomax,hminaniso);

		//Build new mesh Thr
		if (verbosity>1) printf("   Generating Mesh...\n");
		Thr=&BTh,Thb=0;
		Triangles & Th( *(0 ?  new Triangles(*Thr,&Thr->Gh,Thb,maxnbv) :  new Triangles(maxnbv,BTh,bamgopts,bamgopts->KeepVertices)));
		if (Thr != &BTh) delete Thr;
		if(costheta<=1.0) Th.MakeQuadrangles(costheta);
		if (allquad) Th.SplitElement(allquad==2);
		if(bamgopts->splitcorners) Th.SplitInternalEdgeWithBorderVertices();
		Th.ReNumberingTheTriangleBySubDomain();
		if(NbSmooth>0) Th.SmoothingVertex(NbSmooth,omega);

		//display info
		if(verbosity>0) {
			if (Th.nbt-Th.NbOutT-Th.NbOfQuad*2){
				printf("   new number of triangles = %i\n",(Th.nbt-Th.NbOutT-Th.NbOfQuad*2));
			}
			if (Th.NbOfQuad ){
				printf("   new number of quads = %i\n",Th.NbOfQuad);
			}
		}

		//Build output
		if (verbosity>1) printf("   Write Mesh...\n");
		Th.WriteMesh(bamgmesh_out,bamgopts);
		if (verbosity>1) printf("   Write Geometry...\n");
		Th.Gh.WriteGeometry(bamggeom_out,bamgopts);
		if (verbosity>1) printf("   Write Metric...\n");
		BTh.WriteMetric(bamgopts);

		/*clean up*/
		delete &Th;
		//delete &BTh;
		/*}}}*/
	}

	/*No error return*/
	if (verbosity>1) printf("   Exiting Bamg.\n");
	return noerr;

}
