#!/usr/bin/env python
from __future__ import generators
import user
import config.base
import os
import PETSc.package

class Configure(PETSc.package.Package):
  def __init__(self, framework):
    PETSc.package.Package.__init__(self, framework)
    self.download          = ['http://www.columbia.edu/~ma2325/Prometheus-1.8.7.tar.gz']
    self.functions         = []
    self.includes          = []
    self.liblist           = [['libpromfei.a','libprometheus.a']]
    self.compilePrometheus = 0
    return

  def setupDependencies(self, framework):
    PETSc.package.Package.setupDependencies(self, framework)
    self.mpi        = framework.require('config.packages.MPI',self)
    self.blasLapack = framework.require('config.packages.BlasLapack',self)
    self.parmetis   = framework.require('PETSc.packages.ParMetis',self)
    self.deps       = [self.parmetis,self.mpi,self.blasLapack]
    return

  def generateLibList(self,dir):
    '''Normally the one in package.py is used, but Prometheus requires the extra C++ library'''
    alllibs = PETSc.package.Package.generateLibList(self,dir)
    import config.setCompilers
    if self.languages.clanguage == 'C':
      alllibs[0].extend(self.compilers.cxxlibs)
    return alllibs

  def Install(self):

    args = ''
    args += 'SHELL          = '+self.programs.SHELL+'\n'
    args += 'CP             = '+self.programs.cp+'\n'
    args += 'RM             = '+self.programs.RM+'\n'
    args += 'MKDIR          = '+self.programs.mkdir+'\n'


    args += 'PREFIX         = '+self.installDir+'\n'
    args += 'BUILD_DIR      = '+self.packageDir+'\n'
    args += 'LIB_DIR        = $(BUILD_DIR)/lib/\n'
    args += 'PETSC_INCLUDE  = -I'+os.path.join(self.petscdir.dir,self.arch.arch,'include')+' -I'+os.path.join(self.petscdir.dir)+' -I'+os.path.join(self.petscdir.dir,'include')+' '+self.headers.toString(self.mpi.include+self.parmetis.include)+'\n'
    args += 'RANLIB         = '+self.setCompilers.RANLIB+'\n'
    args += 'AR             = '+self.setCompilers.AR+'\n'
    args += 'ARFLAGS        = '+self.setCompilers.AR_FLAGS+'\n'

    self.framework.pushLanguage('C++')
    args += 'CXX            = '+self.framework.getCompiler()
    args += ' -DPROM_HAVE_METIS'

    if self.blasLapack.mangling == 'underscore':
      args += ' -DHAVE_FORTRAN_UNDERSCORE=1'
      if self.compilers.fortranManglingDoubleUnderscore:
        args += ' -DHAVE_FORTRAN_UNDERSCORE_UNDERSCORE=1'
    elif self.blasLapack.mangling == 'unchanged':
      args += ' -DHAVE_FORTRAN_NOUNDERSCORE=1'
    elif self.blasLapack.mangling == 'caps':
      args += ' -DHAVE_FORTRAN_CAPS=1'
    elif self.blasLapack.mangling == 'stdcall':
      args += ' -DHAVE_FORTRAN_STDCALL=1'
      args += ' -DSTDCALL=__stdcall'
      args += ' -DHAVE_FORTRAN_CAPS=1'
      args += ' -DHAVE_FORTRAN_MIXED_STR_ARG=1'

    args += '\n'
    args += 'PETSCFLAGS     = '+self.framework.getCompilerFlags()+'\n'
    self.framework.popLanguage()
    fd = file(os.path.join(self.packageDir,'makefile.petsc'),'w')
    fd.write(args)
    fd.close()

    if self.installNeeded('makefile.petsc'):
      fd = file(os.path.join(self.packageDir,'makefile.in'),'a')
      fd.write('include makefile.petsc\n')
      fd.close()
      output  = config.base.Configure.executeShellCommand('cp -f '+os.path.join(self.packageDir,'makefile.petsc')+' '+self.confDir+'/Prometheus', timeout=5, log = self.framework.log)[0]
      self.compilePrometheus = 1
    return self.installDir

  def configureLibrary(self):
    '''Calls the regular package configureLibrary and then does an additional test needed by Prometheus'''
    '''Normally you do not need to provide this method'''
    PETSc.package.Package.configureLibrary(self)
    # Prometheus requires LAPACK routine dorgqr()
    if not self.blasLapack.checkForRoutine('dorgqr'):
      raise RuntimeError('Prometheus requires the LAPACK routine dorgqr(), the current Lapack libraries '+str(self.blasLapack.lib)+' does not have it\nIf you are using the IBM ESSL library, it does not contain this function. After installing a complete copy of lapack\n You can run config/configure.py with --with-blas-lib=libessl.a --with-lapack-lib=/usr/local/lib/liblapack.a')
    self.framework.log.write('Found dorgqr() in Lapack library as needed by Prometheus\n')
    return

  def postProcess(self):
    if self.compilePrometheus:
      try:
        self.logPrintBox('Compiling Prometheus; this may take several minutes')
        output  = config.base.Configure.executeShellCommand('cd '+self.packageDir+'; make clean cleanlib; make prom minstall',timeout=1000, log = self.framework.log)[0]
        self.framework.log.write(output)
      except RuntimeError, e:
        raise RuntimeError('Error running make on ParMetis: '+str(e))
      self.postInstall(output,'makefile.petsc')

if __name__ == '__main__':
  import config.framework
  import sys
  framework = config.framework.Framework(sys.argv[1:])
  framework.setupLogging(framework.clArgs)
  framework.children.append(Configure(framework))
  framework.configure()
  framework.dumpSubstitutions()
