/*!\file Penta.cpp
 * \brief: implementation of the Penta object
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include "./Penta.h"
#include <string.h>
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../shared/shared.h"
#include "../include/typedefs.h"

/*Object constructors and destructor*/
/*FUNCTION Penta constructor {{{1*/
Penta::Penta(){
	return;
}
/*}}}*/
/*FUNCTION Penta creation {{{1*/
Penta::Penta( int penta_id, int penta_mid, int penta_mparid, int penta_numparid, int penta_node_ids[6], double penta_h[6], double penta_s[6], double penta_b[6], double penta_k[6], int penta_friction_type, 
			double penta_p, double penta_q, int penta_shelf, int penta_onbed, int penta_onsurface,  int penta_collapse, double penta_melting[6], double penta_accumulation[6], double penta_geothermalflux[6], 
			int penta_thermal_steadystate,bool penta_onwater){

	int i;

	id = penta_id;
	mid = penta_mid; 
	mparid = penta_mparid; 
	numparid=penta_numparid;

	for(i =0;i<6;i++){
		node_ids[i] = penta_node_ids[i]; 
		node_offsets[i]=UNDEF;
		nodes[i]=NULL;
		h[i] = penta_h[i]; 
		s[i] = penta_s[i]; 
		b[i] = penta_b[i]; 
		k[i] = penta_k[i]; 
		melting[i] = penta_melting[i]; 
		accumulation[i] = penta_accumulation[i] ;
		geothermalflux[i] = penta_geothermalflux[i]; 
	}
	matice=NULL;
	matice_offset=UNDEF;
	matpar=NULL;
	matpar_offset=UNDEF;
	numpar=NULL;
	numpar_offset=UNDEF;

	friction_type = penta_friction_type; 
	p = penta_p; 
	q = penta_q; 
	shelf = penta_shelf; 
	onbed = penta_onbed; 
	onwater = penta_onwater; 
	onsurface = penta_onsurface; 
	collapse = penta_collapse; 
	thermal_steadystate = penta_thermal_steadystate;

	return;
}
/*}}}*/
/*FUNCTION Penta destructor {{{1*/
Penta::~Penta(){
	return;
}
/*}}}*/

/*Object marshall*/
/*FUNCTION Marshall {{{1*/
void  Penta::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_type=0;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of Penta: */
	enum_type=PentaEnum();

	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);

	/*marshall Penta data: */
	memcpy(marshalled_dataset,&id,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(marshalled_dataset,&mid,sizeof(mid));marshalled_dataset+=sizeof(mid);
	memcpy(marshalled_dataset,&mparid,sizeof(mparid));marshalled_dataset+=sizeof(mparid);
	memcpy(marshalled_dataset,&node_ids,sizeof(node_ids));marshalled_dataset+=sizeof(node_ids);
	memcpy(marshalled_dataset,&nodes,sizeof(nodes));marshalled_dataset+=sizeof(nodes);
	memcpy(marshalled_dataset,&node_offsets,sizeof(node_offsets));marshalled_dataset+=sizeof(node_offsets);
	memcpy(marshalled_dataset,&matice,sizeof(matice));marshalled_dataset+=sizeof(matice);
	memcpy(marshalled_dataset,&matice_offset,sizeof(matice_offset));marshalled_dataset+=sizeof(matice_offset);
	memcpy(marshalled_dataset,&matpar,sizeof(matpar));marshalled_dataset+=sizeof(matpar);
	memcpy(marshalled_dataset,&matpar_offset,sizeof(matpar_offset));marshalled_dataset+=sizeof(matpar_offset);
	memcpy(marshalled_dataset,&numparid,sizeof(numparid));marshalled_dataset+=sizeof(numparid);
	memcpy(marshalled_dataset,&numpar,sizeof(numpar));marshalled_dataset+=sizeof(numpar);
	memcpy(marshalled_dataset,&numpar_offset,sizeof(numpar_offset));marshalled_dataset+=sizeof(numpar_offset);
	memcpy(marshalled_dataset,&h,sizeof(h));marshalled_dataset+=sizeof(h);
	memcpy(marshalled_dataset,&s,sizeof(s));marshalled_dataset+=sizeof(s);
	memcpy(marshalled_dataset,&b,sizeof(b));marshalled_dataset+=sizeof(b);
	memcpy(marshalled_dataset,&k,sizeof(k));marshalled_dataset+=sizeof(k);
	memcpy(marshalled_dataset,&friction_type,sizeof(friction_type));marshalled_dataset+=sizeof(friction_type);
	memcpy(marshalled_dataset,&p,sizeof(p));marshalled_dataset+=sizeof(p);
	memcpy(marshalled_dataset,&q,sizeof(q));marshalled_dataset+=sizeof(q);
	memcpy(marshalled_dataset,&shelf,sizeof(shelf));marshalled_dataset+=sizeof(shelf);
	memcpy(marshalled_dataset,&onbed,sizeof(onbed));marshalled_dataset+=sizeof(onbed);
	memcpy(marshalled_dataset,&onwater,sizeof(onwater));marshalled_dataset+=sizeof(onwater);
	memcpy(marshalled_dataset,&onsurface,sizeof(onsurface));marshalled_dataset+=sizeof(onsurface);
	memcpy(marshalled_dataset,&collapse,sizeof(collapse));marshalled_dataset+=sizeof(collapse);
	memcpy(marshalled_dataset,&melting,sizeof(melting));marshalled_dataset+=sizeof(melting);
	memcpy(marshalled_dataset,&accumulation,sizeof(accumulation));marshalled_dataset+=sizeof(accumulation);
	memcpy(marshalled_dataset,&geothermalflux,sizeof(geothermalflux));marshalled_dataset+=sizeof(geothermalflux);
	memcpy(marshalled_dataset,&thermal_steadystate,sizeof(thermal_steadystate));marshalled_dataset+=sizeof(thermal_steadystate);

	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION MarshallSize {{{1*/
int   Penta::MarshallSize(){

	return sizeof(id)+
	  sizeof(mid)+
	  sizeof(mparid)+
	  sizeof(node_ids)+
	  sizeof(nodes)+
	  sizeof(node_offsets)+
	  sizeof(matice)+
	  sizeof(matice_offset)+
	  sizeof(matpar)+
	  sizeof(matpar_offset)+
	  +sizeof(numparid)+
	  +sizeof(numpar)+
	  +sizeof(numpar_offset)+
	  sizeof(h)+
	  sizeof(s)+
	  sizeof(b)+
	  sizeof(k)+
	  sizeof(friction_type)+
	  sizeof(p)+
	  sizeof(q)+
	  sizeof(shelf)+
	  sizeof(onbed)+
	  sizeof(onwater)+
	  sizeof(onsurface)+
	  sizeof(collapse)+
	  sizeof(melting)+
	  sizeof(accumulation)+
	  sizeof(geothermalflux)+
	  sizeof(thermal_steadystate) +
	  sizeof(int); //sizeof(int) for enum type
}
/*}}}*/
/*FUNCTION Demarshall {{{1*/
void  Penta::Demarshall(char** pmarshalled_dataset){

	int i;
	char* marshalled_dataset=NULL;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/

	memcpy(&id,marshalled_dataset,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(&mid,marshalled_dataset,sizeof(mid));marshalled_dataset+=sizeof(mid);
	memcpy(&mparid,marshalled_dataset,sizeof(mparid));marshalled_dataset+=sizeof(mparid);
	memcpy(&node_ids,marshalled_dataset,sizeof(node_ids));marshalled_dataset+=sizeof(node_ids);
	memcpy(&nodes,marshalled_dataset,sizeof(nodes));marshalled_dataset+=sizeof(nodes);
	memcpy(&node_offsets,marshalled_dataset,sizeof(node_offsets));marshalled_dataset+=sizeof(node_offsets);
	memcpy(&matice,marshalled_dataset,sizeof(matice));marshalled_dataset+=sizeof(matice);
	memcpy(&matice_offset,marshalled_dataset,sizeof(matice_offset));marshalled_dataset+=sizeof(matice_offset);
	memcpy(&matpar,marshalled_dataset,sizeof(matpar));marshalled_dataset+=sizeof(matpar);
	memcpy(&matpar_offset,marshalled_dataset,sizeof(matpar_offset));marshalled_dataset+=sizeof(matpar_offset);
	memcpy(&numparid,marshalled_dataset,sizeof(numparid));marshalled_dataset+=sizeof(numparid);
	memcpy(&numpar,marshalled_dataset,sizeof(numpar));marshalled_dataset+=sizeof(numpar);
	memcpy(&numpar_offset,marshalled_dataset,sizeof(numpar_offset));marshalled_dataset+=sizeof(numpar_offset);
	memcpy(&h,marshalled_dataset,sizeof(h));marshalled_dataset+=sizeof(h);
	memcpy(&s,marshalled_dataset,sizeof(s));marshalled_dataset+=sizeof(s);
	memcpy(&b,marshalled_dataset,sizeof(b));marshalled_dataset+=sizeof(b);
	memcpy(&k,marshalled_dataset,sizeof(k));marshalled_dataset+=sizeof(k);
	memcpy(&friction_type,marshalled_dataset,sizeof(friction_type));marshalled_dataset+=sizeof(friction_type);
	memcpy(&p,marshalled_dataset,sizeof(p));marshalled_dataset+=sizeof(p);
	memcpy(&q,marshalled_dataset,sizeof(q));marshalled_dataset+=sizeof(q);
	memcpy(&shelf,marshalled_dataset,sizeof(shelf));marshalled_dataset+=sizeof(shelf);
	memcpy(&onbed,marshalled_dataset,sizeof(onbed));marshalled_dataset+=sizeof(onbed);
	memcpy(&onwater,marshalled_dataset,sizeof(onwater));marshalled_dataset+=sizeof(onwater);
	memcpy(&onsurface,marshalled_dataset,sizeof(onsurface));marshalled_dataset+=sizeof(onsurface);
	memcpy(&collapse,marshalled_dataset,sizeof(collapse));marshalled_dataset+=sizeof(collapse);
	memcpy(&melting,marshalled_dataset,sizeof(melting));marshalled_dataset+=sizeof(melting);
	memcpy(&accumulation,marshalled_dataset,sizeof(accumulation));marshalled_dataset+=sizeof(accumulation);
	memcpy(&geothermalflux,marshalled_dataset,sizeof(geothermalflux));marshalled_dataset+=sizeof(geothermalflux);
	memcpy(&thermal_steadystate,marshalled_dataset,sizeof(thermal_steadystate));marshalled_dataset+=sizeof(thermal_steadystate);

	/*nodes and materials are not pointing to correct objects anymore:*/
	for(i=0;i<6;i++)nodes[i]=NULL;
	matice=NULL;
	matpar=NULL;
	numpar=NULL;

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/

/*Object functions*/
/*FUNCTION ComputePressure {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::ComputePressure"
void  Penta::ComputePressure(Vec pg){

	int i;
	const int numgrids=6;
	int doflist[numgrids];
	double pressure[numgrids];
	double rho_ice,g;
	double       xyz_list[numgrids][3];

	/*If on water, skip: */
	if(onwater)return;

	/*Get node data: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);

	/*pressure is lithostatic: */
	//md.pressure=md.rho_ice*md.g*(md.surface-md.z); a la matlab

	/*Get dof list on which we will plug the pressure values: */
	GetDofList1(&doflist[0]);

	/*pressure is lithostatic: */
	rho_ice=matpar->GetRhoIce();
	g=matpar->GetG();
	for(i=0;i<numgrids;i++){
		pressure[i]=rho_ice*g*(s[i]-xyz_list[i][2]);
	}

	/*plug local pressure values into global pressure vector: */
	VecSetValues(pg,numgrids,doflist,(const double*)pressure,INSERT_VALUES);

}
/*}}}*/
/*FUNCTION Configure {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::Configure"
void  Penta::Configure(void* ploadsin,void* pnodesin,void* pmaterialsin,void* pparametersin){

	int i;

	DataSet* loadsin=NULL;
	DataSet* nodesin=NULL;
	DataSet* materialsin=NULL;
	DataSet* parametersin=NULL;

	/*Recover pointers :*/
	loadsin=(DataSet*)ploadsin;
	nodesin=(DataSet*)pnodesin;
	materialsin=(DataSet*)pmaterialsin;
	parametersin=(DataSet*)pparametersin;

	/*Link this element with its nodes, ie find pointers to the nodes in the nodes dataset.: */
	ResolvePointers((Object**)nodes,node_ids,node_offsets,6,nodesin);

	/*Same for materials: */
	ResolvePointers((Object**)&matice,&mid,&matice_offset,1,materialsin);
	ResolvePointers((Object**)&matpar,&mparid,&matpar_offset,1,materialsin);

	/*Same for numpar: */
	ResolvePointers((Object**)&numpar,&numparid,&numpar_offset,1,parametersin);

}
/*}}}*/
/*FUNCTION copy {{{1*/
Object* Penta::copy() {
	return new Penta(*this); 
}
/*}}}*/
/*FUNCTION CreateKMatrix {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::CreateKMatrix"

void  Penta::CreateKMatrix(Mat Kgg,void* inputs,int analysis_type,int sub_analysis_type){

	/*Just branch to the correct element stiffness matrix generator, according to the type of analysis we are carrying out: */
	if (analysis_type==ControlAnalysisEnum()){

		CreateKMatrixDiagnosticHoriz( Kgg,inputs,analysis_type,sub_analysis_type);

	}
	else if (analysis_type==DiagnosticAnalysisEnum()){

		if (sub_analysis_type==HorizAnalysisEnum()){

			CreateKMatrixDiagnosticHoriz( Kgg,inputs,analysis_type,sub_analysis_type);
		}
		else if (sub_analysis_type==VertAnalysisEnum()){

			CreateKMatrixDiagnosticVert( Kgg,inputs,analysis_type,sub_analysis_type);
		}
		else if (sub_analysis_type==StokesAnalysisEnum()){

			CreateKMatrixDiagnosticStokes( Kgg,inputs,analysis_type,sub_analysis_type);

		}
		else throw ErrorException(__FUNCT__,exprintf("%s%i%s\n","sub_analysis: ",sub_analysis_type," not supported yet"));
	}
	else if (analysis_type==SlopecomputeAnalysisEnum()){

		CreateKMatrixSlopeCompute( Kgg,inputs,analysis_type,sub_analysis_type);
	}
	else if (analysis_type==PrognosticAnalysisEnum()){

		CreateKMatrixPrognostic( Kgg,inputs,analysis_type,sub_analysis_type);
	}
	else if (analysis_type==BalancedthicknessAnalysisEnum()){

		CreateKMatrixBalancedthickness( Kgg,inputs,analysis_type,sub_analysis_type);
	}
	else if (analysis_type==BalancedvelocitiesAnalysisEnum()){

		CreateKMatrixBalancedvelocities( Kgg,inputs,analysis_type,sub_analysis_type);
	}
	else if (analysis_type==ThermalAnalysisEnum()){

		CreateKMatrixThermal( Kgg,inputs,analysis_type,sub_analysis_type);
	}
	else if (analysis_type==MeltingAnalysisEnum()){

		CreateKMatrixMelting( Kgg,inputs,analysis_type,sub_analysis_type);
	}
	else{
		throw ErrorException(__FUNCT__,exprintf("%s%i%s\n","analysis: ",analysis_type," not supported yet"));
	}

}
/*}}}*/
/*FUNCTION CreateKMatrixBalancedthickness {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::CreateKMatrixBalancedthickness"

void  Penta::CreateKMatrixBalancedthickness(Mat Kgg,void* inputs,int analysis_type,int sub_analysis_type){

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*If on water, skip: */
	if(onwater)return;

	/*Is this element on the bed? :*/
	if(!onbed)return;

	/*Spawn Tria element from the base of the Penta: */
	tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	tria->CreateKMatrix(Kgg,inputs, analysis_type,sub_analysis_type);
	delete tria;
	return;

}
/*}}}*/
/*FUNCTION CreateKMatrixBalancedvelocities {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::CreateKMatrixBalancedvelocities"

void  Penta::CreateKMatrixBalancedvelocities(Mat Kgg,void* inputs,int analysis_type,int sub_analysis_type){

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*If on water, skip: */
	if(onwater)return;

	/*Is this element on the bed? :*/
	if(!onbed)return;

	/*Spawn Tria element from the base of the Penta: */
	tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	tria->CreateKMatrix(Kgg,inputs, analysis_type,sub_analysis_type);
	delete tria;
	return;

}
/*}}}*/
/*FUNCTION CreateKMatrixDiagnosticHoriz {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta:CreateKMatrixDiagnosticHoriz"
void Penta::CreateKMatrixDiagnosticHoriz( Mat Kgg, void* vinputs, int analysis_type,int sub_analysis_type){


	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=6;
	const int    numdof=2*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;


	/* 3d gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* fourth_gauss_vert_coord  =  NULL;
	double* area_gauss_weights           =  NULL;
	double* vert_gauss_weights           =  NULL;
	int     ig1,ig2;
	double  gauss_weight1,gauss_weight2;
	double  gauss_coord[4];
	int     order_area_gauss;
	int     num_vert_gauss;
	int     num_area_gauss;
	double  gauss_weight;

	/* 2d gaussian point: */
	int     num_gauss2d;
	double* first_gauss_area_coord2d  =  NULL;
	double* second_gauss_area_coord2d =  NULL;
	double* third_gauss_area_coord2d  =  NULL;
	double* gauss_weights2d=NULL;
	double  gauss_l1l2l3[3];

	/* material data: */
	double viscosity; //viscosity
	double oldviscosity; //viscosity
	double newviscosity; //viscosity

	/* strain rate: */
	double epsilon[5]; /* epsilon=[exx,eyy,exy,exz,eyz];*/
	double oldepsilon[5]; /* epsilon=[exx,eyy,exy,exz,eyz];*/

	/* matrices: */
	double B[5][numdof];
	double Bprime[5][numdof];
	double L[2][numdof];
	double D[5][5]={{ 0,0,0,0,0 },{0,0,0,0,0},{0,0,0,0,0},{0,0,0,0,0},{0,0,0,0,0}};              // material matrix, simple scalar matrix.
	double D_scalar;
	double DL[2][2]={{ 0,0 },{0,0}}; //for basal drag
	double DL_scalar;

	/* local element matrices: */
	double Ke_gg[numdof][numdof]; //local element stiffness matrix 
	double Ke_gg_gaussian[numdof][numdof]; //stiffness matrix evaluated at the gaussian point.
	double Ke_gg_drag_gaussian[numdof][numdof]; //stiffness matrix contribution from drag
	double Jdet;

	/*slope: */
	double  slope[2]={0.0,0.0};
	double  slope_magnitude;

	/*input parameters for structural analysis (diagnostic): */
	double  vxvy_list[numgrids][2]={{0,0},{0,0},{0,0}};
	double  oldvxvy_list[numgrids][2]={{0,0},{0,0},{0,0}};
	int     dofs[2]={0,1};
	double  thickness;

	/*friction: */
	double  alpha2_list[3];
	double  alpha2;

	double MAXSLOPE=.06; // 6 %
	double MOUNTAINKEXPONENT=10;

	ParameterInputs* inputs=NULL;

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*If on water, skip stiffness: */
	if(onwater)return;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;

	/*Figure out if this pentaelem is collapsed. If so, then bailout, except if it is at the 
	  bedrock, in which case we spawn a tria element using the 3 first grids, and use it to build 
	  the stiffness matrix. */


	if ((collapse==1) && (onbed==0)){
		/*This element should be collapsed, but this element is not on the bedrock, therefore all its 
		 * dofs have already been frozen! Do nothing: */
		return;
	}
	else if ((collapse==1) && (onbed==1)){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 *and use its CreateKMatrix functionality to fill the global stiffness matrix: */
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
		tria->CreateKMatrix(Kgg,inputs, analysis_type,sub_analysis_type);
		delete tria;
		return;
	}
	else{

		/*Implement standard penta element: */

		/*recover extra inputs from users, at current convergence iteration: */
		inputs->Recover("velocity",&vxvy_list[0][0],2,dofs,numgrids,(void**)nodes);
		inputs->Recover("old_velocity",&oldvxvy_list[0][0],2,dofs,numgrids,(void**)nodes);

		/* Get node coordinates and dof list: */
		GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
		GetDofList(&doflist[0],&numberofdofspernode);

		/* Set Ke_gg to 0: */
		for(i=0;i<numdof;i++){
			for(j=0;j<numdof;j++){
				Ke_gg[i][j]=0.0;
			}
		}

#ifdef _DEBUGELEMENTS_
		if(my_rank==RANK && id==ELID){ 
			printf("El id %i Rank %i PentaElement input list before gaussian loop: \n",ELID,RANK); 
			printf("   rho_ice: %g\n",matice->GetRhoIce());
			printf("   rho_water:%g \n",matice->GetRhoWater());
			printf("   gravity: %g\n",matpar->GetG());
			printf("   Velocity: \n");
			for (i=0;i<numgrids;i++){
				printf("      grid %i  [%g,%g,%g]\n",i,vxvy_list[i][0],vxvy_list[i][1],vxvy_list[i][2]);
			}
			printf("   B [%g %g %g %g %g %g]\n",B_list[0],B_list[1],B_list[2],B_list[3],B_list[4],B_list[5]);
			printf("   K [%g %g %g %g %g %g]\n",K_list[0],K_list[1],K_list[2],K_list[3],K_list[4],K_list[5]);
			printf("   thickness [%g %g %g %g %g %g]\n",thickness_list[0],thickness_list[1],thickness_list[2],thickness_list[3],thickness_list[4],thickness_list[5]);
			printf("   surface [%g %g %g %g %g %g]\n",surface_list[0],surface_list[1],surface_list[2],surface_list[3],surface_list[4],surface_list[5]);
			printf("   bed [%g %g %g %g %g %g]\n",bed_list[0],bed_list[1],bed_list[2],bed_list[3],bed_list[4],bed_list[5]);
			printf("   temperature_average [%g %g %g %g %g %g]\n",temperature_average_list[0],temperature_average_list[1],temperature_average_list[2],temperature_average_list[3],temperature_average_list[4],temperature_average_list[5]);
		}
#endif

		/*Get gaussian points and weights. Penta is an extrusion of a Tria, we therefore 
		  get tria gaussian points as well as segment gaussian points. For tria gaussian 
		  points, order of integration is 2, because we need to integrate the product tB*D*B' 
		  which is a polynomial of degree 3 (see GaussTria for more details). For segment gaussian 
		  points, same deal, which yields 3 gaussian points.*/

		order_area_gauss=5;
		num_vert_gauss=5;

		GaussPenta( &num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &area_gauss_weights, &fourth_gauss_vert_coord,&vert_gauss_weights,order_area_gauss,num_vert_gauss);

#ifdef _DEBUGGAUSS_
		if(my_rank==RANK && id==ELID){ 
			printf("El id %i Rank %i PentaElement gauss points\n",ELID,RANK); 
			for (i=0;i<num_area_gauss;i++){
				printf("   Area Gauss coord %i: %lf %lf %lf Weight: %lf\n",i,*(first_gauss_area_coord+i),*(second_gauss_area_coord+i),*(third_gauss_area_coord+i),*(area_gauss_weights+i));
			}
			for (i=0;i<num_vert_gauss;i++){
				printf("   Vert Gauss coord %i: %lf Weight: %lf\n",i,*(fourth_gauss_vert_coord+i),*(vert_gauss_weights+i));
			}	
		}
#endif
		/* Start  looping on the number of gaussian points: */
		for (ig1=0; ig1<num_area_gauss; ig1++){
			for (ig2=0; ig2<num_vert_gauss; ig2++){

				/*Pick up the gaussian point: */
				gauss_weight1=*(area_gauss_weights+ig1);
				gauss_weight2=*(vert_gauss_weights+ig2);
				gauss_weight=gauss_weight1*gauss_weight2;


				gauss_coord[0]=*(first_gauss_area_coord+ig1); 
				gauss_coord[1]=*(second_gauss_area_coord+ig1);
				gauss_coord[2]=*(third_gauss_area_coord+ig1);
				gauss_coord[3]=*(fourth_gauss_vert_coord+ig2);


				/*Get strain rate from velocity: */
				GetStrainRate(&epsilon[0],&vxvy_list[0][0],&xyz_list[0][0],gauss_coord);
				GetStrainRate(&oldepsilon[0],&oldvxvy_list[0][0],&xyz_list[0][0],gauss_coord);

				/*Get viscosity: */
				matice->GetViscosity3d(&viscosity, &epsilon[0]);
				matice->GetViscosity3d(&oldviscosity, &oldepsilon[0]);

				/*Get B and Bprime matrices: */
				GetB(&B[0][0], &xyz_list[0][0], gauss_coord);
				GetBPrime(&Bprime[0][0], &xyz_list[0][0], gauss_coord);

				/* Get Jacobian determinant: */
				GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_coord);

				/*Build the D matrix: we plug the gaussian weight, the thickness, the viscosity, and the jacobian determinant 
				  onto this scalar matrix, so that we win some computational time: */

				newviscosity=viscosity+numpar->viscosity_overshoot*(viscosity-oldviscosity);
				D_scalar=newviscosity*gauss_weight*Jdet;
				for (i=0;i<5;i++){
					D[i][i]=D_scalar;
				}

				/*  Do the triple product tB*D*Bprime: */
				TripleMultiply( &B[0][0],5,numdof,1,
							&D[0][0],5,5,0,
							&Bprime[0][0],5,numdof,0,
							&Ke_gg_gaussian[0][0],0);

				/* Add the Ke_gg_gaussian, and optionally Ke_gg_gaussian onto Ke_gg: */
				for( i=0; i<numdof; i++){
					for(j=0;j<numdof;j++){
						Ke_gg[i][j]+=Ke_gg_gaussian[i][j];
					}
				}
			} //for (ig2=0; ig2<num_vert_gauss; ig2++)
		} //for (ig1=0; ig1<num_area_gauss; ig1++)


		/*Add Ke_gg to global matrix Kgg: */
		MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);

		//Deal with 2d friction at the bedrock interface
		if((onbed && !shelf)){

			/*Build a tria element using the 3 grids of the base of the penta. Then use 
			 * the tria functionality to build a friction stiffness matrix on these 3
			 * grids: */

			tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
			tria->CreateKMatrixDiagnosticHorizFriction(Kgg,inputs,analysis_type,sub_analysis_type);
			delete tria;
		}

	} 

	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&fourth_gauss_vert_coord);
	xfree((void**)&area_gauss_weights);
	xfree((void**)&vert_gauss_weights);
	xfree((void**)&first_gauss_area_coord2d);
	xfree((void**)&second_gauss_area_coord2d);
	xfree((void**)&third_gauss_area_coord2d);
	xfree((void**)&gauss_weights2d);

}
/*}}}*/
/*FUNCTION CreateKMatrixDiagnosticStokes {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta:CreateKMatrixDiagnosticStokes"
void Penta::CreateKMatrixDiagnosticStokes( Mat Kgg, void* vinputs, int analysis_type,int sub_analysis_type){

	int i,j;

	const int numgrids=6;
	const int DOFPERGRID=4;
	const int numdof=numgrids*DOFPERGRID;
	int doflist[numdof];
	int numberofdofspernode;

	const int numgrids2d=3;
	const int numdof2d=numgrids2d*DOFPERGRID;

	int   dofs[3]={0,1,2};

	double K_terms[numdof][numdof];

	/*Material properties: */
	double         gravity,rho_ice,rho_water;

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*Grid data: */
	double        xyz_list[numgrids][3];

	/*parameters: */
	double		   xyz_list_tria[numgrids2d][3];
	double		   surface_normal[3];
	double		   bed_normal[3];
	double         vxvyvz_list[numgrids][3];
	double         thickness;

	/*matrices: */
	double     Ke_temp[27][27]={0.0}; //for the six nodes and the bubble 
	double     Ke_reduced[numdof][numdof]; //for the six nodes only
	double     Ke_gaussian[27][27];
	double     Ke_drag_gaussian[numdof2d][numdof2d];
	double     B[8][27];
	double     B_prime[8][27];
	double     LStokes[14][numdof2d];
	double     LprimeStokes[14][numdof2d];
	double     Jdet;
	double     Jdet2d;
	double     D[8][8]={0.0};
	double     D_scalar;
	double     tBD[27][8];
	double     DLStokes[14][14]={0.0};
	double     tLDStokes[numdof2d][14];

	/* gaussian points: */
	int     num_area_gauss;
	int     igarea,igvert;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* vert_gauss_coord = NULL;
	double* area_gauss_weights  =  NULL;
	double* vert_gauss_weights  =  NULL;

	/* specific gaussian point: */
	double  gauss_weight,area_gauss_weight,vert_gauss_weight;
	double  gauss_coord[4];
	double  gauss_coord_tria[3];
	int area_order=5;
	int	num_vert_gauss=5;

	double  epsilon[6]; /* epsilon=[exx,eyy,ezz,exy,exz,eyz];*/
	double  viscosity;
	double  alpha2_list[numgrids2d];
	double  alpha2_gauss;

	ParameterInputs* inputs=NULL;

	/*If on water, skip stiffness: */
	if(onwater)return;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;

	/* Set K_terms  to 0: */
	for(i=0;i<numdof;i++){
		for(j=0;j<numdof;j++){
			K_terms[i][j]=0.0;
		}
	}

	/*recovre material parameters: */
	rho_water=matpar->GetRhoWater();
	rho_ice=matpar->GetRhoIce();
	gravity=matpar->GetG();

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/*recover extra inputs from users, at current convergence iteration: */
	inputs->Recover("velocity",&vxvyvz_list[0][0],3,dofs,numgrids,(void**)nodes);

	/* Get gaussian points and weights. Penta is an extrusion of a Tria, we therefore 
		get tria gaussian points as well as segment gaussian points. For tria gaussian 
		points, order of integration is 2, because we need to integrate the product tB*D*B' 
		which is a polynomial of degree 3 (see GaussTria for more details). For segment gaussian 
		points, same deal, which yields 3 gaussian points.*/

	area_order=5;
	num_vert_gauss=5;

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussPenta( &num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &area_gauss_weights,&vert_gauss_coord, &vert_gauss_weights, area_order, num_vert_gauss);

	/* Start  looping on the number of gaussian points: */
	for (igarea=0; igarea<num_area_gauss; igarea++){
		for (igvert=0; igvert<num_vert_gauss; igvert++){
			/*Pick up the gaussian point: */
			area_gauss_weight=*(area_gauss_weights+igarea);
			vert_gauss_weight=*(vert_gauss_weights+igvert);
			gauss_weight=area_gauss_weight*vert_gauss_weight;
			gauss_coord[0]=*(first_gauss_area_coord+igarea); 
			gauss_coord[1]=*(second_gauss_area_coord+igarea);
			gauss_coord[2]=*(third_gauss_area_coord+igarea);
			gauss_coord[3]=*(vert_gauss_coord+igvert);

			/*Compute thickness: */

			/*Compute strain rate: */
			GetStrainRateStokes(&epsilon[0],&vxvyvz_list[0][0],&xyz_list[0][0],gauss_coord);

			/*Get viscosity: */
			matice->GetViscosity3dStokes(&viscosity,&epsilon[0]);

			/*Get B and Bprime matrices: */
			GetBStokes(&B[0][0],&xyz_list[0][0],gauss_coord); 
			GetBprimeStokes(&B_prime[0][0],&xyz_list[0][0], gauss_coord); 

			/* Get Jacobian determinant: */
			GetJacobianDeterminant(&Jdet, &xyz_list[0][0],&gauss_coord[0]);

			/* Build the D matrix: we plug the gaussian weight, the thickness, the viscosity, and the jacobian determinant 
			 * onto this scalar matrix, so that we win some computational time: */
			D_scalar=gauss_weight*Jdet;
			for (i=0;i<6;i++){
				D[i][i]=D_scalar*viscosity;
			}
			for (i=6;i<8;i++){
				D[i][i]=-D_scalar*numpar->stokesreconditioning;
			}

			/*  Do the triple product tB*D*Bprime: */
			MatrixMultiply(&B[0][0],8,27,1,&D[0][0],8,8,0,&tBD[0][0],0);
			MatrixMultiply(&tBD[0][0],27,8,0,&B_prime[0][0],8,27,0,&Ke_gaussian[0][0],0);

			/*Add Ke_gaussian and Ke_gaussian to terms in pKe. Watch out for column orientation from matlab: */
			for(i=0;i<27;i++){
				for(j=0;j<27;j++){
					Ke_temp[i][j]+=Ke_gaussian[i][j];
				}
			}
		}
	}

	if((onbed==1) && (shelf==0)){

		/*Build alpha2_list used by drag stiffness matrix*/
		Friction* friction=NewFriction();

		/*Initialize all fields: */
		friction->element_type=(char*)xmalloc((strlen("2d")+1)*sizeof(char));
		strcpy(friction->element_type,"2d");

		friction->gravity=matpar->GetG();
		friction->rho_ice=matpar->GetRhoIce();
		friction->rho_water=matpar->GetRhoWater();
		friction->K=&k[0];
		friction->bed=&b[0];
		friction->thickness=&h[0];
		friction->velocities=&vxvyvz_list[0][0];
		friction->p=p;
		friction->q=q;
		friction->analysis_type=analysis_type;

		/*Compute alpha2_list: */
		FrictionGetAlpha2(&alpha2_list[0],friction);

		/*Erase friction object: */
		DeleteFriction(&friction);

		for(i=0;i<numgrids2d;i++){
			for(j=0;j<3;j++){
				xyz_list_tria[i][j]=xyz_list[i][j];
			}
		}

		xfree((void**)&first_gauss_area_coord); xfree((void**)&second_gauss_area_coord); xfree((void**)&third_gauss_area_coord); xfree((void**)&area_gauss_weights);
		GaussTria (&num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &area_gauss_weights, 2);

		/* Start looping on the number of gauss 2d (nodes on the bedrock) */
		for (igarea=0; igarea<num_area_gauss; igarea++){
			gauss_weight=*(area_gauss_weights+igarea);
			gauss_coord[0]=*(first_gauss_area_coord+igarea); 
			gauss_coord[1]=*(second_gauss_area_coord+igarea);
			gauss_coord[2]=*(third_gauss_area_coord+igarea);
			gauss_coord[3]=-1;

			gauss_coord_tria[0]=*(first_gauss_area_coord+igarea); 
			gauss_coord_tria[1]=*(second_gauss_area_coord+igarea);
			gauss_coord_tria[2]=*(third_gauss_area_coord+igarea);

			/*Get the Jacobian determinant */
			tria->GetJacobianDeterminant3d(&Jdet2d, &xyz_list_tria[0][0], gauss_coord_tria);

			/*Get L matrix if viscous basal drag present: */
			GetLStokes(&LStokes[0][0],  gauss_coord_tria);
			GetLprimeStokes(&LprimeStokes[0][0], &xyz_list[0][0], gauss_coord_tria, gauss_coord);

			/*Compute strain rate: */
			GetStrainRateStokes(&epsilon[0],&vxvyvz_list[0][0],&xyz_list[0][0],gauss_coord);

			/*Get viscosity at last iteration: */
			matice->GetViscosity3dStokes(&viscosity,&epsilon[0]);

			/*Get normal vecyor to the bed */
			SurfaceNormal(&surface_normal[0],xyz_list_tria);

			bed_normal[0]=-surface_normal[0]; //Program is for surface, so the normal to the bed is the opposite of the result
			bed_normal[1]=-surface_normal[1];
			bed_normal[2]=-surface_normal[2];

			/*Calculate DL on gauss point */
			tria->GetParameterValue(&alpha2_gauss,&alpha2_list[0],gauss_coord_tria);

			DLStokes[0][0]=alpha2_gauss*gauss_weight*Jdet2d;
			DLStokes[1][1]=alpha2_gauss*gauss_weight*Jdet2d;
			DLStokes[2][2]=-alpha2_gauss*gauss_weight*Jdet2d*bed_normal[0]*bed_normal[2];
			DLStokes[3][3]=-alpha2_gauss*gauss_weight*Jdet2d*bed_normal[1]*bed_normal[2];
			DLStokes[4][4]=-alpha2_gauss*gauss_weight*Jdet2d*bed_normal[0]*bed_normal[2];
			DLStokes[5][5]=-alpha2_gauss*gauss_weight*Jdet2d*bed_normal[1]*bed_normal[2];
			DLStokes[6][6]=-viscosity*gauss_weight*Jdet2d*bed_normal[0];
			DLStokes[7][7]=-viscosity*gauss_weight*Jdet2d*bed_normal[1];
			DLStokes[8][8]=-viscosity*gauss_weight*Jdet2d*bed_normal[2];
			DLStokes[9][8]=-viscosity*gauss_weight*Jdet2d*bed_normal[0]/2.0;
			DLStokes[10][10]=-viscosity*gauss_weight*Jdet2d*bed_normal[1]/2.0;
			DLStokes[11][11]=numpar->stokesreconditioning*gauss_weight*Jdet2d*bed_normal[0];
			DLStokes[12][12]=numpar->stokesreconditioning*gauss_weight*Jdet2d*bed_normal[1];
			DLStokes[13][13]=numpar->stokesreconditioning*gauss_weight*Jdet2d*bed_normal[2];

			/*  Do the triple product tL*D*L: */
			MatrixMultiply(&LStokes[0][0],14,numdof2d,1,&DLStokes[0][0],14,14,0,&tLDStokes[0][0],0);
			MatrixMultiply(&tLDStokes[0][0],numdof2d,14,0,&LprimeStokes[0][0],14,numdof2d,0,&Ke_drag_gaussian[0][0],0);

			for(i=0;i<numdof2d;i++){
				for(j=0;j<numdof2d;j++){
					Ke_temp[i][j]+=Ke_drag_gaussian[i][j];
				}
			}
		}
	} //if ( (onbed==1) && (shelf==0))

	/*Reduce the matrix */
	ReduceMatrixStokes(&Ke_reduced[0][0], &Ke_temp[0][0]);

	for(i=0;i<numdof;i++){
		for(j=0;j<numdof;j++){
			K_terms[i][j]+=Ke_reduced[i][j];
		}
	}

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)K_terms,ADD_VALUES);


	/*Free ressources:*/
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&area_gauss_weights);
	xfree((void**)&vert_gauss_coord);
	xfree((void**)&vert_gauss_weights);

	return;
}
/*}}}*/
/*FUNCTION CreateKMatrixDiagnosticVert {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta:CreateKMatrixDiagnosticVert"
void Penta::CreateKMatrixDiagnosticVert( Mat Kgg, void* vinputs, int analysis_type,int sub_analysis_type){

	/* local declarations */
	int             i,j;

	/* node data: */
	const int    numgrids=6;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	/* 3d gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* fourth_gauss_vert_coord  =  NULL;
	double* area_gauss_weights           =  NULL;
	double* vert_gauss_weights           =  NULL;
	int     ig1,ig2;
	double  gauss_weight1,gauss_weight2;
	double  gauss_coord[4];
	int     order_area_gauss;
	int     num_vert_gauss;
	int     num_area_gauss;
	double  gauss_weight;

	/* matrices: */
	double  Ke_gg[numdof][numdof];
	double  Ke_gg_gaussian[numdof][numdof];
	double  Jdet;
	double  B[NDOF1][numgrids];
	double  Bprime[NDOF1][numgrids];
	double  DL_scalar;

	ParameterInputs* inputs=NULL;

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*If on water, skip stiffness: */
	if(onwater)return;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;


	/*If this element  is on the surface, we have a dynamic boundary condition that applies, as a stiffness 
	 * matrix: */
	if(onsurface){
		tria=(Tria*)SpawnTria(3,4,5); //nodes 3,4 and 5 are on the surface
		tria->CreateKMatrixDiagnosticSurfaceVert(Kgg,inputs, analysis_type,sub_analysis_type);
		delete tria;
	}

	/*Now, onto the formulation for the vertical velocity: */

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Set Ke_gg to 0: */
	for(i=0;i<numdof;i++){
		for(j=0;j<numdof;j++){
			Ke_gg[i][j]=0.0;
		}
	}

	/*Get gaussian points and weights. Penta is an extrusion of a Tria, we therefore 
	  get tria gaussian points as well as segment gaussian points. For tria gaussian 
	  points, order of integration is 2, because we need to integrate the product tB*D*B' 
	  which is a polynomial of degree 3 (see GaussTria for more details). For segment gaussian 
	  points, same deal, which yields 3 gaussian points.*/

	order_area_gauss=2;
	num_vert_gauss=2;

	GaussPenta( &num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &area_gauss_weights, &fourth_gauss_vert_coord,&vert_gauss_weights,order_area_gauss,num_vert_gauss);
#ifdef _ISSM_DEBUG_ 
	for (i=0;i<num_area_gauss;i++){
		printf("Area Gauss coord %i: %lf %lf %lf Weight: %lf\n",i,*(first_gauss_area_coord+i),*(second_gauss_area_coord+i),*(third_gauss_area_coord+i),*(area_gauss_weights+i));
	}
	for (i=0;i<num_vert_gauss;i++){
		printf("Vert Gauss coord %i: %lf Weight: %lf\n",i,*(fourth_gauss_vert_coord+i),*(vert_gauss_weights+i));
	}
#endif

	/* Start  looping on the number of gaussian points: */
	for (ig1=0; ig1<num_area_gauss; ig1++){
		for (ig2=0; ig2<num_vert_gauss; ig2++){

			/*Pick up the gaussian point: */
			gauss_weight1=*(area_gauss_weights+ig1);
			gauss_weight2=*(vert_gauss_weights+ig2);
			gauss_weight=gauss_weight1*gauss_weight2;

			gauss_coord[0]=*(first_gauss_area_coord+ig1); 
			gauss_coord[1]=*(second_gauss_area_coord+ig1);
			gauss_coord[2]=*(third_gauss_area_coord+ig1);
			gauss_coord[3]=*(fourth_gauss_vert_coord+ig2);

			/*Get B and Bprime matrices: */
			GetB_vert(&B[0][0], &xyz_list[0][0], gauss_coord);
			GetBPrime_vert(&Bprime[0][0], &xyz_list[0][0], gauss_coord);

			/* Get Jacobian determinant: */
			GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_coord);
			DL_scalar=gauss_weight*Jdet;

			/*  Do the triple product tB*D*Bprime: */
			TripleMultiply( &B[0][0],1,numgrids,1,
						&DL_scalar,1,1,0,
						&Bprime[0][0],1,numgrids,0,
						&Ke_gg_gaussian[0][0],0);

			/* Add the Ke_gg_gaussian, and optionally Ke_gg_drag_gaussian onto Ke_gg: */
			for( i=0; i<numdof; i++){
				for(j=0;j<numdof;j++){
					Ke_gg[i][j]+=Ke_gg_gaussian[i][j];
				}
			}	
		} //for (ig2=0; ig2<num_vert_gauss; ig2++)
	} //for (ig1=0; ig1<num_area_gauss; ig1++)

	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)Ke_gg,ADD_VALUES);

	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&fourth_gauss_vert_coord);
	xfree((void**)&area_gauss_weights);
	xfree((void**)&vert_gauss_weights);
}
/*}}}*/
/*FUNCTION CreateKMatrixMelting {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::CreateKMatrixMelting"
void  Penta::CreateKMatrixMelting(Mat Kgg,void* inputs,int analysis_type,int sub_analysis_type){

	Tria* tria=NULL;

	/*If on water, skip: */
	if(onwater)return;

	if (!onbed){
		return;
	}
	else{

		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
		tria->CreateKMatrixMelting(Kgg,inputs, analysis_type,sub_analysis_type);
		delete tria;
		return;
	}
}
/*}}}*/
/*FUNCTION CreateKMatrixPrognostic {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::CreateKMatrixPrognostic"

void  Penta::CreateKMatrixPrognostic(Mat Kgg,void* inputs,int analysis_type,int sub_analysis_type){

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*If on water, skip: */
	if(onwater)return;

	/*Is this element on the bed? :*/
	if(!onbed)return;

	/*Spawn Tria element from the base of the Penta: */
	tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	tria->CreateKMatrix(Kgg,inputs, analysis_type,sub_analysis_type);
	delete tria;
	return;

}
/*}}}*/
/*FUNCTION CreateKMatrixSlopeCompute {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::CreateKMatrixSlopeCompute"

void  Penta::CreateKMatrixSlopeCompute(Mat Kgg,void* inputs,int analysis_type,int sub_analysis_type){

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*If on water, skip: */
	if(onwater)return;

	/*Is this element on the bed? :*/
	if(!onbed)return;

	/*Spawn Tria element from the base of the Penta: */
	tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	tria->CreateKMatrix(Kgg,inputs, analysis_type,sub_analysis_type);
	delete tria;
	return;

}
/*}}}*/
/*FUNCTION CreateKMatrixThermal {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::CreateKMatrixThermal"
void  Penta::CreateKMatrixThermal(Mat Kgg,void* vinputs,int analysis_type,int sub_analysis_type){

	/* local declarations */
	int i,j;
	int found=0;

	/* node data: */
	const int    numgrids=6;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double xyz_list[numgrids][3];
	int    doflist[numdof];
	int    numberofdofspernode;

	/* gaussian points: */
	int     num_area_gauss,igarea,igvert;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* vert_gauss_coord = NULL;
	double* area_gauss_weights  =  NULL;
	double* vert_gauss_weights  =  NULL;
	double  gauss_weight,area_gauss_weight,vert_gauss_weight;
	double  gauss_coord[4];
	double  gauss_l1l2l3[3];

	int area_order=5;
	int num_vert_gauss=5;

	int     dofs[3]={0,1,2};
	double  dt;
	double  K[2][2]={0.0};

	double  vxvyvz_list[numgrids][3];
	double  vx_list[numgrids];
	int     vx_list_exists;
	double  u;
	double  vy_list[numgrids];
	int     vy_list_exists;
	double  v;
	double  vz_list[numgrids];
	int     vz_list_exists;
	double  w;

	/*matrices: */
	double     K_terms[numdof][numdof]={0.0};
	double     Ke_gaussian_conduct[numdof][numdof];
	double     Ke_gaussian_advec[numdof][numdof];
	double     Ke_gaussian_artdiff[numdof][numdof];
	double     Ke_gaussian_transient[numdof][numdof];
	double     B[3][numdof];
	double     Bprime[3][numdof];
	double     B_conduct[3][numdof];
	double     B_advec[3][numdof];
	double     B_artdiff[2][numdof];
	double     Bprime_advec[3][numdof];
	double     L[numdof];
	double     D_scalar;
	double     D[3][3];
	double     l1l2l3[3];
	double     tl1l2l3D[3];
	double     tBD[3][numdof];
	double     tBD_conduct[3][numdof];
	double     tBD_advec[3][numdof];
	double     tBD_artdiff[3][numdof];
	double     tLD[numdof];

	double     Jdet;

	/*Material properties: */
	double     gravity,rho_ice,rho_water;
	double     heatcapacity,thermalconductivity;
	double     mixed_layer_capacity,thermal_exchange_velocity;

	/*Collapsed formulation: */
	Tria*  tria=NULL;
	ParameterInputs* inputs=NULL;

	/*If on water, skip: */
	if(onwater)return;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	// /*recovre material parameters: */
	rho_water=matpar->GetRhoWater();
	rho_ice=matpar->GetRhoIce();
	gravity=matpar->GetG();
	heatcapacity=matpar->GetHeatCapacity();
	thermalconductivity=matpar->GetThermalConductivity();

	/*recover extra inputs from users, dt and velocity: */
	found=inputs->Recover("velocity",&vxvyvz_list[0][0],3,dofs,numgrids,(void**)nodes);
	if(!found)throw ErrorException(__FUNCT__," could not find velocity in inputs!");
	found=inputs->Recover("dt",&dt);
	if(!found)throw ErrorException(__FUNCT__," could not find dt in inputs!");

	for(i=0;i<numgrids;i++){
		vx_list[i]=vxvyvz_list[i][0];
		vy_list[i]=vxvyvz_list[i][1];
		vz_list[i]=vxvyvz_list[i][2];
	}


	/* Get gaussian points and weights. Penta is an extrusion of a Tria, we therefore 
		get tria gaussian points as well as segment gaussian points. For tria gaussian 
		points, order of integration is 2, because we need to integrate the product tB*D*B' 
		which is a polynomial of degree 3 (see GaussTria for more details). For segment gaussian 
		points, same deal, which yields 3 gaussian points.: */

	/*Get gaussian points: */
	area_order=2;
	num_vert_gauss=2;

	GaussPenta( &num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &area_gauss_weights,&vert_gauss_coord, &vert_gauss_weights, area_order, num_vert_gauss);

	/* Start  looping on the number of gaussian points: */
	for (igarea=0; igarea<num_area_gauss; igarea++){
		for (igvert=0; igvert<num_vert_gauss; igvert++){
			/*Pick up the gaussian point: */
			area_gauss_weight=*(area_gauss_weights+igarea);
			vert_gauss_weight=*(vert_gauss_weights+igvert);
			gauss_weight=area_gauss_weight*vert_gauss_weight;
			gauss_coord[0]=*(first_gauss_area_coord+igarea); 
			gauss_coord[1]=*(second_gauss_area_coord+igarea);
			gauss_coord[2]=*(third_gauss_area_coord+igarea);
			gauss_coord[3]=*(vert_gauss_coord+igvert);

			/* Get Jacobian determinant: */
			GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_coord);

			/*Conduction: */

			/*Get B_conduct matrix: */
			GetB_conduct(&B_conduct[0][0],&xyz_list[0][0],gauss_coord); 

			/*Build D: */
			D_scalar=gauss_weight*Jdet*(thermalconductivity/(rho_ice*heatcapacity));

			if(dt){
				D_scalar=D_scalar*dt;
			}

			D[0][0]=D_scalar; D[0][1]=0; D[0][2]=0;
			D[1][0]=0; D[1][1]=D_scalar; D[1][2]=0;
			D[2][0]=0; D[2][1]=0; D[2][2]=D_scalar;

			/*  Do the triple product B'*D*B: */
			MatrixMultiply(&B_conduct[0][0],3,numdof,1,&D[0][0],3,3,0,&tBD_conduct[0][0],0);
			MatrixMultiply(&tBD_conduct[0][0],numdof,3,0,&B_conduct[0][0],3,numdof,0,&Ke_gaussian_conduct[0][0],0);

			/*Advection: */

			/*Get B_advec and Bprime_advec matrices: */
			GetB_advec(&B_advec[0][0],&xyz_list[0][0],gauss_coord); 
			GetBprime_advec(&Bprime_advec[0][0],&xyz_list[0][0],gauss_coord); 

			//Build the D matrix
			GetParameterValue(&u, &vx_list[0],gauss_coord);
			GetParameterValue(&v, &vy_list[0],gauss_coord);
			GetParameterValue(&w, &vz_list[0],gauss_coord);

			D_scalar=gauss_weight*Jdet;

			if(dt){
				D_scalar=D_scalar*dt;
			}

			D[0][0]=D_scalar*u;D[0][1]=0;         D[0][2]=0;
			D[1][0]=0;         D[1][1]=D_scalar*v;D[1][2]=0;
			D[2][0]=0;         D[2][1]=0;         D[2][2]=D_scalar*w;

			/*  Do the triple product B'*D*Bprime: */
			MatrixMultiply(&B_advec[0][0],3,numdof,1,&D[0][0],3,3,0,&tBD_advec[0][0],0);
			MatrixMultiply(&tBD_advec[0][0],numdof,3,0,&Bprime_advec[0][0],3,numdof,0,&Ke_gaussian_advec[0][0],0);

			/*Transient: */
			if(dt){
				GetNodalFunctions(&L[0], gauss_coord);
				D_scalar=gauss_weight*Jdet;
				D_scalar=D_scalar;

				/*  Do the triple product L'*D*L: */
				MatrixMultiply(&L[0],numdof,1,0,&D_scalar,1,1,0,&tLD[0],0);
				MatrixMultiply(&tLD[0],numdof,1,0,&L[0],1,numdof,0,&Ke_gaussian_transient[0][0],0);
			}
			else{
				for(i=0;i<numdof;i++){
					for(j=0;j<numdof;j++){
						Ke_gaussian_transient[i][j]=0;
					}
				}
			}

			/*Artifficial diffusivity*/
			if(numpar->artdiff){
				/*Build K: */
				D_scalar=gauss_weight*Jdet/(pow(u,2)+pow(v,2)+numpar->epsvel);
				if(dt){
					D_scalar=D_scalar*dt;
				}
				K[0][0]=D_scalar*pow(u,2);       K[0][1]=D_scalar*fabs(u)*fabs(v);
				K[1][0]=D_scalar*fabs(u)*fabs(v);K[1][1]=D_scalar*pow(v,2);

				/*Get B_artdiff: */
				GetB_artdiff(&B_artdiff[0][0],&xyz_list[0][0],gauss_coord); 

				/*  Do the triple product B'*K*B: */
				MatrixMultiply(&B_artdiff[0][0],2,numdof,1,&K[0][0],2,2,0,&tBD_artdiff[0][0],0);
				MatrixMultiply(&tBD_artdiff[0][0],numdof,2,0,&B_artdiff[0][0],2,numdof,0,&Ke_gaussian_artdiff[0][0],0);
			}
			else{
				for(i=0;i<numdof;i++){
					for(j=0;j<numdof;j++){
						Ke_gaussian_artdiff[i][j]=0;
					}
				}
			}

			/*Add Ke_gaussian to pKe: */
			for(i=0;i<numdof;i++){
				for(j=0;j<numdof;j++){
					K_terms[i][j]+=Ke_gaussian_conduct[i][j]+Ke_gaussian_advec[i][j]+Ke_gaussian_transient[i][j]+Ke_gaussian_artdiff[i][j];
				}
			}
		}
	}


	/*Add Ke_gg to global matrix Kgg: */
	MatSetValues(Kgg,numdof,doflist,numdof,doflist,(const double*)K_terms,ADD_VALUES);

	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&area_gauss_weights);
	xfree((void**)&vert_gauss_weights);
	xfree((void**)&vert_gauss_coord);

	//Ice/ocean heat exchange flux on ice shelf base 
	if(onbed && shelf){

		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
		tria->CreateKMatrixThermal(Kgg,inputs, analysis_type,sub_analysis_type);
		delete tria;
	}
}
/*}}}*/
/*FUNCTION CreatePVector {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::CreatePVector"
void  Penta::CreatePVector(Vec pg, void* inputs, int analysis_type,int sub_analysis_type){

	/*Just branch to the correct element stiffness matrix generator, according to the type of analysis we are carrying out: */
	if (analysis_type==ControlAnalysisEnum()){

		CreatePVectorDiagnosticHoriz( pg,inputs,analysis_type,sub_analysis_type);
	}
	else if (analysis_type==DiagnosticAnalysisEnum()){

		if (sub_analysis_type==HorizAnalysisEnum()){

			CreatePVectorDiagnosticHoriz( pg,inputs,analysis_type,sub_analysis_type);
		}
		else if (sub_analysis_type==VertAnalysisEnum()){

			CreatePVectorDiagnosticVert( pg,inputs,analysis_type,sub_analysis_type);
		}
		else if (sub_analysis_type==StokesAnalysisEnum()){

			CreatePVectorDiagnosticStokes( pg,inputs,analysis_type,sub_analysis_type);
		}
		else throw ErrorException(__FUNCT__,exprintf("%s%i%s\n","sub_analysis: ",sub_analysis_type," not supported yet"));
	}
	else if (analysis_type==SlopecomputeAnalysisEnum()){

		CreatePVectorSlopeCompute( pg,inputs,analysis_type,sub_analysis_type);
	}
	else if (analysis_type==PrognosticAnalysisEnum()){

		CreatePVectorPrognostic( pg,inputs,analysis_type,sub_analysis_type);
	}
	else if (analysis_type==BalancedthicknessAnalysisEnum()){

		CreatePVectorBalancedthickness( pg,inputs,analysis_type,sub_analysis_type);
	}
	else if (analysis_type==BalancedvelocitiesAnalysisEnum()){

		CreatePVectorBalancedvelocities( pg,inputs,analysis_type,sub_analysis_type);
	}
	else if (analysis_type==ThermalAnalysisEnum()){

		CreatePVectorThermal( pg,inputs,analysis_type,sub_analysis_type);
	}
	else if (analysis_type==MeltingAnalysisEnum()){

		CreatePVectorMelting( pg,inputs,analysis_type,sub_analysis_type);
	}
	else{
		throw ErrorException(__FUNCT__,exprintf("%s%i%s\n","analysis: ",analysis_type," not supported yet"));
	}	

}
/*}}}*/
/*FUNCTION CreatePVectorBalancedthickness {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::CreatePVectorBalancedthickness"

void Penta::CreatePVectorBalancedthickness( Vec pg, void* inputs, int analysis_type,int sub_analysis_type){

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*If on water, skip: */
	if(onwater)return;

	/*Is this element on the bed? :*/
	if(!onbed)return;

	/*Spawn Tria element from the base of the Penta: */
	tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	tria->CreatePVector(pg,inputs, analysis_type,sub_analysis_type);
	delete tria;
	return;
}
/*}}}*/
/*FUNCTION CreatePVectorBalancedvelocities {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::CreatePVectorBalancedvelocities"

void Penta::CreatePVectorBalancedvelocities( Vec pg, void* inputs, int analysis_type,int sub_analysis_type){

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*If on water, skip: */
	if(onwater)return;

	/*Is this element on the bed? :*/
	if(!onbed)return;

	/*Spawn Tria element from the base of the Penta: */
	tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	tria->CreatePVector(pg,inputs, analysis_type,sub_analysis_type);
	delete tria;
	return;
}
/*}}}*/
/*FUNCTION CreatePVectorDiagnosticHoriz {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::CreatePVectorDiagnosticHoriz"
void Penta::CreatePVectorDiagnosticHoriz( Vec pg, void* vinputs,int analysis_type,int sub_analysis_type){

	int i,j;

	/* node data: */
	const int    numgrids=6;
	const int    NDOF2=2;
	const int    numdof=NDOF2*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	/* parameters: */
	double  slope[3]; //do not put 2! this goes into GetParameterDerivativeValue, which addresses slope[3] also!
	double  driving_stress_baseline;
	double  thickness;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* fourth_gauss_vert_coord  =  NULL;
	double* area_gauss_weights      =  NULL;
	double* vert_gauss_weights      =  NULL;
	double  gauss_coord[4];
	int     order_area_gauss;
	int     num_vert_gauss;
	int     num_area_gauss;
	int     ig1,ig2;
	double  gauss_weight1,gauss_weight2;
	double  gauss_weight;

	/* Jacobian: */
	double Jdet;

	/*nodal functions: */
	double l1l6[6];

	/*element vector at the gaussian points: */
	double  pe_g[numdof];
	double  pe_g_gaussian[numdof];

	ParameterInputs* inputs=NULL;

	/*Spawning: */
	Tria* tria=NULL;

	/*If on water, skip load: */
	if(onwater)return;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;


	/*Figure out if this pentaelem is collapsed. If so, then bailout, except if it is at the 
	  bedrock, in which case we spawn a tria element using the 3 first grids, and use it to build 
	  the load vector. */

	if ((collapse==1) && (onbed==0)){
		/*This element should be collapsed, but this element is not on the bedrock, therefore all its 
		 * dofs have already been frozen! Do nothing: */
		return;
	}
	else if ((collapse==1) && (onbed==1)){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 *and use its CreatePVector functionality to return an elementary load vector: */
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
		tria->CreatePVector(pg,inputs, analysis_type,sub_analysis_type);
		delete tria;
		return;
	}
	else{

		/*Implement standard penta element: */

		/* Get node coordinates and dof list: */
		GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
		GetDofList(&doflist[0],&numberofdofspernode);

		/* Set pe_g to 0: */
		for(i=0;i<numdof;i++) pe_g[i]=0.0;

		/*Get gaussian points and weights :*/
		order_area_gauss=2;
		num_vert_gauss=3;

		GaussPenta( &num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &area_gauss_weights, &fourth_gauss_vert_coord,&vert_gauss_weights,order_area_gauss,num_vert_gauss);
#ifdef _ISSM_DEBUG_ 
		for (i=0;i<num_area_gauss;i++){
			printf("Area Gauss coord %i: %lf %lf %lf Weight: %lf\n",i,*(first_gauss_area_coord+i),*(second_gauss_area_coord+i),*(third_gauss_area_coord+i),*(area_gauss_weights+i));
		}
		for (i=0;i<num_vert_gauss;i++){
			printf("Vert Gauss coord %i: %lf Weight: %lf\n",i,*(fourth_gauss_vert_coord+i),*(vert_gauss_weights+i));
		}
#endif

		/* Start  looping on the number of gaussian points: */
		for (ig1=0; ig1<num_area_gauss; ig1++){
			for (ig2=0; ig2<num_vert_gauss; ig2++){

				/*Pick up the gaussian point: */
				gauss_weight1=*(area_gauss_weights+ig1);
				gauss_weight2=*(vert_gauss_weights+ig2);
				gauss_weight=gauss_weight1*gauss_weight2;

				gauss_coord[0]=*(first_gauss_area_coord+ig1); 
				gauss_coord[1]=*(second_gauss_area_coord+ig1);
				gauss_coord[2]=*(third_gauss_area_coord+ig1);
				gauss_coord[3]=*(fourth_gauss_vert_coord+ig2);

				/*Compute thickness at gaussian point: */
				GetParameterValue(&thickness, &h[0],gauss_coord);

				/*Compute slope at gaussian point: */
				GetParameterDerivativeValue(&slope[0], &s[0],&xyz_list[0][0], gauss_coord);

				/* Get Jacobian determinant: */
				GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_coord);

				/*Get nodal functions: */
				GetNodalFunctions(l1l6, gauss_coord);

				/*Compute driving stress: */
				driving_stress_baseline=matpar->GetRhoIce()*matpar->GetG();

				/*Build pe_g_gaussian vector: */
				for (i=0;i<numgrids;i++){
					for (j=0;j<NDOF2;j++){
						pe_g_gaussian[i*NDOF2+j]=-driving_stress_baseline*slope[j]*Jdet*gauss_weight*l1l6[i];
					}
				}

				/*Add pe_g_gaussian vector to pe_g: */
				for( i=0; i<numdof; i++)pe_g[i]+=pe_g_gaussian[i];

			} //for (ig2=0; ig2<num_vert_gauss; ig2++)
		} //for (ig1=0; ig1<num_area_gauss; ig1++)

	} //else if ((collapse==1) && (onbed==1))

	/*Add pe_g to global vector pg: */
	VecSetValues(pg,numdof,doflist,(const double*)pe_g,ADD_VALUES);

	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&fourth_gauss_vert_coord);
	xfree((void**)&area_gauss_weights);
	xfree((void**)&vert_gauss_weights);

}
/*}}}*/
/*FUNCTION CreatePVectorDiagnosticStokes {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::CreatePVectorDiagnosticStokes"
void Penta::CreatePVectorDiagnosticStokes( Vec pg, void* vinputs,int analysis_type,int sub_analysis_type){

	/*indexing: */
	int i,j;

	const int numgrids=6;
	const int DOFPERGRID=4;
	const int numdof=numgrids*DOFPERGRID;
	const int numgrids2d=3;
	int numdof2d=numgrids2d*DOFPERGRID;
	int doflist[numdof];
	int numberofdofspernode;

	/*Material properties: */
	double         gravity,rho_ice,rho_water;

	/*parameters: */
	double		   xyz_list_tria[numgrids2d][3];
	double         xyz_list[numgrids][3];
	double		   surface_normal[3];
	double		   bed_normal[3];
	double         bed;
	double         vxvyvz_list[numgrids][3];

	/* gaussian points: */
	int     num_area_gauss;
	int     igarea,igvert;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* vert_gauss_coord = NULL;
	double* area_gauss_weights  =  NULL;
	double* vert_gauss_weights  =  NULL;

	/* specific gaussian point: */
	double  gauss_weight,area_gauss_weight,vert_gauss_weight;
	double  gauss_coord[4];
	double  gauss_coord_tria[3];

	int     area_order=5;
	int	  num_vert_gauss=5;

	double  epsilon[6]; /* epsilon=[exx,eyy,ezz,exy,exz,eyz];*/
	double  viscosity;
	double  water_pressure;
	int     dofs[3]={0,1,2};

	/*matrices: */
	double     Pe_temp[27]={0.0}; //for the six nodes and the bubble 
	double     Pe_gaussian[27]={0.0}; //for the six nodes and the bubble 
	double     Ke_temp[27][3]={0.0}; //for the six nodes and the bubble 
	double     Pe_reduced[numdof]; //for the six nodes only
	double     Ke_gaussian[27][3];
	double     L[3]; //for the three nodes of the bed
	double     l1l7[7]; //for the six nodes and the bubble 
	double     B[8][27];
	double     B_prime[8][27];
	double     B_prime_bubble[8][3];
	double     Jdet;
	double     Jdet2d;
	double     D[8][8]={0.0};
	double     D_scalar;
	double     tBD[27][8];
	double     P_terms[numdof];

	ParameterInputs* inputs=NULL;
	Tria*            tria=NULL;

	/*If on water, skip load: */
	if(onwater)return;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;

	/* Set P_terms to 0: */
	for(i=0;i<numdof;i++){
		P_terms[i]=0;
	}	

	/*recovre material parameters: */
	rho_water=matpar->GetRhoWater();
	rho_ice=matpar->GetRhoIce();
	gravity=matpar->GetG();

	/*recover extra inputs from users, at current convergence iteration: */
	inputs->Recover("velocity",&vxvyvz_list[0][0],3,dofs,numgrids,(void**)nodes);

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Get gaussian points and weights. Penta is an extrusion of a Tria, we therefore 
		get tria gaussian points as well as segment gaussian points. For tria gaussian 
		points, order of integration is 2, because we need to integrate the product tB*D*B' 
		which is a polynomial of degree 3 (see GaussTria for more details). For segment gaussian 
		points, same deal, which yields 3 gaussian points.*/

	/* Get gaussian points and weights (make this a statically initialized list of points? fstd): */
	GaussPenta( &num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &area_gauss_weights,&vert_gauss_coord, &vert_gauss_weights, area_order, num_vert_gauss);

	/* Start  looping on the number of gaussian points: */
	for (igarea=0; igarea<num_area_gauss; igarea++){
		for (igvert=0; igvert<num_vert_gauss; igvert++){
			/*Pick up the gaussian point: */
			area_gauss_weight=*(area_gauss_weights+igarea);
			vert_gauss_weight=*(vert_gauss_weights+igvert);
			gauss_weight=area_gauss_weight*vert_gauss_weight;
			gauss_coord[0]=*(first_gauss_area_coord+igarea); 
			gauss_coord[1]=*(second_gauss_area_coord+igarea);
			gauss_coord[2]=*(third_gauss_area_coord+igarea);
			gauss_coord[3]=*(vert_gauss_coord+igvert);

			/*Compute strain rate and viscosity: */
			GetStrainRateStokes(&epsilon[0],&vxvyvz_list[0][0],&xyz_list[0][0],gauss_coord);
			matice->GetViscosity3dStokes(&viscosity,&epsilon[0]);

			/* Get Jacobian determinant: */
			GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_coord);

			/* Get nodal functions */
			GetNodalFunctionsStokes(&l1l7[0], gauss_coord);

			/* Build gaussian vector */
			for(i=0;i<numgrids+1;i++){
				Pe_gaussian[i*DOFPERGRID+2]=-rho_ice*gravity*Jdet*gauss_weight*l1l7[i];
			}

			/*Add Pe_gaussian to terms in Pe_temp. Watch out for column orientation from matlab: */
			for(i=0;i<27;i++){
				Pe_temp[i]+=Pe_gaussian[i];
			}

			/*Get B and Bprime matrices: */
			GetBStokes(&B[0][0],&xyz_list[0][0],gauss_coord); 
			GetBprimeStokes(&B_prime[0][0],&xyz_list[0][0], gauss_coord); 

			/*Get bubble part of Bprime */
			for(i=0;i<8;i++){
				for(j=0;j<3;j++){
					B_prime_bubble[i][j]=B_prime[i][j+24];
				}
			}

			/* Build the D matrix: we plug the gaussian weight, the thickness, the viscosity, and the jacobian determinant 
			 * onto this scalar matrix, so that we win some computational time: */
			D_scalar=gauss_weight*Jdet;
			for (i=0;i<6;i++){
				D[i][i]=D_scalar*viscosity;
			}
			for (i=6;i<8;i++){
				D[i][i]=-D_scalar*numpar->stokesreconditioning;
			}

			/*  Do the triple product tB*D*Bprime: */
			MatrixMultiply(&B[0][0],8,27,1,&D[0][0],8,8,0,&tBD[0][0],0);
			MatrixMultiply(&tBD[0][0],27,8,0,&B_prime_bubble[0][0],8,3,0,&Ke_gaussian[0][0],0);

			/*Add Ke_gaussian and Ke_gaussian to terms in pKe. Watch out for column orientation from matlab: */
			for(i=0;i<27;i++){
				for(j=0;j<3;j++){
					Ke_temp[i][j]+=Ke_gaussian[i][j];
				}
			}
		}
	}

	/*Deal with 2d friction at the bedrock interface: */
	if ( (onbed==1) && (shelf==1)){

		for(i=0;i<numgrids2d;i++){
			for(j=0;j<3;j++){
				xyz_list_tria[i][j]=xyz_list[i][j];
			}
		}

		xfree((void**)&first_gauss_area_coord); xfree((void**)&second_gauss_area_coord); xfree((void**)&third_gauss_area_coord); xfree((void**)&area_gauss_weights);
		GaussTria (&num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &area_gauss_weights, 2);

		/* Start looping on the number of gauss 2d (nodes on the bedrock) */
		for (igarea=0; igarea<num_area_gauss; igarea++){
			gauss_weight=*(area_gauss_weights+igarea);
			gauss_coord[0]=*(first_gauss_area_coord+igarea); 
			gauss_coord[1]=*(second_gauss_area_coord+igarea);
			gauss_coord[2]=*(third_gauss_area_coord+igarea);
			gauss_coord[3]=-1;

			gauss_coord_tria[0]=*(first_gauss_area_coord+igarea); 
			gauss_coord_tria[1]=*(second_gauss_area_coord+igarea);
			gauss_coord_tria[2]=*(third_gauss_area_coord+igarea);

			/*Get the Jacobian determinant */
			tria->GetJacobianDeterminant3d(&Jdet2d, &xyz_list_tria[0][0], gauss_coord_tria);

			/* Get bed at gaussian point */
			GetParameterValue(&bed,&b[0],gauss_coord);

			/*Get L matrix : */
			tria->GetL(&L[0], &xyz_list[0][0], gauss_coord_tria,1);

			/*Get water_pressure at gaussian point */
			water_pressure=gravity*rho_water*bed;

			/*Get normal vecyor to the bed */
			SurfaceNormal(&surface_normal[0],xyz_list_tria);

			bed_normal[0]=-surface_normal[0]; //Program is for surface, so the normal to the bed is the opposite of the result
			bed_normal[1]=-surface_normal[1];
			bed_normal[2]=-surface_normal[2];

			for(i=0;i<numgrids2d;i++){
				for(j=0;j<3;j++){
					Pe_temp[i*DOFPERGRID+j]+=water_pressure*gauss_weight*Jdet2d*L[i]*bed_normal[j];
				}
			}
		}
	} //if ( (onbed==1) && (shelf==1))

	/*Reduce the matrix */
	ReduceVectorStokes(&Pe_reduced[0], &Ke_temp[0][0], &Pe_temp[0]);

	for(i=0;i<numdof;i++){
		P_terms[i]+=Pe_reduced[i];
	}

	/*Add P_terms to global vector pg: */
	VecSetValues(pg,numdof,doflist,(const double*)P_terms,ADD_VALUES);

	/*Free ressources:*/
	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&area_gauss_weights);
	xfree((void**)&vert_gauss_coord);
	xfree((void**)&vert_gauss_weights);

}
/*}}}*/
/*FUNCTION CreatePVectorDiagnosticVert {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta:CreatePVectorDiagnosticVert"
void  Penta::CreatePVectorDiagnosticVert( Vec pg, void* vinputs,int analysis_type,int sub_analysis_type){

	int i;


	/* node data: */
	const int    numgrids=6;
	const int    NDOF1=1;
	const int    numdof=NDOF1*numgrids;
	double       xyz_list[numgrids][3];
	int          doflist[numdof];
	int          numberofdofspernode;

	/* gaussian points: */
	int     num_gauss,ig;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* fourth_gauss_vert_coord  =  NULL;
	double* area_gauss_weights      =  NULL;
	double* vert_gauss_weights      =  NULL;
	double  gauss_coord[4];
	int     order_area_gauss;
	int     num_vert_gauss;
	int     num_area_gauss;
	int     ig1,ig2;
	double  gauss_weight1,gauss_weight2;
	double  gauss_weight;

	/* Jacobian: */
	double Jdet;

	/*element vector at the gaussian points: */
	double  pe_g[numdof];
	double  pe_g_gaussian[numdof];
	double l1l6[6];

	/*Spawning: */
	Tria* tria=NULL;

	/*input parameters for structural analysis (diagnostic): */
	double vx_list[numgrids]={0,0,0,0,0,0};
	double vy_list[numgrids]={0,0,0,0,0,0};
	double du[3];
	double dv[3];
	double dudx,dvdy;
	int     dofs1[1]={0};
	int     dofs2[1]={1};

	ParameterInputs* inputs=NULL;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;

	/*If on water, skip: */
	if(onwater)return;

	/*If we are on the bedrock, spawn a tria on the bedrock, and use it to build the 
	 *diagnostic base vertical stifness: */
	if(onbed){
		tria=(Tria*)SpawnTria(0,1,2); //nodes 0, 1 and 2 are on the bedrock
		tria->CreatePVectorDiagnosticBaseVert(pg,inputs, analysis_type,sub_analysis_type);
		delete tria;
	}

	/*Now, handle the standard penta element: */
	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/* Set pe_g to 0: */
	for(i=0;i<numdof;i++) pe_g[i]=0.0;

	/* recover input parameters: */
	if(!inputs->Recover("velocity",&vx_list[0],1,dofs1,numgrids,(void**)nodes))throw ErrorException(__FUNCT__," cannot compute vertical velocity without horizontal velocity!");
	inputs->Recover("velocity",&vy_list[0],1,dofs2,numgrids,(void**)nodes);

	/*Get gaussian points and weights :*/
	order_area_gauss=2;
	num_vert_gauss=2;

	GaussPenta( &num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &area_gauss_weights, &fourth_gauss_vert_coord,&vert_gauss_weights,order_area_gauss,num_vert_gauss);
#ifdef _ISSM_DEBUG_ 
	for (i=0;i<num_area_gauss;i++){
		printf("Area Gauss coord %i: %lf %lf %lf Weight: %lf\n",i,*(first_gauss_area_coord+i),*(second_gauss_area_coord+i),*(third_gauss_area_coord+i),*(area_gauss_weights+i));
	}
	for (i=0;i<num_vert_gauss;i++){
		printf("Vert Gauss coord %i: %lf Weight: %lf\n",i,*(fourth_gauss_vert_coord+i),*(vert_gauss_weights+i));
	}
#endif

	/* Start  looping on the number of gaussian points: */
	for (ig1=0; ig1<num_area_gauss; ig1++){
		for (ig2=0; ig2<num_vert_gauss; ig2++){

			/*Pick up the gaussian point: */
			gauss_weight1=*(area_gauss_weights+ig1);
			gauss_weight2=*(vert_gauss_weights+ig2);
			gauss_weight=gauss_weight1*gauss_weight2;

			gauss_coord[0]=*(first_gauss_area_coord+ig1); 
			gauss_coord[1]=*(second_gauss_area_coord+ig1);
			gauss_coord[2]=*(third_gauss_area_coord+ig1);
			gauss_coord[3]=*(fourth_gauss_vert_coord+ig2);

			/*Get velocity derivative, with respect to x and y: */
			GetParameterDerivativeValue(&du[0], &vx_list[0],&xyz_list[0][0], gauss_coord);
			GetParameterDerivativeValue(&dv[0], &vy_list[0],&xyz_list[0][0], gauss_coord);
			dudx=du[0];
			dvdy=dv[1];


			/* Get Jacobian determinant: */
			GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_coord);
#ifdef _ISSM_DEBUG_ 
			printf("Element id %i Jacobian determinant: %lf\n",GetId(),Jdet);
#endif

			/*Get nodal functions: */
			GetNodalFunctions(l1l6, gauss_coord);

			/*Build pe_g_gaussian vector: */
			for (i=0;i<numgrids;i++){
				pe_g_gaussian[i]=(dudx+dvdy)*Jdet*gauss_weight*l1l6[i];
			}

			/*Add pe_g_gaussian vector to pe_g: */
			for( i=0; i<numdof; i++)pe_g[i]+=pe_g_gaussian[i];

		} //for (ig2=0; ig2<num_vert_gauss; ig2++)
	} //for (ig1=0; ig1<num_area_gauss; ig1++)

	/*Add pe_g to global vector pg: */
	VecSetValues(pg,numdof,doflist,(const double*)pe_g,ADD_VALUES);

	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&fourth_gauss_vert_coord);
	xfree((void**)&area_gauss_weights);
	xfree((void**)&vert_gauss_weights);
}
/*}}}*/
/*FUNCTION CreatePVectorMelting {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::CreatePVectorMelting"
void Penta::CreatePVectorMelting( Vec pg, void* vinputs,int analysis_type,int sub_analysis_type){
	return;
}
/*}}}*/
/*FUNCTION CreatePVectorPrognostic {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::CreatePVectorPrognostic"

void Penta::CreatePVectorPrognostic( Vec pg, void* inputs, int analysis_type,int sub_analysis_type){

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*If on water, skip: */
	if(onwater)return;

	/*Is this element on the bed? :*/
	if(!onbed)return;

	/*Spawn Tria element from the base of the Penta: */
	tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	tria->CreatePVector(pg,inputs, analysis_type,sub_analysis_type);
	delete tria;
	return;
}
/*}}}*/
/*FUNCTION CreatePVectorSlopeCompute {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::CreatePVectorSlopeCompute"

void Penta::CreatePVectorSlopeCompute( Vec pg, void* inputs, int analysis_type,int sub_analysis_type){

	/*Collapsed formulation: */
	Tria*  tria=NULL;

	/*If on water, skip: */
	if(onwater)return;

	/*Is this element on the bed? :*/
	if(!onbed)return;

	/*Spawn Tria element from the base of the Penta: */
	tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
	tria->CreatePVector(pg,inputs, analysis_type,sub_analysis_type);
	delete tria;
	return;
}
/*}}}*/
/*FUNCTION CreatePVectorThermal {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::CreatePVectorThermal"
void Penta::CreatePVectorThermal( Vec pg, void* vinputs,int analysis_type,int sub_analysis_type){


	/*indexing: */
	int i,j;
	int found=0;

	const int  numgrids=6;
	const int  NDOF1=1;
	const int  numdof=numgrids*NDOF1;
	int        doflist[numdof];
	int        numberofdofspernode;

	/*Grid data: */
	double        xyz_list[numgrids][3];

	/* gaussian points: */
	int     num_area_gauss,igarea,igvert;
	double* first_gauss_area_coord  =  NULL;
	double* second_gauss_area_coord =  NULL;
	double* third_gauss_area_coord  =  NULL;
	double* vert_gauss_coord = NULL;
	double* area_gauss_weights  =  NULL;
	double* vert_gauss_weights  =  NULL;
	double  gauss_weight,area_gauss_weight,vert_gauss_weight;
	double  gauss_coord[4];
	int     area_order=2;
	int	  num_vert_gauss=3;

	double dt;
	double vx_list[numgrids];
	double vy_list[numgrids];
	double vz_list[numgrids];
	double vxvyvz_list[numgrids][3];
	double temperature_list[numgrids];
	double temperature;

	/*Material properties: */
	double gravity,rho_ice,rho_water;
	double mixed_layer_capacity,heatcapacity;
	double beta,meltingpoint,thermal_exchange_velocity;

	/* element parameters: */
	int    friction_type;

	int    dofs[3]={0,1,2};
	int    dofs1[1]={0};

	/*matrices: */
	double P_terms[numdof]={0.0};
	double L[numdof];
	double l1l2l3[3];
	double alpha2_list[3];
	double basalfriction_list[3]={0.0};
	double basalfriction;
	double epsilon[6];
	double epsilon_sqr[3][3];
	double epsilon_matrix[3][3];

	double Jdet;
	double viscosity;
	double epsilon_eff;
	double phi;
	double t_pmp;
	double scalar;
	double scalar_def;
	double scalar_ocean;
	double scalar_transient;

	/*Collapsed formulation: */
	Tria*  tria=NULL;
	ParameterInputs* inputs=NULL;

	/*If on water, skip: */
	if(onwater)return;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;

	/* Get node coordinates and dof list: */
	GetVerticesCoordinates(&xyz_list[0][0], nodes, numgrids);
	GetDofList(&doflist[0],&numberofdofspernode);

	/*recovre material parameters: */
	rho_water=matpar->GetRhoWater();
	rho_ice=matpar->GetRhoIce();
	gravity=matpar->GetG();
	heatcapacity=matpar->GetHeatCapacity();
	beta=matpar->GetBeta();
	meltingpoint=matpar->GetMeltingPoint();

	/*recover extra inputs from users, dt and velocity: */
	found=inputs->Recover("velocity",&vxvyvz_list[0][0],3,dofs,numgrids,(void**)nodes);
	if(!found)throw ErrorException(__FUNCT__," could not find velocity in inputs!");
	found=inputs->Recover("dt",&dt);
	if(!found)throw ErrorException(__FUNCT__," could not find dt in inputs!");

	if(dt){
		found=inputs->Recover("temperature",&temperature_list[0],1,dofs1,numgrids,(void**)nodes);
		if(!found)throw ErrorException(__FUNCT__," could not find temperature in inputs!");
	}

	for(i=0;i<numgrids;i++){
		vx_list[i]=vxvyvz_list[i][0];
		vy_list[i]=vxvyvz_list[i][1];
		vz_list[i]=vxvyvz_list[i][2];
	}	

	/* Get gaussian points and weights. Penta is an extrusion of a Tria, we therefore 
		get tria gaussian points as well as segment gaussian points. For tria gaussian 
		points, order of integration is 2, because we need to integrate the product tB*D*B' 
		which is a polynomial of degree 3 (see GaussTria for more details). For segment gaussian 
		points, same deal, which yields 3 gaussian points.: */

	/*Get gaussian points: */
	GaussPenta( &num_area_gauss, &first_gauss_area_coord, &second_gauss_area_coord, &third_gauss_area_coord, &area_gauss_weights,&vert_gauss_coord, &vert_gauss_weights, area_order, num_vert_gauss);

	/* Start  looping on the number of gaussian points: */
	for (igarea=0; igarea<num_area_gauss; igarea++){
		for (igvert=0; igvert<num_vert_gauss; igvert++){
			/*Pick up the gaussian point: */
			area_gauss_weight=*(area_gauss_weights+igarea);
			vert_gauss_weight=*(vert_gauss_weights+igvert);
			gauss_weight=area_gauss_weight*vert_gauss_weight;
			gauss_coord[0]=*(first_gauss_area_coord+igarea); 
			gauss_coord[1]=*(second_gauss_area_coord+igarea);
			gauss_coord[2]=*(third_gauss_area_coord+igarea);
			gauss_coord[3]=*(vert_gauss_coord+igvert);

			/*Compute strain rate and viscosity: */
			GetStrainRateStokes(&epsilon[0],&vxvyvz_list[0][0],&xyz_list[0][0],gauss_coord);
			matice->GetViscosity3dStokes(&viscosity,&epsilon[0]);

			/* Get Jacobian determinant: */
			GetJacobianDeterminant(&Jdet, &xyz_list[0][0],gauss_coord);

			/* Get nodal functions */
			GetNodalFunctions(&L[0], gauss_coord);

			/*Build deformational heating: */
			GetPhi(&phi, &epsilon[0], viscosity);

			/*Build pe_gaussian */
			scalar_def=phi/(rho_ice*heatcapacity)*Jdet*gauss_weight;
			if(dt){
				scalar_def=scalar_def*dt;
			}

			for(i=0;i<numgrids;i++){
				P_terms[i]+=scalar_def*L[i];
			}

			/* Build transient now */
			if(dt){
				GetParameterValue(&temperature, &temperature_list[0],gauss_coord);
				scalar_transient=temperature*Jdet*gauss_weight;
				for(i=0;i<numgrids;i++){
					P_terms[i]+=scalar_transient*L[i];
				}
			}
		}
	}

	/*Add pe_g to global vector pg: */
	VecSetValues(pg,numdof,doflist,(const double*)P_terms,ADD_VALUES);

	/* Ice/ocean heat exchange flux on ice shelf base */
	if(onbed && shelf){

		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
		tria->CreatePVectorThermalShelf(pg,inputs, analysis_type,sub_analysis_type);
		delete tria;
	}

	/* Geothermal flux on ice sheet base and basal friction */
	if(onbed && !shelf){

		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
		tria->CreatePVectorThermalSheet(pg,inputs, analysis_type,sub_analysis_type);
		delete tria;
	}
	extern int my_rank;

	xfree((void**)&first_gauss_area_coord);
	xfree((void**)&second_gauss_area_coord);
	xfree((void**)&third_gauss_area_coord);
	xfree((void**)&vert_gauss_coord);
	xfree((void**)&area_gauss_weights);
	xfree((void**)&vert_gauss_weights);

}
/*}}}*/
/*FUNCTION DeepEcho {{{1*/
void Penta::DeepEcho(void){

	printf("Penta:\n");
	printf("   id: %i\n",id);
	printf("   mid: %i\n",mid);
	printf("   mparid: %i\n",mparid);
	printf("   numparid: %i\n",numparid);

	printf("   node_ids=[%i,%i,%i,%i,%i,%i]\n",node_ids[0],node_ids[1],node_ids[2],node_ids[3],node_ids[4],node_ids[5]);
	printf("   node_offsets=[%i,%i,%i,%i,%i,%i]\n",node_offsets[0],node_offsets[1],node_offsets[2],node_offsets[3],node_offsets[4],node_offsets[5]);
	printf("   matice_offset=%i\n",matice_offset);
	printf("   matpar_offset=%i\n",matpar_offset);

	printf("   h=[%i,%i,%i,%i,%i,%i]\n",h[0],h[1],h[2],h[3],h[4],h[5]);
	printf("   s=[%i,%i,%i,%i,%i,%i]\n",s[0],s[1],s[2],s[3],s[4],s[5]);
	printf("   b=[%i,%i,%i,%i,%i,%i]\n",b[0],b[1],b[2],b[3],b[4],b[5]);
	printf("   k=[%i,%i,%i,%i,%i,%i]\n",k[0],k[1],k[2],k[3],k[4],k[5]);

	printf("   friction_type: %i\n",friction_type);
	printf("   p: %g\n",p);
	printf("   q: %g\n",q);
	printf("   shelf: %i\n",shelf);
	printf("   onbed: %i\n",onbed);
	printf("   onwater: %i\n",onwater);
	printf("   onsurface: %i\n",onsurface);
	printf("   collapse: %i\n",collapse);

	printf("   melting=[%i,%i,%i,%i,%i,%i]\n",melting[0],melting[1],melting[2],melting[3],melting[4],melting[5]);
	printf("   accumulation=[%i,%i,%i,%i,%i,%i]\n",accumulation[0],accumulation[1],accumulation[2],accumulation[3],accumulation[4],accumulation[5]);
	printf("   geothermalflux=[%i,%i,%i,%i,%i,%i]\n",geothermalflux[0],geothermalflux[1],geothermalflux[2],geothermalflux[3],geothermalflux[4],geothermalflux[5]);
	printf("   thermal_steadystate: %i\n",thermal_steadystate);
	return;
}
/*}}}*/
/*FUNCTION Du {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::Du"
void  Penta::Du(Vec du_g,void* inputs,int analysis_type,int sub_analysis_type){

	int i;
	Tria* tria=NULL;

	/*If on water, skip: */
	if(onwater)return;

	/*Bail out if this element if:
	 * -> Non collapsed and not on the surface
	 * -> collapsed (2d model) and not on bed) */
	if ((!collapse && !onsurface) || (collapse && !onbed)){
		return;
	}
	else if (collapse){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 * and compute Du*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria (lower face).
		tria->Du(du_g,inputs,analysis_type,sub_analysis_type);
		delete tria;
		return;
	}
	else{

		tria=(Tria*)SpawnTria(3,4,5); //grids 3, 4 and 5 make the new tria (upper face).
		tria->Du(du_g,inputs,analysis_type,sub_analysis_type);
		delete tria;
		return;
	}
}
/*}}}*/
/*FUNCTION Echo {{{1*/
void Penta::Echo(void){

	printf("Penta:\n");
	printf("   id: %i\n",id);
	printf("   mid: %i\n",mid);
	printf("   mparid: %i\n",mparid);
	printf("   numparid: %i\n",numparid);

	printf("   node_ids=[%i,%i,%i,%i,%i,%i]\n",node_ids[0],node_ids[1],node_ids[2],node_ids[3],node_ids[4],node_ids[5]);
	printf("   node_offsets=[%i,%i,%i,%i,%i,%i]\n",node_offsets[0],node_offsets[1],node_offsets[2],node_offsets[3],node_offsets[4],node_offsets[5]);
	printf("   matice_offset=%i\n",matice_offset);
	printf("   matpar_offset=%i\n",matpar_offset);

	printf("   h=[%g,%g,%g,%g,%g,%g]\n",h[0],h[1],h[2],h[3],h[4],h[5]);
	printf("   s=[%g,%g,%g,%g,%g,%g]\n",s[0],s[1],s[2],s[3],s[4],s[5]);
	printf("   b=[%g,%g,%g,%g,%g,%g]\n",b[0],b[1],b[2],b[3],b[4],b[5]);
	printf("   k=[%g,%g,%g,%g,%g,%g]\n",k[0],k[1],k[2],k[3],k[4],k[5]);

	printf("   friction_type: %i\n",friction_type);
	printf("   p: %g\n",p);
	printf("   q: %g\n",q);
	printf("   shelf: %i\n",shelf);
	printf("   onbed: %i\n",onbed);
	printf("   onwater: %i\n",onwater);
	printf("   onsurface: %i\n",onsurface);
	printf("   collapse: %i\n",collapse);

	printf("   melting=[%g,%g,%g,%g,%g,%g]\n",melting[0],melting[1],melting[2],melting[3],melting[4],melting[5]);
	printf("   accumulation=[%g,%g,%g,%g,%g,%g]\n",accumulation[0],accumulation[1],accumulation[2],accumulation[3],accumulation[4],accumulation[5]);
	printf("   geothermalflux=[%g,%g,%g,%g,%g,%g]\n",geothermalflux[0],geothermalflux[1],geothermalflux[2],geothermalflux[3],geothermalflux[4],geothermalflux[5]);
	printf("   thermal_steadystate: %i\n",thermal_steadystate);
	return;
}
/*}}}*/
/*FUNCTION Enum {{{1*/
int Penta::Enum(void){

	return PentaEnum();

}
/*}}}*/
/*FUNCTION FieldExtrude {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::FieldExtrude"
void  Penta::FieldExtrude(Vec field,double* field_serial,char* field_name, int iscollapsed){

	/* node data: */
	const int    numgrids=6;
	int          numberofdofspernode;
	Node* node=NULL;
	int   i;
	int   extrude=0;

	/*Figure out if we should extrude for this element: */
	if (iscollapsed){
		/*From higher level, we are told to extrude only elements that have the collapse flag on: */
		if (collapse)extrude=1;
		else extrude=0;
	}
	else{
		/*From higher level, we are told to extrude all elements: */
		extrude=1;
	}

	/*Now, extrusion starts from the bed on, so double check this element is on 
	 * the bedrock: */
	if(onbed==0)extrude=0;

	/*Go on and extrude field: */
	if (extrude){

		if (strcmp(field_name,"velocity")==0){

			/* node data: */
			const int    numdof=2*numgrids;
			int          doflist[numdof];
			int          nodedofs[2];
			double       fieldel[2];


			GetDofList(&doflist[0],&numberofdofspernode);

			/*this penta is a collapsed macayeal. For each node on the base of this penta, 
			 * we grab the field. Once we know the field, we follow the upper nodes, 
			 * inserting the same field value into field, until we reach the surface: */
			for(i=0;i<3;i++){

				node=nodes[i]; //base nodes

				/*get field for this base node: */
				fieldel[0]=field_serial[doflist[numberofdofspernode*i+0]];
				fieldel[1]=field_serial[doflist[numberofdofspernode*i+1]];

				//go throfieldn all nodes which sit on top of this node, until we reach the surface, 
				//and plfield  field in field
				for(;;){

					node->GetDofList(&nodedofs[0],&numberofdofspernode);
					VecSetValues(field,1,&nodedofs[0],&fieldel[0],INSERT_VALUES);
					VecSetValues(field,1,&nodedofs[1],&fieldel[1],INSERT_VALUES);

					if (node->IsOnSurface())break;
					/*get next node: */
					node=node->GetUpperNode();
				}
			}
		} //if (strcmp(field_name,"velocity")==0)
		else if (strcmp(field_name,"gradj")==0){

			/* node data: */
			int          dof1;
			double       fieldel;

			/*this penta is a collapsed macayeal. For each node on the base of this penta, 
			 * we grab the field. Once we know the field, we follow the upper nodes, 
			 * inserting the same field value into field, until we reach the surface: */
			for(i=0;i<3;i++){

				node=nodes[i]; //base nodes
				dof1=node->GetDofList1();

				/*get field for this base node: */
				fieldel=field_serial[dof1];

				//go throfieldn all nodes which sit on top of this node, until we reach the surface, 
				//and plfield  field in field
				for(;;){

					dof1=node->GetDofList1();
					VecSetValues(field,1,&dof1,&fieldel,INSERT_VALUES);

					if (node->IsOnSurface())break;
					/*get next node: */
					node=node->GetUpperNode();
				}
			}
		}
		else if ( 
					(strcmp(field_name,"thickness")==0) ||
					(strcmp(field_name,"surface")==0)  ||
					(strcmp(field_name,"bed")==0)  ||
					(strcmp(field_name,"slopex")==0)  ||
					(strcmp(field_name,"slopey")==0)
				  ){

			/* node data: */
			const int    numdof=1*numgrids;
			int          doflist[numdof];
			int          nodedofs;
			double       fieldel;

			GetDofList(&doflist[0],&numberofdofspernode);

			/*this penta is on the bed. For each node on the base of this penta, 
			 * we grab the thickness. Once we know the thickness, we follow the upper nodes, 
			 * inserting the same thickness value into tg, until we reach the surface: */
			for(i=0;i<3;i++){

				node=nodes[i]; //base nodes

				/*get velocity for this base node: */
				fieldel=field_serial[doflist[numberofdofspernode*i+0]];

				//go through all nodes which sit on top of this node, until we reach the surface, 
				//and pltg  fieldel in field:
				for(;;){

					node->GetDofList(&nodedofs,&numberofdofspernode);
					VecSetValues(field,1,&nodedofs,&fieldel,INSERT_VALUES);

					if (node->IsOnSurface())break;
					/*get next node: */
					node=node->GetUpperNode();
				}
			}

		}
		else throw ErrorException(__FUNCT__,exprintf("%s%s%s"," field ",field_name," not supported yet!"));

	} //if (extrude)
}
/*}}}*/
/*FUNCTION GetB {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::GetB"
void Penta::GetB(double* B, double* xyz_list, double* gauss_coord){

	/*Compute B  matrix. B=[B1 B2 B3 B4 B5 B6] where Bi is of size 5*NDOF2. 
	 * For grid i, Bi can be expressed in the actual coordinate system
	 * by: 
	 *       Bi=[ dh/dx          0      ]
	 *                [   0           dh/dy   ]
	 *                [ 1/2*dh/dy  1/2*dh/dx  ]
	 *                [ 1/2*dh/dz      0      ]
	 *                [  0         1/2*dh/dz  ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B has been allocated already, of size: 5x(NDOF2*numgrids)
	 */

	int i;
	const int numgrids=6;
	const int NDOF3=3;
	const int NDOF2=2;

	double dh1dh6[NDOF3][numgrids];

	/*Get dh1dh6 in actual coordinate system: */
	GetNodalFunctionsDerivatives(&dh1dh6[0][0],xyz_list, gauss_coord);

#ifdef _ISSM_DEBUG_ 
	for (i=0;i<numgrids;i++){
		printf("Node %i  dh/dx=%lf dh/dy=%lf dh/dz=%lf\n",i,dh1dh6[0][i],dh1dh6[1][i],dh1dh6[2][i]);
	}
#endif

	/*Build B: */
	for (i=0;i<numgrids;i++){
		*(B+NDOF2*numgrids*0+NDOF2*i)=dh1dh6[0][i]; 
		*(B+NDOF2*numgrids*0+NDOF2*i+1)=0.0;

		*(B+NDOF2*numgrids*1+NDOF2*i)=0.0;
		*(B+NDOF2*numgrids*1+NDOF2*i+1)=dh1dh6[1][i];

		*(B+NDOF2*numgrids*2+NDOF2*i)=(float).5*dh1dh6[1][i]; 
		*(B+NDOF2*numgrids*2+NDOF2*i+1)=(float).5*dh1dh6[0][i]; 

		*(B+NDOF2*numgrids*3+NDOF2*i)=(float).5*dh1dh6[2][i]; 
		*(B+NDOF2*numgrids*3+NDOF2*i+1)=0.0;

		*(B+NDOF2*numgrids*4+NDOF2*i)=0.0;
		*(B+NDOF2*numgrids*4+NDOF2*i+1)=(float).5*dh1dh6[2][i]; 
	}

}
/*}}}*/
/*FUNCTION GetB_artdiff {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::GetB_artdiff"
void Penta::GetB_artdiff(double* B_artdiff, double* xyz_list, double* gauss_coord){

	/*Compute B  matrix. B=[B1 B2 B3 B4 B5 B6] where Bi is of size 5*DOFPERGRID. 
	 * For grid i, Bi' can be expressed in the actual coordinate system
	 * by: 
	 *       Bi_artdiff=[ dh/dx ]
	 *                       [ dh/dy ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B has been allocated already, of size: 2x(DOFPERGRID*numgrids)
	 */

	int i;
	const int calculationdof=3;
	const int numgrids=6;
	int DOFPERGRID=1;

	/*Same thing in the actual coordinate system: */
	double dh1dh6[calculationdof][numgrids];

	/*Get dh1dh2dh3 in actual coordinates system : */
	GetNodalFunctionsDerivatives(&dh1dh6[0][0],xyz_list,gauss_coord);

	/*Build B': */
	for (i=0;i<numgrids;i++){
		*(B_artdiff+DOFPERGRID*numgrids*0+DOFPERGRID*i)=dh1dh6[0][i]; 
		*(B_artdiff+DOFPERGRID*numgrids*1+DOFPERGRID*i)=dh1dh6[1][i]; 
	}
}
/*}}}*/
/*FUNCTION GetB_advec {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::GetB_advec"
void Penta::GetB_advec(double* B_advec, double* xyz_list, double* gauss_coord){

	/*Compute B  matrix. B=[B1 B2 B3 B4 B5 B6] where Bi is of size 5*DOFPERGRID. 
	 * For grid i, Bi' can be expressed in the actual coordinate system
	 * by: 
	 *       Bi_advec =[ h ]
	 *                       [ h ]
	 *                       [ h ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B has been allocated already, of size: 3x(DOFPERGRID*numgrids)
	 */

	int i;
	int calculationdof=3;
	int numgrids=6;
	int DOFPERGRID=1;

	/*Same thing in the actual coordinate system: */
	double l1l6[6];

	/*Get dh1dh2dh3 in actual coordinates system : */
	GetNodalFunctions(l1l6, gauss_coord);

	/*Build B': */
	for (i=0;i<numgrids;i++){
		*(B_advec+DOFPERGRID*numgrids*0+DOFPERGRID*i)=l1l6[i]; 
		*(B_advec+DOFPERGRID*numgrids*1+DOFPERGRID*i)=l1l6[i]; 
		*(B_advec+DOFPERGRID*numgrids*2+DOFPERGRID*i)=l1l6[i]; 
	}
}
/*}}}*/
/*FUNCTION GetB_conduct {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::GetB_conduct"
void Penta::GetB_conduct(double* B_conduct, double* xyz_list, double* gauss_coord){

	/*Compute B  matrix. B=[B1 B2 B3 B4 B5 B6] where Bi is of size 5*DOFPERGRID. 
	 * For grid i, Bi' can be expressed in the actual coordinate system
	 * by: 
	 *       Bi_conduct=[ dh/dx ]
	 *                       [ dh/dy ]
	 *                       [ dh/dz ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B has been allocated already, of size: 3x(DOFPERGRID*numgrids)
	 */

	int i;
	const int calculationdof=3;
	const int numgrids=6;
	int DOFPERGRID=1;

	/*Same thing in the actual coordinate system: */
	double dh1dh6[calculationdof][numgrids];

	/*Get dh1dh2dh3 in actual coordinates system : */
	GetNodalFunctionsDerivatives(&dh1dh6[0][0],xyz_list,gauss_coord);

	/*Build B': */
	for (i=0;i<numgrids;i++){
		*(B_conduct+DOFPERGRID*numgrids*0+DOFPERGRID*i)=dh1dh6[0][i]; 
		*(B_conduct+DOFPERGRID*numgrids*1+DOFPERGRID*i)=dh1dh6[1][i]; 
		*(B_conduct+DOFPERGRID*numgrids*2+DOFPERGRID*i)=dh1dh6[2][i]; 
	}
}
/*}}}*/
/*FUNCTION GetB_vert {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta:GetB_vert"
void Penta::GetB_vert(double* B, double* xyz_list, double* gauss_coord){


	/*	Compute B  matrix. B=[dh1/dz dh2/dz dh3/dz dh4/dz dh5/dz dh6/dz];
		where hi is the interpolation function for grid i.*/

	int i;
	const int NDOF3=3;
	const int numgrids=6;
	double dh1dh6[NDOF3][numgrids];

	/*Get dh1dh6 in actual coordinate system: */
	GetNodalFunctionsDerivatives(&dh1dh6[0][0],xyz_list, gauss_coord);

#ifdef _ISSM_DEBUG_ 
	for (i=0;i<numgrids;i++){
		printf("Node %i  dh/dx=%lf dh/dy=%lf dh/dz=%lf\n",i,dh1dh6[0][i],dh1dh6[1][i],dh1dh6[2][i]);
	}
#endif

	/*Build B: */
	for (i=0;i<numgrids;i++){
		B[i]=dh1dh6[2][i];  
	}

}
/*}}}*/
/*FUNCTION GetBedList {{{1*/
void Penta::GetBedList(double* bed_list){

	int i;
	for(i=0;i<6;i++)bed_list[i]=b[i];

}
/*}}}*/
/*FUNCTION GetBPrime {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::GetBPrime"
void Penta::GetBPrime(double* B, double* xyz_list, double* gauss_coord){

	/*Compute B  prime matrix. B=[B1 B2 B3 B4 B5 B6] where Bi is of size 5*NDOF2. 
	 * For grid i, Bi can be expressed in the actual coordinate system
	 * by: 
	 *       Bi=[ 2*dh/dx     dh/dy   ]
	 *                [   dh/dx    2*dh/dy  ]
	 *                [ dh/dy      dh/dx    ]
	 *                [ dh/dz         0     ]
	 *                [  0         dh/dz    ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B has been allocated already, of size: 5x(NDOF2*numgrids)
	 */

	int i;
	const int NDOF3=3;
	const int NDOF2=2;
	const int numgrids=6;

	double dh1dh6[NDOF3][numgrids];

	/*Get dh1dh6 in actual coordinate system: */
	GetNodalFunctionsDerivatives(&dh1dh6[0][0],xyz_list, gauss_coord);

#ifdef _ISSM_DEBUG_ 
	for (i=0;i<numgrids;i++){
		printf("Node %i  dh/dx=%lf dh/dy=%lf dh/dz=%lf\n",i,dh1dh6[0][i],dh1dh6[1][i],dh1dh6[2][i]);
	}
#endif

	/*Build BPrime: */
	for (i=0;i<numgrids;i++){
		*(B+NDOF2*numgrids*0+NDOF2*i)=2.0*dh1dh6[0][i]; 
		*(B+NDOF2*numgrids*0+NDOF2*i+1)=dh1dh6[1][i];

		*(B+NDOF2*numgrids*1+NDOF2*i)=dh1dh6[0][i];
		*(B+NDOF2*numgrids*1+NDOF2*i+1)=2.0*dh1dh6[1][i];

		*(B+NDOF2*numgrids*2+NDOF2*i)=dh1dh6[1][i]; 
		*(B+NDOF2*numgrids*2+NDOF2*i+1)=dh1dh6[0][i]; 

		*(B+NDOF2*numgrids*3+NDOF2*i)=dh1dh6[2][i]; 
		*(B+NDOF2*numgrids*3+NDOF2*i+1)=0.0;

		*(B+NDOF2*numgrids*4+NDOF2*i)=0.0;
		*(B+NDOF2*numgrids*4+NDOF2*i+1)=dh1dh6[2][i]; 
	}
}
/*}}}*/
/*FUNCTION GetBprime_advec {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::GetBprime_advec"
void Penta::GetBprime_advec(double* Bprime_advec, double* xyz_list, double* gauss_coord){


	/*Compute B  matrix. B=[B1 B2 B3 B4 B5 B6] where Bi is of size 5*DOFPERGRID. 
	 * For grid i, Bi' can be expressed in the actual coordinate system
	 * by: 
	 *       Biprime_advec=[ dh/dx ]
	 *                       [ dh/dy ]
	 *                       [ dh/dz ]
	 * where h is the interpolation function for grid i.
	 *
	 * We assume B has been allocated already, of size: 3x(DOFPERGRID*numgrids)
	 */

	int i;
	const int calculationdof=3;
	const int numgrids=6;
	int DOFPERGRID=1;

	/*Same thing in the actual coordinate system: */
	double dh1dh6[calculationdof][numgrids];

	/*Get dh1dh2dh3 in actual coordinates system : */
	GetNodalFunctionsDerivatives(&dh1dh6[0][0],xyz_list,gauss_coord);

	/*Build B': */
	for (i=0;i<numgrids;i++){
		*(Bprime_advec+DOFPERGRID*numgrids*0+DOFPERGRID*i)=dh1dh6[0][i]; 
		*(Bprime_advec+DOFPERGRID*numgrids*1+DOFPERGRID*i)=dh1dh6[1][i]; 
		*(Bprime_advec+DOFPERGRID*numgrids*2+DOFPERGRID*i)=dh1dh6[2][i]; 
	}
}
/*}}}*/
/*FUNCTION GetBPrime_vert {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta:GetBPrime_vert"
void Penta::GetBPrime_vert(double* B, double* xyz_list, double* gauss_coord){

	// Compute Bprime  matrix. Bprime=[L1 L2 L3 L4 L5 L6] where Li is the nodal function for grid i

	int i;

	GetNodalFunctions(B, gauss_coord);

}
/*}}}*/
/*FUNCTION GetBprimeStokes {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::GetBprimeStokes"
void Penta::GetBprimeStokes(double* B_prime, double* xyz_list, double* gauss_coord){

	/*	Compute B'  matrix. B'=[B1' B2' B3' B4' B5' B6' Bb'] where Bi' is of size 3*NDOF2. 
	 *	For grid i, Bi' can be expressed in the actual coordinate system
	 *	by: 
	 *				Bi'=[ dh/dx   0          0       0]
	 *					 [   0      dh/dy      0       0]
	 *					 [   0      0         dh/dz    0]
	 *					 [  dh/dy   dh/dx      0       0]
	 *					 [  dh/dz   0        dh/dx     0]
	 *					 [   0      dh/dz    dh/dy     0]
	 *					 [  dh/dx   dh/dy    dh/dz     0]
	 *					 [   0      0          0       h]
	 *	where h is the interpolation function for grid i.
	 *
	 * 	Same thing for the bubble fonction except that there is no fourth column
	 */

	int i;
	const int calculationdof=3;
	const int numgrids=6;
	int DOFPERGRID=4;

	double dh1dh7[calculationdof][numgrids+1];
	double l1l6[numgrids];

	/*Get dh1dh7 in actual coordinate system: */
	GetNodalFunctionsDerivativesStokes(&dh1dh7[0][0],xyz_list, gauss_coord);

	GetNodalFunctions(l1l6, gauss_coord);

#ifdef _ISSM_DEBUG_ 
	for (i=0;i<6;i++){
		printf("Node %i  dh/dx=%lf dh/dy=%lf dh/dz=%lf \n",i,dh1dh7[0][i],dh1dh7[1][i]);
	}

#endif

	/*B_primeuild B_prime: */
	for (i=0;i<numgrids+1;i++){
		*(B_prime+(DOFPERGRID*numgrids+3)*0+DOFPERGRID*i)=dh1dh7[0][i]; //B_prime[0][DOFPERGRID*i]=dh1dh6[0][i];
		*(B_prime+(DOFPERGRID*numgrids+3)*0+DOFPERGRID*i+1)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*0+DOFPERGRID*i+2)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*1+DOFPERGRID*i)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*1+DOFPERGRID*i+1)=dh1dh7[1][i];
		*(B_prime+(DOFPERGRID*numgrids+3)*1+DOFPERGRID*i+2)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*2+DOFPERGRID*i)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*2+DOFPERGRID*i+1)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*2+DOFPERGRID*i+2)=dh1dh7[2][i];
		*(B_prime+(DOFPERGRID*numgrids+3)*3+DOFPERGRID*i)=dh1dh7[1][i]; 
		*(B_prime+(DOFPERGRID*numgrids+3)*3+DOFPERGRID*i+1)=dh1dh7[0][i]; 
		*(B_prime+(DOFPERGRID*numgrids+3)*3+DOFPERGRID*i+2)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*4+DOFPERGRID*i)=dh1dh7[2][i];
		*(B_prime+(DOFPERGRID*numgrids+3)*4+DOFPERGRID*i+1)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*4+DOFPERGRID*i+2)=dh1dh7[0][i];
		*(B_prime+(DOFPERGRID*numgrids+3)*5+DOFPERGRID*i)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*5+DOFPERGRID*i+1)=dh1dh7[2][i];
		*(B_prime+(DOFPERGRID*numgrids+3)*5+DOFPERGRID*i+2)=dh1dh7[1][i];
		*(B_prime+(DOFPERGRID*numgrids+3)*6+DOFPERGRID*i)=dh1dh7[0][i];
		*(B_prime+(DOFPERGRID*numgrids+3)*6+DOFPERGRID*i+1)=dh1dh7[1][i];
		*(B_prime+(DOFPERGRID*numgrids+3)*6+DOFPERGRID*i+2)=dh1dh7[2][i];
		*(B_prime+(DOFPERGRID*numgrids+3)*7+DOFPERGRID*i)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*7+DOFPERGRID*i+1)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*7+DOFPERGRID*i+2)=0;
	}

	for (i=0;i<numgrids;i++){ //last column not for the bubble function
		*(B_prime+(DOFPERGRID*numgrids+3)*0+DOFPERGRID*i+3)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*1+DOFPERGRID*i+3)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*2+DOFPERGRID*i+3)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*3+DOFPERGRID*i+3)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*4+DOFPERGRID*i+3)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*5+DOFPERGRID*i+3)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*6+DOFPERGRID*i+3)=0;
		*(B_prime+(DOFPERGRID*numgrids+3)*7+DOFPERGRID*i+3)=l1l6[i];
	}

}
/*}}}*/
/*FUNCTION GetBStokes {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::GetBStokes"
void Penta::GetBStokes(double* B, double* xyz_list, double* gauss_coord){

	/*Compute B  matrix. B=[B1 B2 B3 B4 B5 B6] where Bi is of size 3*DOFPERGRID. 
	 * For grid i, Bi can be expressed in the actual coordinate system
	 * by: 				Bi=[ dh/dx          0             0       0  ]
	 *					[   0           dh/dy           0       0  ]
	 *					[   0             0           dh/dy     0  ]
	 *					[ 1/2*dh/dy    1/2*dh/dx        0       0  ]
	 *					[ 1/2*dh/dz       0         1/2*dh/dx   0  ]
	 *					[   0          1/2*dh/dz    1/2*dh/dy   0  ]
	 *					[   0             0             0       h  ]
	 *					[ dh/dx         dh/dy         dh/dz     0  ]
	 *	where h is the interpolation function for grid i.
	 *	Same thing for Bb except the last column that does not exist.
	 */

	int i;
	const int calculationdof=3;
	const int numgrids=6;
	int DOFPERGRID=4;

	double dh1dh7[calculationdof][numgrids+1];
	double l1l6[numgrids];


	/*Get dh1dh7 in actual coordinate system: */
	GetNodalFunctionsDerivativesStokes(&dh1dh7[0][0],xyz_list, gauss_coord);

	GetNodalFunctions(l1l6, gauss_coord);

#ifdef _ISSM_DEBUG_ 
	for (i=0;i<7;i++){
		printf("Node %i  dh/dx=%lf dh/dy=%lf dh/dz=%lf \n",i,dh1dh7[0][i],dh1dh7[1][i],dh1dh7[2][i]);
	}

#endif

	/*Build B: */
	for (i=0;i<numgrids+1;i++){
		*(B+(DOFPERGRID*numgrids+3)*0+DOFPERGRID*i)=dh1dh7[0][i]; //B[0][DOFPERGRID*i]=dh1dh6[0][i];
		*(B+(DOFPERGRID*numgrids+3)*0+DOFPERGRID*i+1)=0;
		*(B+(DOFPERGRID*numgrids+3)*0+DOFPERGRID*i+2)=0;
		*(B+(DOFPERGRID*numgrids+3)*1+DOFPERGRID*i)=0;
		*(B+(DOFPERGRID*numgrids+3)*1+DOFPERGRID*i+1)=dh1dh7[1][i];
		*(B+(DOFPERGRID*numgrids+3)*1+DOFPERGRID*i+2)=0;
		*(B+(DOFPERGRID*numgrids+3)*2+DOFPERGRID*i)=0;
		*(B+(DOFPERGRID*numgrids+3)*2+DOFPERGRID*i+1)=0;
		*(B+(DOFPERGRID*numgrids+3)*2+DOFPERGRID*i+2)=dh1dh7[2][i];
		*(B+(DOFPERGRID*numgrids+3)*3+DOFPERGRID*i)=(float).5*dh1dh7[1][i]; 
		*(B+(DOFPERGRID*numgrids+3)*3+DOFPERGRID*i+1)=(float).5*dh1dh7[0][i]; 
		*(B+(DOFPERGRID*numgrids+3)*3+DOFPERGRID*i+2)=0;
		*(B+(DOFPERGRID*numgrids+3)*4+DOFPERGRID*i)=(float).5*dh1dh7[2][i];
		*(B+(DOFPERGRID*numgrids+3)*4+DOFPERGRID*i+1)=0;
		*(B+(DOFPERGRID*numgrids+3)*4+DOFPERGRID*i+2)=(float).5*dh1dh7[0][i];
		*(B+(DOFPERGRID*numgrids+3)*5+DOFPERGRID*i)=0;
		*(B+(DOFPERGRID*numgrids+3)*5+DOFPERGRID*i+1)=(float).5*dh1dh7[2][i];
		*(B+(DOFPERGRID*numgrids+3)*5+DOFPERGRID*i+2)=(float).5*dh1dh7[1][i];
		*(B+(DOFPERGRID*numgrids+3)*6+DOFPERGRID*i)=0;
		*(B+(DOFPERGRID*numgrids+3)*6+DOFPERGRID*i+1)=0;
		*(B+(DOFPERGRID*numgrids+3)*6+DOFPERGRID*i+2)=0;
		*(B+(DOFPERGRID*numgrids+3)*7+DOFPERGRID*i)=dh1dh7[0][i];
		*(B+(DOFPERGRID*numgrids+3)*7+DOFPERGRID*i+1)=dh1dh7[1][i];
		*(B+(DOFPERGRID*numgrids+3)*7+DOFPERGRID*i+2)=dh1dh7[2][i];
	}

	for (i=0;i<numgrids;i++){ //last column not for the bubble function
		*(B+(DOFPERGRID*numgrids+3)*0+DOFPERGRID*i+3)=0;
		*(B+(DOFPERGRID*numgrids+3)*1+DOFPERGRID*i+3)=0;
		*(B+(DOFPERGRID*numgrids+3)*2+DOFPERGRID*i+3)=0;
		*(B+(DOFPERGRID*numgrids+3)*3+DOFPERGRID*i+3)=0;
		*(B+(DOFPERGRID*numgrids+3)*4+DOFPERGRID*i+3)=0;
		*(B+(DOFPERGRID*numgrids+3)*5+DOFPERGRID*i+3)=0;
		*(B+(DOFPERGRID*numgrids+3)*6+DOFPERGRID*i+3)=l1l6[i];
		*(B+(DOFPERGRID*numgrids+3)*7+DOFPERGRID*i+3)=0;
	}

}
/*}}}*/
/*FUNCTION GetDofList {{{1*/
void  Penta::GetDofList(int* doflist,int* pnumberofdofspernode){

	int i,j;
	int doflist_per_node[MAXDOFSPERNODE];
	int numberofdofspernode;

	for(i=0;i<6;i++){
		nodes[i]->GetDofList(&doflist_per_node[0],&numberofdofspernode);
		for(j=0;j<numberofdofspernode;j++){
			doflist[i*numberofdofspernode+j]=doflist_per_node[j];
		}
	}

	/*Assign output pointers:*/
	*pnumberofdofspernode=numberofdofspernode;

}
/*}}}*/
/*FUNCTION GetDofList1 {{{1*/
void  Penta::GetDofList1(int* doflist){

	int i;
	for(i=0;i<6;i++){
		doflist[i]=nodes[i]->GetDofList1();
	}

}
/*}}}*/
/*FUNCTION GetId {{{1*/
int    Penta::GetId(void){
	return id; 
}
/*}}}*/
/*FUNCTION GetJacobian {{{1*/
#undef __FUNCT__
#define __FUNCT__ "Penta::GetJacobian" 
void Penta::GetJacobian(double* J, double* xyz_list,double* gauss_coord){

	const int NDOF3=3;
	int i,j;

	/*The Jacobian is constant over the element, discard the gaussian points. 
	 * J is assumed to have been allocated of size NDOF2xNDOF2.*/

	double A1,A2,A3; //area coordinates
	double xi,eta,zi; //parametric coordinates

	double x1,x2,x3,x4,x5,x6;
	double y1,y2,y3,y4,y5,y6;
	double z1,z2,z3,z4,z5,z6;

	double sqrt3=SQRT3;

	/*Figure out xi,eta and zi (parametric coordinates), for this gaussian point: */
	A1=gauss_coord[0];
	A2=gauss_coord[1];
	A3=gauss_coord[2];

	xi=A2-A1;
	eta=sqrt3*A3;
	zi=gauss_coord[3];

	x1=*(xyz_list+3*0+0);
	x2=*(xyz_list+3*1+0);
	x3=*(xyz_list+3*2+0);
	x4=*(xyz_list+3*3+0);
	x5=*(xyz_list+3*4+0);
	x6=*(xyz_list+3*5+0);

	y1=*(xyz_list+3*0+1);
	y2=*(xyz_list+3*1+1);
	y3=*(xyz_list+3*2+1);
	y4=*(xyz_list+3*3+1);
	y5=*(xyz_list+3*4+1);
	y6=*(xyz_list+3*5+1);

	z1=*(xyz_list+3*0+2);
	z2=*(xyz_list+3*1+2);
	z3=*(xyz_list+3*2+2);
	z4=*(xyz_list+3*3+2);
	z5=*(xyz_list+3*4+2);
	z6=*(xyz_list+3*5+2);


	*(J+NDOF3*0+0)=1.0/4.0*(x1-x2-x4+x5)*zi+1.0/4.0*(-x1+x2-x4+x5);
	*(J+NDOF3*1+0)=sqrt3/12.0*(x1+x2-2*x3-x4-x5+2*x6)*zi+sqrt3/12.0*(-x1-x2+2*x3-x4-x5+2*x6);
	*(J+NDOF3*2+0)=sqrt3/12.0*(x1+x2-2*x3-x4-x5+2*x6)*eta+1/4*(x1-x2-x4+x5)*xi +1.0/4.0*(-x1+x5-x2+x4);

	*(J+NDOF3*0+1)=1.0/4.0*(y1-y2-y4+y5)*zi+1.0/4.0*(-y1+y2-y4+y5);
	*(J+NDOF3*1+1)=sqrt3/12.0*(y1+y2-2*y3-y4-y5+2*y6)*zi+sqrt3/12.0*(-y1-y2+2*y3-y4-y5+2*y6);
	*(J+NDOF3*2+1)=sqrt3/12.0*(y1+y2-2*y3-y4-y5+2*y6)*eta+1.0/4.0*(y1-y2-y4+y5)*xi+1.0/4.0*(y4-y1+y5-y2);

	*(J+NDOF3*0+2)=1.0/4.0*(z1-z2-z4+z5)*zi+1.0/4.0*(-z1+z2-z4+z5);
	*(J+NDOF3*1+2)=sqrt3/12.0*(z1+z2-2*z3-z4-z5+2*z6)*zi+sqrt3/12.0*(-z1-z2+2*z3-z4-z5+2*z6);
	*(J+NDOF3*2+2)=sqrt3/12.0*(z1+z2-2*z3-z4-z5+2*z6)*eta+1.0/4.0*(z1-z2-z4+z5)*xi+1.0/4.0*(-z1+z5-z2+z4);

#ifdef _ISSM_DEBUG_
	for(i=0;i<3;i++){
		for (j=0;j<3;j++){
			printf("%lf ",*(J+NDOF3*i+j));
		}
		printf("\n");
	}
#endif
}
/*}}}*/
/*FUNCTION GetJacobianDeterminant {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::GetJacobianDeterminant" 
void Penta::GetJacobianDeterminant(double*  Jdet, double* xyz_list,double* gauss_coord){

	/*On a penta, Jacobian varies according to coordinates. We need to get the Jacobian, and take 
	 * the determinant of it: */
	const int NDOF3=3;

	double J[NDOF3][NDOF3];

	GetJacobian(&J[0][0],xyz_list,gauss_coord);

	*Jdet= J[0][0]*J[1][1]*J[2][2]-J[0][0]*J[1][2]*J[2][1]-J[1][0]*J[0][1]*J[2][2]+J[1][0]*J[0][2]*J[2][1]+J[2][0]*J[0][1]*J[1][2]-J[2][0]*J[0][2]*J[1][1];
	if(*Jdet<0){
		printf("%s%s%i\n",__FUNCT__," error message: negative jacobian determinant on element ",id);
	}
}
/*}}}*/
/*FUNCTION GetLStokes {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::GetLStokes"
void Penta::GetLStokes(double* LStokes, double* gauss_coord_tria){

	/*
	 * Compute L  matrix. L=[L1 L2 L3] where Li is square and of size numdof. 
	 * For grid i, Li can be expressed in the actual coordinate system
	 * by: 
	 *       Li=[ h    0    0   0]
	 *	 	 [ 0    h    0   0]
	 *		 [ 0    0    h   0]
	 *		 [ 0    0    h   0]
	 *	 	 [ h    0    0   0]
	 *	 	 [ 0    h    0   0]
	 *	 	 [ h    0    0   0]
	 *	 	 [ 0    h    0   0]
	 *		 [ 0    0    h   0]
	 *		 [ 0    0    h   0]
	 *		 [ 0    0    h   0]
	 *	 	 [ h    0    0   0]
	 *	 	 [ 0    h    0   0]
	 *		 [ 0    0    h   0]
	 * where h is the interpolation function for grid i.
	 */

	int i;
	const int numgrids2d=3;
	int num_dof=4;

	double l1l2l3[numgrids2d];


	/*Get l1l2l3 in actual coordinate system: */
	l1l2l3[0]=gauss_coord_tria[0];
	l1l2l3[1]=gauss_coord_tria[1];
	l1l2l3[2]=gauss_coord_tria[2];


#ifdef _DELUG_ 
	for (i=0;i<3;i++){
		printf("Node %i  h=%lf \n",i,l1l2l3[i]);
	}
#endif

	/*Build LStokes: */
	for (i=0;i<3;i++){
		*(LStokes+num_dof*numgrids2d*0+num_dof*i)=l1l2l3[i]; //LStokes[0][NDOF2*i]=dh1dh3[0][i];
		*(LStokes+num_dof*numgrids2d*0+num_dof*i+1)=0;
		*(LStokes+num_dof*numgrids2d*0+num_dof*i+2)=0;
		*(LStokes+num_dof*numgrids2d*0+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*1+num_dof*i)=0;
		*(LStokes+num_dof*numgrids2d*1+num_dof*i+1)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*1+num_dof*i+2)=0;
		*(LStokes+num_dof*numgrids2d*1+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*2+num_dof*i)=0;
		*(LStokes+num_dof*numgrids2d*2+num_dof*i+1)=0;
		*(LStokes+num_dof*numgrids2d*2+num_dof*i+2)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*2+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*3+num_dof*i)=0;
		*(LStokes+num_dof*numgrids2d*3+num_dof*i+1)=0;
		*(LStokes+num_dof*numgrids2d*3+num_dof*i+2)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*3+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*4+num_dof*i)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*4+num_dof*i+1)=0;
		*(LStokes+num_dof*numgrids2d*4+num_dof*i+2)=0;
		*(LStokes+num_dof*numgrids2d*4+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*5+num_dof*i)=0;
		*(LStokes+num_dof*numgrids2d*5+num_dof*i+1)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*5+num_dof*i+2)=0;
		*(LStokes+num_dof*numgrids2d*5+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*6+num_dof*i)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*6+num_dof*i+1)=0;
		*(LStokes+num_dof*numgrids2d*6+num_dof*i+2)=0;
		*(LStokes+num_dof*numgrids2d*6+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*7+num_dof*i)=0;
		*(LStokes+num_dof*numgrids2d*7+num_dof*i+1)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*7+num_dof*i+2)=0;
		*(LStokes+num_dof*numgrids2d*7+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*8+num_dof*i)=0;
		*(LStokes+num_dof*numgrids2d*8+num_dof*i+1)=0;
		*(LStokes+num_dof*numgrids2d*8+num_dof*i+2)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*8+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*9+num_dof*i)=0;
		*(LStokes+num_dof*numgrids2d*9+num_dof*i+1)=0;
		*(LStokes+num_dof*numgrids2d*9+num_dof*i+2)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*9+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*10+num_dof*i)=0;
		*(LStokes+num_dof*numgrids2d*10+num_dof*i+1)=0;
		*(LStokes+num_dof*numgrids2d*10+num_dof*i+2)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*10+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*11+num_dof*i)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*11+num_dof*i+1)=0;
		*(LStokes+num_dof*numgrids2d*11+num_dof*i+2)=0;
		*(LStokes+num_dof*numgrids2d*11+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*12+num_dof*i)=0;
		*(LStokes+num_dof*numgrids2d*12+num_dof*i+1)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*12+num_dof*i+2)=0;
		*(LStokes+num_dof*numgrids2d*12+num_dof*i+3)=0;
		*(LStokes+num_dof*numgrids2d*13+num_dof*i)=0;
		*(LStokes+num_dof*numgrids2d*13+num_dof*i+1)=0;
		*(LStokes+num_dof*numgrids2d*13+num_dof*i+2)=l1l2l3[i];
		*(LStokes+num_dof*numgrids2d*13+num_dof*i+3)=0;

	}
}
/*}}}*/
/*FUNCTION GetLprimeStokes {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::GetLprimeStokes"
void Penta::GetLprimeStokes(double* LprimeStokes, double* xyz_list, double* gauss_coord_tria, double* gauss_coord){

	/*
	 * Compute Lprime  matrix. Lprime=[Lp1 Lp2 Lp3] where Lpi is square and of size numdof. 
	 * For grid i, Lpi can be expressed in the actual coordinate system
	 * by: 
	 *       Lpi=[ h    0    0   0]
	 *		 [ 0    h    0   0]
	 *		 [ h    0    0   0]
	 *		 [ 0    h    0   0]
	 *		 [ 0    0    h   0]
	 *		 [ 0    0    h   0]
	 *		 [ 0    0  dh/dz 0]
	 *		 [ 0    0  dh/dz 0]
	 *		 [ 0    0  dh/dz 0]
	 *		 [dh/dz 0  dh/dx 0]
	 *		 [ 0 dh/dz dh/dy 0]
	 * 		 [ 0    0    0   h]
	 * 		 [ 0    0    0   h]
	 * 		 [ 0    0    0   h]
	 * where h is the interpolation function for grid i.
	 */

	int i;
	const int numgrids2d=3;
	int num_dof=4;

	double l1l2l3[numgrids2d];
	double dh1dh6[3][6];


	/*Get l1l2l3 in actual coordinate system: */
	l1l2l3[0]=gauss_coord_tria[0];
	l1l2l3[1]=gauss_coord_tria[1];
	l1l2l3[2]=gauss_coord_tria[2];

	GetNodalFunctionsDerivatives(&dh1dh6[0][0],xyz_list,gauss_coord);

#ifdef _DELUG_ 
	for (i=0;i<3;i++){
		printf("Node %i  h=%lf \n",i,l1l2l3[i]);
	}
#endif

	/*Build LprimeStokes: */
	for (i=0;i<3;i++){
		*(LprimeStokes+num_dof*numgrids2d*0+num_dof*i)=l1l2l3[i]; //LprimeStokes[0][NDOF2*i]=dh1dh3[0][i];
		*(LprimeStokes+num_dof*numgrids2d*0+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*0+num_dof*i+2)=0;
		*(LprimeStokes+num_dof*numgrids2d*0+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*1+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*1+num_dof*i+1)=l1l2l3[i];
		*(LprimeStokes+num_dof*numgrids2d*1+num_dof*i+2)=0;
		*(LprimeStokes+num_dof*numgrids2d*1+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*2+num_dof*i)=l1l2l3[i];
		*(LprimeStokes+num_dof*numgrids2d*2+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*2+num_dof*i+2)=0;
		*(LprimeStokes+num_dof*numgrids2d*2+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*3+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*3+num_dof*i+1)=l1l2l3[i];
		*(LprimeStokes+num_dof*numgrids2d*3+num_dof*i+2)=0;
		*(LprimeStokes+num_dof*numgrids2d*3+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*4+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*4+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*4+num_dof*i+2)=l1l2l3[i];
		*(LprimeStokes+num_dof*numgrids2d*4+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*5+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*5+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*5+num_dof*i+2)=l1l2l3[i];
		*(LprimeStokes+num_dof*numgrids2d*5+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*6+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*6+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*6+num_dof*i+2)=dh1dh6[2][i];
		*(LprimeStokes+num_dof*numgrids2d*6+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*7+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*7+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*7+num_dof*i+2)=dh1dh6[2][i];
		*(LprimeStokes+num_dof*numgrids2d*7+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*8+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*8+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*8+num_dof*i+2)=dh1dh6[2][i];
		*(LprimeStokes+num_dof*numgrids2d*8+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*9+num_dof*i)=dh1dh6[2][i];
		*(LprimeStokes+num_dof*numgrids2d*9+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*9+num_dof*i+2)=dh1dh6[0][i];
		*(LprimeStokes+num_dof*numgrids2d*9+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*10+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*10+num_dof*i+1)=dh1dh6[2][i];
		*(LprimeStokes+num_dof*numgrids2d*10+num_dof*i+2)=dh1dh6[1][i];
		*(LprimeStokes+num_dof*numgrids2d*10+num_dof*i+3)=0;
		*(LprimeStokes+num_dof*numgrids2d*11+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*11+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*11+num_dof*i+2)=0;
		*(LprimeStokes+num_dof*numgrids2d*11+num_dof*i+3)=l1l2l3[i];
		*(LprimeStokes+num_dof*numgrids2d*12+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*12+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*12+num_dof*i+2)=0;
		*(LprimeStokes+num_dof*numgrids2d*12+num_dof*i+3)=l1l2l3[i];
		*(LprimeStokes+num_dof*numgrids2d*13+num_dof*i)=0;
		*(LprimeStokes+num_dof*numgrids2d*13+num_dof*i+1)=0;
		*(LprimeStokes+num_dof*numgrids2d*13+num_dof*i+2)=0;
		*(LprimeStokes+num_dof*numgrids2d*13+num_dof*i+3)=l1l2l3[i];

	}
}
/*}}}*/
/*FUNCTION GetMatrixInvert {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "GetMatrixInvert"
void Penta::GetMatrixInvert(double*  Ke_invert, double* Ke){
	/*Inverse a 3 by 3 matrix only */

	double a,b,c,d,e,f,g,h,i;
	double det;
	int calculationdof=3;

	/*Take the matrix components: */
	a=*(Ke+calculationdof*0+0);
	b=*(Ke+calculationdof*0+1);
	c=*(Ke+calculationdof*0+2);
	d=*(Ke+calculationdof*1+0);
	e=*(Ke+calculationdof*1+1);
	f=*(Ke+calculationdof*1+2);
	g=*(Ke+calculationdof*2+0);
	h=*(Ke+calculationdof*2+1);
	i=*(Ke+calculationdof*2+2);

	det=a*(e*i-f*h)-b*(d*i-f*g)+c*(d*h-e*g);

	*(Ke_invert+calculationdof*0+0)=(e*i-f*h)/det;
	*(Ke_invert+calculationdof*0+1)=(c*h-b*i)/det;
	*(Ke_invert+calculationdof*0+2)=(b*f-c*e)/det;
	*(Ke_invert+calculationdof*1+0)=(f*g-d*i)/det;
	*(Ke_invert+calculationdof*1+1)=(a*i-c*g)/det;
	*(Ke_invert+calculationdof*1+2)=(c*d-a*f)/det;
	*(Ke_invert+calculationdof*2+0)=(d*h-e*g)/det;
	*(Ke_invert+calculationdof*2+1)=(b*g-a*h)/det;
	*(Ke_invert+calculationdof*2+2)=(a*e-b*d)/det;

}
/*}}}*/
/*FUNCTION GetName {{{1*/
char* Penta::GetName(void){
	return "penta";
}
/*}}}*/
/*FUNCTION GetNodalFunctionsDerivatives {{{1*/
#undef __FUNCT__
#define __FUNCT__ "Penta::GetNodalFunctionsDerivatives" 
void Penta::GetNodalFunctionsDerivatives(double* dh1dh6,double* xyz_list, double* gauss_coord){

	/*This routine returns the values of the nodal functions derivatives  (with respect to the actual coordinate system: */


	int i;
	const int NDOF3=3;
	const int numgrids=6;

	double dh1dh6_ref[NDOF3][numgrids];
	double Jinv[NDOF3][NDOF3];

	/*Get derivative values with respect to parametric coordinate system: */
	GetNodalFunctionsDerivativesReference(&dh1dh6_ref[0][0], gauss_coord); 

	/*Get Jacobian invert: */
	GetJacobianInvert(&Jinv[0][0], xyz_list, gauss_coord);

	/*Build dh1dh3: 
	 *
	 * [dhi/dx]= Jinv*[dhi/dr]
	 * [dhi/dy]       [dhi/ds]
	 * [dhi/dz]       [dhi/dn]
	 */

	for (i=0;i<numgrids;i++){
		*(dh1dh6+numgrids*0+i)=Jinv[0][0]*dh1dh6_ref[0][i]+Jinv[0][1]*dh1dh6_ref[1][i]+Jinv[0][2]*dh1dh6_ref[2][i];
		*(dh1dh6+numgrids*1+i)=Jinv[1][0]*dh1dh6_ref[0][i]+Jinv[1][1]*dh1dh6_ref[1][i]+Jinv[1][2]*dh1dh6_ref[2][i];
		*(dh1dh6+numgrids*2+i)=Jinv[2][0]*dh1dh6_ref[0][i]+Jinv[2][1]*dh1dh6_ref[1][i]+Jinv[2][2]*dh1dh6_ref[2][i];
	}

}
/*}}}*/
/*FUNCTION GetNodalFunctionsDerivativesStokes {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::GetNodalFunctionsDerivativesStokes"
void Penta::GetNodalFunctionsDerivativesStokes(double* dh1dh7,double* xyz_list, double* gauss_coord){

	/*This routine returns the values of the nodal functions derivatives  (with respect to the 
	 * actual coordinate system: */

	int i;

	const  int numgrids=7;
	double dh1dh7_ref[3][numgrids];
	double Jinv[3][3];


	/*Get derivative values with respect to parametric coordinate system: */
	GetNodalFunctionsDerivativesReferenceStokes(&dh1dh7_ref[0][0], gauss_coord); 

	/*Get Jacobian invert: */
	GetJacobianInvert(&Jinv[0][0], xyz_list, gauss_coord);

	/*Build dh1dh6: 
	 *
	 * [dhi/dx]= Jinv'*[dhi/dr]
	 * [dhi/dy]        [dhi/ds]
	 * [dhi/dz]        [dhi/dzeta]
	 */

	for (i=0;i<numgrids;i++){
		*(dh1dh7+numgrids*0+i)=Jinv[0][0]*dh1dh7_ref[0][i]+Jinv[0][1]*dh1dh7_ref[1][i]+Jinv[0][2]*dh1dh7_ref[2][i];
		*(dh1dh7+numgrids*1+i)=Jinv[1][0]*dh1dh7_ref[0][i]+Jinv[1][1]*dh1dh7_ref[1][i]+Jinv[1][2]*dh1dh7_ref[2][i];
		*(dh1dh7+numgrids*2+i)=Jinv[2][0]*dh1dh7_ref[0][i]+Jinv[2][1]*dh1dh7_ref[1][i]+Jinv[2][2]*dh1dh7_ref[2][i];
	}

}
/*}}}*/
/*FUNCTION GetNodalFunctionsDerivativesReference {{{1*/
#undef __FUNCT__
#define __FUNCT__ "Penta::GetNodalFunctionsDerivativesReference" 
void Penta::GetNodalFunctionsDerivativesReference(double* dl1dl6,double* gauss_coord){

	/*This routine returns the values of the nodal functions derivatives  (with respect to the 
	 * natural coordinate system) at the gaussian point. Those values vary along xi,eta,z */

	const int numgrids=6;
	double A1,A2,A3,z;
	double sqrt3=SQRT3;

	A1=gauss_coord[0]; //first area coordinate value. In term of xi and eta: A1=(1-xi)/2-eta/(2*SQRT3);
	A2=gauss_coord[1]; //second area coordinate value In term of xi and eta: A2=(1+xi)/2-eta/(2*SQRT3);
	A3=gauss_coord[2]; //third area coordinate value  In term of xi and eta: A3=y/SQRT3;
	z=gauss_coord[3]; //fourth vertical coordinate value. Corresponding nodal function: (1-z)/2 and (1+z)/2


	/*First nodal function derivatives. The corresponding nodal function is N=A1*(1-z)/2. Its derivatives follow*/
	*(dl1dl6+numgrids*0+0)=-1.0/2.0*(1.0-z)/2.0;
	*(dl1dl6+numgrids*1+0)=-1.0/2.0/sqrt3*(1.0-z)/2.0;
	*(dl1dl6+numgrids*2+0)=-1.0/2.0*A1;

	/*Second nodal function: The corresponding nodal function is N=A2*(1-z)/2. Its derivatives follow*/
	*(dl1dl6+numgrids*0+1)=1.0/2.0*(1.0-z)/2.0;
	*(dl1dl6+numgrids*1+1)=-1.0/2.0/sqrt3*(1.0-z)/2.0;
	*(dl1dl6+numgrids*2+1)=-1.0/2.0*A2;

	/*Third nodal function: The corresponding nodal function is N=A3*(1-z)/2. Its derivatives follow*/
	*(dl1dl6+numgrids*0+2)=0.0;
	*(dl1dl6+numgrids*1+2)=1.0/sqrt3*(1.0-z)/2.0;
	*(dl1dl6+numgrids*2+2)=-1.0/2.0*A3;

	/*Fourth nodal function: The corresponding nodal function is N=A1*(1+z)/2. Its derivatives follow*/
	*(dl1dl6+numgrids*0+3)=-1.0/2.0*(1.0+z)/2.0;
	*(dl1dl6+numgrids*1+3)=-1.0/2.0/sqrt3*(1.0+z)/2.0;
	*(dl1dl6+numgrids*2+3)=1.0/2.0*A1;

	/*Fifth nodal function: The corresponding nodal function is N=A2*(1+z)/2. Its derivatives follow*/
	*(dl1dl6+numgrids*0+4)=1.0/2.0*(1.0+z)/2.0;
	*(dl1dl6+numgrids*1+4)=-1.0/2.0/sqrt3*(1.0+z)/2.0;
	*(dl1dl6+numgrids*2+4)=1.0/2.0*A2;

	/*Sixth nodal function: The corresponding nodal function is N=A3*(1+z)/2. Its derivatives follow*/
	*(dl1dl6+numgrids*0+5)=0.0;
	*(dl1dl6+numgrids*1+5)=1.0/sqrt3*(1.0+z)/2.0;
	*(dl1dl6+numgrids*2+5)=1.0/2.0*A3;
}
/*}}}*/
/*FUNCTION GetNodalFunctionsDerivativesReferenceStokes {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::GetNodalFunctionsDerivativesReferenceStokes"
void Penta::GetNodalFunctionsDerivativesReferenceStokes(double* dl1dl7,double* gauss_coord){

	/*This routine returns the values of the nodal functions derivatives  (with respect to the 
	 * natural coordinate system) at the gaussian point. */

	double sqrt3=SQRT3;
	int    numgrids=7; //six plus bubble grids

	double r=gauss_coord[1]-gauss_coord[0];
	double s=-3.0/sqrt3*(gauss_coord[0]+gauss_coord[1]-2.0/3.0);
	double zeta=gauss_coord[3];

	/*First nodal function: */
	*(dl1dl7+numgrids*0+0)=-1.0/2.0*(1.0-zeta)/2.0;
	*(dl1dl7+numgrids*1+0)=-sqrt3/6.0*(1.0-zeta)/2.0;
	*(dl1dl7+numgrids*2+0)=-1.0/2.0*(-1.0/2.0*r-sqrt3/6.0*s+1.0/3.0);

	/*Second nodal function: */
	*(dl1dl7+numgrids*0+1)=1.0/2.0*(1.0-zeta)/2.0;
	*(dl1dl7+numgrids*1+1)=-sqrt3/6.0*(1.0-zeta)/2.0;
	*(dl1dl7+numgrids*2+1)=-1.0/2.0*(1.0/2.0*r-sqrt3/6.0*s+1.0/3.0);

	/*Third nodal function: */
	*(dl1dl7+numgrids*0+2)=0;
	*(dl1dl7+numgrids*1+2)=sqrt3/3.0*(1.0-zeta)/2.0;
	*(dl1dl7+numgrids*2+2)=-1.0/2.0*(sqrt3/3.0*s+1.0/3.0);

	/*Fourth nodal function: */
	*(dl1dl7+numgrids*0+3)=-1.0/2.0*(1.0+zeta)/2.0;
	*(dl1dl7+numgrids*1+3)=-sqrt3/6.0*(1.0+zeta)/2.0;
	*(dl1dl7+numgrids*2+3)=1.0/2.0*(-1.0/2.0*r-sqrt3/6.0*s+1.0/3.0);

	/*Fith nodal function: */
	*(dl1dl7+numgrids*0+4)=1.0/2.0*(1.0+zeta)/2.0;
	*(dl1dl7+numgrids*1+4)=-sqrt3/6.0*(1.0+zeta)/2.0;
	*(dl1dl7+numgrids*2+4)=1.0/2.0*(1.0/2.0*r-sqrt3/6.0*s+1.0/3.0);

	/*Sixth nodal function: */
	*(dl1dl7+numgrids*0+5)=0;
	*(dl1dl7+numgrids*1+5)=sqrt3/3.0*(1.0+zeta)/2.0;
	*(dl1dl7+numgrids*2+5)=1.0/2.0*(sqrt3/3.0*s+1.0/3.0);

	/*Seventh nodal function: */
	*(dl1dl7+numgrids*0+6)=9.0/2.0*r*(1.0+zeta)*(zeta-1.0)*(sqrt3*s+1.0);
	*(dl1dl7+numgrids*1+6)=9.0/4.0*(1+zeta)*(1-zeta)*(sqrt3*pow(s,2.0)-2.0*s-sqrt3*pow(r,2.0));
	*(dl1dl7+numgrids*2+6)=27*gauss_coord[0]*gauss_coord[1]*gauss_coord[2]*(-2.0*zeta);

}
/*}}}*/
/*FUNCTION GetNodalFunctionsStokes {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::GetNodalFunctionsStokes" 
void Penta::GetNodalFunctionsStokes(double* l1l7, double* gauss_coord){

	/*This routine returns the values of the nodal functions  at the gaussian point.*/

	/*First nodal function: */
	l1l7[0]=gauss_coord[0]*(1.0-gauss_coord[3])/2.0;

	/*Second nodal function: */
	l1l7[1]=gauss_coord[1]*(1.0-gauss_coord[3])/2.0;

	/*Third nodal function: */
	l1l7[2]=gauss_coord[2]*(1.0-gauss_coord[3])/2.0;

	/*Fourth nodal function: */
	l1l7[3]=gauss_coord[0]*(1.0+gauss_coord[3])/2.0;

	/*Fifth nodal function: */
	l1l7[4]=gauss_coord[1]*(1.0+gauss_coord[3])/2.0;

	/*Sixth nodal function: */
	l1l7[5]=gauss_coord[2]*(1.0+gauss_coord[3])/2.0;

	/*Seventh nodal function: */
	l1l7[6]=27*gauss_coord[0]*gauss_coord[1]*gauss_coord[2]*(1.0+gauss_coord[3])*(1.0-gauss_coord[3]);

}
/*}}}*/
/*FUNCTION GetParameterDerivativeValue {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::GetParameterDerivativeValue"
void Penta::GetParameterDerivativeValue(double* p, double* p_list,double* xyz_list, double* gauss_coord){

	/*From grid values of parameter p (p_list[0], p_list[1], p_list[2], p_list[3], p_list[4] and p_list[4]), return parameter derivative value at gaussian point specified by gauss_coord:
	 *   dp/dx=p_list[0]*dh1/dx+p_list[1]*dh2/dx+p_list[2]*dh3/dx+p_list[3]*dh4/dx+p_list[4]*dh5/dx+p_list[5]*dh6/dx;
	 *   dp/dy=p_list[0]*dh1/dy+p_list[1]*dh2/dy+p_list[2]*dh3/dy+p_list[3]*dh4/dy+p_list[4]*dh5/dy+p_list[5]*dh6/dy;
	 *   dp/dz=p_list[0]*dh1/dz+p_list[1]*dh2/dz+p_list[2]*dh3/dz+p_list[3]*dh4/dz+p_list[4]*dh5/dz+p_list[5]*dh6/dz;
	 *
	 *   p is a vector of size 3x1 already allocated.
	 */

	const int NDOF3=3;
	const int numgrids=6;
	double dh1dh6[NDOF3][numgrids];

	/*Get dh1dh6 in actual coordinate system: */
	GetNodalFunctionsDerivatives(&dh1dh6[0][0],xyz_list, gauss_coord);

	*(p+0)=p_list[0]*dh1dh6[0][0]+p_list[1]*dh1dh6[0][1]+p_list[2]*dh1dh6[0][2]+p_list[3]*dh1dh6[0][3]+p_list[4]*dh1dh6[0][4]+p_list[5]*dh1dh6[0][5];
	;
	*(p+1)=p_list[0]*dh1dh6[1][0]+p_list[1]*dh1dh6[1][1]+p_list[2]*dh1dh6[1][2]+p_list[3]*dh1dh6[1][3]+p_list[4]*dh1dh6[1][4]+p_list[5]*dh1dh6[1][5];

	*(p+2)=p_list[0]*dh1dh6[2][0]+p_list[1]*dh1dh6[2][1]+p_list[2]*dh1dh6[2][2]+p_list[3]*dh1dh6[2][3]+p_list[4]*dh1dh6[2][4]+p_list[5]*dh1dh6[2][5];

}
/*}}}*/
/*FUNCTION GetParameterValue {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::GetParameterValue"
void Penta::GetParameterValue(double* pvalue, double* v_list,double* gauss_coord){

	const int numgrids=6;
	double l1l6[numgrids];

	GetNodalFunctions(&l1l6[0], gauss_coord);

	*pvalue=l1l6[0]*v_list[0]+l1l6[1]*v_list[1]+l1l6[2]*v_list[2]+l1l6[3]*v_list[3]+l1l6[4]*v_list[4]+l1l6[5]*v_list[5];
}
/*}}}*/
/*FUNCTION GetJacobianInvert {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::GetJacobianInvert"
void Penta::GetJacobianInvert(double*  Jinv, double* xyz_list,double* gauss_coord){

	double Jdet;
	const int NDOF3=3;

	/*Call Jacobian routine to get the jacobian:*/
	GetJacobian(Jinv, xyz_list, gauss_coord);

	/*Invert Jacobian matrix: */
	MatrixInverse(Jinv,NDOF3,NDOF3,NULL,0,&Jdet);
}
/*}}}*/
/*FUNCTION GetMatPar {{{1*/
void* Penta::GetMatPar(){
	return matpar;
}
/*}}}*/
/*FUNCTION GetNodalFunctions {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::GetNodalFunctions"
void Penta::GetNodalFunctions(double* l1l6, double* gauss_coord){

	/*This routine returns the values of the nodal functions  at the gaussian point.*/

	l1l6[0]=gauss_coord[0]*(1-gauss_coord[3])/2.0;

	l1l6[1]=gauss_coord[1]*(1-gauss_coord[3])/2.0;

	l1l6[2]=gauss_coord[2]*(1-gauss_coord[3])/2.0;

	l1l6[3]=gauss_coord[0]*(1+gauss_coord[3])/2.0;

	l1l6[4]=gauss_coord[1]*(1+gauss_coord[3])/2.0;

	l1l6[5]=gauss_coord[2]*(1+gauss_coord[3])/2.0;

}
/*}}}*/
/*FUNCTION GetNodes {{{1*/
void  Penta::GetNodes(void** vpnodes){
	int i;
	Node** pnodes=(Node**)vpnodes;

	for(i=0;i<6;i++){
		pnodes[i]=nodes[i];
	}
}
/*}}}*/
/*FUNCTION GetOnBed {{{1*/
int Penta::GetOnBed(){
	return onbed;
}
/*}}}*/
/*FUNCTION GetPhi {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::GetPhi"
void Penta::GetPhi(double* phi, double*  epsilon, double viscosity){
	/*Compute deformational heating from epsilon and viscosity */

	double epsilon_matrix[3][3];
	double epsilon_eff;
	double epsilon_sqr[3][3];

	/* Build epsilon matrix */
	epsilon_matrix[0][0]=*(epsilon+0);
	epsilon_matrix[1][0]=*(epsilon+3);
	epsilon_matrix[2][0]=*(epsilon+4);
	epsilon_matrix[0][1]=*(epsilon+3);
	epsilon_matrix[1][1]=*(epsilon+1);
	epsilon_matrix[2][1]=*(epsilon+5);
	epsilon_matrix[0][2]=*(epsilon+4);
	epsilon_matrix[1][2]=*(epsilon+5);
	epsilon_matrix[2][2]=*(epsilon+2);

	/* Effective value of epsilon_matrix */
	epsilon_sqr[0][0]=pow(epsilon_matrix[0][0],2);
	epsilon_sqr[1][0]=pow(epsilon_matrix[1][0],2);
	epsilon_sqr[2][0]=pow(epsilon_matrix[2][0],2);
	epsilon_sqr[0][1]=pow(epsilon_matrix[0][1],2);
	epsilon_sqr[1][1]=pow(epsilon_matrix[1][1],2);
	epsilon_sqr[2][1]=pow(epsilon_matrix[2][1],2);
	epsilon_sqr[0][2]=pow(epsilon_matrix[0][2],2);
	epsilon_sqr[1][2]=pow(epsilon_matrix[1][2],2);
	epsilon_sqr[2][2]=pow(epsilon_matrix[2][2],2);

	epsilon_eff=1/pow(2,1.0/2.0)*pow((epsilon_sqr[0][0]+epsilon_sqr[0][1]+ epsilon_sqr[0][2]+ epsilon_sqr[1][0]+ epsilon_sqr[1][1]+ epsilon_sqr[1][2]+ epsilon_sqr[2][0]+ epsilon_sqr[2][1]+ epsilon_sqr[2][2]),1.0/2.0);
	*phi=2*pow(epsilon_eff,2.0)*viscosity;
}
/*}}}*/
/*FUNCTION GetShelf {{{1*/
int   Penta::GetShelf(){
	return shelf;
}
/*}}}*/
/*FUNCTION GetStrainRate {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::GetStrainRate"
void Penta::GetStrainRate(double* epsilon, double* velocity, double* xyz_list, double* gauss_coord){

	int i;
	const int numgrids=6;
	const int NDOF2=2;

	double B[5][NDOF2*numgrids];

	/*Get B matrix: */
	GetB(&B[0][0], xyz_list, gauss_coord);

#ifdef _ISSM_DEBUG_
	printf("B for grid1 : [ %lf   %lf  \n",B[0][0],B[0][1]);
	printf("              [ %lf   %lf  \n",B[1][0],B[1][1]);
	printf("              [ %lf   %lf  ]\n",B[2][0],B[2][1]);
	printf("              [ %lf   %lf  ]\n",B[3][0],B[3][1]);
	printf("              [ %lf   %lf  ]\n",B[4][0],B[4][1]);

	printf("B for grid2 : [ %lf   %lf  \n",B[0][2],B[0][3]);
	printf("              [ %lf   %lf  \n",B[1][2],B[1][3]);
	printf("              [ %lf   %lf  ]\n",B[2][2],B[2][3]);
	printf("              [ %lf   %lf  ]\n",B[3][2],B[3][3]);
	printf("              [ %lf   %lf  ]\n",B[4][2],B[4][3]);

	printf("B for grid3 : [ %lf   %lf  \n", B[0][4],B[0][5]);
	printf("              [ %lf   %lf  \n", B[1][4],B[1][5]);
	printf("              [ %lf   %lf  ]\n",B[2][4],B[2][5]);
	printf("              [ %lf   %lf  ]\n",B[3][4],B[3][5]);
	printf("              [ %lf   %lf  ]\n",B[4][4],B[4][5]);

	printf("B for grid4 : [ %lf   %lf  \n", B[0][6],B[0][7]);
	printf("              [ %lf   %lf  \n", B[1][6],B[1][7]);
	printf("              [ %lf   %lf  ]\n",B[2][6],B[2][7]);
	printf("              [ %lf   %lf  ]\n",B[3][6],B[3][7]);
	printf("              [ %lf   %lf  ]\n",B[4][6],B[4][7]);

	printf("B for grid5 : [ %lf   %lf  \n", B[0][8],B[0][9]);
	printf("              [ %lf   %lf  \n", B[1][8],B[1][9]);
	printf("              [ %lf   %lf  ]\n",B[2][8],B[2][9]);
	printf("              [ %lf   %lf  ]\n",B[3][8],B[3][9]);
	printf("              [ %lf   %lf  ]\n",B[4][8],B[4][9]);

	printf("B for grid6 : [ %lf   %lf  \n", B[0][10],B[0][11]);
	printf("              [ %lf   %lf  \n", B[1][10],B[1][11]);
	printf("              [ %lf   %lf  ]\n",B[2][10],B[2][11]);
	printf("              [ %lf   %lf  ]\n",B[3][10],B[3][11]);
	printf("              [ %lf   %lf  ]\n",B[4][10],B[4][11]);

#endif

	/*Multiply B by velocity, to get strain rate: */
	MatrixMultiply( &B[0][0],5,NDOF2*numgrids,0,
				velocity,NDOF2*numgrids,1,0,
				epsilon,0);

}
/*}}}*/
/*FUNCTION GetStrainRateStokes {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::GetStrainRateStokes"
void Penta::GetStrainRateStokes(double* epsilon, double* velocity, double* xyz_list, double* gauss_coord){

	int i,j;

	const int numgrids=6;
	const int DOFVELOCITY=3;
	double B[8][27];
	double B_reduced[numgrids][DOFVELOCITY*numgrids];

	/*Get B matrix: */
	GetBStokes(&B[0][0], xyz_list, gauss_coord);

	/*Create a reduced matrix of B to get rid of pressure */
	for (i=0;i<6;i++){
		for (j=0;j<3;j++){
			B_reduced[i][j]=B[i][j];
		}
		for (j=4;j<7;j++){
			B_reduced[i][j-1]=B[i][j];
		}
		for (j=8;j<11;j++){
			B_reduced[i][j-2]=B[i][j];
		}
		for (j=12;j<15;j++){
			B_reduced[i][j-3]=B[i][j];
		}
		for (j=16;j<19;j++){
			B_reduced[i][j-4]=B[i][j];
		}
		for (j=20;j<23;j++){
			B_reduced[i][j-5]=B[i][j];
		}
	}
	/*Multiply B by velocity, to get strain rate: */
	MatrixMultiply( &B_reduced[0][0],6,DOFVELOCITY*numgrids, 0, velocity,DOFVELOCITY*numgrids,1,0,epsilon, 0);
}
/*}}}*/
/*FUNCTION GetThicknessList {{{1*/
void Penta::GetThicknessList(double* thickness_list){

	int i;
	for(i=0;i<6;i++)thickness_list[i]=h[i];
}
/*}}}*/
/*FUNCTION Gradj {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::Gradj"
void  Penta::Gradj(Vec grad_g,void* inputs,int analysis_type,int sub_analysis_type,char* control_type){

	/*If on water, skip grad (=0): */
	if(onwater)return;

	if (strcmp(control_type,"drag")==0){
		GradjDrag( grad_g,inputs,analysis_type,sub_analysis_type);
	}
	else if (strcmp(control_type,"B")==0){
		GradjB( grad_g, inputs,analysis_type,sub_analysis_type);
	}
	else throw ErrorException(__FUNCT__,exprintf("%s%s","control type not supported yet: ",control_type));
}
/*}}}*/
/*FUNCTION GradjDrag {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::GradjDrag"
void  Penta::GradjDrag(Vec grad_g,void* inputs,int analysis_type,int sub_analysis_type){

	Tria* tria=NULL;

	/*If on water, skip: */
	if(onwater)return;

	/*If on shelf, skip: */
	if(shelf)return;

	/*Bail out if this element does not touch the bedrock: */
	if (!onbed) return;

	if (sub_analysis_type==HorizAnalysisEnum()){

		/*MacAyeal or Pattyn*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
		tria->GradjDrag( grad_g,inputs,analysis_type,sub_analysis_type);
		delete tria;
		return;
	}
	else if (sub_analysis_type==StokesAnalysisEnum()){

		/*Stokes*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria.
		tria->GradjDragStokes( grad_g,inputs,analysis_type,sub_analysis_type);
		delete tria;
		return;
	}
	else throw ErrorException(__FUNCT__,exprintf("%s%i%s\n","sub_analysis: ",sub_analysis_type," not supported yet"));
}
/*}}}*/
/*FUNCTION GradjB {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::GradjB"
void  Penta::GradjB(Vec grad_g,void* inputs,int analysis_type,int sub_analysis_type){

	Tria* tria=NULL;

	/*If on water, skip: */
	if(onwater)return;

	if (collapse){
		/*Bail out element if collapsed (2d) and not on bed*/
		if (!onbed) return;

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 * and compute gardj*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria (lower face).
		tria->GradjB(grad_g,inputs,analysis_type,sub_analysis_type);
		delete tria;
		return;
	}
	else{
		/*B is a 2d field, use MacAyeal(2d) gradient even if it is Stokes or Pattyn*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria (lower face).
		tria->GradjB(grad_g,inputs,analysis_type,sub_analysis_type);
		delete tria;
		return;
	}
}
/*}}}*/
/*FUNCTION MassFlux {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::MassFlux"
double Penta::MassFlux( double* segment,double* ug){
	throw ErrorException(__FUNCT__," not supported yet!");
}
/*}}}*/
/*FUNCTION Misfit {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::Misfit"
double Penta::Misfit(void* inputs,int analysis_type,int sub_analysis_type,int real){

	double J;
	Tria* tria=NULL;

	/*If on water, return 0: */
	if(onwater)return 0;

	/*Bail out if this element if:
	 * -> Non collapsed and not on the surface
	 * -> collapsed (2d model) and not on bed) */
	if ((!collapse && !onsurface) || (collapse && !onbed)){
		return 0;
	}
	else if (collapse){

		/*This element should be collapsed into a tria element at its base. Create this tria element, 
		 * and compute Misfit*/
		tria=(Tria*)SpawnTria(0,1,2); //grids 0, 1 and 2 make the new tria (lower face).
		J=tria->Misfit(inputs,analysis_type,sub_analysis_type,real);
		delete tria;
		return J;
	}
	else{

		tria=(Tria*)SpawnTria(3,4,5); //grids 3, 4 and 5 make the new tria (upper face).
		J=tria->Misfit(inputs,analysis_type,sub_analysis_type,real);
		delete tria;
		return J;
	}
}
/*}}}*/
/*FUNCTION MyRank {{{1*/
int    Penta::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}*/
/*FUNCTION ReduceMatrixStokes {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "ReduceMatrixStokes" 
void Penta::ReduceMatrixStokes(double* Ke_reduced, double* Ke_temp){

	int i,j;

	double Kii[24][24];
	double Kib[24][3];
	double Kbb[3][3];
	double Kbi[3][24];
	double Kbbinv[3][3];
	double KibKbbinv[24][3];
	double Kright[24][24];

	/*Create the four matrices used for reduction */
	for(i=0;i<24;i++){
		for(j=0;j<24;j++){
			Kii[i][j]=*(Ke_temp+27*i+j);
		}
		for(j=0;j<3;j++){
			Kib[i][j]=*(Ke_temp+27*i+j+24);
		}
	}
	for(i=0;i<3;i++){
		for(j=0;j<24;j++){
			Kbi[i][j]=*(Ke_temp+27*(i+24)+j);
		}
		for(j=0;j<3;j++){
			Kbb[i][j]=*(Ke_temp+27*(i+24)+j+24);
		}
	}

	/*Inverse the matrix corresponding to bubble part Kbb */
	GetMatrixInvert(&Kbbinv[0][0], &Kbb[0][0]);

	/*Multiply matrices to create the reduce matrix Ke_reduced */
	MatrixMultiply(&Kib[0][0],24,3,0,&Kbbinv[0][0],3,3,0,&KibKbbinv[0][0],0);
	MatrixMultiply(&KibKbbinv[0][0],24,3,0,&Kbi[0][0],3,24,0,&Kright[0][0],0);

	/*Affect value to the reduced matrix */
	for(i=0;i<24;i++){
		for(j=0;j<24;j++){
			*(Ke_reduced+24*i+j)=Kii[i][j]-Kright[i][j];
		}
	}
}
/*}}}*/
/*FUNCTION ReduceVectorStokes {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::ReduceVectorStokes" 
void Penta::ReduceVectorStokes(double* Pe_reduced, double* Ke_temp, double* Pe_temp){

	int i,j;

	double Pi[24];
	double Pb[3];
	double Kbb[3][3];
	double Kib[24][3];
	double Kbbinv[3][3];
	double KibKbbinv[24][3];
	double Pright[24];

	/*Create the four matrices used for reduction */
	for(i=0;i<24;i++){
		Pi[i]=*(Pe_temp+i);
	}
	for(i=0;i<3;i++){
		Pb[i]=*(Pe_temp+i+24);
	}
	for(j=0;j<3;j++){
		for(i=0;i<24;i++){
			Kib[i][j]=*(Ke_temp+3*i+j);
		}
		for(i=0;i<3;i++){
			Kbb[i][j]=*(Ke_temp+3*(i+24)+j);
		}
	}

	/*Inverse the matrix corresponding to bubble part Kbb */
	GetMatrixInvert(&Kbbinv[0][0], &Kbb[0][0]);

	/*Multiply matrices to create the reduce matrix Ke_reduced */
	MatrixMultiply(&Kib[0][0],24,3,0,&Kbbinv[0][0],3,3,0,&KibKbbinv[0][0],0);
	MatrixMultiply(&KibKbbinv[0][0],24,3,0,&Pb[0],3,1,0,&Pright[0],0);

	/*Affect value to the reduced matrix */
	for(i=0;i<24;i++){
		*(Pe_reduced+i)=Pi[i]-Pright[i];
	}
}
/*}}}*/
/*FUNCTION SpawnTria {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::SpawnTria"
void*  Penta::SpawnTria(int g0, int g1, int g2){

	/*out of grids g0,g1 and g2 from Penta, build a tria element: */
	Tria* tria=NULL;
	double   tria_h[3];
	double   tria_s[3];
	double   tria_b[3];
	double   tria_c[3];
	double   tria_k[3];
	double   tria_melting[3];
	double   tria_accumulation[3];
	double   tria_geothermalflux[3];

	/*configuration: */
	int tria_node_ids[3];
	Node* tria_nodes[3];
	int tria_node_offsets[3];

	tria_h[0]=h[g0];
	tria_h[1]=h[g1];
	tria_h[2]=h[g2];

	tria_s[0]=s[g0];
	tria_s[1]=s[g1];
	tria_s[2]=s[g2];

	tria_b[0]=b[g0];
	tria_b[1]=b[g1];
	tria_b[2]=b[g2];

	tria_k[0]=k[g0];
	tria_k[1]=k[g1];
	tria_k[2]=k[g2];

	tria_melting[0]=melting[g0];
	tria_melting[1]=melting[g1];
	tria_melting[2]=melting[g2];

	tria_accumulation[0]=accumulation[g0];
	tria_accumulation[1]=accumulation[g1];
	tria_accumulation[2]=accumulation[g2];

	tria_geothermalflux[0]=geothermalflux[g0];
	tria_geothermalflux[1]=geothermalflux[g1];
	tria_geothermalflux[2]=geothermalflux[g2];

	tria_node_ids[0]=node_ids[g0];
	tria_node_ids[1]=node_ids[g1];
	tria_node_ids[2]=node_ids[g2];

	tria_nodes[0]=nodes[g0]; 
	tria_nodes[1]=nodes[g1];
	tria_nodes[2]=nodes[g2];

	tria_node_offsets[0]=node_offsets[g0];
	tria_node_offsets[1]=node_offsets[g1];
	tria_node_offsets[2]=node_offsets[g2];

	tria= new Tria(id,mid,mparid,numparid,tria_node_ids,tria_h,tria_s,tria_b,tria_k, tria_melting, tria_accumulation, tria_geothermalflux,friction_type,p,q,shelf,onwater);

	tria->NodeConfiguration(tria_node_ids,tria_nodes,tria_node_offsets);
	tria->MaticeConfiguration(matice,matice_offset);
	tria->MatparConfiguration(matpar,matpar_offset);
	tria->NumparConfiguration(numpar,numpar_offset);

	return tria;

}
/*}}}*/
/*FUNCTION UpdateFromInputs {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::UpdateFromInputs"
void  Penta::UpdateFromInputs(void* vinputs){

	int     dofs[1]={0};
	double  temperature_list[6];
	double  temperature_average;
	double  B_list[6];
	double  B_average;

	ParameterInputs* inputs=NULL;

	/*If on water, skip: */
	if(onwater)return;

	/*recover pointers: */
	inputs=(ParameterInputs*)vinputs;

	/*Update internal data if inputs holds new values: */
	inputs->Recover("thickness",&h[0],1,dofs,6,(void**)nodes);
	inputs->Recover("surface",&s[0],1,dofs,6,(void**)nodes);
	inputs->Recover("bed",&b[0],1,dofs,6,(void**)nodes);
	inputs->Recover("drag",&k[0],1,dofs,6,(void**)nodes);
	inputs->Recover("melting",&melting[0],1,dofs,6,(void**)nodes);
	inputs->Recover("accumulation_param",&accumulation[0],1,dofs,6,(void**)nodes);

	//Update material if necessary
	if(inputs->Recover("temperature",&temperature_list[0],1,dofs,6,(void**)nodes)){
		if(matice && !collapse){
			//B_average=(Paterson(temperature_list[0])+Paterson(temperature_list[1])+Paterson(temperature_list[2])
			//			+Paterson(temperature_list[3])+Paterson(temperature_list[4])+Paterson(temperature_list[5]))/6.0;
			temperature_average=(temperature_list[0]+temperature_list[1]+temperature_list[2]+temperature_list[3]+temperature_list[4]+temperature_list[5])/6.0;
			B_average=Paterson(temperature_average);
			matice->SetB(B_average);
		}
	}

	if(inputs->Recover("temperature_average",&temperature_list[0],1,dofs,6,(void**)nodes)){
		if(matice && collapse){
			temperature_average=(temperature_list[0]+temperature_list[1]+temperature_list[2]+temperature_list[3]+temperature_list[4]+temperature_list[5])/6.0;
			B_average=Paterson(temperature_average);
			//B_average=(Paterson(temperature_list[0])+Paterson(temperature_list[1])+Paterson(temperature_list[2])
			//			+Paterson(temperature_list[3])+Paterson(temperature_list[4])+Paterson(temperature_list[5]))/6.0;
			matice->SetB(B_average);
		}
	}

	if(inputs->Recover("B",&B_list[0],1,dofs,6,(void**)nodes)){
		if(matice){
			B_average=(B_list[0]+B_list[1]+B_list[2]+B_list[3]+B_list[4]+B_list[5])/6.0;
			matice->SetB(B_average);
		}
	}

}
/*}}}*/
/*FUNCTION SurfaceNormal {{{1*/
#undef __FUNCT__ 
#define __FUNCT__ "Penta::SurfaceNormal"
void Penta::SurfaceNormal(double* surface_normal, double xyz_list[3][3]){

	int i;
	double v13[3];
	double v23[3];
	double normal[3];
	double normal_norm;

	for (i=0;i<3;i++){
		v13[i]=xyz_list[0][i]-xyz_list[2][i];
		v23[i]=xyz_list[1][i]-xyz_list[2][i];
	}

	normal[0]=v13[1]*v23[2]-v13[2]*v23[1];
	normal[1]=v13[2]*v23[0]-v13[0]*v23[2];
	normal[2]=v13[0]*v23[1]-v13[1]*v23[0];

	normal_norm=sqrt( pow(normal[0],2)+pow(normal[1],2)+pow(normal[2],2) );

	*(surface_normal)=normal[0]/normal_norm;
	*(surface_normal+1)=normal[1]/normal_norm;
	*(surface_normal+2)=normal[2]/normal_norm;

}
/*}}}*/
