/*!\file:  thermal.cpp
 * \brief: thermal solution
 */ 

#include "../issm.h"
#include "./parallel.h"

#undef __FUNCT__ 
#define __FUNCT__ "thermal"

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

int main(int argc,char* *argv){

	int i,n;
	
	/*I/O: */
	FILE* fid=NULL;
	char* inputfilename=NULL;
	char* outputfilename=NULL;
	char* lockname=NULL;
	int   qmu_analysis=0;
	int   numberofnodes;
	double waitonlock=0;

	/*Model: */
	Model* model=NULL;

	/*initial velocity and pressure: */
	double* u_g=NULL;
	double* p_g=NULL;

	/*Results: */
	DataSet* results=NULL;
	DataSet* processed_results=NULL;
	Result*  result=NULL;
	
	ParameterInputs* inputs=NULL;
	Param*           param=NULL;
	double  dt;
	double  yts;

	/*time*/
	double   start, finish;
	double   start_core, finish_core;
	double   start_init, finish_init;
		
	MODULEBOOT();

	#if !defined(_PARALLEL_) || (defined(_PARALLEL_) && !defined(_HAVE_PETSC_))
	throw ErrorException(__FUNCT__," parallel executable was compiled without support of parallel libraries!");
	#endif

	/*Initialize Petsc and get start time*/
	PetscInitialize(&argc,&argv,(char *)0,"");  
	MPI_Barrier(MPI_COMM_WORLD); start=MPI_Wtime();

	/*Size and rank: */
	MPI_Comm_rank(MPI_COMM_WORLD,&my_rank);  
	MPI_Comm_size(MPI_COMM_WORLD,&num_procs); 

	inputfilename=argv[2];
	outputfilename=argv[3];
	lockname=argv[4];

	/*Open handle to data on disk: */
	fid=pfopen(inputfilename,"rb");

	/*Initialize model structure: */
	MPI_Barrier(MPI_COMM_WORLD); start_init=MPI_Wtime();
	model=new Model();

	_printf_("read and create thermal finite element model:\n");
	model->AddFormulation(fid,ThermalAnalysisEnum());
	_printf_("read and create melting finite element model:\n");
	model->AddFormulation(fid,MeltingAnalysisEnum());

	/*recover parameters: */
	model->FindParam(&waitonlock,"waitonlock");
	model->FindParam(&qmu_analysis,"qmu_analysis");
	
	_printf_("initialize inputs:\n");
	model->FindParam(&u_g,NULL,NULL,"u_g",ThermalAnalysisEnum());
	model->FindParam(&p_g,NULL,NULL,"p_g",ThermalAnalysisEnum());
	model->FindParam(&numberofnodes,"numberofnodes");
	model->FindParam(&dt,"dt");
	model->FindParam(&yts,"yts");

	inputs=new ParameterInputs;

	inputs->Add("velocity",u_g,3,numberofnodes);
	inputs->Add("pressure",p_g,1,numberofnodes);
	inputs->Add("dt",dt*yts); //convert in seconds 

	_printf_("initialize results:\n");
	results=new DataSet(ResultsEnum());
	MPI_Barrier(MPI_COMM_WORLD); finish_init=MPI_Wtime();

	/*are we running the solutoin sequence, or a qmu wrapper around it? : */
	if(!qmu_analysis){

		/*run thermal analysis: */
		_printf_("call computational core:\n");
		MPI_Barrier(MPI_COMM_WORLD); start_core=MPI_Wtime( );
		thermal_core(results,model,inputs);
		MPI_Barrier(MPI_COMM_WORLD); finish_core=MPI_Wtime( );

		/*Add analysis_type to results: */
		result=new Result(results->Size()+1,0,1,"analysis_type","thermal");
		results->AddObject(result);
			
		_printf_("process results:\n");
		ProcessResults(&processed_results,results,model,ThermalAnalysisEnum());
		
		_printf_("write results to disk:\n");
		OutputResults(processed_results,outputfilename);
	}
	else{
		/*run qmu analysis: */
		_printf_("calling qmu analysis on thermal core:\n");
	
		#ifdef _HAVE_DAKOTA_ 
		MPI_Barrier(MPI_COMM_WORLD); start_core=MPI_Wtime( );
		Qmux(model,inputs,ThermalAnalysisEnum(),NoneAnalysisEnum());
		MPI_Barrier(MPI_COMM_WORLD); finish_core=MPI_Wtime( );
	 	#else
		throw ErrorException(__FUNCT__," Dakota not present, cannot do qmu!");
		#endif
	}

	if (waitonlock>0){
		_printf_("write lock file:\n");
		WriteLockFile(lockname);
	}

	/*Free ressources */
	xfree((void**)&u_g);
	xfree((void**)&p_g);
	delete model;
	delete results;
	delete processed_results;
	delete inputs;

	/*Get finish time and close*/
	MPI_Barrier(MPI_COMM_WORLD); finish = MPI_Wtime( );
	_printf_("\n   %-34s %f seconds  \n","Model initialization elapsed time:",finish_init-start_init);
	_printf_("   %-34s %f seconds  \n","Core solution elapsed time:",finish_core-start_core);
	_printf_("\n   %s %i hrs %i min %i sec\n\n","Total elapsed time:",int((finish-start)/3600),int(int(finish-start)%3600/60),int(finish-start)%60);
	_printf_("closing MPI and Petsc\n");
	PetscFinalize(); 

	/*end module: */
	MODULEEND();

	return 0; //unix success return;
}
