/*!\file: transient_3d_core.cpp
 * \brief: core of the transient_3d solution 
 */ 

#undef __FUNCT__ 
#define __FUNCT__ "transient_3d_core"

#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../shared/shared.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "./parallel.h"
#include "../issm.h"

void transient_core_3d(DataSet* results,Model* model, ParameterInputs* inputs){

	extern int my_rank;

	/*fem models: */
	FemModel* fem_dh=NULL;
	FemModel* fem_dv=NULL;
	FemModel* fem_dhu=NULL;
	FemModel* fem_ds=NULL;
	FemModel* fem_sl=NULL;
	FemModel* fem_p=NULL;
	FemModel* fem_t=NULL;
	FemModel* fem_m=NULL;

	/*output: */
	Result* result=NULL;

	/*solutions: */
	int step;
	double time;
	
	Vec u_g=NULL;
	double* u_g_serial=NULL;
	Vec p_g=NULL;
	double* p_g_serial=NULL;
	Vec h_g=NULL;
	Vec h_g_intermediary=NULL;
	double* h_g_serial=NULL;
	Vec s_g=NULL;
	double* s_g_serial=NULL;
	Vec b_g=NULL;
	double* b_g_serial=NULL;
	Vec t_g=NULL;
	Vec t_g_average=NULL;
	double* t_g_serial=NULL;
	double melting_offset;
	Vec m_g=NULL;
	double* m_g_serial=NULL;

	/*intermediary: */
	double finaltime;
	double dt;
	DataSet* diagnostic_results=NULL;
	DataSet* prognostic_results=NULL;


	/*flags: */
	int verbose=0;
	int numberofnodes;

	/*dof recovery: */
	int dof01[2]={0,1};
	int dof2[1]={2};
	int dof012[3]={0,1,2};
	int dof3[1]={3};

	/*recover fem models: */
	fem_dh=model->GetFormulation(DiagnosticAnalysisEnum(),HorizAnalysisEnum());
	fem_dv=model->GetFormulation(DiagnosticAnalysisEnum(),VertAnalysisEnum());
	fem_ds=model->GetFormulation(DiagnosticAnalysisEnum(),StokesAnalysisEnum());
	fem_dhu=model->GetFormulation(DiagnosticAnalysisEnum(),HutterAnalysisEnum());
	fem_sl=model->GetFormulation(SlopecomputeAnalysisEnum());
	fem_p=model->GetFormulation(PrognosticAnalysisEnum());
	fem_t=model->GetFormulation(ThermalAnalysisEnum());
	fem_m=model->GetFormulation(MeltingAnalysisEnum());


	//first recover parameters common to all solutions
	model->FindParam(&verbose,"verbose");
	model->FindParam(&finaltime,"ndt");
	model->FindParam(&dt,"dt");
	model->FindParam(&numberofnodes,"numberofnodes");

	/*initialize: */
	step=1;
	time=0;
	
	fem_p->parameters->FindParam(&u_g_serial,NULL,NULL,"u_g");
	u_g=SerialToVec(u_g_serial,3*numberofnodes); xfree((void**)&u_g_serial);

	fem_p->parameters->FindParam(&p_g_serial,NULL,NULL,"p_g");
	p_g=SerialToVec(p_g_serial,1*numberofnodes); xfree((void**)&p_g_serial);

	fem_p->parameters->FindParam(&h_g_serial,NULL,NULL,"h_g");
	h_g=SerialToVec(h_g_serial,1*numberofnodes); xfree((void**)&h_g_serial);

	fem_p->parameters->FindParam(&s_g_serial,NULL,NULL,"s_g");
	s_g=SerialToVec(s_g_serial,1*numberofnodes); xfree((void**)&s_g_serial);

	fem_p->parameters->FindParam(&b_g_serial,NULL,NULL,"b_g");
	b_g=SerialToVec(b_g_serial,1*numberofnodes); xfree((void**)&b_g_serial);

	fem_p->parameters->FindParam(&t_g_serial,NULL,NULL,"t_g");
	t_g=SerialToVec(t_g_serial,1*numberofnodes); xfree((void**)&t_g_serial);

	fem_p->parameters->FindParam(&m_g_serial,NULL,NULL,"m_g");
	m_g=SerialToVec(m_g_serial,1*numberofnodes); xfree((void**)&m_g_serial);

	/*Add all these initial vectors in the results!: */
	result=new Result(results->Size()+1,time,step,"u_g",u_g); results->AddObject(result);
	result=new Result(results->Size()+1,time,step,"p_g",p_g); results->AddObject(result);
	result=new Result(results->Size()+1,time,step,"h_g",h_g); results->AddObject(result);
	result=new Result(results->Size()+1,time,step,"s_g",s_g); results->AddObject(result);
	result=new Result(results->Size()+1,time,step,"b_g",b_g); results->AddObject(result);
	result=new Result(results->Size()+1,time,step,"t_g",t_g); results->AddObject(result);
	result=new Result(results->Size()+1,time,step,"m_g",m_g); results->AddObject(result);

	while(time<finaltime){ //make sure we run up to finaltime.
	
		if(verbose)_printf_("%s%g%s%i%s%g\n","time [yr]: ",time,"    iteration number: ",step,"/",floor(finaltime/dt));

		step+=1;
		time+=dt;

		//update inputs
		inputs->Add("thickness",h_g,1,numberofnodes);
		inputs->Add("surface",s_g,1,numberofnodes);
		inputs->Add("bed",b_g,1,numberofnodes);
		inputs->Add("velocity",u_g,3,numberofnodes);
		inputs->Add("pressure",p_g,1,numberofnodes);
		inputs->Add("temperature",t_g,1,numberofnodes);

		//Deal with temperature first 
		if(verbose)_printf_("%s\n","computing temperature");
		thermal_core_nonlinear(&t_g,&melting_offset,fem_t,inputs,ThermalAnalysisEnum(),TransientAnalysisEnum());
		if(verbose)_printf_("%s\n","computing melting");
		inputs->Add("temperature",t_g,1,numberofnodes);
		inputs->Add("melting_offset",melting_offset);
		diagnostic_core_linear(&m_g,fem_m,inputs,MeltingAnalysisEnum(),TransientAnalysisEnum());

		//Compute depth averaged temperature and add to inputs
		if(verbose)_printf_("%s\n","computing depth average temperature");
		VecDuplicatePatch(&t_g_average,t_g); 
		FieldDepthAveragex( t_g_average, fem_t->elements,fem_t->nodes, fem_t->loads, fem_t->materials,fem_t->parameters,"temperature");
		inputs->Add("temperature_average",t_g_average,1,numberofnodes);
		VecFree(&t_g_average); //not needed anymore

		//Deal with velocities.
		diagnostic_results=new DataSet(ResultsEnum()); 
		diagnostic_core(diagnostic_results,model, inputs);

		//extract u_g and p_g from diagnostic_results, and erase diagnostic_results;
		diagnostic_results->FindResult(&u_g,"u_g");
		diagnostic_results->FindResult(&p_g,"p_g");
		delete diagnostic_results;

		//compute new thickness
		if(verbose)_printf_("%s\n","computing new thickness");
		
		inputs->Add("velocity",u_g,3,numberofnodes);
		prognostic_results=new DataSet(ResultsEnum()); prognostic_core(prognostic_results,model, inputs);
	
		//extract h_g prognostic_results, and erase prognostic_results;
		prognostic_results->FindResult(&h_g_intermediary,"h_g");
		delete prognostic_results;

		//update surface and bed using the new thickness
		if(verbose)_printf_("   updating geometry\n");
		UpdateGeometryx(&h_g,&b_g,&s_g, fem_p->elements, fem_p->nodes,fem_p->loads, fem_p->materials, fem_p->parameters,h_g_intermediary,b_g,s_g); 
		VecFree(&h_g_intermediary);
		
		if(verbose)_printf_("%s\n","updating node positions");
		UpdateNodePositionsx( fem_dh->elements,fem_dh->nodes,fem_dh->loads,fem_dh->materials,fem_dh->parameters,h_g,b_g);
		UpdateNodePositionsx( fem_dv->elements,fem_dv->nodes,fem_dv->loads,fem_dv->materials,fem_dv->parameters,h_g,b_g);
		UpdateNodePositionsx( fem_dhu->elements,fem_dhu->nodes,fem_dhu->loads,fem_dhu->materials,fem_dhu->parameters,h_g,b_g);
		UpdateNodePositionsx( fem_ds->elements,fem_ds->nodes,fem_ds->loads,fem_ds->materials,fem_ds->parameters,h_g,b_g);
		UpdateNodePositionsx( fem_sl->elements,fem_sl->nodes,fem_sl->loads,fem_sl->materials,fem_sl->parameters,h_g,b_g);
		UpdateNodePositionsx( fem_p->elements,fem_p->nodes,fem_p->loads,fem_p->materials,fem_p->parameters,h_g,b_g);
		UpdateNodePositionsx( fem_t->elements,fem_t->nodes,fem_t->loads,fem_t->materials,fem_t->parameters,h_g,b_g);
		UpdateNodePositionsx( fem_m->elements,fem_m->nodes,fem_m->loads,fem_m->materials,fem_m->parameters,h_g,b_g);

		//plug into results.
		result=new Result(results->Size()+1,time,step,"u_g",u_g); results->AddObject(result);
		result=new Result(results->Size()+1,time,step,"p_g",p_g); results->AddObject(result);
		result=new Result(results->Size()+1,time,step,"h_g",h_g); results->AddObject(result);
		result=new Result(results->Size()+1,time,step,"s_g",s_g); results->AddObject(result);
		result=new Result(results->Size()+1,time,step,"b_g",b_g); results->AddObject(result);
		result=new Result(results->Size()+1,time,step,"t_g",t_g); results->AddObject(result);
		result=new Result(results->Size()+1,time,step,"m_g",m_g); results->AddObject(result);
	}

	/*Free ressources:*/
	xfree((void**)&u_g_serial);
	xfree((void**)&p_g_serial);
	xfree((void**)&h_g_serial);
	xfree((void**)&s_g_serial);
	xfree((void**)&b_g_serial);
	xfree((void**)&t_g_serial);
	xfree((void**)&m_g_serial);
}
