function md_list=modelsextract(md,flags,minel,varargin)
%modelsextract - extract several self contained models according to a list of element flags.
%
%   The difference between this routine and the modelextract.m routine (without an 's') is that 
%   as many models are extracted as there are closed contours defined in area. 
%   This routine is needed for example when doing data assimilation of ice shelves in Antarctica. 
%   Many independent ice shelves are present, and we don't want data assimilation on one ice shelf 
%   to be hindered by another totally independent ice shelf.
%
%   Usage:
%      md_list=modelsextract(md,elementfalgs,minel);
%
%   Examples:
%      md_list=modelsextract(md,md.elementoniceshelf,1000);
%
%   See also: EXTRUDE, COLLAPSE, MODELEXTRACT

disp('selecting pools of elements');
%go through flags and build as many independent element flags as there are groups of connected 1s
%in flags.

%2D or 3D?
if strcmpi(md.type,'3d'),
	numberofelements=md.numberofelements2d; %this will be forgotten when we get out.
	flags=project2d(md,flags,1);
else
	numberofelements=md.numberofelements;
end

%recover extra arguments: 
distance=0;
if nargin==4,
	distance=varargin{1};
end

flag_list=cell(0,1);

for i=1:size(flags,1),

	if (flags(i)),

		%ok, we are sure element i is part of a new pool.
		pool=zeros(numberofelements,1);
		pool=PropagateFlagsFromConnectivity(md.elementconnectivity,pool,i,flags);
		flag_list{end+1,1}=pool;
		
		%speed up rest of computation by taking pool out of flags: 
		pos=find(pool);flags(pos)=0;

	end
end

%go through flag_list and discard any pool of less than minel elements: 
ex_pos=[];
for i=1:length(flag_list),
	if length(find(flag_list{i}))<minel,
		ex_pos=[ex_pos; i];
	end
end
flag_list(ex_pos)=[];

%now, if distance was specified, expand the flag_list by distance km: 
if distance,
	for i=1:length(flag_list),
		flag_list{i}=PropagateFlagsUntilDistance(md,flag_list{i},distance);
	end
end

%now, go use the pools of flags to extract models: 
disp(['extracting ' num2str(size(flag_list,1)) ' models']);
models=cell(0,1);

for i=1:size(flag_list,1),
	disp(['   ' num2str(i) '/' num2str(size(flag_list,1))]);
	if strcmpi(md.type,'3d'),
		flags2d=flag_list{i};
		realflags=project3d(md,flags2d,'element');
	else
		realflags=flag_list{i};
	end
	models{end+1,1}=modelextract(md,realflags);
end

%return model list
md_list=modellist(models);

end %end of this function
