/*!\file Bamgx
 * \brief: use Bamg capabilities.
 */

#include "./Bamgx.h"

#undef __FUNCT__ 
#define __FUNCT__ "Bamgx"

#include "../shared/shared.h"
#include "../include/macros.h"
#include "../toolkits/toolkits.h"

/*Bamg: */
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "Mesh2.h"
#include "QuadTree.h"

using namespace bamg;
using namespace std;

int Bamgx(BamgMesh* bamgmesh,BamgGeom* bamggeom,BamgOpts* bamgopts){
	

	/*output: */
	double* VerticesOut=NULL;
	double* TrianglesOut=NULL;
	int     NumVerticesOut;
	int     NumTrianglesOut;

	/*Bamg options*/
	int    iso;
	int    maxnbv;
	double hmin,hmax;
	double err,errg,coef;
	int    verbosity;
	int    NbSmooth;
	double omega;

	/*intermediary*/
	int noerr=1;
	int i,j,num;
	int allquad=0;
	double costheta=2;
	double anisomax = 1e6;
	int KeepBackVertices=1;
	double hminaniso=1e-100; 
	const double  boundmaxsubdiv = 10;
	double  maxsubdiv=boundmaxsubdiv;

	Triangles* Thr=NULL;
	Triangles* Thb=NULL;

	/*Bamg options*/
	iso=bamgopts->iso;
	err=bamgopts->err;
	errg=bamgopts->errg;
	NbSmooth=bamgopts->NbSmooth;
	omega=bamgopts->omega;
	coef=bamgopts->coef;
	maxnbv=bamgopts->maxnbv;
	hmin=bamgopts->hmin;
	hmax=bamgopts->hmax;
	verbosity=bamgopts->verbose;

	// some verification
	if ( maxsubdiv > boundmaxsubdiv || maxsubdiv <= 1.0){
		throw ErrorException(__FUNCT__,exprintf("maxsubdiv %g is should be in ]1,%g]",maxsubdiv,boundmaxsubdiv));
	}
	if (iso) anisomax=1;
	// no metric -> no smoothing 
	if (bamgopts->metric==NULL){
		NbSmooth=0; 
	}

	if(bamgmesh->NumTriangles==0){
		/*Mesh generation {{{1*/

		if (verbosity>0) printf("Construction of a mesh from a given geometry\n");
		if (verbosity>1) printf("   Processing geometry...\n");
		Geometry Gh(bamggeom,bamgopts);
		if (verbosity>10){
			Gh.Echo();
		}

		//get hmin and hmax from geometry to generate the metric
		hmin = Max(hmin,Gh.MinimalHmin());
		hmax = Min(hmax,Gh.MaximalHmax());

		//build metric if not given in input
		if (verbosity>1) printf("   Generating Metric...\n");
		for(i=0;i<Gh.nbv;i++){
			MetricAnIso M=Gh[i];
			MatVVP2x2 Vp(M/coef);
			Vp.Maxh(hmax);
			Vp.Minh(hmin);
			Gh.vertices[i].m = Vp;
		}

		//generate mesh
		if (verbosity>1) printf("   Generating Mesh...\n");
		Triangles Th(maxnbv,Gh);
		if(bamgopts->splitcorners) Th.SplitInternalEdgeWithBorderVertices();
		Th.ReNumberingTheTriangleBySubDomain();
		//if(NbSmooth>0) Th.SmoothingVertex(NbSmooth,omega);
		Th.SmoothingVertex(3,omega);

		//Build output
		if (verbosity>1) printf("   Write Mesh...\n");
		Th.WriteMesh(bamgmesh,bamgopts);
		if (verbosity>1) printf("   Write Geometry...\n");
		Gh.WriteGeometry(bamggeom,bamgopts);
		/*}}}*/
	}

	else{
		/*Anisotropic mesh adaptation {{{1*/

		// read background mesh 
		if (verbosity>0) printf("Anisotropic mesh adaptation\n");
		if (verbosity>1) printf("   Processing initial mesh and geometry...\n");
		Triangles BTh(bamgmesh,bamgopts); 
		hmin = Max(hmin,BTh.MinimalHmin());
		hmax = Min(hmax,BTh.MaximalHmax());

		//Make Quadtree from background mesh
		BTh.MakeQuadTree();

		//build metric if not given in input
		if (bamgopts->metric){
			if (verbosity>1) printf("   Processing Metric...\n");
			BTh.ReadMetric(bamgopts,hmin,hmax,coef);
		}
		else { 

			// init with hmax 
			Metric Mhmax(hmax);
			for (Int4 iv=0;iv<BTh.nbv;iv++) BTh[iv].m = Mhmax;

			//use present fields to generate metric if present
			if (bamgopts->field){
				if (verbosity>1) printf("   Generating Metric from solution field...\n");
				BTh.IntersectConsMetric(bamgopts);
			}
		}

		BTh.IntersectGeomMetric(errg,iso);
		if (verbosity>1) printf("   Smoothing Metric...");
		if(bamgopts->gradation) BTh.SmoothMetric(bamgopts->gradation);
		if (verbosity>1) printf(" done\n");
		BTh.MaxSubDivision(maxsubdiv);
		BTh.BoundAnisotropy(anisomax,hminaniso);

		//Build new mesh Thr
		if (verbosity>1) printf("   Generating Mesh...\n");
		Thr=&BTh,Thb=0;
		Triangles & Th( *(0 ?  new Triangles(*Thr,&Thr->Gh,Thb,maxnbv) :  new Triangles(maxnbv,BTh,KeepBackVertices)));
		if (Thr != &BTh) delete Thr;
		if(costheta<=1.0) Th.MakeQuadrangles(costheta);
		if (allquad) Th.SplitElement(allquad==2);
		if(bamgopts->splitcorners) Th.SplitInternalEdgeWithBorderVertices();
		Th.ReNumberingTheTriangleBySubDomain();
		if(NbSmooth>0) Th.SmoothingVertex(NbSmooth,omega);
		Th.BTh.ReMakeTriangleContainingTheVertex();

		//display info
		if(verbosity>0) {
			if (Th.nbt-Th.NbOutT-Th.NbOfQuad*2){
				printf("   new number of triangles = %i\n",(Th.nbt-Th.NbOutT-Th.NbOfQuad*2));
			}
			if (Th.NbOfQuad ){
				printf("   new number of quads = %i\n",Th.NbOfQuad);
			}
		}

		//Build output
		if (verbosity>1) printf("   Write Mesh...\n");
		Th.WriteMesh(bamgmesh,bamgopts);
		if (verbosity>1) printf("   Write Geometry...\n");
		Th.Gh.WriteGeometry(bamggeom,bamgopts);
		if (verbosity>1) printf("   Write Metric...\n");
		BTh.WriteMetric(bamgopts);

		/*clean up*/
		delete &Th; //TEST crash
		/*}}}*/
	}

	/*No error return*/
	if (verbosity>1) printf("   Exiting Bamg.\n");
	return noerr;

}
