#include <cstdio>
#include <string.h>
#include <cmath>
#include <time.h>

#include "../Mesh2.h"
#include "../QuadTree.h"
#include "../SetOfE4.h"

#include "../../shared/shared.h"
#include "../../include/macros.h"
#include "../../toolkits/toolkits.h"

#undef __FUNCT__ 
#define __FUNCT__ "MetricAnIso"

using namespace std;

namespace bamg {

	SaveMetricInterpole  LastMetricInterpole;

	/*Constructor/Destructor*/
	/*FUNCTION MetricAnIso::MetricAnIso(const Real8  a[3],const  MetricAnIso m0, const  MetricAnIso m1,const  MetricAnIso  m2 ){{{1*/ 
	MetricAnIso::MetricAnIso(const Real8  a[3],const  MetricAnIso m0, const  MetricAnIso m1,const  MetricAnIso m2 ){
		MetricAnIso mab(a[0]*m0.a11 + a[1]*m1.a11 + a[2]*m2.a11,
					a[0]*m0.a21 + a[1]*m1.a21 + a[2]*m2.a21,
					a[0]*m0.a22 + a[1]*m1.a22 + a[2]*m2.a22);

		MatVVP2x2 vab(mab);

		R2 v1(vab.v.x,vab.v.y);
		R2 v2(-v1.y,v1.x);

		Real8 h1 = a[0] / m0(v1) + a[1] / m1(v1) + a[2] / m2(v1);
		Real8 h2 = a[0] / m0(v2) + a[1] / m1(v2) + a[2] / m2(v2);

		vab.lambda1 =  1 / (h1*h1);
		vab.lambda2 =  1 / (h2*h2);
		*this = vab;
	}
	/*}}}1*/
	/*FUNCTION MetricAnIso::MetricAnIso( Real8  a,const  MetricAnIso ma, Real8  b,const  MetricAnIso mb){{{1*/
	MetricAnIso::MetricAnIso( Real8  a,const  MetricAnIso ma, Real8  b,const  MetricAnIso mb) { 
		MetricAnIso mab(a*ma.a11+b*mb.a11,a*ma.a21+b*mb.a21,a*ma.a22+b*mb.a22);
		MatVVP2x2 vab(mab);

		R2 v1(vab.v.x,vab.v.y);
		R2 v2(-v1.y,v1.x);


		Real8 h1 = a / ma(v1) + b / mb(v1);
		Real8 h2 = a / ma(v2) + b / mb(v2);
		vab.lambda1 =  1 / (h1*h1);
		vab.lambda2 =  1 / (h2*h2);
		*this = vab;
	}
	/*}}}1*/

	/*Methods*/
	/*FUNCTION MetricAnIso::Echo {{{1*/

	void MetricAnIso::Echo(void){

		printf("MetricAnIso:\n");
		printf("   [a11 a21 a22]: [%g %g %g]\n",a11,a21,a22);

		return;
	}
	/*}}}*/
	/*FUNCTION MetricAnIso::IntersectWith_new{{{1*/
	int MetricAnIso::IntersectWith_new(MetricAnIso M2) {
		/*Get a new metric from an existing metric (M1=this)
		 * and a new metric given in input M2 using a 
		 * Simultaneous Matrix Reduction
		 * */

		MetricAnIso M;

		//Compute M intersection of M1 and M2
		SimultaneousMatrixReduction(*this,M2,&M);

		//check wether something has been done
		if(M.a11!=a11 || M.a21!=a21 || M.a22!=a22){

			//update this with M
			a11=M.a11;
			a21=M.a21;
			a22=M.a22;

			//return 1 -> something has been done
			return 1;
		}
		else{

			//nothing has changed, return 0
			return 0;
		}
	}
	/*}}}1*/
	/*FUNCTION MetricAnIso::IntersectWith{{{1*/
	int MetricAnIso::IntersectWith(const MetricAnIso M2) {
		/*Get a new metric from an existing metric (M1=this)
		 * and a new metric given in input M2 using a 
		 * Simultaneous Matrix Reduction
		 * */

		int r=0;
		MetricAnIso & M1 = *this;
		D2xD2 M;
		double l1,l2;

		SimultaneousMatrixReduction_bamg(*this,M2,l1,l2,M);

		R2 v1(M.x.x,M.y.x);
		R2 v2(M.x.y,M.y.y);
		double l11=M1(v1,v1);
		double l12=M1(v2,v2);
		double l21=M2(v1,v1);
		double l22=M2(v2,v2);
		if ( l11 < l21 )  r=1,l11=l21;
		if ( l12 < l22 )  r=1,l12=l22; 
		if (r) { // change
			D2xD2 M_1(M.inv());
			D2xD2 D(l11,0,0,l12); 
			D2xD2 Mi(M_1.t()*D*M_1);
			a11=Mi.x.x;
			a21=0.5*(Mi.x.y+Mi.y.x);
			a22=Mi.y.y; }
			return r;
	}
	/*}}}1*/

	/*Intermediary*/
	/*FUNCTION eigen2{{{1*/
	void eigen2(double* M,double* lambda,double vp[2][2]) {
		/*m=[a b]
		 *  [c d] */

		double normM,detM,trM;
		double norm;
		const double a=M[2*0+0];
		const double b=M[2*0+1];
		const double c=M[2*1+0];
		const double d=M[2*1+1];

		//compute norm(M)
		normM=pow(pow(a,2.0)+pow(b,2.0)+pow(c,2.0)+pow(d,2.0),0.5);

		//if normM<10-30, M=zeros(2,2)
		if (normM<1.0e-30){
			lambda[0]=0.0;
			lambda[1]=0.0;
			vp[0][0]=vp[0][1]=vp[1][0]=vp[1][1]=0.0;
			return;
		}

		//check that matrix is already diagonal if det(M)=0
		if(Abs(b)<(1.0e-12*normM) & Abs(c)<(1.0e-12*normM)){
			lambda[0]=a;
			lambda[1]=d;
			vp[0][0]=1;
			vp[0][1]=0;
			vp[1][0]=0;
			vp[1][1]=1;
			return;
		}

		//compute det(M)
		detM=a*d-c*b;

		//not symetrical and still det<10^-30
		if (Abs(detM)<1.0e-30){
			throw ErrorException(__FUNCT__,exprintf("Cannot find eigen values of a non invertible matrix"));
		}

		//See http://www.math.harvard.edu/archive/21b_fall_04/exhibits/2dmatrices/index.html
		trM=a+d;

		//general case
		lambda[0] = 0.5*(trM + pow( trM*trM - 4.0*detM ,0.5));
		lambda[1] = 0.5*(trM - pow( trM*trM - 4.0*detM ,0.5));

		if(Abs(b)>1.0e-30){ //if b!=0
			norm=pow(b*b+pow(lambda[0]-a,2.0),0.5);
			vp[0][0]=b/norm;
			vp[1][0]=(lambda[0]-a)/norm;
			norm=pow(b*b+pow(lambda[1]-a,2.0),0.5);
			vp[0][1]=b/norm;
			vp[1][1]=(lambda[1]-a)/norm;
		}
		else{
			norm=pow(c*c+pow(lambda[0]-d,2.0),0.5);
			vp[0][0]=(lambda[0]-d)/norm;
			vp[1][0]=c/norm;
			norm=pow(c*c+pow(lambda[1]-d,2.0),0.5);
			vp[0][1]=(lambda[1]-d)/norm;
			vp[1][1]=c/norm;
		}
	}
	/*}}}1*/
	/*FUNCTION LengthInterpole{{{1*/
	Real8 LengthInterpole(const MetricAnIso Ma,const  MetricAnIso Mb, R2 AB) {
		Real8 k=1./2.;
		int level=0;
		static int kkk=0;
		static  Metric Ms1[32],Ms2[32];
		static Real8 lMs1[32],lMs2[32];
		static double K[32];
		Real8 l=0,sss=0;
		Ms1[level]=Ma;
		Ms2[level]=Mb;
		Real8 sa =  Ma(AB);
		Real8 sb =  Mb(AB);
		lMs1[level]=sa;
		lMs2[level]=sb;
		K[level]=k;
		level++;
		int i=0;
		Real8 * L= LastMetricInterpole.L, *S = LastMetricInterpole.S;
		Real8  sstop = 0.1; // Max(0.6,(sa+sb)/5000);
		while (level) {
			level--;
			Metric M1=Ms1[level];
			Metric M2=Ms2[level];
			k=K[level];
			Real8 s1=  lMs1[level];
			Real8 s2=  lMs2[level];

			Real8 s= (s1+s2)*k;
			if( s > sstop   && level < 30 && i < 500-level ) {
				Metric Mi(0.5,M1,0.5,M2);
				Real8 si = Mi(AB);
				if( Abs((s1+s2)-(si+si)) > s1*0.001) 
				  {
					k=k/2;
					// we begin by the end to walk in the correct sens from a to b
					// due to the stack 
					Ms1[level]=Mi;
					Ms2[level]=M2;
					lMs1[level]=si;
					lMs2[level]=s2;
					K[level]=k;
					level++;
					Ms1[level]=M1;
					Ms2[level]=Mi;
					lMs1[level]=s1;
					lMs2[level]=si;
					K[level]=k;
					level++;
				  }
				else
				 L[i]= l += s,S[i]=sss+=k,i++;
			}
			else 
			 L[i]= l += s,S[i]=sss+=k,i++;
		}
		// warning for optimisation S is in [0:0.5] not in [0:1]
		if (i>=512){
			throw ErrorException(__FUNCT__,exprintf("i>=512"));
		}
		LastMetricInterpole.lab=l;
		LastMetricInterpole.opt=i;
		if (i>200 && kkk++<10) printf("WARNING: LengthInterpole: ( i=%i l=%i sss=%i ) %g\n",i,l,sss,sstop); 
		return l;
	}
	/*}}}1*/
	/*FUNCTION SimultaneousMatrixReduction{{{1*/
	void SimultaneousMatrixReduction(MetricAnIso M1,MetricAnIso M2,MetricAnIso* pM) {

		/*intermediary*/
		double a11,a21,a22;
		double b11,b21,b22;
		double mu1,mu2;
		double lambda1,lambda2;
		double detM1,detP;
		double N[2][2];
		double lambda[2];
		double vp[2][2];
		double invP[2][2];
		MetricAnIso M;

		//get components of both metrics
		a11=M1.a11; a21=M1.a21; a22=M1.a22;
		b11=M2.a11; b21=M2.a21; b22=M2.a22;

		//check diagnonal matrices (easy)
		if (Abs(a21)< 1.0e-20 && Abs(b21)< 1.0e-20){
			M.a11=Max(a11,b11);
			M.a22=Max(a22,b22);
			M.a21=0;
			*pM=M;
			return;
		}

		//Build N=M1^-1 M2 (Alauzet2003 p16)
		detM1=a11*a22-a21*a21;
		if (!detM1){
			printf("a11 a21 a22 = %g %g %g\n",a11,a21,a22);
			throw ErrorException(__FUNCT__,exprintf("One of the metric has a zero eigen value or cannot be inverted... (detP=%g)",detP));
		}
		N[0][0]= 1.0/detM1 * ( a22*b11-a21*b21);
		N[0][1]= 1.0/detM1 * ( a22*b21-a21*b22);
		N[1][0]= 1.0/detM1 * (-a21*b11+a11*b21);
		N[1][1]= 1.0/detM1 * (-a21*b21+a11*b22);

		//now, we must find the eigen values and vectors of N
		eigen2(&N[0][0],lambda,vp);

		//Now that we have P=vp, we can recompute M = M1 inter M2
		//
		//      -T [ max(lambda1, mu1)          0         ]  -1
		// M = P   [                                      ] P
		//         [        0            max(lambda2, mu2)]

		//get det(P)
		detP=vp[0][0]*vp[1][1]-vp[0][1]*vp[1][0];
		if (!detP){
			printf("vp= [%g %g ; %g %g]\n",vp[0][0],vp[0][1],vp[1][0],vp[1][1]);
			throw ErrorException(__FUNCT__,exprintf("One of the metric has a zero eigen value or cannot be inverted... (detP=%g)",detM1));
		}
		invP[0][0]= 1.0/detP * ( vp[1][1]);
		invP[0][1]= 1.0/detP * (-vp[0][1]);
		invP[1][0]= 1.0/detP * (-vp[1][0]);
		invP[1][1]= 1.0/detP * ( vp[0][0]);

		//Compute lambdai and mui using Rayleigh formula: lambdai = ei' M1 ei
		lambda1=vp[0][0]*(a11*vp[0][0]+a21*vp[1][0]) + vp[1][0]*(a21*vp[0][0]+a22*vp[1][0]);
		lambda2=vp[0][1]*(a11*vp[0][1]+a21*vp[1][1]) + vp[1][1]*(a21*vp[0][1]+a22*vp[1][1]);
		mu1    =vp[0][0]*(b11*vp[0][0]+b21*vp[1][0]) + vp[1][0]*(b21*vp[0][0]+b22*vp[1][0]);
		mu2    =vp[0][1]*(b11*vp[0][1]+b21*vp[1][1]) + vp[1][1]*(b21*vp[0][1]+b22*vp[1][1]);

		//finally compute M
		M.a11=invP[0][0]*Max(lambda1,mu1)*invP[0][0] + invP[1][0]*Max(lambda2,mu2)*invP[1][0];
		M.a21=invP[0][0]*Max(lambda1,mu1)*invP[0][1] + invP[1][0]*Max(lambda2,mu2)*invP[1][1];
		M.a22=invP[0][1]*Max(lambda1,mu1)*invP[0][1] + invP[1][1]*Max(lambda2,mu2)*invP[1][1];
		*pM=M;
	}
	/*}}}1*/
	/*FUNCTION SimultaneousMatrixReduction_bamg{{{1*/
	void SimultaneousMatrixReduction_bamg( MetricAnIso M1,  MetricAnIso M2,double & l1,double & l2, D2xD2 & V) {
		double a11=M1.a11,a21=M1.a21,a22=M1.a22;
		double b11=M2.a11,b21=M2.a21,b22=M2.a22;
		//  M1 v = l M2 v
		// (M1 - l M2) v =0
		// det (M1 - l M2) =0
		// det (M1 - l M2) = a l^2 + b l + c;
		// = (a11 - l * b11) * (a22 - l * b22) - (a21 - l * b21 ) ^2
		const double /*c11 = a11*a11,*/ c21= a21*a21;
		const double /*d11 = b11*b11,*/ d21= b21*b21;
		const double a=b11*b22 - d21;
		const double b=-a11*b22-a22*b11+2*a21*b21;
		const double c=-c21+a11*a22;
		const double bb = b*b,ac= a*c;
		const double delta = bb - 4 * ac;
		if (bb + Abs(ac) < 1.0e-20 || (delta< 1.0E-4 * bb ) ){
			// racine double;
			if (Abs(a) < 1.e-30 )
			 l1 = l2 = 0;
			else 
			 l1=l2=-b/(2*a); 
			V= D2xD2(1,0,0,1);
		}
		else {
			const double delta2 = sqrt(delta);
			l1= (-b - delta2)/(2*a);
			l2= (-b + delta2)/(2*a);
			// M1 v = l M2 v
			//  ( (M1 - I M2) x,y)  = (x,(M1 - I M2) y) \forall y
			// so Ker((M1 - I M2)) = Im((M1 - I M2))^\perp
			double v0 = a11-l1*b11, v1 = a21-l1*b21,v2 = a22 - l1*b22;
			double s0 = v0*v0 + v1*v1, s1 = v1*v1 +v2*v2;
			double vp1x,vp1y,vp2x,vp2y;

			if(s1 < s0)
			 s0=sqrt(s0),vp1x=v1/s0,vp1y=-v0/s0;
			else
			 s1=sqrt(s1),vp1x=v2/s1,vp1y=-v1/s1;

			v0 = a11-l2*b11, v1 = a21-l2*b21,v2 = a22 - l2*b22;
			s0 = v0*v0 + v1*v1, s1 = v1*v1 +v2*v2;
			if(s1 < s0)
			 s0=sqrt(s0),vp2x=v1/s0,vp2y=-v0/s0;
			else
			 s1=sqrt(s1),vp2x=v2/s1,vp2y=-v1/s1;
			V=D2xD2(vp1x,vp2x,vp1y,vp2y);
		}
		return;
	}
	/*}}}1*/
	/*FUNCTION abscisseInterpole{{{1*/
	Real8 abscisseInterpole(const MetricAnIso Ma,const  MetricAnIso Mb, R2 AB,Real8 s,int optim) { 
		if(!optim)  LengthInterpole(Ma,Mb,AB);
		Real8 l  = s* LastMetricInterpole.lab,r;
		int j=LastMetricInterpole.opt-1,i=0,k;

		Real8 * L= LastMetricInterpole.L, *S = LastMetricInterpole.S;
		// warning for optimisation S is the abcisse in [0:0.5]
		// and L is le lenght 
		if(l<=L[0])
		 r=2*S[0]*l/L[0];
		else if (l>=L[j])
		 r=1;
		else 
		  {
			while (j-i>1)
			  {
				k= (i+j)/2;
				if(l<=L[k])
				 j=k;// l<=L[j] 
				else
				 i=k; //  L[i]<l
			  };
			if (i==j)
			 r = 2*S[i];
			else
			 r =  2*(S[i]*(L[j]-l)+ S[j]*(l-L[i]))/(L[j]-L[i]);
		  }
		if (r>1 || r<0){
			throw ErrorException(__FUNCT__,exprintf("r>1 || r<0"));
		}
		return r ;
	}
	/*}}}1*/

}
