#include <cstdio>
#include <cstring>
#include <cmath>
#include <ctime>

#include "../Mesh2.h"
#include "../QuadTree.h"
#include "../SetOfE4.h"

#include "../../shared/shared.h"
#include "../../include/macros.h"
#include "../../toolkits/toolkits.h"

#undef __FUNCT__ 
#define __FUNCT__ "Geometry"

namespace bamg {

	static const  Direction NoDirOfSearch=Direction();

	/*Constructors/Destructors*/
	/*FUNCTION  Geometry::Geometry(const Geometry & Gh){{{1*/
	Geometry::Geometry(const Geometry & Gh) {
		Int4 i;
		*this = Gh;
		NbRef =0;
		quadtree=0;
		name = new char[strlen(Gh.name)+4];
		strcpy(name,"cp:");
		strcat(name,Gh.name);
		vertices = nbv ? new GeometricalVertex[nbv] : NULL;
		triangles = nbt ? new  Triangle[nbt]:NULL;
		edges = nbe ? new GeometricalEdge[nbe]:NULL;
		curves= NbOfCurves ? new Curve[NbOfCurves]:NULL;
		subdomains = NbSubDomains ? new GeometricalSubDomain[NbSubDomains]:NULL;
		for (i=0;i<nbv;i++)
		 vertices[i].Set(Gh.vertices[i],Gh,*this);
		for (i=0;i<nbe;i++)
		 edges[i].Set(Gh.edges[i],Gh,*this);
		for (i=0;i<NbOfCurves;i++)
		 curves[i].Set(Gh.curves[i],Gh,*this);
		for (i=0;i<NbSubDomains;i++)
		 subdomains[i].Set(Gh.subdomains[i],Gh,*this);
		if (nbt);  {
			throw ErrorException(__FUNCT__,exprintf("nbt"));
		}
	}
	/*}}}1*/
	/*FUNCTION Geometry::~Geometry(){{{1*/
	Geometry::~Geometry() {
		long int verbosity=0;

		if (NbRef>0){
			throw ErrorException(__FUNCT__,exprintf("NbRef>0"));
		}
		if(vertices)  delete [] vertices;vertices=0;
		if(edges)     delete [] edges;edges=0;
		// if(edgescomponante) delete [] edgescomponante; edgescomponante=0;
		if(triangles) delete [] triangles;triangles=0;
		if(quadtree)  delete  quadtree;quadtree=0;
		if(curves)  delete  []curves;curves=0;NbOfCurves=0;
		if(name) delete [] name;name=0;
		if(subdomains) delete [] subdomains;subdomains=0;
		//  if(ordre)     delete [] ordre;
		EmptyGeometry();
	}
	/*}}}1*/

	/*IO*/
	/*FUNCTION Geometry::WriteGeometry{{{1*/
	void Geometry::WriteGeometry(BamgGeom* bamggeom, BamgOpts* bamgopts){

		int verbose;
		int nbreq=0;
		int nbreqv=0;
		int nbtan=0;
		int nbcracked=0;
		int i,count;

		verbose=bamgopts->verbose;

		//Vertices
		if(verbose>3) printf("      writing Vertices\n");
		bamggeom->NumVertices=nbv;
		xfree((void**)&bamggeom->Vertices);
		if (nbv){
			bamggeom->Vertices=(double*)xmalloc(3*nbv*sizeof(double));
			for (i=0;i<nbv;i++){
				bamggeom->Vertices[i*3+0]=vertices[i].r.x;
				bamggeom->Vertices[i*3+1]=vertices[i].r.y;
				bamggeom->Vertices[i*3+2]=vertices[i].ref();

				//update counters
				if (vertices[i].Required()) nbreqv++;
			}
		}
		else{
			bamggeom->Vertices=NULL;
		}

		//Edges
		if(verbose>3) printf("      writing Edges\n");
		bamggeom->NumEdges=nbe;
		xfree((void**)&bamggeom->Edges);
		if (nbe){
			bamggeom->Edges=(double*)xmalloc(3*nbe*sizeof(double));
			for (i=0;i<nbe;i++){
				bamggeom->Edges[i*3+0]=Number(edges[i][0])+1; //back to Matlab indexing
				bamggeom->Edges[i*3+1]=Number(edges[i][1])+1; //back to Matlab indexing
				//bamggeom->Edges[i*3+2]=(double)edges[i].ref(); //TEST does not compile???
				bamggeom->Edges[i*3+2]=1; //TEST for now

				//update counters
				if (edges[i].Required()) nbreq++;
				if (edges[i].Cracked()){
					if (i<=Number(edges[i].link)) nbcracked++;
				}
				if (edges[i].TgA() && edges[i][0].Corner()) nbtan++;
				if (edges[i].TgB() && edges[i][1].Corner()) nbtan++;
			}
		}
		else{
			bamggeom->Edges=NULL;
		}

		//CrackedEdges
		if(verbose>3) printf("      writing CrackedEdges\n");
		bamggeom->NumCrackedEdges=nbcracked;
		xfree((void**)&bamggeom->CrackedEdges);
		if (nbcracked){
			bamggeom->CrackedEdges=(double*)xmalloc(2*nbcracked*sizeof(double));
			count=0;
			for (i=0;i<nbe;i++){
				if (edges[i].Cracked()){
					if (i<=Number(edges[i].link)){
						bamggeom->CrackedEdges[count*2+0]=i+1;                     //back to Matlab indexing
						bamggeom->CrackedEdges[count*2+1]=Number(edges[i].link)+1; //back to Matlab indexing
						count=count+1;
					}
				}
			}
		}
		else{
			bamggeom->CrackedEdges=NULL;
		}

		//RequiredEdges
		if(verbose>3) printf("      writing RequiredEdges\n");
		bamggeom->NumRequiredEdges=nbreq;
		xfree((void**)&bamggeom->RequiredEdges);
		if (nbreq){
			bamggeom->RequiredEdges=(double*)xmalloc(1*nbreq*sizeof(double));
			count=0;
			for (i=0;i<nbe;i++){
				if (edges[i].Required()){
					bamggeom->RequiredEdges[count]=i+1; //back to Matlab indexing
					count=count+1;
				}
			}
		}
		else{
			bamggeom->RequiredEdges=NULL;
		}

		//No corners

		//RequiredVertices
		if(verbose>3) printf("      writing RequiredVertices\n");
		bamggeom->NumRequiredVertices=nbreqv;
		xfree((void**)&bamggeom->RequiredVertices);
		if (nbreqv){
			bamggeom->RequiredVertices=(double*)xmalloc(1*nbreqv*sizeof(double));
			count=0;
			for (i=0;i<nbe;i++){
				if (vertices[i].Required()){
					bamggeom->RequiredVertices[count]=i+1; //back to Matlab indexing
					count=count+1;
				}
			}
		}
		else{
			bamggeom->RequiredVertices=NULL;
		}

		//SubDomains
		if(verbose>3) printf("      writing SubDomains\n");
		bamggeom->NumSubDomains=NbSubDomains;
		xfree((void**)&bamggeom->SubDomains);
		if (NbSubDomains){
			bamggeom->SubDomains=(double*)xmalloc(4*NbSubDomains*sizeof(double));
			for (i=0;i<NbSubDomains;i++){
				bamggeom->SubDomains[4*i+0]=2;
				bamggeom->SubDomains[4*i+1]=Number(subdomains[i].edge)+1; //back to Matlab indexing
				bamggeom->SubDomains[4*i+2]=subdomains[i].sens;
				bamggeom->SubDomains[4*i+3]=subdomains[i].ref;
			}
		}
		else{
			bamggeom->SubDomains=NULL;
		}

		//TangentAtEdges
		if(verbose>3) printf("      writing TangentAtEdges\n");
		bamggeom->NumTangentAtEdges=nbtan;
		xfree((void**)&bamggeom->TangentAtEdges);
		if (nbtan){
			bamggeom->TangentAtEdges=(double*)xmalloc(4*nbtan*sizeof(double));
			count=0;
			for (i=0;i<nbe;i++){
				if (edges[i].TgA() && edges[i][0].Corner()){
					bamggeom->TangentAtEdges[4*i+0]=i+1; //back to Matlab indexing
					bamggeom->TangentAtEdges[4*i+1]=1;
					bamggeom->TangentAtEdges[4*i+2]=edges[i].tg[0].x;
					bamggeom->TangentAtEdges[4*i+3]=edges[i].tg[0].y;
				}
				if (edges[i].TgB() && edges[i][1].Corner()){
					bamggeom->TangentAtEdges[4*i+0]=i+1; //back to Matlab indexing
					bamggeom->TangentAtEdges[4*i+1]=2;
					bamggeom->TangentAtEdges[4*i+2]=edges[i].tg[1].x;
					bamggeom->TangentAtEdges[4*i+3]=edges[i].tg[2].y;
				}
				count=count+1;
			}
		}
		else{
			bamggeom->TangentAtEdges=NULL;
		}
	}
	/*}}}1*/
	/*FUNCTION Geometry::ReadGeometry{{{1*/
	void Geometry::ReadGeometry(BamgGeom* bamggeom,BamgOpts* bamgopts){

		int verbose;
		nbiv=nbv=nbvx=0;
		nbe=nbt=nbtx=0;
		NbOfCurves=0;

		Real8 Hmin = HUGE_VAL;// the infinie value 
		Int4 hvertices=0;
		int i,j,k,n;

		//initialize some variables
		int Version=1,dim=2;
		nbv=bamggeom->NumVertices;
		nbe=bamggeom->NumEdges;
		nbvx = nbv;
		nbiv = nbv;
		verbose=bamgopts->verbose;

		//some checks
		if (bamggeom->NumVertices<=0 || bamggeom->Vertices==NULL){
			throw ErrorException(__FUNCT__,exprintf("the domain provided does not contain any vertex"));
		}
		if (bamggeom->NumEdges<=0 || bamggeom->Edges==NULL){
			throw ErrorException(__FUNCT__,exprintf("the domain provided does not contain any edge"));
		}

		//Vertices
		if (bamggeom->Vertices){
			if(verbose>3) printf("      processing Vertices\n");
			vertices = new GeometricalVertex[nbvx];
			for (i=0;i<nbv;i++) {
				vertices[i].r.x=(double)bamggeom->Vertices[i*3+0];
				vertices[i].r.y=(double)bamggeom->Vertices[i*3+1];
				vertices[i].ReferenceNumber=(Int4)bamggeom->Vertices[i*3+2];
				vertices[i].DirOfSearch=NoDirOfSearch;
				vertices[i].color =0;
				vertices[i].Set();
			}
			//find domain extrema for pmin,pmax that will define the square box
			//used for by the quadtree
			pmin =  vertices[0].r;
			pmax =  vertices[0].r;
			for (i=0;i<nbv;i++) {
				pmin.x = Min(pmin.x,vertices[i].r.x);
				pmin.y = Min(pmin.y,vertices[i].r.y);
				pmax.x = Max(pmax.x,vertices[i].r.x);
				pmax.y = Max(pmax.y,vertices[i].r.y);
			}
			R2 DD05 = (pmax-pmin)*0.05;
			pmin -=  DD05;
			pmax +=  DD05;
			//coefIcoor is the coefficient used to have coordinates
			//in ]0 1[^2:  x'=coefIcoor*(x-pmin.x)
			coefIcoor= (MaxICoor)/(Max(pmax.x-pmin.x,pmax.y-pmin.y));
			if(coefIcoor <=0){
				throw ErrorException(__FUNCT__,exprintf("coefIcoor should be positive"));
			}
		}
		else{
			throw ErrorException(__FUNCT__,exprintf("No Vertex provided"));
		}

		//Edges
		if (bamggeom->Edges){
			int i1,i2;
			R2 zero2(0,0);
			Real4 *len =0;

			if(verbose>3) printf("      processing Edges\n");
			edges = new GeometricalEdge[nbe];

			if (!hvertices) {
				len = new Real4[nbv];
				for(i=0;i<nbv;i++)
				 len[i]=0;
			}

			for (i=0;i<nbe;i++){
				i1=(int)bamggeom->Edges[i*3+0]-1; //-1 for C indexing
				i2=(int)bamggeom->Edges[i*3+1]-1; //-1 for C indexing
				edges[i].ref=(Int4)bamggeom->Edges[i*3+2];
				edges[i].v[0]=  vertices + i1;
				edges[i].v[1]=  vertices + i2;
				R2 x12 = vertices[i2].r-vertices[i1].r;
				Real8 l12=sqrt((x12,x12));
				edges[i].tg[0]=zero2;
				edges[i].tg[1]=zero2;
				edges[i].DirAdj[0] = edges[i].DirAdj[1] = -1;
				edges[i].Adj[0] = edges[i].Adj[1] = 0;
				edges[i].flag = 0;
				if (!hvertices) {
					vertices[i1].color++;
					vertices[i2].color++;
					len[i1] += l12;
					len[i2] += l12;
				}

				Hmin = Min(Hmin,l12);
			}

			// definition  the default of the given mesh size 
			if (!hvertices){
				for (i=0;i<nbv;i++) 
				 if (vertices[i].color > 0) 
				  vertices[i].m=  Metric(len[i] /(Real4) vertices[i].color);
				 else 
				  vertices[i].m=  Metric(Hmin);
				delete [] len;
			}
		}
		else{
			throw ErrorException(__FUNCT__,exprintf("No edges provided"));
		}

		//hVertices
		if(bamggeom->hVertices){
			if(verbose>3) printf("      processing hVertices\n");
			for (i=0;i< nbv;i++){
				vertices[i].m=Metric((Real4)bamggeom->hVertices[i]);
			}
		}
		else{
			if(verbose>3) printf("      no hVertices found\n");
		}

		//MetricVertices
		if(bamggeom->MetricVertices){
			if(verbose>3) printf("      processing MetricVertices\n");
			hvertices=1;
			for (i=0;i< nbv;i++) {
				vertices[i].m = Metric((Real4)bamggeom->MetricVertices[i*3+0],(Real4)bamggeom->MetricVertices[i*3+1],(Real4)bamggeom->MetricVertices[i*3+2]);
			}
		}
		else{
			if(verbose>3) printf("      no MetricVertices found\n");
		}

		//h1h2VpVertices
		if(bamggeom->h1h2VpVertices){
			if(verbose>3) printf("      processing h1h2VpVertices\n");
			Real4 h1,h2,v1,v2;
			hvertices =1;
			for (i=0;i< nbv;i++) {
				h1=(Real4)bamggeom->MetricVertices[i*4+0];
				h2=(Real4)bamggeom->MetricVertices[i*4+1];
				v1=(Real4)bamggeom->MetricVertices[i*4+2];
				v2=(Real4)bamggeom->MetricVertices[i*4+3];
				vertices[i].m = Metric(MatVVP2x2(1/(h1*h1),1/(h2*h2),D2(v1,v2)));
			}
		}
		else{
			if(verbose>3) printf("      no h1h2VpVertices found\n");
		}

		//MaximalAngleOfCorner
		if (bamgopts->MaximalAngleOfCorner){
			if(verbose>3) printf("      processing MaximalAngleOfCorner\n");
			MaximalAngleOfCorner=bamgopts->MaximalAngleOfCorner*Pi/180;
		}
		else{
			if(verbose>3) printf("      no MaximalAngleOfCorner found\n");
		}

		//TangentAtEdges
		if (bamggeom->TangentAtEdges){
			if(verbose>3) printf("      processing TangentAtEdges");
			int n,i,j,k;
			R2 tg;

			n=bamggeom->NumTangentAtEdges;
			for (k=0;k<n;k++) {
				i=(int)bamggeom->TangentAtEdges[k*4+0]-1; //for C indexing
				j=(int)bamggeom->TangentAtEdges[k*4+1]-1; //for C indexing
				tg.x=bamggeom->TangentAtEdges[k*4+2];
				tg.y=bamggeom->TangentAtEdges[k*4+3];
				if (j!=0 && j!=1){
					throw ErrorException(__FUNCT__,exprintf("TangentAtEdges second index should be 1 or 2 only"));
				}
				edges[i].tg[j] = tg;
			}
		}
		else{
			if(verbose>3) printf("      no TangentAtEdges found\n");
		}

		//Corners
		if(bamggeom->Corners){
			if(verbose>3) printf("      processing Corners");
			n=bamggeom->NumCorners;
			for (i=0;i<n;i++) {     
				j=(int)bamggeom->Corners[i]-1; //for C indexing
				if (j>nbv-1 || j<0){
					throw ErrorException(__FUNCT__,exprintf("Bad corner definition: should in [0 %i]",nbv));
				}
				vertices[j].SetCorner();
				vertices[j].SetRequired();  }
		}
		else{
			if(verbose>3) printf("      no Corners found\n");
		}

		//RequiredVertices
		if(bamggeom->RequiredVertices){
			if(verbose>3) printf("      processing RequiredVertices");
			n=bamggeom->NumRequiredVertices;
			for (i=0;i<n;i++) {     
				j=(int)bamggeom->RequiredVertices[i]-1; //for C indexing
				if (j>nbv-1 || j<0){
					throw ErrorException(__FUNCT__,exprintf("Bad RequiredVerticess  definition: should in [0 %i]",nbv));
				}
				vertices[j].SetRequired();  }
		}
		else{
			if(verbose>3) printf("      no RequiredVertices found\n");
		}

		//RequiredEdges
		if(bamggeom->RequiredEdges){
			if(verbose>3) printf("      processing RequiredEdges");
			n=bamggeom->NumRequiredEdges;
			for (i=0;i<n;i++) {     
				j=(int)bamggeom->RequiredEdges[i]-1; //for C indexing
				if (j>nbe-1 || j<0){
					throw ErrorException(__FUNCT__,exprintf("Bad RequiredEdges definition: should in [0 %i]",nbv));
				}
				edges[j].SetRequired();  }
		}
		else{
			if(verbose>3) printf("      no RequiredEdges found\n");
		}

		//SubDomain
		if(bamggeom->SubDomains){
			Int4 i0,i1,i2,i3;
			if(verbose>3) printf("      processing SubDomains\n");
			NbSubDomains=bamggeom->NumSubDomains;
			subdomains = new GeometricalSubDomain[NbSubDomains];
			for (i=0;i<NbSubDomains;i++) {
				i0=(int)bamggeom->SubDomains[i*4+0];
				i1=(int)bamggeom->SubDomains[i*4+1];
				i2=(int)bamggeom->SubDomains[i*4+2];
				i3=(int)bamggeom->SubDomains[i*4+3];
				if (i0!=2) throw ErrorException(__FUNCT__,exprintf("Bad Subdomain definition: first number should be 2 (for Edges)"));
				if (i1>nbe || i1<=0) throw ErrorException(__FUNCT__,exprintf("Bad Subdomain definition: second number should in [1 %i] (edge number)",nbe));
				subdomains[i].edge=edges + (i1-1);
				subdomains[i].sens = (int) i2;
				subdomains[i].ref = i3;
			}
		}
		else{
			if(verbose>3) printf("      no SubDomains found\n");
		}
	}
	/*}}}1*/

	/*Methods*/
	/*FUNCTION  Geometry::AfterRead(){{{1*/
	void Geometry::AfterRead(){
		long int verbosity=0;

		Int4 i,j,k;
		int jj;
		Int4* head_v=new Int4[nbv];
		Int4* next_p=new Int4[2*nbe];
		float* eangle=new float[nbe];
		double eps=1e-20;
		QuadTree quadtree; // build quadtree to find duplicates
		Vertex* v0=vertices; 
		GeometricalVertex* v0g=(GeometricalVertex*) (void*)v0;   

		k=0;
		//link all vertices to themselves by default
		for (i=0;i<nbv;i++) vertices[i].link = vertices+i;

		//build quadtree for this geometry
		for (i=0;i<nbv;i++){

			/*build integer coordinates (non unique)
			these coordinates are used by the quadtree to group
			the vertices by groups of 5:
			All the coordinates are transformed to ]0,1[^2
			then, the integer coordinates are computed using 
			the transformation ]0,1[^2 -> [0 2^(30-1)[^2 for a quadtree of depth 30*/
			vertices[i].i=toI2(vertices[i].r); 

			//find nearest vertex already present in the quadtree (NULL if empty)
			Vertex* v=quadtree.NearestVertex(vertices[i].i.x,vertices[i].i.y); 

			//if there is a vertex found that is to close to vertices[i] -> error
			if( v && Norme1(v->r - vertices[i]) < eps ){
				// mama's old trick to get j 
				GeometricalVertex* vg=(GeometricalVertex*) (void*)v;
				j=vg-v0g;
				//check that the clostest vertex is not itself...
				if ( v !=  &(Vertex &) vertices[j]){
					throw ErrorException(__FUNCT__,exprintf(" v !=  &(Vertex &) vertices[j]"));
				}
				vertices[i].link = vertices + j;
				k++;	      
			}

			//The nearest vertex was non existent or far enough from vertices[i]
			//Add vertices[i] to the quadtree
			else  quadtree.Add(vertices[i]);
		}

		//if k>0, there are some duplicate vertices -> error
		if (k) {
			printf("number of distinct vertices= %i, over %i\n",nbv - k,nbv);
			printf("List of duplicate vertices:\n");
			for (i=0;i<nbv;i++){
				if (!vertices[i].IsThe()) printf("  %i and %i\n",i,Number(vertices[i].The()));
			}
			throw ErrorException(__FUNCT__,exprintf("See above"));
		}

		//  verification of cracked edge (to be completed)
		for (i=0;i<nbe;i++){
			if (edges[i].Cracked() ) {
				//    verification of crack
				GeometricalEdge & e1=edges[i];
				GeometricalEdge & e2=*e1.link;
				if ( e1[0].The() == e2[0].The() && e1[1].The() == e2[1].The() )
				  {
					//nothing
				  }
				else 
				 if ( e1[0].The() == e2[1].The() && e1[1].The() == e2[0].The() )
					{
					 //nothing
					}
				 else {
					 throw ErrorException(__FUNCT__,exprintf("Cracked edges with no same vertex")); 
				 }
			}
		}

		/* Here we use a powerful chaining algorithm
		 *
		 * 1. What is a chaining algorithm?
		 *
		 * If F is a function that goes from i in [0 n] to j in [0 m]
		 * and we want to compute the reciprocal function F-1 of F
		 * (what are the antecedents of a given j in Im(F) )
		 * We use 2 lists:
		 *    head_F[j] that holds the head of lists
		 *    next_F[i] that holds the list of elements that have the same image
		 *
		 * Example:
		 *    i1, i2, ..., ip in [0,n] are all antecedents of a given j in [0 m]
		 *    head_F[j] = ip
		 *    next_F[ip]= ip-1
		 *    ....
		 *    next_F[i2]= i1
		 *    next_F[i1]= -1  //end of the list
		 *
		 * Algorithm:
		 *    for(j=0;j<m;j++)  head_F[j] = -1 //initialization
		 *    for(i=0;i<n;i++){
		 *       j=F[i];
		 *       next_F[i]= head_F[j];
		 *       head_F[j]=i;
		 *    }
		 * 
		 *    Then, we can go through all the elements that have for image j:
		 *    for(i=head_F[j]; i!=-1; i=next_F[i])
		 *    initialization of i by i=head_F[j]
		 *    stop the loop when i=-1 (end of the chain)
		 *    iterate using i=next_F[i] (next element that have for image j)
		 * 
		 * 2. How to use this algorithm here?
		 * 
		 * Here F is a function that associates two vertices v0 and v1 for a given edge E
		 * We want to build the reciprocal function: what are the edges that contains
		 * a vertex v?
		 * To do so, we use the same chaining algorithm but there is a difficulty
		 * coming from the fact that F we have a couple of vertices and not one 
		 * vertices.
		 * To overcome this difficulty, we use a global indices exactly like in 
		 * C/C++ so that
		 * a member of a 2-column-table can be described by one index p=i*2+j
		 * i=(int)p/2 line number of p
		 * j=p%2       column number of p
		 *
		 * Algorithm:
		 *    for(i=0;i<nbv;i++)  head_v[i] = -1 //initialization
		 *    for(i=0;i<nbe;i++){
		 *       for(j=0;j<2;j++){
		 *          p=2*i+j;
		 *          v=edges(i,j);
		 *          next_p[p]= head_v[v];
		 *          head_v[v]=p;
		 *       }
		 *    }
		 */

		//initialize head_v as -1
		for (i=0;i<nbv;i++) head_v[i]=-1;
		k=0;
		for (i=0;i<nbe;i++) {
			//compute vector of edge i that goes from vertex 0 to vertex 1
			R2 v10=edges[i].v[1]->r - edges[i].v[0]->r;
			Real8 lv10=Norme2(v10);
			//check that its length is not 0
			if(lv10==0) {
				throw ErrorException(__FUNCT__,exprintf("Length of edge %i is 0",i));
			}
			//compute angle in [-Pi Pi]
			eangle[i] = atan2(v10.y,v10.x);
			//build chains head_v and next_p
			for (j=0;j<2;j++){
				Int4 v=Number(edges[i].v[j]);
				next_p[k]=head_v[v];
				head_v[v]=k++; //post increment: head_v[v]=k; and then k=k+1;
			}
		}

		//sort head_v by order of increasing edges angle
		for (i=0;i<nbv;i++) {
			int exch=1, ord=0;      

			//exchange vertices position in head_v and next_p till tey are sorted
			while (exch){
				Int4 *p=head_v+i;               // pointer toward head_v[vertex i]
				Int4 *po=p;                     // copy of pointer p
				Int4  n=*p;                     // next value of edge holding i
				register float angleold=-1000 ; // angle = - infinity
				ord=0; exch=0;

				// loop over the edges that contain the vertex i
				while (n >=0){
					ord++;
					register Int4  i1=n/2;       // i1 = floor (n/2)
					register Int4  j1=n%2;       // j1 = 1 if n is odd
					register Int4* pn=next_p+n;  // pointer to next_p[n]

					//  n = next_p[n] = position in edge of next vertex i
					n=*pn;                       

					//compute angle between horizontal axis and v0->v1
					float angle = j1 ? OppositeAngle(eangle[i1]):  eangle[i1]; 

					//exchange if the current edge angle is smaller than the previous one
					if (angleold > angle){
						exch=1;
						*pn=*po;  // next_p[n] = n + 1
						*po=*p;   // 
						*p=n;     // next_p[n+1] = n
						po=pn;    // po now point toward pn (invert next and current)
					}

					//else, continue going to the next edge position
					else{                        //  to have : po -> p -> pn
						angleold=angle; // update maximum angle
						po=p;           // po now point toward p  (current position)
						p=pn;           // p  now point toward pn (next position)
					}
				}
			}
			printf("ord = %i\n",ord);

			// angular test on current vertex to guess whether it is a corner (ord = number of edges horlding i)
			if(ord == 2) { 
				Int4  n1 = head_v[i];
				Int4  n2 = next_p[n1];
				Int4  i1 = n1/2, i2 = n2/2; // edge number
				Int4  j1 = n1%2, j2 = n2%2; // vertex in the edge 
				float angle1=  j1 ? OppositeAngle(eangle[i1]) : eangle[i1];
				float angle2= !j2 ? OppositeAngle(eangle[i2]) : eangle[i2];
				float da12 = Abs(angle2-angle1);
				if (( da12 >= MaximalAngleOfCorner ) && (da12 <= 2*Pi -MaximalAngleOfCorner)) {
					vertices[i].SetCorner() ; 
				}
				// if the ref a changing then is     SetRequired();
				if (edges[i1].flag != edges[i2].flag || edges[i1].Required()){
					vertices[i].SetRequired();
				}
				if (edges[i1].ref != edges[i2].ref) {
					vertices[i].SetRequired();
				}
			}
			if(ord != 2) {
				vertices[i].SetCorner();
			}

			// close the liste around the vertex 
			Int4 no=-1, ne = head_v[i];
			while (ne >=0) ne = next_p[no=ne];        
			if(no>=0) next_p[no] = head_v[i];
			// now the list around the vertex is circular
		}

		k =0;
		for (i=0;i<nbe;i++){
			for (j=0;j<2;j++){
				Int4 n1 = next_p[k++]; 
				Int4 i1 = n1/2 ,j1=n1%2;
				if( edges[i1].v[j1] != edges[i].v[j]) {
					throw ErrorException(__FUNCT__,exprintf("Bug Adj edge"));
				}
				edges[i1].Adj[j1] = edges + i;
				edges[i1].DirAdj[j1] = j;
			}
		}

		// generation of  all the tangente 
		for (i=0;i<nbe;i++) {
			R2 AB = edges[i].v[1]->r -edges[i].v[0]->r;        
			Real8 lAB = Norme2(AB); // length of current edge AB
			Real8 ltg2[2];
			ltg2[0]=0;ltg2[1]=0;
			for (jj=0;jj<2;jj++) {
				R2 tg =  edges[i].tg[jj];
				Real8 ltg = Norme2(tg); // length of tg
				if(ltg == 0) {// no tg
					if( ! edges[i].v[jj]->Corner())   { // not a Corner       
						tg =  edges[i].v[1-jj]->r 
						  - edges[i].Adj[jj]->v[1-edges[i].DirAdj[jj]]->r;
						ltg =  Norme2(tg);
						tg =  tg *(lAB/ltg),ltg=lAB;
					}
					//else:  a Corner with no tangent => nothing to do    
				}
				else{
					tg = tg *(lAB/ltg),ltg=lAB;
				}
				ltg2[jj] = ltg;
				if ((tg,AB)<0) tg = -tg;
				edges[i].tg[jj] = tg;
			}
			if (ltg2[0]!=0) edges[i].SetTgA();
			if (ltg2[1]!=0) edges[i].SetTgB();
		} 

		for (int step=0;step<2;step++){
			for (i=0;i<nbe;i++) edges[i].SetUnMark();
			NbOfCurves = 0;
			Int4  nbgem=0;
			for (int level=0;level < 2 && nbgem != nbe;level++)
			 for (i=0;i<nbe;i++) {
				 GeometricalEdge & ei = edges[i];   
				 for(jj=0;jj<2;jj++){
					 if (!ei.Mark() && (level || ei[jj].Required())) { 
						 // warning ei.Mark() can be change in loop for(jj=0;jj<2;jj++) 
						 int k0=jj,k1;
						 GeometricalEdge *e = & ei;
						 GeometricalVertex *a=(*e)(k0); // begin 
						 if(curves) {
							 curves[NbOfCurves].be=e;
							 curves[NbOfCurves].kb=k0;
						 }
						 int nee=0;
						 for(;;) { 
							 nee++;
							 k1 = 1-k0; // next vertex of the edge 
							 e->SetMark();
							 nbgem++;
							 e->CurveNumber=NbOfCurves;
							 if(curves) {
								 curves[NbOfCurves].ee=e;
								 curves[NbOfCurves].ke=k1;
							 }
							 GeometricalVertex *b=(*e)(k1);
							 if (a == b ||  b->Required() ) break;
							 k0 = e->DirAdj[k1];//  vertex in next edge
							 e = e->Adj[k1]; // next edge
						 }
						 NbOfCurves++;
						 if(level) a->SetRequired();
					 }
				 }
			 } 
			if (nbgem==0 || nbe==0){
				throw ErrorException(__FUNCT__,exprintf("nbgem==0 || nbe==0"));
			}
			if(step==0) {
				curves = new Curve[NbOfCurves];
			}
		} 
		for(int i=0;i<NbOfCurves ;i++){
			GeometricalEdge * be=curves[i].be, *eqbe=be->link;
			//GeometricalEdge * ee=curves[i].ee, *eqee=be->link;
			curves[i].master=true;
			if(be->Equi() || be->ReverseEqui() ){
				if (!eqbe){
					throw ErrorException(__FUNCT__,exprintf("!eqbe"));
				}
				int nc = eqbe->CurveNumber;
				if (i==nc){
					throw ErrorException(__FUNCT__,exprintf("i==nc"));
				}
				curves[i].next=curves[nc].next;
				curves[i].master=false;
				curves[nc].next=curves+i;
				if(be->ReverseEqui())
				 curves[i].Reverse();           
			}
		}

		/*clean up*/
		delete []next_p;
		delete []head_v;
		delete []eangle;

	}
	/*}}}1*/
	/*FUNCTION  Geometry::Contening{{{1*/
	GeometricalEdge* Geometry::Contening(const R2 P,  GeometricalEdge * start) const {
		GeometricalEdge* on =start,* pon=0;
		// walk with the cos on geometry
		int k=0;
		while(pon != on){  
			k++;
			pon = on;
			if (k>=100){
				throw ErrorException(__FUNCT__,exprintf("k>=100"));
			}
			R2 A= (*on)[0];
			R2 B= (*on)[1];
			R2 AB = B-A;
			R2 AP = P-A;
			R2 BP = P-B;
			if ( (AB,AP) < 0) 
			 on = on->Adj[0];
			else if ( (AB,BP)  > 0) 
			 on = on->Adj[1];
			else
			 return on;
		}
		return on;
	}
	/*}}}1*/
	/*FUNCTION  Geometry::Echo {{{1*/

	void Geometry::Echo(void){

		printf("Geometry:\n");
		printf("   name: %s\n",name);
		printf("   nbvx (maximum number of vertices) : %i\n",nbvx);
		printf("   nbtx (maximum number of triangles): %i\n",nbtx);
		printf("   nbv  (number of vertices) : %i\n",nbv);
		printf("   nbt  (number of triangles): %i\n",nbt);
		printf("   nbe  (number of edges)    : %i\n",nbe);
		printf("   nbv  (number of initial vertices) : %i\n",nbiv);
		printf("   NbSubDomains: %i\n",NbSubDomains);
		printf("   NbEquiEdges: %i\n",NbEquiEdges);
		printf("   NbCrackedEdges: %i\n",NbCrackedEdges);
		printf("   NbOfCurves: %i\n",NbOfCurves);
		printf("   vertices: %p\n",vertices);
		printf("   triangles: %p\n",triangles);
		printf("   edges: %p\n",edges);
		printf("   quadtree: %p\n",quadtree);
		printf("   subdomains: %p\n",subdomains);
		printf("   curves: %p\n",curves);
		printf("   pmin (x,y): (%g %g)\n",pmin.x,pmin.y);
		printf("   pmax (x,y): (%g %g)\n",pmax.x,pmax.y);
		printf("   coefIcoor: %g\n",coefIcoor);
		printf("   MaximalAngleOfCorner: %g\n",MaximalAngleOfCorner);

		return;
	}
	/*}}}*/
	/*FUNCTION  Geometry::EmptyGeometry(){{{1*/
	void Geometry::EmptyGeometry() {
		NbRef=0;
		name =0;
		quadtree=0;
		curves=0;
		// edgescomponante=0;
		triangles=0;
		edges=0;
		vertices=0;
		NbSubDomains=0;
		//  nbtf=0;
		//  BeginOfCurve=0;  
		nbiv=nbv=nbvx=0;
		nbe=nbt=nbtx=0;
		NbOfCurves=0;
		//  BeginOfCurve=0;
		subdomains=0;
		MaximalAngleOfCorner = 10*Pi/180;
	}
	/*}}}1*/
	/*FUNCTION  Geometry::Geometry::ProjectOnCurve {{{1*/
	GeometricalEdge* Geometry::ProjectOnCurve(const Edge & e,Real8 s,Vertex &V,VertexOnGeom &GV ) const {
		Real8 save_s=s;
		int NbTry=0;
retry:    
		s=save_s;
		GeometricalEdge* on = e.on;
		if (!on){
			throw ErrorException(__FUNCT__,exprintf("!on"));
		}
		if (!e[0].on ||  !e[1].on){
			throw ErrorException(__FUNCT__,exprintf("!e[0].on ||  !e[1].on"));
		}
		const Vertex &v0=e[0],&v1=e[1];
		V.m = Metric(1.0-s, v0,s, v1);
		const int mxe =100;
		GeometricalEdge *ge[mxe+1];
		int    sensge[mxe+1];
		Real8  lge[mxe+1];
		int bge=mxe/2,tge=bge;
		ge[bge] = e.on;
		sensge[bge]=1;

		R2 V0 = v0,V1=v1,V01=V1-V0;
		VertexOnGeom  vg0= *v0.on,  vg1=*v1.on;

		//    GeometricalEdge * eg0 = e.on,* eg1 = e.on, *eg=NULL;
		GeometricalEdge * eg0=on, *eg1=on;
		R2 Ag=(R2) (*on)[0],Bg=(R2)(*on)[1],AB=Bg-Ag; 
		int OppositeSens = (V01,AB) < 0;
		int sens0=0,sens1=1;
		if (OppositeSens)
		 s=1-s,Exchange(vg0,vg1),Exchange(V0,V1);
		while (eg0 != (GeometricalEdge*) vg0  &&  (*eg0)(sens0) != (GeometricalVertex*) vg0){ 
			if (bge<=0) {
				//          int kkk;
				if(NbTry) {
					printf("Fatal Error: on the class triangles before call Geometry::ProjectOnCurve\n");
					printf("That bug might come from:\n");
					printf(" 1)  a mesh edge  contening more than %i geometrical edges\n",mxe/2);
					printf(" 2)  code bug : be sure that we call   Triangles::SetVertexFieldOn() before\n");
					printf("To solve the problem do a coarsening of the geometrical mesh or change the constant value of mxe (dangerous)\n");
					throw ErrorException(__FUNCT__,exprintf("see above"));
				  }
				NbTry++;
				goto retry;}
				GeometricalEdge* tmpge = eg0;
				ge[--bge] =eg0 = eg0->Adj[sens0];
				if (bge<0 || bge>mxe){
					throw ErrorException(__FUNCT__,exprintf("bge<0 || bge>mxe"));
				}
				sens0 = 1-( sensge[bge] = tmpge->DirAdj[sens0]);
		  }
		while (eg1 != (GeometricalEdge*) vg1  &&  (*eg1)(sens1) != (GeometricalVertex*) vg1) { 
			if(tge>=mxe ) { 
				printf("WARNING: on the class triangles before call Geometry::ProjectOnCurve is having issues (isn't it Eric?)\n");
				NbTry++;
				if (NbTry<2) goto retry;
				printf("Fatal Error: on the class triangles before call Geometry::ProjectOnCurve\n");
				printf("That bug might come from:\n");
				printf(" 1)  a mesh edge  contening more than %i geometrical edges\n",mxe/2);
				printf(" 2)  code bug : be sure that we call   Triangles::SetVertexFieldOn() before\n");
				printf("To solve the problem do a coarsening of the geometrical mesh or change the constant value of mxe (dangerous)\n");
				throw ErrorException(__FUNCT__,exprintf("see above"));
			}

			GeometricalEdge* tmpge = eg1;
			ge[++tge] =eg1 = eg1->Adj[sens1];
			sensge[tge]= sens1 = 1-tmpge->DirAdj[sens1];
			if (tge<0 || tge>mxe){
				throw ErrorException(__FUNCT__,exprintf("(tge<0 || tge>mxe)"));
			}
		  }


		if ( (*eg0)(sens0) == (GeometricalVertex*) vg0 )
		 vg0 = VertexOnGeom( *(Vertex *) vg0,*eg0,sens0);

		if ( (*eg1)(sens1) == (GeometricalVertex*) vg1)
		 vg1 = VertexOnGeom( *(Vertex *) vg1,*eg1,sens1);

		Real8 sg;
		if (eg0 == eg1) { 
			register Real8 s0= vg0,s1=vg1;
			sg =  s0 * (1.0-s) +  s * s1;
			on=eg0;}
		else {
			R2 AA=V0,BB;
			Real8 s0,s1;
			int i=bge;
			Real8 ll=0;
			for(i=bge;i<tge;i++){
				if ( i<0 || i>mxe){
					throw ErrorException(__FUNCT__,exprintf("i<0 || i>mxe"));
				}
				BB =  (*ge[i])[sensge[i]];
				lge[i]=ll += Norme2(AA-BB);
				AA=BB ;}
				lge[tge]=ll+=Norme2(AA-V1); 
				// search the geometrical edge
				if (s>1.0){
					throw ErrorException(__FUNCT__,exprintf("s>1.0"));
				}
				Real8 ls= s*ll;
				on =0;
				s0 = vg0;
				s1= sensge[bge];
				Real8 l0=0,l1;
				i=bge;
				while (  (l1=lge[i]) < ls ) {
					if (i<0 || i>mxe){
						throw ErrorException(__FUNCT__,exprintf("i<0 || i>mxe"));
					}
					i++,s0=1-(s1=sensge[i]),l0=l1;}
					on=ge[i];
					if (i==tge) 
					 s1=vg1;

					s=(ls-l0)/(l1-l0);
					sg =  s0 * (1.0-s) +  s * s1;    
		} 
		if (!on){
			throw ErrorException(__FUNCT__,exprintf("!on"));
		}
		V.r= on->F(sg);
		GV=VertexOnGeom(V,*on,sg);
		return on;
	}
	/*}}}1*/

} 
