#include <cstdio>
#include <cstring>
#include <cmath>
#include <ctime>

#include "../Mesh2.h"
#include "../QuadTree.h"
#include "../SetOfE4.h"

#include "../../shared/shared.h"
#include "../../include/macros.h"
#include "../../toolkits/toolkits.h"

namespace bamg {

	static const  Direction NoDirOfSearch=Direction();

	/*Constructors/Destructors*/
	/*FUNCTION  Geometry::Geometry(const Geometry & Gh){{{1*/
	Geometry::Geometry(const Geometry & Gh) {
		Int4 i;
		*this = Gh;
		NbRef =0;
		quadtree=0;
		name = new char[strlen(Gh.name)+4];
		strcpy(name,"cp:");
		strcat(name,Gh.name);
		vertices = nbv ? new GeometricalVertex[nbv] : NULL;
		triangles = nbt ? new  Triangle[nbt]:NULL;
		edges = nbe ? new GeometricalEdge[nbe]:NULL;
		curves= NbOfCurves ? new Curve[NbOfCurves]:NULL;
		subdomains = NbSubDomains ? new GeometricalSubDomain[NbSubDomains]:NULL;
		for (i=0;i<nbv;i++)
		 vertices[i].Set(Gh.vertices[i],Gh,*this);
		for (i=0;i<nbe;i++)
		 edges[i].Set(Gh.edges[i],Gh,*this);
		for (i=0;i<NbOfCurves;i++)
		 curves[i].Set(Gh.curves[i],Gh,*this);
		for (i=0;i<NbSubDomains;i++)
		 subdomains[i].Set(Gh.subdomains[i],Gh,*this);

		//    for (i=0;i<nbt;i++)
		//      triangles[i].Set(Gh.triangles[i],Gh,*this);
		assert(!nbt);   
	}
	/*}}}1*/
	/*FUNCTION Geometry::~Geometry(){{{1*/
	Geometry::~Geometry() {
		long int verbosity=0;

		assert(NbRef<=0);
		if(verbosity>9)
		 cout << "DELETE      ~Geometry "<< this  << endl;
		if(vertices)  delete [] vertices;vertices=0;
		if(edges)     delete [] edges;edges=0;
		// if(edgescomponante) delete [] edgescomponante; edgescomponante=0;
		if(triangles) delete [] triangles;triangles=0;
		if(quadtree)  delete  quadtree;quadtree=0;
		if(curves)  delete  []curves;curves=0;NbOfCurves=0;
		if(name) delete [] name;name=0;
		if(subdomains) delete [] subdomains;subdomains=0;
		//  if(ordre)     delete [] ordre;
		EmptyGeometry();
	}
	/*}}}1*/

	/*IO*/
	/*FUNCTION Geometry::WriteGeometry{{{1*/
	void Geometry::WriteGeometry(BamgGeom* bamggeom, BamgOpts* bamgopts){

		int verbose;
		int nbreq=0;
		int nbreqv=0;
		int nbtan=0;
		int nbcracked=0;
		int i,count;

		verbose=bamgopts->verbose;

		//Vertices
		if(verbose>3) printf("      writing Vertices\n");
		bamggeom->NumVertices=nbv;
		xfree((void**)&bamggeom->Vertices);
		if (nbv){
			bamggeom->Vertices=(double*)xmalloc(3*nbv*sizeof(double));
			for (i=0;i<nbv;i++){
				bamggeom->Vertices[i*3+0]=vertices[i].r.x;
				bamggeom->Vertices[i*3+1]=vertices[i].r.y;
				bamggeom->Vertices[i*3+2]=vertices[i].ref();

				//update counters
				if (vertices[i].Required()) nbreqv++;
			}
		}
		else{
			bamggeom->Vertices=NULL;
		}

		//Edges
		if(verbose>3) printf("      writing Edges\n");
		bamggeom->NumEdges=nbe;
		xfree((void**)&bamggeom->Edges);
		if (nbe){
			bamggeom->Edges=(double*)xmalloc(3*nbe*sizeof(double));
			for (i=0;i<nbe;i++){
				bamggeom->Edges[i*3+0]=Number(edges[i][0])+1; //back to Matlab indexing
				bamggeom->Edges[i*3+1]=Number(edges[i][1])+1; //back to Matlab indexing
				//bamggeom->Edges[i*3+2]=(double)edges[i].ref(); //TEST does not compile???
				bamggeom->Edges[i*3+2]=1; //TEST for now

				//update counters
				if (edges[i].Required()) nbreq++;
				if (edges[i].Cracked()){
					if (i<=Number(edges[i].link)) nbcracked++;
				}
				if (edges[i].TgA() && edges[i][0].Corner()) nbtan++;
				if (edges[i].TgB() && edges[i][1].Corner()) nbtan++;
			}
		}
		else{
			bamggeom->Edges=NULL;
		}

		//CrackedEdges
		if(verbose>3) printf("      writing CrackedEdges\n");
		bamggeom->NumCrackedEdges=nbcracked;
		xfree((void**)&bamggeom->CrackedEdges);
		if (nbcracked){
			bamggeom->CrackedEdges=(double*)xmalloc(2*nbcracked*sizeof(double));
			count=0;
			for (i=0;i<nbe;i++){
				if (edges[i].Cracked()){
					if (i<=Number(edges[i].link)){
						bamggeom->CrackedEdges[count*2+0]=i+1;                     //back to Matlab indexing
						bamggeom->CrackedEdges[count*2+1]=Number(edges[i].link)+1; //back to Matlab indexing
						count=count+1;
					}
				}
			}
		}
		else{
			bamggeom->CrackedEdges=NULL;
		}

		//RequiredEdges
		if(verbose>3) printf("      writing RequiredEdges\n");
		bamggeom->NumRequiredEdges=nbreq;
		xfree((void**)&bamggeom->RequiredEdges);
		if (nbreq){
			bamggeom->RequiredEdges=(double*)xmalloc(1*nbreq*sizeof(double));
			count=0;
			for (i=0;i<nbe;i++){
				if (edges[i].Required()){
					bamggeom->RequiredEdges[count]=i+1; //back to Matlab indexing
					count=count+1;
				}
			}
		}
		else{
			bamggeom->RequiredEdges=NULL;
		}

		//No corners

		//RequiredVertices
		if(verbose>3) printf("      writing RequiredVertices\n");
		bamggeom->NumRequiredVertices=nbreqv;
		xfree((void**)&bamggeom->RequiredVertices);
		if (nbreqv){
			bamggeom->RequiredVertices=(double*)xmalloc(1*nbreqv*sizeof(double));
			count=0;
			for (i=0;i<nbe;i++){
				if (vertices[i].Required()){
					bamggeom->RequiredVertices[count]=i+1; //back to Matlab indexing
					count=count+1;
				}
			}
		}
		else{
			bamggeom->RequiredVertices=NULL;
		}

		//SubDomains
		if(verbose>3) printf("      writing SubDomains\n");
		bamggeom->NumSubDomains=NbSubDomains;
		xfree((void**)&bamggeom->SubDomains);
		if (NbSubDomains){
			bamggeom->SubDomains=(double*)xmalloc(4*NbSubDomains*sizeof(double));
			for (i=0;i<NbSubDomains;i++){
				bamggeom->SubDomains[4*i+0]=2;
				bamggeom->SubDomains[4*i+1]=Number(subdomains[i].edge)+1; //back to Matlab indexing
				bamggeom->SubDomains[4*i+2]=subdomains[i].sens;
				bamggeom->SubDomains[4*i+3]=subdomains[i].ref;
			}
		}
		else{
			bamggeom->SubDomains=NULL;
		}

		//TangentAtEdges
		if(verbose>3) printf("      writing TangentAtEdges\n");
		bamggeom->NumTangentAtEdges=nbtan;
		xfree((void**)&bamggeom->TangentAtEdges);
		if (nbtan){
			bamggeom->TangentAtEdges=(double*)xmalloc(4*nbtan*sizeof(double));
			count=0;
			for (i=0;i<nbe;i++){
				if (edges[i].TgA() && edges[i][0].Corner()){
					bamggeom->TangentAtEdges[4*i+0]=i+1; //back to Matlab indexing
					bamggeom->TangentAtEdges[4*i+1]=1;
					bamggeom->TangentAtEdges[4*i+2]=edges[i].tg[0].x;
					bamggeom->TangentAtEdges[4*i+3]=edges[i].tg[0].y;
				}
				if (edges[i].TgB() && edges[i][1].Corner()){
					bamggeom->TangentAtEdges[4*i+0]=i+1; //back to Matlab indexing
					bamggeom->TangentAtEdges[4*i+1]=2;
					bamggeom->TangentAtEdges[4*i+2]=edges[i].tg[1].x;
					bamggeom->TangentAtEdges[4*i+3]=edges[i].tg[2].y;
				}
				count=count+1;
			}
		}
		else{
			bamggeom->TangentAtEdges=NULL;
		}
	}
	/*}}}1*/
	/*FUNCTION Geometry::ReadGeometry{{{1*/
	void Geometry::ReadGeometry(BamgGeom* bamggeom,BamgOpts* bamgopts){

		int verbose;
		nbiv=nbv=nbvx=0;
		nbe=nbt=nbtx=0;
		NbOfCurves=0;

		Real8 Hmin = HUGE_VAL;// the infinie value 
		Int4 hvertices=0;
		int i,j,k,n;

		//initialize some variables
		int Version=1,dim=2;
		nbv=bamggeom->NumVertices;
		nbe=bamggeom->NumEdges;
		nbvx = nbv;
		nbiv = nbv;
		verbose=bamgopts->verbose;

		//some checks
		if (bamggeom->NumVertices<=0 || bamggeom->Vertices==NULL){
			throw ErrorException(__FUNCT__,exprintf("the domain provided does not contain any vertex"));
		}
		if (bamggeom->NumEdges<=0 || bamggeom->Edges==NULL){
			throw ErrorException(__FUNCT__,exprintf("the domain provided does not contain any edge"));
		}

		//Vertices
		if (bamggeom->Vertices){
			if(verbose>3) printf("      processing Vertices\n");
			vertices = new GeometricalVertex[nbvx];
			for (i=0;i<nbv;i++) {
				vertices[i].r.x=(double)bamggeom->Vertices[i*3+0];
				vertices[i].r.y=(double)bamggeom->Vertices[i*3+1];
				vertices[i].ReferenceNumber=(Int4)bamggeom->Vertices[i*3+2];
				vertices[i].DirOfSearch=NoDirOfSearch;
				vertices[i].color =0;
				vertices[i].Set();
			}
			//find domain extrema for pmin,pmax
			pmin =  vertices[0].r;
			pmax =  vertices[0].r;
			for (i=0;i<nbv;i++) {
				pmin.x = Min(pmin.x,vertices[i].r.x);
				pmin.y = Min(pmin.y,vertices[i].r.y);
				pmax.x = Max(pmax.x,vertices[i].r.x);
				pmax.y = Max(pmax.y,vertices[i].r.y);
			}
			R2 DD05 = (pmax-pmin)*0.05;
			pmin -=  DD05;
			pmax +=  DD05;
			coefIcoor= (MaxICoor)/(Max(pmax.x-pmin.x,pmax.y-pmin.y));
			if(coefIcoor <=0){
				throw ErrorException(__FUNCT__,exprintf("coefIcoor should be positive"));
			}
		}
		else{
			throw ErrorException(__FUNCT__,exprintf("No Vertex provided"));
		}

		//Edges
		if (bamggeom->Edges){
			int i1,i2;
			R2 zero2(0,0);
			Real4 *len =0;

			if(verbose>3) printf("      processing Edges\n");
			edges = new GeometricalEdge[nbe];

			if (!hvertices) {
				len = new Real4[nbv];
				for(i=0;i<nbv;i++)
				 len[i]=0;
			}

			for (i=0;i<nbe;i++){
				i1=(int)bamggeom->Edges[i*3+0]-1; //-1 for C indexing
				i2=(int)bamggeom->Edges[i*3+1]-1; //-1 for C indexing
				edges[i].ref=(Int4)bamggeom->Edges[i*3+2];
				edges[i].v[0]=  vertices + i1;
				edges[i].v[1]=  vertices + i2;
				R2 x12 = vertices[i2].r-vertices[i1].r;
				Real8 l12=sqrt((x12,x12));
				edges[i].tg[0]=zero2;
				edges[i].tg[1]=zero2;
				edges[i].SensAdj[0] = edges[i].SensAdj[1] = -1;
				edges[i].Adj[0] = edges[i].Adj[1] = 0;
				edges[i].flag = 0;
				if (!hvertices) {
					vertices[i1].color++;
					vertices[i2].color++;
					len[i1] += l12;
					len[i2] += l12;
				}

				Hmin = Min(Hmin,l12);
			}

			// definition  the default of the given mesh size 
			if (!hvertices){
				for (i=0;i<nbv;i++) 
				 if (vertices[i].color > 0) 
				  vertices[i].m=  Metric(len[i] /(Real4) vertices[i].color);
				 else 
				  vertices[i].m=  Metric(Hmin);
				delete [] len;
			}
		}
		else{
			throw ErrorException(__FUNCT__,exprintf("No edges provided"));
		}

		//hVertices
		if(bamggeom->hVertices){
			if(verbose>3) printf("      processing hVertices\n");
			for (i=0;i< nbv;i++){
				vertices[i].m=Metric((Real4)bamggeom->hVertices[i]);
			}
		}
		else{
			if(verbose>3) printf("      no hVertices found\n");
		}

		//MetricVertices
		if(bamggeom->MetricVertices){
			if(verbose>3) printf("      processing MetricVertices\n");
			hvertices=1;
			for (i=0;i< nbv;i++) {
				vertices[i].m = Metric((Real4)bamggeom->MetricVertices[i*3+0],(Real4)bamggeom->MetricVertices[i*3+1],(Real4)bamggeom->MetricVertices[i*3+2]);
			}
		}
		else{
			if(verbose>3) printf("      no MetricVertices found\n");
		}

		//h1h2VpVertices
		if(bamggeom->h1h2VpVertices){
			if(verbose>3) printf("      processing h1h2VpVertices\n");
			Real4 h1,h2,v1,v2;
			hvertices =1;
			for (i=0;i< nbv;i++) {
				h1=(Real4)bamggeom->MetricVertices[i*4+0];
				h2=(Real4)bamggeom->MetricVertices[i*4+1];
				v1=(Real4)bamggeom->MetricVertices[i*4+2];
				v2=(Real4)bamggeom->MetricVertices[i*4+3];
				vertices[i].m = Metric(MatVVP2x2(1/(h1*h1),1/(h2*h2),D2(v1,v2)));
			}
		}
		else{
			if(verbose>3) printf("      no h1h2VpVertices found\n");
		}

		//MaximalAngleOfCorner
		if (bamgopts->MaximalAngleOfCorner){
			if(verbose>3) printf("      processing MaximalAngleOfCorner\n");
			MaximalAngleOfCorner=bamgopts->MaximalAngleOfCorner*Pi/180;
		}
		else{
			if(verbose>3) printf("      no MaximalAngleOfCorner found\n");
		}

		//TangentAtEdges
		if (bamggeom->TangentAtEdges){
			if(verbose>3) printf("      processing TangentAtEdges");
			int n,i,j,k;
			R2 tg;

			n=bamggeom->NumTangentAtEdges;
			for (k=0;k<n;k++) {
				i=(int)bamggeom->TangentAtEdges[k*4+0]-1; //for C indexing
				j=(int)bamggeom->TangentAtEdges[k*4+1]-1; //for C indexing
				tg.x=bamggeom->TangentAtEdges[k*4+2];
				tg.y=bamggeom->TangentAtEdges[k*4+3];
				if (j!=0 && j!=1){
					throw ErrorException(__FUNCT__,exprintf("TangentAtEdges second index should be 1 or 2 only"));
				}
				edges[i].tg[j] = tg;
			}
		}
		else{
			if(verbose>3) printf("      no TangentAtEdges found\n");
		}

		//Corners
		if(bamggeom->Corners){
			if(verbose>3) printf("      processing Corners");
			n=bamggeom->NumCorners;
			for (i=0;i<n;i++) {     
				j=(int)bamggeom->Corners[i]-1; //for C indexing
				if (j>nbv-1 || j<0){
					throw ErrorException(__FUNCT__,exprintf("Bad corner definition: should in [0 %i]",nbv));
				}
				vertices[j].SetCorner();
				vertices[j].SetRequired();  }
		}
		else{
			if(verbose>3) printf("      no Corners found\n");
		}

		//RequiredVertices
		if(bamggeom->RequiredVertices){
			if(verbose>3) printf("      processing RequiredVertices");
			n=bamggeom->NumRequiredVertices;
			for (i=0;i<n;i++) {     
				j=(int)bamggeom->RequiredVertices[i]-1; //for C indexing
				if (j>nbv-1 || j<0){
					throw ErrorException(__FUNCT__,exprintf("Bad RequiredVerticess  definition: should in [0 %i]",nbv));
				}
				vertices[j].SetRequired();  }
		}
		else{
			if(verbose>3) printf("      no RequiredVertices found\n");
		}

		//RequiredEdges
		if(bamggeom->RequiredEdges){
			if(verbose>3) printf("      processing RequiredEdges");
			n=bamggeom->NumRequiredEdges;
			for (i=0;i<n;i++) {     
				j=(int)bamggeom->RequiredEdges[i]-1; //for C indexing
				if (j>nbe-1 || j<0){
					throw ErrorException(__FUNCT__,exprintf("Bad RequiredEdges definition: should in [0 %i]",nbv));
				}
				edges[j].SetRequired();  }
		}
		else{
			if(verbose>3) printf("      no RequiredEdges found\n");
		}

		//SubDomain
		if(bamggeom->SubDomains){
			Int4 i0,i1,i2,i3;
			if(verbose>3) printf("      processing SubDomains\n");
			NbSubDomains=bamggeom->NumSubDomains;
			subdomains = new GeometricalSubDomain[NbSubDomains];
			for (i=0;i<NbSubDomains;i++) {
				i0=(int)bamggeom->SubDomains[i*4+0];
				i1=(int)bamggeom->SubDomains[i*4+1];
				i2=(int)bamggeom->SubDomains[i*4+2];
				i3=(int)bamggeom->SubDomains[i*4+3];
				if (i0!=2) throw ErrorException(__FUNCT__,exprintf("Bad Subdomain definition: first number should be 2 (for Edges)"));
				if (i1>nbe || i1<=0) throw ErrorException(__FUNCT__,exprintf("Bad Subdomain definition: second number should in [1 %i] (edge number)",nbe));
				subdomains[i].edge=edges + (i1-1);
				subdomains[i].sens = (int) i2;
				subdomains[i].ref = i3;
			}
		}
		else{
			if(verbose>3) printf("      no SubDomains found\n");
		}
	}
	/*}}}1*/

	/*Methods*/
	/*FUNCTION  Geometry::EmptyGeometry(){{{1*/
	void Geometry::EmptyGeometry() {
		OnDisk=0;
		NbRef=0;
		name =0;
		quadtree=0;
		curves=0;
		// edgescomponante=0;
		triangles=0;
		edges=0;
		vertices=0;
		NbSubDomains=0;
		//  nbtf=0;
		//  BeginOfCurve=0;  
		nbiv=nbv=nbvx=0;
		nbe=nbt=nbtx=0;
		NbOfCurves=0;
		//  BeginOfCurve=0;
		subdomains=0;
		MaximalAngleOfCorner = 10*Pi/180;
	}
	/*}}}1*/
	/*FUNCTION  Geometry::Contening{{{1*/
	GeometricalEdge* Geometry::Contening(const R2 P,  GeometricalEdge * start) const {
		GeometricalEdge* on =start,* pon=0;
		// walk with the cos on geometry
		//  cout << P ;
		int k=0;
		while(pon != on)
		  {  
			pon = on;
			assert(k++<100);
			R2 A= (*on)[0];
			R2 B= (*on)[1];
			R2 AB = B-A;
			R2 AP = P-A;
			R2 BP = P-B;
			//   cout << "::  " << on - edges << " "  <<  AB*AP  << " " <<  AB*BP << " " << A << B << endl;
			if ( (AB,AP) < 0) 
			 on = on->Adj[0];
			else if ( (AB,BP)  > 0) 
			 on = on->Adj[1];
			else
			 return on;
		  }
		return on;
	}
	/*}}}1*/
	/*FUNCTION  Geometry::Geometry::ProjectOnCurve {{{1*/
	GeometricalEdge* Geometry::ProjectOnCurve(const Edge & e,Real8 s,Vertex &V,VertexOnGeom &GV ) const {
		Real8 save_s=s;
		int NbTry=0;
retry:    
		s=save_s;
		GeometricalEdge * on = e.on;
		assert(on);
		assert( e[0].on &&  e[1].on);
		const Vertex &v0=e[0],&v1=e[1];
		V.m = Metric(1.0-s, v0,s, v1);
#define MXE__LINE  __LINE__+1
		const int mxe =100;
		GeometricalEdge *ge[mxe+1];
		int    sensge[mxe+1];
		Real8  lge[mxe+1];
		int bge=mxe/2,tge=bge;
		ge[bge] = e.on;
		sensge[bge]=1;

		R2 V0 = v0,V1=v1,V01=V1-V0;
		VertexOnGeom  vg0= *v0.on,  vg1=*v1.on;
		if(NbTry) cout << "bug: s==== " << s << " e=" <<  V0 << " " << V1 << endl;

		//    GeometricalEdge * eg0 = e.on,* eg1 = e.on, *eg=NULL;
		GeometricalEdge * eg0=on, *eg1=on;
		R2 Ag=(R2) (*on)[0],Bg=(R2)(*on)[1],AB=Bg-Ag; 
		if(NbTry) cout <<" G edge= " << Ag << Bg << endl << " v edge" << V01 << " v geom " << AB  <<  (V01,AB) <<endl; 
		int OppositeSens = (V01,AB) < 0;
		int sens0=0,sens1=1;
		if (OppositeSens)
		 s=1-s,Exchange(vg0,vg1),Exchange(V0,V1);
		if(NbTry) cout << "bug: edge = " << v0.r << " -> " << v1.r << endl 
		 << "sg 0 = " << vg0 
			<< " on = " << Number(on) << ":" << Ag << Bg << "; " 
			<<  " sg 1= " << vg1 
			<< "--------------------------------------------" << endl;
		while (eg0 != (GeometricalEdge*) vg0  &&  (*eg0)(sens0) != (GeometricalVertex*) vg0)
		  { 
			if (bge<=0) {
				//          int kkk;
				// if (NbTry) cout <<"Read (int) to Show Sioux window", cin >> kkk ;
				if(NbTry) 
				  {
					cerr << " -- Fatal Error: on the class triangles before call Geometry::ProjectOnCurve" << endl; 
					cerr << "   The mesh of the  Geometry is to fine: ";
					cerr << "     1)  a mesh edge  contening more than "<< mxe/2 << " geometrical edges." << endl;
					cerr << "     2)  code bug : be sure that we call   Triangles::SetVertexFieldOn() before " << endl;
					cerr << "   To solve the problem do a coarsening of the geometrical mesh " << endl;
					cerr << " or change the constant value of mxe in " << __FILE__ << " line " << MXE__LINE << "( dangerous way )" << endl;	  
					MeshError(222);
				  }
				NbTry++;
				goto retry;}
				GeometricalEdge* tmpge = eg0;
				if(NbTry)
				 cout << "bug: --Edge @" <<  Number(tmpge)  << " = "<< Number(eg0) << ":" <<Number(eg0->Adj[0]) << "," <<  
					Number(eg0->Adj[1]) <<"," ;
				ge[--bge] =eg0 = eg0->Adj[sens0];
				assert(bge>=0 && bge <= mxe);
				sens0 = 1-( sensge[bge] = tmpge->SensAdj[sens0]);
				if(NbTry)
				 cout << "bug: Edge "  <<  Number(eg0) << " "<< 1-sens0 <<  " S "
					<< Number((*eg0)[1-sens0]) <<":" << Number(eg0->Adj[0]) << "," 
					<<  Number(eg0->Adj[1]) <<"," << endl
					<<Number(eg0)<< (*eg0)[sens0].r << "v = " << Number((*eg1)(sens0)) << " e = " << eg0 <<  endl;
		  }
		if(NbTry) cout << Number((GeometricalEdge*) vg1) << " " << Number((GeometricalVertex*) vg1) << endl;
		while (eg1 != (GeometricalEdge*) vg1  &&  (*eg1)(sens1) != (GeometricalVertex*) vg1)
		  { 
			if(tge>=mxe ) { 
				cerr << " --Fatal Error: on the class triangles before call Geometry::ProjectOnCurve" << endl; 
				NbTry++;
				if (NbTry<2) goto retry;
				cerr << "   The mesh of the  Geometry is to fine:" ;
				cerr << "     1)  a mesh edge  contening more than "<< mxe/2 << " geometrical edges." << endl;
				cerr << "     2)  code bug : be sure that we call   Triangles::SetVertexFieldOn() before " << endl;
				cerr << "   To solve the problem do a coarsening of the geometrical mesh " << endl;
				cerr << " or change the constant value of mxe in " << __FILE__ << " line " << MXE__LINE << "( dangerous way )" << endl;	
				MeshError(223);
			}

			GeometricalEdge* tmpge = eg1;
			if(NbTry)
			 cout << "++Edge @" << tmpge << " = " <<  Number(eg1) <<"%" << Number(eg1->Adj[0]) << "," 
				<<  Number(eg1->Adj[1]) <<"," ;
			ge[++tge] =eg1 = eg1->Adj[sens1];
			sensge[tge]= sens1 = 1-tmpge->SensAdj[sens1];
			assert(tge>=0 && tge <= mxe);
			if(NbTry)
			 cout << "  Edge "  <<  Number(eg1) << " " << sens1 << " S "
				<<Number((*eg1)[sens1]) <<"%"<< Number(eg1->Adj[0]) << "," <<  Number(eg1->Adj[1]) <<"," 
				<<Number(eg1)<< (*eg1)[sens1].r << "v = " << Number((*eg1)(sens1)) << " e = " << Number(eg1) <<  endl;
		  }

		if(NbTry)    cout << endl;


		if ( (*eg0)(sens0) == (GeometricalVertex*) vg0 )
		 vg0 = VertexOnGeom( *(Vertex *) vg0,*eg0,sens0);

		if ( (*eg1)(sens1) == (GeometricalVertex*) vg1)
		 vg1 = VertexOnGeom( *(Vertex *) vg1,*eg1,sens1);

		Real8 sg;
		//   cout << "           " << Number(on) << " " <<  Number(eg0) << " " <<  Number(eg1) << " "  ; 
		if (eg0 == eg1) { 
			register Real8 s0= vg0,s1=vg1;
			sg =  s0 * (1.0-s) +  s * s1;
			//    cout <<"                s0=" << s0 << " s1=" << s1 
			//             << " s = " << s << " sens= " << OppositeSens << "\t\t sg = " << sg << endl ;
			on=eg0;}
		else {
			R2 AA=V0,BB;
			Real8 s0,s1;

			//cout << endl << "s= " << s << Number(eg0) << " " << (Real8) vg0 << " " 
			//	    << Number(eg1) << " " << (Real8) vg1 << V0 << V1 << "  Interpol = " 
			// << V0*(1-s)+V1*s << ";;; " <<  endl;
			int i=bge;
			Real8 ll=0;
			for(i=bge;i<tge;i++) 
			  {
				assert( i>=0 && i <= mxe);
				BB =  (*ge[i])[sensge[i]];
				lge[i]=ll += Norme2(AA-BB);
				//   cout << " ll " << i << BB << ll << " " <<sensge[i] <<" on = " <<
				// Number(ge[i]) << " sens= " << sensge[i] ;
				AA=BB ;}
				lge[tge]=ll+=Norme2(AA-V1); 
				// cout << " ll " << tge << " " << ll <<  sensge[tge] 
				//	     <<" on = " << Number(ge[tge]) <<  " sens= " << sensge[tge] << endl;
				// search the geometrical edge
				assert(s <= 1.0);
				Real8 ls= s*ll;
				on =0;
				s0 = vg0;
				s1= sensge[bge];
				Real8 l0=0,l1;
				i=bge;
				while (  (l1=lge[i]) < ls ) {
					assert(i >= 0 && i <= mxe);
					i++,s0=1-(s1=sensge[i]),l0=l1;}
					on=ge[i];
					if (i==tge) 
					 s1=vg1;

					s=(ls-l0)/(l1-l0);
					//  cout << "on =" << Number(on) << sens0 << sens1 <<  "s0  " << s0 << " s1 =" 
					//	     << s1 << " l0 =" << l0 << " ls= " << ls << " l1= " << l1 << " s= " << s;
					sg =  s0 * (1.0-s) +  s * s1;    
		} 
		assert(on);
		// assert(sg && sg-1);
		V.r= on->F(sg);
		//  if (eg0 != eg1) 
		//        cout << "----- sg = "<< sg << " Sens =" << OppositeSens << " Edge = " 
		//     << Number(on) <<"  V=" << V << endl;
		GV=VertexOnGeom(V,*on,sg);
		return on;
	}
	/*}}}1*/
	/*FUNCTION  Geometry::AfterRead(){{{1*/
	void Geometry::AfterRead() {
		long int verbosity=0;

		if (verbosity>20)
		 cout << "Geometry::AfterRead()" <<  nbv << " " << nbe << endl;
		Int4 i,k=0;        ;
		int jj; // jj in [0,1]
		Int4 * hv = new Int4 [ nbv];
		Int4 * ev = new Int4 [ 2 * nbe ];
		float  * eangle = new float[ nbe ];
		  {
			double eps = 1e-20;
			QuadTree quadtree; // to find same vertices
			Vertex * v0 = vertices; 
			GeometricalVertex  * v0g = (GeometricalVertex  *) (void *) v0;   
			int k=0;
			for (i=0;i<nbv;i++) 
			 vertices[i].link = vertices +i;
			for (i=0;i<nbv;i++) 
			  {
				vertices[i].i = toI2(vertices[i].r); // set integer coordinate
				Vertex *v= quadtree.NearestVertex(vertices[i].i.x,vertices[i].i.y); 
				if( v && Norme1(v->r - vertices[i]) < eps )
				  { // link v & vertices[i] 
					// vieille ruse pour recuperer j 
					GeometricalVertex * vg = (GeometricalVertex  *) (void *) v;
					int j = vg-v0g;
					assert( v ==  & (Vertex &) vertices[j]);
					vertices[i].link = vertices + j;
					k++;	      
				  }
				else  quadtree.Add(vertices[i]); 
			  }
			if (k) {
				cout << " Number of distinte vertices " << nbv - k << " Over " << nbv << endl;
				//if (verbosity>10) 
				  {
					cout << " The duplicate vertex " << endl;
					for (i=0;i<nbv;i++)
					 if (!vertices[i].IsThe())
					  cout << " " << i << " and " << Number(vertices[i].The()) << endl;
					MeshError(102);
					//throw(ErrorExec("exit",1));    
				  }
			}

			//  verification of cracked edge
			int err =0;
			for (i=0;i<nbe;i++)
			 if (edges[i].Cracked() )
				{
				 //    verification of crack
				 GeometricalEdge & e1=edges[i];
				 GeometricalEdge & e2=*e1.link;
				 cerr << i << " " << e1[0].The() << " " << e2[0].The() << " " <<  e1[1].The() << " " << e2[1].The() << endl;
				 if ( e1[0].The() == e2[0].The() && e1[1].The() == e2[1].The() )
					{
					}
				 else 
				  if ( e1[0].The() == e2[1].The() && e1[1].The() == e2[0].The() )
					 {
					 }
				  else
					 {
					  err++;
					  cerr << " Cracked edges with no same vertex " << &e1-edges << " " << &e2 -edges << endl;
					 }
				}
			 else
				{
				 //  if (!edges[i][0].IsThe()) err++;
				 // if (!edges[i][1].IsThe()) err++;
				}
			if (err)
			  {
				cerr << " Some vertex was not distint and not on cracked edge " << err<< endl;
				MeshError(222);
			  }
		  }
		if(verbosity>7) 
		 for (i=0;i<nbv;i++)
		  if (vertices[i].Required())
			cout << "     The geo vertices  " << i << " is required" << endl;

		for (i=0;i<nbv;i++) 
		 hv[i]=-1;// empty list

		for (i=0;i<nbe;i++) 
		  {
			R2 v10  =  edges[i].v[1]->r -  edges[i].v[0]->r;
			Real8 lv10 = Norme2(v10);
			if(lv10 == 0) {
				cerr << "The length  of " <<i<< "th Egde is 0 " << endl ;
				MeshError(1);}
				eangle[i] = atan2(v10.y,v10.x)  ; // angle in [ -Pi,Pi ]
				if(verbosity>9) 
				 cout << "     angle edge " << i <<" " << eangle[i]*180/Pi<< v10<<endl;
				for (jj=0;jj<2;jj++)
				  { // generation of list
					Int4 v =  Number(edges[i].v[jj]);
					ev[k] = hv[v];
					hv[v] = k++;
				  }
		  }
		// bulle sort on the angle of edge  
		for (i=0;i<nbv;i++) {
			int exch = 1,ord =0;      
			while (exch) {
				exch = 0;
				Int4  *p =  hv + i, *po = p;
				Int4 n = *p;
				register float angleold = -1000 ; // angle = - infini 
				ord = 0;
				while (n >=0) 
				  {
					ord++;
					register Int4 i1= n /2;
					register Int4  j1 = n % 2;
					register Int4 *pn = ev + n;
					float angle = j1 ? OppositeAngle(eangle[i1]):  eangle[i1];
					n = *pn;
					if (angleold > angle) // exch to have : po -> pn -> p 
					 exch=1,*pn = *po,*po=*p,*p=n,po = pn;
					else //  to have : po -> p -> pn 
					 angleold =  angle, po = p,p  = pn;
				  }
			} // end while (exch)

			if (ord >= 1 ) 
			  { /*
					 Int4 n = hv[i];
					 while ( n >=0) 
					 { Int4 i1 = n/2,j1 = n%2;
				//float a = 180*(j1 ? OppositeAngle(eangle[i1]): eangle[i1])/Pi;
				n = ev[n];
				}
				*/
			  } 
			if(ord == 2) { // angulare test to find a corner 
				Int4 n1 = hv[i];
				Int4 n2 = ev[n1];
				Int4 i1 = n1 /2, i2 = n2/2; // edge number
				Int4  j1 = n1 %2, j2 = n2%2; // vertex in the edge 
				float angle1= j1 ? OppositeAngle(eangle[i1]) :  eangle[i1];
				float angle2= !j2 ? OppositeAngle(eangle[i2]) :  eangle[i2];
				float da12 = Abs(angle2-angle1);
				if(verbosity>9)
				 cout <<"     check angle " << i << " " << i1 << " " << i2  << " " << 180*(da12)/Pi 
					<< " " << 180*MaximalAngleOfCorner/Pi << vertices[i] << endl;

				if (( da12 >= MaximalAngleOfCorner ) 
							&& (da12 <= 2*Pi -MaximalAngleOfCorner)) {
					vertices[i].SetCorner() ; 
					if(verbosity>7)
					 cout << "     The vertex " << i << " is a corner (angle) " 
						<< 180*(da12)/ Pi<< " " << 180*MaximalAngleOfCorner/Pi << endl;}
					// if the ref a changing then is     SetRequired();

					if (edges[i1].flag != edges[i2].flag || edges[i1].Required()) 
					  {
						vertices[i].SetRequired();
						if(verbosity>7)
						 cout << "     The vertex " << i << " is Required the flag change (crack or equi, or require)" << endl;}

						if (edges[i1].ref != edges[i2].ref) {
							vertices[i].SetRequired();
							if(verbosity>7)
							 cout << "     The vertex " << i << " is Required ref" << endl;}
			} ;

			if(ord != 2) {
				vertices[i].SetCorner();
				if(verbosity>7)
				 cout << "     the vertex " << i << " is a corner ordre = " << ord << endl;
			}
			// close the liste around the vertex 
			  { Int4 no=-1, ne = hv[i];
				while ( ne >=0) 
				 ne = ev[no=ne];        
				if(no>=0) 
				 ev[no] = hv[i];
			  } // now the list around the vertex is circular

		} // end for (i=0;i<nbv;i++)

		k =0;
		for (i=0;i<nbe;i++)
		 for (jj=0;jj<2;jj++){
			 Int4 n1 = ev[k++]; 
			 Int4 i1 = n1/2 ,j1=n1%2;
			 if( edges[i1].v[j1] != edges[i].v[jj]) 
				{ cerr << " Bug Adj edge " << i << " " << jj << 
				 " et " << i1 << " " << j1 << " k=" << k;
				 cerr << Number(edges[i].v[jj]) <<" <> " 
					<< Number(edges[i1].v[j1])  <<endl;
				 cerr << "edge " << Number(edges[i].v[0]) << " " 
					<< Number(edges[i].v[1]) << endl; 
				 //    cerr << "in file " <<filename <<endl;
				 MeshError(1);
				}
			 edges[i1].Adj[j1] = edges + i;
			 edges[i1].SensAdj[j1] = jj;
			 if (verbosity>10)
			  cout << " edges. Adj " << i1 << " " << j1 << " <--- " << i << " " << jj << endl;
		 }

		// generation of  all the tangente 
		for (i=0;i<nbe;i++) {
			R2 AB = edges[i].v[1]->r -edges[i].v[0]->r;        
			Real8 lAB = Norme2(AB); // length of current edge AB
			Real8 ltg2[2];
			ltg2[0]=0;ltg2[1]=0;
			for (jj=0;jj<2;jj++) {
				R2 tg =  edges[i].tg[jj];
				Real8 ltg = Norme2(tg); // length of tg
				if(ltg == 0) {// no tg
					if( ! edges[i].v[jj]->Corner())   { // not a Corner       
						tg =  edges[i].v[1-jj]->r 
						  - edges[i].Adj[jj]->v[1-edges[i].SensAdj[jj]]->r;
						ltg =  Norme2(tg);
						tg =  tg *(lAB/ltg),ltg=lAB;
						/*
							if(edges[i].ref >=4) 
							cout << " tg " << tg.x << " "<< tg.y  << " " << edges[i].v[1-jj]->r << edges[i].Adj[jj]->v[1-edges[i].SensAdj[jj]]->r << " y-y = "
							<< edges[i].v[1-jj]->r.y -edges[i].Adj[jj]->v[1-edges[i].SensAdj[jj]]->r.y <<  endl;
							*/
					}

					//else ;// a Corner with no tangent => nothing to do    
				} // a tg 
				else 
				 tg = tg *(lAB/ltg),ltg=lAB;
				ltg2[jj] = ltg;
				if ( (tg,AB) < 0) 
				 tg = -tg;
				//if(edges[i].ref >=4) cout << " tg = " << tg << endl;
				edges[i].tg[jj] = tg;
			}     // for (jj=0;jj<2;jj++) 

			if (ltg2[0]!=0) edges[i].SetTgA();
			if (ltg2[1]!=0) edges[i].SetTgB();
		} // for (i=0;i<nbe;i++)

		if(verbosity>7)
		 for (i=0;i<nbv;i++)
		  if (vertices[i].Required())
			cout << "     The  geo  vertices " << i << " is required " << endl;

		for (int step=0;step<2;step++)
		  {
			for (i=0;i<nbe;i++) 
			 edges[i].SetUnMark();

			NbOfCurves = 0;
			Int4  nbgem=0;
			for (int level=0;level < 2 && nbgem != nbe;level++)
			 for (i=0;i<nbe;i++) {
				 GeometricalEdge & ei = edges[i];   
				 for(jj=0;jj<2;jj++) 
				  if (!ei.Mark() && (level || ei[jj].Required())) { 
					  // warning ei.Mark() can be change in loop for(jj=0;jj<2;jj++) 
					  int k0=jj,k1;
					  GeometricalEdge *e = & ei;
					  GeometricalVertex *a=(*e)(k0); // begin 
					  if(curves) {
						  curves[NbOfCurves].be=e;
						  curves[NbOfCurves].kb=k0;
					  }
					  int nee=0;
					  for(;;) { 
						  nee++;
						  k1 = 1-k0; // next vertex of the edge 
						  e->SetMark();
						  nbgem++;
						  e->CurveNumber=NbOfCurves;
						  if(curves) {
							  curves[NbOfCurves].ee=e;
							  curves[NbOfCurves].ke=k1;
						  }

						  GeometricalVertex *b=(*e)(k1);
						  if (a == b ||  b->Required() ) break;
						  k0 = e->SensAdj[k1];//  vertex in next edge
						  e = e->Adj[k1]; // next edge

					  }// for(;;)
					  if(verbosity>10 && curves==0) cout << NbOfCurves <<" curve :  nb edges=  "<< nee<<  endl; 
					  NbOfCurves++;
					  if(level) {
						  if(verbosity>4)
							cout << "    Warning: Curve "<< NbOfCurves << " without required vertex " 
							  << "so the vertex " << Number(a) << " become required " <<endl;
						  a->SetRequired();
					  }

				  }} 
				 assert(nbgem && nbe);

				 if(step==0) {
					 curves = new Curve[NbOfCurves];
				 }
		  } 
		for(int i=0;i<NbOfCurves ;i++)
		  {
			GeometricalEdge * be=curves[i].be, *eqbe=be->link;
			//GeometricalEdge * ee=curves[i].ee, *eqee=be->link;
			curves[i].master=true;
			if(be->Equi() || be->ReverseEqui() ) 
			  {
				assert(eqbe);
				int nc = eqbe->CurveNumber;
				assert(i!=nc);
				curves[i].next=curves[nc].next;
				curves[i].master=false;
				curves[nc].next=curves+i;
				if(be->ReverseEqui())
				 curves[i].Reverse();           
			  }
		  }

		if(verbosity>3)
		 cout << "    End ReadGeometry: Number of curves in geometry is " << NbOfCurves <<endl; 
		if(verbosity>4)
		 for(int i=0;i<NbOfCurves ;i++)
			{
			 cout << " Curve " << i << " begin e=" << Number(curves[i].be) << " k=" << curves[i].kb 
				<< "  end e= " << Number(curves[i].ee) << " k=" << curves[i].ke << endl;
			}
		delete []ev;
		delete []hv;
		delete []eangle;

	}
	/*}}}1*/

} 
