#include "../shared/shared.h"
#include "../include/macros.h"
#include "../toolkits/toolkits.h"

#include <cstdio>
#include <cstring>
#include <cmath>
#include <ctime>
#include "Mesh2.h"
#include "QuadTree.h"
#include "SetOfE4.h"

namespace bamg {

	static const  Direction NoDirOfSearch=Direction();

	/*Constructors/Destructors*/
	/*FUNCTION Triangles::Triangles(BamgMesh* bamgmesh, BamgOpts* bamgopts){{{1*/
	Triangles::Triangles(BamgMesh* bamgmesh, BamgOpts* bamgopts):Gh(*(new Geometry())),BTh(*this){ 

		PreInit(0,"none");
		OnDisk = 1;

		ReadMesh(bamgmesh,bamgopts);
		SetIntCoor();
		FillHoleInMesh();
	}
	/*}}}1*/

	/*IO*/
	/*FUNCTION Triangles::WriteMesh {{{1*/
	void Triangles::WriteMesh(BamgMesh* bamgmesh,BamgOpts* bamgopts){

		int i,j;
		int verbose;

		verbose=bamgopts->verbose;
		Int4 *reft = new Int4[nbt];
		Int4 nbInT = ConsRefTriangle(reft);

		//Vertices
		if(verbose>3) printf("      writing Vertices\n");
		bamgmesh->NumVertices=nbv;
		xfree((void**)&bamgmesh->Vertices);
		if (nbv){
			bamgmesh->Vertices=(double*)xmalloc(3*nbv*sizeof(double));
			for (i=0;i<nbv;i++){
				bamgmesh->Vertices[i*3+0]=vertices[i].r.x;
				bamgmesh->Vertices[i*3+1]=vertices[i].r.y;
				bamgmesh->Vertices[i*3+2]=vertices[i].ref();
			}
		}
		else{
			bamgmesh->Vertices=NULL;
		}

		//Edges
		if(verbose>3) printf("      writing Edges\n");
		bamgmesh->NumEdges=nbe;
		xfree((void**)&bamgmesh->Edges);
		if (nbe){
			bamgmesh->Edges=(double*)xmalloc(3*nbe*sizeof(double));
			for (i=0;i<nbe;i++){
				bamgmesh->Edges[i*3+0]=Number(edges[i][0])+1; //back to M indexing
				bamgmesh->Edges[i*3+1]=Number(edges[i][1])+1; //back to M indexing
				bamgmesh->Edges[i*3+2]=edges[i].ref;
			}
		}
		else{
			bamgmesh->Edges=NULL;
		}

		//CrackedEdges
		if(verbose>3) printf("      writing CrackedEdges\n");
		bamgmesh->NumCrackedEdges=NbCrackedEdges;
		xfree((void**)&bamgmesh->CrackedEdges);
		if (NbCrackedEdges){
			bamgmesh->CrackedEdges=(double*)xmalloc(2*NbCrackedEdges*sizeof(double));
			for (i=0;i<NbCrackedEdges;i++){
				bamgmesh->CrackedEdges[i*2+0]=Number(CrackedEdges[i].a.edge)+1; //back to M indexing
				bamgmesh->CrackedEdges[i*2+1]=Number(CrackedEdges[i].b.edge)+1; //back to M indexing
			}
		}
		else{
			bamgmesh->CrackedEdges=NULL;
		}

		//Triangles
		if(verbose>3) printf("      writing Triangles\n");
		Int4 k=nbInT-NbOfQuad*2;
		Int4 num=0;
		bamgmesh->NumTriangles=k;
		xfree((void**)&bamgmesh->Triangles);
		if (k){
			bamgmesh->Triangles=(double*)xmalloc(4*k*sizeof(double));
			for (i=0;i<nbt;i++){
				Triangle &t=triangles[i];
				if (reft[i]>=0 && !( t.Hidden(0) || t.Hidden(1) || t.Hidden(2) )){
					bamgmesh->Triangles[num*4+0]=Number(t[0])+1; //back to M indexing
					bamgmesh->Triangles[num*4+1]=Number(t[1])+1; //back to M indexing
					bamgmesh->Triangles[num*4+2]=Number(t[2])+1; //back to M indexing
					bamgmesh->Triangles[num*4+3]=subdomains[reft[i]].ref;
					num=num+1;
				}
			}
		}
		else{
			bamgmesh->Triangles=NULL;
		}

		//Quadrilaterals
		if(verbose>3) printf("      writing Quadrilaterals\n");
		bamgmesh->NumQuadrilaterals=NbOfQuad;
		xfree((void**)&bamgmesh->Quadrilaterals);
		if (NbOfQuad){
			bamgmesh->Quadrilaterals=(double*)xmalloc(5*NbOfQuad*sizeof(double));
			for (i=0;i<nbt;i++){
				Triangle &t =triangles[i];
				Triangle* ta;
				Vertex *v0,*v1,*v2,*v3;
				if (reft[i]<0) continue;
				if ((ta=t.Quadrangle(v0,v1,v2,v3)) !=0 && &t<ta) { 
					bamgmesh->Quadrilaterals[i*5+0]=Number(v0)+1; //back to M indexing
					bamgmesh->Quadrilaterals[i*5+1]=Number(v1)+1; //back to M indexing
					bamgmesh->Quadrilaterals[i*5+2]=Number(v2)+1; //back to M indexing
					bamgmesh->Quadrilaterals[i*5+3]=Number(v3)+1; //back to M indexing
					bamgmesh->Quadrilaterals[i*5+4]=subdomains[reft[i]].ref;
				}
			}
		}
		else{
			bamgmesh->Quadrilaterals=NULL;
		}

		//SubDomains
		if(verbose>3) printf("      writing SubDomains\n");
		bamgmesh->NumSubDomains=NbSubDomains;
		xfree((void**)&bamgmesh->SubDomains);
		if (NbSubDomains){
			bamgmesh->SubDomains=(double*)xmalloc(4*NbSubDomains*sizeof(double));
			for (i=0;i<NbSubDomains;i++){
				bamgmesh->SubDomains[i*4+0]=3;
				bamgmesh->SubDomains[i*4+1]=reft[Number(subdomains[i].head)];
				bamgmesh->SubDomains[i*4+2]=1;
				bamgmesh->SubDomains[i*4+3]=subdomains[i].ref;
			}
		}
		else{
			bamgmesh->SubDomains=NULL;
		}

		//SubDomainsFromGeom
		if(verbose>3) printf("      writing SubDomainsFromGeom\n");
		bamgmesh->NumSubDomainsFromGeom=Gh.NbSubDomains;
		xfree((void**)&bamgmesh->SubDomainsFromGeom);
		if (Gh.NbSubDomains){
			bamgmesh->SubDomainsFromGeom=(double*)xmalloc(4*Gh.NbSubDomains*sizeof(double));
			for (i=0;i<Gh.NbSubDomains;i++){
				bamgmesh->SubDomainsFromGeom[i*4+0]=2;
				bamgmesh->SubDomainsFromGeom[i*4+1]=Number(subdomains[i].edge)+1; //back to Matlab indexing
				bamgmesh->SubDomainsFromGeom[i*4+2]=subdomains[i].sens;
				bamgmesh->SubDomainsFromGeom[i*4+3]=Gh.subdomains[i].ref;
			}
		}
		else{
			bamgmesh->SubDomainsFromGeom=NULL;
		}

		//VerticesOnGeomVertex
		if(verbose>3) printf("      writing VerticesOnGeometricVertex\n");
		bamgmesh->NumVerticesOnGeometricVertex=NbVerticesOnGeomVertex;
		xfree((void**)&bamgmesh->VerticesOnGeometricVertex);
		if (NbVerticesOnGeomVertex){
			bamgmesh->VerticesOnGeometricVertex=(double*)xmalloc(2*NbVerticesOnGeomVertex*sizeof(double));
			for (i=0;i<NbVerticesOnGeomVertex;i++){
				VertexOnGeom &v=VerticesOnGeomVertex[i];
				if (!v.OnGeomVertex()){
					throw ErrorException(__FUNCT__,exprintf("A vertices supposed to be OnGeometricVertex is actually not"));
				}
				bamgmesh->VerticesOnGeometricVertex[i*2+0]=Number((Vertex*)v)+1; //back to Matlab indexing
				bamgmesh->VerticesOnGeometricVertex[i*2+1]=Gh.Number(( GeometricalVertex*)v)+1; //back to Matlab indexing
			}
		}
		else{
			bamgmesh->VerticesOnGeometricVertex=NULL;
		}

		//VertexOnGeometricEdge
		if(verbose>3) printf("      writing VerticesOnGeometricEdge\n");
		bamgmesh->NumVerticesOnGeometricEdge=NbVerticesOnGeomEdge;
		xfree((void**)&bamgmesh->VerticesOnGeometricEdge);
		if (NbVerticesOnGeomEdge){
			bamgmesh->VerticesOnGeometricEdge=(double*)xmalloc(3*NbVerticesOnGeomEdge*sizeof(double));
			for (i=0;i<NbVerticesOnGeomEdge;i++){
				const VertexOnGeom &v=VerticesOnGeomEdge[i];
				if (!v.OnGeomEdge()){
					throw ErrorException(__FUNCT__,exprintf("A vertices supposed to be OnGeometricEdge is actually not"));
				}
				bamgmesh->VerticesOnGeometricEdge[i*3+0]=Number((Vertex*)v)+1; //back to Matlab indexing
				bamgmesh->VerticesOnGeometricEdge[i*3+1]=Gh.Number((const GeometricalEdge*)v)+1; //back to Matlab indexing
				bamgmesh->VerticesOnGeometricEdge[i*3+2]=(double)v;
			}
		}
		else{
			bamgmesh->VerticesOnGeometricEdge=NULL;
		}

		//EdgesOnGeometricEdge
		if(verbose>3) printf("      writing EdgesOnGeometricEdge\n");
		k=0;
		for (i=0;i<nbe;i++){
			if (edges[i].on) k=k+1;
		}
		bamgmesh->NumEdgesOnGeometricEdge=k;
		xfree((void**)&bamgmesh->EdgesOnGeometricEdge);
		if (k){
			bamgmesh->EdgesOnGeometricEdge=(double*)xmalloc(2*(int)k*sizeof(double));
			int count=0;
			for (i=0;i<nbe;i++){
				if (edges[i].on){
					bamgmesh->EdgesOnGeometricEdge[count*2+0]=(double)i+1; //back to Matlab indexing
					bamgmesh->EdgesOnGeometricEdge[count*2+1]=(double)Gh.Number(edges[i].on)+1; //back to Matlab indexing
					count=count+1;
				}
			}
		}
		else{
			bamgmesh->EdgesOnGeometricEdge=NULL;
		}
	}
	/*}}}1*/
	/*FUNCTION Triangles::ReadMesh{{{1*/
	void Triangles::ReadMesh(BamgMesh* bamgmesh, BamgOpts* bamgopts){

		int verbose;
		Real8 Hmin = HUGE_VAL;// the infinie value 
		Int4 i1,i2,i3,iref;
		Int4 i,j;
		Int4 hvertices =0;
		Int4 ifgeom=0;
		Metric M1(1);
		int Version=1,dim=2;

		verbose=bamgopts->verbose;

		nbv=bamgmesh->NumVertices;
		nbvx=nbv;
		nbt=bamgmesh->NumTriangles;
		nbiv=nbvx;

		//Vertices
		if(bamgmesh->Vertices){
			if(verbose>3) printf("      processing Vertices\n");

			vertices=(Vertex*)xmalloc(nbv*sizeof(Vertex));
			ordre=(Vertex**)xmalloc(nbv*sizeof(Vertex*));

			for (i=0;i<nbv;i++){
				vertices[i].r.x=bamgmesh->Vertices[i*3+0];
				vertices[i].r.y=bamgmesh->Vertices[i*3+1];
				vertices[i].ReferenceNumber=1;
				vertices[i].DirOfSearch =NoDirOfSearch;
				vertices[i].m=M1;
				vertices[i].color=(Int4)bamgmesh->Vertices[i*3+2];
			}
			nbtx=2*nbvx-2; // for filling The Holes and quadrilaterals 
		}
		else{
			if(verbose>3) throw ErrorException(__FUNCT__,exprintf("no Vertices found in the initial mesh"));
		}

		//Triangles
		if(bamgmesh->Triangles){
			if(verbose>3) printf("      processing Triangles\n");
			triangles =new Triangle[nbtx]; //we cannot allocate only nbt triangles since 
			//other triangles will be added for each edge
			for (i=0;i<nbt;i++){
				Triangle & t = triangles[i];
				i1=(Int4)bamgmesh->Triangles[i*4+0]-1; //for C indexing
				i2=(Int4)bamgmesh->Triangles[i*4+1]-1; //for C indexing
				i3=(Int4)bamgmesh->Triangles[i*4+2]-1; //for C indexing
				t=Triangle(this,i1,i2,i3);
				t.color=(Int4)bamgmesh->Triangles[i*4+3];
			}
		}
		else{
			if(verbose>3) throw ErrorException(__FUNCT__,exprintf("no Triangles found in the initial mesh"));
		}

		//Quadrilaterals
		if(bamgmesh->Quadrilaterals){
			if(verbose>3) printf("      processing Quadrilaterals\n");
			Int4 i1,i2,i3,i4,iref;
			triangles =new Triangle[nbt];
			for (i=0;i<bamgmesh->NumQuadrilaterals;i++){
				Triangle & t1 = triangles[2*i];
				Triangle & t2 = triangles[2*i+1];
				i1=(Int4)bamgmesh->Quadrilaterals[i*4+0]-1; //for C indexing
				i2=(Int4)bamgmesh->Quadrilaterals[i*4+1]-1; //for C indexing
				i3=(Int4)bamgmesh->Quadrilaterals[i*4+2]-1; //for C indexing
				i4=(Int4)bamgmesh->Quadrilaterals[i*4+3]-1; //for C indexing
				t1=Triangle(this,i1,i2,i3);
				t2=Triangle(this,i3,i4,i1);
				t1.color=1; //reference = 1 for all triangles since it has not been specified
				t2.color=1; //reference = 1 for all triangles since it has not been specified
				t1.SetHidden(OppositeEdge[1]); // two times  because the adj was not created 
				t2.SetHidden(OppositeEdge[1]); 
			}
		}
		else{
			if(verbose>3) printf("      no Quadrilaterals found\n");
		}

		//hVertices
		if(bamgmesh->hVertices){
			if(verbose>3) printf("      processing hVertices\n");
			hvertices=1;
			for (i=0;i< nbv;i++){
				vertices[i].m=Metric((Real4)bamgmesh->hVertices[i]);
			}
		}
		else{
			if(verbose>3) printf("      no hVertices found\n");
		}

		//VerVerticesOnGeometricEdge
		if(bamgmesh->VerticesOnGeometricEdge){
			if(verbose>3) printf("      processing VerticesOnGeometricEdge\n");
			NbVerticesOnGeomEdge=bamgmesh->NumVerticesOnGeometricEdge;
			VerticesOnGeomEdge= new  VertexOnGeom[NbVerticesOnGeomEdge] ;
			for (i=0;i<NbVerticesOnGeomEdge;i++){
				Int4  i1,i2;
				Real8 s;
				i1=(Int4)bamgmesh->VerticesOnGeometricEdge[i*3+0]-1; //for C indexing
				i2=(Int4)bamgmesh->VerticesOnGeometricEdge[i*3+1]-1; //for C indexing
				s =(Int4)bamgmesh->VerticesOnGeometricEdge[i*3+2];
				VerticesOnGeomEdge[i]=VertexOnGeom(vertices[i1],Gh.edges[i2],s);
			}
		}
		else{
			if(verbose>3) printf("      no VertexOnGeometricEdge found\n");
		}

		//Edges
		if (bamgmesh->Edges){
			int i1,i2;
			R2 zero2(0,0);
			Real4 *len =0;

			if(verbose>3) printf("      processing Edges\n");
			nbe=bamgmesh->NumEdges;
			edges = new Edge[nbe];

			if (!hvertices) {
				len = new Real4[nbv];
				for(i=0;i<nbv;i++)
				 len[i]=0;
			}

			for (i=0;i<nbe;i++){
				i1=(int)bamgmesh->Edges[i*3+0]-1; //-1 for C indexing
				i2=(int)bamgmesh->Edges[i*3+1]-1; //-1 for C indexing
				edges[i].v[0]= vertices +i1;
				edges[i].v[1]= vertices +i2;
				edges[i].adj[0]=0;
				edges[i].adj[1]=0;
				R2 x12 = vertices[i2].r-vertices[i1].r;
				Real8 l12=sqrt( (x12,x12));        

				if (!hvertices) {
					vertices[i1].color++;
					vertices[i2].color++;
					len[i1]+= l12;
					len[i2] += l12;
				}
				Hmin = Min(Hmin,l12);
			}

			// definition  the default of the given mesh size 
			if (!hvertices){
				for (i=0;i<nbv;i++) 
				 if (vertices[i].color > 0) 
				  vertices[i].m=  Metric(len[i] /(Real4) vertices[i].color);
				 else 
				  vertices[i].m=  Metric(Hmin);
				delete [] len;
			}

			// construction of edges[].adj 
			for (i=0;i<nbv;i++){ 
				vertices[i].color = (vertices[i].color ==2) ? -1 : -2;
			}
			for (i=0;i<nbe;i++){
				for (j=0;j<2;j++) { 
					Vertex *v=edges[i].v[j];
					Int4 i0=v->color,j0;
					if(i0==-1){
						v->color=i*2+j;
					}
					else if (i0>=0) {// i and i0 edge are adjacent by the vertex v
						j0 =  i0%2;
						i0 =  i0/2;
						assert( v ==  edges[i0 ].v[j0]);
						edges[i ].adj[ j ] =edges +i0;
						edges[i0].adj[ j0] =edges +i ;
						assert(edges[i0].v[j0] == v);
						v->color = -3;
					}
				}
			}
		}
		else{
			if(verbose>3) printf("      no Edges found\n");
		}

		//EdgeOnGeometricEdge
		if(bamgmesh->EdgesOnGeometricEdge){
			if(verbose>3) printf("      processing EdgesOnGeometricEdge\n");
			int i1,i2,i,j;
			i2=bamgmesh->NumEdgesOnGeometricEdge;
			for (i1=0;i1<i2;i1++) {
				i=(int)bamgmesh->EdgesOnGeometricEdge[i*2+0];
				j=(int)bamgmesh->EdgesOnGeometricEdge[i*2+1];
				if(!(i>0 && j >0 && i <= nbe && j <= Gh.nbe)) {
					throw ErrorException(__FUNCT__,exprintf("EdgesOnGeometricEdge error: We must have : (i>0 && j >0 && i <= nbe && j <= Gh.nbe)"));
				}
				edges[i-1].on = Gh.edges + j-1;
			}
		}
		else{
			if(verbose>3) printf("      no EdgesOnGeometricEdge found\n");
		}

		//SubDomain
		if(bamgmesh->SubDomains){
			Int4 i3,head,sens;
			if(verbose>3) printf("      processing SubDomains\n");
			NbSubDomains=bamgmesh->NumSubDomains;
			subdomains = new SubDomain [ NbSubDomains ];
			for (i=0;i<NbSubDomains;i++) {
				i3  =(int)bamgmesh->SubDomains[i*3+0];
				head=(int)bamgmesh->SubDomains[i*3+1]-1;//C indexing
				sens=(int)bamgmesh->SubDomains[i*3+2];
				if (i3!=23) throw ErrorException(__FUNCT__,exprintf("Bad Subdomain definition: first number should be 3"));
				if (head<0 || head>=nbt) throw ErrorException(__FUNCT__,exprintf("Bad Subdomain definition: head should in [1 %i] (triangle number)",nbt));
				subdomains[i].head = triangles+head;
			}
		}
		else{
			if(verbose>3) printf("      no SubDomains found\n");
		}

		/*Recreate geometry if needed*/
		if(1!=0) {
			printf("Warning: mesh present but no geometry found. Reconstructing...\n");
			/*Recreate geometry: */
			ConsGeometry(bamgopts->MaximalAngleOfCorner*Pi/180);
			Gh.AfterRead();
		}
	}
	/*}}}1*/

	/*Others*/
	/*FUNCTION Triangles::ConsGeometry{{{1*/
	void Triangles::ConsGeometry(Real8 cutoffradian,int *equiedges) // construct a geometry if no geo 
	  {
		//  if equiedges existe taille nbe 
		//   equiedges[i]/2 == i  original
		//   equiedges[i]/2 = j  =>   equivalence entre i et j => meme maillage
		//   equiedges[i]%2   : 0 meme sens , 1 pas meme sens 
		//       
		// --------------------------
		long int verbosity=0;
		if (verbosity>1) 
		 cout << "  -- construction of the geometry from the 2d mesh " << endl;
		if (nbt<=0 || nbv <=0 ) { MeshError(101);}

		// construction of the edges 
		//  Triangles * OldCurrentTh =CurrentTh;
		CurrentTh=this;
		//  Int4 NbTold = nbt;
		// generation of the integer coor
		// generation of the adjacence of the triangles
		if (cutoffradian>=0)
		 Gh.MaximalAngleOfCorner = cutoffradian;
		SetOfEdges4 * edge4= new SetOfEdges4(nbt*3,nbv);
		Int4 * st = new Int4[nbt*3];
		Int4 i,k;
		int j; 
		if (Gh.name) delete Gh.name;
		Gh.name = new char [ name ? strlen(name) + 15 : 50 ];
		Gh.name[0]=0;
		strcat(Gh.name,"cons from: ");
		if (name) strcat(Gh.name,name);
		else strcat(Gh.name," a mesh with no name");
		for (i=0;i<nbt*3;i++)
		 st[i]=-1;
		Int4 kk =0;

		Int4 nbeold = nbe;
		for (i=0;i<nbe;i++)
		  {
			//      cout << i << " " << Number(edges[i][0]) << " " << Number(edges[i][1]) << endl;
			edge4->addtrie(Number(edges[i][0]),Number(edges[i][1]));
		  }
		if (nbe !=  edge4->nb())
		  { 
			cerr << " Some Double edge in the mesh, the number is " << nbe 
			  << " nbe4=" << edge4->nb()  << endl;
			MeshError(1002);
		  }
		for (i=0;i<nbt;i++)
		 for  (j=0;j<3;j++)
			{
			 // Int4 i0,i1;
			 Int4 k =edge4->addtrie(Number(triangles[i][VerticesOfTriangularEdge[j][0]]),
						 Number(triangles[i][VerticesOfTriangularEdge[j][1]]));
			 Int4 invisible = triangles[i].Hidden(j);
			 if(st[k]==-1)
			  st[k]=3*i+j;
			 else if(st[k]>=0) {
				 assert( ! triangles[i].TriangleAdj(j) && !triangles[st[k] / 3].TriangleAdj((int) (st[k]%3)));

				 triangles[i].SetAdj2(j,triangles + st[k] / 3,(int) (st[k]%3));
				 if (invisible)  triangles[i].SetHidden(j);
				 if (k<nbe) {
					 triangles[i].SetLocked(j);
				 }
				 st[k]=-2-st[k]; }
			 else {
				 cerr << " The edge (" 
					<< Number(triangles[i][VerticesOfTriangularEdge[j][0]])
					<< " , " 
					<< Number(triangles[i][VerticesOfTriangularEdge[j][1]])
					<< " ) is in more than 2 triangles " <<k <<endl;
				 cerr << " Edge " << j << " Of Triangle " << i << endl;
				 cerr << " Edge " << (-st[k]+2)%3 << " Of Triangle " << (-st[k]+2)/3  << endl;
				 cerr << " Edge " << triangles[(-st[k]+2)/3].NuEdgeTriangleAdj((int)((-st[k]+2)%3))
					<< " Of Triangle " <<  Number(triangles[(-st[k]+2)/3].TriangleAdj((int)((-st[k]+2)%3))) 
					<< endl;
				 MeshError(9999);}	


			}
		Int4 nbedges = edge4->nb(); // the total number of edges 
		delete edge4;
		edge4 =0;

		if(verbosity>5) {
			if (name)
			 cout << "    On Mesh " << name << endl;
			cout << "    - The number of Vertices  = " << nbv << endl;
			cout << "    - The number of Triangles = " << nbt << endl;
			cout << "    - The number of given edge = " << nbe << endl;
			cout << "    - The number of all edges = " << nbedges << endl;
			cout << "    - The Euler number = 1-Nb Of Hole = " << nbt-nbedges+nbv << endl; }


			// check the consistant of edge[].adj and the geometrical required  vertex
			k=0;
			kk=0;
			Int4 it;

			for (i=0;i<nbedges;i++)
			 if (st[i] <-1) // edge internal
				{ 
				 it =  (-2-st[i])/3;
				 j  =  (int) ((-2-st[i])%3);
				 Triangle & tt = * triangles[it].TriangleAdj(j);
				 //cout << it << " c="  << triangles[it].color <<  " " << Number(tt) << " c="  << tt.color << endl;
				 if (triangles[it].color != tt.color|| i < nbeold) // Modif FH 06122055 // between 2 sub domai
				  k++;
				}
			 else if (st[i] >=0) // edge alone 
			  // if (i >= nbeold) 
			  kk++;

			if(verbosity>4 && (k+kk) )
			 cout << "    Nb of  ref edge " << kk+k << " (internal " << k << ")"
				<< " in file " << nbe  << endl;
			k += kk;
			kk=0;
			if (k)
			  {

				//      if (nbe) {
				//	cerr << k << " boundary edges  are not defined as edges " << endl;
				//	MeshError(9998);
				// }
				// construction of the edges 
				nbe = k;
				Edge * edgessave = edges;
				edges = new Edge[nbe];
				k =0;
				// construction of the edges 
				if(verbosity>4)
				 cout << "    Construction of the edges  " << nbe << endl;

				for (i=0;i<nbedges;i++)
				  { 
					Int4  add= -1;

					if (st[i] <-1) // edge internal
					  { 
						it =  (-2-st[i])/3;
						j  =  (int) ((-2-st[i])%3);
						Triangle & tt = * triangles[it].TriangleAdj(j);
						if (triangles[it].color !=  tt.color || i < nbeold) // Modif FH 06122055
						 add=k++;
					  }
					else if (st[i] >=0) // edge alone 
					  {
						it = st[i]/3;
						j  = (int) (st[i]%3);
						add=k++;
					  }

					if (add>=0 && add < nbe)
					  {

						edges[add].v[0] = &triangles[it][VerticesOfTriangularEdge[j][0]];
						edges[add].v[1] = &triangles[it][VerticesOfTriangularEdge[j][1]];
						edges[add].on=0; 
						if (i<nbeold) // in file edge // Modif FH 06122055 
						  {
							edges[add].ref = edgessave[i].ref; 		      
							edges[add].on = edgessave[i].on; //  HACK pour recuperer les aretes requise midf FH avril 2006 ???? 
						  }
						else
						 edges[add].ref = Min(edges[add].v[0]->ref(),edges[add].v[1]->ref()); // no a good choice
					  }
				  }
				assert(k==nbe);
				if (edgessave) delete [] edgessave;
			  }

			// construction of edges[].adj 
			for (i=0;i<nbv;i++) 
			 vertices[i].color =0;
			for (i=0;i<nbe;i++)
			 for (j=0;j<2;j++) 
			  edges[i].v[j]->color++;

			for (i=0;i<nbv;i++) 
			 vertices[i].color = (vertices[i].color ==2) ? -1 : -2;
			for (i=0;i<nbe;i++)
			 for (j=0;j<2;j++) 
				{ 
				 Vertex *v=edges[i].v[j];
				 Int4 i0=v->color,j0;
				 if(i0<0)
				  edges[i ].adj[ j ]=0;  // Add FH Jan 2008   
				 if(i0==-1)
				  v->color=i*2+j;
				 else if (i0>=0) {// i and i0 edge are adjacent by the vertex v
					 j0 =  i0%2;
					 i0 =  i0/2;
					 assert( v ==  edges[i0 ].v[j0]);
					 edges[i ].adj[ j ] =edges +i0;
					 edges[i0].adj[ j0] =edges +i ;
					 assert(edges[i0].v[j0] == v);
					 //	    if(verbosity>8)
					 //  cout << " edges adj " << i0 << " "<< j0 << " <-->  "  << i << " " << j << endl;
					 v->color = -3;}
				}
			// now reconstruct the sub domain info 
			assert(!NbSubDomains);
			NbSubDomains=0;

			  { 
				Int4 it;
				// find all the sub domain
				Int4 *colorT = new Int4[nbt];
				Triangle *tt,*t;
				Int4 k;
				for ( it=0;it<nbt;it++)
				 colorT[it]=-1;
				for (it=0;it<nbt;it++)
				  {
					if (colorT[it]<0) 
					  {
						colorT[it]=NbSubDomains;
						Int4 level =1,j,jt,kolor=triangles[it].color;
						st[0]=it; // stack 
						st[1]=0;
						k=1;
						while (level>0)
						 if( ( j=st[level]++) <3)
							{ 
							 t = &triangles[st[level-1]];
							 tt=t->TriangleAdj((int)j);

							 if ( ! t->Locked(j) && tt && (colorT[jt = Number(tt)] == -1) && ( tt->color==kolor))
								{
								 colorT[jt]=NbSubDomains;
								 st[++level]=jt;
								 st[++level]=0;
								 k++;
								}
							}
						 else 
						  level-=2;
						if (verbosity>5) 
						 cout << "   Nb of triangles " << k << " of Subdomain "  
							<<  NbSubDomains << " " << kolor << endl;
						NbSubDomains++;
					  }
				  }
				if (verbosity> 3)
				 cout << "   The Number of sub domain = " << NbSubDomains << endl;

				Int4 isd;
				subdomains = new SubDomain[NbSubDomains];
				for (isd=0;isd<NbSubDomains;isd++)
				  {
					subdomains[isd].head =0;
				  }
				k=0;
				for (it=0;it<nbt;it++)
				 for (int j=0;j<3;j++)
					{
					 tt=triangles[it].TriangleAdj(j);
					 if ((!tt || tt->color != triangles[it].color) && !subdomains[isd=colorT[it]].head)
						{
						 subdomains[isd].head = triangles+it;
						 subdomains[isd].ref =  triangles[it].color;
						 subdomains[isd].sens = j; // hack
						 subdomains[isd].edge = 0;
						 k++;
						}
					}  
				assert(k== NbSubDomains);

				delete [] colorT;


			  }      
			delete [] st;
			// now make the geometry
			// 1 compress the vertices 
			Int4 * colorV = new Int4[nbv];
			for (i=0;i<nbv;i++) 
			 colorV[i]=-1;
			for (i=0;i<nbe;i++)
			 for ( j=0;j<2;j++)
			  colorV[Number(edges[i][j])]=0;
			k=0;
			for (i=0;i<nbv;i++) 
			 if(!colorV[i])
			  colorV[i]=k++;

			Gh.nbv=k;
			Gh.nbe = nbe;
			Gh.vertices = new GeometricalVertex[k];
			Gh.edges = new GeometricalEdge[nbe];
			Gh.NbSubDomains = NbSubDomains;
			Gh.subdomains = new GeometricalSubDomain[NbSubDomains];
			if (verbosity>3)
			 cout << "    Nb of  vertices  = " << Gh.nbv << " Nb of edges = " << Gh.nbe << endl;
			NbVerticesOnGeomVertex = Gh.nbv;
			VerticesOnGeomVertex = new VertexOnGeom[NbVerticesOnGeomVertex];
			NbVerticesOnGeomEdge =0;
			VerticesOnGeomEdge =0;
			  {
				Int4 j;
				for (i=0;i<nbv;i++) 
				 if((j=colorV[i])>=0)
					{

					 Vertex & v = Gh.vertices[j];
					 v = vertices[i];
					 v.color =0;
					 VerticesOnGeomVertex[j] = VertexOnGeom(vertices[i], Gh.vertices[j]);
					}

			  }
			edge4= new SetOfEdges4(nbe,nbv);  

			Real4 * len = new Real4[Gh.nbv];
			for(i=0;i<Gh.nbv;i++)
			 len[i]=0;

			Gh.pmin =  Gh.vertices[0].r;
			Gh.pmax =  Gh.vertices[0].r;
			// recherche des extrema des vertices pmin,pmax
			for (i=0;i<Gh.nbv;i++) {
				Gh.pmin.x = Min(Gh.pmin.x,Gh.vertices[i].r.x);
				Gh.pmin.y = Min(Gh.pmin.y,Gh.vertices[i].r.y);
				Gh.pmax.x = Max(Gh.pmax.x,Gh.vertices[i].r.x);
				Gh.pmax.y = Max(Gh.pmax.y,Gh.vertices[i].r.y);
			}

			R2 DD05 = (Gh.pmax-Gh.pmin)*0.05;
			Gh.pmin -=  DD05;
			Gh.pmax +=  DD05;

			Gh.coefIcoor= (MaxICoor)/(Max(Gh.pmax.x-Gh.pmin.x,Gh.pmax.y-Gh.pmin.y));
			assert(Gh.coefIcoor >0);

			Real8 hmin = HUGE_VAL;
			int kreq=0;
			for (i=0;i<nbe;i++)
			  {
				Int4 i0 = Number(edges[i][0]);
				Int4 i1 = Number(edges[i][1]);
				Int4 j0 =	 colorV[i0];
				Int4 j1 =  colorV[i1];

				Gh.edges[i].v[0] = Gh.vertices +  j0;
				Gh.edges[i].v[1] = Gh.vertices +  j1;
				Gh.edges[i].flag = 0;
				Gh.edges[i].tg[0]=R2();
				Gh.edges[i].tg[1]=R2();
				bool requis= edges[i].on; 
				if(requis) kreq++;
				edges[i].on =  Gh.edges + i;
				if(equiedges && i < nbeold ) {
					int j=equiedges[i]/2;
					int sens=equiedges[i]%2;
					if(i!=j && equiedges[i]>=0) {
						if(verbosity>9)  
						 cout << " Edges Equi " << i << " <=> " << j << " sens = " << sens  << endl;
						if( sens==0)
						 Gh.edges[i].SetEqui();
						else 
						 Gh.edges[i].SetReverseEqui();
						Gh.edges[i].link= & Gh.edges[j];
						//assert(sens==0);//  meme sens pour l'instant
					}

				}
				if(requis)  {  // correction fevr 2009 JYU ...
					Gh.edges[i].v[0]->SetRequired();
					Gh.edges[i].v[1]->SetRequired();
					Gh.edges[i].SetRequired(); // fin modif ... 
				}
				R2 x12 = Gh.vertices[j0].r-Gh.vertices[j1].r;
				Real8 l12=Norme2(x12);        
				hmin = Min(hmin,l12);

				Gh.vertices[j1].color++;
				Gh.vertices[j0].color++;

				len[j0]+= l12;
				len[j1] += l12;
				hmin = Min(hmin,l12);

				Gh.edges[i].ref  = edges[i].ref;

				k = edge4->addtrie(i0,i1);

				assert(k == i);

			  }


			for (i=0;i<Gh.nbv;i++) 
			 if (Gh.vertices[i].color > 0) 
			  Gh.vertices[i].m=  Metric(len[i] /(Real4) Gh.vertices[i].color);
			 else 
			  Gh.vertices[i].m=  Metric(hmin);
			delete [] len;
			for (i=0;i<NbSubDomains;i++)
			  {
				Int4 it = Number(subdomains[i].head);
				int j = subdomains[i].sens;
				Int4 i0 = Number(triangles[it][VerticesOfTriangularEdge[j][0]]);
				Int4 i1 = Number(triangles[it][VerticesOfTriangularEdge[j][1]]);
				k = edge4->findtrie(i0,i1);
				if(k>=0)
				  {
					subdomains[i].sens = (vertices + i0 == edges[k].v[0]) ? 1 : -1;
					subdomains[i].edge = edges+k;
					Gh.subdomains[i].edge = Gh.edges + k;
					Gh.subdomains[i].sens  =  subdomains[i].sens;
					Gh.subdomains[i].ref =  subdomains[i].ref;
				  }
				else
				 MeshError(103);
			  }

			delete edge4;
			delete [] colorV;
			//  -- unset adj
			for (i=0;i<nbt;i++)
			 for ( j=0;j<3;j++)
			  triangles[i].SetAdj2(j,0,triangles[i].GetAllflag(j));

	  }
	/*}}}1*/
	/*FUNCTION Triangles::ProjectOnCurve{{{1*/
	GeometricalEdge*   Triangles::ProjectOnCurve( Edge & BhAB, Vertex &  vA, Vertex & vB,
				Real8 theta,Vertex & R,VertexOnEdge &  BR,VertexOnGeom & GR) {
		void *pA=0,*pB=0;
		Real8 tA=0,tB=0;
		R2 A=vA,B=vB;
		Vertex * pvA=&vA, * pvB=&vB;
		if (vA.vint == IsVertexOnVertex)
		  {
			//  cout << " Debut vertex = " << BTh.Number(vA.onbv) ;
			pA=vA.onbv;
		  }
		else if (vA.vint == IsVertexOnEdge)
		  {
			pA=vA.onbe->be;
			tA=vA.onbe->abcisse;
			// cout << " Debut edge = " << BTh.Number(vA.onbv) << " " << tA ;

		  }
		else
		  {cerr << "ProjectOnCurve On Vertex " << BTh.Number(vA) << " " << endl;
			cerr << " forget call to SetVertexFieldOnBTh" << endl;
			MeshError(-1);
		  } 

		if (vB.vint == IsVertexOnVertex)
		  {
			// cout << " Fin vertex = " << BTh.Number(vB.onbv) << endl;
			pB=vB.onbv;
		  }
		else if(vB.vint == IsVertexOnEdge)
		  {
			pB=vB.onbe->be;
			tB=vB.onbe->abcisse;
			// cout << " Fin edge = " << BTh.Number(vB.onbe->be) << " " <<  tB ;

		  }
		else
		  {cerr << "ProjectOnCurve On Vertex " << BTh.Number(vB) << " " << endl;
			cerr << " forget call to SetVertexFieldOnBTh" << endl;
			MeshError(-1);
		  } 
		Edge * e = &BhAB;
		assert( pA && pB && e);
		// be carefull the back ground edge e is on same geom edge 
		// of the initiale edge def by the 2 vertex A B;
		assert(e>=BTh.edges && e<BTh.edges+BTh.nbe);// Is a background Mesh;   
		// walk on BTh edge 
		//assert(0 /* not finish ProjectOnCurve with BackGround Mesh*/);
		// 1 first find a back ground edge contening the vertex A
		// 2 walk n back gound boundary to find the final vertex B

		if( vA.vint == IsVertexOnEdge) 
		  { // find the start edge 
			e = vA.onbe->be;	 

		  } 
		else if (vB.vint == IsVertexOnEdge) 
		  {
			theta = 1-theta;
			Exchange(tA,tB);
			Exchange(pA,pB);
			Exchange(pvA,pvB);
			Exchange(A,B);
			e =  vB.onbe->be;

			// cout << " EXCHANGE  A et B) " << endl;
		  } 
		else
		  { // do the search by walking 
			assert(0 /* A FAIRE */);
		  }

		// find the direction of walking with sens of edge and pA,PB;
		R2 AB=B-A;

		Real8 cosE01AB = (( (R2) (*e)[1] - (R2) (*e)[0] ) , AB);
		int kkk=0;
		int sens = (cosE01AB>0) ? 1 : 0;

		//   Real8 l=0; // length of the edge AB
		Real8 abscisse = -1;

		for (int cas=0;cas<2;cas++)
		  {// 2 times algo:
			//    1 for computing the length l
			//    2 for find the vertex 
			int  iii;
			Vertex  *v0=pvA,*v1; 
			Edge *neee,*eee;
			Real8 lg =0; // length of the curve 
			Real8 te0;
			// we suppose take the curve's abcisse 
			// cout << kkk << " e = " << BTh.Number(e) << "  v0=  " 
			//    << BTh.Number(v0) << " v1 = " << BTh.Number((*e)[sens]) << endl;
			for ( eee=e,iii=sens,te0=tA;
						eee && ((( void*) eee) != pB) && (( void*) (v1=&((*eee)[iii]))) != pB ;
						neee = eee->adj[iii],iii = 1-neee->Intersection(*eee),eee = neee,v0=v1,te0=1-iii )   
			  { 
				//	cout << kkk << " eee = " << BTh.Number(eee) << "  v0=  " 
				//     << BTh.Number(v0) << " v1 = " << BTh.Number(v1) << endl;

				assert(kkk++<100);
				assert(eee);
				Real8 lg0 = lg;
				Real8 dp = LengthInterpole(v0->m,v1->m,(R2) *v1 - (R2) *v0);
				lg += dp;
				if (cas && abscisse <= lg)
				  { // ok we find the geom edge 
					Real8 sss  =   (abscisse-lg0)/dp;
					Real8 thetab = te0*(1-sss)+ sss*iii;
					assert(thetab>=0 && thetab<=1);
					BR = VertexOnEdge(&R,eee,thetab);

					// cout << Number(R) << " = " <<  thetab << " on  " <<  BTh.Number(eee)
					//	 << " = " << R << endl;

					return  Gh.ProjectOnCurve(*eee,thetab,R,GR);

				  }
			  }
			// we find the end 
			if (v1 != pvB) 
			  {
				if (( void*) v1 == pB)
				 tB = iii;

				Real8 lg0 = lg;
				assert(eee);
				v1 = pvB;
				Real8 dp = LengthInterpole(v0->m,v1->m,(R2) *v1 - (R2) *v0);
				lg += dp;	
				abscisse = lg*theta;
				if (abscisse <= lg && abscisse >= lg0 ) // small optimisation we know the lenght because end
				  { // ok we find the geom edge 
					Real8 sss  =   (abscisse-lg0)/dp;
					Real8 thetab = te0*(1-sss)+ sss*tB;
					assert(thetab>=0 && thetab<=1);
					BR = VertexOnEdge(&R,eee,thetab);

					//	cout << kkk << " eee = " << BTh.Number(eee) << "  v0=  " 
					//     << BTh.Number(v0) << " " << te0
					//      << " v1 = " << BTh.Number(v1) <<  " " << tB  << endl;

					//out << Number(R) << " Opt  = " <<  thetab << " on  " <<  BTh.Number(eee) 
					//	    << " = " << R << endl;

					return  Gh.ProjectOnCurve(*eee,thetab,R,GR);
				  }
			  }
			abscisse = lg*theta;

		  }
		cerr << " Big Bug" << endl;
		MeshError(678);
		return 0; // just for the compiler 

	}                  
	/*}}}1*/
	/*FUNCTION Triangles::MakeQuadrangles{{{1*/
	void Triangles::MakeQuadrangles(double costheta){
		long int verbosity=0;

		if (verbosity>2) 
		 cout << "  -- MakeQuadrangles costheta = " << costheta << endl;
		if (verbosity>5)  
		 cout << "    (in)  Nb of Quadrilaterals = " << NbOfQuad 
			<< " Nb Of Triangles = " << nbt-NbOutT- NbOfQuad*2 
			<< " Nb of outside triangles = " << NbOutT << endl;

		if (costheta >1) {
			if (verbosity>5)
			 cout << "     do nothing costheta >1 "<< endl;
			return;}


			Int4 nbqq = (nbt*3)/2;
			DoubleAndInt4  *qq = new DoubleAndInt4[nbqq];

			Int4 i,ij;
			int j;
			Int4 k=0;
			for (i=0;i<nbt;i++)
			 for (j=0;j<3;j++)
			  if ((qq[k].q=triangles[i].QualityQuad(j))>=costheta)
				qq[k++].i3j=i*3+j;
			//  sort  qq
			HeapSort(qq,k);

			Int4 kk=0;
			for (ij=0;ij<k;ij++)
			  { 
				//      cout << qq[ij].q << " " << endl;
				i=qq[ij].i3j/3;
				j=(int) (qq[ij].i3j%3);
				// optisamition no float computation  
				if (triangles[i].QualityQuad(j,0) >=costheta) 
				 triangles[i].SetHidden(j),kk++;
			  }
			NbOfQuad = kk;
			if (verbosity>2) 
			  {
				cout << "    (out)  Nb of Quadrilaterals = " << NbOfQuad 
				  << " Nb Of Triangles = " << nbt-NbOutT- NbOfQuad*2 
				  << " Nb of outside triangles = " << NbOutT << endl;
			  }
			delete [] qq;
	}
	/*}}}1*/
	/*FUNCTION Triangles::SplitElement{{{1*/
	int  Triangles::SplitElement(int choice){
		long int verbosity=0;

		Direction NoDirOfSearch;
		const  int withBackground = &BTh != this && &BTh;
		if (verbosity>2) 
		 cout << "  -- SplitElement " << (choice? " Q->4Q and T->4T " : " Q->4Q or T->3Q " ) << endl;;
		if (verbosity>5)
		 cout << endl << "    (in)  Nb of Quadrilaterals = " << NbOfQuad 
			<< " Nb Of Triangles = " << nbt-NbOutT- NbOfQuad*2 
			<< " Nb of outside triangles = " << NbOutT << endl;

		ReNumberingTheTriangleBySubDomain();
		//int nswap =0;
		const Int4 nfortria( choice ? 4 : 6);
		if(withBackground) 
		  {
			BTh.SetVertexFieldOn();
			SetVertexFieldOnBTh();
		  }
		else
		 BTh.SetVertexFieldOn();

		Int4 newnbt=0,newnbv=0;
		Int4 * kedge = 0;
		Int4 newNbOfQuad=NbOfQuad;
		Int4 * ksplit= 0, * ksplitarray=0;
		Int4 kkk=0;
		int ret =0;
		if (nbvx<nbv+nbe) return 1;//   
		Triangles *  OCurrentTh= CurrentTh;
		CurrentTh = this;
		// 1) create  the new points by spliting the internal edges 
		// set the 
		Int4 nbvold = nbv;
		Int4 nbtold = nbt;
		Int4 NbOutTold  = NbOutT;
		Int4  NbEdgeOnGeom=0;
		Int4 i;

		nbt = nbt - NbOutT; // remove all the  the ouside triangles 
		Int4 nbtsave = nbt;
		Triangle * lastT = triangles + nbt;
		for (i=0;i<nbe;i++)
		 if(edges[i].on) NbEdgeOnGeom++;
		Int4 newnbe=nbe+nbe;
		//  Int4 newNbVerticesOnGeomVertex=NbVerticesOnGeomVertex;
		Int4 newNbVerticesOnGeomEdge=NbVerticesOnGeomEdge+NbEdgeOnGeom;
		// Int4 newNbVertexOnBThVertex=NbVertexOnBThVertex;
		Int4 newNbVertexOnBThEdge=withBackground ? NbVertexOnBThEdge+NbEdgeOnGeom:0;

		// do allocation for pointeur to the geometry and background
		VertexOnGeom * newVerticesOnGeomEdge = new VertexOnGeom[newNbVerticesOnGeomEdge];
		VertexOnEdge *newVertexOnBThEdge = newNbVertexOnBThEdge ?  new VertexOnEdge[newNbVertexOnBThEdge]:0;
		if (NbVerticesOnGeomEdge)
		 memcpy(newVerticesOnGeomEdge,VerticesOnGeomEdge,sizeof(VertexOnGeom)*NbVerticesOnGeomEdge);
		if (NbVertexOnBThEdge)
		 memcpy(newVertexOnBThEdge,VertexOnBThEdge,sizeof(VertexOnEdge)*NbVertexOnBThEdge);
		Edge *newedges = new Edge [newnbe];
		//  memcpy(newedges,edges,sizeof(Edge)*nbe);
		SetOfEdges4 * edge4= new SetOfEdges4(nbe,nbv);
		Int4 k=nbv;
		Int4 kk=0;
		Int4 kvb = NbVertexOnBThEdge;
		Int4 kvg = NbVerticesOnGeomEdge;
		Int4 ie =0;
		Edge ** edgesGtoB=0;
		if (withBackground)
		 edgesGtoB= BTh.MakeGeometricalEdgeToEdge();
		Int4 ferr=0;
		for (i=0;i<nbe;i++)
		 newedges[ie].on=0;

		for (i=0;i<nbe;i++)
		  {
			GeometricalEdge *ong =  edges[i].on;

			newedges[ie]=edges[i];
			newedges[ie].adj[0]=newedges+(edges[i].adj[0]-edges) ;
			newedges[ie].adj[1]=newedges + ie +1;
			R2 A = edges[i][0],B = edges[i][1];
			// cout << " ie = " << ie <<"  v0 = " <<  Number(newedges[ie][0]) << endl;


			kk += (i == edge4->addtrie(Number(edges[i][0]),Number(edges[i][1])));
			if (ong) // a geometrical edges 
			  { 
				if (withBackground)
				  {
					// walk on back ground mesh 
					//  newVertexOnBThEdge[ibe++] = VertexOnEdge(vertices[k],bedge,absicsseonBedge); 
					// a faire -- difficile 
					// the first PB is to now a background edge between the 2 vertices
					assert(edgesGtoB); 
					// cout << " ie = " << ie <<"  v0 = " <<  Number(newedges[ie][0]) << endl;
					ong= ProjectOnCurve(*edgesGtoB[Gh.Number(edges[i].on)],
								edges[i][0],edges[i][1],0.5,vertices[k],
								newVertexOnBThEdge[kvb],
								newVerticesOnGeomEdge[kvg++]);
					vertices[k].ReferenceNumber= edges[i].ref;
					vertices[k].DirOfSearch =   NoDirOfSearch;        
					;
					// get the Info on background mesh 
					Real8 s =        newVertexOnBThEdge[kvb];
					Vertex &  bv0  = newVertexOnBThEdge[kvb][0];
					Vertex &  bv1  = newVertexOnBThEdge[kvb][1];
					// compute the metrix of the new points 
					vertices[k].m =  Metric(1-s,bv0,s,bv1); 
					kvb++;
					// cout << " ie = " << ie <<"  v0 = " <<  Number(newedges[ie][0]) << endl;
				  }
				else 
				  {
					ong=Gh.ProjectOnCurve(edges[i],
								0.5,vertices[k],newVerticesOnGeomEdge[kvg++]);
					// vertices[k].i = toI2( vertices[k].r);
					vertices[k].ReferenceNumber = edges[i].ref;
					vertices[k].DirOfSearch = NoDirOfSearch;
					vertices[k].m =  Metric(0.5,edges[i][0],0.5,edges[i][1]);	      
				  }  
			  }
			else // straigth line edge ---
			  { 
				vertices[k].r = ((R2) edges[i][0] + (R2)  edges[i][1] )*0.5;
				vertices[k].m =  Metric(0.5,edges[i][0],0.5,edges[i][1]);
				vertices[k].on = 0;
			  }
			//vertices[k].i = toI2( vertices[k].r);
			R2 AB =  vertices[k].r;
			R2 AA = (A+AB)*0.5;
			R2 BB = (AB+B)*0.5;
			vertices[k].ReferenceNumber = edges[i].ref;
			vertices[k].DirOfSearch = NoDirOfSearch;

			newedges[ie].on = Gh.Contening(AA,ong);
			newedges[ie++].v[1]=vertices+k;

			newedges[ie]=edges[i];
			newedges[ie].adj[0]=newedges + ie -1;
			newedges[ie].adj[1]=newedges+(edges[i].adj[1]-edges) ;
			newedges[ie].on =  Gh.Contening(BB,ong);
			newedges[ie++].v[0]=vertices+k;
			// cout << " ie = " << ie-2 << " vm " << k << " v0 = " <<  Number(newedges[ie-2][0])
			//	   << " v1 = " << Number(newedges[ie-1][1])  
			//	   << " ong =" << ong-Gh.edges 
			//	   << " on 0 =" <<  newedges[ie-2].on -Gh.edges << AA
			//	   << " on 1 =" <<  newedges[ie-1].on -Gh.edges << BB 
			//	   << endl;
			k++;
		  }
		if (edgesGtoB) delete [] edgesGtoB;
		edgesGtoB=0;

		newnbv=k;
		newNbVerticesOnGeomEdge=kvg;
		if (newnbv> nbvx) goto Error;// bug 

		nbv = k;


		kedge = new Int4[3*nbt+1];
		ksplitarray = new Int4[nbt+1];
		ksplit = ksplitarray +1; // because ksplit[-1] == ksplitarray[0]

		for (i=0;i<3*nbt;i++)
		 kedge[i]=-1;

		//  

		for (i=0;i<nbt;i++)
		  {  

			Triangle & t = triangles[i];
			assert(t.link);
			for(int j=0;j<3;j++)
			  {
				const TriangleAdjacent ta = t.Adj(j);
				const Triangle & tt = ta;
				if (&tt >= lastT)
				 t.SetAdj2(j,0,0);// unset adj
				const Vertex & v0 = t[VerticesOfTriangularEdge[j][0]];
				const Vertex & v1 = t[VerticesOfTriangularEdge[j][1]];
				Int4  ke =edge4->findtrie(Number(v0),Number(v1));
				if (ke>0) 
				  {
					Int4 ii = Number(tt);
					int  jj = ta;
					Int4 ks = ke + nbvold;
					kedge[3*i+j] = ks;
					if (ii<nbt) // good triangle
					 kedge[3*ii+jj] = ks;
					Vertex &A=vertices[ks];
					Real8 aa,bb,cc,dd;
					if ((dd=Area2(v0.r,v1.r,A.r)) >=0)
					  { // warning PB roundoff error 
						if (t.link && ( (aa=Area2( A.r    , t[1].r , t[2].r )) < 0.0 
										||   (bb=Area2( t[0].r , A.r    , t[2].r )) < 0.0  
										||   (cc=Area2( t[0].r , t[1].r , A.r    )) < 0.0))
						 ferr++, cerr << " Error : " <<  ke + nbvold << " not in triangle " 
							<< i << " In=" << !!t.link
							<< " " <<  aa  << " " << bb << " " << cc << " " << dd << endl;

					  }

					else
					  {
						if (tt.link && ( (aa=Area2( A.r     , tt[1].r , tt[2].r )) < 0 
										||   (bb=Area2( tt[0].r , A.r     , tt[2].r )) < 0 
										||   (cc=Area2( tt[0].r , tt[1].r , A.r     )) < 0)) 
						 ferr++, cerr << " Warning : " <<  ke + nbvold << " not in triangle " << ii 
							<< " In=" << !!tt.link 
							<< " " <<  aa  << " " << bb << " " << cc << " " << dd << endl;

					  } 

				  }
			  }
		  }
		if(ferr)
		  {
			cerr << " Number of triangles with P2 interpolation Probleme " << ferr << endl;;
			MeshError(9);
		  }

		for (i=0;i<nbt;i++)
		  {
			ksplit[i]=1; // no split by default
			const Triangle & t = triangles[ i];
			// cout << " Triangle " << i << " " << t  << !!t.link << ":: " ;
			int nbsplitedge =0;
			int nbinvisible =0;
			int invisibleedge=0;
			int kkk[3];      
			for (int j=0;j<3;j++)
			  {
				if (t.Hidden(j)) invisibleedge=j,nbinvisible++;

				const TriangleAdjacent ta = t.Adj(j);
				const Triangle & tt = ta;


				const Vertex & v0 = t[VerticesOfTriangularEdge[j][0]];
				const Vertex & v1 = t[VerticesOfTriangularEdge[j][1]];
				//  cout << " ke = " << kedge[3*i+j]  << " " << Number(v0) << " " << Number(v1) << "/ ";
				if ( kedge[3*i+j] < 0) 
				  {
					Int4  ke =edge4->findtrie(Number(v0),Number(v1));
					//  cout << ":" << ke << "," << !!t.link << " " <<  &tt ;
					if (ke<0) // new 
					  {
						if (&tt) // internal triangles all the boundary 
						  { // new internal edges 
							Int4 ii = Number(tt);
							int  jj = ta;

							kedge[3*i+j]=k;// save the vertex number 
							kedge[3*ii+jj]=k;
							if (k<nbvx) 
							  {
								vertices[k].r = ((R2) v0+(R2) v1 )/2;
								//vertices[k].i = toI2( vertices[k].r);
								vertices[k].ReferenceNumber=0;
								vertices[k].DirOfSearch =NoDirOfSearch;
								vertices[k].m =  Metric(0.5,v0,0.5,v1);
							  }
							k++;
							kkk[nbsplitedge++]=j;		      
						  } // tt 
						else
						 cerr <<endl <<  " Bug " <<i<< " " << j << " t=" << t << endl;

					  } // ke<0	       
					else
					  { // ke >=0
						kedge[3*i+j]=nbvold+ke;
						kkk[nbsplitedge++]=j;// previously splited
					  }
				  }
				else 
				 kkk[nbsplitedge++]=j;// previously splited

			  } 
			assert (nbinvisible<2);
			// cout << " " <<  nbinvisible << " " <<  nbsplitedge << endl;
			switch (nbsplitedge) {
				case 0: ksplit[i]=10; newnbt++; break;   // nosplit
				case 1: ksplit[i]=20+kkk[0];newnbt += 2; break; // split in 2 
				case 2: ksplit[i]=30+3-kkk[0]-kkk[1];newnbt += 3; break; // split in 3 
				case 3:
						  if (nbinvisible) ksplit[i]=40+invisibleedge,newnbt += 4;
						  else   ksplit[i]=10*nfortria,newnbt+=nfortria;
						  break;
			} 
			assert(ksplit[i]>=40);
		  }
		//  now do the element split
		newNbOfQuad = 4*NbOfQuad;
		nbv = k;
		//  cout << " Nbv = " << nbv << endl;
		kkk = nbt;
		ksplit[-1] = nbt;
		// look on  old true  triangles 

		for (i=0;i<nbtsave;i++)
		  {
			//     cout << "Triangle " << i << " " << ksplit[i] << ":" << triangles[i]
			//	   << "  ----------------------------------------------- " <<endl;
			// Triangle * tc=0;
			int  nbmkadj=0;
			Int4 mkadj [100];
			mkadj[0]=i;
			Int4 kk=ksplit[i]/10;
			int  ke=(int) (ksplit[i]%10);
			assert(kk<7 && kk >0);

			// def the numbering   k (edge) i vertex 
			int k0 = ke;
			int k1 = NextEdge[k0];
			int k2 = PreviousEdge[k0];
			int i0 = OppositeVertex[k0];
			int i1 = OppositeVertex[k1];
			int i2 = OppositeVertex[k2];

			Triangle &t0=triangles[i];
			Vertex * v0=t0(i0);           
			Vertex * v1=t0(i1);           
			Vertex * v2=t0(i2);

			// cout << "nbmkadj " << nbmkadj << " it=" << i <<endl;
			assert(nbmkadj< 10);
			// --------------------------
			TriangleAdjacent ta0(t0.Adj(i0)),ta1(t0.Adj(i1)),ta2(t0.Adj(i2));
			// save the flag Hidden
			int hid[]={t0.Hidden(0),t0.Hidden(1),t0.Hidden(2)};
			// un set all adj -- save Hidden flag --
			t0.SetAdj2(0,0,hid[0]);
			t0.SetAdj2(1,0,hid[1]);
			t0.SetAdj2(2,0,hid[2]);
			// --  remake 
			switch  (kk) {
				case 1: break;// nothing 
				case 2: // 
						  {
							Triangle &t1=triangles[kkk++];
							t1=t0;
							assert (kedge[3*i+i0]>=0);
							Vertex * v3 = vertices + kedge[3*i+k0];

							t0(i2) = v3;
							t1(i1) = v3;
							t0.SetAllFlag(k2,0);
							t1.SetAllFlag(k1,0);
						  } 
						break; 
				case 3: //
						  {
							Triangle &t1=triangles[kkk++];
							Triangle &t2=triangles[kkk++];
							t2=t1=t0;
							assert (kedge[3*i+k1]>=0);
							assert (kedge[3*i+k2]>=0);

							Vertex * v01 = vertices + kedge[3*i+k2];
							Vertex * v02 = vertices + kedge[3*i+k1]; 
							t0(i1) = v01; 
							t0(i2) = v02; 
							t1(i2) = v02;
							t1(i0) = v01; 
							t2(i0) = v02; 
							t0.SetAllFlag(k0,0);
							t1.SetAllFlag(k1,0);
							t1.SetAllFlag(k0,0);
							t2.SetAllFlag(k2,0);
						  } 
						break;
				case 4: // 
				case 6: // split in 4 
						  {
							Triangle &t1=triangles[kkk++];
							Triangle &t2=triangles[kkk++];
							Triangle &t3=triangles[kkk++];
							t3=t2=t1=t0;
							assert(kedge[3*i+k0] >=0 && kedge[3*i+k1] >=0 && kedge[3*i+k2] >=0);
							Vertex * v12 = vertices + kedge[3*i+k0];
							Vertex * v02 = vertices + kedge[3*i+k1]; 
							Vertex * v01 = vertices + kedge[3*i+k2];
							// cout << Number(t0(i0))  << " " << Number(t0(i1)) 
							//     << " " <<  Number(t0(i2)) 
							//     << " " <<  kedge[3*i+k0] 
							//     << " " <<  kedge[3*i+k1] 
							//     << " " <<  kedge[3*i+k2] << endl;
							t0(i1) = v01;
							t0(i2) = v02;
							t0.SetAllFlag(k0,hid[k0]);

							t1(i0) = v01;
							t1(i2) = v12;
							t0.SetAllFlag(k1,hid[k1]);

							t2(i0) = v02;
							t2(i1) = v12;
							t2.SetAllFlag(k2,hid[k2]);

							t3(i0) = v12;
							t3(i1) = v02;
							t3(i2) = v01;

							t3.SetAllFlag(0,hid[0]);	   
							t3.SetAllFlag(1,hid[1]);	   
							t3.SetAllFlag(2,hid[2]);

							if ( kk == 6)
							  {

								Triangle &t4=triangles[kkk++];
								Triangle &t5=triangles[kkk++];

								t4 = t3;
								t5 = t3;

								t0.SetHidden(k0);
								t1.SetHidden(k1);
								t2.SetHidden(k2);
								t3.SetHidden(0);
								t4.SetHidden(1);
								t5.SetHidden(2);

								if (nbv < nbvx ) 
								  {
									vertices[nbv].r = ((R2) *v01 + (R2) *v12  + (R2) *v02 ) / 3.0;
									vertices[nbv].ReferenceNumber =0;
									vertices[nbv].DirOfSearch =NoDirOfSearch;
									//vertices[nbv].i = toI2(vertices[nbv].r);
									Real8 a3[]={1./3.,1./3.,1./3.};
									vertices[nbv].m = Metric(a3,v0->m,v1->m,v2->m);
									Vertex * vc =  vertices +nbv++;
									t3(i0) = vc;
									t4(i1) = vc;
									t5(i2) = vc;

								  }
								else
								 goto Error; 
							  }

						  } 
						break;         
			}

			// cout << "  -- " << i << " " << nbmkadj << " " << kkk << " " << tc << endl;
			//  t0.SetDetf();
			// save all the new triangles
			mkadj[nbmkadj++]=i;
			Int4 jj;
			if (t0.link) 
			 for (jj=nbt;jj<kkk;jj++)
				{
				 triangles[jj].link=t0.link;
				 t0.link= triangles+jj;
				 mkadj[nbmkadj++]=jj;
				 // triangles[jj].SetDet();
				}
			// cout << "  -- " << i << " " << nbmkadj << endl;
			assert(nbmkadj<=13);// 13 = 6 + 4 + 3

			if (kk==6)  newNbOfQuad+=3;
			//	 triangles[i].Draw();       

			for (jj=ksplit[i-1];jj<kkk;jj++)
			 // triangles[jj].SetDet();
			 //	   triangles[jj].Draw();



			 nbt = kkk;
			ksplit[i]= nbt; // save last adresse of the new triangles
			kkk = nbt;

		  }

		//  cout << " nv = " << nbv << " nbt = " << nbt << endl;
		for (i=0;i<nbv;i++)
		 vertices[i].m = vertices[i].m*2.;
		//
		if(withBackground)
		 for (i=0;i<BTh.nbv;i++)
		  BTh.vertices[i].m =  BTh.vertices[i].m*2.;


		ret = 2;
		if (nbt>= nbtx) goto Error; // bug 
		if (nbv>= nbvx) goto Error; // bug 
		// generation of the new triangles 

		SetIntCoor("In SplitElement"); 

		ReMakeTriangleContainingTheVertex();
		if(withBackground)
		 BTh.ReMakeTriangleContainingTheVertex();

		delete [] edges;
		edges = newedges;
		nbe = newnbe;
		NbOfQuad = newNbOfQuad;

		for (i=0;i<NbSubDomains;i++)
		  { 
			Int4 k = subdomains[i].edge- edges;
			subdomains[i].edge =  edges+2*k; // spilt all edge in 2 
		  }

		if (ksplitarray) delete [] ksplitarray;
		if (kedge) delete [] kedge;
		if (edge4) delete edge4;
		if (VerticesOnGeomEdge) delete [] VerticesOnGeomEdge;
		VerticesOnGeomEdge= newVerticesOnGeomEdge;
		if(VertexOnBThEdge) delete []  VertexOnBThEdge;
		VertexOnBThEdge = newVertexOnBThEdge;
		NbVerticesOnGeomEdge = newNbVerticesOnGeomEdge;
		NbVertexOnBThEdge=newNbVertexOnBThEdge;
		//  ReMakeTriangleContainingTheVertex();

		FillHoleInMesh();

		if (verbosity>2)
		 cout << "    (out) Nb of Quadrilaterals = " << NbOfQuad 
			<< " Nb Of Triangles = " << nbt-NbOutT- NbOfQuad*2 
			<< " Nb of outside triangles = " << NbOutT << endl;

		CurrentTh=OCurrentTh;
		return 0; //ok

Error:
		nbv = nbvold;
		nbt = nbtold;
		NbOutT = NbOutTold;
		// cleaning memory ---
		delete newedges;
		if (ksplitarray) delete [] ksplitarray;
		if (kedge) delete [] kedge;
		if (newVerticesOnGeomEdge) delete [] newVerticesOnGeomEdge;
		if (edge4) delete edge4;
		if(newVertexOnBThEdge) delete []  newVertexOnBThEdge;


		CurrentTh= OCurrentTh;
		return ret; // ok 
	}
	/*}}}1*/

} // end of namespace bamg 
