#include "../shared/shared.h"
#include "../include/macros.h"
#include "../toolkits/toolkits.h"

#include <cstdio>
#include <cstring>
#include <cmath>
#include <ctime>
#include "Mesh2.h"
#include "QuadTree.h"
#include "SetOfE4.h"

namespace bamg {

	static const  Direction NoDirOfSearch=Direction();

	/*Constructors/Destructors*/
	/*FUNCTION Triangles::Triangles(BamgMesh* bamgmesh, BamgOpts* bamgopts){{{1*/
	Triangles::Triangles(BamgMesh* bamgmesh, BamgOpts* bamgopts):Gh(*(new Geometry())),BTh(*this){ 

		PreInit(0,"none");
		OnDisk = 1;

		ReadFromMatlabMesh(bamgmesh,bamgopts);
		SetIntCoor();
		FillHoleInMesh();
	}
	/*}}}1*/

	/*IO*/
	/*FUNCTION Triangles::WriteMesh {{{1*/
	void Triangles::WriteMesh(BamgMesh* bamgmesh,BamgOpts* bamgopts){

		int i,j;
		int verbose;

		verbose=bamgopts->verbose;
		Int4 *reft = new Int4[nbt];
		Int4 nbInT = ConsRefTriangle(reft);

		//Vertices
		if(verbose>3) printf("      writing Vertices\n");
		bamgmesh->NumVertices=nbv;
		xfree((void**)&bamgmesh->Vertices);
		if (nbv){
			bamgmesh->Vertices=(double*)xmalloc(3*nbv*sizeof(double));
			for (i=0;i<nbv;i++){
				bamgmesh->Vertices[i*3+0]=vertices[i].r.x;
				bamgmesh->Vertices[i*3+1]=vertices[i].r.y;
				bamgmesh->Vertices[i*3+2]=vertices[i].ref();
			}
		}
		else{
			bamgmesh->Vertices=NULL;
		}

		//Edges
		if(verbose>3) printf("      writing Edges\n");
		bamgmesh->NumEdges=nbe;
		xfree((void**)&bamgmesh->Edges);
		if (nbe){
			bamgmesh->Edges=(double*)xmalloc(3*nbe*sizeof(double));
			for (i=0;i<nbe;i++){
				bamgmesh->Edges[i*3+0]=Number(edges[i][0])+1; //back to M indexing
				bamgmesh->Edges[i*3+1]=Number(edges[i][1])+1; //back to M indexing
				bamgmesh->Edges[i*3+2]=edges[i].ref;
			}
		}
		else{
			bamgmesh->Edges=NULL;
		}

		//CrackedEdges
		if(verbose>3) printf("      writing CrackedEdges\n");
		bamgmesh->NumCrackedEdges=NbCrackedEdges;
		xfree((void**)&bamgmesh->CrackedEdges);
		if (NbCrackedEdges){
			bamgmesh->CrackedEdges=(double*)xmalloc(2*NbCrackedEdges*sizeof(double));
			for (i=0;i<NbCrackedEdges;i++){
				bamgmesh->CrackedEdges[i*2+0]=Number(CrackedEdges[i].a.edge)+1; //back to M indexing
				bamgmesh->CrackedEdges[i*2+1]=Number(CrackedEdges[i].b.edge)+1; //back to M indexing
			}
		}
		else{
			bamgmesh->CrackedEdges=NULL;
		}

		//Triangles
		if(verbose>3) printf("      writing Triangles\n");
		Int4 k=nbInT-NbOfQuad*2;
		Int4 num=0;
		bamgmesh->NumTriangles=k;
		xfree((void**)&bamgmesh->Triangles);
		if (k){
			bamgmesh->Triangles=(double*)xmalloc(4*k*sizeof(double));
			for (i=0;i<nbt;i++){
				Triangle &t=triangles[i];
				if (reft[i]>=0 && !( t.Hidden(0) || t.Hidden(1) || t.Hidden(2) )){
					bamgmesh->Triangles[num*4+0]=Number(t[0])+1; //back to M indexing
					bamgmesh->Triangles[num*4+1]=Number(t[1])+1; //back to M indexing
					bamgmesh->Triangles[num*4+2]=Number(t[2])+1; //back to M indexing
					bamgmesh->Triangles[num*4+3]=subdomains[reft[i]].ref;
					num=num+1;
				}
			}
		}
		else{
			bamgmesh->Triangles=NULL;
		}

		//Quadrilaterals
		if(verbose>3) printf("      writing Quadrilaterals\n");
		bamgmesh->NumQuadrilaterals=NbOfQuad;
		xfree((void**)&bamgmesh->Quadrilaterals);
		if (NbOfQuad){
			bamgmesh->Quadrilaterals=(double*)xmalloc(5*NbOfQuad*sizeof(double));
			for (i=0;i<nbt;i++){
				Triangle &t =triangles[i];
				Triangle* ta;
				Vertex *v0,*v1,*v2,*v3;
				if (reft[i]<0) continue;
				if ((ta=t.Quadrangle(v0,v1,v2,v3)) !=0 && &t<ta) { 
					bamgmesh->Quadrilaterals[i*5+0]=Number(v0)+1; //back to M indexing
					bamgmesh->Quadrilaterals[i*5+1]=Number(v1)+1; //back to M indexing
					bamgmesh->Quadrilaterals[i*5+2]=Number(v2)+1; //back to M indexing
					bamgmesh->Quadrilaterals[i*5+3]=Number(v3)+1; //back to M indexing
					bamgmesh->Quadrilaterals[i*5+4]=subdomains[reft[i]].ref;
				}
			}
		}
		else{
			bamgmesh->Quadrilaterals=NULL;
		}

		//SubDomains
		if(verbose>3) printf("      writing SubDomains\n");
		bamgmesh->NumSubDomains=NbSubDomains;
		xfree((void**)&bamgmesh->SubDomains);
		if (NbSubDomains){
			bamgmesh->SubDomains=(double*)xmalloc(4*NbSubDomains*sizeof(double));
			for (i=0;i<NbSubDomains;i++){
				bamgmesh->SubDomains[i*4+0]=3;
				bamgmesh->SubDomains[i*4+1]=reft[Number(subdomains[i].head)];
				bamgmesh->SubDomains[i*4+2]=1;
				bamgmesh->SubDomains[i*4+3]=subdomains[i].ref;
			}
		}
		else{
			bamgmesh->SubDomains=NULL;
		}

		//SubDomainsFromGeom
		if(verbose>3) printf("      writing SubDomainsFromGeom\n");
		bamgmesh->NumSubDomainsFromGeom=Gh.NbSubDomains;
		xfree((void**)&bamgmesh->SubDomainsFromGeom);
		if (Gh.NbSubDomains){
			bamgmesh->SubDomainsFromGeom=(double*)xmalloc(4*Gh.NbSubDomains*sizeof(double));
			for (i=0;i<Gh.NbSubDomains;i++){
				bamgmesh->SubDomainsFromGeom[i*4+0]=2;
				bamgmesh->SubDomainsFromGeom[i*4+1]=Number(subdomains[i].edge)+1; //back to Matlab indexing
				bamgmesh->SubDomainsFromGeom[i*4+2]=subdomains[i].sens;
				bamgmesh->SubDomainsFromGeom[i*4+3]=Gh.subdomains[i].ref;
			}
		}
		else{
			bamgmesh->SubDomainsFromGeom=NULL;
		}

		//VerticesOnGeomVertex
		if(verbose>3) printf("      writing VerticesOnGeometricVertex\n");
		bamgmesh->NumVerticesOnGeometricVertex=NbVerticesOnGeomVertex;
		xfree((void**)&bamgmesh->VerticesOnGeometricVertex);
		if (NbVerticesOnGeomVertex){
			bamgmesh->VerticesOnGeometricVertex=(double*)xmalloc(2*NbVerticesOnGeomVertex*sizeof(double));
			for (i=0;i<NbVerticesOnGeomVertex;i++){
				VertexOnGeom &v=VerticesOnGeomVertex[i];
				if (!v.OnGeomVertex()){
					throw ErrorException(__FUNCT__,exprintf("A vertices supposed to be OnGeometricVertex is actually not"));
				}
				bamgmesh->VerticesOnGeometricVertex[i*2+0]=Number((Vertex*)v)+1; //back to Matlab indexing
				bamgmesh->VerticesOnGeometricVertex[i*2+1]=Gh.Number(( GeometricalVertex*)v)+1; //back to Matlab indexing
			}
		}
		else{
			bamgmesh->VerticesOnGeometricVertex=NULL;
		}

		//VertexOnGeometricEdge
		if(verbose>3) printf("      writing VerticesOnGeometricEdge\n");
		bamgmesh->NumVerticesOnGeometricEdge=NbVerticesOnGeomEdge;
		xfree((void**)&bamgmesh->VerticesOnGeometricEdge);
		if (NbVerticesOnGeomEdge){
			bamgmesh->VerticesOnGeometricEdge=(double*)xmalloc(3*NbVerticesOnGeomEdge*sizeof(double));
			for (i=0;i<NbVerticesOnGeomEdge;i++){
				const VertexOnGeom &v=VerticesOnGeomEdge[i];
				if (!v.OnGeomEdge()){
					throw ErrorException(__FUNCT__,exprintf("A vertices supposed to be OnGeometricEdge is actually not"));
				}
				bamgmesh->VerticesOnGeometricEdge[i*3+0]=Number((Vertex*)v)+1; //back to Matlab indexing
				bamgmesh->VerticesOnGeometricEdge[i*3+1]=Gh.Number((const GeometricalEdge*)v)+1; //back to Matlab indexing
				bamgmesh->VerticesOnGeometricEdge[i*3+2]=(double)v;
			}
		}
		else{
			bamgmesh->VerticesOnGeometricEdge=NULL;
		}

		//EdgesOnGeometricEdge
		if(verbose>3) printf("      writing EdgesOnGeometricEdge\n");
		k=0;
		for (i=0;i<nbe;i++){
			if (edges[i].on) k=k+1;
		}
		bamgmesh->NumEdgesOnGeometricEdge=k;
		xfree((void**)&bamgmesh->EdgesOnGeometricEdge);
		if (k){
			bamgmesh->EdgesOnGeometricEdge=(double*)xmalloc(2*(int)k*sizeof(double));
			int count=0;
			for (i=0;i<nbe;i++){
				if (edges[i].on){
					bamgmesh->EdgesOnGeometricEdge[count*2+0]=(double)i+1; //back to Matlab indexing
					bamgmesh->EdgesOnGeometricEdge[count*2+1]=(double)Gh.Number(edges[i].on)+1; //back to Matlab indexing
					count=count+1;
				}
			}
		}
		else{
			bamgmesh->EdgesOnGeometricEdge=NULL;
		}
	}
	/*}}}1*/
	/*FUNCTION Triangles::ReadFromMatlabMesh{{{1*/
	void Triangles::ReadFromMatlabMesh(BamgMesh* bamgmesh, BamgOpts* bamgopts){

		int verbose;
		Real8 Hmin = HUGE_VAL;// the infinie value 
		Int4 i1,i2,i3,iref;
		Int4 i,j;
		Int4 hvertices =0;
		Int4 ifgeom=0;
		Metric M1(1);
		int Version=1,dim=2;

		verbose=bamgopts->verbose;

		nbv=bamgmesh->NumVertices;
		nbvx=nbv;
		nbt=bamgmesh->NumTriangles;
		nbiv=nbvx;

		//Vertices
		if(bamgmesh->Vertices){
			if(verbose>3) printf("      processing Vertices\n");

			vertices=(Vertex*)xmalloc(nbv*sizeof(Vertex));
			ordre=(Vertex**)xmalloc(nbv*sizeof(Vertex*));

			for (i=0;i<nbv;i++){
				vertices[i].r.x=bamgmesh->Vertices[i*3+0];
				vertices[i].r.y=bamgmesh->Vertices[i*3+1];
				vertices[i].ReferenceNumber=1;
				vertices[i].DirOfSearch =NoDirOfSearch;
				vertices[i].m=M1;
				vertices[i].color=(Int4)bamgmesh->Vertices[i*3+2];
			}
			nbtx=2*nbvx-2; // for filling The Holes and quadrilaterals 
		}
		else{
			if(verbose>3) throw ErrorException(__FUNCT__,exprintf("no Vertices found in the initial mesh"));
		}

		//Triangles
		if(bamgmesh->Triangles){
			if(verbose>3) printf("      processing Triangles\n");
			triangles =new Triangle[nbtx]; //we cannot allocate only nbt triangles since 
			//other triangles will be added for each edge
			for (i=0;i<nbt;i++){
				Triangle & t = triangles[i];
				i1=(Int4)bamgmesh->Triangles[i*4+0]-1; //for C indexing
				i2=(Int4)bamgmesh->Triangles[i*4+1]-1; //for C indexing
				i3=(Int4)bamgmesh->Triangles[i*4+2]-1; //for C indexing
				t=Triangle(this,i1,i2,i3);
				t.color=(Int4)bamgmesh->Triangles[i*4+3];
			}
		}
		else{
			if(verbose>3) throw ErrorException(__FUNCT__,exprintf("no Triangles found in the initial mesh"));
		}

		//Quadrilaterals
		if(bamgmesh->Quadrilaterals){
			if(verbose>3) printf("      processing Quadrilaterals\n");
			Int4 i1,i2,i3,i4,iref;
			triangles =new Triangle[nbt];
			for (i=0;i<bamgmesh->NumQuadrilaterals;i++){
				Triangle & t1 = triangles[2*i];
				Triangle & t2 = triangles[2*i+1];
				i1=(Int4)bamgmesh->Quadrilaterals[i*4+0]-1; //for C indexing
				i2=(Int4)bamgmesh->Quadrilaterals[i*4+1]-1; //for C indexing
				i3=(Int4)bamgmesh->Quadrilaterals[i*4+2]-1; //for C indexing
				i4=(Int4)bamgmesh->Quadrilaterals[i*4+3]-1; //for C indexing
				t1=Triangle(this,i1,i2,i3);
				t2=Triangle(this,i3,i4,i1);
				t1.color=1; //reference = 1 for all triangles since it has not been specified
				t2.color=1; //reference = 1 for all triangles since it has not been specified
				t1.SetHidden(OppositeEdge[1]); // two times  because the adj was not created 
				t2.SetHidden(OppositeEdge[1]); 
			}
		}
		else{
			if(verbose>3) printf("      no Quadrilaterals found\n");
		}

		//hVertices
		if(bamgmesh->hVertices){
			if(verbose>3) printf("      processing hVertices\n");
			hvertices=1;
			for (i=0;i< nbv;i++){
				vertices[i].m=Metric((Real4)bamgmesh->hVertices[i]);
			}
		}
		else{
			if(verbose>3) printf("      no hVertices found\n");
		}

		//VerVerticesOnGeometricEdge
		if(bamgmesh->VerticesOnGeometricEdge){
			if(verbose>3) printf("      processing VerticesOnGeometricEdge\n");
			NbVerticesOnGeomEdge=bamgmesh->NumVerticesOnGeometricEdge;
			VerticesOnGeomEdge= new  VertexOnGeom[NbVerticesOnGeomEdge] ;
			for (i=0;i<NbVerticesOnGeomEdge;i++){
				Int4  i1,i2;
				Real8 s;
				i1=(Int4)bamgmesh->VerticesOnGeometricEdge[i*3+0]-1; //for C indexing
				i2=(Int4)bamgmesh->VerticesOnGeometricEdge[i*3+1]-1; //for C indexing
				s =(Int4)bamgmesh->VerticesOnGeometricEdge[i*3+2];
				VerticesOnGeomEdge[i]=VertexOnGeom(vertices[i1],Gh.edges[i2],s);
			}
		}
		else{
			if(verbose>3) printf("      no VertexOnGeometricEdge found\n");
		}

		//Edges
		if (bamgmesh->Edges){
			int i1,i2;
			R2 zero2(0,0);
			Real4 *len =0;

			if(verbose>3) printf("      processing Edges\n");
			nbe=bamgmesh->NumEdges;
			edges = new Edge[nbe];

			if (!hvertices) {
				len = new Real4[nbv];
				for(i=0;i<nbv;i++)
				 len[i]=0;
			}

			for (i=0;i<nbe;i++){
				i1=(int)bamgmesh->Edges[i*3+0]-1; //-1 for C indexing
				i2=(int)bamgmesh->Edges[i*3+1]-1; //-1 for C indexing
				edges[i].v[0]= vertices +i1;
				edges[i].v[1]= vertices +i2;
				edges[i].adj[0]=0;
				edges[i].adj[1]=0;
				R2 x12 = vertices[i2].r-vertices[i1].r;
				Real8 l12=sqrt( (x12,x12));        

				if (!hvertices) {
					vertices[i1].color++;
					vertices[i2].color++;
					len[i1]+= l12;
					len[i2] += l12;
				}
				Hmin = Min(Hmin,l12);
			}

			// definition  the default of the given mesh size 
			if (!hvertices){
				for (i=0;i<nbv;i++) 
				 if (vertices[i].color > 0) 
				  vertices[i].m=  Metric(len[i] /(Real4) vertices[i].color);
				 else 
				  vertices[i].m=  Metric(Hmin);
				delete [] len;
			}

			// construction of edges[].adj 
			for (i=0;i<nbv;i++){ 
				vertices[i].color = (vertices[i].color ==2) ? -1 : -2;
			}
			for (i=0;i<nbe;i++){
				for (j=0;j<2;j++) { 
					Vertex *v=edges[i].v[j];
					Int4 i0=v->color,j0;
					if(i0==-1){
						v->color=i*2+j;
					}
					else if (i0>=0) {// i and i0 edge are adjacent by the vertex v
						j0 =  i0%2;
						i0 =  i0/2;
						assert( v ==  edges[i0 ].v[j0]);
						edges[i ].adj[ j ] =edges +i0;
						edges[i0].adj[ j0] =edges +i ;
						assert(edges[i0].v[j0] == v);
						v->color = -3;
					}
				}
			}
		}
		else{
			if(verbose>3) printf("      no Edges found\n");
		}

		//EdgeOnGeometricEdge
		if(bamgmesh->EdgesOnGeometricEdge){
			if(verbose>3) printf("      processing EdgesOnGeometricEdge\n");
			int i1,i2,i,j;
			i2=bamgmesh->NumEdgesOnGeometricEdge;
			for (i1=0;i1<i2;i1++) {
				i=(int)bamgmesh->EdgesOnGeometricEdge[i*2+0];
				j=(int)bamgmesh->EdgesOnGeometricEdge[i*2+1];
				if(!(i>0 && j >0 && i <= nbe && j <= Gh.nbe)) {
					throw ErrorException(__FUNCT__,exprintf("EdgesOnGeometricEdge error: We must have : (i>0 && j >0 && i <= nbe && j <= Gh.nbe)"));
				}
				edges[i-1].on = Gh.edges + j-1;
			}
		}
		else{
			if(verbose>3) printf("      no EdgesOnGeometricEdge found\n");
		}

		//SubDomain
		if(bamgmesh->SubDomains){
			Int4 i3,head,sens;
			if(verbose>3) printf("      processing SubDomains\n");
			NbSubDomains=bamgmesh->NumSubDomains;
			subdomains = new SubDomain [ NbSubDomains ];
			for (i=0;i<NbSubDomains;i++) {
				i3  =(int)bamgmesh->SubDomains[i*3+0];
				head=(int)bamgmesh->SubDomains[i*3+1]-1;//C indexing
				sens=(int)bamgmesh->SubDomains[i*3+2];
				if (i3!=23) throw ErrorException(__FUNCT__,exprintf("Bad Subdomain definition: first number should be 3"));
				if (head<0 || head>=nbt) throw ErrorException(__FUNCT__,exprintf("Bad Subdomain definition: head should in [1 %i] (triangle number)",nbt));
				subdomains[i].head = triangles+head;
			}
		}
		else{
			if(verbose>3) printf("      no SubDomains found\n");
		}

		/*Recreate geometry if needed*/
		if(1!=0) {
			printf("Warning: mesh present but no geometry found. Reconstructing...\n");
			/*Recreate geometry: */
			ConsGeometry(bamgopts->MaximalAngleOfCorner*Pi/180);
			Gh.AfterRead();
		}
	}
	/*}}}1*/

} // end of namespace bamg 
