#include "../shared/shared.h"
#include "../include/macros.h"
#include "../toolkits/toolkits.h"

#include <cstdio>
#include <cstring>
#include <cmath>
#include <ctime>
#include "Mesh2.h"
#include "QuadTree.h"
#include "SetOfE4.h"


namespace bamg {

	static const  Direction NoDirOfSearch=Direction();

	/*Constructors/Destructors*/

	/*IO*/
	/*FUNCTION Geometry::WriteGeometry{{{1*/
	void Geometry::WriteGeometry(BamgGeom* bamggeom, BamgOpts* bamgopts){

		int verbose;
		int nbreq=0;
		int nbreqv=0;
		int nbtan=0;
		int nbcracked=0;
		int i,count;

		verbose=bamgopts->verbose;

		//Vertices
		if(verbose>3) printf("      writing Vertices\n");
		bamggeom->NumVertices=nbv;
		xfree((void**)&bamggeom->Vertices);
		if (nbv){
			bamggeom->Vertices=(double*)xmalloc(3*nbv*sizeof(double));
			for (i=0;i<nbv;i++){
				bamggeom->Vertices[i*3+0]=vertices[i].r.x;
				bamggeom->Vertices[i*3+1]=vertices[i].r.y;
				bamggeom->Vertices[i*3+2]=vertices[i].ref();

				//update counters
				if (vertices[i].Required()) nbreqv++;
			}
		}
		else{
			bamggeom->Vertices=NULL;
		}

		//Edges
		if(verbose>3) printf("      writing Edges\n");
		bamggeom->NumEdges=nbe;
		xfree((void**)&bamggeom->Edges);
		if (nbe){
			bamggeom->Edges=(double*)xmalloc(3*nbe*sizeof(double));
			for (i=0;i<nbe;i++){
				bamggeom->Edges[i*3+0]=Number(edges[i][0])+1; //back to Matlab indexing
				bamggeom->Edges[i*3+1]=Number(edges[i][1])+1; //back to Matlab indexing
				//bamggeom->Edges[i*3+2]=(double)edges[i].ref(); //TEST does not compile???
				bamggeom->Edges[i*3+2]=1; //TEST for now

				//update counters
				if (edges[i].Required()) nbreq++;
				if (edges[i].Cracked()){
					if (i<=Number(edges[i].link)) nbcracked++;
				}
				if (edges[i].TgA() && edges[i][0].Corner()) nbtan++;
				if (edges[i].TgB() && edges[i][1].Corner()) nbtan++;
			}
		}
		else{
			bamggeom->Edges=NULL;
		}

		//CrackedEdges
		if(verbose>3) printf("      writing CrackedEdges\n");
		bamggeom->NumCrackedEdges=nbcracked;
		xfree((void**)&bamggeom->CrackedEdges);
		if (nbcracked){
			bamggeom->CrackedEdges=(double*)xmalloc(2*nbcracked*sizeof(double));
			count=0;
			for (i=0;i<nbe;i++){
				if (edges[i].Cracked()){
					if (i<=Number(edges[i].link)){
						bamggeom->CrackedEdges[count*2+0]=i+1;                     //back to Matlab indexing
						bamggeom->CrackedEdges[count*2+1]=Number(edges[i].link)+1; //back to Matlab indexing
						count=count+1;
					}
				}
			}
		}
		else{
			bamggeom->CrackedEdges=NULL;
		}

		//RequiredEdges
		if(verbose>3) printf("      writing RequiredEdges\n");
		bamggeom->NumRequiredEdges=nbreq;
		xfree((void**)&bamggeom->RequiredEdges);
		if (nbreq){
			bamggeom->RequiredEdges=(double*)xmalloc(1*nbreq*sizeof(double));
			count=0;
			for (i=0;i<nbe;i++){
				if (edges[i].Required()){
					bamggeom->RequiredEdges[count]=i+1; //back to Matlab indexing
					count=count+1;
				}
			}
		}
		else{
			bamggeom->RequiredEdges=NULL;
		}

		//No corners

		//RequiredVertices
		if(verbose>3) printf("      writing RequiredVertices\n");
		bamggeom->NumRequiredVertices=nbreqv;
		xfree((void**)&bamggeom->RequiredVertices);
		if (nbreqv){
			bamggeom->RequiredVertices=(double*)xmalloc(1*nbreqv*sizeof(double));
			count=0;
			for (i=0;i<nbe;i++){
				if (vertices[i].Required()){
					bamggeom->RequiredVertices[count]=i+1; //back to Matlab indexing
					count=count+1;
				}
			}
		}
		else{
			bamggeom->RequiredVertices=NULL;
		}

		//SubDomains
		if(verbose>3) printf("      writing SubDomains\n");
		bamggeom->NumSubDomains=NbSubDomains;
		xfree((void**)&bamggeom->SubDomains);
		if (NbSubDomains){
			bamggeom->SubDomains=(double*)xmalloc(4*NbSubDomains*sizeof(double));
			for (i=0;i<NbSubDomains;i++){
				bamggeom->SubDomains[4*i+0]=2;
				bamggeom->SubDomains[4*i+1]=Number(subdomains[i].edge)+1; //back to Matlab indexing
				bamggeom->SubDomains[4*i+2]=subdomains[i].sens;
				bamggeom->SubDomains[4*i+3]=subdomains[i].ref;
			}
		}
		else{
			bamggeom->SubDomains=NULL;
		}

		//TangentAtEdges
		if(verbose>3) printf("      writing TangentAtEdges\n");
		bamggeom->NumTangentAtEdges=nbtan;
		xfree((void**)&bamggeom->TangentAtEdges);
		if (nbtan){
			bamggeom->TangentAtEdges=(double*)xmalloc(4*nbtan*sizeof(double));
			count=0;
			for (i=0;i<nbe;i++){
				if (edges[i].TgA() && edges[i][0].Corner()){
					bamggeom->TangentAtEdges[4*i+0]=i+1; //back to Matlab indexing
					bamggeom->TangentAtEdges[4*i+1]=1;
					bamggeom->TangentAtEdges[4*i+2]=edges[i].tg[0].x;
					bamggeom->TangentAtEdges[4*i+3]=edges[i].tg[0].y;
				}
				if (edges[i].TgB() && edges[i][1].Corner()){
					bamggeom->TangentAtEdges[4*i+0]=i+1; //back to Matlab indexing
					bamggeom->TangentAtEdges[4*i+1]=2;
					bamggeom->TangentAtEdges[4*i+2]=edges[i].tg[1].x;
					bamggeom->TangentAtEdges[4*i+3]=edges[i].tg[2].y;
				}
				count=count+1;
			}
		}
		else{
			bamggeom->TangentAtEdges=NULL;
		}
	}
	/*}}}1*/
	/*FUNCTION Geometry::ReadGeometry{{{1*/
	void Geometry::ReadGeometry(BamgGeom* bamggeom,BamgOpts* bamgopts){

		int verbose;
		nbiv=nbv=nbvx=0;
		nbe=nbt=nbtx=0;
		NbOfCurves=0;

		Real8 Hmin = HUGE_VAL;// the infinie value 
		Int4 hvertices=0;
		int i,j,k,n;

		//initialize some variables
		int Version=1,dim=2;
		nbv=bamggeom->NumVertices;
		nbe=bamggeom->NumEdges;
		nbvx = nbv;
		nbiv = nbv;
		verbose=bamgopts->verbose;

		//some checks
		if (bamggeom->NumVertices<=0 || bamggeom->Vertices==NULL){
			throw ErrorException(__FUNCT__,exprintf("the domain provided does not contain any vertex"));
		}
		if (bamggeom->NumEdges<=0 || bamggeom->Edges==NULL){
			throw ErrorException(__FUNCT__,exprintf("the domain provided does not contain any edge"));
		}

		//Vertices
		if (bamggeom->Vertices){
			if(verbose>3) printf("      processing Vertices\n");
			vertices = new GeometricalVertex[nbvx];
			for (i=0;i<nbv;i++) {
				vertices[i].r.x=(double)bamggeom->Vertices[i*3+0];
				vertices[i].r.y=(double)bamggeom->Vertices[i*3+1];
				vertices[i].ReferenceNumber=(Int4)bamggeom->Vertices[i*3+2];
				vertices[i].DirOfSearch=NoDirOfSearch;
				vertices[i].color =0;
				vertices[i].Set();
			}
			//find domain extrema for pmin,pmax
			pmin =  vertices[0].r;
			pmax =  vertices[0].r;
			for (i=0;i<nbv;i++) {
				pmin.x = Min(pmin.x,vertices[i].r.x);
				pmin.y = Min(pmin.y,vertices[i].r.y);
				pmax.x = Max(pmax.x,vertices[i].r.x);
				pmax.y = Max(pmax.y,vertices[i].r.y);
			}
			R2 DD05 = (pmax-pmin)*0.05;
			pmin -=  DD05;
			pmax +=  DD05;
			coefIcoor= (MaxICoor)/(Max(pmax.x-pmin.x,pmax.y-pmin.y));
			if(coefIcoor <=0){
				throw ErrorException(__FUNCT__,exprintf("coefIcoor should be positive"));
			}
		}
		else{
			throw ErrorException(__FUNCT__,exprintf("No Vertex provided"));
		}

		//Edges
		if (bamggeom->Edges){
			int i1,i2;
			R2 zero2(0,0);
			Real4 *len =0;

			if(verbose>3) printf("      processing Edges\n");
			edges = new GeometricalEdge[nbe];

			if (!hvertices) {
				len = new Real4[nbv];
				for(i=0;i<nbv;i++)
				 len[i]=0;
			}

			for (i=0;i<nbe;i++){
				i1=(int)bamggeom->Edges[i*3+0]-1; //-1 for C indexing
				i2=(int)bamggeom->Edges[i*3+1]-1; //-1 for C indexing
				edges[i].ref=(Int4)bamggeom->Edges[i*3+2];
				edges[i].v[0]=  vertices + i1;
				edges[i].v[1]=  vertices + i2;
				R2 x12 = vertices[i2].r-vertices[i1].r;
				Real8 l12=sqrt((x12,x12));
				edges[i].tg[0]=zero2;
				edges[i].tg[1]=zero2;
				edges[i].SensAdj[0] = edges[i].SensAdj[1] = -1;
				edges[i].Adj[0] = edges[i].Adj[1] = 0;
				edges[i].flag = 0;
				if (!hvertices) {
					vertices[i1].color++;
					vertices[i2].color++;
					len[i1] += l12;
					len[i2] += l12;
				}

				Hmin = Min(Hmin,l12);
			}

			// definition  the default of the given mesh size 
			if (!hvertices){
				for (i=0;i<nbv;i++) 
				 if (vertices[i].color > 0) 
				  vertices[i].m=  Metric(len[i] /(Real4) vertices[i].color);
				 else 
				  vertices[i].m=  Metric(Hmin);
				delete [] len;
			}
		}
		else{
			throw ErrorException(__FUNCT__,exprintf("No edges provided"));
		}

		//hVertices
		if(bamggeom->hVertices){
			if(verbose>3) printf("      processing hVertices\n");
			for (i=0;i< nbv;i++){
				vertices[i].m=Metric((Real4)bamggeom->hVertices[i]);
			}
		}
		else{
			if(verbose>3) printf("      no hVertices found\n");
		}

		//MetricVertices
		if(bamggeom->MetricVertices){
			if(verbose>3) printf("      processing MetricVertices\n");
			hvertices=1;
			for (i=0;i< nbv;i++) {
				vertices[i].m = Metric((Real4)bamggeom->MetricVertices[i*3+0],(Real4)bamggeom->MetricVertices[i*3+1],(Real4)bamggeom->MetricVertices[i*3+2]);
			}
		}
		else{
			if(verbose>3) printf("      no MetricVertices found\n");
		}

		//h1h2VpVertices
		if(bamggeom->h1h2VpVertices){
			if(verbose>3) printf("      processing h1h2VpVertices\n");
			Real4 h1,h2,v1,v2;
			hvertices =1;
			for (i=0;i< nbv;i++) {
				h1=(Real4)bamggeom->MetricVertices[i*4+0];
				h2=(Real4)bamggeom->MetricVertices[i*4+1];
				v1=(Real4)bamggeom->MetricVertices[i*4+2];
				v2=(Real4)bamggeom->MetricVertices[i*4+3];
				vertices[i].m = Metric(MatVVP2x2(1/(h1*h1),1/(h2*h2),D2(v1,v2)));
			}
		}
		else{
			if(verbose>3) printf("      no h1h2VpVertices found\n");
		}

		//MaximalAngleOfCorner
		if (bamgopts->MaximalAngleOfCorner){
			if(verbose>3) printf("      processing MaximalAngleOfCorner\n");
			MaximalAngleOfCorner=bamgopts->MaximalAngleOfCorner*Pi/180;
		}
		else{
			if(verbose>3) printf("      no MaximalAngleOfCorner found\n");
		}

		//TangentAtEdges
		if (bamggeom->TangentAtEdges){
			if(verbose>3) printf("      processing TangentAtEdges");
			int n,i,j,k;
			R2 tg;

			n=bamggeom->NumTangentAtEdges;
			for (k=0;k<n;k++) {
				i=(int)bamggeom->TangentAtEdges[k*4+0]-1; //for C indexing
				j=(int)bamggeom->TangentAtEdges[k*4+1]-1; //for C indexing
				tg.x=bamggeom->TangentAtEdges[k*4+2];
				tg.y=bamggeom->TangentAtEdges[k*4+3];
				if (j!=0 && j!=1){
					throw ErrorException(__FUNCT__,exprintf("TangentAtEdges second index should be 1 or 2 only"));
				}
				edges[i].tg[j] = tg;
			}
		}
		else{
			if(verbose>3) printf("      no TangentAtEdges found\n");
		}

		//Corners
		if(bamggeom->Corners){
			if(verbose>3) printf("      processing Corners");
			n=bamggeom->NumCorners;
			for (i=0;i<n;i++) {     
				j=(int)bamggeom->Corners[i]-1; //for C indexing
				if (j>nbv-1 || j<0){
					throw ErrorException(__FUNCT__,exprintf("Bad corner definition: should in [0 %i]",nbv));
				}
				vertices[j].SetCorner();
				vertices[j].SetRequired();  }
		}
		else{
			if(verbose>3) printf("      no Corners found\n");
		}

		//RequiredVertices
		if(bamggeom->RequiredVertices){
			if(verbose>3) printf("      processing RequiredVertices");
			n=bamggeom->NumRequiredVertices;
			for (i=0;i<n;i++) {     
				j=(int)bamggeom->RequiredVertices[i]-1; //for C indexing
				if (j>nbv-1 || j<0){
					throw ErrorException(__FUNCT__,exprintf("Bad RequiredVerticess  definition: should in [0 %i]",nbv));
				}
				vertices[j].SetRequired();  }
		}
		else{
			if(verbose>3) printf("      no RequiredVertices found\n");
		}

		//RequiredEdges
		if(bamggeom->RequiredEdges){
			if(verbose>3) printf("      processing RequiredEdges");
			n=bamggeom->NumRequiredEdges;
			for (i=0;i<n;i++) {     
				j=(int)bamggeom->RequiredEdges[i]-1; //for C indexing
				if (j>nbe-1 || j<0){
					throw ErrorException(__FUNCT__,exprintf("Bad RequiredEdges definition: should in [0 %i]",nbv));
				}
				edges[j].SetRequired();  }
		}
		else{
			if(verbose>3) printf("      no RequiredEdges found\n");
		}

		//SubDomain
		if(bamggeom->SubDomains){
			Int4 i0,i1,i2,i3;
			if(verbose>3) printf("      processing SubDomains\n");
			NbSubDomains=bamggeom->NumSubDomains;
			subdomains = new GeometricalSubDomain[NbSubDomains];
			for (i=0;i<NbSubDomains;i++) {
				i0=(int)bamggeom->SubDomains[i*4+0];
				i1=(int)bamggeom->SubDomains[i*4+1];
				i2=(int)bamggeom->SubDomains[i*4+2];
				i3=(int)bamggeom->SubDomains[i*4+3];
				if (i0!=2) throw ErrorException(__FUNCT__,exprintf("Bad Subdomain definition: first number should be 2 (for Edges)"));
				if (i1>nbe || i1<=0) throw ErrorException(__FUNCT__,exprintf("Bad Subdomain definition: second number should in [1 %i] (edge number)",nbe));
				subdomains[i].edge=edges + (i1-1);
				subdomains[i].sens = (int) i2;
				subdomains[i].ref = i3;
			}
		}
		else{
			if(verbose>3) printf("      no SubDomains found\n");
		}
	}
	/*}}}1*/

} 
