function smb = interpISMIP6AntarcticaSMB(md,model_name,scenario)
%interpISMIP6AntarcticaSMB - interpolate chosen ISMIP6 atmospheric forcing to model
%
%   Input:
%     - md (model object)
%     - model_name  (string): name of the climate model 
%                             Examples: CESM2 CNRM_CM6 CNRM_ESM2 CSIRO-Mk3-6-0 HadGEM2-ES IPSL-CM5A-MR 
%                                       ccsm4 miroc-esm-chem noresm1-m
%     - scenario    (string): name of the climate scenario
%                             Examples: rcp26, rcp85, ssp126, ssp585
%   Output:
%     - smb: prepared to be input directly into md.smb
%
%   Examples:
%      md.smb = interpISMIP6AntarcticaSMB(md,'miroc-esm-chem','rcp8.5');

% Find appropriate directory
switch oshostname(),
	case {'totten'}
		path='/totten_1/ModelData/ISMIP6/Projections/AIS/Atmosphere_Forcing/';
	otherwise
		error('machine not supported yet, please provide your own path');
end

%find which files are available and do a case insensitive test
rootname = [path model_name '_' scenario '/'];
list = dir([rootname '/Regridded_2km/']);
found1 = false; found2 = false;
for i=1:numel(list)
	if strcmpi(list(i).name, [model_name '_2km_clim_1995-2014.nc'])
		disp(['Found clim file: ' list(i).name]);
		smbclimnc = [rootname '/Regridded_2km/' list(i).name];
		found1 = true;
	end
	if strcmpi(list(i).name, [model_name '_2km_anomaly_1995-2100.nc'])
		disp(['Found anom file: ' list(i).name]);
		smbanomnc = [rootname '/Regridded_2km/' list(i).name];
		found2 = true;
	end
end
if ~found1 || ~found2
   error(['this path does not exist or the ' model_name ' and ' scenario ' are not available in this combination.']);
end

disp('   == loading TS and SMB climatology data');
lat                 = double(ncread(smbclimnc,'lat'));
lon                 = double(ncread(smbclimnc,'lon'));
smb_clim_data       = double(ncread(smbclimnc,'smb_clim'));
ts_clim_data        = double(ncread(smbclimnc,'ts_clim'));

disp('   == loading TS and SMB anomoly data');
smb_anomaly_data    = double(ncread(smbanomnc,'smb_anomaly'));
ts_anomaly_data     = double(ncread(smbanomnc,'ts_anomaly'));

%Create SMB and TS matrix
disp('   == Interpolating on model');
time = [1995:2100];
[x_n y_n]=ll2xy(lat(:,1),lon(:,1),-1);
y_n = x_n;
smb_clim = InterpFromGridToMesh(x_n,y_n,smb_clim_data',md.mesh.x,md.mesh.y,0);
ts_clim  = InterpFromGridToMesh(x_n,y_n,ts_clim_data',md.mesh.x,md.mesh.y,0);
temp_matrix_smb = []; temp_matrix_ts = [];
for i = 1:size(smb_anomaly_data,3)
	%SMB
	temp_smb        = InterpFromGridToMesh(x_n,y_n,smb_anomaly_data(:,:,i)',md.mesh.x,md.mesh.y,0);
	temp_smb        = temp_smb+smb_clim;
	temp_matrix_smb = [temp_matrix_smb temp_smb];
	%TS
	temp_ts         = InterpFromGridToMesh(x_n,y_n,ts_anomaly_data(:,:,i)',md.mesh.x,md.mesh.y,0);
	temp_ts         = temp_smb+smb_clim;
	temp_matrix_ts  = [temp_matrix_ts temp_ts];
	clear temp_smb; clear temp_ts;
end

% convert to m/yr
rhoi = md.materials.rho_ice;
temp_matrix_smb = temp_matrix_smb*(31556926/1000)*(1000/rhoi);

%Save Data (1995-2100)
smb = SMBforcing();
smb.mass_balance = [temp_matrix_smb ; time];

%What do we do with surface temp?
%md.miscellaneous.dummy.ts = [temp_matrix_ts ; time];
disp('Info: forcings cover 1995 to 2100');
end
