function basalforcings = interpISMIP6AntarcticaOcn(md,model_name,scenario)
%interpISMIP6AntarcticaOcn - interpolate chosen ISMIP6 atmospheric forcing to model
%
%   Input:
%     - md (model object)
%     - model_name  (string): name of the climate model 
%                             Examples: cnrm-esm2-1 ccsm4cesm2 cnrm-cm6-1 csiro-mk3-6-0
%                                       hadgem2-es ipsl-cm5a-mr miroc-esm-chem noresm1-m ukesm1-0-ll
%
%     - scenario    (string): name of the climate scenario
%                             Examples: rcp2.6, rcp8.5, ssp126, ssp585
%   Output:
%     - basalforcings: prepared to be input directly into md.smb
%
%   Examples:
%      md.basalforcings = interpISMIP6AntarcticaOcn(md,'miroc-esm-chem','rcp8.5');

% Find appropriate directory
switch oshostname(),
	case {'totten'}
		path='/totten_1/ModelData/ISMIP6/Projections/AIS/Ocean_Forcing/';
	otherwise
		error('machine not supported yet, please provide your own path');
end

%Find forcing file
rootname = [path model_name '_' scenario '/1995-2100'];
list = dir(rootname);
found = false;
for i=1:numel(list)
	if strcmpi(list(i).name, [model_name '_thermal_forcing_8km_x_60m.nc'])
		disp(['Found tf file: ' list(i).name]);
		tfnc = [rootname '/' list(i).name];
		found = true;
	end
end
if ~found
   error(['this path does not exist or the ' model_name ' and ' scenario ' are not available in this combination.']);
end

%load TF data
disp('   == loading TF');
x_n     = double(ncread(tfnc,'x'));
y_n     = double(ncread(tfnc,'y'));
tf_data = double(ncread(tfnc,'thermal_forcing'));
z_data  = double(ncread(tfnc,'z'));

%Build tf cell array
time = 1995:2100;
tf = cell(1,1,size(tf_data,3));
for i=1:size(tf_data,3)  %Iterate over depths
	disp(['   == Interpolating over depth ' num2str(i) '/' num2str(size(tf_data,3))]);
	
	temp_matrix=[];
	for ii=1:size(tf_data,4) %Iterate over time steps
		%temp_tfdata=InterpFromGridToMesh(x_n,y_n,tf_data(:,:,i,ii)',md.mesh.x,md.mesh.y,0);
		temp_tfdata=InterpFromGrid(x_n,y_n,tf_data(:,:,i,ii)',md.mesh.x,md.mesh.y);
		temp_matrix = [temp_matrix temp_tfdata];
	end
	tf{:,:,i} = [temp_matrix ; time];
end

%load Delta and gamma data
deltatnc_median = [path '/parameterizations/coeff_gamma0_DeltaT_quadratic_non_local_median.nc'];
basin_datanc    = [path '/imbie2/imbie2_basin_numbers_8km.nc'];
deltaT_median   = double(ncread(deltatnc_median,'deltaT_basin'));
gamma0_median   = double(ncread(deltatnc_median,'gamma0'));
basinid_data    = double(ncread(basin_datanc,'basinNumber'));

disp('   == Interpolating basin Id');
num_basins = length(unique(basinid_data));
deltat_median = NaN(length(unique(basinid_data)),1);

for i=0:num_basins-1
	pos = find(basinid_data==i);
	deltat_temp = deltaT_median(pos);
	deltat_temp = deltat_temp(1);
	deltat_median(i+1) = deltat_temp;
end

%Deal with basins ID
x_el = mean(md.mesh.x(md.mesh.elements),2);
y_el = mean(md.mesh.y(md.mesh.elements),2);
basinid = InterpFromGrid(x_n,y_n,basinid_data',x_el, y_el, 'nearest')+1;

%Set ISMIP6 basal melt rate parameters
basalforcings            = basalforcingsismip6(md.basalforcings);
basalforcings.basin_id   = basinid;
basalforcings.num_basins = num_basins;
basalforcings.delta_t    = deltat_median;
basalforcings.tf_depths  = z_data';
basalforcings.gamma_0    = gamma0_median;
basalforcings.tf         = tf;

disp('Info: forcings cover 1995 to 2100');
