#include "../shared/shared.h"
#include "../include/macros.h"
#include "../toolkits/toolkits.h"

#include <cstdio>
#include <cstring>
#include <cmath>
#include <ctime>
#include "Meshio.h"
#include "Mesh2.h"
#include "QuadTree.h"
#include "SetOfE4.h"


namespace bamg {

void Triangles::WriteBamgMesh(BamgMesh* bamgmesh,BamgOpts* bamgopts){

	int i,j;
	int verbose;

	verbose=bamgopts->verbose;
	Int4 *reft = new Int4[nbt];
	Int4 nbInT = ConsRefTriangle(reft);

	//Vertices
	if(verbose>3) printf("      writing Vertices\n");
	bamgmesh->NumVertices=nbv;
	xfree((void**)&bamgmesh->Vertices);
	if (nbv){
		bamgmesh->Vertices=(double*)xmalloc(3*nbv*sizeof(double));
		for (i=0;i<nbv;i++){
			bamgmesh->Vertices[i*3+0]=vertices[i].r.x;
			bamgmesh->Vertices[i*3+1]=vertices[i].r.y;
			bamgmesh->Vertices[i*3+2]=vertices[i].ref();
		}
	}
	else{
		bamgmesh->Vertices=NULL;
	}

	//Edges
	if(verbose>3) printf("      writing Edges\n");
	bamgmesh->NumEdges=nbe;
	xfree((void**)&bamgmesh->Edges);
	if (nbe){
		bamgmesh->Edges=(double*)xmalloc(3*nbe*sizeof(double));
		for (i=0;i<nbe;i++){
			bamgmesh->Edges[i*3+0]=Number(edges[i][0])+1; //back to M indexing
			bamgmesh->Edges[i*3+1]=Number(edges[i][1])+1; //back to M indexing
			bamgmesh->Edges[i*3+2]=edges[i].ref;
		}
	}
	else{
		bamgmesh->Edges=NULL;
	}

	//CrackedEdges
	if(verbose>3) printf("      writing CrackedEdges\n");
	bamgmesh->NumCrackedEdges=NbCrackedEdges;
	xfree((void**)&bamgmesh->CrackedEdges);
	if (NbCrackedEdges){
		bamgmesh->CrackedEdges=(double*)xmalloc(2*NbCrackedEdges*sizeof(double));
		for (i=0;i<NbCrackedEdges;i++){
			bamgmesh->CrackedEdges[i*2+0]=Number(CrackedEdges[i].a.edge)+1; //back to M indexing
			bamgmesh->CrackedEdges[i*2+1]=Number(CrackedEdges[i].b.edge)+1; //back to M indexing
		}
	}
	else{
		bamgmesh->CrackedEdges=NULL;
	}

	//Triangles
	if(verbose>3) printf("      writing Triangles\n");
	Int4 k=nbInT-NbOfQuad*2;
	Int4 num=0;
	bamgmesh->NumTriangles=k;
	xfree((void**)&bamgmesh->Triangles);
	if (k){
		bamgmesh->Triangles=(double*)xmalloc(4*k*sizeof(double));
		for (i=0;i<nbt;i++){
			Triangle &t=triangles[i];
			if (reft[i]>=0 && !( t.Hidden(0) || t.Hidden(1) || t.Hidden(2) )){
				bamgmesh->Triangles[num*4+0]=Number(t[0])+1; //back to M indexing
				bamgmesh->Triangles[num*4+1]=Number(t[1])+1; //back to M indexing
				bamgmesh->Triangles[num*4+2]=Number(t[2])+1; //back to M indexing
				bamgmesh->Triangles[num*4+3]=subdomains[reft[i]].ref;
				num=num+1;
			}
		}
	}
	else{
		bamgmesh->Triangles=NULL;
	}

	//Quadrilaterals
	if(verbose>3) printf("      writing Quadrilaterals\n");
	bamgmesh->NumQuadrilaterals=NbOfQuad;
	xfree((void**)&bamgmesh->Quadrilaterals);
	if (NbOfQuad){
		bamgmesh->Quadrilaterals=(double*)xmalloc(5*NbOfQuad*sizeof(double));
		for (i=0;i<nbt;i++){
			Triangle &t =triangles[i];
			Triangle* ta;
			Vertex *v0,*v1,*v2,*v3;
			if (reft[i]<0) continue;
			if ((ta=t.Quadrangle(v0,v1,v2,v3)) !=0 && &t<ta) { 
				bamgmesh->Quadrilaterals[i*5+0]=Number(v0)+1; //back to M indexing
				bamgmesh->Quadrilaterals[i*5+1]=Number(v1)+1; //back to M indexing
				bamgmesh->Quadrilaterals[i*5+2]=Number(v2)+1; //back to M indexing
				bamgmesh->Quadrilaterals[i*5+3]=Number(v3)+1; //back to M indexing
				bamgmesh->Quadrilaterals[i*5+4]=subdomains[reft[i]].ref;
			}
		}
	}
	else{
		bamgmesh->Quadrilaterals=NULL;
	}

	//SubDomains
	if(verbose>3) printf("      writing SubDomains\n");
	bamgmesh->NumSubDomains=NbSubDomains;
	xfree((void**)&bamgmesh->SubDomains);
	if (NbSubDomains){
		bamgmesh->SubDomains=(double*)xmalloc(4*NbSubDomains*sizeof(double));
		for (i=0;i<NbSubDomains;i++){
			bamgmesh->SubDomains[i*4+0]=3;
			bamgmesh->SubDomains[i*4+1]=reft[Number(subdomains[i].head)];
			bamgmesh->SubDomains[i*4+2]=1;
			bamgmesh->SubDomains[i*4+3]=subdomains[i].ref;
		}
	}
	else{
		bamgmesh->SubDomains=NULL;
	}

	//SubDomainsFromGeom
	if(verbose>3) printf("      writing SubDomainsFromGeom\n");
	bamgmesh->NumSubDomainsFromGeom=Gh.NbSubDomains;
	xfree((void**)&bamgmesh->SubDomainsFromGeom);
	if (Gh.NbSubDomains){
		bamgmesh->SubDomainsFromGeom=(double*)xmalloc(4*Gh.NbSubDomains*sizeof(double));
		for (i=0;i<Gh.NbSubDomains;i++){
			bamgmesh->SubDomainsFromGeom[i*4+0]=2;
			bamgmesh->SubDomainsFromGeom[i*4+1]=Number(subdomains[i].edge)+1; //back to Matlab indexing
			bamgmesh->SubDomainsFromGeom[i*4+2]=subdomains[i].sens;
			bamgmesh->SubDomainsFromGeom[i*4+3]=Gh.subdomains[i].ref;
		}
	}
	else{
		bamgmesh->SubDomainsFromGeom=NULL;
	}

	//VerticesOnGeomVertex
	if(verbose>3) printf("      writing VerticesOnGeometricVertex\n");
	bamgmesh->NumVerticesOnGeometricVertex=NbVerticesOnGeomVertex;
	xfree((void**)&bamgmesh->VerticesOnGeometricVertex);
	if (NbVerticesOnGeomVertex){
		bamgmesh->VerticesOnGeometricVertex=(double*)xmalloc(2*NbVerticesOnGeomVertex*sizeof(double));
		for (i=0;i<NbVerticesOnGeomVertex;i++){
			VertexOnGeom &v=VerticesOnGeomVertex[i];
			if (!v.OnGeomVertex()){
				throw ErrorException(__FUNCT__,exprintf("A vertices supposed to be OnGeometricVertex is actually not"));
			}
			bamgmesh->VerticesOnGeometricVertex[i*2+0]=Number((Vertex*)v)+1; //back to Matlab indexing
			bamgmesh->VerticesOnGeometricVertex[i*2+1]=Gh.Number(( GeometricalVertex*)v)+1; //back to Matlab indexing
		}
	}
	else{
		bamgmesh->VerticesOnGeometricVertex=NULL;
	}

	//VertexOnGeometricEdge
	if(verbose>3) printf("      writing VerticesOnGeometricEdge\n");
	bamgmesh->NumVerticesOnGeometricEdge=NbVerticesOnGeomEdge;
	xfree((void**)&bamgmesh->VerticesOnGeometricEdge);
	if (NbVerticesOnGeomEdge){
		bamgmesh->VerticesOnGeometricEdge=(double*)xmalloc(3*NbVerticesOnGeomEdge*sizeof(double));
		for (i=0;i<NbVerticesOnGeomEdge;i++){
			const VertexOnGeom &v=VerticesOnGeomEdge[i];
			if (!v.OnGeomEdge()){
				throw ErrorException(__FUNCT__,exprintf("A vertices supposed to be OnGeometricEdge is actually not"));
			}
			bamgmesh->VerticesOnGeometricEdge[i*3+0]=Number((Vertex*)v)+1; //back to Matlab indexing
			bamgmesh->VerticesOnGeometricEdge[i*3+1]=Gh.Number((const GeometricalEdge*)v)+1; //back to Matlab indexing
			bamgmesh->VerticesOnGeometricEdge[i*3+2]=(double)v;
		}
	}
	else{
		bamgmesh->VerticesOnGeometricEdge=NULL;
	}

	//EdgesOnGeometricEdge
	if(verbose>3) printf("      writing EdgesOnGeometricEdge\n");
	k=0;
	for (i=0;i<nbe;i++){
		if (edges[i].on) k=k+1;
	}
	bamgmesh->NumEdgesOnGeometricEdge=k;
	xfree((void**)&bamgmesh->EdgesOnGeometricEdge);
	if (k){
		bamgmesh->EdgesOnGeometricEdge=(double*)xmalloc(2*(int)k*sizeof(double));
		int count=0;
		for (i=0;i<nbe;i++){
			if (edges[i].on){
				bamgmesh->EdgesOnGeometricEdge[count*2+0]=(double)i+1; //back to Matlab indexing
				bamgmesh->EdgesOnGeometricEdge[count*2+1]=(double)Gh.Number(edges[i].on)+1; //back to Matlab indexing
				count=count+1;
			}
		}
	}
	else{
		bamgmesh->EdgesOnGeometricEdge=NULL;
	}
}

void Geometry::Write(const char * filename)
{
	long int verbosity=0;

  ofstream f(filename);
  if (f)
    {
      if(verbosity>1)
	cout << "  -- write geometry in file " << filename << endl;
       if (name) delete name;
       name = new char[strlen(filename)+1];
       strcpy(name,filename);
       OnDisk =1;
       f << *this;
    }
}

void Geometry::WriteGeometry(BamgGeom* bamggeom, BamgOpts* bamgopts){

	int verbose;
	int nbreq=0;
	int nbreqv=0;
	int nbtan=0;
	int nbcracked=0;
	int i,count;

	verbose=bamgopts->verbose;

	//Vertices
	if(verbose>3) printf("      writing Vertices\n");
	bamggeom->NumVertices=nbv;
	xfree((void**)&bamggeom->Vertices);
	if (nbv){
		bamggeom->Vertices=(double*)xmalloc(3*nbv*sizeof(double));
		for (i=0;i<nbv;i++){
			bamggeom->Vertices[i*3+0]=vertices[i].r.x;
			bamggeom->Vertices[i*3+1]=vertices[i].r.y;
			bamggeom->Vertices[i*3+2]=vertices[i].ref();

			//update counters
			if (vertices[i].Required()) nbreqv++;
		}
	}
	else{
		bamggeom->Vertices=NULL;
	}

	//Edges
	if(verbose>3) printf("      writing Edges\n");
	bamggeom->NumEdges=nbe;
	xfree((void**)&bamggeom->Edges);
	if (nbe){
		bamggeom->Edges=(double*)xmalloc(3*nbe*sizeof(double));
		for (i=0;i<nbe;i++){
			bamggeom->Edges[i*3+0]=Number(edges[i][0])+1; //back to Matlab indexing
			bamggeom->Edges[i*3+1]=Number(edges[i][1])+1; //back to Matlab indexing
			//bamggeom->Edges[i*3+2]=(double)edges[i].ref(); //TEST does not compile???
			bamggeom->Edges[i*3+2]=1; //TEST for now

			//update counters
			if (edges[i].Required()) nbreq++;
			if (edges[i].Cracked()){
				if (i<=Number(edges[i].link)) nbcracked++;
			}
			if (edges[i].TgA() && edges[i][0].Corner()) nbtan++;
			if (edges[i].TgB() && edges[i][1].Corner()) nbtan++;
		}
	}
	else{
		bamggeom->Edges=NULL;
	}

	//CrackedEdges
	if(verbose>3) printf("      writing CrackedEdges\n");
	bamggeom->NumCrackedEdges=nbcracked;
	xfree((void**)&bamggeom->CrackedEdges);
	if (nbcracked){
		bamggeom->CrackedEdges=(double*)xmalloc(2*nbcracked*sizeof(double));
		count=0;
		for (i=0;i<nbe;i++){
			if (edges[i].Cracked()){
				if (i<=Number(edges[i].link)){
					bamggeom->CrackedEdges[count*2+0]=i+1;                     //back to Matlab indexing
					bamggeom->CrackedEdges[count*2+1]=Number(edges[i].link)+1; //back to Matlab indexing
					count=count+1;
				}
			}
		}
	}
	else{
		bamggeom->CrackedEdges=NULL;
	}

	//RequiredEdges
	if(verbose>3) printf("      writing RequiredEdges\n");
	bamggeom->NumRequiredEdges=nbreq;
	xfree((void**)&bamggeom->RequiredEdges);
	if (nbreq){
		bamggeom->RequiredEdges=(double*)xmalloc(1*nbreq*sizeof(double));
		count=0;
		for (i=0;i<nbe;i++){
			if (edges[i].Required()){
				bamggeom->RequiredEdges[count]=i+1; //back to Matlab indexing
				count=count+1;
			}
		}
	}
	else{
		bamggeom->RequiredEdges=NULL;
	}

	//No corners

	//RequiredVertices
	if(verbose>3) printf("      writing RequiredVertices\n");
	bamggeom->NumRequiredVertices=nbreqv;
	xfree((void**)&bamggeom->RequiredVertices);
	if (nbreqv){
		bamggeom->RequiredVertices=(double*)xmalloc(1*nbreqv*sizeof(double));
		count=0;
		for (i=0;i<nbe;i++){
			if (vertices[i].Required()){
				bamggeom->RequiredVertices[count]=i+1; //back to Matlab indexing
				count=count+1;
			}
		}
	}
	else{
		bamggeom->RequiredVertices=NULL;
	}

	//SubDomains
	if(verbose>3) printf("      writing SubDomains\n");
	bamggeom->NumSubDomains=NbSubDomains;
	xfree((void**)&bamggeom->SubDomains);
	if (NbSubDomains){
		bamggeom->SubDomains=(double*)xmalloc(4*NbSubDomains*sizeof(double));
		for (i=0;i<NbSubDomains;i++){
			bamggeom->SubDomains[4*i+0]=2;
			bamggeom->SubDomains[4*i+1]=Number(subdomains[i].edge)+1; //back to Matlab indexing
			bamggeom->SubDomains[4*i+2]=subdomains[i].sens;
			bamggeom->SubDomains[4*i+3]=subdomains[i].ref;
		}
	}
	else{
		bamggeom->SubDomains=NULL;
	}

	//TangentAtEdges
	if(verbose>3) printf("      writing TangentAtEdges\n");
	bamggeom->NumTangentAtEdges=nbtan;
	xfree((void**)&bamggeom->TangentAtEdges);
	if (nbtan){
		bamggeom->TangentAtEdges=(double*)xmalloc(4*nbtan*sizeof(double));
		count=0;
		for (i=0;i<nbe;i++){
			if (edges[i].TgA() && edges[i][0].Corner()){
				bamggeom->TangentAtEdges[4*i+0]=i+1; //back to Matlab indexing
				bamggeom->TangentAtEdges[4*i+1]=1;
				bamggeom->TangentAtEdges[4*i+2]=edges[i].tg[0].x;
				bamggeom->TangentAtEdges[4*i+3]=edges[i].tg[0].y;
			}
			if (edges[i].TgB() && edges[i][1].Corner()){
				bamggeom->TangentAtEdges[4*i+0]=i+1; //back to Matlab indexing
				bamggeom->TangentAtEdges[4*i+1]=2;
				bamggeom->TangentAtEdges[4*i+2]=edges[i].tg[1].x;
				bamggeom->TangentAtEdges[4*i+3]=edges[i].tg[2].y;
			}
			count=count+1;
		}
	}
	else{
		bamggeom->TangentAtEdges=NULL;
	}
}

ostream& operator <<(ostream& f, const   Geometry & Gh) 
{
   Int4  NbCorner=0;
   {
     f << "MeshVersionFormatted 0" <<endl;
     f << "\nDimension\n"  << 2 << endl;
//     f << "\nIdentifier\n" ;
//     WriteStr(f,Gh.identity);
//     f <<endl;
   }
   int nbreqv=0;
   { 
     
     f.precision(12);
     f << "\nVertices\n" << Gh.nbv <<endl;
     for (Int4 i=0;i<Gh.nbv;i++)
       {
	 GeometricalVertex & v =  Gh.vertices[i];
	 if (v.Required()) nbreqv++;
	 f << v.r.x << " " << v.r.y << " " << v.ref() << endl;
	 if (v.Corner()) NbCorner++;
       }
   }
   
   int nbcracked=0;

   {
     int nbreq=0;
     f << "\nEdges\n"<< Gh.nbe << endl;
     for(Int4 ie=0;ie<Gh.nbe;ie++)
       { 
	 
	 GeometricalEdge & e = Gh.edges[ie];
	 if (e.Required()) nbreq++;
	 if (e.Cracked()) { 
	   Int4 ie1 = Gh.Number(e.link);
	   if (ie <= ie1)  ++nbcracked;}
	 f << Gh.Number(e[0])+1 << " " << Gh.Number(e[1])+1;
	 f << " " << e.ref <<endl;
       }
     
     if (nbcracked)
       {
	 f << "\nCrackedEdges\n"<< nbcracked<< endl;
	 for(Int4 ie=0;ie<Gh.nbe;ie++)
	   {
	     GeometricalEdge & e = Gh.edges[ie];
	     if (e.Cracked()) { 
	       Int4  ie1 = Gh.Number(e.link);
	       if (ie <= ie1)  f << ie+1 << " " << ie1+1<< endl;
	     }
	   }
       }
     if(nbreq)
       {
	 f << "\nRequiredEdges\n"<< nbreq<< endl;
         for(Int4 ie=0;ie<Gh.nbe;ie++)
           {
             GeometricalEdge & e = Gh.edges[ie];
             if (e.Required()) 
	       f << ie+1 << endl;
	   }
       }
     
     
     
   }

    f << "\nAngleOfCornerBound\n" 
     << Gh.MaximalAngleOfCorner*180/Pi << endl;
    if (NbCorner) 
      {
	f << "\nCorners\n" << NbCorner << endl;
	for (Int4 i=0,j=0;i<Gh.nbv;i++)
	  {
	    GeometricalVertex & v =  Gh.vertices[i];
	    if (v.Corner()) 
	      j++,f << Gh.Number(v)+1 << (j % 5 ? ' ' : '\n');
	  }
        
      
      }

    if(nbreqv)
      {
	f << "\nRequiredVertices\n"<< nbreqv<< endl;
	for (Int4 j=0,i=0;i<Gh.nbv;i++)
	  {
	    GeometricalVertex & v =  Gh.vertices[i];
	    
	    if (v.Required()) 
	      j++,f << i+1 << (j % 5 ? ' ' : '\n');
	  }
	f << endl;
      }
    
    { 
       Int4 i;
       f << "\nSubDomainFromGeom\n" ;
       f << Gh.NbSubDomains<< endl;
       for (i=0;i<Gh.NbSubDomains;i++) 
         f << "2 " << Gh.Number(Gh.subdomains[i].edge)+1 << " " << Gh.subdomains[i].sens 
           << " " << Gh.subdomains[i].ref << endl;        
     }
     {
       Int4 n=0,i;

       for(i=0;i< Gh.nbe;i++)
	 {
	   if(Gh.edges[i].TgA() && Gh.edges[i][0].Corner() ) 
	     n++;
	   if(Gh.edges[i].TgB() && Gh.edges[i][1].Corner() ) 
	     n++;
	 }
       if (n) {
	 f << "TangentAtEdges " << n << endl;
	 for(i=0;i< Gh.nbe;i++)
	   {
	     if (Gh.edges[i].TgA() && Gh.edges[i][0].Corner() ) 
	       f << i+1 << " 1 " << Gh.edges[i].tg[0].x 
		 << " " << Gh.edges[i].tg[0].y << endl;
	     if (Gh.edges[i].TgB() && Gh.edges[i][1].Corner() ) 
	       f << i+1 << " 2 " << Gh.edges[i].tg[1].x 
		 << " " << Gh.edges[i].tg[1].y << endl;
	   }
	 
       }}
     //  f << " Not Yet Implemented" << endl;
     
     return f;
}


} // end of namespace bamg 
