/*This file is largely inspired by Freefem++
 
 Freefem++ is free software; you can redistribute it and/or modify
 it under the terms of the GNU Lesser General Public License as published by
 the Free Software Foundation; either version 2.1 of the License, or
 (at your option) any later version.
 
 Freefem++  is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU Lesser General Public License for more details.
 
 You should have received a copy of the GNU Lesser General Public License
 along with Freefem++; if not, write to the Free Software
 Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "../shared/shared.h"
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <time.h>

#include "Meshio.h"
#include "Mesh2.h"
#include "QuadTree.h"
#include "SetOfE4.h"
#ifdef __MWERKS__
#pragma optimization_level 2
//#pragma inline_depth 1
#endif

#ifdef DRAWING1
  extern bool withrgraphique ;
#endif

namespace bamg {

static const  Direction NoDirOfSearch=Direction();

void Triangles::ReadFromMatlabMesh(BamgMesh* bamgmesh, BamgOpts* bamgopts){

	int verbose;
	Real8 Hmin = HUGE_VAL;// the infinie value 
	Int4 i1,i2,i3,iref;
	Int4 i,j;
	Int4 hvertices =0;
	Int4 ifgeom=0;
	Metric M1(1);
	int Version=1,dim=2;

	verbose=bamgopts->verbose;

	nbv=bamgmesh->NumVertices;
	nbvx=nbv;
	nbt=bamgmesh->NumTriangles;
	nbiv=nbvx;

	//Vertices
	if(bamgmesh->Vertices){
		if(verbose>3) printf("      processing Vertices\n");

		vertices=(Vertex*)xmalloc(nbv*sizeof(Vertex));
		ordre=(Vertex**)xmalloc(nbv*sizeof(Vertex*));

		for (i=0;i<nbv;i++){
			vertices[i].r.x=bamgmesh->Vertices[i*3+0];
			vertices[i].r.y=bamgmesh->Vertices[i*3+1];
			vertices[i].ReferenceNumber=1;
			vertices[i].DirOfSearch =NoDirOfSearch;
			vertices[i].m=M1;
			vertices[i].color=(Int4)bamgmesh->Vertices[i*3+2];
		}
		nbtx=2*nbvx-2; // for filling The Holes and quadrilaterals 
	}
	else{
		if(verbose>3) throw ErrorException(__FUNCT__,exprintf("no Vertices found in the initial mesh"));
	}

	//Triangles
	if(bamgmesh->Triangles){
		if(verbose>3) printf("      processing Triangles\n");
		triangles =new Triangle[nbtx]; //we cannot allocate only nbt triangles since 
												 //other triangles will be added for each edge
		for (i=0;i<nbt;i++){
			Triangle & t = triangles[i];
			i1=(Int4)bamgmesh->Triangles[i*4+0]-1; //for C indexing
			i2=(Int4)bamgmesh->Triangles[i*4+1]-1; //for C indexing
			i3=(Int4)bamgmesh->Triangles[i*4+2]-1; //for C indexing
			t=Triangle(this,i1,i2,i3);
			t.color=(Int4)bamgmesh->Triangles[i*4+3];
		}
	}
	else{
		if(verbose>3) throw ErrorException(__FUNCT__,exprintf("no Triangles found in the initial mesh"));
	}

	//Quadrilaterals
	if(bamgmesh->Quadrilaterals){
		if(verbose>3) printf("      processing Quadrilaterals\n");
		Int4 i1,i2,i3,i4,iref;
		triangles =new Triangle[nbt];
		for (i=0;i<bamgmesh->NumQuadrilaterals;i++){
			Triangle & t1 = triangles[2*i];
			Triangle & t2 = triangles[2*i+1];
			i1=(Int4)bamgmesh->Quadrilaterals[i*4+0]-1; //for C indexing
			i2=(Int4)bamgmesh->Quadrilaterals[i*4+1]-1; //for C indexing
			i3=(Int4)bamgmesh->Quadrilaterals[i*4+2]-1; //for C indexing
			i4=(Int4)bamgmesh->Quadrilaterals[i*4+3]-1; //for C indexing
			t1=Triangle(this,i1,i2,i3);
			t2=Triangle(this,i3,i4,i1);
			t1.color=1; //reference = 1 for all triangles since it has not been specified
			t2.color=1; //reference = 1 for all triangles since it has not been specified
			t1.SetHidden(OppositeEdge[1]); // two times  because the adj was not created 
			t2.SetHidden(OppositeEdge[1]); 
		}
	}
	else{
		if(verbose>3) printf("      no Quadrilaterals found\n");
	}

	//hVertices
	if(bamgmesh->hVertices){
		if(verbose>3) printf("      processing hVertices\n");
		hvertices=1;
		for (i=0;i< nbv;i++){
			vertices[i].m=Metric((Real4)bamgmesh->hVertices[i]);
		}
	}
	else{
		if(verbose>3) printf("      no hVertices found\n");
	}

	//VerVerticesOnGeometricEdge
	if(bamgmesh->VerticesOnGeometricEdge){
		if(verbose>3) printf("      processing VerticesOnGeometricEdge\n");
		NbVerticesOnGeomEdge=bamgmesh->NumVerticesOnGeometricEdge;
		VerticesOnGeomEdge= new  VertexOnGeom[NbVerticesOnGeomEdge] ;
		for (i=0;i<NbVerticesOnGeomEdge;i++){
			Int4  i1,i2;
			Real8 s;
			i1=(Int4)bamgmesh->VerticesOnGeometricEdge[i*3+0]-1; //for C indexing
			i2=(Int4)bamgmesh->VerticesOnGeometricEdge[i*3+1]-1; //for C indexing
			s =(Int4)bamgmesh->VerticesOnGeometricEdge[i*3+2];
			VerticesOnGeomEdge[i]=VertexOnGeom(vertices[i1],Gh.edges[i2],s);
		}
	}
	else{
		if(verbose>3) printf("      no VertexOnGeometricEdge found\n");
	}

	//Edges
	if (bamgmesh->Edges){
		int i1,i2;
		R2 zero2(0,0);
		Real4 *len =0;

		if(verbose>3) printf("      processing Edges\n");
		nbe=bamgmesh->NumEdges;
		edges = new Edge[nbe];

		if (!hvertices) {
			len = new Real4[nbv];
			for(i=0;i<nbv;i++)
			 len[i]=0;
		}

		for (i=0;i<nbe;i++){
			i1=(int)bamgmesh->Edges[i*3+0]-1; //-1 for C indexing
			i2=(int)bamgmesh->Edges[i*3+1]-1; //-1 for C indexing
			edges[i].v[0]= vertices +i1;
			edges[i].v[1]= vertices +i2;
			edges[i].adj[0]=0;
			edges[i].adj[1]=0;
			R2 x12 = vertices[i2].r-vertices[i1].r;
			Real8 l12=sqrt( (x12,x12));        

			if (!hvertices) {
				vertices[i1].color++;
				vertices[i2].color++;
				len[i1]+= l12;
				len[i2] += l12;
			}
			Hmin = Min(Hmin,l12);
		}

		// definition  the default of the given mesh size 
		if (!hvertices){
			for (i=0;i<nbv;i++) 
			 if (vertices[i].color > 0) 
			  vertices[i].m=  Metric(len[i] /(Real4) vertices[i].color);
			 else 
			  vertices[i].m=  Metric(Hmin);
			delete [] len;
		}

		// construction of edges[].adj 
		for (i=0;i<nbv;i++){ 
		 vertices[i].color = (vertices[i].color ==2) ? -1 : -2;
		}
		for (i=0;i<nbe;i++){
			for (j=0;j<2;j++) { 
				Vertex *v=edges[i].v[j];
				Int4 i0=v->color,j0;
				if(i0==-1){
					v->color=i*2+j;
				}
				else if (i0>=0) {// i and i0 edge are adjacent by the vertex v
					j0 =  i0%2;
					i0 =  i0/2;
					assert( v ==  edges[i0 ].v[j0]);
					edges[i ].adj[ j ] =edges +i0;
					edges[i0].adj[ j0] =edges +i ;
					assert(edges[i0].v[j0] == v);
					v->color = -3;
				}
			}
		}
	}
	else{
		if(verbose>3) printf("      no Edges found\n");
	}

	//EdgeOnGeometricEdge
	if(bamgmesh->EdgesOnGeometricEdge){
		if(verbose>3) printf("      processing EdgesOnGeometricEdge\n");
		int i1,i2,i,j;
		i2=bamgmesh->NumEdgesOnGeometricEdge;
		for (i1=0;i1<i2;i1++) {
			i=(int)bamgmesh->EdgesOnGeometricEdge[i*2+0];
			j=(int)bamgmesh->EdgesOnGeometricEdge[i*2+1];
			if(!(i>0 && j >0 && i <= nbe && j <= Gh.nbe)) {
				throw ErrorException(__FUNCT__,exprintf("EdgesOnGeometricEdge error: We must have : (i>0 && j >0 && i <= nbe && j <= Gh.nbe)"));
			}
			edges[i-1].on = Gh.edges + j-1;
		}
	}
	else{
		if(verbose>3) printf("      no EdgesOnGeometricEdge found\n");
	}

	//SubDomain
	if(bamgmesh->SubDomains){
		Int4 i3,head,sens;
		if(verbose>3) printf("      processing SubDomains\n");
		NbSubDomains=bamgmesh->NumSubDomains;
		subdomains = new SubDomain [ NbSubDomains ];
		for (i=0;i<NbSubDomains;i++) {
			i3  =(int)bamgmesh->SubDomains[i*3+0];
			head=(int)bamgmesh->SubDomains[i*3+1]-1;//C indexing
			sens=(int)bamgmesh->SubDomains[i*3+2];
			if (i3!=23) throw ErrorException(__FUNCT__,exprintf("Bad Subdomain definition: first number should be 3"));
			if (head<0 || head>=nbt) throw ErrorException(__FUNCT__,exprintf("Bad Subdomain definition: head should in [1 %i] (triangle number)",nbt));
			subdomains[i].head = triangles+head;
		}
	}
	else{
		if(verbose>3) printf("      no SubDomains found\n");
	}

	/*Recreate geometry if needed*/
	if(1!=0) {
		printf("Warning: mesh present but no geometry found. Reconstructing...\n");
		/*Recreate geometry: */
		ConsGeometry(bamgopts->MaximalAngleOfCorner*Pi/180);
		Gh.AfterRead();
	}
}

Triangles::Triangles(BamgMesh* bamgmesh, BamgOpts* bamgopts):Gh(*(new Geometry())),BTh(*this){ 

  PreInit(0,"none");
  OnDisk = 1;

  ReadFromMatlabMesh(bamgmesh,bamgopts);
  SetIntCoor();
  FillHoleInMesh();
}

void Geometry::ReadGeometry(BamgGeom* bamggeom,BamgOpts* bamgopts){

	int verbose;
	nbiv=nbv=nbvx=0;
	nbe=nbt=nbtx=0;
	NbOfCurves=0;

	Real8 Hmin = HUGE_VAL;// the infinie value 
	Int4 hvertices=0;
	int i,j,k,n;

	//initialize some variables
	int Version=1,dim=2;
	nbv=bamggeom->NumVertices;
	nbe=bamggeom->NumEdges;
	nbvx = nbv;
	nbiv = nbv;
	verbose=bamgopts->verbose;

	//some checks
	if (bamggeom->NumVertices<=0 || bamggeom->Vertices==NULL){
		throw ErrorException(__FUNCT__,exprintf("the domain provided does not contain any vertex"));
	}
	if (bamggeom->NumEdges<=0 || bamggeom->Edges==NULL){
		throw ErrorException(__FUNCT__,exprintf("the domain provided does not contain any edge"));
	}

	//Vertices
	if (bamggeom->Vertices){
		if(verbose>3) printf("      processing Vertices\n");
		vertices = new GeometricalVertex[nbvx];
		for (i=0;i<nbv;i++) {
			vertices[i].r.x=(double)bamggeom->Vertices[i*3+0];
			vertices[i].r.y=(double)bamggeom->Vertices[i*3+1];
			vertices[i].ReferenceNumber=(Int4)bamggeom->Vertices[i*3+2];
			vertices[i].DirOfSearch=NoDirOfSearch;
			vertices[i].color =0;
			vertices[i].Set();
		}
		//find domain extrema for pmin,pmax
		pmin =  vertices[0].r;
		pmax =  vertices[0].r;
		for (i=0;i<nbv;i++) {
			pmin.x = Min(pmin.x,vertices[i].r.x);
			pmin.y = Min(pmin.y,vertices[i].r.y);
			pmax.x = Max(pmax.x,vertices[i].r.x);
			pmax.y = Max(pmax.y,vertices[i].r.y);
		}
		R2 DD05 = (pmax-pmin)*0.05;
		pmin -=  DD05;
		pmax +=  DD05;
		coefIcoor= (MaxICoor)/(Max(pmax.x-pmin.x,pmax.y-pmin.y));
		if(coefIcoor <=0){
			throw ErrorException(__FUNCT__,exprintf("coefIcoor should be positive"));
		}
	}
	else{
		throw ErrorException(__FUNCT__,exprintf("No Vertex provided"));
	}

	//Edges
	if (bamggeom->Edges){
		int i1,i2;
		R2 zero2(0,0);
		Real4 *len =0;

		if(verbose>3) printf("      processing Edges\n");
		edges = new GeometricalEdge[nbe];

		if (!hvertices) {
			len = new Real4[nbv];
			for(i=0;i<nbv;i++)
			 len[i]=0;
		}

		for (i=0;i<nbe;i++){
			i1=(int)bamggeom->Edges[i*3+0]-1; //-1 for C indexing
			i2=(int)bamggeom->Edges[i*3+1]-1; //-1 for C indexing
			edges[i].ref=(Int4)bamggeom->Edges[i*3+2];
			edges[i].v[0]=  vertices + i1;
			edges[i].v[1]=  vertices + i2;
			R2 x12 = vertices[i2].r-vertices[i1].r;
			Real8 l12=sqrt((x12,x12));
			edges[i].tg[0]=zero2;
			edges[i].tg[1]=zero2;
			edges[i].SensAdj[0] = edges[i].SensAdj[1] = -1;
			edges[i].Adj[0] = edges[i].Adj[1] = 0;
			edges[i].flag = 0;
			if (!hvertices) {
				vertices[i1].color++;
				vertices[i2].color++;
				len[i1] += l12;
				len[i2] += l12;
			}

			Hmin = Min(Hmin,l12);
		}

		// definition  the default of the given mesh size 
		if (!hvertices){
			for (i=0;i<nbv;i++) 
			 if (vertices[i].color > 0) 
			  vertices[i].m=  Metric(len[i] /(Real4) vertices[i].color);
			 else 
			  vertices[i].m=  Metric(Hmin);
			delete [] len;
		}
	}
	else{
		throw ErrorException(__FUNCT__,exprintf("No edges provided"));
	}

	//hVertices
	if(bamggeom->hVertices){
		if(verbose>3) printf("      processing hVertices\n");
		for (i=0;i< nbv;i++){
			vertices[i].m=Metric((Real4)bamggeom->hVertices[i]);
		}
	}
	else{
		if(verbose>3) printf("      no hVertices found\n");
	}

	//MetricVertices
	if(bamggeom->MetricVertices){
		if(verbose>3) printf("      processing MetricVertices\n");
		hvertices=1;
		for (i=0;i< nbv;i++) {
			vertices[i].m = Metric((Real4)bamggeom->MetricVertices[i*3+0],(Real4)bamggeom->MetricVertices[i*3+1],(Real4)bamggeom->MetricVertices[i*3+2]);
		}
	}
	else{
		if(verbose>3) printf("      no MetricVertices found\n");
	}

	//h1h2VpVertices
	if(bamggeom->h1h2VpVertices){
		if(verbose>3) printf("      processing h1h2VpVertices\n");
		Real4 h1,h2,v1,v2;
		hvertices =1;
		for (i=0;i< nbv;i++) {
			h1=(Real4)bamggeom->MetricVertices[i*4+0];
			h2=(Real4)bamggeom->MetricVertices[i*4+1];
			v1=(Real4)bamggeom->MetricVertices[i*4+2];
			v2=(Real4)bamggeom->MetricVertices[i*4+3];
			vertices[i].m = Metric(MatVVP2x2(1/(h1*h1),1/(h2*h2),D2(v1,v2)));
		}
	}
	else{
		if(verbose>3) printf("      no h1h2VpVertices found\n");
	}

	//MaximalAngleOfCorner
	if (bamgopts->MaximalAngleOfCorner){
		if(verbose>3) printf("      processing MaximalAngleOfCorner\n");
		MaximalAngleOfCorner=bamgopts->MaximalAngleOfCorner*Pi/180;
	}
	else{
		if(verbose>3) printf("      no MaximalAngleOfCorner found\n");
	}

	//TangentAtEdges
	if (bamggeom->TangentAtEdges){
		if(verbose>3) printf("      processing TangentAtEdges");
		int n,i,j,k;
		R2 tg;

		n=bamggeom->NumTangentAtEdges;
		for (k=0;k<n;k++) {
			i=(int)bamggeom->TangentAtEdges[k*4+0]-1; //for C indexing
			j=(int)bamggeom->TangentAtEdges[k*4+1]-1; //for C indexing
			tg.x=bamggeom->TangentAtEdges[k*4+2];
			tg.y=bamggeom->TangentAtEdges[k*4+3];
			if (j!=0 && j!=1){
				throw ErrorException(__FUNCT__,exprintf("TangentAtEdges second index should be 1 or 2 only"));
			}
			edges[i].tg[j] = tg;
		}
	}
	else{
		if(verbose>3) printf("      no TangentAtEdges found\n");
	}

	//Corners
	if(bamggeom->Corners){
		if(verbose>3) printf("      processing Corners");
		n=bamggeom->NumCorners;
		for (i=0;i<n;i++) {     
			j=(int)bamggeom->Corners[i]-1; //for C indexing
			if (j>nbv-1 || j<0){
				throw ErrorException(__FUNCT__,exprintf("Bad corner definition: should in [0 %i]",nbv));
			}
			vertices[j].SetCorner();
			vertices[j].SetRequired();  }
	}
	else{
		if(verbose>3) printf("      no Corners found\n");
	}

	//RequiredVertices
	if(bamggeom->RequiredVertices){
		if(verbose>3) printf("      processing RequiredVertices");
		n=bamggeom->NumRequiredVertices;
		for (i=0;i<n;i++) {     
			j=(int)bamggeom->RequiredVertices[i]-1; //for C indexing
			if (j>nbv-1 || j<0){
				throw ErrorException(__FUNCT__,exprintf("Bad RequiredVerticess  definition: should in [0 %i]",nbv));
			}
			vertices[j].SetRequired();  }
	}
	else{
		if(verbose>3) printf("      no RequiredVertices found\n");
	}

	//RequiredEdges
	if(bamggeom->RequiredEdges){
		if(verbose>3) printf("      processing RequiredEdges");
		n=bamggeom->NumRequiredEdges;
		for (i=0;i<n;i++) {     
			j=(int)bamggeom->RequiredEdges[i]-1; //for C indexing
			if (j>nbe-1 || j<0){
				throw ErrorException(__FUNCT__,exprintf("Bad RequiredEdges definition: should in [0 %i]",nbv));
			}
			edges[j].SetRequired();  }
	}
	else{
		if(verbose>3) printf("      no RequiredEdges found\n");
	}

	//SubDomain
	if(bamggeom->SubDomains){
		Int4 i0,i1,i2,i3;
		if(verbose>3) printf("      processing SubDomains\n");
		NbSubDomains=bamggeom->NumSubDomains;
		subdomains = new GeometricalSubDomain[NbSubDomains];
		for (i=0;i<NbSubDomains;i++) {
			i0=(int)bamggeom->SubDomains[i*4+0];
			i1=(int)bamggeom->SubDomains[i*4+1];
			i2=(int)bamggeom->SubDomains[i*4+2];
			i3=(int)bamggeom->SubDomains[i*4+3];
			if (i0!=2) throw ErrorException(__FUNCT__,exprintf("Bad Subdomain definition: first number should be 2 (for Edges)"));
			if (i1>nbe || i1<=0) throw ErrorException(__FUNCT__,exprintf("Bad Subdomain definition: second number should in [1 %i] (edge number)",nbe));
			subdomains[i].edge=edges + (i1-1);
			subdomains[i].sens = (int) i2;
			subdomains[i].ref = i3;
		}
	}
	else{
		if(verbose>3) printf("      no SubDomains found\n");
	}
}

void Geometry::ReadGeometry(MeshIstream & f_in,const char * filename)  
{
	long int verbosity=2;
  if(verbosity>1)
    cout << "  -- ReadGeometry " << filename << endl;
  assert(empty());
  nbiv=nbv=nbvx=0;
  nbe=nbt=nbtx=0;
  NbOfCurves=0;
 // BeginOfCurve=0;
  name=new char [strlen(filename)+1];
  strcpy(name,filename);
  Real8 Hmin = HUGE_VAL;// the infinie value 
//  Real8 MaximalAngleOfCorner = 10*Pi/180; ; 
  Int4 hvertices =0;
  Int4 i;
  Int4 Version,dim=0;
  int field=0;
  int showfield=0;
  int NbErr=0;

  while (f_in.cm()) 
    { 
      field=0;
      // warning ruse for on allocate fiedname at each time 
      char fieldname[256];
      f_in.cm() >> fieldname ;
      f_in.err();
      if(f_in.eof()) break;
//      cout <<  fieldname <<  " line " << LineNumber  <<endl;
      if (!strcmp(fieldname,"MeshVersionFormatted") )
        f_in  >> Version ;
      else if (!strcmp(fieldname,"End"))
	break;
      else if (!strcmp(fieldname,"end"))
	break;
      else if (!strcmp(fieldname,"Dimension"))
        {
          f_in   >>  dim ;
	  if(verbosity>5) 
	    cout << "     Geom Record Dimension dim = " << dim << endl;        
          assert(dim ==2);
         }
       else if (!strcmp(fieldname,"hVertices"))
         { 
	   if (nbv <=0) {
	     cerr<<"Error: the field Vertex is not found before hVertices " << filename<<endl;
	     NbErr++;}       
	   if(verbosity>5) 
	    cout << "     Geom Record hVertices nbv=" << nbv <<  endl;
	   hvertices =1;
           for (i=0;i< nbv;i++) 
	     {
	       Real4 h;
	       f_in  >>  h ; 
	       vertices[i].m = Metric(h);
	     }
	 }
       else if (!strcmp(fieldname,"MetricVertices"))
         { hvertices =1;
	   if (nbv <=0) {
	     cerr<<"Error: the field Vertex is not found before MetricVertices " << filename<<endl;
	     NbErr++;}       
           if(verbosity>5) 
	     cout << "     Geom Record MetricVertices nbv =" << nbv <<  endl;
           for (i=0;i< nbv;i++) 
	     {
	       Real4 a11,a21,a22;
	       f_in  >>  a11 >> a21 >> a22  ; 
	       vertices[i].m = Metric(a11,a21,a22);
	     }
	 }
       else if (!strcmp(fieldname,"h1h2VpVertices"))
         { hvertices =1;
	   if (nbv <=0) {
	     cerr<<"Error: the field Vertex is not found before h1h2VpVertices " << filename<<endl;
	     NbErr++;}       
           if(verbosity>5) 
	     cout << "     Geom Record h1h2VpVertices nbv=" << nbv << endl;

           for (i=0;i< nbv;i++) 
	     {
	       Real4 h1,h2,v1,v2;
	       f_in  >> h1 >> h2 >>v1 >>v2 ; 
	       vertices[i].m = Metric(MatVVP2x2(1/(h1*h1),1/(h2*h2),D2(v1,v2)));
	     }
	 }
      else if (!strcmp(fieldname,"Vertices"))
        { 
          assert(dim ==2);
          f_in   >>  nbv ;
	  //          if(LineError) break;
          nbvx = nbv;
          
          vertices = new GeometricalVertex[nbvx];
	  if(verbosity>5) 
	    cout << "     Geom Record Vertices nbv = " << nbv << "vertices = " << vertices<<endl;
          assert(nbvx >= nbv);
          nbiv = nbv;
          for (i=0;i<nbv;i++) {
            f_in  >> vertices[i].r.x  ;
            // if(LineError) break;
            f_in  >> vertices[i].r.y ;
	    // if(LineError) break;
            f_in >>   vertices[i].ReferenceNumber   ;
            vertices[i].DirOfSearch=NoDirOfSearch;
	    //            vertices[i].m.h = 0;
            vertices[i].color =0;
            vertices[i].Set();}
	  // if(LineError) break;
	    pmin =  vertices[0].r;
	    pmax =  vertices[0].r;
	    // recherche des extrema des vertices pmin,pmax
	    for (i=0;i<nbv;i++) {
	      pmin.x = Min(pmin.x,vertices[i].r.x);
	      pmin.y = Min(pmin.y,vertices[i].r.y);
	      pmax.x = Max(pmax.x,vertices[i].r.x);
	      pmax.y = Max(pmax.y,vertices[i].r.y);
	    }
	    
	      R2 DD05 = (pmax-pmin)*0.05;
	      pmin -=  DD05;
	      pmax +=  DD05;
	    
	    coefIcoor= (MaxICoor)/(Max(pmax.x-pmin.x,pmax.y-pmin.y));
	    assert(coefIcoor >0);
	    if (verbosity>5) {
	      cout << "     Geom: min="<< pmin << "max ="<< pmax << " hmin = " << MinimalHmin() <<  endl;}
        }
      else if(!strcmp(fieldname,"MaximalAngleOfCorner")||!strcmp(fieldname,"AngleOfCornerBound"))
        {         
          f_in >> MaximalAngleOfCorner;
              
	  if(verbosity>5) 
	    cout << "     Geom Record MaximalAngleOfCorner " << MaximalAngleOfCorner <<endl;
          MaximalAngleOfCorner *= Pi/180;
        }
      else if (!strcmp(fieldname,"Edges"))
        {
	  if (nbv <=0) {
	    cerr<<"Error: the field edges is not found before MetricVertices " << filename<<endl;
	    NbErr++;}   
	  else 
	    {
	      int i1,i2;
	      R2 zero2(0,0);
	      f_in   >>  nbe ;
	      
	      edges = new GeometricalEdge[nbe];
	      if(verbosity>5) 
		cout << "     Record Edges: Nb of Edge " << nbe <<endl;
	      assert(edges);
	      assert (nbv >0); 
	      Real4 *len =0;
	      if (!hvertices) 
		{
		  len = new Real4[nbv];
		  for(i=0;i<nbv;i++)
		    len[i]=0;
		}
	      
	      for (i=0;i<nbe;i++) 
		{
		  f_in  >> i1   >> i2 >>  edges[i].ref  ;
		  
		  i1--;i2--; // for C index
		  edges[i].v[0]=  vertices + i1;
		  edges[i].v[1]=  vertices + i2;
		  R2 x12 = vertices[i2].r-vertices[i1].r;
		  Real8 l12=sqrt((x12,x12));
		  edges[i].tg[0]=zero2;
		  edges[i].tg[1]=zero2;
		  edges[i].SensAdj[0] = edges[i].SensAdj[1] = -1;
		  edges[i].Adj[0] = edges[i].Adj[1] = 0;
		  edges[i].flag = 0;
		  if (!hvertices) 
		    {
		      vertices[i1].color++;
		      vertices[i2].color++;
		      len[i1] += l12;
		      len[i2] += l12;
		    }
		  
		  Hmin = Min(Hmin,l12);
		}
	      // definition  the default of the given mesh size 
	      if (!hvertices) 
		{
		  for (i=0;i<nbv;i++) 
		    if (vertices[i].color > 0) 
		      vertices[i].m=  Metric(len[i] /(Real4) vertices[i].color);
		    else 
		      vertices[i].m=  Metric(Hmin);
		  delete [] len;
		  
		  if(verbosity>3) 
		    cout << "     Geom Hmin " << Hmin << endl;
		}
	      
	    }
	}
      else if (!strcmp(fieldname,"EdgesTangence") ||!strcmp(fieldname,"TangentAtEdges")  )
        { 
          int n,i,j,k;
          R2 tg;
          f_in  >> n ;
          
	  if(verbosity>5) 
	    cout << "     Record TangentAtEdges: Nb of Edge " << n <<endl;
          
          for (k=0;k<n;k++)
            {
	      f_in  >>  i  >> j ;
	      f_in >> tg.x  >> tg.y ;
	      assert( i <= nbe );
	      assert( i > 0 );
	      assert ( j == 1 || j==2 );
	      i--;j--;// for C index
	      edges[i].tg[j] = tg;
            }
        }
      else if (!strcmp(fieldname,"Corners"))
        { 
          int i,j,n;
          f_in  >> n ;
	  if(verbosity>5) 
	    cout << "     Record Corner: Nb of Corner " << n <<endl;
          
          for (i=0;i<n;i++) {     
            f_in  >>  j ;
            assert( j <= nbv );
            assert( j > 0 );
            j--;
            vertices[j].SetCorner();
            vertices[j].SetRequired();  }
        }
      else if (!strcmp(fieldname,"RequiredVertices"))
        { 
          int i,j,n;
          f_in  >> n ;

          for (i=0;i<n;i++) {     
            f_in  >>  j ;
            assert( j <= nbv );
            assert( j > 0 );
            j--;
            vertices[j].SetRequired();  }
      }
      else if (!strcmp(fieldname,"RequiredEdges"))
        { 
          int i,j,n;
          f_in  >> n ;

          for (i=0;i<n;i++) {     
            f_in  >>  j ;
            assert( j <= nbe );
            assert( j > 0 );
            j--;
            edges[j].SetRequired();  }
      }
    else if (!strcmp(fieldname,"SubDomain") || !strcmp(fieldname,"SubDomainFromGeom"))
      { 
	f_in   >>  NbSubDomains ;
	if (NbSubDomains>0) 
	  {
	    subdomains = new GeometricalSubDomain[  NbSubDomains];
	    Int4 i0,i1,i2,i3;
	    for (i=0;i<NbSubDomains;i++) 
	      {
		f_in  >> i0  >>i1 
		      >> i2  >>i3 ; 
		
		assert(i0 == 2);
		assert(i1<=nbe && i1>0);
		subdomains[i].edge=edges + (i1-1);
		subdomains[i].sens = (int) i2;
		subdomains[i].ref = i3;
	      }
	  }
      }
      else
	{ // unkown field
	  field = ++showfield;
	  if(showfield==1) // just to show one time 
	    if (verbosity>3)
	      cout << "    Warning we skip the field " << fieldname << " at line " << f_in.LineNumber << endl;
	}
      showfield=field; // just to show one time 
    } // while !eof()
  // generation  de la geometrie 
  // 1 construction des aretes 
  // construire des aretes en chaque sommets 
  
  if (nbv <=0) {
    cerr<<"Error: the field Vertex is not found in " << filename<<endl;
    NbErr++;}
  if(nbe <=0) {
    cerr <<"Error: the field Edges is not found in "<< filename<<endl
      ;NbErr++;}
  if(NbErr) MeshError(1);

 
}


}  // end of namespace bamg 
