# imports
from netCDF4 import Dataset
import numpy as np
import numpy.ma as ma
from os import path, remove
from model import *
import re
from results import *
from m1qn3inversion import m1qn3inversion
from taoinversion import taoinversion


'''
Given a NetCDF4 file, this set of functions will perform the following:
    1. Enter each group of the file.
    2. For each variable in each group, update an empty model with the variable's data
    3. Enter nested groups and repeat
'''


# make a model framework to fill that is in the scope of this file
model_copy = model()


def read_netCDF(filename):
    print('NetCDF42C v1.1.12')

    # check if path exists
    if path.exists(filename):
        print('Opening {} for reading'.format(filename))

        # open the given netCDF4 file
        global NCData   
        NCData = Dataset(filename, 'r')
        # remove masks from numpy arrays for easy conversion
        NCData.set_auto_mask(False)
    else:
        print('The file you entered does not exist or cannot be found in the current directory')
    
    # continuation of band-aid for results class
    try:
        NCData.groups['results']
        make_results_subclasses()
    except:
        pass

    # similarly, we need to check and see if we have an m1qn3inversion class instance
    try:
        NCData.groups['inversion']
        check_inversion_class()
    except:
        pass
    
    # walk through each group looking for subgroups and variables
    for group in NCData.groups.keys():
        # have to send a custom name to this function: filename.groups['group']
        name = "NCData.groups['" + str(group) + "']"
        walk_nested_groups(name)
    
    return model_copy


def make_results_subclasses():
    for subclass in NCData.groups['results'].variables.keys():
        class_instance = subclass + '()'
        class_instance_name = NCData.groups['results'].variables[subclass][:][...].tobytes().decode()
        setattr(model_copy.results, class_instance_name, eval(class_instance))


def check_inversion_class():
    # get the name of the inversion class: either inversion or m1qn3inversion or taoinversion
    inversion_class_is = NCData.groups['inversion'].variables['inversion_class_name'][:][...].tobytes().decode()
    if inversion_class_is == 'm1qn3inversion':
        # if it is m1qn3inversion we need to instantiate that class since it's not native to model()
        model_copy.inversion = m1qn3inversion(model_copy.inversion)
        print('Conversion successful')
    elif inversion_class_is == 'taoinversion':
        # if it is taoinversion we need to instantiate that class since it's not native to model()
        model_copy.inversion = taoinverion()
        print('Conversion successful')
    else: pass



def walk_nested_groups(group_location_in_file):
    # first, we enter the group by: filename.groups['group_name']
    # second we search the current level for variables: filename.groups['group_name'].variables.keys()
    # third we get nested group keys by: filename.groups['group_name'].groups.keys()
    # if a variables exists, copy the data to the model framework by calling copy function
    # if a nested groups exist, repeat all

    for variable in eval(group_location_in_file + '.variables.keys()'):
        location_of_variable_in_file = group_location_in_file + ".variables['" + str(variable) + "']"
        # group_location_in_file is like filename.groups['group1'].groups['group1.1'].groups['group1.1.1']
        # Define the regex pattern to match the groups within brackets
        pattern = r"\['(.*?)'\]"
        # Use regex to find all matches and return something like 'group1.group1.1.group1.1.1 ...' where the last value is the name of the variable
        matches = re.findall(pattern, location_of_variable_in_file)
        variable_name = matches[-1]
        location_of_variable_in_model = '.'.join(matches[:-1])
        copy_variable_data_to_new_model(location_of_variable_in_file, location_of_variable_in_model, variable_name)

    for nested_group in eval(group_location_in_file + '.groups.keys()'):
        new_nested_group = group_location_in_file + ".groups['" + str(nested_group) + "']"
        walk_nested_groups(new_nested_group)



def copy_variable_data_to_new_model(location_of_variable_in_file, location_of_variable_in_model, variable_name):
    # as simple as navigating to the location_of_variable_in_model and setting it equal to the location_of_variable_in_file

    # NetCDF4 has a property called "_FillValue" that sometimes saves empty lists, so we have to catch those
    FillValue = -9223372036854775806

    # results band-aid...
    if str(location_of_variable_in_model + '.' + variable_name) =='results.solutionstep':
        pass
    # handle any strings:
    elif 'char' in eval(location_of_variable_in_file + '.dimensions[0]'):
        setattr(eval('model_copy.' + location_of_variable_in_model), variable_name, eval(location_of_variable_in_file + '[:][...].tobytes().decode()'))
    # handle ndarrays + lists
    elif len(eval(location_of_variable_in_file + '[:]'))>1:
        # check for bool
        try: # there is only one datatype assigned the attribute 'units' and that is bool, so anything else will go right to except
            if eval(location_of_variable_in_file + '.units') == 'bool':
                setattr(eval('model_copy.' + location_of_variable_in_model), variable_name, np.array(eval(location_of_variable_in_file + '[:]'), dtype = bool))
            else:
                setattr(eval('model_copy.' + location_of_variable_in_model), variable_name, eval(location_of_variable_in_file + '[:]'))
        except:
            setattr(eval('model_copy.' + location_of_variable_in_model), variable_name, eval(location_of_variable_in_file + '[:]'))
    # catch everything else
    else:
        # check for FillValue. use try/except because try block will only work on datatypes like int64, float, single element lists/arrays ect and not nd-arrays/n-lists etc
        try:
            # this try block will only work on single ints/floats/doubles and will skip to the except block for all other cases
            if FillValue == eval(location_of_variable_in_file + '[:][0]'):
                setattr(eval('model_copy.' + location_of_variable_in_model), variable_name, [])
            else:
                # we have to convert numpy datatypes to native python types with .item()
                var_to_save = eval(location_of_variable_in_file + '[:][0]')  # note the [0] on the end
                setattr(eval('model_copy.' + location_of_variable_in_model), variable_name, var_to_save.item())
        except:
            setattr(eval('model_copy.' + location_of_variable_in_model), variable_name, eval(location_of_variable_in_file + '[:]'))
    print('Successfully saved ' + location_of_variable_in_model + '.' + variable_name + ' to model.')








