from math import log

import numpy as np

from checkfield import checkfield
from fielddisplay import fielddisplay
from project3d import project3d
from WriteData import WriteData


class SMBsemic(object):
    """SMBsemic class definition

    Usage:
        SMBsemic = SMBsemic()
    """

    def __init__(self, *args):  # {{{
        self.dailysnowfall = np.nan
        self.dailyrainfall = np.nan
        self.dailydsradiation = np.nan
        self.dailydlradiation = np.nan
        self.dailypressure = np.nan
        self.dailyairdensity = np.nan
        self.dailyairhumidity = np.nan
        self.dailytemperature = np.nan
        self.desfac = 0
        self.rlaps = 0
        self.rdl = 0
        self.s0gcm = np.nan
        self.steps_per_step = 1
        self.averaging = 0
        self.requested_outputs = []

        if len(args) == 0:
            self.setdefaultparameters()
        else:
            raise Exception('constructor not supported')
    #}}}

    def __repr__(self):  # {{{
        s = '   surface forcings parameters:\n'
        s += '   Interface for coupling GCM data to the energy balance model SEMIC (Krapp et al (2017) https://doi.org/10.5194/tc-11-1519-2017).\n'
        s += '   The implemented coupling uses daily mean GCM input to calculate yearly mean smb, accumulation, ablation, and surface temperature.\n'
        s += '   smb and temperatures are updated every year\n'
        s += '\n   SEMIC parameters:\n'
        s += '{}\n'.format(fielddisplay(self, 'dailysnowfall', 'daily surface dailysnowfall [m/s]'))
        s += '{}\n'.format(fielddisplay(self, 'dailyrainfall', 'daily surface dailyrainfall [m/s]'))
        s += '{}\n'.format(fielddisplay(self, 'dailydsradiation', 'daily downwelling shortwave radiation [W/m2]'))
        s += '{}\n'.format(fielddisplay(self, 'dailydlradiation', 'daily downwelling longwave radiation [W/m2]'))
        s += '{}\n'.format(fielddisplay(self, 'dailywindspeed', 'daily surface wind speed [m/s]'))
        s += '{}\n'.format(fielddisplay(self, 'dailypressure', 'daily surface pressure [Pa]'))
        s += '{}\n'.format(fielddisplay(self, 'dailyairdensity', 'daily air density [kg/m3]'))
        s += '{}\n'.format(fielddisplay(self, 'dailyairhumidity', 'daily air specific humidity [kg/kg]'))
        s += '{}\n'.format(fielddisplay(self, 'rlaps', 'present day lapse rate (default is 7.4 [degree/km]; Erokhina et al. 2017)'))
        s += '{}\n'.format(fielddisplay(self, 'desfac', 'desertification elevation factor (default is -log(2.0)/1000 [1/km]; Vizcaino et al. 2010)'))
        s += '{}\n'.format(fielddisplay(self, 'rdl', 'longwave downward radiation decrease (default is 0.29 [W/m^2/km]; Marty et al. 2002)'))
        s += '{}\n'.format(fielddisplay(self, 's0gcm', 'GCM reference elevation; (default is 0) [m]'))
        s += '{}\n'.format(fielddisplay(self, 'steps_per_step', 'number of smb steps per time step'))
        s += '{}\n'.format(fielddisplay(self, 'averaging', 'averaging methods from short to long steps'))
        s += '\t\t{}\n'.format('0: Arithmetic (default)')
        s += '\t\t{}\n'.format('1: Geometric')
        s += '\t\t{}\n'.format('2: Harmonic')
        s += '{}\n'.format(fielddisplay(self, 'requested_outputs', 'additional outputs requested'))
        return s
    #}}}

    def extrude(self, md):  # {{{
        self.dailysnowfall = project3d(md, 'vector', self.dailysnowfall, 'type', 'node')
        self.dailyrainfall = project3d(md, 'vector', self.dailyrainfall, 'type', 'node')
        self.dailydsradiation = project3d(md, 'vector', self.dailydsradiation, 'type', 'node')
        self.dailydlradiation = project3d(md, 'vector', self.dailydlradiation, 'type', 'node')
        self.dailywindspeed = project3d(md, 'vector', self.dailywindspeed, 'type', 'node')
        self.dailypressure = project3d(md, 'vector', self.dailypressure, 'type', 'node')
        self.dailyairdensity = project3d(md, 'vector', self.dailyairdensity, 'type', 'node')
        self.dailyairhumidity = project3d(md, 'vector', self.dailyairhumidity, 'type', 'node')
        self.dailytemperature = project3d(md, 'vector', self.dailytemperature, 'type', 'node')
        self.s0gcm = project3d(md, 'vector', self.s0gcm, 'type', 'node')
        return self
    #}}}

    def defaultoutputs(self, md):  # {{{
        return ['SmbMassBalance']
    #}}}

    def initialize(self, md):  # {{{
        if np.all(np.isnan(self.s0gcm)):
            self.s0gcm = np.zeros((md.mesh.numberofvertices))
            print("      no SMBsemic.s0gcm specified: values set as zero")
        return self
    #}}}

    def setdefaultparameters(self):  #{{{
        self.desfac = -log(2.0) / 1000
        self.rlaps = 7.4
        self.rdl = 0.29
        self.requested_outputs = ['default']
        return self
    # }}}

    def checkconsistency(self, md, solution, analyses):  # {{{
        if 'MasstransportAnalysis' in analyses:
            md = checkfield(md, 'fieldname', 'smb.desfac', '<=', 1, 'numel', 1)
            md = checkfield(md, 'fieldname', 'smb.s0gcm', '>=', 0, 'NaN', 1, 'Inf', 1, 'size', [md.mesh.numberofvertices, 1])
            md = checkfield(md, 'fieldname', 'smb.rlaps', '>=', 0, 'numel', 1)
            md = checkfield(md, 'fieldname', 'smb.rdl', '>=', 0, 'numel', 1)
            md = checkfield(md, 'fieldname', 'smb.dailysnowfall', 'timeseries', 1, 'NaN', 1, 'Inf', 1)
            md = checkfield(md, 'fieldname', 'smb.dailyrainfall', 'timeseries', 1, 'NaN', 1, 'Inf', 1)
            md = checkfield(md, 'fieldname', 'smb.dailydsradiation', 'timeseries', 1, 'NaN', 1, 'Inf', 1)
            md = checkfield(md, 'fieldname', 'smb.dailydlradiation', 'timeseries', 1, 'NaN', 1, 'Inf', 1)
            md = checkfield(md, 'fieldname', 'smb.dailywindspeed', 'timeseries', 1, 'NaN', 1, 'Inf', 1)
            md = checkfield(md, 'fieldname', 'smb.dailypressure', 'timeseries', 1, 'NaN', 1, 'Inf', 1)
            md = checkfield(md, 'fieldname', 'smb.dailyairdensity', 'timeseries', 1, 'NaN', 1, 'Inf', 1)
            md = checkfield(md, 'fieldname', 'smb.dailyairhumidity', 'timeseries', 1, 'NaN', 1, 'Inf', 1)
            md = checkfield(md, 'fieldname', 'smb.dailytemperature', 'timeseries', 1, 'NaN', 1, 'Inf', 1)
        md = checkfield(md, 'fieldname', 'smb.steps_per_step', '>=', 1, 'numel', [1])
        md = checkfield(md, 'fieldname', 'smb.averaging', 'numel', [1], 'values', [0, 1, 2])
        md = checkfield(md, 'fieldname', 'smb.requested_outputs', 'stringrow', 1)
        return md
    # }}}

    def marshall(self, prefix, md, fid):  # {{{
        yts = md.constants.yts
        WriteData(fid, prefix, 'name', 'md.smb.model', 'data', 12, 'format', 'Integer')
        WriteData(fid, prefix, 'object', self, 'class' ,'smb', 'fieldname', 'desfac', 'format', 'Double')
        WriteData(fid, prefix, 'object', self, 'class' ,'smb', 'fieldname', 's0gcm', 'format', 'DoubleMat', 'mattype', 1)
        WriteData(fid, prefix, 'object', self, 'class' ,'smb', 'fieldname', 'rlaps', 'format', 'Double')
        WriteData(fid, prefix, 'object', self, 'class' ,'smb', 'fieldname', 'rdl', 'format', 'Double')
        WriteData(fid, prefix, 'object', self, 'class' ,'smb', 'fieldname', 'dailysnowfall', 'format', 'DoubleMat', 'mattype', 1, 'timeserieslength', md.mesh.numberofvertices + 1, 'yts', yts)
        WriteData(fid, prefix, 'object', self, 'class' ,'smb', 'fieldname', 'dailyrainfall', 'format', 'DoubleMat', 'mattype', 1, 'timeserieslength', md.mesh.numberofvertices + 1, 'yts', yts)
        WriteData(fid, prefix, 'object', self, 'class' ,'smb', 'fieldname', 'dailydsradiation', 'format', 'DoubleMat', 'mattype', 1, 'timeserieslength', md.mesh.numberofvertices + 1, 'yts', yts)
        WriteData(fid, prefix, 'object', self, 'class' ,'smb', 'fieldname', 'dailydlradiation', 'format', 'DoubleMat', 'mattype', 1, 'timeserieslength', md.mesh.numberofvertices + 1, 'yts', yts)
        WriteData(fid, prefix, 'object', self, 'class' ,'smb', 'fieldname', 'dailywindspeed', 'format', 'DoubleMat', 'mattype', 1, 'timeserieslength', md.mesh.numberofvertices + 1, 'yts', yts)
        WriteData(fid, prefix, 'object', self, 'class' ,'smb', 'fieldname', 'dailypressure', 'format', 'DoubleMat', 'mattype', 1, 'timeserieslength', md.mesh.numberofvertices + 1, 'yts', yts)
        WriteData(fid, prefix, 'object', self, 'class' ,'smb', 'fieldname', 'dailyairdensity', 'format', 'DoubleMat', 'mattype', 1, 'timeserieslength', md.mesh.numberofvertices + 1, 'yts', yts)
        WriteData(fid, prefix, 'object', self, 'class' ,'smb', 'fieldname', 'dailyairhumidity', 'format', 'DoubleMat', 'mattype', 1, 'timeserieslength', md.mesh.numberofvertices + 1, 'yts', yts)
        WriteData(fid, prefix, 'object', self, 'class' ,'smb', 'fieldname', 'dailytemperature', 'format', 'DoubleMat', 'mattype', 1, 'timeserieslength', md.mesh.numberofvertices + 1, 'yts', yts)
        WriteData(fid, prefix, 'object', self, 'fieldname', 'steps_per_step', 'format', 'Integer')
        WriteData(fid, prefix, 'object', self, 'fieldname', 'averaging', 'format', 'Integer')

        # Process requested outputs
        outputs = self.requested_outputs
        indices = [i for i, x in enumerate(outputs) if x == 'default']
        if len(indices) > 0:
            outputscopy = outputs[0:max(0, indices[0] - 1)] + self.defaultoutputs(md) + outputs[indices[0] + 1:]
            outputs = outputscopy
        WriteData(fid, prefix, 'data', outputs, 'name', 'md.smb.requested_outputs', 'format', 'StringArray')
    # }}}
