%MODELLIST class definition
%
%   Usage:
%      modellist=modellist({md1 md2 md3});

classdef modellist
	properties (SetAccess=public) 
		models  = cell(0,1);
		cluster = generic();
		name = '';
	end
	properties (SetAccess=public) 
		runtimename = '';
	end
	methods
		function md_list=modelsextract(md,flags,minel,varargin) % {{{
			%modelsextract - extract several self contained models according to a list of element flags.
			%
			%   The difference between this routine and the modelextract.m routine (without an 's') is that 
			%   as many models are extracted as there are closed contours defined in area. 
			%   This routine is needed for example when doing data assimilation of ice shelves in Antarctica. 
			%   Many independent ice shelves are present, and we don't want data assimilation on one ice shelf 
			%   to be hindered by another totally independent ice shelf.
			%
			%   Usage:
			%      md_list=modelsextract(md,elementfalgs,minel);
			%
			%   Examples:
			%      md_list=modelsextract(md...,,1000);
			%
			%   See also: EXTRUDE, COLLAPSE, MODELEXTRACT

			disp('selecting pools of elements');
			%go through flags and build as many independent element flags as there are groups of connected 1s
			%in flags.

			%2D or 3D?
			if dimension(md.mesh)==3,
				numberofelements=md.mesh.numberofelements2d; %this will be forgotten when we get out.
				flags=project2d(md,flags,1);
			else
				numberofelements=md.mesh.numberofelements;
			end

			%recover extra arguments: 
			distance=0;
			if nargin==4,
				distance=varargin{1};
			end

			flag_list=cell(0,1);

			for i=1:size(flags,1),

				if (flags(i)),

					%ok, we are sure element i is part of a new pool.
					pool=zeros(numberofelements,1);
					pool=PropagateFlagsFromConnectivity(md.mesh.elementconnectivity,pool,i,flags);
					flag_list{end+1,1}=pool;

					%speed up rest of computation by taking pool out of flags: 
					pos=find(pool);flags(pos)=0;

				end
			end

			%go through flag_list and discard any pool of less than minel elements: 
			ex_pos=[];
			for i=1:length(flag_list),
				if length(find(flag_list{i}))<minel,
					ex_pos=[ex_pos; i];
				end
			end
			flag_list(ex_pos)=[];

			%now, if distance was specified, expand the flag_list by distance km: 
			if distance,
				for i=1:length(flag_list),
					flag_list{i}=PropagateFlagsUntilDistance(md,flag_list{i},distance);
				end
			end

			%now, go use the pools of flags to extract models: 
			disp(['extracting ' num2str(size(flag_list,1)) ' models']);
			models=cell(0,1);

			for i=1:size(flag_list,1),
				disp(['   ' num2str(i) '/' num2str(size(flag_list,1))]);
				if dimension(md.mesh)==3,
					flags2d=flag_list{i};
					realflags=project3d(md,flags2d,'element');
				else
					realflags=flag_list{i};
				end
				models{end+1,1}=modelextract(md,realflags);
			end

			%return model list
			md_list=modellist(models);

		end %end of this function }}}
		function md_list=modelsextractfromdomains(md,directory) % {{{
			%modelsextractfromdomains- extract several self contained models according to a list of domains
			%
			%   Usage:
			%      md_list=modelsextractfromdomains(md,'Basins/');
			%
			%   Examples:
			%      md_list=modelsextract(md,'Basins/');
			%
			%   See also: MODELSEXTRACTS, MODELEXTRACT

			%go into directory and get list of files.
			cd(directory);
			basins=listfiles;
			cd ..

			models=cell(0,1);
			for i=1:length(basins),
				models{end+1,1}=modelextract(md,[directory '/' basins{i}]);
			end

			%return model list: 
			md_list=modellist(models);

		end % }}}
		function self = modellist(name,cluster,varargin) % {{{

			%initialize list
			if nargin==0,
				%Do nothing,
			else

				self.name=name;
				self.cluster=cluster;
				celllist=varargin{1};

				%check on size of cell list: 
				if (size(celllist,2)~=1),
					error('modellist constructor error message: list of models should be a cell list of column size 1');
				end

				%check that only models are in the celllist: 
				for i=1:size(celllist,1),
					if ~isa(celllist{i},'model')
						error(['modellist constructor error message: element ' num2str(i) ' of cell list is not a model!']);
					end
				end
				self.models  = celllist;
			end
		end % }}}
		function val = get(self, propName)% {{{
		%GET - gets model propertie from a specified object ans returns the value
		% 
		%   Usage:
		%      val = get(a, propName)

			switch propName
				case 'numberofelements'
					val = self.numberofelements;
				case 'numberofnodes'
					val = self.numberofnodes;
				case 'elements' 
					val = self.elements;
				case 'x' 
					val = self.x;
				case 'y' 
					val = self.y;
				case 'z' 
					val = self.z;
				otherwise
					error(['get error message: ' propName,' is not a valid model property'])
			end
		end % }}}
		function self = loadmultipleresultsfromcluster(self) % {{{
			%LOADMULTIPLERESULTSFROMCLUSTER - load multiple results of solution sequences from cluster
			%
			%   Usage:
			%      self=loadresultsfromcluster(self);

			nummodels=length(self.models);

			%Get cluster settings
			cluster=self.cluster;
			name=self.name;
			cluster_rc_location=which('cluster.rc');
			[codepath,executionpath]=ClusterParameters(cluster,cluster_rc_location);

			%Remote tar: 
			disp('tarring results');
			issmssh(cluster,['"cd ' executionpath '/' name ' && rm -rf file_list.txt ModelResults.tar.gz && find -iname ''*-*vs*.outbin'' > file_list.txt && tar zcvf ModelResults.tar.gz --files-from file_list.txt  && rm -rf file_list.txt "']);

			%copy results from cluster to present directory
			scpin(cluster, [executionpath '/' name], {'ModelResults.tar.gz'});

			%untar:
			!tar -zxvf ModelResults.tar.gz

			%ok, go through list and load results from disk: 
			for i=1:nummodels,
				%load  results for this model
				self.models{i}=loadresultsfromdisk(self.models{i},[name '-' num2str(i) 'vs' num2str(nummodels) '.outbin']);

				delete([name '-' num2str(i) 'vs' num2str(nummodels) '.outbin']);
			end

			%erase files 
			delete('ModelResults.tar.gz');
		end % }}}
		function self = solve(self,solutiontype,varargin)% {{{
			%SOLVE - apply solution sequence for  a list of models. Used in batch mode.
			%
			%   Usage:
			%      self=solve(self,solutiontype,varargin)
			%      where varargin is a lit of paired arguments. 
			%
			%   Examples:
			%      self=solve(self,'tr','batch','yes');

			disp('Starting multiple solve');
			

			%come up with a run time name: 
			c=clock;
			self.runtimename=sprintf('%s-%02i-%02i-%04i-%02i-%02i-%02i-%i',self.name,c(2),c(3),c(1),c(4),c(5),floor(c(6)),feature('GetPid'));

			
			%recover options
			options=pairoptions(varargin{:},'solutionstring',solutiontype);

			%length of list
			nummodels=length(self.models);

			%come up with unique names for all our models: 
			for i=1:nummodels,
				self.models{i}.miscellaneous.name=sprintf('%s%i',self.models{i}.miscellaneous.name,i);
			end

			names={};
			
			%solve in batch mode: 
			for i=1:nummodels,

				%model
				mdex=self.models{i};

				%call solve in batch mode:
				disp(sprintf('   marshalling model %i/%i',i,nummodels));
				if strcmpi(self.cluster,oshostname),
					mdex=solve(mdex,varargin{:});
				else
					mdex=solve(mdex,solutiontype,varargin{:},'batch','yes');
				end

				%feed back
				self.models{i}=mdex;
				
				%record name: 
				names{end+1}=sprintf('%s.tar.gz',mdex.private.runtimename);
			end

			%locally, we are done.
			if strcmpi(self.cluster,oshostname),
				return
			end
			
			%tar files: 
			tar('ModelList.tar.gz',names);

			%still have to build a launching script.
			self.BuildQueueingScript(solutiontype);

			%upload data
			self.UploadQueueJob();

			%launch jobs on remote cluster
			self.LaunchQueueJob();

			%erase files: 
			delete('ModelList.tar.gz');

		end % }}}
		function self = loadresults(self) % {{{

			nummodels=length(self.models);
			cluster=self.cluster;
			
			%solve in batch mode: 
			directory=sprintf('%s/Interactive%i',cluster.executionpath,cluster.interactive);
			filelist={};
			for i=1:nummodels,
				filelist{end+1}= sprintf('%s/%s.outbin',self.models{i}.private.runtimename,self.models{i}.miscellaneous.name);
			end
			issmscpin(cluster.name,cluster.login,cluster.port,directory,filelist);

			%load results: 
			for i=1:nummodels,
				self.models{i}=loadresultsfromdisk(self.models{i},sprintf('%s.outbin',self.models{i}.miscellaneous.name));
			end

		end %}}}
		function num= nummodels(self) % {{{
			num=length(self.models);
		end %}}}
		function BuildQueueingScript(self,solution)% {{{

			scriptname=[self.runtimename '.queue'];
			cluster=self.cluster;
			dirname=self.runtimename;
			modelname=self.name;
			nummodels=length(self.models);

			%recover and process solve options %{{{
			if strcmpi(solution,'sb') || strcmpi(solution,'Stressbalance')
				solution = 'StressbalanceSolution';
			elseif strcmpi(solution,'mt') || strcmpi(solution,'Masstransport')
				solution = 'MasstransportSolution';
			elseif strcmpi(solution,'oceant') || strcmpi(solution,'Oceantransport')
				solution = 'OceantransportSolution';
			elseif strcmpi(solution,'th') || strcmpi(solution,'Thermal')
				solution = 'ThermalSolution';
			elseif strcmpi(solution,'ss') || strcmpi(solution,'Steadystate') 
				solution = 'SteadystateSolution';
			elseif strcmpi(solution,'tr') || strcmpi(solution,'Transient')
				solution = 'TransientSolution';
			elseif strcmpi(solution,'mc') || strcmpi(solution,'Balancethickness')
				solution = 'BalancethicknessSolution';
			elseif strcmpi(solution,'Balancethickness2')
				solution = 'Balancethickness2Solution';
			elseif strcmpi(solution,'mcsoft') || strcmpi(solution,'BalancethicknessSoft')
				solution = 'BalancethicknessSoftSolution';
			elseif strcmpi(solution,'bv') || strcmpi(solution,'Balancevelocity')
				solution = 'BalancevelocitySolution';
			elseif strcmpi(solution,'bsl') || strcmpi(solution,'BedSlope')
				solution = 'BedSlopeSolution';
			elseif strcmpi(solution,'ssl') || strcmpi(solution,'SurfaceSlope')
				solution = 'SurfaceSlopeSolution';
			elseif strcmpi(solution,'hy') || strcmpi(solution,'Hydrology')
				solution = 'HydrologySolution';
			elseif strcmpi(solution,'da') || strcmpi(solution,'DamageEvolution')
				solution = 'DamageEvolutionSolution';
			elseif strcmpi(solution,'gia') || strcmpi(solution,'Gia')
				solution = 'GiaSolution';
			elseif strcmpi(solution,'lv') || strcmpi(solution,'Love')
				solution = 'LoveSolution';
			elseif strcmpi(solution,'esa') || strcmpi(solution,'Esa')
				solution = 'EsaSolution';
			elseif strcmpi(solution,'smp') || strcmpi(solution,'Sampling')
				solution = 'SamplingSolution';    
			else
				error(['solution ' solution ' not supported!']);
			end
			%}}}

			executable='issm.exe';

			fid=fopen([modelname '.queue'],'w');
			fprintf(fid,'#PBS -S /bin/bash\n');
			fprintf(fid,'#PBS -l select=%i:ncpus=%i:model=%s\n',cluster.numnodes,cluster.cpuspernode,cluster.processor);
			fprintf(fid,'#PBS -l walltime=%i\n',cluster.time*60); %walltime is in seconds.
			fprintf(fid,'#PBS -q %s\n',cluster.queue);
			fprintf(fid,'#PBS -W group_list=%s\n',cluster.grouplist);
			fprintf(fid,'#PBS -m e\n');
			fprintf(fid,'#PBS -o %s/%s/%s.outlog \n',cluster.executionpath,dirname,modelname);
			fprintf(fid,'#PBS -e %s/%s/%s.errlog \n\n',cluster.executionpath,dirname,modelname);
			fprintf(fid,'. /usr/share/modules/init/bash\n\n');
			for i=1:numel(cluster.modules), fprintf(fid,['module load ' cluster.modules{i} '\n']); end
			fprintf(fid,'export PATH="$PATH:."\n\n');
			fprintf(fid,'export MPI_LAUNCH_TIMEOUT=520\n');
			fprintf(fid,'export MPI_GROUP_MAX=64\n\n');
			fprintf(fid,'export ISSM_DIR="%s"\n',cluster.srcpath); %FIXME

			fprintf(fid,'source $ISSM_DIR/etc/environment.sh\n');       %FIXME
			fprintf(fid,'cd %s/%s/\n\n',cluster.executionpath,dirname);

			for i=1:nummodels,
				fprintf(fid,'cd %s && mpiexec -np %i /u/scicon/tools/bin/mbind.x %s/%s %s %s/%s/%s && cd ..\n',self.models{i}.private.runtimename,cluster.nprocs(),cluster.codepath,executable,solution,cluster.executionpath,self.runtimename,self.models{i}.private.runtimename);
			end
			fclose(fid);

			%in interactive mode, create a run file, and errlog and outlog file
			if cluster.interactive,
				fid=fopen([modelname '.run'],'w');
				for i=1:nummodels,
					fprintf(fid,'cd %s && mpiexec -np %i /u/scicon/tools/bin/mbind.x %s/%s %s %s/%s %s && cd ..\n',self.models{i}.private.runtimename,cluster.nprocs(),cluster.codepath,executable,solution,[cluster.executionpath '/Interactive' num2str(cluster.interactive)] ,self.models{i}.private.runtimename,self.models{i}.miscellaneous.name);
				end
				fclose(fid);
				fid=fopen([modelname '.errlog'],'w'); fclose(fid);
				fid=fopen([modelname '.outlog'],'w'); fclose(fid);
			end
		end% }}}
		function UploadQueueJob(self) % {{{

			cluster=self.cluster;

			%compress the files into one zip.
			compressstring=sprintf('tar -zcf %s.tar.gz ModelList.tar.gz %s.queue ',self.name,self.name);
			if cluster.interactive,
				compressstring = sprintf('%s %s.run %s.errlog %s.outlog',compressstring, self.name,self.name,self.name);
			end
			system(compressstring);

			disp('uploading input file and queueing script');
			if cluster.interactive,
				directory=[cluster.executionpath '/Interactive' num2str(cluster.interactive)];
			else 
				directory=cluster.executionpath;
			end

			issmscpout(cluster.name,directory,cluster.login,cluster.port,{[self.name '.tar.gz']});

		end
		%}}}
		function LaunchQueueJob(self)% {{{
			%LAUNCHQUEUEJOB - LAUNCH MULTIPLE QUEUEING SCRIPT ON PFE CLUSTER
			%
			%   Usage:
			%      LaunchQueueJob(self)

			cluster=self.cluster;
			modelname=self.name;
			nummodels=length(self.models);

			disp('uploading inputs files');
			if cluster.interactive,
				launchcommand=sprintf('cd %s/Interactive%i && tar -zxf %s.tar.gz && tar -zxvf ModelList.tar.gz ',cluster.executionpath,cluster.interactive,modelname);
				for i=1:nummodels,
					launchcommand=sprintf('%s && mkdir %s && cd %s && mv ../%s.tar.gz ./ && tar -zxvf %s.tar.gz && cd .. ',launchcommand,self.models{i}.private.runtimename, ...
						self.models{i}.private.runtimename, self.models{i}.private.runtimename, self.models{i}.private.runtimename);
				end
			else
				launchcommand=['cd ' cluster.executionpath ' && mkdir ' self.runtimename...
					' && cd ' self.runtimename ' && mv ../' self.runtimename '.tar.gz ./ && tar -zxf ' dirname '.tar.gz && qsub ' modelname '.queue '];
			end

			disp('launching solution sequence on remote cluster');
			issmssh(cluster.name,cluster.login,cluster.port,launchcommand);
		end
	end
end
