function plot_overlay(md,data,options,width,i)
%PLOT_OVERLAY - superimpose radar image to a given field
%
%   Usage:
%      plot_overlay(md,options,width,i);
%
%   See also: PLOTMODEL

%process mesh and data
[x y z elements is2d]=processmesh(md,options);
[data isongrid isquiver]=processdata(md,data,options);

%check is2d
if ~is2d, 
	error('buildoverlay error message: overlay not supported for 3d meshes, project on a layer');
end
if isquiver
	error('buildoverlay error message: overlay not supported for quiver plots');
end

%get xlim and ylim
xlim=getfieldvalue(options,'xlim',[min(md.x) max(md.x)]);
ylim=getfieldvalue(options,'ylim',[min(md.y) max(md.y)]);

%radar power
if ~any(isnan(md.sarxm)) & ~any(isnan(md.sarym)) & ~any(isnan(md.sarpwr)),
	disp('plot_overlay info: the radar image held by the model is being used');
else
	t1=clock; fprintf('%s','Extracting radar image...');
	md=radarpower(md,getfieldvalue(options,'hem','s'),xlim,ylim,getfieldvalue(options,'highres',0));
	t2=clock;fprintf('%s\n',[' done (' num2str(etime(t2,t1)) ' seconds)']);
end

%mesh2grid
redo=1;
if (ischar(data) & isstruct(md.mesh2grid_parameters) & ismember(data,mesh2grid_parameters)),
	choice=input(['mesh2grid has already been called for the parameter ''' data '''. Do you want to call it again (y/n)?'],'s');
	if strcmp(choice,'y')
		disp('use previous mesh2grid result');
		x_m=md.mesh2grid_x_m;
		y_m=md.mesh2grid_y_m;
		data_grid=md.mesh2grid_data;
		redo=0;
	end
end
if redo,
	%apply caxis if required
	if exist(options,'caxis'),
		caxis_opt=getfieldvalue(options,'caxis');
		data(find(data<caxis_opt(1)))=caxis_opt(1);
		data(find(data>caxis_opt(2)))=caxis_opt(2);
	end

	%use InterpFromMeshToGrid to get an gridded data to display using imagesc
	cornereast =min(xlim);
	cornernorth=max(ylim);
	xspacing=(max(xlim)-min(xlim))/(length(md.sarxm));
	yspacing=(max(ylim)-min(ylim))/(length(md.sarym));
	nlines=length(md.sarym);
	ncols =length(md.sarxm);
	t1=clock; fprintf('%s','Interpolating data on grid...');
	[x_m y_m data_grid]=InterpFromMeshToGrid(elements,x,y,data,cornereast,cornernorth,xspacing,yspacing,nlines,ncols,0);
	t2=clock;fprintf('%s\n',[' done (' num2str(etime(t2,t1)) ' seconds)']);
end

%Ok, we have two images, double format: 
radar=md.sarpwr;

%Build hsv color image from radar and results
transparency=getfieldvalue(options,'alpha',1.5);  %Rignot's setting: 1.5
border=getfieldvalue(options,'border',0);

%intensity
v=radar/max(radar(:));

%hue
%cut results under 1.5, and log
data_grid(find(data_grid<1.5))=1.5;
h=bytscl(log(data_grid))/(255+1); %1 offset on colormap

%saturation
s=(0.5+data_grid/125)/transparency;s(find(s>1))=1;s(find(s<0))=0;
s(find(data_grid==1.5))=0;

%Include border
%v((1:border),:)=0;  v((end-border+1:end),:)=0; v(:,1:border)=0;v(:,(end-border+1:end))=0;

%Transform hsv to rgb
image_hsv=zeros(size(data_grid,1),size(data_grid,2),3);
image_hsv(:,:,1)=h;
image_hsv(:,:,2)=s;
image_hsv(:,:,3)=v;
image_rgb=hsv2rgb(image_hsv);

%Select plot area 
subplot(width,width,i);

%Plot: 
imagesc(md.sarxm,md.sarym,image_rgb);set(gca,'YDir','normal');

%last step: mesh overlay?
if exist(options,'edgecolor')
	A=elements(:,1); B=elements(:,2); C=elements(:,3); 
	patch( 'Faces', [A B C], 'Vertices', [x y z],'FaceVertexCData', [1 1 1],'FaceColor','none','EdgeColor',getfieldvalue(options,'edgecolor'));
end

%Apply options, without colorbar and without grid
iscolorbar=getfieldvalue(options,'colorbar',1);
options=changefieldvalue(options,'colorbar',0);
options=addfielddefault(options,'axis','equal off');
options=addfielddefault(options,'colorbarpos',[0.80 0.70 0.02 0.15]);
applyoptions(md,data,options);

%colorbar
if iscolorbar,

	%create colorbar with correct colors and position
	colorbar_rgb=buildoverlaycolorbar(md,data,getfieldvalue(options,'aplha',1.5));
	colorbar_handle=colorbar; 
	colorbar_image_handle=get(colorbar_handle,'Children'); 
	set(colorbar_image_handle,'CData',colorbar_rgb);
	set(colorbar_handle,'Position',getfieldvalue(options,'colorbarpos'));
	set(colorbar_handle,'FontSize',getfieldvalue(options,'fontsize',14));

	%modify ticks.
	scalestring=get(colorbar_handle,'YTickLabel');
	numvalues=length(get(colorbar_handle,'YTick'));

	scalestring=[];
	scaleminmax=[max(min(data),1.5) max(data)];
	for i=1:numvalues,
		fraction=(i-1)/(numvalues-1);
		scalevalues(i)=round_ice(scaleminmax(1)+(scaleminmax(2)-scaleminmax(1))*fraction,2);
		scalestring=[scalestring; sprintf('%8.4g',scalevalues(i))];
	end
	set(colorbar_handle,'YTickLabel',scalestring);
	set(colorbar_handle,'YColor','y');
end
