/*!\file FrontalForcingsx
 * \brief: compute ice frontal melting rate
 */

#include "./FrontalForcingsx.h"
#include "../../shared/shared.h"
#include "../../toolkits/toolkits.h"
#include "../../shared/Random/random.h"

void FrontalForcingsx(FemModel* femmodel){/*{{{*/

	/*Recover melt_parameterization*/
	int melt_parameterization;
	femmodel->parameters->FindParam(&melt_parameterization,FrontalForcingsParamEnum);

	/*Calculate melting rate*/
	switch(melt_parameterization){
		case FrontalForcingsDefaultEnum:
			break;
		case FrontalForcingsRignotarmaEnum:
			Thermalforcingarmax(femmodel);
			/*Do not break here, call IcefrontAreax(),RignotMeltParameterizationx()*/
		case FrontalForcingsRignotEnum:
			femmodel->IcefrontAreax();
			femmodel->RignotMeltParameterizationx();
			break;
		default:
			_error_("Frontal forcings "<<EnumToStringx(melt_parameterization)<<" not supported yet");
	}
}/*}}}*/
void Thermalforcingarmax(FemModel* femmodel){/*{{{*/

   /*Get time parameters*/
   IssmDouble time,dt,starttime,tstep_arma;
   femmodel->parameters->FindParam(&time,TimeEnum);
   femmodel->parameters->FindParam(&dt,TimesteppingTimeStepEnum);
   femmodel->parameters->FindParam(&starttime,TimesteppingStartTimeEnum);
   femmodel->parameters->FindParam(&tstep_arma,FrontalForcingsARMATimestepEnum);

   /*Determine if this is a time step for the ARMA model*/
   bool isstepforarma = false;

   #ifndef _HAVE_AD_
   if((fmod(time,tstep_arma)<fmod((time-dt),tstep_arma)) || (time<=starttime+dt) || tstep_arma==dt) isstepforarma = true;
   #else
   _error_("not implemented yet");
   #endif

   /*Load parameters*/
	bool isstochastic;
   bool istfstochastic = false;
	int M,N,Narlagcoefs,Nmalagcoefs,arorder,maorder,numbasins,my_rank;
   femmodel->parameters->FindParam(&numbasins,FrontalForcingsNumberofBasinsEnum);
   femmodel->parameters->FindParam(&arorder,FrontalForcingsARMAarOrderEnum);
   femmodel->parameters->FindParam(&maorder,FrontalForcingsARMAarOrderEnum);
   IssmDouble tinit_arma;
   IssmDouble* termconstant  = NULL;
   IssmDouble* trend         = NULL;
   IssmDouble* arlagcoefs    = NULL;
   IssmDouble* malagcoefs    = NULL;

	femmodel->parameters->FindParam(&tinit_arma,FrontalForcingsARMAInitialTimeEnum);
   femmodel->parameters->FindParam(&termconstant,&M,FrontalForcingsARMAconstEnum);                  _assert_(M==numbasins);
   femmodel->parameters->FindParam(&trend,&M,FrontalForcingsARMAtrendEnum);                         _assert_(M==numbasins);
   femmodel->parameters->FindParam(&arlagcoefs,&M,&Narlagcoefs,FrontalForcingsARMAarlagcoefsEnum);  _assert_(M==numbasins); _assert_(Narlagcoefs==arorder);
   femmodel->parameters->FindParam(&malagcoefs,&M,&Nmalagcoefs,FrontalForcingsARMAmalagcoefsEnum);  _assert_(M==numbasins); _assert_(Nmalagcoefs==maorder);

	femmodel->parameters->FindParam(&isstochastic,StochasticForcingIsStochasticForcingEnum);
	if(isstochastic){
		int  numstochasticfields;
		int* stochasticfields;
		femmodel->parameters->FindParam(&numstochasticfields,StochasticForcingNumFieldsEnum);
		femmodel->parameters->FindParam(&stochasticfields,&N,StochasticForcingFieldsEnum); _assert_(N==numstochasticfields);
		for(int i=0;i<numstochasticfields;i++){
			if(stochasticfields[i]==FrontalForcingsRignotarmaEnum) istfstochastic = true;
		}
		xDelete<int>(stochasticfields);
	}
   /*Time elapsed with respect to ARMA model initial time*/
   IssmDouble telapsed_arma = time-tinit_arma;

   /*Loop over each element to compute Thermal Forcing at vertices*/
   for(Object* &object:femmodel->elements->objects){
      Element* element = xDynamicCast<Element*>(object);
      element->ArmaProcess(isstepforarma,arorder,maorder,telapsed_arma,tstep_arma,termconstant,trend,arlagcoefs,malagcoefs,istfstochastic,FrontalForcingsRignotarmaEnum);
   }

   /*Cleanup*/
   xDelete<IssmDouble>(termconstant);
   xDelete<IssmDouble>(trend);
   xDelete<IssmDouble>(arlagcoefs);
   xDelete<IssmDouble>(malagcoefs);
}/*}}}*/
