%MMEOFFLINESOLIDEARTHSOLUTION class definition
%
%   Usage:
%      addsol=mmeofflinesolidearthsolution(); where the offline solid earth solutions %                               are based on a multi-model ensemble (ex: Caron et al 2017 statistics) 

classdef mmeofflinesolidearthsolution < offlinesolidearthsolution 
	properties (SetAccess=public) 
		modelid; %index into the multi-model ensemble, each ensemble variable being defined 
	         %in the father class.
	end
	methods
		function self = mmeofflinesolidearthsolution(varargin) % {{{
			switch nargin
				case 0
					self=setdefaultparameters(self);
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function self = setdefaultparameters(self) % {{{
			self.setdefaultparameters@offlinesolidearthsolution();
			self.modelid=0;
		end % }}}
		function md = checkconsistency(self,md,solution,analyses) % {{{

			seast=length(self.displacementeast);
			snorth=length(self.displacementnorth);
			sup=length(self.displacementup);
			sgeoid=length(self.geoid);

			if (seast-snorth)~=0,
				error('mmeofflinesolidearthsolution checkconsistency error message: displacementeast and displacementnorth should be the same size');
			end

			if (seast-sup)~=0,
				error('mmeofflinesolidearthsolution checkconsistency error message: displacementeast and displacementup should be the same size');
			end

			if (seast-sgeoid)~=0,
				error('mmeofflinesolidearthsolution checkconsistency error message: displacementeast and geoid should be the same size');
			end
			
			md = checkfield(md,'field',self.modelid,'NaN',1,'Inf',1,'>=',1,'<=',length(self.displacementeast));

			for i=1:seast,
				md = checkfield(md,'field',self.displacementeast{i},'NaN',1,'Inf',1,'timeseries',1);
				md = checkfield(md,'field',self.displacementup{i},'NaN',1,'Inf',1,'timeseries',1);
				md = checkfield(md,'field',self.displacementnorth{i},'NaN',1,'Inf',1,'timeseries',1);
				md = checkfield(md,'field',self.geoid{i},'NaN',1,'Inf',1,'timeseries',1);
			end

			%check time stamps: they should all be in seconds 
			%displacementup {{{
			for i=1:sup, 
				h=self.displacementup{i};
				times=h(end,:);
				avgtime=mean(times);
				if(avgtime<10000)
					error('dslmme checkconsistency error messages: time tags for displacementup should be in seconds!');
				end
			end %}}}
			%geoid {{{
			for i=1:nt, 
				h=self.geoid{i};
				times=h(end,:);
				avgtime=mean(times);
				if(avgtime<10000)
					error('dslmme checkconsistency error messages: time tags for geoid should be in seconds!');
				end
			end %}}}
			%displacementeast {{{
			for i=1:nt, 
				h=self.displacementeast{i};
				times=h(end,:);
				avgtime=mean(times);
				if(avgtime<10000)
					error('dslmme checkconsistency error messages: time tags for displacementeast should be in seconds!');
				end
			end %}}}
			%displacementnorth {{{
			for i=1:nt, 
				h=self.displacementnorth{i};
				times=h(end,:);
				avgtime=mean(times);
				if(avgtime<10000)
					error('dslmme checkconsistency error messages: time tags for displacementnorth should be in seconds!');
				end
			end %}}}

		end % }}}
		function disp(self) % {{{
			disp(sprintf('   external: mmeofflinesolidearth solution:'));
			self.disp@solidearthsolution();
			fielddisplay(self,'modelid','index into the multi-model ensemble, determines which field will be used.');
		end % }}}
		function s=nummodels(self) % {{{
			s=numel(self.displacementup);
		end % }}}
		function marshall(self,prefix,md,fid) % {{{
			WriteData(fid,prefix,'data',4,'name','md.solidearth.external.nature','format','Integer'); %code 4 for mmeofflinesolidearthsolution class
			WriteData(fid,prefix,'data',self.modelid,'name','md.solidearth.external.modelid','format','Integer');
			nummodels=length(self.displacementeast);
			WriteData(fid,prefix,'name','md.solidearth.external.nummodels','data',nummodels,'format','Integer');

			%transform our cell array of time series into cell array of time series of rates 
			for i=1:nummodels,
				displacementeast=self.displacementeast{i}; 
				displacementnorth=self.displacementnorth{i}; 
				displacementup=self.displacementup{i}; 
				geoid=self.geoid{i}; 

				time=displacementeast(end,:);
				dt=diff(time,1,2);
				
				displacementeast_rate=[diff(displacementeast(1:end-1,:),1,2)./dt;time(1:end-1)];
				displacementnorth_rate=[diff(displacementnorth(1:end-1,:),1,2)./dt;time(1:end-1)];
				displacementup_rate=[diff(displacementup(1:end-1,:),1,2)./dt;time(1:end-1)];
				geoid_rate=[diff(geoid(1:end-1,:),1,2)./dt;time(1:end-1)];

				self.displacementeast{i}=displacementeast_rate; 
				self.displacementnorth{i}=displacementnorth_rate; 
				self.displacementup{i}=displacementup_rate; 
				self.geoid{i}=geoid_rate; 
			end
			yts=md.constants.yts;
			WriteData(fid,prefix,'data',self.displacementeast,'name','md.solidearth.external.displacementeast','format','MatArray','timeserieslength',md.mesh.numberofvertices+1,'yts',yts,'scale',1/yts);
			WriteData(fid,prefix,'data',self.displacementup,'name','md.solidearth.external.displacementup','format','MatArray','timeserieslength',md.mesh.numberofvertices+1,'yts',yts,'scale',1/yts);
			WriteData(fid,prefix,'data',self.displacementnorth,'name','md.solidearth.external.displacementnorth','format','MatArray','timeserieslength',md.mesh.numberofvertices+1,'yts',yts,'scale',1/yts);
			WriteData(fid,prefix,'data',self.geoid,'name','md.solidearth.external.geoid','format','MatArray','timeserieslength',md.mesh.numberofvertices+1,'yts',yts,'scale',1/yts);

		end % }}}
		function savemodeljs(self,fid,modelname) % {{{
			error('mmeofflinesolidearthsolution error message: not implemented yet');
		end % }}}
		function self = extrude(self,md) % {{{
		end % }}}
	end
end
