/*!\file:  balancedvelocities.cpp
 * \brief: balancedvelocities solution
 */ 

#include "../issm.h"
#include "./parallel.h"

#undef __FUNCT__ 
#define __FUNCT__ "balancedvelocities"

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif


int main(int argc,char* *argv){
	
	/*I/O: */
	FILE* fid=NULL;
	char* inputfilename=NULL;
	char* outputfilename=NULL;
	char* lockname=NULL;
	int   numberofnodes;
	double waitonlock=0;

	Model* model=NULL;

	double* u_g_serial=NULL;
	double* melting_g=NULL;
	double* accumulation_g=NULL;
	int     qmu_analysis;

	/*Results: */
	DataSet* results=NULL;
	DataSet* processedresults=NULL;
	Result*  result=NULL;

	ParameterInputs* inputs=NULL;
	Param*   param=NULL;

	/*time*/
	double   start, finish;
	double   start_core, finish_core;
	double   start_init, finish_init;

	MODULEBOOT();

	#if !defined(_PARALLEL_) || (defined(_PARALLEL_) && !defined(_HAVE_PETSC_))
	throw ErrorException(__FUNCT__," parallel executable was compiled without support of parallel libraries!");
	#endif

	/*Initialize Petsc and get start time*/
	PetscInitialize(&argc,&argv,(char *)0,"");  
	MPI_Barrier(MPI_COMM_WORLD); start=MPI_Wtime();

	/*Size and rank: */
	MPI_Comm_rank(MPI_COMM_WORLD,&my_rank);  
	MPI_Comm_size(MPI_COMM_WORLD,&num_procs); 

	_printf_("recover , input file name and output file name:\n");
	inputfilename=argv[2];
	outputfilename=argv[3];
	lockname=argv[4];

	/*Initialize model structure: */
	MPI_Barrier(MPI_COMM_WORLD); start_init=MPI_Wtime();
	model=new Model();

	/*Open handle to data on disk: */
	fid=pfopen(inputfilename,"rb");

	/*Initialize model structure: */
	model=new Model();

	_printf_("read and create finite element model:\n");
	model->AddFormulation(fid,BalancedvelocitiesAnalysisEnum());

	/*recover parameters: */
	model->FindParam(&waitonlock,"waitonlock");
	model->FindParam(&qmu_analysis,"qmu_analysis");

	_printf_("initialize inputs:\n");
	
	model->FindParam(&u_g_serial,NULL,NULL,"u_g",BalancedvelocitiesAnalysisEnum());
	model->FindParam(&melting_g,NULL,NULL,"m_g",BalancedvelocitiesAnalysisEnum());
	model->FindParam(&accumulation_g,NULL,NULL,"a_g",BalancedvelocitiesAnalysisEnum());
	model->FindParam(&numberofnodes,"numberofnodes");
	
	inputs=new ParameterInputs;
	inputs->Add("velocity",u_g_serial,3,numberofnodes);
	inputs->Add("melting",melting_g,1,numberofnodes);
	inputs->Add("accumulation",accumulation_g,1,numberofnodes);

	_printf_("initialize results:\n");
	results=new DataSet(ResultsEnum());
	MPI_Barrier(MPI_COMM_WORLD); finish_init=MPI_Wtime();

	/*are we running the solutoin sequence, or a qmu wrapper around it? : */
	if(!qmu_analysis){

		/*run balancedvelocities analysis: */
		_printf_("call computational core:\n");
		MPI_Barrier(MPI_COMM_WORLD); start_core=MPI_Wtime( );
		balancedvelocities_core(results,model,inputs);
		MPI_Barrier(MPI_COMM_WORLD); finish_core=MPI_Wtime( );

	}
	else{

		/*run qmu analysis: */
		_printf_("calling qmu analysis on balancedvelocities core:\n");
	
		#ifdef _HAVE_DAKOTA_ 
		MPI_Barrier(MPI_COMM_WORLD); start_core=MPI_Wtime( );
		Qmux(model,inputs,BalancedvelocitiesAnalysisEnum(),NoneAnalysisEnum());
		MPI_Barrier(MPI_COMM_WORLD); finish_core=MPI_Wtime( );
	 	#else
		throw ErrorException(__FUNCT__," Dakota not present, cannot do qmu!");
		#endif
	}

	/*Add analysis_type to results: */
	result=new Result(results->Size()+1,0,1,"analysis_type","balancedvelocities");
	results->AddObject(result);
	
	_printf_("process results:\n");
	ProcessResults(&processedresults,results,model,BalancedvelocitiesAnalysisEnum());
	
	_printf_("write results to disk:\n");
	OutputResults(processedresults,outputfilename);

	if (waitonlock>0){
		_printf_("write lock file:\n");
		WriteLockFile(lockname);
	}

	/*Free ressources:*/
	delete processedresults;
	delete results;
	delete model;
	delete inputs;

	/*Get finish time and close*/
	MPI_Barrier(MPI_COMM_WORLD); finish = MPI_Wtime( );
	_printf_("\n   %-34s %f seconds  \n","Model initialization elapsed time:",finish_init-start_init);
	_printf_("   %-34s %f seconds  \n","Core solution elapsed time:",finish_core-start_core);
	_printf_("   %-34s %f seconds\n\n","Total elapsed time:",finish-start);
	_printf_("closing MPI and Petsc\n");
	PetscFinalize(); 

	/*end module: */
	MODULEEND();

	return 0; //unix success return;
}
