/* \file IoModel.h
 * \brief  Header file defining the IoModel structure and processing of input data.
 * \sa IoModel.cpp
 */

#ifndef _IOMODEL_H
#define _IOMODEL_H

#include "../io/io.h"
#include "../DataSet/DataSet.h"
#include "../toolkits/toolkits.h"

#define RIFTINFOSIZE 11

struct IoModel {

	char*   name;
	char*	  inputfilename;
	char*	  outputfilename;
	char*   repository;
	char*   meshtype;
	int     analysis_type;
	int     sub_analysis_type;
	int     qmu_analysis;
	int     control_analysis;
	char*   solverstring;

	/*2d mesh: */
	int     numberofelements;
	int     numberofnodes;
	double* x;
	double* y;
	double* z;
	double* elements;
	double* elements_type;

	/*3d mesh: */
	int     numberofnodes2d;
	int     numberofelements2d;
	double* elements2d;
	double* deadgrids;
	int     numlayers;
	double* uppernodes;

	/*elements type: */
	int     ishutter;
	int     ismacayealpattyn;
	int     isstokes;
	double* gridonhutter;
	double* gridonmacayeal;
	double* gridonpattyn;

	/*results: */
	double* vx;
	double* vy;
	double* vz;
	double* pressure;
	double* temperature;

	/*observations: */
	double*  vx_obs;
	double*  vy_obs;

	/*qmu: */
	int      numberofresponses;
	int      numberofvariables;
	int      qmu_npart;

	/*geometry: */
	double* elementonbed;
	double* elementonsurface;
	double* gridonbed;
	double* gridonsurface;
	double* gridonstokes;
	double* borderstokes;
	double* thickness;
	double* surface;
	double* bed;
	double* elementoniceshelf;
	double* elementonwater;
	double* gridonicesheet;
	double* gridoniceshelf;

	/*friction: */
	int     drag_type;
	double* drag;
	double* p;
	double* q;

	/*boundary conditions: */
	int     numberofpressureloads;
	int     numberofpressureloads_stokes;
	double* pressureload;
	double* pressureload_stokes;
	double* spcvelocity;
	double* spctemperature;
	double* spcthickness;
	double* geothermalflux;
	
	/*materials: */
	double  rho_water,rho_ice;
	double  g;
	double* B;
	double* n;

	/*numerical parameters: */
	double  meanvel;
	double  epsvel;
	int     artdiff;
	double  viscosity_overshoot;
	double  stokesreconditioning;
	double  cm_noisedmp;
	double  cm_mindmp_value;
	double  cm_mindmp_slope;
	double  cm_maxdmp_value;
	double  cm_maxdmp_slope;
	double  cm_min;
	double  cm_max;

	double  cm_noisedampening;

	/*control methods: */
	char*	control_type;

	/*solution parameters: */
	double* fit;
	double* cm_jump;
	int     artificial_diffusivity;
	int     nsteps;
	double  eps_cm;
	double  tolx;
	double* maxiter;
	int     verbose;
	int     plot;
	double  eps_res;
	double  eps_rel;
	double  eps_abs;
	double  dt,ndt;
	double  penalty_offset;
	double  penalty_melting;
	int     penalty_lock;
	double  sparsity;
	int     connectivity;
	int     lowmem;
	double* optscal;
	double  yts;
	double  waitonlock;

	/*thermal parameters: */
	double beta;
	double meltingpoint;
	double latentheat;
	double  heatcapacity,thermalconductivity;
	int    min_thermal_constraints;
	int    stabilize_constraints;
	double mixed_layer_capacity;
	double thermal_exchange_velocity;

	/*rifts: */
	int      numrifts;
	double*  riftinfo;

	/*penalties: */
	int      numpenalties;
	double*  penalties;
	
	/*basal: */
	double*  melting;
	double*  accumulation;

	/*parameter output: */
	int      numoutput;

	/*exterior data: */
	int* epart; /*!element partitioning.*/
	int* npart; /*!node partitioning.*/
	int* my_grids; /*! grids that belong to this cpu*/
	double* my_bordergrids; /*! grids that belong to this cpu, and some other cpu also*/
	int*    penaltypartitioning;

};


	/*constructor and destructor: */
	IoModel*	NewIoModel(void);
	void     DeleteIoModel( IoModel** pthis);

	/*Echo: */
	void  IoModelEcho(IoModel* iomodel,int which_part,int rank);

	/*Initialization with matlab workspace data, or marshall binary data: */
	int	IoModelInit(IoModel** piomodel,ConstDataHandle iomodel_handle);

	/*Creation of fem datasets: general drivers*/
	void  CreateDataSets(DataSet** pelements,DataSet** pnodes, DataSet** pmaterials, DataSet** pconstraints, DataSet** ploads,DataSet** pparameters,IoModel* iomodel,ConstDataHandle iomodel_handle);
	void  CreateParameters(DataSet** pparameters,IoModel* iomodel,ConstDataHandle iomodel_handle);

	
	/*Create of fem datasets: specialised drivers: */
	
	/*diagnostic horizontal*/
	void	CreateElementsNodesAndMaterialsDiagnosticHoriz(DataSet** pelements,DataSet** pnodes, DataSet** pmaterials, IoModel* iomodel,ConstDataHandle iomodel_handle);
	void	CreateConstraintsDiagnosticHoriz(DataSet** pconstraints,IoModel* iomodel,ConstDataHandle iomodel_handle);
	void  CreateLoadsDiagnosticHoriz(DataSet** ploads, IoModel* iomodel, ConstDataHandle iomodel_handle);
	void  CreateParametersDiagnosticHoriz(DataSet** pparameters,IoModel* iomodel,ConstDataHandle iomodel_handle);

	/*diagnostic vertical*/
	void	CreateElementsNodesAndMaterialsDiagnosticVert(DataSet** pelements,DataSet** pnodes, DataSet** pmaterials, IoModel* iomodel,ConstDataHandle iomodel_handle);
	void	CreateConstraintsDiagnosticVert(DataSet** pconstraints,IoModel* iomodel,ConstDataHandle iomodel_handle);
	void  CreateLoadsDiagnosticVert(DataSet** ploads, IoModel* iomodel, ConstDataHandle iomodel_handle);

	/*diagnostic hutter*/
	void	CreateElementsNodesAndMaterialsDiagnosticHutter(DataSet** pelements,DataSet** pnodes, DataSet** pmaterials, IoModel* iomodel,ConstDataHandle iomodel_handle);
	void	CreateConstraintsDiagnosticHutter(DataSet** pconstraints,IoModel* iomodel,ConstDataHandle iomodel_handle);
	void  CreateLoadsDiagnosticHutter(DataSet** ploads, IoModel* iomodel, ConstDataHandle iomodel_handle);

	/*diagnostic stokes*/
	void	CreateElementsNodesAndMaterialsDiagnosticStokes(DataSet** pelements,DataSet** pnodes, DataSet** pmaterials, IoModel* iomodel,ConstDataHandle iomodel_handle);
	void	CreateConstraintsDiagnosticStokes(DataSet** pconstraints,IoModel* iomodel,ConstDataHandle iomodel_handle);
	void  CreateLoadsDiagnosticStokes(DataSet** ploads, IoModel* iomodel, ConstDataHandle iomodel_handle);

	/*slope compute*/
	void	CreateElementsNodesAndMaterialsSlopeCompute(DataSet** pelements,DataSet** pnodes, DataSet** pmaterials, IoModel* iomodel,ConstDataHandle iomodel_handle);
	void	CreateConstraintsSlopeCompute(DataSet** pconstraints,IoModel* iomodel,ConstDataHandle iomodel_handle);
	void  CreateLoadsSlopeCompute(DataSet** ploads, IoModel* iomodel, ConstDataHandle iomodel_handle);

	/*control:*/
	void  CreateParametersControl(DataSet** pparameters,IoModel* iomodel,ConstDataHandle iomodel_handle);

	/*thermal:*/
	void	CreateElementsNodesAndMaterialsThermal(DataSet** pelements,DataSet** pnodes, DataSet** pmaterials, IoModel* iomodel,ConstDataHandle iomodel_handle);
	void	CreateConstraintsThermal(DataSet** pconstraints,IoModel* iomodel,ConstDataHandle iomodel_handle);
	void  CreateLoadsThermal(DataSet** ploads, IoModel* iomodel, ConstDataHandle iomodel_handle);
	void  CreateParametersThermal(DataSet** pparameters,IoModel* iomodel,ConstDataHandle iomodel_handle);

	/*melting:*/
	void	CreateElementsNodesAndMaterialsMelting(DataSet** pelements,DataSet** pnodes, DataSet** pmaterials, IoModel* iomodel,ConstDataHandle iomodel_handle);
	void	CreateConstraintsMelting(DataSet** pconstraints,IoModel* iomodel,ConstDataHandle iomodel_handle);
	void  CreateLoadsMelting(DataSet** ploads, IoModel* iomodel, ConstDataHandle iomodel_handle);
	void  CreateParametersMelting(DataSet** pparameters,IoModel* iomodel,ConstDataHandle iomodel_handle);

	/*prognostic:*/
	void	CreateElementsNodesAndMaterialsPrognostic(DataSet** pelements,DataSet** pnodes, DataSet** pmaterials, IoModel* iomodel,ConstDataHandle iomodel_handle);
	void	CreateConstraintsPrognostic(DataSet** pconstraints,IoModel* iomodel,ConstDataHandle iomodel_handle);
	void  CreateLoadsPrognostic(DataSet** ploads, IoModel* iomodel, ConstDataHandle iomodel_handle);
	void  CreateParametersPrognostic(DataSet** pparameters,IoModel* iomodel,ConstDataHandle iomodel_handle);

	/*balancedthickness:*/
	void	CreateElementsNodesAndMaterialsBalancedthickness(DataSet** pelements,DataSet** pnodes, DataSet** pmaterials, IoModel* iomodel,ConstDataHandle iomodel_handle);
	void	CreateConstraintsBalancedthickness(DataSet** pconstraints,IoModel* iomodel,ConstDataHandle iomodel_handle);
	void  CreateLoadsBalancedthickness(DataSet** ploads, IoModel* iomodel, ConstDataHandle iomodel_handle);
	void  CreateParametersBalancedthickness(DataSet** pparameters,IoModel* iomodel,ConstDataHandle iomodel_handle);

	/*balancedvelocities:*/
	void	CreateElementsNodesAndMaterialsBalancedvelocities(DataSet** pelements,DataSet** pnodes, DataSet** pmaterials, IoModel* iomodel,ConstDataHandle iomodel_handle);
	void	CreateConstraintsBalancedvelocities(DataSet** pconstraints,IoModel* iomodel,ConstDataHandle iomodel_handle);
	void  CreateLoadsBalancedvelocities(DataSet** ploads, IoModel* iomodel, ConstDataHandle iomodel_handle);
	void  CreateParametersBalancedvelocities(DataSet** pparameters,IoModel* iomodel,ConstDataHandle iomodel_handle);

	/*qmu: */
	void CreateParametersQmu(DataSet** pparameters,IoModel* iomodel,ConstDataHandle iomodel_handle);

#endif  /* _IOMODEL_H */
