/* \file FetchParams.c:
 * \brief: interface for fetching  matlab parameters into a "C" dataset 
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif


#ifdef _SERIAL_

#include <mex.h>

#include "../objects/Param.h"
#include "./io.h"
#include "../shared/shared.h"

#undef __FUNCT__ 
#define __FUNCT__  "FetchParams"

void FetchParams(DataSet** pparameters, DataHandle dataref){

	int i,j,count;

	/*output: */
	Param* param=NULL;
	Numpar*  numpar=NULL;
	DataSet* parameters=NULL;

	/*intermediary: */
	int M,N;
	double* tmatrix=NULL;
	double* matrix=NULL;
	char**  stringarray=NULL;
	int nfields;
	char* name=NULL;
	mxArray* pfield=NULL;
	mxArray* pfield2=NULL;


	/*First, create parameters : */
	parameters=new DataSet();

	/*Then, create Numpar object, before building the params objects: */
	numpar= new Numpar(1);
	parameters->AddObject(numpar);

	/*now, go through matlab params structure, and create Param object for each field: */

	nfields=mxGetNumberOfFields(dataref);

	for(count=0;count<nfields;count++){

		/*Get i'th field: */
		name=(char*)mxGetFieldNameByNumber(dataref,count);
		pfield=mxGetFieldByNumber(dataref,0,count);

		/*Check type of field: */
		if (mxIsDouble(pfield)){
			/*could be  DOUBLE, DOUBLEVEC or DOUBLEMAT, depends on dimensions: */

			M=mxGetM(pfield);
			N=mxGetN(pfield);

			if (M==0 | N==0){
				throw ErrorException(__FUNCT__,exprintf("%s%s%i (%s) %s%i%s%i%s",__FUNCT__," error message: array in parameters structure field ",count,name," is of size (",M,",",N,")"));
			}

			if (M==1 && N==1){

				/*we have a simple scalar: */
				param= new Param(count+1,name,DOUBLE);
				param->SetDouble(*mxGetPr(pfield));
				parameters->AddObject(param);

			}
			else{
				if (N==1){
					
					/*vector: */
					param= new Param(count+1,name,DOUBLEVEC);
					param->SetDoubleVec(mxGetPr(pfield),M,1);
					parameters->AddObject(param);

				}
				else{

					/*matrix: first, transpose, then plug into Param */
					matrix=mxGetPr(pfield);
					tmatrix=(double*)xmalloc(M*N*sizeof(double));
					for (i=0;i<M;i++){
						for(j=0;j<N;j++){
							*(tmatrix+N*i+j)=*(matrix+M*j+i);
						}
					}

					param= new Param(count+1,name,DOUBLEMAT);
					param->SetDoubleMat(tmatrix,M,N);
					parameters->AddObject(param);
	
					/*Free ressources:*/
					xfree((void**)&tmatrix);

				}
			}

		}
		else if (mxIsChar(pfield)){
			
			/* we have a string parameter:*/
			
			int stringlen;
			char* string=NULL;
			
			stringlen = mxGetM(pfield)*mxGetN(pfield)+1;
			string = (char*)xmalloc(sizeof(mxChar)*stringlen);
			mxGetString(pfield,string,stringlen);

			param= new Param(count+1,name,STRING);
			param->SetString(string);
			parameters->AddObject(param);

			xfree((void**)&string);
		}
		else if (mxIsCell(pfield)){

			/*string array: */
			M=mxGetM(pfield);
			stringarray=(char**)xmalloc(M*sizeof(char*));

			for(i=0;i<M;i++){
				char* descriptor=NULL;
				pfield2=mxGetCell(pfield,i);
				FetchData(&descriptor,pfield2);
				stringarray[i]=descriptor;
			}
		
			param= new Param(count+1,name,STRINGARRAY);
			param->SetStringArray(stringarray,M);
			parameters->AddObject(param);

			/*Free ressources:*/
			for(i=0;i<M;i++){
				char* descriptor=stringarray[i];
				xfree((void**)&descriptor);
			}
			xfree((void**)&stringarray);

		}
		else throw ErrorException(__FUNCT__,exprintf("%s%s%i",__FUNCT__," error message: unknow type in parameters structure field ",i));
	}

	/*Assign output pointers:*/
	*pparameters=parameters;

}
#endif
