%AUTOREGRESSION LINEAR BASAL FORCINGS class definition
%
%   Usage:
%      autoregressionlinearbasalforcings=spatiallinearbasalforcings();

classdef autoregressionlinearbasalforcings
	
	%VV: not verified yet (27Jan2022)
	
	properties (SetAccess=public) 
		num_basins                = 0;
		beta0                     = NaN;
      beta1                     = NaN;
      ar_order                  = 0;
      ar_initialtime            = 0;
      ar_timestep               = 0;
      phi                       = NaN;
      basin_id                  = NaN;
		groundedice_melting_rate  = NaN;
		deepwater_elevation       = NaN;
		upperwater_melting_rate   = NaN;
		upperwater_elevation      = NaN;
		geothermalflux            = NaN;
	end
	methods
		function self = autoregressionlinearbasalforcings(varargin) % {{{
			switch nargin
				case 0
					self=setdefaultparameters(self);
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function self = extrude(self,md) % {{{
			self.groundedice_melting_rate=project3d(md,'vector',self.groundedice_melting_rate,'type','node','layer',1); 
			self.deepwater_elevation=project3d(md,'vector',self.deepwater_elevation,'type','node','layer',1); 
			self.upperwater_melting_rate=project3d(md,'vector',self.upperwater_melting_rate,'type','node','layer',1); 
			self.upperwater_elevation=project3d(md,'vector',self.upperwater_elevation,'type','node','layer',1); 
			self.geothermalflux=project3d(md,'vector',self.geothermalflux,'type','node','layer',1); %bedrock only gets geothermal flux
		end % }}}
		function self = initialize(self,md) % {{{

			if isnan(self.groundedice_melting_rate),
				self.groundedice_melting_rate=zeros(md.mesh.numberofvertices,1);
				disp('      no basalforcings.groundedice_melting_rate specified: values set as zero');
			end

		end % }}}
		function self = setdefaultparameters(self) % {{{
			%Nothing for now
		end % }}}
		function md = checkconsistency(self,md,solution,analyses) % {{{

			if ismember('MasstransportAnalysis',analyses) & ~(strcmp(solution,'TransientSolution') & md.transient.ismasstransport==0),
				md = checkfield(md,'fieldname','basalforcings.num_basins','numel',1,'NaN',1,'Inf',1,'>',0);
				md = checkfield(md,'fieldname','basalforcings.groundedice_melting_rate','NaN',1,'Inf',1,'timeseries',1);
				md = checkfield(md,'fieldname','basalforcings.deepwater_elevation','NaN',1,'Inf',1,'size',[1,md.basalforcings.num_basins],'numel',md.basalforcings.num_basins);
				md = checkfield(md,'fieldname','basalforcings.upperwater_melting_rate','NaN',1,'Inf',1,'>=',0,'size',[1,md.basalforcings.num_basins],'numel',md.basalforcings.num_basins);
				md = checkfield(md,'fieldname','basalforcings.upperwater_elevation','NaN',1,'Inf',1,'<=',0,'size',[1,md.basalforcings.num_basins],'numel',md.basalforcings.num_basins);
            md = checkfield(md,'fieldname','basalforcings.basin_id','Inf',1,'>=',0,'<=',md.basalforcings.num_basins,'size',[md.mesh.numberofelements,1]);
            md = checkfield(md,'fieldname','basalforcings.beta0','NaN',1,'Inf',1,'size',[1,md.basalforcings.num_basins],'numel',md.basalforcings.num_basins); 
				md = checkfield(md,'fieldname','basalforcings.beta1','NaN',1,'Inf',1,'size',[1,md.basalforcings.num_basins],'numel',md.basalforcings.num_basins); 
				md = checkfield(md,'fieldname','basalforcings.ar_order','numel',1,'NaN',1,'Inf',1,'>=',0);
            md = checkfield(md,'fieldname','basalforcings.ar_initialtime','numel',1,'NaN',1,'Inf',1);
            md = checkfield(md,'fieldname','basalforcings.ar_timestep','numel',1,'NaN',1,'Inf',1,'>=',md.timestepping.time_step); %autoregression time step cannot be finer than ISSM timestep
            md = checkfield(md,'fieldname','basalforcings.phi','NaN',1,'Inf',1,'size',[md.basalforcings.num_basins,md.basalforcings.ar_order]);
			end
			if ismember('BalancethicknessAnalysis',analyses),
				error('not implemented yet!');
			end
			if ismember('ThermalAnalysis',analyses) & ~(strcmp(solution,'TransientSolution') & md.transient.isthermal==0),
				error('not implemented yet!');
			end
			if numel(md.basalforcings.geothermalflux)>1
            md = checkfield(md,'fieldname','basalforcings.geothermalflux','NaN',1,'Inf',1,'timeseries',1,'>=',0);
         end
		end % }}}
		function disp(self) % {{{
			disp(sprintf('   autoregression linear basal forcings parameters:'));
			disp(sprintf('   autoregressive model is applied for deepwater_melting_rate'));

			fielddisplay(self,'num_basins','number of different basins [unitless]');
         fielddisplay(self,'basin_id','basin number assigned to each element [unitless]');
         fielddisplay(self,'beta0','basin-specific intercept values [m/yr] (if beta_1==0 mean=beta_0/(1-sum(phi)))');
         fielddisplay(self,'beta1','basin-specific trend values [m  yr^(-2)]');
         fielddisplay(self,'ar_order','order of the autoregressive model [unitless]');
         fielddisplay(self,'ar_initialtime','initial time assumed in the autoregressive model parameterization [yr]');
         fielddisplay(self,'ar_timestep','time resolution of the autoregressive model [yr]');
         fielddisplay(self,'phi','basin-specific vectors of lag coefficients [unitless]');
			fielddisplay(self,'deepwater_elevation','basin-specific elevation of ocean deepwater [m]');
			fielddisplay(self,'upperwater_melting_rate','basin-specific basal melting rate (positive if melting applied for floating ice whith base >= upperwater_elevation) [m/yr]');
			fielddisplay(self,'upperwater_elevation','basin-specific elevation of ocean upperwater [m]');
			fielddisplay(self,'groundedice_melting_rate','node-specific basal melting rate (positive if melting) [m/yr]');
			fielddisplay(self,'geothermalflux','node-specific geothermal heat flux [W/m^2]');

		end % }}}
		function marshall(self,prefix,md,fid) % {{{

			yts=md.constants.yts;

			WriteData(fid,prefix,'name','md.basalforcings.model','data',9,'format','Integer');
			WriteData(fid,prefix,'object',self,'fieldname','groundedice_melting_rate','format','DoubleMat','name','md.basalforcings.groundedice_melting_rate','mattype',1,'scale',1./yts,'timeserieslength',md.mesh.numberofvertices+1,'yts',md.constants.yts);
			WriteData(fid,prefix,'object',self,'fieldname','geothermalflux','name','md.basalforcings.geothermalflux','format','DoubleMat','mattype',1,'timeserieslength',md.mesh.numberofvertices+1,'yts',md.constants.yts);
			WriteData(fid,prefix,'object',self,'fieldname','num_basins','format','Integer');
         WriteData(fid,prefix,'object',self,'fieldname','ar_order','format','Integer');
         WriteData(fid,prefix,'object',self,'fieldname','ar_initialtime','format','Double','scale',yts);
         WriteData(fid,prefix,'object',self,'fieldname','ar_timestep','format','Double','scale',yts);
         WriteData(fid,prefix,'object',self,'fieldname','basin_id','data',self.basin_id-1,'name','md.basalforcings.basin_id','format','IntMat','mattype',2); %0-indexed
         WriteData(fid,prefix,'object',self,'fieldname','beta0','format','DoubleMat','name','md.basalforcings.beta0','scale',1./yts,'yts',yts);
         WriteData(fid,prefix,'object',self,'fieldname','beta1','format','DoubleMat','name','md.basalforcings.beta1','scale',1./(yts^2),'yts',yts);
         WriteData(fid,prefix,'object',self,'fieldname','phi','format','DoubleMat','name','md.basalforcings.phi','yts',yts);	
			WriteData(fid,prefix,'object',self,'fieldname','deepwater_elevation','format','DoubleMat','name','md.basalforcings.deepwater_elevation');
			WriteData(fid,prefix,'object',self,'fieldname','upperwater_melting_rate','format','DoubleMat','name','md.basalforcings.upperwater_melting_rate','scale',1./yts);
			WriteData(fid,prefix,'object',self,'fieldname','upperwater_elevation','format','DoubleMat','name','md.basalforcings.upperwater_elevation');
		end % }}}
	end
end
