function series=getlovenumbers(varargin)
%GETLOVENUMBERS: provide love numbers retrieved from: 
%http://www.srosat.com/iag-jsg/loveNb.php in a chosen reference frame 
%
%	Usage: 
%		series = getlovenumbers('type','loadingverticaldisplacement','referenceframe','CM','maxdeg',1000) 
% 
%	- type = one of 'loadingverticaldisplacement', 
%	'loadinggravitationalpotential', 'loadinghorizontaldisplacement', 
%	'tidalverticaldisplacement', 'tidalgravitationalpotential', 
%	'tidalhorizontaldisplacement'
%	- reference_frame = one of 'CM' (default) and 'CF'.
%   - maxdeg = default 1000
%
%	Example:  
%		h=getlovenumbers('type','loadingverticaldisplacement','referenceframe','CM','maxdeg',maxdeg);
%		k=getlovenumbers('type','loadinggravitationalpotential','referenceframe','CM','maxdeg',maxdeg);
%		l=getlovenumbers('type','loadinghorizontaldisplacement','referenceframe','CM','maxdeg',maxdeg);
%		th=getlovenumbers('type','tidalverticaldisplacement','referenceframe','CM','maxdeg',maxdeg);
%		tk=getlovenumbers('type','tidalgravitationalpotential','referenceframe','CM','maxdeg',maxdeg);
%		tl=getlovenumbers('type','tidalhorizontaldisplacement','referenceframe','CM','maxdeg',maxdeg);

%recover options: 
options=pairoptions(varargin{:});
type=getfieldvalue(options,'type');
frame=getfieldvalue(options,'referenceframe','CM');
maxdeg=getfieldvalue(options,'maxdeg',1000);

if(maxdeg>10000),
	error('PREM love numbers computed only for deg < 10,000. Request lower maxdeg' );
end

if ~( strcmpi(type,'loadingverticaldisplacement') | strcmpi(type,'loadinggravitationalpotential') | strcmpi(type,'loadinghorizontaldisplacement') ...
		| strcmpi(type,'tidalverticaldisplacement') | strcmpi(type,'tidalgravitationalpotential') | strcmpi(type,'tidalhorizontaldisplacement')), 
	error('type should be one of ''loadingverticaldisplacement'', ''loadinggravitationalpotential'', ''loadinghorizontaldisplacement'',''tidalverticaldisplacement'', ''tidalgravitationalpotential'', ''tidalhorizontaldisplacement''');
end

	love_numbers=[...
		0.00000000	0.00000000	0.00000000	0.00000000	0.00000000	0.00000000
		-1.28740059	-1.00000000	-0.89858519	0.00000000	0.00000000	0.00000000
		-1.00025365	-0.30922675	0.02060926	0.62933648	0.30136172	0.08745784
		-1.06243501	-0.19927948	0.06801636	0.29799672	0.09304210	0.01525258
		-1.06779588	-0.13649834	0.05667027	0.18024117	0.04181593	0.01055799
		-1.10365923	-0.10736896	0.04401221	0.13257950	0.02455390	0.00874595
		-1.16440348	-0.09295485	0.03638747	0.10999739	0.01694141	0.00701045
		-1.23634156	-0.08469861	0.03202759	0.09727833	0.01279626	0.00556581
		-1.31140380	-0.07921412	0.02937593	0.08896691	0.01019313	0.00445157
		-1.38582399	-0.07513541	0.02762338	0.08291950	0.00840274	0.00360620
		-1.45807465	-0.07187005	0.02638627	0.07819716	0.00709641	0.00295944
		-1.52763314	-0.06913154	0.02547640	0.07433538	0.00610349	0.00245649
		-1.59437866	-0.06676258	0.02479080	0.07107472	0.00532548	0.00205857
		-1.65833071	-0.06466619	0.02426511	0.06825469	0.00470107	0.00173889
		-1.71954820	-0.06277732	0.02385464	0.06576832	0.00418998	0.00147872
		-1.77809640	-0.06105001	0.02352654	0.06354080	0.00376467	0.00126474
		-1.83403970	-0.05945081	0.02325609	0.06151811	0.00340573	0.00108719
		-1.88744242	-0.05795502	0.02302469	0.05966045	0.00309910	0.00093879
		-1.93837115	-0.05654418	0.02281843	0.05793806	0.00283441	0.00081397
		-1.98689666	-0.05520447	0.02262706	0.05632844	0.00260382	0.00070843
		-2.03309477	-0.05392545	0.02244322	0.05481440	0.00240132	0.00061877
		-2.07704643	-0.05269926	0.02226173	0.05338269	0.00222224	0.00054227
		-2.11883714	-0.05151988	0.02207909	0.05202300	0.00206289	0.00047675
		-2.15855611	-0.05038274	0.02189307	0.05072723	0.00192031	0.00042043
		-2.19629514	-0.04928430	0.02170238	0.04948894	0.00179212	0.00037187
		-2.23214747	-0.04822179	0.02150643	0.04830294	0.00167634	0.00032986
		-2.26620674	-0.04719301	0.02130509	0.04716500	0.00157138	0.00029341
		-2.29856595	-0.04619619	0.02109858	0.04607163	0.00147587	0.00026171
		-2.32931659	-0.04522983	0.02088735	0.04501989	0.00138869	0.00023405
		-2.35854794	-0.04429270	0.02067197	0.04400725	0.00130886	0.00020985
		-2.38634650	-0.04338368	0.02045310	0.04303154	0.00123559	0.00018864
		-2.41279547	-0.04250179	0.02023142	0.04209082	0.00116814	0.00017000
		-2.43797451	-0.04164613	0.02000761	0.04118337	0.00110593	0.00015358
		-2.46195951	-0.04081583	0.01978231	0.04030761	0.00104841	0.00013908
		-2.48482241	-0.04001011	0.01955614	0.03946212	0.00099513	0.00012625
		-2.50663126	-0.03922817	0.01932966	0.03864555	0.00094568	0.00011488
		-2.52745016	-0.03846928	0.01910337	0.03785666	0.00089971	0.00010477
		-2.54733938	-0.03773269	0.01887774	0.03709428	0.00085689	0.00009577
		-2.56635547	-0.03701769	0.01865317	0.03635731	0.00081696	0.00008775
		-2.58455138	-0.03632358	0.01843000	0.03564469	0.00077965	0.00008057
		-2.60197665	-0.03564968	0.01820854	0.03495543	0.00074474	0.00007415
		-2.61867756	-0.03499532	0.01798905	0.03428858	0.00071205	0.00006839
		-2.63469733	-0.03435985	0.01777176	0.03364323	0.00068137	0.00006322
		-2.65007629	-0.03374263	0.01755683	0.03301851	0.00065257	0.00005857
		-2.66485208	-0.03314303	0.01734443	0.03241359	0.00062548	0.00005438
		-2.67905981	-0.03256047	0.01713468	0.03182769	0.00059998	0.00005059
		-2.69273222	-0.03199435	0.01692767	0.03126003	0.00057595	0.00004717
		-2.70589990	-0.03144411	0.01672347	0.03070991	0.00055327	0.00004407
		-2.71859139	-0.03090919	0.01652215	0.03017662	0.00053186	0.00004127
		-2.73083334	-0.03038907	0.01632374	0.02965950	0.00051162	0.00003872
		-2.74265068	-0.02988323	0.01612826	0.02915791	0.00049246	0.00003641
		-2.75406669	-0.02939118	0.01593573	0.02867125	0.00047432	0.00003431
		-2.76510320	-0.02891245	0.01574615	0.02819894	0.00045712	0.00003239
		-2.77578063	-0.02844656	0.01555950	0.02774043	0.00044081	0.00003065
		-2.78611812	-0.02799309	0.01537578	0.02729517	0.00042531	0.00002905
		-2.79613364	-0.02755161	0.01519496	0.02686267	0.00041059	0.00002760
		-2.80584405	-0.02712170	0.01501701	0.02644244	0.00039659	0.00002627
		-2.81526521	-0.02670298	0.01484191	0.02603402	0.00038326	0.00002505
		-2.82441204	-0.02629506	0.01466961	0.02563696	0.00037057	0.00002393
		-2.83329857	-0.02589759	0.01450009	0.02525085	0.00035847	0.00002291
		-2.84193804	-0.02551021	0.01433329	0.02487527	0.00034693	0.00002197
		-2.85034293	-0.02513260	0.01416919	0.02450984	0.00033591	0.00002111
		-2.85852503	-0.02476443	0.01400773	0.02415420	0.00032539	0.00002031
		-2.86649548	-0.02440538	0.01384888	0.02380798	0.00031534	0.00001958
		-2.87426481	-0.02405518	0.01369258	0.02347086	0.00030573	0.00001891
		-2.88184299	-0.02371352	0.01353880	0.02314250	0.00029653	0.00001829
		-2.88923945	-0.02338014	0.01338749	0.02282261	0.00028772	0.00001772
		-2.89646316	-0.02305478	0.01323861	0.02251087	0.00027928	0.00001720
		-2.90352261	-0.02273718	0.01309211	0.02220702	0.00027120	0.00001671
		-2.91042585	-0.02242710	0.01294795	0.02191078	0.00026344	0.00001626
		-2.91718054	-0.02212431	0.01280609	0.02162189	0.00025600	0.00001585
		-2.92379397	-0.02182859	0.01266648	0.02134010	0.00024886	0.00001546
		-2.93027306	-0.02153971	0.01252908	0.02106517	0.00024200	0.00001511
		-2.93662439	-0.02125748	0.01239386	0.02079689	0.00023540	0.00001478
		-2.94285425	-0.02098169	0.01226077	0.02053502	0.00022907	0.00001447
		-2.94896860	-0.02071215	0.01212977	0.02027936	0.00022297	0.00001419
		-2.95497314	-0.02044868	0.01200082	0.02002971	0.00021710	0.00001392
		-2.96087331	-0.02019110	0.01187388	0.01978587	0.00021146	0.00001368
		-2.96667427	-0.01993924	0.01174893	0.01954767	0.00020602	0.00001345
		-2.97238097	-0.01969293	0.01162591	0.01931492	0.00020078	0.00001324
		-2.97799813	-0.01945201	0.01150481	0.01908746	0.00019573	0.00001305
		-2.98353025	-0.01921634	0.01138557	0.01886511	0.00019086	0.00001286
		-2.98898162	-0.01898576	0.01126817	0.01864773	0.00018616	0.00001269
		-2.99435636	-0.01876014	0.01115257	0.01843516	0.00018162	0.00001253
		-2.99965838	-0.01853932	0.01103875	0.01822725	0.00017725	0.00001238
		-3.00489143	-0.01832319	0.01092666	0.01802387	0.00017302	0.00001224
		-3.01005909	-0.01811161	0.01081628	0.01782489	0.00016893	0.00001211
		-3.01516479	-0.01790446	0.01070757	0.01763017	0.00016499	0.00001199
		-3.02021180	-0.01770162	0.01060052	0.01743958	0.00016117	0.00001187
		-3.02520323	-0.01750298	0.01049508	0.01725302	0.00015748	0.00001177
		-3.03014209	-0.01730842	0.01039123	0.01707036	0.00015391	0.00001166
		-3.03503122	-0.01711783	0.01028894	0.01689149	0.00015045	0.00001157
		-3.03987336	-0.01693111	0.01018819	0.01671631	0.00014711	0.00001148
		-3.04467112	-0.01674816	0.01008894	0.01654470	0.00014387	0.00001139
		-3.04942699	-0.01656889	0.00999117	0.01637658	0.00014073	0.00001131
		-3.05414335	-0.01639319	0.00989485	0.01621185	0.00013769	0.00001124
		-3.05882250	-0.01622097	0.00979997	0.01605041	0.00013474	0.00001116
		-3.06346660	-0.01605215	0.00970649	0.01589217	0.00013189	0.00001109
		-3.06807773	-0.01588664	0.00961439	0.01573705	0.00012912	0.00001103
		-3.07265789	-0.01572436	0.00952364	0.01558496	0.00012643	0.00001097
		-3.07720897	-0.01556522	0.00943423	0.01543582	0.00012382	0.00001091
		-3.08173279	-0.01540916	0.00934613	0.01528956	0.00012129	0.00001085
		-3.08623109	-0.01525608	0.00925931	0.01514610	0.00011883	0.00001080
		-3.09070551	-0.01510592	0.00917376	0.01500536	0.00011645	0.00001074
		-3.09515765	-0.01495861	0.00908946	0.01486728	0.00011413	0.00001069
		-3.09958899	-0.01481408	0.00900637	0.01473179	0.00011188	0.00001064
		-3.10400100	-0.01467225	0.00892449	0.01459882	0.00010969	0.00001060
		-3.10839504	-0.01453308	0.00884379	0.01446831	0.00010756	0.00001055
		-3.11277241	-0.01439648	0.00876425	0.01434019	0.00010549	0.00001051
		-3.11713438	-0.01426240	0.00868586	0.01421441	0.00010348	0.00001047
		-3.12148213	-0.01413079	0.00860858	0.01409091	0.00010152	0.00001042
		-3.12581680	-0.01400157	0.00853241	0.01396963	0.00009962	0.00001038
		-3.13013947	-0.01387471	0.00845733	0.01385052	0.00009776	0.00001034
		-3.13445117	-0.01375013	0.00838331	0.01373352	0.00009596	0.00001031
		-3.13875289	-0.01362779	0.00831034	0.01361859	0.00009420	0.00001027
		-3.14304556	-0.01350764	0.00823841	0.01350567	0.00009249	0.00001023
		-3.14733008	-0.01338963	0.00816748	0.01339472	0.00009083	0.00001020
		-3.15160728	-0.01327370	0.00809756	0.01328569	0.00008920	0.00001016
		-3.15587797	-0.01315981	0.00802862	0.01317853	0.00008762	0.00001012
		-3.16014293	-0.01304792	0.00796064	0.01307320	0.00008608	0.00001009
		-3.16440288	-0.01293797	0.00789361	0.01296967	0.00008458	0.00001006
		-3.16865852	-0.01282993	0.00782751	0.01286788	0.00008311	0.00001002
		-3.17291049	-0.01272375	0.00776233	0.01276779	0.00008169	0.00000999
		-3.17715942	-0.01261940	0.00769805	0.01266938	0.00008029	0.00000996
		-3.18140591	-0.01251682	0.00763466	0.01257259	0.00007893	0.00000992
		-3.18565052	-0.01241598	0.00757215	0.01247740	0.00007761	0.00000989
		-3.18989378	-0.01231685	0.00751049	0.01238376	0.00007631	0.00000986
		-3.19413619	-0.01221938	0.00744968	0.01229165	0.00007505	0.00000983
		-3.19837823	-0.01212354	0.00738970	0.01220103	0.00007381	0.00000979
		-3.20262035	-0.01202930	0.00733053	0.01211186	0.00007261	0.00000976
		-3.20686298	-0.01193661	0.00727217	0.01202412	0.00007143	0.00000973
		-3.21110653	-0.01184546	0.00721461	0.01193777	0.00007028	0.00000970
		-3.21535137	-0.01175579	0.00715782	0.01185279	0.00006916	0.00000967
		-3.21959786	-0.01166759	0.00710179	0.01176914	0.00006806	0.00000964
		-3.22384634	-0.01158082	0.00704652	0.01168680	0.00006699	0.00000961
		-3.22809714	-0.01149545	0.00699199	0.01160573	0.00006594	0.00000957
		-3.23235055	-0.01141146	0.00693819	0.01152591	0.00006492	0.00000954
		-3.23660685	-0.01132880	0.00688511	0.01144732	0.00006392	0.00000951
		-3.24086631	-0.01124746	0.00683273	0.01136993	0.00006294	0.00000948
		-3.24512918	-0.01116741	0.00678105	0.01129371	0.00006198	0.00000945
		-3.24939569	-0.01108862	0.00673005	0.01121864	0.00006105	0.00000942
		-3.25366606	-0.01101107	0.00667973	0.01114470	0.00006013	0.00000939
		-3.25794050	-0.01093473	0.00663007	0.01107186	0.00005923	0.00000936
		-3.26221918	-0.01085957	0.00658106	0.01100010	0.00005835	0.00000932
		-3.26650230	-0.01078557	0.00653269	0.01092939	0.00005750	0.00000929
		-3.27079000	-0.01071272	0.00648495	0.01085973	0.00005665	0.00000926
		-3.27508246	-0.01064097	0.00643784	0.01079108	0.00005583	0.00000923
		-3.27937980	-0.01057032	0.00639134	0.01072343	0.00005503	0.00000920
		-3.28368216	-0.01050074	0.00634544	0.01065676	0.00005424	0.00000917
		-3.28798965	-0.01043222	0.00630013	0.01059104	0.00005346	0.00000914
		-3.29230239	-0.01036472	0.00625541	0.01052626	0.00005271	0.00000910
		-3.29662047	-0.01029823	0.00621126	0.01046241	0.00005196	0.00000907
		-3.30094399	-0.01023273	0.00616768	0.01039945	0.00005124	0.00000904
		-3.30527303	-0.01016819	0.00612465	0.01033738	0.00005053	0.00000901
		-3.30960766	-0.01010461	0.00608218	0.01027618	0.00004983	0.00000898
		-3.31394795	-0.01004197	0.00604024	0.01021583	0.00004914	0.00000895
		-3.31829395	-0.00998024	0.00599883	0.01015631	0.00004847	0.00000891
		-3.32264573	-0.00991940	0.00595795	0.01009762	0.00004781	0.00000888
		-3.32700331	-0.00985945	0.00591759	0.01003973	0.00004717	0.00000885
		-3.33136675	-0.00980035	0.00587773	0.00998262	0.00004654	0.00000882
		-3.33573607	-0.00974211	0.00583838	0.00992630	0.00004592	0.00000879
		-3.34011130	-0.00968470	0.00579951	0.00987073	0.00004531	0.00000875
		-3.34449246	-0.00962810	0.00576113	0.00981591	0.00004471	0.00000872
		-3.34887956	-0.00957230	0.00572323	0.00976182	0.00004413	0.00000869
		-3.35327261	-0.00951729	0.00568581	0.00970845	0.00004355	0.00000866
		-3.35767163	-0.00946304	0.00564884	0.00965578	0.00004299	0.00000863
		-3.36207660	-0.00940956	0.00561233	0.00960381	0.00004244	0.00000859
		-3.36648753	-0.00935681	0.00557627	0.00955252	0.00004189	0.00000856
		-3.37090440	-0.00930480	0.00554066	0.00950190	0.00004136	0.00000853
		-3.37532721	-0.00925350	0.00550548	0.00945194	0.00004084	0.00000850
		-3.37975593	-0.00920290	0.00547073	0.00940262	0.00004032	0.00000847
		-3.38419056	-0.00915300	0.00543641	0.00935393	0.00003982	0.00000843
		-3.38863105	-0.00910377	0.00540251	0.00930586	0.00003932	0.00000840
		-3.39307740	-0.00905520	0.00536901	0.00925841	0.00003884	0.00000837
		-3.39752956	-0.00900729	0.00533593	0.00921156	0.00003836	0.00000834
		-3.40198751	-0.00896002	0.00530324	0.00916530	0.00003789	0.00000830
		-3.40645121	-0.00891338	0.00527095	0.00911962	0.00003743	0.00000827
		-3.41092063	-0.00886736	0.00523904	0.00907451	0.00003697	0.00000824
		-3.41539571	-0.00882195	0.00520752	0.00902996	0.00003653	0.00000821
		-3.41987643	-0.00877713	0.00517637	0.00898596	0.00003609	0.00000817
		-3.42436272	-0.00873290	0.00514560	0.00894250	0.00003566	0.00000814
		-3.42885456	-0.00868925	0.00511520	0.00889958	0.00003524	0.00000811
		-3.43335188	-0.00864617	0.00508515	0.00885718	0.00003482	0.00000808
		-3.43785464	-0.00860364	0.00505546	0.00881530	0.00003441	0.00000804
		-3.44236278	-0.00856166	0.00502613	0.00877392	0.00003401	0.00000801
		-3.44687625	-0.00852021	0.00499714	0.00873304	0.00003361	0.00000798
		-3.45139500	-0.00847930	0.00496849	0.00869264	0.00003323	0.00000795
		-3.45591895	-0.00843890	0.00494017	0.00865273	0.00003284	0.00000791
		-3.46044807	-0.00839902	0.00491219	0.00861330	0.00003247	0.00000788
		-3.46498227	-0.00835964	0.00488454	0.00857433	0.00003210	0.00000785
		-3.46952151	-0.00832075	0.00485721	0.00853582	0.00003173	0.00000782
		-3.47406572	-0.00828234	0.00483019	0.00849776	0.00003138	0.00000778
		-3.47861484	-0.00824442	0.00480349	0.00846014	0.00003102	0.00000775
		-3.48316880	-0.00820696	0.00477710	0.00842296	0.00003068	0.00000772
		-3.48772753	-0.00816996	0.00475102	0.00838622	0.00003034	0.00000769
		-3.49229097	-0.00813342	0.00472523	0.00834989	0.00003000	0.00000766
		-3.49685904	-0.00809733	0.00469975	0.00831398	0.00002967	0.00000762
		-3.50143169	-0.00806167	0.00467455	0.00827848	0.00002935	0.00000759
		-3.50600884	-0.00802644	0.00464964	0.00824338	0.00002903	0.00000756
		-3.51059042	-0.00799164	0.00462502	0.00820868	0.00002871	0.00000753
		-3.51517637	-0.00795726	0.00460068	0.00817437	0.00002840	0.00000750
		-3.51976660	-0.00792329	0.00457662	0.00814045	0.00002810	0.00000746
		-3.52436105	-0.00788972	0.00455283	0.00810691	0.00002779	0.00000743
		-3.52895964	-0.00785655	0.00452930	0.00807373	0.00002750	0.00000740
		-3.53356231	-0.00782377	0.00450605	0.00804093	0.00002721	0.00000737
		-3.53816898	-0.00779138	0.00448306	0.00800849	0.00002692	0.00000734
		-3.54277957	-0.00775937	0.00446032	0.00797640	0.00002664	0.00000730
		-3.54739402	-0.00772773	0.00443784	0.00794466	0.00002636	0.00000727
		-3.55201224	-0.00769645	0.00441562	0.00791327	0.00002608	0.00000724
		-3.55663417	-0.00766554	0.00439364	0.00788222	0.00002581	0.00000721
		-3.56125973	-0.00763498	0.00437190	0.00785151	0.00002555	0.00000718
		-3.56588885	-0.00760478	0.00435041	0.00782113	0.00002528	0.00000715
		-3.57052145	-0.00757491	0.00432916	0.00779107	0.00002502	0.00000712
		-3.57515745	-0.00754539	0.00430814	0.00776133	0.00002477	0.00000708
		-3.57979678	-0.00751620	0.00428736	0.00773191	0.00002452	0.00000705
		-3.58443937	-0.00748734	0.00426681	0.00770279	0.00002427	0.00000702
		-3.58908514	-0.00745880	0.00424648	0.00767399	0.00002403	0.00000699
		-3.59373401	-0.00743059	0.00422637	0.00764549	0.00002379	0.00000696
		-3.59838592	-0.00740268	0.00420649	0.00761728	0.00002355	0.00000693
		-3.60304078	-0.00737509	0.00418682	0.00758937	0.00002331	0.00000690
		-3.60769852	-0.00734780	0.00416737	0.00756175	0.00002308	0.00000687
		-3.61235907	-0.00732081	0.00414813	0.00753441	0.00002286	0.00000684
		-3.61702235	-0.00729411	0.00412910	0.00750735	0.00002263	0.00000681
		-3.62168828	-0.00726771	0.00411028	0.00748058	0.00002241	0.00000678
		-3.62635680	-0.00724159	0.00409166	0.00745407	0.00002219	0.00000674
		-3.63102782	-0.00721575	0.00407325	0.00742783	0.00002198	0.00000671
		-3.63570128	-0.00719020	0.00405503	0.00740186	0.00002176	0.00000668
		-3.64037709	-0.00716491	0.00403700	0.00737615	0.00002156	0.00000665
		-3.64505519	-0.00713990	0.00401918	0.00735070	0.00002135	0.00000662
		-3.64973550	-0.00711515	0.00400154	0.00732550	0.00002115	0.00000659
		-3.65441795	-0.00709066	0.00398409	0.00730055	0.00002094	0.00000656
		-3.65910247	-0.00706643	0.00396683	0.00727585	0.00002075	0.00000653
		-3.66378898	-0.00704246	0.00394975	0.00725140	0.00002055	0.00000650
		-3.66847740	-0.00701873	0.00393286	0.00722718	0.00002036	0.00000647
		-3.67316767	-0.00699526	0.00391614	0.00720320	0.00002017	0.00000644
		-3.67785972	-0.00697202	0.00389960	0.00717946	0.00001998	0.00000641
		-3.68255347	-0.00694903	0.00388324	0.00715594	0.00001979	0.00000638
		-3.68724885	-0.00692627	0.00386705	0.00713266	0.00001961	0.00000636
		-3.69194579	-0.00690374	0.00385103	0.00710959	0.00001943	0.00000633
		-3.69664421	-0.00688145	0.00383518	0.00708675	0.00001925	0.00000630
		-3.70134406	-0.00685938	0.00381949	0.00706413	0.00001907	0.00000627
		-3.70604525	-0.00683753	0.00380397	0.00704173	0.00001890	0.00000624
		-3.71074772	-0.00681590	0.00378861	0.00701953	0.00001873	0.00000621
		-3.71545140	-0.00679449	0.00377342	0.00699755	0.00001856	0.00000618
		-3.72015622	-0.00677330	0.00375838	0.00697577	0.00001839	0.00000615
		-3.72486211	-0.00675231	0.00374349	0.00695420	0.00001823	0.00000612
		-3.72956899	-0.00673153	0.00372877	0.00693283	0.00001806	0.00000609
		-3.73427682	-0.00671096	0.00371419	0.00691165	0.00001790	0.00000607
		-3.73898550	-0.00669059	0.00369976	0.00689068	0.00001774	0.00000604
		-3.74369498	-0.00667042	0.00368549	0.00686989	0.00001759	0.00000601
		-3.74840519	-0.00665044	0.00367136	0.00684930	0.00001743	0.00000598
		-3.75311607	-0.00663066	0.00365738	0.00682890	0.00001728	0.00000595
		-3.75782754	-0.00661107	0.00364354	0.00680868	0.00001713	0.00000592
		-3.76253955	-0.00659167	0.00362984	0.00678865	0.00001698	0.00000590
		-3.76725202	-0.00657245	0.00361628	0.00676880	0.00001683	0.00000587
		-3.77196489	-0.00655341	0.00360286	0.00674913	0.00001668	0.00000584
		-3.77667809	-0.00653456	0.00358958	0.00672963	0.00001654	0.00000581
		-3.78139156	-0.00651589	0.00357643	0.00671031	0.00001640	0.00000579
		-3.78610525	-0.00649739	0.00356342	0.00669116	0.00001625	0.00000576
		-3.79081907	-0.00647906	0.00355053	0.00667218	0.00001612	0.00000573
		-3.79553298	-0.00646091	0.00353778	0.00665337	0.00001598	0.00000570
		-3.80024690	-0.00644292	0.00352516	0.00663472	0.00001584	0.00000568
		-3.80496078	-0.00642510	0.00351266	0.00661624	0.00001571	0.00000565
		-3.80967455	-0.00640745	0.00350029	0.00659792	0.00001557	0.00000562
		-3.81438815	-0.00638996	0.00348804	0.00657976	0.00001544	0.00000560
		-3.81910152	-0.00637262	0.00347592	0.00656176	0.00001531	0.00000557
		-3.82381460	-0.00635545	0.00346392	0.00654391	0.00001519	0.00000554
		-3.82852732	-0.00633843	0.00345204	0.00652622	0.00001506	0.00000552
		-3.83323964	-0.00632157	0.00344027	0.00650867	0.00001493	0.00000549
		-3.83795149	-0.00630485	0.00342863	0.00649128	0.00001481	0.00000546
		-3.84266280	-0.00628829	0.00341709	0.00647404	0.00001469	0.00000544
		-3.84737353	-0.00627187	0.00340568	0.00645694	0.00001457	0.00000541
		-3.85208361	-0.00625560	0.00339437	0.00643998	0.00001445	0.00000538
		-3.85679299	-0.00623948	0.00338318	0.00642317	0.00001433	0.00000536
		-3.86150160	-0.00622349	0.00337210	0.00640650	0.00001421	0.00000533
		-3.86620939	-0.00620765	0.00336112	0.00638997	0.00001410	0.00000531
		-3.87091631	-0.00619194	0.00335026	0.00637358	0.00001398	0.00000528
		-3.87562229	-0.00617637	0.00333950	0.00635732	0.00001387	0.00000526
		-3.88032729	-0.00616094	0.00332885	0.00634120	0.00001376	0.00000523
		-3.88503124	-0.00614564	0.00331830	0.00632520	0.00001364	0.00000521
		-3.88973410	-0.00613047	0.00330785	0.00630934	0.00001354	0.00000518
		-3.89443580	-0.00611543	0.00329750	0.00629361	0.00001343	0.00000515
		-3.89913629	-0.00610051	0.00328726	0.00627801	0.00001332	0.00000513
		-3.90383552	-0.00608573	0.00327711	0.00626253	0.00001321	0.00000510
		-3.90853343	-0.00607107	0.00326707	0.00624718	0.00001311	0.00000508
		-3.91322998	-0.00605653	0.00325712	0.00623195	0.00001301	0.00000506
		-3.91792511	-0.00604212	0.00324726	0.00621684	0.00001290	0.00000503
		-3.92261876	-0.00602782	0.00323750	0.00620185	0.00001280	0.00000501
		-3.92731089	-0.00601364	0.00322784	0.00618698	0.00001270	0.00000498
		-3.93200144	-0.00599958	0.00321826	0.00617223	0.00001260	0.00000496
		-3.93669036	-0.00598564	0.00320878	0.00615760	0.00001250	0.00000493
		-3.94137761	-0.00597181	0.00319939	0.00614308	0.00001241	0.00000491
		-3.94606313	-0.00595809	0.00319009	0.00612867	0.00001231	0.00000488
		-3.95074687	-0.00594449	0.00318088	0.00611438	0.00001221	0.00000486
		-3.95542878	-0.00593099	0.00317175	0.00610019	0.00001212	0.00000484
		-3.96010882	-0.00591761	0.00316271	0.00608612	0.00001203	0.00000481
		-3.96478693	-0.00590433	0.00315376	0.00607215	0.00001193	0.00000479
		-3.96946306	-0.00589116	0.00314489	0.00605829	0.00001184	0.00000477
		-3.97413718	-0.00587809	0.00313611	0.00604454	0.00001175	0.00000474
		-3.97880922	-0.00586512	0.00312740	0.00603089	0.00001166	0.00000472
		-3.98347915	-0.00585226	0.00311878	0.00601735	0.00001157	0.00000470
		-3.98814692	-0.00583950	0.00311024	0.00600390	0.00001149	0.00000467
		-3.99281247	-0.00582684	0.00310178	0.00599056	0.00001140	0.00000465
		-3.99747577	-0.00581428	0.00309340	0.00597732	0.00001131	0.00000463
		-4.00213677	-0.00580181	0.00308510	0.00596417	0.00001123	0.00000460
		-4.00679542	-0.00578944	0.00307688	0.00595113	0.00001114	0.00000458
		-4.01145168	-0.00577717	0.00306873	0.00593818	0.00001106	0.00000456
		-4.01610551	-0.00576499	0.00306065	0.00592532	0.00001098	0.00000454
		-4.02075685	-0.00575290	0.00305266	0.00591256	0.00001090	0.00000451
		-4.02540567	-0.00574091	0.00304473	0.00589990	0.00001082	0.00000449
		-4.03005191	-0.00572900	0.00303688	0.00588732	0.00001073	0.00000447
		-4.03469555	-0.00571719	0.00302910	0.00587484	0.00001066	0.00000445
		-4.03933654	-0.00570546	0.00302139	0.00586244	0.00001058	0.00000443
		-4.04397482	-0.00569382	0.00301375	0.00585014	0.00001050	0.00000440
		-4.04861037	-0.00568227	0.00300619	0.00583792	0.00001042	0.00000438
		-4.05324314	-0.00567080	0.00299869	0.00582579	0.00001035	0.00000436
		-4.05787308	-0.00565942	0.00299126	0.00581375	0.00001027	0.00000434
		-4.06250017	-0.00564812	0.00298390	0.00580179	0.00001019	0.00000432
		-4.06712435	-0.00563690	0.00297660	0.00578992	0.00001012	0.00000430
		-4.07174558	-0.00562577	0.00296937	0.00577813	0.00001005	0.00000427
		-4.07636383	-0.00561471	0.00296221	0.00576642	0.00000997	0.00000425
		-4.08097906	-0.00560374	0.00295511	0.00575480	0.00000990	0.00000423
		-4.08559122	-0.00559284	0.00294807	0.00574325	0.00000983	0.00000421
		-4.09020028	-0.00558202	0.00294110	0.00573179	0.00000976	0.00000419
		-4.09480620	-0.00557128	0.00293419	0.00572040	0.00000969	0.00000417
		-4.09940894	-0.00556061	0.00292734	0.00570909	0.00000962	0.00000415
		-4.10400846	-0.00555002	0.00292056	0.00569786	0.00000955	0.00000413
		-4.10860473	-0.00553950	0.00291383	0.00568670	0.00000948	0.00000411
		-4.11319770	-0.00552906	0.00290717	0.00567562	0.00000942	0.00000409
		-4.11778734	-0.00551869	0.00290056	0.00566462	0.00000935	0.00000407
		-4.12237362	-0.00550839	0.00289401	0.00565369	0.00000928	0.00000405
		-4.12695649	-0.00549816	0.00288752	0.00564283	0.00000922	0.00000403
		-4.13153592	-0.00548801	0.00288109	0.00563204	0.00000915	0.00000401
		-4.13611188	-0.00547792	0.00287471	0.00562133	0.00000909	0.00000399
		-4.14068433	-0.00546790	0.00286839	0.00561069	0.00000902	0.00000397
		-4.14525323	-0.00545795	0.00286213	0.00560011	0.00000896	0.00000395
		-4.14981854	-0.00544806	0.00285592	0.00558961	0.00000890	0.00000393
		-4.15438025	-0.00543824	0.00284976	0.00557918	0.00000883	0.00000391
		-4.15893830	-0.00542849	0.00284366	0.00556881	0.00000877	0.00000389
		-4.16349267	-0.00541880	0.00283761	0.00555851	0.00000871	0.00000387
		-4.16804332	-0.00540918	0.00283162	0.00554827	0.00000865	0.00000385
		-4.17259021	-0.00539962	0.00282567	0.00553811	0.00000859	0.00000383
		-4.17713333	-0.00539012	0.00281978	0.00552800	0.00000853	0.00000381
		-4.18167262	-0.00538069	0.00281394	0.00551796	0.00000847	0.00000379
		-4.18620807	-0.00537131	0.00280815	0.00550799	0.00000841	0.00000377
		-4.19073963	-0.00536200	0.00280241	0.00549808	0.00000836	0.00000375
		-4.19526728	-0.00535274	0.00279671	0.00548823	0.00000830	0.00000373
		-4.19979098	-0.00534355	0.00279107	0.00547844	0.00000824	0.00000372
		-4.20431070	-0.00533441	0.00278548	0.00546871	0.00000818	0.00000370
		-4.20882641	-0.00532534	0.00277993	0.00545905	0.00000813	0.00000368
		-4.21333809	-0.00531632	0.00277443	0.00544944	0.00000807	0.00000366
		-4.21784569	-0.00530735	0.00276897	0.00543989	0.00000802	0.00000364
		-4.22234919	-0.00529845	0.00276357	0.00543040	0.00000796	0.00000362
		-4.22684856	-0.00528959	0.00275820	0.00542097	0.00000791	0.00000361
		-4.23134377	-0.00528080	0.00275289	0.00541160	0.00000786	0.00000359
		-4.23583479	-0.00527206	0.00274762	0.00540228	0.00000780	0.00000357
		-4.24032159	-0.00526337	0.00274239	0.00539302	0.00000775	0.00000355
		-4.24480413	-0.00525473	0.00273720	0.00538382	0.00000770	0.00000353
		-4.24928241	-0.00524615	0.00273206	0.00537467	0.00000765	0.00000352
		-4.25375637	-0.00523762	0.00272697	0.00536557	0.00000759	0.00000350
		-4.25822600	-0.00522914	0.00272191	0.00535653	0.00000754	0.00000348
		-4.26269127	-0.00522071	0.00271690	0.00534754	0.00000749	0.00000346
		-4.26715214	-0.00521233	0.00271193	0.00533861	0.00000744	0.00000345
		-4.27160860	-0.00520401	0.00270700	0.00532973	0.00000739	0.00000343
		-4.27606061	-0.00519573	0.00270211	0.00532090	0.00000734	0.00000341
		-4.28050816	-0.00518750	0.00269726	0.00531212	0.00000729	0.00000339
		-4.28495120	-0.00517932	0.00269245	0.00530339	0.00000725	0.00000338
		-4.28938971	-0.00517119	0.00268767	0.00529471	0.00000720	0.00000336
		-4.29382368	-0.00516310	0.00268294	0.00528608	0.00000715	0.00000334
		-4.29825306	-0.00515506	0.00267825	0.00527750	0.00000710	0.00000333
		-4.30267785	-0.00514707	0.00267359	0.00526897	0.00000706	0.00000331
		-4.30709800	-0.00513912	0.00266898	0.00526049	0.00000701	0.00000329
		-4.31151350	-0.00513122	0.00266440	0.00525206	0.00000696	0.00000328
		-4.31592431	-0.00512337	0.00265985	0.00524367	0.00000692	0.00000326
		-4.32033043	-0.00511555	0.00265535	0.00523533	0.00000687	0.00000324
		-4.32473181	-0.00510779	0.00265088	0.00522704	0.00000683	0.00000323
		-4.32912844	-0.00510006	0.00264644	0.00521879	0.00000678	0.00000321
		-4.33352030	-0.00509238	0.00264204	0.00521059	0.00000674	0.00000319
		-4.33790735	-0.00508474	0.00263768	0.00520243	0.00000669	0.00000318
		-4.34228957	-0.00507715	0.00263335	0.00519432	0.00000665	0.00000316
		-4.34666695	-0.00506959	0.00262906	0.00518625	0.00000661	0.00000315
		-4.35103946	-0.00506208	0.00262479	0.00517822	0.00000656	0.00000313
		-4.35540706	-0.00505461	0.00262057	0.00517024	0.00000652	0.00000311
		-4.35976976	-0.00504718	0.00261637	0.00516230	0.00000648	0.00000310
		-4.36412751	-0.00503978	0.00261221	0.00515441	0.00000644	0.00000308
		-4.36848029	-0.00503243	0.00260808	0.00514656	0.00000639	0.00000307
		-4.37282810	-0.00502512	0.00260399	0.00513874	0.00000635	0.00000305
		-4.37717089	-0.00501785	0.00259992	0.00513097	0.00000631	0.00000304
		-4.38150866	-0.00501061	0.00259589	0.00512324	0.00000627	0.00000302
		-4.38584137	-0.00500341	0.00259189	0.00511555	0.00000623	0.00000301
		-4.39016901	-0.00499626	0.00258792	0.00510791	0.00000619	0.00000299
		-4.39449156	-0.00498913	0.00258397	0.00510030	0.00000615	0.00000298
		-4.39880900	-0.00498205	0.00258006	0.00509273	0.00000611	0.00000296
		-4.40312130	-0.00497500	0.00257618	0.00508520	0.00000607	0.00000295
		-4.40742845	-0.00496799	0.00257233	0.00507770	0.00000603	0.00000293
		-4.41173042	-0.00496101	0.00256851	0.00507025	0.00000599	0.00000292
		-4.41602719	-0.00495407	0.00256472	0.00506283	0.00000596	0.00000290
		-4.42031875	-0.00494717	0.00256096	0.00505545	0.00000592	0.00000289
		-4.42460508	-0.00494030	0.00255722	0.00504811	0.00000588	0.00000287
		-4.42888615	-0.00493346	0.00255351	0.00504081	0.00000584	0.00000286
		-4.43316194	-0.00492666	0.00254984	0.00503354	0.00000581	0.00000284
		-4.43743244	-0.00491989	0.00254618	0.00502631	0.00000577	0.00000283
		-4.44169763	-0.00491316	0.00254256	0.00501911	0.00000573	0.00000281
		-4.44595749	-0.00490646	0.00253896	0.00501195	0.00000570	0.00000280
		-4.45021200	-0.00489979	0.00253539	0.00500482	0.00000566	0.00000279
		-4.45446115	-0.00489316	0.00253185	0.00499773	0.00000562	0.00000277
		-4.45870490	-0.00488655	0.00252833	0.00499068	0.00000559	0.00000276
		-4.46294326	-0.00487998	0.00252484	0.00498365	0.00000555	0.00000274
		-4.46717619	-0.00487344	0.00252137	0.00497667	0.00000552	0.00000273
		-4.47140368	-0.00486693	0.00251793	0.00496971	0.00000548	0.00000272
		-4.47562571	-0.00486046	0.00251451	0.00496279	0.00000545	0.00000270
		-4.47984227	-0.00485401	0.00251112	0.00495590	0.00000541	0.00000269
		-4.48405334	-0.00484759	0.00250775	0.00494904	0.00000538	0.00000268
		-4.48825891	-0.00484121	0.00250441	0.00494222	0.00000535	0.00000266
		-4.49245894	-0.00483485	0.00250109	0.00493543	0.00000531	0.00000265
		-4.49665344	-0.00482852	0.00249780	0.00492867	0.00000528	0.00000264
		-4.50084238	-0.00482223	0.00249453	0.00492194	0.00000525	0.00000262
		-4.50502575	-0.00481596	0.00249128	0.00491524	0.00000521	0.00000261
		-4.50920352	-0.00480972	0.00248805	0.00490857	0.00000518	0.00000260
		-4.51337569	-0.00480350	0.00248485	0.00490194	0.00000515	0.00000258
		-4.51754224	-0.00479732	0.00248167	0.00489533	0.00000512	0.00000257
		-4.52170315	-0.00479117	0.00247851	0.00488876	0.00000508	0.00000256
		-4.52585842	-0.00478504	0.00247538	0.00488221	0.00000505	0.00000254
		-4.53000801	-0.00477894	0.00247227	0.00487569	0.00000502	0.00000253
		-4.53415192	-0.00477286	0.00246917	0.00486921	0.00000499	0.00000252
		-4.53829014	-0.00476682	0.00246610	0.00486275	0.00000496	0.00000251
		-4.54242264	-0.00476080	0.00246306	0.00485632	0.00000493	0.00000249
		-4.54654942	-0.00475480	0.00246003	0.00484992	0.00000490	0.00000248
		-4.55067046	-0.00474884	0.00245702	0.00484354	0.00000487	0.00000247
		-4.55478574	-0.00474290	0.00245404	0.00483720	0.00000484	0.00000245
		-4.55889526	-0.00473698	0.00245107	0.00483088	0.00000481	0.00000244
		-4.56299899	-0.00473109	0.00244812	0.00482459	0.00000478	0.00000243
		-4.56709693	-0.00472522	0.00244520	0.00481833	0.00000475	0.00000242
		-4.57118906	-0.00471938	0.00244229	0.00481209	0.00000472	0.00000241
		-4.57527536	-0.00471357	0.00243940	0.00480588	0.00000469	0.00000239
		-4.57935583	-0.00470778	0.00243654	0.00479970	0.00000466	0.00000238
		-4.58343045	-0.00470201	0.00243369	0.00479354	0.00000463	0.00000237
		-4.58749921	-0.00469627	0.00243086	0.00478741	0.00000460	0.00000236
		-4.59156210	-0.00469055	0.00242805	0.00478131	0.00000457	0.00000235
		-4.59561910	-0.00468485	0.00242526	0.00477523	0.00000454	0.00000233
		-4.59967020	-0.00467918	0.00242248	0.00476918	0.00000452	0.00000232
		-4.60371538	-0.00467353	0.00241973	0.00476315	0.00000449	0.00000231
		-4.60775465	-0.00466791	0.00241699	0.00475714	0.00000446	0.00000230
		-4.61178797	-0.00466230	0.00241427	0.00475116	0.00000443	0.00000229
		-4.61581536	-0.00465672	0.00241157	0.00474521	0.00000441	0.00000228
		-4.61983678	-0.00465116	0.00240889	0.00473928	0.00000438	0.00000226
		-4.62385223	-0.00464563	0.00240622	0.00473337	0.00000435	0.00000225
		-4.62786170	-0.00464011	0.00240357	0.00472749	0.00000432	0.00000224
		-4.63186517	-0.00463462	0.00240093	0.00472163	0.00000430	0.00000223
		-4.63586264	-0.00462915	0.00239832	0.00471580	0.00000427	0.00000222
		-4.63985410	-0.00462370	0.00239572	0.00470998	0.00000425	0.00000221
		-4.64383953	-0.00461827	0.00239313	0.00470419	0.00000422	0.00000220
		-4.64781892	-0.00461286	0.00239057	0.00469843	0.00000419	0.00000219
		-4.65179227	-0.00460748	0.00238802	0.00469268	0.00000417	0.00000217
		-4.65575956	-0.00460211	0.00238548	0.00468696	0.00000414	0.00000216
		-4.65972078	-0.00459677	0.00238296	0.00468126	0.00000412	0.00000215
		-4.66367592	-0.00459144	0.00238046	0.00467559	0.00000409	0.00000214
		-4.66762497	-0.00458614	0.00237797	0.00466993	0.00000407	0.00000213
		-4.67156793	-0.00458085	0.00237549	0.00466430	0.00000404	0.00000212
		-4.67550478	-0.00457558	0.00237303	0.00465869	0.00000402	0.00000211
		-4.67943552	-0.00457034	0.00237059	0.00465310	0.00000399	0.00000210
		-4.68336012	-0.00456511	0.00236816	0.00464753	0.00000397	0.00000209
		-4.68727860	-0.00455990	0.00236575	0.00464198	0.00000394	0.00000208
		-4.69119092	-0.00455472	0.00236335	0.00463645	0.00000392	0.00000207
		-4.69509710	-0.00454955	0.00236096	0.00463095	0.00000389	0.00000206
		-4.69899711	-0.00454440	0.00235859	0.00462546	0.00000387	0.00000205
		-4.70289095	-0.00453926	0.00235623	0.00461999	0.00000385	0.00000204
		-4.70677861	-0.00453415	0.00235389	0.00461455	0.00000382	0.00000203
		-4.71066008	-0.00452905	0.00235156	0.00460912	0.00000380	0.00000202
		-4.71453535	-0.00452398	0.00234924	0.00460372	0.00000378	0.00000200
		-4.71840442	-0.00451892	0.00234694	0.00459833	0.00000375	0.00000199
		-4.72226728	-0.00451387	0.00234465	0.00459296	0.00000373	0.00000198
		-4.72612392	-0.00450885	0.00234237	0.00458761	0.00000371	0.00000197
		-4.72997433	-0.00450384	0.00234011	0.00458229	0.00000369	0.00000196
		-4.73381850	-0.00449885	0.00233786	0.00457698	0.00000366	0.00000195
		-4.73765643	-0.00449388	0.00233562	0.00457168	0.00000364	0.00000195
		-4.74148810	-0.00448893	0.00233340	0.00456641	0.00000362	0.00000194
		-4.74531352	-0.00448399	0.00233119	0.00456116	0.00000360	0.00000193
		-4.74913267	-0.00447907	0.00232899	0.00455592	0.00000357	0.00000192
		-4.75294555	-0.00447416	0.00232680	0.00455071	0.00000355	0.00000191
		-4.75675214	-0.00446927	0.00232462	0.00454551	0.00000353	0.00000190
		-4.76055246	-0.00446440	0.00232246	0.00454033	0.00000351	0.00000189
		-4.76434647	-0.00445955	0.00232031	0.00453516	0.00000349	0.00000188
		-4.76813419	-0.00445471	0.00231816	0.00453002	0.00000347	0.00000187
		-4.77191560	-0.00444988	0.00231604	0.00452489	0.00000344	0.00000186
		-4.77569070	-0.00444507	0.00231392	0.00451978	0.00000342	0.00000185
		-4.77945947	-0.00444028	0.00231181	0.00451469	0.00000340	0.00000184
		-4.78322192	-0.00443550	0.00230972	0.00450961	0.00000338	0.00000183
		-4.78697804	-0.00443074	0.00230763	0.00450455	0.00000336	0.00000182
		-4.79072783	-0.00442600	0.00230556	0.00449951	0.00000334	0.00000181
		-4.79447127	-0.00442127	0.00230350	0.00449448	0.00000332	0.00000180
		-4.79820836	-0.00441655	0.00230145	0.00448947	0.00000330	0.00000179
		-4.80193909	-0.00441185	0.00229941	0.00448448	0.00000328	0.00000178
		-4.80566347	-0.00440716	0.00229738	0.00447951	0.00000326	0.00000178
		-4.80938148	-0.00440249	0.00229536	0.00447454	0.00000324	0.00000177
		-4.81309312	-0.00439783	0.00229335	0.00446960	0.00000322	0.00000176
		-4.81679838	-0.00439319	0.00229135	0.00446467	0.00000320	0.00000175
		-4.82049726	-0.00438856	0.00228937	0.00445976	0.00000318	0.00000174
		-4.82418976	-0.00438395	0.00228739	0.00445486	0.00000316	0.00000173
		-4.82787587	-0.00437935	0.00228542	0.00444998	0.00000314	0.00000172
		-4.83155558	-0.00437476	0.00228346	0.00444512	0.00000312	0.00000171
		-4.83522889	-0.00437019	0.00228151	0.00444027	0.00000310	0.00000171
		-4.83889580	-0.00436563	0.00227957	0.00443543	0.00000308	0.00000170
		-4.84255630	-0.00436109	0.00227764	0.00443061	0.00000306	0.00000169
		-4.84621038	-0.00435656	0.00227572	0.00442580	0.00000305	0.00000168
		-4.84985805	-0.00435204	0.00227381	0.00442101	0.00000303	0.00000167
		-4.85349930	-0.00434753	0.00227191	0.00441624	0.00000301	0.00000166
		-4.85713412	-0.00434304	0.00227002	0.00441148	0.00000299	0.00000165
		-4.86076252	-0.00433856	0.00226813	0.00440673	0.00000297	0.00000165
		-4.86438448	-0.00433410	0.00226626	0.00440200	0.00000295	0.00000164
		-4.86800001	-0.00432965	0.00226439	0.00439728	0.00000294	0.00000163
		-4.87160909	-0.00432521	0.00226253	0.00439258	0.00000292	0.00000162
		-4.87521174	-0.00432078	0.00226069	0.00438789	0.00000290	0.00000161
		-4.87880793	-0.00431637	0.00225885	0.00438321	0.00000288	0.00000161
		-4.88239768	-0.00431196	0.00225701	0.00437855	0.00000286	0.00000160
		-4.88598098	-0.00430758	0.00225519	0.00437390	0.00000285	0.00000159
		-4.88955781	-0.00430320	0.00225338	0.00436927	0.00000283	0.00000158
		-4.89312819	-0.00429883	0.00225157	0.00436465	0.00000281	0.00000157
		-4.89669211	-0.00429448	0.00224977	0.00436004	0.00000279	0.00000157
		-4.90024957	-0.00429014	0.00224798	0.00435545	0.00000278	0.00000156
		-4.90380055	-0.00428581	0.00224620	0.00435087	0.00000276	0.00000155
		-4.90734507	-0.00428150	0.00224442	0.00434630	0.00000274	0.00000154
		-4.91088312	-0.00427719	0.00224266	0.00434174	0.00000272	0.00000153
		-4.91441469	-0.00427290	0.00224090	0.00433720	0.00000271	0.00000153
		-4.91793978	-0.00426862	0.00223915	0.00433267	0.00000269	0.00000152
		-4.92145840	-0.00426435	0.00223740	0.00432816	0.00000267	0.00000151
		-4.92497053	-0.00426009	0.00223566	0.00432366	0.00000266	0.00000150
		-4.92847618	-0.00425584	0.00223394	0.00431916	0.00000264	0.00000150
		-4.93197535	-0.00425160	0.00223221	0.00431469	0.00000262	0.00000149
		-4.93546803	-0.00424738	0.00223050	0.00431022	0.00000261	0.00000148
		-4.93895423	-0.00424317	0.00222879	0.00430577	0.00000259	0.00000147
		-4.94243393	-0.00423896	0.00222709	0.00430133	0.00000258	0.00000147
		-4.94590714	-0.00423477	0.00222540	0.00429690	0.00000256	0.00000146
		-4.94937386	-0.00423059	0.00222371	0.00429248	0.00000254	0.00000145
		-4.95283409	-0.00422642	0.00222203	0.00428808	0.00000253	0.00000144
		-4.95628782	-0.00422226	0.00222036	0.00428368	0.00000251	0.00000144
		-4.95973505	-0.00421811	0.00221869	0.00427930	0.00000250	0.00000143
		-4.96317579	-0.00421397	0.00221703	0.00427493	0.00000248	0.00000142
		-4.96661002	-0.00420984	0.00221538	0.00427058	0.00000246	0.00000142
		-4.97003776	-0.00420573	0.00221373	0.00426623	0.00000245	0.00000141
		-4.97345900	-0.00420162	0.00221209	0.00426189	0.00000243	0.00000140
		-4.97687374	-0.00419752	0.00221046	0.00425757	0.00000242	0.00000139
		-4.98028197	-0.00419344	0.00220883	0.00425326	0.00000240	0.00000139
		-4.98368371	-0.00418936	0.00220721	0.00424896	0.00000239	0.00000138
		-4.98707894	-0.00418529	0.00220559	0.00424467	0.00000237	0.00000137
		-4.99046767	-0.00418123	0.00220398	0.00424039	0.00000236	0.00000137
		-4.99384989	-0.00417719	0.00220238	0.00423612	0.00000234	0.00000136
		-4.99722561	-0.00417315	0.00220078	0.00423187	0.00000233	0.00000135
		-5.00059483	-0.00416912	0.00219919	0.00422762	0.00000231	0.00000135
		-5.00395754	-0.00416511	0.00219761	0.00422339	0.00000230	0.00000134
		-5.00731375	-0.00416110	0.00219603	0.00421916	0.00000228	0.00000133
		-5.01066346	-0.00415710	0.00219445	0.00421495	0.00000227	0.00000133
		-5.01400667	-0.00415311	0.00219288	0.00421075	0.00000226	0.00000132
		-5.01734337	-0.00414913	0.00219132	0.00420655	0.00000224	0.00000131
		-5.02067356	-0.00414516	0.00218976	0.00420237	0.00000223	0.00000131
		-5.02399726	-0.00414120	0.00218821	0.00419820	0.00000221	0.00000130
		-5.02731445	-0.00413724	0.00218666	0.00419404	0.00000220	0.00000129
		-5.03062515	-0.00413330	0.00218512	0.00418989	0.00000218	0.00000129
		-5.03392934	-0.00412937	0.00218359	0.00418574	0.00000217	0.00000128
		-5.03722703	-0.00412544	0.00218206	0.00418161	0.00000216	0.00000127
		-5.04051822	-0.00412153	0.00218053	0.00417749	0.00000214	0.00000127
		-5.04380292	-0.00411762	0.00217901	0.00417338	0.00000213	0.00000126
		-5.04708112	-0.00411372	0.00217750	0.00416928	0.00000212	0.00000125
		-5.05035282	-0.00410983	0.00217599	0.00416519	0.00000210	0.00000125
		-5.05361802	-0.00410595	0.00217448	0.00416111	0.00000209	0.00000124
		-5.05687674	-0.00410208	0.00217298	0.00415703	0.00000207	0.00000124
		-5.06012896	-0.00409821	0.00217148	0.00415297	0.00000206	0.00000123
		-5.06337469	-0.00409436	0.00216999	0.00414892	0.00000205	0.00000122
		-5.06661393	-0.00409051	0.00216851	0.00414488	0.00000203	0.00000122
		-5.06984668	-0.00408668	0.00216703	0.00414084	0.00000202	0.00000121
		-5.07307294	-0.00408285	0.00216555	0.00413682	0.00000201	0.00000121
		-5.07629272	-0.00407902	0.00216408	0.00413280	0.00000199	0.00000120
		-5.07950602	-0.00407521	0.00216261	0.00412880	0.00000198	0.00000119
		-5.08271283	-0.00407141	0.00216115	0.00412480	0.00000197	0.00000119
		-5.08591317	-0.00406761	0.00215969	0.00412081	0.00000196	0.00000118
		-5.08910703	-0.00406382	0.00215823	0.00411683	0.00000194	0.00000118
		-5.09229441	-0.00406004	0.00215678	0.00411286	0.00000193	0.00000117
		-5.09547532	-0.00405627	0.00215534	0.00410890	0.00000192	0.00000116
		-5.09864975	-0.00405250	0.00215390	0.00410495	0.00000191	0.00000116
		-5.10181772	-0.00404874	0.00215246	0.00410101	0.00000189	0.00000115
		-5.10497922	-0.00404500	0.00215102	0.00409707	0.00000188	0.00000115
		-5.10813425	-0.00404125	0.00214960	0.00409315	0.00000187	0.00000114
		-5.11128282	-0.00403752	0.00214817	0.00408923	0.00000186	0.00000114
		-5.11442494	-0.00403379	0.00214675	0.00408532	0.00000184	0.00000113
		-5.11756059	-0.00403008	0.00214533	0.00408142	0.00000183	0.00000112
		-5.12068979	-0.00402637	0.00214392	0.00407753	0.00000182	0.00000112
		-5.12381254	-0.00402266	0.00214251	0.00407365	0.00000181	0.00000111
		-5.12692884	-0.00401897	0.00214110	0.00406978	0.00000179	0.00000111
		-5.13003869	-0.00401528	0.00213970	0.00406591	0.00000178	0.00000110
		-5.13314210	-0.00401160	0.00213831	0.00406205	0.00000177	0.00000110
		-5.13623906	-0.00400792	0.00213691	0.00405820	0.00000176	0.00000109
		-5.13932959	-0.00400426	0.00213552	0.00405436	0.00000175	0.00000109
		-5.14241368	-0.00400060	0.00213413	0.00405053	0.00000173	0.00000108
		-5.14549135	-0.00399695	0.00213275	0.00404670	0.00000172	0.00000107
		-5.14856258	-0.00399330	0.00213137	0.00404289	0.00000171	0.00000107
		-5.15162739	-0.00398967	0.00213000	0.00403908	0.00000170	0.00000106
		-5.15468577	-0.00398604	0.00212862	0.00403528	0.00000169	0.00000106
		-5.15773774	-0.00398241	0.00212725	0.00403148	0.00000168	0.00000105
		-5.16078329	-0.00397880	0.00212589	0.00402770	0.00000167	0.00000105
		-5.16382243	-0.00397519	0.00212453	0.00402392	0.00000165	0.00000104
		-5.16685516	-0.00397159	0.00212317	0.00402015	0.00000164	0.00000104
		-5.16988149	-0.00396799	0.00212181	0.00401639	0.00000163	0.00000103
		-5.17290141	-0.00396440	0.00212046	0.00401264	0.00000162	0.00000103
		-5.17591494	-0.00396082	0.00211911	0.00400889	0.00000161	0.00000102
		-5.17892208	-0.00395724	0.00211776	0.00400515	0.00000160	0.00000102
		-5.18192282	-0.00395368	0.00211642	0.00400142	0.00000159	0.00000101
		-5.18491718	-0.00395011	0.00211508	0.00399770	0.00000158	0.00000101
		-5.18790516	-0.00394656	0.00211374	0.00399398	0.00000156	0.00000100
		-5.19088675	-0.00394301	0.00211241	0.00399027	0.00000155	0.00000100
		-5.19386198	-0.00393947	0.00211108	0.00398657	0.00000154	0.00000099
		-5.19683083	-0.00393593	0.00210975	0.00398288	0.00000153	0.00000099
		-5.19979332	-0.00393241	0.00210843	0.00397919	0.00000152	0.00000098
		-5.20274945	-0.00392888	0.00210710	0.00397551	0.00000151	0.00000098
		-5.20569922	-0.00392537	0.00210578	0.00397184	0.00000150	0.00000097
		-5.20864264	-0.00392186	0.00210447	0.00396817	0.00000149	0.00000097
		-5.21157971	-0.00391835	0.00210315	0.00396451	0.00000148	0.00000096
		-5.21451043	-0.00391486	0.00210184	0.00396086	0.00000147	0.00000096
		-5.21743482	-0.00391137	0.00210054	0.00395722	0.00000146	0.00000095
		-5.22035287	-0.00390788	0.00209923	0.00395358	0.00000145	0.00000095
		-5.22326459	-0.00390440	0.00209793	0.00394995	0.00000144	0.00000094
		-5.22616999	-0.00390093	0.00209663	0.00394633	0.00000143	0.00000094
		-5.22906906	-0.00389747	0.00209533	0.00394271	0.00000142	0.00000093
		-5.23196182	-0.00389401	0.00209404	0.00393910	0.00000141	0.00000093
		-5.23484826	-0.00389055	0.00209274	0.00393550	0.00000140	0.00000093
		-5.23772840	-0.00388711	0.00209145	0.00393191	0.00000139	0.00000092
		-5.24060224	-0.00388366	0.00209017	0.00392832	0.00000138	0.00000092
		-5.24346978	-0.00388023	0.00208888	0.00392473	0.00000137	0.00000091
		-5.24633103	-0.00387680	0.00208760	0.00392116	0.00000136	0.00000091
		-5.24918599	-0.00387338	0.00208632	0.00391759	0.00000135	0.00000090
		-5.25203467	-0.00386996	0.00208504	0.00391403	0.00000134	0.00000090
		-5.25487707	-0.00386655	0.00208377	0.00391047	0.00000133	0.00000089
		-5.25771320	-0.00386314	0.00208250	0.00390692	0.00000132	0.00000089
		-5.26054307	-0.00385974	0.00208123	0.00390338	0.00000131	0.00000088
		-5.26336668	-0.00385634	0.00207996	0.00389984	0.00000130	0.00000088
		-5.26618402	-0.00385295	0.00207869	0.00389631	0.00000129	0.00000088
		-5.26899512	-0.00384957	0.00207743	0.00389279	0.00000128	0.00000087
		-5.27179998	-0.00384619	0.00207617	0.00388927	0.00000127	0.00000087
		-5.27459860	-0.00384282	0.00207491	0.00388576	0.00000126	0.00000086
		-5.27739098	-0.00383945	0.00207365	0.00388226	0.00000125	0.00000086
		-5.28017713	-0.00383609	0.00207240	0.00387876	0.00000124	0.00000085
		-5.28295706	-0.00383274	0.00207115	0.00387527	0.00000123	0.00000085
		-5.28573078	-0.00382939	0.00206989	0.00387178	0.00000122	0.00000085
		-5.28849828	-0.00382604	0.00206865	0.00386830	0.00000121	0.00000084
		-5.29125958	-0.00382271	0.00206740	0.00386483	0.00000120	0.00000084
		-5.29401468	-0.00381937	0.00206616	0.00386136	0.00000119	0.00000083
		-5.29676358	-0.00381604	0.00206491	0.00385790	0.00000118	0.00000083
		-5.29950630	-0.00381272	0.00206367	0.00385445	0.00000118	0.00000083
		-5.30224283	-0.00380940	0.00206243	0.00385100	0.00000117	0.00000082
		-5.30497319	-0.00380609	0.00206120	0.00384755	0.00000116	0.00000082
		-5.30769738	-0.00380279	0.00205996	0.00384412	0.00000115	0.00000081
		-5.31041540	-0.00379948	0.00205873	0.00384068	0.00000114	0.00000081
		-5.31312727	-0.00379619	0.00205750	0.00383726	0.00000113	0.00000081
		-5.31583299	-0.00379290	0.00205627	0.00383384	0.00000112	0.00000080
		-5.31853255	-0.00378961	0.00205504	0.00383043	0.00000111	0.00000080
		-5.32122598	-0.00378633	0.00205382	0.00382702	0.00000110	0.00000079
		-5.32391328	-0.00378306	0.00205259	0.00382362	0.00000110	0.00000079
		-5.32659445	-0.00377979	0.00205137	0.00382022	0.00000109	0.00000079
		-5.32926950	-0.00377652	0.00205015	0.00381683	0.00000108	0.00000078
		-5.33193843	-0.00377326	0.00204893	0.00381344	0.00000107	0.00000078
		-5.33460126	-0.00377000	0.00204772	0.00381006	0.00000106	0.00000077
		-5.33725798	-0.00376675	0.00204650	0.00380669	0.00000105	0.00000077
		-5.33990861	-0.00376351	0.00204529	0.00380332	0.00000104	0.00000077
		-5.34255316	-0.00376027	0.00204408	0.00379996	0.00000103	0.00000076
		-5.34519162	-0.00375703	0.00204287	0.00379660	0.00000103	0.00000076
		-5.34782400	-0.00375380	0.00204166	0.00379325	0.00000102	0.00000075
		-5.35045032	-0.00375058	0.00204045	0.00378991	0.00000101	0.00000075
		-5.35307057	-0.00374736	0.00203925	0.00378657	0.00000100	0.00000075
		-5.35568477	-0.00374414	0.00203804	0.00378323	0.00000099	0.00000074
		-5.35829292	-0.00374093	0.00203684	0.00377990	0.00000098	0.00000074
		-5.36089503	-0.00373773	0.00203564	0.00377658	0.00000098	0.00000074
		-5.36349110	-0.00373453	0.00203444	0.00377326	0.00000097	0.00000073
		-5.36608114	-0.00373133	0.00203324	0.00376994	0.00000096	0.00000073
		-5.36866517	-0.00372814	0.00203205	0.00376664	0.00000095	0.00000073
		-5.37124318	-0.00372495	0.00203085	0.00376333	0.00000094	0.00000072
		-5.37381518	-0.00372177	0.00202966	0.00376004	0.00000094	0.00000072
		-5.37638118	-0.00371859	0.00202847	0.00375674	0.00000093	0.00000071
		-5.37894119	-0.00371542	0.00202728	0.00375346	0.00000092	0.00000071
		-5.38149521	-0.00371225	0.00202609	0.00375018	0.00000091	0.00000071
		-5.38404325	-0.00370909	0.00202490	0.00374690	0.00000090	0.00000070
		-5.38658532	-0.00370593	0.00202371	0.00374363	0.00000090	0.00000070
		-5.38912142	-0.00370278	0.00202253	0.00374036	0.00000089	0.00000070
		-5.39165157	-0.00369963	0.00202134	0.00373710	0.00000088	0.00000069
		-5.39417576	-0.00369648	0.00202016	0.00373385	0.00000087	0.00000069
		-5.39669401	-0.00369334	0.00201898	0.00373059	0.00000087	0.00000069
		-5.39920633	-0.00369021	0.00201780	0.00372735	0.00000086	0.00000068
		-5.40171272	-0.00368707	0.00201662	0.00372411	0.00000085	0.00000068
		-5.40421318	-0.00368395	0.00201544	0.00372087	0.00000084	0.00000068
		-5.40670773	-0.00368082	0.00201427	0.00371764	0.00000083	0.00000067
		-5.40919637	-0.00367771	0.00201309	0.00371442	0.00000083	0.00000067
		-5.41167912	-0.00367459	0.00201192	0.00371119	0.00000082	0.00000067
		-5.41415597	-0.00367148	0.00201075	0.00370798	0.00000081	0.00000066
		-5.41662693	-0.00366838	0.00200957	0.00370477	0.00000080	0.00000066
		-5.41909202	-0.00366528	0.00200840	0.00370156	0.00000080	0.00000066
		-5.42155124	-0.00366218	0.00200724	0.00369836	0.00000079	0.00000065
		-5.42400460	-0.00365909	0.00200607	0.00369516	0.00000078	0.00000065
		-5.42645210	-0.00365600	0.00200490	0.00369197	0.00000077	0.00000065
		-5.42889376	-0.00365292	0.00200374	0.00368879	0.00000077	0.00000064
		-5.43132958	-0.00364984	0.00200257	0.00368560	0.00000076	0.00000064
		-5.43375956	-0.00364677	0.00200141	0.00368243	0.00000075	0.00000064
		-5.43618372	-0.00364370	0.00200024	0.00367926	0.00000075	0.00000063
		-5.43860207	-0.00364063	0.00199908	0.00367609	0.00000074	0.00000063
		-5.44101460	-0.00363757	0.00199792	0.00367293	0.00000073	0.00000063
		-5.44342134	-0.00363451	0.00199676	0.00366977	0.00000072	0.00000063
		-5.44582228	-0.00363146	0.00199561	0.00366661	0.00000072	0.00000062
		-5.44821744	-0.00362841	0.00199445	0.00366346	0.00000071	0.00000062
		-5.45060682	-0.00362536	0.00199329	0.00366032	0.00000070	0.00000062
		-5.45299043	-0.00362232	0.00199214	0.00365718	0.00000070	0.00000061
		-5.45536828	-0.00361929	0.00199098	0.00365405	0.00000069	0.00000061
		-5.45774037	-0.00361625	0.00198983	0.00365092	0.00000068	0.00000061
		-5.46010672	-0.00361323	0.00198868	0.00364779	0.00000067	0.00000060
		-5.46246734	-0.00361020	0.00198753	0.00364467	0.00000067	0.00000060
		-5.46482222	-0.00360718	0.00198638	0.00364155	0.00000066	0.00000060
		-5.46717138	-0.00360417	0.00198523	0.00363844	0.00000065	0.00000060
		-5.46951483	-0.00360115	0.00198408	0.00363533	0.00000065	0.00000059
		-5.47185257	-0.00359814	0.00198293	0.00363223	0.00000064	0.00000059
		-5.47418461	-0.00359514	0.00198178	0.00362913	0.00000063	0.00000059
		-5.47651097	-0.00359214	0.00198064	0.00362604	0.00000063	0.00000058
		-5.47883164	-0.00358914	0.00197949	0.00362295	0.00000062	0.00000058
		-5.48114664	-0.00358615	0.00197835	0.00361986	0.00000061	0.00000058
		-5.48345598	-0.00358316	0.00197721	0.00361678	0.00000061	0.00000058
		-5.48575966	-0.00358018	0.00197606	0.00361370	0.00000060	0.00000057
		-5.48805769	-0.00357720	0.00197492	0.00361063	0.00000059	0.00000057
		-5.49035007	-0.00357422	0.00197378	0.00360756	0.00000059	0.00000057
		-5.49263683	-0.00357125	0.00197264	0.00360450	0.00000058	0.00000056
		-5.49491797	-0.00356828	0.00197150	0.00360144	0.00000057	0.00000056
		-5.49719348	-0.00356532	0.00197036	0.00359839	0.00000057	0.00000056
		-5.49946339	-0.00356236	0.00196923	0.00359534	0.00000056	0.00000056
		-5.50172771	-0.00355940	0.00196809	0.00359229	0.00000056	0.00000055
		-5.50398643	-0.00355645	0.00196695	0.00358925	0.00000055	0.00000055
		-5.50623957	-0.00355350	0.00196582	0.00358621	0.00000054	0.00000055
		-5.50848713	-0.00355055	0.00196468	0.00358318	0.00000054	0.00000054
		-5.51072913	-0.00354761	0.00196355	0.00358015	0.00000053	0.00000054
		-5.51296557	-0.00354467	0.00196242	0.00357712	0.00000052	0.00000054
		-5.51519647	-0.00354174	0.00196129	0.00357410	0.00000052	0.00000054
		-5.51742182	-0.00353881	0.00196016	0.00357108	0.00000051	0.00000053
		-5.51964164	-0.00353588	0.00195902	0.00356807	0.00000050	0.00000053
		-5.52185594	-0.00353296	0.00195789	0.00356506	0.00000050	0.00000053
		-5.52406473	-0.00353004	0.00195677	0.00356206	0.00000049	0.00000053
		-5.52626800	-0.00352712	0.00195564	0.00355906	0.00000049	0.00000052
		-5.52846578	-0.00352421	0.00195451	0.00355606	0.00000048	0.00000052
		-5.53065808	-0.00352130	0.00195338	0.00355307	0.00000047	0.00000052
		-5.53284489	-0.00351840	0.00195226	0.00355008	0.00000047	0.00000052
		-5.53502623	-0.00351550	0.00195113	0.00354710	0.00000046	0.00000051
		-5.53720210	-0.00351260	0.00195001	0.00354412	0.00000046	0.00000051
		-5.53937252	-0.00350971	0.00194888	0.00354114	0.00000045	0.00000051
		-5.54153750	-0.00350682	0.00194776	0.00353817	0.00000044	0.00000051
		-5.54369704	-0.00350393	0.00194663	0.00353520	0.00000044	0.00000050
		-5.54585115	-0.00350105	0.00194551	0.00353224	0.00000043	0.00000050
		-5.54799984	-0.00349817	0.00194439	0.00352928	0.00000043	0.00000050
		-5.55014311	-0.00349529	0.00194327	0.00352632	0.00000042	0.00000050
		-5.55228099	-0.00349242	0.00194215	0.00352337	0.00000041	0.00000049
		-5.55441347	-0.00348955	0.00194103	0.00352042	0.00000041	0.00000049
		-5.55654057	-0.00348669	0.00193991	0.00351748	0.00000040	0.00000049
		-5.55866229	-0.00348383	0.00193879	0.00351454	0.00000040	0.00000049
		-5.56077864	-0.00348097	0.00193767	0.00351160	0.00000039	0.00000048
		-5.56288964	-0.00347811	0.00193655	0.00350867	0.00000039	0.00000048
		-5.56499528	-0.00347526	0.00193544	0.00350574	0.00000038	0.00000048
		-5.56709558	-0.00347242	0.00193432	0.00350282	0.00000037	0.00000048
		-5.56919055	-0.00346957	0.00193321	0.00349990	0.00000037	0.00000047
		-5.57128020	-0.00346673	0.00193209	0.00349698	0.00000036	0.00000047
		-5.57336453	-0.00346390	0.00193098	0.00349407	0.00000036	0.00000047
		-5.57544356	-0.00346106	0.00192986	0.00349116	0.00000035	0.00000047
		-5.57751729	-0.00345823	0.00192875	0.00348825	0.00000035	0.00000047
		-5.57958573	-0.00345541	0.00192764	0.00348535	0.00000034	0.00000046
		-5.58164889	-0.00345259	0.00192652	0.00348245	0.00000033	0.00000046
		-5.58370679	-0.00344977	0.00192541	0.00347956	0.00000033	0.00000046
		-5.58575942	-0.00344695	0.00192430	0.00347667	0.00000032	0.00000046
		-5.58780679	-0.00344414	0.00192319	0.00347378	0.00000032	0.00000045
		-5.58984893	-0.00344133	0.00192208	0.00347090	0.00000031	0.00000045
		-5.59188583	-0.00343852	0.00192097	0.00346802	0.00000031	0.00000045
		-5.59391750	-0.00343572	0.00191986	0.00346515	0.00000030	0.00000045
		-5.59594396	-0.00343292	0.00191875	0.00346228	0.00000030	0.00000045
		-5.59796521	-0.00343013	0.00191764	0.00345941	0.00000029	0.00000044
		-5.59998126	-0.00342733	0.00191653	0.00345654	0.00000029	0.00000044
		-5.60199212	-0.00342455	0.00191543	0.00345368	0.00000028	0.00000044
		-5.60399781	-0.00342176	0.00191432	0.00345083	0.00000028	0.00000044
		-5.60599831	-0.00341898	0.00191321	0.00344797	0.00000027	0.00000043
		-5.60799366	-0.00341620	0.00191211	0.00344513	0.00000026	0.00000043
		-5.60998385	-0.00341342	0.00191100	0.00344228	0.00000026	0.00000043
		-5.61196890	-0.00341065	0.00190990	0.00343944	0.00000025	0.00000043
		-5.61394881	-0.00340788	0.00190879	0.00343660	0.00000025	0.00000043
		-5.61592360	-0.00340512	0.00190769	0.00343376	0.00000024	0.00000042
		-5.61789327	-0.00340235	0.00190659	0.00343093	0.00000024	0.00000042
		-5.61985783	-0.00339960	0.00190548	0.00342811	0.00000023	0.00000042
		-5.62181729	-0.00339684	0.00190438	0.00342528	0.00000023	0.00000042
		-5.62377166	-0.00339409	0.00190328	0.00342246	0.00000022	0.00000042
		-5.62572095	-0.00339134	0.00190218	0.00341964	0.00000022	0.00000041
		-5.62766517	-0.00338859	0.00190108	0.00341683	0.00000021	0.00000041
		-5.62960432	-0.00338585	0.00189998	0.00341402	0.00000021	0.00000041
		-5.63153842	-0.00338311	0.00189888	0.00341122	0.00000020	0.00000041
		-5.63346748	-0.00338037	0.00189778	0.00340841	0.00000020	0.00000041
		-5.63539150	-0.00337764	0.00189668	0.00340561	0.00000019	0.00000040
		-5.63731050	-0.00337491	0.00189558	0.00340282	0.00000019	0.00000040
		-5.63922447	-0.00337218	0.00189448	0.00340003	0.00000018	0.00000040
		-5.64113344	-0.00336946	0.00189338	0.00339724	0.00000018	0.00000040
		-5.64303741	-0.00336674	0.00189229	0.00339445	0.00000017	0.00000040
		-5.64493639	-0.00336402	0.00189119	0.00339167	0.00000017	0.00000039
		-5.64683039	-0.00336131	0.00189009	0.00338889	0.00000016	0.00000039
		-5.64871942	-0.00335860	0.00188899	0.00338612	0.00000016	0.00000039
		-5.65060348	-0.00335589	0.00188790	0.00338335	0.00000015	0.00000039
		-5.65248259	-0.00335319	0.00188680	0.00338058	0.00000015	0.00000039
		-5.65435676	-0.00335049	0.00188571	0.00337781	0.00000014	0.00000038
		-5.65622600	-0.00334779	0.00188461	0.00337505	0.00000014	0.00000038
		-5.65809030	-0.00334509	0.00188352	0.00337229	0.00000013	0.00000038
		-5.65994970	-0.00334240	0.00188243	0.00336954	0.00000013	0.00000038
		-5.66180418	-0.00333971	0.00188133	0.00336679	0.00000013	0.00000038
		-5.66365377	-0.00333703	0.00188024	0.00336404	0.00000012	0.00000038
		-5.66549846	-0.00333434	0.00187915	0.00336130	0.00000012	0.00000037
		-5.66733828	-0.00333166	0.00187805	0.00335856	0.00000011	0.00000037
		-5.66917323	-0.00332899	0.00187696	0.00335582	0.00000011	0.00000037
		-5.67100331	-0.00332631	0.00187587	0.00335309	0.00000010	0.00000037
		-5.67282855	-0.00332364	0.00187478	0.00335035	0.00000010	0.00000037
		-5.67464894	-0.00332098	0.00187369	0.00334763	0.00000009	0.00000036
		-5.67646450	-0.00331831	0.00187260	0.00334490	0.00000009	0.00000036
		-5.67827523	-0.00331565	0.00187151	0.00334218	0.00000008	0.00000036
		-5.68008115	-0.00331299	0.00187042	0.00333947	0.00000008	0.00000036
		-5.68188226	-0.00331034	0.00186933	0.00333675	0.00000007	0.00000036
		-5.68367858	-0.00330769	0.00186824	0.00333404	0.00000007	0.00000036
		-5.68547011	-0.00330504	0.00186715	0.00333133	0.00000007	0.00000035
		-5.68725686	-0.00330239	0.00186606	0.00332863	0.00000006	0.00000035
		-5.68903884	-0.00329975	0.00186498	0.00332593	0.00000006	0.00000035
		-5.69081606	-0.00329711	0.00186389	0.00332323	0.00000005	0.00000035
		-5.69258853	-0.00329447	0.00186280	0.00332054	0.00000005	0.00000035
		-5.69435627	-0.00329184	0.00186172	0.00331785	0.00000004	0.00000035
		-5.69611926	-0.00328921	0.00186063	0.00331516	0.00000004	0.00000034
		-5.69787754	-0.00328658	0.00185954	0.00331247	0.00000003	0.00000034
		-5.69963110	-0.00328396	0.00185846	0.00330979	0.00000003	0.00000034
		-5.70137996	-0.00328133	0.00185737	0.00330711	0.00000003	0.00000034
		-5.70312413	-0.00327872	0.00185629	0.00330444	0.00000002	0.00000034
		-5.70486360	-0.00327610	0.00185520	0.00330177	0.00000002	0.00000034
		-5.70659840	-0.00327349	0.00185412	0.00329910	0.00000001	0.00000033
		-5.70832854	-0.00327088	0.00185304	0.00329643	0.00000001	0.00000033
		-5.71005402	-0.00326827	0.00185195	0.00329377	0.00000000	0.00000033
		-5.71177484	-0.00326567	0.00185087	0.00329111	0.00000000	0.00000033
		-5.71349103	-0.00326306	0.00184979	0.00328846	-0.00000000	0.00000033
		-5.71520259	-0.00326047	0.00184870	0.00328580	-0.00000001	0.00000033
		-5.71690953	-0.00325787	0.00184762	0.00328316	-0.00000001	0.00000032
		-5.71861185	-0.00325528	0.00184654	0.00328051	-0.00000002	0.00000032
		-5.72030958	-0.00325269	0.00184546	0.00327787	-0.00000002	0.00000032
		-5.72200271	-0.00325010	0.00184438	0.00327523	-0.00000002	0.00000032
		-5.72369126	-0.00324752	0.00184330	0.00327259	-0.00000003	0.00000032
		-5.72537523	-0.00324494	0.00184222	0.00326996	-0.00000003	0.00000032
		-5.72705463	-0.00324236	0.00184114	0.00326733	-0.00000004	0.00000032
		-5.72872948	-0.00323979	0.00184006	0.00326470	-0.00000004	0.00000031
		-5.73039979	-0.00323721	0.00183898	0.00326207	-0.00000005	0.00000031
		-5.73206555	-0.00323464	0.00183790	0.00325945	-0.00000005	0.00000031
		-5.73372679	-0.00323208	0.00183682	0.00325683	-0.00000005	0.00000031
		-5.73538351	-0.00322952	0.00183574	0.00325422	-0.00000006	0.00000031
		-5.73703572	-0.00322695	0.00183466	0.00325161	-0.00000006	0.00000031
		-5.73868343	-0.00322440	0.00183359	0.00324900	-0.00000007	0.00000030
		-5.74032664	-0.00322184	0.00183251	0.00324639	-0.00000007	0.00000030
		-5.74196538	-0.00321929	0.00183143	0.00324379	-0.00000007	0.00000030
		-5.74359964	-0.00321674	0.00183036	0.00324119	-0.00000008	0.00000030
		-5.74522943	-0.00321419	0.00182928	0.00323859	-0.00000008	0.00000030
		-5.74685478	-0.00321165	0.00182820	0.00323600	-0.00000009	0.00000030
		-5.74847567	-0.00320911	0.00182713	0.00323341	-0.00000009	0.00000030
		-5.75009213	-0.00320657	0.00182605	0.00323082	-0.00000009	0.00000029
		-5.75170417	-0.00320404	0.00182498	0.00322824	-0.00000010	0.00000029
		-5.75331179	-0.00320151	0.00182390	0.00322566	-0.00000010	0.00000029
		-5.75491499	-0.00319898	0.00182283	0.00322308	-0.00000010	0.00000029
		-5.75651380	-0.00319645	0.00182176	0.00322050	-0.00000011	0.00000029
		-5.75810822	-0.00319393	0.00182068	0.00321793	-0.00000011	0.00000029
		-5.75969826	-0.00319141	0.00181961	0.00321536	-0.00000012	0.00000029
		-5.76128392	-0.00318889	0.00181854	0.00321280	-0.00000012	0.00000028
		-5.76286523	-0.00318637	0.00181746	0.00321023	-0.00000012	0.00000028
		-5.76444218	-0.00318386	0.00181639	0.00320767	-0.00000013	0.00000028
		-5.76601479	-0.00318135	0.00181532	0.00320512	-0.00000013	0.00000028
		-5.76758306	-0.00317884	0.00181425	0.00320256	-0.00000013	0.00000028
		-5.76914700	-0.00317634	0.00181318	0.00320001	-0.00000014	0.00000028
		-5.77070663	-0.00317384	0.00181210	0.00319746	-0.00000014	0.00000028
		-5.77226195	-0.00317134	0.00181103	0.00319492	-0.00000015	0.00000028
		-5.77381298	-0.00316884	0.00180996	0.00319237	-0.00000015	0.00000027
		-5.77535971	-0.00316635	0.00180889	0.00318984	-0.00000015	0.00000027
		-5.77690216	-0.00316386	0.00180782	0.00318730	-0.00000016	0.00000027
		-5.77844035	-0.00316137	0.00180675	0.00318477	-0.00000016	0.00000027
		-5.77997427	-0.00315889	0.00180569	0.00318223	-0.00000016	0.00000027
		-5.78150394	-0.00315640	0.00180462	0.00317971	-0.00000017	0.00000027
		-5.78302936	-0.00315393	0.00180355	0.00317718	-0.00000017	0.00000027
		-5.78455056	-0.00315145	0.00180248	0.00317466	-0.00000017	0.00000026
		-5.78606752	-0.00314897	0.00180141	0.00317214	-0.00000018	0.00000026
		-5.78758027	-0.00314650	0.00180035	0.00316963	-0.00000018	0.00000026
		-5.78908882	-0.00314403	0.00179928	0.00316711	-0.00000019	0.00000026
		-5.79059316	-0.00314157	0.00179821	0.00316460	-0.00000019	0.00000026
		-5.79209332	-0.00313911	0.00179715	0.00316209	-0.00000019	0.00000026
		-5.79358930	-0.00313665	0.00179608	0.00315959	-0.00000020	0.00000026
		-5.79508110	-0.00313419	0.00179501	0.00315709	-0.00000020	0.00000026
		-5.79656875	-0.00313173	0.00179395	0.00315459	-0.00000020	0.00000025
		-5.79805224	-0.00312928	0.00179288	0.00315209	-0.00000021	0.00000025
		-5.79953159	-0.00312683	0.00179182	0.00314960	-0.00000021	0.00000025
		-5.80100681	-0.00312438	0.00179075	0.00314711	-0.00000021	0.00000025
		-5.80247790	-0.00312194	0.00178969	0.00314462	-0.00000022	0.00000025
		-5.80394487	-0.00311950	0.00178863	0.00314214	-0.00000022	0.00000025
		-5.80540774	-0.00311706	0.00178756	0.00313966	-0.00000022	0.00000025
		-5.80686651	-0.00311462	0.00178650	0.00313718	-0.00000023	0.00000025
		-5.80832119	-0.00311219	0.00178544	0.00313470	-0.00000023	0.00000024
		-5.80977179	-0.00310976	0.00178438	0.00313223	-0.00000023	0.00000024
		-5.81121832	-0.00310733	0.00178331	0.00312976	-0.00000024	0.00000024
		-5.81266079	-0.00310490	0.00178225	0.00312729	-0.00000024	0.00000024
		-5.81409921	-0.00310248	0.00178119	0.00312483	-0.00000024	0.00000024
		-5.81553358	-0.00310006	0.00178013	0.00312237	-0.00000025	0.00000024
		-5.81696392	-0.00309764	0.00177907	0.00311991	-0.00000025	0.00000024
		-5.81839023	-0.00309523	0.00177801	0.00311745	-0.00000025	0.00000024
		-5.81981252	-0.00309282	0.00177695	0.00311500	-0.00000026	0.00000024
		-5.82123081	-0.00309041	0.00177589	0.00311255	-0.00000026	0.00000023
		-5.82264509	-0.00308800	0.00177483	0.00311010	-0.00000026	0.00000023
		-5.82405539	-0.00308560	0.00177377	0.00310766	-0.00000027	0.00000023
		-5.82546171	-0.00308319	0.00177271	0.00310521	-0.00000027	0.00000023
		-5.82686405	-0.00308079	0.00177165	0.00310277	-0.00000027	0.00000023
		-5.82826243	-0.00307840	0.00177059	0.00310034	-0.00000028	0.00000023
		-5.82965686	-0.00307600	0.00176954	0.00309790	-0.00000028	0.00000023
		-5.83104734	-0.00307361	0.00176848	0.00309547	-0.00000028	0.00000023
		-5.83243389	-0.00307122	0.00176742	0.00309305	-0.00000029	0.00000023
		-5.83381650	-0.00306884	0.00176637	0.00309062	-0.00000029	0.00000022
		-5.83519520	-0.00306645	0.00176531	0.00308820	-0.00000029	0.00000022
		-5.83656999	-0.00306407	0.00176425	0.00308578	-0.00000030	0.00000022
		-5.83794088	-0.00306169	0.00176320	0.00308336	-0.00000030	0.00000022
		-5.83930788	-0.00305932	0.00176214	0.00308095	-0.00000030	0.00000022
		-5.84067099	-0.00305695	0.00176109	0.00307853	-0.00000031	0.00000022
		-5.84203023	-0.00305458	0.00176003	0.00307612	-0.00000031	0.00000022
		-5.84338560	-0.00305221	0.00175898	0.00307372	-0.00000031	0.00000022
		-5.84473712	-0.00304984	0.00175793	0.00307132	-0.00000031	0.00000022
		-5.84608479	-0.00304748	0.00175687	0.00306891	-0.00000032	0.00000022
		-5.84742862	-0.00304512	0.00175582	0.00306652	-0.00000032	0.00000021
		-5.84876862	-0.00304276	0.00175477	0.00306412	-0.00000032	0.00000021
		-5.85010480	-0.00304041	0.00175371	0.00306173	-0.00000033	0.00000021
		-5.85143717	-0.00303805	0.00175266	0.00305934	-0.00000033	0.00000021
		-5.85276573	-0.00303570	0.00175161	0.00305695	-0.00000033	0.00000021
		-5.85409050	-0.00303336	0.00175056	0.00305457	-0.00000034	0.00000021
		-5.85541148	-0.00303101	0.00174951	0.00305218	-0.00000034	0.00000021
		-5.85672868	-0.00302867	0.00174846	0.00304981	-0.00000034	0.00000021
		-5.85804212	-0.00302633	0.00174741	0.00304743	-0.00000034	0.00000021
		-5.85935180	-0.00302399	0.00174636	0.00304506	-0.00000035	0.00000020
		-5.86065772	-0.00302166	0.00174531	0.00304268	-0.00000035	0.00000020
		-5.86195991	-0.00301932	0.00174426	0.00304032	-0.00000035	0.00000020
		-5.86325835	-0.00301699	0.00174321	0.00303795	-0.00000036	0.00000020
		-5.86455308	-0.00301467	0.00174216	0.00303559	-0.00000036	0.00000020
		-5.86584409	-0.00301234	0.00174111	0.00303323	-0.00000036	0.00000020
		-5.86713139	-0.00301002	0.00174006	0.00303087	-0.00000037	0.00000020
		-5.86841499	-0.00300770	0.00173902	0.00302851	-0.00000037	0.00000020
		-5.86969490	-0.00300538	0.00173797	0.00302616	-0.00000037	0.00000020
		-5.87097113	-0.00300307	0.00173692	0.00302381	-0.00000037	0.00000020
		-5.87224368	-0.00300076	0.00173588	0.00302146	-0.00000038	0.00000020
		-5.87351257	-0.00299845	0.00173483	0.00301912	-0.00000038	0.00000019
		-5.87477781	-0.00299614	0.00173378	0.00301678	-0.00000038	0.00000019
		-5.87603940	-0.00299383	0.00173274	0.00301444	-0.00000039	0.00000019
		-5.87729735	-0.00299153	0.00173169	0.00301210	-0.00000039	0.00000019
		-5.87855167	-0.00298923	0.00173065	0.00300977	-0.00000039	0.00000019
		-5.87980237	-0.00298694	0.00172961	0.00300744	-0.00000039	0.00000019
		-5.88104946	-0.00298464	0.00172856	0.00300511	-0.00000040	0.00000019
		-5.88229294	-0.00298235	0.00172752	0.00300278	-0.00000040	0.00000019
		-5.88353283	-0.00298006	0.00172648	0.00300046	-0.00000040	0.00000019
		-5.88476913	-0.00297777	0.00172543	0.00299814	-0.00000040	0.00000019
		-5.88600185	-0.00297549	0.00172439	0.00299582	-0.00000041	0.00000019
		-5.88723100	-0.00297320	0.00172335	0.00299350	-0.00000041	0.00000018
		-5.88845659	-0.00297093	0.00172231	0.00299119	-0.00000041	0.00000018
		-5.88967862	-0.00296865	0.00172127	0.00298888	-0.00000042	0.00000018
		-5.89089712	-0.00296637	0.00172022	0.00298657	-0.00000042	0.00000018
		-5.89211207	-0.00296410	0.00171918	0.00298427	-0.00000042	0.00000018
		-5.89332350	-0.00296183	0.00171814	0.00298196	-0.00000042	0.00000018
		-5.89453141	-0.00295956	0.00171710	0.00297966	-0.00000043	0.00000018
		-5.89573581	-0.00295730	0.00171607	0.00297737	-0.00000043	0.00000018
		-5.89693670	-0.00295504	0.00171503	0.00297507	-0.00000043	0.00000018
		-5.89813411	-0.00295278	0.00171399	0.00297278	-0.00000043	0.00000018
		-5.89932802	-0.00295052	0.00171295	0.00297049	-0.00000044	0.00000018
		-5.90051846	-0.00294826	0.00171191	0.00296820	-0.00000044	0.00000017
		-5.90170544	-0.00294601	0.00171088	0.00296592	-0.00000044	0.00000017
		-5.90288895	-0.00294376	0.00170984	0.00296363	-0.00000045	0.00000017
		-5.90406901	-0.00294151	0.00170880	0.00296136	-0.00000045	0.00000017
		-5.90524562	-0.00293927	0.00170777	0.00295908	-0.00000045	0.00000017
		-5.90641881	-0.00293702	0.00170673	0.00295680	-0.00000045	0.00000017
		-5.90758856	-0.00293478	0.00170569	0.00295453	-0.00000046	0.00000017
		-5.90875490	-0.00293254	0.00170466	0.00295226	-0.00000046	0.00000017
		-5.90991782	-0.00293031	0.00170363	0.00295000	-0.00000046	0.00000017
		-5.91107735	-0.00292807	0.00170259	0.00294773	-0.00000046	0.00000017
		-5.91223348	-0.00292584	0.00170156	0.00294547	-0.00000047	0.00000017
		-5.91338622	-0.00292362	0.00170052	0.00294321	-0.00000047	0.00000017
		-5.91453559	-0.00292139	0.00169949	0.00294095	-0.00000047	0.00000016
		-5.91568159	-0.00291917	0.00169846	0.00293870	-0.00000047	0.00000016
		-5.91682423	-0.00291694	0.00169743	0.00293645	-0.00000048	0.00000016
		-5.91796352	-0.00291472	0.00169639	0.00293420	-0.00000048	0.00000016
		-5.91909946	-0.00291251	0.00169536	0.00293195	-0.00000048	0.00000016
		-5.92023207	-0.00291029	0.00169433	0.00292971	-0.00000048	0.00000016
		-5.92136135	-0.00290808	0.00169330	0.00292746	-0.00000049	0.00000016
		-5.92248731	-0.00290587	0.00169227	0.00292523	-0.00000049	0.00000016
		-5.92360996	-0.00290367	0.00169124	0.00292299	-0.00000049	0.00000016
		-5.92472930	-0.00290146	0.00169021	0.00292075	-0.00000049	0.00000016
		-5.92584536	-0.00289926	0.00168918	0.00291852	-0.00000050	0.00000016
		-5.92695812	-0.00289706	0.00168816	0.00291629	-0.00000050	0.00000016
		-5.92806761	-0.00289486	0.00168713	0.00291407	-0.00000050	0.00000016
		-5.92917382	-0.00289267	0.00168610	0.00291184	-0.00000050	0.00000015
		-5.93027677	-0.00289047	0.00168507	0.00290962	-0.00000051	0.00000015
		-5.93137647	-0.00288828	0.00168405	0.00290740	-0.00000051	0.00000015
		-5.93247293	-0.00288610	0.00168302	0.00290518	-0.00000051	0.00000015
		-5.93356614	-0.00288391	0.00168199	0.00290297	-0.00000051	0.00000015
		-5.93465613	-0.00288173	0.00168097	0.00290076	-0.00000051	0.00000015
		-5.93574289	-0.00287955	0.00167994	0.00289855	-0.00000052	0.00000015
		-5.93682644	-0.00287737	0.00167892	0.00289634	-0.00000052	0.00000015
		-5.93790678	-0.00287519	0.00167789	0.00289414	-0.00000052	0.00000015
		-5.93898392	-0.00287302	0.00167687	0.00289193	-0.00000052	0.00000015
		-5.94005788	-0.00287085	0.00167585	0.00288973	-0.00000053	0.00000015
		-5.94112865	-0.00286868	0.00167482	0.00288754	-0.00000053	0.00000015
		-5.94219625	-0.00286651	0.00167380	0.00288534	-0.00000053	0.00000015
		-5.94326068	-0.00286435	0.00167278	0.00288315	-0.00000053	0.00000015
		-5.94432196	-0.00286218	0.00167176	0.00288096	-0.00000054	0.00000014
		-5.94538008	-0.00286002	0.00167074	0.00287877	-0.00000054	0.00000014
		-5.94643506	-0.00285787	0.00166971	0.00287659	-0.00000054	0.00000014
		-5.94748691	-0.00285571	0.00166869	0.00287441	-0.00000054	0.00000014
		-5.94853563	-0.00285356	0.00166767	0.00287222	-0.00000055	0.00000014
		-5.94958124	-0.00285141	0.00166666	0.00287005	-0.00000055	0.00000014
		-5.95062373	-0.00284926	0.00166564	0.00286787	-0.00000055	0.00000014
		-5.95166312	-0.00284711	0.00166462	0.00286570	-0.00000055	0.00000014
		-5.95269941	-0.00284497	0.00166360	0.00286353	-0.00000055	0.00000014
		-5.95373262	-0.00284283	0.00166258	0.00286136	-0.00000056	0.00000014
		-5.95476275	-0.00284069	0.00166156	0.00285919	-0.00000056	0.00000014
		-5.95578980	-0.00283855	0.00166055	0.00285703	-0.00000056	0.00000014
		-5.95681380	-0.00283642	0.00165953	0.00285487	-0.00000056	0.00000014
		-5.95783474	-0.00283429	0.00165852	0.00285271	-0.00000057	0.00000014
		-5.95885262	-0.00283216	0.00165750	0.00285056	-0.00000057	0.00000014
		-5.95986747	-0.00283003	0.00165648	0.00284840	-0.00000057	0.00000013
		-5.96087929	-0.00282791	0.00165547	0.00284625	-0.00000057	0.00000013
		-5.96188808	-0.00282578	0.00165446	0.00284410	-0.00000057	0.00000013
		-5.96289385	-0.00282366	0.00165344	0.00284196	-0.00000058	0.00000013
		-5.96389662	-0.00282154	0.00165243	0.00283981	-0.00000058	0.00000013
		-5.96489638	-0.00281943	0.00165142	0.00283767	-0.00000058	0.00000013
		-5.96589315	-0.00281732	0.00165040	0.00283553	-0.00000058	0.00000013
		-5.96688693	-0.00281520	0.00164939	0.00283339	-0.00000058	0.00000013
		-5.96787774	-0.00281310	0.00164838	0.00283126	-0.00000059	0.00000013
		-5.96886557	-0.00281099	0.00164737	0.00282913	-0.00000059	0.00000013
		-5.96985044	-0.00280888	0.00164636	0.00282700	-0.00000059	0.00000013
		-5.97083235	-0.00280678	0.00164535	0.00282487	-0.00000059	0.00000013
		-5.97181132	-0.00280468	0.00164434	0.00282274	-0.00000060	0.00000013
		-5.97278734	-0.00280258	0.00164333	0.00282062	-0.00000060	0.00000013
		-5.97376044	-0.00280049	0.00164232	0.00281850	-0.00000060	0.00000013
		-5.97473060	-0.00279840	0.00164131	0.00281638	-0.00000060	0.00000013
		-5.97569785	-0.00279631	0.00164031	0.00281427	-0.00000060	0.00000012
		-5.97666219	-0.00279422	0.00163930	0.00281215	-0.00000061	0.00000012
		-5.97762362	-0.00279213	0.00163829	0.00281004	-0.00000061	0.00000012
		-5.97858216	-0.00279005	0.00163729	0.00280793	-0.00000061	0.00000012
		-5.97953782	-0.00278796	0.00163628	0.00280583	-0.00000061	0.00000012
		-5.98049059	-0.00278589	0.00163527	0.00280372	-0.00000061	0.00000012
		-5.98144049	-0.00278381	0.00163427	0.00280162	-0.00000062	0.00000012
		-5.98238752	-0.00278173	0.00163326	0.00279952	-0.00000062	0.00000012
		-5.98333170	-0.00277966	0.00163226	0.00279742	-0.00000062	0.00000012
		-5.98427302	-0.00277759	0.00163126	0.00279533	-0.00000062	0.00000012
		-5.98521150	-0.00277552	0.00163025	0.00279324	-0.00000062	0.00000012
		-5.98614715	-0.00277346	0.00162925	0.00279115	-0.00000063	0.00000012
		-5.98707997	-0.00277139	0.00162825	0.00278906	-0.00000063	0.00000012
		-5.98800996	-0.00276933	0.00162725	0.00278697	-0.00000063	0.00000012
		-5.98893715	-0.00276727	0.00162625	0.00278489	-0.00000063	0.00000012
		-5.98986153	-0.00276521	0.00162525	0.00278281	-0.00000063	0.00000012
		-5.99078310	-0.00276316	0.00162425	0.00278073	-0.00000064	0.00000012
		-5.99170189	-0.00276111	0.00162325	0.00277865	-0.00000064	0.00000012
		-5.99261789	-0.00275906	0.00162225	0.00277658	-0.00000064	0.00000011
		-5.99353112	-0.00275701	0.00162125	0.00277451	-0.00000064	0.00000011
		-5.99444158	-0.00275496	0.00162025	0.00277244	-0.00000064	0.00000011
		-5.99534927	-0.00275292	0.00161925	0.00277037	-0.00000065	0.00000011
		-5.99625421	-0.00275088	0.00161826	0.00276831	-0.00000065	0.00000011
		-5.99715640	-0.00274884	0.00161726	0.00276624	-0.00000065	0.00000011
		-5.99805585	-0.00274680	0.00161626	0.00276418	-0.00000065	0.00000011
		-5.99895257	-0.00274476	0.00161527	0.00276213	-0.00000065	0.00000011
		-5.99984656	-0.00274273	0.00161427	0.00276007	-0.00000066	0.00000011
		-6.00073784	-0.00274070	0.00161328	0.00275802	-0.00000066	0.00000011
		-6.00162640	-0.00273867	0.00161228	0.00275597	-0.00000066	0.00000011
		-6.00251225	-0.00273665	0.00161129	0.00275392	-0.00000066	0.00000011
		-6.00339541	-0.00273462	0.00161030	0.00275187	-0.00000066	0.00000011
		-6.00427588	-0.00273260	0.00160931	0.00274983	-0.00000066	0.00000011
		-6.00515367	-0.00273058	0.00160831	0.00274778	-0.00000067	0.00000011
		-6.00602878	-0.00272856	0.00160732	0.00274574	-0.00000067	0.00000011
		-6.00690122	-0.00272655	0.00160633	0.00274371	-0.00000067	0.00000011
		-6.00777100	-0.00272454	0.00160534	0.00274167	-0.00000067	0.00000011
		-6.00863812	-0.00272252	0.00160435	0.00273964	-0.00000067	0.00000011
		-6.00950260	-0.00272052	0.00160336	0.00273761	-0.00000068	0.00000010
		-6.01036444	-0.00271851	0.00160237	0.00273558	-0.00000068	0.00000010
		-6.01122364	-0.00271651	0.00160138	0.00273355	-0.00000068	0.00000010
		-6.01208022	-0.00271450	0.00160039	0.00273153	-0.00000068	0.00000010
		-6.01293417	-0.00271250	0.00159941	0.00272951	-0.00000068	0.00000010
		-6.01378552	-0.00271051	0.00159842	0.00272749	-0.00000068	0.00000010
		-6.01463426	-0.00270851	0.00159743	0.00272547	-0.00000069	0.00000010
		-6.01548040	-0.00270652	0.00159645	0.00272346	-0.00000069	0.00000010
		-6.01632394	-0.00270453	0.00159546	0.00272144	-0.00000069	0.00000010
		-6.01716491	-0.00270254	0.00159448	0.00271943	-0.00000069	0.00000010
		-6.01800329	-0.00270055	0.00159349	0.00271742	-0.00000069	0.00000010
		-6.01883911	-0.00269856	0.00159251	0.00271542	-0.00000070	0.00000010
		-6.01967236	-0.00269658	0.00159153	0.00271341	-0.00000070	0.00000010
		-6.02050306	-0.00269460	0.00159054	0.00271141	-0.00000070	0.00000010
		-6.02133120	-0.00269262	0.00158956	0.00270941	-0.00000070	0.00000010
		-6.02215680	-0.00269065	0.00158858	0.00270741	-0.00000070	0.00000010
		-6.02297987	-0.00268867	0.00158760	0.00270542	-0.00000070	0.00000010
		-6.02380040	-0.00268670	0.00158662	0.00270343	-0.00000071	0.00000010
		-6.02461841	-0.00268473	0.00158564	0.00270144	-0.00000071	0.00000010
		-6.02543391	-0.00268276	0.00158466	0.00269945	-0.00000071	0.00000010
		-6.02624690	-0.00268080	0.00158368	0.00269746	-0.00000071	0.00000010
		-6.02705738	-0.00267883	0.00158270	0.00269548	-0.00000071	0.00000009
		-6.02786537	-0.00267687	0.00158172	0.00269350	-0.00000071	0.00000009
		-6.02867087	-0.00267491	0.00158074	0.00269152	-0.00000072	0.00000009
		-6.02947389	-0.00267296	0.00157977	0.00268954	-0.00000072	0.00000009
		-6.03027443	-0.00267100	0.00157879	0.00268756	-0.00000072	0.00000009
		-6.03107251	-0.00266905	0.00157781	0.00268559	-0.00000072	0.00000009
		-6.03186812	-0.00266710	0.00157684	0.00268362	-0.00000072	0.00000009
		-6.03266127	-0.00266515	0.00157586	0.00268165	-0.00000072	0.00000009
		-6.03345198	-0.00266320	0.00157489	0.00267968	-0.00000073	0.00000009
		-6.03424025	-0.00266126	0.00157391	0.00267772	-0.00000073	0.00000009
		-6.03502608	-0.00265932	0.00157294	0.00267576	-0.00000073	0.00000009
		-6.03580948	-0.00265738	0.00157197	0.00267380	-0.00000073	0.00000009
		-6.03659046	-0.00265544	0.00157100	0.00267184	-0.00000073	0.00000009
		-6.03736903	-0.00265350	0.00157002	0.00266988	-0.00000073	0.00000009
		-6.03814518	-0.00265157	0.00156905	0.00266793	-0.00000074	0.00000009
		-6.03891894	-0.00264964	0.00156808	0.00266598	-0.00000074	0.00000009
		-6.03969029	-0.00264771	0.00156711	0.00266403	-0.00000074	0.00000009
		-6.04045926	-0.00264578	0.00156614	0.00266208	-0.00000074	0.00000009
		-6.04122585	-0.00264386	0.00156517	0.00266014	-0.00000074	0.00000009
		-6.04199006	-0.00264193	0.00156421	0.00265819	-0.00000074	0.00000009
		-6.04275190	-0.00264001	0.00156324	0.00265625	-0.00000075	0.00000009
		-6.04351138	-0.00263809	0.00156227	0.00265431	-0.00000075	0.00000009
		-6.04426850	-0.00263618	0.00156130	0.00265238	-0.00000075	0.00000009
		-6.04502327	-0.00263426	0.00156034	0.00265044	-0.00000075	0.00000008
		-6.04577570	-0.00263235	0.00155937	0.00264851	-0.00000075	0.00000008
		-6.04652579	-0.00263044	0.00155841	0.00264658	-0.00000075	0.00000008
		-6.04727354	-0.00262853	0.00155744	0.00264465	-0.00000075	0.00000008
		-6.04801898	-0.00262662	0.00155648	0.00264273	-0.00000076	0.00000008
		-6.04876210	-0.00262472	0.00155552	0.00264080	-0.00000076	0.00000008
		-6.04950290	-0.00262282	0.00155455	0.00263888	-0.00000076	0.00000008
		-6.05024140	-0.00262092	0.00155359	0.00263696	-0.00000076	0.00000008
		-6.05097760	-0.00261902	0.00155263	0.00263505	-0.00000076	0.00000008
		-6.05171151	-0.00261712	0.00155167	0.00263313	-0.00000076	0.00000008
		-6.05244313	-0.00261523	0.00155071	0.00263122	-0.00000077	0.00000008
		-6.05317248	-0.00261334	0.00154975	0.00262931	-0.00000077	0.00000008
		-6.05389955	-0.00261145	0.00154879	0.00262740	-0.00000077	0.00000008
		-6.05462435	-0.00260956	0.00154783	0.00262549	-0.00000077	0.00000008
		-6.05534689	-0.00260767	0.00154687	0.00262359	-0.00000077	0.00000008
		-6.05606718	-0.00260579	0.00154591	0.00262169	-0.00000077	0.00000008
		-6.05678521	-0.00260391	0.00154495	0.00261979	-0.00000077	0.00000008
		-6.05750101	-0.00260203	0.00154400	0.00261789	-0.00000078	0.00000008
		-6.05821457	-0.00260015	0.00154304	0.00261599	-0.00000078	0.00000008
		-6.05892590	-0.00259827	0.00154209	0.00261410	-0.00000078	0.00000008
		-6.05963501	-0.00259640	0.00154113	0.00261221	-0.00000078	0.00000008
		-6.06034190	-0.00259453	0.00154018	0.00261032	-0.00000078	0.00000008
		-6.06104657	-0.00259266	0.00153922	0.00260843	-0.00000078	0.00000008
		-6.06174905	-0.00259079	0.00153827	0.00260654	-0.00000078	0.00000008
		-6.06244932	-0.00258893	0.00153732	0.00260466	-0.00000079	0.00000008
		-6.06314740	-0.00258706	0.00153637	0.00260278	-0.00000079	0.00000008
		-6.06384330	-0.00258520	0.00153541	0.00260090	-0.00000079	0.00000008
		-6.06453701	-0.00258334	0.00153446	0.00259902	-0.00000079	0.00000007
		-6.06522855	-0.00258148	0.00153351	0.00259715	-0.00000079	0.00000007
		-6.06591793	-0.00257963	0.00153256	0.00259527	-0.00000079	0.00000007
		-6.06660514	-0.00257778	0.00153161	0.00259340	-0.00000079	0.00000007
		-6.06729019	-0.00257592	0.00153067	0.00259153	-0.00000080	0.00000007
		-6.06797309	-0.00257408	0.00152972	0.00258967	-0.00000080	0.00000007
		-6.06865385	-0.00257223	0.00152877	0.00258780	-0.00000080	0.00000007
		-6.06933248	-0.00257038	0.00152782	0.00258594	-0.00000080	0.00000007
		-6.07000896	-0.00256854	0.00152688	0.00258408	-0.00000080	0.00000007
		-6.07068333	-0.00256670	0.00152593	0.00258222	-0.00000080	0.00000007
		-6.07135557	-0.00256486	0.00152499	0.00258036	-0.00000080	0.00000007
		-6.07202570	-0.00256302	0.00152404	0.00257851	-0.00000081	0.00000007
		-6.07269372	-0.00256119	0.00152310	0.00257666	-0.00000081	0.00000007
		-6.07335964	-0.00255935	0.00152215	0.00257480	-0.00000081	0.00000007
		-6.07402346	-0.00255752	0.00152121	0.00257296	-0.00000081	0.00000007
		-6.07468520	-0.00255569	0.00152027	0.00257111	-0.00000081	0.00000007
		-6.07534484	-0.00255387	0.00151933	0.00256927	-0.00000081	0.00000007
		-6.07600241	-0.00255204	0.00151839	0.00256742	-0.00000081	0.00000007
		-6.07665791	-0.00255022	0.00151745	0.00256558	-0.00000082	0.00000007
		-6.07731134	-0.00254840	0.00151651	0.00256374	-0.00000082	0.00000007
		-6.07796271	-0.00254658	0.00151557	0.00256191	-0.00000082	0.00000007
		-6.07861202	-0.00254476	0.00151463	0.00256007	-0.00000082	0.00000007
		-6.07925928	-0.00254294	0.00151369	0.00255824	-0.00000082	0.00000007
		-6.07990450	-0.00254113	0.00151275	0.00255641	-0.00000082	0.00000007
		-6.08054769	-0.00253932	0.00151182	0.00255458	-0.00000082	0.00000007
		-6.08118884	-0.00253751	0.00151088	0.00255276	-0.00000082	0.00000007
		-6.08182796	-0.00253570	0.00150994	0.00255093	-0.00000083	0.00000007
		-6.08246507	-0.00253390	0.00150901	0.00254911	-0.00000083	0.00000007
		-6.08310016	-0.00253210	0.00150807	0.00254729	-0.00000083	0.00000007
		-6.08373324	-0.00253029	0.00150714	0.00254547	-0.00000083	0.00000007
		-6.08436431	-0.00252849	0.00150621	0.00254366	-0.00000083	0.00000006
		-6.08499339	-0.00252670	0.00150527	0.00254184	-0.00000083	0.00000006
		-6.08562048	-0.00252490	0.00150434	0.00254003	-0.00000083	0.00000006
		-6.08624558	-0.00252311	0.00150341	0.00253822	-0.00000084	0.00000006
		-6.08686870	-0.00252132	0.00150248	0.00253641	-0.00000084	0.00000006
		-6.08748985	-0.00251953	0.00150155	0.00253461	-0.00000084	0.00000006
		-6.08810902	-0.00251774	0.00150062	0.00253280	-0.00000084	0.00000006
		-6.08872624	-0.00251595	0.00149969	0.00253100	-0.00000084	0.00000006
		-6.08934149	-0.00251417	0.00149876	0.00252920	-0.00000084	0.00000006
		-6.08995479	-0.00251239	0.00149784	0.00252740	-0.00000084	0.00000006
		-6.09056615	-0.00251061	0.00149691	0.00252561	-0.00000084	0.00000006
		-6.09117556	-0.00250883	0.00149598	0.00252381	-0.00000085	0.00000006
		-6.09178304	-0.00250705	0.00149506	0.00252202	-0.00000085	0.00000006
		-6.09238859	-0.00250528	0.00149413	0.00252023	-0.00000085	0.00000006
		-6.09299221	-0.00250351	0.00149320	0.00251844	-0.00000085	0.00000006
		-6.09359392	-0.00250174	0.00149228	0.00251666	-0.00000085	0.00000006
		-6.09419371	-0.00249997	0.00149136	0.00251487	-0.00000085	0.00000006
		-6.09479159	-0.00249820	0.00149043	0.00251309	-0.00000085	0.00000006
		-6.09538757	-0.00249644	0.00148951	0.00251131	-0.00000085	0.00000006
		-6.09598165	-0.00249468	0.00148859	0.00250953	-0.00000086	0.00000006
		-6.09657385	-0.00249292	0.00148767	0.00250776	-0.00000086	0.00000006
		-6.09716415	-0.00249116	0.00148675	0.00250598	-0.00000086	0.00000006
		-6.09775257	-0.00248940	0.00148583	0.00250421	-0.00000086	0.00000006
		-6.09833912	-0.00248765	0.00148491	0.00250244	-0.00000086	0.00000006
		-6.09892380	-0.00248589	0.00148399	0.00250067	-0.00000086	0.00000006
		-6.09950662	-0.00248414	0.00148307	0.00249891	-0.00000086	0.00000006
		-6.10008757	-0.00248239	0.00148215	0.00249714	-0.00000086	0.00000006
		-6.10066667	-0.00248065	0.00148124	0.00249538	-0.00000086	0.00000006
		-6.10124393	-0.00247890	0.00148032	0.00249362	-0.00000087	0.00000006
		-6.10181934	-0.00247716	0.00147941	0.00249186	-0.00000087	0.00000006
		-6.10239291	-0.00247542	0.00147849	0.00249010	-0.00000087	0.00000006
		-6.10296465	-0.00247368	0.00147758	0.00248835	-0.00000087	0.00000006
		-6.10353456	-0.00247194	0.00147666	0.00248660	-0.00000087	0.00000006
		-6.10410265	-0.00247020	0.00147575	0.00248484	-0.00000087	0.00000006
		-6.10466892	-0.00246847	0.00147484	0.00248310	-0.00000087	0.00000006
		-6.10523338	-0.00246674	0.00147393	0.00248135	-0.00000087	0.00000005
		-6.10579604	-0.00246501	0.00147301	0.00247960	-0.00000088	0.00000005
		-6.10635689	-0.00246328	0.00147210	0.00247786	-0.00000088	0.00000005
		-6.10691595	-0.00246155	0.00147119	0.00247612	-0.00000088	0.00000005
		-6.10747322	-0.00245983	0.00147028	0.00247438	-0.00000088	0.00000005
		-6.10802871	-0.00245811	0.00146937	0.00247264	-0.00000088	0.00000005
		-6.10858241	-0.00245639	0.00146847	0.00247091	-0.00000088	0.00000005
		-6.10913434	-0.00245467	0.00146756	0.00246917	-0.00000088	0.00000005
		-6.10968450	-0.00245295	0.00146665	0.00246744	-0.00000088	0.00000005
		-6.11023289	-0.00245123	0.00146575	0.00246571	-0.00000088	0.00000005
		-6.11077953	-0.00244952	0.00146484	0.00246398	-0.00000089	0.00000005
		-6.11132441	-0.00244781	0.00146393	0.00246226	-0.00000089	0.00000005
		-6.11186754	-0.00244610	0.00146303	0.00246053	-0.00000089	0.00000005
		-6.11240893	-0.00244439	0.00146213	0.00245881	-0.00000089	0.00000005
		-6.11294858	-0.00244269	0.00146122	0.00245709	-0.00000089	0.00000005
		-6.11348649	-0.00244098	0.00146032	0.00245537	-0.00000089	0.00000005
		-6.11402268	-0.00243928	0.00145942	0.00245366	-0.00000089	0.00000005
		-6.11455714	-0.00243758	0.00145852	0.00245194	-0.00000089	0.00000005
		-6.11508989	-0.00243588	0.00145762	0.00245023	-0.00000089	0.00000005
		-6.11562092	-0.00243419	0.00145672	0.00244852	-0.00000090	0.00000005
		-6.11615024	-0.00243249	0.00145582	0.00244681	-0.00000090	0.00000005
		-6.11667786	-0.00243080	0.00145492	0.00244510	-0.00000090	0.00000005
		-6.11720378	-0.00242911	0.00145402	0.00244340	-0.00000090	0.00000005
		-6.11772800	-0.00242742	0.00145312	0.00244170	-0.00000090	0.00000005
		-6.11825054	-0.00242574	0.00145223	0.00244000	-0.00000090	0.00000005
		-6.11877140	-0.00242405	0.00145133	0.00243830	-0.00000090	0.00000005
		-6.11929057	-0.00242237	0.00145043	0.00243660	-0.00000090	0.00000005
		-6.11980807	-0.00242069	0.00144954	0.00243490	-0.00000090	0.00000005
		-6.12032390	-0.00241901	0.00144864	0.00243321	-0.00000090	0.00000005
		-6.12083807	-0.00241733	0.00144775	0.00243152	-0.00000091	0.00000005
		-6.12135058	-0.00241565	0.00144686	0.00242983	-0.00000091	0.00000005
		-6.12186144	-0.00241398	0.00144597	0.00242814	-0.00000091	0.00000005
		-6.12237064	-0.00241231	0.00144507	0.00242645	-0.00000091	0.00000005
		-6.12287820	-0.00241064	0.00144418	0.00242477	-0.00000091	0.00000005
		-6.12338412	-0.00240897	0.00144329	0.00242309	-0.00000091	0.00000005
		-6.12388841	-0.00240730	0.00144240	0.00242141	-0.00000091	0.00000005
		-6.12439107	-0.00240563	0.00144151	0.00241973	-0.00000091	0.00000005
		-6.12489210	-0.00240397	0.00144062	0.00241805	-0.00000091	0.00000005
		-6.12539151	-0.00240231	0.00143974	0.00241637	-0.00000092	0.00000005
		-6.12588931	-0.00240065	0.00143885	0.00241470	-0.00000092	0.00000005
		-6.12638549	-0.00239899	0.00143796	0.00241303	-0.00000092	0.00000005
		-6.12688007	-0.00239734	0.00143708	0.00241136	-0.00000092	0.00000005
		-6.12737305	-0.00239568	0.00143619	0.00240969	-0.00000092	0.00000005
		-6.12786443	-0.00239403	0.00143531	0.00240803	-0.00000092	0.00000004
		-6.12835422	-0.00239238	0.00143442	0.00240636	-0.00000092	0.00000004
		-6.12884242	-0.00239073	0.00143354	0.00240470	-0.00000092	0.00000004
		-6.12932904	-0.00238908	0.00143266	0.00240304	-0.00000092	0.00000004
		-6.12981408	-0.00238744	0.00143177	0.00240138	-0.00000092	0.00000004
		-6.13029755	-0.00238580	0.00143089	0.00239972	-0.00000093	0.00000004
		-6.13077945	-0.00238416	0.00143001	0.00239807	-0.00000093	0.00000004
		-6.13125979	-0.00238252	0.00142913	0.00239642	-0.00000093	0.00000004
		-6.13173857	-0.00238088	0.00142825	0.00239477	-0.00000093	0.00000004
		-6.13221579	-0.00237924	0.00142737	0.00239312	-0.00000093	0.00000004
		-6.13269147	-0.00237761	0.00142649	0.00239147	-0.00000093	0.00000004
		-6.13316560	-0.00237598	0.00142562	0.00238982	-0.00000093	0.00000004
		-6.13363819	-0.00237434	0.00142474	0.00238818	-0.00000093	0.00000004
		-6.13410924	-0.00237272	0.00142386	0.00238654	-0.00000093	0.00000004
		-6.13457876	-0.00237109	0.00142299	0.00238490	-0.00000093	0.00000004
		-6.13504676	-0.00236946	0.00142211	0.00238326	-0.00000093	0.00000004
		-6.13551323	-0.00236784	0.00142124	0.00238162	-0.00000094	0.00000004
		-6.13597818	-0.00236622	0.00142036	0.00237999	-0.00000094	0.00000004
		-6.13644163	-0.00236460	0.00141949	0.00237835	-0.00000094	0.00000004
		-6.13690356	-0.00236298	0.00141862	0.00237672	-0.00000094	0.00000004
		-6.13736399	-0.00236136	0.00141774	0.00237509	-0.00000094	0.00000004
		-6.13782292	-0.00235975	0.00141687	0.00237347	-0.00000094	0.00000004
		-6.13828035	-0.00235814	0.00141600	0.00237184	-0.00000094	0.00000004
		-6.13873630	-0.00235652	0.00141513	0.00237022	-0.00000094	0.00000004
		-6.13919076	-0.00235492	0.00141426	0.00236859	-0.00000094	0.00000004
		-6.13964373	-0.00235331	0.00141339	0.00236697	-0.00000094	0.00000004
		-6.14009523	-0.00235170	0.00141253	0.00236535	-0.00000094	0.00000004
		-6.14054526	-0.00235010	0.00141166	0.00236374	-0.00000095	0.00000004
		-6.14099381	-0.00234850	0.00141079	0.00236212	-0.00000095	0.00000004
		-6.14144091	-0.00234689	0.00140993	0.00236051	-0.00000095	0.00000004
		-6.14188654	-0.00234530	0.00140906	0.00235890	-0.00000095	0.00000004
		-6.14233072	-0.00234370	0.00140820	0.00235729	-0.00000095	0.00000004
		-6.14277345	-0.00234210	0.00140733	0.00235568	-0.00000095	0.00000004
		-6.14321473	-0.00234051	0.00140647	0.00235407	-0.00000095	0.00000004
		-6.14365457	-0.00233892	0.00140561	0.00235247	-0.00000095	0.00000004
		-6.14409297	-0.00233733	0.00140474	0.00235087	-0.00000095	0.00000004
		-6.14452994	-0.00233574	0.00140388	0.00234926	-0.00000095	0.00000004
		-6.14496548	-0.00233415	0.00140302	0.00234767	-0.00000095	0.00000004
		-6.14539959	-0.00233257	0.00140216	0.00234607	-0.00000096	0.00000004
		-6.14583228	-0.00233098	0.00140130	0.00234447	-0.00000096	0.00000004
		-6.14626356	-0.00232940	0.00140044	0.00234288	-0.00000096	0.00000004
		-6.14669342	-0.00232782	0.00139958	0.00234129	-0.00000096	0.00000004
		-6.14712188	-0.00232625	0.00139873	0.00233970	-0.00000096	0.00000004
		-6.14754894	-0.00232467	0.00139787	0.00233811	-0.00000096	0.00000004
		-6.14797459	-0.00232309	0.00139701	0.00233652	-0.00000096	0.00000004
		-6.14839885	-0.00232152	0.00139616	0.00233493	-0.00000096	0.00000004
		-6.14882172	-0.00231995	0.00139530	0.00233335	-0.00000096	0.00000004
		-6.14924320	-0.00231838	0.00139445	0.00233177	-0.00000096	0.00000004
		-6.14966330	-0.00231681	0.00139359	0.00233019	-0.00000096	0.00000004
		-6.15008202	-0.00231525	0.00139274	0.00232861	-0.00000097	0.00000004
		-6.15049937	-0.00231368	0.00139189	0.00232704	-0.00000097	0.00000004
		-6.15091535	-0.00231212	0.00139104	0.00232546	-0.00000097	0.00000004
		-6.15132996	-0.00231056	0.00139018	0.00232389	-0.00000097	0.00000004
		-6.15174321	-0.00230900	0.00138933	0.00232232	-0.00000097	0.00000003
		-6.15215510	-0.00230744	0.00138848	0.00232075	-0.00000097	0.00000003
		-6.15256564	-0.00230589	0.00138763	0.00231918	-0.00000097	0.00000003
		-6.15297483	-0.00230433	0.00138679	0.00231761	-0.00000097	0.00000003
		-6.15338268	-0.00230278	0.00138594	0.00231605	-0.00000097	0.00000003
		-6.15378918	-0.00230123	0.00138509	0.00231449	-0.00000097	0.00000003
		-6.15419435	-0.00229968	0.00138424	0.00231293	-0.00000097	0.00000003
		-6.15459818	-0.00229814	0.00138340	0.00231137	-0.00000097	0.00000003
		-6.15500069	-0.00229659	0.00138255	0.00230981	-0.00000097	0.00000003
		-6.15540187	-0.00229505	0.00138171	0.00230825	-0.00000098	0.00000003
		-6.15580173	-0.00229350	0.00138087	0.00230670	-0.00000098	0.00000003
		-6.15620027	-0.00229196	0.00138002	0.00230515	-0.00000098	0.00000003
		-6.15659750	-0.00229042	0.00137918	0.00230360	-0.00000098	0.00000003
		-6.15699342	-0.00228889	0.00137834	0.00230205	-0.00000098	0.00000003
		-6.15738804	-0.00228735	0.00137750	0.00230050	-0.00000098	0.00000003
		-6.15778136	-0.00228582	0.00137666	0.00229896	-0.00000098	0.00000003
		-6.15817338	-0.00228429	0.00137582	0.00229741	-0.00000098	0.00000003
		-6.15856410	-0.00228276	0.00137498	0.00229587	-0.00000098	0.00000003
		-6.15895354	-0.00228123	0.00137414	0.00229433	-0.00000098	0.00000003
		-6.15934170	-0.00227970	0.00137330	0.00229279	-0.00000098	0.00000003
		-6.15972857	-0.00227818	0.00137246	0.00229125	-0.00000098	0.00000003
		-6.16011416	-0.00227665	0.00137162	0.00228972	-0.00000099	0.00000003
		-6.16049848	-0.00227513	0.00137079	0.00228818	-0.00000099	0.00000003
		-6.16088154	-0.00227361	0.00136995	0.00228665	-0.00000099	0.00000003
		-6.16126332	-0.00227209	0.00136912	0.00228512	-0.00000099	0.00000003
		-6.16164385	-0.00227057	0.00136828	0.00228359	-0.00000099	0.00000003
		-6.16202312	-0.00226906	0.00136745	0.00228207	-0.00000099	0.00000003
		-6.16240113	-0.00226754	0.00136662	0.00228054	-0.00000099	0.00000003
		-6.16277790	-0.00226603	0.00136579	0.00227902	-0.00000099	0.00000003
		-6.16315341	-0.00226452	0.00136495	0.00227750	-0.00000099	0.00000003
		-6.16352769	-0.00226301	0.00136412	0.00227598	-0.00000099	0.00000003
		-6.16390073	-0.00226151	0.00136329	0.00227446	-0.00000099	0.00000003
		-6.16427253	-0.00226000	0.00136246	0.00227294	-0.00000099	0.00000003
		-6.16464310	-0.00225850	0.00136164	0.00227143	-0.00000099	0.00000003
		-6.16501245	-0.00225700	0.00136081	0.00226991	-0.00000100	0.00000003
		-6.16538057	-0.00225549	0.00135998	0.00226840	-0.00000100	0.00000003
		-6.16574747	-0.00225400	0.00135915	0.00226689	-0.00000100	0.00000003
		-6.16611316	-0.00225250	0.00135833	0.00226538	-0.00000100	0.00000003
		-6.16647763	-0.00225100	0.00135750	0.00226388	-0.00000100	0.00000003
		-6.16684090	-0.00224951	0.00135668	0.00226237	-0.00000100	0.00000003
		-6.16720296	-0.00224802	0.00135585	0.00226087	-0.00000100	0.00000003
		-6.16756382	-0.00224653	0.00135503	0.00225936	-0.00000100	0.00000003
		-6.16792348	-0.00224504	0.00135421	0.00225786	-0.00000100	0.00000003
		-6.16828195	-0.00224355	0.00135338	0.00225637	-0.00000100	0.00000003
		-6.16863923	-0.00224206	0.00135256	0.00225487	-0.00000100	0.00000003
		-6.16899533	-0.00224058	0.00135174	0.00225337	-0.00000100	0.00000003
		-6.16935024	-0.00223910	0.00135092	0.00225188	-0.00000100	0.00000003
		-6.16970397	-0.00223762	0.00135010	0.00225039	-0.00000100	0.00000003
		-6.17005653	-0.00223614	0.00134928	0.00224890	-0.00000101	0.00000003
		-6.17040791	-0.00223466	0.00134846	0.00224741	-0.00000101	0.00000003
		-6.17075813	-0.00223318	0.00134765	0.00224592	-0.00000101	0.00000003
		-6.17110719	-0.00223171	0.00134683	0.00224444	-0.00000101	0.00000003
		-6.17145508	-0.00223024	0.00134601	0.00224295	-0.00000101	0.00000003
		-6.17180182	-0.00222876	0.00134520	0.00224147	-0.00000101	0.00000003
		-6.17214740	-0.00222729	0.00134438	0.00223999	-0.00000101	0.00000003
		-6.17249183	-0.00222583	0.00134357	0.00223851	-0.00000101	0.00000003
		-6.17283512	-0.00222436	0.00134275	0.00223703	-0.00000101	0.00000003
		-6.17317726	-0.00222289	0.00134194	0.00223556	-0.00000101	0.00000003
		-6.17351827	-0.00222143	0.00134113	0.00223408	-0.00000101	0.00000003
		-6.17385813	-0.00221997	0.00134032	0.00223261	-0.00000101	0.00000003
		-6.17419687	-0.00221851	0.00133950	0.00223114	-0.00000101	0.00000003
		-6.17453448	-0.00221705	0.00133869	0.00222967	-0.00000101	0.00000003
		-6.17487096	-0.00221559	0.00133788	0.00222820	-0.00000101	0.00000003
		-6.17520632	-0.00221414	0.00133708	0.00222674	-0.00000102	0.00000003
		-6.17554056	-0.00221269	0.00133627	0.00222527	-0.00000102	0.00000003
		-6.17587369	-0.00221123	0.00133546	0.00222381	-0.00000102	0.00000003
		-6.17620571	-0.00220978	0.00133465	0.00222235	-0.00000102	0.00000003
		-6.17653662	-0.00220833	0.00133385	0.00222089	-0.00000102	0.00000003
		-6.17686642	-0.00220689	0.00133304	0.00221943	-0.00000102	0.00000003
		-6.17719513	-0.00220544	0.00133223	0.00221798	-0.00000102	0.00000003
		-6.17752273	-0.00220400	0.00133143	0.00221652	-0.00000102	0.00000003
		-6.17784925	-0.00220255	0.00133063	0.00221507	-0.00000102	0.00000003
		-6.17817467	-0.00220111	0.00132982	0.00221362	-0.00000102	0.00000002
		-6.17849900	-0.00219967	0.00132902	0.00221217	-0.00000102	0.00000002
		-6.17882225	-0.00219823	0.00132822	0.00221072	-0.00000102	0.00000002
		-6.17914442	-0.00219680	0.00132742	0.00220927	-0.00000102	0.00000002
		-6.17946552	-0.00219536	0.00132662	0.00220782	-0.00000102	0.00000002
		-6.17978554	-0.00219393	0.00132582	0.00220638	-0.00000102	0.00000002
		-6.18010448	-0.00219250	0.00132502	0.00220494	-0.00000103	0.00000002
		-6.18042237	-0.00219107	0.00132422	0.00220350	-0.00000103	0.00000002
		-6.18073919	-0.00218964	0.00132342	0.00220206	-0.00000103	0.00000002
		-6.18105494	-0.00218821	0.00132262	0.00220062	-0.00000103	0.00000002
		-6.18136964	-0.00218679	0.00132183	0.00219919	-0.00000103	0.00000002
		-6.18168329	-0.00218536	0.00132103	0.00219775	-0.00000103	0.00000002
		-6.18199589	-0.00218394	0.00132024	0.00219632	-0.00000103	0.00000002
		-6.18230744	-0.00218252	0.00131944	0.00219489	-0.00000103	0.00000002
		-6.18261795	-0.00218110	0.00131865	0.00219346	-0.00000103	0.00000002
		-6.18292741	-0.00217968	0.00131785	0.00219203	-0.00000103	0.00000002
		-6.18323584	-0.00217827	0.00131706	0.00219060	-0.00000103	0.00000002
		-6.18354323	-0.00217685	0.00131627	0.00218918	-0.00000103	0.00000002
		-6.18384960	-0.00217544	0.00131548	0.00218776	-0.00000103	0.00000002
		-6.18415493	-0.00217403	0.00131469	0.00218633	-0.00000103	0.00000002
		-6.18445924	-0.00217262	0.00131390	0.00218491	-0.00000103	0.00000002
		-6.18476253	-0.00217121	0.00131311	0.00218350	-0.00000103	0.00000002
		-6.18506481	-0.00216980	0.00131232	0.00218208	-0.00000104	0.00000002
		-6.18536606	-0.00216840	0.00131153	0.00218066	-0.00000104	0.00000002
		-6.18566631	-0.00216699	0.00131074	0.00217925	-0.00000104	0.00000002
		-6.18596555	-0.00216559	0.00130996	0.00217784	-0.00000104	0.00000002
		-6.18626378	-0.00216419	0.00130917	0.00217643	-0.00000104	0.00000002
		-6.18656101	-0.00216279	0.00130839	0.00217502	-0.00000104	0.00000002
		-6.18685724	-0.00216139	0.00130760	0.00217361	-0.00000104	0.00000002
		-6.18715248	-0.00216000	0.00130682	0.00217220	-0.00000104	0.00000002
		-6.18744673	-0.00215860	0.00130603	0.00217080	-0.00000104	0.00000002
		-6.18773998	-0.00215721	0.00130525	0.00216939	-0.00000104	0.00000002
		-6.18803225	-0.00215582	0.00130447	0.00216799	-0.00000104	0.00000002
		-6.18832353	-0.00215443	0.00130369	0.00216659	-0.00000104	0.00000002
		-6.18861384	-0.00215304	0.00130291	0.00216519	-0.00000104	0.00000002
		-6.18890316	-0.00215165	0.00130213	0.00216380	-0.00000104	0.00000002
		-6.18919152	-0.00215026	0.00130135	0.00216240	-0.00000104	0.00000002
		-6.18947890	-0.00214888	0.00130057	0.00216101	-0.00000104	0.00000002
		-6.18976532	-0.00214750	0.00129979	0.00215961	-0.00000104	0.00000002
		-6.19005077	-0.00214612	0.00129901	0.00215822	-0.00000105	0.00000002
		-6.19033526	-0.00214474	0.00129823	0.00215683	-0.00000105	0.00000002
		-6.19061879	-0.00214336	0.00129746	0.00215545	-0.00000105	0.00000002
		-6.19090136	-0.00214198	0.00129668	0.00215406	-0.00000105	0.00000002
		-6.19118298	-0.00214061	0.00129591	0.00215268	-0.00000105	0.00000002
		-6.19146366	-0.00213923	0.00129513	0.00215129	-0.00000105	0.00000002
		-6.19174338	-0.00213786	0.00129436	0.00214991	-0.00000105	0.00000002
		-6.19202217	-0.00213649	0.00129359	0.00214853	-0.00000105	0.00000002
		-6.19230001	-0.00213512	0.00129281	0.00214715	-0.00000105	0.00000002
		-6.19257691	-0.00213375	0.00129204	0.00214577	-0.00000105	0.00000002
		-6.19285288	-0.00213239	0.00129127	0.00214440	-0.00000105	0.00000002
		-6.19312792	-0.00213102	0.00129050	0.00214302	-0.00000105	0.00000002
		-6.19340203	-0.00212966	0.00128973	0.00214165	-0.00000105	0.00000002
		-6.19367521	-0.00212830	0.00128896	0.00214028	-0.00000105	0.00000002
		-6.19394748	-0.00212694	0.00128819	0.00213891	-0.00000105	0.00000002
		-6.19421882	-0.00212558	0.00128743	0.00213754	-0.00000105	0.00000002
		-6.19448924	-0.00212422	0.00128666	0.00213617	-0.00000105	0.00000002
		-6.19475875	-0.00212286	0.00128589	0.00213481	-0.00000105	0.00000002
		-6.19502735	-0.00212151	0.00128513	0.00213344	-0.00000105	0.00000002
		-6.19529504	-0.00212016	0.00128436	0.00213208	-0.00000106	0.00000002
		-6.19556183	-0.00211880	0.00128360	0.00213072	-0.00000106	0.00000002
		-6.19582771	-0.00211745	0.00128283	0.00212936	-0.00000106	0.00000002
		-6.19609270	-0.00211611	0.00128207	0.00212800	-0.00000106	0.00000002
		-6.19635678	-0.00211476	0.00128131	0.00212665	-0.00000106	0.00000002
		-6.19661998	-0.00211341	0.00128054	0.00212529	-0.00000106	0.00000002
		-6.19688228	-0.00211207	0.00127978	0.00212394	-0.00000106	0.00000002
		-6.19714369	-0.00211073	0.00127902	0.00212259	-0.00000106	0.00000002
		-6.19740422	-0.00210938	0.00127826	0.00212123	-0.00000106	0.00000002
		-6.19766387	-0.00210804	0.00127750	0.00211989	-0.00000106	0.00000002
		-6.19792264	-0.00210671	0.00127674	0.00211854	-0.00000106	0.00000002
		-6.19818053	-0.00210537	0.00127599	0.00211719	-0.00000106	0.00000002
		-6.19843754	-0.00210403	0.00127523	0.00211585	-0.00000106	0.00000002
		-6.19869369	-0.00210270	0.00127447	0.00211450	-0.00000106	0.00000002
		-6.19894896	-0.00210137	0.00127372	0.00211316	-0.00000106	0.00000002
		-6.19920337	-0.00210004	0.00127296	0.00211182	-0.00000106	0.00000002
		-6.19945692	-0.00209871	0.00127221	0.00211048	-0.00000106	0.00000002
		-6.19970961	-0.00209738	0.00127145	0.00210914	-0.00000106	0.00000002
		-6.19996144	-0.00209605	0.00127070	0.00210781	-0.00000106	0.00000002
		-6.20021242	-0.00209473	0.00126995	0.00210647	-0.00000107	0.00000002
		-6.20046254	-0.00209340	0.00126919	0.00210514	-0.00000107	0.00000002
		-6.20071181	-0.00209208	0.00126844	0.00210381	-0.00000107	0.00000002
		-6.20096024	-0.00209076	0.00126769	0.00210248	-0.00000107	0.00000002
		-6.20120782	-0.00208944	0.00126694	0.00210115	-0.00000107	0.00000002
		-6.20145457	-0.00208812	0.00126619	0.00209982	-0.00000107	0.00000002
		-6.20170047	-0.00208680	0.00126544	0.00209850	-0.00000107	0.00000002
		-6.20194554	-0.00208549	0.00126469	0.00209717	-0.00000107	0.00000002
		-6.20218978	-0.00208417	0.00126395	0.00209585	-0.00000107	0.00000002
		-6.20243318	-0.00208286	0.00126320	0.00209453	-0.00000107	0.00000002
		-6.20267576	-0.00208155	0.00126245	0.00209321	-0.00000107	0.00000002
		-6.20291751	-0.00208024	0.00126171	0.00209189	-0.00000107	0.00000002
		-6.20315844	-0.00207893	0.00126096	0.00209057	-0.00000107	0.00000002
		-6.20339855	-0.00207762	0.00126022	0.00208925	-0.00000107	0.00000002
		-6.20363785	-0.00207632	0.00125947	0.00208794	-0.00000107	0.00000002
		-6.20387633	-0.00207501	0.00125873	0.00208663	-0.00000107	0.00000002
		-6.20411399	-0.00207371	0.00125799	0.00208532	-0.00000107	0.00000002
		-6.20435085	-0.00207241	0.00125725	0.00208400	-0.00000107	0.00000002
		-6.20458690	-0.00207111	0.00125650	0.00208270	-0.00000107	0.00000002
		-6.20482215	-0.00206981	0.00125576	0.00208139	-0.00000107	0.00000002
		-6.20505660	-0.00206852	0.00125502	0.00208008	-0.00000107	0.00000002
		-6.20529024	-0.00206722	0.00125428	0.00207878	-0.00000108	0.00000002
		-6.20552310	-0.00206593	0.00125355	0.00207748	-0.00000108	0.00000002
		-6.20575515	-0.00206463	0.00125281	0.00207617	-0.00000108	0.00000002
		-6.20598642	-0.00206334	0.00125207	0.00207487	-0.00000108	0.00000002
		-6.20621690	-0.00206205	0.00125133	0.00207357	-0.00000108	0.00000002
		-6.20644659	-0.00206076	0.00125060	0.00207228	-0.00000108	0.00000002
		-6.20667550	-0.00205947	0.00124986	0.00207098	-0.00000108	0.00000002
		-6.20690362	-0.00205819	0.00124913	0.00206969	-0.00000108	0.00000002
		-6.20713097	-0.00205690	0.00124839	0.00206839	-0.00000108	0.00000002
		-6.20735754	-0.00205562	0.00124766	0.00206710	-0.00000108	0.00000002
		-6.20758334	-0.00205434	0.00124693	0.00206581	-0.00000108	0.00000002
		-6.20780837	-0.00205306	0.00124619	0.00206452	-0.00000108	0.00000001
		-6.20803263	-0.00205178	0.00124546	0.00206323	-0.00000108	0.00000001
		-6.20825612	-0.00205050	0.00124473	0.00206195	-0.00000108	0.00000001
		-6.20847885	-0.00204923	0.00124400	0.00206066	-0.00000108	0.00000001
		-6.20870082	-0.00204795	0.00124327	0.00205938	-0.00000108	0.00000001
		-6.20892203	-0.00204668	0.00124254	0.00205810	-0.00000108	0.00000001
		-6.20914249	-0.00204540	0.00124181	0.00205681	-0.00000108	0.00000001
		-6.20936219	-0.00204413	0.00124109	0.00205554	-0.00000108	0.00000001
		-6.20958114	-0.00204286	0.00124036	0.00205426	-0.00000108	0.00000001
		-6.20979934	-0.00204160	0.00123963	0.00205298	-0.00000108	0.00000001
		-6.21001679	-0.00204033	0.00123891	0.00205171	-0.00000108	0.00000001
		-6.21023350	-0.00203906	0.00123818	0.00205043	-0.00000108	0.00000001
		-6.21044947	-0.00203780	0.00123746	0.00204916	-0.00000109	0.00000001
		-6.21066470	-0.00203654	0.00123673	0.00204789	-0.00000109	0.00000001
		-6.21087920	-0.00203528	0.00123601	0.00204662	-0.00000109	0.00000001
		-6.21109295	-0.00203402	0.00123529	0.00204535	-0.00000109	0.00000001
		-6.21130598	-0.00203276	0.00123456	0.00204408	-0.00000109	0.00000001
		-6.21151828	-0.00203150	0.00123384	0.00204282	-0.00000109	0.00000001
		-6.21172985	-0.00203024	0.00123312	0.00204155	-0.00000109	0.00000001
		-6.21194069	-0.00202899	0.00123240	0.00204029	-0.00000109	0.00000001
		-6.21215082	-0.00202774	0.00123168	0.00203903	-0.00000109	0.00000001
		-6.21236022	-0.00202648	0.00123096	0.00203777	-0.00000109	0.00000001
		-6.21256890	-0.00202523	0.00123025	0.00203651	-0.00000109	0.00000001
		-6.21277687	-0.00202398	0.00122953	0.00203525	-0.00000109	0.00000001
		-6.21298413	-0.00202274	0.00122881	0.00203399	-0.00000109	0.00000001
		-6.21319067	-0.00202149	0.00122809	0.00203274	-0.00000109	0.00000001
		-6.21339651	-0.00202025	0.00122738	0.00203149	-0.00000109	0.00000001
		-6.21360164	-0.00201900	0.00122666	0.00203023	-0.00000109	0.00000001
		-6.21380607	-0.00201776	0.00122595	0.00202898	-0.00000109	0.00000001
		-6.21400979	-0.00201652	0.00122523	0.00202773	-0.00000109	0.00000001
		-6.21421282	-0.00201528	0.00122452	0.00202649	-0.00000109	0.00000001
		-6.21441515	-0.00201404	0.00122381	0.00202524	-0.00000109	0.00000001
		-6.21461678	-0.00201280	0.00122310	0.00202399	-0.00000109	0.00000001
		-6.21481773	-0.00201157	0.00122239	0.00202275	-0.00000109	0.00000001
		-6.21501798	-0.00201033	0.00122167	0.00202151	-0.00000109	0.00000001
		-6.21521754	-0.00200910	0.00122096	0.00202026	-0.00000109	0.00000001
		-6.21541642	-0.00200787	0.00122026	0.00201902	-0.00000110	0.00000001
		-6.21561461	-0.00200664	0.00121955	0.00201779	-0.00000110	0.00000001
		-6.21581213	-0.00200541	0.00121884	0.00201655	-0.00000110	0.00000001
		-6.21600896	-0.00200418	0.00121813	0.00201531	-0.00000110	0.00000001
		-6.21620512	-0.00200295	0.00121742	0.00201408	-0.00000110	0.00000001
		-6.21640060	-0.00200173	0.00121672	0.00201284	-0.00000110	0.00000001
		-6.21659541	-0.00200050	0.00121601	0.00201161	-0.00000110	0.00000001
		-6.21678955	-0.00199928	0.00121531	0.00201038	-0.00000110	0.00000001
		-6.21698302	-0.00199806	0.00121460	0.00200915	-0.00000110	0.00000001
		-6.21717583	-0.00199684	0.00121390	0.00200792	-0.00000110	0.00000001
		-6.21736797	-0.00199562	0.00121320	0.00200670	-0.00000110	0.00000001
		-6.21755945	-0.00199440	0.00121249	0.00200547	-0.00000110	0.00000001
		-6.21775028	-0.00199319	0.00121179	0.00200425	-0.00000110	0.00000001
		-6.21794044	-0.00199197	0.00121109	0.00200302	-0.00000110	0.00000001
		-6.21812995	-0.00199076	0.00121039	0.00200180	-0.00000110	0.00000001
		-6.21831881	-0.00198955	0.00120969	0.00200058	-0.00000110	0.00000001
		-6.21850702	-0.00198833	0.00120899	0.00199936	-0.00000110	0.00000001
		-6.21869457	-0.00198712	0.00120829	0.00199815	-0.00000110	0.00000001
		-6.21888149	-0.00198592	0.00120759	0.00199693	-0.00000110	0.00000001
		-6.21906775	-0.00198471	0.00120689	0.00199571	-0.00000110	0.00000001
		-6.21925338	-0.00198350	0.00120620	0.00199450	-0.00000110	0.00000001
		-6.21943836	-0.00198230	0.00120550	0.00199329	-0.00000110	0.00000001
		-6.21962271	-0.00198110	0.00120481	0.00199208	-0.00000110	0.00000001
		-6.21980642	-0.00197989	0.00120411	0.00199087	-0.00000110	0.00000001
		-6.21998950	-0.00197869	0.00120342	0.00198966	-0.00000110	0.00000001
		-6.22017194	-0.00197749	0.00120272	0.00198845	-0.00000110	0.00000001
		-6.22035376	-0.00197630	0.00120203	0.00198724	-0.00000110	0.00000001
		-6.22053495	-0.00197510	0.00120134	0.00198604	-0.00000110	0.00000001
		-6.22071551	-0.00197390	0.00120064	0.00198484	-0.00000111	0.00000001
		-6.22089545	-0.00197271	0.00119995	0.00198363	-0.00000111	0.00000001
		-6.22107477	-0.00197152	0.00119926	0.00198243	-0.00000111	0.00000001
		-6.22125347	-0.00197032	0.00119857	0.00198123	-0.00000111	0.00000001
		-6.22143155	-0.00196913	0.00119788	0.00198003	-0.00000111	0.00000001
		-6.22160902	-0.00196794	0.00119719	0.00197884	-0.00000111	0.00000001
		-6.22178587	-0.00196676	0.00119650	0.00197764	-0.00000111	0.00000001
		-6.22196211	-0.00196557	0.00119582	0.00197645	-0.00000111	0.00000001
		-6.22213774	-0.00196438	0.00119513	0.00197525	-0.00000111	0.00000001
		-6.22231277	-0.00196320	0.00119444	0.00197406	-0.00000111	0.00000001
		-6.22248719	-0.00196202	0.00119376	0.00197287	-0.00000111	0.00000001
		-6.22266101	-0.00196083	0.00119307	0.00197168	-0.00000111	0.00000001
		-6.22283422	-0.00195965	0.00119239	0.00197049	-0.00000111	0.00000001
		-6.22300684	-0.00195848	0.00119170	0.00196931	-0.00000111	0.00000001
		-6.22317886	-0.00195730	0.00119102	0.00196812	-0.00000111	0.00000001
		-6.22335028	-0.00195612	0.00119034	0.00196694	-0.00000111	0.00000001
		-6.22352111	-0.00195495	0.00118965	0.00196575	-0.00000111	0.00000001
		-6.22369135	-0.00195377	0.00118897	0.00196457	-0.00000111	0.00000001
		-6.22386100	-0.00195260	0.00118829	0.00196339	-0.00000111	0.00000001
		-6.22403007	-0.00195143	0.00118761	0.00196221	-0.00000111	0.00000001
		-6.22419854	-0.00195026	0.00118693	0.00196103	-0.00000111	0.00000001
		-6.22436644	-0.00194909	0.00118625	0.00195986	-0.00000111	0.00000001
		-6.22453375	-0.00194792	0.00118557	0.00195868	-0.00000111	0.00000001
		-6.22470048	-0.00194675	0.00118490	0.00195751	-0.00000111	0.00000001
		-6.22486664	-0.00194559	0.00118422	0.00195633	-0.00000111	0.00000001
		-6.22503222	-0.00194442	0.00118354	0.00195516	-0.00000111	0.00000001
		-6.22519722	-0.00194326	0.00118287	0.00195399	-0.00000111	0.00000001
		-6.22536166	-0.00194210	0.00118219	0.00195282	-0.00000111	0.00000001
		-6.22552552	-0.00194094	0.00118152	0.00195165	-0.00000111	0.00000001
		-6.22568881	-0.00193978	0.00118084	0.00195049	-0.00000111	0.00000001
		-6.22585154	-0.00193862	0.00118017	0.00194932	-0.00000112	0.00000001
		-6.22601370	-0.00193746	0.00117950	0.00194816	-0.00000112	0.00000001
		-6.22617531	-0.00193631	0.00117882	0.00194699	-0.00000112	0.00000001
		-6.22633635	-0.00193515	0.00117815	0.00194583	-0.00000112	0.00000001
		-6.22649683	-0.00193400	0.00117748	0.00194467	-0.00000112	0.00000001
		-6.22665675	-0.00193285	0.00117681	0.00194351	-0.00000112	0.00000001
		-6.22681612	-0.00193170	0.00117614	0.00194235	-0.00000112	0.00000001
		-6.22697494	-0.00193055	0.00117547	0.00194120	-0.00000112	0.00000001
		-6.22713320	-0.00192940	0.00117480	0.00194004	-0.00000112	0.00000001
		-6.22729092	-0.00192825	0.00117413	0.00193889	-0.00000112	0.00000001
		-6.22744808	-0.00192711	0.00117346	0.00193773	-0.00000112	0.00000001
		-6.22760470	-0.00192596	0.00117280	0.00193658	-0.00000112	0.00000001
		-6.22776078	-0.00192482	0.00117213	0.00193543	-0.00000112	0.00000001
		-6.22791631	-0.00192368	0.00117147	0.00193428	-0.00000112	0.00000001
		-6.22807131	-0.00192253	0.00117080	0.00193313	-0.00000112	0.00000001
		-6.22822576	-0.00192139	0.00117014	0.00193198	-0.00000112	0.00000001
		-6.22837968	-0.00192026	0.00116947	0.00193084	-0.00000112	0.00000001
		-6.22853306	-0.00191912	0.00116881	0.00192969	-0.00000112	0.00000001
		-6.22868591	-0.00191798	0.00116815	0.00192855	-0.00000112	0.00000001
		-6.22883823	-0.00191685	0.00116748	0.00192741	-0.00000112	0.00000001
		-6.22899002	-0.00191571	0.00116682	0.00192627	-0.00000112	0.00000001
		-6.22914128	-0.00191458	0.00116616	0.00192513	-0.00000112	0.00000001
		-6.22929201	-0.00191345	0.00116550	0.00192399	-0.00000112	0.00000001
		-6.22944222	-0.00191232	0.00116484	0.00192285	-0.00000112	0.00000001
		-6.22959190	-0.00191119	0.00116418	0.00192171	-0.00000112	0.00000001
		-6.22974107	-0.00191006	0.00116352	0.00192058	-0.00000112	0.00000001
		-6.22988971	-0.00190893	0.00116286	0.00191944	-0.00000112	0.00000001
		-6.23003784	-0.00190781	0.00116221	0.00191831	-0.00000112	0.00000001
		-6.23018545	-0.00190669	0.00116155	0.00191718	-0.00000112	0.00000001
		-6.23033255	-0.00190556	0.00116089	0.00191605	-0.00000112	0.00000001
		-6.23047913	-0.00190444	0.00116024	0.00191492	-0.00000112	0.00000001
		-6.23062521	-0.00190332	0.00115958	0.00191379	-0.00000112	0.00000001
		-6.23077077	-0.00190220	0.00115893	0.00191267	-0.00000112	0.00000001
		-6.23091583	-0.00190108	0.00115828	0.00191154	-0.00000112	0.00000001
		-6.23106038	-0.00189996	0.00115762	0.00191042	-0.00000113	0.00000001
		-6.23120443	-0.00189885	0.00115697	0.00190929	-0.00000113	0.00000001
		-6.23134798	-0.00189773	0.00115632	0.00190817	-0.00000113	0.00000001
		-6.23149102	-0.00189662	0.00115567	0.00190705	-0.00000113	0.00000001
		-6.23163357	-0.00189551	0.00115501	0.00190593	-0.00000113	0.00000001
		-6.23177562	-0.00189439	0.00115436	0.00190481	-0.00000113	0.00000001
		-6.23191717	-0.00189328	0.00115371	0.00190369	-0.00000113	0.00000001
		-6.23205823	-0.00189218	0.00115307	0.00190258	-0.00000113	0.00000001
		-6.23219880	-0.00189107	0.00115242	0.00190146	-0.00000113	0.00000001
		-6.23233888	-0.00188996	0.00115177	0.00190035	-0.00000113	0.00000001
		-6.23247846	-0.00188886	0.00115112	0.00189924	-0.00000113	0.00000001
		-6.23261756	-0.00188775	0.00115047	0.00189812	-0.00000113	0.00000001
		-6.23275618	-0.00188665	0.00114983	0.00189701	-0.00000113	0.00000001
		-6.23289431	-0.00188555	0.00114918	0.00189591	-0.00000113	0.00000001
		-6.23303196	-0.00188444	0.00114854	0.00189480	-0.00000113	0.00000001
		-6.23316912	-0.00188334	0.00114789	0.00189369	-0.00000113	0.00000001
		-6.23330581	-0.00188225	0.00114725	0.00189258	-0.00000113	0.00000001
		-6.23344202	-0.00188115	0.00114661	0.00189148	-0.00000113	0.00000001
		-6.23357776	-0.00188005	0.00114596	0.00189038	-0.00000113	0.00000001
		-6.23371302	-0.00187896	0.00114532	0.00188927	-0.00000113	0.00000001
		-6.23384781	-0.00187786	0.00114468	0.00188817	-0.00000113	0.00000001
		-6.23398212	-0.00187677	0.00114404	0.00188707	-0.00000113	0.00000001
		-6.23411597	-0.00187568	0.00114340	0.00188598	-0.00000113	0.00000001
		-6.23424935	-0.00187459	0.00114276	0.00188488	-0.00000113	0.00000001
		-6.23438226	-0.00187350	0.00114212	0.00188378	-0.00000113	0.00000001
		-6.23451471	-0.00187241	0.00114148	0.00188269	-0.00000113	0.00000001
		-6.23464669	-0.00187132	0.00114084	0.00188159	-0.00000113	0.00000001
		-6.23477821	-0.00187024	0.00114021	0.00188050	-0.00000113	0.00000001
		-6.23490927	-0.00186915	0.00113957	0.00187941	-0.00000113	0.00000001
		-6.23503988	-0.00186807	0.00113893	0.00187832	-0.00000113	0.00000001
		-6.23517002	-0.00186698	0.00113830	0.00187723	-0.00000113	0.00000001
		-6.23529971	-0.00186590	0.00113766	0.00187614	-0.00000113	0.00000001
		-6.23542895	-0.00186482	0.00113703	0.00187505	-0.00000113	0.00000001
		-6.23555773	-0.00186374	0.00113639	0.00187396	-0.00000113	0.00000001
		-6.23568606	-0.00186267	0.00113576	0.00187288	-0.00000113	0.00000001
		-6.23581395	-0.00186159	0.00113513	0.00187180	-0.00000113	0.00000001
		-6.23594138	-0.00186051	0.00113449	0.00187071	-0.00000113	0.00000001
		-6.23606837	-0.00185944	0.00113386	0.00186963	-0.00000113	0.00000001
		-6.23619491	-0.00185836	0.00113323	0.00186855	-0.00000113	0.00000001
		-6.23632102	-0.00185729	0.00113260	0.00186747	-0.00000113	0.00000001
		-6.23644667	-0.00185622	0.00113197	0.00186639	-0.00000114	0.00000001
		-6.23657189	-0.00185515	0.00113134	0.00186532	-0.00000114	0.00000001
		-6.23669667	-0.00185408	0.00113071	0.00186424	-0.00000114	0.00000001
		-6.23682101	-0.00185301	0.00113009	0.00186317	-0.00000114	0.00000001
		-6.23694492	-0.00185195	0.00112946	0.00186209	-0.00000114	0.00000001
		-6.23706839	-0.00185088	0.00112883	0.00186102	-0.00000114	0.00000001
		-6.23719143	-0.00184981	0.00112820	0.00185995	-0.00000114	0.00000001
		-6.23731403	-0.00184875	0.00112758	0.00185888	-0.00000114	0.00000001
		-6.23743621	-0.00184769	0.00112695	0.00185781	-0.00000114	0.00000001
		-6.23755795	-0.00184663	0.00112633	0.00185674	-0.00000114	0.00000001
		-6.23767927	-0.00184557	0.00112570	0.00185567	-0.00000114	0.00000001
		-6.23780017	-0.00184451	0.00112508	0.00185461	-0.00000114	0.00000001
		-6.23792064	-0.00184345	0.00112446	0.00185354	-0.00000114	0.00000001
		-6.23804068	-0.00184239	0.00112383	0.00185248	-0.00000114	0.00000001
		-6.23816031	-0.00184134	0.00112321	0.00185142	-0.00000114	0.00000001
		-6.23827952	-0.00184028	0.00112259	0.00185035	-0.00000114	0.00000001
		-6.23839830	-0.00183923	0.00112197	0.00184929	-0.00000114	0.00000001
		-6.23851667	-0.00183817	0.00112135	0.00184823	-0.00000114	0.00000001
		-6.23863462	-0.00183712	0.00112073	0.00184718	-0.00000114	0.00000001
		-6.23875216	-0.00183607	0.00112011	0.00184612	-0.00000114	0.00000001
		-6.23886929	-0.00183502	0.00111949	0.00184506	-0.00000114	0.00000001
		-6.23898600	-0.00183398	0.00111887	0.00184401	-0.00000114	0.00000001
		-6.23910231	-0.00183293	0.00111826	0.00184295	-0.00000114	0.00000001
		-6.23921820	-0.00183188	0.00111764	0.00184190	-0.00000114	0.00000001
		-6.23933369	-0.00183084	0.00111702	0.00184085	-0.00000114	0.00000001
		-6.23944877	-0.00182979	0.00111641	0.00183980	-0.00000114	0.00000001
		-6.23956345	-0.00182875	0.00111579	0.00183875	-0.00000114	0.00000001
		-6.23967773	-0.00182771	0.00111518	0.00183770	-0.00000114	0.00000001
		-6.23979160	-0.00182667	0.00111456	0.00183665	-0.00000114	0.00000001
		-6.23990507	-0.00182563	0.00111395	0.00183561	-0.00000114	0.00000001
		-6.24001814	-0.00182459	0.00111334	0.00183456	-0.00000114	0.00000001
		-6.24013082	-0.00182355	0.00111273	0.00183352	-0.00000114	0.00000001
		-6.24024309	-0.00182251	0.00111211	0.00183248	-0.00000114	0.00000001
		-6.24035498	-0.00182148	0.00111150	0.00183143	-0.00000114	0.00000001
		-6.24046647	-0.00182045	0.00111089	0.00183039	-0.00000114	0.00000001
		-6.24057756	-0.00181941	0.00111028	0.00182935	-0.00000114	0.00000001
		-6.24068827	-0.00181838	0.00110967	0.00182831	-0.00000114	0.00000001
		-6.24079858	-0.00181735	0.00110906	0.00182728	-0.00000114	0.00000001
		-6.24090851	-0.00181632	0.00110846	0.00182624	-0.00000114	0.00000001
		-6.24101805	-0.00181529	0.00110785	0.00182521	-0.00000114	0.00000001
		-6.24112720	-0.00181426	0.00110724	0.00182417	-0.00000114	0.00000001
		-6.24123597	-0.00181324	0.00110663	0.00182314	-0.00000114	0.00000001
		-6.24134436	-0.00181221	0.00110603	0.00182211	-0.00000114	0.00000001
		-6.24145237	-0.00181118	0.00110542	0.00182107	-0.00000114	0.00000001
		-6.24155999	-0.00181016	0.00110482	0.00182004	-0.00000114	0.00000001
		-6.24166723	-0.00180914	0.00110421	0.00181902	-0.00000114	0.00000001
		-6.24177410	-0.00180812	0.00110361	0.00181799	-0.00000114	0.00000001
		-6.24188059	-0.00180710	0.00110300	0.00181696	-0.00000115	0.00000001
		-6.24198671	-0.00180608	0.00110240	0.00181593	-0.00000115	0.00000001
		-6.24209245	-0.00180506	0.00110180	0.00181491	-0.00000115	0.00000001
		-6.24219781	-0.00180404	0.00110120	0.00181389	-0.00000115	0.00000001
		-6.24230281	-0.00180302	0.00110060	0.00181286	-0.00000115	0.00000001
		-6.24240743	-0.00180201	0.00110000	0.00181184	-0.00000115	0.00000001
		-6.24251169	-0.00180100	0.00109940	0.00181082	-0.00000115	0.00000001
		-6.24261558	-0.00179998	0.00109880	0.00180980	-0.00000115	0.00000001
		-6.24271910	-0.00179897	0.00109820	0.00180878	-0.00000115	0.00000001
		-6.24282226	-0.00179796	0.00109760	0.00180777	-0.00000115	0.00000001
		-6.24292505	-0.00179695	0.00109700	0.00180675	-0.00000115	0.00000001
		-6.24302748	-0.00179594	0.00109640	0.00180573	-0.00000115	0.00000001
		-6.24312955	-0.00179493	0.00109581	0.00180472	-0.00000115	0.00000001
		-6.24323126	-0.00179392	0.00109521	0.00180371	-0.00000115	0.00000001
		-6.24333261	-0.00179292	0.00109461	0.00180269	-0.00000115	0.00000001
		-6.24343360	-0.00179191	0.00109402	0.00180168	-0.00000115	0.00000001
		-6.24353423	-0.00179091	0.00109342	0.00180067	-0.00000115	0.00000001
		-6.24363451	-0.00178991	0.00109283	0.00179966	-0.00000115	0.00000001
		-6.24373444	-0.00178891	0.00109224	0.00179866	-0.00000115	0.00000001
		-6.24383401	-0.00178790	0.00109164	0.00179765	-0.00000115	0.00000001
		-6.24393323	-0.00178690	0.00109105	0.00179664	-0.00000115	0.00000001
		-6.24403210	-0.00178591	0.00109046	0.00179564	-0.00000115	0.00000001
		-6.24413062	-0.00178491	0.00108987	0.00179463	-0.00000115	0.00000001
		-6.24422879	-0.00178391	0.00108928	0.00179363	-0.00000115	0.00000001
		-6.24432661	-0.00178292	0.00108869	0.00179263	-0.00000115	0.00000001
		-6.24442409	-0.00178192	0.00108810	0.00179163	-0.00000115	0.00000001
		-6.24452123	-0.00178093	0.00108751	0.00179063	-0.00000115	0.00000000
		-6.24461802	-0.00177993	0.00108692	0.00178963	-0.00000115	0.00000000
		-6.24471446	-0.00177894	0.00108633	0.00178863	-0.00000115	0.00000000
		-6.24481057	-0.00177795	0.00108574	0.00178764	-0.00000115	0.00000000
		-6.24490634	-0.00177696	0.00108516	0.00178664	-0.00000115	0.00000000
		-6.24500177	-0.00177597	0.00108457	0.00178565	-0.00000115	0.00000000
		-6.24509686	-0.00177499	0.00108398	0.00178465	-0.00000115	0.00000000
		-6.24519161	-0.00177400	0.00108340	0.00178366	-0.00000115	0.00000000
		-6.24528603	-0.00177301	0.00108281	0.00178267	-0.00000115	0.00000000
		-6.24538012	-0.00177203	0.00108223	0.00178168	-0.00000115	0.00000000
		-6.24547387	-0.00177105	0.00108164	0.00178069	-0.00000115	0.00000000
		-6.24556729	-0.00177006	0.00108106	0.00177970	-0.00000115	0.00000000
		-6.24566038	-0.00176908	0.00108048	0.00177871	-0.00000115	0.00000000
		-6.24575314	-0.00176810	0.00107990	0.00177772	-0.00000115	0.00000000
		-6.24584557	-0.00176712	0.00107931	0.00177674	-0.00000115	0.00000000
		-6.24593768	-0.00176614	0.00107873	0.00177575	-0.00000115	0.00000000
		-6.24602946	-0.00176517	0.00107815	0.00177477	-0.00000115	0.00000000
		-6.24612091	-0.00176419	0.00107757	0.00177379	-0.00000115	0.00000000
		-6.24621204	-0.00176321	0.00107699	0.00177281	-0.00000115	0.00000000
		-6.24630285	-0.00176224	0.00107641	0.00177183	-0.00000115	0.00000000
		-6.24639333	-0.00176127	0.00107584	0.00177085	-0.00000115	0.00000000
		-6.24648350	-0.00176029	0.00107526	0.00176987	-0.00000115	0.00000000
		-6.24657334	-0.00175932	0.00107468	0.00176889	-0.00000115	0.00000000
		-6.24666287	-0.00175835	0.00107410	0.00176791	-0.00000115	0.00000000
		-6.24675208	-0.00175738	0.00107353	0.00176694	-0.00000115	0.00000000
		-6.24684097	-0.00175641	0.00107295	0.00176596	-0.00000115	0.00000000
		-6.24692955	-0.00175545	0.00107238	0.00176499	-0.00000115	0.00000000
		-6.24701781	-0.00175448	0.00107180	0.00176402	-0.00000115	0.00000000
		-6.24710576	-0.00175351	0.00107123	0.00176305	-0.00000115	0.00000000
		-6.24719340	-0.00175255	0.00107065	0.00176207	-0.00000115	0.00000000
		-6.24728073	-0.00175158	0.00107008	0.00176110	-0.00000115	0.00000000
		-6.24736775	-0.00175062	0.00106951	0.00176014	-0.00000115	0.00000000
		-6.24745446	-0.00174966	0.00106893	0.00175917	-0.00000115	0.00000000
		-6.24754086	-0.00174870	0.00106836	0.00175820	-0.00000115	0.00000000
		-6.24762695	-0.00174774	0.00106779	0.00175724	-0.00000116	0.00000000
		-6.24771274	-0.00174678	0.00106722	0.00175627	-0.00000116	0.00000000
		-6.24779823	-0.00174582	0.00106665	0.00175531	-0.00000116	0.00000000
		-6.24788341	-0.00174487	0.00106608	0.00175434	-0.00000116	0.00000000
		-6.24796829	-0.00174391	0.00106551	0.00175338	-0.00000116	0.00000000
		-6.24805287	-0.00174296	0.00106494	0.00175242	-0.00000116	0.00000000
		-6.24813715	-0.00174200	0.00106438	0.00175146	-0.00000116	0.00000000
		-6.24822113	-0.00174105	0.00106381	0.00175050	-0.00000116	0.00000000
		-6.24830481	-0.00174010	0.00106324	0.00174955	-0.00000116	0.00000000
		-6.24838819	-0.00173915	0.00106267	0.00174859	-0.00000116	0.00000000
		-6.24847128	-0.00173819	0.00106211	0.00174763	-0.00000116	0.00000000
		-6.24855407	-0.00173725	0.00106154	0.00174668	-0.00000116	0.00000000
		-6.24863657	-0.00173630	0.00106098	0.00174572	-0.00000116	0.00000000
		-6.24871878	-0.00173535	0.00106041	0.00174477	-0.00000116	0.00000000
		-6.24880069	-0.00173440	0.00105985	0.00174382	-0.00000116	0.00000000
		-6.24888231	-0.00173346	0.00105929	0.00174287	-0.00000116	0.00000000
		-6.24896365	-0.00173251	0.00105872	0.00174192	-0.00000116	0.00000000
		-6.24904469	-0.00173157	0.00105816	0.00174097	-0.00000116	0.00000000
		-6.24912545	-0.00173063	0.00105760	0.00174002	-0.00000116	0.00000000
		-6.24920591	-0.00172969	0.00105704	0.00173907	-0.00000116	0.00000000
		-6.24928610	-0.00172875	0.00105648	0.00173813	-0.00000116	0.00000000
		-6.24936599	-0.00172781	0.00105592	0.00173718	-0.00000116	0.00000000
		-6.24944561	-0.00172687	0.00105536	0.00173624	-0.00000116	0.00000000
		-6.24952494	-0.00172593	0.00105480	0.00173529	-0.00000116	0.00000000
		-6.24960399	-0.00172499	0.00105424	0.00173435	-0.00000116	0.00000000
		-6.24968275	-0.00172406	0.00105368	0.00173341	-0.00000116	0.00000000
		-6.24976124	-0.00172312	0.00105312	0.00173247	-0.00000116	0.00000000
		-6.24983945	-0.00172219	0.00105257	0.00173153	-0.00000116	0.00000000
		-6.24991738	-0.00172126	0.00105201	0.00173059	-0.00000116	0.00000000
		-6.24999503	-0.00172032	0.00105145	0.00172965	-0.00000116	0.00000000
		-6.25007241	-0.00171939	0.00105090	0.00172871	-0.00000116	0.00000000
		-6.25014951	-0.00171846	0.00105034	0.00172778	-0.00000116	0.00000000
		-6.25022634	-0.00171753	0.00104979	0.00172684	-0.00000116	0.00000000
		-6.25030289	-0.00171660	0.00104923	0.00172591	-0.00000116	0.00000000
		-6.25037917	-0.00171568	0.00104868	0.00172498	-0.00000116	0.00000000
		-6.25045518	-0.00171475	0.00104813	0.00172404	-0.00000116	0.00000000
		-6.25053092	-0.00171382	0.00104758	0.00172311	-0.00000116	0.00000000
		-6.25060639	-0.00171290	0.00104702	0.00172218	-0.00000116	0.00000000
		-6.25068159	-0.00171198	0.00104647	0.00172125	-0.00000116	0.00000000
		-6.25075652	-0.00171105	0.00104592	0.00172032	-0.00000116	0.00000000
		-6.25083119	-0.00171013	0.00104537	0.00171940	-0.00000116	0.00000000
		-6.25090559	-0.00170921	0.00104482	0.00171847	-0.00000116	0.00000000
		-6.25097972	-0.00170829	0.00104427	0.00171754	-0.00000116	0.00000000
		-6.25105359	-0.00170737	0.00104372	0.00171662	-0.00000116	0.00000000
		-6.25112720	-0.00170645	0.00104317	0.00171570	-0.00000116	0.00000000
		-6.25120055	-0.00170554	0.00104262	0.00171477	-0.00000116	0.00000000
		-6.25127363	-0.00170462	0.00104208	0.00171385	-0.00000116	0.00000000
		-6.25134646	-0.00170370	0.00104153	0.00171293	-0.00000116	0.00000000
		-6.25141902	-0.00170279	0.00104098	0.00171201	-0.00000116	0.00000000
		-6.25149133	-0.00170188	0.00104044	0.00171109	-0.00000116	0.00000000
		-6.25156338	-0.00170096	0.00103989	0.00171017	-0.00000116	0.00000000
		-6.25163517	-0.00170005	0.00103935	0.00170926	-0.00000116	0.00000000
		-6.25170670	-0.00169914	0.00103880	0.00170834	-0.00000116	0.00000000
		-6.25177798	-0.00169823	0.00103826	0.00170742	-0.00000116	0.00000000
		-6.25184901	-0.00169732	0.00103771	0.00170651	-0.00000116	0.00000000
		-6.25191978	-0.00169641	0.00103717	0.00170560	-0.00000116	0.00000000
		-6.25199031	-0.00169551	0.00103663	0.00170468	-0.00000116	0.00000000
		-6.25206057	-0.00169460	0.00103609	0.00170377	-0.00000116	0.00000000
		-6.25213059	-0.00169369	0.00103555	0.00170286	-0.00000116	0.00000000
		-6.25220036	-0.00169279	0.00103500	0.00170195	-0.00000116	0.00000000
		-6.25226988	-0.00169189	0.00103446	0.00170104	-0.00000116	0.00000000
		-6.25233916	-0.00169098	0.00103392	0.00170013	-0.00000116	0.00000000
		-6.25240818	-0.00169008	0.00103338	0.00169923	-0.00000116	0.00000000
		-6.25247696	-0.00168918	0.00103285	0.00169832	-0.00000116	0.00000000
		-6.25254550	-0.00168828	0.00103231	0.00169741	-0.00000116	0.00000000
		-6.25261379	-0.00168738	0.00103177	0.00169651	-0.00000116	0.00000000
		-6.25268183	-0.00168648	0.00103123	0.00169560	-0.00000116	0.00000000
		-6.25274963	-0.00168559	0.00103069	0.00169470	-0.00000116	0.00000000
		-6.25281720	-0.00168469	0.00103016	0.00169380	-0.00000116	0.00000000
		-6.25288452	-0.00168379	0.00102962	0.00169290	-0.00000116	0.00000000
		-6.25295160	-0.00168290	0.00102909	0.00169200	-0.00000116	0.00000000
		-6.25301844	-0.00168200	0.00102855	0.00169110	-0.00000116	0.00000000
		-6.25308504	-0.00168111	0.00102802	0.00169020	-0.00000116	0.00000000
		-6.25315140	-0.00168022	0.00102748	0.00168930	-0.00000116	0.00000000
		-6.25321753	-0.00167933	0.00102695	0.00168841	-0.00000116	0.00000000
		-6.25328342	-0.00167844	0.00102641	0.00168751	-0.00000116	0.00000000
		-6.25334907	-0.00167755	0.00102588	0.00168662	-0.00000116	0.00000000
		-6.25341450	-0.00167666	0.00102535	0.00168572	-0.00000116	0.00000000
		-6.25347968	-0.00167577	0.00102482	0.00168483	-0.00000116	0.00000000
		-6.25354464	-0.00167489	0.00102429	0.00168394	-0.00000116	0.00000000
		-6.25360936	-0.00167400	0.00102376	0.00168305	-0.00000116	0.00000000
		-6.25367385	-0.00167312	0.00102323	0.00168216	-0.00000116	0.00000000
		-6.25373811	-0.00167223	0.00102270	0.00168127	-0.00000116	0.00000000
		-6.25380215	-0.00167135	0.00102217	0.00168038	-0.00000116	0.00000000
		-6.25386595	-0.00167047	0.00102164	0.00167949	-0.00000117	0.00000000
		-6.25392952	-0.00166958	0.00102111	0.00167860	-0.00000117	0.00000000
		-6.25399287	-0.00166870	0.00102058	0.00167772	-0.00000117	0.00000000
		-6.25405599	-0.00166782	0.00102005	0.00167683	-0.00000117	0.00000000
		-6.25411889	-0.00166694	0.00101953	0.00167595	-0.00000117	0.00000000
		-6.25418156	-0.00166607	0.00101900	0.00167507	-0.00000117	0.00000000
		-6.25424401	-0.00166519	0.00101847	0.00167418	-0.00000117	0.00000000
		-6.25430623	-0.00166431	0.00101795	0.00167330	-0.00000117	0.00000000
		-6.25436823	-0.00166344	0.00101742	0.00167242	-0.00000117	0.00000000
		-6.25443001	-0.00166256	0.00101690	0.00167154	-0.00000117	0.00000000
		-6.25449157	-0.00166169	0.00101638	0.00167066	-0.00000117	0.00000000
		-6.25455291	-0.00166082	0.00101585	0.00166978	-0.00000117	0.00000000
		-6.25461403	-0.00165994	0.00101533	0.00166891	-0.00000117	0.00000000
		-6.25467493	-0.00165907	0.00101481	0.00166803	-0.00000117	0.00000000
		-6.25473561	-0.00165820	0.00101428	0.00166715	-0.00000117	0.00000000
		-6.25479608	-0.00165733	0.00101376	0.00166628	-0.00000117	0.00000000
		-6.25485633	-0.00165647	0.00101324	0.00166541	-0.00000117	0.00000000
		-6.25491636	-0.00165560	0.00101272	0.00166453	-0.00000117	0.00000000
		-6.25497618	-0.00165473	0.00101220	0.00166366	-0.00000117	0.00000000
		-6.25503579	-0.00165386	0.00101168	0.00166279	-0.00000117	0.00000000
		-6.25509518	-0.00165300	0.00101116	0.00166192	-0.00000117	0.00000000
		-6.25515436	-0.00165214	0.00101064	0.00166105	-0.00000117	0.00000000
		-6.25521333	-0.00165127	0.00101012	0.00166018	-0.00000117	0.00000000
		-6.25527209	-0.00165041	0.00100961	0.00165931	-0.00000117	0.00000000
		-6.25533063	-0.00164955	0.00100909	0.00165845	-0.00000117	0.00000000
		-6.25538897	-0.00164869	0.00100857	0.00165758	-0.00000117	0.00000000
		-6.25544710	-0.00164783	0.00100806	0.00165672	-0.00000117	0.00000000
		-6.25550502	-0.00164697	0.00100754	0.00165585	-0.00000117	0.00000000
		-6.25556273	-0.00164611	0.00100702	0.00165499	-0.00000117	0.00000000
		-6.25562024	-0.00164525	0.00100651	0.00165413	-0.00000117	0.00000000
		-6.25567754	-0.00164440	0.00100599	0.00165326	-0.00000117	0.00000000
		-6.25573464	-0.00164354	0.00100548	0.00165240	-0.00000117	0.00000000
		-6.25579153	-0.00164268	0.00100497	0.00165154	-0.00000117	0.00000000
		-6.25584821	-0.00164183	0.00100445	0.00165068	-0.00000117	0.00000000
		-6.25590470	-0.00164098	0.00100394	0.00164982	-0.00000117	0.00000000
		-6.25596098	-0.00164012	0.00100343	0.00164897	-0.00000117	0.00000000
		-6.25601706	-0.00163927	0.00100292	0.00164811	-0.00000117	0.00000000
		-6.25607294	-0.00163842	0.00100241	0.00164725	-0.00000117	0.00000000
		-6.25612862	-0.00163757	0.00100189	0.00164640	-0.00000117	0.00000000
		-6.25618410	-0.00163672	0.00100138	0.00164554	-0.00000117	0.00000000
		-6.25623938	-0.00163587	0.00100087	0.00164469	-0.00000117	0.00000000
		-6.25629447	-0.00163503	0.00100036	0.00164384	-0.00000117	0.00000000
		-6.25634936	-0.00163418	0.00099986	0.00164299	-0.00000117	0.00000000
		-6.25640405	-0.00163333	0.00099935	0.00164214	-0.00000117	0.00000000
		-6.25645854	-0.00163249	0.00099884	0.00164129	-0.00000117	0.00000000
		-6.25651284	-0.00163164	0.00099833	0.00164044	-0.00000117	0.00000000
		-6.25656694	-0.00163080	0.00099782	0.00163959	-0.00000117	0.00000000
		-6.25662085	-0.00162996	0.00099732	0.00163874	-0.00000117	0.00000000
		-6.25667457	-0.00162912	0.00099681	0.00163789	-0.00000117	0.00000000
		-6.25672809	-0.00162828	0.00099631	0.00163705	-0.00000117	0.00000000
		-6.25678143	-0.00162744	0.00099580	0.00163620	-0.00000117	0.00000000
		-6.25683457	-0.00162660	0.00099530	0.00163536	-0.00000117	0.00000000
		-6.25688752	-0.00162576	0.00099479	0.00163451	-0.00000117	0.00000000
		-6.25694028	-0.00162492	0.00099429	0.00163367	-0.00000117	0.00000000
		-6.25699285	-0.00162408	0.00099378	0.00163283	-0.00000117	0.00000000
		-6.25704524	-0.00162325	0.00099328	0.00163199	-0.00000117	0.00000000
		-6.25709743	-0.00162241	0.00099278	0.00163115	-0.00000117	0.00000000
		-6.25714944	-0.00162158	0.00099228	0.00163031	-0.00000117	0.00000000
		-6.25720127	-0.00162074	0.00099177	0.00162947	-0.00000117	0.00000000
		-6.25725290	-0.00161991	0.00099127	0.00162863	-0.00000117	0.00000000
		-6.25730435	-0.00161908	0.00099077	0.00162780	-0.00000117	0.00000000
		-6.25735562	-0.00161825	0.00099027	0.00162696	-0.00000117	0.00000000
		-6.25740670	-0.00161742	0.00098977	0.00162613	-0.00000117	0.00000000
		-6.25745761	-0.00161659	0.00098927	0.00162529	-0.00000117	0.00000000
		-6.25750832	-0.00161576	0.00098877	0.00162446	-0.00000117	0.00000000
		-6.25755886	-0.00161493	0.00098828	0.00162362	-0.00000117	0.00000000
		-6.25760921	-0.00161411	0.00098778	0.00162279	-0.00000117	0.00000000
		-6.25765939	-0.00161328	0.00098728	0.00162196	-0.00000117	0.00000000
		-6.25770938	-0.00161245	0.00098678	0.00162113	-0.00000117	0.00000000
		-6.25775920	-0.00161163	0.00098629	0.00162030	-0.00000117	0.00000000
		-6.25780883	-0.00161080	0.00098579	0.00161947	-0.00000117	0.00000000
		-6.25785829	-0.00160998	0.00098529	0.00161864	-0.00000117	0.00000000
		-6.25790757	-0.00160916	0.00098480	0.00161782	-0.00000117	0.00000000
		-6.25795667	-0.00160834	0.00098430	0.00161699	-0.00000117	0.00000000
		-6.25800560	-0.00160752	0.00098381	0.00161616	-0.00000117	0.00000000
		-6.25805435	-0.00160670	0.00098332	0.00161534	-0.00000117	0.00000000
		-6.25810293	-0.00160588	0.00098282	0.00161452	-0.00000117	0.00000000
		-6.25815133	-0.00160506	0.00098233	0.00161369	-0.00000117	0.00000000
		-6.25819956	-0.00160424	0.00098184	0.00161287	-0.00000117	0.00000000
		-6.25824762	-0.00160342	0.00098134	0.00161205	-0.00000117	0.00000000
		-6.25829550	-0.00160261	0.00098085	0.00161123	-0.00000117	0.00000000
		-6.25834321	-0.00160179	0.00098036	0.00161041	-0.00000117	0.00000000
		-6.25839075	-0.00160098	0.00097987	0.00160959	-0.00000117	0.00000000
		-6.25843812	-0.00160017	0.00097938	0.00160877	-0.00000117	0.00000000
		-6.25848532	-0.00159935	0.00097889	0.00160795	-0.00000117	0.00000000
		-6.25853235	-0.00159854	0.00097840	0.00160713	-0.00000117	0.00000000
		-6.25857922	-0.00159773	0.00097791	0.00160632	-0.00000117	0.00000000
		-6.25862591	-0.00159692	0.00097742	0.00160550	-0.00000117	0.00000000
		-6.25867244	-0.00159611	0.00097693	0.00160469	-0.00000117	0.00000000
		-6.25871879	-0.00159530	0.00097645	0.00160387	-0.00000117	0.00000000
		-6.25876499	-0.00159449	0.00097596	0.00160306	-0.00000117	0.00000000
		-6.25881101	-0.00159368	0.00097547	0.00160225	-0.00000117	0.00000000
		-6.25885688	-0.00159288	0.00097499	0.00160144	-0.00000117	0.00000000
		-6.25890257	-0.00159207	0.00097450	0.00160063	-0.00000117	0.00000000
		-6.25894810	-0.00159127	0.00097402	0.00159982	-0.00000117	0.00000000
		-6.25899347	-0.00159046	0.00097353	0.00159901	-0.00000117	0.00000000
		-6.25903868	-0.00158966	0.00097305	0.00159820	-0.00000117	0.00000000
		-6.25908372	-0.00158886	0.00097256	0.00159739	-0.00000117	0.00000000
		-6.25912861	-0.00158805	0.00097208	0.00159659	-0.00000117	0.00000000
		-6.25917333	-0.00158725	0.00097159	0.00159578	-0.00000117	0.00000000
		-6.25921789	-0.00158645	0.00097111	0.00159497	-0.00000117	0.00000000
		-6.25926229	-0.00158565	0.00097063	0.00159417	-0.00000117	0.00000000
		-6.25930653	-0.00158485	0.00097015	0.00159337	-0.00000117	0.00000000
		-6.25935061	-0.00158406	0.00096966	0.00159256	-0.00000117	0.00000000
		-6.25939453	-0.00158326	0.00096918	0.00159176	-0.00000117	0.00000000
		-6.25943830	-0.00158246	0.00096870	0.00159096	-0.00000117	0.00000000
		-6.25948191	-0.00158167	0.00096822	0.00159016	-0.00000117	0.00000000
		-6.25952536	-0.00158087	0.00096774	0.00158936	-0.00000117	0.00000000
		-6.25956866	-0.00158008	0.00096726	0.00158856	-0.00000117	0.00000000
		-6.25961180	-0.00157928	0.00096678	0.00158776	-0.00000117	0.00000000
		-6.25965478	-0.00157849	0.00096631	0.00158696	-0.00000117	0.00000000
		-6.25969761	-0.00157770	0.00096583	0.00158617	-0.00000117	0.00000000
		-6.25974029	-0.00157691	0.00096535	0.00158537	-0.00000117	0.00000000
		-6.25978282	-0.00157612	0.00096487	0.00158458	-0.00000117	0.00000000
		-6.25982519	-0.00157533	0.00096440	0.00158378	-0.00000117	0.00000000
		-6.25986741	-0.00157454	0.00096392	0.00158299	-0.00000117	0.00000000
		-6.25990947	-0.00157375	0.00096344	0.00158220	-0.00000117	0.00000000
		-6.25995139	-0.00157296	0.00096297	0.00158140	-0.00000117	0.00000000
		-6.25999315	-0.00157217	0.00096249	0.00158061	-0.00000117	0.00000000
		-6.26003477	-0.00157139	0.00096202	0.00157982	-0.00000117	0.00000000
		-6.26007624	-0.00157060	0.00096154	0.00157903	-0.00000117	0.00000000
		-6.26011755	-0.00156982	0.00096107	0.00157824	-0.00000117	0.00000000
		-6.26015872	-0.00156903	0.00096060	0.00157745	-0.00000117	0.00000000
		-6.26019974	-0.00156825	0.00096012	0.00157667	-0.00000117	0.00000000
		-6.26024061	-0.00156747	0.00095965	0.00157588	-0.00000117	0.00000000
		-6.26028134	-0.00156669	0.00095918	0.00157509	-0.00000117	0.00000000
		-6.26032192	-0.00156591	0.00095871	0.00157431	-0.00000117	0.00000000
		-6.26036235	-0.00156513	0.00095823	0.00157352	-0.00000117	0.00000000
		-6.26040264	-0.00156435	0.00095776	0.00157274	-0.00000117	0.00000000
		-6.26044279	-0.00156357	0.00095729	0.00157195	-0.00000117	0.00000000
		-6.26048279	-0.00156279	0.00095682	0.00157117	-0.00000117	0.00000000
		-6.26052264	-0.00156201	0.00095635	0.00157039	-0.00000117	0.00000000
		-6.26056235	-0.00156124	0.00095588	0.00156961	-0.00000117	0.00000000
		-6.26060192	-0.00156046	0.00095542	0.00156883	-0.00000117	0.00000000
		-6.26064135	-0.00155969	0.00095495	0.00156805	-0.00000117	0.00000000
		-6.26068064	-0.00155891	0.00095448	0.00156727	-0.00000117	0.00000000
		-6.26071978	-0.00155814	0.00095401	0.00156649	-0.00000117	0.00000000
		-6.26075879	-0.00155737	0.00095354	0.00156572	-0.00000117	0.00000000
		-6.26079765	-0.00155659	0.00095308	0.00156494	-0.00000117	0.00000000
		-6.26083637	-0.00155582	0.00095261	0.00156416	-0.00000117	0.00000000
		-6.26087496	-0.00155505	0.00095215	0.00156339	-0.00000117	0.00000000
		-6.26091341	-0.00155428	0.00095168	0.00156261	-0.00000117	0.00000000
		-6.26095171	-0.00155351	0.00095122	0.00156184	-0.00000117	0.00000000
		-6.26098988	-0.00155274	0.00095075	0.00156107	-0.00000117	0.00000000
		-6.26102792	-0.00155198	0.00095029	0.00156029	-0.00000117	0.00000000
		-6.26106581	-0.00155121	0.00094982	0.00155952	-0.00000117	0.00000000
		-6.26110357	-0.00155044	0.00094936	0.00155875	-0.00000117	0.00000000
		-6.26114120	-0.00154968	0.00094890	0.00155798	-0.00000117	0.00000000
		-6.26117869	-0.00154891	0.00094843	0.00155721	-0.00000117	0.00000000
		-6.26121604	-0.00154815	0.00094797	0.00155644	-0.00000117	0.00000000
		-6.26125326	-0.00154739	0.00094751	0.00155568	-0.00000117	0.00000000
		-6.26129035	-0.00154662	0.00094705	0.00155491	-0.00000117	0.00000000
		-6.26132730	-0.00154586	0.00094659	0.00155414	-0.00000117	0.00000000
		-6.26136412	-0.00154510	0.00094613	0.00155338	-0.00000117	0.00000000
		-6.26140081	-0.00154434	0.00094567	0.00155261	-0.00000117	0.00000000
		-6.26143736	-0.00154358	0.00094521	0.00155185	-0.00000117	0.00000000
		-6.26147379	-0.00154282	0.00094475	0.00155109	-0.00000117	0.00000000
		-6.26151008	-0.00154206	0.00094429	0.00155032	-0.00000117	0.00000000
		-6.26154624	-0.00154131	0.00094383	0.00154956	-0.00000117	0.00000000
		-6.26158228	-0.00154055	0.00094337	0.00154880	-0.00000117	0.00000000
		-6.26161818	-0.00153979	0.00094292	0.00154804	-0.00000117	0.00000000
		-6.26165395	-0.00153904	0.00094246	0.00154728	-0.00000117	0.00000000
		-6.26168960	-0.00153828	0.00094200	0.00154652	-0.00000117	0.00000000
		-6.26172512	-0.00153753	0.00094155	0.00154576	-0.00000117	0.00000000
		-6.26176051	-0.00153677	0.00094109	0.00154500	-0.00000117	0.00000000
		-6.26179577	-0.00153602	0.00094063	0.00154425	-0.00000117	0.00000000
		-6.26183090	-0.00153527	0.00094018	0.00154349	-0.00000117	0.00000000
		-6.26186591	-0.00153452	0.00093972	0.00154273	-0.00000117	0.00000000
		-6.26190080	-0.00153377	0.00093927	0.00154198	-0.00000117	0.00000000
		-6.26193555	-0.00153302	0.00093882	0.00154123	-0.00000117	0.00000000
		-6.26197019	-0.00153227	0.00093836	0.00154047	-0.00000117	0.00000000
		-6.26200470	-0.00153152	0.00093791	0.00153972	-0.00000117	0.00000000
		-6.26203908	-0.00153077	0.00093746	0.00153897	-0.00000117	0.00000000
		-6.26207334	-0.00153003	0.00093700	0.00153822	-0.00000118	0.00000000
		-6.26210748	-0.00152928	0.00093655	0.00153746	-0.00000118	0.00000000
		-6.26214149	-0.00152853	0.00093610	0.00153671	-0.00000118	0.00000000
		-6.26217539	-0.00152779	0.00093565	0.00153597	-0.00000118	0.00000000
		-6.26220916	-0.00152705	0.00093520	0.00153522	-0.00000118	0.00000000
		-6.26224280	-0.00152630	0.00093475	0.00153447	-0.00000118	0.00000000
		-6.26227633	-0.00152556	0.00093430	0.00153372	-0.00000118	0.00000000
		-6.26230974	-0.00152482	0.00093385	0.00153298	-0.00000118	0.00000000
		-6.26234303	-0.00152408	0.00093340	0.00153223	-0.00000118	0.00000000
		-6.26237620	-0.00152333	0.00093295	0.00153148	-0.00000118	0.00000000
		-6.26240924	-0.00152259	0.00093250	0.00153074	-0.00000118	0.00000000
		-6.26244217	-0.00152186	0.00093205	0.00153000	-0.00000118	0.00000000
		-6.26247499	-0.00152112	0.00093161	0.00152925	-0.00000118	0.00000000
		-6.26250768	-0.00152038	0.00093116	0.00152851	-0.00000118	0.00000000
		-6.26254025	-0.00151964	0.00093071	0.00152777	-0.00000118	0.00000000
		-6.26257271	-0.00151890	0.00093027	0.00152703	-0.00000118	0.00000000
		-6.26260505	-0.00151817	0.00092982	0.00152629	-0.00000118	0.00000000
		-6.26263728	-0.00151743	0.00092937	0.00152555	-0.00000118	0.00000000
		-6.26266939	-0.00151670	0.00092893	0.00152481	-0.00000118	0.00000000
		-6.26270138	-0.00151596	0.00092848	0.00152407	-0.00000118	0.00000000
		-6.26273326	-0.00151523	0.00092804	0.00152333	-0.00000118	0.00000000
		-6.26276503	-0.00151450	0.00092760	0.00152260	-0.00000118	0.00000000
		-6.26279668	-0.00151377	0.00092715	0.00152186	-0.00000118	0.00000000
		-6.26282822	-0.00151304	0.00092671	0.00152113	-0.00000118	0.00000000
		-6.26285964	-0.00151231	0.00092627	0.00152039	-0.00000118	0.00000000
		-6.26289095	-0.00151158	0.00092582	0.00151966	-0.00000118	0.00000000
		-6.26292215	-0.00151085	0.00092538	0.00151892	-0.00000118	0.00000000
		-6.26295323	-0.00151012	0.00092494	0.00151819	-0.00000118	0.00000000
		-6.26298421	-0.00150939	0.00092450	0.00151746	-0.00000118	0.00000000
		-6.26301507	-0.00150866	0.00092406	0.00151673	-0.00000118	0.00000000
		-6.26304582	-0.00150794	0.00092362	0.00151600	-0.00000118	0.00000000
		-6.26307646	-0.00150721	0.00092318	0.00151527	-0.00000118	0.00000000
		-6.26310699	-0.00150649	0.00092274	0.00151454	-0.00000118	0.00000000
		-6.26313741	-0.00150576	0.00092230	0.00151381	-0.00000118	0.00000000
		-6.26316772	-0.00150504	0.00092186	0.00151308	-0.00000118	0.00000000
		-6.26319793	-0.00150431	0.00092142	0.00151235	-0.00000118	0.00000000
		-6.26322802	-0.00150359	0.00092098	0.00151163	-0.00000118	0.00000000
		-6.26325801	-0.00150287	0.00092055	0.00151090	-0.00000118	0.00000000
		-6.26328788	-0.00150215	0.00092011	0.00151018	-0.00000118	0.00000000
		-6.26331765	-0.00150143	0.00091967	0.00150945	-0.00000118	0.00000000
		-6.26334732	-0.00150071	0.00091924	0.00150873	-0.00000118	0.00000000
		-6.26337687	-0.00149999	0.00091880	0.00150800	-0.00000118	0.00000000
		-6.26340633	-0.00149927	0.00091836	0.00150728	-0.00000118	0.00000000
		-6.26343567	-0.00149855	0.00091793	0.00150656	-0.00000118	0.00000000
		-6.26346491	-0.00149784	0.00091749	0.00150584	-0.00000118	0.00000000
		-6.26349404	-0.00149712	0.00091706	0.00150512	-0.00000118	0.00000000
		-6.26352307	-0.00149640	0.00091662	0.00150440	-0.00000118	0.00000000
		-6.26355200	-0.00149569	0.00091619	0.00150368	-0.00000118	0.00000000
		-6.26358082	-0.00149498	0.00091576	0.00150296	-0.00000118	0.00000000
		-6.26360954	-0.00149426	0.00091532	0.00150224	-0.00000118	0.00000000
		-6.26363815	-0.00149355	0.00091489	0.00150152	-0.00000118	0.00000000
		-6.26366666	-0.00149284	0.00091446	0.00150081	-0.00000118	0.00000000
		-6.26369507	-0.00149212	0.00091403	0.00150009	-0.00000118	0.00000000
		-6.26372338	-0.00149141	0.00091359	0.00149938	-0.00000118	0.00000000
		-6.26375159	-0.00149070	0.00091316	0.00149866	-0.00000118	0.00000000
		-6.26377969	-0.00148999	0.00091273	0.00149795	-0.00000118	0.00000000
		-6.26380769	-0.00148928	0.00091230	0.00149723	-0.00000118	0.00000000
		-6.26383560	-0.00148857	0.00091187	0.00149652	-0.00000118	0.00000000
		-6.26386340	-0.00148787	0.00091144	0.00149581	-0.00000118	0.00000000
		-6.26389110	-0.00148716	0.00091101	0.00149510	-0.00000118	0.00000000
		-6.26391870	-0.00148645	0.00091058	0.00149439	-0.00000118	0.00000000
		-6.26394621	-0.00148575	0.00091016	0.00149368	-0.00000118	0.00000000
		-6.26397361	-0.00148504	0.00090973	0.00149297	-0.00000118	0.00000000
		-6.26400092	-0.00148434	0.00090930	0.00149226	-0.00000118	0.00000000
		-6.26402813	-0.00148363	0.00090887	0.00149155	-0.00000118	0.00000000
		-6.26405524	-0.00148293	0.00090844	0.00149084	-0.00000118	0.00000000
		-6.26408225	-0.00148223	0.00090802	0.00149014	-0.00000118	0.00000000
		-6.26410916	-0.00148152	0.00090759	0.00148943	-0.00000118	0.00000000
		-6.26413598	-0.00148082	0.00090717	0.00148872	-0.00000118	0.00000000
		-6.26416271	-0.00148012	0.00090674	0.00148802	-0.00000118	0.00000000
		-6.26418933	-0.00147942	0.00090631	0.00148732	-0.00000118	0.00000000
		-6.26421586	-0.00147872	0.00090589	0.00148661	-0.00000118	0.00000000
		-6.26424230	-0.00147802	0.00090547	0.00148591	-0.00000118	0.00000000
		-6.26426864	-0.00147733	0.00090504	0.00148521	-0.00000118	0.00000000
		-6.26429489	-0.00147663	0.00090462	0.00148450	-0.00000118	0.00000000
		-6.26432104	-0.00147593	0.00090419	0.00148380	-0.00000118	0.00000000
		-6.26434710	-0.00147523	0.00090377	0.00148310	-0.00000118	0.00000000
		-6.26437306	-0.00147454	0.00090335	0.00148240	-0.00000118	0.00000000
		-6.26439893	-0.00147384	0.00090293	0.00148170	-0.00000118	0.00000000
		-6.26442471	-0.00147315	0.00090250	0.00148101	-0.00000118	0.00000000
		-6.26445040	-0.00147246	0.00090208	0.00148031	-0.00000118	0.00000000
		-6.26447599	-0.00147176	0.00090166	0.00147961	-0.00000118	0.00000000
		-6.26450150	-0.00147107	0.00090124	0.00147891	-0.00000118	0.00000000
		-6.26452691	-0.00147038	0.00090082	0.00147822	-0.00000118	0.00000000
		-6.26455222	-0.00146969	0.00090040	0.00147752	-0.00000118	0.00000000
		-6.26457745	-0.00146900	0.00089998	0.00147683	-0.00000118	0.00000000
		-6.26460259	-0.00146831	0.00089956	0.00147613	-0.00000118	0.00000000
		-6.26462764	-0.00146762	0.00089914	0.00147544	-0.00000118	0.00000000
		-6.26465259	-0.00146693	0.00089872	0.00147475	-0.00000118	0.00000000
		-6.26467746	-0.00146624	0.00089831	0.00147406	-0.00000118	0.00000000
		-6.26470224	-0.00146555	0.00089789	0.00147336	-0.00000118	0.00000000
		-6.26472693	-0.00146486	0.00089747	0.00147267	-0.00000118	0.00000000
		-6.26475153	-0.00146418	0.00089705	0.00147198	-0.00000118	0.00000000
		-6.26477604	-0.00146349	0.00089664	0.00147129	-0.00000118	0.00000000
		-6.26480047	-0.00146281	0.00089622	0.00147060	-0.00000118	0.00000000
		-6.26482481	-0.00146212	0.00089580	0.00146992	-0.00000118	0.00000000
		-6.26484906	-0.00146144	0.00089539	0.00146923	-0.00000118	0.00000000
		-6.26487322	-0.00146076	0.00089497	0.00146854	-0.00000118	0.00000000
		-6.26489730	-0.00146007	0.00089456	0.00146785	-0.00000118	0.00000000
		-6.26492129	-0.00145939	0.00089414	0.00146717	-0.00000118	0.00000000
		-6.26494519	-0.00145871	0.00089373	0.00146648	-0.00000118	0.00000000
		-6.26496901	-0.00145803	0.00089332	0.00146580	-0.00000118	0.00000000
		-6.26499274	-0.00145735	0.00089290	0.00146511	-0.00000118	0.00000000
		-6.26501639	-0.00145667	0.00089249	0.00146443	-0.00000118	0.00000000
		-6.26503995	-0.00145599	0.00089208	0.00146375	-0.00000118	0.00000000
		-6.26506343	-0.00145531	0.00089166	0.00146307	-0.00000118	0.00000000
		-6.26508682	-0.00145463	0.00089125	0.00146238	-0.00000118	0.00000000
		-6.26511013	-0.00145396	0.00089084	0.00146170	-0.00000118	0.00000000
		-6.26513336	-0.00145328	0.00089043	0.00146102	-0.00000118	0.00000000
		-6.26515650	-0.00145261	0.00089002	0.00146034	-0.00000118	0.00000000
		-6.26517956	-0.00145193	0.00088961	0.00145966	-0.00000118	0.00000000
		-6.26520254	-0.00145126	0.00088920	0.00145898	-0.00000118	0.00000000
		-6.26522543	-0.00145058	0.00088879	0.00145831	-0.00000118	0.00000000
		-6.26524824	-0.00144991	0.00088838	0.00145763	-0.00000118	0.00000000
		-6.26527098	-0.00144924	0.00088797	0.00145695	-0.00000118	0.00000000
		-6.26529363	-0.00144856	0.00088756	0.00145628	-0.00000118	0.00000000
		-6.26531619	-0.00144789	0.00088715	0.00145560	-0.00000118	0.00000000
		-6.26533868	-0.00144722	0.00088674	0.00145493	-0.00000118	0.00000000
		-6.26536109	-0.00144655	0.00088634	0.00145425	-0.00000118	0.00000000
		-6.26538341	-0.00144588	0.00088593	0.00145358	-0.00000118	0.00000000
		-6.26540566	-0.00144521	0.00088552	0.00145291	-0.00000118	0.00000000
		-6.26542783	-0.00144454	0.00088511	0.00145223	-0.00000118	0.00000000
		-6.26544992	-0.00144387	0.00088471	0.00145156	-0.00000118	0.00000000
		-6.26547192	-0.00144321	0.00088430	0.00145089	-0.00000118	0.00000000
		-6.26549385	-0.00144254	0.00088390	0.00145022	-0.00000118	0.00000000
		-6.26551570	-0.00144187	0.00088349	0.00144955	-0.00000118	0.00000000
		-6.26553748	-0.00144121	0.00088309	0.00144888	-0.00000118	0.00000000
		-6.26555917	-0.00144054	0.00088268	0.00144821	-0.00000118	0.00000000
		-6.26558079	-0.00143988	0.00088228	0.00144754	-0.00000118	0.00000000
		-6.26560233	-0.00143922	0.00088187	0.00144687	-0.00000118	0.00000000
		-6.26562379	-0.00143855	0.00088147	0.00144621	-0.00000118	0.00000000
		-6.26564518	-0.00143789	0.00088107	0.00144554	-0.00000118	0.00000000
		-6.26566648	-0.00143723	0.00088066	0.00144488	-0.00000118	0.00000000
		-6.26568772	-0.00143657	0.00088026	0.00144421	-0.00000118	0.00000000
		-6.26570887	-0.00143590	0.00087986	0.00144355	-0.00000118	0.00000000
		-6.26572995	-0.00143524	0.00087946	0.00144288	-0.00000118	0.00000000
		-6.26575096	-0.00143458	0.00087906	0.00144222	-0.00000118	0.00000000
		-6.26577189	-0.00143393	0.00087865	0.00144155	-0.00000118	0.00000000
		-6.26579274	-0.00143327	0.00087825	0.00144089	-0.00000118	0.00000000
		-6.26581352	-0.00143261	0.00087785	0.00144023	-0.00000118	0.00000000
		-6.26583423	-0.00143195	0.00087745	0.00143957	-0.00000118	0.00000000
		-6.26585486	-0.00143130	0.00087705	0.00143891	-0.00000118	0.00000000
		-6.26587542	-0.00143064	0.00087665	0.00143825	-0.00000118	0.00000000
		-6.26589591	-0.00142998	0.00087626	0.00143759	-0.00000118	0.00000000
		-6.26591632	-0.00142933	0.00087586	0.00143693	-0.00000118	0.00000000
		-6.26593666	-0.00142867	0.00087546	0.00143627	-0.00000118	0.00000000
		-6.26595692	-0.00142802	0.00087506	0.00143562	-0.00000118	0.00000000
		-6.26597711	-0.00142737	0.00087466	0.00143496	-0.00000118	0.00000000
		-6.26599724	-0.00142671	0.00087426	0.00143430	-0.00000118	0.00000000
		-6.26601728	-0.00142606	0.00087387	0.00143365	-0.00000118	0.00000000
		-6.26603726	-0.00142541	0.00087347	0.00143299	-0.00000118	0.00000000
		-6.26605717	-0.00142476	0.00087307	0.00143234	-0.00000118	0.00000000
		-6.26607700	-0.00142411	0.00087268	0.00143168	-0.00000118	0.00000000
		-6.26609676	-0.00142346	0.00087228	0.00143103	-0.00000118	0.00000000
		-6.26611646	-0.00142281	0.00087189	0.00143038	-0.00000118	0.00000000
		-6.26613608	-0.00142216	0.00087149	0.00142972	-0.00000118	0.00000000
		-6.26615563	-0.00142151	0.00087110	0.00142907	-0.00000118	0.00000000
		-6.26617512	-0.00142087	0.00087070	0.00142842	-0.00000118	0.00000000
		-6.26619453	-0.00142022	0.00087031	0.00142777	-0.00000118	0.00000000
		-6.26621387	-0.00141957	0.00086992	0.00142712	-0.00000118	0.00000000
		-6.26623315	-0.00141893	0.00086952	0.00142647	-0.00000118	0.00000000
		-6.26625235	-0.00141828	0.00086913	0.00142582	-0.00000118	0.00000000
		-6.26627149	-0.00141764	0.00086874	0.00142517	-0.00000118	0.00000000
		-6.26629056	-0.00141699	0.00086835	0.00142452	-0.00000118	0.00000000
		-6.26630956	-0.00141635	0.00086795	0.00142388	-0.00000118	0.00000000
		-6.26632849	-0.00141571	0.00086756	0.00142323	-0.00000118	0.00000000
		-6.26634735	-0.00141506	0.00086717	0.00142258	-0.00000118	0.00000000
		-6.26636615	-0.00141442	0.00086678	0.00142194	-0.00000118	0.00000000
		-6.26638488	-0.00141378	0.00086639	0.00142129	-0.00000118	0.00000000
		-6.26640355	-0.00141314	0.00086600	0.00142065	-0.00000118	0.00000000
		-6.26642214	-0.00141250	0.00086561	0.00142001	-0.00000118	0.00000000
		-6.26644067	-0.00141186	0.00086522	0.00141936	-0.00000118	0.00000000
		-6.26645914	-0.00141122	0.00086483	0.00141872	-0.00000118	0.00000000
		-6.26647754	-0.00141058	0.00086444	0.00141808	-0.00000118	0.00000000
		-6.26649587	-0.00140995	0.00086405	0.00141744	-0.00000118	0.00000000
		-6.26651414	-0.00140931	0.00086366	0.00141680	-0.00000118	0.00000000
		-6.26653234	-0.00140867	0.00086328	0.00141616	-0.00000118	0.00000000
		-6.26655048	-0.00140804	0.00086289	0.00141552	-0.00000118	0.00000000
		-6.26656855	-0.00140740	0.00086250	0.00141488	-0.00000118	0.00000000
		-6.26658656	-0.00140677	0.00086211	0.00141424	-0.00000118	0.00000000
		-6.26660450	-0.00140613	0.00086173	0.00141360	-0.00000118	0.00000000
		-6.26662238	-0.00140550	0.00086134	0.00141296	-0.00000118	0.00000000
		-6.26664020	-0.00140486	0.00086096	0.00141233	-0.00000118	0.00000000
		-6.26665795	-0.00140423	0.00086057	0.00141169	-0.00000118	0.00000000
		-6.26667564	-0.00140360	0.00086018	0.00141105	-0.00000118	0.00000000
		-6.26669326	-0.00140297	0.00085980	0.00141042	-0.00000118	0.00000000
		-6.26671083	-0.00140234	0.00085942	0.00140978	-0.00000118	0.00000000
		-6.26672833	-0.00140171	0.00085903	0.00140915	-0.00000118	0.00000000
		-6.26674577	-0.00140108	0.00085865	0.00140852	-0.00000118	0.00000000
		-6.26676314	-0.00140045	0.00085826	0.00140788	-0.00000118	0.00000000
		-6.26678046	-0.00139982	0.00085788	0.00140725	-0.00000118	0.00000000
		-6.26679771	-0.00139919	0.00085750	0.00140662	-0.00000118	0.00000000
		-6.26681490	-0.00139856	0.00085711	0.00140599	-0.00000118	0.00000000
		-6.26683203	-0.00139793	0.00085673	0.00140536	-0.00000118	0.00000000
		-6.26684910	-0.00139731	0.00085635	0.00140472	-0.00000118	0.00000000
		-6.26686611	-0.00139668	0.00085597	0.00140409	-0.00000118	0.00000000
		-6.26688306	-0.00139605	0.00085559	0.00140347	-0.00000118	0.00000000
		-6.26689994	-0.00139543	0.00085521	0.00140284	-0.00000118	0.00000000
		-6.26691677	-0.00139481	0.00085482	0.00140221	-0.00000118	0.00000000
		-6.26693354	-0.00139418	0.00085444	0.00140158	-0.00000118	0.00000000
		-6.26695024	-0.00139356	0.00085406	0.00140095	-0.00000118	0.00000000
		-6.26696689	-0.00139293	0.00085368	0.00140033	-0.00000118	0.00000000
		-6.26698348	-0.00139231	0.00085331	0.00139970	-0.00000118	0.00000000
		-6.26700001	-0.00139169	0.00085293	0.00139908	-0.00000118	0.00000000
		-6.26701648	-0.00139107	0.00085255	0.00139845	-0.00000118	0.00000000
		-6.26703289	-0.00139045	0.00085217	0.00139783	-0.00000118	0.00000000
		-6.26704925	-0.00138983	0.00085179	0.00139720	-0.00000118	0.00000000
		-6.26706554	-0.00138921	0.00085141	0.00139658	-0.00000118	0.00000000
		-6.26708178	-0.00138859	0.00085104	0.00139596	-0.00000118	0.00000000
		-6.26709796	-0.00138797	0.00085066	0.00139534	-0.00000118	0.00000000
		-6.26711408	-0.00138735	0.00085028	0.00139471	-0.00000118	0.00000000
		-6.26713015	-0.00138674	0.00084990	0.00139409	-0.00000118	0.00000000
		-6.26714616	-0.00138612	0.00084953	0.00139347	-0.00000118	0.00000000
		-6.26716211	-0.00138550	0.00084915	0.00139285	-0.00000118	0.00000000
		-6.26717800	-0.00138489	0.00084878	0.00139223	-0.00000118	0.00000000
		-6.26719384	-0.00138427	0.00084840	0.00139161	-0.00000118	0.00000000
		-6.26720962	-0.00138366	0.00084803	0.00139100	-0.00000118	0.00000000
		-6.26722535	-0.00138304	0.00084765	0.00139038	-0.00000118	0.00000000
		-6.26724102	-0.00138243	0.00084728	0.00138976	-0.00000118	0.00000000
		-6.26725663	-0.00138181	0.00084690	0.00138914	-0.00000118	0.00000000
		-6.26727219	-0.00138120	0.00084653	0.00138853	-0.00000118	0.00000000
		-6.26728769	-0.00138059	0.00084616	0.00138791	-0.00000118	0.00000000
		-6.26730314	-0.00137998	0.00084578	0.00138730	-0.00000118	0.00000000
		-6.26731854	-0.00137937	0.00084541	0.00138668	-0.00000118	0.00000000
		-6.26733388	-0.00137876	0.00084504	0.00138607	-0.00000118	0.00000000
		-6.26734916	-0.00137815	0.00084467	0.00138545	-0.00000118	0.00000000
		-6.26736439	-0.00137754	0.00084429	0.00138484	-0.00000118	0.00000000
		-6.26737957	-0.00137693	0.00084392	0.00138423	-0.00000118	0.00000000
		-6.26739469	-0.00137632	0.00084355	0.00138362	-0.00000118	0.00000000
		-6.26740976	-0.00137571	0.00084318	0.00138301	-0.00000118	0.00000000
		-6.26742477	-0.00137510	0.00084281	0.00138239	-0.00000118	0.00000000
		-6.26743974	-0.00137450	0.00084244	0.00138178	-0.00000118	0.00000000
		-6.26745464	-0.00137389	0.00084207	0.00138117	-0.00000118	0.00000000
		-6.26746950	-0.00137328	0.00084170	0.00138056	-0.00000118	0.00000000
		-6.26748430	-0.00137268	0.00084133	0.00137996	-0.00000117	0.00000000
		-6.26749906	-0.00137207	0.00084096	0.00137935	-0.00000117	0.00000000
		-6.26751375	-0.00137147	0.00084059	0.00137874	-0.00000117	0.00000000
		-6.26752840	-0.00137087	0.00084022	0.00137813	-0.00000117	0.00000000
		-6.26754300	-0.00137026	0.00083986	0.00137753	-0.00000117	0.00000000
		-6.26755754	-0.00136966	0.00083949	0.00137692	-0.00000117	0.00000000
		-6.26757203	-0.00136906	0.00083912	0.00137631	-0.00000117	0.00000000
		-6.26758647	-0.00136846	0.00083875	0.00137571	-0.00000117	0.00000000
		-6.26760086	-0.00136786	0.00083839	0.00137510	-0.00000117	0.00000000
		-6.26761520	-0.00136725	0.00083802	0.00137450	-0.00000117	0.00000000
		-6.26762949	-0.00136665	0.00083765	0.00137390	-0.00000117	0.00000000
		-6.26764372	-0.00136605	0.00083729	0.00137329	-0.00000117	0.00000000
		-6.26765791	-0.00136546	0.00083692	0.00137269	-0.00000117	0.00000000
		-6.26767205	-0.00136486	0.00083656	0.00137209	-0.00000117	0.00000000
		-6.26768613	-0.00136426	0.00083619	0.00137149	-0.00000117	0.00000000
		-6.26770017	-0.00136366	0.00083583	0.00137089	-0.00000117	0.00000000
		-6.26771416	-0.00136306	0.00083546	0.00137029	-0.00000117	0.00000000
		-6.26772809	-0.00136247	0.00083510	0.00136969	-0.00000117	0.00000000
		-6.26774198	-0.00136187	0.00083474	0.00136909	-0.00000117	0.00000000
		-6.26775582	-0.00136128	0.00083437	0.00136849	-0.00000117	0.00000000
		-6.26776961	-0.00136068	0.00083401	0.00136789	-0.00000117	0.00000000
		-6.26778335	-0.00136009	0.00083365	0.00136729	-0.00000117	0.00000000
		-6.26779704	-0.00135949	0.00083328	0.00136669	-0.00000117	0.00000000
		-6.26781069	-0.00135890	0.00083292	0.00136610	-0.00000117	0.00000000
		-6.26782428	-0.00135831	0.00083256	0.00136550	-0.00000117	0.00000000
		-6.26783783	-0.00135771	0.00083220	0.00136490	-0.00000117	0.00000000
		-6.26785133	-0.00135712	0.00083184	0.00136431	-0.00000117	0.00000000
		-6.26786479	-0.00135653	0.00083148	0.00136371	-0.00000117	0.00000000
		-6.26787819	-0.00135594	0.00083111	0.00136312	-0.00000117	0.00000000
		-6.26789155	-0.00135535	0.00083075	0.00136253	-0.00000117	0.00000000
		-6.26790486	-0.00135476	0.00083039	0.00136193	-0.00000117	0.00000000
		-6.26791812	-0.00135417	0.00083003	0.00136134	-0.00000117	0.00000000
		-6.26793134	-0.00135358	0.00082967	0.00136075	-0.00000117	0.00000000
		-6.26794451	-0.00135299	0.00082931	0.00136016	-0.00000117	0.00000000
		-6.26795764	-0.00135240	0.00082896	0.00135956	-0.00000117	0.00000000
		-6.26797071	-0.00135182	0.00082860	0.00135897	-0.00000117	0.00000000
		-6.26798374	-0.00135123	0.00082824	0.00135838	-0.00000117	0.00000000
		-6.26799673	-0.00135064	0.00082788	0.00135779	-0.00000117	0.00000000
		-6.26800967	-0.00135006	0.00082752	0.00135720	-0.00000117	0.00000000
		-6.26802256	-0.00134947	0.00082717	0.00135662	-0.00000117	0.00000000
		-6.26803541	-0.00134889	0.00082681	0.00135603	-0.00000117	0.00000000
		-6.26804822	-0.00134830	0.00082645	0.00135544	-0.00000117	0.00000000
		-6.26806098	-0.00134772	0.00082609	0.00135485	-0.00000117	0.00000000
		-6.26807369	-0.00134713	0.00082574	0.00135427	-0.00000117	0.00000000
		-6.26808636	-0.00134655	0.00082538	0.00135368	-0.00000117	0.00000000
		-6.26809898	-0.00134597	0.00082503	0.00135309	-0.00000117	0.00000000
		-6.26811156	-0.00134539	0.00082467	0.00135251	-0.00000117	0.00000000
		-6.26812410	-0.00134481	0.00082432	0.00135192	-0.00000117	0.00000000
		-6.26813659	-0.00134422	0.00082396	0.00135134	-0.00000117	0.00000000
		-6.26814904	-0.00134364	0.00082361	0.00135075	-0.00000117	0.00000000
		-6.26816144	-0.00134306	0.00082325	0.00135017	-0.00000117	0.00000000
		-6.26817380	-0.00134248	0.00082290	0.00134959	-0.00000117	0.00000000
		-6.26818612	-0.00134191	0.00082255	0.00134901	-0.00000117	0.00000000
		-6.26819839	-0.00134133	0.00082219	0.00134842	-0.00000117	0.00000000
		-6.26821062	-0.00134075	0.00082184	0.00134784	-0.00000117	0.00000000
		-6.26822281	-0.00134017	0.00082149	0.00134726	-0.00000117	0.00000000
		-6.26823495	-0.00133959	0.00082113	0.00134668	-0.00000117	0.00000000
		-6.26824705	-0.00133902	0.00082078	0.00134610	-0.00000117	0.00000000
		-6.26825911	-0.00133844	0.00082043	0.00134552	-0.00000117	0.00000000
		-6.26827113	-0.00133787	0.00082008	0.00134494	-0.00000117	0.00000000
		-6.26828310	-0.00133729	0.00081973	0.00134437	-0.00000117	0.00000000
		-6.26829504	-0.00133672	0.00081938	0.00134379	-0.00000117	0.00000000
		-6.26830693	-0.00133614	0.00081902	0.00134321	-0.00000117	0.00000000
		-6.26831878	-0.00133557	0.00081867	0.00134263	-0.00000117	0.00000000
		-6.26833058	-0.00133499	0.00081832	0.00134206	-0.00000117	0.00000000
		-6.26834235	-0.00133442	0.00081797	0.00134148	-0.00000117	0.00000000
		-6.26835407	-0.00133385	0.00081762	0.00134090	-0.00000117	0.00000000
		-6.26836576	-0.00133328	0.00081728	0.00134033	-0.00000117	0.00000000
		-6.26837740	-0.00133271	0.00081693	0.00133976	-0.00000117	0.00000000
		-6.26838900	-0.00133214	0.00081658	0.00133918	-0.00000117	0.00000000
		-6.26840056	-0.00133157	0.00081623	0.00133861	-0.00000117	0.00000000
		-6.26841209	-0.00133100	0.00081588	0.00133803	-0.00000117	0.00000000
		-6.26842357	-0.00133043	0.00081553	0.00133746	-0.00000117	0.00000000
		-6.26843501	-0.00132986	0.00081519	0.00133689	-0.00000117	0.00000000
		-6.26844641	-0.00132929	0.00081484	0.00133632	-0.00000117	0.00000000
		-6.26845777	-0.00132872	0.00081449	0.00133575	-0.00000117	0.00000000
		-6.26846909	-0.00132815	0.00081414	0.00133518	-0.00000117	0.00000000
		-6.26848037	-0.00132759	0.00081380	0.00133461	-0.00000117	0.00000000
		-6.26849161	-0.00132702	0.00081345	0.00133404	-0.00000117	0.00000000
		-6.26850281	-0.00132645	0.00081311	0.00133347	-0.00000117	0.00000000
		-6.26851397	-0.00132589	0.00081276	0.00133290	-0.00000117	0.00000000
		-6.26852510	-0.00132532	0.00081242	0.00133233	-0.00000117	0.00000000
		-6.26853618	-0.00132476	0.00081207	0.00133176	-0.00000117	0.00000000
		-6.26854723	-0.00132419	0.00081173	0.00133120	-0.00000117	0.00000000
		-6.26855824	-0.00132363	0.00081138	0.00133063	-0.00000117	0.00000000
		-6.26856921	-0.00132307	0.00081104	0.00133006	-0.00000117	0.00000000
		-6.26858014	-0.00132251	0.00081069	0.00132950	-0.00000117	0.00000000
		-6.26859103	-0.00132194	0.00081035	0.00132893	-0.00000117	0.00000000
		-6.26860188	-0.00132138	0.00081001	0.00132837	-0.00000117	0.00000000
		-6.26861270	-0.00132082	0.00080966	0.00132780	-0.00000117	0.00000000
		-6.26862348	-0.00132026	0.00080932	0.00132724	-0.00000117	0.00000000
		-6.26863422	-0.00131970	0.00080898	0.00132667	-0.00000117	0.00000000
		-6.26864493	-0.00131914	0.00080864	0.00132611	-0.00000117	0.00000000
		-6.26865559	-0.00131858	0.00080829	0.00132555	-0.00000117	0.00000000
		-6.26866622	-0.00131802	0.00080795	0.00132499	-0.00000117	0.00000000
		-6.26867681	-0.00131746	0.00080761	0.00132442	-0.00000117	0.00000000
		-6.26868737	-0.00131690	0.00080727	0.00132386	-0.00000117	0.00000000
		-6.26869789	-0.00131635	0.00080693	0.00132330	-0.00000117	0.00000000
		-6.26870837	-0.00131579	0.00080659	0.00132274	-0.00000117	0.00000000
		-6.26871882	-0.00131523	0.00080625	0.00132218	-0.00000117	0.00000000
		-6.26872923	-0.00131468	0.00080591	0.00132162	-0.00000117	0.00000000
		-6.26873960	-0.00131412	0.00080557	0.00132106	-0.00000117	0.00000000
		-6.26874993	-0.00131357	0.00080523	0.00132050	-0.00000117	0.00000000
		-6.26876024	-0.00131301	0.00080489	0.00131995	-0.00000117	0.00000000
		-6.26877050	-0.00131246	0.00080455	0.00131939	-0.00000117	0.00000000
		-6.26878073	-0.00131190	0.00080421	0.00131883	-0.00000117	0.00000000
		-6.26879092	-0.00131135	0.00080387	0.00131828	-0.00000117	0.00000000
		-6.26880108	-0.00131080	0.00080354	0.00131772	-0.00000117	0.00000000
		-6.26881120	-0.00131024	0.00080320	0.00131716	-0.00000117	0.00000000
		-6.26882129	-0.00130969	0.00080286	0.00131661	-0.00000117	0.00000000
		-6.26883135	-0.00130914	0.00080252	0.00131605	-0.00000117	0.00000000
		-6.26884136	-0.00130859	0.00080219	0.00131550	-0.00000117	0.00000000
		-6.26885135	-0.00130804	0.00080185	0.00131495	-0.00000117	0.00000000
		-6.26886129	-0.00130749	0.00080151	0.00131439	-0.00000117	0.00000000
		-6.26887121	-0.00130694	0.00080118	0.00131384	-0.00000117	0.00000000
		-6.26888109	-0.00130639	0.00080084	0.00131329	-0.00000117	0.00000000
		-6.26889093	-0.00130584	0.00080051	0.00131274	-0.00000117	0.00000000
		-6.26890074	-0.00130529	0.00080017	0.00131218	-0.00000117	0.00000000
		-6.26891052	-0.00130474	0.00079984	0.00131163	-0.00000117	0.00000000
		-6.26892026	-0.00130420	0.00079950	0.00131108	-0.00000117	0.00000000
		-6.26892997	-0.00130365	0.00079917	0.00131053	-0.00000117	0.00000000
		-6.26893965	-0.00130310	0.00079883	0.00130998	-0.00000117	0.00000000
		-6.26894929	-0.00130256	0.00079850	0.00130943	-0.00000117	0.00000000
		-6.26895890	-0.00130201	0.00079816	0.00130888	-0.00000117	0.00000000
		-6.26896848	-0.00130146	0.00079783	0.00130834	-0.00000117	0.00000000
		-6.26897802	-0.00130092	0.00079750	0.00130779	-0.00000117	0.00000000
		-6.26898753	-0.00130038	0.00079717	0.00130724	-0.00000117	0.00000000
		-6.26899700	-0.00129983	0.00079683	0.00130669	-0.00000117	0.00000000
		-6.26900645	-0.00129929	0.00079650	0.00130615	-0.00000117	0.00000000
		-6.26901586	-0.00129875	0.00079617	0.00130560	-0.00000117	0.00000000
		-6.26902524	-0.00129820	0.00079584	0.00130506	-0.00000117	0.00000000
		-6.26903458	-0.00129766	0.00079550	0.00130451	-0.00000117	0.00000000
		-6.26904390	-0.00129712	0.00079517	0.00130397	-0.00000117	0.00000000
		-6.26905318	-0.00129658	0.00079484	0.00130342	-0.00000117	0.00000000
		-6.26906243	-0.00129604	0.00079451	0.00130288	-0.00000117	0.00000000
		-6.26907164	-0.00129550	0.00079418	0.00130233	-0.00000117	0.00000000
		-6.26908083	-0.00129496	0.00079385	0.00130179	-0.00000117	0.00000000
		-6.26908998	-0.00129442	0.00079352	0.00130125	-0.00000117	0.00000000
		-6.26909911	-0.00129388	0.00079319	0.00130071	-0.00000117	0.00000000
		-6.26910820	-0.00129334	0.00079286	0.00130016	-0.00000117	0.00000000
		-6.26911726	-0.00129280	0.00079253	0.00129962	-0.00000117	0.00000000
		-6.26912629	-0.00129226	0.00079220	0.00129908	-0.00000117	0.00000000
		-6.26913528	-0.00129173	0.00079188	0.00129854	-0.00000117	0.00000000
		-6.26914425	-0.00129119	0.00079155	0.00129800	-0.00000117	0.00000000
		-6.26915318	-0.00129065	0.00079122	0.00129746	-0.00000117	0.00000000
		-6.26916209	-0.00129012	0.00079089	0.00129692	-0.00000117	0.00000000
		-6.26917096	-0.00128958	0.00079056	0.00129639	-0.00000117	0.00000000
		-6.26917981	-0.00128905	0.00079024	0.00129585	-0.00000117	0.00000000
		-6.26918862	-0.00128851	0.00078991	0.00129531	-0.00000117	0.00000000
		-6.26919740	-0.00128798	0.00078958	0.00129477	-0.00000117	0.00000000
		-6.26920615	-0.00128744	0.00078926	0.00129424	-0.00000117	0.00000000
		-6.26921488	-0.00128691	0.00078893	0.00129370	-0.00000117	0.00000000
		-6.26922357	-0.00128638	0.00078860	0.00129316	-0.00000117	0.00000000
		-6.26923223	-0.00128585	0.00078828	0.00129263	-0.00000117	0.00000000
		-6.26924086	-0.00128531	0.00078795	0.00129209	-0.00000117	0.00000000
		-6.26924947	-0.00128478	0.00078763	0.00129156	-0.00000117	0.00000000
		-6.26925804	-0.00128425	0.00078730	0.00129102	-0.00000117	0.00000000
		-6.26926658	-0.00128372	0.00078698	0.00129049	-0.00000117	0.00000000
		-6.26927510	-0.00128319	0.00078665	0.00128996	-0.00000117	0.00000000
		-6.26928358	-0.00128266	0.00078633	0.00128942	-0.00000117	0.00000000
		-6.26929204	-0.00128213	0.00078601	0.00128889	-0.00000117	0.00000000
		-6.26930047	-0.00128160	0.00078568	0.00128836	-0.00000117	0.00000000
		-6.26930887	-0.00128107	0.00078536	0.00128783	-0.00000117	0.00000000
		-6.26931724	-0.00128054	0.00078504	0.00128730	-0.00000117	0.00000000
		-6.26932558	-0.00128002	0.00078471	0.00128677	-0.00000117	0.00000000
		-6.26933389	-0.00127949	0.00078439	0.00128624	-0.00000117	0.00000000
		-6.26934217	-0.00127896	0.00078407	0.00128571	-0.00000117	0.00000000
		-6.26935043	-0.00127844	0.00078375	0.00128518	-0.00000117	0.00000000
		-6.26935866	-0.00127791	0.00078342	0.00128465	-0.00000117	0.00000000
		-6.26936686	-0.00127738	0.00078310	0.00128412	-0.00000117	0.00000000
		-6.26937503	-0.00127686	0.00078278	0.00128359	-0.00000117	0.00000000
		-6.26938317	-0.00127634	0.00078246	0.00128306	-0.00000117	0.00000000
		-6.26939129	-0.00127581	0.00078214	0.00128254	-0.00000117	0.00000000
		-6.26939937	-0.00127529	0.00078182	0.00128201	-0.00000117	0.00000000
		-6.26940743	-0.00127476	0.00078150	0.00128148	-0.00000117	0.00000000
		-6.26941547	-0.00127424	0.00078118	0.00128096	-0.00000117	0.00000000
		-6.26942347	-0.00127372	0.00078086	0.00128043	-0.00000117	0.00000000
		-6.26943145	-0.00127320	0.00078054	0.00127991	-0.00000117	0.00000000
		-6.26943940	-0.00127267	0.00078022	0.00127938	-0.00000117	0.00000000
		-6.26944732	-0.00127215	0.00077990	0.00127886	-0.00000117	0.00000000
		-6.26945522	-0.00127163	0.00077958	0.00127833	-0.00000117	0.00000000
		-6.26946309	-0.00127111	0.00077926	0.00127781	-0.00000117	0.00000000
		-6.26947093	-0.00127059	0.00077895	0.00127729	-0.00000117	0.00000000
		-6.26947875	-0.00127007	0.00077863	0.00127677	-0.00000117	0.00000000
		-6.26948654	-0.00126955	0.00077831	0.00127624	-0.00000117	0.00000000
		-6.26949430	-0.00126903	0.00077799	0.00127572	-0.00000117	0.00000000
		-6.26950204	-0.00126852	0.00077768	0.00127520	-0.00000117	0.00000000
		-6.26950975	-0.00126800	0.00077736	0.00127468	-0.00000117	0.00000000
		-6.26951743	-0.00126748	0.00077704	0.00127416	-0.00000117	0.00000000
		-6.26952509	-0.00126696	0.00077673	0.00127364	-0.00000117	0.00000000
		-6.26953272	-0.00126645	0.00077641	0.00127312	-0.00000117	0.00000000
		-6.26954033	-0.00126593	0.00077609	0.00127260	-0.00000117	0.00000000
		-6.26954791	-0.00126541	0.00077578	0.00127208	-0.00000117	0.00000000
		-6.26955546	-0.00126490	0.00077546	0.00127156	-0.00000117	0.00000000
		-6.26956299	-0.00126438	0.00077515	0.00127104	-0.00000117	0.00000000
		-6.26957049	-0.00126387	0.00077483	0.00127053	-0.00000117	0.00000000
		-6.26957797	-0.00126335	0.00077452	0.00127001	-0.00000117	0.00000000
		-6.26958542	-0.00126284	0.00077420	0.00126949	-0.00000117	0.00000000
		-6.26959285	-0.00126233	0.00077389	0.00126898	-0.00000117	0.00000000
		-6.26960025	-0.00126181	0.00077357	0.00126846	-0.00000117	0.00000000
		-6.26960763	-0.00126130	0.00077326	0.00126795	-0.00000117	0.00000000
		-6.26961498	-0.00126079	0.00077295	0.00126743	-0.00000117	0.00000000
		-6.26962230	-0.00126028	0.00077263	0.00126692	-0.00000117	0.00000000
		-6.26962961	-0.00125977	0.00077232	0.00126640	-0.00000117	0.00000000
		-6.26963688	-0.00125926	0.00077201	0.00126589	-0.00000117	0.00000000
		-6.26964414	-0.00125874	0.00077170	0.00126537	-0.00000117	0.00000000
		-6.26965137	-0.00125823	0.00077138	0.00126486	-0.00000117	0.00000000
		-6.26965857	-0.00125772	0.00077107	0.00126435	-0.00000117	0.00000000
		-6.26966575	-0.00125722	0.00077076	0.00126384	-0.00000117	0.00000000
		-6.26967290	-0.00125671	0.00077045	0.00126333	-0.00000117	0.00000000
		-6.26968003	-0.00125620	0.00077014	0.00126281	-0.00000117	0.00000000
		-6.26968714	-0.00125569	0.00076983	0.00126230	-0.00000117	0.00000000
		-6.26969422	-0.00125518	0.00076952	0.00126179	-0.00000117	0.00000000
		-6.26970128	-0.00125468	0.00076920	0.00126128	-0.00000117	0.00000000
		-6.26970832	-0.00125417	0.00076889	0.00126077	-0.00000117	0.00000000
		-6.26971533	-0.00125366	0.00076858	0.00126026	-0.00000117	0.00000000
		-6.26972232	-0.00125316	0.00076827	0.00125975	-0.00000117	0.00000000
		-6.26972928	-0.00125265	0.00076797	0.00125925	-0.00000117	0.00000000
		-6.26973622	-0.00125214	0.00076766	0.00125874	-0.00000117	0.00000000
		-6.26974314	-0.00125164	0.00076735	0.00125823	-0.00000117	0.00000000
		-6.26975003	-0.00125114	0.00076704	0.00125772	-0.00000117	0.00000000
		-6.26975691	-0.00125063	0.00076673	0.00125722	-0.00000117	0.00000000
		-6.26976375	-0.00125013	0.00076642	0.00125671	-0.00000117	0.00000000
		-6.26977058	-0.00124962	0.00076611	0.00125620	-0.00000117	0.00000000
		-6.26977738	-0.00124912	0.00076581	0.00125570	-0.00000117	0.00000000
		-6.26978416	-0.00124862	0.00076550	0.00125519	-0.00000117	0.00000000
		-6.26979092	-0.00124812	0.00076519	0.00125469	-0.00000117	0.00000000
		-6.26979765	-0.00124762	0.00076488	0.00125418	-0.00000117	0.00000000
		-6.26980436	-0.00124711	0.00076458	0.00125368	-0.00000117	0.00000000
		-6.26981105	-0.00124661	0.00076427	0.00125318	-0.00000117	0.00000000
		-6.26981772	-0.00124611	0.00076396	0.00125267	-0.00000117	0.00000000
		-6.26982436	-0.00124561	0.00076366	0.00125217	-0.00000117	0.00000000
		-6.26983098	-0.00124511	0.00076335	0.00125167	-0.00000117	0.00000000
		-6.26983758	-0.00124461	0.00076305	0.00125116	-0.00000117	0.00000000
		-6.26984416	-0.00124412	0.00076274	0.00125066	-0.00000117	0.00000000
		-6.26985071	-0.00124362	0.00076244	0.00125016	-0.00000117	0.00000000
		-6.26985725	-0.00124312	0.00076213	0.00124966	-0.00000117	0.00000000
		-6.26986376	-0.00124262	0.00076183	0.00124916	-0.00000117	0.00000000
		-6.26987025	-0.00124212	0.00076152	0.00124866	-0.00000117	0.00000000
		-6.26987672	-0.00124163	0.00076122	0.00124816	-0.00000117	0.00000000
		-6.26988316	-0.00124113	0.00076091	0.00124766	-0.00000117	0.00000000
		-6.26988959	-0.00124063	0.00076061	0.00124716	-0.00000117	0.00000000
		-6.26989599	-0.00124014	0.00076031	0.00124666	-0.00000117	0.00000000
		-6.26990237	-0.00123964	0.00076000	0.00124617	-0.00000117	0.00000000
		-6.26990873	-0.00123915	0.00075970	0.00124567	-0.00000117	0.00000000
		-6.26991507	-0.00123865	0.00075940	0.00124517	-0.00000117	0.00000000
		-6.26992139	-0.00123816	0.00075909	0.00124467	-0.00000117	0.00000000
		-6.26992769	-0.00123767	0.00075879	0.00124418	-0.00000117	0.00000000
		-6.26993396	-0.00123717	0.00075849	0.00124368	-0.00000117	0.00000000
		-6.26994022	-0.00123668	0.00075819	0.00124319	-0.00000117	0.00000000
		-6.26994645	-0.00123619	0.00075789	0.00124269	-0.00000117	0.00000000
		-6.26995267	-0.00123569	0.00075758	0.00124219	-0.00000117	0.00000000
		-6.26995886	-0.00123520	0.00075728	0.00124170	-0.00000117	0.00000000
		-6.26996503	-0.00123471	0.00075698	0.00124121	-0.00000117	0.00000000
		-6.26997119	-0.00123422	0.00075668	0.00124071	-0.00000117	0.00000000
		-6.26997732	-0.00123373	0.00075638	0.00124022	-0.00000117	0.00000000
		-6.26998343	-0.00123324	0.00075608	0.00123973	-0.00000117	0.00000000
		-6.26998952	-0.00123275	0.00075578	0.00123923	-0.00000117	0.00000000
		-6.26999559	-0.00123226	0.00075548	0.00123874	-0.00000117	0.00000000
		-6.27000164	-0.00123177	0.00075518	0.00123825	-0.00000117	0.00000000
		-6.27000767	-0.00123128	0.00075488	0.00123776	-0.00000117	0.00000000
		-6.27001368	-0.00123079	0.00075458	0.00123727	-0.00000117	0.00000000
		-6.27001968	-0.00123031	0.00075428	0.00123678	-0.00000117	0.00000000
		-6.27002565	-0.00122982	0.00075399	0.00123629	-0.00000117	0.00000000
		-6.27003160	-0.00122933	0.00075369	0.00123580	-0.00000117	0.00000000
		-6.27003753	-0.00122884	0.00075339	0.00123531	-0.00000117	0.00000000
		-6.27004344	-0.00122836	0.00075309	0.00123482	-0.00000117	0.00000000
		-6.27004933	-0.00122787	0.00075279	0.00123433	-0.00000117	0.00000000
		-6.27005521	-0.00122739	0.00075250	0.00123384	-0.00000117	0.00000000
		-6.27006106	-0.00122690	0.00075220	0.00123335	-0.00000117	0.00000000
		-6.27006690	-0.00122642	0.00075190	0.00123286	-0.00000117	0.00000000
		-6.27007271	-0.00122593	0.00075160	0.00123238	-0.00000117	0.00000000
		-6.27007851	-0.00122545	0.00075131	0.00123189	-0.00000117	0.00000000
		-6.27008428	-0.00122496	0.00075101	0.00123140	-0.00000117	0.00000000
		-6.27009004	-0.00122448	0.00075072	0.00123092	-0.00000117	0.00000000
		-6.27009578	-0.00122400	0.00075042	0.00123043	-0.00000117	0.00000000
		-6.27010150	-0.00122351	0.00075012	0.00122995	-0.00000117	0.00000000
		-6.27010720	-0.00122303	0.00074983	0.00122946	-0.00000117	0.00000000
		-6.27011288	-0.00122255	0.00074953	0.00122898	-0.00000117	0.00000000
		-6.27011855	-0.00122207	0.00074924	0.00122849	-0.00000117	0.00000000
		-6.27012419	-0.00122159	0.00074894	0.00122801	-0.00000117	0.00000000
		-6.27012982	-0.00122111	0.00074865	0.00122752	-0.00000117	0.00000000
		-6.27013543	-0.00122063	0.00074836	0.00122704	-0.00000117	0.00000000
		-6.27014102	-0.00122015	0.00074806	0.00122656	-0.00000117	0.00000000
		-6.27014659	-0.00121967	0.00074777	0.00122608	-0.00000117	0.00000000
		-6.27015214	-0.00121919	0.00074747	0.00122559	-0.00000117	0.00000000
		-6.27015767	-0.00121871	0.00074718	0.00122511	-0.00000117	0.00000000
		-6.27016319	-0.00121823	0.00074689	0.00122463	-0.00000117	0.00000000
		-6.27016869	-0.00121775	0.00074659	0.00122415	-0.00000117	0.00000000
		-6.27017417	-0.00121727	0.00074630	0.00122367	-0.00000117	0.00000000
		-6.27017963	-0.00121680	0.00074601	0.00122319	-0.00000117	0.00000000
		-6.27018508	-0.00121632	0.00074572	0.00122271	-0.00000117	0.00000000
		-6.27019050	-0.00121584	0.00074543	0.00122223	-0.00000117	0.00000000
		-6.27019591	-0.00121537	0.00074513	0.00122175	-0.00000117	0.00000000
		-6.27020130	-0.00121489	0.00074484	0.00122127	-0.00000117	0.00000000
		-6.27020668	-0.00121441	0.00074455	0.00122080	-0.00000117	0.00000000
		-6.27021203	-0.00121394	0.00074426	0.00122032	-0.00000117	0.00000000
		-6.27021737	-0.00121346	0.00074397	0.00121984	-0.00000117	0.00000000
		-6.27022270	-0.00121299	0.00074368	0.00121936	-0.00000117	0.00000000
		-6.27022800	-0.00121252	0.00074339	0.00121889	-0.00000117	0.00000000
		-6.27023329	-0.00121204	0.00074310	0.00121841	-0.00000117	0.00000000
		-6.27023856	-0.00121157	0.00074281	0.00121793	-0.00000117	0.00000000
		-6.27024381	-0.00121110	0.00074252	0.00121746	-0.00000117	0.00000000
		-6.27024904	-0.00121062	0.00074223	0.00121698	-0.00000117	0.00000000
		-6.27025426	-0.00121015	0.00074194	0.00121651	-0.00000117	0.00000000
		-6.27025946	-0.00120968	0.00074165	0.00121603	-0.00000117	0.00000000
		-6.27026465	-0.00120921	0.00074136	0.00121556	-0.00000117	0.00000000
		-6.27026982	-0.00120874	0.00074107	0.00121509	-0.00000117	0.00000000
		-6.27027497	-0.00120827	0.00074078	0.00121461	-0.00000117	0.00000000
		-6.27028010	-0.00120780	0.00074050	0.00121414	-0.00000117	0.00000000
		-6.27028522	-0.00120732	0.00074021	0.00121367	-0.00000116	0.00000000
		-6.27029032	-0.00120686	0.00073992	0.00121319	-0.00000116	0.00000000
		-6.27029541	-0.00120639	0.00073963	0.00121272	-0.00000116	0.00000000
		-6.27030047	-0.00120592	0.00073934	0.00121225	-0.00000116	0.00000000
		-6.27030552	-0.00120545	0.00073906	0.00121178	-0.00000116	0.00000000
		-6.27031056	-0.00120498	0.00073877	0.00121131	-0.00000116	0.00000000
		-6.27031558	-0.00120451	0.00073848	0.00121084	-0.00000116	0.00000000
		-6.27032058	-0.00120404	0.00073820	0.00121037	-0.00000116	0.00000000
		-6.27032557	-0.00120358	0.00073791	0.00120990	-0.00000116	0.00000000
		-6.27033054	-0.00120311	0.00073763	0.00120943	-0.00000116	0.00000000
		-6.27033549	-0.00120264	0.00073734	0.00120896	-0.00000116	0.00000000
		-6.27034043	-0.00120218	0.00073705	0.00120849	-0.00000116	0.00000000
		-6.27034536	-0.00120171	0.00073677	0.00120802	-0.00000116	0.00000000
		-6.27035026	-0.00120125	0.00073648	0.00120755	-0.00000116	0.00000000
		-6.27035515	-0.00120078	0.00073620	0.00120709	-0.00000116	0.00000000
		-6.27036003	-0.00120032	0.00073591	0.00120662	-0.00000116	0.00000000
		-6.27036489	-0.00119985	0.00073563	0.00120615	-0.00000116	0.00000000
		-6.27036973	-0.00119939	0.00073535	0.00120569	-0.00000116	0.00000000
		-6.27037456	-0.00119892	0.00073506	0.00120522	-0.00000116	0.00000000
		-6.27037937	-0.00119846	0.00073478	0.00120475	-0.00000116	0.00000000
		-6.27038417	-0.00119800	0.00073449	0.00120429	-0.00000116	0.00000000
		-6.27038895	-0.00119754	0.00073421	0.00120382	-0.00000116	0.00000000
		-6.27039372	-0.00119707	0.00073393	0.00120336	-0.00000116	0.00000000
		-6.27039847	-0.00119661	0.00073364	0.00120289	-0.00000116	0.00000000
		-6.27040321	-0.00119615	0.00073336	0.00120243	-0.00000116	0.00000000
		-6.27040793	-0.00119569	0.00073308	0.00120197	-0.00000116	0.00000000
		-6.27041263	-0.00119523	0.00073280	0.00120150	-0.00000116	0.00000000
		-6.27041732	-0.00119477	0.00073251	0.00120104	-0.00000116	0.00000000
		-6.27042200	-0.00119431	0.00073223	0.00120058	-0.00000116	0.00000000
		-6.27042666	-0.00119385	0.00073195	0.00120011	-0.00000116	0.00000000
		-6.27043131	-0.00119339	0.00073167	0.00119965	-0.00000116	0.00000000
		-6.27043594	-0.00119293	0.00073139	0.00119919	-0.00000116	0.00000000
		-6.27044055	-0.00119247	0.00073111	0.00119873	-0.00000116	0.00000000
		-6.27044516	-0.00119201	0.00073083	0.00119827	-0.00000116	0.00000000
		-6.27044974	-0.00119155	0.00073054	0.00119781	-0.00000116	0.00000000
		-6.27045432	-0.00119110	0.00073026	0.00119735	-0.00000116	0.00000000
		-6.27045887	-0.00119064	0.00072998	0.00119689	-0.00000116	0.00000000
		-6.27046342	-0.00119018	0.00072970	0.00119643	-0.00000116	0.00000000
		-6.27046795	-0.00118973	0.00072942	0.00119597	-0.00000116	0.00000000
		-6.27047246	-0.00118927	0.00072914	0.00119551	-0.00000116	0.00000000
		-6.27047696	-0.00118881	0.00072887	0.00119505	-0.00000116	0.00000000
		-6.27048145	-0.00118836	0.00072859	0.00119459	-0.00000116	0.00000000
		-6.27048592	-0.00118790	0.00072831	0.00119414	-0.00000116	0.00000000
		-6.27049038	-0.00118745	0.00072803	0.00119368	-0.00000116	0.00000000
		-6.27049482	-0.00118699	0.00072775	0.00119322	-0.00000116	0.00000000
		-6.27049925	-0.00118654	0.00072747	0.00119277	-0.00000116	0.00000000
		-6.27050366	-0.00118609	0.00072719	0.00119231	-0.00000116	0.00000000
		-6.27050807	-0.00118563	0.00072692	0.00119185	-0.00000116	0.00000000
		-6.27051245	-0.00118518	0.00072664	0.00119140	-0.00000116	0.00000000
		-6.27051683	-0.00118473	0.00072636	0.00119094	-0.00000116	0.00000000
		-6.27052119	-0.00118427	0.00072608	0.00119049	-0.00000116	0.00000000
		-6.27052553	-0.00118382	0.00072581	0.00119003	-0.00000116	0.00000000
		-6.27052986	-0.00118337	0.00072553	0.00118958	-0.00000116	0.00000000
		-6.27053418	-0.00118292	0.00072525	0.00118912	-0.00000116	0.00000000
		-6.27053849	-0.00118247	0.00072498	0.00118867	-0.00000116	0.00000000
		-6.27054278	-0.00118202	0.00072470	0.00118822	-0.00000116	0.00000000
		-6.27054706	-0.00118157	0.00072442	0.00118776	-0.00000116	0.00000000
		-6.27055132	-0.00118112	0.00072415	0.00118731	-0.00000116	0.00000000
		-6.27055557	-0.00118067	0.00072387	0.00118686	-0.00000116	0.00000000
		-6.27055981	-0.00118022	0.00072360	0.00118641	-0.00000116	0.00000000
		-6.27056403	-0.00117977	0.00072332	0.00118596	-0.00000116	0.00000000
		-6.27056825	-0.00117932	0.00072305	0.00118550	-0.00000116	0.00000000
		-6.27057244	-0.00117887	0.00072277	0.00118505	-0.00000116	0.00000000
		-6.27057663	-0.00117842	0.00072250	0.00118460	-0.00000116	0.00000000
		-6.27058080	-0.00117798	0.00072222	0.00118415	-0.00000116	0.00000000
		-6.27058496	-0.00117753	0.00072195	0.00118370	-0.00000116	0.00000000
		-6.27058910	-0.00117708	0.00072167	0.00118325	-0.00000116	0.00000000
		-6.27059324	-0.00117663	0.00072140	0.00118280	-0.00000116	0.00000000
		-6.27059736	-0.00117619	0.00072113	0.00118236	-0.00000116	0.00000000
		-6.27060146	-0.00117574	0.00072085	0.00118191	-0.00000116	0.00000000
		-6.27060556	-0.00117530	0.00072058	0.00118146	-0.00000116	0.00000000
		-6.27060964	-0.00117485	0.00072031	0.00118101	-0.00000116	0.00000000
		-6.27061371	-0.00117441	0.00072004	0.00118056	-0.00000116	0.00000000
		-6.27061776	-0.00117396	0.00071976	0.00118012	-0.00000116	0.00000000
		-6.27062180	-0.00117352	0.00071949	0.00117967	-0.00000116	0.00000000
		-6.27062583	-0.00117307	0.00071922	0.00117922	-0.00000116	0.00000000
		-6.27062985	-0.00117263	0.00071895	0.00117878	-0.00000116	0.00000000
		-6.27063386	-0.00117219	0.00071867	0.00117833	-0.00000116	0.00000000
		-6.27063785	-0.00117174	0.00071840	0.00117789	-0.00000116	0.00000000
		-6.27064183	-0.00117130	0.00071813	0.00117744	-0.00000116	0.00000000
		-6.27064580	-0.00117086	0.00071786	0.00117700	-0.00000116	0.00000000
		-6.27064976	-0.00117042	0.00071759	0.00117655	-0.00000116	0.00000000
		-6.27065370	-0.00116997	0.00071732	0.00117611	-0.00000116	0.00000000
		-6.27065763	-0.00116953	0.00071705	0.00117566	-0.00000116	0.00000000
		-6.27066155	-0.00116909	0.00071678	0.00117522	-0.00000116	0.00000000
		-6.27066546	-0.00116865	0.00071651	0.00117478	-0.00000116	0.00000000
		-6.27066935	-0.00116821	0.00071624	0.00117433	-0.00000116	0.00000000
		-6.27067323	-0.00116777	0.00071597	0.00117389	-0.00000116	0.00000000
		-6.27067711	-0.00116733	0.00071570	0.00117345	-0.00000116	0.00000000
		-6.27068096	-0.00116689	0.00071543	0.00117301	-0.00000116	0.00000000
		-6.27068481	-0.00116645	0.00071516	0.00117257	-0.00000116	0.00000000
		-6.27068865	-0.00116601	0.00071489	0.00117212	-0.00000116	0.00000000
		-6.27069247	-0.00116558	0.00071462	0.00117168	-0.00000116	0.00000000
		-6.27069628	-0.00116514	0.00071435	0.00117124	-0.00000116	0.00000000
		-6.27070008	-0.00116470	0.00071409	0.00117080	-0.00000116	0.00000000
		-6.27070387	-0.00116426	0.00071382	0.00117036	-0.00000116	0.00000000
		-6.27070765	-0.00116383	0.00071355	0.00116992	-0.00000116	0.00000000
		-6.27071141	-0.00116339	0.00071328	0.00116948	-0.00000116	0.00000000
		-6.27071516	-0.00116295	0.00071301	0.00116905	-0.00000116	0.00000000
		-6.27071891	-0.00116252	0.00071275	0.00116861	-0.00000116	0.00000000
		-6.27072264	-0.00116208	0.00071248	0.00116817	-0.00000116	0.00000000
		-6.27072636	-0.00116165	0.00071221	0.00116773	-0.00000116	0.00000000
		-6.27073006	-0.00116121	0.00071195	0.00116729	-0.00000116	0.00000000
		-6.27073376	-0.00116078	0.00071168	0.00116686	-0.00000116	0.00000000
		-6.27073744	-0.00116034	0.00071141	0.00116642	-0.00000116	0.00000000
		-6.27074112	-0.00115991	0.00071115	0.00116598	-0.00000116	0.00000000
		-6.27074478	-0.00115947	0.00071088	0.00116555	-0.00000116	0.00000000
		-6.27074843	-0.00115904	0.00071062	0.00116511	-0.00000116	0.00000000
		-6.27075207	-0.00115861	0.00071035	0.00116468	-0.00000116	0.00000000
		-6.27075570	-0.00115817	0.00071008	0.00116424	-0.00000116	0.00000000
		-6.27075932	-0.00115774	0.00070982	0.00116381	-0.00000116	0.00000000
		-6.27076293	-0.00115731	0.00070955	0.00116337	-0.00000116	0.00000000
		-6.27076652	-0.00115688	0.00070929	0.00116294	-0.00000116	0.00000000
		-6.27077011	-0.00115644	0.00070902	0.00116250	-0.00000116	0.00000000
		-6.27077368	-0.00115601	0.00070876	0.00116207	-0.00000116	0.00000000
		-6.27077724	-0.00115558	0.00070850	0.00116164	-0.00000116	0.00000000
		-6.27078080	-0.00115515	0.00070823	0.00116120	-0.00000116	0.00000000
		-6.27078434	-0.00115472	0.00070797	0.00116077	-0.00000116	0.00000000
		-6.27078787	-0.00115429	0.00070770	0.00116034	-0.00000116	0.00000000
		-6.27079139	-0.00115386	0.00070744	0.00115991	-0.00000116	0.00000000
		-6.27079490	-0.00115343	0.00070718	0.00115947	-0.00000116	0.00000000
		-6.27079840	-0.00115300	0.00070692	0.00115904	-0.00000116	0.00000000
		-6.27080189	-0.00115257	0.00070665	0.00115861	-0.00000116	0.00000000
		-6.27080537	-0.00115215	0.00070639	0.00115818	-0.00000116	0.00000000
		-6.27080883	-0.00115172	0.00070613	0.00115775	-0.00000116	0.00000000
		-6.27081229	-0.00115129	0.00070586	0.00115732	-0.00000116	0.00000000
		-6.27081574	-0.00115086	0.00070560	0.00115689	-0.00000116	0.00000000
		-6.27081917	-0.00115044	0.00070534	0.00115646	-0.00000116	0.00000000
		-6.27082260	-0.00115001	0.00070508	0.00115603	-0.00000116	0.00000000
		-6.27082601	-0.00114958	0.00070482	0.00115560	-0.00000116	0.00000000
		-6.27082942	-0.00114916	0.00070456	0.00115517	-0.00000116	0.00000000
		-6.27083281	-0.00114873	0.00070430	0.00115475	-0.00000116	0.00000000
		-6.27083620	-0.00114830	0.00070403	0.00115432	-0.00000116	0.00000000
		-6.27083957	-0.00114788	0.00070377	0.00115389	-0.00000116	0.00000000
		-6.27084294	-0.00114745	0.00070351	0.00115346	-0.00000116	0.00000000
		-6.27084629	-0.00114703	0.00070325	0.00115304	-0.00000116	0.00000000
		-6.27084964	-0.00114661	0.00070299	0.00115261	-0.00000116	0.00000000
		-6.27085297	-0.00114618	0.00070273	0.00115218	-0.00000116	0.00000000
		-6.27085630	-0.00114576	0.00070247	0.00115176	-0.00000116	0.00000000
		-6.27085961	-0.00114533	0.00070221	0.00115133	-0.00000116	0.00000000
		-6.27086292	-0.00114491	0.00070195	0.00115091	-0.00000116	0.00000000
		-6.27086621	-0.00114449	0.00070169	0.00115048	-0.00000116	0.00000000
		-6.27086950	-0.00114407	0.00070144	0.00115006	-0.00000116	0.00000000
		-6.27087277	-0.00114364	0.00070118	0.00114963	-0.00000116	0.00000000
		-6.27087604	-0.00114322	0.00070092	0.00114921	-0.00000116	0.00000000
		-6.27087929	-0.00114280	0.00070066	0.00114878	-0.00000116	0.00000000
		-6.27088254	-0.00114238	0.00070040	0.00114836	-0.00000116	0.00000000
		-6.27088577	-0.00114196	0.00070014	0.00114794	-0.00000116	0.00000000
		-6.27088900	-0.00114154	0.00069989	0.00114751	-0.00000116	0.00000000
		-6.27089222	-0.00114112	0.00069963	0.00114709	-0.00000116	0.00000000
		-6.27089543	-0.00114070	0.00069937	0.00114667	-0.00000116	0.00000000
		-6.27089862	-0.00114028	0.00069911	0.00114625	-0.00000116	0.00000000
		-6.27090181	-0.00113986	0.00069886	0.00114582	-0.00000116	0.00000000
		-6.27090499	-0.00113944	0.00069860	0.00114540	-0.00000116	0.00000000
		-6.27090816	-0.00113902	0.00069834	0.00114498	-0.00000116	0.00000000
		-6.27091132	-0.00113860	0.00069809	0.00114456	-0.00000116	0.00000000
		-6.27091447	-0.00113818	0.00069783	0.00114414	-0.00000116	0.00000000
		-6.27091762	-0.00113777	0.00069757	0.00114372	-0.00000116	0.00000000
		-6.27092075	-0.00113735	0.00069732	0.00114330	-0.00000116	0.00000000
		-6.27092387	-0.00113693	0.00069706	0.00114288	-0.00000116	0.00000000
		-6.27092699	-0.00113651	0.00069681	0.00114246	-0.00000116	0.00000000
		-6.27093009	-0.00113610	0.00069655	0.00114204	-0.00000116	0.00000000
		-6.27093319	-0.00113568	0.00069629	0.00114162	-0.00000116	0.00000000
		-6.27093627	-0.00113527	0.00069604	0.00114121	-0.00000116	0.00000000
		-6.27093935	-0.00113485	0.00069578	0.00114079	-0.00000116	0.00000000
		-6.27094242	-0.00113443	0.00069553	0.00114037	-0.00000116	0.00000000
		-6.27094548	-0.00113402	0.00069528	0.00113995	-0.00000116	0.00000000
		-6.27094853	-0.00113360	0.00069502	0.00113954	-0.00000116	0.00000000
		-6.27095157	-0.00113319	0.00069477	0.00113912	-0.00000116	0.00000000
		-6.27095461	-0.00113278	0.00069451	0.00113870	-0.00000116	0.00000000
		-6.27095763	-0.00113236	0.00069426	0.00113829	-0.00000116	0.00000000
		-6.27096065	-0.00113195	0.00069401	0.00113787	-0.00000116	0.00000000
		-6.27096365	-0.00113154	0.00069375	0.00113745	-0.00000116	0.00000000
		-6.27096665	-0.00113112	0.00069350	0.00113704	-0.00000116	0.00000000
		-6.27096964	-0.00113071	0.00069325	0.00113662	-0.00000116	0.00000000
		-6.27097262	-0.00113030	0.00069299	0.00113621	-0.00000116	0.00000000
		-6.27097559	-0.00112989	0.00069274	0.00113580	-0.00000116	0.00000000
		-6.27097856	-0.00112947	0.00069249	0.00113538	-0.00000116	0.00000000
		-6.27098151	-0.00112906	0.00069224	0.00113497	-0.00000116	0.00000000
		-6.27098446	-0.00112865	0.00069198	0.00113455	-0.00000116	0.00000000
		-6.27098739	-0.00112824	0.00069173	0.00113414	-0.00000116	0.00000000
		-6.27099032	-0.00112783	0.00069148	0.00113373	-0.00000116	0.00000000
		-6.27099324	-0.00112742	0.00069123	0.00113332	-0.00000116	0.00000000
		-6.27099616	-0.00112701	0.00069098	0.00113290	-0.00000116	0.00000000
		-6.27099906	-0.00112660	0.00069072	0.00113249	-0.00000116	0.00000000
		-6.27100196	-0.00112619	0.00069047	0.00113208	-0.00000116	0.00000000
		-6.27100484	-0.00112578	0.00069022	0.00113167	-0.00000116	0.00000000
		-6.27100772	-0.00112537	0.00068997	0.00113126	-0.00000116	0.00000000
		-6.27101059	-0.00112496	0.00068972	0.00113085	-0.00000116	0.00000000
		-6.27101345	-0.00112455	0.00068947	0.00113044	-0.00000116	0.00000000
		-6.27101631	-0.00112415	0.00068922	0.00113003	-0.00000116	0.00000000
		-6.27101915	-0.00112374	0.00068897	0.00112962	-0.00000116	0.00000000
		-6.27102199	-0.00112333	0.00068872	0.00112921	-0.00000116	0.00000000
		-6.27102482	-0.00112292	0.00068847	0.00112880	-0.00000116	0.00000000
		-6.27102764	-0.00112252	0.00068822	0.00112839	-0.00000116	0.00000000
		-6.27103045	-0.00112211	0.00068797	0.00112798	-0.00000116	0.00000000
		-6.27103326	-0.00112171	0.00068772	0.00112757	-0.00000116	0.00000000
		-6.27103606	-0.00112130	0.00068747	0.00112716	-0.00000116	0.00000000
		-6.27103885	-0.00112089	0.00068723	0.00112675	-0.00000116	0.00000000
		-6.27104163	-0.00112049	0.00068698	0.00112635	-0.00000116	0.00000000
		-6.27104440	-0.00112008	0.00068673	0.00112594	-0.00000116	0.00000000
		-6.27104717	-0.00111968	0.00068648	0.00112553	-0.00000116	0.00000000
		-6.27104992	-0.00111927	0.00068623	0.00112513	-0.00000116	0.00000000
		-6.27105267	-0.00111887	0.00068599	0.00112472	-0.00000116	0.00000000
		-6.27105541	-0.00111847	0.00068574	0.00112431	-0.00000116	0.00000000
		-6.27105815	-0.00111806	0.00068549	0.00112391	-0.00000116	0.00000000
		-6.27106088	-0.00111766	0.00068524	0.00112350	-0.00000116	0.00000000
		-6.27106359	-0.00111726	0.00068500	0.00112310	-0.00000116	0.00000000
		-6.27106630	-0.00111685	0.00068475	0.00112269	-0.00000116	0.00000000
		-6.27106901	-0.00111645	0.00068450	0.00112229	-0.00000116	0.00000000
		-6.27107170	-0.00111605	0.00068426	0.00112188	-0.00000116	0.00000000
		-6.27107439	-0.00111565	0.00068401	0.00112148	-0.00000116	0.00000000
		-6.27107707	-0.00111525	0.00068376	0.00112107	-0.00000116	0.00000000
		-6.27107974	-0.00111484	0.00068352	0.00112067	-0.00000116	0.00000000
		-6.27108241	-0.00111444	0.00068327	0.00112027	-0.00000116	0.00000000
		-6.27108507	-0.00111404	0.00068302	0.00111987	-0.00000116	0.00000000
		-6.27108772	-0.00111364	0.00068278	0.00111946	-0.00000116	0.00000000
		-6.27109036	-0.00111324	0.00068253	0.00111906	-0.00000116	0.00000000
		-6.27109300	-0.00111284	0.00068229	0.00111866	-0.00000116	0.00000000
		-6.27109562	-0.00111244	0.00068204	0.00111826	-0.00000116	0.00000000
		-6.27109825	-0.00111204	0.00068180	0.00111786	-0.00000116	0.00000000
		-6.27110086	-0.00111164	0.00068155	0.00111745	-0.00000116	0.00000000
		-6.27110347	-0.00111125	0.00068131	0.00111705	-0.00000116	0.00000000
		-6.27110606	-0.00111085	0.00068106	0.00111665	-0.00000116	0.00000000
		-6.27110866	-0.00111045	0.00068082	0.00111625	-0.00000116	0.00000000
		-6.27111124	-0.00111005	0.00068058	0.00111585	-0.00000116	0.00000000
		-6.27111382	-0.00110965	0.00068033	0.00111545	-0.00000116	0.00000000
		-6.27111639	-0.00110926	0.00068009	0.00111505	-0.00000116	0.00000000
		-6.27111895	-0.00110886	0.00067985	0.00111465	-0.00000116	0.00000000
		-6.27112151	-0.00110846	0.00067960	0.00111425	-0.00000116	0.00000000
		-6.27112405	-0.00110807	0.00067936	0.00111386	-0.00000116	0.00000000
		-6.27112660	-0.00110767	0.00067912	0.00111346	-0.00000116	0.00000000
		-6.27112913	-0.00110728	0.00067887	0.00111306	-0.00000116	0.00000000
		-6.27113166	-0.00110688	0.00067863	0.00111266	-0.00000116	0.00000000
		-6.27113418	-0.00110648	0.00067839	0.00111226	-0.00000116	0.00000000
		-6.27113669	-0.00110609	0.00067815	0.00111187	-0.00000116	0.00000000
		-6.27113920	-0.00110569	0.00067790	0.00111147	-0.00000116	0.00000000
		-6.27114170	-0.00110530	0.00067766	0.00111107	-0.00000116	0.00000000
		-6.27114419	-0.00110491	0.00067742	0.00111068	-0.00000116	0.00000000
		-6.27114668	-0.00110451	0.00067718	0.00111028	-0.00000116	0.00000000
		-6.27114915	-0.00110412	0.00067694	0.00110989	-0.00000116	0.00000000
		-6.27115163	-0.00110372	0.00067670	0.00110949	-0.00000116	0.00000000
		-6.27115409	-0.00110333	0.00067646	0.00110910	-0.00000116	0.00000000
		-6.27115655	-0.00110294	0.00067621	0.00110870	-0.00000116	0.00000000
		-6.27115900	-0.00110255	0.00067597	0.00110831	-0.00000116	0.00000000
		-6.27116145	-0.00110215	0.00067573	0.00110791	-0.00000116	0.00000000
		-6.27116389	-0.00110176	0.00067549	0.00110752	-0.00000116	0.00000000
		-6.27116632	-0.00110137	0.00067525	0.00110712	-0.00000116	0.00000000
		-6.27116874	-0.00110098	0.00067501	0.00110673	-0.00000116	0.00000000
		-6.27117116	-0.00110059	0.00067477	0.00110634	-0.00000115	0.00000000
		-6.27117357	-0.00110020	0.00067453	0.00110594	-0.00000115	0.00000000
		-6.27117598	-0.00109981	0.00067429	0.00110555	-0.00000115	0.00000000
		-6.27117838	-0.00109942	0.00067405	0.00110516	-0.00000115	0.00000000
		-6.27118077	-0.00109903	0.00067381	0.00110477	-0.00000115	0.00000000
		-6.27118315	-0.00109864	0.00067358	0.00110437	-0.00000115	0.00000000
		-6.27118553	-0.00109825	0.00067334	0.00110398	-0.00000115	0.00000000
		-6.27118791	-0.00109786	0.00067310	0.00110359	-0.00000115	0.00000000
		-6.27119027	-0.00109747	0.00067286	0.00110320	-0.00000115	0.00000000
		-6.27119263	-0.00109708	0.00067262	0.00110281	-0.00000115	0.00000000
		-6.27119499	-0.00109669	0.00067238	0.00110242	-0.00000115	0.00000000
		-6.27119733	-0.00109630	0.00067214	0.00110203	-0.00000115	0.00000000
		-6.27119968	-0.00109592	0.00067191	0.00110164	-0.00000115	0.00000000
		-6.27120201	-0.00109553	0.00067167	0.00110125	-0.00000115	0.00000000
		-6.27120434	-0.00109514	0.00067143	0.00110086	-0.00000115	0.00000000
		-6.27120666	-0.00109475	0.00067119	0.00110047	-0.00000115	0.00000000
		-6.27120898	-0.00109437	0.00067096	0.00110008	-0.00000115	0.00000000
		-6.27121129	-0.00109398	0.00067072	0.00109969	-0.00000115	0.00000000
		-6.27121359	-0.00109360	0.00067048	0.00109930	-0.00000115	0.00000000
		-6.27121589	-0.00109321	0.00067025	0.00109892	-0.00000115	0.00000000
		-6.27121818	-0.00109282	0.00067001	0.00109853	-0.00000115	0.00000000
		-6.27122046	-0.00109244	0.00066977	0.00109814	-0.00000115	0.00000000
		-6.27122274	-0.00109205	0.00066954	0.00109775	-0.00000115	0.00000000
		-6.27122501	-0.00109167	0.00066930	0.00109737	-0.00000115	0.00000000
		-6.27122728	-0.00109128	0.00066907	0.00109698	-0.00000115	0.00000000
		-6.27122954	-0.00109090	0.00066883	0.00109659	-0.00000115	0.00000000
		-6.27123179	-0.00109052	0.00066859	0.00109621	-0.00000115	0.00000000
		-6.27123404	-0.00109013	0.00066836	0.00109582	-0.00000115	0.00000000
		-6.27123629	-0.00108975	0.00066812	0.00109544	-0.00000115	0.00000000
		-6.27123852	-0.00108937	0.00066789	0.00109505	-0.00000115	0.00000000
		-6.27124075	-0.00108898	0.00066765	0.00109467	-0.00000115	0.00000000
		-6.27124298	-0.00108860	0.00066742	0.00109428	-0.00000115	0.00000000
		-6.27124520	-0.00108822	0.00066718	0.00109390	-0.00000115	0.00000000
		-6.27124741	-0.00108784	0.00066695	0.00109351	-0.00000115	0.00000000
		-6.27124962	-0.00108745	0.00066672	0.00109313	-0.00000115	0.00000000
		-6.27125182	-0.00108707	0.00066648	0.00109275	-0.00000115	0.00000000
		-6.27125401	-0.00108669	0.00066625	0.00109236	-0.00000115	0.00000000
		-6.27125620	-0.00108631	0.00066601	0.00109198	-0.00000115	0.00000000
		-6.27125839	-0.00108593	0.00066578	0.00109160	-0.00000115	0.00000000
		-6.27126057	-0.00108555	0.00066555	0.00109121	-0.00000115	0.00000000
		-6.27126274	-0.00108517	0.00066531	0.00109083	-0.00000115	0.00000000
		-6.27126490	-0.00108479	0.00066508	0.00109045	-0.00000115	0.00000000
		-6.27126707	-0.00108441	0.00066485	0.00109007	-0.00000115	0.00000000
		-6.27126922	-0.00108403	0.00066461	0.00108969	-0.00000115	0.00000000
		-6.27127137	-0.00108365	0.00066438	0.00108930	-0.00000115	0.00000000
		-6.27127351	-0.00108327	0.00066415	0.00108892	-0.00000115	0.00000000
		-6.27127565	-0.00108289	0.00066392	0.00108854	-0.00000115	0.00000000
		-6.27127779	-0.00108251	0.00066369	0.00108816	-0.00000115	0.00000000
		-6.27127991	-0.00108214	0.00066345	0.00108778	-0.00000115	0.00000000
		-6.27128204	-0.00108176	0.00066322	0.00108740	-0.00000115	0.00000000
		-6.27128415	-0.00108138	0.00066299	0.00108702	-0.00000115	0.00000000
		-6.27128626	-0.00108100	0.00066276	0.00108664	-0.00000115	0.00000000
		-6.27128837	-0.00108063	0.00066253	0.00108626	-0.00000115	0.00000000
		-6.27129047	-0.00108025	0.00066230	0.00108588	-0.00000115	0.00000000
		-6.27129256	-0.00107987	0.00066206	0.00108551	-0.00000115	0.00000000
		-6.27129465	-0.00107950	0.00066183	0.00108513	-0.00000115	0.00000000
		-6.27129674	-0.00107912	0.00066160	0.00108475	-0.00000115	0.00000000
		-6.27129881	-0.00107874	0.00066137	0.00108437	-0.00000115	0.00000000
		-6.27130089	-0.00107837	0.00066114	0.00108399	-0.00000115	0.00000000
		-6.27130295	-0.00107799	0.00066091	0.00108362	-0.00000115	0.00000000
		-6.27130502	-0.00107762	0.00066068	0.00108324	-0.00000115	0.00000000
		-6.27130707	-0.00107724	0.00066045	0.00108286	-0.00000115	0.00000000
		-6.27130913	-0.00107687	0.00066022	0.00108249	-0.00000115	0.00000000
		-6.27131117	-0.00107649	0.00065999	0.00108211	-0.00000115	0.00000000
		-6.27131321	-0.00107612	0.00065976	0.00108173	-0.00000115	0.00000000
		-6.27131525	-0.00107575	0.00065953	0.00108136	-0.00000115	0.00000000
		-6.27131728	-0.00107537	0.00065931	0.00108098	-0.00000115	0.00000000
		-6.27131930	-0.00107500	0.00065908	0.00108061	-0.00000115	0.00000000
		-6.27132133	-0.00107463	0.00065885	0.00108023	-0.00000115	0.00000000
		-6.27132334	-0.00107425	0.00065862	0.00107986	-0.00000115	0.00000000
		-6.27132535	-0.00107388	0.00065839	0.00107948	-0.00000115	0.00000000
		-6.27132736	-0.00107351	0.00065816	0.00107911	-0.00000115	0.00000000
		-6.27132936	-0.00107314	0.00065793	0.00107873	-0.00000115	0.00000000
		-6.27133135	-0.00107277	0.00065771	0.00107836	-0.00000115	0.00000000
		-6.27133334	-0.00107239	0.00065748	0.00107799	-0.00000115	0.00000000
		-6.27133532	-0.00107202	0.00065725	0.00107761	-0.00000115	0.00000000
		-6.27133730	-0.00107165	0.00065702	0.00107724	-0.00000115	0.00000000
		-6.27133928	-0.00107128	0.00065680	0.00107687	-0.00000115	0.00000000
		-6.27134125	-0.00107091	0.00065657	0.00107650	-0.00000115	0.00000000
		-6.27134321	-0.00107054	0.00065634	0.00107612	-0.00000115	0.00000000
		-6.27134517	-0.00107017	0.00065611	0.00107575	-0.00000115	0.00000000
		-6.27134713	-0.00106980	0.00065589	0.00107538	-0.00000115	0.00000000
		-6.27134908	-0.00106943	0.00065566	0.00107501	-0.00000115	0.00000000
		-6.27135102	-0.00106906	0.00065543	0.00107464	-0.00000115	0.00000000
		-6.27135296	-0.00106870	0.00065521	0.00107427	-0.00000115	0.00000000
		-6.27135490	-0.00106833	0.00065498	0.00107390	-0.00000115	0.00000000
		-6.27135683	-0.00106796	0.00065476	0.00107353	-0.00000115	0.00000000
		-6.27135875	-0.00106759	0.00065453	0.00107316	-0.00000115	0.00000000
		-6.27136067	-0.00106722	0.00065430	0.00107279	-0.00000115	0.00000000
		-6.27136259	-0.00106685	0.00065408	0.00107242	-0.00000115	0.00000000
		-6.27136450	-0.00106649	0.00065385	0.00107205	-0.00000115	0.00000000
		-6.27136641	-0.00106612	0.00065363	0.00107168	-0.00000115	0.00000000
		-6.27136831	-0.00106575	0.00065340	0.00107131	-0.00000115	0.00000000
		-6.27137021	-0.00106539	0.00065318	0.00107094	-0.00000115	0.00000000
		-6.27137210	-0.00106502	0.00065295	0.00107057	-0.00000115	0.00000000
		-6.27137399	-0.00106465	0.00065273	0.00107020	-0.00000115	0.00000000
		-6.27137587	-0.00106429	0.00065250	0.00106984	-0.00000115	0.00000000
		-6.27137775	-0.00106392	0.00065228	0.00106947	-0.00000115	0.00000000
		-6.27137962	-0.00106356	0.00065206	0.00106910	-0.00000115	0.00000000
		-6.27138149	-0.00106319	0.00065183	0.00106873	-0.00000115	0.00000000
		-6.27138335	-0.00106283	0.00065161	0.00106837	-0.00000115	0.00000000
		-6.27138521	-0.00106246	0.00065139	0.00106800	-0.00000115	0.00000000
		-6.27138707	-0.00106210	0.00065116	0.00106763	-0.00000115	0.00000000
		-6.27138892	-0.00106173	0.00065094	0.00106727	-0.00000115	0.00000000
		-6.27139077	-0.00106137	0.00065072	0.00106690	-0.00000115	0.00000000
		-6.27139261	-0.00106101	0.00065049	0.00106654	-0.00000115	0.00000000
		-6.27139445	-0.00106064	0.00065027	0.00106617	-0.00000115	0.00000000
		-6.27139628	-0.00106028	0.00065005	0.00106581	-0.00000115	0.00000000
		-6.27139811	-0.00105992	0.00064982	0.00106544	-0.00000115	0.00000000
		-6.27139993	-0.00105956	0.00064960	0.00106508	-0.00000115	0.00000000
		-6.27140175	-0.00105919	0.00064938	0.00106471	-0.00000115	0.00000000
		-6.27140356	-0.00105883	0.00064916	0.00106435	-0.00000115	0.00000000
		-6.27140537	-0.00105847	0.00064893	0.00106399	-0.00000115	0.00000000
		-6.27140718	-0.00105811	0.00064871	0.00106362	-0.00000115	0.00000000
		-6.27140898	-0.00105775	0.00064849	0.00106326	-0.00000115	0.00000000
		-6.27141078	-0.00105739	0.00064827	0.00106290	-0.00000115	0.00000000
		-6.27141257	-0.00105702	0.00064805	0.00106253	-0.00000115	0.00000000
		-6.27141436	-0.00105666	0.00064783	0.00106217	-0.00000115	0.00000000
		-6.27141614	-0.00105630	0.00064761	0.00106181	-0.00000115	0.00000000
		-6.27141792	-0.00105594	0.00064739	0.00106145	-0.00000115	0.00000000
		-6.27141970	-0.00105558	0.00064716	0.00106108	-0.00000115	0.00000000
		-6.27142147	-0.00105522	0.00064694	0.00106072	-0.00000115	0.00000000
		-6.27142324	-0.00105486	0.00064672	0.00106036	-0.00000115	0.00000000
		-6.27142500	-0.00105451	0.00064650	0.00106000	-0.00000115	0.00000000
		-6.27142676	-0.00105415	0.00064628	0.00105964	-0.00000115	0.00000000
		-6.27142851	-0.00105379	0.00064606	0.00105928	-0.00000115	0.00000000
		-6.27143026	-0.00105343	0.00064584	0.00105892	-0.00000115	0.00000000
		-6.27143201	-0.00105307	0.00064562	0.00105856	-0.00000115	0.00000000
		-6.27143375	-0.00105271	0.00064540	0.00105820	-0.00000115	0.00000000
		-6.27143549	-0.00105236	0.00064518	0.00105784	-0.00000115	0.00000000
		-6.27143722	-0.00105200	0.00064496	0.00105748	-0.00000115	0.00000000
		-6.27143895	-0.00105164	0.00064475	0.00105712	-0.00000115	0.00000000
		-6.27144068	-0.00105128	0.00064453	0.00105676	-0.00000115	0.00000000
		-6.27144240	-0.00105093	0.00064431	0.00105640	-0.00000115	0.00000000
		-6.27144412	-0.00105057	0.00064409	0.00105604	-0.00000115	0.00000000
		-6.27144583	-0.00105021	0.00064387	0.00105569	-0.00000115	0.00000000
		-6.27144754	-0.00104986	0.00064365	0.00105533	-0.00000115	0.00000000
		-6.27144924	-0.00104950	0.00064343	0.00105497	-0.00000115	0.00000000
		-6.27145094	-0.00104915	0.00064322	0.00105461	-0.00000115	0.00000000
		-6.27145264	-0.00104879	0.00064300	0.00105425	-0.00000115	0.00000000
		-6.27145433	-0.00104844	0.00064278	0.00105390	-0.00000115	0.00000000
		-6.27145602	-0.00104808	0.00064256	0.00105354	-0.00000115	0.00000000
		-6.27145771	-0.00104773	0.00064234	0.00105318	-0.00000115	0.00000000
		-6.27145939	-0.00104737	0.00064213	0.00105283	-0.00000115	0.00000000
		-6.27146107	-0.00104702	0.00064191	0.00105247	-0.00000115	0.00000000
		-6.27146274	-0.00104667	0.00064169	0.00105212	-0.00000115	0.00000000
		-6.27146441	-0.00104631	0.00064148	0.00105176	-0.00000115	0.00000000
		-6.27146607	-0.00104596	0.00064126	0.00105141	-0.00000115	0.00000000
		-6.27146774	-0.00104561	0.00064104	0.00105105	-0.00000115	0.00000000
		-6.27146939	-0.00104525	0.00064083	0.00105070	-0.00000115	0.00000000
		-6.27147105	-0.00104490	0.00064061	0.00105034	-0.00000115	0.00000000
		-6.27147270	-0.00104455	0.00064039	0.00104999	-0.00000115	0.00000000
		-6.27147434	-0.00104420	0.00064018	0.00104963	-0.00000115	0.00000000
		-6.27147599	-0.00104384	0.00063996	0.00104928	-0.00000115	0.00000000
		-6.27147762	-0.00104349	0.00063975	0.00104893	-0.00000115	0.00000000
		-6.27147926	-0.00104314	0.00063953	0.00104857	-0.00000115	0.00000000
		-6.27148089	-0.00104279	0.00063931	0.00104822	-0.00000115	0.00000000
		-6.27148252	-0.00104244	0.00063910	0.00104787	-0.00000115	0.00000000
		-6.27148414	-0.00104209	0.00063888	0.00104751	-0.00000115	0.00000000
		-6.27148576	-0.00104174	0.00063867	0.00104716	-0.00000115	0.00000000
		-6.27148738	-0.00104139	0.00063845	0.00104681	-0.00000115	0.00000000
		-6.27148899	-0.00104104	0.00063824	0.00104646	-0.00000115	0.00000000
		-6.27149060	-0.00104069	0.00063802	0.00104610	-0.00000115	0.00000000
		-6.27149220	-0.00104034	0.00063781	0.00104575	-0.00000115	0.00000000
		-6.27149380	-0.00103999	0.00063760	0.00104540	-0.00000115	0.00000000
		-6.27149540	-0.00103964	0.00063738	0.00104505	-0.00000115	0.00000000
		-6.27149699	-0.00103929	0.00063717	0.00104470	-0.00000115	0.00000000
		-6.27149859	-0.00103894	0.00063695	0.00104435	-0.00000115	0.00000000
		-6.27150017	-0.00103859	0.00063674	0.00104400	-0.00000115	0.00000000
		-6.27150175	-0.00103824	0.00063653	0.00104365	-0.00000115	0.00000000
		-6.27150333	-0.00103790	0.00063631	0.00104330	-0.00000115	0.00000000
		-6.27150491	-0.00103755	0.00063610	0.00104295	-0.00000115	0.00000000
		-6.27150648	-0.00103720	0.00063589	0.00104260	-0.00000115	0.00000000
		-6.27150805	-0.00103685	0.00063567	0.00104225	-0.00000115	0.00000000
		-6.27150962	-0.00103651	0.00063546	0.00104190	-0.00000115	0.00000000
		-6.27151118	-0.00103616	0.00063525	0.00104155	-0.00000115	0.00000000
		-6.27151274	-0.00103581	0.00063503	0.00104120	-0.00000115	0.00000000
		-6.27151429	-0.00103547	0.00063482	0.00104086	-0.00000115	0.00000000
		-6.27151584	-0.00103512	0.00063461	0.00104051	-0.00000115	0.00000000
		-6.27151739	-0.00103478	0.00063440	0.00104016	-0.00000115	0.00000000
		-6.27151893	-0.00103443	0.00063419	0.00103981	-0.00000115	0.00000000
		-6.27152048	-0.00103408	0.00063397	0.00103947	-0.00000115	0.00000000
		-6.27152201	-0.00103374	0.00063376	0.00103912	-0.00000115	0.00000000
		-6.27152355	-0.00103339	0.00063355	0.00103877	-0.00000115	0.00000000
		-6.27152508	-0.00103305	0.00063334	0.00103843	-0.00000115	0.00000000
		-6.27152660	-0.00103270	0.00063313	0.00103808	-0.00000115	0.00000000
		-6.27152813	-0.00103236	0.00063292	0.00103773	-0.00000115	0.00000000
		-6.27152965	-0.00103202	0.00063271	0.00103739	-0.00000115	0.00000000
		-6.27153117	-0.00103167	0.00063249	0.00103704	-0.00000115	0.00000000
		-6.27153268	-0.00103133	0.00063228	0.00103670	-0.00000115	0.00000000
		-6.27153419	-0.00103099	0.00063207	0.00103635	-0.00000115	0.00000000
		-6.27153570	-0.00103064	0.00063186	0.00103601	-0.00000115	0.00000000
		-6.27153720	-0.00103030	0.00063165	0.00103566	-0.00000115	0.00000000
		-6.27153870	-0.00102996	0.00063144	0.00103532	-0.00000115	0.00000000
		-6.27154020	-0.00102961	0.00063123	0.00103497	-0.00000115	0.00000000
		-6.27154169	-0.00102927	0.00063102	0.00103463	-0.00000115	0.00000000
		-6.27154318	-0.00102893	0.00063081	0.00103428	-0.00000115	0.00000000
		-6.27154467	-0.00102859	0.00063060	0.00103394	-0.00000115	0.00000000
		-6.27154615	-0.00102825	0.00063039	0.00103360	-0.00000115	0.00000000
		-6.27154763	-0.00102791	0.00063018	0.00103325	-0.00000115	0.00000000
		-6.27154911	-0.00102756	0.00062997	0.00103291	-0.00000115	0.00000000
		-6.27155059	-0.00102722	0.00062976	0.00103257	-0.00000115	0.00000000
		-6.27155206	-0.00102688	0.00062956	0.00103223	-0.00000115	0.00000000
		-6.27155352	-0.00102654	0.00062935	0.00103188	-0.00000115	0.00000000
		-6.27155499	-0.00102620	0.00062914	0.00103154	-0.00000115	0.00000000
		-6.27155645	-0.00102586	0.00062893	0.00103120	-0.00000115	0.00000000
		-6.27155791	-0.00102552	0.00062872	0.00103086	-0.00000115	0.00000000
		-6.27155936	-0.00102518	0.00062851	0.00103052	-0.00000115	0.00000000
		-6.27156082	-0.00102484	0.00062830	0.00103018	-0.00000115	0.00000000
		-6.27156226	-0.00102451	0.00062810	0.00102983	-0.00000115	0.00000000
		-6.27156371	-0.00102417	0.00062789	0.00102949	-0.00000115	0.00000000
		-6.27156515	-0.00102383	0.00062768	0.00102915	-0.00000115	0.00000000
		-6.27156659	-0.00102349	0.00062747	0.00102881	-0.00000115	0.00000000
		-6.27156803	-0.00102315	0.00062727	0.00102847	-0.00000115	0.00000000
		-6.27156946	-0.00102281	0.00062706	0.00102813	-0.00000115	0.00000000
		-6.27157089	-0.00102248	0.00062685	0.00102779	-0.00000115	0.00000000
		-6.27157232	-0.00102214	0.00062664	0.00102745	-0.00000115	0.00000000
		-6.27157374	-0.00102180	0.00062644	0.00102712	-0.00000115	0.00000000
		-6.27157517	-0.00102146	0.00062623	0.00102678	-0.00000115	0.00000000
		-6.27157658	-0.00102113	0.00062602	0.00102644	-0.00000115	0.00000000
		-6.27157800	-0.00102079	0.00062582	0.00102610	-0.00000115	0.00000000
		-6.27157941	-0.00102045	0.00062561	0.00102576	-0.00000115	0.00000000
		-6.27158082	-0.00102012	0.00062540	0.00102542	-0.00000115	0.00000000
		-6.27158223	-0.00101978	0.00062520	0.00102509	-0.00000115	0.00000000
		-6.27158363	-0.00101945	0.00062499	0.00102475	-0.00000115	0.00000000
		-6.27158503	-0.00101911	0.00062479	0.00102441	-0.00000115	0.00000000
		-6.27158643	-0.00101878	0.00062458	0.00102407	-0.00000115	0.00000000
		-6.27158782	-0.00101844	0.00062438	0.00102374	-0.00000115	0.00000000
		-6.27158921	-0.00101811	0.00062417	0.00102340	-0.00000115	0.00000000
		-6.27159060	-0.00101777	0.00062396	0.00102306	-0.00000115	0.00000000
		-6.27159199	-0.00101744	0.00062376	0.00102273	-0.00000115	0.00000000
		-6.27159337	-0.00101710	0.00062355	0.00102239	-0.00000115	0.00000000
		-6.27159475	-0.00101677	0.00062335	0.00102206	-0.00000115	0.00000000
		-6.27159613	-0.00101644	0.00062314	0.00102172	-0.00000115	0.00000000
		-6.27159750	-0.00101610	0.00062294	0.00102138	-0.00000115	0.00000000
		-6.27159887	-0.00101577	0.00062274	0.00102105	-0.00000115	0.00000000
		-6.27160024	-0.00101544	0.00062253	0.00102071	-0.00000115	0.00000000
		-6.27160160	-0.00101510	0.00062233	0.00102038	-0.00000115	0.00000000
		-6.27160297	-0.00101477	0.00062212	0.00102005	-0.00000115	0.00000000
		-6.27160433	-0.00101444	0.00062192	0.00101971	-0.00000115	0.00000000
		-6.27160568	-0.00101410	0.00062171	0.00101938	-0.00000115	0.00000000
		-6.27160704	-0.00101377	0.00062151	0.00101904	-0.00000115	0.00000000
		-6.27160839	-0.00101344	0.00062131	0.00101871	-0.00000115	0.00000000
		-6.27160974	-0.00101311	0.00062110	0.00101838	-0.00000115	0.00000000
		-6.27161108	-0.00101278	0.00062090	0.00101804	-0.00000115	0.00000000
		-6.27161243	-0.00101245	0.00062070	0.00101771	-0.00000115	0.00000000
		-6.27161377	-0.00101212	0.00062049	0.00101738	-0.00000115	0.00000000
		-6.27161511	-0.00101179	0.00062029	0.00101705	-0.00000115	0.00000000
		-6.27161644	-0.00101146	0.00062009	0.00101671	-0.00000115	0.00000000
		-6.27161777	-0.00101112	0.00061989	0.00101638	-0.00000115	0.00000000
		-6.27161910	-0.00101079	0.00061968	0.00101605	-0.00000115	0.00000000
		-6.27162043	-0.00101046	0.00061948	0.00101572	-0.00000115	0.00000000
		-6.27162175	-0.00101014	0.00061928	0.00101539	-0.00000115	0.00000000
		-6.27162307	-0.00100981	0.00061908	0.00101505	-0.00000115	0.00000000
		-6.27162439	-0.00100948	0.00061887	0.00101472	-0.00000115	0.00000000
		-6.27162571	-0.00100915	0.00061867	0.00101439	-0.00000115	0.00000000
		-6.27162702	-0.00100882	0.00061847	0.00101406	-0.00000115	0.00000000
		-6.27162833	-0.00100849	0.00061827	0.00101373	-0.00000115	0.00000000
		-6.27162964	-0.00100816	0.00061807	0.00101340	-0.00000115	0.00000000
		-6.27163095	-0.00100783	0.00061787	0.00101307	-0.00000115	0.00000000
		-6.27163225	-0.00100751	0.00061767	0.00101274	-0.00000115	0.00000000
		-6.27163355	-0.00100718	0.00061746	0.00101241	-0.00000115	0.00000000
		-6.27163485	-0.00100685	0.00061726	0.00101208	-0.00000115	0.00000000
		-6.27163614	-0.00100652	0.00061706	0.00101175	-0.00000115	0.00000000
		-6.27163743	-0.00100620	0.00061686	0.00101143	-0.00000115	0.00000000
		-6.27163872	-0.00100587	0.00061666	0.00101110	-0.00000115	0.00000000
		-6.27164001	-0.00100554	0.00061646	0.00101077	-0.00000115	0.00000000
		-6.27164130	-0.00100522	0.00061626	0.00101044	-0.00000115	0.00000000
		-6.27164258	-0.00100489	0.00061606	0.00101011	-0.00000115	0.00000000
		-6.27164386	-0.00100456	0.00061586	0.00100978	-0.00000115	0.00000000
		-6.27164513	-0.00100424	0.00061566	0.00100946	-0.00000115	0.00000000
		-6.27164641	-0.00100391	0.00061546	0.00100913	-0.00000114	0.00000000
		-6.27164768	-0.00100359	0.00061526	0.00100880	-0.00000114	0.00000000
		-6.27164895	-0.00100326	0.00061506	0.00100848	-0.00000114	0.00000000
		-6.27165022	-0.00100294	0.00061486	0.00100815	-0.00000114	0.00000000
		-6.27165148	-0.00100261	0.00061466	0.00100782	-0.00000114	0.00000000
		-6.27165274	-0.00100229	0.00061446	0.00100750	-0.00000114	0.00000000
		-6.27165400	-0.00100196	0.00061427	0.00100717	-0.00000114	0.00000000
		-6.27165526	-0.00100164	0.00061407	0.00100684	-0.00000114	0.00000000
		-6.27165651	-0.00100132	0.00061387	0.00100652	-0.00000114	0.00000000
		-6.27165777	-0.00100099	0.00061367	0.00100619	-0.00000114	0.00000000
		-6.27165902	-0.00100067	0.00061347	0.00100587	-0.00000114	0.00000000
		-6.27166026	-0.00100035	0.00061327	0.00100554	-0.00000114	0.00000000
		-6.27166151	-0.00100002	0.00061307	0.00100522	-0.00000114	0.00000000
		-6.27166275	-0.00099970	0.00061288	0.00100489	-0.00000114	0.00000000
		-6.27166399	-0.00099938	0.00061268	0.00100457	-0.00000114	0.00000000
		-6.27166523	-0.00099905	0.00061248	0.00100424	-0.00000114	0.00000000
		-6.27166646	-0.00099873	0.00061228	0.00100392	-0.00000114	0.00000000
		-6.27166770	-0.00099841	0.00061208	0.00100360	-0.00000114	0.00000000
		-6.27166893	-0.00099809	0.00061189	0.00100327	-0.00000114	0.00000000
		-6.27167015	-0.00099777	0.00061169	0.00100295	-0.00000114	0.00000000
		-6.27167138	-0.00099745	0.00061149	0.00100263	-0.00000114	0.00000000
		-6.27167260	-0.00099712	0.00061130	0.00100230	-0.00000114	0.00000000
		-6.27167382	-0.00099680	0.00061110	0.00100198	-0.00000114	0.00000000
		-6.27167504	-0.00099648	0.00061090	0.00100166	-0.00000114	0.00000000
		-6.27167626	-0.00099616	0.00061070	0.00100134	-0.00000114	0.00000000
		-6.27167747	-0.00099584	0.00061051	0.00100101	-0.00000114	0.00000000
		-6.27167868	-0.00099552	0.00061031	0.00100069	-0.00000114	0.00000000
		-6.27167989	-0.00099520	0.00061012	0.00100037	-0.00000114	0.00000000
		-6.27168110	-0.00099488	0.00060992	0.00100005	-0.00000114	0.00000000
		-6.27168231	-0.00099456	0.00060972	0.00099973	-0.00000114	0.00000000
		-6.27168351	-0.00099424	0.00060953	0.00099941	-0.00000114	0.00000000
		-6.27168471	-0.00099392	0.00060933	0.00099909	-0.00000114	0.00000000
		-6.27168591	-0.00099360	0.00060914	0.00099877	-0.00000114	0.00000000
		-6.27168710	-0.00099329	0.00060894	0.00099844	-0.00000114	0.00000000
		-6.27168829	-0.00099297	0.00060874	0.00099812	-0.00000114	0.00000000
		-6.27168949	-0.00099265	0.00060855	0.00099780	-0.00000114	0.00000000
		-6.27169067	-0.00099233	0.00060835	0.00099748	-0.00000114	0.00000000
		-6.27169186	-0.00099201	0.00060816	0.00099716	-0.00000114	0.00000000
		-6.27169305	-0.00099170	0.00060796	0.00099685	-0.00000114	0.00000000
		-6.27169423	-0.00099138	0.00060777	0.00099653	-0.00000114	0.00000000
		-6.27169541	-0.00099106	0.00060757	0.00099621	-0.00000114	0.00000000
		-6.27169659	-0.00099074	0.00060738	0.00099589	-0.00000114	0.00000000
		-6.27169776	-0.00099043	0.00060719	0.00099557	-0.00000114	0.00000000
		-6.27169893	-0.00099011	0.00060699	0.00099525	-0.00000114	0.00000000
		-6.27170011	-0.00098979	0.00060680	0.00099493	-0.00000114	0.00000000
		-6.27170128	-0.00098948	0.00060660	0.00099461	-0.00000114	0.00000000
		-6.27170244	-0.00098916	0.00060641	0.00099430	-0.00000114	0.00000000
		-6.27170361	-0.00098884	0.00060622	0.00099398	-0.00000114	0.00000000
		-6.27170477	-0.00098853	0.00060602	0.00099366	-0.00000114	0.00000000
		-6.27170593	-0.00098821	0.00060583	0.00099334	-0.00000114	0.00000000
		-6.27170709	-0.00098790	0.00060563	0.00099303	-0.00000114	0.00000000
		-6.27170825	-0.00098758	0.00060544	0.00099271	-0.00000114	0.00000000
		-6.27170940	-0.00098727	0.00060525	0.00099239	-0.00000114	0.00000000
		-6.27171055	-0.00098695	0.00060505	0.00099208	-0.00000114	0.00000000
		-6.27171170	-0.00098664	0.00060486	0.00099176	-0.00000114	0.00000000
		-6.27171285	-0.00098633	0.00060467	0.00099145	-0.00000114	0.00000000
		-6.27171399	-0.00098601	0.00060448	0.00099113	-0.00000114	0.00000000
		-6.27171514	-0.00098570	0.00060428	0.00099081	-0.00000114	0.00000000
		-6.27171628	-0.00098538	0.00060409	0.00099050	-0.00000114	0.00000000
		-6.27171742	-0.00098507	0.00060390	0.00099018	-0.00000114	0.00000000
		-6.27171856	-0.00098476	0.00060371	0.00098987	-0.00000114	0.00000000
		-6.27171969	-0.00098444	0.00060351	0.00098955	-0.00000114	0.00000000
		-6.27172082	-0.00098413	0.00060332	0.00098924	-0.00000114	0.00000000
		-6.27172196	-0.00098382	0.00060313	0.00098893	-0.00000114	0.00000000
		-6.27172309	-0.00098351	0.00060294	0.00098861	-0.00000114	0.00000000
		-6.27172421	-0.00098319	0.00060275	0.00098830	-0.00000114	0.00000000
		-6.27172534	-0.00098288	0.00060256	0.00098798	-0.00000114	0.00000000
		-6.27172646	-0.00098257	0.00060236	0.00098767	-0.00000114	0.00000000
		-6.27172758	-0.00098226	0.00060217	0.00098736	-0.00000114	0.00000000
		-6.27172870	-0.00098195	0.00060198	0.00098704	-0.00000114	0.00000000
		-6.27172982	-0.00098163	0.00060179	0.00098673	-0.00000114	0.00000000
		-6.27173093	-0.00098132	0.00060160	0.00098642	-0.00000114	0.00000000
		-6.27173205	-0.00098101	0.00060141	0.00098610	-0.00000114	0.00000000
		-6.27173316	-0.00098070	0.00060122	0.00098579	-0.00000114	0.00000000
		-6.27173427	-0.00098039	0.00060103	0.00098548	-0.00000114	0.00000000
		-6.27173537	-0.00098008	0.00060084	0.00098517	-0.00000114	0.00000000
		-6.27173648	-0.00097977	0.00060065	0.00098486	-0.00000114	0.00000000
		-6.27173758	-0.00097946	0.00060046	0.00098454	-0.00000114	0.00000000
		-6.27173868	-0.00097915	0.00060027	0.00098423	-0.00000114	0.00000000
		-6.27173978	-0.00097884	0.00060008	0.00098392	-0.00000114	0.00000000
		-6.27174088	-0.00097853	0.00059989	0.00098361	-0.00000114	0.00000000
		-6.27174198	-0.00097822	0.00059970	0.00098330	-0.00000114	0.00000000
		-6.27174307	-0.00097791	0.00059951	0.00098299	-0.00000114	0.00000000
		-6.27174416	-0.00097761	0.00059932	0.00098268	-0.00000114	0.00000000
		-6.27174525	-0.00097730	0.00059913	0.00098237	-0.00000114	0.00000000
		-6.27174634	-0.00097699	0.00059894	0.00098206	-0.00000114	0.00000000
		-6.27174743	-0.00097668	0.00059875	0.00098175	-0.00000114	0.00000000
		-6.27174851	-0.00097637	0.00059856	0.00098144	-0.00000114	0.00000000
		-6.27174959	-0.00097606	0.00059837	0.00098113	-0.00000114	0.00000000
		-6.27175067	-0.00097576	0.00059818	0.00098082	-0.00000114	0.00000000
		-6.27175175	-0.00097545	0.00059800	0.00098051	-0.00000114	0.00000000
		-6.27175283	-0.00097514	0.00059781	0.00098020	-0.00000114	0.00000000
		-6.27175390	-0.00097484	0.00059762	0.00097989	-0.00000114	0.00000000
		-6.27175497	-0.00097453	0.00059743	0.00097958	-0.00000114	0.00000000
		-6.27175605	-0.00097422	0.00059724	0.00097928	-0.00000114	0.00000000
		-6.27175712	-0.00097392	0.00059705	0.00097897	-0.00000114	0.00000000
		-6.27175818	-0.00097361	0.00059687	0.00097866	-0.00000114	0.00000000
		-6.27175925	-0.00097330	0.00059668	0.00097835	-0.00000114	0.00000000
		-6.27176031	-0.00097300	0.00059649	0.00097805	-0.00000114	0.00000000
		-6.27176137	-0.00097269	0.00059630	0.00097774	-0.00000114	0.00000000
		-6.27176243	-0.00097239	0.00059612	0.00097743	-0.00000114	0.00000000
		-6.27176349	-0.00097208	0.00059593	0.00097712	-0.00000114	0.00000000
		-6.27176455	-0.00097178	0.00059574	0.00097682	-0.00000114	0.00000000
		-6.27176560	-0.00097147	0.00059555	0.00097651	-0.00000114	0.00000000
		-6.27176666	-0.00097117	0.00059537	0.00097620	-0.00000114	0.00000000
		-6.27176771	-0.00097086	0.00059518	0.00097590	-0.00000114	0.00000000
		-6.27176876	-0.00097056	0.00059499	0.00097559	-0.00000114	0.00000000
		-6.27176980	-0.00097025	0.00059481	0.00097529	-0.00000114	0.00000000
		-6.27177085	-0.00096995	0.00059462	0.00097498	-0.00000114	0.00000000
		-6.27177189	-0.00096965	0.00059443	0.00097468	-0.00000114	0.00000000
		-6.27177294	-0.00096934	0.00059425	0.00097437	-0.00000114	0.00000000
		-6.27177398	-0.00096904	0.00059406	0.00097407	-0.00000114	0.00000000
		-6.27177502	-0.00096874	0.00059388	0.00097376	-0.00000114	0.00000000
		-6.27177605	-0.00096843	0.00059369	0.00097346	-0.00000114	0.00000000
		-6.27177709	-0.00096813	0.00059350	0.00097315	-0.00000114	0.00000000
		-6.27177812	-0.00096783	0.00059332	0.00097285	-0.00000114	0.00000000
		-6.27177915	-0.00096752	0.00059313	0.00097254	-0.00000114	0.00000000
		-6.27178018	-0.00096722	0.00059295	0.00097224	-0.00000114	0.00000000
		-6.27178121	-0.00096692	0.00059276	0.00097194	-0.00000114	0.00000000
		-6.27178224	-0.00096662	0.00059258	0.00097163	-0.00000114	0.00000000
		-6.27178326	-0.00096632	0.00059239	0.00097133	-0.00000114	0.00000000
		-6.27178429	-0.00096602	0.00059221	0.00097103	-0.00000114	0.00000000
		-6.27178531	-0.00096571	0.00059202	0.00097072	-0.00000114	0.00000000
		-6.27178633	-0.00096541	0.00059184	0.00097042	-0.00000114	0.00000000
		-6.27178735	-0.00096511	0.00059165	0.00097012	-0.00000114	0.00000000
		-6.27178836	-0.00096481	0.00059147	0.00096981	-0.00000114	0.00000000
		-6.27178938	-0.00096451	0.00059128	0.00096951	-0.00000114	0.00000000
		-6.27179039	-0.00096421	0.00059110	0.00096921	-0.00000114	0.00000000
		-6.27179140	-0.00096391	0.00059091	0.00096891	-0.00000114	0.00000000
		-6.27179242	-0.00096361	0.00059073	0.00096861	-0.00000114	0.00000000
		-6.27179342	-0.00096331	0.00059055	0.00096831	-0.00000114	0.00000000
		-6.27179443	-0.00096301	0.00059036	0.00096800	-0.00000114	0.00000000
		-6.27179544	-0.00096271	0.00059018	0.00096770	-0.00000114	0.00000000
		-6.27179644	-0.00096241	0.00058999	0.00096740	-0.00000114	0.00000000
		-6.27179744	-0.00096211	0.00058981	0.00096710	-0.00000114	0.00000000
		-6.27179844	-0.00096181	0.00058963	0.00096680	-0.00000114	0.00000000
		-6.27179944	-0.00096152	0.00058944	0.00096650	-0.00000114	0.00000000
		-6.27180044	-0.00096122	0.00058926	0.00096620	-0.00000114	0.00000000
		-6.27180143	-0.00096092	0.00058908	0.00096590	-0.00000114	0.00000000
		-6.27180243	-0.00096062	0.00058890	0.00096560	-0.00000114	0.00000000
		-6.27180342	-0.00096032	0.00058871	0.00096530	-0.00000114	0.00000000
		-6.27180441	-0.00096002	0.00058853	0.00096500	-0.00000114	0.00000000
		-6.27180540	-0.00095973	0.00058835	0.00096470	-0.00000114	0.00000000
		-6.27180639	-0.00095943	0.00058816	0.00096440	-0.00000114	0.00000000
		-6.27180737	-0.00095913	0.00058798	0.00096410	-0.00000114	0.00000000
		-6.27180836	-0.00095884	0.00058780	0.00096381	-0.00000114	0.00000000
		-6.27180934	-0.00095854	0.00058762	0.00096351	-0.00000114	0.00000000
		-6.27181032	-0.00095824	0.00058744	0.00096321	-0.00000114	0.00000000
		-6.27181130	-0.00095795	0.00058725	0.00096291	-0.00000114	0.00000000
		-6.27181228	-0.00095765	0.00058707	0.00096261	-0.00000114	0.00000000
		-6.27181325	-0.00095735	0.00058689	0.00096232	-0.00000114	0.00000000
		-6.27181423	-0.00095706	0.00058671	0.00096202	-0.00000114	0.00000000
		-6.27181520	-0.00095676	0.00058653	0.00096172	-0.00000114	0.00000000
		-6.27181618	-0.00095647	0.00058635	0.00096142	-0.00000114	0.00000000
		-6.27181715	-0.00095617	0.00058616	0.00096113	-0.00000114	0.00000000
		-6.27181811	-0.00095588	0.00058598	0.00096083	-0.00000114	0.00000000
		-6.27181908	-0.00095558	0.00058580	0.00096053	-0.00000114	0.00000000
		-6.27182005	-0.00095529	0.00058562	0.00096024	-0.00000114	0.00000000
		-6.27182101	-0.00095499	0.00058544	0.00095994	-0.00000114	0.00000000
		-6.27182198	-0.00095470	0.00058526	0.00095964	-0.00000114	0.00000000
		-6.27182294	-0.00095440	0.00058508	0.00095935	-0.00000114	0.00000000
		-6.27182390	-0.00095411	0.00058490	0.00095905	-0.00000114	0.00000000
		-6.27182486	-0.00095381	0.00058472	0.00095876	-0.00000114	0.00000000
		-6.27182581	-0.00095352	0.00058454	0.00095846	-0.00000114	0.00000000
		-6.27182677	-0.00095323	0.00058436	0.00095817	-0.00000114	0.00000000
		-6.27182772	-0.00095293	0.00058418	0.00095787	-0.00000114	0.00000000
		-6.27182867	-0.00095264	0.00058400	0.00095758	-0.00000114	0.00000000
		-6.27182963	-0.00095235	0.00058382	0.00095728	-0.00000114	0.00000000
		-6.27183058	-0.00095205	0.00058364	0.00095699	-0.00000114	0.00000000
		-6.27183152	-0.00095176	0.00058346	0.00095669	-0.00000114	0.00000000
		-6.27183247	-0.00095147	0.00058328	0.00095640	-0.00000114	0.00000000
		-6.27183342	-0.00095118	0.00058310	0.00095611	-0.00000114	0.00000000
		-6.27183436	-0.00095088	0.00058292	0.00095581	-0.00000114	0.00000000
		-6.27183530	-0.00095059	0.00058274	0.00095552	-0.00000114	0.00000000
		-6.27183624	-0.00095030	0.00058256	0.00095523	-0.00000114	0.00000000
		-6.27183718	-0.00095001	0.00058238	0.00095493	-0.00000114	0.00000000
		-6.27183812	-0.00094972	0.00058221	0.00095464	-0.00000114	0.00000000
		-6.27183906	-0.00094943	0.00058203	0.00095435	-0.00000114	0.00000000
		-6.27183999	-0.00094914	0.00058185	0.00095405	-0.00000114	0.00000000
		-6.27184093	-0.00094884	0.00058167	0.00095376	-0.00000114	0.00000000
		-6.27184186	-0.00094855	0.00058149	0.00095347	-0.00000114	0.00000000
		-6.27184279	-0.00094826	0.00058131	0.00095318	-0.00000114	0.00000000
		-6.27184372	-0.00094797	0.00058113	0.00095288	-0.00000114	0.00000000
		-6.27184465	-0.00094768	0.00058096	0.00095259	-0.00000114	0.00000000
		-6.27184558	-0.00094739	0.00058078	0.00095230	-0.00000114	0.00000000
		-6.27184650	-0.00094710	0.00058060	0.00095201	-0.00000114	0.00000000
		-6.27184743	-0.00094681	0.00058042	0.00095172	-0.00000114	0.00000000
		-6.27184835	-0.00094652	0.00058025	0.00095143	-0.00000114	0.00000000
		-6.27184927	-0.00094623	0.00058007	0.00095114	-0.00000114	0.00000000
		-6.27185019	-0.00094595	0.00057989	0.00095085	-0.00000114	0.00000000
		-6.27185111	-0.00094566	0.00057971	0.00095056	-0.00000114	0.00000000
		-6.27185203	-0.00094537	0.00057954	0.00095027	-0.00000114	0.00000000
		-6.27185294	-0.00094508	0.00057936	0.00094998	-0.00000114	0.00000000
		-6.27185386	-0.00094479	0.00057918	0.00094969	-0.00000114	0.00000000
		-6.27185477	-0.00094450	0.00057901	0.00094940	-0.00000114	0.00000000
		-6.27185568	-0.00094421	0.00057883	0.00094911	-0.00000114	0.00000000
		-6.27185659	-0.00094393	0.00057865	0.00094882	-0.00000114	0.00000000
		-6.27185750	-0.00094364	0.00057848	0.00094853	-0.00000114	0.00000000
		-6.27185841	-0.00094335	0.00057830	0.00094824	-0.00000114	0.00000000
		-6.27185932	-0.00094306	0.00057812	0.00094795	-0.00000114	0.00000000
		-6.27186022	-0.00094278	0.00057795	0.00094766	-0.00000114	0.00000000
		-6.27186113	-0.00094249	0.00057777	0.00094737	-0.00000114	0.00000000
		-6.27186203	-0.00094220	0.00057759	0.00094708	-0.00000114	0.00000000
		-6.27186293	-0.00094192	0.00057742	0.00094680	-0.00000114	0.00000000
		-6.27186383	-0.00094163	0.00057724	0.00094651	-0.00000114	0.00000000
		-6.27186473	-0.00094134	0.00057707	0.00094622	-0.00000114	0.00000000
		-6.27186563	-0.00094106	0.00057689	0.00094593	-0.00000114	0.00000000
		-6.27186652	-0.00094077	0.00057672	0.00094564	-0.00000114	0.00000000
		-6.27186742	-0.00094049	0.00057654	0.00094536	-0.00000114	0.00000000
		-6.27186831	-0.00094020	0.00057636	0.00094507	-0.00000114	0.00000000
		-6.27186920	-0.00093991	0.00057619	0.00094478	-0.00000114	0.00000000
		-6.27187010	-0.00093963	0.00057601	0.00094450	-0.00000114	0.00000000
		-6.27187098	-0.00093934	0.00057584	0.00094421	-0.00000114	0.00000000
		-6.27187187	-0.00093906	0.00057567	0.00094392	-0.00000114	0.00000000
		-6.27187276	-0.00093877	0.00057549	0.00094364	-0.00000114	0.00000000
		-6.27187365	-0.00093849	0.00057532	0.00094335	-0.00000114	0.00000000
		-6.27187453	-0.00093821	0.00057514	0.00094306	-0.00000114	0.00000000
		-6.27187541	-0.00093792	0.00057497	0.00094278	-0.00000114	0.00000000
		-6.27187630	-0.00093764	0.00057479	0.00094249	-0.00000114	0.00000000
		-6.27187718	-0.00093735	0.00057462	0.00094221	-0.00000114	0.00000000
		-6.27187806	-0.00093707	0.00057444	0.00094192	-0.00000114	0.00000000
		-6.27187894	-0.00093679	0.00057427	0.00094164	-0.00000114	0.00000000
		-6.27187981	-0.00093650	0.00057410	0.00094135	-0.00000114	0.00000000
		-6.27188069	-0.00093622	0.00057392	0.00094107	-0.00000114	0.00000000
		-6.27188156	-0.00093594	0.00057375	0.00094078	-0.00000114	0.00000000
		-6.27188244	-0.00093565	0.00057358	0.00094050	-0.00000114	0.00000000
		-6.27188331	-0.00093537	0.00057340	0.00094021	-0.00000114	0.00000000
		-6.27188418	-0.00093509	0.00057323	0.00093993	-0.00000114	0.00000000
		-6.27188505	-0.00093481	0.00057306	0.00093965	-0.00000114	0.00000000
		-6.27188592	-0.00093452	0.00057288	0.00093936	-0.00000114	0.00000000
		-6.27188679	-0.00093424	0.00057271	0.00093908	-0.00000114	0.00000000
		-6.27188765	-0.00093396	0.00057254	0.00093880	-0.00000114	0.00000000
		-6.27188852	-0.00093368	0.00057236	0.00093851	-0.00000114	0.00000000
		-6.27188938	-0.00093340	0.00057219	0.00093823	-0.00000114	0.00000000
		-6.27189025	-0.00093312	0.00057202	0.00093795	-0.00000114	0.00000000
		-6.27189111	-0.00093283	0.00057185	0.00093766	-0.00000114	0.00000000
		-6.27189197	-0.00093255	0.00057167	0.00093738	-0.00000114	0.00000000
		-6.27189283	-0.00093227	0.00057150	0.00093710	-0.00000114	0.00000000
		-6.27189369	-0.00093199	0.00057133	0.00093682	-0.00000114	0.00000000
		-6.27189454	-0.00093171	0.00057116	0.00093654	-0.00000114	0.00000000
		-6.27189540	-0.00093143	0.00057098	0.00093625	-0.00000114	0.00000000
		-6.27189625	-0.00093115	0.00057081	0.00093597	-0.00000114	0.00000000
		-6.27189711	-0.00093087	0.00057064	0.00093569	-0.00000114	0.00000000
		-6.27189796	-0.00093059	0.00057047	0.00093541	-0.00000114	0.00000000
		-6.27189881	-0.00093031	0.00057030	0.00093513	-0.00000114	0.00000000
		-6.27189966	-0.00093003	0.00057013	0.00093485	-0.00000114	0.00000000
		-6.27190051	-0.00092975	0.00056995	0.00093457	-0.00000114	0.00000000
		-6.27190136	-0.00092947	0.00056978	0.00093429	-0.00000114	0.00000000
		-6.27190220	-0.00092919	0.00056961	0.00093401	-0.00000114	0.00000000
		-6.27190305	-0.00092892	0.00056944	0.00093372	-0.00000114	0.00000000
		-6.27190389	-0.00092864	0.00056927	0.00093344	-0.00000114	0.00000000
		-6.27190474	-0.00092836	0.00056910	0.00093316	-0.00000114	0.00000000
		-6.27190558	-0.00092808	0.00056893	0.00093288	-0.00000114	0.00000000
		-6.27190642	-0.00092780	0.00056876	0.00093261	-0.00000114	0.00000000
		-6.27190726	-0.00092752	0.00056859	0.00093233	-0.00000114	0.00000000
		-6.27190810	-0.00092725	0.00056842	0.00093205	-0.00000114	0.00000000
		-6.27190893	-0.00092697	0.00056825	0.00093177	-0.00000114	0.00000000
		-6.27190977	-0.00092669	0.00056808	0.00093149	-0.00000114	0.00000000
		-6.27191061	-0.00092641	0.00056791	0.00093121	-0.00000114	0.00000000
		-6.27191144	-0.00092614	0.00056774	0.00093093	-0.00000114	0.00000000
		-6.27191227	-0.00092586	0.00056757	0.00093065	-0.00000114	0.00000000
		-6.27191311	-0.00092558	0.00056740	0.00093037	-0.00000114	0.00000000
		-6.27191394	-0.00092531	0.00056723	0.00093010	-0.00000114	0.00000000
		-6.27191477	-0.00092503	0.00056706	0.00092982	-0.00000114	0.00000000
		-6.27191559	-0.00092475	0.00056689	0.00092954	-0.00000114	0.00000000
		-6.27191642	-0.00092448	0.00056672	0.00092926	-0.00000114	0.00000000
		-6.27191725	-0.00092420	0.00056655	0.00092899	-0.00000114	0.00000000
		-6.27191807	-0.00092393	0.00056638	0.00092871	-0.00000114	0.00000000
		-6.27191890	-0.00092365	0.00056621	0.00092843	-0.00000114	0.00000000
		-6.27191972	-0.00092338	0.00056604	0.00092815	-0.00000114	0.00000000
		-6.27192054	-0.00092310	0.00056587	0.00092788	-0.00000114	0.00000000
		-6.27192136	-0.00092282	0.00056570	0.00092760	-0.00000114	0.00000000
		-6.27192218	-0.00092255	0.00056553	0.00092732	-0.00000114	0.00000000
		-6.27192300	-0.00092227	0.00056537	0.00092705	-0.00000114	0.00000000
		-6.27192382	-0.00092200	0.00056520	0.00092677	-0.00000114	0.00000000
		-6.27192464	-0.00092173	0.00056503	0.00092650	-0.00000114	0.00000000
		-6.27192545	-0.00092145	0.00056486	0.00092622	-0.00000114	0.00000000
		-6.27192627	-0.00092118	0.00056469	0.00092594	-0.00000114	0.00000000
		-6.27192708	-0.00092090	0.00056452	0.00092567	-0.00000114	0.00000000
		-6.27192790	-0.00092063	0.00056436	0.00092539	-0.00000114	0.00000000
		-6.27192871	-0.00092036	0.00056419	0.00092512	-0.00000114	0.00000000
		-6.27192952	-0.00092008	0.00056402	0.00092484	-0.00000114	0.00000000
		-6.27193033	-0.00091981	0.00056385	0.00092457	-0.00000114	0.00000000
		-6.27193114	-0.00091954	0.00056368	0.00092429	-0.00000114	0.00000000
		-6.27193194	-0.00091926	0.00056352	0.00092402	-0.00000114	0.00000000
		-6.27193275	-0.00091899	0.00056335	0.00092374	-0.00000114	0.00000000
		-6.27193356	-0.00091872	0.00056318	0.00092347	-0.00000114	0.00000000
		-6.27193436	-0.00091844	0.00056301	0.00092320	-0.00000114	0.00000000
		-6.27193516	-0.00091817	0.00056285	0.00092292	-0.00000114	0.00000000
		-6.27193597	-0.00091790	0.00056268	0.00092265	-0.00000114	0.00000000
		-6.27193677	-0.00091763	0.00056251	0.00092238	-0.00000114	0.00000000
		-6.27193757	-0.00091736	0.00056235	0.00092210	-0.00000114	0.00000000
		-6.27193837	-0.00091708	0.00056218	0.00092183	-0.00000114	0.00000000
		-6.27193917	-0.00091681	0.00056201	0.00092156	-0.00000114	0.00000000
		-6.27193996	-0.00091654	0.00056185	0.00092128	-0.00000114	0.00000000
		-6.27194076	-0.00091627	0.00056168	0.00092101	-0.00000114	0.00000000
		-6.27194155	-0.00091600	0.00056151	0.00092074	-0.00000114	0.00000000
		-6.27194235	-0.00091573	0.00056135	0.00092046	-0.00000114	0.00000000
		-6.27194314	-0.00091546	0.00056118	0.00092019	-0.00000114	0.00000000
		-6.27194394	-0.00091519	0.00056102	0.00091992	-0.00000114	0.00000000
		-6.27194473	-0.00091492	0.00056085	0.00091965	-0.00000114	0.00000000
		-6.27194552	-0.00091465	0.00056068	0.00091938	-0.00000114	0.00000000
		-6.27194631	-0.00091438	0.00056052	0.00091911	-0.00000114	0.00000000
		-6.27194710	-0.00091411	0.00056035	0.00091883	-0.00000114	0.00000000
		-6.27194788	-0.00091384	0.00056019	0.00091856	-0.00000114	0.00000000
		-6.27194867	-0.00091357	0.00056002	0.00091829	-0.00000114	0.00000000
		-6.27194946	-0.00091330	0.00055986	0.00091802	-0.00000114	0.00000000
		-6.27195024	-0.00091303	0.00055969	0.00091775	-0.00000114	0.00000000
		-6.27195102	-0.00091276	0.00055953	0.00091748	-0.00000114	0.00000000
		-6.27195181	-0.00091249	0.00055936	0.00091721	-0.00000114	0.00000000
		-6.27195259	-0.00091222	0.00055920	0.00091694	-0.00000114	0.00000000
		-6.27195337	-0.00091195	0.00055903	0.00091667	-0.00000114	0.00000000
		-6.27195415	-0.00091168	0.00055887	0.00091640	-0.00000114	0.00000000
		-6.27195493	-0.00091141	0.00055870	0.00091613	-0.00000114	0.00000000
		-6.27195571	-0.00091115	0.00055854	0.00091586	-0.00000114	0.00000000
		-6.27195648	-0.00091088	0.00055837	0.00091559	-0.00000114	0.00000000
		-6.27195726	-0.00091061	0.00055821	0.00091532	-0.00000114	0.00000000
		-6.27195803	-0.00091034	0.00055804	0.00091505	-0.00000114	0.00000000
		-6.27195881	-0.00091007	0.00055788	0.00091478	-0.00000114	0.00000000
		-6.27195958	-0.00090981	0.00055771	0.00091451	-0.00000114	0.00000000
		-6.27196035	-0.00090954	0.00055755	0.00091424	-0.00000114	0.00000000
		-6.27196113	-0.00090927	0.00055739	0.00091397	-0.00000114	0.00000000
		-6.27196190	-0.00090901	0.00055722	0.00091371	-0.00000114	0.00000000
		-6.27196267	-0.00090874	0.00055706	0.00091344	-0.00000114	0.00000000
		-6.27196344	-0.00090847	0.00055690	0.00091317	-0.00000114	0.00000000
		-6.27196420	-0.00090821	0.00055673	0.00091290	-0.00000114	0.00000000
		-6.27196497	-0.00090794	0.00055657	0.00091263	-0.00000114	0.00000000
		-6.27196574	-0.00090767	0.00055641	0.00091237	-0.00000114	0.00000000
		-6.27196650	-0.00090741	0.00055624	0.00091210	-0.00000114	0.00000000
		-6.27196726	-0.00090714	0.00055608	0.00091183	-0.00000114	0.00000000
		-6.27196803	-0.00090688	0.00055592	0.00091156	-0.00000114	0.00000000
		-6.27196879	-0.00090661	0.00055575	0.00091130	-0.00000114	0.00000000
		-6.27196955	-0.00090634	0.00055559	0.00091103	-0.00000114	0.00000000
		-6.27197031	-0.00090608	0.00055543	0.00091076	-0.00000114	0.00000000
		-6.27197107	-0.00090581	0.00055526	0.00091050	-0.00000114	0.00000000
		-6.27197183	-0.00090555	0.00055510	0.00091023	-0.00000114	0.00000000
		-6.27197259	-0.00090528	0.00055494	0.00090997	-0.00000114	0.00000000
		-6.27197335	-0.00090502	0.00055478	0.00090970	-0.00000114	0.00000000
		-6.27197410	-0.00090476	0.00055461	0.00090943	-0.00000114	0.00000000
		-6.27197486	-0.00090449	0.00055445	0.00090917	-0.00000114	0.00000000
		-6.27197561	-0.00090423	0.00055429	0.00090890	-0.00000114	0.00000000
		-6.27197637	-0.00090396	0.00055413	0.00090864	-0.00000114	0.00000000
		-6.27197712	-0.00090370	0.00055397	0.00090837	-0.00000114	0.00000000
		-6.27197787	-0.00090344	0.00055380	0.00090811	-0.00000114	0.00000000
		-6.27197862	-0.00090317	0.00055364	0.00090784	-0.00000114	0.00000000
		-6.27197937	-0.00090291	0.00055348	0.00090758	-0.00000114	0.00000000
		-6.27198012	-0.00090264	0.00055332	0.00090731	-0.00000114	0.00000000
		-6.27198087	-0.00090238	0.00055316	0.00090705	-0.00000114	0.00000000
		-6.27198162	-0.00090212	0.00055300	0.00090678	-0.00000114	0.00000000
		-6.27198236	-0.00090186	0.00055284	0.00090652	-0.00000114	0.00000000
		-6.27198311	-0.00090159	0.00055267	0.00090625	-0.00000114	0.00000000
		-6.27198385	-0.00090133	0.00055251	0.00090599	-0.00000113	0.00000000
		-6.27198460	-0.00090107	0.00055235	0.00090573	-0.00000113	0.00000000
		-6.27198534	-0.00090081	0.00055219	0.00090546	-0.00000113	0.00000000
		-6.27198608	-0.00090054	0.00055203	0.00090520	-0.00000113	0.00000000
		-6.27198682	-0.00090028	0.00055187	0.00090494	-0.00000113	0.00000000
		-6.27198756	-0.00090002	0.00055171	0.00090467	-0.00000113	0.00000000
		-6.27198830	-0.00089976	0.00055155	0.00090441	-0.00000113	0.00000000
		-6.27198904	-0.00089950	0.00055139	0.00090415	-0.00000113	0.00000000
		-6.27198978	-0.00089924	0.00055123	0.00090388	-0.00000113	0.00000000
		-6.27199052	-0.00089898	0.00055107	0.00090362	-0.00000113	0.00000000
		-6.27199126	-0.00089871	0.00055091	0.00090336	-0.00000113	0.00000000
		-6.27199199	-0.00089845	0.00055075	0.00090310	-0.00000113	0.00000000
		-6.27199273	-0.00089819	0.00055059	0.00090284	-0.00000113	0.00000000
		-6.27199346	-0.00089793	0.00055043	0.00090257	-0.00000113	0.00000000
		-6.27199419	-0.00089767	0.00055027	0.00090231	-0.00000113	0.00000000
		-6.27199493	-0.00089741	0.00055011	0.00090205	-0.00000113	0.00000000
		-6.27199566	-0.00089715	0.00054995	0.00090179	-0.00000113	0.00000000
		-6.27199639	-0.00089689	0.00054979	0.00090153	-0.00000113	0.00000000
		-6.27199712	-0.00089663	0.00054963	0.00090127	-0.00000113	0.00000000
		-6.27199785	-0.00089637	0.00054947	0.00090100	-0.00000113	0.00000000
		-6.27199857	-0.00089611	0.00054931	0.00090074	-0.00000113	0.00000000
		-6.27199930	-0.00089585	0.00054915	0.00090048	-0.00000113	0.00000000
		-6.27200003	-0.00089559	0.00054899	0.00090022	-0.00000113	0.00000000
		-6.27200076	-0.00089534	0.00054883	0.00089996	-0.00000113	0.00000000
		-6.27200148	-0.00089508	0.00054868	0.00089970	-0.00000113	0.00000000
		-6.27200220	-0.00089482	0.00054852	0.00089944	-0.00000113	0.00000000
		-6.27200293	-0.00089456	0.00054836	0.00089918	-0.00000113	0.00000000
		-6.27200365	-0.00089430	0.00054820	0.00089892	-0.00000113	0.00000000
		-6.27200437	-0.00089404	0.00054804	0.00089866	-0.00000113	0.00000000
		-6.27200509	-0.00089378	0.00054788	0.00089840	-0.00000113	0.00000000
		-6.27200581	-0.00089353	0.00054772	0.00089814	-0.00000113	0.00000000
		-6.27200653	-0.00089327	0.00054757	0.00089788	-0.00000113	0.00000000
		-6.27200725	-0.00089301	0.00054741	0.00089763	-0.00000113	0.00000000
		-6.27200797	-0.00089275	0.00054725	0.00089737	-0.00000113	0.00000000
		-6.27200869	-0.00089250	0.00054709	0.00089711	-0.00000113	0.00000000
		-6.27200940	-0.00089224	0.00054693	0.00089685	-0.00000113	0.00000000
		-6.27201012	-0.00089198	0.00054678	0.00089659	-0.00000113	0.00000000
		-6.27201083	-0.00089172	0.00054662	0.00089633	-0.00000113	0.00000000
		-6.27201155	-0.00089147	0.00054646	0.00089607	-0.00000113	0.00000000
		-6.27201226	-0.00089121	0.00054630	0.00089582	-0.00000113	0.00000000
		-6.27201297	-0.00089095	0.00054615	0.00089556	-0.00000113	0.00000000
		-6.27201369	-0.00089070	0.00054599	0.00089530	-0.00000113	0.00000000
		-6.27201440	-0.00089044	0.00054583	0.00089504	-0.00000113	0.00000000
		-6.27201511	-0.00089019	0.00054568	0.00089479	-0.00000113	0.00000000
		-6.27201582	-0.00088993	0.00054552	0.00089453	-0.00000113	0.00000000
		-6.27201653	-0.00088967	0.00054536	0.00089427	-0.00000113	0.00000000
		-6.27201723	-0.00088942	0.00054520	0.00089401	-0.00000113	0.00000000
		-6.27201794	-0.00088916	0.00054505	0.00089376	-0.00000113	0.00000000
		-6.27201865	-0.00088891	0.00054489	0.00089350	-0.00000113	0.00000000
		-6.27201935	-0.00088865	0.00054473	0.00089324	-0.00000113	0.00000000
		-6.27202006	-0.00088840	0.00054458	0.00089299	-0.00000113	0.00000000
		-6.27202076	-0.00088814	0.00054442	0.00089273	-0.00000113	0.00000000
		-6.27202147	-0.00088789	0.00054427	0.00089248	-0.00000113	0.00000000
		-6.27202217	-0.00088763	0.00054411	0.00089222	-0.00000113	0.00000000
		-6.27202287	-0.00088738	0.00054395	0.00089196	-0.00000113	0.00000000
		-6.27202357	-0.00088713	0.00054380	0.00089171	-0.00000113	0.00000000
		-6.27202427	-0.00088687	0.00054364	0.00089145	-0.00000113	0.00000000
		-6.27202497	-0.00088662	0.00054349	0.00089120	-0.00000113	0.00000000
		-6.27202567	-0.00088636	0.00054333	0.00089094	-0.00000113	0.00000000
		-6.27202637	-0.00088611	0.00054317	0.00089069	-0.00000113	0.00000000
		-6.27202707	-0.00088586	0.00054302	0.00089043	-0.00000113	0.00000000
		-6.27202777	-0.00088560	0.00054286	0.00089018	-0.00000113	0.00000000
		-6.27202846	-0.00088535	0.00054271	0.00088992	-0.00000113	0.00000000
		-6.27202916	-0.00088510	0.00054255	0.00088967	-0.00000113	0.00000000
		-6.27202985	-0.00088484	0.00054240	0.00088941	-0.00000113	0.00000000
		-6.27203055	-0.00088459	0.00054224	0.00088916	-0.00000113	0.00000000
		-6.27203124	-0.00088434	0.00054209	0.00088891	-0.00000113	0.00000000
		-6.27203193	-0.00088409	0.00054193	0.00088865	-0.00000113	0.00000000
		-6.27203262	-0.00088383	0.00054178	0.00088840	-0.00000113	0.00000000
		-6.27203332	-0.00088358	0.00054162	0.00088814	-0.00000113	0.00000000
		-6.27203401	-0.00088333	0.00054147	0.00088789	-0.00000113	0.00000000
		-6.27203470	-0.00088308	0.00054131	0.00088764	-0.00000113	0.00000000
		-6.27203539	-0.00088282	0.00054116	0.00088738	-0.00000113	0.00000000
		-6.27203607	-0.00088257	0.00054100	0.00088713	-0.00000113	0.00000000
		-6.27203676	-0.00088232	0.00054085	0.00088688	-0.00000113	0.00000000
		-6.27203745	-0.00088207	0.00054069	0.00088663	-0.00000113	0.00000000
		-6.27203813	-0.00088182	0.00054054	0.00088637	-0.00000113	0.00000000
		-6.27203882	-0.00088157	0.00054039	0.00088612	-0.00000113	0.00000000
		-6.27203950	-0.00088132	0.00054023	0.00088587	-0.00000113	0.00000000
		-6.27204019	-0.00088107	0.00054008	0.00088562	-0.00000113	0.00000000
		-6.27204087	-0.00088082	0.00053992	0.00088536	-0.00000113	0.00000000
		-6.27204156	-0.00088056	0.00053977	0.00088511	-0.00000113	0.00000000
		-6.27204224	-0.00088031	0.00053962	0.00088486	-0.00000113	0.00000000
		-6.27204292	-0.00088006	0.00053946	0.00088461	-0.00000113	0.00000000
		-6.27204360	-0.00087981	0.00053931	0.00088436	-0.00000113	0.00000000
		-6.27204428	-0.00087956	0.00053916	0.00088411	-0.00000113	0.00000000
		-6.27204496	-0.00087931	0.00053900	0.00088385	-0.00000113	0.00000000
		-6.27204564	-0.00087906	0.00053885	0.00088360	-0.00000113	0.00000000
		-6.27204632	-0.00087881	0.00053870	0.00088335	-0.00000113	0.00000000
		-6.27204699	-0.00087857	0.00053854	0.00088310	-0.00000113	0.00000000
		-6.27204767	-0.00087832	0.00053839	0.00088285	-0.00000113	0.00000000
		-6.27204835	-0.00087807	0.00053824	0.00088260	-0.00000113	0.00000000
		-6.27204902	-0.00087782	0.00053809	0.00088235	-0.00000113	0.00000000
		-6.27204970	-0.00087757	0.00053793	0.00088210	-0.00000113	0.00000000
		-6.27205037	-0.00087732	0.00053778	0.00088185	-0.00000113	0.00000000
		-6.27205104	-0.00087707	0.00053763	0.00088160	-0.00000113	0.00000000
		-6.27205172	-0.00087682	0.00053748	0.00088135	-0.00000113	0.00000000
		-6.27205239	-0.00087658	0.00053732	0.00088110	-0.00000113	0.00000000
		-6.27205306	-0.00087633	0.00053717	0.00088085	-0.00000113	0.00000000
		-6.27205373	-0.00087608	0.00053702	0.00088060	-0.00000113	0.00000000
		-6.27205440	-0.00087583	0.00053687	0.00088035	-0.00000113	0.00000000
		-6.27205507	-0.00087558	0.00053671	0.00088010	-0.00000113	0.00000000
		-6.27205574	-0.00087534	0.00053656	0.00087985	-0.00000113	0.00000000
		-6.27205641	-0.00087509	0.00053641	0.00087961	-0.00000113	0.00000000
		-6.27205707	-0.00087484	0.00053626	0.00087936	-0.00000113	0.00000000
		-6.27205774	-0.00087459	0.00053611	0.00087911	-0.00000113	0.00000000
		-6.27205841	-0.00087435	0.00053596	0.00087886	-0.00000113	0.00000000
		-6.27205907	-0.00087410	0.00053580	0.00087861	-0.00000113	0.00000000
		-6.27205974	-0.00087385	0.00053565	0.00087836	-0.00000113	0.00000000
		-6.27206040	-0.00087361	0.00053550	0.00087812	-0.00000113	0.00000000
		-6.27206106	-0.00087336	0.00053535	0.00087787	-0.00000113	0.00000000
		-6.27206173	-0.00087311	0.00053520	0.00087762	-0.00000113	0.00000000
		-6.27206239	-0.00087287	0.00053505	0.00087737	-0.00000113	0.00000000
		-6.27206305	-0.00087262	0.00053490	0.00087713	-0.00000113	0.00000000
		-6.27206371	-0.00087238	0.00053475	0.00087688	-0.00000113	0.00000000
		-6.27206437	-0.00087213	0.00053460	0.00087663	-0.00000113	0.00000000
		-6.27206503	-0.00087188	0.00053445	0.00087638	-0.00000113	0.00000000
		-6.27206569	-0.00087164	0.00053429	0.00087614	-0.00000113	0.00000000
		-6.27206635	-0.00087139	0.00053414	0.00087589	-0.00000113	0.00000000
		-6.27206701	-0.00087115	0.00053399	0.00087564	-0.00000113	0.00000000
		-6.27206767	-0.00087090	0.00053384	0.00087540	-0.00000113	0.00000000
		-6.27206832	-0.00087066	0.00053369	0.00087515	-0.00000113	0.00000000
		-6.27206898	-0.00087041	0.00053354	0.00087491	-0.00000113	0.00000000
		-6.27206963	-0.00087017	0.00053339	0.00087466	-0.00000113	0.00000000
		-6.27207029	-0.00086992	0.00053324	0.00087441	-0.00000113	0.00000000
		-6.27207094	-0.00086968	0.00053309	0.00087417	-0.00000113	0.00000000
		-6.27207160	-0.00086944	0.00053294	0.00087392	-0.00000113	0.00000000
		-6.27207225	-0.00086919	0.00053279	0.00087368	-0.00000113	0.00000000
		-6.27207290	-0.00086895	0.00053264	0.00087343	-0.00000113	0.00000000
		-6.27207355	-0.00086870	0.00053249	0.00087319	-0.00000113	0.00000000
		-6.27207420	-0.00086846	0.00053234	0.00087294	-0.00000113	0.00000000
		-6.27207485	-0.00086822	0.00053219	0.00087270	-0.00000113	0.00000000
		-6.27207550	-0.00086797	0.00053205	0.00087245	-0.00000113	0.00000000
		-6.27207615	-0.00086773	0.00053190	0.00087221	-0.00000113	0.00000000
		-6.27207680	-0.00086749	0.00053175	0.00087196	-0.00000113	0.00000000
		-6.27207745	-0.00086724	0.00053160	0.00087172	-0.00000113	0.00000000
		-6.27207810	-0.00086700	0.00053145	0.00087147	-0.00000113	0.00000000
		-6.27207874	-0.00086676	0.00053130	0.00087123	-0.00000113	0.00000000
		-6.27207939	-0.00086652	0.00053115	0.00087099	-0.00000113	0.00000000
		-6.27208004	-0.00086627	0.00053100	0.00087074	-0.00000113	0.00000000
		-6.27208068	-0.00086603	0.00053085	0.00087050	-0.00000113	0.00000000
		-6.27208132	-0.00086579	0.00053070	0.00087026	-0.00000113	0.00000000
		-6.27208197	-0.00086555	0.00053056	0.00087001	-0.00000113	0.00000000
		-6.27208261	-0.00086530	0.00053041	0.00086977	-0.00000113	0.00000000
		-6.27208325	-0.00086506	0.00053026	0.00086953	-0.00000113	0.00000000
		-6.27208390	-0.00086482	0.00053011	0.00086928	-0.00000113	0.00000000
		-6.27208454	-0.00086458	0.00052996	0.00086904	-0.00000113	0.00000000
		-6.27208518	-0.00086434	0.00052981	0.00086880	-0.00000113	0.00000000
		-6.27208582	-0.00086410	0.00052967	0.00086856	-0.00000113	0.00000000
		-6.27208646	-0.00086386	0.00052952	0.00086831	-0.00000113	0.00000000
		-6.27208710	-0.00086361	0.00052937	0.00086807	-0.00000113	0.00000000
		-6.27208774	-0.00086337	0.00052922	0.00086783	-0.00000113	0.00000000
		-6.27208837	-0.00086313	0.00052908	0.00086759	-0.00000113	0.00000000
		-6.27208901	-0.00086289	0.00052893	0.00086734	-0.00000113	0.00000000
		-6.27208965	-0.00086265	0.00052878	0.00086710	-0.00000113	0.00000000
		-6.27209028	-0.00086241	0.00052863	0.00086686	-0.00000113	0.00000000
		-6.27209092	-0.00086217	0.00052849	0.00086662	-0.00000113	0.00000000
		-6.27209155	-0.00086193	0.00052834	0.00086638	-0.00000113	0.00000000
		-6.27209219	-0.00086169	0.00052819	0.00086614	-0.00000113	0.00000000
		-6.27209282	-0.00086145	0.00052804	0.00086590	-0.00000113	0.00000000
		-6.27209346	-0.00086121	0.00052790	0.00086565	-0.00000113	0.00000000
		-6.27209409	-0.00086097	0.00052775	0.00086541	-0.00000113	0.00000000
		-6.27209472	-0.00086073	0.00052760	0.00086517	-0.00000113	0.00000000
		-6.27209535	-0.00086049	0.00052746	0.00086493	-0.00000113	0.00000000
		-6.27209598	-0.00086025	0.00052731	0.00086469	-0.00000113	0.00000000
		-6.27209661	-0.00086002	0.00052716	0.00086445	-0.00000113	0.00000000
		-6.27209724	-0.00085978	0.00052702	0.00086421	-0.00000113	0.00000000
		-6.27209787	-0.00085954	0.00052687	0.00086397	-0.00000113	0.00000000
		-6.27209850	-0.00085930	0.00052672	0.00086373	-0.00000113	0.00000000
		-6.27209913	-0.00085906	0.00052658	0.00086349	-0.00000113	0.00000000
		-6.27209976	-0.00085882	0.00052643	0.00086325	-0.00000113	0.00000000
		-6.27210038	-0.00085858	0.00052628	0.00086301	-0.00000113	0.00000000
		-6.27210101	-0.00085835	0.00052614	0.00086277	-0.00000113	0.00000000
		-6.27210164	-0.00085811	0.00052599	0.00086253	-0.00000113	0.00000000
		-6.27210226	-0.00085787	0.00052585	0.00086230	-0.00000113	0.00000000
		-6.27210289	-0.00085763	0.00052570	0.00086206	-0.00000113	0.00000000
		-6.27210351	-0.00085740	0.00052555	0.00086182	-0.00000113	0.00000000
		-6.27210414	-0.00085716	0.00052541	0.00086158	-0.00000113	0.00000000
		-6.27210476	-0.00085692	0.00052526	0.00086134	-0.00000113	0.00000000
		-6.27210538	-0.00085668	0.00052512	0.00086110	-0.00000113	0.00000000
		-6.27210600	-0.00085645	0.00052497	0.00086086	-0.00000113	0.00000000
		-6.27210663	-0.00085621	0.00052483	0.00086063	-0.00000113	0.00000000
		-6.27210725	-0.00085597	0.00052468	0.00086039	-0.00000113	0.00000000
		-6.27210787	-0.00085574	0.00052454	0.00086015	-0.00000113	0.00000000
		-6.27210849	-0.00085550	0.00052439	0.00085991	-0.00000113	0.00000000
		-6.27210911	-0.00085526	0.00052425	0.00085967	-0.00000113	0.00000000
		-6.27210973	-0.00085503	0.00052410	0.00085944	-0.00000113	0.00000000
		-6.27211034	-0.00085479	0.00052396	0.00085920	-0.00000113	0.00000000
		-6.27211096	-0.00085456	0.00052381	0.00085896	-0.00000113	0.00000000
		-6.27211158	-0.00085432	0.00052367	0.00085873	-0.00000113	0.00000000
		-6.27211220	-0.00085408	0.00052352	0.00085849	-0.00000113	0.00000000
		-6.27211281	-0.00085385	0.00052338	0.00085825	-0.00000113	0.00000000
		-6.27211343	-0.00085361	0.00052323	0.00085801	-0.00000113	0.00000000
		-6.27211404	-0.00085338	0.00052309	0.00085778	-0.00000113	0.00000000
		-6.27211466	-0.00085314	0.00052295	0.00085754	-0.00000113	0.00000000
		-6.27211527	-0.00085291	0.00052280	0.00085731	-0.00000113	0.00000000
		-6.27211589	-0.00085267	0.00052266	0.00085707	-0.00000113	0.00000000
		-6.27211650	-0.00085244	0.00052251	0.00085683	-0.00000113	0.00000000
		-6.27211711	-0.00085220	0.00052237	0.00085660	-0.00000113	0.00000000
		-6.27211772	-0.00085197	0.00052222	0.00085636	-0.00000113	0.00000000
		-6.27211833	-0.00085173	0.00052208	0.00085613	-0.00000113	0.00000000
		-6.27211895	-0.00085150	0.00052194	0.00085589	-0.00000113	0.00000000
		-6.27211956	-0.00085127	0.00052179	0.00085565	-0.00000113	0.00000000
		-6.27212017	-0.00085103	0.00052165	0.00085542	-0.00000113	0.00000000
		-6.27212078	-0.00085080	0.00052151	0.00085518	-0.00000113	0.00000000
		-6.27212138	-0.00085056	0.00052136	0.00085495	-0.00000113	0.00000000
		-6.27212199	-0.00085033	0.00052122	0.00085471	-0.00000113	0.00000000
		-6.27212260	-0.00085010	0.00052108	0.00085448	-0.00000113	0.00000000
		-6.27212321	-0.00084986	0.00052093	0.00085425	-0.00000113	0.00000000
		-6.27212381	-0.00084963	0.00052079	0.00085401	-0.00000113	0.00000000
		-6.27212442	-0.00084940	0.00052065	0.00085378	-0.00000113	0.00000000
		-6.27212503	-0.00084916	0.00052050	0.00085354	-0.00000113	0.00000000
		-6.27212563	-0.00084893	0.00052036	0.00085331	-0.00000113	0.00000000
		-6.27212624	-0.00084870	0.00052022	0.00085307	-0.00000113	0.00000000
		-6.27212684	-0.00084847	0.00052008	0.00085284	-0.00000113	0.00000000
		-6.27212744	-0.00084823	0.00051993	0.00085261	-0.00000113	0.00000000
		-6.27212805	-0.00084800	0.00051979	0.00085237	-0.00000113	0.00000000
		-6.27212865	-0.00084777	0.00051965	0.00085214	-0.00000113	0.00000000
		-6.27212925	-0.00084754	0.00051951	0.00085191	-0.00000113	0.00000000
		-6.27212986	-0.00084730	0.00051936	0.00085167	-0.00000113	0.00000000
		-6.27213046	-0.00084707	0.00051922	0.00085144	-0.00000113	0.00000000
		-6.27213106	-0.00084684	0.00051908	0.00085121	-0.00000113	0.00000000
		-6.27213166	-0.00084661	0.00051894	0.00085097	-0.00000113	0.00000000
		-6.27213226	-0.00084638	0.00051879	0.00085074	-0.00000113	0.00000000
		-6.27213286	-0.00084615	0.00051865	0.00085051	-0.00000113	0.00000000
		-6.27213345	-0.00084592	0.00051851	0.00085028	-0.00000113	0.00000000
		-6.27213405	-0.00084568	0.00051837	0.00085004	-0.00000113	0.00000000
		-6.27213465	-0.00084545	0.00051823	0.00084981	-0.00000113	0.00000000
		-6.27213525	-0.00084522	0.00051809	0.00084958	-0.00000113	0.00000000
		-6.27213585	-0.00084499	0.00051794	0.00084935	-0.00000113	0.00000000
		-6.27213644	-0.00084476	0.00051780	0.00084912	-0.00000113	0.00000000
		-6.27213704	-0.00084453	0.00051766	0.00084888	-0.00000113	0.00000000
		-6.27213763	-0.00084430	0.00051752	0.00084865	-0.00000113	0.00000000
		-6.27213823	-0.00084407	0.00051738	0.00084842	-0.00000113	0.00000000
		-6.27213882	-0.00084384	0.00051724	0.00084819	-0.00000113	0.00000000
		-6.27213942	-0.00084361	0.00051710	0.00084796	-0.00000113	0.00000000
		-6.27214001	-0.00084338	0.00051696	0.00084773	-0.00000113	0.00000000
		-6.27214060	-0.00084315	0.00051681	0.00084750	-0.00000113	0.00000000
		-6.27214120	-0.00084292	0.00051667	0.00084727	-0.00000113	0.00000000
		-6.27214179	-0.00084269	0.00051653	0.00084703	-0.00000113	0.00000000
		-6.27214238	-0.00084246	0.00051639	0.00084680	-0.00000113	0.00000000
		-6.27214297	-0.00084223	0.00051625	0.00084657	-0.00000113	0.00000000
		-6.27214356	-0.00084200	0.00051611	0.00084634	-0.00000113	0.00000000
		-6.27214415	-0.00084178	0.00051597	0.00084611	-0.00000113	0.00000000
		-6.27214474	-0.00084155	0.00051583	0.00084588	-0.00000113	0.00000000
		-6.27214533	-0.00084132	0.00051569	0.00084565	-0.00000113	0.00000000
		-6.27214592	-0.00084109	0.00051555	0.00084542	-0.00000113	0.00000000
		-6.27214651	-0.00084086	0.00051541	0.00084519	-0.00000113	0.00000000
		-6.27214709	-0.00084063	0.00051527	0.00084496	-0.00000113	0.00000000
		-6.27214768	-0.00084040	0.00051513	0.00084473	-0.00000113	0.00000000
		-6.27214827	-0.00084018	0.00051499	0.00084451	-0.00000113	0.00000000
		-6.27214885	-0.00083995	0.00051485	0.00084428	-0.00000113	0.00000000
		-6.27214944	-0.00083972	0.00051471	0.00084405	-0.00000113	0.00000000
		-6.27215002	-0.00083949	0.00051457	0.00084382	-0.00000113	0.00000000
		-6.27215061	-0.00083926	0.00051443	0.00084359	-0.00000113	0.00000000
		-6.27215119	-0.00083904	0.00051429	0.00084336	-0.00000113	0.00000000
		-6.27215178	-0.00083881	0.00051415	0.00084313	-0.00000113	0.00000000
		-6.27215236	-0.00083858	0.00051401	0.00084290	-0.00000113	0.00000000
		-6.27215294	-0.00083836	0.00051387	0.00084268	-0.00000113	0.00000000
		-6.27215353	-0.00083813	0.00051373	0.00084245	-0.00000113	0.00000000
		-6.27215411	-0.00083790	0.00051359	0.00084222	-0.00000113	0.00000000
		-6.27215469	-0.00083768	0.00051345	0.00084199	-0.00000113	0.00000000
		-6.27215527	-0.00083745	0.00051332	0.00084176	-0.00000113	0.00000000
		-6.27215585	-0.00083722	0.00051318	0.00084154	-0.00000113	0.00000000
		-6.27215643	-0.00083700	0.00051304	0.00084131	-0.00000113	0.00000000
		-6.27215701	-0.00083677	0.00051290	0.00084108	-0.00000113	0.00000000
		-6.27215759	-0.00083654	0.00051276	0.00084085	-0.00000113	0.00000000
		-6.27215817	-0.00083632	0.00051262	0.00084063	-0.00000113	0.00000000
		-6.27215875	-0.00083609	0.00051248	0.00084040	-0.00000113	0.00000000
		-6.27215933	-0.00083587	0.00051234	0.00084017	-0.00000113	0.00000000
		-6.27215990	-0.00083564	0.00051221	0.00083995	-0.00000113	0.00000000
		-6.27216048	-0.00083541	0.00051207	0.00083972	-0.00000113	0.00000000
		-6.27216106	-0.00083519	0.00051193	0.00083949	-0.00000113	0.00000000
		-6.27216164	-0.00083496	0.00051179	0.00083927	-0.00000113	0.00000000
		-6.27216221	-0.00083474	0.00051165	0.00083904	-0.00000113	0.00000000
		-6.27216279	-0.00083451	0.00051151	0.00083881	-0.00000113	0.00000000
		-6.27216336	-0.00083429	0.00051138	0.00083859	-0.00000113	0.00000000
		-6.27216394	-0.00083406	0.00051124	0.00083836	-0.00000113	0.00000000
		-6.27216451	-0.00083384	0.00051110	0.00083813	-0.00000113	0.00000000
		-6.27216508	-0.00083361	0.00051096	0.00083791	-0.00000113	0.00000000
		-6.27216566	-0.00083339	0.00051083	0.00083768	-0.00000113	0.00000000
		-6.27216623	-0.00083317	0.00051069	0.00083746	-0.00000113	0.00000000
		-6.27216680	-0.00083294	0.00051055	0.00083723	-0.00000113	0.00000000
		-6.27216737	-0.00083272	0.00051041	0.00083701	-0.00000113	0.00000000
		-6.27216794	-0.00083249	0.00051028	0.00083678	-0.00000113	0.00000000
		-6.27216851	-0.00083227	0.00051014	0.00083656	-0.00000113	0.00000000
		-6.27216909	-0.00083205	0.00051000	0.00083633	-0.00000113	0.00000000
		-6.27216966	-0.00083182	0.00050986	0.00083611	-0.00000113	0.00000000
		-6.27217022	-0.00083160	0.00050973	0.00083588	-0.00000113	0.00000000
		-6.27217079	-0.00083138	0.00050959	0.00083566	-0.00000113	0.00000000
		-6.27217136	-0.00083115	0.00050945	0.00083543	-0.00000113	0.00000000
		-6.27217193	-0.00083093	0.00050932	0.00083521	-0.00000113	0.00000000
		-6.27217250	-0.00083071	0.00050918	0.00083499	-0.00000113	0.00000000
		-6.27217307	-0.00083048	0.00050904	0.00083476	-0.00000113	0.00000000
		-6.27217363	-0.00083026	0.00050891	0.00083454	-0.00000113	0.00000000
		-6.27217420	-0.00083004	0.00050877	0.00083431	-0.00000113	0.00000000
		-6.27217477	-0.00082982	0.00050863	0.00083409	-0.00000113	0.00000000
		-6.27217533	-0.00082959	0.00050850	0.00083387	-0.00000113	0.00000000
		-6.27217590	-0.00082937	0.00050836	0.00083364	-0.00000113	0.00000000
		-6.27217646	-0.00082915	0.00050822	0.00083342	-0.00000113	0.00000000
		-6.27217703	-0.00082893	0.00050809	0.00083320	-0.00000113	0.00000000
		-6.27217759	-0.00082871	0.00050795	0.00083297	-0.00000113	0.00000000
		-6.27217815	-0.00082848	0.00050782	0.00083275	-0.00000113	0.00000000
		-6.27217872	-0.00082826	0.00050768	0.00083253	-0.00000113	0.00000000
		-6.27217928	-0.00082804	0.00050754	0.00083230	-0.00000113	0.00000000
		-6.27217984	-0.00082782	0.00050741	0.00083208	-0.00000113	0.00000000
		-6.27218040	-0.00082760	0.00050727	0.00083186	-0.00000113	0.00000000
		-6.27218097	-0.00082738	0.00050714	0.00083164	-0.00000113	0.00000000
		-6.27218153	-0.00082716	0.00050700	0.00083141	-0.00000113	0.00000000
		-6.27218209	-0.00082693	0.00050686	0.00083119	-0.00000113	0.00000000
		-6.27218265	-0.00082671	0.00050673	0.00083097	-0.00000113	0.00000000
		-6.27218321	-0.00082649	0.00050659	0.00083075	-0.00000113	0.00000000
		-6.27218377	-0.00082627	0.00050646	0.00083053	-0.00000113	0.00000000
		-6.27218433	-0.00082605	0.00050632	0.00083031	-0.00000113	0.00000000
		-6.27218488	-0.00082583	0.00050619	0.00083008	-0.00000113	0.00000000
		-6.27218544	-0.00082561	0.00050605	0.00082986	-0.00000113	0.00000000
		-6.27218600	-0.00082539	0.00050592	0.00082964	-0.00000113	0.00000000
		-6.27218656	-0.00082517	0.00050578	0.00082942	-0.00000113	0.00000000
		-6.27218711	-0.00082495	0.00050565	0.00082920	-0.00000113	0.00000000
		-6.27218767	-0.00082473	0.00050551	0.00082898	-0.00000113	0.00000000
		-6.27218823	-0.00082451	0.00050538	0.00082876	-0.00000113	0.00000000
		-6.27218878	-0.00082429	0.00050524	0.00082854	-0.00000113	0.00000000
		-6.27218934	-0.00082407	0.00050511	0.00082832	-0.00000113	0.00000000
		-6.27218989	-0.00082385	0.00050497	0.00082810	-0.00000113	0.00000000
		-6.27219045	-0.00082363	0.00050484	0.00082787	-0.00000113	0.00000000
		-6.27219100	-0.00082341	0.00050471	0.00082765	-0.00000113	0.00000000
		-6.27219155	-0.00082320	0.00050457	0.00082743	-0.00000113	0.00000000
		-6.27219211	-0.00082298	0.00050444	0.00082721	-0.00000113	0.00000000
		-6.27219266	-0.00082276	0.00050430	0.00082699	-0.00000113	0.00000000
		-6.27219321	-0.00082254	0.00050417	0.00082677	-0.00000113	0.00000000
		-6.27219376	-0.00082232	0.00050403	0.00082656	-0.00000113	0.00000000
		-6.27219432	-0.00082210	0.00050390	0.00082634	-0.00000113	0.00000000
		-6.27219487	-0.00082188	0.00050377	0.00082612	-0.00000113	0.00000000
		-6.27219542	-0.00082167	0.00050363	0.00082590	-0.00000113	0.00000000
		-6.27219597	-0.00082145	0.00050350	0.00082568	-0.00000113	0.00000000
		-6.27219652	-0.00082123	0.00050337	0.00082546	-0.00000113	0.00000000
		-6.27219707	-0.00082101	0.00050323	0.00082524	-0.00000113	0.00000000
		-6.27219762	-0.00082080	0.00050310	0.00082502	-0.00000113	0.00000000
		-6.27219817	-0.00082058	0.00050296	0.00082480	-0.00000113	0.00000000
		-6.27219872	-0.00082036	0.00050283	0.00082458	-0.00000113	0.00000000
		-6.27219926	-0.00082014	0.00050270	0.00082436	-0.00000113	0.00000000
		-6.27219981	-0.00081993	0.00050256	0.00082415	-0.00000113	0.00000000
		-6.27220036	-0.00081971	0.00050243	0.00082393	-0.00000113	0.00000000
		-6.27220091	-0.00081949	0.00050230	0.00082371	-0.00000113	0.00000000
		-6.27220145	-0.00081927	0.00050216	0.00082349	-0.00000113	0.00000000
		-6.27220200	-0.00081906	0.00050203	0.00082327	-0.00000113	0.00000000
		-6.27220254	-0.00081884	0.00050190	0.00082306	-0.00000113	0.00000000
		-6.27220309	-0.00081862	0.00050177	0.00082284	-0.00000113	0.00000000
		-6.27220363	-0.00081841	0.00050163	0.00082262	-0.00000113	0.00000000
		-6.27220418	-0.00081819	0.00050150	0.00082240	-0.00000113	0.00000000
		-6.27220472	-0.00081798	0.00050137	0.00082219	-0.00000113	0.00000000
		-6.27220527	-0.00081776	0.00050124	0.00082197	-0.00000113	0.00000000
		-6.27220581	-0.00081754	0.00050110	0.00082175	-0.00000113	0.00000000
		-6.27220635	-0.00081733	0.00050097	0.00082153	-0.00000113	0.00000000
		-6.27220690	-0.00081711	0.00050084	0.00082132	-0.00000113	0.00000000
		-6.27220744	-0.00081690	0.00050071	0.00082110	-0.00000113	0.00000000
		-6.27220798	-0.00081668	0.00050057	0.00082088	-0.00000113	0.00000000
		-6.27220852	-0.00081647	0.00050044	0.00082067	-0.00000113	0.00000000
		-6.27220906	-0.00081625	0.00050031	0.00082045	-0.00000113	0.00000000
		-6.27220960	-0.00081604	0.00050018	0.00082024	-0.00000113	0.00000000
		-6.27221014	-0.00081582	0.00050005	0.00082002	-0.00000113	0.00000000
		-6.27221068	-0.00081561	0.00049991	0.00081980	-0.00000113	0.00000000
		-6.27221122	-0.00081539	0.00049978	0.00081959	-0.00000113	0.00000000
		-6.27221176	-0.00081518	0.00049965	0.00081937	-0.00000113	0.00000000
		-6.27221230	-0.00081496	0.00049952	0.00081916	-0.00000113	0.00000000
		-6.27221284	-0.00081475	0.00049939	0.00081894	-0.00000113	0.00000000
		-6.27221338	-0.00081453	0.00049926	0.00081872	-0.00000113	0.00000000
		-6.27221391	-0.00081432	0.00049912	0.00081851	-0.00000113	0.00000000
		-6.27221445	-0.00081410	0.00049899	0.00081829	-0.00000113	0.00000000
		-6.27221499	-0.00081389	0.00049886	0.00081808	-0.00000113	0.00000000
		-6.27221552	-0.00081368	0.00049873	0.00081786	-0.00000113	0.00000000
		-6.27221606	-0.00081346	0.00049860	0.00081765	-0.00000113	0.00000000
		-6.27221660	-0.00081325	0.00049847	0.00081743	-0.00000113	0.00000000
		-6.27221713	-0.00081303	0.00049834	0.00081722	-0.00000113	0.00000000
		-6.27221767	-0.00081282	0.00049821	0.00081700	-0.00000113	0.00000000
		-6.27221820	-0.00081261	0.00049807	0.00081679	-0.00000113	0.00000000
		-6.27221874	-0.00081239	0.00049794	0.00081657	-0.00000113	0.00000000
		-6.27221927	-0.00081218	0.00049781	0.00081636	-0.00000113	0.00000000
		-6.27221980	-0.00081197	0.00049768	0.00081615	-0.00000113	0.00000000
		-6.27222034	-0.00081176	0.00049755	0.00081593	-0.00000113	0.00000000
		-6.27222087	-0.00081154	0.00049742	0.00081572	-0.00000113	0.00000000
		-6.27222140	-0.00081133	0.00049729	0.00081550	-0.00000113	0.00000000
		-6.27222193	-0.00081112	0.00049716	0.00081529	-0.00000113	0.00000000
		-6.27222247	-0.00081090	0.00049703	0.00081508	-0.00000113	0.00000000
		-6.27222300	-0.00081069	0.00049690	0.00081486	-0.00000113	0.00000000
		-6.27222353	-0.00081048	0.00049677	0.00081465	-0.00000113	0.00000000
		-6.27222406	-0.00081027	0.00049664	0.00081444	-0.00000113	0.00000000
		-6.27222459	-0.00081006	0.00049651	0.00081422	-0.00000113	0.00000000
		-6.27222512	-0.00080984	0.00049638	0.00081401	-0.00000113	0.00000000
		-6.27222565	-0.00080963	0.00049625	0.00081380	-0.00000113	0.00000000
		-6.27222618	-0.00080942	0.00049612	0.00081359	-0.00000113	0.00000000
		-6.27222671	-0.00080921	0.00049599	0.00081337	-0.00000113	0.00000000
		-6.27222723	-0.00080900	0.00049586	0.00081316	-0.00000113	0.00000000
		-6.27222776	-0.00080879	0.00049573	0.00081295	-0.00000113	0.00000000
		-6.27222829	-0.00080857	0.00049560	0.00081273	-0.00000113	0.00000000
		-6.27222882	-0.00080836	0.00049547	0.00081252	-0.00000113	0.00000000
		-6.27222934	-0.00080815	0.00049534	0.00081231	-0.00000113	0.00000000
		-6.27222987	-0.00080794	0.00049521	0.00081210	-0.00000113	0.00000000
		-6.27223040	-0.00080773	0.00049508	0.00081189	-0.00000113	0.00000000
		-6.27223092	-0.00080752	0.00049495	0.00081167	-0.00000113	0.00000000
		-6.27223145	-0.00080731	0.00049482	0.00081146	-0.00000113	0.00000000
		-6.27223197	-0.00080710	0.00049469	0.00081125	-0.00000113	0.00000000
		-6.27223250	-0.00080689	0.00049457	0.00081104	-0.00000113	0.00000000
		-6.27223302	-0.00080668	0.00049444	0.00081083	-0.00000113	0.00000000
		-6.27223355	-0.00080647	0.00049431	0.00081062	-0.00000113	0.00000000
		-6.27223407	-0.00080626	0.00049418	0.00081041	-0.00000113	0.00000000
		-6.27223460	-0.00080605	0.00049405	0.00081019	-0.00000113	0.00000000
		-6.27223512	-0.00080584	0.00049392	0.00080998	-0.00000113	0.00000000
		-6.27223564	-0.00080563	0.00049379	0.00080977	-0.00000113	0.00000000
		-6.27223616	-0.00080542	0.00049366	0.00080956	-0.00000113	0.00000000
		-6.27223669	-0.00080521	0.00049354	0.00080935	-0.00000113	0.00000000
		-6.27223721	-0.00080500	0.00049341	0.00080914	-0.00000113	0.00000000
		-6.27223773	-0.00080479	0.00049328	0.00080893	-0.00000113	0.00000000
		-6.27223825	-0.00080458	0.00049315	0.00080872	-0.00000113	0.00000000
		-6.27223877	-0.00080437	0.00049302	0.00080851	-0.00000113	0.00000000
		-6.27223929	-0.00080416	0.00049289	0.00080830	-0.00000113	0.00000000
		-6.27223981	-0.00080395	0.00049277	0.00080809	-0.00000113	0.00000000
		-6.27224033	-0.00080375	0.00049264	0.00080788	-0.00000113	0.00000000
		-6.27224085	-0.00080354	0.00049251	0.00080767	-0.00000113	0.00000000
		-6.27224137	-0.00080333	0.00049238	0.00080746	-0.00000113	0.00000000
		-6.27224189	-0.00080312	0.00049225	0.00080725	-0.00000113	0.00000000
		-6.27224241	-0.00080291	0.00049213	0.00080704	-0.00000113	0.00000000
		-6.27224292	-0.00080270	0.00049200	0.00080683	-0.00000113	0.00000000
		-6.27224344	-0.00080250	0.00049187	0.00080662	-0.00000113	0.00000000
		-6.27224396	-0.00080229	0.00049174	0.00080641	-0.00000113	0.00000000
		-6.27224447	-0.00080208	0.00049162	0.00080621	-0.00000113	0.00000000
		-6.27224499	-0.00080187	0.00049149	0.00080600	-0.00000113	0.00000000
		-6.27224551	-0.00080166	0.00049136	0.00080579	-0.00000113	0.00000000
		-6.27224602	-0.00080146	0.00049123	0.00080558	-0.00000113	0.00000000
		-6.27224654	-0.00080125	0.00049111	0.00080537	-0.00000113	0.00000000
		-6.27224705	-0.00080104	0.00049098	0.00080516	-0.00000113	0.00000000
		-6.27224757	-0.00080084	0.00049085	0.00080495	-0.00000113	0.00000000
		-6.27224808	-0.00080063	0.00049072	0.00080475	-0.00000113	0.00000000
		-6.27224860	-0.00080042	0.00049060	0.00080454	-0.00000113	0.00000000
		-6.27224911	-0.00080021	0.00049047	0.00080433	-0.00000113	0.00000000
		-6.27224962	-0.00080001	0.00049034	0.00080412	-0.00000113	0.00000000
		-6.27225014	-0.00079980	0.00049022	0.00080391	-0.00000113	0.00000000
		-6.27225065	-0.00079959	0.00049009	0.00080371	-0.00000113	0.00000000
		-6.27225116	-0.00079939	0.00048996	0.00080350	-0.00000113	0.00000000
		-6.27225168	-0.00079918	0.00048984	0.00080329	-0.00000113	0.00000000
		-6.27225219	-0.00079897	0.00048971	0.00080308	-0.00000113	0.00000000
		-6.27225270	-0.00079877	0.00048958	0.00080288	-0.00000113	0.00000000
		-6.27225321	-0.00079856	0.00048946	0.00080267	-0.00000113	0.00000000
		-6.27225372	-0.00079836	0.00048933	0.00080246	-0.00000113	0.00000000
		-6.27225423	-0.00079815	0.00048921	0.00080226	-0.00000113	0.00000000
		-6.27225474	-0.00079795	0.00048908	0.00080205	-0.00000113	0.00000000
		-6.27225525	-0.00079774	0.00048895	0.00080184	-0.00000113	0.00000000
		-6.27225576	-0.00079753	0.00048883	0.00080164	-0.00000113	0.00000000
		-6.27225627	-0.00079733	0.00048870	0.00080143	-0.00000113	0.00000000
		-6.27225678	-0.00079712	0.00048857	0.00080122	-0.00000113	0.00000000
		-6.27225729	-0.00079692	0.00048845	0.00080102	-0.00000113	0.00000000
		-6.27225780	-0.00079671	0.00048832	0.00080081	-0.00000113	0.00000000
		-6.27225830	-0.00079651	0.00048820	0.00080060	-0.00000113	0.00000000
		-6.27225881	-0.00079630	0.00048807	0.00080040	-0.00000113	0.00000000
		-6.27225932	-0.00079610	0.00048795	0.00080019	-0.00000113	0.00000000
		-6.27225983	-0.00079589	0.00048782	0.00079999	-0.00000113	0.00000000
		-6.27226033	-0.00079569	0.00048769	0.00079978	-0.00000113	0.00000000
		-6.27226084	-0.00079548	0.00048757	0.00079957	-0.00000113	0.00000000
		-6.27226134	-0.00079528	0.00048744	0.00079937	-0.00000113	0.00000000
		-6.27226185	-0.00079508	0.00048732	0.00079916	-0.00000113	0.00000000
		-6.27226235	-0.00079487	0.00048719	0.00079896	-0.00000113	0.00000000
		-6.27226286	-0.00079467	0.00048707	0.00079875	-0.00000113	0.00000000
		-6.27226336	-0.00079446	0.00048694	0.00079855	-0.00000113	0.00000000
		-6.27226387	-0.00079426	0.00048682	0.00079834	-0.00000113	0.00000000
		-6.27226437	-0.00079406	0.00048669	0.00079814	-0.00000113	0.00000000
		-6.27226488	-0.00079385	0.00048657	0.00079793	-0.00000113	0.00000000
		-6.27226538	-0.00079365	0.00048644	0.00079773	-0.00000113	0.00000000
		-6.27226588	-0.00079345	0.00048632	0.00079752	-0.00000113	0.00000000
		-6.27226639	-0.00079324	0.00048619	0.00079732	-0.00000113	0.00000000
		-6.27226689	-0.00079304	0.00048607	0.00079712	-0.00000113	0.00000000
		-6.27226739	-0.00079284	0.00048594	0.00079691	-0.00000113	0.00000000
		-6.27226789	-0.00079263	0.00048582	0.00079671	-0.00000113	0.00000000
		-6.27226839	-0.00079243	0.00048570	0.00079650	-0.00000113	0.00000000
		-6.27226889	-0.00079223	0.00048557	0.00079630	-0.00000113	0.00000000
		-6.27226939	-0.00079202	0.00048545	0.00079610	-0.00000113	0.00000000
		-6.27226990	-0.00079182	0.00048532	0.00079589	-0.00000113	0.00000000
		-6.27227040	-0.00079162	0.00048520	0.00079569	-0.00000113	0.00000000
		-6.27227090	-0.00079142	0.00048507	0.00079549	-0.00000113	0.00000000
		-6.27227139	-0.00079121	0.00048495	0.00079528	-0.00000113	0.00000000
		-6.27227189	-0.00079101	0.00048483	0.00079508	-0.00000113	0.00000000
		-6.27227239	-0.00079081	0.00048470	0.00079488	-0.00000113	0.00000000
		-6.27227289	-0.00079061	0.00048458	0.00079467	-0.00000113	0.00000000
		-6.27227339	-0.00079041	0.00048445	0.00079447	-0.00000113	0.00000000
		-6.27227389	-0.00079020	0.00048433	0.00079427	-0.00000113	0.00000000
		-6.27227439	-0.00079000	0.00048421	0.00079406	-0.00000113	0.00000000
		-6.27227488	-0.00078980	0.00048408	0.00079386	-0.00000113	0.00000000
		-6.27227538	-0.00078960	0.00048396	0.00079366	-0.00000113	0.00000000
		-6.27227588	-0.00078940	0.00048384	0.00079346	-0.00000113	0.00000000
		-6.27227637	-0.00078920	0.00048371	0.00079325	-0.00000113	0.00000000
		-6.27227687	-0.00078900	0.00048359	0.00079305	-0.00000113	0.00000000
		-6.27227736	-0.00078880	0.00048347	0.00079285	-0.00000113	0.00000000
		-6.27227786	-0.00078859	0.00048334	0.00079265	-0.00000113	0.00000000
		-6.27227836	-0.00078839	0.00048322	0.00079245	-0.00000113	0.00000000
		-6.27227885	-0.00078819	0.00048310	0.00079224	-0.00000113	0.00000000
		-6.27227934	-0.00078799	0.00048297	0.00079204	-0.00000113	0.00000000
		-6.27227984	-0.00078779	0.00048285	0.00079184	-0.00000113	0.00000000
		-6.27228033	-0.00078759	0.00048273	0.00079164	-0.00000113	0.00000000
		-6.27228083	-0.00078739	0.00048260	0.00079144	-0.00000113	0.00000000
		-6.27228132	-0.00078719	0.00048248	0.00079124	-0.00000113	0.00000000
		-6.27228181	-0.00078699	0.00048236	0.00079104	-0.00000113	0.00000000
		-6.27228231	-0.00078679	0.00048224	0.00079083	-0.00000113	0.00000000
		-6.27228280	-0.00078659	0.00048211	0.00079063	-0.00000113	0.00000000
		-6.27228329	-0.00078639	0.00048199	0.00079043	-0.00000113	0.00000000
		-6.27228378	-0.00078619	0.00048187	0.00079023	-0.00000113	0.00000000
		-6.27228427	-0.00078599	0.00048175	0.00079003	-0.00000113	0.00000000
		-6.27228476	-0.00078579	0.00048162	0.00078983	-0.00000113	0.00000000
		-6.27228526	-0.00078559	0.00048150	0.00078963	-0.00000113	0.00000000
		-6.27228575	-0.00078539	0.00048138	0.00078943	-0.00000113	0.00000000
		-6.27228624	-0.00078519	0.00048126	0.00078923	-0.00000113	0.00000000
		-6.27228673	-0.00078500	0.00048113	0.00078903	-0.00000113	0.00000000
		-6.27228722	-0.00078480	0.00048101	0.00078883	-0.00000113	0.00000000
		-6.27228771	-0.00078460	0.00048089	0.00078863	-0.00000113	0.00000000
		-6.27228819	-0.00078440	0.00048077	0.00078843	-0.00000113	0.00000000
		-6.27228868	-0.00078420	0.00048065	0.00078823	-0.00000113	0.00000000
		-6.27228917	-0.00078400	0.00048052	0.00078803	-0.00000113	0.00000000
		-6.27228966	-0.00078380	0.00048040	0.00078783	-0.00000113	0.00000000
		-6.27229015	-0.00078360	0.00048028	0.00078763	-0.00000113	0.00000000
		-6.27229064	-0.00078341	0.00048016	0.00078743	-0.00000113	0.00000000
		-6.27229112	-0.00078321	0.00048004	0.00078723	-0.00000113	0.00000000
		-6.27229161	-0.00078301	0.00047992	0.00078703	-0.00000113	0.00000000
		-6.27229210	-0.00078281	0.00047979	0.00078683	-0.00000113	0.00000000
		-6.27229258	-0.00078261	0.00047967	0.00078663	-0.00000113	0.00000000
		-6.27229307	-0.00078242	0.00047955	0.00078644	-0.00000113	0.00000000
		-6.27229356	-0.00078222	0.00047943	0.00078624	-0.00000113	0.00000000
		-6.27229404	-0.00078202	0.00047931	0.00078604	-0.00000113	0.00000000
		-6.27229453	-0.00078182	0.00047919	0.00078584	-0.00000113	0.00000000
		-6.27229501	-0.00078163	0.00047907	0.00078564	-0.00000113	0.00000000
		-6.27229550	-0.00078143	0.00047895	0.00078544	-0.00000113	0.00000000
		-6.27229598	-0.00078123	0.00047882	0.00078524	-0.00000113	0.00000000
		-6.27229646	-0.00078103	0.00047870	0.00078505	-0.00000113	0.00000000
		-6.27229695	-0.00078084	0.00047858	0.00078485	-0.00000113	0.00000000
		-6.27229743	-0.00078064	0.00047846	0.00078465	-0.00000113	0.00000000
		-6.27229791	-0.00078044	0.00047834	0.00078445	-0.00000113	0.00000000
		-6.27229840	-0.00078025	0.00047822	0.00078426	-0.00000113	0.00000000
		-6.27229888	-0.00078005	0.00047810	0.00078406	-0.00000113	0.00000000
		-6.27229936	-0.00077985	0.00047798	0.00078386	-0.00000113	0.00000000
		-6.27229984	-0.00077966	0.00047786	0.00078366	-0.00000113	0.00000000
		-6.27230033	-0.00077946	0.00047774	0.00078346	-0.00000113	0.00000000
		-6.27230081	-0.00077926	0.00047762	0.00078327	-0.00000113	0.00000000
		-6.27230129	-0.00077907	0.00047750	0.00078307	-0.00000113	0.00000000
		-6.27230177	-0.00077887	0.00047738	0.00078287	-0.00000113	0.00000000
		-6.27230225	-0.00077868	0.00047726	0.00078268	-0.00000113	0.00000000
		-6.27230273	-0.00077848	0.00047714	0.00078248	-0.00000113	0.00000000
		-6.27230321	-0.00077828	0.00047702	0.00078228	-0.00000113	0.00000000
		-6.27230369	-0.00077809	0.00047690	0.00078209	-0.00000113	0.00000000
		-6.27230417	-0.00077789	0.00047678	0.00078189	-0.00000113	0.00000000
		-6.27230465	-0.00077770	0.00047666	0.00078169	-0.00000113	0.00000000
		-6.27230513	-0.00077750	0.00047654	0.00078150	-0.00000113	0.00000000
		-6.27230561	-0.00077731	0.00047642	0.00078130	-0.00000113	0.00000000
		-6.27230609	-0.00077711	0.00047630	0.00078110	-0.00000113	0.00000000
		-6.27230656	-0.00077692	0.00047618	0.00078091	-0.00000113	0.00000000
		-6.27230704	-0.00077672	0.00047606	0.00078071	-0.00000113	0.00000000
		-6.27230752	-0.00077653	0.00047594	0.00078052	-0.00000113	0.00000000
		-6.27230800	-0.00077633	0.00047582	0.00078032	-0.00000113	0.00000000
		-6.27230847	-0.00077614	0.00047570	0.00078012	-0.00000113	0.00000000
		-6.27230895	-0.00077594	0.00047558	0.00077993	-0.00000113	0.00000000
		-6.27230943	-0.00077575	0.00047546	0.00077973	-0.00000113	0.00000000
		-6.27230990	-0.00077556	0.00047534	0.00077954	-0.00000113	0.00000000
		-6.27231038	-0.00077536	0.00047522	0.00077934	-0.00000113	0.00000000
		-6.27231085	-0.00077517	0.00047510	0.00077915	-0.00000113	0.00000000
		-6.27231133	-0.00077497	0.00047498	0.00077895	-0.00000113	0.00000000
		-6.27231180	-0.00077478	0.00047487	0.00077876	-0.00000113	0.00000000
		-6.27231228	-0.00077458	0.00047475	0.00077856	-0.00000113	0.00000000
		-6.27231275	-0.00077439	0.00047463	0.00077837	-0.00000113	0.00000000
		-6.27231323	-0.00077420	0.00047451	0.00077817	-0.00000113	0.00000000
		-6.27231370	-0.00077400	0.00047439	0.00077798	-0.00000113	0.00000000
		-6.27231417	-0.00077381	0.00047427	0.00077778	-0.00000113	0.00000000
		-6.27231465	-0.00077362	0.00047415	0.00077759	-0.00000113	0.00000000
		-6.27231512	-0.00077342	0.00047403	0.00077740	-0.00000113	0.00000000
		-6.27231559	-0.00077323	0.00047392	0.00077720	-0.00000113	0.00000000
		-6.27231607	-0.00077304	0.00047380	0.00077701	-0.00000113	0.00000000
		-6.27231654	-0.00077284	0.00047368	0.00077681	-0.00000113	0.00000000
		-6.27231701	-0.00077265	0.00047356	0.00077662	-0.00000113	0.00000000
		-6.27231748	-0.00077246	0.00047344	0.00077643	-0.00000113	0.00000000
		-6.27231795	-0.00077227	0.00047332	0.00077623	-0.00000113	0.00000000
		-6.27231842	-0.00077207	0.00047321	0.00077604	-0.00000113	0.00000000
		-6.27231889	-0.00077188	0.00047309	0.00077584	-0.00000113	0.00000000
		-6.27231936	-0.00077169	0.00047297	0.00077565	-0.00000113	0.00000000
		-6.27231984	-0.00077150	0.00047285	0.00077546	-0.00000113	0.00000000
		-6.27232031	-0.00077130	0.00047273	0.00077526	-0.00000113	0.00000000
		-6.27232077	-0.00077111	0.00047262	0.00077507	-0.00000113	0.00000000
		-6.27232124	-0.00077092	0.00047250	0.00077488	-0.00000113	0.00000000
		-6.27232171	-0.00077073	0.00047238	0.00077469	-0.00000113	0.00000000
		-6.27232218	-0.00077054	0.00047226	0.00077449	-0.00000113	0.00000000
		-6.27232265	-0.00077034	0.00047215	0.00077430	-0.00000113	0.00000000
		-6.27232312	-0.00077015	0.00047203	0.00077411	-0.00000113	0.00000000
		-6.27232359	-0.00076996	0.00047191	0.00077391	-0.00000113	0.00000000
		-6.27232406	-0.00076977	0.00047179	0.00077372	-0.00000113	0.00000000
		-6.27232452	-0.00076958	0.00047168	0.00077353	-0.00000113	0.00000000
		-6.27232499	-0.00076939	0.00047156	0.00077334	-0.00000113	0.00000000
		-6.27232546	-0.00076920	0.00047144	0.00077315	-0.00000113	0.00000000
		-6.27232592	-0.00076900	0.00047132	0.00077295	-0.00000113	0.00000000
		-6.27232639	-0.00076881	0.00047121	0.00077276	-0.00000113	0.00000000
		-6.27232686	-0.00076862	0.00047109	0.00077257	-0.00000113	0.00000000
		-6.27232732	-0.00076843	0.00047097	0.00077238	-0.00000113	0.00000000
		-6.27232779	-0.00076824	0.00047086	0.00077219	-0.00000113	0.00000000
		-6.27232825	-0.00076805	0.00047074	0.00077199	-0.00000113	0.00000000
		-6.27232872	-0.00076786	0.00047062	0.00077180	-0.00000113	0.00000000
		-6.27232918	-0.00076767	0.00047050	0.00077161	-0.00000113	0.00000000
		-6.27232965	-0.00076748	0.00047039	0.00077142	-0.00000113	0.00000000
		-6.27233011	-0.00076729	0.00047027	0.00077123	-0.00000113	0.00000000
		-6.27233058	-0.00076710	0.00047015	0.00077104	-0.00000113	0.00000000
		-6.27233104	-0.00076691	0.00047004	0.00077085	-0.00000113	0.00000000
		-6.27233150	-0.00076672	0.00046992	0.00077066	-0.00000113	0.00000000
		-6.27233197	-0.00076653	0.00046981	0.00077046	-0.00000113	0.00000000
		-6.27233243	-0.00076634	0.00046969	0.00077027	-0.00000113	0.00000000
		-6.27233289	-0.00076615	0.00046957	0.00077008	-0.00000113	0.00000000
		-6.27233336	-0.00076596	0.00046946	0.00076989	-0.00000113	0.00000000
		-6.27233382	-0.00076577	0.00046934	0.00076970	-0.00000113	0.00000000
		-6.27233428	-0.00076558	0.00046922	0.00076951	-0.00000113	0.00000000
		-6.27233474	-0.00076539	0.00046911	0.00076932	-0.00000113	0.00000000
		-6.27233520	-0.00076520	0.00046899	0.00076913	-0.00000113	0.00000000
		-6.27233566	-0.00076501	0.00046888	0.00076894	-0.00000113	0.00000000
		-6.27233612	-0.00076482	0.00046876	0.00076875	-0.00000113	0.00000000
		-6.27233658	-0.00076464	0.00046864	0.00076856	-0.00000113	0.00000000
		-6.27233705	-0.00076445	0.00046853	0.00076837	-0.00000113	0.00000000
		-6.27233751	-0.00076426	0.00046841	0.00076818	-0.00000113	0.00000000
		-6.27233797	-0.00076407	0.00046830	0.00076799	-0.00000113	0.00000000
		-6.27233843	-0.00076388	0.00046818	0.00076780	-0.00000113	0.00000000
		-6.27233888	-0.00076369	0.00046806	0.00076761	-0.00000113	0.00000000
		-6.27233934	-0.00076350	0.00046795	0.00076742	-0.00000113	0.00000000
		-6.27233980	-0.00076332	0.00046783	0.00076723	-0.00000113	0.00000000
		-6.27234026	-0.00076313	0.00046772	0.00076704	-0.00000113	0.00000000
		-6.27234072	-0.00076294	0.00046760	0.00076686	-0.00000113	0.00000000
		-6.27234118	-0.00076275	0.00046749	0.00076667	-0.00000113	0.00000000
		-6.27234163	-0.00076256	0.00046737	0.00076648	-0.00000113	0.00000000
		-6.27234209	-0.00076238	0.00046726	0.00076629	-0.00000113	0.00000000
		-6.27234255	-0.00076219	0.00046714	0.00076610	-0.00000113	0.00000000
		-6.27234301	-0.00076200	0.00046703	0.00076591	-0.00000113	0.00000000
		-6.27234346	-0.00076181	0.00046691	0.00076572	-0.00000113	0.00000000
		-6.27234392	-0.00076163	0.00046680	0.00076553	-0.00000113	0.00000000
		-6.27234438	-0.00076144	0.00046668	0.00076535	-0.00000113	0.00000000
		-6.27234483	-0.00076125	0.00046657	0.00076516	-0.00000113	0.00000000
		-6.27234529	-0.00076106	0.00046645	0.00076497	-0.00000113	0.00000000
		-6.27234574	-0.00076088	0.00046634	0.00076478	-0.00000113	0.00000000
		-6.27234620	-0.00076069	0.00046622	0.00076459	-0.00000113	0.00000000
		-6.27234665	-0.00076050	0.00046611	0.00076441	-0.00000113	0.00000000
		-6.27234711	-0.00076032	0.00046599	0.00076422	-0.00000113	0.00000000
		-6.27234756	-0.00076013	0.00046588	0.00076403	-0.00000113	0.00000000
		-6.27234802	-0.00075994	0.00046576	0.00076384	-0.00000113	0.00000000
		-6.27234847	-0.00075976	0.00046565	0.00076366	-0.00000113	0.00000000
		-6.27234892	-0.00075957	0.00046554	0.00076347	-0.00000113	0.00000000
		-6.27234938	-0.00075938	0.00046542	0.00076328	-0.00000113	0.00000000
		-6.27234983	-0.00075920	0.00046531	0.00076309	-0.00000113	0.00000000
		-6.27235028	-0.00075901	0.00046519	0.00076291	-0.00000113	0.00000000
		-6.27235074	-0.00075882	0.00046508	0.00076272	-0.00000113	0.00000000
		-6.27235119	-0.00075864	0.00046496	0.00076253	-0.00000113	0.00000000
		-6.27235164	-0.00075845	0.00046485	0.00076235	-0.00000113	0.00000000
		-6.27235209	-0.00075827	0.00046474	0.00076216	-0.00000113	0.00000000
		-6.27235255	-0.00075808	0.00046462	0.00076197	-0.00000113	0.00000000
		-6.27235300	-0.00075790	0.00046451	0.00076179	-0.00000113	0.00000000
		-6.27235345	-0.00075771	0.00046440	0.00076160	-0.00000113	0.00000000
		-6.27235390	-0.00075753	0.00046428	0.00076141	-0.00000113	0.00000000
		-6.27235435	-0.00075734	0.00046417	0.00076123	-0.00000113	0.00000000
		-6.27235480	-0.00075715	0.00046405	0.00076104	-0.00000113	0.00000000
		-6.27235525	-0.00075697	0.00046394	0.00076085	-0.00000113	0.00000000
		-6.27235570	-0.00075678	0.00046383	0.00076067	-0.00000113	0.00000000
		-6.27235615	-0.00075660	0.00046371	0.00076048	-0.00000113	0.00000000
		-6.27235660	-0.00075641	0.00046360	0.00076030	-0.00000113	0.00000000
		-6.27235705	-0.00075623	0.00046349	0.00076011	-0.00000113	0.00000000
		-6.27235750	-0.00075605	0.00046337	0.00075992	-0.00000113	0.00000000
		-6.27235795	-0.00075586	0.00046326	0.00075974	-0.00000113	0.00000000
		-6.27235840	-0.00075568	0.00046315	0.00075955	-0.00000113	0.00000000
		-6.27235884	-0.00075549	0.00046303	0.00075937	-0.00000113	0.00000000
		-6.27235929	-0.00075531	0.00046292	0.00075918	-0.00000113	0.00000000
		-6.27235974	-0.00075512	0.00046281	0.00075900	-0.00000113	0.00000000
		-6.27236019	-0.00075494	0.00046269	0.00075881	-0.00000113	0.00000000
		-6.27236063	-0.00075475	0.00046258	0.00075863	-0.00000113	0.00000000
		-6.27236108	-0.00075457	0.00046247	0.00075844	-0.00000113	0.00000000
		-6.27236153	-0.00075439	0.00046236	0.00075826	-0.00000113	0.00000000
		-6.27236197	-0.00075420	0.00046224	0.00075807	-0.00000113	0.00000000
		-6.27236242	-0.00075402	0.00046213	0.00075789	-0.00000113	0.00000000
		-6.27236287	-0.00075384	0.00046202	0.00075770	-0.00000113	0.00000000
		-6.27236331	-0.00075365	0.00046191	0.00075752	-0.00000113	0.00000000
		-6.27236376	-0.00075347	0.00046179	0.00075733	-0.00000113	0.00000000
		-6.27236420	-0.00075329	0.00046168	0.00075715	-0.00000113	0.00000000
		-6.27236465	-0.00075310	0.00046157	0.00075697	-0.00000113	0.00000000
		-6.27236509	-0.00075292	0.00046146	0.00075678	-0.00000113	0.00000000
		-6.27236554	-0.00075274	0.00046134	0.00075660	-0.00000113	0.00000000
		-6.27236598	-0.00075255	0.00046123	0.00075641	-0.00000113	0.00000000
		-6.27236643	-0.00075237	0.00046112	0.00075623	-0.00000113	0.00000000
		-6.27236687	-0.00075219	0.00046101	0.00075605	-0.00000113	0.00000000
		-6.27236731	-0.00075200	0.00046089	0.00075586	-0.00000113	0.00000000
		-6.27236776	-0.00075182	0.00046078	0.00075568	-0.00000113	0.00000000
		-6.27236820	-0.00075164	0.00046067	0.00075550	-0.00000113	0.00000000
		-6.27236864	-0.00075146	0.00046056	0.00075531	-0.00000113	0.00000000
		-6.27236909	-0.00075127	0.00046045	0.00075513	-0.00000113	0.00000000
		-6.27236953	-0.00075109	0.00046034	0.00075495	-0.00000113	0.00000000
		-6.27236997	-0.00075091	0.00046022	0.00075476	-0.00000113	0.00000000
		-6.27237041	-0.00075073	0.00046011	0.00075458	-0.00000113	0.00000000
		-6.27237085	-0.00075055	0.00046000	0.00075440	-0.00000113	0.00000000
		-6.27237130	-0.00075036	0.00045989	0.00075421	-0.00000113	0.00000000
		-6.27237174	-0.00075018	0.00045978	0.00075403	-0.00000113	0.00000000
		-6.27237218	-0.00075000	0.00045967	0.00075385	-0.00000113	0.00000000
		-6.27237262	-0.00074982	0.00045955	0.00075366	-0.00000113	0.00000000
		-6.27237306	-0.00074964	0.00045944	0.00075348	-0.00000113	0.00000000
		-6.27237350	-0.00074946	0.00045933	0.00075330	-0.00000113	0.00000000
		-6.27237394	-0.00074927	0.00045922	0.00075312	-0.00000113	0.00000000
		-6.27237438	-0.00074909	0.00045911	0.00075294	-0.00000113	0.00000000
		-6.27237482	-0.00074891	0.00045900	0.00075275	-0.00000113	0.00000000
		-6.27237526	-0.00074873	0.00045889	0.00075257	-0.00000113	0.00000000
		-6.27237570	-0.00074855	0.00045878	0.00075239	-0.00000113	0.00000000
		-6.27237614	-0.00074837	0.00045866	0.00075221	-0.00000113	0.00000000
		-6.27237657	-0.00074819	0.00045855	0.00075203	-0.00000113	0.00000000
		-6.27237701	-0.00074801	0.00045844	0.00075184	-0.00000113	0.00000000
		-6.27237745	-0.00074783	0.00045833	0.00075166	-0.00000113	0.00000000
		-6.27237789	-0.00074765	0.00045822	0.00075148	-0.00000113	0.00000000
		-6.27237833	-0.00074747	0.00045811	0.00075130	-0.00000113	0.00000000
		-6.27237876	-0.00074728	0.00045800	0.00075112	-0.00000113	0.00000000
		-6.27237920	-0.00074710	0.00045789	0.00075094	-0.00000113	0.00000000
		-6.27237964	-0.00074692	0.00045778	0.00075075	-0.00000113	0.00000000
		-6.27238007	-0.00074674	0.00045767	0.00075057	-0.00000113	0.00000000
		-6.27238051	-0.00074656	0.00045756	0.00075039	-0.00000113	0.00000000
		-6.27238095	-0.00074638	0.00045745	0.00075021	-0.00000113	0.00000000
		-6.27238138	-0.00074620	0.00045734	0.00075003	-0.00000113	0.00000000
		-6.27238182	-0.00074602	0.00045723	0.00074985	-0.00000113	0.00000000
		-6.27238225	-0.00074584	0.00045712	0.00074967	-0.00000113	0.00000000
		-6.27238269	-0.00074566	0.00045701	0.00074949	-0.00000113	0.00000000
		-6.27238312	-0.00074548	0.00045690	0.00074931	-0.00000113	0.00000000
		-6.27238356	-0.00074531	0.00045679	0.00074913	-0.00000113	0.00000000
		-6.27238399	-0.00074513	0.00045668	0.00074895	-0.00000113	0.00000000
		-6.27238443	-0.00074495	0.00045657	0.00074877	-0.00000113	0.00000000
		-6.27238486	-0.00074477	0.00045646	0.00074859	-0.00000113	0.00000000
		-6.27238530	-0.00074459	0.00045635	0.00074841	-0.00000113	0.00000000
		-6.27238573	-0.00074441	0.00045624	0.00074823	-0.00000113	0.00000000
		-6.27238616	-0.00074423	0.00045613	0.00074805	-0.00000113	0.00000000
		-6.27238660	-0.00074405	0.00045602	0.00074787	-0.00000113	0.00000000
		-6.27238703	-0.00074387	0.00045591	0.00074769	-0.00000113	0.00000000
		-6.27238746	-0.00074369	0.00045580	0.00074751	-0.00000113	0.00000000
		-6.27238790	-0.00074352	0.00045569	0.00074733	-0.00000113	0.00000000
		-6.27238833	-0.00074334	0.00045558	0.00074715	-0.00000113	0.00000000
		-6.27238876	-0.00074316	0.00045547	0.00074697	-0.00000113	0.00000000
		-6.27238919	-0.00074298	0.00045536	0.00074679	-0.00000113	0.00000000
		-6.27238962	-0.00074280	0.00045525	0.00074661	-0.00000113	0.00000000
		-6.27239006	-0.00074262	0.00045514	0.00074643	-0.00000113	0.00000000
		-6.27239049	-0.00074245	0.00045503	0.00074625	-0.00000113	0.00000000
		-6.27239092	-0.00074227	0.00045492	0.00074607	-0.00000113	0.00000000
		-6.27239135	-0.00074209	0.00045481	0.00074589	-0.00000113	0.00000000
		-6.27239178	-0.00074191	0.00045470	0.00074572	-0.00000113	0.00000000
		-6.27239221	-0.00074173	0.00045459	0.00074554	-0.00000113	0.00000000
		-6.27239264	-0.00074156	0.00045449	0.00074536	-0.00000113	0.00000000
		-6.27239307	-0.00074138	0.00045438	0.00074518	-0.00000113	0.00000000
		-6.27239350	-0.00074120	0.00045427	0.00074500	-0.00000113	0.00000000
		-6.27239393	-0.00074102	0.00045416	0.00074482	-0.00000113	0.00000000
		-6.27239436	-0.00074085	0.00045405	0.00074464	-0.00000113	0.00000000
		-6.27239479	-0.00074067	0.00045394	0.00074447	-0.00000113	0.00000000
		-6.27239522	-0.00074049	0.00045383	0.00074429	-0.00000113	0.00000000
		-6.27239564	-0.00074031	0.00045372	0.00074411	-0.00000113	0.00000000
		-6.27239607	-0.00074014	0.00045362	0.00074393	-0.00000113	0.00000000
		-6.27239650	-0.00073996	0.00045351	0.00074375	-0.00000113	0.00000000
		-6.27239693	-0.00073978	0.00045340	0.00074358	-0.00000112	0.00000000
		-6.27239736	-0.00073961	0.00045329	0.00074340	-0.00000112	0.00000000
		-6.27239778	-0.00073943	0.00045318	0.00074322	-0.00000112	0.00000000
		-6.27239821	-0.00073925	0.00045307	0.00074304	-0.00000112	0.00000000
		-6.27239864	-0.00073908	0.00045296	0.00074287	-0.00000112	0.00000000
		-6.27239906	-0.00073890	0.00045286	0.00074269	-0.00000112	0.00000000
		-6.27239949	-0.00073872	0.00045275	0.00074251	-0.00000112	0.00000000
		-6.27239992	-0.00073855	0.00045264	0.00074233	-0.00000112	0.00000000
		-6.27240034	-0.00073837	0.00045253	0.00074216	-0.00000112	0.00000000
		-6.27240077	-0.00073820	0.00045242	0.00074198	-0.00000112	0.00000000
		-6.27240119	-0.00073802	0.00045232	0.00074180	-0.00000112	0.00000000
		-6.27240162	-0.00073784	0.00045221	0.00074163	-0.00000112	0.00000000
		-6.27240205	-0.00073767	0.00045210	0.00074145	-0.00000112	0.00000000
		-6.27240247	-0.00073749	0.00045199	0.00074127	-0.00000112	0.00000000
		-6.27240289	-0.00073732	0.00045188	0.00074110	-0.00000112	0.00000000
		-6.27240332	-0.00073714	0.00045178	0.00074092	-0.00000112	0.00000000
		-6.27240374	-0.00073697	0.00045167	0.00074074	-0.00000112	0.00000000
		-6.27240417	-0.00073679	0.00045156	0.00074057	-0.00000112	0.00000000
		-6.27240459	-0.00073662	0.00045145	0.00074039	-0.00000112	0.00000000
		-6.27240502	-0.00073644	0.00045135	0.00074022	-0.00000112	0.00000000
		-6.27240544	-0.00073626	0.00045124	0.00074004	-0.00000112	0.00000000
		-6.27240586	-0.00073609	0.00045113	0.00073986	-0.00000112	0.00000000
		-6.27240628	-0.00073591	0.00045102	0.00073969	-0.00000112	0.00000000
		-6.27240671	-0.00073574	0.00045092	0.00073951	-0.00000112	0.00000000
		-6.27240713	-0.00073556	0.00045081	0.00073934	-0.00000112	0.00000000
		-6.27240755	-0.00073539	0.00045070	0.00073916	-0.00000112	0.00000000
		-6.27240797	-0.00073522	0.00045060	0.00073898	-0.00000112	0.00000000
		-6.27240840	-0.00073504	0.00045049	0.00073881	-0.00000112	0.00000000
		-6.27240882	-0.00073487	0.00045038	0.00073863	-0.00000112	0.00000000
		-6.27240924	-0.00073469	0.00045027	0.00073846	-0.00000112	0.00000000
		-6.27240966	-0.00073452	0.00045017	0.00073828	-0.00000112	0.00000000
		-6.27241008	-0.00073434	0.00045006	0.00073811	-0.00000112	0.00000000
		-6.27241050	-0.00073417	0.00044995	0.00073793	-0.00000112	0.00000000
		-6.27241092	-0.00073400	0.00044985	0.00073776	-0.00000112	0.00000000
		-6.27241134	-0.00073382	0.00044974	0.00073758	-0.00000112	0.00000000
		-6.27241176	-0.00073365	0.00044963	0.00073741	-0.00000112	0.00000000
		-6.27241218	-0.00073347	0.00044953	0.00073723	-0.00000112	0.00000000
		-6.27241260	-0.00073330	0.00044942	0.00073706	-0.00000112	0.00000000
		-6.27241302	-0.00073313	0.00044931	0.00073688	-0.00000112	0.00000000
		-6.27241344	-0.00073295	0.00044921	0.00073671	-0.00000112	0.00000000
		-6.27241386	-0.00073278	0.00044910	0.00073653	-0.00000112	0.00000000
		-6.27241428	-0.00073261	0.00044899	0.00073636	-0.00000112	0.00000000
		-6.27241470	-0.00073243	0.00044889	0.00073619	-0.00000112	0.00000000
		-6.27241512	-0.00073226	0.00044878	0.00073601	-0.00000112	0.00000000
		-6.27241554	-0.00073209	0.00044868	0.00073584	-0.00000112	0.00000000
		-6.27241595	-0.00073191	0.00044857	0.00073566	-0.00000112	0.00000000
		-6.27241637	-0.00073174	0.00044846	0.00073549	-0.00000112	0.00000000
		-6.27241679	-0.00073157	0.00044836	0.00073532	-0.00000112	0.00000000
		-6.27241721	-0.00073139	0.00044825	0.00073514	-0.00000112	0.00000000
		-6.27241762	-0.00073122	0.00044815	0.00073497	-0.00000112	0.00000000
		-6.27241804	-0.00073105	0.00044804	0.00073479	-0.00000112	0.00000000
		-6.27241846	-0.00073088	0.00044793	0.00073462	-0.00000112	0.00000000
		-6.27241887	-0.00073070	0.00044783	0.00073445	-0.00000112	0.00000000
		-6.27241929	-0.00073053	0.00044772	0.00073427	-0.00000112	0.00000000
		-6.27241971	-0.00073036	0.00044762	0.00073410	-0.00000112	0.00000000
		-6.27242012	-0.00073019	0.00044751	0.00073393	-0.00000112	0.00000000
		-6.27242054	-0.00073001	0.00044741	0.00073376	-0.00000112	0.00000000
		-6.27242095	-0.00072984	0.00044730	0.00073358	-0.00000112	0.00000000
		-6.27242137	-0.00072967	0.00044719	0.00073341	-0.00000112	0.00000000
		-6.27242178	-0.00072950	0.00044709	0.00073324	-0.00000112	0.00000000
		-6.27242220	-0.00072933	0.00044698	0.00073306	-0.00000112	0.00000000
		-6.27242261	-0.00072915	0.00044688	0.00073289	-0.00000112	0.00000000
		-6.27242303	-0.00072898	0.00044677	0.00073272	-0.00000112	0.00000000
		-6.27242344	-0.00072881	0.00044667	0.00073255	-0.00000112	0.00000000
		-6.27242386	-0.00072864	0.00044656	0.00073237	-0.00000112	0.00000000
		-6.27242427	-0.00072847	0.00044646	0.00073220	-0.00000112	0.00000000
		-6.27242468	-0.00072830	0.00044635	0.00073203	-0.00000112	0.00000000
		-6.27242510	-0.00072813	0.00044625	0.00073186	-0.00000112	0.00000000
		-6.27242551	-0.00072795	0.00044614	0.00073168	-0.00000112	0.00000000
		-6.27242592	-0.00072778	0.00044604	0.00073151	-0.00000112	0.00000000
		-6.27242634	-0.00072761	0.00044593	0.00073134	-0.00000112	0.00000000
		-6.27242675	-0.00072744	0.00044583	0.00073117	-0.00000112	0.00000000
		-6.27242716	-0.00072727	0.00044572	0.00073100	-0.00000112	0.00000000
		-6.27242757	-0.00072710	0.00044562	0.00073082	-0.00000112	0.00000000
		-6.27242798	-0.00072693	0.00044551	0.00073065	-0.00000112	0.00000000
		-6.27242840	-0.00072676	0.00044541	0.00073048	-0.00000112	0.00000000
		-6.27242881	-0.00072659	0.00044530	0.00073031	-0.00000112	0.00000000
		-6.27242922	-0.00072642	0.00044520	0.00073014	-0.00000112	0.00000000
		-6.27242963	-0.00072625	0.00044509	0.00072997	-0.00000112	0.00000000
		-6.27243004	-0.00072608	0.00044499	0.00072980	-0.00000112	0.00000000
		-6.27243045	-0.00072591	0.00044488	0.00072962	-0.00000112	0.00000000
		-6.27243086	-0.00072574	0.00044478	0.00072945	-0.00000112	0.00000000
		-6.27243127	-0.00072557	0.00044468	0.00072928	-0.00000112	0.00000000
		-6.27243168	-0.00072540	0.00044457	0.00072911	-0.00000112	0.00000000
		-6.27243209	-0.00072523	0.00044447	0.00072894	-0.00000112	0.00000000
		-6.27243250	-0.00072506	0.00044436	0.00072877	-0.00000112	0.00000000
		-6.27243291	-0.00072489	0.00044426	0.00072860	-0.00000112	0.00000000
		-6.27243332	-0.00072472	0.00044416	0.00072843	-0.00000112	0.00000000
		-6.27243373	-0.00072455	0.00044405	0.00072826	-0.00000112	0.00000000
		-6.27243414	-0.00072438	0.00044395	0.00072809	-0.00000112	0.00000000
		-6.27243455	-0.00072421	0.00044384	0.00072792	-0.00000112	0.00000000
		-6.27243496	-0.00072404	0.00044374	0.00072775	-0.00000112	0.00000000
		-6.27243536	-0.00072387	0.00044364	0.00072758	-0.00000112	0.00000000
		-6.27243577	-0.00072370	0.00044353	0.00072741	-0.00000112	0.00000000
		-6.27243618	-0.00072353	0.00044343	0.00072724	-0.00000112	0.00000000
		-6.27243659	-0.00072336	0.00044332	0.00072707	-0.00000112	0.00000000
		-6.27243700	-0.00072319	0.00044322	0.00072690	-0.00000112	0.00000000
		-6.27243740	-0.00072302	0.00044312	0.00072673	-0.00000112	0.00000000
		-6.27243781	-0.00072285	0.00044301	0.00072656	-0.00000112	0.00000000
		-6.27243822	-0.00072269	0.00044291	0.00072639	-0.00000112	0.00000000
		-6.27243862	-0.00072252	0.00044281	0.00072622	-0.00000112	0.00000000
		-6.27243903	-0.00072235	0.00044270	0.00072605	-0.00000112	0.00000000
		-6.27243944	-0.00072218	0.00044260	0.00072588	-0.00000112	0.00000000
		-6.27243984	-0.00072201	0.00044250	0.00072571	-0.00000112	0.00000000
		-6.27244025	-0.00072184	0.00044239	0.00072554	-0.00000112	0.00000000
		-6.27244065	-0.00072167	0.00044229	0.00072537	-0.00000112	0.00000000
		-6.27244106	-0.00072151	0.00044219	0.00072520	-0.00000112	0.00000000
		-6.27244146	-0.00072134	0.00044208	0.00072503	-0.00000112	0.00000000
		-6.27244187	-0.00072117	0.00044198	0.00072486	-0.00000112	0.00000000
		-6.27244227	-0.00072100	0.00044188	0.00072470	-0.00000112	0.00000000
		-6.27244268	-0.00072083	0.00044177	0.00072453	-0.00000112	0.00000000
		-6.27244308	-0.00072067	0.00044167	0.00072436	-0.00000112	0.00000000
		-6.27244349	-0.00072050	0.00044157	0.00072419	-0.00000112	0.00000000
		-6.27244389	-0.00072033	0.00044147	0.00072402	-0.00000112	0.00000000
		-6.27244429	-0.00072016	0.00044136	0.00072385	-0.00000112	0.00000000
		-6.27244470	-0.00072000	0.00044126	0.00072368	-0.00000112	0.00000000
		-6.27244510	-0.00071983	0.00044116	0.00072352	-0.00000112	0.00000000
		-6.27244550	-0.00071966	0.00044105	0.00072335	-0.00000112	0.00000000
		-6.27244591	-0.00071949	0.00044095	0.00072318	-0.00000112	0.00000000
		-6.27244631	-0.00071933	0.00044085	0.00072301	-0.00000112	0.00000000
		-6.27244671	-0.00071916	0.00044075	0.00072284	-0.00000112	0.00000000
		-6.27244712	-0.00071899	0.00044064	0.00072268	-0.00000112	0.00000000
		-6.27244752	-0.00071883	0.00044054	0.00072251	-0.00000112	0.00000000
		-6.27244792	-0.00071866	0.00044044	0.00072234	-0.00000112	0.00000000
		-6.27244832	-0.00071849	0.00044034	0.00072217	-0.00000112	0.00000000
		-6.27244872	-0.00071833	0.00044024	0.00072200	-0.00000112	0.00000000
		-6.27244912	-0.00071816	0.00044013	0.00072184	-0.00000112	0.00000000
		-6.27244953	-0.00071799	0.00044003	0.00072167	-0.00000112	0.00000000
		-6.27244993	-0.00071783	0.00043993	0.00072150	-0.00000112	0.00000000
		-6.27245033	-0.00071766	0.00043983	0.00072133	-0.00000112	0.00000000
		-6.27245073	-0.00071749	0.00043972	0.00072117	-0.00000112	0.00000000
		-6.27245113	-0.00071733	0.00043962	0.00072100	-0.00000112	0.00000000
		-6.27245153	-0.00071716	0.00043952	0.00072083	-0.00000112	0.00000000
		-6.27245193	-0.00071699	0.00043942	0.00072067	-0.00000112	0.00000000
		-6.27245233	-0.00071683	0.00043932	0.00072050	-0.00000112	0.00000000
		-6.27245273	-0.00071666	0.00043922	0.00072033	-0.00000112	0.00000000
		-6.27245313	-0.00071650	0.00043911	0.00072017	-0.00000112	0.00000000
		-6.27245353	-0.00071633	0.00043901	0.00072000	-0.00000112	0.00000000
		-6.27245393	-0.00071617	0.00043891	0.00071983	-0.00000112	0.00000000
		-6.27245432	-0.00071600	0.00043881	0.00071967	-0.00000112	0.00000000
		-6.27245472	-0.00071583	0.00043871	0.00071950	-0.00000112	0.00000000
		-6.27245512	-0.00071567	0.00043861	0.00071933	-0.00000112	0.00000000
		-6.27245552	-0.00071550	0.00043850	0.00071917	-0.00000112	0.00000000
		-6.27245592	-0.00071534	0.00043840	0.00071900	-0.00000112	0.00000000
		-6.27245632	-0.00071517	0.00043830	0.00071883	-0.00000112	0.00000000
		-6.27245671	-0.00071501	0.00043820	0.00071867	-0.00000112	0.00000000
		-6.27245711	-0.00071484	0.00043810	0.00071850	-0.00000112	0.00000000
		-6.27245751	-0.00071468	0.00043800	0.00071834	-0.00000112	0.00000000
		-6.27245791	-0.00071451	0.00043790	0.00071817	-0.00000112	0.00000000
		-6.27245830	-0.00071435	0.00043779	0.00071800	-0.00000112	0.00000000
		-6.27245870	-0.00071418	0.00043769	0.00071784	-0.00000112	0.00000000
		-6.27245910	-0.00071402	0.00043759	0.00071767	-0.00000112	0.00000000
		-6.27245949	-0.00071385	0.00043749	0.00071751	-0.00000112	0.00000000
		-6.27245989	-0.00071369	0.00043739	0.00071734	-0.00000112	0.00000000
		-6.27246028	-0.00071352	0.00043729	0.00071718	-0.00000112	0.00000000
		-6.27246068	-0.00071336	0.00043719	0.00071701	-0.00000112	0.00000000
		-6.27246108	-0.00071320	0.00043709	0.00071685	-0.00000112	0.00000000
		-6.27246147	-0.00071303	0.00043699	0.00071668	-0.00000112	0.00000000
		-6.27246187	-0.00071287	0.00043689	0.00071652	-0.00000112	0.00000000
		-6.27246226	-0.00071270	0.00043679	0.00071635	-0.00000112	0.00000000
		-6.27246266	-0.00071254	0.00043669	0.00071619	-0.00000112	0.00000000
		-6.27246305	-0.00071237	0.00043658	0.00071602	-0.00000112	0.00000000
		-6.27246345	-0.00071221	0.00043648	0.00071586	-0.00000112	0.00000000
		-6.27246384	-0.00071205	0.00043638	0.00071569	-0.00000112	0.00000000
		-6.27246423	-0.00071188	0.00043628	0.00071553	-0.00000112	0.00000000
		-6.27246463	-0.00071172	0.00043618	0.00071536	-0.00000112	0.00000000
		-6.27246502	-0.00071156	0.00043608	0.00071520	-0.00000112	0.00000000
		-6.27246541	-0.00071139	0.00043598	0.00071503	-0.00000112	0.00000000
		-6.27246581	-0.00071123	0.00043588	0.00071487	-0.00000112	0.00000000
		-6.27246620	-0.00071107	0.00043578	0.00071471	-0.00000112	0.00000000
		-6.27246659	-0.00071090	0.00043568	0.00071454	-0.00000112	0.00000000
		-6.27246699	-0.00071074	0.00043558	0.00071438	-0.00000112	0.00000000
		-6.27246738	-0.00071058	0.00043548	0.00071421	-0.00000112	0.00000000
		-6.27246777	-0.00071041	0.00043538	0.00071405	-0.00000112	0.00000000
		-6.27246816	-0.00071025	0.00043528	0.00071389	-0.00000112	0.00000000
		-6.27246856	-0.00071009	0.00043518	0.00071372	-0.00000112	0.00000000
		-6.27246895	-0.00070992	0.00043508	0.00071356	-0.00000112	0.00000000
		-6.27246934	-0.00070976	0.00043498	0.00071339	-0.00000112	0.00000000
		-6.27246973	-0.00070960	0.00043488	0.00071323	-0.00000112	0.00000000
		-6.27247012	-0.00070944	0.00043478	0.00071307	-0.00000112	0.00000000
		-6.27247051	-0.00070927	0.00043468	0.00071290	-0.00000112	0.00000000
		-6.27247090	-0.00070911	0.00043458	0.00071274	-0.00000112	0.00000000
		-6.27247129	-0.00070895	0.00043448	0.00071258	-0.00000112	0.00000000
		-6.27247168	-0.00070879	0.00043438	0.00071241	-0.00000112	0.00000000
		-6.27247207	-0.00070862	0.00043428	0.00071225	-0.00000112	0.00000000
		-6.27247246	-0.00070846	0.00043418	0.00071209	-0.00000112	0.00000000
		-6.27247285	-0.00070830	0.00043409	0.00071193	-0.00000112	0.00000000
		-6.27247324	-0.00070814	0.00043399	0.00071176	-0.00000112	0.00000000
		-6.27247363	-0.00070798	0.00043389	0.00071160	-0.00000112	0.00000000
		-6.27247402	-0.00070781	0.00043379	0.00071144	-0.00000112	0.00000000
		-6.27247441	-0.00070765	0.00043369	0.00071127	-0.00000112	0.00000000
		-6.27247480	-0.00070749	0.00043359	0.00071111	-0.00000112	0.00000000
		-6.27247519	-0.00070733	0.00043349	0.00071095	-0.00000112	0.00000000
		-6.27247558	-0.00070717	0.00043339	0.00071079	-0.00000112	0.00000000
		-6.27247597	-0.00070701	0.00043329	0.00071062	-0.00000112	0.00000000
		-6.27247636	-0.00070684	0.00043319	0.00071046	-0.00000112	0.00000000
		-6.27247674	-0.00070668	0.00043309	0.00071030	-0.00000112	0.00000000
		-6.27247713	-0.00070652	0.00043299	0.00071014	-0.00000112	0.00000000
		-6.27247752	-0.00070636	0.00043290	0.00070998	-0.00000112	0.00000000
		-6.27247791	-0.00070620	0.00043280	0.00070981	-0.00000112	0.00000000
		-6.27247829	-0.00070604	0.00043270	0.00070965	-0.00000112	0.00000000
		-6.27247868	-0.00070588	0.00043260	0.00070949	-0.00000112	0.00000000
		-6.27247907	-0.00070572	0.00043250	0.00070933	-0.00000112	0.00000000
		-6.27247945	-0.00070556	0.00043240	0.00070917	-0.00000112	0.00000000
		-6.27247984	-0.00070539	0.00043230	0.00070900	-0.00000112	0.00000000
		-6.27248023	-0.00070523	0.00043220	0.00070884	-0.00000112	0.00000000
		-6.27248061	-0.00070507	0.00043211	0.00070868	-0.00000112	0.00000000
		-6.27248100	-0.00070491	0.00043201	0.00070852	-0.00000112	0.00000000
		-6.27248139	-0.00070475	0.00043191	0.00070836	-0.00000112	0.00000000
		-6.27248177	-0.00070459	0.00043181	0.00070820	-0.00000112	0.00000000
		-6.27248216	-0.00070443	0.00043171	0.00070804	-0.00000112	0.00000000
		-6.27248254	-0.00070427	0.00043161	0.00070788	-0.00000112	0.00000000
		-6.27248293	-0.00070411	0.00043152	0.00070771	-0.00000112	0.00000000
		-6.27248331	-0.00070395	0.00043142	0.00070755	-0.00000112	0.00000000
		-6.27248370	-0.00070379	0.00043132	0.00070739	-0.00000112	0.00000000
		-6.27248408	-0.00070363	0.00043122	0.00070723	-0.00000112	0.00000000
		-6.27248447	-0.00070347	0.00043112	0.00070707	-0.00000112	0.00000000
		-6.27248485	-0.00070331	0.00043103	0.00070691	-0.00000112	0.00000000
		-6.27248523	-0.00070315	0.00043093	0.00070675	-0.00000112	0.00000000
		-6.27248562	-0.00070299	0.00043083	0.00070659	-0.00000112	0.00000000
		-6.27248600	-0.00070283	0.00043073	0.00070643	-0.00000112	0.00000000
		-6.27248638	-0.00070267	0.00043063	0.00070627	-0.00000112	0.00000000
		-6.27248677	-0.00070251	0.00043054	0.00070611	-0.00000112	0.00000000
		-6.27248715	-0.00070235	0.00043044	0.00070595	-0.00000112	0.00000000
		-6.27248753	-0.00070219	0.00043034	0.00070579	-0.00000112	0.00000000
		-6.27248792	-0.00070203	0.00043024	0.00070563	-0.00000112	0.00000000
		-6.27248830	-0.00070188	0.00043014	0.00070547	-0.00000112	0.00000000
		-6.27248868	-0.00070172	0.00043005	0.00070531	-0.00000112	0.00000000
		-6.27248906	-0.00070156	0.00042995	0.00070515	-0.00000112	0.00000000
		-6.27248945	-0.00070140	0.00042985	0.00070499	-0.00000112	0.00000000
		-6.27248983	-0.00070124	0.00042975	0.00070483	-0.00000112	0.00000000
		-6.27249021	-0.00070108	0.00042966	0.00070467	-0.00000112	0.00000000
		-6.27249059	-0.00070092	0.00042956	0.00070451	-0.00000112	0.00000000
		-6.27249097	-0.00070076	0.00042946	0.00070435	-0.00000112	0.00000000
		-6.27249135	-0.00070060	0.00042937	0.00070419	-0.00000112	0.00000000
		-6.27249173	-0.00070045	0.00042927	0.00070403	-0.00000112	0.00000000
		-6.27249211	-0.00070029	0.00042917	0.00070387	-0.00000112	0.00000000
		-6.27249250	-0.00070013	0.00042907	0.00070371	-0.00000112	0.00000000
		-6.27249288	-0.00069997	0.00042898	0.00070355	-0.00000112	0.00000000
		-6.27249326	-0.00069981	0.00042888	0.00070339	-0.00000112	0.00000000
		-6.27249364	-0.00069965	0.00042878	0.00070323	-0.00000112	0.00000000
		-6.27249402	-0.00069950	0.00042869	0.00070308	-0.00000112	0.00000000
		-6.27249440	-0.00069934	0.00042859	0.00070292	-0.00000112	0.00000000
		-6.27249477	-0.00069918	0.00042849	0.00070276	-0.00000112	0.00000000
		-6.27249515	-0.00069902	0.00042840	0.00070260	-0.00000112	0.00000000
		-6.27249553	-0.00069887	0.00042830	0.00070244	-0.00000112	0.00000000
		-6.27249591	-0.00069871	0.00042820	0.00070228	-0.00000112	0.00000000
		-6.27249629	-0.00069855	0.00042810	0.00070212	-0.00000112	0.00000000
		-6.27249667	-0.00069839	0.00042801	0.00070197	-0.00000112	0.00000000
		-6.27249705	-0.00069823	0.00042791	0.00070181	-0.00000112	0.00000000
		-6.27249743	-0.00069808	0.00042781	0.00070165	-0.00000112	0.00000000
		-6.27249780	-0.00069792	0.00042772	0.00070149	-0.00000112	0.00000000
		-6.27249818	-0.00069776	0.00042762	0.00070133	-0.00000112	0.00000000
		-6.27249856	-0.00069761	0.00042753	0.00070117	-0.00000112	0.00000000
		-6.27249894	-0.00069745	0.00042743	0.00070102	-0.00000112	0.00000000
		-6.27249931	-0.00069729	0.00042733	0.00070086	-0.00000112	0.00000000
		-6.27249969	-0.00069713	0.00042724	0.00070070	-0.00000112	0.00000000
		-6.27250007	-0.00069698	0.00042714	0.00070054	-0.00000112	0.00000000
		-6.27250045	-0.00069682	0.00042704	0.00070038	-0.00000112	0.00000000
		-6.27250082	-0.00069666	0.00042695	0.00070023	-0.00000112	0.00000000
		-6.27250120	-0.00069651	0.00042685	0.00070007	-0.00000112	0.00000000
		-6.27250158	-0.00069635	0.00042676	0.00069991	-0.00000112	0.00000000
		-6.27250195	-0.00069619	0.00042666	0.00069975	-0.00000112	0.00000000
		-6.27250233	-0.00069604	0.00042656	0.00069960	-0.00000112	0.00000000
		-6.27250270	-0.00069588	0.00042647	0.00069944	-0.00000112	0.00000000
		-6.27250308	-0.00069572	0.00042637	0.00069928	-0.00000112	0.00000000
		-6.27250345	-0.00069557	0.00042628	0.00069913	-0.00000112	0.00000000
		-6.27250383	-0.00069541	0.00042618	0.00069897	-0.00000112	0.00000000
		-6.27250420	-0.00069525	0.00042608	0.00069881	-0.00000112	0.00000000
		-6.27250458	-0.00069510	0.00042599	0.00069865	-0.00000112	0.00000000
		-6.27250495	-0.00069494	0.00042589	0.00069850	-0.00000112	0.00000000
		-6.27250533	-0.00069479	0.00042580	0.00069834	-0.00000112	0.00000000
		-6.27250570	-0.00069463	0.00042570	0.00069818	-0.00000112	0.00000000
		-6.27250608	-0.00069447	0.00042561	0.00069803	-0.00000112	0.00000000
		-6.27250645	-0.00069432	0.00042551	0.00069787	-0.00000112	0.00000000
		-6.27250683	-0.00069416	0.00042541	0.00069771	-0.00000112	0.00000000
		-6.27250720	-0.00069401	0.00042532	0.00069756	-0.00000112	0.00000000
		-6.27250757	-0.00069385	0.00042522	0.00069740	-0.00000112	0.00000000
		-6.27250795	-0.00069370	0.00042513	0.00069725	-0.00000112	0.00000000
		-6.27250832	-0.00069354	0.00042503	0.00069709	-0.00000112	0.00000000
		-6.27250869	-0.00069339	0.00042494	0.00069693	-0.00000112	0.00000000
		-6.27250906	-0.00069323	0.00042484	0.00069678	-0.00000112	0.00000000
		-6.27250944	-0.00069308	0.00042475	0.00069662	-0.00000112	0.00000000
		-6.27250981	-0.00069292	0.00042465	0.00069646	-0.00000112	0.00000000
		-6.27251018	-0.00069277	0.00042456	0.00069631	-0.00000112	0.00000000
		-6.27251055	-0.00069261	0.00042446	0.00069615	-0.00000112	0.00000000
		-6.27251093	-0.00069246	0.00042437	0.00069600	-0.00000112	0.00000000
		-6.27251130	-0.00069230	0.00042427	0.00069584	-0.00000112	0.00000000
		-6.27251167	-0.00069215	0.00042418	0.00069569	-0.00000112	0.00000000
		-6.27251204	-0.00069199	0.00042408	0.00069553	-0.00000112	0.00000000
		-6.27251241	-0.00069184	0.00042399	0.00069537	-0.00000112	0.00000000
		-6.27251278	-0.00069168	0.00042389	0.00069522	-0.00000112	0.00000000
		-6.27251315	-0.00069153	0.00042380	0.00069506	-0.00000112	0.00000000
		-6.27251353	-0.00069137	0.00042370	0.00069491	-0.00000112	0.00000000
		-6.27251390	-0.00069122	0.00042361	0.00069475	-0.00000112	0.00000000
		-6.27251427	-0.00069106	0.00042351	0.00069460	-0.00000112	0.00000000
		-6.27251464	-0.00069091	0.00042342	0.00069444	-0.00000112	0.00000000
		-6.27251501	-0.00069076	0.00042332	0.00069429	-0.00000112	0.00000000
		-6.27251538	-0.00069060	0.00042323	0.00069413	-0.00000112	0.00000000
		-6.27251575	-0.00069045	0.00042314	0.00069398	-0.00000112	0.00000000
		-6.27251612	-0.00069029	0.00042304	0.00069382	-0.00000112	0.00000000
		-6.27251649	-0.00069014	0.00042295	0.00069367	-0.00000112	0.00000000
		-6.27251685	-0.00068999	0.00042285	0.00069351	-0.00000112	0.00000000
		-6.27251722	-0.00068983	0.00042276	0.00069336	-0.00000112	0.00000000
		-6.27251759	-0.00068968	0.00042266	0.00069321	-0.00000112	0.00000000
		-6.27251796	-0.00068952	0.00042257	0.00069305	-0.00000112	0.00000000
		-6.27251833	-0.00068937	0.00042248	0.00069290	-0.00000112	0.00000000
		-6.27251870	-0.00068922	0.00042238	0.00069274	-0.00000112	0.00000000
		-6.27251907	-0.00068906	0.00042229	0.00069259	-0.00000112	0.00000000
		-6.27251943	-0.00068891	0.00042219	0.00069243	-0.00000112	0.00000000
		-6.27251980	-0.00068876	0.00042210	0.00069228	-0.00000112	0.00000000
		-6.27252017	-0.00068860	0.00042200	0.00069213	-0.00000112	0.00000000
		-6.27252054	-0.00068845	0.00042191	0.00069197	-0.00000112	0.00000000
		-6.27252091	-0.00068830	0.00042182	0.00069182	-0.00000112	0.00000000
		-6.27252127	-0.00068815	0.00042172	0.00069166	-0.00000112	0.00000000
		-6.27252164	-0.00068799	0.00042163	0.00069151	-0.00000112	0.00000000
		-6.27252201	-0.00068784	0.00042154	0.00069136	-0.00000112	0.00000000
		-6.27252237	-0.00068769	0.00042144	0.00069120	-0.00000112	0.00000000
		-6.27252274	-0.00068753	0.00042135	0.00069105	-0.00000112	0.00000000
		-6.27252311	-0.00068738	0.00042125	0.00069090	-0.00000112	0.00000000
		-6.27252347	-0.00068723	0.00042116	0.00069074	-0.00000112	0.00000000
		-6.27252384	-0.00068708	0.00042107	0.00069059	-0.00000112	0.00000000
		-6.27252420	-0.00068692	0.00042097	0.00069044	-0.00000112	0.00000000
		-6.27252457	-0.00068677	0.00042088	0.00069028	-0.00000112	0.00000000
		-6.27252494	-0.00068662	0.00042079	0.00069013	-0.00000112	0.00000000
		-6.27252530	-0.00068647	0.00042069	0.00068998	-0.00000112	0.00000000
		-6.27252567	-0.00068631	0.00042060	0.00068982	-0.00000112	0.00000000
		-6.27252603	-0.00068616	0.00042051	0.00068967	-0.00000112	0.00000000
		-6.27252640	-0.00068601	0.00042041	0.00068952	-0.00000112	0.00000000
		-6.27252676	-0.00068586	0.00042032	0.00068937	-0.00000112	0.00000000
		-6.27252712	-0.00068571	0.00042023	0.00068921	-0.00000112	0.00000000
		-6.27252749	-0.00068555	0.00042013	0.00068906	-0.00000112	0.00000000
		-6.27252785	-0.00068540	0.00042004	0.00068891	-0.00000112	0.00000000
		-6.27252822	-0.00068525	0.00041995	0.00068875	-0.00000112	0.00000000
		-6.27252858	-0.00068510	0.00041985	0.00068860	-0.00000112	0.00000000
		-6.27252895	-0.00068495	0.00041976	0.00068845	-0.00000112	0.00000000
		-6.27252931	-0.00068480	0.00041967	0.00068830	-0.00000112	0.00000000
		-6.27252967	-0.00068464	0.00041958	0.00068815	-0.00000112	0.00000000
		-6.27253004	-0.00068449	0.00041948	0.00068799	-0.00000112	0.00000000
		-6.27253040	-0.00068434	0.00041939	0.00068784	-0.00000112	0.00000000
		-6.27253076	-0.00068419	0.00041930	0.00068769	-0.00000112	0.00000000
		-6.27253112	-0.00068404	0.00041920	0.00068754	-0.00000112	0.00000000
		-6.27253149	-0.00068389	0.00041911	0.00068738	-0.00000112	0.00000000
		-6.27253185	-0.00068374	0.00041902	0.00068723	-0.00000112	0.00000000
		-6.27253221	-0.00068359	0.00041893	0.00068708	-0.00000112	0.00000000
		-6.27253257	-0.00068344	0.00041883	0.00068693	-0.00000112	0.00000000
		-6.27253293	-0.00068328	0.00041874	0.00068678	-0.00000112	0.00000000
		-6.27253330	-0.00068313	0.00041865	0.00068663	-0.00000112	0.00000000
		-6.27253366	-0.00068298	0.00041856	0.00068647	-0.00000112	0.00000000
		-6.27253402	-0.00068283	0.00041846	0.00068632	-0.00000112	0.00000000
		-6.27253438	-0.00068268	0.00041837	0.00068617	-0.00000112	0.00000000
		-6.27253474	-0.00068253	0.00041828	0.00068602	-0.00000112	0.00000000
		-6.27253510	-0.00068238	0.00041819	0.00068587	-0.00000112	0.00000000
		-6.27253546	-0.00068223	0.00041810	0.00068572	-0.00000112	0.00000000
		-6.27253582	-0.00068208	0.00041800	0.00068557	-0.00000112	0.00000000
		-6.27253618	-0.00068193	0.00041791	0.00068542	-0.00000112	0.00000000
		-6.27253654	-0.00068178	0.00041782	0.00068526	-0.00000112	0.00000000
		-6.27253690	-0.00068163	0.00041773	0.00068511	-0.00000112	0.00000000
		-6.27253726	-0.00068148	0.00041763	0.00068496	-0.00000112	0.00000000
		-6.27253762	-0.00068133	0.00041754	0.00068481	-0.00000112	0.00000000
		-6.27253798	-0.00068118	0.00041745	0.00068466	-0.00000112	0.00000000
		-6.27253834	-0.00068103	0.00041736	0.00068451	-0.00000112	0.00000000
		-6.27253870	-0.00068088	0.00041727	0.00068436	-0.00000112	0.00000000
		-6.27253906	-0.00068073	0.00041718	0.00068421	-0.00000112	0.00000000
		-6.27253942	-0.00068058	0.00041708	0.00068406	-0.00000112	0.00000000
		-6.27253978	-0.00068043	0.00041699	0.00068391	-0.00000112	0.00000000
		-6.27254014	-0.00068028	0.00041690	0.00068376	-0.00000112	0.00000000
		-6.27254050	-0.00068013	0.00041681	0.00068361	-0.00000112	0.00000000
		-6.27254086	-0.00067998	0.00041672	0.00068346	-0.00000112	0.00000000
		-6.27254121	-0.00067983	0.00041662	0.00068331	-0.00000112	0.00000000
		-6.27254157	-0.00067968	0.00041653	0.00068316	-0.00000112	0.00000000
		-6.27254193	-0.00067953	0.00041644	0.00068301	-0.00000112	0.00000000
		-6.27254229	-0.00067939	0.00041635	0.00068286	-0.00000112	0.00000000
		-6.27254264	-0.00067924	0.00041626	0.00068271	-0.00000112	0.00000000
		-6.27254300	-0.00067909	0.00041617	0.00068256	-0.00000112	0.00000000
		-6.27254336	-0.00067894	0.00041608	0.00068241	-0.00000112	0.00000000
		-6.27254372	-0.00067879	0.00041598	0.00068226	-0.00000112	0.00000000
		-6.27254407	-0.00067864	0.00041589	0.00068211	-0.00000112	0.00000000
		-6.27254443	-0.00067849	0.00041580	0.00068196	-0.00000112	0.00000000
		-6.27254479	-0.00067834	0.00041571	0.00068181	-0.00000112	0.00000000
		-6.27254514	-0.00067819	0.00041562	0.00068166	-0.00000112	0.00000000
		-6.27254550	-0.00067805	0.00041553	0.00068151	-0.00000112	0.00000000
		-6.27254585	-0.00067790	0.00041544	0.00068136	-0.00000112	0.00000000
		-6.27254621	-0.00067775	0.00041535	0.00068121	-0.00000112	0.00000000
		-6.27254657	-0.00067760	0.00041526	0.00068106	-0.00000112	0.00000000
		-6.27254692	-0.00067745	0.00041516	0.00068092	-0.00000112	0.00000000
		-6.27254728	-0.00067730	0.00041507	0.00068077	-0.00000112	0.00000000
		-6.27254763	-0.00067716	0.00041498	0.00068062	-0.00000112	0.00000000
		-6.27254799	-0.00067701	0.00041489	0.00068047	-0.00000112	0.00000000
		-6.27254834	-0.00067686	0.00041480	0.00068032	-0.00000112	0.00000000
		-6.27254870	-0.00067671	0.00041471	0.00068017	-0.00000112	0.00000000
		-6.27254905	-0.00067656	0.00041462	0.00068002	-0.00000112	0.00000000
		-6.27254941	-0.00067642	0.00041453	0.00067987	-0.00000112	0.00000000
		-6.27254976	-0.00067627	0.00041444	0.00067972	-0.00000112	0.00000000
		-6.27255012	-0.00067612	0.00041435	0.00067958	-0.00000112	0.00000000
		-6.27255047	-0.00067597	0.00041426	0.00067943	-0.00000112	0.00000000
		-6.27255082	-0.00067583	0.00041417	0.00067928	-0.00000112	0.00000000
		-6.27255118	-0.00067568	0.00041408	0.00067913	-0.00000112	0.00000000
		-6.27255153	-0.00067553	0.00041399	0.00067898	-0.00000112	0.00000000
		-6.27255188	-0.00067538	0.00041390	0.00067883	-0.00000112	0.00000000
		-6.27255224	-0.00067524	0.00041381	0.00067869	-0.00000112	0.00000000
		-6.27255259	-0.00067509	0.00041372	0.00067854	-0.00000112	0.00000000
		-6.27255294	-0.00067494	0.00041362	0.00067839	-0.00000112	0.00000000
		-6.27255330	-0.00067479	0.00041353	0.00067824	-0.00000112	0.00000000
		-6.27255365	-0.00067465	0.00041344	0.00067810	-0.00000112	0.00000000
		-6.27255400	-0.00067450	0.00041335	0.00067795	-0.00000112	0.00000000
		-6.27255435	-0.00067435	0.00041326	0.00067780	-0.00000112	0.00000000
		-6.27255470	-0.00067421	0.00041317	0.00067765	-0.00000112	0.00000000
		-6.27255506	-0.00067406	0.00041308	0.00067750	-0.00000112	0.00000000
		-6.27255541	-0.00067391	0.00041299	0.00067736	-0.00000112	0.00000000
		-6.27255576	-0.00067377	0.00041290	0.00067721	-0.00000112	0.00000000
		-6.27255611	-0.00067362	0.00041281	0.00067706	-0.00000112	0.00000000
		-6.27255646	-0.00067347	0.00041272	0.00067691	-0.00000112	0.00000000
		-6.27255681	-0.00067333	0.00041263	0.00067677	-0.00000112	0.00000000
		-6.27255717	-0.00067318	0.00041254	0.00067662	-0.00000112	0.00000000
		-6.27255752	-0.00067303	0.00041245	0.00067647	-0.00000112	0.00000000
		-6.27255787	-0.00067289	0.00041236	0.00067633	-0.00000112	0.00000000
		-6.27255822	-0.00067274	0.00041228	0.00067618	-0.00000112	0.00000000
		-6.27255857	-0.00067260	0.00041219	0.00067603	-0.00000112	0.00000000
		-6.27255892	-0.00067245	0.00041210	0.00067589	-0.00000112	0.00000000
		-6.27255927	-0.00067230	0.00041201	0.00067574	-0.00000112	0.00000000
		-6.27255962	-0.00067216	0.00041192	0.00067559	-0.00000112	0.00000000
		-6.27255997	-0.00067201	0.00041183	0.00067544	-0.00000112	0.00000000
		-6.27256032	-0.00067187	0.00041174	0.00067530	-0.00000112	0.00000000
		-6.27256067	-0.00067172	0.00041165	0.00067515	-0.00000112	0.00000000
		-6.27256102	-0.00067157	0.00041156	0.00067501	-0.00000112	0.00000000
		-6.27256137	-0.00067143	0.00041147	0.00067486	-0.00000112	0.00000000
		-6.27256171	-0.00067128	0.00041138	0.00067471	-0.00000112	0.00000000
		-6.27256206	-0.00067114	0.00041129	0.00067457	-0.00000112	0.00000000
		-6.27256241	-0.00067099	0.00041120	0.00067442	-0.00000112	0.00000000
		-6.27256276	-0.00067085	0.00041111	0.00067427	-0.00000112	0.00000000
		-6.27256311	-0.00067070	0.00041102	0.00067413	-0.00000112	0.00000000
		-6.27256346	-0.00067056	0.00041093	0.00067398	-0.00000112	0.00000000
		-6.27256381	-0.00067041	0.00041085	0.00067384	-0.00000112	0.00000000
		-6.27256415	-0.00067027	0.00041076	0.00067369	-0.00000112	0.00000000
		-6.27256450	-0.00067012	0.00041067	0.00067354	-0.00000112	0.00000000
		-6.27256485	-0.00066998	0.00041058	0.00067340	-0.00000112	0.00000000
		-6.27256520	-0.00066983	0.00041049	0.00067325	-0.00000112	0.00000000
		-6.27256554	-0.00066969	0.00041040	0.00067311	-0.00000112	0.00000000
		-6.27256589	-0.00066954	0.00041031	0.00067296	-0.00000112	0.00000000
		-6.27256624	-0.00066940	0.00041022	0.00067282	-0.00000112	0.00000000
		-6.27256658	-0.00066925	0.00041013	0.00067267	-0.00000112	0.00000000
		-6.27256693	-0.00066911	0.00041005	0.00067252	-0.00000112	0.00000000
		-6.27256728	-0.00066896	0.00040996	0.00067238	-0.00000112	0.00000000
		-6.27256762	-0.00066882	0.00040987	0.00067223	-0.00000112	0.00000000
		-6.27256797	-0.00066867	0.00040978	0.00067209	-0.00000112	0.00000000
		-6.27256832	-0.00066853	0.00040969	0.00067194	-0.00000112	0.00000000
		-6.27256866	-0.00066838	0.00040960	0.00067180	-0.00000112	0.00000000
		-6.27256901	-0.00066824	0.00040951	0.00067165	-0.00000112	0.00000000
		-6.27256935	-0.00066810	0.00040943	0.00067151	-0.00000112	0.00000000
		-6.27256970	-0.00066795	0.00040934	0.00067136	-0.00000112	0.00000000
		-6.27257004	-0.00066781	0.00040925	0.00067122	-0.00000112	0.00000000
		-6.27257039	-0.00066766	0.00040916	0.00067107	-0.00000112	0.00000000
		-6.27257073	-0.00066752	0.00040907	0.00067093	-0.00000112	0.00000000
		-6.27257108	-0.00066738	0.00040898	0.00067078	-0.00000112	0.00000000
		-6.27257142	-0.00066723	0.00040890	0.00067064	-0.00000112	0.00000000
		-6.27257177	-0.00066709	0.00040881	0.00067050	-0.00000112	0.00000000
		-6.27257211	-0.00066694	0.00040872	0.00067035	-0.00000112	0.00000000
		-6.27257246	-0.00066680	0.00040863	0.00067021	-0.00000112	0.00000000
		-6.27257280	-0.00066666	0.00040854	0.00067006	-0.00000112	0.00000000
		-6.27257315	-0.00066651	0.00040846	0.00066992	-0.00000112	0.00000000
		-6.27257349	-0.00066637	0.00040837	0.00066977	-0.00000112	0.00000000
		-6.27257383	-0.00066623	0.00040828	0.00066963	-0.00000112	0.00000000
		-6.27257418	-0.00066608	0.00040819	0.00066949	-0.00000112	0.00000000
		-6.27257452	-0.00066594	0.00040810	0.00066934	-0.00000112	0.00000000
		-6.27257486	-0.00066580	0.00040802	0.00066920	-0.00000112	0.00000000
		-6.27257521	-0.00066565	0.00040793	0.00066905	-0.00000112	0.00000000
		-6.27257555	-0.00066551	0.00040784	0.00066891	-0.00000112	0.00000000
		-6.27257589	-0.00066537	0.00040775	0.00066877	-0.00000112	0.00000000
		-6.27257623	-0.00066522	0.00040766	0.00066862	-0.00000112	0.00000000
		-6.27257658	-0.00066508	0.00040758	0.00066848	-0.00000112	0.00000000
		-6.27257692	-0.00066494	0.00040749	0.00066833	-0.00000112	0.00000000
		-6.27257726	-0.00066480	0.00040740	0.00066819	-0.00000112	0.00000000
		-6.27257760	-0.00066465	0.00040731	0.00066805	-0.00000112	0.00000000
		-6.27257795	-0.00066451	0.00040723	0.00066790	-0.00000112	0.00000000
		-6.27257829	-0.00066437	0.00040714	0.00066776	-0.00000112	0.00000000
		-6.27257863	-0.00066422	0.00040705	0.00066762	-0.00000112	0.00000000
		-6.27257897	-0.00066408	0.00040696	0.00066747	-0.00000112	0.00000000
		-6.27257931	-0.00066394	0.00040688	0.00066733	-0.00000112	0.00000000
		-6.27257965	-0.00066380	0.00040679	0.00066719	-0.00000112	0.00000000
		-6.27257999	-0.00066366	0.00040670	0.00066704	-0.00000112	0.00000000
		-6.27258033	-0.00066351	0.00040662	0.00066690	-0.00000112	0.00000000
		-6.27258068	-0.00066337	0.00040653	0.00066676	-0.00000112	0.00000000
		-6.27258102	-0.00066323	0.00040644	0.00066662	-0.00000112	0.00000000
		-6.27258136	-0.00066309	0.00040635	0.00066647	-0.00000112	0.00000000
		-6.27258170	-0.00066294	0.00040627	0.00066633	-0.00000112	0.00000000
		-6.27258204	-0.00066280	0.00040618	0.00066619	-0.00000112	0.00000000
		-6.27258238	-0.00066266	0.00040609	0.00066604	-0.00000112	0.00000000
		-6.27258272	-0.00066252	0.00040601	0.00066590	-0.00000112	0.00000000
		-6.27258306	-0.00066238	0.00040592	0.00066576	-0.00000112	0.00000000
		-6.27258340	-0.00066224	0.00040583	0.00066562	-0.00000112	0.00000000
		-6.27258373	-0.00066209	0.00040574	0.00066547	-0.00000112	0.00000000
		-6.27258407	-0.00066195	0.00040566	0.00066533	-0.00000112	0.00000000
		-6.27258441	-0.00066181	0.00040557	0.00066519	-0.00000112	0.00000000
		-6.27258475	-0.00066167	0.00040548	0.00066505	-0.00000112	0.00000000
		-6.27258509	-0.00066153	0.00040540	0.00066491	-0.00000112	0.00000000
		-6.27258543	-0.00066139	0.00040531	0.00066476	-0.00000112	0.00000000
		-6.27258577	-0.00066125	0.00040522	0.00066462	-0.00000112	0.00000000
		-6.27258611	-0.00066110	0.00040514	0.00066448	-0.00000112	0.00000000
		-6.27258644	-0.00066096	0.00040505	0.00066434	-0.00000112	0.00000000
		-6.27258678	-0.00066082	0.00040496	0.00066420	-0.00000112	0.00000000
		-6.27258712	-0.00066068	0.00040488	0.00066405	-0.00000112	0.00000000
		-6.27258746	-0.00066054	0.00040479	0.00066391	-0.00000112	0.00000000
		-6.27258780	-0.00066040	0.00040471	0.00066377	-0.00000112	0.00000000
		-6.27258813	-0.00066026	0.00040462	0.00066363	-0.00000112	0.00000000
		-6.27258847	-0.00066012	0.00040453	0.00066349	-0.00000112	0.00000000
		-6.27258881	-0.00065998	0.00040445	0.00066335	-0.00000112	0.00000000
		-6.27258914	-0.00065984	0.00040436	0.00066320	-0.00000112	0.00000000
		-6.27258948	-0.00065970	0.00040427	0.00066306	-0.00000112	0.00000000
		-6.27258982	-0.00065955	0.00040419	0.00066292	-0.00000112	0.00000000
		-6.27259015	-0.00065941	0.00040410	0.00066278	-0.00000112	0.00000000
		-6.27259049	-0.00065927	0.00040401	0.00066264	-0.00000112	0.00000000
		-6.27259083	-0.00065913	0.00040393	0.00066250	-0.00000112	0.00000000
		-6.27259116	-0.00065899	0.00040384	0.00066236	-0.00000112	0.00000000
		-6.27259150	-0.00065885	0.00040376	0.00066222	-0.00000112	0.00000000
		-6.27259184	-0.00065871	0.00040367	0.00066208	-0.00000112	0.00000000
		-6.27259217	-0.00065857	0.00040358	0.00066193	-0.00000112	0.00000000
		-6.27259251	-0.00065843	0.00040350	0.00066179	-0.00000112	0.00000000
		-6.27259284	-0.00065829	0.00040341	0.00066165	-0.00000112	0.00000000
		-6.27259318	-0.00065815	0.00040333	0.00066151	-0.00000112	0.00000000
		-6.27259351	-0.00065801	0.00040324	0.00066137	-0.00000112	0.00000000
		-6.27259385	-0.00065787	0.00040316	0.00066123	-0.00000112	0.00000000
		-6.27259418	-0.00065773	0.00040307	0.00066109	-0.00000112	0.00000000
		-6.27259452	-0.00065759	0.00040298	0.00066095	-0.00000112	0.00000000
		-6.27259485	-0.00065745	0.00040290	0.00066081	-0.00000112	0.00000000
		-6.27259519	-0.00065731	0.00040281	0.00066067	-0.00000112	0.00000000
		-6.27259552	-0.00065717	0.00040273	0.00066053	-0.00000112	0.00000000
		-6.27259585	-0.00065703	0.00040264	0.00066039	-0.00000112	0.00000000
		-6.27259619	-0.00065690	0.00040256	0.00066025	-0.00000112	0.00000000
		-6.27259652	-0.00065676	0.00040247	0.00066011	-0.00000112	0.00000000
		-6.27259686	-0.00065662	0.00040239	0.00065997	-0.00000112	0.00000000
		-6.27259719	-0.00065648	0.00040230	0.00065983	-0.00000112	0.00000000
		-6.27259752	-0.00065634	0.00040221	0.00065969	-0.00000112	0.00000000
		-6.27259786	-0.00065620	0.00040213	0.00065955	-0.00000112	0.00000000
		-6.27259819	-0.00065606	0.00040204	0.00065941	-0.00000112	0.00000000
		-6.27259852	-0.00065592	0.00040196	0.00065927	-0.00000112	0.00000000
		-6.27259885	-0.00065578	0.00040187	0.00065913	-0.00000112	0.00000000
		-6.27259919	-0.00065564	0.00040179	0.00065899	-0.00000112	0.00000000
		-6.27259952	-0.00065550	0.00040170	0.00065885	-0.00000112	0.00000000
		-6.27259985	-0.00065537	0.00040162	0.00065871	-0.00000112	0.00000000
		-6.27260018	-0.00065523	0.00040153	0.00065857	-0.00000112	0.00000000
		-6.27260052	-0.00065509	0.00040145	0.00065843	-0.00000112	0.00000000
		-6.27260085	-0.00065495	0.00040136	0.00065829	-0.00000112	0.00000000
		-6.27260118	-0.00065481	0.00040128	0.00065815	-0.00000112	0.00000000
		-6.27260151	-0.00065467	0.00040119	0.00065801	-0.00000112	0.00000000
		-6.27260184	-0.00065453	0.00040111	0.00065788	-0.00000112	0.00000000
		-6.27260217	-0.00065440	0.00040102	0.00065774	-0.00000112	0.00000000
		-6.27260251	-0.00065426	0.00040094	0.00065760	-0.00000112	0.00000000
		-6.27260284	-0.00065412	0.00040085	0.00065746	-0.00000112	0.00000000
		-6.27260317	-0.00065398	0.00040077	0.00065732	-0.00000112	0.00000000
		-6.27260350	-0.00065384	0.00040068	0.00065718	-0.00000112	0.00000000
		-6.27260383	-0.00065370	0.00040060	0.00065704	-0.00000112	0.00000000
		-6.27260416	-0.00065357	0.00040051	0.00065690	-0.00000112	0.00000000
		-6.27260449	-0.00065343	0.00040043	0.00065676	-0.00000112	0.00000000
		-6.27260482	-0.00065329	0.00040035	0.00065663	-0.00000112	0.00000000
		-6.27260515	-0.00065315	0.00040026	0.00065649	-0.00000112	0.00000000
		-6.27260548	-0.00065302	0.00040018	0.00065635	-0.00000112	0.00000000
		-6.27260581	-0.00065288	0.00040009	0.00065621	-0.00000112	0.00000000
		-6.27260614	-0.00065274	0.00040001	0.00065607	-0.00000112	0.00000000
		-6.27260647	-0.00065260	0.00039992	0.00065593	-0.00000112	0.00000000
		-6.27260680	-0.00065246	0.00039984	0.00065580	-0.00000112	0.00000000
		-6.27260713	-0.00065233	0.00039975	0.00065566	-0.00000112	0.00000000
		-6.27260746	-0.00065219	0.00039967	0.00065552	-0.00000112	0.00000000
		-6.27260779	-0.00065205	0.00039959	0.00065538	-0.00000112	0.00000000
		-6.27260812	-0.00065192	0.00039950	0.00065524	-0.00000112	0.00000000
		-6.27260844	-0.00065178	0.00039942	0.00065510	-0.00000112	0.00000000
		-6.27260877	-0.00065164	0.00039933	0.00065497	-0.00000112	0.00000000
		-6.27260910	-0.00065150	0.00039925	0.00065483	-0.00000112	0.00000000
		-6.27260943	-0.00065137	0.00039917	0.00065469	-0.00000112	0.00000000
		-6.27260976	-0.00065123	0.00039908	0.00065455	-0.00000112	0.00000000
		-6.27261009	-0.00065109	0.00039900	0.00065442	-0.00000112	0.00000000
		-6.27261041	-0.00065096	0.00039891	0.00065428	-0.00000112	0.00000000
		-6.27261074	-0.00065082	0.00039883	0.00065414	-0.00000112	0.00000000
		-6.27261107	-0.00065068	0.00039875	0.00065400	-0.00000112	0.00000000
		-6.27261140	-0.00065055	0.00039866	0.00065387	-0.00000112	0.00000000
		-6.27261172	-0.00065041	0.00039858	0.00065373	-0.00000112	0.00000000
		-6.27261205	-0.00065027	0.00039849	0.00065359	-0.00000112	0.00000000
		-6.27261238	-0.00065014	0.00039841	0.00065345	-0.00000112	0.00000000
		-6.27261271	-0.00065000	0.00039833	0.00065332	-0.00000112	0.00000000
		-6.27261303	-0.00064986	0.00039824	0.00065318	-0.00000112	0.00000000
		-6.27261336	-0.00064973	0.00039816	0.00065304	-0.00000112	0.00000000
		-6.27261369	-0.00064959	0.00039808	0.00065290	-0.00000112	0.00000000
		-6.27261401	-0.00064945	0.00039799	0.00065277	-0.00000112	0.00000000
		-6.27261434	-0.00064932	0.00039791	0.00065263	-0.00000112	0.00000000
		-6.27261466	-0.00064918	0.00039782	0.00065249	-0.00000112	0.00000000
		-6.27261499	-0.00064904	0.00039774	0.00065236	-0.00000112	0.00000000
		-6.27261532	-0.00064891	0.00039766	0.00065222	-0.00000112	0.00000000
		-6.27261564	-0.00064877	0.00039757	0.00065208	-0.00000112	0.00000000
		-6.27261597	-0.00064864	0.00039749	0.00065195	-0.00000112	0.00000000
		-6.27261629	-0.00064850	0.00039741	0.00065181	-0.00000112	0.00000000
		-6.27261662	-0.00064836	0.00039732	0.00065167	-0.00000112	0.00000000
		-6.27261694	-0.00064823	0.00039724	0.00065154	-0.00000112	0.00000000
		-6.27261727	-0.00064809	0.00039716	0.00065140	-0.00000112	0.00000000
		-6.27261759	-0.00064796	0.00039707	0.00065126	-0.00000112	0.00000000
		-6.27261792	-0.00064782	0.00039699	0.00065113	-0.00000112	0.00000000
		-6.27261824	-0.00064769	0.00039691	0.00065099	-0.00000112	0.00000000
		-6.27261857	-0.00064755	0.00039683	0.00065086	-0.00000112	0.00000000
		-6.27261889	-0.00064742	0.00039674	0.00065072	-0.00000112	0.00000000
		-6.27261921	-0.00064728	0.00039666	0.00065058	-0.00000112	0.00000000
		-6.27261954	-0.00064715	0.00039658	0.00065045	-0.00000112	0.00000000
		-6.27261986	-0.00064701	0.00039649	0.00065031	-0.00000112	0.00000000
		-6.27262019	-0.00064687	0.00039641	0.00065018	-0.00000112	0.00000000
		-6.27262051	-0.00064674	0.00039633	0.00065004	-0.00000112	0.00000000
		-6.27262083	-0.00064660	0.00039624	0.00064990	-0.00000112	0.00000000
		-6.27262116	-0.00064647	0.00039616	0.00064977	-0.00000112	0.00000000
		-6.27262148	-0.00064633	0.00039608	0.00064963	-0.00000112	0.00000000
		-6.27262180	-0.00064620	0.00039600	0.00064950	-0.00000112	0.00000000
		-6.27262213	-0.00064606	0.00039591	0.00064936	-0.00000112	0.00000000
		-6.27262245	-0.00064593	0.00039583	0.00064923	-0.00000112	0.00000000
		-6.27262277	-0.00064579	0.00039575	0.00064909	-0.00000112	0.00000000
		-6.27262309	-0.00064566	0.00039567	0.00064895	-0.00000112	0.00000000
		-6.27262342	-0.00064553	0.00039558	0.00064882	-0.00000112	0.00000000
		-6.27262374	-0.00064539	0.00039550	0.00064868	-0.00000112	0.00000000
		-6.27262406	-0.00064526	0.00039542	0.00064855	-0.00000112	0.00000000
		-6.27262438	-0.00064512	0.00039534	0.00064841	-0.00000112	0.00000000
		-6.27262470	-0.00064499	0.00039525	0.00064828	-0.00000112	0.00000000
		-6.27262503	-0.00064485	0.00039517	0.00064814	-0.00000112	0.00000000
		-6.27262535	-0.00064472	0.00039509	0.00064801	-0.00000112	0.00000000
		-6.27262567	-0.00064458	0.00039501	0.00064787	-0.00000112	0.00000000
		-6.27262599	-0.00064445	0.00039492	0.00064774	-0.00000112	0.00000000
		-6.27262631	-0.00064432	0.00039484	0.00064760	-0.00000112	0.00000000
		-6.27262663	-0.00064418	0.00039476	0.00064747	-0.00000112	0.00000000
		-6.27262695	-0.00064405	0.00039468	0.00064733	-0.00000112	0.00000000
		-6.27262727	-0.00064391	0.00039460	0.00064720	-0.00000112	0.00000000
		-6.27262759	-0.00064378	0.00039451	0.00064706	-0.00000112	0.00000000
		-6.27262792	-0.00064365	0.00039443	0.00064693	-0.00000112	0.00000000
		-6.27262824	-0.00064351	0.00039435	0.00064680	-0.00000112	0.00000000
		-6.27262856	-0.00064338	0.00039427	0.00064666	-0.00000112	0.00000000
		-6.27262888	-0.00064325	0.00039418	0.00064653	-0.00000112	0.00000000
		-6.27262920	-0.00064311	0.00039410	0.00064639	-0.00000112	0.00000000
		-6.27262952	-0.00064298	0.00039402	0.00064626	-0.00000112	0.00000000
		-6.27262984	-0.00064284	0.00039394	0.00064612	-0.00000112	0.00000000
		-6.27263015	-0.00064271	0.00039386	0.00064599	-0.00000112	0.00000000
		-6.27263047	-0.00064258	0.00039378	0.00064586	-0.00000112	0.00000000
		-6.27263079	-0.00064244	0.00039369	0.00064572	-0.00000112	0.00000000
		-6.27263111	-0.00064231	0.00039361	0.00064559	-0.00000112	0.00000000
		-6.27263143	-0.00064218	0.00039353	0.00064545	-0.00000112	0.00000000
		-6.27263175	-0.00064204	0.00039345	0.00064532	-0.00000112	0.00000000
		-6.27263207	-0.00064191	0.00039337	0.00064519	-0.00000112	0.00000000
		-6.27263239	-0.00064178	0.00039329	0.00064505	-0.00000112	0.00000000
		-6.27263271	-0.00064165	0.00039320	0.00064492	-0.00000112	0.00000000
		-6.27263302	-0.00064151	0.00039312	0.00064478	-0.00000112	0.00000000
		-6.27263334	-0.00064138	0.00039304	0.00064465	-0.00000112	0.00000000
		-6.27263366	-0.00064125	0.00039296	0.00064452	-0.00000112	0.00000000
		-6.27263398	-0.00064111	0.00039288	0.00064438	-0.00000112	0.00000000
		-6.27263430	-0.00064098	0.00039280	0.00064425	-0.00000112	0.00000000
		-6.27263461	-0.00064085	0.00039271	0.00064412	-0.00000112	0.00000000
		-6.27263493	-0.00064072	0.00039263	0.00064398	-0.00000112	0.00000000
		-6.27263525	-0.00064058	0.00039255	0.00064385	-0.00000112	0.00000000
		-6.27263557	-0.00064045	0.00039247	0.00064372	-0.00000112	0.00000000
		-6.27263588	-0.00064032	0.00039239	0.00064358	-0.00000112	0.00000000
		-6.27263620	-0.00064019	0.00039231	0.00064345	-0.00000112	0.00000000
		-6.27263652	-0.00064005	0.00039223	0.00064332	-0.00000112	0.00000000
		-6.27263683	-0.00063992	0.00039215	0.00064319	-0.00000112	0.00000000
		-6.27263715	-0.00063979	0.00039206	0.00064305	-0.00000112	0.00000000
		-6.27263747	-0.00063966	0.00039198	0.00064292	-0.00000112	0.00000000
		-6.27263778	-0.00063952	0.00039190	0.00064279	-0.00000112	0.00000000
		-6.27263810	-0.00063939	0.00039182	0.00064265	-0.00000112	0.00000000
		-6.27263842	-0.00063926	0.00039174	0.00064252	-0.00000112	0.00000000
		-6.27263873	-0.00063913	0.00039166	0.00064239	-0.00000112	0.00000000
		-6.27263905	-0.00063900	0.00039158	0.00064226	-0.00000112	0.00000000
		-6.27263936	-0.00063886	0.00039150	0.00064212	-0.00000112	0.00000000
		-6.27263968	-0.00063873	0.00039142	0.00064199	-0.00000112	0.00000000
		-6.27263999	-0.00063860	0.00039134	0.00064186	-0.00000112	0.00000000
		-6.27264031	-0.00063847	0.00039126	0.00064173	-0.00000112	0.00000000
		-6.27264063	-0.00063834	0.00039117	0.00064159	-0.00000112	0.00000000
		-6.27264094	-0.00063821	0.00039109	0.00064146	-0.00000112	0.00000000
		-6.27264126	-0.00063807	0.00039101	0.00064133	-0.00000112	0.00000000
		-6.27264157	-0.00063794	0.00039093	0.00064120	-0.00000112	0.00000000
		-6.27264188	-0.00063781	0.00039085	0.00064106	-0.00000112	0.00000000
		-6.27264220	-0.00063768	0.00039077	0.00064093	-0.00000112	0.00000000
		-6.27264251	-0.00063755	0.00039069	0.00064080	-0.00000113	0.00000000
		-6.27264283	-0.00063742	0.00039061	0.00064067	-0.00000113	0.00000000
		-6.27264314	-0.00063729	0.00039053	0.00064054	-0.00000113	0.00000000
		-6.27264346	-0.00063716	0.00039045	0.00064040	-0.00000113	0.00000000
		-6.27264377	-0.00063702	0.00039037	0.00064027	-0.00000113	0.00000000
		-6.27264408	-0.00063689	0.00039029	0.00064014	-0.00000113	0.00000000
		-6.27264440	-0.00063676	0.00039021	0.00064001	-0.00000113	0.00000000
		-6.27264471	-0.00063663	0.00039013	0.00063988	-0.00000113	0.00000000
		-6.27264502	-0.00063650	0.00039005	0.00063975	-0.00000113	0.00000000
		-6.27264534	-0.00063637	0.00038997	0.00063961	-0.00000113	0.00000000
		-6.27264565	-0.00063624	0.00038989	0.00063948	-0.00000113	0.00000000
		-6.27264596	-0.00063611	0.00038981	0.00063935	-0.00000113	0.00000000
		-6.27264628	-0.00063598	0.00038973	0.00063922	-0.00000113	0.00000000
		-6.27264659	-0.00063585	0.00038965	0.00063909	-0.00000113	0.00000000
		-6.27264690	-0.00063572	0.00038957	0.00063896	-0.00000113	0.00000000
		-6.27264721	-0.00063559	0.00038949	0.00063883	-0.00000113	0.00000000
		-6.27264753	-0.00063545	0.00038941	0.00063870	-0.00000113	0.00000000
		-6.27264784	-0.00063532	0.00038933	0.00063856	-0.00000113	0.00000000
		-6.27264815	-0.00063519	0.00038925	0.00063843	-0.00000113	0.00000000
		-6.27264846	-0.00063506	0.00038917	0.00063830	-0.00000113	0.00000000
		-6.27264877	-0.00063493	0.00038909	0.00063817	-0.00000113	0.00000000
		-6.27264909	-0.00063480	0.00038901	0.00063804	-0.00000113	0.00000000
		-6.27264940	-0.00063467	0.00038893	0.00063791	-0.00000113	0.00000000
		-6.27264971	-0.00063454	0.00038885	0.00063778	-0.00000113	0.00000000
		-6.27265002	-0.00063441	0.00038877	0.00063765	-0.00000113	0.00000000
		-6.27265033	-0.00063428	0.00038869	0.00063752	-0.00000113	0.00000000
		-6.27265064	-0.00063415	0.00038861	0.00063739	-0.00000113	0.00000000
		-6.27265095	-0.00063402	0.00038853	0.00063726	-0.00000113	0.00000000
		-6.27265126	-0.00063389	0.00038845	0.00063713	-0.00000113	0.00000000
		-6.27265157	-0.00063376	0.00038837	0.00063700	-0.00000113	0.00000000
		-6.27265189	-0.00063363	0.00038829	0.00063686	-0.00000113	0.00000000
		-6.27265220	-0.00063350	0.00038821	0.00063673	-0.00000113	0.00000000
		-6.27265251	-0.00063337	0.00038813	0.00063660	-0.00000113	0.00000000
		-6.27265282	-0.00063324	0.00038805	0.00063647	-0.00000113	0.00000000
		-6.27265313	-0.00063312	0.00038797	0.00063634	-0.00000113	0.00000000
		-6.27265344	-0.00063299	0.00038789	0.00063621	-0.00000113	0.00000000
		-6.27265375	-0.00063286	0.00038781	0.00063608	-0.00000113	0.00000000
		-6.27265406	-0.00063273	0.00038773	0.00063595	-0.00000113	0.00000000
		-6.27265436	-0.00063260	0.00038765	0.00063582	-0.00000113	0.00000000
		-6.27265467	-0.00063247	0.00038758	0.00063569	-0.00000113	0.00000000
		-6.27265498	-0.00063234	0.00038750	0.00063556	-0.00000113	0.00000000
		-6.27265529	-0.00063221	0.00038742	0.00063543	-0.00000113	0.00000000
		-6.27265560	-0.00063208	0.00038734	0.00063530	-0.00000113	0.00000000
		-6.27265591	-0.00063195	0.00038726	0.00063517	-0.00000113	0.00000000
		-6.27265622	-0.00063182	0.00038718	0.00063504	-0.00000113	0.00000000
		-6.27265653	-0.00063169	0.00038710	0.00063492	-0.00000113	0.00000000
		-6.27265684	-0.00063157	0.00038702	0.00063479	-0.00000113	0.00000000
		-6.27265714	-0.00063144	0.00038694	0.00063466	-0.00000113	0.00000000
		-6.27265745	-0.00063131	0.00038686	0.00063453	-0.00000113	0.00000000
		-6.27265776	-0.00063118	0.00038678	0.00063440	-0.00000113	0.00000000
		-6.27265807	-0.00063105	0.00038671	0.00063427	-0.00000113	0.00000000
		-6.27265838	-0.00063092	0.00038663	0.00063414	-0.00000113	0.00000000
		-6.27265868	-0.00063079	0.00038655	0.00063401	-0.00000113	0.00000000
		-6.27265899	-0.00063066	0.00038647	0.00063388	-0.00000113	0.00000000
		-6.27265930	-0.00063054	0.00038639	0.00063375	-0.00000113	0.00000000
		-6.27265961	-0.00063041	0.00038631	0.00063362	-0.00000113	0.00000000
		-6.27265991	-0.00063028	0.00038623	0.00063349	-0.00000113	0.00000000
		-6.27266022	-0.00063015	0.00038615	0.00063336	-0.00000113	0.00000000
		-6.27266053	-0.00063002	0.00038608	0.00063323	-0.00000113	0.00000000
		-6.27266083	-0.00062989	0.00038600	0.00063311	-0.00000113	0.00000000
		-6.27266114	-0.00062977	0.00038592	0.00063298	-0.00000113	0.00000000
		-6.27266145	-0.00062964	0.00038584	0.00063285	-0.00000113	0.00000000
		-6.27266175	-0.00062951	0.00038576	0.00063272	-0.00000113	0.00000000
		-6.27266206	-0.00062938	0.00038568	0.00063259	-0.00000113	0.00000000
		-6.27266237	-0.00062925	0.00038560	0.00063246	-0.00000113	0.00000000
		-6.27266267	-0.00062913	0.00038553	0.00063233	-0.00000113	0.00000000
		-6.27266298	-0.00062900	0.00038545	0.00063221	-0.00000113	0.00000000
		-6.27266329	-0.00062887	0.00038537	0.00063208	-0.00000113	0.00000000
		-6.27266359	-0.00062874	0.00038529	0.00063195	-0.00000113	0.00000000
		-6.27266390	-0.00062862	0.00038521	0.00063182	-0.00000113	0.00000000
		-6.27266420	-0.00062849	0.00038513	0.00063169	-0.00000113	0.00000000
		-6.27266451	-0.00062836	0.00038506	0.00063156	-0.00000113	0.00000000
		-6.27266481	-0.00062823	0.00038498	0.00063144	-0.00000113	0.00000000
		-6.27266512	-0.00062811	0.00038490	0.00063131	-0.00000113	0.00000000
		-6.27266542	-0.00062798	0.00038482	0.00063118	-0.00000113	0.00000000
		-6.27266573	-0.00062785	0.00038474	0.00063105	-0.00000113	0.00000000
		-6.27266603	-0.00062772	0.00038467	0.00063092	-0.00000113	0.00000000
		-6.27266634	-0.00062760	0.00038459	0.00063080	-0.00000113	0.00000000
		-6.27266664	-0.00062747	0.00038451	0.00063067	-0.00000113	0.00000000
		-6.27266695	-0.00062734	0.00038443	0.00063054	-0.00000113	0.00000000
		-6.27266725	-0.00062721	0.00038435	0.00063041	-0.00000113	0.00000000
		-6.27266755	-0.00062709	0.00038428	0.00063028	-0.00000113	0.00000000
		-6.27266786	-0.00062696	0.00038420	0.00063016	-0.00000113	0.00000000
		-6.27266816	-0.00062683	0.00038412	0.00063003	-0.00000113	0.00000000
		-6.27266847	-0.00062671	0.00038404	0.00062990	-0.00000113	0.00000000
		-6.27266877	-0.00062658	0.00038396	0.00062977	-0.00000113	0.00000000
		-6.27266907	-0.00062645	0.00038389	0.00062965	-0.00000113	0.00000000
		-6.27266938	-0.00062633	0.00038381	0.00062952	-0.00000113	0.00000000
		-6.27266968	-0.00062620	0.00038373	0.00062939	-0.00000113	0.00000000
		-6.27266998	-0.00062607	0.00038365	0.00062926	-0.00000113	0.00000000
		-6.27267028	-0.00062595	0.00038358	0.00062914	-0.00000113	0.00000000
		-6.27267059	-0.00062582	0.00038350	0.00062901	-0.00000113	0.00000000
		-6.27267089	-0.00062569	0.00038342	0.00062888	-0.00000113	0.00000000
		-6.27267119	-0.00062557	0.00038334	0.00062876	-0.00000113	0.00000000
		-6.27267150	-0.00062544	0.00038327	0.00062863	-0.00000113	0.00000000
		-6.27267180	-0.00062531	0.00038319	0.00062850	-0.00000113	0.00000000
		-6.27267210	-0.00062519	0.00038311	0.00062837	-0.00000113	0.00000000
		-6.27267240	-0.00062506	0.00038303	0.00062825	-0.00000113	0.00000000
		-6.27267270	-0.00062494	0.00038296	0.00062812	-0.00000113	0.00000000
		-6.27267301	-0.00062481	0.00038288	0.00062799	-0.00000113	0.00000000
		-6.27267331	-0.00062468	0.00038280	0.00062787	-0.00000113	0.00000000
		-6.27267361	-0.00062456	0.00038272	0.00062774	-0.00000113	0.00000000
		-6.27267391	-0.00062443	0.00038265	0.00062761	-0.00000113	0.00000000
		-6.27267421	-0.00062431	0.00038257	0.00062749	-0.00000113	0.00000000
		-6.27267451	-0.00062418	0.00038249	0.00062736	-0.00000113	0.00000000
		-6.27267481	-0.00062405	0.00038241	0.00062723	-0.00000113	0.00000000
		-6.27267512	-0.00062393	0.00038234	0.00062711	-0.00000113	0.00000000
		-6.27267542	-0.00062380	0.00038226	0.00062698	-0.00000113	0.00000000
		-6.27267572	-0.00062368	0.00038218	0.00062685	-0.00000113	0.00000000
		-6.27267602	-0.00062355	0.00038211	0.00062673	-0.00000113	0.00000000
		-6.27267632	-0.00062342	0.00038203	0.00062660	-0.00000113	0.00000000
		-6.27267662	-0.00062330	0.00038195	0.00062648	-0.00000113	0.00000000
		-6.27267692	-0.00062317	0.00038188	0.00062635	-0.00000113	0.00000000
		-6.27267722	-0.00062305	0.00038180	0.00062622	-0.00000113	0.00000000
		-6.27267752	-0.00062292	0.00038172	0.00062610	-0.00000113	0.00000000
		-6.27267782	-0.00062280	0.00038164	0.00062597	-0.00000113	0.00000000
		-6.27267812	-0.00062267	0.00038157	0.00062585	-0.00000113	0.00000000
		-6.27267842	-0.00062255	0.00038149	0.00062572	-0.00000113	0.00000000
		-6.27267872	-0.00062242	0.00038141	0.00062559	-0.00000113	0.00000000
		-6.27267902	-0.00062230	0.00038134	0.00062547	-0.00000113	0.00000000
		-6.27267932	-0.00062217	0.00038126	0.00062534	-0.00000113	0.00000000
		-6.27267962	-0.00062205	0.00038118	0.00062522	-0.00000113	0.00000000
		-6.27267992	-0.00062192	0.00038111	0.00062509	-0.00000113	0.00000000
		-6.27268021	-0.00062180	0.00038103	0.00062497	-0.00000113	0.00000000
		-6.27268051	-0.00062167	0.00038095	0.00062484	-0.00000113	0.00000000
		-6.27268081	-0.00062155	0.00038088	0.00062471	-0.00000113	0.00000000
		-6.27268111	-0.00062142	0.00038080	0.00062459	-0.00000113	0.00000000
		-6.27268141	-0.00062130	0.00038072	0.00062446	-0.00000113	0.00000000
		-6.27268171	-0.00062117	0.00038065	0.00062434	-0.00000113	0.00000000
		-6.27268201	-0.00062105	0.00038057	0.00062421	-0.00000113	0.00000000
		-6.27268230	-0.00062092	0.00038050	0.00062409	-0.00000113	0.00000000
		-6.27268260	-0.00062080	0.00038042	0.00062396	-0.00000113	0.00000000
		-6.27268290	-0.00062067	0.00038034	0.00062384	-0.00000113	0.00000000
		-6.27268320	-0.00062055	0.00038027	0.00062371	-0.00000113	0.00000000
		-6.27268350	-0.00062043	0.00038019	0.00062359	-0.00000113	0.00000000
		-6.27268379	-0.00062030	0.00038011	0.00062346	-0.00000113	0.00000000
		-6.27268409	-0.00062018	0.00038004	0.00062334	-0.00000113	0.00000000
		-6.27268439	-0.00062005	0.00037996	0.00062321	-0.00000113	0.00000000
		-6.27268468	-0.00061993	0.00037989	0.00062309	-0.00000113	0.00000000
		-6.27268498	-0.00061980	0.00037981	0.00062296	-0.00000113	0.00000000
		-6.27268528	-0.00061968	0.00037973	0.00062284	-0.00000113	0.00000000
		-6.27268558	-0.00061956	0.00037966	0.00062271	-0.00000113	0.00000000
		-6.27268587	-0.00061943	0.00037958	0.00062259	-0.00000113	0.00000000
		-6.27268617	-0.00061931	0.00037951	0.00062246	-0.00000113	0.00000000
		-6.27268647	-0.00061919	0.00037943	0.00062234	-0.00000113	0.00000000
		-6.27268676	-0.00061906	0.00037935	0.00062222	-0.00000113	0.00000000
		-6.27268706	-0.00061894	0.00037928	0.00062209	-0.00000113	0.00000000
		-6.27268735	-0.00061881	0.00037920	0.00062197	-0.00000113	0.00000000
		-6.27268765	-0.00061869	0.00037913	0.00062184	-0.00000113	0.00000000
		-6.27268795	-0.00061857	0.00037905	0.00062172	-0.00000113	0.00000000
		-6.27268824	-0.00061844	0.00037897	0.00062159	-0.00000113	0.00000000
		-6.27268854	-0.00061832	0.00037890	0.00062147	-0.00000113	0.00000000
		-6.27268883	-0.00061820	0.00037882	0.00062135	-0.00000113	0.00000000
		-6.27268913	-0.00061807	0.00037875	0.00062122	-0.00000113	0.00000000
		-6.27268942	-0.00061795	0.00037867	0.00062110	-0.00000113	0.00000000
		-6.27268972	-0.00061783	0.00037860	0.00062097	-0.00000113	0.00000000
		-6.27269001	-0.00061770	0.00037852	0.00062085	-0.00000113	0.00000000
		-6.27269031	-0.00061758	0.00037844	0.00062073	-0.00000113	0.00000000
		-6.27269060	-0.00061746	0.00037837	0.00062060	-0.00000113	0.00000000
		-6.27269090	-0.00061733	0.00037829	0.00062048	-0.00000113	0.00000000
		-6.27269119	-0.00061721	0.00037822	0.00062035	-0.00000113	0.00000000
		-6.27269149	-0.00061709	0.00037814	0.00062023	-0.00000113	0.00000000
		-6.27269178	-0.00061696	0.00037807	0.00062011	-0.00000113	0.00000000
		-6.27269208	-0.00061684	0.00037799	0.00061998	-0.00000113	0.00000000
		-6.27269237	-0.00061672	0.00037792	0.00061986	-0.00000113	0.00000000
		-6.27269266	-0.00061660	0.00037784	0.00061974	-0.00000113	0.00000000
		-6.27269296	-0.00061647	0.00037777	0.00061961	-0.00000113	0.00000000
		-6.27269325	-0.00061635	0.00037769	0.00061949	-0.00000113	0.00000000
		-6.27269354	-0.00061623	0.00037762	0.00061937	-0.00000113	0.00000000
		-6.27269384	-0.00061610	0.00037754	0.00061924	-0.00000113	0.00000000
		-6.27269413	-0.00061598	0.00037746	0.00061912	-0.00000113	0.00000000
		-6.27269442	-0.00061586	0.00037739	0.00061900	-0.00000113	0.00000000
		-6.27269472	-0.00061574	0.00037731	0.00061887	-0.00000113	0.00000000
		-6.27269501	-0.00061561	0.00037724	0.00061875	-0.00000113	0.00000000
		-6.27269530	-0.00061549	0.00037716	0.00061863	-0.00000113	0.00000000
		-6.27269560	-0.00061537	0.00037709	0.00061850	-0.00000113	0.00000000
		-6.27269589	-0.00061525	0.00037701	0.00061838	-0.00000113	0.00000000
		-6.27269618	-0.00061512	0.00037694	0.00061826	-0.00000113	0.00000000
		-6.27269647	-0.00061500	0.00037686	0.00061814	-0.00000113	0.00000000
		-6.27269677	-0.00061488	0.00037679	0.00061801	-0.00000113	0.00000000
		-6.27269706	-0.00061476	0.00037671	0.00061789	-0.00000113	0.00000000
		-6.27269735	-0.00061464	0.00037664	0.00061777	-0.00000113	0.00000000
		-6.27269764	-0.00061451	0.00037656	0.00061764	-0.00000113	0.00000000
		-6.27269793	-0.00061439	0.00037649	0.00061752	-0.00000113	0.00000000
		-6.27269823	-0.00061427	0.00037642	0.00061740	-0.00000113	0.00000000
		-6.27269852	-0.00061415	0.00037634	0.00061728	-0.00000113	0.00000000
		-6.27269881	-0.00061403	0.00037627	0.00061715	-0.00000113	0.00000000
		-6.27269910	-0.00061390	0.00037619	0.00061703	-0.00000113	0.00000000
		-6.27269939	-0.00061378	0.00037612	0.00061691	-0.00000113	0.00000000
		-6.27269968	-0.00061366	0.00037604	0.00061679	-0.00000113	0.00000000
		-6.27269997	-0.00061354	0.00037597	0.00061667	-0.00000113	0.00000000
		-6.27270027	-0.00061342	0.00037589	0.00061654	-0.00000113	0.00000000
		-6.27270056	-0.00061330	0.00037582	0.00061642	-0.00000113	0.00000000
		-6.27270085	-0.00061318	0.00037574	0.00061630	-0.00000113	0.00000000
		-6.27270114	-0.00061305	0.00037567	0.00061618	-0.00000113	0.00000000
		-6.27270143	-0.00061293	0.00037559	0.00061605	-0.00000113	0.00000000
		-6.27270172	-0.00061281	0.00037552	0.00061593	-0.00000113	0.00000000
		-6.27270201	-0.00061269	0.00037545	0.00061581	-0.00000113	0.00000000
		-6.27270230	-0.00061257	0.00037537	0.00061569	-0.00000113	0.00000000
		-6.27270259	-0.00061245	0.00037530	0.00061557	-0.00000113	0.00000000
		-6.27270288	-0.00061233	0.00037522	0.00061545	-0.00000113	0.00000000
		-6.27270317	-0.00061221	0.00037515	0.00061532	-0.00000113	0.00000000
		-6.27270346	-0.00061208	0.00037507	0.00061520	-0.00000113	0.00000000
		-6.27270375	-0.00061196	0.00037500	0.00061508	-0.00000113	0.00000000
		-6.27270404	-0.00061184	0.00037493	0.00061496	-0.00000113	0.00000000
		-6.27270433	-0.00061172	0.00037485	0.00061484	-0.00000113	0.00000000
		-6.27270462	-0.00061160	0.00037478	0.00061472	-0.00000113	0.00000000
		-6.27270490	-0.00061148	0.00037470	0.00061459	-0.00000113	0.00000000
		-6.27270519	-0.00061136	0.00037463	0.00061447	-0.00000113	0.00000000
		-6.27270548	-0.00061124	0.00037456	0.00061435	-0.00000113	0.00000000
		-6.27270577	-0.00061112	0.00037448	0.00061423	-0.00000113	0.00000000
		-6.27270606	-0.00061100	0.00037441	0.00061411	-0.00000113	0.00000000
		-6.27270635	-0.00061088	0.00037433	0.00061399	-0.00000113	0.00000000
		-6.27270664	-0.00061076	0.00037426	0.00061387	-0.00000113	0.00000000
		-6.27270692	-0.00061064	0.00037419	0.00061375	-0.00000113	0.00000000
		-6.27270721	-0.00061051	0.00037411	0.00061362	-0.00000113	0.00000000
		-6.27270750	-0.00061039	0.00037404	0.00061350	-0.00000113	0.00000000
		-6.27270779	-0.00061027	0.00037396	0.00061338	-0.00000113	0.00000000
		-6.27270808	-0.00061015	0.00037389	0.00061326	-0.00000113	0.00000000
		-6.27270836	-0.00061003	0.00037382	0.00061314	-0.00000113	0.00000000
		-6.27270865	-0.00060991	0.00037374	0.00061302	-0.00000113	0.00000000
		-6.27270894	-0.00060979	0.00037367	0.00061290	-0.00000113	0.00000000
		-6.27270923	-0.00060967	0.00037360	0.00061278	-0.00000113	0.00000000
		-6.27270951	-0.00060955	0.00037352	0.00061266	-0.00000113	0.00000000
		-6.27270980	-0.00060943	0.00037345	0.00061254	-0.00000113	0.00000000
		-6.27271009	-0.00060931	0.00037338	0.00061242	-0.00000113	0.00000000
		-6.27271038	-0.00060919	0.00037330	0.00061230	-0.00000113	0.00000000
		-6.27271066	-0.00060907	0.00037323	0.00061218	-0.00000113	0.00000000
		-6.27271095	-0.00060895	0.00037315	0.00061205	-0.00000113	0.00000000
		-6.27271124	-0.00060883	0.00037308	0.00061193	-0.00000113	0.00000000
		-6.27271152	-0.00060871	0.00037301	0.00061181	-0.00000113	0.00000000
		-6.27271181	-0.00060859	0.00037293	0.00061169	-0.00000113	0.00000000
		-6.27271209	-0.00060847	0.00037286	0.00061157	-0.00000113	0.00000000
		-6.27271238	-0.00060836	0.00037279	0.00061145	-0.00000113	0.00000000
		-6.27271267	-0.00060824	0.00037271	0.00061133	-0.00000113	0.00000000
		-6.27271295	-0.00060812	0.00037264	0.00061121	-0.00000113	0.00000000
		-6.27271324	-0.00060800	0.00037257	0.00061109	-0.00000113	0.00000000
		-6.27271352	-0.00060788	0.00037249	0.00061097	-0.00000113	0.00000000
		-6.27271381	-0.00060776	0.00037242	0.00061085	-0.00000113	0.00000000
		-6.27271410	-0.00060764	0.00037235	0.00061073	-0.00000113	0.00000000
		-6.27271438	-0.00060752	0.00037228	0.00061061	-0.00000113	0.00000000
		-6.27271467	-0.00060740	0.00037220	0.00061049	-0.00000113	0.00000000
		-6.27271495	-0.00060728	0.00037213	0.00061037	-0.00000113	0.00000000
		-6.27271524	-0.00060716	0.00037206	0.00061025	-0.00000113	0.00000000
		-6.27271552	-0.00060704	0.00037198	0.00061013	-0.00000113	0.00000000
		-6.27271581	-0.00060692	0.00037191	0.00061001	-0.00000113	0.00000000
		-6.27271609	-0.00060680	0.00037184	0.00060989	-0.00000113	0.00000000
		-6.27271637	-0.00060669	0.00037176	0.00060978	-0.00000113	0.00000000
		-6.27271666	-0.00060657	0.00037169	0.00060966	-0.00000113	0.00000000
		-6.27271694	-0.00060645	0.00037162	0.00060954	-0.00000113	0.00000000
		-6.27271723	-0.00060633	0.00037155	0.00060942	-0.00000113	0.00000000
		-6.27271751	-0.00060621	0.00037147	0.00060930	-0.00000113	0.00000000
		-6.27271780	-0.00060609	0.00037140	0.00060918	-0.00000113	0.00000000
		-6.27271808	-0.00060597	0.00037133	0.00060906	-0.00000113	0.00000000
		-6.27271836	-0.00060585	0.00037125	0.00060894	-0.00000113	0.00000000
		-6.27271865	-0.00060574	0.00037118	0.00060882	-0.00000113	0.00000000
		-6.27271893	-0.00060562	0.00037111	0.00060870	-0.00000113	0.00000000
		-6.27271921	-0.00060550	0.00037104	0.00060858	-0.00000113	0.00000000
		-6.27271950	-0.00060538	0.00037096	0.00060846	-0.00000113	0.00000000
		-6.27271978	-0.00060526	0.00037089	0.00060834	-0.00000113	0.00000000
		-6.27272006	-0.00060514	0.00037082	0.00060823	-0.00000113	0.00000000
		-6.27272035	-0.00060503	0.00037075	0.00060811	-0.00000113	0.00000000
		-6.27272063	-0.00060491	0.00037067	0.00060799	-0.00000113	0.00000000
		-6.27272091	-0.00060479	0.00037060	0.00060787	-0.00000113	0.00000000
		-6.27272120	-0.00060467	0.00037053	0.00060775	-0.00000113	0.00000000
		-6.27272148	-0.00060455	0.00037046	0.00060763	-0.00000113	0.00000000
		-6.27272176	-0.00060444	0.00037038	0.00060751	-0.00000113	0.00000000
		-6.27272204	-0.00060432	0.00037031	0.00060739	-0.00000113	0.00000000
		-6.27272232	-0.00060420	0.00037024	0.00060728	-0.00000113	0.00000000
		-6.27272261	-0.00060408	0.00037017	0.00060716	-0.00000113	0.00000000
		-6.27272289	-0.00060396	0.00037009	0.00060704	-0.00000113	0.00000000
		-6.27272317	-0.00060385	0.00037002	0.00060692	-0.00000113	0.00000000
		-6.27272345	-0.00060373	0.00036995	0.00060680	-0.00000113	0.00000000
		-6.27272373	-0.00060361	0.00036988	0.00060668	-0.00000113	0.00000000
		-6.27272402	-0.00060349	0.00036981	0.00060657	-0.00000113	0.00000000
		-6.27272430	-0.00060337	0.00036973	0.00060645	-0.00000113	0.00000000
		-6.27272458	-0.00060326	0.00036966	0.00060633	-0.00000113	0.00000000
		-6.27272486	-0.00060314	0.00036959	0.00060621	-0.00000113	0.00000000
		-6.27272514	-0.00060302	0.00036952	0.00060609	-0.00000113	0.00000000
		-6.27272542	-0.00060290	0.00036945	0.00060597	-0.00000113	0.00000000
		-6.27272570	-0.00060279	0.00036937	0.00060586	-0.00000113	0.00000000
		-6.27272598	-0.00060267	0.00036930	0.00060574	-0.00000113	0.00000000
		-6.27272626	-0.00060255	0.00036923	0.00060562	-0.00000113	0.00000000
		-6.27272654	-0.00060244	0.00036916	0.00060550	-0.00000113	0.00000000
		-6.27272683	-0.00060232	0.00036909	0.00060538	-0.00000113	0.00000000
		-6.27272711	-0.00060220	0.00036901	0.00060527	-0.00000113	0.00000000
		-6.27272739	-0.00060208	0.00036894	0.00060515	-0.00000113	0.00000000
		-6.27272767	-0.00060197	0.00036887	0.00060503	-0.00000113	0.00000000
		-6.27272795	-0.00060185	0.00036880	0.00060491	-0.00000113	0.00000000
		-6.27272823	-0.00060173	0.00036873	0.00060480	-0.00000113	0.00000000
		-6.27272851	-0.00060162	0.00036865	0.00060468	-0.00000113	0.00000000
		-6.27272879	-0.00060150	0.00036858	0.00060456	-0.00000113	0.00000000
		-6.27272907	-0.00060138	0.00036851	0.00060444	-0.00000113	0.00000000
		-6.27272934	-0.00060127	0.00036844	0.00060433	-0.00000113	0.00000000
		-6.27272962	-0.00060115	0.00036837	0.00060421	-0.00000113	0.00000000
		-6.27272990	-0.00060103	0.00036830	0.00060409	-0.00000113	0.00000000
		-6.27273018	-0.00060092	0.00036823	0.00060397	-0.00000113	0.00000000
		-6.27273046	-0.00060080	0.00036815	0.00060386	-0.00000113	0.00000000
		-6.27273074	-0.00060068	0.00036808	0.00060374	-0.00000113	0.00000000
		-6.27273102	-0.00060057	0.00036801	0.00060362	-0.00000113	0.00000000
		-6.27273130	-0.00060045	0.00036794	0.00060351	-0.00000113	0.00000000
		-6.27273158	-0.00060033	0.00036787	0.00060339	-0.00000113	0.00000000
		-6.27273186	-0.00060022	0.00036780	0.00060327	-0.00000113	0.00000000
		-6.27273213	-0.00060010	0.00036773	0.00060315	-0.00000113	0.00000000
		-6.27273241	-0.00059998	0.00036765	0.00060304	-0.00000113	0.00000000
		-6.27273269	-0.00059987	0.00036758	0.00060292	-0.00000113	0.00000000
		-6.27273297	-0.00059975	0.00036751	0.00060280	-0.00000113	0.00000000
		-6.27273325	-0.00059963	0.00036744	0.00060269	-0.00000113	0.00000000
		-6.27273352	-0.00059952	0.00036737	0.00060257	-0.00000113	0.00000000
		-6.27273380	-0.00059940	0.00036730	0.00060245	-0.00000113	0.00000000
		-6.27273408	-0.00059929	0.00036723	0.00060234	-0.00000113	0.00000000
		-6.27273436	-0.00059917	0.00036716	0.00060222	-0.00000113	0.00000000
		-6.27273463	-0.00059905	0.00036708	0.00060210	-0.00000113	0.00000000
		-6.27273491	-0.00059894	0.00036701	0.00060199	-0.00000113	0.00000000
		-6.27273519	-0.00059882	0.00036694	0.00060187	-0.00000113	0.00000000
		-6.27273547	-0.00059871	0.00036687	0.00060175	-0.00000113	0.00000000
		-6.27273574	-0.00059859	0.00036680	0.00060164	-0.00000113	0.00000000
		-6.27273602	-0.00059848	0.00036673	0.00060152	-0.00000113	0.00000000
		-6.27273630	-0.00059836	0.00036666	0.00060141	-0.00000113	0.00000000
		-6.27273657	-0.00059824	0.00036659	0.00060129	-0.00000113	0.00000000
		-6.27273685	-0.00059813	0.00036652	0.00060117	-0.00000113	0.00000000
		-6.27273713	-0.00059801	0.00036645	0.00060106	-0.00000113	0.00000000
		-6.27273740	-0.00059790	0.00036637	0.00060094	-0.00000113	0.00000000
		-6.27273768	-0.00059778	0.00036630	0.00060082	-0.00000113	0.00000000
		-6.27273796	-0.00059767	0.00036623	0.00060071	-0.00000113	0.00000000
		-6.27273823	-0.00059755	0.00036616	0.00060059	-0.00000113	0.00000000
		-6.27273851	-0.00059744	0.00036609	0.00060048	-0.00000113	0.00000000
		-6.27273878	-0.00059732	0.00036602	0.00060036	-0.00000113	0.00000000
		-6.27273906	-0.00059721	0.00036595	0.00060025	-0.00000113	0.00000000
		-6.27273933	-0.00059709	0.00036588	0.00060013	-0.00000113	0.00000000
		-6.27273961	-0.00059698	0.00036581	0.00060001	-0.00000113	0.00000000
		-6.27273989	-0.00059686	0.00036574	0.00059990	-0.00000113	0.00000000
		-6.27274016	-0.00059674	0.00036567	0.00059978	-0.00000113	0.00000000
		-6.27274044	-0.00059663	0.00036560	0.00059967	-0.00000113	0.00000000
		-6.27274071	-0.00059651	0.00036553	0.00059955	-0.00000113	0.00000000
		-6.27274099	-0.00059640	0.00036546	0.00059944	-0.00000113	0.00000000
		-6.27274126	-0.00059629	0.00036539	0.00059932	-0.00000113	0.00000000
		-6.27274154	-0.00059617	0.00036532	0.00059920	-0.00000113	0.00000000
		-6.27274181	-0.00059606	0.00036525	0.00059909	-0.00000113	0.00000000
		-6.27274208	-0.00059594	0.00036517	0.00059897	-0.00000113	0.00000000
		-6.27274236	-0.00059583	0.00036510	0.00059886	-0.00000113	0.00000000
		-6.27274263	-0.00059571	0.00036503	0.00059874	-0.00000113	0.00000000
		-6.27274291	-0.00059560	0.00036496	0.00059863	-0.00000113	0.00000000
		-6.27274318	-0.00059548	0.00036489	0.00059851	-0.00000113	0.00000000
		-6.27274346	-0.00059537	0.00036482	0.00059840	-0.00000113	0.00000000
		-6.27274373	-0.00059525	0.00036475	0.00059828	-0.00000113	0.00000000
		-6.27274400	-0.00059514	0.00036468	0.00059817	-0.00000113	0.00000000
		-6.27274428	-0.00059502	0.00036461	0.00059805	-0.00000113	0.00000000
		-6.27274455	-0.00059491	0.00036454	0.00059794	-0.00000113	0.00000000
		-6.27274482	-0.00059480	0.00036447	0.00059782	-0.00000113	0.00000000
		-6.27274510	-0.00059468	0.00036440	0.00059771	-0.00000113	0.00000000
		-6.27274537	-0.00059457	0.00036433	0.00059759	-0.00000113	0.00000000
		-6.27274564	-0.00059445	0.00036426	0.00059748	-0.00000113	0.00000000
		-6.27274592	-0.00059434	0.00036419	0.00059736	-0.00000113	0.00000000
		-6.27274619	-0.00059422	0.00036412	0.00059725	-0.00000113	0.00000000
		-6.27274646	-0.00059411	0.00036405	0.00059713	-0.00000113	0.00000000
		-6.27274674	-0.00059400	0.00036398	0.00059702	-0.00000113	0.00000000
		-6.27274701	-0.00059388	0.00036391	0.00059690	-0.00000113	0.00000000
		-6.27274728	-0.00059377	0.00036384	0.00059679	-0.00000113	0.00000000
		-6.27274755	-0.00059365	0.00036377	0.00059668	-0.00000113	0.00000000
		-6.27274783	-0.00059354	0.00036370	0.00059656	-0.00000113	0.00000000
		-6.27274810	-0.00059343	0.00036363	0.00059645	-0.00000113	0.00000000
		-6.27274837	-0.00059331	0.00036356	0.00059633	-0.00000113	0.00000000
		-6.27274864	-0.00059320	0.00036349	0.00059622	-0.00000113	0.00000000
		-6.27274891	-0.00059309	0.00036342	0.00059610	-0.00000113	0.00000000
		-6.27274919	-0.00059297	0.00036335	0.00059599	-0.00000113	0.00000000
		-6.27274946	-0.00059286	0.00036329	0.00059588	-0.00000113	0.00000000
		-6.27274973	-0.00059275	0.00036322	0.00059576	-0.00000113	0.00000000
		-6.27275000	-0.00059263	0.00036315	0.00059565	-0.00000113	0.00000000
		-6.27275027	-0.00059252	0.00036308	0.00059553	-0.00000113	0.00000000
		-6.27275054	-0.00059241	0.00036301	0.00059542	-0.00000113	0.00000000
		-6.27275081	-0.00059229	0.00036294	0.00059531	-0.00000113	0.00000000
		-6.27275109	-0.00059218	0.00036287	0.00059519	-0.00000113	0.00000000
		-6.27275136	-0.00059207	0.00036280	0.00059508	-0.00000113	0.00000000
		-6.27275163	-0.00059195	0.00036273	0.00059496	-0.00000113	0.00000000
		-6.27275190	-0.00059184	0.00036266	0.00059485	-0.00000113	0.00000000
		-6.27275217	-0.00059173	0.00036259	0.00059474	-0.00000113	0.00000000
		-6.27275244	-0.00059161	0.00036252	0.00059462	-0.00000113	0.00000000
		-6.27275271	-0.00059150	0.00036245	0.00059451	-0.00000113	0.00000000
		-6.27275298	-0.00059139	0.00036238	0.00059440	-0.00000113	0.00000000
		-6.27275325	-0.00059127	0.00036231	0.00059428	-0.00000113	0.00000000
		-6.27275352	-0.00059116	0.00036224	0.00059417	-0.00000113	0.00000000
		-6.27275379	-0.00059105	0.00036217	0.00059406	-0.00000113	0.00000000
		-6.27275406	-0.00059094	0.00036211	0.00059394	-0.00000113	0.00000000
		-6.27275433	-0.00059082	0.00036204	0.00059383	-0.00000113	0.00000000
		-6.27275460	-0.00059071	0.00036197	0.00059372	-0.00000113	0.00000000
		-6.27275487	-0.00059060	0.00036190	0.00059360	-0.00000113	0.00000000
		-6.27275514	-0.00059048	0.00036183	0.00059349	-0.00000113	0.00000000
		-6.27275541	-0.00059037	0.00036176	0.00059338	-0.00000113	0.00000000
		-6.27275568	-0.00059026	0.00036169	0.00059326	-0.00000113	0.00000000
		-6.27275595	-0.00059015	0.00036162	0.00059315	-0.00000113	0.00000000
		-6.27275622	-0.00059003	0.00036155	0.00059304	-0.00000113	0.00000000
		-6.27275649	-0.00058992	0.00036148	0.00059292	-0.00000113	0.00000000
		-6.27275675	-0.00058981	0.00036141	0.00059281	-0.00000113	0.00000000
		-6.27275702	-0.00058970	0.00036135	0.00059270	-0.00000113	0.00000000
		-6.27275729	-0.00058958	0.00036128	0.00059258	-0.00000113	0.00000000
		-6.27275756	-0.00058947	0.00036121	0.00059247	-0.00000113	0.00000000
		-6.27275783	-0.00058936	0.00036114	0.00059236	-0.00000113	0.00000000
		-6.27275810	-0.00058925	0.00036107	0.00059225	-0.00000113	0.00000000
		-6.27275837	-0.00058914	0.00036100	0.00059213	-0.00000113	0.00000000
		-6.27275863	-0.00058902	0.00036093	0.00059202	-0.00000113	0.00000000
		-6.27275890	-0.00058891	0.00036086	0.00059191	-0.00000113	0.00000000
		-6.27275917	-0.00058880	0.00036080	0.00059180	-0.00000113	0.00000000
		-6.27275944	-0.00058869	0.00036073	0.00059168	-0.00000113	0.00000000
		-6.27275971	-0.00058858	0.00036066	0.00059157	-0.00000113	0.00000000
		-6.27275997	-0.00058846	0.00036059	0.00059146	-0.00000113	0.00000000
		-6.27276024	-0.00058835	0.00036052	0.00059135	-0.00000113	0.00000000
		-6.27276051	-0.00058824	0.00036045	0.00059123	-0.00000113	0.00000000
		-6.27276078	-0.00058813	0.00036038	0.00059112	-0.00000113	0.00000000
		-6.27276104	-0.00058802	0.00036032	0.00059101	-0.00000113	0.00000000
		-6.27276131	-0.00058790	0.00036025	0.00059090	-0.00000113	0.00000000
		-6.27276158	-0.00058779	0.00036018	0.00059078	-0.00000113	0.00000000
		-6.27276184	-0.00058768	0.00036011	0.00059067	-0.00000113	0.00000000
		-6.27276211	-0.00058757	0.00036004	0.00059056	-0.00000113	0.00000000
		-6.27276238	-0.00058746	0.00035997	0.00059045	-0.00000113	0.00000000
		-6.27276265	-0.00058735	0.00035991	0.00059034	-0.00000113	0.00000000
		-6.27276291	-0.00058724	0.00035984	0.00059022	-0.00000113	0.00000000
		-6.27276318	-0.00058712	0.00035977	0.00059011	-0.00000113	0.00000000
		-6.27276344	-0.00058701	0.00035970	0.00059000	-0.00000113	0.00000000
		-6.27276371	-0.00058690	0.00035963	0.00058989	-0.00000113	0.00000000
		-6.27276398	-0.00058679	0.00035956	0.00058978	-0.00000113	0.00000000
		-6.27276424	-0.00058668	0.00035950	0.00058966	-0.00000113	0.00000000
		-6.27276451	-0.00058657	0.00035943	0.00058955	-0.00000113	0.00000000
		-6.27276478	-0.00058646	0.00035936	0.00058944	-0.00000113	0.00000000
		-6.27276504	-0.00058635	0.00035929	0.00058933	-0.00000113	0.00000000
		-6.27276531	-0.00058623	0.00035922	0.00058922	-0.00000113	0.00000000
		-6.27276557	-0.00058612	0.00035916	0.00058911	-0.00000113	0.00000000
		-6.27276584	-0.00058601	0.00035909	0.00058899	-0.00000113	0.00000000
		-6.27276610	-0.00058590	0.00035902	0.00058888	-0.00000113	0.00000000
		-6.27276637	-0.00058579	0.00035895	0.00058877	-0.00000113	0.00000000
		-6.27276663	-0.00058568	0.00035888	0.00058866	-0.00000113	0.00000000
		-6.27276690	-0.00058557	0.00035882	0.00058855	-0.00000113	0.00000000
		-6.27276716	-0.00058546	0.00035875	0.00058844	-0.00000113	0.00000000
		-6.27276743	-0.00058535	0.00035868	0.00058833	-0.00000113	0.00000000
		-6.27276769	-0.00058524	0.00035861	0.00058821	-0.00000113	0.00000000
		-6.27276796	-0.00058513	0.00035854	0.00058810	-0.00000113	0.00000000
		-6.27276822	-0.00058502	0.00035848	0.00058799	-0.00000113	0.00000000
		-6.27276849	-0.00058491	0.00035841	0.00058788	-0.00000113	0.00000000
		-6.27276875	-0.00058480	0.00035834	0.00058777	-0.00000113	0.00000000
		-6.27276902	-0.00058468	0.00035827	0.00058766	-0.00000113	0.00000000
		-6.27276928	-0.00058457	0.00035820	0.00058755	-0.00000113	0.00000000
		-6.27276954	-0.00058446	0.00035814	0.00058744	-0.00000113	0.00000000
		-6.27276981	-0.00058435	0.00035807	0.00058733	-0.00000113	0.00000000
		-6.27277007	-0.00058424	0.00035800	0.00058721	-0.00000113	0.00000000
		-6.27277034	-0.00058413	0.00035793	0.00058710	-0.00000113	0.00000000
		-6.27277060	-0.00058402	0.00035787	0.00058699	-0.00000113	0.00000000
		-6.27277086	-0.00058391	0.00035780	0.00058688	-0.00000113	0.00000000
		-6.27277113	-0.00058380	0.00035773	0.00058677	-0.00000113	0.00000000
		-6.27277139	-0.00058369	0.00035766	0.00058666	-0.00000113	0.00000000
		-6.27277165	-0.00058358	0.00035760	0.00058655	-0.00000113	0.00000000
		-6.27277192	-0.00058347	0.00035753	0.00058644	-0.00000113	0.00000000
		-6.27277218	-0.00058336	0.00035746	0.00058633	-0.00000113	0.00000000
		-6.27277244	-0.00058325	0.00035739	0.00058622	-0.00000113	0.00000000
		-6.27277270	-0.00058314	0.00035733	0.00058611	-0.00000113	0.00000000
		-6.27277297	-0.00058303	0.00035726	0.00058600	-0.00000113	0.00000000
		-6.27277323	-0.00058292	0.00035719	0.00058589	-0.00000113	0.00000000
		-6.27277349	-0.00058281	0.00035712	0.00058578	-0.00000113	0.00000000
		-6.27277376	-0.00058270	0.00035706	0.00058567	-0.00000113	0.00000000
		-6.27277402	-0.00058259	0.00035699	0.00058556	-0.00000113	0.00000000
		-6.27277428	-0.00058248	0.00035692	0.00058545	-0.00000113	0.00000000
		-6.27277454	-0.00058237	0.00035686	0.00058534	-0.00000113	0.00000000
		-6.27277480	-0.00058227	0.00035679	0.00058523	-0.00000113	0.00000000
		-6.27277507	-0.00058216	0.00035672	0.00058512	-0.00000113	0.00000000
		-6.27277533	-0.00058205	0.00035665	0.00058501	-0.00000113	0.00000000
		-6.27277559	-0.00058194	0.00035659	0.00058490	-0.00000113	0.00000000
		-6.27277585	-0.00058183	0.00035652	0.00058479	-0.00000113	0.00000000
		-6.27277611	-0.00058172	0.00035645	0.00058468	-0.00000113	0.00000000
		-6.27277637	-0.00058161	0.00035639	0.00058457	-0.00000113	0.00000000
		-6.27277664	-0.00058150	0.00035632	0.00058446	-0.00000113	0.00000000
		-6.27277690	-0.00058139	0.00035625	0.00058435	-0.00000113	0.00000000
		-6.27277716	-0.00058128	0.00035619	0.00058424	-0.00000113	0.00000000
		-6.27277742	-0.00058117	0.00035612	0.00058413	-0.00000113	0.00000000
		-6.27277768	-0.00058106	0.00035605	0.00058402	-0.00000113	0.00000000
		-6.27277794	-0.00058095	0.00035598	0.00058391	-0.00000113	0.00000000
		-6.27277820	-0.00058084	0.00035592	0.00058380	-0.00000113	0.00000000
		-6.27277846	-0.00058074	0.00035585	0.00058369	-0.00000113	0.00000000
		-6.27277872	-0.00058063	0.00035578	0.00058358	-0.00000113	0.00000000
		-6.27277898	-0.00058052	0.00035572	0.00058347	-0.00000113	0.00000000
		-6.27277924	-0.00058041	0.00035565	0.00058336	-0.00000113	0.00000000
		-6.27277950	-0.00058030	0.00035558	0.00058325	-0.00000113	0.00000000
		-6.27277977	-0.00058019	0.00035552	0.00058314	-0.00000113	0.00000000
		-6.27278003	-0.00058008	0.00035545	0.00058303	-0.00000113	0.00000000
		-6.27278029	-0.00057997	0.00035538	0.00058292	-0.00000113	0.00000000
		-6.27278055	-0.00057987	0.00035532	0.00058281	-0.00000113	0.00000000
		-6.27278080	-0.00057976	0.00035525	0.00058271	-0.00000113	0.00000000
		-6.27278106	-0.00057965	0.00035518	0.00058260	-0.00000113	0.00000000
		-6.27278132	-0.00057954	0.00035512	0.00058249	-0.00000113	0.00000000
		-6.27278158	-0.00057943	0.00035505	0.00058238	-0.00000113	0.00000000
		-6.27278184	-0.00057932	0.00035498	0.00058227	-0.00000113	0.00000000
		-6.27278210	-0.00057921	0.00035492	0.00058216	-0.00000113	0.00000000
		-6.27278236	-0.00057911	0.00035485	0.00058205	-0.00000113	0.00000000
		-6.27278262	-0.00057900	0.00035479	0.00058194	-0.00000113	0.00000000
		-6.27278288	-0.00057889	0.00035472	0.00058183	-0.00000113	0.00000000
		-6.27278314	-0.00057878	0.00035465	0.00058172	-0.00000113	0.00000000
		-6.27278340	-0.00057867	0.00035459	0.00058162	-0.00000113	0.00000000
		-6.27278366	-0.00057857	0.00035452	0.00058151	-0.00000113	0.00000000
		-6.27278392	-0.00057846	0.00035445	0.00058140	-0.00000113	0.00000000
		-6.27278417	-0.00057835	0.00035439	0.00058129	-0.00000113	0.00000000
		-6.27278443	-0.00057824	0.00035432	0.00058118	-0.00000113	0.00000000
		-6.27278469	-0.00057813	0.00035425	0.00058107	-0.00000113	0.00000000
		-6.27278495	-0.00057802	0.00035419	0.00058096	-0.00000113	0.00000000
		-6.27278521	-0.00057792	0.00035412	0.00058086	-0.00000113	0.00000000
		-6.27278547	-0.00057781	0.00035406	0.00058075	-0.00000113	0.00000000
		-6.27278572	-0.00057770	0.00035399	0.00058064	-0.00000113	0.00000000
		-6.27278598	-0.00057759	0.00035392	0.00058053	-0.00000113	0.00000000
		-6.27278624	-0.00057749	0.00035386	0.00058042	-0.00000113	0.00000000
		-6.27278650	-0.00057738	0.00035379	0.00058031	-0.00000113	0.00000000
		-6.27278675	-0.00057727	0.00035373	0.00058021	-0.00000113	0.00000000
		-6.27278701	-0.00057716	0.00035366	0.00058010	-0.00000113	0.00000000
		-6.27278727	-0.00057706	0.00035359	0.00057999	-0.00000113	0.00000000
		-6.27278753	-0.00057695	0.00035353	0.00057988	-0.00000113	0.00000000
		-6.27278778	-0.00057684	0.00035346	0.00057977	-0.00000113	0.00000000
		-6.27278804	-0.00057673	0.00035340	0.00057966	-0.00000113	0.00000000
		-6.27278830	-0.00057663	0.00035333	0.00057956	-0.00000113	0.00000000
		-6.27278855	-0.00057652	0.00035326	0.00057945	-0.00000113	0.00000000
		-6.27278881	-0.00057641	0.00035320	0.00057934	-0.00000113	0.00000000
		-6.27278907	-0.00057630	0.00035313	0.00057923	-0.00000113	0.00000000
		-6.27278933	-0.00057620	0.00035307	0.00057913	-0.00000113	0.00000000
		-6.27278958	-0.00057609	0.00035300	0.00057902	-0.00000113	0.00000000
		-6.27278984	-0.00057598	0.00035294	0.00057891	-0.00000113	0.00000000
		-6.27279009	-0.00057587	0.00035287	0.00057880	-0.00000113	0.00000000
		-6.27279035	-0.00057577	0.00035280	0.00057869	-0.00000113	0.00000000
		-6.27279061	-0.00057566	0.00035274	0.00057859	-0.00000113	0.00000000
		-6.27279086	-0.00057555	0.00035267	0.00057848	-0.00000113	0.00000000
		-6.27279112	-0.00057545	0.00035261	0.00057837	-0.00000113	0.00000000
		-6.27279137	-0.00057534	0.00035254	0.00057826	-0.00000113	0.00000000
		-6.27279163	-0.00057523	0.00035248	0.00057816	-0.00000113	0.00000000
		-6.27279189	-0.00057513	0.00035241	0.00057805	-0.00000113	0.00000000
		-6.27279214	-0.00057502	0.00035235	0.00057794	-0.00000113	0.00000000
		-6.27279240	-0.00057491	0.00035228	0.00057783	-0.00000113	0.00000000
		-6.27279265	-0.00057480	0.00035221	0.00057773	-0.00000113	0.00000000
		-6.27279291	-0.00057470	0.00035215	0.00057762	-0.00000113	0.00000000
		-6.27279316	-0.00057459	0.00035208	0.00057751	-0.00000113	0.00000000
		-6.27279342	-0.00057448	0.00035202	0.00057741	-0.00000113	0.00000000
		-6.27279367	-0.00057438	0.00035195	0.00057730	-0.00000113	0.00000000
		-6.27279393	-0.00057427	0.00035189	0.00057719	-0.00000113	0.00000000
		-6.27279418	-0.00057417	0.00035182	0.00057708	-0.00000113	0.00000000
		-6.27279444	-0.00057406	0.00035176	0.00057698	-0.00000113	0.00000000
		-6.27279469	-0.00057395	0.00035169	0.00057687	-0.00000113	0.00000000
		-6.27279495	-0.00057385	0.00035163	0.00057676	-0.00000113	0.00000000
		-6.27279520	-0.00057374	0.00035156	0.00057666	-0.00000113	0.00000000
		-6.27279546	-0.00057363	0.00035150	0.00057655	-0.00000113	0.00000000
		-6.27279571	-0.00057353	0.00035143	0.00057644	-0.00000113	0.00000000
		-6.27279596	-0.00057342	0.00035137	0.00057634	-0.00000113	0.00000000
		-6.27279622	-0.00057331	0.00035130	0.00057623	-0.00000113	0.00000000
		-6.27279647	-0.00057321	0.00035124	0.00057612	-0.00000113	0.00000000
		-6.27279673	-0.00057310	0.00035117	0.00057602	-0.00000113	0.00000000
		-6.27279698	-0.00057300	0.00035111	0.00057591	-0.00000113	0.00000000
		-6.27279723	-0.00057289	0.00035104	0.00057580	-0.00000113	0.00000000
		-6.27279749	-0.00057278	0.00035098	0.00057570	-0.00000113	0.00000000
		-6.27279774	-0.00057268	0.00035091	0.00057559	-0.00000113	0.00000000
		-6.27279799	-0.00057257	0.00035085	0.00057548	-0.00000113	0.00000000
		-6.27279825	-0.00057247	0.00035078	0.00057538	-0.00000113	0.00000000
		-6.27279850	-0.00057236	0.00035072	0.00057527	-0.00000113	0.00000000
		-6.27279875	-0.00057225	0.00035065	0.00057516	-0.00000113	0.00000000
		-6.27279901	-0.00057215	0.00035059	0.00057506	-0.00000113	0.00000000
		-6.27279926	-0.00057204	0.00035052	0.00057495	-0.00000113	0.00000000
		-6.27279951	-0.00057194	0.00035046	0.00057484	-0.00000113	0.00000000
		-6.27279977	-0.00057183	0.00035039	0.00057474	-0.00000113	0.00000000
		-6.27280002	-0.00057173	0.00035033	0.00057463	-0.00000113	0.00000000
		-6.27280027	-0.00057162	0.00035026	0.00057453	-0.00000113	0.00000000
		-6.27280052	-0.00057152	0.00035020	0.00057442	-0.00000113	0.00000000
		-6.27280078	-0.00057141	0.00035013	0.00057431	-0.00000113	0.00000000
		-6.27280103	-0.00057130	0.00035007	0.00057421	-0.00000113	0.00000000
		-6.27280128	-0.00057120	0.00035000	0.00057410	-0.00000113	0.00000000
		-6.27280153	-0.00057109	0.00034994	0.00057400	-0.00000113	0.00000000
		-6.27280178	-0.00057099	0.00034987	0.00057389	-0.00000113	0.00000000
		-6.27280204	-0.00057088	0.00034981	0.00057378	-0.00000113	0.00000000
		-6.27280229	-0.00057078	0.00034974	0.00057368	-0.00000113	0.00000000
		-6.27280254	-0.00057067	0.00034968	0.00057357	-0.00000113	0.00000000
		-6.27280279	-0.00057057	0.00034962	0.00057347	-0.00000113	0.00000000
		-6.27280304	-0.00057046	0.00034955	0.00057336	-0.00000113	0.00000000
		-6.27280329	-0.00057036	0.00034949	0.00057326	-0.00000113	0.00000000
		-6.27280355	-0.00057025	0.00034942	0.00057315	-0.00000113	0.00000000
		-6.27280380	-0.00057015	0.00034936	0.00057304	-0.00000113	0.00000000
		-6.27280405	-0.00057004	0.00034929	0.00057294	-0.00000113	0.00000000
		-6.27280430	-0.00056994	0.00034923	0.00057283	-0.00000113	0.00000000
		-6.27280455	-0.00056983	0.00034916	0.00057273	-0.00000113	0.00000000
		-6.27280480	-0.00056973	0.00034910	0.00057262	-0.00000113	0.00000000
		-6.27280505	-0.00056962	0.00034904	0.00057252	-0.00000113	0.00000000
		-6.27280530	-0.00056952	0.00034897	0.00057241	-0.00000113	0.00000000
		-6.27280555	-0.00056941	0.00034891	0.00057231	-0.00000113	0.00000000
		-6.27280580	-0.00056931	0.00034884	0.00057220	-0.00000113	0.00000000
		-6.27280606	-0.00056920	0.00034878	0.00057210	-0.00000113	0.00000000
		-6.27280631	-0.00056910	0.00034872	0.00057199	-0.00000113	0.00000000
		-6.27280656	-0.00056899	0.00034865	0.00057189	-0.00000113	0.00000000
		-6.27280681	-0.00056889	0.00034859	0.00057178	-0.00000113	0.00000000
		-6.27280706	-0.00056878	0.00034852	0.00057168	-0.00000113	0.00000000
		-6.27280731	-0.00056868	0.00034846	0.00057157	-0.00000113	0.00000000
		-6.27280756	-0.00056858	0.00034839	0.00057147	-0.00000113	0.00000000
		-6.27280781	-0.00056847	0.00034833	0.00057136	-0.00000113	0.00000000
		-6.27280806	-0.00056837	0.00034827	0.00057126	-0.00000113	0.00000000
		-6.27280830	-0.00056826	0.00034820	0.00057115	-0.00000113	0.00000000
		-6.27280855	-0.00056816	0.00034814	0.00057105	-0.00000113	0.00000000
		-6.27280880	-0.00056805	0.00034807	0.00057094	-0.00000113	0.00000000
		-6.27280905	-0.00056795	0.00034801	0.00057084	-0.00000113	0.00000000
		-6.27280930	-0.00056785	0.00034795	0.00057073	-0.00000113	0.00000000
		-6.27280955	-0.00056774	0.00034788	0.00057063	-0.00000113	0.00000000
		-6.27280980	-0.00056764	0.00034782	0.00057052	-0.00000113	0.00000000
		-6.27281005	-0.00056753	0.00034776	0.00057042	-0.00000113	0.00000000
		-6.27281030	-0.00056743	0.00034769	0.00057031	-0.00000113	0.00000000
		-6.27281055	-0.00056733	0.00034763	0.00057021	-0.00000113	0.00000000
		-6.27281080	-0.00056722	0.00034756	0.00057010	-0.00000113	0.00000000
		-6.27281104	-0.00056712	0.00034750	0.00057000	-0.00000113	0.00000000
		-6.27281129	-0.00056701	0.00034744	0.00056990	-0.00000113	0.00000000
		-6.27281154	-0.00056691	0.00034737	0.00056979	-0.00000113	0.00000000
		-6.27281179	-0.00056681	0.00034731	0.00056969	-0.00000113	0.00000000
		-6.27281204	-0.00056670	0.00034725	0.00056958	-0.00000113	0.00000000
		-6.27281229	-0.00056660	0.00034718	0.00056948	-0.00000113	0.00000000
		-6.27281253	-0.00056649	0.00034712	0.00056937	-0.00000113	0.00000000
		-6.27281278	-0.00056639	0.00034706	0.00056927	-0.00000113	0.00000000
		-6.27281303	-0.00056629	0.00034699	0.00056917	-0.00000113	0.00000000
		-6.27281328	-0.00056618	0.00034693	0.00056906	-0.00000113	0.00000000
		-6.27281353	-0.00056608	0.00034686	0.00056896	-0.00000113	0.00000000
		-6.27281377	-0.00056598	0.00034680	0.00056885	-0.00000113	0.00000000
		-6.27281402	-0.00056587	0.00034674	0.00056875	-0.00000113	0.00000000
		-6.27281427	-0.00056577	0.00034667	0.00056865	-0.00000113	0.00000000
		-6.27281452	-0.00056567	0.00034661	0.00056854	-0.00000113	0.00000000
		-6.27281476	-0.00056556	0.00034655	0.00056844	-0.00000113	0.00000000
		-6.27281501	-0.00056546	0.00034648	0.00056833	-0.00000113	0.00000000
		-6.27281526	-0.00056536	0.00034642	0.00056823	-0.00000113	0.00000000
		-6.27281550	-0.00056525	0.00034636	0.00056813	-0.00000113	0.00000000
		-6.27281575	-0.00056515	0.00034629	0.00056802	-0.00000113	0.00000000
		-6.27281600	-0.00056505	0.00034623	0.00056792	-0.00000113	0.00000000
		-6.27281624	-0.00056494	0.00034617	0.00056781	-0.00000113	0.00000000
		-6.27281649	-0.00056484	0.00034610	0.00056771	-0.00000113	0.00000000
		-6.27281674	-0.00056474	0.00034604	0.00056761	-0.00000113	0.00000000
		-6.27281698	-0.00056463	0.00034598	0.00056750	-0.00000113	0.00000000
		-6.27281723	-0.00056453	0.00034592	0.00056740	-0.00000113	0.00000000
		-6.27281748	-0.00056443	0.00034585	0.00056730	-0.00000113	0.00000000
		-6.27281772	-0.00056433	0.00034579	0.00056719	-0.00000113	0.00000000
		-6.27281797	-0.00056422	0.00034573	0.00056709	-0.00000113	0.00000000
		-6.27281822	-0.00056412	0.00034566	0.00056699	-0.00000113	0.00000000
		-6.27281846	-0.00056402	0.00034560	0.00056688	-0.00000113	0.00000000
		-6.27281871	-0.00056391	0.00034554	0.00056678	-0.00000113	0.00000000
		-6.27281895	-0.00056381	0.00034547	0.00056668	-0.00000113	0.00000000
		-6.27281920	-0.00056371	0.00034541	0.00056657	-0.00000113	0.00000000
		-6.27281944	-0.00056361	0.00034535	0.00056647	-0.00000113	0.00000000
		-6.27281969	-0.00056350	0.00034528	0.00056637	-0.00000113	0.00000000
		-6.27281993	-0.00056340	0.00034522	0.00056626	-0.00000113	0.00000000
		-6.27282018	-0.00056330	0.00034516	0.00056616	-0.00000113	0.00000000
		-6.27282043	-0.00056320	0.00034510	0.00056606	-0.00000113	0.00000000
		-6.27282067	-0.00056309	0.00034503	0.00056596	-0.00000113	0.00000000
		-6.27282092	-0.00056299	0.00034497	0.00056585	-0.00000113	0.00000000
		-6.27282116	-0.00056289	0.00034491	0.00056575	-0.00000113	0.00000000
		-6.27282141	-0.00056279	0.00034485	0.00056565	-0.00000113	0.00000000
		-6.27282165	-0.00056268	0.00034478	0.00056554	-0.00000113	0.00000000
		-6.27282189	-0.00056258	0.00034472	0.00056544	-0.00000113	0.00000000
		-6.27282214	-0.00056248	0.00034466	0.00056534	-0.00000113	0.00000000
		-6.27282238	-0.00056238	0.00034459	0.00056524	-0.00000113	0.00000000
		-6.27282263	-0.00056228	0.00034453	0.00056513	-0.00000113	0.00000000
		-6.27282287	-0.00056217	0.00034447	0.00056503	-0.00000113	0.00000000
		-6.27282312	-0.00056207	0.00034441	0.00056493	-0.00000113	0.00000000
		-6.27282336	-0.00056197	0.00034434	0.00056483	-0.00000113	0.00000000
		-6.27282360	-0.00056187	0.00034428	0.00056472	-0.00000113	0.00000000
		-6.27282385	-0.00056177	0.00034422	0.00056462	-0.00000113	0.00000000
		-6.27282409	-0.00056166	0.00034416	0.00056452	-0.00000113	0.00000000
		-6.27282434	-0.00056156	0.00034409	0.00056442	-0.00000113	0.00000000
		-6.27282458	-0.00056146	0.00034403	0.00056431	-0.00000113	0.00000000
		-6.27282482	-0.00056136	0.00034397	0.00056421	-0.00000113	0.00000000
		-6.27282507	-0.00056126	0.00034391	0.00056411	-0.00000113	0.00000000
		-6.27282531	-0.00056115	0.00034384	0.00056401	-0.00000113	0.00000000
		-6.27282555	-0.00056105	0.00034378	0.00056390	-0.00000113	0.00000000
		-6.27282580	-0.00056095	0.00034372	0.00056380	-0.00000113	0.00000000
		-6.27282604	-0.00056085	0.00034366	0.00056370	-0.00000113	0.00000000
		-6.27282628	-0.00056075	0.00034360	0.00056360	-0.00000113	0.00000000
		-6.27282653	-0.00056065	0.00034353	0.00056349	-0.00000113	0.00000000
		-6.27282677	-0.00056055	0.00034347	0.00056339	-0.00000113	0.00000000
		-6.27282701	-0.00056044	0.00034341	0.00056329	-0.00000113	0.00000000
		-6.27282725	-0.00056034	0.00034335	0.00056319	-0.00000113	0.00000000
		-6.27282750	-0.00056024	0.00034328	0.00056309	-0.00000113	0.00000000
		-6.27282774	-0.00056014	0.00034322	0.00056299	-0.00000113	0.00000000
		-6.27282798	-0.00056004	0.00034316	0.00056288	-0.00000113	0.00000000
		-6.27282822	-0.00055994	0.00034310	0.00056278	-0.00000113	0.00000000
		-6.27282847	-0.00055984	0.00034304	0.00056268	-0.00000113	0.00000000
		-6.27282871	-0.00055973	0.00034297	0.00056258	-0.00000113	0.00000000
		-6.27282895	-0.00055963	0.00034291	0.00056248	-0.00000113	0.00000000
		-6.27282919	-0.00055953	0.00034285	0.00056237	-0.00000113	0.00000000
		-6.27282944	-0.00055943	0.00034279	0.00056227	-0.00000113	0.00000000
		-6.27282968	-0.00055933	0.00034273	0.00056217	-0.00000113	0.00000000
		-6.27282992	-0.00055923	0.00034266	0.00056207	-0.00000113	0.00000000
		-6.27283016	-0.00055913	0.00034260	0.00056197	-0.00000113	0.00000000
		-6.27283040	-0.00055903	0.00034254	0.00056187	-0.00000113	0.00000000
		-6.27283064	-0.00055893	0.00034248	0.00056177	-0.00000113	0.00000000
		-6.27283089	-0.00055882	0.00034242	0.00056166	-0.00000113	0.00000000
		-6.27283113	-0.00055872	0.00034235	0.00056156	-0.00000113	0.00000000
		-6.27283137	-0.00055862	0.00034229	0.00056146	-0.00000113	0.00000000
		-6.27283161	-0.00055852	0.00034223	0.00056136	-0.00000113	0.00000000
		-6.27283185	-0.00055842	0.00034217	0.00056126	-0.00000113	0.00000000
		-6.27283209	-0.00055832	0.00034211	0.00056116	-0.00000113	0.00000000
		-6.27283233	-0.00055822	0.00034204	0.00056106	-0.00000113	0.00000000
		-6.27283257	-0.00055812	0.00034198	0.00056095	-0.00000113	0.00000000
		-6.27283281	-0.00055802	0.00034192	0.00056085	-0.00000113	0.00000000
		-6.27283305	-0.00055792	0.00034186	0.00056075	-0.00000113	0.00000000
		-6.27283329	-0.00055782	0.00034180	0.00056065	-0.00000113	0.00000000
		-6.27283353	-0.00055772	0.00034174	0.00056055	-0.00000113	0.00000000
		-6.27283378	-0.00055762	0.00034168	0.00056045	-0.00000113	0.00000000
		-6.27283402	-0.00055752	0.00034161	0.00056035	-0.00000113	0.00000000
		-6.27283426	-0.00055742	0.00034155	0.00056025	-0.00000113	0.00000000
		-6.27283450	-0.00055732	0.00034149	0.00056015	-0.00000113	0.00000000
		-6.27283474	-0.00055722	0.00034143	0.00056005	-0.00000113	0.00000000
		-6.27283498	-0.00055712	0.00034137	0.00055994	-0.00000113	0.00000000
		-6.27283522	-0.00055701	0.00034131	0.00055984	-0.00000113	0.00000000
		-6.27283546	-0.00055691	0.00034124	0.00055974	-0.00000113	0.00000000
		-6.27283570	-0.00055681	0.00034118	0.00055964	-0.00000113	0.00000000
		-6.27283593	-0.00055671	0.00034112	0.00055954	-0.00000113	0.00000000
		-6.27283617	-0.00055661	0.00034106	0.00055944	-0.00000113	0.00000000
		-6.27283641	-0.00055651	0.00034100	0.00055934	-0.00000113	0.00000000
		-6.27283665	-0.00055641	0.00034094	0.00055924	-0.00000113	0.00000000
		-6.27283689	-0.00055631	0.00034088	0.00055914	-0.00000113	0.00000000
		-6.27283713	-0.00055621	0.00034081	0.00055904	-0.00000113	0.00000000
		-6.27283737	-0.00055611	0.00034075	0.00055894	-0.00000113	0.00000000
		-6.27283761	-0.00055601	0.00034069	0.00055884	-0.00000113	0.00000000
		-6.27283785	-0.00055591	0.00034063	0.00055874	-0.00000113	0.00000000
		-6.27283809	-0.00055581	0.00034057	0.00055864	-0.00000113	0.00000000
		-6.27283833	-0.00055571	0.00034051	0.00055854	-0.00000113	0.00000000
		-6.27283856	-0.00055562	0.00034045	0.00055844	-0.00000113	0.00000000
		-6.27283880	-0.00055552	0.00034039	0.00055834	-0.00000113	0.00000000
		-6.27283904	-0.00055542	0.00034033	0.00055824	-0.00000113	0.00000000
		-6.27283928	-0.00055532	0.00034026	0.00055814	-0.00000113	0.00000000
		-6.27283952	-0.00055522	0.00034020	0.00055804	-0.00000113	0.00000000
		-6.27283976	-0.00055512	0.00034014	0.00055794	-0.00000113	0.00000000
		-6.27283999	-0.00055502	0.00034008	0.00055784	-0.00000113	0.00000000
		-6.27284023	-0.00055492	0.00034002	0.00055774	-0.00000113	0.00000000
		-6.27284047	-0.00055482	0.00033996	0.00055764	-0.00000113	0.00000000
		-6.27284071	-0.00055472	0.00033990	0.00055754	-0.00000113	0.00000000
		-6.27284095	-0.00055462	0.00033984	0.00055744	-0.00000113	0.00000000
		-6.27284118	-0.00055452	0.00033978	0.00055734	-0.00000113	0.00000000
		-6.27284142	-0.00055442	0.00033972	0.00055724	-0.00000113	0.00000000
		-6.27284166	-0.00055432	0.00033965	0.00055714	-0.00000113	0.00000000
		-6.27284190	-0.00055422	0.00033959	0.00055704	-0.00000113	0.00000000
		-6.27284213	-0.00055412	0.00033953	0.00055694	-0.00000113	0.00000000
		-6.27284237	-0.00055402	0.00033947	0.00055684	-0.00000113	0.00000000
		-6.27284261	-0.00055393	0.00033941	0.00055674	-0.00000113	0.00000000
		-6.27284285	-0.00055383	0.00033935	0.00055664	-0.00000113	0.00000000
		-6.27284308	-0.00055373	0.00033929	0.00055654	-0.00000113	0.00000000
		-6.27284332	-0.00055363	0.00033923	0.00055644	-0.00000113	0.00000000
		-6.27284356	-0.00055353	0.00033917	0.00055634	-0.00000113	0.00000000
		-6.27284379	-0.00055343	0.00033911	0.00055624	-0.00000113	0.00000000
		-6.27284403	-0.00055333	0.00033905	0.00055614	-0.00000113	0.00000000
		-6.27284427	-0.00055323	0.00033899	0.00055604	-0.00000113	0.00000000
		-6.27284450	-0.00055313	0.00033893	0.00055594	-0.00000113	0.00000000
		-6.27284474	-0.00055303	0.00033887	0.00055584	-0.00000113	0.00000000
		-6.27284498	-0.00055294	0.00033880	0.00055574	-0.00000113	0.00000000
		-6.27284521	-0.00055284	0.00033874	0.00055564	-0.00000113	0.00000000
		-6.27284545	-0.00055274	0.00033868	0.00055555	-0.00000113	0.00000000
		-6.27284568	-0.00055264	0.00033862	0.00055545	-0.00000113	0.00000000
		-6.27284592	-0.00055254	0.00033856	0.00055535	-0.00000113	0.00000000
		-6.27284616	-0.00055244	0.00033850	0.00055525	-0.00000113	0.00000000
		-6.27284639	-0.00055234	0.00033844	0.00055515	-0.00000113	0.00000000
		-6.27284663	-0.00055225	0.00033838	0.00055505	-0.00000113	0.00000000
		-6.27284686	-0.00055215	0.00033832	0.00055495	-0.00000113	0.00000000
		-6.27284710	-0.00055205	0.00033826	0.00055485	-0.00000113	0.00000000
		-6.27284733	-0.00055195	0.00033820	0.00055475	-0.00000113	0.00000000
		-6.27284757	-0.00055185	0.00033814	0.00055465	-0.00000113	0.00000000
		-6.27284781	-0.00055175	0.00033808	0.00055455	-0.00000113	0.00000000
		-6.27284804	-0.00055165	0.00033802	0.00055446	-0.00000113	0.00000000
		-6.27284828	-0.00055156	0.00033796	0.00055436	-0.00000113	0.00000000
		-6.27284851	-0.00055146	0.00033790	0.00055426	-0.00000113	0.00000000
		-6.27284875	-0.00055136	0.00033784	0.00055416	-0.00000113	0.00000000
		-6.27284898	-0.00055126	0.00033778	0.00055406	-0.00000113	0.00000000
		-6.27284922	-0.00055116	0.00033772	0.00055396	-0.00000113	0.00000000
		-6.27284945	-0.00055107	0.00033766	0.00055386	-0.00000113	0.00000000
		-6.27284968	-0.00055097	0.00033760	0.00055377	-0.00000113	0.00000000
		-6.27284992	-0.00055087	0.00033754	0.00055367	-0.00000113	0.00000000
		-6.27285015	-0.00055077	0.00033748	0.00055357	-0.00000113	0.00000000
		-6.27285039	-0.00055067	0.00033742	0.00055347	-0.00000113	0.00000000
		-6.27285062	-0.00055058	0.00033736	0.00055337	-0.00000113	0.00000000
		-6.27285086	-0.00055048	0.00033730	0.00055327	-0.00000113	0.00000000
		-6.27285109	-0.00055038	0.00033724	0.00055317	-0.00000113	0.00000000
		-6.27285133	-0.00055028	0.00033718	0.00055308	-0.00000113	0.00000000
		-6.27285156	-0.00055018	0.00033712	0.00055298	-0.00000113	0.00000000
		-6.27285179	-0.00055009	0.00033706	0.00055288	-0.00000113	0.00000000
		-6.27285203	-0.00054999	0.00033700	0.00055278	-0.00000113	0.00000000
		-6.27285226	-0.00054989	0.00033694	0.00055268	-0.00000113	0.00000000
		-6.27285249	-0.00054979	0.00033688	0.00055258	-0.00000113	0.00000000
		-6.27285273	-0.00054970	0.00033682	0.00055249	-0.00000113	0.00000000
		-6.27285296	-0.00054960	0.00033676	0.00055239	-0.00000113	0.00000000
		-6.27285319	-0.00054950	0.00033670	0.00055229	-0.00000113	0.00000000
		-6.27285343	-0.00054940	0.00033664	0.00055219	-0.00000113	0.00000000
		-6.27285366	-0.00054931	0.00033658	0.00055209	-0.00000113	0.00000000
		-6.27285389	-0.00054921	0.00033652	0.00055200	-0.00000113	0.00000000
		-6.27285413	-0.00054911	0.00033646	0.00055190	-0.00000113	0.00000000
		-6.27285436	-0.00054901	0.00033640	0.00055180	-0.00000113	0.00000000
		-6.27285459	-0.00054892	0.00033634	0.00055170	-0.00000113	0.00000000
		-6.27285483	-0.00054882	0.00033628	0.00055160	-0.00000113	0.00000000
		-6.27285506	-0.00054872	0.00033622	0.00055151	-0.00000113	0.00000000
		-6.27285529	-0.00054862	0.00033616	0.00055141	-0.00000113	0.00000000
		-6.27285552	-0.00054853	0.00033610	0.00055131	-0.00000113	0.00000000
		-6.27285576	-0.00054843	0.00033604	0.00055121	-0.00000113	0.00000000
		-6.27285599	-0.00054833	0.00033598	0.00055112	-0.00000113	0.00000000
		-6.27285622	-0.00054824	0.00033592	0.00055102	-0.00000113	0.00000000
		-6.27285645	-0.00054814	0.00033586	0.00055092	-0.00000113	0.00000000
		-6.27285669	-0.00054804	0.00033580	0.00055082	-0.00000113	0.00000000
		-6.27285692	-0.00054794	0.00033574	0.00055073	-0.00000113	0.00000000
		-6.27285715	-0.00054785	0.00033568	0.00055063	-0.00000113	0.00000000
		-6.27285738	-0.00054775	0.00033563	0.00055053	-0.00000113	0.00000000
		-6.27285761	-0.00054765	0.00033557	0.00055043	-0.00000113	0.00000000
		-6.27285785	-0.00054756	0.00033551	0.00055034	-0.00000113	0.00000000
		-6.27285808	-0.00054746	0.00033545	0.00055024	-0.00000113	0.00000000
		-6.27285831	-0.00054736	0.00033539	0.00055014	-0.00000113	0.00000000
		-6.27285854	-0.00054727	0.00033533	0.00055004	-0.00000113	0.00000000
		-6.27285877	-0.00054717	0.00033527	0.00054995	-0.00000113	0.00000000
		-6.27285900	-0.00054707	0.00033521	0.00054985	-0.00000113	0.00000000
		-6.27285923	-0.00054698	0.00033515	0.00054975	-0.00000113	0.00000000
		-6.27285947	-0.00054688	0.00033509	0.00054966	-0.00000113	0.00000000
		-6.27285970	-0.00054678	0.00033503	0.00054956	-0.00000113	0.00000000
		-6.27285993	-0.00054669	0.00033497	0.00054946	-0.00000113	0.00000000
		-6.27286016	-0.00054659	0.00033491	0.00054936	-0.00000113	0.00000000
		-6.27286039	-0.00054649	0.00033485	0.00054927	-0.00000113	0.00000000
		-6.27286062	-0.00054640	0.00033480	0.00054917	-0.00000113	0.00000000
		-6.27286085	-0.00054630	0.00033474	0.00054907	-0.00000113	0.00000000
		-6.27286108	-0.00054620	0.00033468	0.00054898	-0.00000113	0.00000000
		-6.27286131	-0.00054611	0.00033462	0.00054888	-0.00000113	0.00000000
		-6.27286154	-0.00054601	0.00033456	0.00054878	-0.00000113	0.00000000
		-6.27286177	-0.00054591	0.00033450	0.00054869	-0.00000113	0.00000000
		-6.27286200	-0.00054582	0.00033444	0.00054859	-0.00000113	0.00000000
		-6.27286223	-0.00054572	0.00033438	0.00054849	-0.00000113	0.00000000
		-6.27286246	-0.00054563	0.00033432	0.00054840	-0.00000113	0.00000000
		-6.27286269	-0.00054553	0.00033426	0.00054830	-0.00000113	0.00000000
		-6.27286292	-0.00054543	0.00033420	0.00054820	-0.00000113	0.00000000
		-6.27286315	-0.00054534	0.00033415	0.00054811	-0.00000113	0.00000000
		-6.27286338	-0.00054524	0.00033409	0.00054801	-0.00000113	0.00000000
		-6.27286361	-0.00054515	0.00033403	0.00054791	-0.00000113	0.00000000
		-6.27286384	-0.00054505	0.00033397	0.00054782	-0.00000113	0.00000000
		-6.27286407	-0.00054495	0.00033391	0.00054772	-0.00000113	0.00000000
		-6.27286430	-0.00054486	0.00033385	0.00054762	-0.00000113	0.00000000
		-6.27286453	-0.00054476	0.00033379	0.00054753	-0.00000113	0.00000000
		-6.27286476	-0.00054467	0.00033373	0.00054743	-0.00000113	0.00000000
		-6.27286499	-0.00054457	0.00033368	0.00054733	-0.00000113	0.00000000
		-6.27286522	-0.00054447	0.00033362	0.00054724	-0.00000113	0.00000000
		-6.27286545	-0.00054438	0.00033356	0.00054714	-0.00000113	0.00000000
		-6.27286568	-0.00054428	0.00033350	0.00054705	-0.00000113	0.00000000
		-6.27286591	-0.00054419	0.00033344	0.00054695	-0.00000113	0.00000000
		-6.27286614	-0.00054409	0.00033338	0.00054685	-0.00000113	0.00000000
		-6.27286636	-0.00054400	0.00033332	0.00054676	-0.00000113	0.00000000
		-6.27286659	-0.00054390	0.00033326	0.00054666	-0.00000113	0.00000000
		-6.27286682	-0.00054380	0.00033321	0.00054657	-0.00000113	0.00000000
		-6.27286705	-0.00054371	0.00033315	0.00054647	-0.00000113	0.00000000
		-6.27286728	-0.00054361	0.00033309	0.00054637	-0.00000113	0.00000000
		-6.27286751	-0.00054352	0.00033303	0.00054628	-0.00000113	0.00000000
		-6.27286774	-0.00054342	0.00033297	0.00054618	-0.00000113	0.00000000
		-6.27286796	-0.00054333	0.00033291	0.00054609	-0.00000113	0.00000000
		-6.27286819	-0.00054323	0.00033286	0.00054599	-0.00000113	0.00000000
		-6.27286842	-0.00054314	0.00033280	0.00054589	-0.00000113	0.00000000
		-6.27286865	-0.00054304	0.00033274	0.00054580	-0.00000113	0.00000000
		-6.27286888	-0.00054295	0.00033268	0.00054570	-0.00000113	0.00000000
		-6.27286910	-0.00054285	0.00033262	0.00054561	-0.00000113	0.00000000
		-6.27286933	-0.00054276	0.00033256	0.00054551	-0.00000113	0.00000000
		-6.27286956	-0.00054266	0.00033250	0.00054542	-0.00000113	0.00000000
		-6.27286979	-0.00054257	0.00033245	0.00054532	-0.00000113	0.00000000
		-6.27287001	-0.00054247	0.00033239	0.00054522	-0.00000113	0.00000000
		-6.27287024	-0.00054238	0.00033233	0.00054513	-0.00000113	0.00000000
		-6.27287047	-0.00054228	0.00033227	0.00054503	-0.00000113	0.00000000
		-6.27287070	-0.00054219	0.00033221	0.00054494	-0.00000113	0.00000000
		-6.27287092	-0.00054209	0.00033216	0.00054484	-0.00000113	0.00000000
		-6.27287115	-0.00054200	0.00033210	0.00054475	-0.00000113	0.00000000
		-6.27287138	-0.00054190	0.00033204	0.00054465	-0.00000113	0.00000000
		-6.27287160	-0.00054181	0.00033198	0.00054456	-0.00000113	0.00000000
		-6.27287183	-0.00054171	0.00033192	0.00054446	-0.00000113	0.00000000
		-6.27287206	-0.00054162	0.00033186	0.00054437	-0.00000113	0.00000000
		-6.27287228	-0.00054152	0.00033181	0.00054427	-0.00000113	0.00000000
		-6.27287251	-0.00054143	0.00033175	0.00054417	-0.00000113	0.00000000
		-6.27287274	-0.00054133	0.00033169	0.00054408	-0.00000113	0.00000000
		-6.27287296	-0.00054124	0.00033163	0.00054398	-0.00000113	0.00000000
		-6.27287319	-0.00054114	0.00033157	0.00054389	-0.00000113	0.00000000
		-6.27287342	-0.00054105	0.00033152	0.00054379	-0.00000113	0.00000000
		-6.27287364	-0.00054095	0.00033146	0.00054370	-0.00000113	0.00000000
		-6.27287387	-0.00054086	0.00033140	0.00054360	-0.00000113	0.00000000
		-6.27287409	-0.00054076	0.00033134	0.00054351	-0.00000113	0.00000000
		-6.27287432	-0.00054067	0.00033128	0.00054341	-0.00000113	0.00000000
		-6.27287455	-0.00054058	0.00033123	0.00054332	-0.00000113	0.00000000
		-6.27287477	-0.00054048	0.00033117	0.00054322	-0.00000113	0.00000000
		-6.27287500	-0.00054039	0.00033111	0.00054313	-0.00000113	0.00000000
		-6.27287522	-0.00054029	0.00033105	0.00054304	-0.00000113	0.00000000
		-6.27287545	-0.00054020	0.00033100	0.00054294	-0.00000113	0.00000000
		-6.27287568	-0.00054010	0.00033094	0.00054285	-0.00000113	0.00000000
		-6.27287590	-0.00054001	0.00033088	0.00054275	-0.00000113	0.00000000
		-6.27287613	-0.00053992	0.00033082	0.00054266	-0.00000113	0.00000000
		-6.27287635	-0.00053982	0.00033076	0.00054256	-0.00000113	0.00000000
		-6.27287658	-0.00053973	0.00033071	0.00054247	-0.00000113	0.00000000
		-6.27287680	-0.00053963	0.00033065	0.00054237	-0.00000113	0.00000000
		-6.27287703	-0.00053954	0.00033059	0.00054228	-0.00000113	0.00000000
		-6.27287725	-0.00053945	0.00033053	0.00054218	-0.00000113	0.00000000
		-6.27287748	-0.00053935	0.00033048	0.00054209	-0.00000113	0.00000000
		-6.27287770	-0.00053926	0.00033042	0.00054199	-0.00000113	0.00000000
		-6.27287793	-0.00053916	0.00033036	0.00054190	-0.00000113	0.00000000
		-6.27287815	-0.00053907	0.00033030	0.00054181	-0.00000113	0.00000000
		-6.27287838	-0.00053898	0.00033025	0.00054171	-0.00000113	0.00000000
		-6.27287860	-0.00053888	0.00033019	0.00054162	-0.00000113	0.00000000
		-6.27287882	-0.00053879	0.00033013	0.00054152	-0.00000113	0.00000000
		-6.27287905	-0.00053869	0.00033007	0.00054143	-0.00000113	0.00000000
		-6.27287927	-0.00053860	0.00033002	0.00054133	-0.00000113	0.00000000
		-6.27287950	-0.00053851	0.00032996	0.00054124	-0.00000113	0.00000000
		-6.27287972	-0.00053841	0.00032990	0.00054115	-0.00000113	0.00000000
		-6.27287994	-0.00053832	0.00032984	0.00054105	-0.00000113	0.00000000
		-6.27288017	-0.00053823	0.00032979	0.00054096	-0.00000113	0.00000000
		-6.27288039	-0.00053813	0.00032973	0.00054086	-0.00000113	0.00000000
		-6.27288062	-0.00053804	0.00032967	0.00054077	-0.00000113	0.00000000
		-6.27288084	-0.00053795	0.00032961	0.00054068	-0.00000113	0.00000000
		-6.27288106	-0.00053785	0.00032956	0.00054058	-0.00000113	0.00000000
		-6.27288129	-0.00053776	0.00032950	0.00054049	-0.00000113	0.00000000
		-6.27288151	-0.00053767	0.00032944	0.00054039	-0.00000113	0.00000000
		-6.27288173	-0.00053757	0.00032938	0.00054030	-0.00000113	0.00000000
		-6.27288196	-0.00053748	0.00032933	0.00054021	-0.00000113	0.00000000
		-6.27288218	-0.00053739	0.00032927	0.00054011	-0.00000113	0.00000000
		-6.27288240	-0.00053729	0.00032921	0.00054002	-0.00000113	0.00000000
		-6.27288263	-0.00053720	0.00032916	0.00053992	-0.00000113	0.00000000
		-6.27288285	-0.00053711	0.00032910	0.00053983	-0.00000113	0.00000000
		-6.27288307	-0.00053701	0.00032904	0.00053974	-0.00000113	0.00000000
		-6.27288330	-0.00053692	0.00032898	0.00053964	-0.00000113	0.00000000
		-6.27288352	-0.00053683	0.00032893	0.00053955	-0.00000113	0.00000000
		-6.27288374	-0.00053673	0.00032887	0.00053946	-0.00000113	0.00000000
		-6.27288396	-0.00053664	0.00032881	0.00053936	-0.00000113	0.00000000
		-6.27288419	-0.00053655	0.00032876	0.00053927	-0.00000113	0.00000000
		-6.27288441	-0.00053645	0.00032870	0.00053918	-0.00000113	0.00000000
		-6.27288463	-0.00053636	0.00032864	0.00053908	-0.00000113	0.00000000
		-6.27288485	-0.00053627	0.00032859	0.00053899	-0.00000113	0.00000000
		-6.27288508	-0.00053618	0.00032853	0.00053890	-0.00000113	0.00000000
		-6.27288530	-0.00053608	0.00032847	0.00053880	-0.00000113	0.00000000
		-6.27288552	-0.00053599	0.00032841	0.00053871	-0.00000113	0.00000000
		-6.27288574	-0.00053590	0.00032836	0.00053862	-0.00000113	0.00000000
		-6.27288596	-0.00053580	0.00032830	0.00053852	-0.00000113	0.00000000
		-6.27288619	-0.00053571	0.00032824	0.00053843	-0.00000113	0.00000000
		-6.27288641	-0.00053562	0.00032819	0.00053834	-0.00000113	0.00000000
		-6.27288663	-0.00053553	0.00032813	0.00053824	-0.00000113	0.00000000
		-6.27288685	-0.00053543	0.00032807	0.00053815	-0.00000113	0.00000000
		-6.27288707	-0.00053534	0.00032802	0.00053806	-0.00000113	0.00000000
		-6.27288729	-0.00053525	0.00032796	0.00053796	-0.00000113	0.00000000
		-6.27288752	-0.00053516	0.00032790	0.00053787	-0.00000113	0.00000000
		-6.27288774	-0.00053506	0.00032785	0.00053778	-0.00000113	0.00000000
		-6.27288796	-0.00053497	0.00032779	0.00053769	-0.00000113	0.00000000
		-6.27288818	-0.00053488	0.00032773	0.00053759	-0.00000113	0.00000000
		-6.27288840	-0.00053479	0.00032768	0.00053750	-0.00000113	0.00000000
		-6.27288862	-0.00053469	0.00032762	0.00053741	-0.00000113	0.00000000
		-6.27288884	-0.00053460	0.00032756	0.00053731	-0.00000113	0.00000000
		-6.27288906	-0.00053451	0.00032751	0.00053722	-0.00000113	0.00000000
		-6.27288928	-0.00053442	0.00032745	0.00053713	-0.00000113	0.00000000
		-6.27288950	-0.00053432	0.00032739	0.00053704	-0.00000113	0.00000000
		-6.27288973	-0.00053423	0.00032734	0.00053694	-0.00000113	0.00000000
		-6.27288995	-0.00053414	0.00032728	0.00053685	-0.00000113	0.00000000
		-6.27289017	-0.00053405	0.00032722	0.00053676	-0.00000113	0.00000000
		-6.27289039	-0.00053396	0.00032717	0.00053666	-0.00000113	0.00000000
		-6.27289061	-0.00053386	0.00032711	0.00053657	-0.00000113	0.00000000
		-6.27289083	-0.00053377	0.00032705	0.00053648	-0.00000113	0.00000000
		-6.27289105	-0.00053368	0.00032700	0.00053639	-0.00000113	0.00000000
		-6.27289127	-0.00053359	0.00032694	0.00053629	-0.00000113	0.00000000
		-6.27289149	-0.00053350	0.00032689	0.00053620	-0.00000113	0.00000000
		-6.27289171	-0.00053340	0.00032683	0.00053611	-0.00000113	0.00000000
		-6.27289193	-0.00053331	0.00032677	0.00053602	-0.00000113	0.00000000
		-6.27289215	-0.00053322	0.00032672	0.00053593	-0.00000113	0.00000000
		-6.27289237	-0.00053313	0.00032666	0.00053583	-0.00000113	0.00000000
		-6.27289259	-0.00053304	0.00032660	0.00053574	-0.00000113	0.00000000
		-6.27289281	-0.00053294	0.00032655	0.00053565	-0.00000113	0.00000000
		-6.27289303	-0.00053285	0.00032649	0.00053556	-0.00000113	0.00000000
		-6.27289325	-0.00053276	0.00032643	0.00053546	-0.00000113	0.00000000
		-6.27289346	-0.00053267	0.00032638	0.00053537	-0.00000113	0.00000000
		-6.27289368	-0.00053258	0.00032632	0.00053528	-0.00000113	0.00000000
		-6.27289390	-0.00053249	0.00032627	0.00053519	-0.00000113	0.00000000
		-6.27289412	-0.00053239	0.00032621	0.00053510	-0.00000113	0.00000000
		-6.27289434	-0.00053230	0.00032615	0.00053500	-0.00000113	0.00000000
		-6.27289456	-0.00053221	0.00032610	0.00053491	-0.00000113	0.00000000
		-6.27289478	-0.00053212	0.00032604	0.00053482	-0.00000113	0.00000000
		-6.27289500	-0.00053203	0.00032599	0.00053473	-0.00000113	0.00000000
		-6.27289522	-0.00053194	0.00032593	0.00053464	-0.00000113	0.00000000
		-6.27289544	-0.00053185	0.00032587	0.00053454	-0.00000113	0.00000000
		-6.27289565	-0.00053175	0.00032582	0.00053445	-0.00000113	0.00000000
		-6.27289587	-0.00053166	0.00032576	0.00053436	-0.00000113	0.00000000
		-6.27289609	-0.00053157	0.00032571	0.00053427	-0.00000113	0.00000000
		-6.27289631	-0.00053148	0.00032565	0.00053418	-0.00000113	0.00000000
		-6.27289653	-0.00053139	0.00032559	0.00053408	-0.00000113	0.00000000
		-6.27289675	-0.00053130	0.00032554	0.00053399	-0.00000113	0.00000000
		-6.27289696	-0.00053121	0.00032548	0.00053390	-0.00000113	0.00000000
		-6.27289718	-0.00053112	0.00032543	0.00053381	-0.00000113	0.00000000
		-6.27289740	-0.00053102	0.00032537	0.00053372	-0.00000113	0.00000000
		-6.27289762	-0.00053093	0.00032531	0.00053363	-0.00000113	0.00000000
		-6.27289784	-0.00053084	0.00032526	0.00053354	-0.00000113	0.00000000
		-6.27289805	-0.00053075	0.00032520	0.00053344	-0.00000113	0.00000000
		-6.27289827	-0.00053066	0.00032515	0.00053335	-0.00000113	0.00000000
		-6.27289849	-0.00053057	0.00032509	0.00053326	-0.00000113	0.00000000
		-6.27289871	-0.00053048	0.00032504	0.00053317	-0.00000113	0.00000000
		-6.27289892	-0.00053039	0.00032498	0.00053308	-0.00000113	0.00000000
		-6.27289914	-0.00053030	0.00032492	0.00053299	-0.00000113	0.00000000
		-6.27289936	-0.00053021	0.00032487	0.00053290	-0.00000113	0.00000000
		-6.27289958	-0.00053012	0.00032481	0.00053280	-0.00000113	0.00000000
		-6.27289979	-0.00053002	0.00032476	0.00053271	-0.00000113	0.00000000
		-6.27290001	-0.00052993	0.00032470	0.00053262	-0.00000113	0.00000000
		-6.27290023	-0.00052984	0.00032465	0.00053253	-0.00000113	0.00000000
		-6.27290044	-0.00052975	0.00032459	0.00053244	-0.00000113	0.00000000
		-6.27290066	-0.00052966	0.00032453	0.00053235	-0.00000113	0.00000000
		-6.27290088	-0.00052957	0.00032448	0.00053226	-0.00000113	0.00000000
		-6.27290109	-0.00052948	0.00032442	0.00053217	-0.00000113	0.00000000
		-6.27290131	-0.00052939	0.00032437	0.00053208	-0.00000113	0.00000000
		-6.27290153	-0.00052930	0.00032431	0.00053198	-0.00000113	0.00000000
		-6.27290174	-0.00052921	0.00032426	0.00053189	-0.00000113	0.00000000
		-6.27290196	-0.00052912	0.00032420	0.00053180	-0.00000113	0.00000000
		-6.27290218	-0.00052903	0.00032415	0.00053171	-0.00000113	0.00000000
		-6.27290239	-0.00052894	0.00032409	0.00053162	-0.00000113	0.00000000
		-6.27290261	-0.00052885	0.00032404	0.00053153	-0.00000113	0.00000000
		-6.27290283	-0.00052876	0.00032398	0.00053144	-0.00000113	0.00000000
		-6.27290304	-0.00052867	0.00032392	0.00053135	-0.00000113	0.00000000
		-6.27290326	-0.00052858	0.00032387	0.00053126	-0.00000113	0.00000000
		-6.27290347	-0.00052849	0.00032381	0.00053117	-0.00000113	0.00000000
		-6.27290369	-0.00052840	0.00032376	0.00053108	-0.00000113	0.00000000
		-6.27290391	-0.00052831	0.00032370	0.00053099	-0.00000113	0.00000000
		-6.27290412	-0.00052822	0.00032365	0.00053089	-0.00000113	0.00000000
		-6.27290434	-0.00052813	0.00032359	0.00053080	-0.00000113	0.00000000
		-6.27290455	-0.00052804	0.00032354	0.00053071	-0.00000113	0.00000000
		-6.27290477	-0.00052795	0.00032348	0.00053062	-0.00000113	0.00000000
		-6.27290498	-0.00052786	0.00032343	0.00053053	-0.00000113	0.00000000
		-6.27290520	-0.00052777	0.00032337	0.00053044	-0.00000113	0.00000000
		-6.27290541	-0.00052768	0.00032332	0.00053035	-0.00000113	0.00000000
		-6.27290563	-0.00052759	0.00032326	0.00053026	-0.00000113	0.00000000
		-6.27290584	-0.00052750	0.00032321	0.00053017	-0.00000113	0.00000000
		-6.27290606	-0.00052741	0.00032315	0.00053008	-0.00000113	0.00000000
		-6.27290627	-0.00052732	0.00032310	0.00052999	-0.00000113	0.00000000
		-6.27290649	-0.00052723	0.00032304	0.00052990	-0.00000113	0.00000000
		-6.27290670	-0.00052714	0.00032299	0.00052981	-0.00000113	0.00000000
		-6.27290692	-0.00052705	0.00032293	0.00052972	-0.00000113	0.00000000
		-6.27290713	-0.00052696	0.00032288	0.00052963	-0.00000113	0.00000000
		-6.27290735	-0.00052687	0.00032282	0.00052954	-0.00000113	0.00000000
		-6.27290756	-0.00052678	0.00032277	0.00052945	-0.00000113	0.00000000
		-6.27290778	-0.00052669	0.00032271	0.00052936	-0.00000113	0.00000000
		-6.27290799	-0.00052660	0.00032266	0.00052927	-0.00000113	0.00000000
		-6.27290821	-0.00052651	0.00032260	0.00052918	-0.00000113	0.00000000
		-6.27290842	-0.00052642	0.00032255	0.00052909	-0.00000113	0.00000000
		-6.27290863	-0.00052633	0.00032249	0.00052900	-0.00000113	0.00000000
		-6.27290885	-0.00052624	0.00032244	0.00052891	-0.00000113	0.00000000
		-6.27290906	-0.00052615	0.00032238	0.00052882	-0.00000113	0.00000000
		-6.27290928	-0.00052606	0.00032233	0.00052873	-0.00000113	0.00000000
		-6.27290949	-0.00052597	0.00032227	0.00052864	-0.00000113	0.00000000
		-6.27290970	-0.00052588	0.00032222	0.00052855	-0.00000113	0.00000000
		-6.27290992	-0.00052579	0.00032216	0.00052846	-0.00000113	0.00000000
		-6.27291013	-0.00052570	0.00032211	0.00052837	-0.00000113	0.00000000
		-6.27291034	-0.00052561	0.00032205	0.00052828	-0.00000113	0.00000000
		-6.27291056	-0.00052553	0.00032200	0.00052819	-0.00000113	0.00000000
		-6.27291077	-0.00052544	0.00032194	0.00052810	-0.00000113	0.00000000
		-6.27291098	-0.00052535	0.00032189	0.00052801	-0.00000113	0.00000000
		-6.27291120	-0.00052526	0.00032183	0.00052792	-0.00000113	0.00000000
		-6.27291141	-0.00052517	0.00032178	0.00052783	-0.00000113	0.00000000
		-6.27291162	-0.00052508	0.00032173	0.00052774	-0.00000113	0.00000000
		-6.27291184	-0.00052499	0.00032167	0.00052765	-0.00000113	0.00000000
		-6.27291205	-0.00052490	0.00032162	0.00052756	-0.00000113	0.00000000
		-6.27291226	-0.00052481	0.00032156	0.00052747	-0.00000113	0.00000000
		-6.27291248	-0.00052472	0.00032151	0.00052738	-0.00000113	0.00000000
		-6.27291269	-0.00052463	0.00032145	0.00052730	-0.00000113	0.00000000
		-6.27291290	-0.00052455	0.00032140	0.00052721	-0.00000113	0.00000000
		-6.27291311	-0.00052446	0.00032134	0.00052712	-0.00000113	0.00000000
		-6.27291333	-0.00052437	0.00032129	0.00052703	-0.00000113	0.00000000
		-6.27291354	-0.00052428	0.00032123	0.00052694	-0.00000113	0.00000000
		-6.27291375	-0.00052419	0.00032118	0.00052685	-0.00000113	0.00000000
		-6.27291396	-0.00052410	0.00032113	0.00052676	-0.00000113	0.00000000
		-6.27291418	-0.00052401	0.00032107	0.00052667	-0.00000113	0.00000000
		-6.27291439	-0.00052392	0.00032102	0.00052658	-0.00000113	0.00000000
		-6.27291460	-0.00052384	0.00032096	0.00052649	-0.00000113	0.00000000
		-6.27291481	-0.00052375	0.00032091	0.00052640	-0.00000113	0.00000000
		-6.27291503	-0.00052366	0.00032085	0.00052631	-0.00000113	0.00000000
		-6.27291524	-0.00052357	0.00032080	0.00052622	-0.00000113	0.00000000
		-6.27291545	-0.00052348	0.00032075	0.00052614	-0.00000113	0.00000000
		-6.27291566	-0.00052339	0.00032069	0.00052605	-0.00000113	0.00000000
		-6.27291587	-0.00052330	0.00032064	0.00052596	-0.00000113	0.00000000
		-6.27291608	-0.00052322	0.00032058	0.00052587	-0.00000113	0.00000000
		-6.27291630	-0.00052313	0.00032053	0.00052578	-0.00000113	0.00000000
		-6.27291651	-0.00052304	0.00032047	0.00052569	-0.00000113	0.00000000
		-6.27291672	-0.00052295	0.00032042	0.00052560	-0.00000113	0.00000000
		-6.27291693	-0.00052286	0.00032037	0.00052551	-0.00000113	0.00000000
		-6.27291714	-0.00052277	0.00032031	0.00052542	-0.00000113	0.00000000
		-6.27291735	-0.00052269	0.00032026	0.00052534	-0.00000113	0.00000000
		-6.27291756	-0.00052260	0.00032020	0.00052525	-0.00000113	0.00000000
		-6.27291777	-0.00052251	0.00032015	0.00052516	-0.00000113	0.00000000
		-6.27291799	-0.00052242	0.00032010	0.00052507	-0.00000113	0.00000000
		-6.27291820	-0.00052233	0.00032004	0.00052498	-0.00000113	0.00000000
		-6.27291841	-0.00052225	0.00031999	0.00052489	-0.00000113	0.00000000
		-6.27291862	-0.00052216	0.00031993	0.00052480	-0.00000113	0.00000000
		-6.27291883	-0.00052207	0.00031988	0.00052472	-0.00000113	0.00000000
		-6.27291904	-0.00052198	0.00031983	0.00052463	-0.00000113	0.00000000
		-6.27291925	-0.00052189	0.00031977	0.00052454	-0.00000113	0.00000000
		-6.27291946	-0.00052180	0.00031972	0.00052445	-0.00000113	0.00000000
		-6.27291967	-0.00052172	0.00031966	0.00052436	-0.00000113	0.00000000
		-6.27291988	-0.00052163	0.00031961	0.00052427	-0.00000113	0.00000000
		-6.27292009	-0.00052154	0.00031956	0.00052419	-0.00000113	0.00000000
		-6.27292030	-0.00052145	0.00031950	0.00052410	-0.00000113	0.00000000
		-6.27292051	-0.00052137	0.00031945	0.00052401	-0.00000113	0.00000000
		-6.27292072	-0.00052128	0.00031939	0.00052392	-0.00000113	0.00000000
		-6.27292093	-0.00052119	0.00031934	0.00052383	-0.00000113	0.00000000
		-6.27292114	-0.00052110	0.00031929	0.00052374	-0.00000113	0.00000000
		-6.27292135	-0.00052101	0.00031923	0.00052366	-0.00000113	0.00000000
		-6.27292156	-0.00052093	0.00031918	0.00052357	-0.00000113	0.00000000
		-6.27292177	-0.00052084	0.00031913	0.00052348	-0.00000113	0.00000000
		-6.27292198	-0.00052075	0.00031907	0.00052339	-0.00000113	0.00000000
		-6.27292219	-0.00052066	0.00031902	0.00052330	-0.00000113	0.00000000
		-6.27292240	-0.00052058	0.00031896	0.00052322	-0.00000113	0.00000000
		-6.27292261	-0.00052049	0.00031891	0.00052313	-0.00000113	0.00000000
		-6.27292282	-0.00052040	0.00031886	0.00052304	-0.00000113	0.00000000
		-6.27292303	-0.00052031	0.00031880	0.00052295	-0.00000113	0.00000000
		-6.27292324	-0.00052023	0.00031875	0.00052286	-0.00000113	0.00000000
		-6.27292345	-0.00052014	0.00031870	0.00052278	-0.00000113	0.00000000
		-6.27292366	-0.00052005	0.00031864	0.00052269	-0.00000113	0.00000000
		-6.27292386	-0.00051996	0.00031859	0.00052260	-0.00000113	0.00000000
		-6.27292407	-0.00051988	0.00031854	0.00052251	-0.00000113	0.00000000
		-6.27292428	-0.00051979	0.00031848	0.00052242	-0.00000113	0.00000000
		-6.27292449	-0.00051970	0.00031843	0.00052234	-0.00000113	0.00000000
		-6.27292470	-0.00051962	0.00031837	0.00052225	-0.00000113	0.00000000
		-6.27292491	-0.00051953	0.00031832	0.00052216	-0.00000113	0.00000000
		-6.27292512	-0.00051944	0.00031827	0.00052207	-0.00000113	0.00000000
		-6.27292533	-0.00051935	0.00031821	0.00052199	-0.00000113	0.00000000
		-6.27292553	-0.00051927	0.00031816	0.00052190	-0.00000113	0.00000000
		-6.27292574	-0.00051918	0.00031811	0.00052181	-0.00000113	0.00000000
		-6.27292595	-0.00051909	0.00031805	0.00052172	-0.00000113	0.00000000
		-6.27292616	-0.00051901	0.00031800	0.00052164	-0.00000113	0.00000000
		-6.27292637	-0.00051892	0.00031795	0.00052155	-0.00000113	0.00000000
		-6.27292658	-0.00051883	0.00031789	0.00052146	-0.00000113	0.00000000
		-6.27292678	-0.00051874	0.00031784	0.00052137	-0.00000113	0.00000000
		-6.27292699	-0.00051866	0.00031779	0.00052129	-0.00000113	0.00000000
		-6.27292720	-0.00051857	0.00031773	0.00052120	-0.00000113	0.00000000
		-6.27292741	-0.00051848	0.00031768	0.00052111	-0.00000113	0.00000000
		-6.27292762	-0.00051840	0.00031763	0.00052102	-0.00000113	0.00000000
		-6.27292782	-0.00051831	0.00031757	0.00052094	-0.00000113	0.00000000
		-6.27292803	-0.00051822	0.00031752	0.00052085	-0.00000113	0.00000000
		-6.27292824	-0.00051814	0.00031747	0.00052076	-0.00000113	0.00000000
		-6.27292845	-0.00051805	0.00031742	0.00052068	-0.00000113	0.00000000
		-6.27292865	-0.00051796	0.00031736	0.00052059	-0.00000113	0.00000000
		-6.27292886	-0.00051788	0.00031731	0.00052050	-0.00000113	0.00000000
		-6.27292907	-0.00051779	0.00031726	0.00052041	-0.00000113	0.00000000
		-6.27292927	-0.00051770	0.00031720	0.00052033	-0.00000113	0.00000000
		-6.27292948	-0.00051762	0.00031715	0.00052024	-0.00000113	0.00000000
		-6.27292969	-0.00051753	0.00031710	0.00052015	-0.00000113	0.00000000
		-6.27292990	-0.00051744	0.00031704	0.00052007	-0.00000113	0.00000000
		-6.27293010	-0.00051736	0.00031699	0.00051998	-0.00000113	0.00000000
		-6.27293031	-0.00051727	0.00031694	0.00051989	-0.00000113	0.00000000
		-6.27293052	-0.00051718	0.00031688	0.00051981	-0.00000113	0.00000000
		-6.27293072	-0.00051710	0.00031683	0.00051972	-0.00000113	0.00000000
		-6.27293093	-0.00051701	0.00031678	0.00051963	-0.00000113	0.00000000
		-6.27293114	-0.00051693	0.00031673	0.00051955	-0.00000113	0.00000000
		-6.27293134	-0.00051684	0.00031667	0.00051946	-0.00000113	0.00000000
		-6.27293155	-0.00051675	0.00031662	0.00051937	-0.00000113	0.00000000
		-6.27293176	-0.00051667	0.00031657	0.00051929	-0.00000113	0.00000000
		-6.27293196	-0.00051658	0.00031651	0.00051920	-0.00000113	0.00000000
		-6.27293217	-0.00051649	0.00031646	0.00051911	-0.00000113	0.00000000
		-6.27293237	-0.00051641	0.00031641	0.00051903	-0.00000113	0.00000000
		-6.27293258	-0.00051632	0.00031636	0.00051894	-0.00000113	0.00000000
		-6.27293279	-0.00051624	0.00031630	0.00051885	-0.00000113	0.00000000
		-6.27293299	-0.00051615	0.00031625	0.00051877	-0.00000113	0.00000000
		-6.27293320	-0.00051606	0.00031620	0.00051868	-0.00000113	0.00000000
		-6.27293340	-0.00051598	0.00031614	0.00051859	-0.00000113	0.00000000
		-6.27293361	-0.00051589	0.00031609	0.00051851	-0.00000113	0.00000000
		-6.27293382	-0.00051581	0.00031604	0.00051842	-0.00000113	0.00000000
		-6.27293402	-0.00051572	0.00031599	0.00051833	-0.00000113	0.00000000
		-6.27293423	-0.00051563	0.00031593	0.00051825	-0.00000113	0.00000000
		-6.27293443	-0.00051555	0.00031588	0.00051816	-0.00000113	0.00000000
		-6.27293464	-0.00051546	0.00031583	0.00051807	-0.00000113	0.00000000
		-6.27293484	-0.00051538	0.00031578	0.00051799	-0.00000113	0.00000000
		-6.27293505	-0.00051529	0.00031572	0.00051790	-0.00000113	0.00000000
		-6.27293525	-0.00051520	0.00031567	0.00051782	-0.00000113	0.00000000
		-6.27293546	-0.00051512	0.00031562	0.00051773	-0.00000113	0.00000000
		-6.27293566	-0.00051503	0.00031557	0.00051764	-0.00000113	0.00000000
		-6.27293587	-0.00051495	0.00031551	0.00051756	-0.00000113	0.00000000
		-6.27293607	-0.00051486	0.00031546	0.00051747	-0.00000113	0.00000000
		-6.27293628	-0.00051478	0.00031541	0.00051739	-0.00000113	0.00000000
		-6.27293648	-0.00051469	0.00031536	0.00051730	-0.00000113	0.00000000
		-6.27293669	-0.00051461	0.00031530	0.00051721	-0.00000113	0.00000000
		-6.27293689	-0.00051452	0.00031525	0.00051713	-0.00000113	0.00000000
		-6.27293710	-0.00051443	0.00031520	0.00051704	-0.00000113	0.00000000
		-6.27293730	-0.00051435	0.00031515	0.00051696	-0.00000113	0.00000000
		-6.27293751	-0.00051426	0.00031509	0.00051687	-0.00000113	0.00000000
		-6.27293771	-0.00051418	0.00031504	0.00051678	-0.00000113	0.00000000
		-6.27293792	-0.00051409	0.00031499	0.00051670	-0.00000113	0.00000000
		-6.27293812	-0.00051401	0.00031494	0.00051661	-0.00000113	0.00000000
		-6.27293832	-0.00051392	0.00031488	0.00051653	-0.00000113	0.00000000
		-6.27293853	-0.00051384	0.00031483	0.00051644	-0.00000113	0.00000000
		-6.27293873	-0.00051375	0.00031478	0.00051635	-0.00000113	0.00000000
		-6.27293894	-0.00051367	0.00031473	0.00051627	-0.00000113	0.00000000
		-6.27293914	-0.00051358	0.00031467	0.00051618	-0.00000113	0.00000000
		-6.27293934	-0.00051350	0.00031462	0.00051610	-0.00000113	0.00000000
		-6.27293955	-0.00051341	0.00031457	0.00051601	-0.00000113	0.00000000
		-6.27293975	-0.00051333	0.00031452	0.00051593	-0.00000113	0.00000000
		-6.27293995	-0.00051324	0.00031447	0.00051584	-0.00000113	0.00000000
		-6.27294016	-0.00051315	0.00031441	0.00051576	-0.00000113	0.00000000
		-6.27294036	-0.00051307	0.00031436	0.00051567	-0.00000113	0.00000000
		-6.27294057	-0.00051298	0.00031431	0.00051558	-0.00000113	0.00000000
		-6.27294077	-0.00051290	0.00031426	0.00051550	-0.00000113	0.00000000
		-6.27294097	-0.00051281	0.00031421	0.00051541	-0.00000113	0.00000000
		-6.27294118	-0.00051273	0.00031415	0.00051533	-0.00000113	0.00000000
		-6.27294138	-0.00051265	0.00031410	0.00051524	-0.00000113	0.00000000
		-6.27294158	-0.00051256	0.00031405	0.00051516	-0.00000113	0.00000000
		-6.27294178	-0.00051248	0.00031400	0.00051507	-0.00000113	0.00000000
		-6.27294199	-0.00051239	0.00031395	0.00051499	-0.00000113	0.00000000
		-6.27294219	-0.00051231	0.00031389	0.00051490	-0.00000113	0.00000000
		-6.27294239	-0.00051222	0.00031384	0.00051482	-0.00000113	0.00000000
		-6.27294260	-0.00051214	0.00031379	0.00051473	-0.00000113	0.00000000
		-6.27294280	-0.00051205	0.00031374	0.00051465	-0.00000113	0.00000000
		-6.27294300	-0.00051197	0.00031369	0.00051456	-0.00000113	0.00000000
		-6.27294320	-0.00051188	0.00031363	0.00051448	-0.00000113	0.00000000
		-6.27294341	-0.00051180	0.00031358	0.00051439	-0.00000113	0.00000000
		-6.27294361	-0.00051171	0.00031353	0.00051431	-0.00000113	0.00000000
		-6.27294381	-0.00051163	0.00031348	0.00051422	-0.00000113	0.00000000
		-6.27294401	-0.00051154	0.00031343	0.00051414	-0.00000113	0.00000000
		-6.27294422	-0.00051146	0.00031337	0.00051405	-0.00000113	0.00000000
		-6.27294442	-0.00051137	0.00031332	0.00051397	-0.00000113	0.00000000
		-6.27294462	-0.00051129	0.00031327	0.00051388	-0.00000113	0.00000000
		-6.27294482	-0.00051121	0.00031322	0.00051380	-0.00000113	0.00000000
		-6.27294502	-0.00051112	0.00031317	0.00051371	-0.00000113	0.00000000
		-6.27294523	-0.00051104	0.00031312	0.00051363	-0.00000113	0.00000000
		-6.27294543	-0.00051095	0.00031306	0.00051354	-0.00000113	0.00000000
		-6.27294563	-0.00051087	0.00031301	0.00051346	-0.00000113	0.00000000
		-6.27294583	-0.00051078	0.00031296	0.00051337	-0.00000113	0.00000000
		-6.27294603	-0.00051070	0.00031291	0.00051329	-0.00000113	0.00000000
		-6.27294623	-0.00051062	0.00031286	0.00051320	-0.00000113	0.00000000
		-6.27294644	-0.00051053	0.00031281	0.00051312	-0.00000113	0.00000000
		-6.27294664	-0.00051045	0.00031275	0.00051303	-0.00000113	0.00000000
		-6.27294684	-0.00051036	0.00031270	0.00051295	-0.00000113	0.00000000
		-6.27294704	-0.00051028	0.00031265	0.00051286	-0.00000113	0.00000000
		-6.27294724	-0.00051019	0.00031260	0.00051278	-0.00000113	0.00000000
		-6.27294744	-0.00051011	0.00031255	0.00051270	-0.00000113	0.00000000
		-6.27294764	-0.00051003	0.00031250	0.00051261	-0.00000113	0.00000000
		-6.27294784	-0.00050994	0.00031244	0.00051253	-0.00000113	0.00000000
		-6.27294805	-0.00050986	0.00031239	0.00051244	-0.00000113	0.00000000
		-6.27294825	-0.00050977	0.00031234	0.00051236	-0.00000113	0.00000000
		-6.27294845	-0.00050969	0.00031229	0.00051227	-0.00000113	0.00000000
		-6.27294865	-0.00050961	0.00031224	0.00051219	-0.00000113	0.00000000
		-6.27294885	-0.00050952	0.00031219	0.00051210	-0.00000113	0.00000000
		-6.27294905	-0.00050944	0.00031214	0.00051202	-0.00000113	0.00000000
		-6.27294925	-0.00050936	0.00031208	0.00051194	-0.00000113	0.00000000
		-6.27294945	-0.00050927	0.00031203	0.00051185	-0.00000113	0.00000000
		-6.27294965	-0.00050919	0.00031198	0.00051177	-0.00000113	0.00000000
		-6.27294985	-0.00050910	0.00031193	0.00051168	-0.00000113	0.00000000
		-6.27295005	-0.00050902	0.00031188	0.00051160	-0.00000113	0.00000000
		-6.27295025	-0.00050894	0.00031183	0.00051152	-0.00000113	0.00000000
		-6.27295045	-0.00050885	0.00031178	0.00051143	-0.00000113	0.00000000
		-6.27295065	-0.00050877	0.00031173	0.00051135	-0.00000113	0.00000000
		-6.27295085	-0.00050869	0.00031167	0.00051126	-0.00000113	0.00000000
		-6.27295105	-0.00050860	0.00031162	0.00051118	-0.00000113	0.00000000
		-6.27295125	-0.00050852	0.00031157	0.00051109	-0.00000113	0.00000000
		-6.27295145	-0.00050844	0.00031152	0.00051101	-0.00000113	0.00000000
		-6.27295165	-0.00050835	0.00031147	0.00051093	-0.00000113	0.00000000
		-6.27295185	-0.00050827	0.00031142	0.00051084	-0.00000113	0.00000000
		-6.27295205	-0.00050818	0.00031137	0.00051076	-0.00000113	0.00000000
		-6.27295225	-0.00050810	0.00031132	0.00051068	-0.00000113	0.00000000
		-6.27295245	-0.00050802	0.00031126	0.00051059	-0.00000113	0.00000000
		-6.27295265	-0.00050793	0.00031121	0.00051051	-0.00000113	0.00000000
		-6.27295285	-0.00050785	0.00031116	0.00051042	-0.00000113	0.00000000
		-6.27295305	-0.00050777	0.00031111	0.00051034	-0.00000113	0.00000000
		-6.27295325	-0.00050768	0.00031106	0.00051026	-0.00000113	0.00000000
		-6.27295345	-0.00050760	0.00031101	0.00051017	-0.00000113	0.00000000
		-6.27295365	-0.00050752	0.00031096	0.00051009	-0.00000113	0.00000000
		-6.27295385	-0.00050743	0.00031091	0.00051001	-0.00000113	0.00000000
		-6.27295404	-0.00050735	0.00031086	0.00050992	-0.00000113	0.00000000
		-6.27295424	-0.00050727	0.00031081	0.00050984	-0.00000113	0.00000000
		-6.27295444	-0.00050719	0.00031075	0.00050975	-0.00000113	0.00000000
		-6.27295464	-0.00050710	0.00031070	0.00050967	-0.00000113	0.00000000
		-6.27295484	-0.00050702	0.00031065	0.00050959	-0.00000113	0.00000000
		-6.27295504	-0.00050694	0.00031060	0.00050950	-0.00000113	0.00000000
		-6.27295524	-0.00050685	0.00031055	0.00050942	-0.00000113	0.00000000
		-6.27295544	-0.00050677	0.00031050	0.00050934	-0.00000113	0.00000000
		-6.27295563	-0.00050669	0.00031045	0.00050925	-0.00000113	0.00000000
		-6.27295583	-0.00050660	0.00031040	0.00050917	-0.00000113	0.00000000
		-6.27295603	-0.00050652	0.00031035	0.00050909	-0.00000113	0.00000000
		-6.27295623	-0.00050644	0.00031030	0.00050900	-0.00000113	0.00000000
		-6.27295643	-0.00050636	0.00031025	0.00050892	-0.00000113	0.00000000
		-6.27295663	-0.00050627	0.00031019	0.00050884	-0.00000113	0.00000000
		-6.27295682	-0.00050619	0.00031014	0.00050875	-0.00000113	0.00000000
		-6.27295702	-0.00050611	0.00031009	0.00050867	-0.00000113	0.00000000
		-6.27295722	-0.00050602	0.00031004	0.00050859	-0.00000113	0.00000000
		-6.27295742	-0.00050594	0.00030999	0.00050851	-0.00000113	0.00000000
		-6.27295762	-0.00050586	0.00030994	0.00050842	-0.00000113	0.00000000
		-6.27295781	-0.00050578	0.00030989	0.00050834	-0.00000113	0.00000000
		-6.27295801	-0.00050569	0.00030984	0.00050826	-0.00000113	0.00000000
		-6.27295821	-0.00050561	0.00030979	0.00050817	-0.00000113	0.00000000
		-6.27295841	-0.00050553	0.00030974	0.00050809	-0.00000113	0.00000000
		-6.27295860	-0.00050545	0.00030969	0.00050801	-0.00000113	0.00000000
		-6.27295880	-0.00050536	0.00030964	0.00050792	-0.00000113	0.00000000
		-6.27295900	-0.00050528	0.00030959	0.00050784	-0.00000113	0.00000000
		-6.27295920	-0.00050520	0.00030954	0.00050776	-0.00000113	0.00000000
		-6.27295939	-0.00050512	0.00030949	0.00050768	-0.00000113	0.00000000
		-6.27295959	-0.00050503	0.00030944	0.00050759	-0.00000113	0.00000000
		-6.27295979	-0.00050495	0.00030938	0.00050751	-0.00000113	0.00000000
		-6.27295998	-0.00050487	0.00030933	0.00050743	-0.00000113	0.00000000
		-6.27296018	-0.00050479	0.00030928	0.00050734	-0.00000113	0.00000000
		-6.27296038	-0.00050470	0.00030923	0.00050726	-0.00000113	0.00000000
		-6.27296058	-0.00050462	0.00030918	0.00050718	-0.00000113	0.00000000
		-6.27296077	-0.00050454	0.00030913	0.00050710	-0.00000113	0.00000000
		-6.27296097	-0.00050446	0.00030908	0.00050701	-0.00000113	0.00000000
		-6.27296117	-0.00050438	0.00030903	0.00050693	-0.00000113	0.00000000
		-6.27296136	-0.00050429	0.00030898	0.00050685	-0.00000113	0.00000000
		-6.27296156	-0.00050421	0.00030893	0.00050677	-0.00000113	0.00000000
		-6.27296176	-0.00050413	0.00030888	0.00050668	-0.00000113	0.00000000
		-6.27296195	-0.00050405	0.00030883	0.00050660	-0.00000113	0.00000000
		-6.27296215	-0.00050397	0.00030878	0.00050652	-0.00000113	0.00000000
		-6.27296234	-0.00050388	0.00030873	0.00050644	-0.00000113	0.00000000
		-6.27296254	-0.00050380	0.00030868	0.00050635	-0.00000113	0.00000000
		-6.27296274	-0.00050372	0.00030863	0.00050627	-0.00000113	0.00000000
		-6.27296293	-0.00050364	0.00030858	0.00050619	-0.00000113	0.00000000
		-6.27296313	-0.00050356	0.00030853	0.00050611	-0.00000113	0.00000000
		-6.27296332	-0.00050347	0.00030848	0.00050602	-0.00000113	0.00000000
		-6.27296352	-0.00050339	0.00030843	0.00050594	-0.00000113	0.00000000
		-6.27296372	-0.00050331	0.00030838	0.00050586	-0.00000113	0.00000000
		-6.27296391	-0.00050323	0.00030833	0.00050578	-0.00000113	0.00000000
		-6.27296411	-0.00050315	0.00030828	0.00050569	-0.00000113	0.00000000
		-6.27296430	-0.00050306	0.00030823	0.00050561	-0.00000113	0.00000000
		-6.27296450	-0.00050298	0.00030818	0.00050553	-0.00000113	0.00000000
		-6.27296469	-0.00050290	0.00030813	0.00050545	-0.00000113	0.00000000
		-6.27296489	-0.00050282	0.00030808	0.00050537	-0.00000113	0.00000000
		-6.27296509	-0.00050274	0.00030803	0.00050528	-0.00000113	0.00000000
		-6.27296528	-0.00050266	0.00030798	0.00050520	-0.00000113	0.00000000
		-6.27296548	-0.00050257	0.00030793	0.00050512	-0.00000113	0.00000000
		-6.27296567	-0.00050249	0.00030788	0.00050504	-0.00000113	0.00000000
		-6.27296587	-0.00050241	0.00030783	0.00050496	-0.00000113	0.00000000
		-6.27296606	-0.00050233	0.00030778	0.00050487	-0.00000113	0.00000000
		-6.27296626	-0.00050225	0.00030773	0.00050479	-0.00000113	0.00000000
		-6.27296645	-0.00050217	0.00030768	0.00050471	-0.00000113	0.00000000
		-6.27296665	-0.00050209	0.00030763	0.00050463	-0.00000113	0.00000000
		-6.27296684	-0.00050200	0.00030758	0.00050455	-0.00000113	0.00000000
		-6.27296704	-0.00050192	0.00030753	0.00050446	-0.00000113	0.00000000
		-6.27296723	-0.00050184	0.00030748	0.00050438	-0.00000113	0.00000000
		-6.27296743	-0.00050176	0.00030743	0.00050430	-0.00000113	0.00000000
		-6.27296762	-0.00050168	0.00030738	0.00050422	-0.00000113	0.00000000
		-6.27296781	-0.00050160	0.00030733	0.00050414	-0.00000113	0.00000000
		-6.27296801	-0.00050152	0.00030728	0.00050406	-0.00000113	0.00000000
		-6.27296820	-0.00050143	0.00030723	0.00050397	-0.00000113	0.00000000
		-6.27296840	-0.00050135	0.00030718	0.00050389	-0.00000113	0.00000000
		-6.27296859	-0.00050127	0.00030713	0.00050381	-0.00000113	0.00000000
		-6.27296879	-0.00050119	0.00030708	0.00050373	-0.00000113	0.00000000
		-6.27296898	-0.00050111	0.00030703	0.00050365	-0.00000113	0.00000000
		-6.27296917	-0.00050103	0.00030698	0.00050357	-0.00000113	0.00000000
		-6.27296937	-0.00050095	0.00030693	0.00050348	-0.00000113	0.00000000
		-6.27296956	-0.00050087	0.00030688	0.00050340	-0.00000113	0.00000000
		-6.27296976	-0.00050079	0.00030683	0.00050332	-0.00000113	0.00000000
		-6.27296995	-0.00050070	0.00030678	0.00050324	-0.00000113	0.00000000
		-6.27297014	-0.00050062	0.00030673	0.00050316	-0.00000113	0.00000000
		-6.27297034	-0.00050054	0.00030668	0.00050308	-0.00000113	0.00000000
		-6.27297053	-0.00050046	0.00030663	0.00050300	-0.00000113	0.00000000
		-6.27297073	-0.00050038	0.00030658	0.00050291	-0.00000113	0.00000000
		-6.27297092	-0.00050030	0.00030653	0.00050283	-0.00000113	0.00000000
		-6.27297111	-0.00050022	0.00030648	0.00050275	-0.00000113	0.00000000
		-6.27297131	-0.00050014	0.00030643	0.00050267	-0.00000113	0.00000000
		-6.27297150	-0.00050006	0.00030638	0.00050259	-0.00000113	0.00000000
		-6.27297169	-0.00049998	0.00030633	0.00050251	-0.00000113	0.00000000
		-6.27297189	-0.00049990	0.00030629	0.00050243	-0.00000113	0.00000000
		-6.27297208	-0.00049982	0.00030624	0.00050235	-0.00000113	0.00000000
		-6.27297227	-0.00049973	0.00030619	0.00050227	-0.00000113	0.00000000
		-6.27297246	-0.00049965	0.00030614	0.00050218	-0.00000113	0.00000000
		-6.27297266	-0.00049957	0.00030609	0.00050210	-0.00000113	0.00000000
		-6.27297285	-0.00049949	0.00030604	0.00050202	-0.00000113	0.00000000
		-6.27297304	-0.00049941	0.00030599	0.00050194	-0.00000113	0.00000000
		-6.27297324	-0.00049933	0.00030594	0.00050186	-0.00000113	0.00000000
		-6.27297343	-0.00049925	0.00030589	0.00050178	-0.00000113	0.00000000
		-6.27297362	-0.00049917	0.00030584	0.00050170	-0.00000113	0.00000000
		-6.27297381	-0.00049909	0.00030579	0.00050162	-0.00000113	0.00000000
		-6.27297401	-0.00049901	0.00030574	0.00050154	-0.00000113	0.00000000
		-6.27297420	-0.00049893	0.00030569	0.00050146	-0.00000113	0.00000000
		-6.27297439	-0.00049885	0.00030564	0.00050137	-0.00000113	0.00000000
		-6.27297458	-0.00049877	0.00030559	0.00050129	-0.00000113	0.00000000
		-6.27297478	-0.00049869	0.00030554	0.00050121	-0.00000113	0.00000000
		-6.27297497	-0.00049861	0.00030550	0.00050113	-0.00000113	0.00000000
		-6.27297516	-0.00049853	0.00030545	0.00050105	-0.00000113	0.00000000
		-6.27297535	-0.00049845	0.00030540	0.00050097	-0.00000113	0.00000000
		-6.27297554	-0.00049837	0.00030535	0.00050089	-0.00000113	0.00000000
		-6.27297574	-0.00049829	0.00030530	0.00050081	-0.00000113	0.00000000
		-6.27297593	-0.00049821	0.00030525	0.00050073	-0.00000113	0.00000000
		-6.27297612	-0.00049813	0.00030520	0.00050065	-0.00000113	0.00000000
		-6.27297631	-0.00049805	0.00030515	0.00050057	-0.00000113	0.00000000
		-6.27297650	-0.00049797	0.00030510	0.00050049	-0.00000113	0.00000000
		-6.27297670	-0.00049789	0.00030505	0.00050041	-0.00000113	0.00000000
		-6.27297689	-0.00049781	0.00030500	0.00050033	-0.00000113	0.00000000
		-6.27297708	-0.00049773	0.00030495	0.00050025	-0.00000113	0.00000000
		-6.27297727	-0.00049765	0.00030491	0.00050017	-0.00000113	0.00000000
		-6.27297746	-0.00049757	0.00030486	0.00050009	-0.00000113	0.00000000
		-6.27297765	-0.00049749	0.00030481	0.00050000	-0.00000113	0.00000000
		-6.27297784	-0.00049741	0.00030476	0.00049992	-0.00000113	0.00000000
		-6.27297804	-0.00049733	0.00030471	0.00049984	-0.00000113	0.00000000
		-6.27297823	-0.00049725	0.00030466	0.00049976	-0.00000113	0.00000000
		-6.27297842	-0.00049717	0.00030461	0.00049968	-0.00000113	0.00000000
		-6.27297861	-0.00049709	0.00030456	0.00049960	-0.00000113	0.00000000
		-6.27297880	-0.00049701	0.00030451	0.00049952	-0.00000113	0.00000000
		-6.27297899	-0.00049693	0.00030446	0.00049944	-0.00000113	0.00000000
		-6.27297918	-0.00049685	0.00030442	0.00049936	-0.00000113	0.00000000
		-6.27297937	-0.00049677	0.00030437	0.00049928	-0.00000113	0.00000000
		-6.27297956	-0.00049669	0.00030432	0.00049920	-0.00000113	0.00000000
		-6.27297975	-0.00049661	0.00030427	0.00049912	-0.00000113	0.00000000
		-6.27297994	-0.00049653	0.00030422	0.00049904	-0.00000113	0.00000000
		-6.27298013	-0.00049645	0.00030417	0.00049896	-0.00000113	0.00000000
		-6.27298033	-0.00049637	0.00030412	0.00049888	-0.00000114	0.00000000
		-6.27298052	-0.00049629	0.00030407	0.00049880	-0.00000114	0.00000000
		-6.27298071	-0.00049621	0.00030403	0.00049872	-0.00000114	0.00000000
		-6.27298090	-0.00049613	0.00030398	0.00049864	-0.00000114	0.00000000
		-6.27298109	-0.00049605	0.00030393	0.00049856	-0.00000114	0.00000000
		-6.27298128	-0.00049597	0.00030388	0.00049848	-0.00000114	0.00000000
		-6.27298147	-0.00049589	0.00030383	0.00049840	-0.00000114	0.00000000
		-6.27298166	-0.00049581	0.00030378	0.00049832	-0.00000114	0.00000000
		-6.27298185	-0.00049573	0.00030373	0.00049824	-0.00000114	0.00000000
		-6.27298204	-0.00049565	0.00030368	0.00049816	-0.00000114	0.00000000
		-6.27298223	-0.00049557	0.00030364	0.00049808	-0.00000114	0.00000000
		-6.27298242	-0.00049550	0.00030359	0.00049800	-0.00000114	0.00000000
		-6.27298261	-0.00049542	0.00030354	0.00049792	-0.00000114	0.00000000
		-6.27298280	-0.00049534	0.00030349	0.00049784	-0.00000114	0.00000000
		-6.27298299	-0.00049526	0.00030344	0.00049776	-0.00000114	0.00000000
		-6.27298318	-0.00049518	0.00030339	0.00049769	-0.00000114	0.00000000
		-6.27298336	-0.00049510	0.00030334	0.00049761	-0.00000114	0.00000000
		-6.27298355	-0.00049502	0.00030330	0.00049753	-0.00000114	0.00000000
		-6.27298374	-0.00049494	0.00030325	0.00049745	-0.00000114	0.00000000
		-6.27298393	-0.00049486	0.00030320	0.00049737	-0.00000114	0.00000000
		-6.27298412	-0.00049478	0.00030315	0.00049729	-0.00000114	0.00000000
		-6.27298431	-0.00049470	0.00030310	0.00049721	-0.00000114	0.00000000
		-6.27298450	-0.00049462	0.00030305	0.00049713	-0.00000114	0.00000000
		-6.27298469	-0.00049455	0.00030300	0.00049705	-0.00000114	0.00000000
		-6.27298488	-0.00049447	0.00030296	0.00049697	-0.00000114	0.00000000
		-6.27298507	-0.00049439	0.00030291	0.00049689	-0.00000114	0.00000000
		-6.27298526	-0.00049431	0.00030286	0.00049681	-0.00000114	0.00000000
		-6.27298545	-0.00049423	0.00030281	0.00049673	-0.00000114	0.00000000
		-6.27298563	-0.00049415	0.00030276	0.00049665	-0.00000114	0.00000000
		-6.27298582	-0.00049407	0.00030271	0.00049657	-0.00000114	0.00000000
		-6.27298601	-0.00049399	0.00030267	0.00049649	-0.00000114	0.00000000
		-6.27298620	-0.00049391	0.00030262	0.00049641	-0.00000114	0.00000000
		-6.27298639	-0.00049384	0.00030257	0.00049634	-0.00000114	0.00000000
		-6.27298658	-0.00049376	0.00030252	0.00049626	-0.00000114	0.00000000
		-6.27298677	-0.00049368	0.00030247	0.00049618	-0.00000114	0.00000000
		-6.27298695	-0.00049360	0.00030242	0.00049610	-0.00000114	0.00000000
		-6.27298714	-0.00049352	0.00030238	0.00049602	-0.00000114	0.00000000
		-6.27298733	-0.00049344	0.00030233	0.00049594	-0.00000114	0.00000000
		-6.27298752	-0.00049336	0.00030228	0.00049586	-0.00000114	0.00000000
		-6.27298771	-0.00049328	0.00030223	0.00049578	-0.00000114	0.00000000
		-6.27298790	-0.00049321	0.00030218	0.00049570	-0.00000114	0.00000000
		-6.27298808	-0.00049313	0.00030214	0.00049562	-0.00000114	0.00000000
		-6.27298827	-0.00049305	0.00030209	0.00049554	-0.00000114	0.00000000
		-6.27298846	-0.00049297	0.00030204	0.00049547	-0.00000114	0.00000000
		-6.27298865	-0.00049289	0.00030199	0.00049539	-0.00000114	0.00000000
		-6.27298883	-0.00049281	0.00030194	0.00049531	-0.00000114	0.00000000
		-6.27298902	-0.00049273	0.00030189	0.00049523	-0.00000114	0.00000000
		-6.27298921	-0.00049266	0.00030185	0.00049515	-0.00000114	0.00000000
		-6.27298940	-0.00049258	0.00030180	0.00049507	-0.00000114	0.00000000
		-6.27298959	-0.00049250	0.00030175	0.00049499	-0.00000114	0.00000000
		-6.27298977	-0.00049242	0.00030170	0.00049491	-0.00000114	0.00000000
		-6.27298996	-0.00049234	0.00030165	0.00049484	-0.00000114	0.00000000
		-6.27299015	-0.00049226	0.00030161	0.00049476	-0.00000114	0.00000000
		-6.27299033	-0.00049219	0.00030156	0.00049468	-0.00000114	0.00000000
		-6.27299052	-0.00049211	0.00030151	0.00049460	-0.00000114	0.00000000
		-6.27299071	-0.00049203	0.00030146	0.00049452	-0.00000114	0.00000000
		-6.27299090	-0.00049195	0.00030141	0.00049444	-0.00000114	0.00000000
		-6.27299108	-0.00049187	0.00030137	0.00049436	-0.00000114	0.00000000
		-6.27299127	-0.00049180	0.00030132	0.00049428	-0.00000114	0.00000000
		-6.27299146	-0.00049172	0.00030127	0.00049421	-0.00000114	0.00000000
		-6.27299164	-0.00049164	0.00030122	0.00049413	-0.00000114	0.00000000
		-6.27299183	-0.00049156	0.00030118	0.00049405	-0.00000114	0.00000000
		-6.27299202	-0.00049148	0.00030113	0.00049397	-0.00000114	0.00000000
		-6.27299220	-0.00049140	0.00030108	0.00049389	-0.00000114	0.00000000
		-6.27299239	-0.00049133	0.00030103	0.00049381	-0.00000114	0.00000000
		-6.27299258	-0.00049125	0.00030098	0.00049374	-0.00000114	0.00000000
		-6.27299276	-0.00049117	0.00030094	0.00049366	-0.00000114	0.00000000
		-6.27299295	-0.00049109	0.00030089	0.00049358	-0.00000114	0.00000000
		-6.27299314	-0.00049102	0.00030084	0.00049350	-0.00000114	0.00000000
		-6.27299332	-0.00049094	0.00030079	0.00049342	-0.00000114	0.00000000
		-6.27299351	-0.00049086	0.00030075	0.00049334	-0.00000114	0.00000000
		-6.27299370	-0.00049078	0.00030070	0.00049327	-0.00000114	0.00000000
		-6.27299388	-0.00049070	0.00030065	0.00049319	-0.00000114	0.00000000
		-6.27299407	-0.00049063	0.00030060	0.00049311	-0.00000114	0.00000000
		-6.27299425	-0.00049055	0.00030055	0.00049303	-0.00000114	0.00000000
		-6.27299444	-0.00049047	0.00030051	0.00049295	-0.00000114	0.00000000
		-6.27299463	-0.00049039	0.00030046	0.00049288	-0.00000114	0.00000000
		-6.27299481	-0.00049032	0.00030041	0.00049280	-0.00000114	0.00000000
		-6.27299500	-0.00049024	0.00030036	0.00049272	-0.00000114	0.00000000
		-6.27299518	-0.00049016	0.00030032	0.00049264	-0.00000114	0.00000000
		-6.27299537	-0.00049008	0.00030027	0.00049256	-0.00000114	0.00000000
		-6.27299556	-0.00049000	0.00030022	0.00049249	-0.00000114	0.00000000
		-6.27299574	-0.00048993	0.00030017	0.00049241	-0.00000114	0.00000000
		-6.27299593	-0.00048985	0.00030013	0.00049233	-0.00000114	0.00000000
		-6.27299611	-0.00048977	0.00030008	0.00049225	-0.00000114	0.00000000
		-6.27299630	-0.00048969	0.00030003	0.00049217	-0.00000114	0.00000000
		-6.27299648	-0.00048962	0.00029998	0.00049210	-0.00000114	0.00000000
		-6.27299667	-0.00048954	0.00029994	0.00049202	-0.00000114	0.00000000
		-6.27299685	-0.00048946	0.00029989	0.00049194	-0.00000114	0.00000000
		-6.27299704	-0.00048938	0.00029984	0.00049186	-0.00000114	0.00000000
		-6.27299722	-0.00048931	0.00029979	0.00049178	-0.00000114	0.00000000
		-6.27299741	-0.00048923	0.00029975	0.00049171	-0.00000114	0.00000000
		-6.27299759	-0.00048915	0.00029970	0.00049163	-0.00000114	0.00000000
		-6.27299778	-0.00048908	0.00029965	0.00049155	-0.00000114	0.00000000
		-6.27299796	-0.00048900	0.00029960	0.00049147	-0.00000114	0.00000000
		-6.27299815	-0.00048892	0.00029956	0.00049140	-0.00000114	0.00000000
		-6.27299833	-0.00048884	0.00029951	0.00049132	-0.00000114	0.00000000
		-6.27299852	-0.00048877	0.00029946	0.00049124	-0.00000114	0.00000000
		-6.27299870	-0.00048869	0.00029941	0.00049116	-0.00000114	0.00000000
		-6.27299889	-0.00048861	0.00029937	0.00049109	-0.00000114	0.00000000
		-6.27299907	-0.00048854	0.00029932	0.00049101	-0.00000114	0.00000000
		-6.27299926	-0.00048846	0.00029927	0.00049093	-0.00000114	0.00000000
		-6.27299944	-0.00048838	0.00029923	0.00049085	-0.00000114	0.00000000
		-6.27299962	-0.00048830	0.00029918	0.00049078	-0.00000114	0.00000000
		-6.27299981	-0.00048823	0.00029913	0.00049070	-0.00000114	0.00000000
		-6.27299999	-0.00048815	0.00029908	0.00049062	-0.00000114	0.00000000
		-6.27300018	-0.00048807	0.00029904	0.00049054	-0.00000114	0.00000000
		-6.27300036	-0.00048800	0.00029899	0.00049047	-0.00000114	0.00000000
		-6.27300055	-0.00048792	0.00029894	0.00049039	-0.00000114	0.00000000
		-6.27300073	-0.00048784	0.00029890	0.00049031	-0.00000114	0.00000000
		-6.27300091	-0.00048777	0.00029885	0.00049023	-0.00000114	0.00000000
		-6.27300110	-0.00048769	0.00029880	0.00049016	-0.00000114	0.00000000
		-6.27300128	-0.00048761	0.00029875	0.00049008	-0.00000114	0.00000000
		-6.27300146	-0.00048754	0.00029871	0.00049000	-0.00000114	0.00000000
		-6.27300165	-0.00048746	0.00029866	0.00048993	-0.00000114	0.00000000
		-6.27300183	-0.00048738	0.00029861	0.00048985	-0.00000114	0.00000000
		-6.27300202	-0.00048730	0.00029857	0.00048977	-0.00000114	0.00000000
		-6.27300220	-0.00048723	0.00029852	0.00048969	-0.00000114	0.00000000
		-6.27300238	-0.00048715	0.00029847	0.00048962	-0.00000114	0.00000000
		-6.27300257	-0.00048707	0.00029842	0.00048954	-0.00000114	0.00000000
		-6.27300275	-0.00048700	0.00029838	0.00048946	-0.00000114	0.00000000
		-6.27300293	-0.00048692	0.00029833	0.00048939	-0.00000114	0.00000000
		-6.27300312	-0.00048685	0.00029828	0.00048931	-0.00000114	0.00000000
		-6.27300330	-0.00048677	0.00029824	0.00048923	-0.00000114	0.00000000
		-6.27300348	-0.00048669	0.00029819	0.00048916	-0.00000114	0.00000000
		-6.27300367	-0.00048662	0.00029814	0.00048908	-0.00000114	0.00000000
		-6.27300385	-0.00048654	0.00029810	0.00048900	-0.00000114	0.00000000
		-6.27300403	-0.00048646	0.00029805	0.00048892	-0.00000114	0.00000000
		-6.27300421	-0.00048639	0.00029800	0.00048885	-0.00000114	0.00000000
		-6.27300440	-0.00048631	0.00029796	0.00048877	-0.00000114	0.00000000
		-6.27300458	-0.00048623	0.00029791	0.00048869	-0.00000114	0.00000000
		-6.27300476	-0.00048616	0.00029786	0.00048862	-0.00000114	0.00000000
		-6.27300495	-0.00048608	0.00029782	0.00048854	-0.00000114	0.00000000
		-6.27300513	-0.00048600	0.00029777	0.00048846	-0.00000114	0.00000000
		-6.27300531	-0.00048593	0.00029772	0.00048839	-0.00000114	0.00000000
		-6.27300549	-0.00048585	0.00029768	0.00048831	-0.00000114	0.00000000
		-6.27300568	-0.00048578	0.00029763	0.00048823	-0.00000114	0.00000000
		-6.27300586	-0.00048570	0.00029758	0.00048816	-0.00000114	0.00000000
		-6.27300604	-0.00048562	0.00029753	0.00048808	-0.00000114	0.00000000
		-6.27300622	-0.00048555	0.00029749	0.00048800	-0.00000114	0.00000000
		-6.27300640	-0.00048547	0.00029744	0.00048793	-0.00000114	0.00000000
		-6.27300659	-0.00048539	0.00029739	0.00048785	-0.00000114	0.00000000
		-6.27300677	-0.00048532	0.00029735	0.00048777	-0.00000114	0.00000000
		-6.27300695	-0.00048524	0.00029730	0.00048770	-0.00000114	0.00000000
		-6.27300713	-0.00048517	0.00029725	0.00048762	-0.00000114	0.00000000
		-6.27300732	-0.00048509	0.00029721	0.00048755	-0.00000114	0.00000000
		-6.27300750	-0.00048501	0.00029716	0.00048747	-0.00000114	0.00000000
		-6.27300768	-0.00048494	0.00029712	0.00048739	-0.00000114	0.00000000
		-6.27300786	-0.00048486	0.00029707	0.00048732	-0.00000114	0.00000000
		-6.27300804	-0.00048479	0.00029702	0.00048724	-0.00000114	0.00000000
		-6.27300822	-0.00048471	0.00029698	0.00048716	-0.00000114	0.00000000
		-6.27300841	-0.00048463	0.00029693	0.00048709	-0.00000114	0.00000000
		-6.27300859	-0.00048456	0.00029688	0.00048701	-0.00000114	0.00000000
		-6.27300877	-0.00048448	0.00029684	0.00048693	-0.00000114	0.00000000
		-6.27300895	-0.00048441	0.00029679	0.00048686	-0.00000114	0.00000000
		-6.27300913	-0.00048433	0.00029674	0.00048678	-0.00000114	0.00000000
		-6.27300931	-0.00048426	0.00029670	0.00048671	-0.00000114	0.00000000
		-6.27300949	-0.00048418	0.00029665	0.00048663	-0.00000114	0.00000000
		-6.27300968	-0.00048410	0.00029660	0.00048655	-0.00000114	0.00000000
		-6.27300986	-0.00048403	0.00029656	0.00048648	-0.00000114	0.00000000
		-6.27301004	-0.00048395	0.00029651	0.00048640	-0.00000114	0.00000000
		-6.27301022	-0.00048388	0.00029646	0.00048633	-0.00000114	0.00000000
		-6.27301040	-0.00048380	0.00029642	0.00048625	-0.00000114	0.00000000
		-6.27301058	-0.00048373	0.00029637	0.00048617	-0.00000114	0.00000000
		-6.27301076	-0.00048365	0.00029633	0.00048610	-0.00000114	0.00000000
		-6.27301094	-0.00048357	0.00029628	0.00048602	-0.00000114	0.00000000
		-6.27301112	-0.00048350	0.00029623	0.00048595	-0.00000114	0.00000000
		-6.27301130	-0.00048342	0.00029619	0.00048587	-0.00000114	0.00000000
		-6.27301148	-0.00048335	0.00029614	0.00048579	-0.00000114	0.00000000
		-6.27301166	-0.00048327	0.00029609	0.00048572	-0.00000114	0.00000000
		-6.27301185	-0.00048320	0.00029605	0.00048564	-0.00000114	0.00000000
		-6.27301203	-0.00048312	0.00029600	0.00048557	-0.00000114	0.00000000
		-6.27301221	-0.00048305	0.00029596	0.00048549	-0.00000114	0.00000000
		-6.27301239	-0.00048297	0.00029591	0.00048542	-0.00000114	0.00000000
		-6.27301257	-0.00048290	0.00029586	0.00048534	-0.00000114	0.00000000
		-6.27301275	-0.00048282	0.00029582	0.00048526	-0.00000114	0.00000000
		-6.27301293	-0.00048275	0.00029577	0.00048519	-0.00000114	0.00000000
		-6.27301311	-0.00048267	0.00029572	0.00048511	-0.00000114	0.00000000
		-6.27301329	-0.00048259	0.00029568	0.00048504	-0.00000114	0.00000000
		-6.27301347	-0.00048252	0.00029563	0.00048496	-0.00000114	0.00000000
		-6.27301365	-0.00048244	0.00029559	0.00048489	-0.00000114	0.00000000
		-6.27301383	-0.00048237	0.00029554	0.00048481	-0.00000114	0.00000000
		-6.27301401	-0.00048229	0.00029549	0.00048473	-0.00000114	0.00000000
		-6.27301419	-0.00048222	0.00029545	0.00048466	-0.00000114	0.00000000
		-6.27301437	-0.00048214	0.00029540	0.00048458	-0.00000114	0.00000000
		-6.27301455	-0.00048207	0.00029536	0.00048451	-0.00000114	0.00000000
		-6.27301473	-0.00048199	0.00029531	0.00048443	-0.00000114	0.00000000
		-6.27301491	-0.00048192	0.00029526	0.00048436	-0.00000114	0.00000000
		-6.27301509	-0.00048184	0.00029522	0.00048428	-0.00000114	0.00000000
		-6.27301526	-0.00048177	0.00029517	0.00048421	-0.00000114	0.00000000
		-6.27301544	-0.00048169	0.00029513	0.00048413	-0.00000114	0.00000000
		-6.27301562	-0.00048162	0.00029508	0.00048406	-0.00000114	0.00000000
		-6.27301580	-0.00048154	0.00029503	0.00048398	-0.00000114	0.00000000
		-6.27301598	-0.00048147	0.00029499	0.00048390	-0.00000114	0.00000000
		-6.27301616	-0.00048139	0.00029494	0.00048383	-0.00000114	0.00000000
		-6.27301634	-0.00048132	0.00029490	0.00048375	-0.00000114	0.00000000
		-6.27301652	-0.00048124	0.00029485	0.00048368	-0.00000114	0.00000000
		-6.27301670	-0.00048117	0.00029480	0.00048360	-0.00000114	0.00000000
		-6.27301688	-0.00048109	0.00029476	0.00048353	-0.00000114	0.00000000
		-6.27301706	-0.00048102	0.00029471	0.00048345	-0.00000114	0.00000000
		-6.27301724	-0.00048095	0.00029467	0.00048338	-0.00000114	0.00000000
		-6.27301741	-0.00048087	0.00029462	0.00048330	-0.00000114	0.00000000
		-6.27301759	-0.00048080	0.00029458	0.00048323	-0.00000114	0.00000000
		-6.27301777	-0.00048072	0.00029453	0.00048315	-0.00000114	0.00000000
		-6.27301795	-0.00048065	0.00029448	0.00048308	-0.00000114	0.00000000
		-6.27301813	-0.00048057	0.00029444	0.00048300	-0.00000114	0.00000000
		-6.27301831	-0.00048050	0.00029439	0.00048293	-0.00000114	0.00000000
		-6.27301849	-0.00048042	0.00029435	0.00048285	-0.00000114	0.00000000
		-6.27301866	-0.00048035	0.00029430	0.00048278	-0.00000114	0.00000000
		-6.27301884	-0.00048027	0.00029426	0.00048270	-0.00000114	0.00000000
		-6.27301902	-0.00048020	0.00029421	0.00048263	-0.00000114	0.00000000
		-6.27301920	-0.00048012	0.00029416	0.00048255	-0.00000114	0.00000000
		-6.27301938	-0.00048005	0.00029412	0.00048248	-0.00000114	0.00000000
		-6.27301956	-0.00047998	0.00029407	0.00048240	-0.00000114	0.00000000
		-6.27301973	-0.00047990	0.00029403	0.00048233	-0.00000114	0.00000000
		-6.27301991	-0.00047983	0.00029398	0.00048225	-0.00000114	0.00000000
		-6.27302009	-0.00047975	0.00029394	0.00048218	-0.00000114	0.00000000
		-6.27302027	-0.00047968	0.00029389	0.00048211	-0.00000114	0.00000000
		-6.27302044	-0.00047960	0.00029384	0.00048203	-0.00000114	0.00000000
		-6.27302062	-0.00047953	0.00029380	0.00048196	-0.00000114	0.00000000
		-6.27302080	-0.00047946	0.00029375	0.00048188	-0.00000114	0.00000000
		-6.27302098	-0.00047938	0.00029371	0.00048181	-0.00000114	0.00000000
		-6.27302116	-0.00047931	0.00029366	0.00048173	-0.00000114	0.00000000
		-6.27302133	-0.00047923	0.00029362	0.00048166	-0.00000114	0.00000000
		-6.27302151	-0.00047916	0.00029357	0.00048158	-0.00000114	0.00000000
		-6.27302169	-0.00047908	0.00029353	0.00048151	-0.00000114	0.00000000
		-6.27302187	-0.00047901	0.00029348	0.00048143	-0.00000114	0.00000000
		-6.27302204	-0.00047894	0.00029344	0.00048136	-0.00000114	0.00000000
		-6.27302222	-0.00047886	0.00029339	0.00048128	-0.00000114	0.00000000
		-6.27302240	-0.00047879	0.00029334	0.00048121	-0.00000114	0.00000000
		-6.27302257	-0.00047871	0.00029330	0.00048114	-0.00000114	0.00000000
		-6.27302275	-0.00047864	0.00029325	0.00048106	-0.00000114	0.00000000
		-6.27302293	-0.00047857	0.00029321	0.00048099	-0.00000114	0.00000000
		-6.27302311	-0.00047849	0.00029316	0.00048091	-0.00000114	0.00000000
		-6.27302328	-0.00047842	0.00029312	0.00048084	-0.00000114	0.00000000
		-6.27302346	-0.00047834	0.00029307	0.00048076	-0.00000114	0.00000000
		-6.27302364	-0.00047827	0.00029303	0.00048069	-0.00000114	0.00000000
		-6.27302381	-0.00047820	0.00029298	0.00048062	-0.00000114	0.00000000
		-6.27302399	-0.00047812	0.00029294	0.00048054	-0.00000114	0.00000000
		-6.27302417	-0.00047805	0.00029289	0.00048047	-0.00000114	0.00000000
		-6.27302434	-0.00047797	0.00029285	0.00048039	-0.00000114	0.00000000
		-6.27302452	-0.00047790	0.00029280	0.00048032	-0.00000114	0.00000000
		-6.27302470	-0.00047783	0.00029276	0.00048024	-0.00000114	0.00000000
		-6.27302487	-0.00047775	0.00029271	0.00048017	-0.00000114	0.00000000
		-6.27302505	-0.00047768	0.00029267	0.00048010	-0.00000114	0.00000000
		-6.27302523	-0.00047761	0.00029262	0.00048002	-0.00000114	0.00000000
		-6.27302540	-0.00047753	0.00029257	0.00047995	-0.00000114	0.00000000
		-6.27302558	-0.00047746	0.00029253	0.00047987	-0.00000114	0.00000000
		-6.27302575	-0.00047739	0.00029248	0.00047980	-0.00000114	0.00000000
		-6.27302593	-0.00047731	0.00029244	0.00047973	-0.00000114	0.00000000
		-6.27302611	-0.00047724	0.00029239	0.00047965	-0.00000114	0.00000000
		-6.27302628	-0.00047716	0.00029235	0.00047958	-0.00000114	0.00000000
		-6.27302646	-0.00047709	0.00029230	0.00047950	-0.00000114	0.00000000
		-6.27302664	-0.00047702	0.00029226	0.00047943	-0.00000114	0.00000000
		-6.27302681	-0.00047694	0.00029221	0.00047936	-0.00000114	0.00000000
		-6.27302699	-0.00047687	0.00029217	0.00047928	-0.00000114	0.00000000
		-6.27302716	-0.00047680	0.00029212	0.00047921	-0.00000114	0.00000000
		-6.27302734	-0.00047672	0.00029208	0.00047913	-0.00000114	0.00000000
		-6.27302751	-0.00047665	0.00029203	0.00047906	-0.00000114	0.00000000
		-6.27302769	-0.00047658	0.00029199	0.00047899	-0.00000114	0.00000000
		-6.27302787	-0.00047650	0.00029194	0.00047891	-0.00000114	0.00000000
		-6.27302804	-0.00047643	0.00029190	0.00047884	-0.00000114	0.00000000
		-6.27302822	-0.00047636	0.00029185	0.00047877	-0.00000114	0.00000000
		-6.27302839	-0.00047628	0.00029181	0.00047869	-0.00000114	0.00000000
		-6.27302857	-0.00047621	0.00029176	0.00047862	-0.00000114	0.00000000
		-6.27302874	-0.00047614	0.00029172	0.00047855	-0.00000114	0.00000000
		-6.27302892	-0.00047606	0.00029167	0.00047847	-0.00000114	0.00000000
		-6.27302909	-0.00047599	0.00029163	0.00047840	-0.00000114	0.00000000
		-6.27302927	-0.00047592	0.00029158	0.00047832	-0.00000114	0.00000000
		-6.27302944	-0.00047584	0.00029154	0.00047825	-0.00000114	0.00000000
		-6.27302962	-0.00047577	0.00029150	0.00047818	-0.00000114	0.00000000
		-6.27302979	-0.00047570	0.00029145	0.00047810	-0.00000114	0.00000000
		-6.27302997	-0.00047562	0.00029141	0.00047803	-0.00000114	0.00000000
		-6.27303014	-0.00047555	0.00029136	0.00047796	-0.00000114	0.00000000
		-6.27303032	-0.00047548	0.00029132	0.00047788	-0.00000114	0.00000000
		-6.27303049	-0.00047541	0.00029127	0.00047781	-0.00000114	0.00000000
		-6.27303067	-0.00047533	0.00029123	0.00047774	-0.00000114	0.00000000
		-6.27303084	-0.00047526	0.00029118	0.00047766	-0.00000114	0.00000000
		-6.27303102	-0.00047519	0.00029114	0.00047759	-0.00000114	0.00000000
		-6.27303119	-0.00047511	0.00029109	0.00047752	-0.00000114	0.00000000
		-6.27303137	-0.00047504	0.00029105	0.00047744	-0.00000114	0.00000000
		-6.27303154	-0.00047497	0.00029100	0.00047737	-0.00000114	0.00000000
		-6.27303171	-0.00047490	0.00029096	0.00047730	-0.00000114	0.00000000
		-6.27303189	-0.00047482	0.00029091	0.00047722	-0.00000114	0.00000000
		-6.27303206	-0.00047475	0.00029087	0.00047715	-0.00000114	0.00000000
		-6.27303224	-0.00047468	0.00029082	0.00047708	-0.00000114	0.00000000
		-6.27303241	-0.00047460	0.00029078	0.00047700	-0.00000114	0.00000000
		-6.27303259	-0.00047453	0.00029073	0.00047693	-0.00000114	0.00000000
		-6.27303276	-0.00047446	0.00029069	0.00047686	-0.00000114	0.00000000
		-6.27303293	-0.00047439	0.00029065	0.00047678	-0.00000114	0.00000000
		-6.27303311	-0.00047431	0.00029060	0.00047671	-0.00000114	0.00000000
		-6.27303328	-0.00047424	0.00029056	0.00047664	-0.00000114	0.00000000
		-6.27303345	-0.00047417	0.00029051	0.00047657	-0.00000114	0.00000000
		-6.27303363	-0.00047410	0.00029047	0.00047649	-0.00000114	0.00000000
		-6.27303380	-0.00047402	0.00029042	0.00047642	-0.00000114	0.00000000
		-6.27303398	-0.00047395	0.00029038	0.00047635	-0.00000114	0.00000000
		-6.27303415	-0.00047388	0.00029033	0.00047627	-0.00000114	0.00000000
		-6.27303432	-0.00047380	0.00029029	0.00047620	-0.00000114	0.00000000
		-6.27303450	-0.00047373	0.00029025	0.00047613	-0.00000114	0.00000000
		-6.27303467	-0.00047366	0.00029020	0.00047606	-0.00000114	0.00000000
		-6.27303484	-0.00047359	0.00029016	0.00047598	-0.00000114	0.00000000
		-6.27303502	-0.00047351	0.00029011	0.00047591	-0.00000114	0.00000000
		-6.27303519	-0.00047344	0.00029007	0.00047584	-0.00000114	0.00000000
		-6.27303536	-0.00047337	0.00029002	0.00047576	-0.00000114	0.00000000
		-6.27303554	-0.00047330	0.00028998	0.00047569	-0.00000114	0.00000000
		-6.27303571	-0.00047323	0.00028993	0.00047562	-0.00000114	0.00000000
		-6.27303588	-0.00047315	0.00028989	0.00047555	-0.00000114	0.00000000
		-6.27303606	-0.00047308	0.00028985	0.00047547	-0.00000114	0.00000000
		-6.27303623	-0.00047301	0.00028980	0.00047540	-0.00000114	0.00000000
		-6.27303640	-0.00047294	0.00028976	0.00047533	-0.00000114	0.00000000
		-6.27303658	-0.00047286	0.00028971	0.00047526	-0.00000114	0.00000000
		-6.27303675	-0.00047279	0.00028967	0.00047518	-0.00000114	0.00000000
		-6.27303692	-0.00047272	0.00028962	0.00047511	-0.00000114	0.00000000
		-6.27303709	-0.00047265	0.00028958	0.00047504	-0.00000114	0.00000000
		-6.27303727	-0.00047258	0.00028954	0.00047497	-0.00000114	0.00000000
		-6.27303744	-0.00047250	0.00028949	0.00047489	-0.00000114	0.00000000
		-6.27303761	-0.00047243	0.00028945	0.00047482	-0.00000114	0.00000000
		-6.27303778	-0.00047236	0.00028940	0.00047475	-0.00000114	0.00000000
		-6.27303796	-0.00047229	0.00028936	0.00047468	-0.00000114	0.00000000
		-6.27303813	-0.00047221	0.00028931	0.00047460	-0.00000114	0.00000000
		-6.27303830	-0.00047214	0.00028927	0.00047453	-0.00000114	0.00000000
		-6.27303847	-0.00047207	0.00028923	0.00047446	-0.00000114	0.00000000
		-6.27303865	-0.00047200	0.00028918	0.00047439	-0.00000114	0.00000000
		-6.27303882	-0.00047193	0.00028914	0.00047431	-0.00000114	0.00000000
		-6.27303899	-0.00047185	0.00028909	0.00047424	-0.00000114	0.00000000
		-6.27303916	-0.00047178	0.00028905	0.00047417	-0.00000114	0.00000000
		-6.27303933	-0.00047171	0.00028901	0.00047410	-0.00000114	0.00000000
		-6.27303951	-0.00047164	0.00028896	0.00047402	-0.00000114	0.00000000
		-6.27303968	-0.00047157	0.00028892	0.00047395	-0.00000114	0.00000000
		-6.27303985	-0.00047150	0.00028887	0.00047388	-0.00000114	0.00000000
		-6.27304002	-0.00047142	0.00028883	0.00047381	-0.00000114	0.00000000
		-6.27304019	-0.00047135	0.00028879	0.00047374	-0.00000114	0.00000000
		-6.27304037	-0.00047128	0.00028874	0.00047366	-0.00000114	0.00000000
		-6.27304054	-0.00047121	0.00028870	0.00047359	-0.00000114	0.00000000
		-6.27304071	-0.00047114	0.00028865	0.00047352	-0.00000114	0.00000000
		-6.27304088	-0.00047106	0.00028861	0.00047345	-0.00000114	0.00000000
		-6.27304105	-0.00047099	0.00028857	0.00047337	-0.00000114	0.00000000
		-6.27304122	-0.00047092	0.00028852	0.00047330	-0.00000114	0.00000000
		-6.27304139	-0.00047085	0.00028848	0.00047323	-0.00000114	0.00000000
		-6.27304157	-0.00047078	0.00028843	0.00047316	-0.00000114	0.00000000
		-6.27304174	-0.00047071	0.00028839	0.00047309	-0.00000114	0.00000000
		-6.27304191	-0.00047064	0.00028835	0.00047302	-0.00000114	0.00000000
		-6.27304208	-0.00047056	0.00028830	0.00047294	-0.00000114	0.00000000
		-6.27304225	-0.00047049	0.00028826	0.00047287	-0.00000114	0.00000000
		-6.27304242	-0.00047042	0.00028822	0.00047280	-0.00000114	0.00000000
		-6.27304259	-0.00047035	0.00028817	0.00047273	-0.00000114	0.00000000
		-6.27304276	-0.00047028	0.00028813	0.00047266	-0.00000114	0.00000000
		-6.27304294	-0.00047021	0.00028808	0.00047258	-0.00000114	0.00000000
		-6.27304311	-0.00047013	0.00028804	0.00047251	-0.00000114	0.00000000
		-6.27304328	-0.00047006	0.00028800	0.00047244	-0.00000114	0.00000000
		-6.27304345	-0.00046999	0.00028795	0.00047237	-0.00000114	0.00000000
		-6.27304362	-0.00046992	0.00028791	0.00047230	-0.00000114	0.00000000
		-6.27304379	-0.00046985	0.00028787	0.00047223	-0.00000114	0.00000000
		-6.27304396	-0.00046978	0.00028782	0.00047215	-0.00000114	0.00000000
		-6.27304413	-0.00046971	0.00028778	0.00047208	-0.00000114	0.00000000
		-6.27304430	-0.00046964	0.00028773	0.00047201	-0.00000114	0.00000000
		-6.27304447	-0.00046956	0.00028769	0.00047194	-0.00000114	0.00000000
		-6.27304464	-0.00046949	0.00028765	0.00047187	-0.00000114	0.00000000
		-6.27304481	-0.00046942	0.00028760	0.00047180	-0.00000114	0.00000000
		-6.27304498	-0.00046935	0.00028756	0.00047172	-0.00000114	0.00000000
		-6.27304515	-0.00046928	0.00028752	0.00047165	-0.00000114	0.00000000
		-6.27304532	-0.00046921	0.00028747	0.00047158	-0.00000114	0.00000000
		-6.27304549	-0.00046914	0.00028743	0.00047151	-0.00000114	0.00000000
		-6.27304566	-0.00046907	0.00028738	0.00047144	-0.00000114	0.00000000
		-6.27304583	-0.00046900	0.00028734	0.00047137	-0.00000114	0.00000000
		-6.27304600	-0.00046892	0.00028730	0.00047130	-0.00000114	0.00000000
		-6.27304617	-0.00046885	0.00028725	0.00047122	-0.00000114	0.00000000
		-6.27304634	-0.00046878	0.00028721	0.00047115	-0.00000114	0.00000000
		-6.27304651	-0.00046871	0.00028717	0.00047108	-0.00000114	0.00000000
		-6.27304668	-0.00046864	0.00028712	0.00047101	-0.00000114	0.00000000
		-6.27304685	-0.00046857	0.00028708	0.00047094	-0.00000114	0.00000000
		-6.27304702	-0.00046850	0.00028704	0.00047087	-0.00000114	0.00000000
		-6.27304719	-0.00046843	0.00028699	0.00047080	-0.00000114	0.00000000
		-6.27304736	-0.00046836	0.00028695	0.00047072	-0.00000114	0.00000000
		-6.27304753	-0.00046829	0.00028691	0.00047065	-0.00000114	0.00000000
		-6.27304770	-0.00046821	0.00028686	0.00047058	-0.00000114	0.00000000
		-6.27304787	-0.00046814	0.00028682	0.00047051	-0.00000114	0.00000000
		-6.27304804	-0.00046807	0.00028678	0.00047044	-0.00000114	0.00000000
		-6.27304821	-0.00046800	0.00028673	0.00047037	-0.00000114	0.00000000
		-6.27304838	-0.00046793	0.00028669	0.00047030	-0.00000114	0.00000000
		-6.27304855	-0.00046786	0.00028665	0.00047023	-0.00000114	0.00000000
		-6.27304872	-0.00046779	0.00028660	0.00047016	-0.00000114	0.00000000
		-6.27304889	-0.00046772	0.00028656	0.00047008	-0.00000114	0.00000000
		-6.27304906	-0.00046765	0.00028652	0.00047001	-0.00000114	0.00000000
		-6.27304922	-0.00046758	0.00028647	0.00046994	-0.00000114	0.00000000
		-6.27304939	-0.00046751	0.00028643	0.00046987	-0.00000114	0.00000000
		-6.27304956	-0.00046744	0.00028639	0.00046980	-0.00000114	0.00000000
		-6.27304973	-0.00046737	0.00028634	0.00046973	-0.00000114	0.00000000
		-6.27304990	-0.00046730	0.00028630	0.00046966	-0.00000114	0.00000000
		-6.27305007	-0.00046723	0.00028626	0.00046959	-0.00000114	0.00000000
		-6.27305024	-0.00046715	0.00028621	0.00046952	-0.00000114	0.00000000
		-6.27305041	-0.00046708	0.00028617	0.00046945	-0.00000114	0.00000000
		-6.27305057	-0.00046701	0.00028613	0.00046938	-0.00000114	0.00000000
		-6.27305074	-0.00046694	0.00028608	0.00046930	-0.00000114	0.00000000
		-6.27305091	-0.00046687	0.00028604	0.00046923	-0.00000114	0.00000000
		-6.27305108	-0.00046680	0.00028600	0.00046916	-0.00000114	0.00000000
		-6.27305125	-0.00046673	0.00028595	0.00046909	-0.00000114	0.00000000
		-6.27305142	-0.00046666	0.00028591	0.00046902	-0.00000114	0.00000000
		-6.27305159	-0.00046659	0.00028587	0.00046895	-0.00000114	0.00000000
		-6.27305175	-0.00046652	0.00028582	0.00046888	-0.00000114	0.00000000
		-6.27305192	-0.00046645	0.00028578	0.00046881	-0.00000114	0.00000000
		-6.27305209	-0.00046638	0.00028574	0.00046874	-0.00000114	0.00000000
		-6.27305226	-0.00046631	0.00028570	0.00046867	-0.00000114	0.00000000
		-6.27305243	-0.00046624	0.00028565	0.00046860	-0.00000114	0.00000000
		-6.27305259	-0.00046617	0.00028561	0.00046853	-0.00000114	0.00000000
		-6.27305276	-0.00046610	0.00028557	0.00046846	-0.00000114	0.00000000
		-6.27305293	-0.00046603	0.00028552	0.00046839	-0.00000114	0.00000000
		-6.27305310	-0.00046596	0.00028548	0.00046832	-0.00000114	0.00000000
		-6.27305327	-0.00046589	0.00028544	0.00046824	-0.00000114	0.00000000
		-6.27305343	-0.00046582	0.00028539	0.00046817	-0.00000114	0.00000000
		-6.27305360	-0.00046575	0.00028535	0.00046810	-0.00000114	0.00000000
		-6.27305377	-0.00046568	0.00028531	0.00046803	-0.00000114	0.00000000
		-6.27305394	-0.00046561	0.00028527	0.00046796	-0.00000114	0.00000000
		-6.27305410	-0.00046554	0.00028522	0.00046789	-0.00000114	0.00000000
		-6.27305427	-0.00046547	0.00028518	0.00046782	-0.00000114	0.00000000
		-6.27305444	-0.00046540	0.00028514	0.00046775	-0.00000114	0.00000000
		-6.27305461	-0.00046533	0.00028509	0.00046768	-0.00000114	0.00000000
		-6.27305477	-0.00046526	0.00028505	0.00046761	-0.00000114	0.00000000
		-6.27305494	-0.00046519	0.00028501	0.00046754	-0.00000114	0.00000000
		-6.27305511	-0.00046512	0.00028497	0.00046747	-0.00000114	0.00000000
		-6.27305528	-0.00046505	0.00028492	0.00046740	-0.00000114	0.00000000
		-6.27305544	-0.00046498	0.00028488	0.00046733	-0.00000114	0.00000000
		-6.27305561	-0.00046491	0.00028484	0.00046726	-0.00000114	0.00000000
		-6.27305578	-0.00046484	0.00028479	0.00046719	-0.00000114	0.00000000
		-6.27305594	-0.00046477	0.00028475	0.00046712	-0.00000114	0.00000000
		-6.27305611	-0.00046470	0.00028471	0.00046705	-0.00000114	0.00000000
		-6.27305628	-0.00046463	0.00028467	0.00046698	-0.00000114	0.00000000
		-6.27305645	-0.00046456	0.00028462	0.00046691	-0.00000114	0.00000000
		-6.27305661	-0.00046449	0.00028458	0.00046684	-0.00000114	0.00000000
		-6.27305678	-0.00046442	0.00028454	0.00046677	-0.00000114	0.00000000
		-6.27305695	-0.00046435	0.00028449	0.00046670	-0.00000114	0.00000000
		-6.27305711	-0.00046428	0.00028445	0.00046663	-0.00000114	0.00000000
		-6.27305728	-0.00046421	0.00028441	0.00046656	-0.00000114	0.00000000
		-6.27305745	-0.00046414	0.00028437	0.00046649	-0.00000114	0.00000000
		-6.27305761	-0.00046407	0.00028432	0.00046642	-0.00000114	0.00000000
		-6.27305778	-0.00046400	0.00028428	0.00046635	-0.00000114	0.00000000
		-6.27305795	-0.00046393	0.00028424	0.00046628	-0.00000114	0.00000000
		-6.27305811	-0.00046386	0.00028420	0.00046621	-0.00000114	0.00000000
		-6.27305828	-0.00046380	0.00028415	0.00046614	-0.00000114	0.00000000
		-6.27305844	-0.00046373	0.00028411	0.00046607	-0.00000114	0.00000000
		-6.27305861	-0.00046366	0.00028407	0.00046600	-0.00000114	0.00000000
		-6.27305878	-0.00046359	0.00028403	0.00046593	-0.00000114	0.00000000
		-6.27305894	-0.00046352	0.00028398	0.00046586	-0.00000114	0.00000000
		-6.27305911	-0.00046345	0.00028394	0.00046579	-0.00000114	0.00000000
		-6.27305927	-0.00046338	0.00028390	0.00046572	-0.00000114	0.00000000
		-6.27305944	-0.00046331	0.00028386	0.00046565	-0.00000114	0.00000000
		-6.27305961	-0.00046324	0.00028381	0.00046558	-0.00000114	0.00000000
		-6.27305977	-0.00046317	0.00028377	0.00046551	-0.00000114	0.00000000
		-6.27305994	-0.00046310	0.00028373	0.00046544	-0.00000114	0.00000000
		-6.27306010	-0.00046303	0.00028369	0.00046537	-0.00000114	0.00000000
		-6.27306027	-0.00046296	0.00028364	0.00046530	-0.00000114	0.00000000
		-6.27306044	-0.00046289	0.00028360	0.00046523	-0.00000114	0.00000000
		-6.27306060	-0.00046282	0.00028356	0.00046516	-0.00000114	0.00000000
		-6.27306077	-0.00046276	0.00028352	0.00046509	-0.00000114	0.00000000
		-6.27306093	-0.00046269	0.00028347	0.00046502	-0.00000114	0.00000000
		-6.27306110	-0.00046262	0.00028343	0.00046496	-0.00000114	0.00000000
		-6.27306126	-0.00046255	0.00028339	0.00046489	-0.00000114	0.00000000
		-6.27306143	-0.00046248	0.00028335	0.00046482	-0.00000114	0.00000000
		-6.27306159	-0.00046241	0.00028330	0.00046475	-0.00000114	0.00000000
		-6.27306176	-0.00046234	0.00028326	0.00046468	-0.00000114	0.00000000
		-6.27306192	-0.00046227	0.00028322	0.00046461	-0.00000114	0.00000000
		-6.27306209	-0.00046220	0.00028318	0.00046454	-0.00000114	0.00000000
		-6.27306225	-0.00046213	0.00028314	0.00046447	-0.00000114	0.00000000
		-6.27306242	-0.00046206	0.00028309	0.00046440	-0.00000114	0.00000000
		-6.27306259	-0.00046200	0.00028305	0.00046433	-0.00000114	0.00000000
		-6.27306275	-0.00046193	0.00028301	0.00046426	-0.00000114	0.00000000
		-6.27306292	-0.00046186	0.00028297	0.00046419	-0.00000114	0.00000000
		-6.27306308	-0.00046179	0.00028292	0.00046412	-0.00000114	0.00000000
		-6.27306324	-0.00046172	0.00028288	0.00046405	-0.00000114	0.00000000
		-6.27306341	-0.00046165	0.00028284	0.00046398	-0.00000114	0.00000000
		-6.27306357	-0.00046158	0.00028280	0.00046392	-0.00000114	0.00000000
		-6.27306374	-0.00046151	0.00028275	0.00046385	-0.00000114	0.00000000
		-6.27306390	-0.00046144	0.00028271	0.00046378	-0.00000114	0.00000000
		-6.27306407	-0.00046138	0.00028267	0.00046371	-0.00000114	0.00000000
		-6.27306423	-0.00046131	0.00028263	0.00046364	-0.00000114	0.00000000
		-6.27306440	-0.00046124	0.00028259	0.00046357	-0.00000114	0.00000000
		-6.27306456	-0.00046117	0.00028254	0.00046350	-0.00000114	0.00000000
		-6.27306473	-0.00046110	0.00028250	0.00046343	-0.00000114	0.00000000
		-6.27306489	-0.00046103	0.00028246	0.00046336	-0.00000114	0.00000000
		-6.27306505	-0.00046096	0.00028242	0.00046329	-0.00000114	0.00000000
		-6.27306522	-0.00046090	0.00028238	0.00046322	-0.00000114	0.00000000
		-6.27306538	-0.00046083	0.00028233	0.00046316	-0.00000114	0.00000000
		-6.27306555	-0.00046076	0.00028229	0.00046309	-0.00000114	0.00000000
		-6.27306571	-0.00046069	0.00028225	0.00046302	-0.00000114	0.00000000
		-6.27306588	-0.00046062	0.00028221	0.00046295	-0.00000114	0.00000000
		-6.27306604	-0.00046055	0.00028217	0.00046288	-0.00000114	0.00000000
		-6.27306620	-0.00046048	0.00028212	0.00046281	-0.00000114	0.00000000
		-6.27306637	-0.00046042	0.00028208	0.00046274	-0.00000114	0.00000000
		-6.27306653	-0.00046035	0.00028204	0.00046267	-0.00000114	0.00000000
		-6.27306670	-0.00046028	0.00028200	0.00046260	-0.00000114	0.00000000
		-6.27306686	-0.00046021	0.00028196	0.00046254	-0.00000114	0.00000000
		-6.27306702	-0.00046014	0.00028191	0.00046247	-0.00000114	0.00000000
		-6.27306719	-0.00046007	0.00028187	0.00046240	-0.00000114	0.00000000
		-6.27306735	-0.00046000	0.00028183	0.00046233	-0.00000114	0.00000000
		-6.27306751	-0.00045994	0.00028179	0.00046226	-0.00000114	0.00000000
		-6.27306768	-0.00045987	0.00028175	0.00046219	-0.00000114	0.00000000
		-6.27306784	-0.00045980	0.00028170	0.00046212	-0.00000114	0.00000000
		-6.27306800	-0.00045973	0.00028166	0.00046205	-0.00000114	0.00000000
		-6.27306817	-0.00045966	0.00028162	0.00046199	-0.00000114	0.00000000
		-6.27306833	-0.00045959	0.00028158	0.00046192	-0.00000114	0.00000000
		-6.27306849	-0.00045953	0.00028154	0.00046185	-0.00000114	0.00000000
		-6.27306866	-0.00045946	0.00028150	0.00046178	-0.00000114	0.00000000
		-6.27306882	-0.00045939	0.00028145	0.00046171	-0.00000114	0.00000000
		-6.27306898	-0.00045932	0.00028141	0.00046164	-0.00000114	0.00000000
		-6.27306915	-0.00045925	0.00028137	0.00046158	-0.00000114	0.00000000
		-6.27306931	-0.00045919	0.00028133	0.00046151	-0.00000114	0.00000000
		-6.27306947	-0.00045912	0.00028129	0.00046144	-0.00000114	0.00000000
		-6.27306964	-0.00045905	0.00028124	0.00046137	-0.00000114	0.00000000
		-6.27306980	-0.00045898	0.00028120	0.00046130	-0.00000114	0.00000000
		-6.27306996	-0.00045891	0.00028116	0.00046123	-0.00000114	0.00000000
		-6.27307012	-0.00045885	0.00028112	0.00046116	-0.00000114	0.00000000
		-6.27307029	-0.00045878	0.00028108	0.00046110	-0.00000114	0.00000000
		-6.27307045	-0.00045871	0.00028104	0.00046103	-0.00000114	0.00000000
		-6.27307061	-0.00045864	0.00028099	0.00046096	-0.00000114	0.00000000
		-6.27307078	-0.00045857	0.00028095	0.00046089	-0.00000114	0.00000000
		-6.27307094	-0.00045851	0.00028091	0.00046082	-0.00000114	0.00000000
		-6.27307110	-0.00045844	0.00028087	0.00046075	-0.00000114	0.00000000
		-6.27307126	-0.00045837	0.00028083	0.00046069	-0.00000114	0.00000000
		-6.27307142	-0.00045830	0.00028079	0.00046062	-0.00000114	0.00000000
		-6.27307159	-0.00045823	0.00028074	0.00046055	-0.00000114	0.00000000
		-6.27307175	-0.00045817	0.00028070	0.00046048	-0.00000114	0.00000000
		-6.27307191	-0.00045810	0.00028066	0.00046041	-0.00000114	0.00000000
		-6.27307207	-0.00045803	0.00028062	0.00046035	-0.00000114	0.00000000
		-6.27307224	-0.00045796	0.00028058	0.00046028	-0.00000114	0.00000000
		-6.27307240	-0.00045790	0.00028054	0.00046021	-0.00000114	0.00000000
		-6.27307256	-0.00045783	0.00028050	0.00046014	-0.00000114	0.00000000
		-6.27307272	-0.00045776	0.00028045	0.00046007	-0.00000114	0.00000000
		-6.27307288	-0.00045769	0.00028041	0.00046001	-0.00000114	0.00000000
		-6.27307305	-0.00045762	0.00028037	0.00045994	-0.00000114	0.00000000
		-6.27307321	-0.00045756	0.00028033	0.00045987	-0.00000114	0.00000000
		-6.27307337	-0.00045749	0.00028029	0.00045980	-0.00000114	0.00000000
		-6.27307353	-0.00045742	0.00028025	0.00045973	-0.00000114	0.00000000
		-6.27307369	-0.00045735	0.00028021	0.00045967	-0.00000114	0.00000000
		-6.27307386	-0.00045729	0.00028016	0.00045960	-0.00000114	0.00000000
		-6.27307402	-0.00045722	0.00028012	0.00045953	-0.00000114	0.00000000
		-6.27307418	-0.00045715	0.00028008	0.00045946	-0.00000114	0.00000000
		-6.27307434	-0.00045708	0.00028004	0.00045939	-0.00000114	0.00000000
		-6.27307450	-0.00045702	0.00028000	0.00045933	-0.00000114	0.00000000
		-6.27307466	-0.00045695	0.00027996	0.00045926	-0.00000114	0.00000000
		-6.27307482	-0.00045688	0.00027992	0.00045919	-0.00000114	0.00000000
		-6.27307499	-0.00045681	0.00027987	0.00045912	-0.00000114	0.00000000
		-6.27307515	-0.00045675	0.00027983	0.00045905	-0.00000114	0.00000000
		-6.27307531	-0.00045668	0.00027979	0.00045899	-0.00000114	0.00000000
		-6.27307547	-0.00045661	0.00027975	0.00045892	-0.00000114	0.00000000
		-6.27307563	-0.00045654	0.00027971	0.00045885	-0.00000114	0.00000000
		-6.27307579	-0.00045648	0.00027967	0.00045878	-0.00000114	0.00000000
		-6.27307595	-0.00045641	0.00027963	0.00045872	-0.00000114	0.00000000
		-6.27307611	-0.00045634	0.00027959	0.00045865	-0.00000114	0.00000000
		-6.27307628	-0.00045628	0.00027954	0.00045858	-0.00000114	0.00000000
		-6.27307644	-0.00045621	0.00027950	0.00045851	-0.00000114	0.00000000
		-6.27307660	-0.00045614	0.00027946	0.00045845	-0.00000114	0.00000000
		-6.27307676	-0.00045607	0.00027942	0.00045838	-0.00000114	0.00000000
		-6.27307692	-0.00045601	0.00027938	0.00045831	-0.00000114	0.00000000
		-6.27307708	-0.00045594	0.00027934	0.00045824	-0.00000114	0.00000000
		-6.27307724	-0.00045587	0.00027930	0.00045818	-0.00000114	0.00000000
		-6.27307740	-0.00045581	0.00027926	0.00045811	-0.00000114	0.00000000
		-6.27307756	-0.00045574	0.00027921	0.00045804	-0.00000114	0.00000000
		-6.27307772	-0.00045567	0.00027917	0.00045797	-0.00000114	0.00000000
		-6.27307788	-0.00045560	0.00027913	0.00045791	-0.00000114	0.00000000
		-6.27307804	-0.00045554	0.00027909	0.00045784	-0.00000114	0.00000000
		-6.27307820	-0.00045547	0.00027905	0.00045777	-0.00000114	0.00000000
		-6.27307836	-0.00045540	0.00027901	0.00045770	-0.00000114	0.00000000
		-6.27307852	-0.00045534	0.00027897	0.00045764	-0.00000114	0.00000000
		-6.27307868	-0.00045527	0.00027893	0.00045757	-0.00000114	0.00000000
		-6.27307884	-0.00045520	0.00027889	0.00045750	-0.00000114	0.00000000
		-6.27307900	-0.00045514	0.00027884	0.00045743	-0.00000114	0.00000000
		-6.27307916	-0.00045507	0.00027880	0.00045737	-0.00000114	0.00000000
		-6.27307932	-0.00045500	0.00027876	0.00045730	-0.00000114	0.00000000
		-6.27307949	-0.00045493	0.00027872	0.00045723	-0.00000114	0.00000000
		-6.27307964	-0.00045487	0.00027868	0.00045717	-0.00000114	0.00000000
		-6.27307980	-0.00045480	0.00027864	0.00045710	-0.00000114	0.00000000
		-6.27307996	-0.00045473	0.00027860	0.00045703	-0.00000114	0.00000000
		-6.27308012	-0.00045467	0.00027856	0.00045696	-0.00000114	0.00000000
		-6.27308028	-0.00045460	0.00027852	0.00045690	-0.00000114	0.00000000
		-6.27308044	-0.00045453	0.00027848	0.00045683	-0.00000114	0.00000000
		-6.27308060	-0.00045447	0.00027844	0.00045676	-0.00000114	0.00000000
		-6.27308076	-0.00045440	0.00027839	0.00045670	-0.00000114	0.00000000
		-6.27308092	-0.00045433	0.00027835	0.00045663	-0.00000114	0.00000000
		-6.27308108	-0.00045427	0.00027831	0.00045656	-0.00000114	0.00000000
		-6.27308124	-0.00045420	0.00027827	0.00045650	-0.00000114	0.00000000
		-6.27308140	-0.00045413	0.00027823	0.00045643	-0.00000114	0.00000000
		-6.27308156	-0.00045407	0.00027819	0.00045636	-0.00000114	0.00000000
		-6.27308172	-0.00045400	0.00027815	0.00045629	-0.00000114	0.00000000
		-6.27308188	-0.00045393	0.00027811	0.00045623	-0.00000114	0.00000000
		-6.27308204	-0.00045387	0.00027807	0.00045616	-0.00000114	0.00000000
		-6.27308220	-0.00045380	0.00027803	0.00045609	-0.00000114	0.00000000
		-6.27308236	-0.00045373	0.00027799	0.00045603	-0.00000114	0.00000000
		-6.27308252	-0.00045367	0.00027795	0.00045596	-0.00000114	0.00000000
		-6.27308268	-0.00045360	0.00027790	0.00045589	-0.00000114	0.00000000
		-6.27308284	-0.00045353	0.00027786	0.00045583	-0.00000114	0.00000000
		-6.27308299	-0.00045347	0.00027782	0.00045576	-0.00000114	0.00000000
		-6.27308315	-0.00045340	0.00027778	0.00045569	-0.00000114	0.00000000
		-6.27308331	-0.00045334	0.00027774	0.00045563	-0.00000114	0.00000000
		-6.27308347	-0.00045327	0.00027770	0.00045556	-0.00000114	0.00000000
		-6.27308363	-0.00045320	0.00027766	0.00045549	-0.00000114	0.00000000
		-6.27308379	-0.00045314	0.00027762	0.00045543	-0.00000114	0.00000000
		-6.27308395	-0.00045307	0.00027758	0.00045536	-0.00000114	0.00000000
		-6.27308411	-0.00045300	0.00027754	0.00045529	-0.00000114	0.00000000
		-6.27308426	-0.00045294	0.00027750	0.00045523	-0.00000114	0.00000000
		-6.27308442	-0.00045287	0.00027746	0.00045516	-0.00000114	0.00000000
		-6.27308458	-0.00045281	0.00027742	0.00045509	-0.00000114	0.00000000
		-6.27308474	-0.00045274	0.00027738	0.00045503	-0.00000114	0.00000000
		-6.27308490	-0.00045267	0.00027734	0.00045496	-0.00000114	0.00000000
		-6.27308506	-0.00045261	0.00027729	0.00045489	-0.00000114	0.00000000
		-6.27308522	-0.00045254	0.00027725	0.00045483	-0.00000114	0.00000000
		-6.27308537	-0.00045247	0.00027721	0.00045476	-0.00000114	0.00000000
		-6.27308553	-0.00045241	0.00027717	0.00045469	-0.00000114	0.00000000
		-6.27308569	-0.00045234	0.00027713	0.00045463	-0.00000114	0.00000000
		-6.27308585	-0.00045228	0.00027709	0.00045456	-0.00000114	0.00000000
		-6.27308601	-0.00045221	0.00027705	0.00045449	-0.00000114	0.00000000
		-6.27308616	-0.00045214	0.00027701	0.00045443	-0.00000114	0.00000000
		-6.27308632	-0.00045208	0.00027697	0.00045436	-0.00000114	0.00000000
		-6.27308648	-0.00045201	0.00027693	0.00045430	-0.00000114	0.00000000
		-6.27308664	-0.00045195	0.00027689	0.00045423	-0.00000114	0.00000000
		-6.27308680	-0.00045188	0.00027685	0.00045416	-0.00000114	0.00000000
		-6.27308695	-0.00045181	0.00027681	0.00045410	-0.00000114	0.00000000
		-6.27308711	-0.00045175	0.00027677	0.00045403	-0.00000114	0.00000000
		-6.27308727	-0.00045168	0.00027673	0.00045396	-0.00000114	0.00000000
		-6.27308743	-0.00045162	0.00027669	0.00045390	-0.00000114	0.00000000
		-6.27308759	-0.00045155	0.00027665	0.00045383	-0.00000114	0.00000000
		-6.27308774	-0.00045148	0.00027661	0.00045376	-0.00000114	0.00000000
		-6.27308790	-0.00045142	0.00027657	0.00045370	-0.00000114	0.00000000
		-6.27308806	-0.00045135	0.00027653	0.00045363	-0.00000114	0.00000000
		-6.27308822	-0.00045129	0.00027649	0.00045357	-0.00000114	0.00000000
		-6.27308837	-0.00045122	0.00027645	0.00045350	-0.00000114	0.00000000
		-6.27308853	-0.00045116	0.00027641	0.00045343	-0.00000114	0.00000000
		-6.27308869	-0.00045109	0.00027636	0.00045337	-0.00000114	0.00000000
		-6.27308884	-0.00045102	0.00027632	0.00045330	-0.00000114	0.00000000
		-6.27308900	-0.00045096	0.00027628	0.00045324	-0.00000114	0.00000000
		-6.27308916	-0.00045089	0.00027624	0.00045317	-0.00000114	0.00000000
		-6.27308932	-0.00045083	0.00027620	0.00045310	-0.00000114	0.00000000
		-6.27308947	-0.00045076	0.00027616	0.00045304	-0.00000114	0.00000000
		-6.27308963	-0.00045070	0.00027612	0.00045297	-0.00000114	0.00000000
		-6.27308979	-0.00045063	0.00027608	0.00045291	-0.00000114	0.00000000
		-6.27308994	-0.00045056	0.00027604	0.00045284	-0.00000114	0.00000000
		-6.27309010	-0.00045050	0.00027600	0.00045277	-0.00000114	0.00000000
		-6.27309026	-0.00045043	0.00027596	0.00045271	-0.00000114	0.00000000
		-6.27309042	-0.00045037	0.00027592	0.00045264	-0.00000114	0.00000000
		-6.27309057	-0.00045030	0.00027588	0.00045258	-0.00000114	0.00000000
		-6.27309073	-0.00045024	0.00027584	0.00045251	-0.00000114	0.00000000
		-6.27309089	-0.00045017	0.00027580	0.00045245	-0.00000114	0.00000000
		-6.27309104	-0.00045011	0.00027576	0.00045238	-0.00000114	0.00000000
		-6.27309120	-0.00045004	0.00027572	0.00045231	-0.00000114	0.00000000
		-6.27309136	-0.00044997	0.00027568	0.00045225	-0.00000114	0.00000000
		-6.27309151	-0.00044991	0.00027564	0.00045218	-0.00000114	0.00000000
		-6.27309167	-0.00044984	0.00027560	0.00045212	-0.00000114	0.00000000
		-6.27309183	-0.00044978	0.00027556	0.00045205	-0.00000114	0.00000000
		-6.27309198	-0.00044971	0.00027552	0.00045198	-0.00000114	0.00000000
		-6.27309214	-0.00044965	0.00027548	0.00045192	-0.00000114	0.00000000
		-6.27309229	-0.00044958	0.00027544	0.00045185	-0.00000114	0.00000000
		-6.27309245	-0.00044952	0.00027540	0.00045179	-0.00000114	0.00000000
		-6.27309261	-0.00044945	0.00027536	0.00045172	-0.00000114	0.00000000
		-6.27309276	-0.00044939	0.00027532	0.00045166	-0.00000114	0.00000000
		-6.27309292	-0.00044932	0.00027528	0.00045159	-0.00000114	0.00000000
		-6.27309307	-0.00044926	0.00027524	0.00045153	-0.00000114	0.00000000
		-6.27309323	-0.00044919	0.00027520	0.00045146	-0.00000114	0.00000000
		-6.27309339	-0.00044913	0.00027516	0.00045139	-0.00000114	0.00000000
		-6.27309354	-0.00044906	0.00027512	0.00045133	-0.00000114	0.00000000
		-6.27309370	-0.00044900	0.00027508	0.00045126	-0.00000114	0.00000000
		-6.27309385	-0.00044893	0.00027504	0.00045120	-0.00000114	0.00000000
		-6.27309401	-0.00044887	0.00027500	0.00045113	-0.00000114	0.00000000
		-6.27309417	-0.00044880	0.00027496	0.00045107	-0.00000114	0.00000000
		-6.27309432	-0.00044874	0.00027492	0.00045100	-0.00000114	0.00000000
		-6.27309448	-0.00044867	0.00027488	0.00045094	-0.00000114	0.00000000
		-6.27309463	-0.00044861	0.00027484	0.00045087	-0.00000114	0.00000000
		-6.27309479	-0.00044854	0.00027480	0.00045081	-0.00000114	0.00000000
		-6.27309494	-0.00044848	0.00027476	0.00045074	-0.00000114	0.00000000
		-6.27309510	-0.00044841	0.00027472	0.00045068	-0.00000114	0.00000000
		-6.27309526	-0.00044835	0.00027468	0.00045061	-0.00000114	0.00000000
		-6.27309541	-0.00044828	0.00027464	0.00045054	-0.00000114	0.00000000
		-6.27309557	-0.00044822	0.00027460	0.00045048	-0.00000114	0.00000000
		-6.27309572	-0.00044815	0.00027456	0.00045041	-0.00000114	0.00000000
		-6.27309588	-0.00044809	0.00027452	0.00045035	-0.00000114	0.00000000
		-6.27309603	-0.00044802	0.00027448	0.00045028	-0.00000114	0.00000000
		-6.27309619	-0.00044796	0.00027444	0.00045022	-0.00000114	0.00000000
		-6.27309634	-0.00044789	0.00027440	0.00045015	-0.00000114	0.00000000
		-6.27309650	-0.00044783	0.00027436	0.00045009	-0.00000114	0.00000000
		-6.27309665	-0.00044776	0.00027432	0.00045002	-0.00000114	0.00000000
		-6.27309681	-0.00044770	0.00027429	0.00044996	-0.00000114	0.00000000
		-6.27309696	-0.00044763	0.00027425	0.00044989	-0.00000114	0.00000000
		-6.27309712	-0.00044757	0.00027421	0.00044983	-0.00000114	0.00000000
		-6.27309727	-0.00044750	0.00027417	0.00044976	-0.00000114	0.00000000
		-6.27309743	-0.00044744	0.00027413	0.00044970	-0.00000114	0.00000000
		-6.27309758	-0.00044737	0.00027409	0.00044963	-0.00000114	0.00000000
		-6.27309774	-0.00044731	0.00027405	0.00044957	-0.00000114	0.00000000
		-6.27309789	-0.00044724	0.00027401	0.00044950	-0.00000114	0.00000000
		-6.27309805	-0.00044718	0.00027397	0.00044944	-0.00000114	0.00000000
		-6.27309820	-0.00044712	0.00027393	0.00044937	-0.00000114	0.00000000
		-6.27309835	-0.00044705	0.00027389	0.00044931	-0.00000114	0.00000000
		-6.27309851	-0.00044699	0.00027385	0.00044924	-0.00000114	0.00000000
		-6.27309866	-0.00044692	0.00027381	0.00044918	-0.00000114	0.00000000
		-6.27309882	-0.00044686	0.00027377	0.00044911	-0.00000114	0.00000000
		-6.27309897	-0.00044679	0.00027373	0.00044905	-0.00000114	0.00000000
		-6.27309913	-0.00044673	0.00027369	0.00044898	-0.00000114	0.00000000
		-6.27309928	-0.00044666	0.00027365	0.00044892	-0.00000114	0.00000000
		-6.27309944	-0.00044660	0.00027361	0.00044885	-0.00000114	0.00000000
		-6.27309959	-0.00044653	0.00027357	0.00044879	-0.00000114	0.00000000
		-6.27309974	-0.00044647	0.00027353	0.00044873	-0.00000114	0.00000000
		-6.27309990	-0.00044641	0.00027349	0.00044866	-0.00000114	0.00000000
		-6.27310005	-0.00044634	0.00027345	0.00044860	-0.00000114	0.00000000
		-6.27310021	-0.00044628	0.00027341	0.00044853	-0.00000114	0.00000000
		-6.27310036	-0.00044621	0.00027338	0.00044847	-0.00000114	0.00000000
		-6.27310051	-0.00044615	0.00027334	0.00044840	-0.00000114	0.00000000
		-6.27310067	-0.00044608	0.00027330	0.00044834	-0.00000114	0.00000000
		-6.27310082	-0.00044602	0.00027326	0.00044827	-0.00000114	0.00000000
		-6.27310097	-0.00044596	0.00027322	0.00044821	-0.00000114	0.00000000
		-6.27310113	-0.00044589	0.00027318	0.00044814	-0.00000114	0.00000000
		-6.27310128	-0.00044583	0.00027314	0.00044808	-0.00000114	0.00000000
		-6.27310144	-0.00044576	0.00027310	0.00044801	-0.00000114	0.00000000
		-6.27310159	-0.00044570	0.00027306	0.00044795	-0.00000114	0.00000000
		-6.27310174	-0.00044563	0.00027302	0.00044789	-0.00000114	0.00000000
		-6.27310190	-0.00044557	0.00027298	0.00044782	-0.00000114	0.00000000
		-6.27310205	-0.00044551	0.00027294	0.00044776	-0.00000114	0.00000000
		-6.27310220	-0.00044544	0.00027290	0.00044769	-0.00000114	0.00000000
		-6.27310236	-0.00044538	0.00027286	0.00044763	-0.00000114	0.00000000
		-6.27310251	-0.00044531	0.00027282	0.00044756	-0.00000114	0.00000000
		-6.27310266	-0.00044525	0.00027279	0.00044750	-0.00000114	0.00000000
		-6.27310282	-0.00044519	0.00027275	0.00044743	-0.00000114	0.00000000
		-6.27310297	-0.00044512	0.00027271	0.00044737	-0.00000114	0.00000000
		-6.27310312	-0.00044506	0.00027267	0.00044731	-0.00000114	0.00000000
		-6.27310328	-0.00044499	0.00027263	0.00044724	-0.00000114	0.00000000
		-6.27310343	-0.00044493	0.00027259	0.00044718	-0.00000114	0.00000000
		-6.27310358	-0.00044487	0.00027255	0.00044711	-0.00000114	0.00000000
		-6.27310374	-0.00044480	0.00027251	0.00044705	-0.00000114	0.00000000
		-6.27310389	-0.00044474	0.00027247	0.00044698	-0.00000114	0.00000000
		-6.27310404	-0.00044467	0.00027243	0.00044692	-0.00000114	0.00000000
		-6.27310419	-0.00044461	0.00027239	0.00044686	-0.00000114	0.00000000
		-6.27310435	-0.00044455	0.00027235	0.00044679	-0.00000114	0.00000000
		-6.27310450	-0.00044448	0.00027231	0.00044673	-0.00000114	0.00000000
		-6.27310465	-0.00044442	0.00027228	0.00044666	-0.00000114	0.00000000
		-6.27310481	-0.00044436	0.00027224	0.00044660	-0.00000114	0.00000000
		-6.27310496	-0.00044429	0.00027220	0.00044654	-0.00000114	0.00000000
		-6.27310511	-0.00044423	0.00027216	0.00044647	-0.00000114	0.00000000
		-6.27310526	-0.00044416	0.00027212	0.00044641	-0.00000114	0.00000000
		-6.27310542	-0.00044410	0.00027208	0.00044634	-0.00000114	0.00000000
		-6.27310557	-0.00044404	0.00027204	0.00044628	-0.00000114	0.00000000
		-6.27310572	-0.00044397	0.00027200	0.00044621	-0.00000114	0.00000000
		-6.27310587	-0.00044391	0.00027196	0.00044615	-0.00000114	0.00000000
		-6.27310603	-0.00044385	0.00027192	0.00044609	-0.00000114	0.00000000
		-6.27310618	-0.00044378	0.00027189	0.00044602	-0.00000114	0.00000000
		-6.27310633	-0.00044372	0.00027185	0.00044596	-0.00000114	0.00000000
		-6.27310648	-0.00044365	0.00027181	0.00044590	-0.00000114	0.00000000
		-6.27310663	-0.00044359	0.00027177	0.00044583	-0.00000114	0.00000000
		-6.27310679	-0.00044353	0.00027173	0.00044577	-0.00000114	0.00000000
		-6.27310694	-0.00044346	0.00027169	0.00044570	-0.00000114	0.00000000
		-6.27310709	-0.00044340	0.00027165	0.00044564	-0.00000114	0.00000000
		-6.27310724	-0.00044334	0.00027161	0.00044558	-0.00000114	0.00000000
		-6.27310739	-0.00044327	0.00027157	0.00044551	-0.00000114	0.00000000
		-6.27310755	-0.00044321	0.00027153	0.00044545	-0.00000114	0.00000000
		-6.27310770	-0.00044315	0.00027150	0.00044538	-0.00000114	0.00000000
		-6.27310785	-0.00044308	0.00027146	0.00044532	-0.00000114	0.00000000
		-6.27310800	-0.00044302	0.00027142	0.00044526	-0.00000114	0.00000000
		-6.27310815	-0.00044296	0.00027138	0.00044519	-0.00000114	0.00000000
		-6.27310831	-0.00044289	0.00027134	0.00044513	-0.00000114	0.00000000
		-6.27310846	-0.00044283	0.00027130	0.00044507	-0.00000114	0.00000000
		-6.27310861	-0.00044277	0.00027126	0.00044500	-0.00000114	0.00000000
		-6.27310876	-0.00044270	0.00027122	0.00044494	-0.00000114	0.00000000
		-6.27310891	-0.00044264	0.00027119	0.00044487	-0.00000114	0.00000000
		-6.27310906	-0.00044258	0.00027115	0.00044481	-0.00000114	0.00000000
		-6.27310921	-0.00044251	0.00027111	0.00044475	-0.00000114	0.00000000
		-6.27310937	-0.00044245	0.00027107	0.00044468	-0.00000114	0.00000000
		-6.27310952	-0.00044239	0.00027103	0.00044462	-0.00000114	0.00000000
		-6.27310967	-0.00044232	0.00027099	0.00044456	-0.00000114	0.00000000
		-6.27310982	-0.00044226	0.00027095	0.00044449	-0.00000114	0.00000000
		-6.27310997	-0.00044220	0.00027091	0.00044443	-0.00000114	0.00000000
		-6.27311012	-0.00044213	0.00027088	0.00044437	-0.00000114	0.00000000
		-6.27311027	-0.00044207	0.00027084	0.00044430	-0.00000114	0.00000000
		-6.27311042	-0.00044201	0.00027080	0.00044424	-0.00000114	0.00000000
		-6.27311058	-0.00044194	0.00027076	0.00044418	-0.00000114	0.00000000
		-6.27311073	-0.00044188	0.00027072	0.00044411	-0.00000114	0.00000000
		-6.27311088	-0.00044182	0.00027068	0.00044405	-0.00000114	0.00000000
		-6.27311103	-0.00044176	0.00027064	0.00044399	-0.00000114	0.00000000
		-6.27311118	-0.00044169	0.00027060	0.00044392	-0.00000114	0.00000000
		-6.27311133	-0.00044163	0.00027057	0.00044386	-0.00000114	0.00000000
		-6.27311148	-0.00044157	0.00027053	0.00044380	-0.00000114	0.00000000
		-6.27311163	-0.00044150	0.00027049	0.00044373	-0.00000114	0.00000000
		-6.27311178	-0.00044144	0.00027045	0.00044367	-0.00000114	0.00000000
		-6.27311193	-0.00044138	0.00027041	0.00044361	-0.00000114	0.00000000
		-6.27311208	-0.00044131	0.00027037	0.00044354	-0.00000114	0.00000000
		-6.27311223	-0.00044125	0.00027033	0.00044348	-0.00000114	0.00000000
		-6.27311238	-0.00044119	0.00027030	0.00044342	-0.00000114	0.00000000
		-6.27311253	-0.00044113	0.00027026	0.00044335	-0.00000114	0.00000000
		-6.27311269	-0.00044106	0.00027022	0.00044329	-0.00000114	0.00000000
		-6.27311284	-0.00044100	0.00027018	0.00044323	-0.00000114	0.00000000
		-6.27311299	-0.00044094	0.00027014	0.00044316	-0.00000114	0.00000000
		-6.27311314	-0.00044087	0.00027010	0.00044310	-0.00000114	0.00000000
		-6.27311329	-0.00044081	0.00027006	0.00044304	-0.00000114	0.00000000
		-6.27311344	-0.00044075	0.00027003	0.00044297	-0.00000114	0.00000000
		-6.27311359	-0.00044069	0.00026999	0.00044291	-0.00000114	0.00000000
		-6.27311374	-0.00044062	0.00026995	0.00044285	-0.00000114	0.00000000
		-6.27311389	-0.00044056	0.00026991	0.00044279	-0.00000114	0.00000000
		-6.27311404	-0.00044050	0.00026987	0.00044272	-0.00000114	0.00000000
		-6.27311419	-0.00044044	0.00026983	0.00044266	-0.00000114	0.00000000
		-6.27311434	-0.00044037	0.00026980	0.00044260	-0.00000114	0.00000000
		-6.27311449	-0.00044031	0.00026976	0.00044253	-0.00000114	0.00000000
		-6.27311464	-0.00044025	0.00026972	0.00044247	-0.00000114	0.00000000
		-6.27311479	-0.00044018	0.00026968	0.00044241	-0.00000114	0.00000000
		-6.27311494	-0.00044012	0.00026964	0.00044234	-0.00000114	0.00000000
		-6.27311509	-0.00044006	0.00026960	0.00044228	-0.00000114	0.00000000
		-6.27311524	-0.00044000	0.00026957	0.00044222	-0.00000114	0.00000000
		-6.27311539	-0.00043993	0.00026953	0.00044216	-0.00000114	0.00000000
		-6.27311554	-0.00043987	0.00026949	0.00044209	-0.00000114	0.00000000
		-6.27311569	-0.00043981	0.00026945	0.00044203	-0.00000114	0.00000000
		-6.27311583	-0.00043975	0.00026941	0.00044197	-0.00000114	0.00000000
		-6.27311598	-0.00043968	0.00026937	0.00044190	-0.00000114	0.00000000
		-6.27311613	-0.00043962	0.00026934	0.00044184	-0.00000114	0.00000000
		-6.27311628	-0.00043956	0.00026930	0.00044178	-0.00000114	0.00000000
		-6.27311643	-0.00043950	0.00026926	0.00044172	-0.00000114	0.00000000
		-6.27311658	-0.00043944	0.00026922	0.00044165	-0.00000114	0.00000000
		-6.27311673	-0.00043937	0.00026918	0.00044159	-0.00000114	0.00000000
		-6.27311688	-0.00043931	0.00026914	0.00044153	-0.00000114	0.00000000
		-6.27311703	-0.00043925	0.00026911	0.00044147	-0.00000114	0.00000000
		-6.27311718	-0.00043919	0.00026907	0.00044140	-0.00000114	0.00000000
		-6.27311733	-0.00043912	0.00026903	0.00044134	-0.00000114	0.00000000
		-6.27311748	-0.00043906	0.00026899	0.00044128	-0.00000114	0.00000000
		-6.27311763	-0.00043900	0.00026895	0.00044122	-0.00000114	0.00000000
		-6.27311777	-0.00043894	0.00026892	0.00044115	-0.00000114	0.00000000
		-6.27311792	-0.00043887	0.00026888	0.00044109	-0.00000114	0.00000000
		-6.27311807	-0.00043881	0.00026884	0.00044103	-0.00000114	0.00000000
		-6.27311822	-0.00043875	0.00026880	0.00044096	-0.00000114	0.00000000
		-6.27311837	-0.00043869	0.00026876	0.00044090	-0.00000114	0.00000000
		-6.27311852	-0.00043863	0.00026872	0.00044084	-0.00000114	0.00000000
		-6.27311867	-0.00043856	0.00026869	0.00044078	-0.00000114	0.00000000
		-6.27311882	-0.00043850	0.00026865	0.00044072	-0.00000114	0.00000000
		-6.27311897	-0.00043844	0.00026861	0.00044065	-0.00000114	0.00000000
		-6.27311911	-0.00043838	0.00026857	0.00044059	-0.00000114	0.00000000
		-6.27311926	-0.00043832	0.00026853	0.00044053	-0.00000114	0.00000000
		-6.27311941	-0.00043825	0.00026850	0.00044047	-0.00000114	0.00000000
		-6.27311956	-0.00043819	0.00026846	0.00044040	-0.00000114	0.00000000
		-6.27311971	-0.00043813	0.00026842	0.00044034	-0.00000114	0.00000000
		-6.27311986	-0.00043807	0.00026838	0.00044028	-0.00000114	0.00000000
		-6.27312000	-0.00043800	0.00026834	0.00044022	-0.00000114	0.00000000
		-6.27312015	-0.00043794	0.00026831	0.00044015	-0.00000114	0.00000000
		-6.27312030	-0.00043788	0.00026827	0.00044009	-0.00000114	0.00000000
		-6.27312045	-0.00043782	0.00026823	0.00044003	-0.00000114	0.00000000
		-6.27312060	-0.00043776	0.00026819	0.00043997	-0.00000114	0.00000000
		-6.27312075	-0.00043770	0.00026815	0.00043990	-0.00000114	0.00000000
		-6.27312089	-0.00043763	0.00026812	0.00043984	-0.00000114	0.00000000
		-6.27312104	-0.00043757	0.00026808	0.00043978	-0.00000114	0.00000000
		-6.27312119	-0.00043751	0.00026804	0.00043972	-0.00000114	0.00000000
		-6.27312134	-0.00043745	0.00026800	0.00043966	-0.00000114	0.00000000
		-6.27312149	-0.00043739	0.00026796	0.00043959	-0.00000114	0.00000000
		-6.27312163	-0.00043732	0.00026793	0.00043953	-0.00000114	0.00000000
		-6.27312178	-0.00043726	0.00026789	0.00043947	-0.00000114	0.00000000
		-6.27312193	-0.00043720	0.00026785	0.00043941	-0.00000114	0.00000000
		-6.27312208	-0.00043714	0.00026781	0.00043935	-0.00000114	0.00000000
		-6.27312223	-0.00043708	0.00026778	0.00043928	-0.00000114	0.00000000
		-6.27312237	-0.00043702	0.00026774	0.00043922	-0.00000114	0.00000000
		-6.27312252	-0.00043695	0.00026770	0.00043916	-0.00000114	0.00000000
		-6.27312267	-0.00043689	0.00026766	0.00043910	-0.00000114	0.00000000
		-6.27312282	-0.00043683	0.00026762	0.00043904	-0.00000114	0.00000000
		-6.27312296	-0.00043677	0.00026759	0.00043897	-0.00000114	0.00000000
		-6.27312311	-0.00043671	0.00026755	0.00043891	-0.00000114	0.00000000
		-6.27312326	-0.00043665	0.00026751	0.00043885	-0.00000114	0.00000000
		-6.27312341	-0.00043658	0.00026747	0.00043879	-0.00000114	0.00000000
		-6.27312355	-0.00043652	0.00026744	0.00043873	-0.00000114	0.00000000
		-6.27312370	-0.00043646	0.00026740	0.00043866	-0.00000114	0.00000000
		-6.27312385	-0.00043640	0.00026736	0.00043860	-0.00000114	0.00000000
		-6.27312399	-0.00043634	0.00026732	0.00043854	-0.00000114	0.00000000
		-6.27312414	-0.00043628	0.00026728	0.00043848	-0.00000114	0.00000000
		-6.27312429	-0.00043622	0.00026725	0.00043842	-0.00000114	0.00000000
		-6.27312444	-0.00043615	0.00026721	0.00043836	-0.00000114	0.00000000
		-6.27312458	-0.00043609	0.00026717	0.00043829	-0.00000114	0.00000000
		-6.27312473	-0.00043603	0.00026713	0.00043823	-0.00000114	0.00000000
		-6.27312488	-0.00043597	0.00026710	0.00043817	-0.00000114	0.00000000
		-6.27312502	-0.00043591	0.00026706	0.00043811	-0.00000114	0.00000000
		-6.27312517	-0.00043585	0.00026702	0.00043805	-0.00000114	0.00000000
		-6.27312532	-0.00043579	0.00026698	0.00043798	-0.00000114	0.00000000
		-6.27312546	-0.00043572	0.00026695	0.00043792	-0.00000114	0.00000000
		-6.27312561	-0.00043566	0.00026691	0.00043786	-0.00000114	0.00000000
		-6.27312576	-0.00043560	0.00026687	0.00043780	-0.00000114	0.00000000
		-6.27312590	-0.00043554	0.00026683	0.00043774	-0.00000114	0.00000000
		-6.27312605	-0.00043548	0.00026680	0.00043768	-0.00000114	0.00000000
		-6.27312620	-0.00043542	0.00026676	0.00043762	-0.00000114	0.00000000
		-6.27312634	-0.00043536	0.00026672	0.00043755	-0.00000114	0.00000000
		-6.27312649	-0.00043529	0.00026668	0.00043749	-0.00000114	0.00000000
		-6.27312664	-0.00043523	0.00026665	0.00043743	-0.00000114	0.00000000
		-6.27312678	-0.00043517	0.00026661	0.00043737	-0.00000114	0.00000000
		-6.27312693	-0.00043511	0.00026657	0.00043731	-0.00000114	0.00000000
		-6.27312708	-0.00043505	0.00026653	0.00043725	-0.00000114	0.00000000
		-6.27312722	-0.00043499	0.00026650	0.00043718	-0.00000114	0.00000000
		-6.27312737	-0.00043493	0.00026646	0.00043712	-0.00000114	0.00000000
		-6.27312752	-0.00043487	0.00026642	0.00043706	-0.00000114	0.00000000
		-6.27312766	-0.00043481	0.00026638	0.00043700	-0.00000114	0.00000000
		-6.27312781	-0.00043474	0.00026635	0.00043694	-0.00000114	0.00000000
		-6.27312795	-0.00043468	0.00026631	0.00043688	-0.00000114	0.00000000
		-6.27312810	-0.00043462	0.00026627	0.00043682	-0.00000114	0.00000000
		-6.27312825	-0.00043456	0.00026623	0.00043676	-0.00000114	0.00000000
		-6.27312839	-0.00043450	0.00026620	0.00043669	-0.00000114	0.00000000
		-6.27312854	-0.00043444	0.00026616	0.00043663	-0.00000114	0.00000000
		-6.27312868	-0.00043438	0.00026612	0.00043657	-0.00000114	0.00000000
		-6.27312883	-0.00043432	0.00026608	0.00043651	-0.00000114	0.00000000
		-6.27312898	-0.00043426	0.00026605	0.00043645	-0.00000114	0.00000000
		-6.27312912	-0.00043420	0.00026601	0.00043639	-0.00000114	0.00000000
		-6.27312927	-0.00043413	0.00026597	0.00043633	-0.00000114	0.00000000
		-6.27312941	-0.00043407	0.00026593	0.00043626	-0.00000114	0.00000000
		-6.27312956	-0.00043401	0.00026590	0.00043620	-0.00000114	0.00000000
		-6.27312970	-0.00043395	0.00026586	0.00043614	-0.00000114	0.00000000
		-6.27312985	-0.00043389	0.00026582	0.00043608	-0.00000114	0.00000000
		-6.27313000	-0.00043383	0.00026579	0.00043602	-0.00000114	0.00000000
		-6.27313014	-0.00043377	0.00026575	0.00043596	-0.00000114	0.00000000
		-6.27313029	-0.00043371	0.00026571	0.00043590	-0.00000114	0.00000000
		-6.27313043	-0.00043365	0.00026567	0.00043584	-0.00000114	0.00000000
		-6.27313058	-0.00043359	0.00026564	0.00043578	-0.00000114	0.00000000
		-6.27313072	-0.00043353	0.00026560	0.00043572	-0.00000114	0.00000000
		-6.27313087	-0.00043347	0.00026556	0.00043565	-0.00000114	0.00000000
		-6.27313101	-0.00043341	0.00026552	0.00043559	-0.00000114	0.00000000
		-6.27313116	-0.00043335	0.00026549	0.00043553	-0.00000114	0.00000000
		-6.27313130	-0.00043328	0.00026545	0.00043547	-0.00000114	0.00000000
		-6.27313145	-0.00043322	0.00026541	0.00043541	-0.00000114	0.00000000
		-6.27313159	-0.00043316	0.00026538	0.00043535	-0.00000114	0.00000000
		-6.27313174	-0.00043310	0.00026534	0.00043529	-0.00000114	0.00000000
		-6.27313188	-0.00043304	0.00026530	0.00043523	-0.00000114	0.00000000
		-6.27313203	-0.00043298	0.00026526	0.00043517	-0.00000114	0.00000000
		-6.27313217	-0.00043292	0.00026523	0.00043511	-0.00000114	0.00000000
		-6.27313232	-0.00043286	0.00026519	0.00043504	-0.00000114	0.00000000
		-6.27313246	-0.00043280	0.00026515	0.00043498	-0.00000114	0.00000000
		-6.27313261	-0.00043274	0.00026512	0.00043492	-0.00000114	0.00000000
		-6.27313275	-0.00043268	0.00026508	0.00043486	-0.00000114	0.00000000
		-6.27313290	-0.00043262	0.00026504	0.00043480	-0.00000114	0.00000000
		-6.27313304	-0.00043256	0.00026501	0.00043474	-0.00000114	0.00000000
		-6.27313319	-0.00043250	0.00026497	0.00043468	-0.00000114	0.00000000
		-6.27313333	-0.00043244	0.00026493	0.00043462	-0.00000114	0.00000000
		-6.27313348	-0.00043238	0.00026489	0.00043456	-0.00000114	0.00000000
		-6.27313362	-0.00043232	0.00026486	0.00043450	-0.00000114	0.00000000
		-6.27313377	-0.00043226	0.00026482	0.00043444	-0.00000114	0.00000000
		-6.27313391	-0.00043220	0.00026478	0.00043438	-0.00000114	0.00000000
		-6.27313405	-0.00043214	0.00026475	0.00043432	-0.00000114	0.00000000
		-6.27313420	-0.00043207	0.00026471	0.00043426	-0.00000114	0.00000000
		-6.27313434	-0.00043201	0.00026467	0.00043419	-0.00000114	0.00000000
		-6.27313449	-0.00043195	0.00026464	0.00043413	-0.00000114	0.00000000
		-6.27313463	-0.00043189	0.00026460	0.00043407	-0.00000114	0.00000000
		-6.27313478	-0.00043183	0.00026456	0.00043401	-0.00000114	0.00000000
		-6.27313492	-0.00043177	0.00026452	0.00043395	-0.00000114	0.00000000
		-6.27313506	-0.00043171	0.00026449	0.00043389	-0.00000114	0.00000000
		-6.27313521	-0.00043165	0.00026445	0.00043383	-0.00000114	0.00000000
		-6.27313535	-0.00043159	0.00026441	0.00043377	-0.00000114	0.00000000
		-6.27313550	-0.00043153	0.00026438	0.00043371	-0.00000114	0.00000000
		-6.27313564	-0.00043147	0.00026434	0.00043365	-0.00000114	0.00000000
		-6.27313578	-0.00043141	0.00026430	0.00043359	-0.00000114	0.00000000
		-6.27313593	-0.00043135	0.00026427	0.00043353	-0.00000114	0.00000000
		-6.27313607	-0.00043129	0.00026423	0.00043347	-0.00000114	0.00000000
		-6.27313622	-0.00043123	0.00026419	0.00043341	-0.00000114	0.00000000
		-6.27313636	-0.00043117	0.00026416	0.00043335	-0.00000114	0.00000000
		-6.27313650	-0.00043111	0.00026412	0.00043329	-0.00000114	0.00000000
		-6.27313665	-0.00043105	0.00026408	0.00043323	-0.00000114	0.00000000
		-6.27313679	-0.00043099	0.00026405	0.00043317	-0.00000114	0.00000000
		-6.27313693	-0.00043093	0.00026401	0.00043311	-0.00000114	0.00000000
		-6.27313708	-0.00043087	0.00026397	0.00043305	-0.00000114	0.00000000
		-6.27313722	-0.00043081	0.00026394	0.00043299	-0.00000114	0.00000000
		-6.27313737	-0.00043075	0.00026390	0.00043293	-0.00000114	0.00000000
		-6.27313751	-0.00043069	0.00026386	0.00043287	-0.00000114	0.00000000
		-6.27313765	-0.00043063	0.00026382	0.00043281	-0.00000114	0.00000000
		-6.27313780	-0.00043057	0.00026379	0.00043275	-0.00000114	0.00000000
		-6.27313794	-0.00043051	0.00026375	0.00043268	-0.00000114	0.00000000
		-6.27313808	-0.00043045	0.00026371	0.00043262	-0.00000114	0.00000000
		-6.27313823	-0.00043039	0.00026368	0.00043256	-0.00000114	0.00000000
		-6.27313837	-0.00043033	0.00026364	0.00043250	-0.00000114	0.00000000
		-6.27313851	-0.00043027	0.00026360	0.00043244	-0.00000114	0.00000000
		-6.27313865	-0.00043021	0.00026357	0.00043238	-0.00000114	0.00000000
		-6.27313880	-0.00043015	0.00026353	0.00043232	-0.00000114	0.00000000
		-6.27313894	-0.00043009	0.00026349	0.00043226	-0.00000114	0.00000000
		-6.27313908	-0.00043003	0.00026346	0.00043220	-0.00000114	0.00000000
		-6.27313923	-0.00042997	0.00026342	0.00043214	-0.00000114	0.00000000
		-6.27313937	-0.00042991	0.00026338	0.00043208	-0.00000114	0.00000000
		-6.27313951	-0.00042985	0.00026335	0.00043202	-0.00000114	0.00000000
		-6.27313966	-0.00042980	0.00026331	0.00043196	-0.00000114	0.00000000
		-6.27313980	-0.00042974	0.00026328	0.00043190	-0.00000114	0.00000000
		-6.27313994	-0.00042968	0.00026324	0.00043184	-0.00000114	0.00000000
		-6.27314008	-0.00042962	0.00026320	0.00043178	-0.00000114	0.00000000
		-6.27314023	-0.00042956	0.00026317	0.00043172	-0.00000114	0.00000000
		-6.27314037	-0.00042950	0.00026313	0.00043166	-0.00000114	0.00000000
		-6.27314051	-0.00042944	0.00026309	0.00043160	-0.00000114	0.00000000
		-6.27314065	-0.00042938	0.00026306	0.00043154	-0.00000114	0.00000000
		-6.27314080	-0.00042932	0.00026302	0.00043148	-0.00000114	0.00000000
		-6.27314094	-0.00042926	0.00026298	0.00043142	-0.00000114	0.00000000
		-6.27314108	-0.00042920	0.00026295	0.00043136	-0.00000114	0.00000000
		-6.27314122	-0.00042914	0.00026291	0.00043131	-0.00000114	0.00000000
		-6.27314137	-0.00042908	0.00026287	0.00043125	-0.00000114	0.00000000
		-6.27314151	-0.00042902	0.00026284	0.00043119	-0.00000114	0.00000000
		-6.27314165	-0.00042896	0.00026280	0.00043113	-0.00000114	0.00000000
		-6.27314179	-0.00042890	0.00026276	0.00043107	-0.00000114	0.00000000
		-6.27314194	-0.00042884	0.00026273	0.00043101	-0.00000114	0.00000000
		-6.27314208	-0.00042878	0.00026269	0.00043095	-0.00000114	0.00000000
		-6.27314222	-0.00042872	0.00026265	0.00043089	-0.00000114	0.00000000
		-6.27314236	-0.00042866	0.00026262	0.00043083	-0.00000114	0.00000000
		-6.27314250	-0.00042861	0.00026258	0.00043077	-0.00000114	0.00000000
		-6.27314265	-0.00042855	0.00026255	0.00043071	-0.00000114	0.00000000
		-6.27314279	-0.00042849	0.00026251	0.00043065	-0.00000114	0.00000000
		-6.27314293	-0.00042843	0.00026247	0.00043059	-0.00000114	0.00000000
		-6.27314307	-0.00042837	0.00026244	0.00043053	-0.00000114	0.00000000
		-6.27314321	-0.00042831	0.00026240	0.00043047	-0.00000114	0.00000000
		-6.27314336	-0.00042825	0.00026236	0.00043041	-0.00000114	0.00000000
		-6.27314350	-0.00042819	0.00026233	0.00043035	-0.00000114	0.00000000
		-6.27314364	-0.00042813	0.00026229	0.00043029	-0.00000114	0.00000000
		-6.27314378	-0.00042807	0.00026226	0.00043023	-0.00000114	0.00000000
		-6.27314392	-0.00042801	0.00026222	0.00043017	-0.00000114	0.00000000
		-6.27314406	-0.00042795	0.00026218	0.00043011	-0.00000114	0.00000000
		-6.27314421	-0.00042789	0.00026215	0.00043005	-0.00000114	0.00000000
		-6.27314435	-0.00042783	0.00026211	0.00042999	-0.00000114	0.00000000
		-6.27314449	-0.00042778	0.00026207	0.00042993	-0.00000114	0.00000000
		-6.27314463	-0.00042772	0.00026204	0.00042987	-0.00000114	0.00000000
		-6.27314477	-0.00042766	0.00026200	0.00042982	-0.00000114	0.00000000
		-6.27314491	-0.00042760	0.00026197	0.00042976	-0.00000114	0.00000000
		-6.27314505	-0.00042754	0.00026193	0.00042970	-0.00000114	0.00000000
		-6.27314520	-0.00042748	0.00026189	0.00042964	-0.00000114	0.00000000
		-6.27314534	-0.00042742	0.00026186	0.00042958	-0.00000114	0.00000000
		-6.27314548	-0.00042736	0.00026182	0.00042952	-0.00000114	0.00000000
		-6.27314562	-0.00042730	0.00026178	0.00042946	-0.00000114	0.00000000
		-6.27314576	-0.00042724	0.00026175	0.00042940	-0.00000114	0.00000000
		-6.27314590	-0.00042719	0.00026171	0.00042934	-0.00000114	0.00000000
		-6.27314604	-0.00042713	0.00026168	0.00042928	-0.00000114	0.00000000
		-6.27314618	-0.00042707	0.00026164	0.00042922	-0.00000114	0.00000000
		-6.27314633	-0.00042701	0.00026160	0.00042916	-0.00000114	0.00000000
		-6.27314647	-0.00042695	0.00026157	0.00042910	-0.00000114	0.00000000
		-6.27314661	-0.00042689	0.00026153	0.00042904	-0.00000114	0.00000000
		-6.27314675	-0.00042683	0.00026150	0.00042899	-0.00000114	0.00000000
		-6.27314689	-0.00042677	0.00026146	0.00042893	-0.00000114	0.00000000
		-6.27314703	-0.00042671	0.00026142	0.00042887	-0.00000114	0.00000000
		-6.27314717	-0.00042666	0.00026139	0.00042881	-0.00000114	0.00000000
		-6.27314731	-0.00042660	0.00026135	0.00042875	-0.00000114	0.00000000
		-6.27314745	-0.00042654	0.00026132	0.00042869	-0.00000114	0.00000000
		-6.27314759	-0.00042648	0.00026128	0.00042863	-0.00000114	0.00000000
		-6.27314773	-0.00042642	0.00026124	0.00042857	-0.00000114	0.00000000
		-6.27314787	-0.00042636	0.00026121	0.00042851	-0.00000114	0.00000000
		-6.27314801	-0.00042630	0.00026117	0.00042845	-0.00000114	0.00000000
		-6.27314815	-0.00042624	0.00026114	0.00042839	-0.00000114	0.00000000
		-6.27314830	-0.00042619	0.00026110	0.00042834	-0.00000114	0.00000000
		-6.27314844	-0.00042613	0.00026106	0.00042828	-0.00000114	0.00000000
		-6.27314858	-0.00042607	0.00026103	0.00042822	-0.00000114	0.00000000
		-6.27314872	-0.00042601	0.00026099	0.00042816	-0.00000114	0.00000000
		-6.27314886	-0.00042595	0.00026096	0.00042810	-0.00000114	0.00000000
		-6.27314900	-0.00042589	0.00026092	0.00042804	-0.00000114	0.00000000
		-6.27314914	-0.00042583	0.00026088	0.00042798	-0.00000114	0.00000000
		-6.27314928	-0.00042578	0.00026085	0.00042792	-0.00000114	0.00000000
		-6.27314942	-0.00042572	0.00026081	0.00042786	-0.00000114	0.00000000
		-6.27314956	-0.00042566	0.00026078	0.00042781	-0.00000114	0.00000000
		-6.27314970	-0.00042560	0.00026074	0.00042775	-0.00000114	0.00000000
		-6.27314984	-0.00042554	0.00026070	0.00042769	-0.00000114	0.00000000
		-6.27314998	-0.00042548	0.00026067	0.00042763	-0.00000114	0.00000000
		-6.27315012	-0.00042542	0.00026063	0.00042757	-0.00000114	0.00000000
		-6.27315026	-0.00042537	0.00026060	0.00042751	-0.00000114	0.00000000
		-6.27315040	-0.00042531	0.00026056	0.00042745	-0.00000114	0.00000000
		-6.27315054	-0.00042525	0.00026052	0.00042739	-0.00000114	0.00000000
		-6.27315068	-0.00042519	0.00026049	0.00042734	-0.00000114	0.00000000
		-6.27315082	-0.00042513	0.00026045	0.00042728	-0.00000114	0.00000000
		-6.27315096	-0.00042507	0.00026042	0.00042722	-0.00000114	0.00000000
		-6.27315110	-0.00042502	0.00026038	0.00042716	-0.00000114	0.00000000
		-6.27315124	-0.00042496	0.00026035	0.00042710	-0.00000114	0.00000000
		-6.27315138	-0.00042490	0.00026031	0.00042704	-0.00000114	0.00000000
		-6.27315152	-0.00042484	0.00026027	0.00042698	-0.00000114	0.00000000
		-6.27315165	-0.00042478	0.00026024	0.00042692	-0.00000114	0.00000000
		-6.27315179	-0.00042472	0.00026020	0.00042687	-0.00000114	0.00000000
		-6.27315193	-0.00042467	0.00026017	0.00042681	-0.00000114	0.00000000
		-6.27315207	-0.00042461	0.00026013	0.00042675	-0.00000114	0.00000000
		-6.27315221	-0.00042455	0.00026010	0.00042669	-0.00000114	0.00000000
		-6.27315235	-0.00042449	0.00026006	0.00042663	-0.00000114	0.00000000
		-6.27315249	-0.00042443	0.00026002	0.00042657	-0.00000114	0.00000000
		-6.27315263	-0.00042437	0.00025999	0.00042652	-0.00000114	0.00000000
		-6.27315277	-0.00042432	0.00025995	0.00042646	-0.00000114	0.00000000
		-6.27315291	-0.00042426	0.00025992	0.00042640	-0.00000114	0.00000000
		-6.27315305	-0.00042420	0.00025988	0.00042634	-0.00000114	0.00000000
		-6.27315319	-0.00042414	0.00025985	0.00042628	-0.00000114	0.00000000
		-6.27315333	-0.00042408	0.00025981	0.00042622	-0.00000114	0.00000000
		-6.27315347	-0.00042403	0.00025978	0.00042616	-0.00000114	0.00000000
		-6.27315360	-0.00042397	0.00025974	0.00042611	-0.00000114	0.00000000
		-6.27315374	-0.00042391	0.00025970	0.00042605	-0.00000114	0.00000000
		-6.27315388	-0.00042385	0.00025967	0.00042599	-0.00000114	0.00000000
		-6.27315402	-0.00042379	0.00025963	0.00042593	-0.00000114	0.00000000
		-6.27315416	-0.00042374	0.00025960	0.00042587	-0.00000114	0.00000000
		-6.27315430	-0.00042368	0.00025956	0.00042581	-0.00000114	0.00000000
		-6.27315444	-0.00042362	0.00025953	0.00042576	-0.00000114	0.00000000
		-6.27315458	-0.00042356	0.00025949	0.00042570	-0.00000114	0.00000000
		-6.27315472	-0.00042350	0.00025946	0.00042564	-0.00000114	0.00000000
		-6.27315485	-0.00042345	0.00025942	0.00042558	-0.00000114	0.00000000
		-6.27315499	-0.00042339	0.00025938	0.00042552	-0.00000114	0.00000000
		-6.27315513	-0.00042333	0.00025935	0.00042547	-0.00000114	0.00000000
		-6.27315527	-0.00042327	0.00025931	0.00042541	-0.00000114	0.00000000
		-6.27315541	-0.00042321	0.00025928	0.00042535	-0.00000114	0.00000000
		-6.27315555	-0.00042316	0.00025924	0.00042529	-0.00000114	0.00000000
		-6.27315569	-0.00042310	0.00025921	0.00042523	-0.00000114	0.00000000
		-6.27315582	-0.00042304	0.00025917	0.00042517	-0.00000114	0.00000000
		-6.27315596	-0.00042298	0.00025914	0.00042512	-0.00000114	0.00000000
		-6.27315610	-0.00042292	0.00025910	0.00042506	-0.00000114	0.00000000
		-6.27315624	-0.00042287	0.00025907	0.00042500	-0.00000114	0.00000000
		-6.27315638	-0.00042281	0.00025903	0.00042494	-0.00000114	0.00000000
		-6.27315652	-0.00042275	0.00025899	0.00042488	-0.00000114	0.00000000
		-6.27315665	-0.00042269	0.00025896	0.00042483	-0.00000114	0.00000000
		-6.27315679	-0.00042264	0.00025892	0.00042477	-0.00000114	0.00000000
		-6.27315693	-0.00042258	0.00025889	0.00042471	-0.00000114	0.00000000
		-6.27315707	-0.00042252	0.00025885	0.00042465	-0.00000114	0.00000000
		-6.27315721	-0.00042246	0.00025882	0.00042459	-0.00000114	0.00000000
		-6.27315734	-0.00042241	0.00025878	0.00042454	-0.00000114	0.00000000
		-6.27315748	-0.00042235	0.00025875	0.00042448	-0.00000114	0.00000000
		-6.27315762	-0.00042229	0.00025871	0.00042442	-0.00000114	0.00000000
		-6.27315776	-0.00042223	0.00025868	0.00042436	-0.00000114	0.00000000
		-6.27315790	-0.00042218	0.00025864	0.00042430	-0.00000114	0.00000000
		-6.27315803	-0.00042212	0.00025861	0.00042425	-0.00000114	0.00000000
		-6.27315817	-0.00042206	0.00025857	0.00042419	-0.00000114	0.00000000
		-6.27315831	-0.00042200	0.00025854	0.00042413	-0.00000114	0.00000000
		-6.27315845	-0.00042194	0.00025850	0.00042407	-0.00000114	0.00000000
		-6.27315858	-0.00042189	0.00025846	0.00042402	-0.00000114	0.00000000
		-6.27315872	-0.00042183	0.00025843	0.00042396	-0.00000114	0.00000000
		-6.27315886	-0.00042177	0.00025839	0.00042390	-0.00000114	0.00000000
		-6.27315900	-0.00042171	0.00025836	0.00042384	-0.00000114	0.00000000
		-6.27315913	-0.00042166	0.00025832	0.00042378	-0.00000114	0.00000000
		-6.27315927	-0.00042160	0.00025829	0.00042373	-0.00000114	0.00000000
		-6.27315941	-0.00042154	0.00025825	0.00042367	-0.00000114	0.00000000
		-6.27315955	-0.00042149	0.00025822	0.00042361	-0.00000114	0.00000000
		-6.27315968	-0.00042143	0.00025818	0.00042355	-0.00000114	0.00000000
		-6.27315982	-0.00042137	0.00025815	0.00042350	-0.00000114	0.00000000
		-6.27315996	-0.00042131	0.00025811	0.00042344	-0.00000114	0.00000000
		-6.27316010	-0.00042126	0.00025808	0.00042338	-0.00000114	0.00000000
		-6.27316023	-0.00042120	0.00025804	0.00042332	-0.00000114	0.00000000
		-6.27316037	-0.00042114	0.00025801	0.00042327	-0.00000114	0.00000000
		-6.27316051	-0.00042108	0.00025797	0.00042321	-0.00000114	0.00000000
		-6.27316064	-0.00042103	0.00025794	0.00042315	-0.00000114	0.00000000
		-6.27316078	-0.00042097	0.00025790	0.00042309	-0.00000114	0.00000000
		-6.27316092	-0.00042091	0.00025787	0.00042304	-0.00000114	0.00000000
		-6.27316105	-0.00042085	0.00025783	0.00042298	-0.00000114	0.00000000
		-6.27316119	-0.00042080	0.00025780	0.00042292	-0.00000114	0.00000000
		-6.27316133	-0.00042074	0.00025776	0.00042286	-0.00000114	0.00000000
		-6.27316147	-0.00042068	0.00025773	0.00042281	-0.00000114	0.00000000
		-6.27316160	-0.00042063	0.00025769	0.00042275	-0.00000114	0.00000000
		-6.27316174	-0.00042057	0.00025766	0.00042269	-0.00000114	0.00000000
		-6.27316188	-0.00042051	0.00025762	0.00042263	-0.00000114	0.00000000
		-6.27316201	-0.00042045	0.00025759	0.00042258	-0.00000114	0.00000000
		-6.27316215	-0.00042040	0.00025755	0.00042252	-0.00000114	0.00000000
		-6.27316229	-0.00042034	0.00025752	0.00042246	-0.00000114	0.00000000
		-6.27316242	-0.00042028	0.00025748	0.00042240	-0.00000114	0.00000000
		-6.27316256	-0.00042023	0.00025745	0.00042235	-0.00000114	0.00000000
		-6.27316270	-0.00042017	0.00025741	0.00042229	-0.00000114	0.00000000
		-6.27316283	-0.00042011	0.00025738	0.00042223	-0.00000114	0.00000000
		-6.27316297	-0.00042006	0.00025734	0.00042217	-0.00000114	0.00000000
		-6.27316311	-0.00042000	0.00025731	0.00042212	-0.00000114	0.00000000
		-6.27316324	-0.00041994	0.00025727	0.00042206	-0.00000114	0.00000000
		-6.27316338	-0.00041988	0.00025724	0.00042200	-0.00000114	0.00000000
		-6.27316351	-0.00041983	0.00025720	0.00042194	-0.00000114	0.00000000
		-6.27316365	-0.00041977	0.00025717	0.00042189	-0.00000114	0.00000000
		-6.27316379	-0.00041971	0.00025713	0.00042183	-0.00000114	0.00000000
		-6.27316392	-0.00041966	0.00025710	0.00042177	-0.00000114	0.00000000
		-6.27316406	-0.00041960	0.00025706	0.00042172	-0.00000114	0.00000000
		-6.27316420	-0.00041954	0.00025703	0.00042166	-0.00000114	0.00000000
		-6.27316433	-0.00041949	0.00025699	0.00042160	-0.00000114	0.00000000
		-6.27316447	-0.00041943	0.00025696	0.00042154	-0.00000114	0.00000000
		-6.27316460	-0.00041937	0.00025692	0.00042149	-0.00000114	0.00000000
		-6.27316474	-0.00041932	0.00025689	0.00042143	-0.00000114	0.00000000
		-6.27316488	-0.00041926	0.00025685	0.00042137	-0.00000114	0.00000000
		-6.27316501	-0.00041920	0.00025682	0.00042132	-0.00000114	0.00000000
		-6.27316515	-0.00041915	0.00025678	0.00042126	-0.00000114	0.00000000
		-6.27316528	-0.00041909	0.00025675	0.00042120	-0.00000114	0.00000000
		-6.27316542	-0.00041903	0.00025671	0.00042115	-0.00000114	0.00000000
		-6.27316555	-0.00041898	0.00025668	0.00042109	-0.00000114	0.00000000
		-6.27316569	-0.00041892	0.00025664	0.00042103	-0.00000114	0.00000000
		-6.27316583	-0.00041886	0.00025661	0.00042097	-0.00000114	0.00000000
		-6.27316596	-0.00041881	0.00025658	0.00042092	-0.00000114	0.00000000
		-6.27316610	-0.00041875	0.00025654	0.00042086	-0.00000114	0.00000000
		-6.27316623	-0.00041869	0.00025651	0.00042080	-0.00000114	0.00000000
		-6.27316637	-0.00041864	0.00025647	0.00042075	-0.00000114	0.00000000
		-6.27316650	-0.00041858	0.00025644	0.00042069	-0.00000114	0.00000000
		-6.27316664	-0.00041852	0.00025640	0.00042063	-0.00000114	0.00000000
		-6.27316677	-0.00041847	0.00025637	0.00042058	-0.00000114	0.00000000
		-6.27316691	-0.00041841	0.00025633	0.00042052	-0.00000114	0.00000000
		-6.27316705	-0.00041835	0.00025630	0.00042046	-0.00000114	0.00000000
		-6.27316718	-0.00041830	0.00025626	0.00042041	-0.00000114	0.00000000
		-6.27316732	-0.00041824	0.00025623	0.00042035	-0.00000114	0.00000000
		-6.27316745	-0.00041818	0.00025619	0.00042029	-0.00000114	0.00000000
		-6.27316759	-0.00041813	0.00025616	0.00042023	-0.00000114	0.00000000
		-6.27316772	-0.00041807	0.00025612	0.00042018	-0.00000114	0.00000000
		-6.27316786	-0.00041801	0.00025609	0.00042012	-0.00000114	0.00000000
		-6.27316799	-0.00041796	0.00025606	0.00042006	-0.00000114	0.00000000
		-6.27316813	-0.00041790	0.00025602	0.00042001	-0.00000114	0.00000000
		-6.27316826	-0.00041784	0.00025599	0.00041995	-0.00000114	0.00000000
		-6.27316840	-0.00041779	0.00025595	0.00041989	-0.00000114	0.00000000
		-6.27316853	-0.00041773	0.00025592	0.00041984	-0.00000114	0.00000000
		-6.27316867	-0.00041768	0.00025588	0.00041978	-0.00000114	0.00000000
		-6.27316880	-0.00041762	0.00025585	0.00041972	-0.00000114	0.00000000
		-6.27316894	-0.00041756	0.00025581	0.00041967	-0.00000114	0.00000000
		-6.27316907	-0.00041751	0.00025578	0.00041961	-0.00000114	0.00000000
		-6.27316921	-0.00041745	0.00025574	0.00041955	-0.00000114	0.00000000
		-6.27316934	-0.00041739	0.00025571	0.00041950	-0.00000114	0.00000000
		-6.27316948	-0.00041734	0.00025568	0.00041944	-0.00000114	0.00000000
		-6.27316961	-0.00041728	0.00025564	0.00041939	-0.00000114	0.00000000
		-6.27316974	-0.00041722	0.00025561	0.00041933	-0.00000114	0.00000000
		-6.27316988	-0.00041717	0.00025557	0.00041927	-0.00000114	0.00000000
		-6.27317001	-0.00041711	0.00025554	0.00041922	-0.00000114	0.00000000
		-6.27317015	-0.00041706	0.00025550	0.00041916	-0.00000114	0.00000000
		-6.27317028	-0.00041700	0.00025547	0.00041910	-0.00000114	0.00000000
		-6.27317042	-0.00041694	0.00025543	0.00041905	-0.00000114	0.00000000
		-6.27317055	-0.00041689	0.00025540	0.00041899	-0.00000114	0.00000000
		-6.27317069	-0.00041683	0.00025537	0.00041893	-0.00000114	0.00000000
		-6.27317082	-0.00041678	0.00025533	0.00041888	-0.00000114	0.00000000
		-6.27317095	-0.00041672	0.00025530	0.00041882	-0.00000114	0.00000000
		-6.27317109	-0.00041666	0.00025526	0.00041876	-0.00000114	0.00000000
		-6.27317122	-0.00041661	0.00025523	0.00041871	-0.00000114	0.00000000
		-6.27317136	-0.00041655	0.00025519	0.00041865	-0.00000114	0.00000000
		-6.27317149	-0.00041649	0.00025516	0.00041860	-0.00000114	0.00000000
		-6.27317163	-0.00041644	0.00025512	0.00041854	-0.00000114	0.00000000
		-6.27317176	-0.00041638	0.00025509	0.00041848	-0.00000114	0.00000000
		-6.27317189	-0.00041633	0.00025506	0.00041843	-0.00000114	0.00000000
		-6.27317203	-0.00041627	0.00025502	0.00041837	-0.00000114	0.00000000
		-6.27317216	-0.00041621	0.00025499	0.00041831	-0.00000114	0.00000000
		-6.27317230	-0.00041616	0.00025495	0.00041826	-0.00000114	0.00000000
		-6.27317243	-0.00041610	0.00025492	0.00041820	-0.00000114	0.00000000
		-6.27317256	-0.00041605	0.00025488	0.00041814	-0.00000114	0.00000000
		-6.27317270	-0.00041599	0.00025485	0.00041809	-0.00000114	0.00000000
		-6.27317283	-0.00041594	0.00025482	0.00041803	-0.00000114	0.00000000
		-6.27317297	-0.00041588	0.00025478	0.00041798	-0.00000114	0.00000000
		-6.27317310	-0.00041582	0.00025475	0.00041792	-0.00000115	0.00000000
		-6.27317323	-0.00041577	0.00025471	0.00041786	-0.00000115	0.00000000
		-6.27317337	-0.00041571	0.00025468	0.00041781	-0.00000115	0.00000000
		-6.27317350	-0.00041566	0.00025465	0.00041775	-0.00000115	0.00000000
		-6.27317363	-0.00041560	0.00025461	0.00041770	-0.00000115	0.00000000
		-6.27317377	-0.00041554	0.00025458	0.00041764	-0.00000115	0.00000000
		-6.27317390	-0.00041549	0.00025454	0.00041758	-0.00000115	0.00000000
		-6.27317403	-0.00041543	0.00025451	0.00041753	-0.00000115	0.00000000
		-6.27317417	-0.00041538	0.00025447	0.00041747	-0.00000115	0.00000000
		-6.27317430	-0.00041532	0.00025444	0.00041742	-0.00000115	0.00000000
		-6.27317443	-0.00041527	0.00025441	0.00041736	-0.00000115	0.00000000
		-6.27317457	-0.00041521	0.00025437	0.00041730	-0.00000115	0.00000000
		-6.27317470	-0.00041515	0.00025434	0.00041725	-0.00000115	0.00000000
		-6.27317483	-0.00041510	0.00025430	0.00041719	-0.00000115	0.00000000
		-6.27317497	-0.00041504	0.00025427	0.00041714	-0.00000115	0.00000000
		-6.27317510	-0.00041499	0.00025424	0.00041708	-0.00000115	0.00000000
		-6.27317523	-0.00041493	0.00025420	0.00041702	-0.00000115	0.00000000
		-6.27317537	-0.00041488	0.00025417	0.00041697	-0.00000115	0.00000000
		-6.27317550	-0.00041482	0.00025413	0.00041691	-0.00000115	0.00000000
		-6.27317563	-0.00041476	0.00025410	0.00041686	-0.00000115	0.00000000
		-6.27317577	-0.00041471	0.00025406	0.00041680	-0.00000115	0.00000000
		-6.27317590	-0.00041465	0.00025403	0.00041674	-0.00000115	0.00000000
		-6.27317603	-0.00041460	0.00025400	0.00041669	-0.00000115	0.00000000
		-6.27317616	-0.00041454	0.00025396	0.00041663	-0.00000115	0.00000000
		-6.27317630	-0.00041449	0.00025393	0.00041658	-0.00000115	0.00000000
		-6.27317643	-0.00041443	0.00025389	0.00041652	-0.00000115	0.00000000
		-6.27317656	-0.00041438	0.00025386	0.00041647	-0.00000115	0.00000000
		-6.27317670	-0.00041432	0.00025383	0.00041641	-0.00000115	0.00000000
		-6.27317683	-0.00041427	0.00025379	0.00041635	-0.00000115	0.00000000
		-6.27317696	-0.00041421	0.00025376	0.00041630	-0.00000115	0.00000000
		-6.27317709	-0.00041415	0.00025372	0.00041624	-0.00000115	0.00000000
		-6.27317723	-0.00041410	0.00025369	0.00041619	-0.00000115	0.00000000
		-6.27317736	-0.00041404	0.00025366	0.00041613	-0.00000115	0.00000000
		-6.27317749	-0.00041399	0.00025362	0.00041608	-0.00000115	0.00000000
		-6.27317762	-0.00041393	0.00025359	0.00041602	-0.00000115	0.00000000
		-6.27317776	-0.00041388	0.00025355	0.00041596	-0.00000115	0.00000000
		-6.27317789	-0.00041382	0.00025352	0.00041591	-0.00000115	0.00000000
		-6.27317802	-0.00041377	0.00025349	0.00041585	-0.00000115	0.00000000
		-6.27317815	-0.00041371	0.00025345	0.00041580	-0.00000115	0.00000000
		-6.27317829	-0.00041366	0.00025342	0.00041574	-0.00000115	0.00000000
		-6.27317842	-0.00041360	0.00025339	0.00041569	-0.00000115	0.00000000
		-6.27317855	-0.00041355	0.00025335	0.00041563	-0.00000115	0.00000000
		-6.27317868	-0.00041349	0.00025332	0.00041558	-0.00000115	0.00000000
		-6.27317882	-0.00041344	0.00025328	0.00041552	-0.00000115	0.00000000
		-6.27317895	-0.00041338	0.00025325	0.00041546	-0.00000115	0.00000000
		-6.27317908	-0.00041333	0.00025322	0.00041541	-0.00000115	0.00000000
		-6.27317921	-0.00041327	0.00025318	0.00041535	-0.00000115	0.00000000
		-6.27317934	-0.00041321	0.00025315	0.00041530	-0.00000115	0.00000000
		-6.27317948	-0.00041316	0.00025311	0.00041524	-0.00000115	0.00000000
		-6.27317961	-0.00041310	0.00025308	0.00041519	-0.00000115	0.00000000
		-6.27317974	-0.00041305	0.00025305	0.00041513	-0.00000115	0.00000000
		-6.27317987	-0.00041299	0.00025301	0.00041508	-0.00000115	0.00000000
		-6.27318000	-0.00041294	0.00025298	0.00041502	-0.00000115	0.00000000
		-6.27318014	-0.00041288	0.00025295	0.00041497	-0.00000115	0.00000000
		-6.27318027	-0.00041283	0.00025291	0.00041491	-0.00000115	0.00000000
		-6.27318040	-0.00041277	0.00025288	0.00041485	-0.00000115	0.00000000
		-6.27318053	-0.00041272	0.00025284	0.00041480	-0.00000115	0.00000000
		-6.27318066	-0.00041266	0.00025281	0.00041474	-0.00000115	0.00000000
		-6.27318079	-0.00041261	0.00025278	0.00041469	-0.00000115	0.00000000
		-6.27318093	-0.00041255	0.00025274	0.00041463	-0.00000115	0.00000000
		-6.27318106	-0.00041250	0.00025271	0.00041458	-0.00000115	0.00000000
		-6.27318119	-0.00041244	0.00025268	0.00041452	-0.00000115	0.00000000
		-6.27318132	-0.00041239	0.00025264	0.00041447	-0.00000115	0.00000000
		-6.27318145	-0.00041233	0.00025261	0.00041441	-0.00000115	0.00000000
		-6.27318158	-0.00041228	0.00025258	0.00041436	-0.00000115	0.00000000
		-6.27318171	-0.00041222	0.00025254	0.00041430	-0.00000115	0.00000000
		-6.27318185	-0.00041217	0.00025251	0.00041425	-0.00000115	0.00000000
		-6.27318198	-0.00041211	0.00025247	0.00041419	-0.00000115	0.00000000
		-6.27318211	-0.00041206	0.00025244	0.00041414	-0.00000115	0.00000000
		-6.27318224	-0.00041200	0.00025241	0.00041408	-0.00000115	0.00000000
		-6.27318237	-0.00041195	0.00025237	0.00041403	-0.00000115	0.00000000
		-6.27318250	-0.00041190	0.00025234	0.00041397	-0.00000115	0.00000000
		-6.27318263	-0.00041184	0.00025231	0.00041392	-0.00000115	0.00000000
		-6.27318276	-0.00041179	0.00025227	0.00041386	-0.00000115	0.00000000
		-6.27318290	-0.00041173	0.00025224	0.00041381	-0.00000115	0.00000000
		-6.27318303	-0.00041168	0.00025221	0.00041375	-0.00000115	0.00000000
		-6.27318316	-0.00041162	0.00025217	0.00041370	-0.00000115	0.00000000
		-6.27318329	-0.00041157	0.00025214	0.00041364	-0.00000115	0.00000000
		-6.27318342	-0.00041151	0.00025210	0.00041359	-0.00000115	0.00000000
		-6.27318355	-0.00041146	0.00025207	0.00041353	-0.00000115	0.00000000
		-6.27318368	-0.00041140	0.00025204	0.00041348	-0.00000115	0.00000000
		-6.27318381	-0.00041135	0.00025200	0.00041342	-0.00000115	0.00000000
		-6.27318394	-0.00041129	0.00025197	0.00041337	-0.00000115	0.00000000
		-6.27318407	-0.00041124	0.00025194	0.00041331	-0.00000115	0.00000000
		-6.27318420	-0.00041118	0.00025190	0.00041326	-0.00000115	0.00000000
		-6.27318434	-0.00041113	0.00025187	0.00041320	-0.00000115	0.00000000
		-6.27318447	-0.00041107	0.00025184	0.00041315	-0.00000115	0.00000000
		-6.27318460	-0.00041102	0.00025180	0.00041309	-0.00000115	0.00000000
		-6.27318473	-0.00041097	0.00025177	0.00041304	-0.00000115	0.00000000
		-6.27318486	-0.00041091	0.00025174	0.00041298	-0.00000115	0.00000000
		-6.27318499	-0.00041086	0.00025170	0.00041293	-0.00000115	0.00000000
		-6.27318512	-0.00041080	0.00025167	0.00041287	-0.00000115	0.00000000
		-6.27318525	-0.00041075	0.00025164	0.00041282	-0.00000115	0.00000000
		-6.27318538	-0.00041069	0.00025160	0.00041276	-0.00000115	0.00000000
		-6.27318551	-0.00041064	0.00025157	0.00041271	-0.00000115	0.00000000
		-6.27318564	-0.00041058	0.00025154	0.00041265	-0.00000115	0.00000000
		-6.27318577	-0.00041053	0.00025150	0.00041260	-0.00000115	0.00000000
		-6.27318590	-0.00041048	0.00025147	0.00041254	-0.00000115	0.00000000
		-6.27318603	-0.00041042	0.00025144	0.00041249	-0.00000115	0.00000000
		-6.27318616	-0.00041037	0.00025140	0.00041243	-0.00000115	0.00000000
		-6.27318629	-0.00041031	0.00025137	0.00041238	-0.00000115	0.00000000
		-6.27318642	-0.00041026	0.00025134	0.00041233	-0.00000115	0.00000000
		-6.27318655	-0.00041020	0.00025130	0.00041227	-0.00000115	0.00000000
		-6.27318668	-0.00041015	0.00025127	0.00041222	-0.00000115	0.00000000
		-6.27318681	-0.00041009	0.00025124	0.00041216	-0.00000115	0.00000000
		-6.27318694	-0.00041004	0.00025120	0.00041211	-0.00000115	0.00000000
		-6.27318707	-0.00040999	0.00025117	0.00041205	-0.00000115	0.00000000
		-6.27318720	-0.00040993	0.00025114	0.00041200	-0.00000115	0.00000000
		-6.27318733	-0.00040988	0.00025110	0.00041194	-0.00000115	0.00000000
		-6.27318746	-0.00040982	0.00025107	0.00041189	-0.00000115	0.00000000
		-6.27318759	-0.00040977	0.00025104	0.00041183	-0.00000115	0.00000000
		-6.27318772	-0.00040971	0.00025100	0.00041178	-0.00000115	0.00000000
		-6.27318785	-0.00040966	0.00025097	0.00041173	-0.00000115	0.00000000
		-6.27318798	-0.00040961	0.00025094	0.00041167	-0.00000115	0.00000000
		-6.27318811	-0.00040955	0.00025090	0.00041162	-0.00000115	0.00000000
		-6.27318824	-0.00040950	0.00025087	0.00041156	-0.00000115	0.00000000
		-6.27318837	-0.00040944	0.00025084	0.00041151	-0.00000115	0.00000000
		-6.27318850	-0.00040939	0.00025080	0.00041145	-0.00000115	0.00000000
		-6.27318863	-0.00040934	0.00025077	0.00041140	-0.00000115	0.00000000
		-6.27318876	-0.00040928	0.00025074	0.00041134	-0.00000115	0.00000000
		-6.27318889	-0.00040923	0.00025070	0.00041129	-0.00000115	0.00000000
		-6.27318902	-0.00040917	0.00025067	0.00041124	-0.00000115	0.00000000
		-6.27318915	-0.00040912	0.00025064	0.00041118	-0.00000115	0.00000000
		-6.27318928	-0.00040906	0.00025060	0.00041113	-0.00000115	0.00000000
		-6.27318941	-0.00040901	0.00025057	0.00041107	-0.00000115	0.00000000
		-6.27318954	-0.00040896	0.00025054	0.00041102	-0.00000115	0.00000000
		-6.27318967	-0.00040890	0.00025051	0.00041096	-0.00000115	0.00000000
		-6.27318979	-0.00040885	0.00025047	0.00041091	-0.00000115	0.00000000
		-6.27318992	-0.00040879	0.00025044	0.00041086	-0.00000115	0.00000000
		-6.27319005	-0.00040874	0.00025041	0.00041080	-0.00000115	0.00000000
		-6.27319018	-0.00040869	0.00025037	0.00041075	-0.00000115	0.00000000
		-6.27319031	-0.00040863	0.00025034	0.00041069	-0.00000115	0.00000000
		-6.27319044	-0.00040858	0.00025031	0.00041064	-0.00000115	0.00000000
		-6.27319057	-0.00040852	0.00025027	0.00041058	-0.00000115	0.00000000
		-6.27319070	-0.00040847	0.00025024	0.00041053	-0.00000115	0.00000000
		-6.27319083	-0.00040842	0.00025021	0.00041048	-0.00000115	0.00000000
		-6.27319096	-0.00040836	0.00025017	0.00041042	-0.00000115	0.00000000
		-6.27319109	-0.00040831	0.00025014	0.00041037	-0.00000115	0.00000000
		-6.27319121	-0.00040826	0.00025011	0.00041031	-0.00000115	0.00000000
		-6.27319134	-0.00040820	0.00025008	0.00041026	-0.00000115	0.00000000
		-6.27319147	-0.00040815	0.00025004	0.00041021	-0.00000115	0.00000000
		-6.27319160	-0.00040809	0.00025001	0.00041015	-0.00000115	0.00000000
		-6.27319173	-0.00040804	0.00024998	0.00041010	-0.00000115	0.00000000
		-6.27319186	-0.00040799	0.00024994	0.00041004	-0.00000115	0.00000000
		-6.27319199	-0.00040793	0.00024991	0.00040999	-0.00000115	0.00000000
		-6.27319212	-0.00040788	0.00024988	0.00040993	-0.00000115	0.00000000
		-6.27319224	-0.00040783	0.00024985	0.00040988	-0.00000115	0.00000000
		-6.27319237	-0.00040777	0.00024981	0.00040983	-0.00000115	0.00000000
		-6.27319250	-0.00040772	0.00024978	0.00040977	-0.00000115	0.00000000
		-6.27319263	-0.00040766	0.00024975	0.00040972	-0.00000115	0.00000000
		-6.27319276	-0.00040761	0.00024971	0.00040966	-0.00000115	0.00000000
		-6.27319289	-0.00040756	0.00024968	0.00040961	-0.00000115	0.00000000
		-6.27319302	-0.00040750	0.00024965	0.00040956	-0.00000115	0.00000000
		-6.27319314	-0.00040745	0.00024961	0.00040950	-0.00000115	0.00000000
		-6.27319327	-0.00040740	0.00024958	0.00040945	-0.00000115	0.00000000
		-6.27319340	-0.00040734	0.00024955	0.00040940	-0.00000115	0.00000000
		-6.27319353	-0.00040729	0.00024952	0.00040934	-0.00000115	0.00000000
		-6.27319366	-0.00040724	0.00024948	0.00040929	-0.00000115	0.00000000
		-6.27319379	-0.00040718	0.00024945	0.00040923	-0.00000115	0.00000000
		-6.27319391	-0.00040713	0.00024942	0.00040918	-0.00000115	0.00000000
		-6.27319404	-0.00040707	0.00024938	0.00040913	-0.00000115	0.00000000
		-6.27319417	-0.00040702	0.00024935	0.00040907	-0.00000115	0.00000000
		-6.27319430	-0.00040697	0.00024932	0.00040902	-0.00000115	0.00000000
		-6.27319443	-0.00040691	0.00024929	0.00040896	-0.00000115	0.00000000
		-6.27319455	-0.00040686	0.00024925	0.00040891	-0.00000115	0.00000000
		-6.27319468	-0.00040681	0.00024922	0.00040886	-0.00000115	0.00000000
		-6.27319481	-0.00040675	0.00024919	0.00040880	-0.00000115	0.00000000
		-6.27319494	-0.00040670	0.00024916	0.00040875	-0.00000115	0.00000000
		-6.27319507	-0.00040665	0.00024912	0.00040870	-0.00000115	0.00000000
		-6.27319519	-0.00040659	0.00024909	0.00040864	-0.00000115	0.00000000
		-6.27319532	-0.00040654	0.00024906	0.00040859	-0.00000115	0.00000000
		-6.27319545	-0.00040649	0.00024902	0.00040854	-0.00000115	0.00000000
		-6.27319558	-0.00040643	0.00024899	0.00040848	-0.00000115	0.00000000
		-6.27319571	-0.00040638	0.00024896	0.00040843	-0.00000115	0.00000000
		-6.27319583	-0.00040633	0.00024893	0.00040837	-0.00000115	0.00000000
		-6.27319596	-0.00040627	0.00024889	0.00040832	-0.00000115	0.00000000
		-6.27319609	-0.00040622	0.00024886	0.00040827	-0.00000115	0.00000000
		-6.27319622	-0.00040617	0.00024883	0.00040821	-0.00000115	0.00000000
		-6.27319634	-0.00040611	0.00024880	0.00040816	-0.00000115	0.00000000
		-6.27319647	-0.00040606	0.00024876	0.00040811	-0.00000115	0.00000000
		-6.27319660	-0.00040601	0.00024873	0.00040805	-0.00000115	0.00000000
		-6.27319673	-0.00040595	0.00024870	0.00040800	-0.00000115	0.00000000
		-6.27319685	-0.00040590	0.00024867	0.00040795	-0.00000115	0.00000000
		-6.27319698	-0.00040585	0.00024863	0.00040789	-0.00000115	0.00000000
		-6.27319711	-0.00040579	0.00024860	0.00040784	-0.00000115	0.00000000
		-6.27319724	-0.00040574	0.00024857	0.00040779	-0.00000115	0.00000000
		-6.27319736	-0.00040569	0.00024853	0.00040773	-0.00000115	0.00000000
		-6.27319749	-0.00040563	0.00024850	0.00040768	-0.00000115	0.00000000
		-6.27319762	-0.00040558	0.00024847	0.00040763	-0.00000115	0.00000000
		-6.27319774	-0.00040553	0.00024844	0.00040757	-0.00000115	0.00000000
		-6.27319787	-0.00040547	0.00024840	0.00040752	-0.00000115	0.00000000
		-6.27319800	-0.00040542	0.00024837	0.00040746	-0.00000115	0.00000000
		-6.27319813	-0.00040537	0.00024834	0.00040741	-0.00000115	0.00000000
		-6.27319825	-0.00040532	0.00024831	0.00040736	-0.00000115	0.00000000
		-6.27319838	-0.00040526	0.00024827	0.00040730	-0.00000115	0.00000000
		-6.27319851	-0.00040521	0.00024824	0.00040725	-0.00000115	0.00000000
		-6.27319863	-0.00040516	0.00024821	0.00040720	-0.00000115	0.00000000
		-6.27319876	-0.00040510	0.00024818	0.00040714	-0.00000115	0.00000000
		-6.27319889	-0.00040505	0.00024814	0.00040709	-0.00000115	0.00000000
		-6.27319901	-0.00040500	0.00024811	0.00040704	-0.00000115	0.00000000
		-6.27319914	-0.00040494	0.00024808	0.00040699	-0.00000115	0.00000000
		-6.27319927	-0.00040489	0.00024805	0.00040693	-0.00000115	0.00000000
		-6.27319939	-0.00040484	0.00024801	0.00040688	-0.00000115	0.00000000
		-6.27319952	-0.00040479	0.00024798	0.00040683	-0.00000115	0.00000000
		-6.27319965	-0.00040473	0.00024795	0.00040677	-0.00000115	0.00000000
		-6.27319977	-0.00040468	0.00024792	0.00040672	-0.00000115	0.00000000
		-6.27319990	-0.00040463	0.00024788	0.00040667	-0.00000115	0.00000000
		-6.27320003	-0.00040457	0.00024785	0.00040661	-0.00000115	0.00000000
		-6.27320015	-0.00040452	0.00024782	0.00040656	-0.00000115	0.00000000
		-6.27320028	-0.00040447	0.00024779	0.00040651	-0.00000115	0.00000000
		-6.27320041	-0.00040442	0.00024776	0.00040645	-0.00000115	0.00000000
		-6.27320053	-0.00040436	0.00024772	0.00040640	-0.00000115	0.00000000
		-6.27320066	-0.00040431	0.00024769	0.00040635	-0.00000115	0.00000000
		-6.27320079	-0.00040426	0.00024766	0.00040629	-0.00000115	0.00000000
		-6.27320091	-0.00040420	0.00024763	0.00040624	-0.00000115	0.00000000
		-6.27320104	-0.00040415	0.00024759	0.00040619	-0.00000115	0.00000000
		-6.27320117	-0.00040410	0.00024756	0.00040613	-0.00000115	0.00000000
		-6.27320129	-0.00040405	0.00024753	0.00040608	-0.00000115	0.00000000
		-6.27320142	-0.00040399	0.00024750	0.00040603	-0.00000115	0.00000000
		-6.27320154	-0.00040394	0.00024746	0.00040598	-0.00000115	0.00000000
		-6.27320167	-0.00040389	0.00024743	0.00040592	-0.00000115	0.00000000
		-6.27320180	-0.00040383	0.00024740	0.00040587	-0.00000115	0.00000000
		-6.27320192	-0.00040378	0.00024737	0.00040582	-0.00000115	0.00000000
		-6.27320205	-0.00040373	0.00024733	0.00040576	-0.00000115	0.00000000
		-6.27320217	-0.00040368	0.00024730	0.00040571	-0.00000115	0.00000000
		-6.27320230	-0.00040362	0.00024727	0.00040566	-0.00000115	0.00000000
		-6.27320243	-0.00040357	0.00024724	0.00040561	-0.00000115	0.00000000
		-6.27320255	-0.00040352	0.00024721	0.00040555	-0.00000115	0.00000000
		-6.27320268	-0.00040347	0.00024717	0.00040550	-0.00000115	0.00000000
		-6.27320280	-0.00040341	0.00024714	0.00040545	-0.00000115	0.00000000
		-6.27320293	-0.00040336	0.00024711	0.00040539	-0.00000115	0.00000000
		-6.27320306	-0.00040331	0.00024708	0.00040534	-0.00000115	0.00000000
		-6.27320318	-0.00040326	0.00024704	0.00040529	-0.00000115	0.00000000
		-6.27320331	-0.00040320	0.00024701	0.00040524	-0.00000115	0.00000000
		-6.27320343	-0.00040315	0.00024698	0.00040518	-0.00000115	0.00000000
		-6.27320356	-0.00040310	0.00024695	0.00040513	-0.00000115	0.00000000
		-6.27320368	-0.00040305	0.00024692	0.00040508	-0.00000115	0.00000000
		-6.27320381	-0.00040299	0.00024688	0.00040502	-0.00000115	0.00000000
		-6.27320394	-0.00040294	0.00024685	0.00040497	-0.00000115	0.00000000
		-6.27320406	-0.00040289	0.00024682	0.00040492	-0.00000115	0.00000000
		-6.27320419	-0.00040284	0.00024679	0.00040487	-0.00000115	0.00000000
		-6.27320431	-0.00040278	0.00024676	0.00040481	-0.00000115	0.00000000
		-6.27320444	-0.00040273	0.00024672	0.00040476	-0.00000115	0.00000000
		-6.27320456	-0.00040268	0.00024669	0.00040471	-0.00000115	0.00000000
		-6.27320469	-0.00040263	0.00024666	0.00040466	-0.00000115	0.00000000
		-6.27320481	-0.00040257	0.00024663	0.00040460	-0.00000115	0.00000000
		-6.27320494	-0.00040252	0.00024659	0.00040455	-0.00000115	0.00000000
		-6.27320506	-0.00040247	0.00024656	0.00040450	-0.00000115	0.00000000
		-6.27320519	-0.00040242	0.00024653	0.00040445	-0.00000115	0.00000000
		-6.27320532	-0.00040237	0.00024650	0.00040439	-0.00000115	0.00000000
		-6.27320544	-0.00040231	0.00024647	0.00040434	-0.00000115	0.00000000
		-6.27320557	-0.00040226	0.00024643	0.00040429	-0.00000115	0.00000000
		-6.27320569	-0.00040221	0.00024640	0.00040424	-0.00000115	0.00000000
		-6.27320582	-0.00040216	0.00024637	0.00040418	-0.00000115	0.00000000
		-6.27320594	-0.00040210	0.00024634	0.00040413	-0.00000115	0.00000000
		-6.27320607	-0.00040205	0.00024631	0.00040408	-0.00000115	0.00000000
		-6.27320619	-0.00040200	0.00024627	0.00040403	-0.00000115	0.00000000
		-6.27320632	-0.00040195	0.00024624	0.00040397	-0.00000115	0.00000000
		-6.27320644	-0.00040190	0.00024621	0.00040392	-0.00000115	0.00000000
		-6.27320657	-0.00040184	0.00024618	0.00040387	-0.00000115	0.00000000
		-6.27320669	-0.00040179	0.00024615	0.00040382	-0.00000115	0.00000000
		-6.27320682	-0.00040174	0.00024611	0.00040376	-0.00000115	0.00000000
		-6.27320694	-0.00040169	0.00024608	0.00040371	-0.00000115	0.00000000
		-6.27320706	-0.00040163	0.00024605	0.00040366	-0.00000115	0.00000000
		-6.27320719	-0.00040158	0.00024602	0.00040361	-0.00000115	0.00000000
		-6.27320731	-0.00040153	0.00024599	0.00040355	-0.00000115	0.00000000
		-6.27320744	-0.00040148	0.00024595	0.00040350	-0.00000115	0.00000000
		-6.27320756	-0.00040143	0.00024592	0.00040345	-0.00000115	0.00000000
		-6.27320769	-0.00040137	0.00024589	0.00040340	-0.00000115	0.00000000
		-6.27320781	-0.00040132	0.00024586	0.00040334	-0.00000115	0.00000000
		-6.27320794	-0.00040127	0.00024583	0.00040329	-0.00000115	0.00000000
		-6.27320806	-0.00040122	0.00024580	0.00040324	-0.00000115	0.00000000
		-6.27320819	-0.00040117	0.00024576	0.00040319	-0.00000115	0.00000000
		-6.27320831	-0.00040111	0.00024573	0.00040313	-0.00000115	0.00000000
		-6.27320843	-0.00040106	0.00024570	0.00040308	-0.00000115	0.00000000
		-6.27320856	-0.00040101	0.00024567	0.00040303	-0.00000115	0.00000000
		-6.27320868	-0.00040096	0.00024564	0.00040298	-0.00000115	0.00000000
		-6.27320881	-0.00040091	0.00024560	0.00040293	-0.00000115	0.00000000
		-6.27320893	-0.00040085	0.00024557	0.00040287	-0.00000115	0.00000000
		-6.27320906	-0.00040080	0.00024554	0.00040282	-0.00000115	0.00000000
		-6.27320918	-0.00040075	0.00024551	0.00040277	-0.00000115	0.00000000
		-6.27320930	-0.00040070	0.00024548	0.00040272	-0.00000115	0.00000000
		-6.27320943	-0.00040065	0.00024545	0.00040267	-0.00000115	0.00000000
		-6.27320955	-0.00040059	0.00024541	0.00040261	-0.00000115	0.00000000
		-6.27320968	-0.00040054	0.00024538	0.00040256	-0.00000115	0.00000000
		-6.27320980	-0.00040049	0.00024535	0.00040251	-0.00000115	0.00000000
		-6.27320993	-0.00040044	0.00024532	0.00040246	-0.00000115	0.00000000
		-6.27321005	-0.00040039	0.00024529	0.00040240	-0.00000115	0.00000000
		-6.27321017	-0.00040034	0.00024525	0.00040235	-0.00000115	0.00000000
		-6.27321030	-0.00040028	0.00024522	0.00040230	-0.00000115	0.00000000
		-6.27321042	-0.00040023	0.00024519	0.00040225	-0.00000115	0.00000000
		-6.27321054	-0.00040018	0.00024516	0.00040220	-0.00000115	0.00000000
		-6.27321067	-0.00040013	0.00024513	0.00040214	-0.00000115	0.00000000
		-6.27321079	-0.00040008	0.00024510	0.00040209	-0.00000115	0.00000000
		-6.27321092	-0.00040003	0.00024506	0.00040204	-0.00000115	0.00000000
		-6.27321104	-0.00039997	0.00024503	0.00040199	-0.00000115	0.00000000
		-6.27321116	-0.00039992	0.00024500	0.00040194	-0.00000115	0.00000000
		-6.27321129	-0.00039987	0.00024497	0.00040189	-0.00000115	0.00000000
		-6.27321141	-0.00039982	0.00024494	0.00040183	-0.00000115	0.00000000
		-6.27321153	-0.00039977	0.00024491	0.00040178	-0.00000115	0.00000000
		-6.27321166	-0.00039972	0.00024487	0.00040173	-0.00000115	0.00000000
		-6.27321178	-0.00039966	0.00024484	0.00040168	-0.00000115	0.00000000
		-6.27321191	-0.00039961	0.00024481	0.00040163	-0.00000115	0.00000000
		-6.27321203	-0.00039956	0.00024478	0.00040157	-0.00000115	0.00000000
		-6.27321215	-0.00039951	0.00024475	0.00040152	-0.00000115	0.00000000
		-6.27321228	-0.00039946	0.00024472	0.00040147	-0.00000115	0.00000000
		-6.27321240	-0.00039941	0.00024468	0.00040142	-0.00000115	0.00000000
		-6.27321252	-0.00039935	0.00024465	0.00040137	-0.00000115	0.00000000
		-6.27321265	-0.00039930	0.00024462	0.00040131	-0.00000115	0.00000000
		-6.27321277	-0.00039925	0.00024459	0.00040126	-0.00000115	0.00000000
		-6.27321289	-0.00039920	0.00024456	0.00040121	-0.00000115	0.00000000
		-6.27321302	-0.00039915	0.00024453	0.00040116	-0.00000115	0.00000000
		-6.27321314	-0.00039910	0.00024450	0.00040111	-0.00000115	0.00000000
		-6.27321326	-0.00039905	0.00024446	0.00040106	-0.00000115	0.00000000
		-6.27321338	-0.00039899	0.00024443	0.00040100	-0.00000115	0.00000000
		-6.27321351	-0.00039894	0.00024440	0.00040095	-0.00000115	0.00000000
		-6.27321363	-0.00039889	0.00024437	0.00040090	-0.00000115	0.00000000
		-6.27321375	-0.00039884	0.00024434	0.00040085	-0.00000115	0.00000000
		-6.27321388	-0.00039879	0.00024431	0.00040080	-0.00000115	0.00000000
		-6.27321400	-0.00039874	0.00024427	0.00040075	-0.00000115	0.00000000
		-6.27321412	-0.00039869	0.00024424	0.00040069	-0.00000115	0.00000000
		-6.27321425	-0.00039863	0.00024421	0.00040064	-0.00000115	0.00000000
		-6.27321437	-0.00039858	0.00024418	0.00040059	-0.00000115	0.00000000
		-6.27321449	-0.00039853	0.00024415	0.00040054	-0.00000115	0.00000000
		-6.27321461	-0.00039848	0.00024412	0.00040049	-0.00000115	0.00000000
		-6.27321474	-0.00039843	0.00024409	0.00040044	-0.00000115	0.00000000
		-6.27321486	-0.00039838	0.00024405	0.00040039	-0.00000115	0.00000000
		-6.27321498	-0.00039833	0.00024402	0.00040033	-0.00000115	0.00000000
		-6.27321511	-0.00039828	0.00024399	0.00040028	-0.00000115	0.00000000
		-6.27321523	-0.00039822	0.00024396	0.00040023	-0.00000115	0.00000000
		-6.27321535	-0.00039817	0.00024393	0.00040018	-0.00000115	0.00000000
		-6.27321547	-0.00039812	0.00024390	0.00040013	-0.00000115	0.00000000
		-6.27321560	-0.00039807	0.00024387	0.00040008	-0.00000115	0.00000000
		-6.27321572	-0.00039802	0.00024383	0.00040002	-0.00000115	0.00000000
		-6.27321584	-0.00039797	0.00024380	0.00039997	-0.00000115	0.00000000
		-6.27321596	-0.00039792	0.00024377	0.00039992	-0.00000115	0.00000000
		-6.27321609	-0.00039787	0.00024374	0.00039987	-0.00000115	0.00000000
		-6.27321621	-0.00039781	0.00024371	0.00039982	-0.00000115	0.00000000
		-6.27321633	-0.00039776	0.00024368	0.00039977	-0.00000115	0.00000000
		-6.27321645	-0.00039771	0.00024365	0.00039972	-0.00000115	0.00000000
		-6.27321658	-0.00039766	0.00024362	0.00039966	-0.00000115	0.00000000
		-6.27321670	-0.00039761	0.00024358	0.00039961	-0.00000115	0.00000000
		-6.27321682	-0.00039756	0.00024355	0.00039956	-0.00000115	0.00000000
		-6.27321694	-0.00039751	0.00024352	0.00039951	-0.00000115	0.00000000
		-6.27321706	-0.00039746	0.00024349	0.00039946	-0.00000115	0.00000000
		-6.27321719	-0.00039741	0.00024346	0.00039941	-0.00000115	0.00000000
		-6.27321731	-0.00039736	0.00024343	0.00039936	-0.00000115	0.00000000
		-6.27321743	-0.00039730	0.00024340	0.00039931	-0.00000115	0.00000000
		-6.27321755	-0.00039725	0.00024337	0.00039925	-0.00000115	0.00000000
		-6.27321767	-0.00039720	0.00024333	0.00039920	-0.00000115	0.00000000
		-6.27321780	-0.00039715	0.00024330	0.00039915	-0.00000115	0.00000000
		-6.27321792	-0.00039710	0.00024327	0.00039910	-0.00000115	0.00000000
		-6.27321804	-0.00039705	0.00024324	0.00039905	-0.00000115	0.00000000
		-6.27321816	-0.00039700	0.00024321	0.00039900	-0.00000115	0.00000000
		-6.27321828	-0.00039695	0.00024318	0.00039895	-0.00000115	0.00000000
		-6.27321841	-0.00039690	0.00024315	0.00039890	-0.00000115	0.00000000
		-6.27321853	-0.00039685	0.00024312	0.00039884	-0.00000115	0.00000000
		-6.27321865	-0.00039680	0.00024308	0.00039879	-0.00000115	0.00000000
		-6.27321877	-0.00039674	0.00024305	0.00039874	-0.00000115	0.00000000
		-6.27321889	-0.00039669	0.00024302	0.00039869	-0.00000115	0.00000000
		-6.27321901	-0.00039664	0.00024299	0.00039864	-0.00000115	0.00000000
		-6.27321914	-0.00039659	0.00024296	0.00039859	-0.00000115	0.00000000
		-6.27321926	-0.00039654	0.00024293	0.00039854	-0.00000115	0.00000000
		-6.27321938	-0.00039649	0.00024290	0.00039849	-0.00000115	0.00000000
		-6.27321950	-0.00039644	0.00024287	0.00039844	-0.00000115	0.00000000
		-6.27321962	-0.00039639	0.00024284	0.00039839	-0.00000115	0.00000000
		-6.27321974	-0.00039634	0.00024280	0.00039833	-0.00000115	0.00000000
		-6.27321987	-0.00039629	0.00024277	0.00039828	-0.00000115	0.00000000
		-6.27321999	-0.00039624	0.00024274	0.00039823	-0.00000115	0.00000000
		-6.27322011	-0.00039619	0.00024271	0.00039818	-0.00000115	0.00000000
		-6.27322023	-0.00039613	0.00024268	0.00039813	-0.00000115	0.00000000
		-6.27322035	-0.00039608	0.00024265	0.00039808	-0.00000115	0.00000000
		-6.27322047	-0.00039603	0.00024262	0.00039803	-0.00000115	0.00000000
		-6.27322059	-0.00039598	0.00024259	0.00039798	-0.00000115	0.00000000
		-6.27322071	-0.00039593	0.00024256	0.00039793	-0.00000115	0.00000000
		-6.27322084	-0.00039588	0.00024252	0.00039788	-0.00000115	0.00000000
		-6.27322096	-0.00039583	0.00024249	0.00039782	-0.00000115	0.00000000
		-6.27322108	-0.00039578	0.00024246	0.00039777	-0.00000115	0.00000000
		-6.27322120	-0.00039573	0.00024243	0.00039772	-0.00000115	0.00000000
		-6.27322132	-0.00039568	0.00024240	0.00039767	-0.00000115	0.00000000
		-6.27322144	-0.00039563	0.00024237	0.00039762	-0.00000115	0.00000000
		-6.27322156	-0.00039558	0.00024234	0.00039757	-0.00000115	0.00000000
		-6.27322168	-0.00039553	0.00024231	0.00039752	-0.00000115	0.00000000
		-6.27322180	-0.00039548	0.00024228	0.00039747	-0.00000115	0.00000000
		-6.27322193	-0.00039543	0.00024225	0.00039742	-0.00000115	0.00000000
		-6.27322205	-0.00039538	0.00024221	0.00039737	-0.00000115	0.00000000
		-6.27322217	-0.00039533	0.00024218	0.00039732	-0.00000115	0.00000000
		-6.27322229	-0.00039528	0.00024215	0.00039727	-0.00000115	0.00000000
		-6.27322241	-0.00039522	0.00024212	0.00039722	-0.00000115	0.00000000
		-6.27322253	-0.00039517	0.00024209	0.00039716	-0.00000115	0.00000000
		-6.27322265	-0.00039512	0.00024206	0.00039711	-0.00000115	0.00000000
		-6.27322277	-0.00039507	0.00024203	0.00039706	-0.00000115	0.00000000
		-6.27322289	-0.00039502	0.00024200	0.00039701	-0.00000115	0.00000000
		-6.27322301	-0.00039497	0.00024197	0.00039696	-0.00000115	0.00000000
		-6.27322313	-0.00039492	0.00024194	0.00039691	-0.00000115	0.00000000
		-6.27322325	-0.00039487	0.00024191	0.00039686	-0.00000115	0.00000000
		-6.27322337	-0.00039482	0.00024187	0.00039681	-0.00000115	0.00000000
		-6.27322349	-0.00039477	0.00024184	0.00039676	-0.00000115	0.00000000
		-6.27322362	-0.00039472	0.00024181	0.00039671	-0.00000115	0.00000000
		-6.27322374	-0.00039467	0.00024178	0.00039666	-0.00000115	0.00000000
		-6.27322386	-0.00039462	0.00024175	0.00039661	-0.00000115	0.00000000
		-6.27322398	-0.00039457	0.00024172	0.00039656	-0.00000115	0.00000000
		-6.27322410	-0.00039452	0.00024169	0.00039651	-0.00000115	0.00000000
		-6.27322422	-0.00039447	0.00024166	0.00039646	-0.00000115	0.00000000
		-6.27322434	-0.00039442	0.00024163	0.00039641	-0.00000115	0.00000000
		-6.27322446	-0.00039437	0.00024160	0.00039635	-0.00000115	0.00000000
		-6.27322458	-0.00039432	0.00024157	0.00039630	-0.00000115	0.00000000
		-6.27322470	-0.00039427	0.00024154	0.00039625	-0.00000115	0.00000000
		-6.27322482	-0.00039422	0.00024150	0.00039620	-0.00000115	0.00000000
		-6.27322494	-0.00039417	0.00024147	0.00039615	-0.00000115	0.00000000
		-6.27322506	-0.00039412	0.00024144	0.00039610	-0.00000115	0.00000000
		-6.27322518	-0.00039407	0.00024141	0.00039605	-0.00000115	0.00000000
		-6.27322530	-0.00039402	0.00024138	0.00039600	-0.00000115	0.00000000
		-6.27322542	-0.00039397	0.00024135	0.00039595	-0.00000115	0.00000000
		-6.27322554	-0.00039392	0.00024132	0.00039590	-0.00000115	0.00000000
		-6.27322566	-0.00039387	0.00024129	0.00039585	-0.00000115	0.00000000
		-6.27322578	-0.00039382	0.00024126	0.00039580	-0.00000115	0.00000000
		-6.27322590	-0.00039377	0.00024123	0.00039575	-0.00000115	0.00000000
		-6.27322602	-0.00039372	0.00024120	0.00039570	-0.00000115	0.00000000
		-6.27322614	-0.00039367	0.00024117	0.00039565	-0.00000115	0.00000000
		-6.27322626	-0.00039362	0.00024114	0.00039560	-0.00000115	0.00000000
		-6.27322638	-0.00039357	0.00024111	0.00039555	-0.00000115	0.00000000
		-6.27322650	-0.00039352	0.00024107	0.00039550	-0.00000115	0.00000000
		-6.27322662	-0.00039347	0.00024104	0.00039545	-0.00000115	0.00000000
		-6.27322674	-0.00039342	0.00024101	0.00039540	-0.00000115	0.00000000
		-6.27322686	-0.00039337	0.00024098	0.00039535	-0.00000115	0.00000000
		-6.27322698	-0.00039332	0.00024095	0.00039530	-0.00000115	0.00000000
		-6.27322710	-0.00039327	0.00024092	0.00039525	-0.00000115	0.00000000
		-6.27322722	-0.00039322	0.00024089	0.00039520	-0.00000115	0.00000000
		-6.27322734	-0.00039317	0.00024086	0.00039515	-0.00000115	0.00000000
		-6.27322746	-0.00039312	0.00024083	0.00039510	-0.00000115	0.00000000
		-6.27322757	-0.00039307	0.00024080	0.00039505	-0.00000115	0.00000000
		-6.27322769	-0.00039302	0.00024077	0.00039500	-0.00000115	0.00000000
		-6.27322781	-0.00039297	0.00024074	0.00039495	-0.00000115	0.00000000
		-6.27322793	-0.00039292	0.00024071	0.00039490	-0.00000115	0.00000000
		-6.27322805	-0.00039287	0.00024068	0.00039485	-0.00000115	0.00000000
		-6.27322817	-0.00039282	0.00024065	0.00039480	-0.00000115	0.00000000
		-6.27322829	-0.00039277	0.00024062	0.00039475	-0.00000115	0.00000000
		-6.27322841	-0.00039272	0.00024059	0.00039470	-0.00000115	0.00000000
		-6.27322853	-0.00039267	0.00024055	0.00039465	-0.00000115	0.00000000
		-6.27322865	-0.00039262	0.00024052	0.00039460	-0.00000115	0.00000000
		-6.27322877	-0.00039257	0.00024049	0.00039455	-0.00000115	0.00000000
		-6.27322889	-0.00039252	0.00024046	0.00039450	-0.00000115	0.00000000
		-6.27322901	-0.00039247	0.00024043	0.00039445	-0.00000115	0.00000000
		-6.27322913	-0.00039242	0.00024040	0.00039440	-0.00000115	0.00000000
		-6.27322924	-0.00039237	0.00024037	0.00039435	-0.00000115	0.00000000
		-6.27322936	-0.00039232	0.00024034	0.00039430	-0.00000115	0.00000000
		-6.27322948	-0.00039227	0.00024031	0.00039425	-0.00000115	0.00000000
		-6.27322960	-0.00039222	0.00024028	0.00039420	-0.00000115	0.00000000
		-6.27322972	-0.00039217	0.00024025	0.00039415	-0.00000115	0.00000000
		-6.27322984	-0.00039212	0.00024022	0.00039410	-0.00000115	0.00000000
		-6.27322996	-0.00039207	0.00024019	0.00039405	-0.00000115	0.00000000
		-6.27323008	-0.00039202	0.00024016	0.00039400	-0.00000115	0.00000000
		-6.27323020	-0.00039197	0.00024013	0.00039395	-0.00000115	0.00000000
		-6.27323032	-0.00039192	0.00024010	0.00039390	-0.00000115	0.00000000
		-6.27323043	-0.00039187	0.00024007	0.00039385	-0.00000115	0.00000000
		-6.27323055	-0.00039182	0.00024004	0.00039380	-0.00000115	0.00000000
		-6.27323067	-0.00039177	0.00024001	0.00039375	-0.00000115	0.00000000
		-6.27323079	-0.00039172	0.00023998	0.00039370	-0.00000115	0.00000000
		-6.27323091	-0.00039167	0.00023995	0.00039365	-0.00000115	0.00000000
		-6.27323103	-0.00039162	0.00023992	0.00039360	-0.00000115	0.00000000
		-6.27323115	-0.00039158	0.00023989	0.00039355	-0.00000115	0.00000000
		-6.27323126	-0.00039153	0.00023985	0.00039350	-0.00000115	0.00000000
		-6.27323138	-0.00039148	0.00023982	0.00039345	-0.00000115	0.00000000
		-6.27323150	-0.00039143	0.00023979	0.00039340	-0.00000115	0.00000000
		-6.27323162	-0.00039138	0.00023976	0.00039335	-0.00000115	0.00000000
		-6.27323174	-0.00039133	0.00023973	0.00039330	-0.00000115	0.00000000
		-6.27323186	-0.00039128	0.00023970	0.00039325	-0.00000115	0.00000000
		-6.27323198	-0.00039123	0.00023967	0.00039320	-0.00000115	0.00000000
		-6.27323209	-0.00039118	0.00023964	0.00039315	-0.00000115	0.00000000
		-6.27323221	-0.00039113	0.00023961	0.00039310	-0.00000115	0.00000000
		-6.27323233	-0.00039108	0.00023958	0.00039305	-0.00000115	0.00000000
		-6.27323245	-0.00039103	0.00023955	0.00039300	-0.00000115	0.00000000
		-6.27323257	-0.00039098	0.00023952	0.00039295	-0.00000115	0.00000000
		-6.27323268	-0.00039093	0.00023949	0.00039290	-0.00000115	0.00000000
		-6.27323280	-0.00039088	0.00023946	0.00039285	-0.00000115	0.00000000
		-6.27323292	-0.00039083	0.00023943	0.00039280	-0.00000115	0.00000000
		-6.27323304	-0.00039078	0.00023940	0.00039275	-0.00000115	0.00000000
		-6.27323316	-0.00039073	0.00023937	0.00039270	-0.00000115	0.00000000
		-6.27323328	-0.00039069	0.00023934	0.00039265	-0.00000115	0.00000000
		-6.27323339	-0.00039064	0.00023931	0.00039260	-0.00000115	0.00000000
		-6.27323351	-0.00039059	0.00023928	0.00039255	-0.00000115	0.00000000
		-6.27323363	-0.00039054	0.00023925	0.00039250	-0.00000115	0.00000000
		-6.27323375	-0.00039049	0.00023922	0.00039245	-0.00000115	0.00000000
		-6.27323387	-0.00039044	0.00023919	0.00039241	-0.00000115	0.00000000
		-6.27323398	-0.00039039	0.00023916	0.00039236	-0.00000115	0.00000000
		-6.27323410	-0.00039034	0.00023913	0.00039231	-0.00000115	0.00000000
		-6.27323422	-0.00039029	0.00023910	0.00039226	-0.00000115	0.00000000
		-6.27323434	-0.00039024	0.00023907	0.00039221	-0.00000115	0.00000000
		-6.27323445	-0.00039019	0.00023904	0.00039216	-0.00000115	0.00000000
		-6.27323457	-0.00039014	0.00023901	0.00039211	-0.00000115	0.00000000
		-6.27323469	-0.00039009	0.00023898	0.00039206	-0.00000115	0.00000000
		-6.27323481	-0.00039005	0.00023895	0.00039201	-0.00000115	0.00000000
		-6.27323493	-0.00039000	0.00023892	0.00039196	-0.00000115	0.00000000
		-6.27323504	-0.00038995	0.00023889	0.00039191	-0.00000115	0.00000000
		-6.27323516	-0.00038990	0.00023886	0.00039186	-0.00000115	0.00000000
		-6.27323528	-0.00038985	0.00023883	0.00039181	-0.00000115	0.00000000
		-6.27323540	-0.00038980	0.00023880	0.00039176	-0.00000115	0.00000000
		-6.27323551	-0.00038975	0.00023877	0.00039171	-0.00000115	0.00000000
		-6.27323563	-0.00038970	0.00023874	0.00039166	-0.00000115	0.00000000
		-6.27323575	-0.00038965	0.00023871	0.00039161	-0.00000115	0.00000000
		-6.27323587	-0.00038960	0.00023868	0.00039157	-0.00000115	0.00000000
		-6.27323598	-0.00038955	0.00023865	0.00039152	-0.00000115	0.00000000
		-6.27323610	-0.00038951	0.00023862	0.00039147	-0.00000115	0.00000000
		-6.27323622	-0.00038946	0.00023859	0.00039142	-0.00000115	0.00000000
		-6.27323633	-0.00038941	0.00023856	0.00039137	-0.00000115	0.00000000
		-6.27323645	-0.00038936	0.00023853	0.00039132	-0.00000115	0.00000000
		-6.27323657	-0.00038931	0.00023850	0.00039127	-0.00000115	0.00000000
		-6.27323669	-0.00038926	0.00023847	0.00039122	-0.00000115	0.00000000
		-6.27323680	-0.00038921	0.00023844	0.00039117	-0.00000115	0.00000000
		-6.27323692	-0.00038916	0.00023841	0.00039112	-0.00000115	0.00000000
		-6.27323704	-0.00038911	0.00023838	0.00039107	-0.00000115	0.00000000
		-6.27323715	-0.00038907	0.00023835	0.00039102	-0.00000115	0.00000000
		-6.27323727	-0.00038902	0.00023832	0.00039097	-0.00000115	0.00000000
		-6.27323739	-0.00038897	0.00023829	0.00039093	-0.00000115	0.00000000
		-6.27323751	-0.00038892	0.00023826	0.00039088	-0.00000115	0.00000000
		-6.27323762	-0.00038887	0.00023823	0.00039083	-0.00000115	0.00000000
		-6.27323774	-0.00038882	0.00023820	0.00039078	-0.00000115	0.00000000
		-6.27323786	-0.00038877	0.00023817	0.00039073	-0.00000115	0.00000000
		-6.27323797	-0.00038872	0.00023814	0.00039068	-0.00000115	0.00000000
		-6.27323809	-0.00038867	0.00023811	0.00039063	-0.00000115	0.00000000
		-6.27323821	-0.00038863	0.00023808	0.00039058	-0.00000115	0.00000000
		-6.27323832	-0.00038858	0.00023805	0.00039053	-0.00000115	0.00000000
		-6.27323844	-0.00038853	0.00023802	0.00039048	-0.00000115	0.00000000
		-6.27323856	-0.00038848	0.00023799	0.00039044	-0.00000115	0.00000000
		-6.27323867	-0.00038843	0.00023796	0.00039039	-0.00000115	0.00000000
		-6.27323879	-0.00038838	0.00023793	0.00039034	-0.00000115	0.00000000
		-6.27323891	-0.00038833	0.00023790	0.00039029	-0.00000115	0.00000000
		-6.27323902	-0.00038828	0.00023787	0.00039024	-0.00000115	0.00000000
		-6.27323914	-0.00038824	0.00023784	0.00039019	-0.00000115	0.00000000
		-6.27323926	-0.00038819	0.00023781	0.00039014	-0.00000115	0.00000000
		-6.27323937	-0.00038814	0.00023778	0.00039009	-0.00000115	0.00000000
		-6.27323949	-0.00038809	0.00023775	0.00039004	-0.00000115	0.00000000
		-6.27323961	-0.00038804	0.00023772	0.00038999	-0.00000115	0.00000000
		-6.27323972	-0.00038799	0.00023769	0.00038995	-0.00000115	0.00000000
		-6.27323984	-0.00038794	0.00023766	0.00038990	-0.00000115	0.00000000
		-6.27323996	-0.00038789	0.00023763	0.00038985	-0.00000115	0.00000000
		-6.27324007	-0.00038785	0.00023760	0.00038980	-0.00000115	0.00000000
		-6.27324019	-0.00038780	0.00023757	0.00038975	-0.00000115	0.00000000
		-6.27324030	-0.00038775	0.00023754	0.00038970	-0.00000115	0.00000000
		-6.27324042	-0.00038770	0.00023751	0.00038965	-0.00000115	0.00000000
		-6.27324054	-0.00038765	0.00023748	0.00038960	-0.00000115	0.00000000
		-6.27324065	-0.00038760	0.00023745	0.00038956	-0.00000115	0.00000000
		-6.27324077	-0.00038755	0.00023742	0.00038951	-0.00000115	0.00000000
		-6.27324089	-0.00038751	0.00023739	0.00038946	-0.00000115	0.00000000
		-6.27324100	-0.00038746	0.00023736	0.00038941	-0.00000115	0.00000000
		-6.27324112	-0.00038741	0.00023733	0.00038936	-0.00000115	0.00000000
		-6.27324123	-0.00038736	0.00023730	0.00038931	-0.00000115	0.00000000
		-6.27324135	-0.00038731	0.00023727	0.00038926	-0.00000115	0.00000000
		-6.27324147	-0.00038726	0.00023724	0.00038921	-0.00000115	0.00000000
		-6.27324158	-0.00038722	0.00023721	0.00038917	-0.00000115	0.00000000
		-6.27324170	-0.00038717	0.00023718	0.00038912	-0.00000115	0.00000000
		-6.27324181	-0.00038712	0.00023715	0.00038907	-0.00000115	0.00000000
		-6.27324193	-0.00038707	0.00023712	0.00038902	-0.00000115	0.00000000
		-6.27324205	-0.00038702	0.00023709	0.00038897	-0.00000115	0.00000000
		-6.27324216	-0.00038697	0.00023706	0.00038892	-0.00000115	0.00000000
		-6.27324228	-0.00038693	0.00023704	0.00038887	-0.00000115	0.00000000
		-6.27324239	-0.00038688	0.00023701	0.00038882	-0.00000115	0.00000000
		-6.27324251	-0.00038683	0.00023698	0.00038878	-0.00000115	0.00000000
		-6.27324262	-0.00038678	0.00023695	0.00038873	-0.00000115	0.00000000
		-6.27324274	-0.00038673	0.00023692	0.00038868	-0.00000115	0.00000000
		-6.27324286	-0.00038668	0.00023689	0.00038863	-0.00000115	0.00000000
		-6.27324297	-0.00038664	0.00023686	0.00038858	-0.00000115	0.00000000
		-6.27324309	-0.00038659	0.00023683	0.00038853	-0.00000115	0.00000000
		-6.27324320	-0.00038654	0.00023680	0.00038848	-0.00000115	0.00000000
		-6.27324332	-0.00038649	0.00023677	0.00038844	-0.00000115	0.00000000
		-6.27324343	-0.00038644	0.00023674	0.00038839	-0.00000115	0.00000000
		-6.27324355	-0.00038639	0.00023671	0.00038834	-0.00000115	0.00000000
		-6.27324366	-0.00038635	0.00023668	0.00038829	-0.00000115	0.00000000
		-6.27324378	-0.00038630	0.00023665	0.00038824	-0.00000115	0.00000000
		-6.27324390	-0.00038625	0.00023662	0.00038819	-0.00000115	0.00000000
		-6.27324401	-0.00038620	0.00023659	0.00038815	-0.00000115	0.00000000
		-6.27324413	-0.00038615	0.00023656	0.00038810	-0.00000115	0.00000000
		-6.27324424	-0.00038610	0.00023653	0.00038805	-0.00000115	0.00000000
		-6.27324436	-0.00038606	0.00023650	0.00038800	-0.00000115	0.00000000
		-6.27324447	-0.00038601	0.00023647	0.00038795	-0.00000115	0.00000000
		-6.27324459	-0.00038596	0.00023644	0.00038790	-0.00000115	0.00000000
		-6.27324470	-0.00038591	0.00023641	0.00038786	-0.00000115	0.00000000
		-6.27324482	-0.00038586	0.00023638	0.00038781	-0.00000115	0.00000000
		-6.27324493	-0.00038582	0.00023636	0.00038776	-0.00000115	0.00000000
		-6.27324505	-0.00038577	0.00023633	0.00038771	-0.00000115	0.00000000
		-6.27324516	-0.00038572	0.00023630	0.00038766	-0.00000115	0.00000000
		-6.27324528	-0.00038567	0.00023627	0.00038761	-0.00000115	0.00000000
		-6.27324539	-0.00038562	0.00023624	0.00038757	-0.00000115	0.00000000
		-6.27324551	-0.00038558	0.00023621	0.00038752	-0.00000115	0.00000000
		-6.27324562	-0.00038553	0.00023618	0.00038747	-0.00000115	0.00000000
		-6.27324574	-0.00038548	0.00023615	0.00038742	-0.00000115	0.00000000
		-6.27324585	-0.00038543	0.00023612	0.00038737	-0.00000115	0.00000000
		-6.27324597	-0.00038538	0.00023609	0.00038732	-0.00000115	0.00000000
		-6.27324608	-0.00038534	0.00023606	0.00038728	-0.00000115	0.00000000
		-6.27324620	-0.00038529	0.00023603	0.00038723	-0.00000115	0.00000000
		-6.27324631	-0.00038524	0.00023600	0.00038718	-0.00000115	0.00000000
		-6.27324643	-0.00038519	0.00023597	0.00038713	-0.00000115	0.00000000
		-6.27324654	-0.00038514	0.00023594	0.00038708	-0.00000115	0.00000000
		-6.27324666	-0.00038510	0.00023591	0.00038703	-0.00000115	0.00000000
		-6.27324677	-0.00038505	0.00023588	0.00038699	-0.00000115	0.00000000
		-6.27324689	-0.00038500	0.00023586	0.00038694	-0.00000115	0.00000000
		-6.27324700	-0.00038495	0.00023583	0.00038689	-0.00000115	0.00000000
		-6.27324711	-0.00038490	0.00023580	0.00038684	-0.00000115	0.00000000
		-6.27324723	-0.00038486	0.00023577	0.00038679	-0.00000115	0.00000000
		-6.27324734	-0.00038481	0.00023574	0.00038675	-0.00000115	0.00000000
		-6.27324746	-0.00038476	0.00023571	0.00038670	-0.00000115	0.00000000
		-6.27324757	-0.00038471	0.00023568	0.00038665	-0.00000115	0.00000000
		-6.27324769	-0.00038467	0.00023565	0.00038660	-0.00000115	0.00000000
		-6.27324780	-0.00038462	0.00023562	0.00038655	-0.00000115	0.00000000
		-6.27324792	-0.00038457	0.00023559	0.00038651	-0.00000115	0.00000000
		-6.27324803	-0.00038452	0.00023556	0.00038646	-0.00000115	0.00000000
		-6.27324814	-0.00038447	0.00023553	0.00038641	-0.00000115	0.00000000
		-6.27324826	-0.00038443	0.00023550	0.00038636	-0.00000115	0.00000000
		-6.27324837	-0.00038438	0.00023547	0.00038631	-0.00000115	0.00000000
		-6.27324849	-0.00038433	0.00023545	0.00038627	-0.00000115	0.00000000
		-6.27324860	-0.00038428	0.00023542	0.00038622	-0.00000115	0.00000000
		-6.27324872	-0.00038424	0.00023539	0.00038617	-0.00000115	0.00000000
		-6.27324883	-0.00038419	0.00023536	0.00038612	-0.00000115	0.00000000
		-6.27324894	-0.00038414	0.00023533	0.00038607	-0.00000115	0.00000000
		-6.27324906	-0.00038409	0.00023530	0.00038603	-0.00000115	0.00000000
		-6.27324917	-0.00038405	0.00023527	0.00038598	-0.00000115	0.00000000
		-6.27324929	-0.00038400	0.00023524	0.00038593	-0.00000115	0.00000000
		-6.27324940	-0.00038395	0.00023521	0.00038588	-0.00000115	0.00000000
		-6.27324951	-0.00038390	0.00023518	0.00038583	-0.00000115	0.00000000
		-6.27324963	-0.00038385	0.00023515	0.00038579	-0.00000115	0.00000000
		-6.27324974	-0.00038381	0.00023512	0.00038574	-0.00000115	0.00000000
		-6.27324986	-0.00038376	0.00023509	0.00038569	-0.00000115	0.00000000
		-6.27324997	-0.00038371	0.00023507	0.00038564	-0.00000115	0.00000000
		-6.27325008	-0.00038366	0.00023504	0.00038560	-0.00000115	0.00000000
		-6.27325020	-0.00038362	0.00023501	0.00038555	-0.00000115	0.00000000
		-6.27325031	-0.00038357	0.00023498	0.00038550	-0.00000115	0.00000000
		-6.27325043	-0.00038352	0.00023495	0.00038545	-0.00000115	0.00000000
		-6.27325054	-0.00038347	0.00023492	0.00038540	-0.00000115	0.00000000
		-6.27325065	-0.00038343	0.00023489	0.00038536	-0.00000115	0.00000000
		-6.27325077	-0.00038338	0.00023486	0.00038531	-0.00000115	0.00000000
		-6.27325088	-0.00038333	0.00023483	0.00038526	-0.00000115	0.00000000
		-6.27325099	-0.00038328	0.00023480	0.00038521	-0.00000115	0.00000000
		-6.27325111	-0.00038324	0.00023477	0.00038517	-0.00000115	0.00000000
		-6.27325122	-0.00038319	0.00023475	0.00038512	-0.00000115	0.00000000
		-6.27325133	-0.00038314	0.00023472	0.00038507	-0.00000115	0.00000000
		-6.27325145	-0.00038309	0.00023469	0.00038502	-0.00000115	0.00000000
		-6.27325156	-0.00038305	0.00023466	0.00038498	-0.00000115	0.00000000
		-6.27325168	-0.00038300	0.00023463	0.00038493	-0.00000115	0.00000000
		-6.27325179	-0.00038295	0.00023460	0.00038488	-0.00000115	0.00000000
		-6.27325190	-0.00038291	0.00023457	0.00038483	-0.00000115	0.00000000
		-6.27325202	-0.00038286	0.00023454	0.00038478	-0.00000115	0.00000000
		-6.27325213	-0.00038281	0.00023451	0.00038474	-0.00000115	0.00000000
		-6.27325224	-0.00038276	0.00023448	0.00038469	-0.00000115	0.00000000
		-6.27325236	-0.00038272	0.00023446	0.00038464	-0.00000115	0.00000000
		-6.27325247	-0.00038267	0.00023443	0.00038459	-0.00000115	0.00000000
		-6.27325258	-0.00038262	0.00023440	0.00038455	-0.00000115	0.00000000
		-6.27325269	-0.00038257	0.00023437	0.00038450	-0.00000115	0.00000000
		-6.27325281	-0.00038253	0.00023434	0.00038445	-0.00000115	0.00000000
		-6.27325292	-0.00038248	0.00023431	0.00038440	-0.00000115	0.00000000
		-6.27325303	-0.00038243	0.00023428	0.00038436	-0.00000115	0.00000000
		-6.27325315	-0.00038238	0.00023425	0.00038431	-0.00000115	0.00000000
		-6.27325326	-0.00038234	0.00023422	0.00038426	-0.00000115	0.00000000
		-6.27325337	-0.00038229	0.00023419	0.00038421	-0.00000115	0.00000000
		-6.27325349	-0.00038224	0.00023417	0.00038417	-0.00000115	0.00000000
		-6.27325360	-0.00038220	0.00023414	0.00038412	-0.00000115	0.00000000
		-6.27325371	-0.00038215	0.00023411	0.00038407	-0.00000115	0.00000000
		-6.27325383	-0.00038210	0.00023408	0.00038402	-0.00000115	0.00000000
		-6.27325394	-0.00038205	0.00023405	0.00038398	-0.00000115	0.00000000
		-6.27325405	-0.00038201	0.00023402	0.00038393	-0.00000115	0.00000000
		-6.27325416	-0.00038196	0.00023399	0.00038388	-0.00000115	0.00000000
		-6.27325428	-0.00038191	0.00023396	0.00038384	-0.00000115	0.00000000
		-6.27325439	-0.00038187	0.00023393	0.00038379	-0.00000115	0.00000000
		-6.27325450	-0.00038182	0.00023391	0.00038374	-0.00000115	0.00000000
		-6.27325462	-0.00038177	0.00023388	0.00038369	-0.00000115	0.00000000
		-6.27325473	-0.00038172	0.00023385	0.00038365	-0.00000115	0.00000000
		-6.27325484	-0.00038168	0.00023382	0.00038360	-0.00000115	0.00000000
		-6.27325495	-0.00038163	0.00023379	0.00038355	-0.00000115	0.00000000
		-6.27325507	-0.00038158	0.00023376	0.00038350	-0.00000115	0.00000000
		-6.27325518	-0.00038154	0.00023373	0.00038346	-0.00000115	0.00000000
		-6.27325529	-0.00038149	0.00023370	0.00038341	-0.00000115	0.00000000
		-6.27325540	-0.00038144	0.00023367	0.00038336	-0.00000115	0.00000000
		-6.27325552	-0.00038140	0.00023365	0.00038332	-0.00000115	0.00000000
		-6.27325563	-0.00038135	0.00023362	0.00038327	-0.00000115	0.00000000
		-6.27325574	-0.00038130	0.00023359	0.00038322	-0.00000115	0.00000000
		-6.27325585	-0.00038125	0.00023356	0.00038317	-0.00000115	0.00000000
		-6.27325597	-0.00038121	0.00023353	0.00038313	-0.00000115	0.00000000
		-6.27325608	-0.00038116	0.00023350	0.00038308	-0.00000115	0.00000000
		-6.27325619	-0.00038111	0.00023347	0.00038303	-0.00000115	0.00000000
		-6.27325630	-0.00038107	0.00023344	0.00038298	-0.00000115	0.00000000
		-6.27325642	-0.00038102	0.00023342	0.00038294	-0.00000115	0.00000000
		-6.27325653	-0.00038097	0.00023339	0.00038289	-0.00000115	0.00000000
		-6.27325664	-0.00038093	0.00023336	0.00038284	-0.00000115	0.00000000
		-6.27325675	-0.00038088	0.00023333	0.00038280	-0.00000115	0.00000000
		-6.27325686	-0.00038083	0.00023330	0.00038275	-0.00000115	0.00000000
		-6.27325698	-0.00038079	0.00023327	0.00038270	-0.00000115	0.00000000
		-6.27325709	-0.00038074	0.00023324	0.00038265	-0.00000115	0.00000000
		-6.27325720	-0.00038069	0.00023321	0.00038261	-0.00000115	0.00000000
		-6.27325731	-0.00038065	0.00023319	0.00038256	-0.00000115	0.00000000
		-6.27325742	-0.00038060	0.00023316	0.00038251	-0.00000115	0.00000000
		-6.27325754	-0.00038055	0.00023313	0.00038247	-0.00000115	0.00000000
		-6.27325765	-0.00038050	0.00023310	0.00038242	-0.00000115	0.00000000
		-6.27325776	-0.00038046	0.00023307	0.00038237	-0.00000115	0.00000000
		-6.27325787	-0.00038041	0.00023304	0.00038233	-0.00000115	0.00000000
		-6.27325798	-0.00038036	0.00023301	0.00038228	-0.00000115	0.00000000
		-6.27325810	-0.00038032	0.00023299	0.00038223	-0.00000115	0.00000000
		-6.27325821	-0.00038027	0.00023296	0.00038218	-0.00000115	0.00000000
		-6.27325832	-0.00038022	0.00023293	0.00038214	-0.00000115	0.00000000
		-6.27325843	-0.00038018	0.00023290	0.00038209	-0.00000115	0.00000000
		-6.27325854	-0.00038013	0.00023287	0.00038204	-0.00000115	0.00000000
		-6.27325866	-0.00038008	0.00023284	0.00038200	-0.00000115	0.00000000
		-6.27325877	-0.00038004	0.00023281	0.00038195	-0.00000115	0.00000000
		-6.27325888	-0.00037999	0.00023279	0.00038190	-0.00000115	0.00000000
		-6.27325899	-0.00037994	0.00023276	0.00038186	-0.00000115	0.00000000
		-6.27325910	-0.00037990	0.00023273	0.00038181	-0.00000115	0.00000000
		-6.27325921	-0.00037985	0.00023270	0.00038176	-0.00000115	0.00000000
		-6.27325932	-0.00037980	0.00023267	0.00038172	-0.00000115	0.00000000
		-6.27325944	-0.00037976	0.00023264	0.00038167	-0.00000115	0.00000000
		-6.27325955	-0.00037971	0.00023261	0.00038162	-0.00000115	0.00000000
		-6.27325966	-0.00037966	0.00023259	0.00038158	-0.00000115	0.00000000
		-6.27325977	-0.00037962	0.00023256	0.00038153	-0.00000115	0.00000000
		-6.27325988	-0.00037957	0.00023253	0.00038148	-0.00000115	0.00000000
		-6.27325999	-0.00037953	0.00023250	0.00038144	-0.00000115	0.00000000
		-6.27326011	-0.00037948	0.00023247	0.00038139	-0.00000115	0.00000000
		-6.27326022	-0.00037943	0.00023244	0.00038134	-0.00000115	0.00000000
		-6.27326033	-0.00037939	0.00023241	0.00038129	-0.00000115	0.00000000
		-6.27326044	-0.00037934	0.00023239	0.00038125	-0.00000115	0.00000000
		-6.27326055	-0.00037929	0.00023236	0.00038120	-0.00000115	0.00000000
		-6.27326066	-0.00037925	0.00023233	0.00038115	-0.00000115	0.00000000
		-6.27326077	-0.00037920	0.00023230	0.00038111	-0.00000115	0.00000000
		-6.27326088	-0.00037915	0.00023227	0.00038106	-0.00000115	0.00000000
		-6.27326100	-0.00037911	0.00023224	0.00038101	-0.00000115	0.00000000
		-6.27326111	-0.00037906	0.00023221	0.00038097	-0.00000115	0.00000000
		-6.27326122	-0.00037901	0.00023219	0.00038092	-0.00000115	0.00000000
		-6.27326133	-0.00037897	0.00023216	0.00038087	-0.00000115	0.00000000
		-6.27326144	-0.00037892	0.00023213	0.00038083	-0.00000115	0.00000000
		-6.27326155	-0.00037887	0.00023210	0.00038078	-0.00000115	0.00000000
		-6.27326166	-0.00037883	0.00023207	0.00038073	-0.00000115	0.00000000
		-6.27326177	-0.00037878	0.00023204	0.00038069	-0.00000115	0.00000000
		-6.27326188	-0.00037874	0.00023202	0.00038064	-0.00000115	0.00000000
		-6.27326199	-0.00037869	0.00023199	0.00038060	-0.00000115	0.00000000
		-6.27326211	-0.00037864	0.00023196	0.00038055	-0.00000115	0.00000000
		-6.27326222	-0.00037860	0.00023193	0.00038050	-0.00000115	0.00000000
		-6.27326233	-0.00037855	0.00023190	0.00038046	-0.00000115	0.00000000
		-6.27326244	-0.00037850	0.00023187	0.00038041	-0.00000115	0.00000000
		-6.27326255	-0.00037846	0.00023185	0.00038036	-0.00000115	0.00000000
		-6.27326266	-0.00037841	0.00023182	0.00038032	-0.00000115	0.00000000
		-6.27326277	-0.00037837	0.00023179	0.00038027	-0.00000115	0.00000000
		-6.27326288	-0.00037832	0.00023176	0.00038022	-0.00000115	0.00000000
		-6.27326299	-0.00037827	0.00023173	0.00038018	-0.00000115	0.00000000
		-6.27326310	-0.00037823	0.00023170	0.00038013	-0.00000115	0.00000000
		-6.27326321	-0.00037818	0.00023168	0.00038008	-0.00000115	0.00000000
		-6.27326332	-0.00037813	0.00023165	0.00038004	-0.00000115	0.00000000
		-6.27326343	-0.00037809	0.00023162	0.00037999	-0.00000115	0.00000000
		-6.27326354	-0.00037804	0.00023159	0.00037994	-0.00000115	0.00000000
		-6.27326366	-0.00037800	0.00023156	0.00037990	-0.00000115	0.00000000
		-6.27326377	-0.00037795	0.00023153	0.00037985	-0.00000115	0.00000000
		-6.27326388	-0.00037790	0.00023151	0.00037981	-0.00000115	0.00000000
		-6.27326399	-0.00037786	0.00023148	0.00037976	-0.00000115	0.00000000
		-6.27326410	-0.00037781	0.00023145	0.00037971	-0.00000115	0.00000000
		-6.27326421	-0.00037777	0.00023142	0.00037967	-0.00000115	0.00000000
		-6.27326432	-0.00037772	0.00023139	0.00037962	-0.00000115	0.00000000
		-6.27326443	-0.00037767	0.00023136	0.00037957	-0.00000115	0.00000000
		-6.27326454	-0.00037763	0.00023134	0.00037953	-0.00000115	0.00000000
		-6.27326465	-0.00037758	0.00023131	0.00037948	-0.00000115	0.00000000
		-6.27326476	-0.00037753	0.00023128	0.00037943	-0.00000115	0.00000000
		-6.27326487	-0.00037749	0.00023125	0.00037939	-0.00000115	0.00000000
		-6.27326498	-0.00037744	0.00023122	0.00037934	-0.00000115	0.00000000
		-6.27326509	-0.00037740	0.00023120	0.00037930	-0.00000115	0.00000000
		-6.27326520	-0.00037735	0.00023117	0.00037925	-0.00000115	0.00000000
		-6.27326531	-0.00037730	0.00023114	0.00037920	-0.00000115	0.00000000
		-6.27326542	-0.00037726	0.00023111	0.00037916	-0.00000115	0.00000000
		-6.27326553	-0.00037721	0.00023108	0.00037911	-0.00000115	0.00000000
		-6.27326564	-0.00037717	0.00023105	0.00037906	-0.00000115	0.00000000
		-6.27326575	-0.00037712	0.00023103	0.00037902	-0.00000115	0.00000000
		-6.27326586	-0.00037707	0.00023100	0.00037897	-0.00000115	0.00000000
		-6.27326597	-0.00037703	0.00023097	0.00037893	-0.00000115	0.00000000
		-6.27326608	-0.00037698	0.00023094	0.00037888	-0.00000115	0.00000000
		-6.27326619	-0.00037694	0.00023091	0.00037883	-0.00000115	0.00000000
		-6.27326630	-0.00037689	0.00023089	0.00037879	-0.00000115	0.00000000
		-6.27326641	-0.00037685	0.00023086	0.00037874	-0.00000115	0.00000000
		-6.27326652	-0.00037680	0.00023083	0.00037870	-0.00000115	0.00000000
		-6.27326663	-0.00037675	0.00023080	0.00037865	-0.00000115	0.00000000
		-6.27326674	-0.00037671	0.00023077	0.00037860	-0.00000115	0.00000000
		-6.27326685	-0.00037666	0.00023074	0.00037856	-0.00000115	0.00000000
		-6.27326696	-0.00037662	0.00023072	0.00037851	-0.00000115	0.00000000
		-6.27326707	-0.00037657	0.00023069	0.00037847	-0.00000115	0.00000000
		-6.27326718	-0.00037652	0.00023066	0.00037842	-0.00000115	0.00000000
		-6.27326729	-0.00037648	0.00023063	0.00037837	-0.00000115	0.00000000
		-6.27326740	-0.00037643	0.00023060	0.00037833	-0.00000115	0.00000000
		-6.27326751	-0.00037639	0.00023058	0.00037828	-0.00000115	0.00000000
		-6.27326762	-0.00037634	0.00023055	0.00037823	-0.00000115	0.00000000
		-6.27326773	-0.00037630	0.00023052	0.00037819	-0.00000115	0.00000000
		-6.27326783	-0.00037625	0.00023049	0.00037814	-0.00000115	0.00000000
		-6.27326794	-0.00037620	0.00023046	0.00037810	-0.00000115	0.00000000
		-6.27326805	-0.00037616	0.00023044	0.00037805	-0.00000115	0.00000000
		-6.27326816	-0.00037611	0.00023041	0.00037801	-0.00000115	0.00000000
		-6.27326827	-0.00037607	0.00023038	0.00037796	-0.00000115	0.00000000
		-6.27326838	-0.00037602	0.00023035	0.00037791	-0.00000115	0.00000000
		-6.27326849	-0.00037598	0.00023032	0.00037787	-0.00000115	0.00000000
		-6.27326860	-0.00037593	0.00023030	0.00037782	-0.00000115	0.00000000
		-6.27326871	-0.00037588	0.00023027	0.00037778	-0.00000115	0.00000000
		-6.27326882	-0.00037584	0.00023024	0.00037773	-0.00000115	0.00000000
		-6.27326893	-0.00037579	0.00023021	0.00037768	-0.00000115	0.00000000
		-6.27326904	-0.00037575	0.00023018	0.00037764	-0.00000115	0.00000000
		-6.27326915	-0.00037570	0.00023016	0.00037759	-0.00000115	0.00000000
		-6.27326926	-0.00037566	0.00023013	0.00037755	-0.00000115	0.00000000
		-6.27326937	-0.00037561	0.00023010	0.00037750	-0.00000115	0.00000000
		-6.27326947	-0.00037557	0.00023007	0.00037745	-0.00000115	0.00000000
		-6.27326958	-0.00037552	0.00023004	0.00037741	-0.00000115	0.00000000
		-6.27326969	-0.00037547	0.00023002	0.00037736	-0.00000115	0.00000000
		-6.27326980	-0.00037543	0.00022999	0.00037732	-0.00000115	0.00000000
		-6.27326991	-0.00037538	0.00022996	0.00037727	-0.00000115	0.00000000
		-6.27327002	-0.00037534	0.00022993	0.00037723	-0.00000115	0.00000000
		-6.27327013	-0.00037529	0.00022991	0.00037718	-0.00000115	0.00000000
		-6.27327024	-0.00037525	0.00022988	0.00037713	-0.00000115	0.00000000
		-6.27327035	-0.00037520	0.00022985	0.00037709	-0.00000115	0.00000000
		-6.27327045	-0.00037516	0.00022982	0.00037704	-0.00000115	0.00000000
		-6.27327056	-0.00037511	0.00022979	0.00037700	-0.00000115	0.00000000
		-6.27327067	-0.00037506	0.00022977	0.00037695	-0.00000115	0.00000000
		-6.27327078	-0.00037502	0.00022974	0.00037691	-0.00000115	0.00000000
		-6.27327089	-0.00037497	0.00022971	0.00037686	-0.00000115	0.00000000
		-6.27327100	-0.00037493	0.00022968	0.00037681	-0.00000115	0.00000000
		-6.27327111	-0.00037488	0.00022965	0.00037677	-0.00000115	0.00000000
		-6.27327122	-0.00037484	0.00022963	0.00037672	-0.00000115	0.00000000
		-6.27327132	-0.00037479	0.00022960	0.00037668	-0.00000115	0.00000000
		-6.27327143	-0.00037475	0.00022957	0.00037663	-0.00000115	0.00000000
		-6.27327154	-0.00037470	0.00022954	0.00037659	-0.00000115	0.00000000
		-6.27327165	-0.00037466	0.00022952	0.00037654	-0.00000115	0.00000000
		-6.27327176	-0.00037461	0.00022949	0.00037650	-0.00000115	0.00000000
		-6.27327187	-0.00037457	0.00022946	0.00037645	-0.00000115	0.00000000
		-6.27327198	-0.00037452	0.00022943	0.00037640	-0.00000115	0.00000000
		-6.27327208	-0.00037447	0.00022940	0.00037636	-0.00000115	0.00000000
		-6.27327219	-0.00037443	0.00022938	0.00037631	-0.00000115	0.00000000
		-6.27327230	-0.00037438	0.00022935	0.00037627	-0.00000115	0.00000000
		-6.27327241	-0.00037434	0.00022932	0.00037622	-0.00000115	0.00000000
		-6.27327252	-0.00037429	0.00022929	0.00037618	-0.00000115	0.00000000
		-6.27327263	-0.00037425	0.00022927	0.00037613	-0.00000115	0.00000000
		-6.27327273	-0.00037420	0.00022924	0.00037609	-0.00000115	0.00000000
		-6.27327284	-0.00037416	0.00022921	0.00037604	-0.00000115	0.00000000
		-6.27327295	-0.00037411	0.00022918	0.00037599	-0.00000115	0.00000000
		-6.27327306	-0.00037407	0.00022915	0.00037595	-0.00000115	0.00000000
		-6.27327317	-0.00037402	0.00022913	0.00037590	-0.00000115	0.00000000
		-6.27327328	-0.00037398	0.00022910	0.00037586	-0.00000115	0.00000000
		-6.27327338	-0.00037393	0.00022907	0.00037581	-0.00000115	0.00000000
		-6.27327349	-0.00037389	0.00022904	0.00037577	-0.00000115	0.00000000
		-6.27327360	-0.00037384	0.00022902	0.00037572	-0.00000115	0.00000000
		-6.27327371	-0.00037380	0.00022899	0.00037568	-0.00000115	0.00000000
		-6.27327382	-0.00037375	0.00022896	0.00037563	-0.00000115	0.00000000
		-6.27327392	-0.00037371	0.00022893	0.00037559	-0.00000115	0.00000000
		-6.27327403	-0.00037366	0.00022891	0.00037554	-0.00000115	0.00000000
		-6.27327414	-0.00037362	0.00022888	0.00037550	-0.00000115	0.00000000
		-6.27327425	-0.00037357	0.00022885	0.00037545	-0.00000115	0.00000000
		-6.27327436	-0.00037353	0.00022882	0.00037541	-0.00000115	0.00000000
		-6.27327446	-0.00037348	0.00022880	0.00037536	-0.00000115	0.00000000
		-6.27327457	-0.00037344	0.00022877	0.00037531	-0.00000115	0.00000000
		-6.27327468	-0.00037339	0.00022874	0.00037527	-0.00000115	0.00000000
		-6.27327479	-0.00037335	0.00022871	0.00037522	-0.00000115	0.00000000
		-6.27327490	-0.00037330	0.00022868	0.00037518	-0.00000115	0.00000000
		-6.27327500	-0.00037326	0.00022866	0.00037513	-0.00000115	0.00000000
		-6.27327511	-0.00037321	0.00022863	0.00037509	-0.00000115	0.00000000
		-6.27327522	-0.00037317	0.00022860	0.00037504	-0.00000115	0.00000000
		-6.27327533	-0.00037312	0.00022857	0.00037500	-0.00000115	0.00000000
		-6.27327543	-0.00037308	0.00022855	0.00037495	-0.00000115	0.00000000
		-6.27327554	-0.00037303	0.00022852	0.00037491	-0.00000115	0.00000000
		-6.27327565	-0.00037299	0.00022849	0.00037486	-0.00000115	0.00000000
		-6.27327576	-0.00037294	0.00022846	0.00037482	-0.00000115	0.00000000
		-6.27327586	-0.00037290	0.00022844	0.00037477	-0.00000115	0.00000000
		-6.27327597	-0.00037285	0.00022841	0.00037473	-0.00000115	0.00000000
		-6.27327608	-0.00037281	0.00022838	0.00037468	-0.00000115	0.00000000
		-6.27327619	-0.00037276	0.00022835	0.00037464	-0.00000115	0.00000000
		-6.27327629	-0.00037272	0.00022833	0.00037459	-0.00000115	0.00000000
		-6.27327640	-0.00037267	0.00022830	0.00037455	-0.00000115	0.00000000
		-6.27327651	-0.00037263	0.00022827	0.00037450	-0.00000115	0.00000000
		-6.27327662	-0.00037258	0.00022824	0.00037446	-0.00000115	0.00000000
		-6.27327672	-0.00037254	0.00022822	0.00037441	-0.00000115	0.00000000
		-6.27327683	-0.00037249	0.00022819	0.00037437	-0.00000115	0.00000000
		-6.27327694	-0.00037245	0.00022816	0.00037432	-0.00000115	0.00000000
		-6.27327705	-0.00037240	0.00022813	0.00037428	-0.00000115	0.00000000
		-6.27327715	-0.00037236	0.00022811	0.00037423	-0.00000115	0.00000000
		-6.27327726	-0.00037231	0.00022808	0.00037419	-0.00000115	0.00000000
		-6.27327737	-0.00037227	0.00022805	0.00037414	-0.00000115	0.00000000
		-6.27327747	-0.00037222	0.00022802	0.00037410	-0.00000115	0.00000000
		-6.27327758	-0.00037218	0.00022800	0.00037405	-0.00000115	0.00000000
		-6.27327769	-0.00037213	0.00022797	0.00037401	-0.00000115	0.00000000
		-6.27327780	-0.00037209	0.00022794	0.00037396	-0.00000115	0.00000000
		-6.27327790	-0.00037204	0.00022791	0.00037392	-0.00000115	0.00000000
		-6.27327801	-0.00037200	0.00022789	0.00037387	-0.00000115	0.00000000
		-6.27327812	-0.00037195	0.00022786	0.00037383	-0.00000115	0.00000000
		-6.27327822	-0.00037191	0.00022783	0.00037378	-0.00000115	0.00000000
		-6.27327833	-0.00037187	0.00022781	0.00037374	-0.00000115	0.00000000
		-6.27327844	-0.00037182	0.00022778	0.00037369	-0.00000115	0.00000000
		-6.27327855	-0.00037178	0.00022775	0.00037365	-0.00000115	0.00000000
		-6.27327865	-0.00037173	0.00022772	0.00037360	-0.00000115	0.00000000
		-6.27327876	-0.00037169	0.00022770	0.00037356	-0.00000115	0.00000000
		-6.27327887	-0.00037164	0.00022767	0.00037351	-0.00000115	0.00000000
		-6.27327897	-0.00037160	0.00022764	0.00037347	-0.00000115	0.00000000
		-6.27327908	-0.00037155	0.00022761	0.00037342	-0.00000115	0.00000000
		-6.27327919	-0.00037151	0.00022759	0.00037338	-0.00000115	0.00000000
		-6.27327929	-0.00037146	0.00022756	0.00037333	-0.00000115	0.00000000
		-6.27327940	-0.00037142	0.00022753	0.00037329	-0.00000115	0.00000000
		-6.27327951	-0.00037137	0.00022750	0.00037324	-0.00000115	0.00000000
		-6.27327961	-0.00037133	0.00022748	0.00037320	-0.00000115	0.00000000
		-6.27327972	-0.00037129	0.00022745	0.00037315	-0.00000115	0.00000000
		-6.27327983	-0.00037124	0.00022742	0.00037311	-0.00000115	0.00000000
		-6.27327993	-0.00037120	0.00022740	0.00037306	-0.00000115	0.00000000
		-6.27328004	-0.00037115	0.00022737	0.00037302	-0.00000115	0.00000000
		-6.27328015	-0.00037111	0.00022734	0.00037297	-0.00000115	0.00000000
		-6.27328025	-0.00037106	0.00022731	0.00037293	-0.00000115	0.00000000
		-6.27328036	-0.00037102	0.00022729	0.00037289	-0.00000115	0.00000000
		-6.27328047	-0.00037097	0.00022726	0.00037284	-0.00000115	0.00000000
		-6.27328057	-0.00037093	0.00022723	0.00037280	-0.00000115	0.00000000
		-6.27328068	-0.00037089	0.00022720	0.00037275	-0.00000115	0.00000000
		-6.27328078	-0.00037084	0.00022718	0.00037271	-0.00000115	0.00000000
		-6.27328089	-0.00037080	0.00022715	0.00037266	-0.00000115	0.00000000
		-6.27328100	-0.00037075	0.00022712	0.00037262	-0.00000115	0.00000000
		-6.27328110	-0.00037071	0.00022710	0.00037257	-0.00000115	0.00000000
		-6.27328121	-0.00037066	0.00022707	0.00037253	-0.00000115	0.00000000
		-6.27328132	-0.00037062	0.00022704	0.00037248	-0.00000115	0.00000000
		-6.27328142	-0.00037057	0.00022701	0.00037244	-0.00000115	0.00000000
		-6.27328153	-0.00037053	0.00022699	0.00037239	-0.00000115	0.00000000
		-6.27328163	-0.00037049	0.00022696	0.00037235	-0.00000115	0.00000000
		-6.27328174	-0.00037044	0.00022693	0.00037230	-0.00000115	0.00000000
		-6.27328185	-0.00037040	0.00022691	0.00037226	-0.00000115	0.00000000
		-6.27328195	-0.00037035	0.00022688	0.00037222	-0.00000115	0.00000000
		-6.27328206	-0.00037031	0.00022685	0.00037217	-0.00000115	0.00000000
		-6.27328217	-0.00037026	0.00022682	0.00037213	-0.00000115	0.00000000
		-6.27328227	-0.00037022	0.00022680	0.00037208	-0.00000115	0.00000000
		-6.27328238	-0.00037018	0.00022677	0.00037204	-0.00000115	0.00000000
		-6.27328248	-0.00037013	0.00022674	0.00037199	-0.00000115	0.00000000
		-6.27328259	-0.00037009	0.00022672	0.00037195	-0.00000115	0.00000000
		-6.27328269	-0.00037004	0.00022669	0.00037190	-0.00000115	0.00000000
		-6.27328280	-0.00037000	0.00022666	0.00037186	-0.00000115	0.00000000
		-6.27328291	-0.00036995	0.00022663	0.00037182	-0.00000115	0.00000000
		-6.27328301	-0.00036991	0.00022661	0.00037177	-0.00000115	0.00000000
		-6.27328312	-0.00036987	0.00022658	0.00037173	-0.00000115	0.00000000
		-6.27328322	-0.00036982	0.00022655	0.00037168	-0.00000115	0.00000000
		-6.27328333	-0.00036978	0.00022653	0.00037164	-0.00000115	0.00000000
		-6.27328344	-0.00036973	0.00022650	0.00037159	-0.00000115	0.00000000
		-6.27328354	-0.00036969	0.00022647	0.00037155	-0.00000115	0.00000000
		-6.27328365	-0.00036965	0.00022644	0.00037150	-0.00000115	0.00000000
		-6.27328375	-0.00036960	0.00022642	0.00037146	-0.00000115	0.00000000
		-6.27328386	-0.00036956	0.00022639	0.00037142	-0.00000115	0.00000000
		-6.27328396	-0.00036951	0.00022636	0.00037137	-0.00000115	0.00000000
		-6.27328407	-0.00036947	0.00022634	0.00037133	-0.00000115	0.00000000
		-6.27328418	-0.00036942	0.00022631	0.00037128	-0.00000115	0.00000000
		-6.27328428	-0.00036938	0.00022628	0.00037124	-0.00000115	0.00000000
		-6.27328439	-0.00036934	0.00022626	0.00037119	-0.00000115	0.00000000
		-6.27328449	-0.00036929	0.00022623	0.00037115	-0.00000115	0.00000000
		-6.27328460	-0.00036925	0.00022620	0.00037111	-0.00000115	0.00000000
		-6.27328470	-0.00036920	0.00022617	0.00037106	-0.00000115	0.00000000
		-6.27328481	-0.00036916	0.00022615	0.00037102	-0.00000115	0.00000000
		-6.27328491	-0.00036912	0.00022612	0.00037097	-0.00000115	0.00000000
		-6.27328502	-0.00036907	0.00022609	0.00037093	-0.00000115	0.00000000
		-6.27328512	-0.00036903	0.00022607	0.00037088	-0.00000115	0.00000000
		-6.27328523	-0.00036898	0.00022604	0.00037084	-0.00000115	0.00000000
		-6.27328533	-0.00036894	0.00022601	0.00037080	-0.00000115	0.00000000
		-6.27328544	-0.00036890	0.00022599	0.00037075	-0.00000115	0.00000000
		-6.27328555	-0.00036885	0.00022596	0.00037071	-0.00000115	0.00000000
		-6.27328565	-0.00036881	0.00022593	0.00037066	-0.00000115	0.00000000
		-6.27328576	-0.00036876	0.00022590	0.00037062	-0.00000115	0.00000000
		-6.27328586	-0.00036872	0.00022588	0.00037058	-0.00000115	0.00000000
		-6.27328597	-0.00036868	0.00022585	0.00037053	-0.00000115	0.00000000
		-6.27328607	-0.00036863	0.00022582	0.00037049	-0.00000115	0.00000000
		-6.27328618	-0.00036859	0.00022580	0.00037044	-0.00000115	0.00000000
		-6.27328628	-0.00036855	0.00022577	0.00037040	-0.00000115	0.00000000
		-6.27328639	-0.00036850	0.00022574	0.00037035	-0.00000115	0.00000000
		-6.27328649	-0.00036846	0.00022572	0.00037031	-0.00000115	0.00000000
		-6.27328660	-0.00036841	0.00022569	0.00037027	-0.00000115	0.00000000
		-6.27328670	-0.00036837	0.00022566	0.00037022	-0.00000115	0.00000000
		-6.27328681	-0.00036833	0.00022564	0.00037018	-0.00000115	0.00000000
		-6.27328691	-0.00036828	0.00022561	0.00037013	-0.00000115	0.00000000
		-6.27328702	-0.00036824	0.00022558	0.00037009	-0.00000115	0.00000000
		-6.27328712	-0.00036819	0.00022556	0.00037005	-0.00000115	0.00000000
		-6.27328723	-0.00036815	0.00022553	0.00037000	-0.00000115	0.00000000
		-6.27328733	-0.00036811	0.00022550	0.00036996	-0.00000115	0.00000000
		-6.27328744	-0.00036806	0.00022547	0.00036991	-0.00000115	0.00000000
		-6.27328754	-0.00036802	0.00022545	0.00036987	-0.00000115	0.00000000
		-6.27328764	-0.00036798	0.00022542	0.00036983	-0.00000115	0.00000000
		-6.27328775	-0.00036793	0.00022539	0.00036978	-0.00000115	0.00000000
		-6.27328785	-0.00036789	0.00022537	0.00036974	-0.00000115	0.00000000
		-6.27328796	-0.00036784	0.00022534	0.00036969	-0.00000115	0.00000000
		-6.27328806	-0.00036780	0.00022531	0.00036965	-0.00000115	0.00000000
		-6.27328817	-0.00036776	0.00022529	0.00036961	-0.00000115	0.00000000
		-6.27328827	-0.00036771	0.00022526	0.00036956	-0.00000115	0.00000000
		-6.27328838	-0.00036767	0.00022523	0.00036952	-0.00000115	0.00000000
		-6.27328848	-0.00036763	0.00022521	0.00036947	-0.00000115	0.00000000
		-6.27328859	-0.00036758	0.00022518	0.00036943	-0.00000115	0.00000000
		-6.27328869	-0.00036754	0.00022515	0.00036939	-0.00000115	0.00000000
		-6.27328880	-0.00036749	0.00022513	0.00036934	-0.00000115	0.00000000
		-6.27328890	-0.00036745	0.00022510	0.00036930	-0.00000115	0.00000000
		-6.27328900	-0.00036741	0.00022507	0.00036926	-0.00000115	0.00000000
		-6.27328911	-0.00036736	0.00022505	0.00036921	-0.00000115	0.00000000
		-6.27328921	-0.00036732	0.00022502	0.00036917	-0.00000115	0.00000000
		-6.27328932	-0.00036728	0.00022499	0.00036912	-0.00000115	0.00000000
		-6.27328942	-0.00036723	0.00022497	0.00036908	-0.00000115	0.00000000
		-6.27328953	-0.00036719	0.00022494	0.00036904	-0.00000115	0.00000000
		-6.27328963	-0.00036715	0.00022491	0.00036899	-0.00000115	0.00000000
		-6.27328973	-0.00036710	0.00022489	0.00036895	-0.00000115	0.00000000
		-6.27328984	-0.00036706	0.00022486	0.00036891	-0.00000115	0.00000000
		-6.27328994	-0.00036702	0.00022483	0.00036886	-0.00000115	0.00000000
		-6.27329005	-0.00036697	0.00022481	0.00036882	-0.00000115	0.00000000
		-6.27329015	-0.00036693	0.00022478	0.00036877	-0.00000115	0.00000000
		-6.27329025	-0.00036689	0.00022475	0.00036873	-0.00000115	0.00000000
		-6.27329036	-0.00036684	0.00022473	0.00036869	-0.00000115	0.00000000
		-6.27329046	-0.00036680	0.00022470	0.00036864	-0.00000115	0.00000000
		-6.27329057	-0.00036675	0.00022467	0.00036860	-0.00000115	0.00000000
		-6.27329067	-0.00036671	0.00022465	0.00036856	-0.00000115	0.00000000
		-6.27329078	-0.00036667	0.00022462	0.00036851	-0.00000115	0.00000000
		-6.27329088	-0.00036662	0.00022459	0.00036847	-0.00000115	0.00000000
		-6.27329098	-0.00036658	0.00022457	0.00036842	-0.00000115	0.00000000
		-6.27329109	-0.00036654	0.00022454	0.00036838	-0.00000115	0.00000000
		-6.27329119	-0.00036649	0.00022451	0.00036834	-0.00000115	0.00000000
		-6.27329129	-0.00036645	0.00022449	0.00036829	-0.00000115	0.00000000
		-6.27329140	-0.00036641	0.00022446	0.00036825	-0.00000115	0.00000000
		-6.27329150	-0.00036636	0.00022443	0.00036821	-0.00000115	0.00000000
		-6.27329161	-0.00036632	0.00022441	0.00036816	-0.00000115	0.00000000
		-6.27329171	-0.00036628	0.00022438	0.00036812	-0.00000115	0.00000000
		-6.27329181	-0.00036623	0.00022435	0.00036808	-0.00000115	0.00000000
		-6.27329192	-0.00036619	0.00022433	0.00036803	-0.00000115	0.00000000
		-6.27329202	-0.00036615	0.00022430	0.00036799	-0.00000115	0.00000000
		-6.27329212	-0.00036610	0.00022427	0.00036795	-0.00000115	0.00000000
		-6.27329223	-0.00036606	0.00022425	0.00036790	-0.00000115	0.00000000
		-6.27329233	-0.00036602	0.00022422	0.00036786	-0.00000115	0.00000000
		-6.27329244	-0.00036597	0.00022419	0.00036781	-0.00000115	0.00000000
		-6.27329254	-0.00036593	0.00022417	0.00036777	-0.00000115	0.00000000
		-6.27329264	-0.00036589	0.00022414	0.00036773	-0.00000115	0.00000000
		-6.27329275	-0.00036584	0.00022412	0.00036768	-0.00000115	0.00000000
		-6.27329285	-0.00036580	0.00022409	0.00036764	-0.00000115	0.00000000
		-6.27329295	-0.00036576	0.00022406	0.00036760	-0.00000115	0.00000000
		-6.27329306	-0.00036571	0.00022404	0.00036755	-0.00000115	0.00000000
		-6.27329316	-0.00036567	0.00022401	0.00036751	-0.00000115	0.00000000
		-6.27329326	-0.00036563	0.00022398	0.00036747	-0.00000115	0.00000000
		-6.27329337	-0.00036559	0.00022396	0.00036742	-0.00000115	0.00000000
		-6.27329347	-0.00036554	0.00022393	0.00036738	-0.00000115	0.00000000
		-6.27329357	-0.00036550	0.00022390	0.00036734	-0.00000115	0.00000000
		-6.27329368	-0.00036546	0.00022388	0.00036729	-0.00000115	0.00000000
		-6.27329378	-0.00036541	0.00022385	0.00036725	-0.00000115	0.00000000
		-6.27329388	-0.00036537	0.00022382	0.00036721	-0.00000115	0.00000000
		-6.27329399	-0.00036533	0.00022380	0.00036716	-0.00000115	0.00000000
		-6.27329409	-0.00036528	0.00022377	0.00036712	-0.00000115	0.00000000
		-6.27329419	-0.00036524	0.00022374	0.00036708	-0.00000115	0.00000000
		-6.27329430	-0.00036520	0.00022372	0.00036703	-0.00000115	0.00000000
		-6.27329440	-0.00036515	0.00022369	0.00036699	-0.00000115	0.00000000
		-6.27329450	-0.00036511	0.00022367	0.00036695	-0.00000115	0.00000000
		-6.27329461	-0.00036507	0.00022364	0.00036690	-0.00000115	0.00000000
		-6.27329471	-0.00036502	0.00022361	0.00036686	-0.00000115	0.00000000
		-6.27329481	-0.00036498	0.00022359	0.00036682	-0.00000115	0.00000000
		-6.27329491	-0.00036494	0.00022356	0.00036677	-0.00000115	0.00000000
		-6.27329502	-0.00036490	0.00022353	0.00036673	-0.00000115	0.00000000
		-6.27329512	-0.00036485	0.00022351	0.00036669	-0.00000115	0.00000000
		-6.27329522	-0.00036481	0.00022348	0.00036664	-0.00000115	0.00000000
		-6.27329533	-0.00036477	0.00022345	0.00036660	-0.00000115	0.00000000
		-6.27329543	-0.00036472	0.00022343	0.00036656	-0.00000115	0.00000000
		-6.27329553	-0.00036468	0.00022340	0.00036651	-0.00000115	0.00000000
		-6.27329564	-0.00036464	0.00022338	0.00036647	-0.00000115	0.00000000
		-6.27329574	-0.00036460	0.00022335	0.00036643	-0.00000115	0.00000000
		-6.27329584	-0.00036455	0.00022332	0.00036639	-0.00000115	0.00000000
		-6.27329594	-0.00036451	0.00022330	0.00036634	-0.00000115	0.00000000
		-6.27329605	-0.00036447	0.00022327	0.00036630	-0.00000115	0.00000000
		-6.27329615	-0.00036442	0.00022324	0.00036626	-0.00000115	0.00000000
		-6.27329625	-0.00036438	0.00022322	0.00036621	-0.00000115	0.00000000
		-6.27329635	-0.00036434	0.00022319	0.00036617	-0.00000115	0.00000000
		-6.27329646	-0.00036429	0.00022317	0.00036613	-0.00000115	0.00000000
		-6.27329656	-0.00036425	0.00022314	0.00036608	-0.00000115	0.00000000
		-6.27329666	-0.00036421	0.00022311	0.00036604	-0.00000115	0.00000000
		-6.27329677	-0.00036417	0.00022309	0.00036600	-0.00000115	0.00000000
		-6.27329687	-0.00036412	0.00022306	0.00036595	-0.00000115	0.00000000
		-6.27329697	-0.00036408	0.00022303	0.00036591	-0.00000115	0.00000000
		-6.27329707	-0.00036404	0.00022301	0.00036587	-0.00000115	0.00000000
		-6.27329718	-0.00036399	0.00022298	0.00036583	-0.00000115	0.00000000
		-6.27329728	-0.00036395	0.00022296	0.00036578	-0.00000115	0.00000000
		-6.27329738	-0.00036391	0.00022293	0.00036574	-0.00000115	0.00000000
		-6.27329748	-0.00036387	0.00022290	0.00036570	-0.00000115	0.00000000
		-6.27329758	-0.00036382	0.00022288	0.00036565	-0.00000115	0.00000000
		-6.27329769	-0.00036378	0.00022285	0.00036561	-0.00000115	0.00000000
		-6.27329779	-0.00036374	0.00022282	0.00036557	-0.00000115	0.00000000
		-6.27329789	-0.00036370	0.00022280	0.00036552	-0.00000115	0.00000000
		-6.27329799	-0.00036365	0.00022277	0.00036548	-0.00000115	0.00000000
		-6.27329810	-0.00036361	0.00022275	0.00036544	-0.00000115	0.00000000
		-6.27329820	-0.00036357	0.00022272	0.00036540	-0.00000115	0.00000000
		-6.27329830	-0.00036352	0.00022269	0.00036535	-0.00000115	0.00000000
		-6.27329840	-0.00036348	0.00022267	0.00036531	-0.00000115	0.00000000
		-6.27329851	-0.00036344	0.00022264	0.00036527	-0.00000115	0.00000000
		-6.27329861	-0.00036340	0.00022261	0.00036522	-0.00000115	0.00000000
		-6.27329871	-0.00036335	0.00022259	0.00036518	-0.00000115	0.00000000
		-6.27329881	-0.00036331	0.00022256	0.00036514	-0.00000115	0.00000000
		-6.27329891	-0.00036327	0.00022254	0.00036510	-0.00000115	0.00000000
		-6.27329902	-0.00036323	0.00022251	0.00036505	-0.00000115	0.00000000
		-6.27329912	-0.00036318	0.00022248	0.00036501	-0.00000115	0.00000000
		-6.27329922	-0.00036314	0.00022246	0.00036497	-0.00000115	0.00000000
		-6.27329932	-0.00036310	0.00022243	0.00036492	-0.00000115	0.00000000
		-6.27329942	-0.00036306	0.00022241	0.00036488	-0.00000115	0.00000000
		-6.27329953	-0.00036301	0.00022238	0.00036484	-0.00000115	0.00000000
		-6.27329963	-0.00036297	0.00022235	0.00036480	-0.00000115	0.00000000
		-6.27329973	-0.00036293	0.00022233	0.00036475	-0.00000115	0.00000000
		-6.27329983	-0.00036289	0.00022230	0.00036471	-0.00000115	0.00000000
		-6.27329993	-0.00036284	0.00022228	0.00036467	-0.00000115	0.00000000
		-6.27330004	-0.00036280	0.00022225	0.00036462	-0.00000115	0.00000000
		-6.27330014	-0.00036276	0.00022222	0.00036458	-0.00000115	0.00000000
		-6.27330024	-0.00036272	0.00022220	0.00036454	-0.00000115	0.00000000
		-6.27330034	-0.00036267	0.00022217	0.00036450	-0.00000115	0.00000000
		-6.27330044	-0.00036263	0.00022215	0.00036445	-0.00000115	0.00000000
		-6.27330054	-0.00036259	0.00022212	0.00036441	-0.00000115	0.00000000
		-6.27330065	-0.00036255	0.00022209	0.00036437	-0.00000115	0.00000000
		-6.27330075	-0.00036250	0.00022207	0.00036433	-0.00000115	0.00000000
		-6.27330085	-0.00036246	0.00022204	0.00036428	-0.00000115	0.00000000
		-6.27330095	-0.00036242	0.00022202	0.00036424	-0.00000115	0.00000000
		-6.27330105	-0.00036238	0.00022199	0.00036420	-0.00000115	0.00000000
		-6.27330115	-0.00036233	0.00022196	0.00036416	-0.00000115	0.00000000
		-6.27330126	-0.00036229	0.00022194	0.00036411	-0.00000115	0.00000000
		-6.27330136	-0.00036225	0.00022191	0.00036407	-0.00000115	0.00000000
		-6.27330146	-0.00036221	0.00022189	0.00036403	-0.00000115	0.00000000
		-6.27330156	-0.00036216	0.00022186	0.00036398	-0.00000115	0.00000000
		-6.27330166	-0.00036212	0.00022183	0.00036394	-0.00000115	0.00000000
		-6.27330176	-0.00036208	0.00022181	0.00036390	-0.00000115	0.00000000
		-6.27330186	-0.00036204	0.00022178	0.00036386	-0.00000115	0.00000000
		-6.27330197	-0.00036199	0.00022176	0.00036381	-0.00000115	0.00000000
		-6.27330207	-0.00036195	0.00022173	0.00036377	-0.00000115	0.00000000
		-6.27330217	-0.00036191	0.00022170	0.00036373	-0.00000115	0.00000000
		-6.27330227	-0.00036187	0.00022168	0.00036369	-0.00000115	0.00000000
		-6.27330237	-0.00036182	0.00022165	0.00036364	-0.00000115	0.00000000
		-6.27330247	-0.00036178	0.00022163	0.00036360	-0.00000115	0.00000000
		-6.27330257	-0.00036174	0.00022160	0.00036356	-0.00000115	0.00000000
		-6.27330267	-0.00036170	0.00022157	0.00036352	-0.00000115	0.00000000
		-6.27330278	-0.00036166	0.00022155	0.00036347	-0.00000115	0.00000000
		-6.27330288	-0.00036161	0.00022152	0.00036343	-0.00000115	0.00000000
		-6.27330298	-0.00036157	0.00022150	0.00036339	-0.00000115	0.00000000
		-6.27330308	-0.00036153	0.00022147	0.00036335	-0.00000115	0.00000000
		-6.27330318	-0.00036149	0.00022144	0.00036330	-0.00000115	0.00000000
		-6.27330328	-0.00036144	0.00022142	0.00036326	-0.00000115	0.00000000
		-6.27330338	-0.00036140	0.00022139	0.00036322	-0.00000115	0.00000000
		-6.27330348	-0.00036136	0.00022137	0.00036318	-0.00000115	0.00000000
		-6.27330358	-0.00036132	0.00022134	0.00036313	-0.00000115	0.00000000
		-6.27330368	-0.00036128	0.00022132	0.00036309	-0.00000115	0.00000000
		-6.27330379	-0.00036123	0.00022129	0.00036305	-0.00000115	0.00000000
		-6.27330389	-0.00036119	0.00022126	0.00036301	-0.00000115	0.00000000
		-6.27330399	-0.00036115	0.00022124	0.00036297	-0.00000115	0.00000000
		-6.27330409	-0.00036111	0.00022121	0.00036292	-0.00000115	0.00000000
		-6.27330419	-0.00036107	0.00022119	0.00036288	-0.00000115	0.00000000
		-6.27330429	-0.00036102	0.00022116	0.00036284	-0.00000115	0.00000000
		-6.27330439	-0.00036098	0.00022113	0.00036280	-0.00000115	0.00000000
		-6.27330449	-0.00036094	0.00022111	0.00036275	-0.00000115	0.00000000
		-6.27330459	-0.00036090	0.00022108	0.00036271	-0.00000115	0.00000000
		-6.27330469	-0.00036085	0.00022106	0.00036267	-0.00000115	0.00000000
		-6.27330479	-0.00036081	0.00022103	0.00036263	-0.00000115	0.00000000
		-6.27330489	-0.00036077	0.00022101	0.00036258	-0.00000115	0.00000000
		-6.27330499	-0.00036073	0.00022098	0.00036254	-0.00000115	0.00000000
		-6.27330510	-0.00036069	0.00022095	0.00036250	-0.00000115	0.00000000
		-6.27330520	-0.00036064	0.00022093	0.00036246	-0.00000115	0.00000000
		-6.27330530	-0.00036060	0.00022090	0.00036242	-0.00000115	0.00000000
		-6.27330540	-0.00036056	0.00022088	0.00036237	-0.00000115	0.00000000
		-6.27330550	-0.00036052	0.00022085	0.00036233	-0.00000115	0.00000000
		-6.27330560	-0.00036048	0.00022083	0.00036229	-0.00000115	0.00000000
		-6.27330570	-0.00036043	0.00022080	0.00036225	-0.00000115	0.00000000
		-6.27330580	-0.00036039	0.00022077	0.00036220	-0.00000115	0.00000000
		-6.27330590	-0.00036035	0.00022075	0.00036216	-0.00000115	0.00000000
		-6.27330600	-0.00036031	0.00022072	0.00036212	-0.00000115	0.00000000
		-6.27330610	-0.00036027	0.00022070	0.00036208	-0.00000115	0.00000000
		-6.27330620	-0.00036023	0.00022067	0.00036204	-0.00000115	0.00000000
		-6.27330630	-0.00036018	0.00022065	0.00036199	-0.00000115	0.00000000
		-6.27330640	-0.00036014	0.00022062	0.00036195	-0.00000115	0.00000000
		-6.27330650	-0.00036010	0.00022059	0.00036191	-0.00000115	0.00000000
		-6.27330660	-0.00036006	0.00022057	0.00036187	-0.00000115	0.00000000
		-6.27330670	-0.00036002	0.00022054	0.00036183	-0.00000115	0.00000000
		-6.27330680	-0.00035997	0.00022052	0.00036178	-0.00000115	0.00000000
		-6.27330690	-0.00035993	0.00022049	0.00036174	-0.00000115	0.00000000
		-6.27330700	-0.00035989	0.00022047	0.00036170	-0.00000115	0.00000000
		-6.27330710	-0.00035985	0.00022044	0.00036166	-0.00000115	0.00000000
		-6.27330720	-0.00035981	0.00022041	0.00036162	-0.00000115	0.00000000
		-6.27330730	-0.00035976	0.00022039	0.00036157	-0.00000115	0.00000000
		-6.27330740	-0.00035972	0.00022036	0.00036153	-0.00000115	0.00000000
		-6.27330750	-0.00035968	0.00022034	0.00036149	-0.00000115	0.00000000
		-6.27330760	-0.00035964	0.00022031	0.00036145	-0.00000115	0.00000000
		-6.27330770	-0.00035960	0.00022029	0.00036141	-0.00000115	0.00000000
		-6.27330780	-0.00035956	0.00022026	0.00036136	-0.00000115	0.00000000
		-6.27330790	-0.00035951	0.00022024	0.00036132	-0.00000115	0.00000000
		-6.27330800	-0.00035947	0.00022021	0.00036128	-0.00000115	0.00000000
		-6.27330810	-0.00035943	0.00022018	0.00036124	-0.00000115	0.00000000
		-6.27330820	-0.00035939	0.00022016	0.00036120	-0.00000115	0.00000000
		-6.27330830	-0.00035935	0.00022013	0.00036115	-0.00000115	0.00000000
		-6.27330840	-0.00035931	0.00022011	0.00036111	-0.00000115	0.00000000
		-6.27330850	-0.00035926	0.00022008	0.00036107	-0.00000115	0.00000000
		-6.27330860	-0.00035922	0.00022006	0.00036103	-0.00000115	0.00000000
		-6.27330870	-0.00035918	0.00022003	0.00036099	-0.00000115	0.00000000
		-6.27330880	-0.00035914	0.00022001	0.00036094	-0.00000115	0.00000000
		-6.27330890	-0.00035910	0.00021998	0.00036090	-0.00000115	0.00000000
		-6.27330900	-0.00035906	0.00021995	0.00036086	-0.00000115	0.00000000
		-6.27330910	-0.00035901	0.00021993	0.00036082	-0.00000115	0.00000000
		-6.27330920	-0.00035897	0.00021990	0.00036078	-0.00000115	0.00000000
		-6.27330930	-0.00035893	0.00021988	0.00036073	-0.00000115	0.00000000
		-6.27330940	-0.00035889	0.00021985	0.00036069	-0.00000115	0.00000000
		-6.27330950	-0.00035885	0.00021983	0.00036065	-0.00000115	0.00000000
		-6.27330960	-0.00035881	0.00021980	0.00036061	-0.00000115	0.00000000
		-6.27330970	-0.00035876	0.00021978	0.00036057	-0.00000115	0.00000000
		-6.27330980	-0.00035872	0.00021975	0.00036053	-0.00000115	0.00000000
		-6.27330990	-0.00035868	0.00021972	0.00036048	-0.00000115	0.00000000
		-6.27331000	-0.00035864	0.00021970	0.00036044	-0.00000115	0.00000000
		-6.27331010	-0.00035860	0.00021967	0.00036040	-0.00000115	0.00000000
		-6.27331020	-0.00035856	0.00021965	0.00036036	-0.00000115	0.00000000
		-6.27331030	-0.00035851	0.00021962	0.00036032	-0.00000115	0.00000000
		-6.27331040	-0.00035847	0.00021960	0.00036028	-0.00000115	0.00000000
		-6.27331050	-0.00035843	0.00021957	0.00036023	-0.00000115	0.00000000
		-6.27331059	-0.00035839	0.00021955	0.00036019	-0.00000115	0.00000000
		-6.27331069	-0.00035835	0.00021952	0.00036015	-0.00000115	0.00000000
		-6.27331079	-0.00035831	0.00021950	0.00036011	-0.00000115	0.00000000
		-6.27331089	-0.00035827	0.00021947	0.00036007	-0.00000115	0.00000000
		-6.27331099	-0.00035822	0.00021944	0.00036002	-0.00000115	0.00000000
		-6.27331109	-0.00035818	0.00021942	0.00035998	-0.00000115	0.00000000
		-6.27331119	-0.00035814	0.00021939	0.00035994	-0.00000115	0.00000000
		-6.27331129	-0.00035810	0.00021937	0.00035990	-0.00000115	0.00000000
		-6.27331139	-0.00035806	0.00021934	0.00035986	-0.00000115	0.00000000
		-6.27331149	-0.00035802	0.00021932	0.00035982	-0.00000115	0.00000000
		-6.27331159	-0.00035798	0.00021929	0.00035978	-0.00000115	0.00000000
		-6.27331169	-0.00035793	0.00021927	0.00035973	-0.00000115	0.00000000
		-6.27331179	-0.00035789	0.00021924	0.00035969	-0.00000115	0.00000000
		-6.27331188	-0.00035785	0.00021922	0.00035965	-0.00000115	0.00000000
		-6.27331198	-0.00035781	0.00021919	0.00035961	-0.00000115	0.00000000
		-6.27331208	-0.00035777	0.00021917	0.00035957	-0.00000115	0.00000000
		-6.27331218	-0.00035773	0.00021914	0.00035953	-0.00000115	0.00000000
		-6.27331228	-0.00035769	0.00021911	0.00035948	-0.00000115	0.00000000
		-6.27331238	-0.00035764	0.00021909	0.00035944	-0.00000115	0.00000000
		-6.27331248	-0.00035760	0.00021906	0.00035940	-0.00000115	0.00000000
		-6.27331258	-0.00035756	0.00021904	0.00035936	-0.00000115	0.00000000
		-6.27331268	-0.00035752	0.00021901	0.00035932	-0.00000115	0.00000000
		-6.27331277	-0.00035748	0.00021899	0.00035928	-0.00000115	0.00000000
		-6.27331287	-0.00035744	0.00021896	0.00035923	-0.00000115	0.00000000
		-6.27331297	-0.00035740	0.00021894	0.00035919	-0.00000115	0.00000000
		-6.27331307	-0.00035736	0.00021891	0.00035915	-0.00000115	0.00000000
		-6.27331317	-0.00035731	0.00021889	0.00035911	-0.00000115	0.00000000
		-6.27331327	-0.00035727	0.00021886	0.00035907	-0.00000115	0.00000000
		-6.27331337	-0.00035723	0.00021884	0.00035903	-0.00000115	0.00000000
		-6.27331347	-0.00035719	0.00021881	0.00035899	-0.00000115	0.00000000
		-6.27331356	-0.00035715	0.00021879	0.00035894	-0.00000115	0.00000000
		-6.27331366	-0.00035711	0.00021876	0.00035890	-0.00000115	0.00000000
		-6.27331376	-0.00035707	0.00021874	0.00035886	-0.00000115	0.00000000
		-6.27331386	-0.00035703	0.00021871	0.00035882	-0.00000115	0.00000000
		-6.27331396	-0.00035698	0.00021869	0.00035878	-0.00000115	0.00000000
		-6.27331406	-0.00035694	0.00021866	0.00035874	-0.00000115	0.00000000
		-6.27331416	-0.00035690	0.00021863	0.00035870	-0.00000115	0.00000000
		-6.27331425	-0.00035686	0.00021861	0.00035866	-0.00000115	0.00000000
		-6.27331435	-0.00035682	0.00021858	0.00035861	-0.00000115	0.00000000
		-6.27331445	-0.00035678	0.00021856	0.00035857	-0.00000115	0.00000000
		-6.27331455	-0.00035674	0.00021853	0.00035853	-0.00000115	0.00000000
		-6.27331465	-0.00035670	0.00021851	0.00035849	-0.00000115	0.00000000
		-6.27331475	-0.00035666	0.00021848	0.00035845	-0.00000115	0.00000000
		-6.27331485	-0.00035661	0.00021846	0.00035841	-0.00000115	0.00000000
		-6.27331494	-0.00035657	0.00021843	0.00035837	-0.00000115	0.00000000
		-6.27331504	-0.00035653	0.00021841	0.00035832	-0.00000115	0.00000000
		-6.27331514	-0.00035649	0.00021838	0.00035828	-0.00000115	0.00000000
		-6.27331524	-0.00035645	0.00021836	0.00035824	-0.00000115	0.00000000
		-6.27331534	-0.00035641	0.00021833	0.00035820	-0.00000115	0.00000000
		-6.27331543	-0.00035637	0.00021831	0.00035816	-0.00000115	0.00000000
		-6.27331553	-0.00035633	0.00021828	0.00035812	-0.00000115	0.00000000
		-6.27331563	-0.00035629	0.00021826	0.00035808	-0.00000115	0.00000000
		-6.27331573	-0.00035625	0.00021823	0.00035804	-0.00000115	0.00000000
		-6.27331583	-0.00035620	0.00021821	0.00035799	-0.00000115	0.00000000
		-6.27331593	-0.00035616	0.00021818	0.00035795	-0.00000115	0.00000000
		-6.27331602	-0.00035612	0.00021816	0.00035791	-0.00000115	0.00000000
		-6.27331612	-0.00035608	0.00021813	0.00035787	-0.00000115	0.00000000
		-6.27331622	-0.00035604	0.00021811	0.00035783	-0.00000115	0.00000000
		-6.27331632	-0.00035600	0.00021808	0.00035779	-0.00000115	0.00000000
		-6.27331642	-0.00035596	0.00021806	0.00035775	-0.00000115	0.00000000
		-6.27331651	-0.00035592	0.00021803	0.00035771	-0.00000115	0.00000000
		-6.27331661	-0.00035588	0.00021801	0.00035767	-0.00000115	0.00000000
		-6.27331671	-0.00035584	0.00021798	0.00035762	-0.00000115	0.00000000
		-6.27331681	-0.00035579	0.00021796	0.00035758	-0.00000115	0.00000000
		-6.27331691	-0.00035575	0.00021793	0.00035754	-0.00000115	0.00000000
		-6.27331700	-0.00035571	0.00021791	0.00035750	-0.00000115	0.00000000
		-6.27331710	-0.00035567	0.00021788	0.00035746	-0.00000115	0.00000000
		-6.27331720	-0.00035563	0.00021786	0.00035742	-0.00000115	0.00000000
		-6.27331730	-0.00035559	0.00021783	0.00035738	-0.00000115	0.00000000
		-6.27331740	-0.00035555	0.00021781	0.00035734	-0.00000115	0.00000000
		-6.27331749	-0.00035551	0.00021778	0.00035730	-0.00000115	0.00000000
		-6.27331759	-0.00035547	0.00021776	0.00035725	-0.00000115	0.00000000
		-6.27331769	-0.00035543	0.00021773	0.00035721	-0.00000115	0.00000000
		-6.27331779	-0.00035539	0.00021771	0.00035717	-0.00000115	0.00000000
		-6.27331788	-0.00035535	0.00021768	0.00035713	-0.00000115	0.00000000
		-6.27331798	-0.00035530	0.00021766	0.00035709	-0.00000115	0.00000000
		-6.27331808	-0.00035526	0.00021763	0.00035705	-0.00000115	0.00000000
		-6.27331818	-0.00035522	0.00021761	0.00035701	-0.00000115	0.00000000
		-6.27331827	-0.00035518	0.00021758	0.00035697	-0.00000115	0.00000000
		-6.27331837	-0.00035514	0.00021756	0.00035693	-0.00000115	0.00000000
		-6.27331847	-0.00035510	0.00021753	0.00035689	-0.00000115	0.00000000
		-6.27331857	-0.00035506	0.00021751	0.00035684	-0.00000115	0.00000000
		-6.27331866	-0.00035502	0.00021748	0.00035680	-0.00000115	0.00000000
		-6.27331876	-0.00035498	0.00021746	0.00035676	-0.00000115	0.00000000
		-6.27331886	-0.00035494	0.00021743	0.00035672	-0.00000115	0.00000000
		-6.27331896	-0.00035490	0.00021741	0.00035668	-0.00000115	0.00000000
		-6.27331905	-0.00035486	0.00021738	0.00035664	-0.00000115	0.00000000
		-6.27331915	-0.00035482	0.00021736	0.00035660	-0.00000115	0.00000000
		-6.27331925	-0.00035478	0.00021733	0.00035656	-0.00000115	0.00000000
		-6.27331935	-0.00035473	0.00021731	0.00035652	-0.00000115	0.00000000
		-6.27331944	-0.00035469	0.00021728	0.00035648	-0.00000115	0.00000000
		-6.27331954	-0.00035465	0.00021726	0.00035644	-0.00000115	0.00000000
		-6.27331964	-0.00035461	0.00021723	0.00035640	-0.00000115	0.00000000
		-6.27331974	-0.00035457	0.00021721	0.00035635	-0.00000115	0.00000000
		-6.27331983	-0.00035453	0.00021718	0.00035631	-0.00000115	0.00000000
		-6.27331993	-0.00035449	0.00021716	0.00035627	-0.00000115	0.00000000
		-6.27332003	-0.00035445	0.00021713	0.00035623	-0.00000115	0.00000000
		-6.27332012	-0.00035441	0.00021711	0.00035619	-0.00000115	0.00000000
		-6.27332022	-0.00035437	0.00021708	0.00035615	-0.00000115	0.00000000
		-6.27332032	-0.00035433	0.00021706	0.00035611	-0.00000115	0.00000000
		-6.27332042	-0.00035429	0.00021703	0.00035607	-0.00000115	0.00000000
		-6.27332051	-0.00035425	0.00021701	0.00035603	-0.00000115	0.00000000
		-6.27332061	-0.00035421	0.00021698	0.00035599	-0.00000115	0.00000000
		-6.27332071	-0.00035417	0.00021696	0.00035595	-0.00000115	0.00000000
		-6.27332080	-0.00035413	0.00021693	0.00035591	-0.00000115	0.00000000
		-6.27332090	-0.00035409	0.00021691	0.00035587	-0.00000115	0.00000000
		-6.27332100	-0.00035405	0.00021688	0.00035582	-0.00000115	0.00000000
		-6.27332109	-0.00035400	0.00021686	0.00035578	-0.00000115	0.00000000
		-6.27332119	-0.00035396	0.00021683	0.00035574	-0.00000115	0.00000000
		-6.27332129	-0.00035392	0.00021681	0.00035570	-0.00000115	0.00000000
		-6.27332139	-0.00035388	0.00021678	0.00035566	-0.00000115	0.00000000
		-6.27332148	-0.00035384	0.00021676	0.00035562	-0.00000115	0.00000000
		-6.27332158	-0.00035380	0.00021673	0.00035558	-0.00000115	0.00000000
		-6.27332168	-0.00035376	0.00021671	0.00035554	-0.00000115	0.00000000
		-6.27332177	-0.00035372	0.00021669	0.00035550	-0.00000115	0.00000000
		-6.27332187	-0.00035368	0.00021666	0.00035546	-0.00000115	0.00000000
		-6.27332197	-0.00035364	0.00021664	0.00035542	-0.00000115	0.00000000
		-6.27332206	-0.00035360	0.00021661	0.00035538	-0.00000115	0.00000000
		-6.27332216	-0.00035356	0.00021659	0.00035534	-0.00000115	0.00000000
		-6.27332226	-0.00035352	0.00021656	0.00035530	-0.00000115	0.00000000
		-6.27332235	-0.00035348	0.00021654	0.00035526	-0.00000115	0.00000000
		-6.27332245	-0.00035344	0.00021651	0.00035522	-0.00000115	0.00000000
		-6.27332255	-0.00035340	0.00021649	0.00035517	-0.00000115	0.00000000
		-6.27332264	-0.00035336	0.00021646	0.00035513	-0.00000115	0.00000000
		-6.27332274	-0.00035332	0.00021644	0.00035509	-0.00000115	0.00000000
		-6.27332284	-0.00035328	0.00021641	0.00035505	-0.00000115	0.00000000
		-6.27332293	-0.00035324	0.00021639	0.00035501	-0.00000115	0.00000000
		-6.27332303	-0.00035320	0.00021636	0.00035497	-0.00000115	0.00000000
		-6.27332313	-0.00035316	0.00021634	0.00035493	-0.00000115	0.00000000
		-6.27332322	-0.00035312	0.00021631	0.00035489	-0.00000115	0.00000000
		-6.27332332	-0.00035308	0.00021629	0.00035485	-0.00000115	0.00000000
		-6.27332341	-0.00035304	0.00021626	0.00035481	-0.00000115	0.00000000
		-6.27332351	-0.00035300	0.00021624	0.00035477	-0.00000115	0.00000000
		-6.27332361	-0.00035296	0.00021622	0.00035473	-0.00000115	0.00000000
		-6.27332370	-0.00035291	0.00021619	0.00035469	-0.00000115	0.00000000
		-6.27332380	-0.00035287	0.00021617	0.00035465	-0.00000115	0.00000000
		-6.27332390	-0.00035283	0.00021614	0.00035461	-0.00000115	0.00000000
		-6.27332399	-0.00035279	0.00021612	0.00035457	-0.00000115	0.00000000
		-6.27332409	-0.00035275	0.00021609	0.00035453	-0.00000115	0.00000000
		-6.27332418	-0.00035271	0.00021607	0.00035449	-0.00000115	0.00000000
		-6.27332428	-0.00035267	0.00021604	0.00035445	-0.00000115	0.00000000
		-6.27332438	-0.00035263	0.00021602	0.00035441	-0.00000115	0.00000000
		-6.27332447	-0.00035259	0.00021599	0.00035437	-0.00000115	0.00000000
		-6.27332457	-0.00035255	0.00021597	0.00035432	-0.00000115	0.00000000
		-6.27332467	-0.00035251	0.00021594	0.00035428	-0.00000115	0.00000000
		-6.27332476	-0.00035247	0.00021592	0.00035424	-0.00000115	0.00000000
		-6.27332486	-0.00035243	0.00021590	0.00035420	-0.00000115	0.00000000
		-6.27332495	-0.00035239	0.00021587	0.00035416	-0.00000115	0.00000000
		-6.27332505	-0.00035235	0.00021585	0.00035412	-0.00000115	0.00000000
		-6.27332515	-0.00035231	0.00021582	0.00035408	-0.00000115	0.00000000
		-6.27332524	-0.00035227	0.00021580	0.00035404	-0.00000115	0.00000000
		-6.27332534	-0.00035223	0.00021577	0.00035400	-0.00000115	0.00000000
		-6.27332543	-0.00035219	0.00021575	0.00035396	-0.00000115	0.00000000
		-6.27332553	-0.00035215	0.00021572	0.00035392	-0.00000115	0.00000000
		-6.27332563	-0.00035211	0.00021570	0.00035388	-0.00000115	0.00000000
		-6.27332572	-0.00035207	0.00021567	0.00035384	-0.00000115	0.00000000
		-6.27332582	-0.00035203	0.00021565	0.00035380	-0.00000115	0.00000000
		-6.27332591	-0.00035199	0.00021563	0.00035376	-0.00000115	0.00000000
		-6.27332601	-0.00035195	0.00021560	0.00035372	-0.00000115	0.00000000
		-6.27332611	-0.00035191	0.00021558	0.00035368	-0.00000115	0.00000000
		-6.27332620	-0.00035187	0.00021555	0.00035364	-0.00000115	0.00000000
		-6.27332630	-0.00035183	0.00021553	0.00035360	-0.00000115	0.00000000
		-6.27332639	-0.00035179	0.00021550	0.00035356	-0.00000115	0.00000000
		-6.27332649	-0.00035175	0.00021548	0.00035352	-0.00000115	0.00000000
		-6.27332658	-0.00035171	0.00021545	0.00035348	-0.00000115	0.00000000
		-6.27332668	-0.00035167	0.00021543	0.00035344	-0.00000115	0.00000000
		-6.27332678	-0.00035163	0.00021540	0.00035340	-0.00000115	0.00000000
		-6.27332687	-0.00035159	0.00021538	0.00035336	-0.00000115	0.00000000
		-6.27332697	-0.00035155	0.00021536	0.00035332	-0.00000115	0.00000000
		-6.27332706	-0.00035151	0.00021533	0.00035328	-0.00000115	0.00000000
		-6.27332716	-0.00035147	0.00021531	0.00035324	-0.00000115	0.00000000
		-6.27332725	-0.00035143	0.00021528	0.00035320	-0.00000115	0.00000000
		-6.27332735	-0.00035139	0.00021526	0.00035316	-0.00000115	0.00000000
		-6.27332744	-0.00035135	0.00021523	0.00035312	-0.00000115	0.00000000
		-6.27332754	-0.00035131	0.00021521	0.00035308	-0.00000115	0.00000000
		-6.27332763	-0.00035127	0.00021518	0.00035304	-0.00000115	0.00000000
		-6.27332773	-0.00035123	0.00021516	0.00035300	-0.00000115	0.00000000
		-6.27332783	-0.00035119	0.00021514	0.00035296	-0.00000115	0.00000000
		-6.27332792	-0.00035115	0.00021511	0.00035292	-0.00000115	0.00000000
		-6.27332802	-0.00035111	0.00021509	0.00035288	-0.00000115	0.00000000
		-6.27332811	-0.00035107	0.00021506	0.00035284	-0.00000115	0.00000000
		-6.27332821	-0.00035103	0.00021504	0.00035280	-0.00000115	0.00000000
		-6.27332830	-0.00035099	0.00021501	0.00035276	-0.00000115	0.00000000
		-6.27332840	-0.00035095	0.00021499	0.00035272	-0.00000115	0.00000000
		-6.27332849	-0.00035091	0.00021496	0.00035268	-0.00000115	0.00000000
		-6.27332859	-0.00035087	0.00021494	0.00035264	-0.00000115	0.00000000
		-6.27332868	-0.00035084	0.00021492	0.00035260	-0.00000115	0.00000000
		-6.27332878	-0.00035080	0.00021489	0.00035256	-0.00000115	0.00000000
		-6.27332887	-0.00035076	0.00021487	0.00035252	-0.00000115	0.00000000
		-6.27332897	-0.00035072	0.00021484	0.00035248	-0.00000115	0.00000000
		-6.27332906	-0.00035068	0.00021482	0.00035244	-0.00000115	0.00000000
		-6.27332916	-0.00035064	0.00021479	0.00035240	-0.00000115	0.00000000
		-6.27332925	-0.00035060	0.00021477	0.00035236	-0.00000115	0.00000000
		-6.27332935	-0.00035056	0.00021475	0.00035232	-0.00000115	0.00000000
		-6.27332944	-0.00035052	0.00021472	0.00035228	-0.00000115	0.00000000
		-6.27332954	-0.00035048	0.00021470	0.00035224	-0.00000115	0.00000000
		-6.27332963	-0.00035044	0.00021467	0.00035220	-0.00000115	0.00000000
		-6.27332973	-0.00035040	0.00021465	0.00035216	-0.00000115	0.00000000
		-6.27332982	-0.00035036	0.00021462	0.00035212	-0.00000115	0.00000000
		-6.27332992	-0.00035032	0.00021460	0.00035208	-0.00000115	0.00000000
		-6.27333001	-0.00035028	0.00021458	0.00035204	-0.00000115	0.00000000
		-6.27333011	-0.00035024	0.00021455	0.00035200	-0.00000115	0.00000000
		-6.27333020	-0.00035020	0.00021453	0.00035196	-0.00000115	0.00000000
		-6.27333030	-0.00035016	0.00021450	0.00035192	-0.00000115	0.00000000
		-6.27333039	-0.00035012	0.00021448	0.00035188	-0.00000115	0.00000000
		-6.27333049	-0.00035008	0.00021445	0.00035184	-0.00000115	0.00000000
		-6.27333058	-0.00035004	0.00021443	0.00035180	-0.00000115	0.00000000
		-6.27333068	-0.00035000	0.00021441	0.00035176	-0.00000115	0.00000000
		-6.27333077	-0.00034996	0.00021438	0.00035172	-0.00000115	0.00000000
		-6.27333087	-0.00034992	0.00021436	0.00035168	-0.00000115	0.00000000
		-6.27333096	-0.00034988	0.00021433	0.00035164	-0.00000115	0.00000000
		-6.27333106	-0.00034984	0.00021431	0.00035160	-0.00000115	0.00000000
		-6.27333115	-0.00034980	0.00021428	0.00035156	-0.00000115	0.00000000
		-6.27333125	-0.00034976	0.00021426	0.00035152	-0.00000115	0.00000000
		-6.27333134	-0.00034973	0.00021424	0.00035148	-0.00000115	0.00000000
		-6.27333144	-0.00034969	0.00021421	0.00035144	-0.00000115	0.00000000
		-6.27333153	-0.00034965	0.00021419	0.00035140	-0.00000115	0.00000000
		-6.27333162	-0.00034961	0.00021416	0.00035136	-0.00000115	0.00000000
		-6.27333172	-0.00034957	0.00021414	0.00035132	-0.00000115	0.00000000
		-6.27333181	-0.00034953	0.00021412	0.00035128	-0.00000115	0.00000000
		-6.27333191	-0.00034949	0.00021409	0.00035124	-0.00000115	0.00000000
		-6.27333200	-0.00034945	0.00021407	0.00035120	-0.00000115	0.00000000
		-6.27333210	-0.00034941	0.00021404	0.00035117	-0.00000115	0.00000000
		-6.27333219	-0.00034937	0.00021402	0.00035113	-0.00000115	0.00000000
		-6.27333229	-0.00034933	0.00021399	0.00035109	-0.00000115	0.00000000
		-6.27333238	-0.00034929	0.00021397	0.00035105	-0.00000115	0.00000000
		-6.27333247	-0.00034925	0.00021395	0.00035101	-0.00000115	0.00000000
		-6.27333257	-0.00034921	0.00021392	0.00035097	-0.00000115	0.00000000
		-6.27333266	-0.00034917	0.00021390	0.00035093	-0.00000115	0.00000000
		-6.27333276	-0.00034913	0.00021387	0.00035089	-0.00000115	0.00000000
		-6.27333285	-0.00034909	0.00021385	0.00035085	-0.00000115	0.00000000
		-6.27333295	-0.00034905	0.00021383	0.00035081	-0.00000115	0.00000000
		-6.27333304	-0.00034902	0.00021380	0.00035077	-0.00000115	0.00000000
		-6.27333313	-0.00034898	0.00021378	0.00035073	-0.00000115	0.00000000
		-6.27333323	-0.00034894	0.00021375	0.00035069	-0.00000115	0.00000000
		-6.27333332	-0.00034890	0.00021373	0.00035065	-0.00000115	0.00000000
		-6.27333342	-0.00034886	0.00021370	0.00035061	-0.00000115	0.00000000
		-6.27333351	-0.00034882	0.00021368	0.00035057	-0.00000115	0.00000000
		-6.27333360	-0.00034878	0.00021366	0.00035053	-0.00000115	0.00000000
		-6.27333370	-0.00034874	0.00021363	0.00035049	-0.00000115	0.00000000
		-6.27333379	-0.00034870	0.00021361	0.00035045	-0.00000115	0.00000000
		-6.27333389	-0.00034866	0.00021358	0.00035041	-0.00000115	0.00000000
		-6.27333398	-0.00034862	0.00021356	0.00035037	-0.00000115	0.00000000
		-6.27333408	-0.00034858	0.00021354	0.00035033	-0.00000115	0.00000000
		-6.27333417	-0.00034854	0.00021351	0.00035030	-0.00000115	0.00000000
		-6.27333426	-0.00034850	0.00021349	0.00035026	-0.00000115	0.00000000
		-6.27333436	-0.00034847	0.00021346	0.00035022	-0.00000115	0.00000000
		-6.27333445	-0.00034843	0.00021344	0.00035018	-0.00000115	0.00000000
		-6.27333454	-0.00034839	0.00021342	0.00035014	-0.00000115	0.00000000
		-6.27333464	-0.00034835	0.00021339	0.00035010	-0.00000115	0.00000000
		-6.27333473	-0.00034831	0.00021337	0.00035006	-0.00000115	0.00000000
		-6.27333483	-0.00034827	0.00021334	0.00035002	-0.00000115	0.00000000
		-6.27333492	-0.00034823	0.00021332	0.00034998	-0.00000115	0.00000000
		-6.27333501	-0.00034819	0.00021330	0.00034994	-0.00000115	0.00000000
		-6.27333511	-0.00034815	0.00021327	0.00034990	-0.00000115	0.00000000
		-6.27333520	-0.00034811	0.00021325	0.00034986	-0.00000115	0.00000000
		-6.27333530	-0.00034807	0.00021322	0.00034982	-0.00000115	0.00000000
		-6.27333539	-0.00034803	0.00021320	0.00034978	-0.00000115	0.00000000
		-6.27333548	-0.00034800	0.00021318	0.00034974	-0.00000115	0.00000000
		-6.27333558	-0.00034796	0.00021315	0.00034970	-0.00000115	0.00000000
		-6.27333567	-0.00034792	0.00021313	0.00034967	-0.00000115	0.00000000
		-6.27333576	-0.00034788	0.00021310	0.00034963	-0.00000115	0.00000000
		-6.27333586	-0.00034784	0.00021308	0.00034959	-0.00000115	0.00000000
		-6.27333595	-0.00034780	0.00021306	0.00034955	-0.00000115	0.00000000
		-6.27333604	-0.00034776	0.00021303	0.00034951	-0.00000115	0.00000000
		-6.27333614	-0.00034772	0.00021301	0.00034947	-0.00000115	0.00000000
		-6.27333623	-0.00034768	0.00021298	0.00034943	-0.00000115	0.00000000
		-6.27333632	-0.00034764	0.00021296	0.00034939	-0.00000115	0.00000000
		-6.27333642	-0.00034760	0.00021294	0.00034935	-0.00000115	0.00000000
		-6.27333651	-0.00034757	0.00021291	0.00034931	-0.00000115	0.00000000
		-6.27333661	-0.00034753	0.00021289	0.00034927	-0.00000115	0.00000000
		-6.27333670	-0.00034749	0.00021286	0.00034923	-0.00000115	0.00000000
		-6.27333679	-0.00034745	0.00021284	0.00034919	-0.00000115	0.00000000
		-6.27333689	-0.00034741	0.00021282	0.00034915	-0.00000115	0.00000000
		-6.27333698	-0.00034737	0.00021279	0.00034912	-0.00000115	0.00000000
		-6.27333707	-0.00034733	0.00021277	0.00034908	-0.00000115	0.00000000
		-6.27333717	-0.00034729	0.00021275	0.00034904	-0.00000115	0.00000000
		-6.27333726	-0.00034725	0.00021272	0.00034900	-0.00000115	0.00000000
		-6.27333735	-0.00034721	0.00021270	0.00034896	-0.00000115	0.00000000
		-6.27333745	-0.00034718	0.00021267	0.00034892	-0.00000115	0.00000000
		-6.27333754	-0.00034714	0.00021265	0.00034888	-0.00000115	0.00000000
		-6.27333763	-0.00034710	0.00021263	0.00034884	-0.00000115	0.00000000
		-6.27333772	-0.00034706	0.00021260	0.00034880	-0.00000115	0.00000000
		-6.27333782	-0.00034702	0.00021258	0.00034876	-0.00000115	0.00000000
		-6.27333791	-0.00034698	0.00021255	0.00034872	-0.00000115	0.00000000
		-6.27333800	-0.00034694	0.00021253	0.00034869	-0.00000115	0.00000000
		-6.27333810	-0.00034690	0.00021251	0.00034865	-0.00000115	0.00000000
		-6.27333819	-0.00034686	0.00021248	0.00034861	-0.00000115	0.00000000
		-6.27333828	-0.00034683	0.00021246	0.00034857	-0.00000115	0.00000000
		-6.27333838	-0.00034679	0.00021244	0.00034853	-0.00000115	0.00000000
		-6.27333847	-0.00034675	0.00021241	0.00034849	-0.00000115	0.00000000
		-6.27333856	-0.00034671	0.00021239	0.00034845	-0.00000115	0.00000000
		-6.27333866	-0.00034667	0.00021236	0.00034841	-0.00000115	0.00000000
		-6.27333875	-0.00034663	0.00021234	0.00034837	-0.00000115	0.00000000
		-6.27333884	-0.00034659	0.00021232	0.00034833	-0.00000115	0.00000000
		-6.27333893	-0.00034655	0.00021229	0.00034829	-0.00000115	0.00000000
		-6.27333903	-0.00034652	0.00021227	0.00034826	-0.00000115	0.00000000
		-6.27333912	-0.00034648	0.00021225	0.00034822	-0.00000115	0.00000000
		-6.27333921	-0.00034644	0.00021222	0.00034818	-0.00000115	0.00000000
		-6.27333931	-0.00034640	0.00021220	0.00034814	-0.00000115	0.00000000
		-6.27333940	-0.00034636	0.00021217	0.00034810	-0.00000115	0.00000000
		-6.27333949	-0.00034632	0.00021215	0.00034806	-0.00000115	0.00000000
		-6.27333958	-0.00034628	0.00021213	0.00034802	-0.00000115	0.00000000
		-6.27333968	-0.00034624	0.00021210	0.00034798	-0.00000115	0.00000000
		-6.27333977	-0.00034620	0.00021208	0.00034794	-0.00000115	0.00000000
		-6.27333986	-0.00034617	0.00021206	0.00034791	-0.00000115	0.00000000
		-6.27333995	-0.00034613	0.00021203	0.00034787	-0.00000115	0.00000000
		-6.27334005	-0.00034609	0.00021201	0.00034783	-0.00000115	0.00000000
		-6.27334014	-0.00034605	0.00021198	0.00034779	-0.00000115	0.00000000
		-6.27334023	-0.00034601	0.00021196	0.00034775	-0.00000115	0.00000000
		-6.27334033	-0.00034597	0.00021194	0.00034771	-0.00000115	0.00000000
		-6.27334042	-0.00034593	0.00021191	0.00034767	-0.00000115	0.00000000
		-6.27334051	-0.00034590	0.00021189	0.00034763	-0.00000115	0.00000000
		-6.27334060	-0.00034586	0.00021187	0.00034759	-0.00000115	0.00000000
		-6.27334070	-0.00034582	0.00021184	0.00034756	-0.00000115	0.00000000
		-6.27334079	-0.00034578	0.00021182	0.00034752	-0.00000115	0.00000000
		-6.27334088	-0.00034574	0.00021179	0.00034748	-0.00000115	0.00000000
		-6.27334097	-0.00034570	0.00021177	0.00034744	-0.00000115	0.00000000
		-6.27334107	-0.00034566	0.00021175	0.00034740	-0.00000115	0.00000000
		-6.27334116	-0.00034563	0.00021172	0.00034736	-0.00000115	0.00000000
		-6.27334125	-0.00034559	0.00021170	0.00034732	-0.00000115	0.00000000
		-6.27334134	-0.00034555	0.00021168	0.00034728	-0.00000115	0.00000000
		-6.27334143	-0.00034551	0.00021165	0.00034725	-0.00000115	0.00000000
		-6.27334153	-0.00034547	0.00021163	0.00034721	-0.00000115	0.00000000
		-6.27334162	-0.00034543	0.00021161	0.00034717	-0.00000115	0.00000000
		-6.27334171	-0.00034539	0.00021158	0.00034713	-0.00000115	0.00000000
		-6.27334180	-0.00034536	0.00021156	0.00034709	-0.00000115	0.00000000
		-6.27334190	-0.00034532	0.00021153	0.00034705	-0.00000115	0.00000000
		-6.27334199	-0.00034528	0.00021151	0.00034701	-0.00000115	0.00000000
		-6.27334208	-0.00034524	0.00021149	0.00034697	-0.00000115	0.00000000
		-6.27334217	-0.00034520	0.00021146	0.00034694	-0.00000115	0.00000000
		-6.27334226	-0.00034516	0.00021144	0.00034690	-0.00000115	0.00000000
		-6.27334236	-0.00034512	0.00021142	0.00034686	-0.00000115	0.00000000
		-6.27334245	-0.00034509	0.00021139	0.00034682	-0.00000115	0.00000000
		-6.27334254	-0.00034505	0.00021137	0.00034678	-0.00000115	0.00000000
		-6.27334263	-0.00034501	0.00021135	0.00034674	-0.00000115	0.00000000
		-6.27334273	-0.00034497	0.00021132	0.00034670	-0.00000115	0.00000000
		-6.27334282	-0.00034493	0.00021130	0.00034666	-0.00000115	0.00000000
		-6.27334291	-0.00034489	0.00021128	0.00034663	-0.00000115	0.00000000
		-6.27334300	-0.00034486	0.00021125	0.00034659	-0.00000115	0.00000000
		-6.27334309	-0.00034482	0.00021123	0.00034655	-0.00000115	0.00000000
		-6.27334319	-0.00034478	0.00021120	0.00034651	-0.00000115	0.00000000
		-6.27334328	-0.00034474	0.00021118	0.00034647	-0.00000115	0.00000000
		-6.27334337	-0.00034470	0.00021116	0.00034643	-0.00000115	0.00000000
		-6.27334346	-0.00034466	0.00021113	0.00034639	-0.00000115	0.00000000
		-6.27334355	-0.00034462	0.00021111	0.00034636	-0.00000115	0.00000000
		-6.27334364	-0.00034459	0.00021109	0.00034632	-0.00000115	0.00000000
		-6.27334374	-0.00034455	0.00021106	0.00034628	-0.00000115	0.00000000
		-6.27334383	-0.00034451	0.00021104	0.00034624	-0.00000115	0.00000000
		-6.27334392	-0.00034447	0.00021102	0.00034620	-0.00000115	0.00000000
		-6.27334401	-0.00034443	0.00021099	0.00034616	-0.00000115	0.00000000
		-6.27334410	-0.00034439	0.00021097	0.00034612	-0.00000115	0.00000000
		-6.27334420	-0.00034436	0.00021095	0.00034609	-0.00000115	0.00000000
		-6.27334429	-0.00034432	0.00021092	0.00034605	-0.00000115	0.00000000
		-6.27334438	-0.00034428	0.00021090	0.00034601	-0.00000115	0.00000000
		-6.27334447	-0.00034424	0.00021088	0.00034597	-0.00000115	0.00000000
		-6.27334456	-0.00034420	0.00021085	0.00034593	-0.00000115	0.00000000
		-6.27334465	-0.00034417	0.00021083	0.00034589	-0.00000115	0.00000000
		-6.27334475	-0.00034413	0.00021081	0.00034586	-0.00000115	0.00000000
		-6.27334484	-0.00034409	0.00021078	0.00034582	-0.00000115	0.00000000
		-6.27334493	-0.00034405	0.00021076	0.00034578	-0.00000115	0.00000000
		-6.27334502	-0.00034401	0.00021073	0.00034574	-0.00000115	0.00000000
		-6.27334511	-0.00034397	0.00021071	0.00034570	-0.00000115	0.00000000
		-6.27334520	-0.00034394	0.00021069	0.00034566	-0.00000115	0.00000000
		-6.27334529	-0.00034390	0.00021066	0.00034562	-0.00000115	0.00000000
		-6.27334539	-0.00034386	0.00021064	0.00034559	-0.00000115	0.00000000
		-6.27334548	-0.00034382	0.00021062	0.00034555	-0.00000115	0.00000000
		-6.27334557	-0.00034378	0.00021059	0.00034551	-0.00000115	0.00000000
		-6.27334566	-0.00034374	0.00021057	0.00034547	-0.00000115	0.00000000
		-6.27334575	-0.00034371	0.00021055	0.00034543	-0.00000116	0.00000000
		-6.27334584	-0.00034367	0.00021052	0.00034539	-0.00000116	0.00000000
		-6.27334593	-0.00034363	0.00021050	0.00034536	-0.00000116	0.00000000
		-6.27334603	-0.00034359	0.00021048	0.00034532	-0.00000116	0.00000000
		-6.27334612	-0.00034355	0.00021045	0.00034528	-0.00000116	0.00000000
		-6.27334621	-0.00034352	0.00021043	0.00034524	-0.00000116	0.00000000
		-6.27334630	-0.00034348	0.00021041	0.00034520	-0.00000116	0.00000000
		-6.27334639	-0.00034344	0.00021038	0.00034516	-0.00000116	0.00000000
		-6.27334648	-0.00034340	0.00021036	0.00034513	-0.00000116	0.00000000
		-6.27334657	-0.00034336	0.00021034	0.00034509	-0.00000116	0.00000000
		-6.27334666	-0.00034333	0.00021031	0.00034505	-0.00000116	0.00000000
		-6.27334676	-0.00034329	0.00021029	0.00034501	-0.00000116	0.00000000
		-6.27334685	-0.00034325	0.00021027	0.00034497	-0.00000116	0.00000000
		-6.27334694	-0.00034321	0.00021024	0.00034494	-0.00000116	0.00000000
		-6.27334703	-0.00034317	0.00021022	0.00034490	-0.00000116	0.00000000
		-6.27334712	-0.00034313	0.00021020	0.00034486	-0.00000116	0.00000000
		-6.27334721	-0.00034310	0.00021017	0.00034482	-0.00000116	0.00000000
		-6.27334730	-0.00034306	0.00021015	0.00034478	-0.00000116	0.00000000
		-6.27334739	-0.00034302	0.00021013	0.00034474	-0.00000116	0.00000000
		-6.27334748	-0.00034298	0.00021010	0.00034471	-0.00000116	0.00000000
		-6.27334758	-0.00034294	0.00021008	0.00034467	-0.00000116	0.00000000
		-6.27334767	-0.00034291	0.00021006	0.00034463	-0.00000116	0.00000000
		-6.27334776	-0.00034287	0.00021003	0.00034459	-0.00000116	0.00000000
		-6.27334785	-0.00034283	0.00021001	0.00034455	-0.00000116	0.00000000
		-6.27334794	-0.00034279	0.00020999	0.00034451	-0.00000116	0.00000000
		-6.27334803	-0.00034276	0.00020996	0.00034448	-0.00000116	0.00000000
		-6.27334812	-0.00034272	0.00020994	0.00034444	-0.00000116	0.00000000
		-6.27334821	-0.00034268	0.00020992	0.00034440	-0.00000116	0.00000000
		-6.27334830	-0.00034264	0.00020989	0.00034436	-0.00000116	0.00000000
		-6.27334839	-0.00034260	0.00020987	0.00034432	-0.00000116	0.00000000
		-6.27334848	-0.00034257	0.00020985	0.00034429	-0.00000116	0.00000000
		-6.27334857	-0.00034253	0.00020983	0.00034425	-0.00000116	0.00000000
		-6.27334867	-0.00034249	0.00020980	0.00034421	-0.00000116	0.00000000
		-6.27334876	-0.00034245	0.00020978	0.00034417	-0.00000116	0.00000000
		-6.27334885	-0.00034241	0.00020976	0.00034413	-0.00000116	0.00000000
		-6.27334894	-0.00034238	0.00020973	0.00034410	-0.00000116	0.00000000
		-6.27334903	-0.00034234	0.00020971	0.00034406	-0.00000116	0.00000000
		-6.27334912	-0.00034230	0.00020969	0.00034402	-0.00000116	0.00000000
		-6.27334921	-0.00034226	0.00020966	0.00034398	-0.00000116	0.00000000
		-6.27334930	-0.00034222	0.00020964	0.00034394	-0.00000116	0.00000000
		-6.27334939	-0.00034219	0.00020962	0.00034391	-0.00000116	0.00000000
		-6.27334948	-0.00034215	0.00020959	0.00034387	-0.00000116	0.00000000
		-6.27334957	-0.00034211	0.00020957	0.00034383	-0.00000116	0.00000000
		-6.27334966	-0.00034207	0.00020955	0.00034379	-0.00000116	0.00000000
		-6.27334975	-0.00034204	0.00020952	0.00034375	-0.00000116	0.00000000
		-6.27334984	-0.00034200	0.00020950	0.00034372	-0.00000116	0.00000000
		-6.27334993	-0.00034196	0.00020948	0.00034368	-0.00000116	0.00000000
		-6.27335002	-0.00034192	0.00020945	0.00034364	-0.00000116	0.00000000
		-6.27335011	-0.00034188	0.00020943	0.00034360	-0.00000116	0.00000000
		-6.27335021	-0.00034185	0.00020941	0.00034356	-0.00000116	0.00000000
		-6.27335030	-0.00034181	0.00020938	0.00034353	-0.00000116	0.00000000
		-6.27335039	-0.00034177	0.00020936	0.00034349	-0.00000116	0.00000000
		-6.27335048	-0.00034173	0.00020934	0.00034345	-0.00000116	0.00000000
		-6.27335057	-0.00034170	0.00020932	0.00034341	-0.00000116	0.00000000
		-6.27335066	-0.00034166	0.00020929	0.00034337	-0.00000116	0.00000000
		-6.27335075	-0.00034162	0.00020927	0.00034334	-0.00000116	0.00000000
		-6.27335084	-0.00034158	0.00020925	0.00034330	-0.00000116	0.00000000
		-6.27335093	-0.00034154	0.00020922	0.00034326	-0.00000116	0.00000000
		-6.27335102	-0.00034151	0.00020920	0.00034322	-0.00000116	0.00000000
		-6.27335111	-0.00034147	0.00020918	0.00034318	-0.00000116	0.00000000
		-6.27335120	-0.00034143	0.00020915	0.00034315	-0.00000116	0.00000000
		-6.27335129	-0.00034139	0.00020913	0.00034311	-0.00000116	0.00000000
		-6.27335138	-0.00034136	0.00020911	0.00034307	-0.00000116	0.00000000
		-6.27335147	-0.00034132	0.00020908	0.00034303	-0.00000116	0.00000000
		-6.27335156	-0.00034128	0.00020906	0.00034300	-0.00000116	0.00000000
		-6.27335165	-0.00034124	0.00020904	0.00034296	-0.00000116	0.00000000
		-6.27335174	-0.00034121	0.00020902	0.00034292	-0.00000116	0.00000000
		-6.27335183	-0.00034117	0.00020899	0.00034288	-0.00000116	0.00000000
		-6.27335192	-0.00034113	0.00020897	0.00034284	-0.00000116	0.00000000
		-6.27335201	-0.00034109	0.00020895	0.00034281	-0.00000116	0.00000000
		-6.27335210	-0.00034106	0.00020892	0.00034277	-0.00000116	0.00000000
		-6.27335219	-0.00034102	0.00020890	0.00034273	-0.00000116	0.00000000
		-6.27335228	-0.00034098	0.00020888	0.00034269	-0.00000116	0.00000000
		-6.27335237	-0.00034094	0.00020885	0.00034266	-0.00000116	0.00000000
		-6.27335246	-0.00034091	0.00020883	0.00034262	-0.00000116	0.00000000
		-6.27335255	-0.00034087	0.00020881	0.00034258	-0.00000116	0.00000000
		-6.27335264	-0.00034083	0.00020878	0.00034254	-0.00000116	0.00000000
		-6.27335273	-0.00034079	0.00020876	0.00034250	-0.00000116	0.00000000
		-6.27335282	-0.00034076	0.00020874	0.00034247	-0.00000116	0.00000000
		-6.27335291	-0.00034072	0.00020872	0.00034243	-0.00000116	0.00000000
		-6.27335300	-0.00034068	0.00020869	0.00034239	-0.00000116	0.00000000
		-6.27335309	-0.00034064	0.00020867	0.00034235	-0.00000116	0.00000000
		-6.27335318	-0.00034061	0.00020865	0.00034232	-0.00000116	0.00000000
		-6.27335327	-0.00034057	0.00020862	0.00034228	-0.00000116	0.00000000
		-6.27335336	-0.00034053	0.00020860	0.00034224	-0.00000116	0.00000000
		-6.27335345	-0.00034049	0.00020858	0.00034220	-0.00000116	0.00000000
		-6.27335354	-0.00034046	0.00020856	0.00034217	-0.00000116	0.00000000
		-6.27335363	-0.00034042	0.00020853	0.00034213	-0.00000116	0.00000000
		-6.27335372	-0.00034038	0.00020851	0.00034209	-0.00000116	0.00000000
		-6.27335380	-0.00034034	0.00020849	0.00034205	-0.00000116	0.00000000
		-6.27335389	-0.00034031	0.00020846	0.00034201	-0.00000116	0.00000000
		-6.27335398	-0.00034027	0.00020844	0.00034198	-0.00000116	0.00000000
		-6.27335407	-0.00034023	0.00020842	0.00034194	-0.00000116	0.00000000
		-6.27335416	-0.00034019	0.00020839	0.00034190	-0.00000116	0.00000000
		-6.27335425	-0.00034016	0.00020837	0.00034186	-0.00000116	0.00000000
		-6.27335434	-0.00034012	0.00020835	0.00034183	-0.00000116	0.00000000
		-6.27335443	-0.00034008	0.00020833	0.00034179	-0.00000116	0.00000000
		-6.27335452	-0.00034004	0.00020830	0.00034175	-0.00000116	0.00000000
		-6.27335461	-0.00034001	0.00020828	0.00034171	-0.00000116	0.00000000
		-6.27335470	-0.00033997	0.00020826	0.00034168	-0.00000116	0.00000000
		-6.27335479	-0.00033993	0.00020823	0.00034164	-0.00000116	0.00000000
		-6.27335488	-0.00033989	0.00020821	0.00034160	-0.00000116	0.00000000
		-6.27335497	-0.00033986	0.00020819	0.00034156	-0.00000116	0.00000000
		-6.27335506	-0.00033982	0.00020817	0.00034153	-0.00000116	0.00000000
		-6.27335515	-0.00033978	0.00020814	0.00034149	-0.00000116	0.00000000
		-6.27335524	-0.00033975	0.00020812	0.00034145	-0.00000116	0.00000000
		-6.27335533	-0.00033971	0.00020810	0.00034141	-0.00000116	0.00000000
		-6.27335541	-0.00033967	0.00020807	0.00034138	-0.00000116	0.00000000
		-6.27335550	-0.00033963	0.00020805	0.00034134	-0.00000116	0.00000000
		-6.27335559	-0.00033960	0.00020803	0.00034130	-0.00000116	0.00000000
		-6.27335568	-0.00033956	0.00020801	0.00034126	-0.00000116	0.00000000
		-6.27335577	-0.00033952	0.00020798	0.00034123	-0.00000116	0.00000000
		-6.27335586	-0.00033948	0.00020796	0.00034119	-0.00000116	0.00000000
		-6.27335595	-0.00033945	0.00020794	0.00034115	-0.00000116	0.00000000
		-6.27335604	-0.00033941	0.00020791	0.00034111	-0.00000116	0.00000000
		-6.27335613	-0.00033937	0.00020789	0.00034108	-0.00000116	0.00000000
		-6.27335622	-0.00033934	0.00020787	0.00034104	-0.00000116	0.00000000
		-6.27335631	-0.00033930	0.00020785	0.00034100	-0.00000116	0.00000000
		-6.27335639	-0.00033926	0.00020782	0.00034097	-0.00000116	0.00000000
		-6.27335648	-0.00033922	0.00020780	0.00034093	-0.00000116	0.00000000
		-6.27335657	-0.00033919	0.00020778	0.00034089	-0.00000116	0.00000000
		-6.27335666	-0.00033915	0.00020776	0.00034085	-0.00000116	0.00000000
		-6.27335675	-0.00033911	0.00020773	0.00034082	-0.00000116	0.00000000
		-6.27335684	-0.00033908	0.00020771	0.00034078	-0.00000116	0.00000000
		-6.27335693	-0.00033904	0.00020769	0.00034074	-0.00000116	0.00000000
		-6.27335702	-0.00033900	0.00020766	0.00034070	-0.00000116	0.00000000
		-6.27335711	-0.00033896	0.00020764	0.00034067	-0.00000116	0.00000000
		-6.27335720	-0.00033893	0.00020762	0.00034063	-0.00000116	0.00000000
		-6.27335728	-0.00033889	0.00020760	0.00034059	-0.00000116	0.00000000
		-6.27335737	-0.00033885	0.00020757	0.00034055	-0.00000116	0.00000000
		-6.27335746	-0.00033882	0.00020755	0.00034052	-0.00000116	0.00000000
		-6.27335755	-0.00033878	0.00020753	0.00034048	-0.00000116	0.00000000
		-6.27335764	-0.00033874	0.00020751	0.00034044	-0.00000116	0.00000000
		-6.27335773	-0.00033870	0.00020748	0.00034041	-0.00000116	0.00000000
		-6.27335782	-0.00033867	0.00020746	0.00034037	-0.00000116	0.00000000
		-6.27335791	-0.00033863	0.00020744	0.00034033	-0.00000116	0.00000000
		-6.27335799	-0.00033859	0.00020741	0.00034029	-0.00000116	0.00000000
		-6.27335808	-0.00033856	0.00020739	0.00034026	-0.00000116	0.00000000
		-6.27335817	-0.00033852	0.00020737	0.00034022	-0.00000116	0.00000000
		-6.27335826	-0.00033848	0.00020735	0.00034018	-0.00000116	0.00000000
		-6.27335835	-0.00033845	0.00020732	0.00034015	-0.00000116	0.00000000
		-6.27335844	-0.00033841	0.00020730	0.00034011	-0.00000116	0.00000000
		-6.27335853	-0.00033837	0.00020728	0.00034007	-0.00000116	0.00000000
		-6.27335861	-0.00033833	0.00020726	0.00034003	-0.00000116	0.00000000
		-6.27335870	-0.00033830	0.00020723	0.00034000	-0.00000116	0.00000000
		-6.27335879	-0.00033826	0.00020721	0.00033996	-0.00000116	0.00000000
		-6.27335888	-0.00033822	0.00020719	0.00033992	-0.00000116	0.00000000
		-6.27335897	-0.00033819	0.00020716	0.00033988	-0.00000116	0.00000000
		-6.27335906	-0.00033815	0.00020714	0.00033985	-0.00000116	0.00000000
		-6.27335915	-0.00033811	0.00020712	0.00033981	-0.00000116	0.00000000
		-6.27335923	-0.00033808	0.00020710	0.00033977	-0.00000116	0.00000000
		-6.27335932	-0.00033804	0.00020707	0.00033974	-0.00000116	0.00000000
		-6.27335941	-0.00033800	0.00020705	0.00033970	-0.00000116	0.00000000
		-6.27335950	-0.00033797	0.00020703	0.00033966	-0.00000116	0.00000000
		-6.27335959	-0.00033793	0.00020701	0.00033963	-0.00000116	0.00000000
		-6.27335968	-0.00033789	0.00020698	0.00033959	-0.00000116	0.00000000
		-6.27335976	-0.00033785	0.00020696	0.00033955	-0.00000116	0.00000000
		-6.27335985	-0.00033782	0.00020694	0.00033951	-0.00000116	0.00000000
		-6.27335994	-0.00033778	0.00020692	0.00033948	-0.00000116	0.00000000
		-6.27336003	-0.00033774	0.00020689	0.00033944	-0.00000116	0.00000000
		-6.27336012	-0.00033771	0.00020687	0.00033940	-0.00000116	0.00000000
		-6.27336021	-0.00033767	0.00020685	0.00033937	-0.00000116	0.00000000
		-6.27336029	-0.00033763	0.00020683	0.00033933	-0.00000116	0.00000000
		-6.27336038	-0.00033760	0.00020680	0.00033929	-0.00000116	0.00000000
		-6.27336047	-0.00033756	0.00020678	0.00033925	-0.00000116	0.00000000
		-6.27336056	-0.00033752	0.00020676	0.00033922	-0.00000116	0.00000000
		-6.27336065	-0.00033749	0.00020674	0.00033918	-0.00000116	0.00000000
		-6.27336073	-0.00033745	0.00020671	0.00033914	-0.00000116	0.00000000
		-6.27336082	-0.00033741	0.00020669	0.00033911	-0.00000116	0.00000000
		-6.27336091	-0.00033738	0.00020667	0.00033907	-0.00000116	0.00000000
		-6.27336100	-0.00033734	0.00020665	0.00033903	-0.00000116	0.00000000
		-6.27336109	-0.00033730	0.00020662	0.00033900	-0.00000116	0.00000000
		-6.27336117	-0.00033727	0.00020660	0.00033896	-0.00000116	0.00000000
		-6.27336126	-0.00033723	0.00020658	0.00033892	-0.00000116	0.00000000
		-6.27336135	-0.00033719	0.00020656	0.00033889	-0.00000116	0.00000000
		-6.27336144	-0.00033716	0.00020653	0.00033885	-0.00000116	0.00000000
		-6.27336153	-0.00033712	0.00020651	0.00033881	-0.00000116	0.00000000
		-6.27336161	-0.00033708	0.00020649	0.00033877	-0.00000116	0.00000000
		-6.27336170	-0.00033705	0.00020647	0.00033874	-0.00000116	0.00000000
		-6.27336179	-0.00033701	0.00020644	0.00033870	-0.00000116	0.00000000
		-6.27336188	-0.00033697	0.00020642	0.00033866	-0.00000116	0.00000000
		-6.27336197	-0.00033694	0.00020640	0.00033863	-0.00000116	0.00000000
		-6.27336205	-0.00033690	0.00020638	0.00033859	-0.00000116	0.00000000
		-6.27336214	-0.00033686	0.00020635	0.00033855	-0.00000116	0.00000000
		-6.27336223	-0.00033683	0.00020633	0.00033852	-0.00000116	0.00000000
		-6.27336232	-0.00033679	0.00020631	0.00033848	-0.00000116	0.00000000
		-6.27336240	-0.00033675	0.00020629	0.00033844	-0.00000116	0.00000000
		-6.27336249	-0.00033672	0.00020626	0.00033841	-0.00000116	0.00000000
		-6.27336258	-0.00033668	0.00020624	0.00033837	-0.00000116	0.00000000
		-6.27336267	-0.00033664	0.00020622	0.00033833	-0.00000116	0.00000000
		-6.27336275	-0.00033661	0.00020620	0.00033830	-0.00000116	0.00000000
		-6.27336284	-0.00033657	0.00020617	0.00033826	-0.00000116	0.00000000
		-6.27336293	-0.00033653	0.00020615	0.00033822	-0.00000116	0.00000000
		-6.27336302	-0.00033650	0.00020613	0.00033819	-0.00000116	0.00000000
		-6.27336311	-0.00033646	0.00020611	0.00033815	-0.00000116	0.00000000
		-6.27336319	-0.00033642	0.00020608	0.00033811	-0.00000116	0.00000000
		-6.27336328	-0.00033639	0.00020606	0.00033808	-0.00000116	0.00000000
		-6.27336337	-0.00033635	0.00020604	0.00033804	-0.00000116	0.00000000
		-6.27336346	-0.00033631	0.00020602	0.00033800	-0.00000116	0.00000000
		-6.27336354	-0.00033628	0.00020599	0.00033796	-0.00000116	0.00000000
		-6.27336363	-0.00033624	0.00020597	0.00033793	-0.00000116	0.00000000
		-6.27336372	-0.00033620	0.00020595	0.00033789	-0.00000116	0.00000000
		-6.27336381	-0.00033617	0.00020593	0.00033785	-0.00000116	0.00000000
		-6.27336389	-0.00033613	0.00020590	0.00033782	-0.00000116	0.00000000
		-6.27336398	-0.00033609	0.00020588	0.00033778	-0.00000116	0.00000000
		-6.27336407	-0.00033606	0.00020586	0.00033774	-0.00000116	0.00000000
		-6.27336415	-0.00033602	0.00020584	0.00033771	-0.00000116	0.00000000
		-6.27336424	-0.00033598	0.00020582	0.00033767	-0.00000116	0.00000000
		-6.27336433	-0.00033595	0.00020579	0.00033763	-0.00000116	0.00000000
		-6.27336442	-0.00033591	0.00020577	0.00033760	-0.00000116	0.00000000
		-6.27336450	-0.00033587	0.00020575	0.00033756	-0.00000116	0.00000000
		-6.27336459	-0.00033584	0.00020573	0.00033752	-0.00000116	0.00000000
		-6.27336468	-0.00033580	0.00020570	0.00033749	-0.00000116	0.00000000
		-6.27336477	-0.00033577	0.00020568	0.00033745	-0.00000116	0.00000000
		-6.27336485	-0.00033573	0.00020566	0.00033742	-0.00000116	0.00000000
		-6.27336494	-0.00033569	0.00020564	0.00033738	-0.00000116	0.00000000
		-6.27336503	-0.00033566	0.00020561	0.00033734	-0.00000116	0.00000000
		-6.27336511	-0.00033562	0.00020559	0.00033731	-0.00000116	0.00000000
		-6.27336520	-0.00033558	0.00020557	0.00033727	-0.00000116	0.00000000
		-6.27336529	-0.00033555	0.00020555	0.00033723	-0.00000116	0.00000000
		-6.27336538	-0.00033551	0.00020553	0.00033720	-0.00000116	0.00000000
		-6.27336546	-0.00033547	0.00020550	0.00033716	-0.00000116	0.00000000
		-6.27336555	-0.00033544	0.00020548	0.00033712	-0.00000116	0.00000000
		-6.27336564	-0.00033540	0.00020546	0.00033709	-0.00000116	0.00000000
		-6.27336572	-0.00033537	0.00020544	0.00033705	-0.00000116	0.00000000
		-6.27336581	-0.00033533	0.00020541	0.00033701	-0.00000116	0.00000000
		-6.27336590	-0.00033529	0.00020539	0.00033698	-0.00000116	0.00000000
		-6.27336598	-0.00033526	0.00020537	0.00033694	-0.00000116	0.00000000
		-6.27336607	-0.00033522	0.00020535	0.00033690	-0.00000116	0.00000000
		-6.27336616	-0.00033518	0.00020532	0.00033687	-0.00000116	0.00000000
		-6.27336625	-0.00033515	0.00020530	0.00033683	-0.00000116	0.00000000
		-6.27336633	-0.00033511	0.00020528	0.00033679	-0.00000116	0.00000000
		-6.27336642	-0.00033508	0.00020526	0.00033676	-0.00000116	0.00000000
		-6.27336651	-0.00033504	0.00020524	0.00033672	-0.00000116	0.00000000
		-6.27336659	-0.00033500	0.00020521	0.00033668	-0.00000116	0.00000000
		-6.27336668	-0.00033497	0.00020519	0.00033665	-0.00000116	0.00000000
		-6.27336677	-0.00033493	0.00020517	0.00033661	-0.00000116	0.00000000
		-6.27336685	-0.00033489	0.00020515	0.00033658	-0.00000116	0.00000000
		-6.27336694	-0.00033486	0.00020512	0.00033654	-0.00000116	0.00000000
		-6.27336703	-0.00033482	0.00020510	0.00033650	-0.00000116	0.00000000
		-6.27336711	-0.00033479	0.00020508	0.00033647	-0.00000116	0.00000000
		-6.27336720	-0.00033475	0.00020506	0.00033643	-0.00000116	0.00000000
		-6.27336729	-0.00033471	0.00020504	0.00033639	-0.00000116	0.00000000
		-6.27336737	-0.00033468	0.00020501	0.00033636	-0.00000116	0.00000000
		-6.27336746	-0.00033464	0.00020499	0.00033632	-0.00000116	0.00000000
		-6.27336755	-0.00033460	0.00020497	0.00033628	-0.00000116	0.00000000
		-6.27336763	-0.00033457	0.00020495	0.00033625	-0.00000116	0.00000000
		-6.27336772	-0.00033453	0.00020493	0.00033621	-0.00000116	0.00000000
		-6.27336781	-0.00033450	0.00020490	0.00033618	-0.00000116	0.00000000
		-6.27336789	-0.00033446	0.00020488	0.00033614	-0.00000116	0.00000000
		-6.27336798	-0.00033442	0.00020486	0.00033610	-0.00000116	0.00000000
		-6.27336807	-0.00033439	0.00020484	0.00033607	-0.00000116	0.00000000
		-6.27336815	-0.00033435	0.00020481	0.00033603	-0.00000116	0.00000000
		-6.27336824	-0.00033432	0.00020479	0.00033599	-0.00000116	0.00000000
		-6.27336833	-0.00033428	0.00020477	0.00033596	-0.00000116	0.00000000
		-6.27336841	-0.00033424	0.00020475	0.00033592	-0.00000116	0.00000000
		-6.27336850	-0.00033421	0.00020473	0.00033588	-0.00000116	0.00000000
		-6.27336858	-0.00033417	0.00020470	0.00033585	-0.00000116	0.00000000
		-6.27336867	-0.00033413	0.00020468	0.00033581	-0.00000116	0.00000000
		-6.27336876	-0.00033410	0.00020466	0.00033578	-0.00000116	0.00000000
		-6.27336884	-0.00033406	0.00020464	0.00033574	-0.00000116	0.00000000
		-6.27336893	-0.00033403	0.00020462	0.00033570	-0.00000116	0.00000000
		-6.27336902	-0.00033399	0.00020459	0.00033567	-0.00000116	0.00000000
		-6.27336910	-0.00033395	0.00020457	0.00033563	-0.00000116	0.00000000
		-6.27336919	-0.00033392	0.00020455	0.00033559	-0.00000116	0.00000000
		-6.27336927	-0.00033388	0.00020453	0.00033556	-0.00000116	0.00000000
		-6.27336936	-0.00033385	0.00020451	0.00033552	-0.00000116	0.00000000
		-6.27336945	-0.00033381	0.00020448	0.00033549	-0.00000116	0.00000000
		-6.27336953	-0.00033377	0.00020446	0.00033545	-0.00000116	0.00000000
		-6.27336962	-0.00033374	0.00020444	0.00033541	-0.00000116	0.00000000
		-6.27336971	-0.00033370	0.00020442	0.00033538	-0.00000116	0.00000000
		-6.27336979	-0.00033367	0.00020439	0.00033534	-0.00000116	0.00000000
		-6.27336988	-0.00033363	0.00020437	0.00033531	-0.00000116	0.00000000
		-6.27336996	-0.00033359	0.00020435	0.00033527	-0.00000116	0.00000000
		-6.27337005	-0.00033356	0.00020433	0.00033523	-0.00000116	0.00000000
		-6.27337014	-0.00033352	0.00020431	0.00033520	-0.00000116	0.00000000
		-6.27337022	-0.00033349	0.00020428	0.00033516	-0.00000116	0.00000000
		-6.27337031	-0.00033345	0.00020426	0.00033513	-0.00000116	0.00000000
		-6.27337039	-0.00033342	0.00020424	0.00033509	-0.00000116	0.00000000
		-6.27337048	-0.00033338	0.00020422	0.00033505	-0.00000116	0.00000000
		-6.27337057	-0.00033334	0.00020420	0.00033502	-0.00000116	0.00000000
		-6.27337065	-0.00033331	0.00020417	0.00033498	-0.00000116	0.00000000
		-6.27337074	-0.00033327	0.00020415	0.00033494	-0.00000116	0.00000000
		-6.27337082	-0.00033324	0.00020413	0.00033491	-0.00000116	0.00000000
		-6.27337091	-0.00033320	0.00020411	0.00033487	-0.00000116	0.00000000
		-6.27337100	-0.00033316	0.00020409	0.00033484	-0.00000116	0.00000000
		-6.27337108	-0.00033313	0.00020406	0.00033480	-0.00000116	0.00000000
		-6.27337117	-0.00033309	0.00020404	0.00033476	-0.00000116	0.00000000
		-6.27337125	-0.00033306	0.00020402	0.00033473	-0.00000116	0.00000000
		-6.27337134	-0.00033302	0.00020400	0.00033469	-0.00000116	0.00000000
		-6.27337142	-0.00033298	0.00020398	0.00033466	-0.00000116	0.00000000
		-6.27337151	-0.00033295	0.00020395	0.00033462	-0.00000116	0.00000000
		-6.27337160	-0.00033291	0.00020393	0.00033458	-0.00000116	0.00000000
		-6.27337168	-0.00033288	0.00020391	0.00033455	-0.00000116	0.00000000
		-6.27337177	-0.00033284	0.00020389	0.00033451	-0.00000116	0.00000000
		-6.27337185	-0.00033281	0.00020387	0.00033448	-0.00000116	0.00000000
		-6.27337194	-0.00033277	0.00020385	0.00033444	-0.00000116	0.00000000
		-6.27337202	-0.00033273	0.00020382	0.00033440	-0.00000116	0.00000000
		-6.27337211	-0.00033270	0.00020380	0.00033437	-0.00000116	0.00000000
		-6.27337220	-0.00033266	0.00020378	0.00033433	-0.00000116	0.00000000
		-6.27337228	-0.00033263	0.00020376	0.00033430	-0.00000116	0.00000000
		-6.27337237	-0.00033259	0.00020374	0.00033426	-0.00000116	0.00000000
		-6.27337245	-0.00033256	0.00020371	0.00033422	-0.00000116	0.00000000
		-6.27337254	-0.00033252	0.00020369	0.00033419	-0.00000116	0.00000000
		-6.27337262	-0.00033248	0.00020367	0.00033415	-0.00000116	0.00000000
		-6.27337271	-0.00033245	0.00020365	0.00033412	-0.00000116	0.00000000
		-6.27337279	-0.00033241	0.00020363	0.00033408	-0.00000116	0.00000000
		-6.27337288	-0.00033238	0.00020360	0.00033405	-0.00000116	0.00000000
		-6.27337297	-0.00033234	0.00020358	0.00033401	-0.00000116	0.00000000
		-6.27337305	-0.00033231	0.00020356	0.00033397	-0.00000116	0.00000000
		-6.27337314	-0.00033227	0.00020354	0.00033394	-0.00000116	0.00000000
		-6.27337322	-0.00033223	0.00020352	0.00033390	-0.00000116	0.00000000
		-6.27337331	-0.00033220	0.00020350	0.00033387	-0.00000116	0.00000000
		-6.27337339	-0.00033216	0.00020347	0.00033383	-0.00000116	0.00000000
		-6.27337348	-0.00033213	0.00020345	0.00033379	-0.00000116	0.00000000
		-6.27337356	-0.00033209	0.00020343	0.00033376	-0.00000116	0.00000000
		-6.27337365	-0.00033206	0.00020341	0.00033372	-0.00000116	0.00000000
		-6.27337373	-0.00033202	0.00020339	0.00033369	-0.00000116	0.00000000
		-6.27337382	-0.00033198	0.00020336	0.00033365	-0.00000116	0.00000000
		-6.27337390	-0.00033195	0.00020334	0.00033362	-0.00000116	0.00000000
		-6.27337399	-0.00033191	0.00020332	0.00033358	-0.00000116	0.00000000
		-6.27337407	-0.00033188	0.00020330	0.00033354	-0.00000116	0.00000000
		-6.27337416	-0.00033184	0.00020328	0.00033351	-0.00000116	0.00000000
		-6.27337425	-0.00033181	0.00020326	0.00033347	-0.00000116	0.00000000
		-6.27337433	-0.00033177	0.00020323	0.00033344	-0.00000116	0.00000000
		-6.27337442	-0.00033174	0.00020321	0.00033340	-0.00000116	0.00000000
		-6.27337450	-0.00033170	0.00020319	0.00033337	-0.00000116	0.00000000
		-6.27337459	-0.00033166	0.00020317	0.00033333	-0.00000116	0.00000000
		-6.27337467	-0.00033163	0.00020315	0.00033329	-0.00000116	0.00000000
		-6.27337476	-0.00033159	0.00020312	0.00033326	-0.00000116	0.00000000
		-6.27337484	-0.00033156	0.00020310	0.00033322	-0.00000116	0.00000000
		-6.27337493	-0.00033152	0.00020308	0.00033319	-0.00000116	0.00000000
		-6.27337501	-0.00033149	0.00020306	0.00033315	-0.00000116	0.00000000
		-6.27337510	-0.00033145	0.00020304	0.00033312	-0.00000116	0.00000000
		-6.27337518	-0.00033142	0.00020302	0.00033308	-0.00000116	0.00000000
		-6.27337527	-0.00033138	0.00020299	0.00033304	-0.00000116	0.00000000
		-6.27337535	-0.00033135	0.00020297	0.00033301	-0.00000116	0.00000000
		-6.27337544	-0.00033131	0.00020295	0.00033297	-0.00000116	0.00000000
		-6.27337552	-0.00033127	0.00020293	0.00033294	-0.00000116	0.00000000
		-6.27337561	-0.00033124	0.00020291	0.00033290	-0.00000116	0.00000000
		-6.27337569	-0.00033120	0.00020289	0.00033287	-0.00000116	0.00000000
		-6.27337578	-0.00033117	0.00020286	0.00033283	-0.00000116	0.00000000
		-6.27337586	-0.00033113	0.00020284	0.00033279	-0.00000116	0.00000000
		-6.27337594	-0.00033110	0.00020282	0.00033276	-0.00000116	0.00000000
		-6.27337603	-0.00033106	0.00020280	0.00033272	-0.00000116	0.00000000
		-6.27337611	-0.00033103	0.00020278	0.00033269	-0.00000116	0.00000000
		-6.27337620	-0.00033099	0.00020275	0.00033265	-0.00000116	0.00000000
		-6.27337628	-0.00033096	0.00020273	0.00033262	-0.00000116	0.00000000
		-6.27337637	-0.00033092	0.00020271	0.00033258	-0.00000116	0.00000000
		-6.27337645	-0.00033088	0.00020269	0.00033255	-0.00000116	0.00000000
		-6.27337654	-0.00033085	0.00020267	0.00033251	-0.00000116	0.00000000
		-6.27337662	-0.00033081	0.00020265	0.00033247	-0.00000116	0.00000000
		-6.27337671	-0.00033078	0.00020262	0.00033244	-0.00000116	0.00000000
		-6.27337679	-0.00033074	0.00020260	0.00033240	-0.00000116	0.00000000
		-6.27337688	-0.00033071	0.00020258	0.00033237	-0.00000116	0.00000000
		-6.27337696	-0.00033067	0.00020256	0.00033233	-0.00000116	0.00000000
		-6.27337705	-0.00033064	0.00020254	0.00033230	-0.00000116	0.00000000
		-6.27337713	-0.00033060	0.00020252	0.00033226	-0.00000116	0.00000000
		-6.27337721	-0.00033057	0.00020250	0.00033223	-0.00000116	0.00000000
		-6.27337730	-0.00033053	0.00020247	0.00033219	-0.00000116	0.00000000
		-6.27337738	-0.00033050	0.00020245	0.00033216	-0.00000116	0.00000000
		-6.27337747	-0.00033046	0.00020243	0.00033212	-0.00000116	0.00000000
		-6.27337755	-0.00033043	0.00020241	0.00033208	-0.00000116	0.00000000
		-6.27337764	-0.00033039	0.00020239	0.00033205	-0.00000116	0.00000000
		-6.27337772	-0.00033035	0.00020237	0.00033201	-0.00000116	0.00000000
		-6.27337781	-0.00033032	0.00020234	0.00033198	-0.00000116	0.00000000
		-6.27337789	-0.00033028	0.00020232	0.00033194	-0.00000116	0.00000000
		-6.27337797	-0.00033025	0.00020230	0.00033191	-0.00000116	0.00000000
		-6.27337806	-0.00033021	0.00020228	0.00033187	-0.00000116	0.00000000
		-6.27337814	-0.00033018	0.00020226	0.00033184	-0.00000116	0.00000000
		-6.27337823	-0.00033014	0.00020224	0.00033180	-0.00000116	0.00000000
		-6.27337831	-0.00033011	0.00020221	0.00033177	-0.00000116	0.00000000
		-6.27337840	-0.00033007	0.00020219	0.00033173	-0.00000116	0.00000000
		-6.27337848	-0.00033004	0.00020217	0.00033169	-0.00000116	0.00000000
		-6.27337856	-0.00033000	0.00020215	0.00033166	-0.00000116	0.00000000
		-6.27337865	-0.00032997	0.00020213	0.00033162	-0.00000116	0.00000000
		-6.27337873	-0.00032993	0.00020211	0.00033159	-0.00000116	0.00000000
		-6.27337882	-0.00032990	0.00020208	0.00033155	-0.00000116	0.00000000
		-6.27337890	-0.00032986	0.00020206	0.00033152	-0.00000116	0.00000000
		-6.27337899	-0.00032983	0.00020204	0.00033148	-0.00000116	0.00000000
		-6.27337907	-0.00032979	0.00020202	0.00033145	-0.00000116	0.00000000
		-6.27337915	-0.00032976	0.00020200	0.00033141	-0.00000116	0.00000000
		-6.27337924	-0.00032972	0.00020198	0.00033138	-0.00000116	0.00000000
		-6.27337932	-0.00032969	0.00020196	0.00033134	-0.00000116	0.00000000
		-6.27337941	-0.00032965	0.00020193	0.00033131	-0.00000116	0.00000000
		-6.27337949	-0.00032962	0.00020191	0.00033127	-0.00000116	0.00000000
		-6.27337957	-0.00032958	0.00020189	0.00033124	-0.00000116	0.00000000
		-6.27337966	-0.00032955	0.00020187	0.00033120	-0.00000116	0.00000000
		-6.27337974	-0.00032951	0.00020185	0.00033116	-0.00000116	0.00000000
		-6.27337983	-0.00032948	0.00020183	0.00033113	-0.00000116	0.00000000
		-6.27337991	-0.00032944	0.00020181	0.00033109	-0.00000116	0.00000000
		-6.27337999	-0.00032941	0.00020178	0.00033106	-0.00000116	0.00000000
		-6.27338008	-0.00032937	0.00020176	0.00033102	-0.00000116	0.00000000
		-6.27338016	-0.00032934	0.00020174	0.00033099	-0.00000116	0.00000000
		-6.27338025	-0.00032930	0.00020172	0.00033095	-0.00000116	0.00000000
		-6.27338033	-0.00032927	0.00020170	0.00033092	-0.00000116	0.00000000
		-6.27338041	-0.00032923	0.00020168	0.00033088	-0.00000116	0.00000000
		-6.27338050	-0.00032920	0.00020165	0.00033085	-0.00000116	0.00000000
		-6.27338058	-0.00032916	0.00020163	0.00033081	-0.00000116	0.00000000
		-6.27338067	-0.00032913	0.00020161	0.00033078	-0.00000116	0.00000000
		-6.27338075	-0.00032909	0.00020159	0.00033074	-0.00000116	0.00000000
		-6.27338083	-0.00032906	0.00020157	0.00033071	-0.00000116	0.00000000
		-6.27338092	-0.00032902	0.00020155	0.00033067	-0.00000116	0.00000000
		-6.27338100	-0.00032899	0.00020153	0.00033064	-0.00000116	0.00000000
		-6.27338108	-0.00032895	0.00020150	0.00033060	-0.00000116	0.00000000
		-6.27338117	-0.00032892	0.00020148	0.00033057	-0.00000116	0.00000000
		-6.27338125	-0.00032888	0.00020146	0.00033053	-0.00000116	0.00000000
		-6.27338134	-0.00032885	0.00020144	0.00033050	-0.00000116	0.00000000
		-6.27338142	-0.00032881	0.00020142	0.00033046	-0.00000116	0.00000000
		-6.27338150	-0.00032878	0.00020140	0.00033043	-0.00000116	0.00000000
		-6.27338159	-0.00032874	0.00020138	0.00033039	-0.00000116	0.00000000
		-6.27338167	-0.00032871	0.00020135	0.00033036	-0.00000116	0.00000000
		-6.27338175	-0.00032867	0.00020133	0.00033032	-0.00000116	0.00000000
		-6.27338184	-0.00032864	0.00020131	0.00033029	-0.00000116	0.00000000
		-6.27338192	-0.00032860	0.00020129	0.00033025	-0.00000116	0.00000000
		-6.27338200	-0.00032857	0.00020127	0.00033022	-0.00000116	0.00000000
		-6.27338209	-0.00032853	0.00020125	0.00033018	-0.00000116	0.00000000
		-6.27338217	-0.00032850	0.00020123	0.00033015	-0.00000116	0.00000000
		-6.27338225	-0.00032846	0.00020121	0.00033011	-0.00000116	0.00000000
		-6.27338234	-0.00032843	0.00020118	0.00033008	-0.00000116	0.00000000
		-6.27338242	-0.00032839	0.00020116	0.00033004	-0.00000116	0.00000000
		-6.27338250	-0.00032836	0.00020114	0.00033001	-0.00000116	0.00000000
		-6.27338259	-0.00032832	0.00020112	0.00032997	-0.00000116	0.00000000
		-6.27338267	-0.00032829	0.00020110	0.00032994	-0.00000116	0.00000000
		-6.27338275	-0.00032825	0.00020108	0.00032990	-0.00000116	0.00000000
		-6.27338284	-0.00032822	0.00020106	0.00032987	-0.00000116	0.00000000
		-6.27338292	-0.00032818	0.00020103	0.00032983	-0.00000116	0.00000000
		-6.27338300	-0.00032815	0.00020101	0.00032980	-0.00000116	0.00000000
		-6.27338309	-0.00032811	0.00020099	0.00032976	-0.00000116	0.00000000
		-6.27338317	-0.00032808	0.00020097	0.00032973	-0.00000116	0.00000000
		-6.27338325	-0.00032804	0.00020095	0.00032969	-0.00000116	0.00000000
		-6.27338334	-0.00032801	0.00020093	0.00032966	-0.00000116	0.00000000
		-6.27338342	-0.00032797	0.00020091	0.00032962	-0.00000116	0.00000000
		-6.27338350	-0.00032794	0.00020089	0.00032959	-0.00000116	0.00000000
		-6.27338359	-0.00032791	0.00020086	0.00032955	-0.00000116	0.00000000
		-6.27338367	-0.00032787	0.00020084	0.00032952	-0.00000116	0.00000000
		-6.27338375	-0.00032784	0.00020082	0.00032948	-0.00000116	0.00000000
		-6.27338384	-0.00032780	0.00020080	0.00032945	-0.00000116	0.00000000
		-6.27338392	-0.00032777	0.00020078	0.00032941	-0.00000116	0.00000000
		-6.27338400	-0.00032773	0.00020076	0.00032938	-0.00000116	0.00000000
		-6.27338409	-0.00032770	0.00020074	0.00032934	-0.00000116	0.00000000
		-6.27338417	-0.00032766	0.00020072	0.00032931	-0.00000116	0.00000000
		-6.27338425	-0.00032763	0.00020069	0.00032927	-0.00000116	0.00000000
		-6.27338433	-0.00032759	0.00020067	0.00032924	-0.00000116	0.00000000
		-6.27338442	-0.00032756	0.00020065	0.00032920	-0.00000116	0.00000000
		-6.27338450	-0.00032752	0.00020063	0.00032917	-0.00000116	0.00000000
		-6.27338458	-0.00032749	0.00020061	0.00032913	-0.00000116	0.00000000
		-6.27338467	-0.00032745	0.00020059	0.00032910	-0.00000116	0.00000000
		-6.27338475	-0.00032742	0.00020057	0.00032906	-0.00000116	0.00000000
		-6.27338483	-0.00032739	0.00020055	0.00032903	-0.00000116	0.00000000
		-6.27338492	-0.00032735	0.00020052	0.00032899	-0.00000116	0.00000000
		-6.27338500	-0.00032732	0.00020050	0.00032896	-0.00000116	0.00000000
		-6.27338508	-0.00032728	0.00020048	0.00032892	-0.00000116	0.00000000
		-6.27338516	-0.00032725	0.00020046	0.00032889	-0.00000116	0.00000000
		-6.27338525	-0.00032721	0.00020044	0.00032885	-0.00000116	0.00000000
		-6.27338533	-0.00032718	0.00020042	0.00032882	-0.00000116	0.00000000
		-6.27338541	-0.00032714	0.00020040	0.00032878	-0.00000116	0.00000000
		-6.27338550	-0.00032711	0.00020038	0.00032875	-0.00000116	0.00000000
		-6.27338558	-0.00032707	0.00020035	0.00032872	-0.00000116	0.00000000
		-6.27338566	-0.00032704	0.00020033	0.00032868	-0.00000116	0.00000000
		-6.27338574	-0.00032700	0.00020031	0.00032865	-0.00000116	0.00000000
		-6.27338583	-0.00032697	0.00020029	0.00032861	-0.00000116	0.00000000
		-6.27338591	-0.00032694	0.00020027	0.00032858	-0.00000116	0.00000000
		-6.27338599	-0.00032690	0.00020025	0.00032854	-0.00000116	0.00000000
		-6.27338607	-0.00032687	0.00020023	0.00032851	-0.00000116	0.00000000
		-6.27338616	-0.00032683	0.00020021	0.00032847	-0.00000116	0.00000000
		-6.27338624	-0.00032680	0.00020019	0.00032844	-0.00000116	0.00000000
		-6.27338632	-0.00032676	0.00020016	0.00032840	-0.00000116	0.00000000
		-6.27338640	-0.00032673	0.00020014	0.00032837	-0.00000116	0.00000000
		-6.27338649	-0.00032669	0.00020012	0.00032833	-0.00000116	0.00000000
		-6.27338657	-0.00032666	0.00020010	0.00032830	-0.00000116	0.00000000
		-6.27338665	-0.00032663	0.00020008	0.00032826	-0.00000116	0.00000000
		-6.27338674	-0.00032659	0.00020006	0.00032823	-0.00000116	0.00000000
		-6.27338682	-0.00032656	0.00020004	0.00032820	-0.00000116	0.00000000
		-6.27338690	-0.00032652	0.00020002	0.00032816	-0.00000116	0.00000000
		-6.27338698	-0.00032649	0.00020000	0.00032813	-0.00000116	0.00000000
		-6.27338706	-0.00032645	0.00019997	0.00032809	-0.00000116	0.00000000
		-6.27338715	-0.00032642	0.00019995	0.00032806	-0.00000116	0.00000000
		-6.27338723	-0.00032638	0.00019993	0.00032802	-0.00000116	0.00000000
		-6.27338731	-0.00032635	0.00019991	0.00032799	-0.00000116	0.00000000
		-6.27338739	-0.00032632	0.00019989	0.00032795	-0.00000116	0.00000000
		-6.27338748	-0.00032628	0.00019987	0.00032792	-0.00000116	0.00000000
		-6.27338756	-0.00032625	0.00019985	0.00032788	-0.00000116	0.00000000
		-6.27338764	-0.00032621	0.00019983	0.00032785	-0.00000116	0.00000000
		-6.27338772	-0.00032618	0.00019981	0.00032781	-0.00000116	0.00000000
		-6.27338781	-0.00032614	0.00019978	0.00032778	-0.00000116	0.00000000
		-6.27338789	-0.00032611	0.00019976	0.00032775	-0.00000116	0.00000000
		-6.27338797	-0.00032607	0.00019974	0.00032771	-0.00000116	0.00000000
		-6.27338805	-0.00032604	0.00019972	0.00032768	-0.00000116	0.00000000
		-6.27338814	-0.00032601	0.00019970	0.00032764	-0.00000116	0.00000000
		-6.27338822	-0.00032597	0.00019968	0.00032761	-0.00000116	0.00000000
		-6.27338830	-0.00032594	0.00019966	0.00032757	-0.00000116	0.00000000
		-6.27338838	-0.00032590	0.00019964	0.00032754	-0.00000116	0.00000000
		-6.27338846	-0.00032587	0.00019962	0.00032750	-0.00000116	0.00000000
		-6.27338855	-0.00032583	0.00019960	0.00032747	-0.00000116	0.00000000
		-6.27338863	-0.00032580	0.00019957	0.00032744	-0.00000116	0.00000000
		-6.27338871	-0.00032577	0.00019955	0.00032740	-0.00000116	0.00000000
		-6.27338879	-0.00032573	0.00019953	0.00032737	-0.00000116	0.00000000
		-6.27338887	-0.00032570	0.00019951	0.00032733	-0.00000116	0.00000000
		-6.27338896	-0.00032566	0.00019949	0.00032730	-0.00000116	0.00000000
		-6.27338904	-0.00032563	0.00019947	0.00032726	-0.00000116	0.00000000
		-6.27338912	-0.00032559	0.00019945	0.00032723	-0.00000116	0.00000000
		-6.27338920	-0.00032556	0.00019943	0.00032719	-0.00000116	0.00000000
		-6.27338928	-0.00032553	0.00019941	0.00032716	-0.00000116	0.00000000
		-6.27338937	-0.00032549	0.00019939	0.00032713	-0.00000116	0.00000000
		-6.27338945	-0.00032546	0.00019936	0.00032709	-0.00000116	0.00000000
		-6.27338953	-0.00032542	0.00019934	0.00032706	-0.00000116	0.00000000
		-6.27338961	-0.00032539	0.00019932	0.00032702	-0.00000116	0.00000000
		-6.27338969	-0.00032536	0.00019930	0.00032699	-0.00000116	0.00000000
		-6.27338978	-0.00032532	0.00019928	0.00032695	-0.00000116	0.00000000
		-6.27338986	-0.00032529	0.00019926	0.00032692	-0.00000116	0.00000000
		-6.27338994	-0.00032525	0.00019924	0.00032688	-0.00000116	0.00000000
		-6.27339002	-0.00032522	0.00019922	0.00032685	-0.00000116	0.00000000
		-6.27339010	-0.00032518	0.00019920	0.00032682	-0.00000116	0.00000000
		-6.27339019	-0.00032515	0.00019918	0.00032678	-0.00000116	0.00000000
		-6.27339027	-0.00032512	0.00019915	0.00032675	-0.00000116	0.00000000
		-6.27339035	-0.00032508	0.00019913	0.00032671	-0.00000116	0.00000000
		-6.27339043	-0.00032505	0.00019911	0.00032668	-0.00000116	0.00000000
		-6.27339051	-0.00032501	0.00019909	0.00032664	-0.00000116	0.00000000
		-6.27339059	-0.00032498	0.00019907	0.00032661	-0.00000116	0.00000000
		-6.27339068	-0.00032495	0.00019905	0.00032658	-0.00000116	0.00000000
		-6.27339076	-0.00032491	0.00019903	0.00032654	-0.00000116	0.00000000
		-6.27339084	-0.00032488	0.00019901	0.00032651	-0.00000116	0.00000000
		-6.27339092	-0.00032484	0.00019899	0.00032647	-0.00000116	0.00000000
		-6.27339100	-0.00032481	0.00019897	0.00032644	-0.00000116	0.00000000
		-6.27339108	-0.00032477	0.00019895	0.00032640	-0.00000116	0.00000000
		-6.27339117	-0.00032474	0.00019892	0.00032637	-0.00000116	0.00000000
		-6.27339125	-0.00032471	0.00019890	0.00032634	-0.00000116	0.00000000
		-6.27339133	-0.00032467	0.00019888	0.00032630	-0.00000116	0.00000000
		-6.27339141	-0.00032464	0.00019886	0.00032627	-0.00000116	0.00000000
		-6.27339149	-0.00032460	0.00019884	0.00032623	-0.00000116	0.00000000
		-6.27339157	-0.00032457	0.00019882	0.00032620	-0.00000116	0.00000000
		-6.27339165	-0.00032454	0.00019880	0.00032617	-0.00000116	0.00000000
		-6.27339174	-0.00032450	0.00019878	0.00032613	-0.00000116	0.00000000
		-6.27339182	-0.00032447	0.00019876	0.00032610	-0.00000116	0.00000000
		-6.27339190	-0.00032443	0.00019874	0.00032606	-0.00000116	0.00000000
		-6.27339198	-0.00032440	0.00019872	0.00032603	-0.00000116	0.00000000
		-6.27339206	-0.00032437	0.00019870	0.00032599	-0.00000116	0.00000000
		-6.27339214	-0.00032433	0.00019867	0.00032596	-0.00000116	0.00000000
		-6.27339222	-0.00032430	0.00019865	0.00032593	-0.00000116	0.00000000
		-6.27339231	-0.00032426	0.00019863	0.00032589	-0.00000116	0.00000000
		-6.27339239	-0.00032423	0.00019861	0.00032586	-0.00000116	0.00000000
		-6.27339247	-0.00032420	0.00019859	0.00032582	-0.00000116	0.00000000
		-6.27339255	-0.00032416	0.00019857	0.00032579	-0.00000116	0.00000000
		-6.27339263	-0.00032413	0.00019855	0.00032576	-0.00000116	0.00000000
		-6.27339271	-0.00032409	0.00019853	0.00032572	-0.00000116	0.00000000
		-6.27339279	-0.00032406	0.00019851	0.00032569	-0.00000116	0.00000000
		-6.27339287	-0.00032403	0.00019849	0.00032565	-0.00000116	0.00000000
		-6.27339296	-0.00032399	0.00019847	0.00032562	-0.00000116	0.00000000
		-6.27339304	-0.00032396	0.00019845	0.00032558	-0.00000116	0.00000000
		-6.27339312	-0.00032393	0.00019843	0.00032555	-0.00000116	0.00000000
		-6.27339320	-0.00032389	0.00019840	0.00032552	-0.00000116	0.00000000
		-6.27339328	-0.00032386	0.00019838	0.00032548	-0.00000116	0.00000000
		-6.27339336	-0.00032382	0.00019836	0.00032545	-0.00000116	0.00000000
		-6.27339344	-0.00032379	0.00019834	0.00032541	-0.00000116	0.00000000
		-6.27339352	-0.00032376	0.00019832	0.00032538	-0.00000116	0.00000000
		-6.27339360	-0.00032372	0.00019830	0.00032535	-0.00000116	0.00000000
		-6.27339369	-0.00032369	0.00019828	0.00032531	-0.00000116	0.00000000
		-6.27339377	-0.00032365	0.00019826	0.00032528	-0.00000116	0.00000000
		-6.27339385	-0.00032362	0.00019824	0.00032524	-0.00000116	0.00000000
		-6.27339393	-0.00032359	0.00019822	0.00032521	-0.00000116	0.00000000
		-6.27339401	-0.00032355	0.00019820	0.00032518	-0.00000116	0.00000000
		-6.27339409	-0.00032352	0.00019818	0.00032514	-0.00000116	0.00000000
		-6.27339417	-0.00032349	0.00019816	0.00032511	-0.00000116	0.00000000
		-6.27339425	-0.00032345	0.00019813	0.00032507	-0.00000116	0.00000000
		-6.27339433	-0.00032342	0.00019811	0.00032504	-0.00000116	0.00000000
		-6.27339441	-0.00032338	0.00019809	0.00032501	-0.00000116	0.00000000
		-6.27339450	-0.00032335	0.00019807	0.00032497	-0.00000116	0.00000000
		-6.27339458	-0.00032332	0.00019805	0.00032494	-0.00000116	0.00000000
		-6.27339466	-0.00032328	0.00019803	0.00032490	-0.00000116	0.00000000
		-6.27339474	-0.00032325	0.00019801	0.00032487	-0.00000116	0.00000000
		-6.27339482	-0.00032321	0.00019799	0.00032484	-0.00000116	0.00000000
		-6.27339490	-0.00032318	0.00019797	0.00032480	-0.00000116	0.00000000
		-6.27339498	-0.00032315	0.00019795	0.00032477	-0.00000116	0.00000000
		-6.27339506	-0.00032311	0.00019793	0.00032473	-0.00000116	0.00000000
		-6.27339514	-0.00032308	0.00019791	0.00032470	-0.00000116	0.00000000
		-6.27339522	-0.00032305	0.00019789	0.00032467	-0.00000116	0.00000000
		-6.27339530	-0.00032301	0.00019787	0.00032463	-0.00000116	0.00000000
		-6.27339538	-0.00032298	0.00019785	0.00032460	-0.00000116	0.00000000
		-6.27339546	-0.00032295	0.00019782	0.00032457	-0.00000116	0.00000000
		-6.27339555	-0.00032291	0.00019780	0.00032453	-0.00000116	0.00000000
		-6.27339563	-0.00032288	0.00019778	0.00032450	-0.00000116	0.00000000
		-6.27339571	-0.00032284	0.00019776	0.00032446	-0.00000116	0.00000000
		-6.27339579	-0.00032281	0.00019774	0.00032443	-0.00000116	0.00000000
		-6.27339587	-0.00032278	0.00019772	0.00032440	-0.00000116	0.00000000
		-6.27339595	-0.00032274	0.00019770	0.00032436	-0.00000116	0.00000000
		-6.27339603	-0.00032271	0.00019768	0.00032433	-0.00000116	0.00000000
		-6.27339611	-0.00032268	0.00019766	0.00032429	-0.00000116	0.00000000
		-6.27339619	-0.00032264	0.00019764	0.00032426	-0.00000116	0.00000000
		-6.27339627	-0.00032261	0.00019762	0.00032423	-0.00000116	0.00000000
		-6.27339635	-0.00032258	0.00019760	0.00032419	-0.00000116	0.00000000
		-6.27339643	-0.00032254	0.00019758	0.00032416	-0.00000116	0.00000000
		-6.27339651	-0.00032251	0.00019756	0.00032413	-0.00000116	0.00000000
		-6.27339659	-0.00032247	0.00019754	0.00032409	-0.00000116	0.00000000
		-6.27339667	-0.00032244	0.00019752	0.00032406	-0.00000116	0.00000000
		-6.27339675	-0.00032241	0.00019749	0.00032402	-0.00000116	0.00000000
		-6.27339683	-0.00032237	0.00019747	0.00032399	-0.00000116	0.00000000
		-6.27339691	-0.00032234	0.00019745	0.00032396	-0.00000116	0.00000000
		-6.27339699	-0.00032231	0.00019743	0.00032392	-0.00000116	0.00000000
		-6.27339708	-0.00032227	0.00019741	0.00032389	-0.00000116	0.00000000
		-6.27339716	-0.00032224	0.00019739	0.00032386	-0.00000116	0.00000000
		-6.27339724	-0.00032221	0.00019737	0.00032382	-0.00000116	0.00000000
		-6.27339732	-0.00032217	0.00019735	0.00032379	-0.00000116	0.00000000
		-6.27339740	-0.00032214	0.00019733	0.00032375	-0.00000116	0.00000000
		-6.27339748	-0.00032210	0.00019731	0.00032372	-0.00000116	0.00000000
		-6.27339756	-0.00032207	0.00019729	0.00032369	-0.00000116	0.00000000
		-6.27339764	-0.00032204	0.00019727	0.00032365	-0.00000116	0.00000000
		-6.27339772	-0.00032200	0.00019725	0.00032362	-0.00000116	0.00000000
		-6.27339780	-0.00032197	0.00019723	0.00032359	-0.00000116	0.00000000
		-6.27339788	-0.00032194	0.00019721	0.00032355	-0.00000116	0.00000000
		-6.27339796	-0.00032190	0.00019719	0.00032352	-0.00000116	0.00000000
		-6.27339804	-0.00032187	0.00019717	0.00032349	-0.00000116	0.00000000
		-6.27339812	-0.00032184	0.00019715	0.00032345	-0.00000116	0.00000000
		-6.27339820	-0.00032180	0.00019713	0.00032342	-0.00000116	0.00000000
		-6.27339828	-0.00032177	0.00019710	0.00032338	-0.00000116	0.00000000
		-6.27339836	-0.00032174	0.00019708	0.00032335	-0.00000116	0.00000000
		-6.27339844	-0.00032170	0.00019706	0.00032332	-0.00000116	0.00000000
		-6.27339852	-0.00032167	0.00019704	0.00032328	-0.00000116	0.00000000
		-6.27339860	-0.00032164	0.00019702	0.00032325	-0.00000116	0.00000000
		-6.27339868	-0.00032160	0.00019700	0.00032322	-0.00000116	0.00000000
		-6.27339876	-0.00032157	0.00019698	0.00032318	-0.00000116	0.00000000
		-6.27339884	-0.00032154	0.00019696	0.00032315	-0.00000116	0.00000000
		-6.27339892	-0.00032150	0.00019694	0.00032312	-0.00000116	0.00000000
		-6.27339900	-0.00032147	0.00019692	0.00032308	-0.00000116	0.00000000
		-6.27339908	-0.00032144	0.00019690	0.00032305	-0.00000116	0.00000000
		-6.27339916	-0.00032140	0.00019688	0.00032302	-0.00000116	0.00000000
		-6.27339924	-0.00032137	0.00019686	0.00032298	-0.00000116	0.00000000
		-6.27339932	-0.00032134	0.00019684	0.00032295	-0.00000116	0.00000000
		-6.27339940	-0.00032130	0.00019682	0.00032291	-0.00000116	0.00000000
		-6.27339948	-0.00032127	0.00019680	0.00032288	-0.00000116	0.00000000
		-6.27339956	-0.00032124	0.00019678	0.00032285	-0.00000116	0.00000000
		-6.27339964	-0.00032120	0.00019676	0.00032281	-0.00000116	0.00000000
		-6.27339972	-0.00032117	0.00019674	0.00032278	-0.00000116	0.00000000
		-6.27339980	-0.00032114	0.00019672	0.00032275	-0.00000116	0.00000000
		-6.27339988	-0.00032110	0.00019670	0.00032271	-0.00000116	0.00000000
		-6.27339996	-0.00032107	0.00019668	0.00032268	-0.00000116	0.00000000
		-6.27340004	-0.00032104	0.00019665	0.00032265	-0.00000116	0.00000000
		-6.27340012	-0.00032100	0.00019663	0.00032261	-0.00000116	0.00000000
		-6.27340020	-0.00032097	0.00019661	0.00032258	-0.00000116	0.00000000
		-6.27340028	-0.00032094	0.00019659	0.00032255	-0.00000116	0.00000000
		-6.27340036	-0.00032090	0.00019657	0.00032251	-0.00000116	0.00000000
		-6.27340043	-0.00032087	0.00019655	0.00032248	-0.00000116	0.00000000
		-6.27340051	-0.00032084	0.00019653	0.00032245	-0.00000116	0.00000000
		-6.27340059	-0.00032080	0.00019651	0.00032241	-0.00000116	0.00000000
		-6.27340067	-0.00032077	0.00019649	0.00032238	-0.00000116	0.00000000
		-6.27340075	-0.00032074	0.00019647	0.00032235	-0.00000116	0.00000000
		-6.27340083	-0.00032070	0.00019645	0.00032231	-0.00000116	0.00000000
		-6.27340091	-0.00032067	0.00019643	0.00032228	-0.00000116	0.00000000
		-6.27340099	-0.00032064	0.00019641	0.00032225	-0.00000116	0.00000000
		-6.27340107	-0.00032060	0.00019639	0.00032221	-0.00000116	0.00000000
		-6.27340115	-0.00032057	0.00019637	0.00032218	-0.00000116	0.00000000
		-6.27340123	-0.00032054	0.00019635	0.00032215	-0.00000116	0.00000000
		-6.27340131	-0.00032050	0.00019633	0.00032211	-0.00000116	0.00000000
		-6.27340139	-0.00032047	0.00019631	0.00032208	-0.00000116	0.00000000
		-6.27340147	-0.00032044	0.00019629	0.00032205	-0.00000116	0.00000000
		-6.27340155	-0.00032040	0.00019627	0.00032201	-0.00000116	0.00000000
		-6.27340163	-0.00032037	0.00019625	0.00032198	-0.00000116	0.00000000
		-6.27340171	-0.00032034	0.00019623	0.00032195	-0.00000116	0.00000000
		-6.27340179	-0.00032031	0.00019621	0.00032191	-0.00000116	0.00000000
		-6.27340187	-0.00032027	0.00019619	0.00032188	-0.00000116	0.00000000
		-6.27340194	-0.00032024	0.00019617	0.00032185	-0.00000116	0.00000000
		-6.27340202	-0.00032021	0.00019615	0.00032181	-0.00000116	0.00000000
		-6.27340210	-0.00032017	0.00019613	0.00032178	-0.00000116	0.00000000
		-6.27340218	-0.00032014	0.00019611	0.00032175	-0.00000116	0.00000000
		-6.27340226	-0.00032011	0.00019608	0.00032171	-0.00000116	0.00000000
		-6.27340234	-0.00032007	0.00019606	0.00032168	-0.00000116	0.00000000
		-6.27340242	-0.00032004	0.00019604	0.00032165	-0.00000116	0.00000000
		-6.27340250	-0.00032001	0.00019602	0.00032161	-0.00000116	0.00000000
		-6.27340258	-0.00031997	0.00019600	0.00032158	-0.00000116	0.00000000
		-6.27340266	-0.00031994	0.00019598	0.00032155	-0.00000116	0.00000000
		-6.27340274	-0.00031991	0.00019596	0.00032151	-0.00000116	0.00000000
		-6.27340282	-0.00031987	0.00019594	0.00032148	-0.00000116	0.00000000
		-6.27340290	-0.00031984	0.00019592	0.00032145	-0.00000116	0.00000000
		-6.27340297	-0.00031981	0.00019590	0.00032141	-0.00000116	0.00000000
		-6.27340305	-0.00031978	0.00019588	0.00032138	-0.00000116	0.00000000
		-6.27340313	-0.00031974	0.00019586	0.00032135	-0.00000116	0.00000000
		-6.27340321	-0.00031971	0.00019584	0.00032131	-0.00000116	0.00000000
		-6.27340329	-0.00031968	0.00019582	0.00032128	-0.00000116	0.00000000
		-6.27340337	-0.00031964	0.00019580	0.00032125	-0.00000116	0.00000000
		-6.27340345	-0.00031961	0.00019578	0.00032121	-0.00000116	0.00000000
		-6.27340353	-0.00031958	0.00019576	0.00032118	-0.00000116	0.00000000
		-6.27340361	-0.00031954	0.00019574	0.00032115	-0.00000116	0.00000000
		-6.27340369	-0.00031951	0.00019572	0.00032111	-0.00000116	0.00000000
		-6.27340377	-0.00031948	0.00019570	0.00032108	-0.00000116	0.00000000
		-6.27340384	-0.00031945	0.00019568	0.00032105	-0.00000116	0.00000000
		-6.27340392	-0.00031941	0.00019566	0.00032102	-0.00000116	0.00000000
		-6.27340400	-0.00031938	0.00019564	0.00032098	-0.00000116	0.00000000
		-6.27340408	-0.00031935	0.00019562	0.00032095	-0.00000116	0.00000000
		-6.27340416	-0.00031931	0.00019560	0.00032092	-0.00000116	0.00000000
		-6.27340424	-0.00031928	0.00019558	0.00032088	-0.00000116	0.00000000
		-6.27340432	-0.00031925	0.00019556	0.00032085	-0.00000116	0.00000000
		-6.27340440	-0.00031922	0.00019554	0.00032082	-0.00000116	0.00000000
		-6.27340448	-0.00031918	0.00019552	0.00032078	-0.00000116	0.00000000
		-6.27340455	-0.00031915	0.00019550	0.00032075	-0.00000116	0.00000000
		-6.27340463	-0.00031912	0.00019548	0.00032072	-0.00000116	0.00000000
		-6.27340471	-0.00031908	0.00019546	0.00032068	-0.00000116	0.00000000
		-6.27340479	-0.00031905	0.00019544	0.00032065	-0.00000116	0.00000000
		-6.27340487	-0.00031902	0.00019542	0.00032062	-0.00000116	0.00000000
		-6.27340495	-0.00031899	0.00019540	0.00032059	-0.00000116	0.00000000
		-6.27340503	-0.00031895	0.00019538	0.00032055	-0.00000116	0.00000000
		-6.27340511	-0.00031892	0.00019536	0.00032052	-0.00000116	0.00000000
		-6.27340518	-0.00031889	0.00019534	0.00032049	-0.00000116	0.00000000
		-6.27340526	-0.00031885	0.00019532	0.00032045	-0.00000116	0.00000000
		-6.27340534	-0.00031882	0.00019530	0.00032042	-0.00000116	0.00000000
		-6.27340542	-0.00031879	0.00019528	0.00032039	-0.00000116	0.00000000
		-6.27340550	-0.00031876	0.00019526	0.00032035	-0.00000116	0.00000000
		-6.27340558	-0.00031872	0.00019524	0.00032032	-0.00000116	0.00000000
		-6.27340566	-0.00031869	0.00019522	0.00032029	-0.00000116	0.00000000
		-6.27340573	-0.00031866	0.00019520	0.00032026	-0.00000116	0.00000000
		-6.27340581	-0.00031862	0.00019518	0.00032022	-0.00000116	0.00000000
		-6.27340589	-0.00031859	0.00019516	0.00032019	-0.00000116	0.00000000
		-6.27340597	-0.00031856	0.00019514	0.00032016	-0.00000116	0.00000000
		-6.27340605	-0.00031853	0.00019512	0.00032012	-0.00000116	0.00000000
		-6.27340613	-0.00031849	0.00019510	0.00032009	-0.00000116	0.00000000
		-6.27340621	-0.00031846	0.00019508	0.00032006	-0.00000116	0.00000000
		-6.27340628	-0.00031843	0.00019506	0.00032002	-0.00000116	0.00000000
		-6.27340636	-0.00031839	0.00019504	0.00031999	-0.00000116	0.00000000
		-6.27340644	-0.00031836	0.00019502	0.00031996	-0.00000116	0.00000000
		-6.27340652	-0.00031833	0.00019500	0.00031993	-0.00000116	0.00000000
		-6.27340660	-0.00031830	0.00019498	0.00031989	-0.00000116	0.00000000
		-6.27340668	-0.00031826	0.00019496	0.00031986	-0.00000116	0.00000000
		-6.27340675	-0.00031823	0.00019494	0.00031983	-0.00000116	0.00000000
		-6.27340683	-0.00031820	0.00019492	0.00031979	-0.00000116	0.00000000
		-6.27340691	-0.00031817	0.00019490	0.00031976	-0.00000116	0.00000000
		-6.27340699	-0.00031813	0.00019488	0.00031973	-0.00000116	0.00000000
		-6.27340707	-0.00031810	0.00019486	0.00031970	-0.00000116	0.00000000
		-6.27340715	-0.00031807	0.00019484	0.00031966	-0.00000116	0.00000000
		-6.27340722	-0.00031803	0.00019482	0.00031963	-0.00000116	0.00000000
		-6.27340730	-0.00031800	0.00019480	0.00031960	-0.00000116	0.00000000
		-6.27340738	-0.00031797	0.00019478	0.00031956	-0.00000116	0.00000000
		-6.27340746	-0.00031794	0.00019476	0.00031953	-0.00000116	0.00000000
		-6.27340754	-0.00031790	0.00019474	0.00031950	-0.00000116	0.00000000
		-6.27340761	-0.00031787	0.00019472	0.00031947	-0.00000116	0.00000000
		-6.27340769	-0.00031784	0.00019470	0.00031943	-0.00000116	0.00000000
		-6.27340777	-0.00031781	0.00019468	0.00031940	-0.00000116	0.00000000
		-6.27340785	-0.00031777	0.00019466	0.00031937	-0.00000116	0.00000000
		-6.27340793	-0.00031774	0.00019464	0.00031933	-0.00000116	0.00000000
		-6.27340801	-0.00031771	0.00019462	0.00031930	-0.00000116	0.00000000
		-6.27340808	-0.00031768	0.00019460	0.00031927	-0.00000116	0.00000000
		-6.27340816	-0.00031764	0.00019458	0.00031924	-0.00000116	0.00000000
		-6.27340824	-0.00031761	0.00019456	0.00031920	-0.00000116	0.00000000
		-6.27340832	-0.00031758	0.00019454	0.00031917	-0.00000116	0.00000000
		-6.27340840	-0.00031755	0.00019452	0.00031914	-0.00000116	0.00000000
		-6.27340847	-0.00031751	0.00019450	0.00031911	-0.00000116	0.00000000
		-6.27340855	-0.00031748	0.00019448	0.00031907	-0.00000116	0.00000000
		-6.27340863	-0.00031745	0.00019446	0.00031904	-0.00000116	0.00000000
		-6.27340871	-0.00031742	0.00019444	0.00031901	-0.00000116	0.00000000
		-6.27340879	-0.00031738	0.00019442	0.00031897	-0.00000116	0.00000000
		-6.27340886	-0.00031735	0.00019440	0.00031894	-0.00000116	0.00000000
		-6.27340894	-0.00031732	0.00019438	0.00031891	-0.00000116	0.00000000
		-6.27340902	-0.00031728	0.00019436	0.00031888	-0.00000116	0.00000000
		-6.27340910	-0.00031725	0.00019434	0.00031884	-0.00000116	0.00000000
		-6.27340917	-0.00031722	0.00019432	0.00031881	-0.00000116	0.00000000
		-6.27340925	-0.00031719	0.00019430	0.00031878	-0.00000116	0.00000000
		-6.27340933	-0.00031715	0.00019428	0.00031875	-0.00000116	0.00000000
		-6.27340941	-0.00031712	0.00019426	0.00031871	-0.00000116	0.00000000
		-6.27340949	-0.00031709	0.00019424	0.00031868	-0.00000116	0.00000000
		-6.27340956	-0.00031706	0.00019422	0.00031865	-0.00000116	0.00000000
		-6.27340964	-0.00031702	0.00019420	0.00031862	-0.00000116	0.00000000
		-6.27340972	-0.00031699	0.00019418	0.00031858	-0.00000116	0.00000000
		-6.27340980	-0.00031696	0.00019416	0.00031855	-0.00000116	0.00000000
		-6.27340987	-0.00031693	0.00019414	0.00031852	-0.00000116	0.00000000
		-6.27340995	-0.00031690	0.00019412	0.00031848	-0.00000116	0.00000000
		-6.27341003	-0.00031686	0.00019410	0.00031845	-0.00000116	0.00000000
		-6.27341011	-0.00031683	0.00019408	0.00031842	-0.00000116	0.00000000
		-6.27341019	-0.00031680	0.00019406	0.00031839	-0.00000116	0.00000000
		-6.27341026	-0.00031677	0.00019404	0.00031835	-0.00000116	0.00000000
		-6.27341034	-0.00031673	0.00019402	0.00031832	-0.00000116	0.00000000
		-6.27341042	-0.00031670	0.00019400	0.00031829	-0.00000116	0.00000000
		-6.27341050	-0.00031667	0.00019398	0.00031826	-0.00000116	0.00000000
		-6.27341057	-0.00031664	0.00019396	0.00031822	-0.00000116	0.00000000
		-6.27341065	-0.00031660	0.00019394	0.00031819	-0.00000116	0.00000000
		-6.27341073	-0.00031657	0.00019392	0.00031816	-0.00000116	0.00000000
		-6.27341081	-0.00031654	0.00019390	0.00031813	-0.00000116	0.00000000
		-6.27341088	-0.00031651	0.00019388	0.00031809	-0.00000116	0.00000000
		-6.27341096	-0.00031647	0.00019386	0.00031806	-0.00000116	0.00000000
		-6.27341104	-0.00031644	0.00019384	0.00031803	-0.00000116	0.00000000
		-6.27341112	-0.00031641	0.00019382	0.00031800	-0.00000116	0.00000000
		-6.27341119	-0.00031638	0.00019380	0.00031796	-0.00000116	0.00000000
		-6.27341127	-0.00031634	0.00019378	0.00031793	-0.00000116	0.00000000
		-6.27341135	-0.00031631	0.00019376	0.00031790	-0.00000116	0.00000000
		-6.27341143	-0.00031628	0.00019374	0.00031787	-0.00000116	0.00000000
		-6.27341150	-0.00031625	0.00019372	0.00031783	-0.00000116	0.00000000
		-6.27341158	-0.00031622	0.00019370	0.00031780	-0.00000116	0.00000000
		-6.27341166	-0.00031618	0.00019368	0.00031777	-0.00000116	0.00000000
		-6.27341173	-0.00031615	0.00019366	0.00031774	-0.00000116	0.00000000
		-6.27341181	-0.00031612	0.00019364	0.00031770	-0.00000116	0.00000000
		-6.27341189	-0.00031609	0.00019362	0.00031767	-0.00000116	0.00000000
		-6.27341197	-0.00031605	0.00019360	0.00031764	-0.00000116	0.00000000
		-6.27341204	-0.00031602	0.00019358	0.00031761	-0.00000116	0.00000000
		-6.27341212	-0.00031599	0.00019356	0.00031757	-0.00000116	0.00000000
		-6.27341220	-0.00031596	0.00019354	0.00031754	-0.00000116	0.00000000
		-6.27341228	-0.00031592	0.00019352	0.00031751	-0.00000116	0.00000000
		-6.27341235	-0.00031589	0.00019350	0.00031748	-0.00000116	0.00000000
		-6.27341243	-0.00031586	0.00019348	0.00031744	-0.00000116	0.00000000
		-6.27341251	-0.00031583	0.00019346	0.00031741	-0.00000116	0.00000000
		-6.27341258	-0.00031580	0.00019344	0.00031738	-0.00000116	0.00000000
		-6.27341266	-0.00031576	0.00019342	0.00031735	-0.00000116	0.00000000
		-6.27341274	-0.00031573	0.00019340	0.00031732	-0.00000116	0.00000000
		-6.27341282	-0.00031570	0.00019338	0.00031728	-0.00000116	0.00000000
		-6.27341289	-0.00031567	0.00019336	0.00031725	-0.00000116	0.00000000
		-6.27341297	-0.00031563	0.00019334	0.00031722	-0.00000116	0.00000000
		-6.27341305	-0.00031560	0.00019333	0.00031719	-0.00000116	0.00000000
		-6.27341312	-0.00031557	0.00019331	0.00031715	-0.00000116	0.00000000
		-6.27341320	-0.00031554	0.00019329	0.00031712	-0.00000116	0.00000000
		-6.27341328	-0.00031551	0.00019327	0.00031709	-0.00000116	0.00000000
		-6.27341336	-0.00031547	0.00019325	0.00031706	-0.00000116	0.00000000
		-6.27341343	-0.00031544	0.00019323	0.00031702	-0.00000116	0.00000000
		-6.27341351	-0.00031541	0.00019321	0.00031699	-0.00000116	0.00000000
		-6.27341359	-0.00031538	0.00019319	0.00031696	-0.00000116	0.00000000
		-6.27341366	-0.00031535	0.00019317	0.00031693	-0.00000116	0.00000000
		-6.27341374	-0.00031531	0.00019315	0.00031689	-0.00000116	0.00000000
		-6.27341382	-0.00031528	0.00019313	0.00031686	-0.00000116	0.00000000
		-6.27341389	-0.00031525	0.00019311	0.00031683	-0.00000116	0.00000000
		-6.27341397	-0.00031522	0.00019309	0.00031680	-0.00000116	0.00000000
		-6.27341405	-0.00031519	0.00019307	0.00031677	-0.00000116	0.00000000
		-6.27341412	-0.00031515	0.00019305	0.00031673	-0.00000116	0.00000000
		-6.27341420	-0.00031512	0.00019303	0.00031670	-0.00000116	0.00000000
		-6.27341428	-0.00031509	0.00019301	0.00031667	-0.00000116	0.00000000
		-6.27341435	-0.00031506	0.00019299	0.00031664	-0.00000116	0.00000000
		-6.27341443	-0.00031502	0.00019297	0.00031660	-0.00000116	0.00000000
		-6.27341451	-0.00031499	0.00019295	0.00031657	-0.00000116	0.00000000
		-6.27341459	-0.00031496	0.00019293	0.00031654	-0.00000116	0.00000000
		-6.27341466	-0.00031493	0.00019291	0.00031651	-0.00000116	0.00000000
		-6.27341474	-0.00031490	0.00019289	0.00031648	-0.00000116	0.00000000
		-6.27341482	-0.00031486	0.00019287	0.00031644	-0.00000116	0.00000000
		-6.27341489	-0.00031483	0.00019285	0.00031641	-0.00000116	0.00000000
		-6.27341497	-0.00031480	0.00019283	0.00031638	-0.00000116	0.00000000
		-6.27341505	-0.00031477	0.00019281	0.00031635	-0.00000116	0.00000000
		-6.27341512	-0.00031474	0.00019279	0.00031631	-0.00000116	0.00000000
		-6.27341520	-0.00031470	0.00019277	0.00031628	-0.00000116	0.00000000
		-6.27341528	-0.00031467	0.00019275	0.00031625	-0.00000116	0.00000000
		-6.27341535	-0.00031464	0.00019274	0.00031622	-0.00000116	0.00000000
		-6.27341543	-0.00031461	0.00019272	0.00031619	-0.00000116	0.00000000
		-6.27341550	-0.00031458	0.00019270	0.00031615	-0.00000116	0.00000000
		-6.27341558	-0.00031454	0.00019268	0.00031612	-0.00000116	0.00000000
		-6.27341566	-0.00031451	0.00019266	0.00031609	-0.00000116	0.00000000
		-6.27341573	-0.00031448	0.00019264	0.00031606	-0.00000116	0.00000000
		-6.27341581	-0.00031445	0.00019262	0.00031603	-0.00000116	0.00000000
		-6.27341589	-0.00031442	0.00019260	0.00031599	-0.00000116	0.00000000
		-6.27341596	-0.00031438	0.00019258	0.00031596	-0.00000116	0.00000000
		-6.27341604	-0.00031435	0.00019256	0.00031593	-0.00000116	0.00000000
		-6.27341612	-0.00031432	0.00019254	0.00031590	-0.00000116	0.00000000
		-6.27341619	-0.00031429	0.00019252	0.00031587	-0.00000116	0.00000000
		-6.27341627	-0.00031426	0.00019250	0.00031583	-0.00000116	0.00000000
		-6.27341635	-0.00031423	0.00019248	0.00031580	-0.00000116	0.00000000
		-6.27341642	-0.00031419	0.00019246	0.00031577	-0.00000116	0.00000000
		-6.27341650	-0.00031416	0.00019244	0.00031574	-0.00000116	0.00000000
		-6.27341658	-0.00031413	0.00019242	0.00031571	-0.00000116	0.00000000
		-6.27341665	-0.00031410	0.00019240	0.00031567	-0.00000116	0.00000000
		-6.27341673	-0.00031407	0.00019238	0.00031564	-0.00000116	0.00000000
		-6.27341680	-0.00031403	0.00019236	0.00031561	-0.00000116	0.00000000
		-6.27341688	-0.00031400	0.00019234	0.00031558	-0.00000116	0.00000000
		-6.27341696	-0.00031397	0.00019232	0.00031554	-0.00000116	0.00000000
		-6.27341703	-0.00031394	0.00019231	0.00031551	-0.00000116	0.00000000
		-6.27341711	-0.00031391	0.00019229	0.00031548	-0.00000116	0.00000000
		-6.27341719	-0.00031387	0.00019227	0.00031545	-0.00000116	0.00000000
		-6.27341726	-0.00031384	0.00019225	0.00031542	-0.00000116	0.00000000
		-6.27341734	-0.00031381	0.00019223	0.00031538	-0.00000116	0.00000000
		-6.27341741	-0.00031378	0.00019221	0.00031535	-0.00000116	0.00000000
		-6.27341749	-0.00031375	0.00019219	0.00031532	-0.00000116	0.00000000
		-6.27341757	-0.00031372	0.00019217	0.00031529	-0.00000116	0.00000000
		-6.27341764	-0.00031368	0.00019215	0.00031526	-0.00000116	0.00000000
		-6.27341772	-0.00031365	0.00019213	0.00031523	-0.00000116	0.00000000
		-6.27341779	-0.00031362	0.00019211	0.00031519	-0.00000116	0.00000000
		-6.27341787	-0.00031359	0.00019209	0.00031516	-0.00000116	0.00000000
		-6.27341795	-0.00031356	0.00019207	0.00031513	-0.00000116	0.00000000
		-6.27341802	-0.00031353	0.00019205	0.00031510	-0.00000116	0.00000000
		-6.27341810	-0.00031349	0.00019203	0.00031507	-0.00000116	0.00000000
		-6.27341817	-0.00031346	0.00019201	0.00031503	-0.00000116	0.00000000
		-6.27341825	-0.00031343	0.00019199	0.00031500	-0.00000116	0.00000000
		-6.27341833	-0.00031340	0.00019197	0.00031497	-0.00000116	0.00000000
		-6.27341840	-0.00031337	0.00019195	0.00031494	-0.00000116	0.00000000
		-6.27341848	-0.00031333	0.00019194	0.00031491	-0.00000116	0.00000000
		-6.27341855	-0.00031330	0.00019192	0.00031487	-0.00000116	0.00000000
		-6.27341863	-0.00031327	0.00019190	0.00031484	-0.00000116	0.00000000
		-6.27341871	-0.00031324	0.00019188	0.00031481	-0.00000116	0.00000000
		-6.27341878	-0.00031321	0.00019186	0.00031478	-0.00000116	0.00000000
		-6.27341886	-0.00031318	0.00019184	0.00031475	-0.00000116	0.00000000
		-6.27341893	-0.00031314	0.00019182	0.00031471	-0.00000116	0.00000000
		-6.27341901	-0.00031311	0.00019180	0.00031468	-0.00000116	0.00000000
		-6.27341909	-0.00031308	0.00019178	0.00031465	-0.00000116	0.00000000
		-6.27341916	-0.00031305	0.00019176	0.00031462	-0.00000116	0.00000000
		-6.27341924	-0.00031302	0.00019174	0.00031459	-0.00000116	0.00000000
		-6.27341931	-0.00031299	0.00019172	0.00031456	-0.00000116	0.00000000
		-6.27341939	-0.00031295	0.00019170	0.00031452	-0.00000116	0.00000000
		-6.27341946	-0.00031292	0.00019168	0.00031449	-0.00000116	0.00000000
		-6.27341954	-0.00031289	0.00019166	0.00031446	-0.00000116	0.00000000
		-6.27341962	-0.00031286	0.00019164	0.00031443	-0.00000116	0.00000000
		-6.27341969	-0.00031283	0.00019162	0.00031440	-0.00000116	0.00000000
		-6.27341977	-0.00031280	0.00019161	0.00031437	-0.00000116	0.00000000
		-6.27341984	-0.00031276	0.00019159	0.00031433	-0.00000116	0.00000000
		-6.27341992	-0.00031273	0.00019157	0.00031430	-0.00000116	0.00000000
		-6.27341999	-0.00031270	0.00019155	0.00031427	-0.00000116	0.00000000
		-6.27342007	-0.00031267	0.00019153	0.00031424	-0.00000116	0.00000000
		-6.27342015	-0.00031264	0.00019151	0.00031421	-0.00000116	0.00000000
		-6.27342022	-0.00031261	0.00019149	0.00031417	-0.00000116	0.00000000
		-6.27342030	-0.00031258	0.00019147	0.00031414	-0.00000116	0.00000000
		-6.27342037	-0.00031254	0.00019145	0.00031411	-0.00000116	0.00000000
		-6.27342045	-0.00031251	0.00019143	0.00031408	-0.00000116	0.00000000
		-6.27342052	-0.00031248	0.00019141	0.00031405	-0.00000116	0.00000000
		-6.27342060	-0.00031245	0.00019139	0.00031402	-0.00000116	0.00000000
		-6.27342067	-0.00031242	0.00019137	0.00031398	-0.00000116	0.00000000
		-6.27342075	-0.00031239	0.00019135	0.00031395	-0.00000116	0.00000000
		-6.27342083	-0.00031235	0.00019133	0.00031392	-0.00000116	0.00000000
		-6.27342090	-0.00031232	0.00019132	0.00031389	-0.00000116	0.00000000
		-6.27342098	-0.00031229	0.00019130	0.00031386	-0.00000116	0.00000000
		-6.27342105	-0.00031226	0.00019128	0.00031383	-0.00000116	0.00000000
		-6.27342113	-0.00031223	0.00019126	0.00031379	-0.00000116	0.00000000
		-6.27342120	-0.00031220	0.00019124	0.00031376	-0.00000116	0.00000000
		-6.27342128	-0.00031217	0.00019122	0.00031373	-0.00000116	0.00000000
		-6.27342135	-0.00031213	0.00019120	0.00031370	-0.00000116	0.00000000
		-6.27342143	-0.00031210	0.00019118	0.00031367	-0.00000116	0.00000000
		-6.27342150	-0.00031207	0.00019116	0.00031364	-0.00000116	0.00000000
		-6.27342158	-0.00031204	0.00019114	0.00031360	-0.00000116	0.00000000
		-6.27342166	-0.00031201	0.00019112	0.00031357	-0.00000116	0.00000000
		-6.27342173	-0.00031198	0.00019110	0.00031354	-0.00000116	0.00000000
		-6.27342181	-0.00031195	0.00019108	0.00031351	-0.00000116	0.00000000
		-6.27342188	-0.00031191	0.00019106	0.00031348	-0.00000116	0.00000000
		-6.27342196	-0.00031188	0.00019105	0.00031345	-0.00000116	0.00000000
		-6.27342203	-0.00031185	0.00019103	0.00031341	-0.00000116	0.00000000
		-6.27342211	-0.00031182	0.00019101	0.00031338	-0.00000116	0.00000000
		-6.27342218	-0.00031179	0.00019099	0.00031335	-0.00000116	0.00000000
		-6.27342226	-0.00031176	0.00019097	0.00031332	-0.00000116	0.00000000
		-6.27342233	-0.00031173	0.00019095	0.00031329	-0.00000116	0.00000000
		-6.27342241	-0.00031169	0.00019093	0.00031326	-0.00000116	0.00000000
		-6.27342248	-0.00031166	0.00019091	0.00031323	-0.00000116	0.00000000
		-6.27342256	-0.00031163	0.00019089	0.00031319	-0.00000116	0.00000000
		-6.27342263	-0.00031160	0.00019087	0.00031316	-0.00000116	0.00000000
		-6.27342271	-0.00031157	0.00019085	0.00031313	-0.00000116	0.00000000
		-6.27342278	-0.00031154	0.00019083	0.00031310	-0.00000116	0.00000000
		-6.27342286	-0.00031151	0.00019081	0.00031307	-0.00000116	0.00000000
		-6.27342293	-0.00031147	0.00019080	0.00031304	-0.00000116	0.00000000
		-6.27342301	-0.00031144	0.00019078	0.00031300	-0.00000116	0.00000000
		-6.27342308	-0.00031141	0.00019076	0.00031297	-0.00000116	0.00000000
		-6.27342316	-0.00031138	0.00019074	0.00031294	-0.00000116	0.00000000
		-6.27342323	-0.00031135	0.00019072	0.00031291	-0.00000116	0.00000000
		-6.27342331	-0.00031132	0.00019070	0.00031288	-0.00000116	0.00000000
		-6.27342338	-0.00031129	0.00019068	0.00031285	-0.00000116	0.00000000
		-6.27342346	-0.00031126	0.00019066	0.00031282	-0.00000116	0.00000000
		-6.27342353	-0.00031122	0.00019064	0.00031278	-0.00000116	0.00000000
		-6.27342361	-0.00031119	0.00019062	0.00031275	-0.00000116	0.00000000
		-6.27342368	-0.00031116	0.00019060	0.00031272	-0.00000116	0.00000000
		-6.27342376	-0.00031113	0.00019058	0.00031269	-0.00000116	0.00000000
		-6.27342383	-0.00031110	0.00019057	0.00031266	-0.00000116	0.00000000
		-6.27342391	-0.00031107	0.00019055	0.00031263	-0.00000116	0.00000000
		-6.27342398	-0.00031104	0.00019053	0.00031260	-0.00000116	0.00000000
		-6.27342406	-0.00031101	0.00019051	0.00031256	-0.00000116	0.00000000
		-6.27342413	-0.00031097	0.00019049	0.00031253	-0.00000116	0.00000000
		-6.27342421	-0.00031094	0.00019047	0.00031250	-0.00000116	0.00000000
		-6.27342428	-0.00031091	0.00019045	0.00031247	-0.00000116	0.00000000
		-6.27342436	-0.00031088	0.00019043	0.00031244	-0.00000116	0.00000000
		-6.27342443	-0.00031085	0.00019041	0.00031241	-0.00000116	0.00000000
		-6.27342451	-0.00031082	0.00019039	0.00031238	-0.00000116	0.00000000
		-6.27342458	-0.00031079	0.00019037	0.00031235	-0.00000116	0.00000000
		-6.27342466	-0.00031076	0.00019035	0.00031231	-0.00000116	0.00000000
		-6.27342473	-0.00031072	0.00019034	0.00031228	-0.00000116	0.00000000
		-6.27342480	-0.00031069	0.00019032	0.00031225	-0.00000116	0.00000000
		-6.27342488	-0.00031066	0.00019030	0.00031222	-0.00000116	0.00000000
		-6.27342495	-0.00031063	0.00019028	0.00031219	-0.00000116	0.00000000
		-6.27342503	-0.00031060	0.00019026	0.00031216	-0.00000116	0.00000000
		-6.27342510	-0.00031057	0.00019024	0.00031213	-0.00000116	0.00000000
		-6.27342518	-0.00031054	0.00019022	0.00031209	-0.00000116	0.00000000
		-6.27342525	-0.00031051	0.00019020	0.00031206	-0.00000116	0.00000000
		-6.27342533	-0.00031047	0.00019018	0.00031203	-0.00000116	0.00000000
		-6.27342540	-0.00031044	0.00019016	0.00031200	-0.00000116	0.00000000
		-6.27342548	-0.00031041	0.00019014	0.00031197	-0.00000116	0.00000000
		-6.27342555	-0.00031038	0.00019013	0.00031194	-0.00000116	0.00000000
		-6.27342563	-0.00031035	0.00019011	0.00031191	-0.00000116	0.00000000
		-6.27342570	-0.00031032	0.00019009	0.00031188	-0.00000116	0.00000000
		-6.27342577	-0.00031029	0.00019007	0.00031184	-0.00000116	0.00000000
		-6.27342585	-0.00031026	0.00019005	0.00031181	-0.00000116	0.00000000
		-6.27342592	-0.00031023	0.00019003	0.00031178	-0.00000116	0.00000000
		-6.27342600	-0.00031019	0.00019001	0.00031175	-0.00000116	0.00000000
		-6.27342607	-0.00031016	0.00018999	0.00031172	-0.00000116	0.00000000
		-6.27342615	-0.00031013	0.00018997	0.00031169	-0.00000116	0.00000000
		-6.27342622	-0.00031010	0.00018995	0.00031166	-0.00000116	0.00000000
		-6.27342630	-0.00031007	0.00018993	0.00031163	-0.00000116	0.00000000
		-6.27342637	-0.00031004	0.00018992	0.00031159	-0.00000116	0.00000000
		-6.27342644	-0.00031001	0.00018990	0.00031156	-0.00000116	0.00000000
		-6.27342652	-0.00030998	0.00018988	0.00031153	-0.00000116	0.00000000
		-6.27342659	-0.00030995	0.00018986	0.00031150	-0.00000116	0.00000000
		-6.27342667	-0.00030992	0.00018984	0.00031147	-0.00000116	0.00000000
		-6.27342674	-0.00030988	0.00018982	0.00031144	-0.00000116	0.00000000
		-6.27342682	-0.00030985	0.00018980	0.00031141	-0.00000116	0.00000000
		-6.27342689	-0.00030982	0.00018978	0.00031138	-0.00000116	0.00000000
		-6.27342696	-0.00030979	0.00018976	0.00031134	-0.00000116	0.00000000
		-6.27342704	-0.00030976	0.00018974	0.00031131	-0.00000116	0.00000000
		-6.27342711	-0.00030973	0.00018973	0.00031128	-0.00000116	0.00000000
		-6.27342719	-0.00030970	0.00018971	0.00031125	-0.00000116	0.00000000
		-6.27342726	-0.00030967	0.00018969	0.00031122	-0.00000116	0.00000000
		-6.27342733	-0.00030964	0.00018967	0.00031119	-0.00000116	0.00000000
		-6.27342741	-0.00030961	0.00018965	0.00031116	-0.00000116	0.00000000
		-6.27342748	-0.00030957	0.00018963	0.00031113	-0.00000116	0.00000000
		-6.27342756	-0.00030954	0.00018961	0.00031110	-0.00000116	0.00000000
		-6.27342763	-0.00030951	0.00018959	0.00031106	-0.00000116	0.00000000
		-6.27342771	-0.00030948	0.00018957	0.00031103	-0.00000116	0.00000000
		-6.27342778	-0.00030945	0.00018956	0.00031100	-0.00000116	0.00000000]; 

	%cut off series at maxdeg: 
	love_numbers(maxdeg+2:end,:)=[];

	%retrive right type: 
	if strcmpi(type,'loadingverticaldisplacement'),
		series=love_numbers(:,1);
	elseif strcmpi(type,'loadinggravitationalpotential'),
		series=love_numbers(:,2);
	elseif strcmpi(type,'loadinghorizontaldisplacement'),
		series=love_numbers(:,3);
	elseif strcmpi(type,'tidalverticaldisplacement'),
		series=love_numbers(:,4);
	elseif strcmpi(type,'tidalgravitationalpotential'),
		series=love_numbers(:,5);
	elseif strcmpi(type,'tidalhorizontaldisplacement'),
		series=love_numbers(:,6);
	else 
		error(['love_numbers error message: unknown type:' type]);
	end

	% choose degree 1 term for CF reference system 
	if strcmpi(frame,'CM'), 
		return; 
	elseif strcmpi(frame,'CF'), % from Blewitt, 2003, JGR
		if strcmpi(type,'loadingverticaldisplacement'),
			series(2,1) = -0.269; 
		elseif strcmpi(type,'loadinggravitationalpotential'),
			series(2,1) = 0.021;  
		elseif strcmpi(type,'loadinghorizontaldisplacement'),
			series(2,1) = 0.134; 
		end
	else 
		error(['love_numbers error message: unknown reference frame:' frame]);
	end



