%SMBautoregression Class definition
%
%   Usage:
%      SMBautoregression=SMBautoregression();

classdef SMBautoregression
	properties (SetAccess=public)
		num_basins        = 0;
		beta0             = NaN;
		beta1             = NaN;
		ar_order          = 0;
		ar_initialtime    = 0;
		ar_timestep       = 0;
		phi               = NaN;
		basin_id          = NaN;
		steps_per_step    = 1;
		averaging         = 0;
		requested_outputs = {};
	end
	methods
		function self = SMBautoregression(varargin) % {{{
			switch nargin
				case 0
					self=setdefaultparameters(self);
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function self = extrude(self,md) % {{{
			%Nothing for now
		end % }}}
		function list = defaultoutputs(self,md) % {{{
			list = {''};
		end % }}}
		function self = initialize(self,md) % {{{
			if isnan(self.beta1)
				self.beta1 = zeros(1,self.num_basins); %no trend in SMB
				disp('      smb.beta1 (trend) not specified: value set to 0');
			end
			if (self.ar_order==0)
				self.ar_order = 1; %dummy 1 value for autoregression
				self.phi      = zeros(self.num_basins,self.ar_order); %autoregression coefficients all set to 0 
				disp('      smb.ar_order (order of autoregressive model) not specified: order of autoregressive model set to 0');
			end
			if (self.ar_initialtime==0)
				self.ar_initialtime = md.timestepping.start_time; %autoregression model has no prescribed initial time
				disp('      smb.ar_initialtime (initial time in the autoregressive model parameterization) not specified: set to md.timestepping.start_time');
			end
			if (self.ar_timestep==0)
				self.ar_timestep = md.timestepping.time_step; %autoregression model has no prescribed time step
				disp('      smb.ar_timestep (timestep of autoregressive model) not specified: set to md.timestepping.time_step');
			end
			if isnan(self.phi)
				self.phi = zeros(self.num_basins,self.ar_order); %autoregression model of order 0 
				disp('      smb.phi (lag coefficients) not specified: order of autoregressive model set to 0');
			end
		end % }}}
		function self = setdefaultparameters(self) % {{{
			self.ar_order    = 0.0; %autoregression model of order 0
		end % }}}
		function md = checkconsistency(self,md,solution,analyses) 

			if ismember('MasstransportAnalysis',analyses),
				md = checkfield(md,'fieldname','smb.num_basins','numel',1,'NaN',1,'Inf',1,'>',0);
				md = checkfield(md,'fieldname','smb.basin_id','Inf',1,'>=',0,'<=',md.smb.num_basins,'size',[md.mesh.numberofelements,1]);
				md = checkfield(md,'fieldname','smb.beta0','NaN',1,'Inf',1,'size',[1,md.smb.num_basins],'numel',md.smb.num_basins); %scheme fails if passed as column vector
				md = checkfield(md,'fieldname','smb.beta1','NaN',1,'Inf',1,'size',[1,md.smb.num_basins],'numel',md.smb.num_basins); %scheme fails if passed as column vector
				md = checkfield(md,'fieldname','smb.ar_order','numel',1,'NaN',1,'Inf',1,'>=',0);
				md = checkfield(md,'fieldname','smb.ar_initialtime','numel',1,'NaN',1,'Inf',1); 
				md = checkfield(md,'fieldname','smb.ar_timestep','numel',1,'NaN',1,'Inf',1,'>=',md.timestepping.time_step); %autoregression time step cannot be finer than ISSM timestep
				md = checkfield(md,'fieldname','smb.phi','NaN',1,'Inf',1,'size',[md.smb.num_basins,md.smb.ar_order]);
			end
			md = checkfield(md,'fieldname','smb.steps_per_step','>=',1,'numel',[1]);
			md = checkfield(md,'fieldname','smb.averaging','numel',[1],'values',[0 1 2]);
			md = checkfield(md,'fieldname','smb.requested_outputs','stringrow',1);
		end 
		function disp(self) % {{{
			disp(sprintf('   surface forcings parameters:'));
			fielddisplay(self,'num_basins','number of different basins [unitless]');
			fielddisplay(self,'basin_id','basin number assigned to each element [unitless]');
			fielddisplay(self,'beta0','basin-specific intercept values [m ice eq./yr]');
			fielddisplay(self,'beta1','basin-specific trend values [m ice eq. yr^(-2)]');
			fielddisplay(self,'ar_order','order of the autoregressive model [unitless]');
			fielddisplay(self,'ar_initialtime','initial time assumed in the autoregressive model parameterization [yr]');
			fielddisplay(self,'ar_timestep','time resolution of the autoregressive model [yr]');
			fielddisplay(self,'phi','basin-specific vectors of lag coefficients [unitless]');
			fielddisplay(self, 'steps_per_step', 'number of smb steps per time step');
			fielddisplay(self, 'averaging', 'averaging methods from short to long steps');
			disp(sprintf('%51s  0: Arithmetic (default)',' '));
			disp(sprintf('%51s  1: Geometric',' '));
			disp(sprintf('%51s  2: Harmonic',' '));
			fielddisplay(self,'requested_outputs','additional outputs requested');

		end % }}}
		function marshall(self,prefix,md,fid) % {{{

			yts=md.constants.yts;

			WriteData(fid,prefix,'name','md.smb.model','data',55,'format','Integer');
			WriteData(fid,prefix,'object',self,'class','smb','fieldname','num_basins','format','Integer');
			WriteData(fid,prefix,'object',self,'class','smb','fieldname','ar_order','format','Integer');
			WriteData(fid,prefix,'object',self,'class','smb','fieldname','ar_initialtime','format','Double','scale',yts);
			WriteData(fid,prefix,'object',self,'class','smb','fieldname','ar_timestep','format','Double','scale',yts);
			WriteData(fid,prefix,'object',self,'class','smb','fieldname','basin_id','data',self.basin_id-1,'name','md.smb.basin_id','format','IntMat','mattype',2); %0-indexed
			WriteData(fid,prefix,'object',self,'class','smb','fieldname','beta0','format','DoubleMat','name','md.smb.beta0','scale',1./yts,'yts',yts);
			WriteData(fid,prefix,'object',self,'class','smb','fieldname','beta1','format','DoubleMat','name','md.smb.beta1','scale',1./(yts^2),'yts',yts);
			WriteData(fid,prefix,'object',self,'class','smb','fieldname','phi','format','DoubleMat','name','md.smb.phi','yts',yts); 
			WriteData(fid,prefix,'object',self,'fieldname','steps_per_step','format','Integer');
			WriteData(fid,prefix,'object',self,'fieldname','averaging','format','Integer');

			%process requested outputs
			outputs = self.requested_outputs;
			pos  = find(ismember(outputs,'default'));
			if ~isempty(pos),
				outputs(pos) = [];                         %remove 'default' from outputs
				outputs      = [outputs defaultoutputs(self,md)]; %add defaults
			end
			WriteData(fid,prefix,'data',outputs,'name','md.smb.requested_outputs','format','StringArray');

		end % }}}
	end
end
